'''
Defines an L{AEEvent} indicating an event of interest occurred in an 
L{AEChooser}.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base, Constants
import AEChooser

class ChooserChange(Base.AccessEngineEvent):
  '''  
  Event that fires when a L{AEChooser} indicates important input from a user.
  
  This class registers its name and whether it should be monitored by default in
  an L{AEMonitor.EventMonitor} using the L{Constants.registerEventType} function
  when this module is first imported. The L{AEMonitor.EventMonitor} can use this
  information to build its menus.
  
  @ivar chooser: L{AEChooser} that fired this event
  @type chooser: L{AEChooser}
  @ivar kind: Kind of event, one of OK, CANCEL, or APPLY from L{AEChooser}
  @type kind: integer
  @ivar kwargs: Aribitrary data to be passed to the handler of this event
  @type kwargs: dictionary
  '''
  Constants.registerEventType('ChooserGesture', False)
  def __init__(self, chooser, kind, **kwargs):
    '''
    Calls the base class 
    
    @param gesture: Gestures detected on an L{AEInput} device
    @type gesture: L{AEInput.Gesture}
    '''
    Base.AccessEngineEvent.__init__(self, focused=True, **kwargs)
    self.kwargs = kwargs
    self.kind = kind
    self.chooser = chooser
    
  def __str__(self):
    '''
    Returns a human readable representation of this event including its name,
    its gesture codes, and its device.
    
    @return: Information about this event
    @rtype: string
    '''
    name = Base.AccessEngineEvent.__str__(self)
    if self.kind == AEChooser.OK:
      action = 'ok'
    elif self.kind == AEChooser.CANCEL:
      action = 'cancel'
    else:
      action = 'apply'
    return '%s:\n\tchooser: %s\n\taction: %s' % \
           (name, self.chooser.getName(), action)

  def execute(self, tier_manager, **kwargs):
    '''
    Contacts the L{TierManager} and asks it to manage this chooser event.
    
    @param tier_manager: TierManager that will handle the event
    @type tier_manager: L{TierManager}
    @param kwargs: Packed references to other managers not of interest here
    @type kwargs: dictionary
    @return: True to indicate the event executed properly
    @rtype: boolean
    '''
    tier_manager.manageChooser(self)
    return True
  
  def getChooser(self):
    '''
    Gets the L{AEChooser} that triggered this event.
    
    @return: Chooser that fired the event
    @rtype: L{AEChooser}
    '''
    return self.chooser
  
  def getDataForTask(self):
    '''
    Fetches data out of this L{ChooserChange} for use by a L{Task.ChooserTask}.
    
    @return: Dictionary of parameters to be passed to a L{Task.ChooserTask} as 
      follows:
        - chooser: The chooser that fired the event
        - kind: The kind of event
        - any addition data in the L{kwargs} instance variable
    @rtype: dictionary
    '''
    self.kwargs['chooser'] = chooser
    self.kwargs['kind'] = kind
    return self.kwargs