'''
Defines the abstract base class for all L{AEMonitor} subclasses.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging
from AEMonitor import AEMonitor
from AEEvent import LAYER_NAMES
from LSRConstants import SEMANTIC_NAMES
from DeviceManager import InputEvent, OutputEvent, DeviceEvent
from i18n import _

__uie__ = dict(kind='monitor')

# get a reference to the logging system
log = logging.getLogger('TestLogMonitor')

class TestLogMonitor(AEMonitor):
  '''  
  Logs information sent to output devices to using the LSR logging system. The
  format of the output is intended to be used in automated regression testing.
  '''
  def getName(self):
    '''
    Gets the localized name of this monitor.
    
    @return: Monitor name
    @rtype: string
    '''
    return _('Test logging monitor')
  
  def getEventNames(self):
    '''
    Gets the event categories to be logged.
    
    @return: Event categories
    @rtype: list of string
    '''
    return OutputEvent.getNames()
  
  def getEventDefaults(self):
    '''
    Gets the default event categories to log.
    
    @return: Event categories
    @rtype: list of string
    '''
    return OutputEvent.getDefaults()
  
  def getEventType(self):
    '''
    Gets the L{DeviceManager.DeviceEvent} base class to indicate the type 
    of events this monitor wants to buffer.
    
    @return: Base type of the event this monitor should buffer
    @rtype: L{DeviceManager.DeviceEvent} class
    '''
    return OutputEvent
      
  def show(self, event, text=None, sem=None, layer=None, *args, **kwargs):
    '''
    Buffers additional details about how the L{AEEvent} was handled by various
    L{Perk}s and L{Task}s. The event is only provided as a means of filtering 
    information.

    @param text: Text or filename sent to the device
    @type text: string
    @param sem: Semantic constant indicating the kind of output
    @type sem: integer
    @param layer: Layer constant indicating from where the event came
    @type layer: integer
    @raise IOError: When the monitor is no longer accepting data to buffer 
    '''
    if not self.isInitialized():
      raise IOError
    # convert to human readable names
    sem = SEMANTIC_NAMES.get(sem)
    layer = LAYER_NAMES[layer]
    if text:
      # log text
      log.info('%s ** %s %s', text, sem, layer)
    else:
      # log a command (stop, talk, index, etc.)
      log.info('%s ** %s %s', event, sem, layer)
