'''
Defines tools for assisting L{Perk} developers.

Keys registered

Alt-CapsLock-P: Announce all Perks loaded on the current Tier
Alt-CapsLock-R: Refresh all Perks loaded on the current Tier
Alt-CapsLock-M: Show/hide all monitors

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
# import useful modules for Perks
import Perk, Task
from POR import POR
from i18n import _

__uie__ = dict(kind='perk', all_tiers=True)

class DeveloperPerk(Perk.Perk):
  '''  
  Defines keyboard commands for Perk developers that report all Perks in the
  current Tier, refresh all Perks in the current Tier, show/hide all
  monitors registered in the current profile, and mute/unmute all output.
  '''
  def init(self):
    '''
    Registers keys for all L{Task}s in the module.
    '''
    kbd = self.getInputDevice('Keyboard')
    # set Alt+CapsLock as the modifier
    self.addInputModifiers(kbd, kbd.AEK_ALT_L, kbd.AEK_ALT_R, 
                           kbd.AEK_CAPS_LOCK)
  
    # register named tasks
    self.registerTask(FocusDebug(focus=True, tier=True, background=True))
    self.registerTask(ViewDebug())
    self.registerTask(SayPerks('say perks'))
    self.registerTask(ReloadPerks('reload perks'))
    self.registerTask(ShowHideMonitors('toggle monitors'))
    self.registerTask(Mute('toggle mute'))
    
    # map named tasks to commands
    self.registerCommand(kbd, 'toggle mute', False,
                         [kbd.AEK_CONTROL_R, kbd.AEK_CONTROL_L])    
    
    pairs = [[kbd.AEK_ALT_L, kbd.AEK_CAPS_LOCK],
             [kbd.AEK_ALT_R, kbd.AEK_CAPS_LOCK]]
    for pair in pairs:
      self.registerCommand(kbd, 'say perks', False, pair+[kbd.AEK_P])
      self.registerCommand(kbd, 'reload perks', False, pair+[kbd.AEK_R])
      self.registerCommand(kbd, 'toggle monitors', False, pair+[kbd.AEK_M])

class Mute(Task.InputTask):
  '''
  Task to mute output indefinitely or unmute it if it has already been muted.
  '''
  def execute(self, **kwargs):
    self.stopAll()
    mute = self.getMute() 
    if mute:
      self.setMute(False)
      self.sayInfo(text=_('unmuted'))
    else:
      self.sayInfo(text=_('muted'))
      self.setMute(True)

class SayPerks(Task.InputTask):
  '''
  Says the number of L{Perk}s loaded on the active L{Tier} followed by their
  names.
  '''
  def execute(self, **kwargs):
    '''
    Stops speech then makes the announcement.
    '''
    self.stopNow()
    names = self.getPerkNames()
    n = len(names)
    self.sayInfo(text=self.getAppName(), template=str(n)+' perks in %s.')
    self.sayInfo(text=', '.join(names))

class ReloadPerks(Task.InputTask):
  '''
  Reloads all L{Perk}s in the current L{Tier}. Useful during L{Perk} 
  development when changes have been made to a L{Perk} and those changes should
  be tested without restarting LSR.
  '''
  def execute(self, **kwargs):
    '''
    Reloads L{Perk}s then makes an announcement.
    '''
    self.stopNow()
    self.sayInfo(text=self.getAppName(), template='reloaded perks in %s')
    self.reloadPerks()

class ShowHideMonitors(Task.InputTask):
  '''
  Shows all monitors associated with the current profile if all are hidden.
  Hides all monitors associated with the current profile if any one is shown.
  '''
  def execute(self, **kwargs):
    '''
    Reloads L{Perk}s then makes an announcement.
    '''
    self.stopNow()
    self.inhibitMayStop()
    if not self.loadAllMonitors():
      self.unloadAllMonitors()
      self.sayInfo(text=_('hiding monitors'))
    else:
      self.sayInfo(text=_('showing monitors'))
      
class FocusDebug(Task.FocusTask):
  '''
  Prints the focus L{POR}.
  '''
  def executeGained(self, por, **kwargs):
    print 'focus:', por
    
  def executeLost(self, por, **kwargs):
    print 'unfocus:', por
    
class ViewDebug(Task.ViewTask):
  '''
  Prints the view L{POR}.
  '''
  def execute(self, por, gained, **kwargs):
    print 'view:', por, gained