'''
Defines an L{AEEvent} indicating the view has been updated by the
L{ViewManager}.

@author: Peter Parente
@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''

import Base, Constants
from LSRInterfaces import *

class ViewChange(Base.AccessEngineEvent):
  '''
  Event that fires when the L{ViewManager} is creating or updating its view.
  
  This class registers its name and whether it should be monitored by default
  in an L{AEMonitor} using the L{Constants.registerEventType} function when
  this module is first imported. An L{AEMonitor} can use this information to
  build its menus.

  @ivar gained: Type of view change: True if gained, False if lost, None if 
    mutated
  @type gained: boolean
  @ivar title: Name of the root element of the new view (e.g. title of the
    foreground window)
  @type title: string
  @cvar GAINED: View was gained by activating it
  @type GAINED: integer
  @cvar LOST: View was lost by deactivating it
  @type LOST: integer
  @cvar STARTUP: View was identified at LSR startup
  @type STARTUP: integer
  @cvar FIRST_GAINED: View was gained by activating it during LSR startup
  @type FIRST_GAINED: integer
  '''
  LOST = 0
  GAINED = 1 
  STARTUP = 2
  FIRST_GAINED = 3
  Constants.registerEventType('ViewChange', True)
  def __init__(self, por, gained, **kwargs):
    '''
    Stores the type of view change and intializes the title attribute to
    an empty string.

    @param por: Point-of-regard to the accessible at the root of the new view
    @type por: L{POR}    
    @param gained: One of the integer class variables of this class
    @type gained: integer
    '''
    focused = gained in (ViewChange.GAINED, ViewChange.FIRST_GAINED)
    Base.AccessEngineEvent.__init__(self, focused=focused, **kwargs)
    self.por = por
    self.gained = gained
    self.title = ''
    
  def __str__(self):
    '''
    Returns a human readable representation of this event including its name,
    its L{POR}, its type, and the title of the new view.
    
    @return: Information about this event
    @rtype: string
    '''
    name = Base.AccessEngineEvent.__str__(self)
    if self.gained == self.GAINED:
      action = 'gained'
    elif self.gained == self.LOST:
      action = 'lost'
    elif self.gained == self.FIRST_GAINED:
      action = 'first gained'
    elif self.gained == self.STARTUP:
      action = 'startup'
    return '%s:\n\tPOR: %s\n\ttitle: %s\n\taction: %s' % (name, self.por,
                                                          self.title, action)

  def execute(self, tier_manager, view_manager, **kwargs):
    '''
    Stores the name of the root element of the new view in the title
    attribute. Calls L{TierManager.TierManager.switchTier} to switch to the 
    appropriate L{Tier} for this view. Calls 
    L{TierManager.TierManager.manageEvent} to allow it to dispatch this
    L{AEEvent} to the active L{Tier} and its registered L{Perk}.

    @param tier_manager: TierManager that will handle the event
    @type tier_manager: L{TierManager}
    @param view_manager: ViewManager defining the current View
    @type view_manager: L{ViewManager}
    @param kwargs: Packed references to other managers not of interest here
    @type kwargs: dictionary
    @return: True to indicate the event executed properly
    @rtype: boolean
    '''
    # get the accessible name of the root element
    ai = IAccessibleInfo(self.por)
    try:
      self.title = ai.getAccName()
    except LookupError:
      # view is dead
      return True
    if self.gained in (self.GAINED, self.FIRST_GAINED):
      # get information about the accessible's application
      try:
        aid = ai.getAccAppID()
        aname = ai.getAccAppName()
      except LookupError:
        # view is dead
        return True
      # ask the tier_manager to switch tiers first
      tier_manager.switchTier(aname, aid, self.por)
    # then process the view change event
    tier_manager.manageEvent(self)
    return True
  
  def getDataForTask(self):
    '''
    Fetches data out of this L{ViewChange} for use by a L{Task.ViewTask}.
    
    @return: Dictionary of parameters to be passed to a L{Task.ViewTask} as
      follows:
        - por: Point of regard to the top of the new view
        - title: String containing the window title of the view which changed
        - gained: The type of view change, indicated by one of the integer
          class variables
    @rtype: dictionary
    '''
    return {'title':self.title, 'gained':self.gained, 'por':self.getPOR()}  