/* data-entry.c
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "data-entry-private.h"
#include "marshal.h"


static void data_entry_class_init (DataEntryClass * class);
static void data_entry_init (DataEntry * wid);
static void data_entry_dispose (GObject   * object);

static void data_entry_refresh_status_display (DataEntry *de);
static void m_modified (DataEntry * wid);

/* signals */
enum
{
	CONTENTS_MODIF_SIGNAL,
	LAST_SIGNAL
};

static gint data_entry_signals[LAST_SIGNAL] = { 0 };

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
data_entry_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (DataEntryClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) data_entry_class_init,
			NULL,
			NULL,
			sizeof (DataEntry),
			0,
			(GInstanceInitFunc) data_entry_init
		};		

		type = g_type_register_static (GTK_TYPE_VBOX, "DataEntry", &info, 0);
	}
	return type;
}


static void
data_entry_class_init (DataEntryClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	data_entry_signals[CONTENTS_MODIF_SIGNAL] =
		g_signal_new ("contents_modified",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (DataEntryClass, contents_modified),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE, 0);

	class->contents_modified = m_modified;
	object_class->dispose = data_entry_dispose;
}

static void 
m_modified (DataEntry * wid)
{
	data_entry_refresh_display (wid);
}

void       
data_entry_refresh_display (DataEntry *wid)
{
	GdaValue *now;

	g_return_if_fail (wid && IS_DATA_ENTRY (wid));
	
	/* determine the is_null and is_modified values */
	now = data_handler_get_value_from_widget (wid->priv->data_handler, GTK_WIDGET (wid));
	if (now) {
		wid->priv->value_is_null = gda_value_is_null (now) ? TRUE : FALSE;
		
		if (wid->priv->orig_value) {
			if (gda_value_is_null (wid->priv->orig_value)) {
				if (gda_value_is_null (now))
					wid->priv->value_is_modified = FALSE;
				else
					wid->priv->value_is_modified = TRUE;
			}
			else {
				if (gda_value_is_null (now))
					wid->priv->value_is_modified = TRUE;
				else 
					wid->priv->value_is_modified = 
						gda_value_compare (now, wid->priv->orig_value) ? TRUE: FALSE;
			}
		}
		else
			wid->priv->value_is_modified = TRUE;
		gda_value_free (now);
	}
	else
		wid->priv->value_is_null = TRUE;
	
	/* the default value flag comes down if we have modified the entry */
	if (wid->priv->value_is_modified)
		wid->priv->value_is_defaultval = FALSE;
	else {
		if (wid->priv->value_is_defaultval)
			wid->priv->value_is_modified = TRUE;
	}
	
	data_entry_refresh_status_display (wid);

	g_print ("null: %d, modif: %d, default: %d\n", wid->priv->value_is_null, 
		 wid->priv->value_is_modified, wid->priv->value_is_defaultval);
}

void
data_entry_update_with_value (DataEntry *de, 
			      const GdaValue *value, GdaValueType fallback_orig_type,
			      gboolean value_as_init)
{
	g_return_if_fail (de && IS_DATA_ENTRY (de));
	g_return_if_fail (de->priv->data_handler);
	
	data_handler_widget_update (de->priv->data_handler, GTK_WIDGET (de), value, value_as_init);
	if ((de->priv->orig_type == GDA_VALUE_TYPE_NULL) ||
	    (de->priv->orig_type == GDA_VALUE_TYPE_UNKNOWN))
		de->priv->orig_type = fallback_orig_type;
}

void
data_entry_set_query_field (DataEntry *de, QueryField *qf)
{
	g_return_if_fail (de && IS_DATA_ENTRY (de));

	de->priv->query_field = qf;
}

void
data_entry_set_is_default_set (DataEntry *de, gboolean is_default)
{
	g_return_if_fail (de && IS_DATA_ENTRY (de));

	de->priv->value_is_defaultval = is_default;
	if (is_default) 
		data_entry_set_is_default_allowed (de, TRUE);
}

void
data_entry_set_is_default_allowed (DataEntry *de, gboolean is_default_possible)
{
	g_return_if_fail (de && IS_DATA_ENTRY (de));

	de->priv->is_default_possible = is_default_possible;	
}

void
data_entry_set_is_null_allowed (DataEntry *de, gboolean null_allowed)
{
	g_return_if_fail (de && IS_DATA_ENTRY (de));

	de->priv->value_required = !null_allowed;
}

void
data_entry_set_show_actions_menu (DataEntry *de, gboolean show)
{
	g_return_if_fail (de && IS_DATA_ENTRY (de));

	de->priv->show_actions = show;
	if (show)
		gtk_widget_show (de->priv->button);
	else
		gtk_widget_hide (de->priv->button);
}

gboolean
data_entry_get_is_default_allowed (DataEntry *de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), FALSE);

	return de->priv->is_default_possible;
}

gboolean
data_entry_get_is_null_allowed (DataEntry *de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), FALSE);

	return !de->priv->value_required;
}

DataHandler *
data_entry_get_data_handler (DataEntry *de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), NULL);

	return de->priv->data_handler;
}

const GdaValue *
data_entry_get_orig_value (DataEntry *de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), NULL);

	return de->priv->orig_value;
}

GdaValueType
data_entry_get_orig_type (DataEntry *de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), GDA_VALUE_TYPE_UNKNOWN);

	return de->priv->orig_type;
}


static void
data_entry_init (DataEntry * wid)
{
	wid->priv = g_new0 (DataEntryPriv, 1);
	wid->priv->top_box = NULL;
	wid->priv->button = NULL;
	wid->priv->show_actions = TRUE;

	wid->extension = NULL;
	wid->free_extension = NULL;

	wid->priv->query_field = NULL;
	wid->priv->data_handler = NULL;
	wid->priv->is_default_possible = FALSE;
	wid->priv->value_required = FALSE;

	wid->priv->orig_type = GDA_VALUE_TYPE_UNKNOWN;
	wid->priv->orig_value = NULL;
	wid->priv->value_is_null = FALSE;
	wid->priv->value_is_defaultval = FALSE;
	wid->priv->value_is_modified = FALSE;
}

static gint event_cb (GtkWidget *widget, GdkEvent *event, DataEntry *de);
GtkWidget *
data_entry_new (DataHandler *dh)
{
	GObject   *obj;
	DataEntry *de;
	GtkWidget *button, *hbox, *vbox, *arrow;
	GValue *gval;

	g_return_val_if_fail (dh && IS_DATA_HANDLER (dh), NULL);

	obj = g_object_new (DATA_ENTRY_TYPE, NULL);
	gtk_container_set_border_width (GTK_CONTAINER (obj), 0);

	de = DATA_ENTRY (obj);
	de->priv->data_handler = dh;
	de->priv->orig_style = gtk_style_copy (gtk_widget_get_style (GTK_WIDGET (obj)));

	/* hbox */
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (de), hbox, TRUE, TRUE, 0);  
	gtk_widget_show (hbox);

	/* vbox */
	vbox = gtk_vbox_new (FALSE, 0); 
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 3); 
	de->priv->top_box = vbox;
	gtk_widget_show (vbox);

	/* button to change the entry's state and to display that state */
	arrow = gtk_arrow_new (GTK_ARROW_RIGHT, GTK_SHADOW_NONE);
	button = gtk_button_new ();
	gtk_container_add (GTK_CONTAINER (button), arrow);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);  
	de->priv->button = button;
	gtk_widget_show_all (button);
	gtk_widget_set_size_request (button, 15, 15);

	g_signal_connect (G_OBJECT (button), "event",
			  G_CALLBACK (event_cb), de);

	/* focus */
	gval = g_new0 (GValue, 1);
	g_value_init (gval, G_TYPE_BOOLEAN);
	g_value_set_boolean (gval, TRUE);
	g_object_set_property (G_OBJECT (button), "can-focus", gval);
	g_free (gval);

	return GTK_WIDGET (obj);
}


static GtkWidget *build_actions_menu (DataEntry *de);
static gint
event_cb (GtkWidget *widget, GdkEvent *event, DataEntry *de)
{
	gboolean done = FALSE;

	if (event->type == GDK_BUTTON_PRESS) {
		GdkEventButton *bevent = (GdkEventButton *) event; 
		if ((bevent->button == 1) || (bevent->button == 3)) {
			GtkWidget *menu;
			
			menu = build_actions_menu (de);
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
					bevent->button, bevent->time);
			done = TRUE;
		}
	}

	if (event->type == GDK_KEY_PRESS) {
		GtkWidget *menu;
		GdkEventKey *kevent = (GdkEventKey *) event;

		if (kevent->keyval == GDK_space) {
			menu = build_actions_menu (de);
			gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
					0, kevent->time);
			done = TRUE;
		}
		else {
			if (kevent->keyval == GDK_Tab)
				done = FALSE;
			else
				done = TRUE;
		}
	}
	
	return done;
}

static void mitem_reset_to_orig_activated_cb (GtkWidget *mitem, DataEntry *de);
static void mitem_to_null_activated_cb (GtkWidget *mitem, DataEntry *de);
static void mitem_to_defval_activated_cb (GtkWidget *mitem, DataEntry *de);
static GtkWidget *
build_actions_menu (DataEntry *de)
{
	GtkWidget *menu, *mitem;
	gchar *str;
	const gchar *name = NULL;
	gboolean nullval = FALSE;
	gboolean defval = de->priv->is_default_possible;
	gboolean reset = FALSE;

	menu = gtk_menu_new ();

	/* which menu items do make sensitive ? */
	if (!de->priv->value_required) {
		gboolean tmp = TRUE;

		if (de->priv->value_is_null && de->priv->value_is_defaultval)
			tmp = FALSE;

		if (de->priv->query_field && !de->priv->value_is_null) {
			name = query_field_get_name (de->priv->query_field);

			if (query_field_get_ftype (de->priv->query_field) == QUERY_FIELD_FIELD) {
				DbField *field;
				field = query_field_field_get_db_field (de->priv->query_field);
				if (field) 
					nullval = field->null_allowed;
			}
		}

		nullval = tmp;
		/* if ((de->priv->query_field && !de->priv->value_is_null) || */
/* 		    (de->priv->value_is_null && de->priv->value_is_defaultval)) { */
/* 			name = de->priv->query_field->name; */
					
/* 			if (de->priv->query_field->field_type == QUERY_FIELD_FIELD) { */
/* 				DbField *field; */
/* 				field = query_field_field_get_db_field (de->priv->query_field); */
/* 				if (field)  */
/* 					nullval = field->null_allowed; */
/* 			} */
/* 		} */
	}

	if (defval && de->priv->value_is_defaultval)
		defval = FALSE;

	if (de->priv->orig_value)
		reset = TRUE;
	if (!de->priv->value_is_modified)
		reset = FALSE;

	/* set to NULL item */
	if (name)
		str = g_strdup_printf (_("Set '%s' to NULL"), name);
	else
		str = g_strdup (_("Set to NULL"));
	
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), de->priv->value_is_null);
	gtk_widget_show (mitem);
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (mitem_to_null_activated_cb), de);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, nullval);

	/* default value item */
	if (name)
		str = g_strdup_printf (_("Set '%s' to default value"), name);
	else
		str = g_strdup (_("Set to default value"));
	
	mitem = gtk_check_menu_item_new_with_label (str);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mitem), de->priv->value_is_defaultval);
	gtk_widget_show (mitem);
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (mitem_to_defval_activated_cb), de);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	g_free (str);
	gtk_widget_set_sensitive (mitem, defval);
		
	/* reset to original value item */
	if (name)
		str = g_strdup_printf (_("Reset '%s' to its original value"), name);
	else
		str = g_strdup (_("Reset to original value"));
	mitem = gtk_menu_item_new_with_label (str);
	g_free (str);
	gtk_widget_show (mitem);
	g_signal_connect (G_OBJECT (mitem), "activate",
			  G_CALLBACK (mitem_reset_to_orig_activated_cb), de);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), mitem);
	gtk_widget_set_sensitive (mitem, reset);


	return menu;
}

static void 
mitem_reset_to_orig_activated_cb (GtkWidget *mitem, DataEntry *de)
{
	data_handler_widget_update (de->priv->data_handler, GTK_WIDGET (de), de->priv->orig_value, FALSE);
}

static void 
mitem_to_null_activated_cb (GtkWidget *mitem, DataEntry *de)
{
	GdaValue *value;
	
	de->priv->value_is_defaultval = FALSE;
	value = gda_value_new_null ();
	data_handler_widget_update (de->priv->data_handler, GTK_WIDGET (de), value, FALSE);
	gda_value_free (value);
}

static void
mitem_to_defval_activated_cb (GtkWidget *mitem, DataEntry *de)
{
	de->priv->value_is_defaultval = TRUE;
	data_handler_widget_update (de->priv->data_handler, GTK_WIDGET (de), NULL, FALSE);
	data_entry_refresh_status_display (de);
#ifdef debug_signal
	g_print (">> 'CONTENTS_MODIFIED' from emit_contents_modified de=%p\n", de);
#endif
	g_signal_emit (G_OBJECT (de), data_entry_signals [CONTENTS_MODIF_SIGNAL], 0);
#ifdef debug_signal
	g_print ("<< 'CONTENTS_MODIFIED' from emit_contents_modified\n");
#endif
}



void
data_entry_set_orig_value (DataEntry *de, const GdaValue *value)
{
	g_return_if_fail (de && IS_DATA_ENTRY (de));

	if (de->priv->orig_value) {
		gda_value_free (de->priv->orig_value);
		de->priv->orig_value = NULL;
		de->priv->orig_type = GDA_VALUE_TYPE_UNKNOWN;
	}
	
	if (value) {
		de->priv->orig_value = gda_value_copy (value);
		de->priv->value_is_modified = FALSE;
		if (gda_value_is_null (value))
			de->priv->value_is_null = TRUE;
		else
			de->priv->value_is_null = FALSE;
		de->priv->orig_type = gda_value_get_type (value);
	}
	else {
		/* failsafe defaults */
		de->priv->value_is_null = TRUE;
		de->priv->value_is_modified = FALSE;
		de->priv->orig_type = GDA_VALUE_TYPE_NULL;
	}
}

static void
data_entry_dispose (GObject   * object)
{
	DataEntry *de;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_DATA_ENTRY (object));

	de = DATA_ENTRY (object);

	if (de->priv) {
		if (de->free_extension) {
			(de->free_extension)(de);
		}

		if (de->priv->orig_value) {
			gda_value_free (de->priv->orig_value);
			de->priv->orig_value = NULL;
		}
		
		if (de->priv->orig_style) {
			g_free (de->priv->orig_style);
			de->priv->orig_style = NULL;
		}
		
		g_free (de->priv);
		de->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}


static void 
data_entry_refresh_status_display (DataEntry *de)
{
	static GdkColor **colors = NULL;
	GdkColor *normal = NULL, *prelight = NULL;
	GdkColor *orig_normal, *orig_prelight;

	g_return_if_fail (de && IS_DATA_ENTRY (de));

	orig_normal = & (de->priv->orig_style->bg[GTK_STATE_NORMAL]);
	orig_prelight = & (de->priv->orig_style->bg[GTK_STATE_PRELIGHT]);

	if (!colors) {
		GdkColor *color;

		colors = g_new0 (GdkColor *, 6);

		/* Green color */
		color = g_new0 (GdkColor, 1);
		gdk_color_parse ("#00cd66", color);
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
			g_free (color);
			color = NULL;
		}
		colors[0] = color;

		color = g_new0 (GdkColor, 1);
		gdk_color_parse ("#00ef77", color);
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
			g_free (color);
			color = NULL;
		}
		colors[1] = color;


		/* Blue color */
		color = g_new0 (GdkColor, 1);
		gdk_color_parse ("#6495ed", color);
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
			g_free (color);
			color = NULL;
		}
		colors[2] = color;

		color = g_new0 (GdkColor, 1);
		gdk_color_parse ("#75a6fe", color);
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
			g_free (color);
			color = NULL;
		}
		colors[3] = color;


		/* Red color */
		color = g_new0 (GdkColor, 1);
		gdk_color_parse ("#ff6a6a", color);
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
			g_free (color);
			color = NULL;
		}
		colors[4] = color;

		color = g_new0 (GdkColor, 1);
		gdk_color_parse ("#ff7b7b", color);
		if (!gdk_colormap_alloc_color (gtk_widget_get_default_colormap (), color, FALSE, TRUE)) {
			g_free (color);
			color = NULL;
		}
		colors[5] = color;		
	}

	if (de->priv->value_is_null) {
		normal = colors[0];
		prelight = colors[1];
	}

	if (de->priv->value_is_defaultval) {
		normal = colors[2];
		prelight = colors[3];
	}

	if (de->priv->value_required && data_entry_is_value_null (de) && !de->priv->value_is_defaultval) {
		normal = colors[4];
		prelight = colors[5];
	}

	if (!normal)
		normal = orig_normal;
	if (!prelight)
		prelight = orig_prelight;

	gtk_widget_modify_bg (de->priv->button, GTK_STATE_NORMAL, normal);
	gtk_widget_modify_bg (de->priv->button, GTK_STATE_ACTIVE, normal);
	gtk_widget_modify_bg (de->priv->button, GTK_STATE_PRELIGHT, prelight);
}

void
data_entry_pack_default (DataEntry * de, GtkWidget * wid)
{
	gtk_box_pack_start (GTK_BOX (de->priv->top_box), wid, TRUE, TRUE, 7);
}

GdaValue *
data_entry_get_gdavalue (DataEntry * de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), NULL);

	return data_handler_get_value_from_widget (de->priv->data_handler, GTK_WIDGET (de));
}

gboolean
data_entry_is_value_null (DataEntry * de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), FALSE);

	return de->priv->value_is_null;
}

gboolean
data_entry_is_value_default (DataEntry * de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), FALSE);

	return de->priv->value_is_defaultval;
}

gboolean
data_entry_is_value_modified (DataEntry * de)
{
	g_return_val_if_fail (de && IS_DATA_ENTRY (de), FALSE);

	return de->priv->value_is_modified;
}
