/*
 *
 * bg-config-gui.c :
 *
 * Copyright (C) 2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Richard Hestilow <tvgm@ximian.com>
 *
 */

#include <mtm/mtm-config-gui.h>
#include <mtm/mtm-ext-handler.h>
#include "bg-config-gui.h"
#include <sys/types.h>
#include <dirent.h>
#include <gtk/gtk.h>
#include <libgnome/libgnome.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <glade/glade-xml.h>
#include <string.h>
#include "editor.h"

/* Eeeeep -- this is required for scale_for_screen */
#include <gdk/gdkx.h>

#define GLADEFILE (SHAREDIR "bg-config-gui.glade")

typedef struct 
{
	MtmExt *ext;
	GtkWidget *pixmap;
	GtkWidget *box;
	GladeXML *xml;
	gboolean realized;
	GdkPixbuf *buf;
} ConfigPreview;

/* The following code nicked from background-properties-capplet */

#define MONITOR_CONTENTS_X 20
#define MONITOR_CONTENTS_Y 10
#define MONITOR_CONTENTS_WIDTH 157
#define MONITOR_CONTENTS_HEIGHT 111

static void
fill_gradient (guchar *d, gint w, gint h,
	       GdkColor *c1, GdkColor *c2, int vertical)
{
    gint i, j;
    gint dr, dg, db;
    gint gs1, w3;
    gint vc = (!vertical || (c1 == c2));
    guchar *b, *row;

#define R1 c1->red
#define G1 c1->green
#define B1 c1->blue
#define R2 c2->red
#define G2 c2->green
#define B2 c2->blue

    dr = R2 - R1;
    dg = G2 - G1;
    db = B2 - B1;

    gs1 = (vertical) ? h-1 : w-1;
    w3 = w*3 + 1;

    row = g_new (unsigned char, w3);

    if (vc) {
	b = row;
	for (j = 0; j < w; j++) {
	    *b++ = (R1 + (j * dr) / gs1) >> 8;
	    *b++ = (G1 + (j * dg) / gs1) >> 8;
	    *b++ = (B1 + (j * db) / gs1) >> 8;
	}
    }

    for (i = 0; i < h; i++) {
	if (!vc) {
	    guchar cr, cg, cb;
	    cr = (R1 + (i * dr) / gs1) >> 8;
	    cg = (G1 + (i * dg) / gs1) >> 8;
	    cb = (B1 + (i * db) / gs1) >> 8;
	    b = row;
	    for (j = 0; j < w; j++) {
		*b++ = cr;
		*b++ = cg;
		*b++ = cb;
	    }
	}
	memcpy (d, row, w3);
	d += w3;
    }

#undef R1
#undef G1
#undef B1
#undef R2
#undef G2
#undef B2

    g_free (row);
}
/* end bg-properties code */

static gint
expose_event_cb (GtkWidget *pixmap, GdkEventExpose *event, ConfigPreview *preview)
{
	guchar *pixels;
	int rowstride;
	GdkRectangle r1, r2, dest;
	
	g_return_val_if_fail (preview != NULL, TRUE);
	if (!preview->buf)
		return TRUE;

	r1.x = MONITOR_CONTENTS_X;
	r1.y = MONITOR_CONTENTS_Y;
	r1.width = gdk_pixbuf_get_width (preview->buf);
	r1.height = gdk_pixbuf_get_height (preview->buf);
	
	r2.x = event->area.x;
	r2.y = event->area.y;
	r2.width = event->area.width;
	r2.height = event->area.height;
	
	if (gdk_rectangle_intersect (&r1, &r2, &dest))
	{
		rowstride = gdk_pixbuf_get_rowstride (preview->buf);
		pixels = gdk_pixbuf_get_pixels (preview->buf);
		/* Because of the hacky after-expose way we do this,
		 * we have to redraw the whole preview */
		gdk_draw_rgb_image_dithalign (preview->pixmap->window,
					      preview->pixmap->style->black_gc,
					      MONITOR_CONTENTS_X,
					      MONITOR_CONTENTS_Y,
					      gdk_pixbuf_get_width (preview->buf),
					      gdk_pixbuf_get_height (preview->buf),
				 	      GDK_RGB_DITHER_NORMAL,
					      pixels, rowstride,
					      0, 0);
	}

	return TRUE;
}

static gchar**
generate_file_list (void)
{
	GArray *files;
	gchar **ret;
	/* FIXME */
	gchar *dirnames[] = { "/usr/share/sawfish/themes", "/.sawfish/themes", NULL};
	int i;

	dirnames[1] = g_strconcat (g_get_home_dir (), dirnames[1], NULL);
	
	files = g_array_new (TRUE, TRUE, sizeof (gchar*));
	
	for (i = 0; dirnames[i] != NULL; i++)
	{
		DIR *dir;
		struct dirent *de;
		dir = opendir (dirnames[i]);
		if (!dir)
			continue;

		while ((de = readdir (dir)))
		{
			gchar *filename;
			
			if (de->d_name[0] == '.')
				continue;
			
			filename = g_strconcat (dirnames[i], "/", de->d_name, NULL);
			g_array_append_val (files, filename);
		}
		
		closedir (dir);
	}

	g_free (dirnames[1]);
	ret = files->data;
	g_array_free (files, FALSE);

	return ret;
}

static void
destroy_cb (MtmConfigGui *gui, ConfigPreview *preview)
{
	g_return_if_fail (preview != NULL);
	gtk_object_destroy (GTK_OBJECT (preview->xml));
	if (preview->buf)
		gdk_pixbuf_unref (preview->buf);
	g_free (preview);
}

static void
compute_scale_factors (GdkPixbuf *pixbuf, int *xoff, int *yoff, int *w, int *h)
{
	double aspectx, aspecty;
	int ow, oh;
	
	g_return_if_fail (pixbuf != NULL);
	g_return_if_fail (xoff != NULL);
	g_return_if_fail (yoff != NULL);
	g_return_if_fail (w != NULL);
	g_return_if_fail (h != NULL);

	ow = *w;
	oh = *h;
	
	aspectx = (double) gdk_pixbuf_get_width (pixbuf) / ow;
	aspecty = (double) gdk_pixbuf_get_height (pixbuf) / oh;

	if (aspectx < aspecty)
	{
		*w = gdk_pixbuf_get_width (pixbuf) / aspecty;
		*xoff = (ow - *w) / 2;
		*yoff = 0;
	}
	else
	{
		*h = gdk_pixbuf_get_height (pixbuf) / aspectx;
		*xoff = 0;
		*yoff = (oh - *h) / 2;
	}
}

static GdkPixbuf*
scale_for_screen (GdkPixbuf *pixbuf)
{
	int sw, sh, w, h;
	
	g_return_val_if_fail (pixbuf != NULL, NULL);
	
	gdk_window_get_size (GDK_ROOT_PARENT (), &sw, &sh);
	w = gdk_pixbuf_get_width (pixbuf) * MONITOR_CONTENTS_WIDTH / sw;
	h = gdk_pixbuf_get_height (pixbuf) * MONITOR_CONTENTS_HEIGHT / sh;

	return gdk_pixbuf_scale_simple (pixbuf, w, h, GDK_INTERP_BILINEAR);
}

static void
update_preview (ConfigPreview *preview)
{
	MtmExtHandler *handler;
	GdkColor *color1, *color2;
	gboolean gradient;
	int gtype, ltype;
	GdkPixbuf *buf, *image, *scaled;
	
	g_return_if_fail (preview != NULL);
	g_return_if_fail (preview->ext != NULL);

	handler = mtm_ext_get_handler (preview->ext);
	gtk_object_set (GTK_OBJECT (handler), "ext_context", preview->ext, NULL);
	gtk_object_get (GTK_OBJECT (handler), "Color1", &color1,
					      "Color2", &color2,
					      "ColorGradient", &gradient,
					      "GradientType", &gtype,
					      "WallpaperLayout", &ltype,
					      NULL);
	
	if (!preview->buf)
		preview->buf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8,
			      MONITOR_CONTENTS_WIDTH, MONITOR_CONTENTS_HEIGHT);
	
	buf = preview->buf;
	
	if (!gradient)
		color2 = color1;
	
	fill_gradient (gdk_pixbuf_get_pixels (buf),
		       gdk_pixbuf_get_width (buf),
		       gdk_pixbuf_get_height (buf),
		       color1, color2, (gtype == VERTICAL));
	
	/* An image? */
	if (preview->ext->file && strcmp(g_basename (preview->ext->file), "none")
	    && (image = gdk_pixbuf_new_from_file (preview->ext->file)))
	{
		int xoff, yoff, w, h, nx, ny;
		int pxoff, pyoff;
		
		scaled = NULL;
		
		switch (ltype)
		{
			case WALLPAPER_SCALED:
				gdk_pixbuf_scale (image, buf,
						  0, 0,
						  MONITOR_CONTENTS_WIDTH,
						  MONITOR_CONTENTS_HEIGHT,
						  0, 0,
						  (double) MONITOR_CONTENTS_WIDTH / gdk_pixbuf_get_width (image),
						  (double) MONITOR_CONTENTS_HEIGHT / gdk_pixbuf_get_height (image),
						  GDK_INTERP_BILINEAR);
				break;
			
			case WALLPAPER_SCALED_KEEP:
				w = MONITOR_CONTENTS_WIDTH;
				h = MONITOR_CONTENTS_HEIGHT;
				compute_scale_factors (image, &xoff, &yoff, &w, &h);
				scaled = gdk_pixbuf_scale_simple (image, w, h, GDK_INTERP_BILINEAR);
				gdk_pixbuf_copy_area (scaled, 0, 0, w, h,
						      buf, xoff, yoff);
				break;
				
			case WALLPAPER_TILED:
				scaled = scale_for_screen (image);
				w = gdk_pixbuf_get_width (scaled);
				h = gdk_pixbuf_get_height (scaled);
				nx = MONITOR_CONTENTS_WIDTH / gdk_pixbuf_get_width (scaled);
				ny = MONITOR_CONTENTS_HEIGHT / gdk_pixbuf_get_height (scaled);
				for (xoff = 0; xoff < nx; xoff++)
				{
					for (yoff = 0; yoff < ny; yoff++)
					{
						gdk_pixbuf_copy_area (
							scaled, 0, 0, w, h,
							buf,
							xoff * w, yoff * h);
					}
				}
				
				if (w * nx < MONITOR_CONTENTS_WIDTH
				    || h * ny < MONITOR_CONTENTS_HEIGHT)
				{
					gdk_pixbuf_copy_area (
						scaled, 0, 0,
						MONITOR_CONTENTS_WIDTH - w * nx,
						MONITOR_CONTENTS_HEIGHT - h * ny,
						buf, nx * w, ny * h);
				}
				break;
				
			case WALLPAPER_CENTERED:
				scaled = scale_for_screen (image);
				xoff = (MONITOR_CONTENTS_WIDTH - gdk_pixbuf_get_width (scaled)) / 2;
				yoff = (MONITOR_CONTENTS_HEIGHT - gdk_pixbuf_get_height (scaled)) / 2;
				if (xoff < 0)
				{
					pxoff = -xoff;
					xoff = 0;
					w = MONITOR_CONTENTS_WIDTH;
				}
				else
				{
					pxoff = 0;
					w = gdk_pixbuf_get_width (scaled);
				}
				
				if (yoff < 0)
				{
					pyoff = -yoff;
					yoff = 0;
					h = MONITOR_CONTENTS_HEIGHT;
				}
				else
				{
					pyoff = 0;
					h = gdk_pixbuf_get_height (scaled);
				}

				gdk_pixbuf_copy_area (scaled, pxoff, pyoff,
						      w, h, buf, xoff, yoff);
				break;
		}

		if (scaled)
			gdk_pixbuf_unref (scaled);
	}

	gtk_widget_queue_draw (preview->box);
}

static void
update_widgets (ConfigPreview *preview)
{
	GtkWidget *w;
	MtmExtHandler *handler;
	GdkColor *color1, *color2;
	gboolean gradient;
	int gtype, ltype;

	g_return_if_fail (preview != NULL);
	g_return_if_fail (preview->ext != NULL);
	
	handler = mtm_ext_get_handler (preview->ext);
	gtk_object_set (GTK_OBJECT (handler), "ext_context", preview->ext, NULL);
	gtk_object_get (GTK_OBJECT (handler), "Color1", &color1,
					      "Color2", &color2,
					      "ColorGradient", &gradient,
					      "GradientType", &gtype,
					      "WallpaperLayout", &ltype,
					      NULL);

	w = glade_xml_get_widget (preview->xml, "colorpicker1");
	gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (w),
				    color1->red, color1->green, color1->blue,
				    0);
	w = glade_xml_get_widget (preview->xml, "colorpicker2");
	gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (w),
				    color2->red, color2->green, color2->blue,
				    0);
	w = glade_xml_get_widget (preview->xml, "pattern_menu");
	gtk_option_menu_set_history (GTK_OPTION_MENU (w),
				     gradient ? (gradient + gtype) : 0);
	w = glade_xml_get_widget (preview->xml, "align_menu");
	gtk_option_menu_set_history (GTK_OPTION_MENU (w), ltype);
}

static void
ext_changed_cb (MtmConfigGui *gui, MtmExt *ext, ConfigPreview *preview)
{
	g_return_if_fail (preview != NULL);
	update_widgets (preview);
	update_preview (preview);
}
	
static void
set_ext_cb (MtmConfigGui *gui, MtmExt *ext, ConfigPreview *preview)
{
	g_return_if_fail (preview != NULL);
	preview->ext = ext;
	update_widgets (preview);
	update_preview (preview);
}

static void
realize_cb (GtkWidget *pixmap, ConfigPreview *preview)
{
	g_return_if_fail (preview != NULL);
	preview->realized = TRUE;
}

static void
set_ext_arg (ConfigPreview *preview, gchar *arg, gboolean is_int,
	     gpointer pvalue, int ivalue)
{
	g_return_if_fail (preview != NULL);
	g_return_if_fail (arg != NULL);
	
	if (!preview->ext)
		return;

	gtk_object_set (GTK_OBJECT (preview->ext->handler), "ext_context",
			preview->ext, NULL);
	if (is_int)
		gtk_object_set (GTK_OBJECT (preview->ext->handler), arg, ivalue, NULL);
	else
		gtk_object_set (GTK_OBJECT (preview->ext->handler), arg, pvalue, NULL);

	update_preview (preview);
}

static void
generic_color_set_cb (GnomeColorPicker *cp, guint r, guint g, guint b, ConfigPreview *preview, gchar *arg)
{
	GdkColor color;

	g_return_if_fail (preview != NULL);
	g_return_if_fail (arg != NULL);
	
	color.red = r;
	color.green = g;
	color.blue = b;
	
	set_ext_arg (preview, arg, FALSE, &color, -1);
}

static void
color1_set_cb (GnomeColorPicker *cp, guint r, guint g, guint b, guint a,
	       ConfigPreview *preview)
{
	generic_color_set_cb (cp, r, g, b, preview, "Color1");
}

static void
color2_set_cb (GnomeColorPicker *cp, guint r, guint g, guint b, guint a,
	       ConfigPreview *preview)
{
	generic_color_set_cb (cp, r, g, b, preview, "Color2");
}

static void
pattern_set_cb (GtkMenuItem *item, ConfigPreview *preview)
{
	int index = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (item), "index"));
	gboolean gradient;
	GradType type = VERTICAL;
	
	if (index == 0)
		gradient = FALSE;
	else
	{
		gradient = TRUE;
		type = index - 1;
	}
	
	set_ext_arg (preview, "ColorGradient", TRUE, NULL, gradient);
	if (gradient)
		set_ext_arg (preview, "GradientType", TRUE, NULL, type);
}

static void
align_set_cb (GtkMenuItem *item, ConfigPreview *preview)
{
	WpType type = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (item), "index"));
	set_ext_arg (preview, "WallpaperLayout", TRUE, NULL, type);
}

static void
connect_menu (gchar *wname, GtkSignalFunc func, ConfigPreview *preview)
{
	GtkWidget *w, *menu;
	GtkMenuItem *item;
	GList *l;
	int index = 0;
	
	g_return_if_fail (wname != NULL);
	g_return_if_fail (preview != NULL);
	g_return_if_fail (preview->xml != NULL);
	
	w = glade_xml_get_widget (preview->xml, wname);
	g_return_if_fail (w != NULL);
	
	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (w));
	for (l = GTK_MENU_SHELL (menu)->children; l != NULL; l = l->next)
	{
		item = GTK_MENU_ITEM (l->data);
		gtk_object_set_data (GTK_OBJECT (item), "index", GINT_TO_POINTER (index++));
		gtk_signal_connect (GTK_OBJECT (item), "activate", func, preview);
	}
}

MtmConfigGui*
bg_config_gui_new (MtmGuiHandler *handler)
{
	GladeXML *xml;
	GtkWidget *w;
	gchar *imfile;
	GdkPixbuf *buf;
	GtkWidget *pixmap;
	GdkPixmap *pmap;
	GdkBitmap *bmap;
	MtmConfigGui *gui;
	gchar **files;
	ConfigPreview *preview;
	GtkWidget *box, *hbox, *vbox;

	xml = glade_xml_new (GLADEFILE, "hbox1");
	if (!xml)
		return NULL;
	imfile = gnome_pixmap_file ("monitor.png");
	buf = gdk_pixbuf_new_from_file (imfile);
	g_free (imfile);
	if (!buf)
	{
		gtk_object_destroy (GTK_OBJECT (xml));
		return NULL;
	}

	gui = mtm_config_gui_new ();
#if 0
	files = generate_file_list ();
	mtm_config_gui_set_file_list (gui, files);
	g_strfreev (files);
#endif

	gdk_pixbuf_render_pixmap_and_mask (buf, &pmap, &bmap, 255);
	pixmap = gtk_pixmap_new (pmap, bmap);
	gdk_pixbuf_unref (buf);

	box = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER (box), pixmap);
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), box, TRUE, FALSE, 0);
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, FALSE, 0);

	w = glade_xml_get_widget (xml, "preview_frame");
	gtk_container_add (GTK_CONTAINER (w), vbox);
	
	w = glade_xml_get_widget (xml, "hbox1");
	mtm_config_gui_set_config_area (gui, w); 

	preview = g_new0 (ConfigPreview, 1);
	preview->pixmap = pixmap;
	preview->box = box;
	preview->xml = xml;
	preview->ext = NULL;
	preview->realized = FALSE;

	gtk_signal_connect_after (GTK_OBJECT (gui), "destroy", destroy_cb, preview);
	
	gtk_signal_connect (GTK_OBJECT (gui), "set_ext", set_ext_cb, preview);
	gtk_signal_connect (GTK_OBJECT (gui), "ext_modified", ext_changed_cb, preview);
	gtk_signal_connect (GTK_OBJECT (pixmap), "realize", realize_cb, preview);
	gtk_signal_connect_after (GTK_OBJECT (pixmap), "expose_event",
				  expose_event_cb, preview);

	w = glade_xml_get_widget (xml, "colorpicker1");
	gtk_signal_connect (GTK_OBJECT (w), "color_set", color1_set_cb, preview);
	w = glade_xml_get_widget (xml, "colorpicker2");
	gtk_signal_connect (GTK_OBJECT (w), "color_set", color2_set_cb, preview);
	connect_menu ("pattern_menu", pattern_set_cb, preview);
	connect_menu ("align_menu", align_set_cb, preview);

	gtk_widget_show_all (w);

	return gui;
}
