/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */

#ifndef __MLVIEW_SETTINGS_MANAGEMENT_H__
#  define __MLVIEW_SETTINGS_MANAGEMENT_H__

#  include <libxml/tree.h>
#  include <libxml/xpath.h>
#  include <libxml/xpathInternals.h>
#  include "mlview-app-context.h"

typedef struct _MlViewSettingsEntryHandlers
 MlViewSettingsEntryHandlers;
typedef struct _MlViewSettingsEntry MlViewSettingsEntry;
typedef struct _MlViewSettingsEntryPrivate
 MlViewSettingsEntryPrivate;

/*===============================================================
 *These are the interfaces that each part of MlView must implement
 *to install a settings entry.
 *==============================================================*/

/*Callback that is called by the settings manager when the settings have been modified by the user*/
typedef void (*NotifySettingsChanged) (MlViewSettingsEntry *
                                       a_settings,
                                       void *a_user_data);

/*Function used to load the settings from an xml doc into a given MlViewSettingsEntry->settings structure.
 *This function is called by the settings manager before at MlView load time and each time, it is asked to reread
 *the MlView Settings.
 */
typedef void (*LoadSettingsFromXMLDocIntoSettingsEntry)
 (MlViewSettingsEntry * a_settings, xmlDoc * a_xml_doc);

/*Function used to get the settings from the settings UI and set populate the custom settings struct.
 *This function is called by the settings manager each time the user modifies the settings via the UI.
 */
typedef void (*GetSettingsFromSettingsWidget)
 (MlViewSettingsEntry * a_settings);

/*Function used to get the settings from the custom settings struct and to populate the UI.
 *This function is called by the settings manager each time the user wants to visualize a settings entry.
 */
typedef void (*SetSettingsToSettingsWidget) (MlViewSettingsEntry
                                             * a_settings);

/*Each part of the application that wants to define a set of settings
 *must add field to this enum.
 */

typedef enum {
        MLVIEW_EDITOR_SETTINGS = 1, /*The settings of the MlViewEditor widget */
        MLVIEW_EDITOR_VALIDATION_SETTINGS, /*a subset of the MLVIEW_EDITOR_SETTINGS */
        MLVIEW_TREE_EDITOR_SETTINGS,

/*=========This field must be the last one !!!!*/
        MLVIEW_SETTINGS_ROOT    /*The parent of all the settings. This type _MUST NOT_ be used !!! */
} MlViewSettingsType;


/*This structs contains the handlers called by the Settings manager to perform the
 *tasks of load/update/notify needed for the settings entries.
 */
struct _MlViewSettingsEntryHandlers {
        /*Settings modif callback and the custom data passed to it */
        NotifySettingsChanged notify_settings_changed;
        gpointer settings_changed_notifier_data;

        /*function called to populate the "settings" custom stucture from an xml document. Each app has to overloead this function */
         LoadSettingsFromXMLDocIntoSettingsEntry
                load_settings_from_xml_doc_into_settings_entry;

        /*Function called to populate the "settings" custom stucture from the settings UI. Each app has to overload this function */
         GetSettingsFromSettingsWidget
                get_settings_from_settings_widget;

        /*Function used to populate the settings widget from the "settings" custom structure. Each app has to overload this function */
         SetSettingsToSettingsWidget
                set_settings_to_settings_widget;
};

/*
 * Each entry of the setting dialog box is described by this
 * Structure that inherits GNode.
 */
struct _MlViewSettingsEntry {
        GNode *tree_node;
        MlViewSettingsType settings_type;
        MlViewSettingsEntryPrivate *private;
};

MlViewSettingsEntry *mlview_settings_entry_new (gchar * a_name,
                                                gpointer
                                                a_settings,
                                                GtkWidget *
                                                a_settings_widget,
                                                gpointer
                                                a_settings_widget_structure);

void mlview_settings_add_child_entry (MlViewSettingsEntry *
                                      a_parent_entry,
                                      MlViewSettingsEntry *
                                      a_child_entry);
void
 mlview_settings_add_previous_sibling_entry (MlViewSettingsEntry *
                                             a_parent_entry,
                                             MlViewSettingsEntry
                                             * a_sibling_entry,
                                             MlViewSettingsEntry
                                             * a_node);

MlViewSettingsEntry
        * mlview_settings_unlink_entry (MlViewSettingsEntry *
                                        a_entry);
void mlview_settings_entry_destroy (MlViewSettingsEntry *
                                    a_settings_entry);
void
 mlview_settings_entry_entry_set_handlers (MlViewSettingsEntry *
                                           a_settings_entry,
                                           MlViewSettingsEntryHandlers
                                           * a_handlers);


/*--------------MlViewSettingsEntry accessors--------------*/
gchar *mlview_settings_entry_get_name (MlViewSettingsEntry *
                                       a_settings_entry);
void mlview_settings_entry_set_name (MlViewSettingsEntry *
                                     a_settings_entry,
                                     gchar * a_name);
gint mlview_settings_entry_get_type (MlViewSettingsEntry *
                                     a_settings_entry);
void mlview_settings_entry_set_type (MlViewSettingsEntry *
                                     a_settings_entry,
                                     MlViewSettingsType a_type);
void *mlview_settings_entry_get_settings (MlViewSettingsEntry *
                                          a_settings_entry);
void mlview_settings_entry_set_settings (MlViewSettingsEntry *
                                         a_settings_entry, void
                                         *a_settings_struct);

GtkWidget
        * mlview_settings_entry_get_settings_widget
        (MlViewSettingsEntry * a_settings_entry);
void *mlview_settings_entry_get_settings_widgets_struct
        (MlViewSettingsEntry * a_settings_entry);
MlViewSettingsEntryHandlers
        * mlview_settings_entry_get_handlers (MlViewSettingsEntry
                                              *
                                              a_settings_entry);
void mlview_settings_entry_set_handlers (MlViewSettingsEntry *
                                         a_settings_entry,
                                         MlViewSettingsEntryHandlers
                                         * a_handlers);
/*-------------------------------
 * Settings manager definition.
 *-------------------------------*/
typedef struct _MlViewSettingsManager MlViewSettingsManager;
typedef struct _MlViewSettingsManagerPrivate
 MlViewSettingsManagerPrivate;

struct _MlViewSettingsManager {
        MlViewSettingsManagerPrivate *private;
};

MlViewSettingsManager
        * mlview_settings_manager_new (MlViewAppContext *
                                       a_app_context);

GtkDialog
        * mlview_settings_manager_get_settings_dialog
        (MlViewSettingsManager * a_manager);
void
 mlview_settings_manager_edit_settings_interactive
        (MlViewSettingsManager * a_manager);
gint
mlview_settings_manager_install_settings (MlViewSettingsManager *
                                          a_settings_manager,
                                          MlViewSettingsEntry *
                                          a_entry);
MlViewSettingsEntry
        * mlview_settings_manager_uninstall_settings
        (MlViewSettingsManager * a_settings_manager,
         MlViewSettingsEntry * a_entry);
void
 mlview_settings_manager_set_settings_dialog_proportions
        (MlViewSettingsManager * a_manager, guint a_percentage);
gint
mlview_settings_manager_create_personal_settings_file (xmlDoc **
                                                       a_xml_doc);
gboolean
mlview_settings_manager_personal_settings_file_exists (void);
xmlDoc *mlview_settings_manager_load_settings_from_disk
        (MlViewSettingsManager * a_manager);
void
 mlview_settings_manager_post_settings_install_init
        (MlViewSettingsManager * a_manager,
         xmlDoc * a_settings_doc);
void mlview_settings_manager_destroy (MlViewSettingsManager *
                                      a_manager);


/*Settings management Utils funtions*/
gint
mlview_settings_management_get_settings_xml_nodes (xmlXPathObject
                                                   **
                                                   a_xpath_object,
                                                   gchar *
                                                   a_xpath_expression,
                                                   xmlDoc *
                                                   a_xml_doc);


#endif /*__MLVIEW_PROPERTIES_MANAGEMENT_H__*/
