/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute it and/or 
 *modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView.
 *see the file COPYING. 
 *If not, write to the 
 *Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <stdio.h>
#include "config.h"
#include "mlview-editor.h"
#include "mlview-app.h"
#include "mlview-app-context.h"
#include "mlview-preferences.h"

/**
 *@file 
 *the mlview application class definition.
 *It instanciates mlview's instance of  GnomeApp, 
 *fills it with an instance of MlViewEditor, builds the application menus
 *and toolbars and creates an instance of MlViewAppContext.
 */


/*
 *The current instance of the mlview application context.
 *Must be set by mlview_app. For the moment, 
 *there must be one application
 *context per process. That should be changed in the 
 *future to enforce
 *reentrency.
 */
static MlViewAppContext *gv_app_ctxt = NULL;


/*common function declarations*/

/*callbacks ...*/
static void
display_about_dialog (void);

static void
open_cb (GtkWidget * a_widget, MlViewAppContext * a_context);

static void
about_cb (GtkWidget * a_widget, MlViewAppContext * a_context);

static void
paste_node_as_child_cb (GtkWidget * a_widget,
                        MlViewAppContext * a_context);

static void
paste_node_as_prev_sibling_cb (GtkWidget * a_widget,
                               MlViewAppContext * a_context);

static void
paste_node_as_next_sibling_cb (GtkWidget * a_widget,
                               MlViewAppContext * a_context);

static void copy_node_cb (GtkWidget * a_widget,
                          MlViewAppContext * a_context);

static void cut_node_cb (GtkWidget * a_widget,
                         MlViewAppContext * a_context);

static void new_cb (GtkWidget * a_menu_item, MlViewAppContext * a_context);

static void new_view_on_document_cb (GtkWidget * a_menu_item,
                                     MlViewAppContext * a_context);

static void close_cb (GtkWidget * a_menu_item,
                      MlViewAppContext * a_context);

static gint exit_cb (GtkWidget * a_widget, MlViewAppContext * a_context);

static void add_child_node_cb (GtkWidget * a_menu_item,
                               MlViewAppContext * a_context);

static void insert_prev_sibling_node_cb (GtkWidget * a_menu_item,
                                         MlViewAppContext * a_context);

static void insert_next_sibling_node_cb (GtkWidget * a_menu_item,
                                         MlViewAppContext * a_context);

static void save_cb (GtkWidget * a_menu_item, 
                     MlViewAppContext * a_context);

static void save_as_cb (GtkWidget * a_menu_item,
                        MlViewAppContext * a_context);

static void go_url_home_cb (void) ;

/*other functions*/
static void install_menu_and_toolbar (GnomeApp * a_appwin);

/*
 *Common functions functions that perform various generic tasks such as:
 *displaying an about dialog box for example, fetching and opening a file, ... 
 *
 */


/*
 *Displays the about dialog.
 *@author Dodji Seketeli <dodji@seketeli.org> 
 *
 */
static void
display_about_dialog ()
{
        static GtkWidget *about_dialog = NULL;
        guchar *path = NULL ;
        GdkPixbuf *pixbuf = NULL;

        const gchar *authors[] =
                {"Dodji Seketeli<dodji@mlview.org>",
                 "Gael Chamoulaud<strider@mlview.org>",
                 "Nicolas Centa<happypeng@free.fr>"
                 "",
                NULL
        };

        const gchar *documenters[] =
                {"Dodji Seketeli<dodji@mlview.org>",
                 "Gael Chamoulaud <strider@mlview.org>",
                 NULL
        };

        const gchar *translator_credits = _("translator_credits");

        if (about_dialog) {
                gtk_window_present (GTK_WINDOW (about_dialog));
                return;
        }
        path = gnome_program_locate_file (NULL, 
                                          GNOME_FILE_DOMAIN_APP_DATADIR,
                                          "pixmaps/mlview-app-icon.xpm",
                                          TRUE, NULL) ;
        if (path) {
                pixbuf = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path) ;
                path = NULL ;
        }
        about_dialog = gnome_about_new 
                (PACKAGE, VERSION,
                 "Copyright \xc2\xa9 2001-2003 Dodji Seketeli\n",
                 _("A simple xml editor for GNOME"),
                 (const char **) authors,
                 (const char **) documenters,
                 strcmp
                 (translator_credits,
                  "translator_credits") !=
                 0 ? translator_credits :
                 "No translators, English by\n"
                 "Dodji Seketeli  <dodji@mlview.org>\n"
                 "Gael Chamoulaud <strider@mlview.org>",
                 pixbuf);

        if (pixbuf != NULL) {
                g_object_unref (pixbuf) ;
                pixbuf = NULL ;
        }
        g_object_add_weak_pointer (G_OBJECT (about_dialog),
                                   (void **) &about_dialog);
        gtk_widget_show (about_dialog);
}

/*=========================================================
 *General signal handlers for the differents menuitems
 *========================================================*/
static void
open_cb (GtkWidget * a_widget, MlViewAppContext * a_context)
{
        g_return_if_fail (a_widget != NULL);
        g_return_if_fail (a_context != NULL);

        mlview_editor_open_local_xml_document_interactive
                (mlview_app_context_get_element
                 (a_context, "MlViewEditor"));
}

static void
about_cb (GtkWidget * a_widget, MlViewAppContext * a_context)
{
        display_about_dialog ();
}

static void
paste_node_as_child_cb (GtkWidget * a_widget,
                        MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL ;
        MlViewAction action = {0} ;
        
        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        action.name= (guchar *) "paste-node-as-child" ;
        mlview_editor_execute_action (editor, &action) ;
        /*mlview_editor_paste_node_as_child (editor);*/
}

static void
paste_node_as_prev_sibling_cb (GtkWidget * a_widget,
                               MlViewAppContext * a_context)
{
        MlViewEditor *editor;
        MlViewAction action = {0} ;

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");

        action.name = (guchar*) "paste-node-as-prev-sibling" ;
        /*mlview_editor_paste_node_as_prev_sibling (editor);*/
        mlview_editor_execute_action (editor, &action) ;
}

static void
paste_node_as_next_sibling_cb (GtkWidget * a_widget,
                               MlViewAppContext * a_context)
{
        MlViewEditor *editor;
        MlViewAction action = {0} ;

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        action.name = (guchar*) "paste-node-as-next-sibling" ;
        /*mlview_editor_paste_node_as_next_sibling (editor);*/        
        mlview_editor_execute_action (editor, &action) ;
}

static void
copy_node_cb (GtkWidget * a_widget, MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;
        MlViewAction action = {0} ;

        g_return_if_fail (a_context);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        g_return_if_fail (editor != NULL);
        action.name = (guchar*) "copy-node" ;
        /*mlview_editor_copy_node (editor);*/
        mlview_editor_execute_action (editor, &action) ;
}


static void
cut_node_cb (GtkWidget * a_widget, MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;
        MlViewAction action = {0} ;

        g_return_if_fail (a_context);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        g_return_if_fail (editor != NULL);
        action.name = (guchar*) "cut-node" ;
        /*mlview_editor_cut_node (editor);*/
        mlview_editor_execute_action (editor, &action) ;
}

static void
new_cb (GtkWidget * a_menu_item, MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_create_new_xml_document (editor);
}


static void
new_view_on_document_cb (GtkWidget * a_menu_item,
                         MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_create_new_view_on_current_document_interactive
                (editor);
}

static void
close_cb (GtkWidget * a_menu_item, MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_close_xml_document_interactive (editor);
}

static void
close_application (GtkWidget *a_widget, MlViewAppContext *a_context)
{
        GnomeApp *app = NULL;
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_widget
                          && GTK_IS_WIDGET (a_widget)
                          && a_context
                          && MLVIEW_IS_APP_CONTEXT (a_context));

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");

        if (editor)
                mlview_editor_close_all_xml_documents_interactive
                        (editor);
        app = mlview_app_context_get_element (a_context,
                                              "GnomeApp");
        g_object_unref (G_OBJECT (a_context));
        gtk_main_quit ();
}

static gboolean
delete_event_cb (GtkWidget *a_widget, 
                 GdkEvent *a_event,
                 MlViewAppContext *a_context)
{
        g_return_val_if_fail (GTK_IS_WIDGET (a_widget)
                              && MLVIEW_IS_APP_CONTEXT (a_context),
                              FALSE) ;

        close_application (a_widget, a_context) ;
        return FALSE ;
}

static gint
exit_cb (GtkWidget * a_widget, MlViewAppContext * a_context)
{
        
        close_application (a_widget, a_context) ;
        return FALSE;
}

static void
add_child_node_cb (GtkWidget * a_menu_item,
                   MlViewAppContext * a_context)
{
        MlViewEditor *editor;
        MlViewAction action = {0} ;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item);
        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        action.name = (guchar*) "add-child-node-interactive" ;
        /*mlview_editor_add_child_node_interactive (editor);*/
        mlview_editor_execute_action (editor, &action) ;
}

static void
insert_prev_sibling_node_cb (GtkWidget * a_menu_item,
                             MlViewAppContext * a_context)
{
        MlViewEditor *editor;
        MlViewAction action = {0} ;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");

        action.name = (guchar*) "insert-prev-sibling-node-interactive" ;
        /*mlview_editor_insert_prev_sibling_node_interactive
          (editor); */
        mlview_editor_execute_action (editor, &action) ;
}


static void
insert_next_sibling_node_cb (GtkWidget * a_menu_item,
                             MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;
        MlViewAction action = {0} ;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);
        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        action.name = (guchar*) "insert-next-sibling-node-interactive" ;
        /*mlview_editor_insert_next_sibling_node_interactive
          (editor);*/
        mlview_editor_execute_action (editor, &action) ;
}

static void
save_cb (GtkWidget * a_menu_item, MlViewAppContext * a_context)
{
        MlViewEditor *editor;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_save_xml_document (editor);
}

static void
save_as_cb (GtkWidget * a_menu_item,
            MlViewAppContext * a_context)
{
        MlViewEditor *editor;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_save_xml_document_as_interactive (editor);
}


static void
search_cb (GtkWidget * a_menu_item, MlViewAppContext * a_context)
{
        MlViewEditor *editor;
        MlViewAction action = {0} ;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);
        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        action.name = (guchar *) "find-node-that-contains-str-interactive" ;
        /*mlview_editor_find_xml_node_that_contains_str_interactive
          (editor);*/
        mlview_editor_execute_action (editor, &action) ;
}

static void
associate_a_dtd_cb (GtkWidget * a_menu_item,
                    MlViewAppContext * a_context)
{
        MlViewEditor *editor;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");

        mlview_editor_associate_dtd_interactive (editor);
}

static void
validate_document_cb (GtkWidget * a_menu_item,
                      MlViewAppContext * a_context)
{
        MlViewEditor *editor;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_validate (editor);
}

static void
expand_tree_to_depth_cb (GtkWidget * a_menu_item,
                         MlViewAppContext * a_context)
{
        MlViewEditor *editor;
        MlViewAction action = {0} ;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        action.name = (guchar*) "expand-tree-to-depth-interactive" ;
        /*mlview_editor_expand_tree_to_depth_interactive (editor);*/
        mlview_editor_execute_action (editor, &action) ;
}

static void
edit_settings_cb (GtkWidget * a_menu_item,
                  MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");

        mlview_preferences_dialog_show (a_context);
}

static void
change_view_name_cb (GtkWidget * a_menu_item,
                     MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_set_current_view_name_interactive (editor);
}

static void
go_url_home_cb (void)
{
	gnome_url_show ("http://www.mlview.org", NULL);
}

/*===============================================================
 *The applications menus definitions.
 *FIXME: Manage to comply with the gnome application menu rules.
 *===============================================================*/

/*the menuitems of the /file menu*/
static GnomeUIInfo g_menu_files_tree[] = {
        GNOMEUIINFO_MENU_OPEN_ITEM (open_cb, NULL),

        GNOMEUIINFO_MENU_CLOSE_ITEM (close_cb, NULL),

        GNOMEUIINFO_MENU_NEW_ITEM (N_("New xml document"),
                                   N_
                                   ("Creates new xml document"),
                                   new_cb, NULL),

        GNOMEUIINFO_MENU_NEW_ITEM (N_
                                   ("New view on xml document"),
                                   N_
                                   ("Creates a new view on current xml document"),
                                   new_view_on_document_cb,
                                   NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_MENU_SAVE_ITEM (save_cb, NULL),

        GNOMEUIINFO_MENU_SAVE_AS_ITEM (save_as_cb, NULL),

        GNOMEUIINFO_ITEM (N_("Change view name"),
                          N_("Rename the current view"),
                          change_view_name_cb, NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_MENU_EXIT_ITEM (exit_cb, NULL),

        GNOMEUIINFO_END
};

/*The menuitems of the /edit menu */
static GnomeUIInfo g_menu_edit_tree[] = {
        GNOMEUIINFO_ITEM_STOCK (N_("Add child node"),
                                N_
                                ("Adds a child node to the currently selected xml node"),
                                add_child_node_cb,
                                GNOME_STOCK_PIXMAP_ADD),

        GNOMEUIINFO_ITEM (N_("Insert previous node"),
                          N_
                          ("Inserts a sibling node just before the current selected node"),
                          insert_prev_sibling_node_cb, NULL),

        GNOMEUIINFO_ITEM (N_("Insert next node"),
                          N_
                          ("Inserts a sibling node just after the current selected node"),
                          insert_next_sibling_node_cb, NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_STOCK (N_("Copy node"),
                                N_
                                ("Copies the current selected node and it children"),
                                copy_node_cb,
                                GNOME_STOCK_PIXMAP_COPY),

        GNOMEUIINFO_ITEM_STOCK (N_("Cut node"),
                                N_
                                ("Cuts the current selected node and it children"),
                                cut_node_cb,
                                GNOME_STOCK_PIXMAP_CUT),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_STOCK (N_("Paste node as child"),
                                N_
                                ("Pastes the last copied node as a child of the currently selected node"),
                                paste_node_as_child_cb,
                                GNOME_STOCK_PIXMAP_PASTE),

        GNOMEUIINFO_ITEM_STOCK (N_("Paste node as prev"),
                                N_
                                ("Pastes the last copied node as the previous sibling of the currently selected node"),
                                paste_node_as_prev_sibling_cb,
                                GNOME_STOCK_PIXMAP_PASTE),

        GNOMEUIINFO_ITEM_STOCK (N_("Paste node as next"),
                                N_
                                ("Pastes the last copied node as the next sibling of the currently selected node"),
                                paste_node_as_next_sibling_cb,
                                GNOME_STOCK_PIXMAP_PASTE),

        GNOMEUIINFO_SEPARATOR,
        GNOMEUIINFO_ITEM (N_("Find an xml node"),
                          N_
                          ("Finds an xml node that matches some criteria"),
                          search_cb, NULL),
        /*
        GNOMEUIINFO_SEPARATOR,
        GNOMEUIINFO_ITEM_STOCK (N_("Settings"),
                                N_
                                ("Edits Gnome MlView Settings"),
                                edit_settings_cb,
                                GNOME_STOCK_PIXMAP_PREFERENCES),
        */

        GNOMEUIINFO_END
};

/*The */
static GnomeUIInfo g_menu_actions_tree[] = {
        GNOMEUIINFO_ITEM (N_("Associate a Dtd"),
                          N_
                          ("Associates a Dtd to the current xml document"),
                          associate_a_dtd_cb, NULL),

        GNOMEUIINFO_ITEM (N_("Validate document"),
                          N_
                          ("Validates the document against it current Dtd"),
                          validate_document_cb, NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM (N_("Expand current node"),
                          N_
                          ("Expands the current selected node to an absolute depth"),
                          expand_tree_to_depth_cb, NULL),

        GNOMEUIINFO_END
};

/*The menuitems of the /help menu*/
static GnomeUIInfo g_menu_help_tree[] = {
	GNOMEUIINFO_ITEM_STOCK (N_("MlView _Home..."),
			  	NULL,
                          	go_url_home_cb, 
				GNOME_STOCK_PIXMAP_HOME),

	GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_MENU_ABOUT_ITEM (about_cb, NULL),
        GNOMEUIINFO_END
};

/*The menuitems of the /search menu
static GnomeUIInfo g_menu_search_tree[] = {
        GNOMEUIINFO_ITEM (N_("Find an xml node"),
                          N_
                          ("Finds an xml node that matches some criteria"),
                          search_cb, NULL),
        GNOMEUIINFO_END
};
*/


/*The application menubar*/
static GnomeUIInfo g_app_menu_bar[] = {
        GNOMEUIINFO_MENU_FILE_TREE (g_menu_files_tree),
        GNOMEUIINFO_MENU_EDIT_TREE (g_menu_edit_tree),
        /*GNOMEUIINFO_SUBTREE (N_("_Search"), g_menu_search_tree),*/
        GNOMEUIINFO_SUBTREE (N_("Actions"), g_menu_actions_tree),
        GNOMEUIINFO_MENU_HELP_TREE (g_menu_help_tree),
        GNOMEUIINFO_END
};

/*the application toolbar*/
static GnomeUIInfo g_toolbar[] = {
        GNOMEUIINFO_ITEM_STOCK (N_("Open"),
                                N_("open an existing file"),
                                open_cb,
                                GNOME_STOCK_PIXMAP_OPEN),

        GNOMEUIINFO_ITEM_STOCK (N_("New"),
                                N_("creates a new xml document"),
                                new_cb, GNOME_STOCK_PIXMAP_NEW),

        GNOMEUIINFO_ITEM_STOCK (N_("Close"),
                                N_
                                ("closes the currently open file"),
                                close_cb,
                                GNOME_STOCK_PIXMAP_CLOSE),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_STOCK (N_("Save"),
                                N_("save this file"),
                                save_cb,
                                GNOME_STOCK_PIXMAP_SAVE),

        GNOMEUIINFO_ITEM_STOCK (N_("Save as"),
                                N_
                                ("save this file with a new name"),
                                save_as_cb,
                                GNOME_STOCK_PIXMAP_SAVE_AS),
        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_STOCK (N_("Exit"),
                                N_("Exits MlView application"),
                                exit_cb,
                                GNOME_STOCK_PIXMAP_QUIT),
        GNOMEUIINFO_END
};

/*
 *FIXME: define a general popup menu that contains 
 *some edit menu stuffs, some search stuffs and also som mdi stuffs.
 */

/*be carefull. The fields of this array are modified in install_menu_and_toolbar() so if you add a field, update the function !!!*/
static GnomeUIInfo g_edition_toolbar[] = {
        GNOMEUIINFO_ITEM_DATA (N_("Add child"),
                               N_
                               ("Add a child node to the current selected node"),
                               add_child_node_cb,
                               GNOME_STOCK_PIXMAP_ADD, NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_DATA (N_("Insert prev"),
                               N_
                               ("Inserts a sibling node just before the current selected node"),
                               insert_prev_sibling_node_cb, NULL,
                               NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_DATA (N_("Insert next"),
                               N_
                               ("Inserts a sibling node just after the current selected node"),
                               insert_next_sibling_node_cb, NULL,
                               NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_DATA (N_("Copy"),
                               N_
                               ("Copies the current selected node and it children"),
                               copy_node_cb, NULL, NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_DATA (N_("Cut"),
                               N_
                               ("Cuts the current selected node and it children"),
                               cut_node_cb, NULL, NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_DATA (N_("Paste child"),
                               N_
                               ("Pastes the last copied node as a child of the currently selected node"),
                               paste_node_as_child_cb, NULL,
                               NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_DATA (N_("Paste prev"),
                               N_
                               ("Pastes the last copied node as the previous sibling node of the currently selected node"),
                               paste_node_as_prev_sibling_cb,
                               NULL, NULL),

        GNOMEUIINFO_SEPARATOR,

        GNOMEUIINFO_ITEM_DATA (N_("Paste next"),
                               N_
                               ("Pastes the last copied node as the next sibling node of the currently selected node"),
                               paste_node_as_next_sibling_cb,
                               NULL, NULL),
        GNOMEUIINFO_END
};


/*
 *install a main menu and a main toolbar to the
 *main application root window.
 *@param a_appwin the main application root win. 
 *
 */
static void
install_menu_and_toolbar (GnomeApp * a_appwin)
{
        GtkToolbar *tb = NULL;

        if (!a_appwin)
                return;
        gnome_app_create_menus_with_data (a_appwin,
                                          g_app_menu_bar,
                                          gv_app_ctxt);

        /* tb = GTK_TOOLBAR (gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,
           GTK_TOOLBAR_BOTH)) ; */

        tb = GTK_TOOLBAR (gtk_toolbar_new ());

        /*gtk_toolbar_set_space_size (tb,0) ; */

        gnome_app_create_toolbar_with_data (a_appwin, g_toolbar,
                                            gv_app_ctxt);

        g_edition_toolbar[0].user_data = gv_app_ctxt;
        g_edition_toolbar[2].user_data = gv_app_ctxt;
        g_edition_toolbar[4].user_data = gv_app_ctxt;
        g_edition_toolbar[6].user_data = gv_app_ctxt;
        g_edition_toolbar[8].user_data = gv_app_ctxt;
        g_edition_toolbar[10].user_data = gv_app_ctxt;
        g_edition_toolbar[12].user_data = gv_app_ctxt;
        g_edition_toolbar[14].user_data = gv_app_ctxt;

        gnome_app_fill_toolbar (tb, g_edition_toolbar, NULL);

        gnome_app_add_toolbar (a_appwin, tb, "edition",
                               BONOBO_DOCK_ITEM_BEH_EXCLUSIVE,
                               BONOBO_DOCK_TOP, 2, 0, 0);

        gtk_toolbar_set_style (tb, GTK_TOOLBAR_TEXT);
}


/**
 *Sets the icon of the current instance of GnomeApp to
 *the MlView's icon.
 *@param a_app instance of GnomeApp to consider.
 */
void
mlview_app_set_icon (GnomeApp * a_app)
{
        MlViewAppContext *app_context = NULL;

        g_return_if_fail (a_app != NULL);
        g_return_if_fail (GNOME_IS_APP (a_app));
        g_return_if_fail (GTK_WIDGET (a_app)->window != NULL);

        app_context = gtk_object_get_data (GTK_OBJECT (a_app),
                                           "mlview-app-context");

        g_return_if_fail (app_context != NULL);
        g_return_if_fail (MLVIEW_IS_APP_CONTEXT (app_context));

        mlview_app_context_set_window_icon (app_context,
                                            GTK_WINDOW (a_app));
}



/**
 *Builds the mlview application.
 *@param a_appname the name of the mlview application.
 *@return the instance of GnomeApp that holds mlview.
 */
GnomeApp *
mlview_app_new (const guchar * a_appname)
{
        GnomeApp *mlview_application = NULL;
        GtkWidget *editor = NULL,
                *appbar = NULL;

        mlview_application =
                GNOME_APP (gnome_app_new (a_appname, a_appname));

        /*creates the application context */
        gv_app_ctxt =
                MLVIEW_APP_CONTEXT
                (mlview_app_context_get_instance ());
        gtk_object_set_data (GTK_OBJECT (mlview_application),
                             "mlview-app-context", gv_app_ctxt);

        /*add the gnome app to the app context */
        mlview_app_context_set_element (gv_app_ctxt, "GnomeApp",
                                        mlview_application);

        /*create the main vbox of the root window. */
        editor = mlview_editor_new ("mlview:empty", gv_app_ctxt);
        gnome_app_set_contents (GNOME_APP (mlview_application),
                                editor);

        /*add the mlview editor widget to the app context */
        mlview_app_context_set_element (gv_app_ctxt,
                                        "MlViewEditor", editor);
        mlview_editor_set_app_context (MLVIEW_EDITOR (editor),
                                       gv_app_ctxt);

        /*
         *create the app status bar, add it to the gnome app 
         *and store the pointer into the app context
         */
        appbar = gnome_appbar_new (TRUE, TRUE,
                                   GNOME_PREFERENCES_NEVER);
        gnome_app_set_statusbar (mlview_application, appbar);
        mlview_app_context_set_element (gv_app_ctxt,
                                        "GnomeAppBar", appbar);

        /*Install the menu and toolbar */
        install_menu_and_toolbar (mlview_application);

        gtk_widget_show (editor);

        /*FIXME: the message printing is just for test purpose */
        mlview_app_context_sbar_set_default_message (gv_app_ctxt,
                                                     "MlView is ready");

        /*
         *Make sure we exit the glib main loop when the user clicks the
         *window close button.
         */
        g_signal_connect (G_OBJECT (mlview_application),
                          "delete_event",
                          G_CALLBACK (delete_event_cb), gv_app_ctxt);

        return mlview_application;
}


/**
 *Getter of the instance of MlViewEditor associated
 *to the current instance of GnomeApp.
 *
 *@param a_app the instance of GnomeApp to consider.
 *@return the instance of MlViewEditor associated to a_app or
 *NULL if nothing is associated.
 */
MlViewEditor *
mlview_app_get_editor (GnomeApp * a_app)
{
        GnomeApp *app_candidate = NULL;
        MlViewEditor *result = NULL;

        g_return_val_if_fail (a_app != NULL, NULL);
        g_return_val_if_fail (GNOME_IS_APP (a_app), NULL);

        app_candidate =
                mlview_app_context_get_element (gv_app_ctxt,
                                                "GnomeApp");

        if (app_candidate) {
                result = mlview_app_context_get_element
                        (gv_app_ctxt, "MlViewEditor");
        }

        return result;
}
