///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "docview.h"

#include <cassert>
#include <string>
#include <cstdlib>
#include <vector>
#include <algorithm>

#include <gtkmm/style.h>

#include "document/loader.h"
#include "document/textframe.h"
#include "document/imageframe.h"
#include "document/rasterframe.h"
#include "document/document.h"
#include "document/page.h"

#include "icons/wait.xpm"
#include "icons/missing.xpm"
#include "icons/broken.xpm"

#include "util/warning.h"
#include "util/stringutil.h"
#include "util/filesys.h"

#include "widget/usererror.h"

#include "groupmeta.h"
#include "pptcore.h"
#include "pagesel.h"
#include "config.h"
#include "lengthunits.h"

IVector DocumentView::get_origin() const {
  return IVector(margin_size, margin_size);
}

float DocumentView::pt2scr(float pt) const {
   return int(resolution * zoom_factor 
	      * length_units.from_base(pt, "in") + 0.5);
}

float DocumentView::scr2pt(float scr) const {
   return length_units.to_base(scr / (resolution * zoom_factor), "in");
}  

Gdk::Point DocumentView::pt2scr(const Vector& pt) const {
  return Gdk::Point(int(get_origin().x + pt2scr(pt.x) + 0.5),
		    int(get_origin().y + page_height() - pt2scr(pt.y) + 0.5));
}

Vector DocumentView::scr2pt(const Gdk::Point& scr) const {
  return Vector(scr2pt(scr.get_x() - get_origin().x),
		scr2pt(page_height() - (scr.get_y() - get_origin().y)));
}

int DocumentView::page_height() const {
  return int(document ? pt2scr(document->get_height()) + 0.5 : 0);
}

int DocumentView::page_width() const {
  return int(document ? pt2scr(document->get_width()) + 0.5 : 0);
}


void DocumentView::insert_page_before() {
  if(document)
    template_page_dialog->show_it(true, 
				  document->get_num_of_pages() 
				  ? current_page_num
				  : document->get_first_page_num());
}

void DocumentView::insert_page_after() {
  if(document)
    template_page_dialog->show_it(true, 
				  document->get_num_of_pages()
				  ? current_page_num + 1
				  : document->get_first_page_num());
}

void DocumentView::delete_page() {
  try {
    if(document)
      document->delete_page(current_page_num);
  } 
  catch(const Error::InvalidPageNum &e) {
    warning << "DocumentView::delete_page: " << e.what() << std::endl;
  }
}

Page *DocumentView::get_page() {
  if(document) {
    Page *page = document->get_page(current_page_num);
    // if page has changed, release memory from old page
    if(old_page && page != old_page)
      pageview.clear();
    old_page = page;
    return page;
  } else
    return 0;
}

void DocumentView::act_on_document_change(DocRef document_) {
  if(!document || document != document_)
    return;
  int first = document->get_first_page_num();
  int num = document->get_num_of_pages();
  int last = first + num - 1;
  if(current_page_num < first) {
    current_page_num = first;
    current_page_num_changed_signal();
  } else if(current_page_num > last) {
    current_page_num = last;
    current_page_num_changed_signal();
  }
  draw();
  document_changed_signal();
}

void DocumentView::set_current_page_num(int current) {
  if(!document)
    return;
  int first = document->get_first_page_num();
  int num = document->get_num_of_pages();
  int last = first + num - 1;
  if(current < first)
    current = first;
  else if(current > last)
    current = last;
  if(current != current_page_num) {
      current_page_num = current;
      current_page_num_changed_signal();
      draw(); 
    }
}

void DocumentView::set_document(DocMeta d) {
  if(document == d)
    return;

  pageview.clear();

  document = d;
  if(document) {
    current_page_num = document->get_first_page_num();
    Document::changed_signal.connect
      (slot(*this, &DocumentView::act_on_document_change));
    Document::size_changed_signal.connect
      (slot(*this, &DocumentView::act_on_size_change));
  }

  document_changed_signal(); //?
  document_set_signal();

  adjust_size();
  draw();
}

DocumentView::DocumentView(DocMeta d)
  : snap_mode(snap::NONE), document(d),
    old_page(0),
    wait_pixmap(0), missing_pixmap(0), broken_pixmap(0)
{
  set_events(Gdk::EXPOSURE_MASK
	     | Gdk::LEAVE_NOTIFY_MASK
	     | Gdk::BUTTON_PRESS_MASK
	     | Gdk::BUTTON_RELEASE_MASK
	     | Gdk::POINTER_MOTION_MASK
	     | Gdk::POINTER_MOTION_HINT_MASK);
  Glib::RefPtr<Gdk::Colormap> colormap = get_default_colormap();
  white = Gdk::Color("white");
  black = Gdk::Color("black");
  gray = Gdk::Color("gray");
  red = Gdk::Color("red");
  colormap->alloc_color(white);
  colormap->alloc_color(black);
  colormap->alloc_color(gray);
  colormap->alloc_color(red);
  zoom_factor = 0.5;
  
  Glib::RefPtr<Gdk::Screen> screen =
    Gdk::Display::get_default()->get_default_screen();
  float xresolution = screen->get_width() / (screen->get_width_mm() / 25.4);
  float yresolution = screen->get_height() / (screen->get_height_mm() / 25.4);
  resolution = (xresolution + yresolution) / 2;  // average :-)
  verbose << "Screen resolution: " << resolution << " dpi" << std::endl;

  set_document(d);
  reshaping = moving = false;
  margin_size = 30;
  Page::ready_to_draw_page_signal.connect
    (slot(*this, &DocumentView::maybe_page_ready_to_draw));

  // drag'n drop stuff:
  std::list<Gtk::TargetEntry> targets;
  //  targets.push_back(Gtk::TargetEntry(""));
  targets.push_back(Gtk::TargetEntry("text/uri-list"));
  targets.push_back(Gtk::TargetEntry("_NETSCAPE_URL"));
  targets.push_back(Gtk::TargetEntry("text/unicode"));
  targets.push_back(Gtk::TargetEntry("text/plain"));
  targets.push_back(Gtk::TargetEntry("UTF8_STRING"));
  targets.push_back(Gtk::TargetEntry("STRING"));
  drag_dest_set(targets);
}

void DocumentView::maybe_page_ready_to_draw(Pagent *pagent) {
  if(pagent == get_page())
    draw(moving || reshaping);
}

DocumentView::~DocumentView() {
}

void DocumentView::adjust_size() {
  if(document)
    set_size_request(2 * margin_size 
		     + int(pt2scr(document->get_width()) + 0.5), 
		     2 * margin_size
		     + int(pt2scr(document->get_height()) + 0.5));
}

void DocumentView::act_on_size_change(DocRef document_) {
  if(document && document == document_) {
    hide();
    adjust_size();
    show();
  }
}

void DocumentView::set_zoom_factor(float factor) {
  if(factor > 0 && document) {
      zoom_factor = factor;
      hide();
      adjust_size();
      zoom_change_signal(zoom_factor);
      show();
    }
}

bool DocumentView::in_move_area(int x, int y) {
  if(Page* page = get_page()) {
    const Document::Selection selected = get_document()->selected();
    // "paper coordinates"
    const Vector pos = scr2pt(Gdk::Point(x, y));
    const float dist = scr2pt(2);

    for(Document::Selection::const_iterator i=selected.begin(); 
 	i != selected.end(); 
 	++i) 
      if(&Page::containing(**i) == page 
	 && (*i)->get_box()->isInsideOrClose(pos, dist))
 	return true;
  }
  return false;
}

Pagent* DocumentView::in_select_area(int x, int y) {
  if(Page* page = get_page()) {
    const Vector pos = scr2pt(Gdk::Point(x, y));
    const float dist = scr2pt(2);
    
    for(Group::ChildVec::const_iterator i = page->pbegin(); 
	i != page->pend(); 
	i++) {
      if((*i)->get_box()->isInsideOrClose(pos, dist))
	return *i;
    }
  }
  return 0;
}

namespace {
  typedef std::vector<Vector> ResizeHandles;
  ResizeHandles get_resize_handles(const Pagent &p) {
    ResizeHandles handles;
    Polygon poly = p.get_box()->get_polygon();
    for(Polygon::const_iterator i = poly.begin(); i != poly.end(); i++) {
      handles.push_back(*i);
      Polygon::const_iterator j = i; j++;
      if(j == poly.end())
	j = poly.begin();
      handles.push_back(Vector((i->x + j->x) / 2, (i->y + j->y) / 2));
    }
    return handles;
  }
}

namespace {
  // finds out if there is a resize handle at (x, y) and puts its number in j
  bool hover_reshape(const DocumentView& view, int x, int y, int& j) {
    if(!view.get_document())
      return false;

    const Document::Selection selected = view.get_document()->selected();
    if(selected.size() != 1) 
      return false;
    const Pagent& obj = *selected.front();
    
    // "paper coordinates"
    const int dist = int(config.ReshapeBoxSize.values.front()) / 2;

    ResizeHandles handles = get_resize_handles(obj);
    j = 0;
    for(ResizeHandles::const_iterator i = handles.begin();
	i != handles.end(); i++) {
      const Gdk::Point pos = view.pt2scr(*i);
      if(chessboard_dist(Vector(pos.get_x(), pos.get_y()), Vector(x, y)) 
	 <= dist) {
	return true;
      }
      j++;
    }

    return false;
  }
}

bool DocumentView::on_button_press_event(GdkEventButton *event) {
  // Cursor-browsing hack:
  //  static GdkCursorType foo=GdkCursorType(0);
  //  cerr << foo << endl;
  //  Gdk::Cursor moving_cursor(foo);
  //  foo=GdkCursorType(foo+2);
  //  win.set_cursor(moving_cursor);

  Page *page = get_page();
  if(page && event->button == 1) {
    const Document::Selection selected = get_document()->selected();

    int j;
    bool on_handle = hover_reshape(*this, int(event->x), int(event->y), j);
    if(on_handle && selected.size() == 1
       && !selected.front()->get_lock()) { // maybe start moving/resizing?
      // reshape

      begin_reshape(int(event->x), int(event->y), j);
    } else if(Pagent* select = in_select_area(int(event->x), int(event->y))) {
      // select / move

      // is it already selected?
      if(find(selected.begin(), selected.end(), select) != selected.end()) {
	if(event->state & Gdk::CONTROL_MASK) { // deselect
	  get_document()->deselect(select);
	} else 
	  begin_move(int(event->x), int(event->y));
      } else { // select and begin moving
	get_document()->select(select, !(event->state & Gdk::CONTROL_MASK));
	if(!select->get_lock())
	  begin_move(int(event->x), int(event->y));
      }
    } else if((event->state & Gdk::CONTROL_MASK) == 0) { // deselect all
      get_document()->select_all(false);
    }
    
    draw();
  }

  return true;
}

void DocumentView::new_text_frame(TextStream *stream) {
  Page *page = get_page();
  if(!page || !document)
    return;
  
  int w = 200, h = 300;
  // default to the first (alphabetically) stream
  page->addObject(new TextFrame(page, stream, w, h));

  draw();
}

void DocumentView::new_image_frame(std::string filename, float res) {
  Page *page = get_page();
  if(!page)
    return;
  get_document()->select_all(false);
  
  /// \todo more helpful error messages
  if(ImageFrame::is_postscript(filename))
    page->addObject(new ImageFrame(page, filename, 72, 72));
  else {
    try {
      page->addObject(new RasterFrame(page, filename, res));
    } catch(const std::exception& e){
      throw UserError("Could not open \"" + filename
		      + "\"", e);
    }
  }
  draw();
}

void DocumentView::select_all_frames() {
  if(Page *page = get_page()) {
      page->select_all(true); 
      draw();
    }
}

void DocumentView::unselect_all_frames() {
  if(Page *page = get_page()) {
      page->select_all(false); 
      draw();
    }
}

// *** copy/paste stuff ***/

namespace {
  std::string clip; // a global clip should suffice
  const std::string pptout_target = "PPTOUT_PAGENTS_2"; // with version number
}

void DocumentView::copy() {
  DocRef doc = get_document();
  if(!doc)
    return;

  /// \todo make sure the selection is sorted
  const Document::Selection selected = document->selected();
  if(selected.empty())
    return;
  
  Glib::RefPtr<Gtk::Clipboard> clipboard = Gtk::Clipboard::get();
  std::list<Gtk::TargetEntry> target_list;
  target_list.push_back(Gtk::TargetEntry(pptout_target)); 
  clipboard->set(target_list, 
		 SigC::slot(*this, &DocumentView::on_clipboard_get), 
		 SigC::slot(*this, &DocumentView::on_clipboard_clear));

  // store an XML representation of the selected pagents
  xmlpp::Document rep;
  xmlpp::Element *root = rep.create_root_node("copy_data");

  // save relevant streams
  typedef std::set<const TextStream*> Streams;
  Streams streams;
  for(Document::Selection::const_iterator i = selected.begin();
      i != selected.end(); i++) {
    const TextFrame *frame = dynamic_cast<const TextFrame *>(*i);
    if(!frame)
      continue;
    const TextStream *stream = frame->get_stream();
    if(stream && streams.find(stream) == streams.end()) {
      streams.insert(stream);
      stream->save(*root, FileContext());
    }
  }

  // save pagents
  for(Document::Selection::const_iterator i = selected.begin();
      i != selected.end(); i++) {
    (*i)->save(*root, FileContext());
  }

  clip = rep.write_to_string();
  //std::cerr << clip << std::endl;
}

void DocumentView::on_clipboard_get(Gtk::SelectionData& selection_data, 
				     guint info)
{ 
  const Glib::ustring& target = selection_data.get_target(); 
  
  if(target == pptout_target)
    selection_data.set(pptout_target, clip);
}

void DocumentView::on_clipboard_clear() {
  // not really necessary
  clip.clear(); 
}

void DocumentView::cut() {
  copy(); 
  delete_selected();
}

void DocumentView::paste() {
  Page *page = get_page();
  if(!page)
    return;

  Glib::RefPtr<Gtk::Clipboard> clipboard = Gtk::Clipboard::get();
  clipboard->request_contents
    (pptout_target, SigC::slot(*this, &DocumentView::on_clipboard_received));
}

void 
DocumentView::on_clipboard_received(const Gtk::SelectionData& selection_data)
{
  Page *page = get_page();
  if(!page)
    return;

  Glib::ustring clipboard_data = selection_data.get_data_as_string();
  try {
    xmlpp::DomParser parser;  
    parser.parse_memory(clipboard_data);
    xmlpp::Element *root = parser.get_document()->get_root_node();
    if(!root)
      throw std::runtime_error("Root node is NULL.");
    xmlpp::Element::NodeList children = root->get_children();

    get_document()->select_all(false);
    for(xmlpp::Element::NodeList::const_iterator i = children.begin();
	i != children.end();
	i++)
      {
	if(xmlpp::Element *elem = dynamic_cast<xmlpp::Element*>(*i)) {
	  if(elem->get_name() == "text_stream") {
	    std::auto_ptr<TextStream> stream(new TextStream(*elem, 
							    FileContext()));
	    // Don't try to add a stream with a name that already exists
	    /// \todo rename the stream automatically?
	    if(!document->get_text_stream(stream->get_name()))
	      document->add_text_stream(stream.release());
	  } else {
	    Pagent *pagent = load(*elem, page, FileContext());
	    page->add(pagent);
	    get_document()->select(pagent, false); // don't deselect
	  }
	}
      }
    draw();
  } catch(const std::exception& e) {
    throw std::runtime_error("Error encountered when pasting:\n" 
			     + std::string(e.what()));
  }
}  

void DocumentView::delete_selected() {
  if(DocRef doc = get_document()) {
    doc->delete_selected();
    draw();
  }
}  

void DocumentView::group_selected() {
  if(Page *page = get_page()) {
    page->group_selected();
    draw();
  }
}

void DocumentView::ungroup_selected() {
  if(Page *page = get_page()) {
    page->ungroup_selected();
    draw();
  }
}

void DocumentView::rearrange_selected(RearrangeTarget target) {
  if(Page *page = get_page()) {
    page->rearrange_selected(target);
    draw();
  }
}  

void DocumentView::refresh_streams() {
  if(Page *page = get_page()) {
    Document::StreamVec streams;
    for(Group::ChildVec::const_iterator i = page->pbegin();
	i != page->pend();
	i++)
      {	
	TextFrame *tmp = dynamic_cast<TextFrame*>(*i);
	if(tmp) {
	  TextStream *stream = tmp->get_stream();
	  if(stream 
	     && find(streams.begin(), streams.end(), stream) == streams.end())
	    streams.push_back(stream);
	}
      }
    for(Document::StreamVec::iterator i = streams.begin();
	i != streams.end();
	i++)
      {
	(*i)->run_typesetter();
	debug << (*i)->get_name() << ", " 
	      << (*i)->get_association() << std::endl;
      }
  }
}

void DocumentView::begin_reshape(int x, int y, int box) {
  DocRef document = get_document();
  if(!get_page() || !document)
    return;

  const Document::Selection selected = document->selected();
  if(selected.size() != 1) 
    return;
  const Pagent& obj = *selected.front();

  ResizeHandles handles = get_resize_handles(obj);
  if(box >= int(handles.size()))
    return;
  Gdk::Point handle = pt2scr(handles[box]);
  // pointer is probably not in center of handle
  offset = IVector(x - handle.get_x(), 
		   y - handle.get_y());
  old_width = obj.get_width(); old_height = obj.get_height();
  old_matrix = obj.get_matrix();
  reshape_box = box; reshaping = true;
}

void DocumentView::end_reshape(bool revert) {
  reshaping = false;
  if(DocRef document = get_document()) {
    if(revert) {
      const Document::Selection selected = document->selected();
      if(selected.size() != 1) {
	Pagent& obj = *selected.front();
	obj.set_matrix(old_matrix);
	if(obj.is_resizable()) {
	  obj.set_size(old_width, old_height);
	}
      }
    }
    draw(false);
    refresh_streams();
  }
}


namespace {
  // get bounding box of selected pagents
  void box_of_selected(Vector &ll, Vector &ur, DocRef document) {
    bool first = true;
    const Document::Selection &selected = document->selected();
    Document::Selection::const_iterator i;
    for(i = selected.begin(); i != selected.end(); i++) {
      Boundary b = (*i)->get_box();
      Vector b_ur = b->getCorner(Corner::UR);
      Vector b_ll = b->getCorner(Corner::LL);
      if(first) {
	ll = b_ll; ur = b_ur;
      } else {
	ll.x = std::min(ll.x, b_ll.x);  ll.y = std::min(ll.y, b_ll.y);
	ur.x = std::max(ur.x, b_ur.x);  ur.y = std::max(ur.y, b_ur.y);
      }
      first &= false;
    }
  } 
}

void DocumentView::begin_move(int x, int y) {
  if(get_page()) {
    Vector ur, ll;
    box_of_selected(ll, ur, get_document());
    Gdk::Point ll_scr = pt2scr(ll);
    offset = IVector(x, y) - IVector(ll_scr.get_x(), ll_scr.get_y());
    moving = true;  
  }
}

void DocumentView::end_move(bool revert) {
  if(get_page()) {
    moving = false;
    refresh_streams();
    draw(false);
  }
}

bool DocumentView::on_key_press_event(GdkEventKey* key) {
  if(key->keyval == 65307) { // escape
    if(moving)
      end_move(true);
    if(reshaping)
      end_reshape(true);
  }
  return false;
}

bool DocumentView::on_button_release_event(GdkEventButton *event)
{
  if(moving)
    end_move(false);
  if(reshaping)
    end_reshape(false);
  return true;
}

Gdk::Cursor DocumentView::get_cursor(int x, int y) {
  DocRef document = get_document();
  if(!document)
    return Gdk::LEFT_PTR;
    
  if(moving)
    return Gdk::FLEUR;

  const Document::Selection selection = document->selected();
  Pagent *obj = in_select_area(x, y);
  
  if(!reshaping && obj
     && find(selection.begin(), selection.end(), obj) == selection.end())
    return Gdk::TOP_LEFT_ARROW;
  
  int j = reshape_box;
  if(reshaping || hover_reshape(*this, x, y, j))
    switch(j) {
    case 0: return Gdk::BOTTOM_LEFT_CORNER;
    case 1: return Gdk::LEFT_SIDE;
    case 2: return Gdk::TOP_LEFT_CORNER;	
    case 3: return Gdk::TOP_SIDE;
    case 4: return Gdk::TOP_RIGHT_CORNER;
    case 5: return Gdk::RIGHT_SIDE;
    case 6: return Gdk::BOTTOM_RIGHT_CORNER;
    case 7: return Gdk::BOTTOM_SIDE;
    default: return Gdk::QUESTION_ARROW;
    }

  if(in_move_area(x, y))
    return Gdk::FLEUR;

  if(in_select_area(x, y))
    return Gdk::TOP_LEFT_ARROW;
  
  return Gdk::LEFT_PTR;
}

void DocumentView::update_cursor(int x, int y) {
  win->set_cursor(get_cursor(x, y));
}

namespace {
  float grid_size_x = 24, grid_size_y = 24;
  Vector grid_offset(0, 0);
  float snap_to_grid(float x, float offset, float step) {
    x -= offset;
    int n = int(x / step);
    float low = n * step, high = (n + 1) * step;
    float dlow = x - low, dhigh = high - x;
    x = dlow < dhigh ? low : high;
    x += offset;
    return x;
  }
}

void DocumentView::move_reshape_box(int x, int y) {
  const Document::Selection selected = document->selected();
  if(selected.size() != 1) 
    return;
  Pagent& obj = *selected.front();

  const Matrix &m = obj.get_matrix();
  // cursor wasn't necessarily in the center of the handle
  x -= offset.x; y -= offset.y;
  Vector p = scr2pt(Gdk::Point(x, y));

  // snap to grid
  if(snap_mode == snap::GRID) {
    p = Vector(snap_to_grid(p.x, grid_offset.x, grid_size_x),
	       snap_to_grid(p.y, grid_offset.y, grid_size_y));
  }


  // get pos in matrix coordinates
  Vector pos = m.inv().transform(p);

  float w = obj.get_width(), h = obj.get_height();

  Vector ll(0, 0), ur(w, h); // lower left / upper right
  switch(reshape_box) {
  case 0: // bottom left
    ll = pos;
    break;
  case 1: // left
    ll.x = pos.x;
    break;
  case 2: // top left
    ll.x = pos.x;
    ur.y = pos.y;
    break;
  case 3: // top
    ur.y = pos.y;
    break;
  case 4: // top right
    ur = pos;
    break;
  case 5: // right
    ur.x = pos.x;
    break;
  case 6: // bottom right;
    ur.x = pos.x;
    ll.y = pos.y;
    break;
  case 7: // bottom
    ll.y = pos.y;
    break;
  default:
    break;
  }

  bool negx = ll.x > ur.x, negy = ll.y > ur.y;
  // change reshape_box if going past zero
  if(negx) {
    switch(reshape_box) {
    case 0: reshape_box = 6; break;
    case 1: reshape_box = 5; break;
    case 2: reshape_box = 4; break;
    case 4: reshape_box = 2; break;
    case 5: reshape_box = 1; break;
    case 6: reshape_box = 0; break;
    default: break;
    }
  }
  if(negy) {
    switch(reshape_box) {
    case 0: reshape_box = 2; break;
    case 2: reshape_box = 0; break;
    case 3: reshape_box = 7; break;
    case 4: reshape_box = 6; break;
    case 6: reshape_box = 4; break;
    case 7: reshape_box = 3; break;
    default: break;
    }
  }

  // avoid negative scaling
  Vector _ll(std::min(ll.x, ur.x), std::min(ll.y, ur.y));
  // Vector _ur(std::max(ll.x, ur.x), std::max(ll.y, ur.y));

  Vector size = ur - ll;

  // can't set size to zero
  float minsize = 1;
  if(fabs(size.x) < minsize || fabs(size.y) < minsize) {
    return;
  }

  if(obj.is_resizable()) {
    obj.set_size(fabs(size.x), fabs(size.y));
  } else {
    obj.set_scaling(m.sc_x() * size.x / obj.get_width(), 
		    m.sc_y() * size.y / obj.get_height());
  }
  obj.set_matrix(Matrix::translation(_ll) * obj.get_matrix());
}

bool DocumentView::on_motion_notify_event(GdkEventMotion *event) {
  if(!get_page())
    return true;

  // find out where the pointer is
  int x, y;
  Gdk::ModifierType state;
  Glib::RefPtr<Gdk::Window> window = win;
  if(event->is_hint)
    window->get_pointer(x, y, state);
  else {
    x = int(event->x);
    y = int(event->y);
    state = Gdk::ModifierType(event->state);
  }

  update_cursor(x, y);

  Document::Selection selected = get_document()->selected();
  Document::Selection::const_iterator i;

  if(moving || reshaping) {
    bool locked = false;
    for(i = selected.begin(); i != selected.end(); i++) {
      locked |= (*i)->get_lock();  // if one is locked, all are locked
    }

    if(moving) {
      Vector ur, ll;
      box_of_selected(ll, ur, get_document());
      // new ll is at pointer pos minus offset
      Vector move = scr2pt(Gdk::Point(x - offset.x, y - offset.y)) - ll;  

      if(snap_mode == snap::GRID && !locked) {
	ll += move;  ur += move;
	Vector ll_s(snap_to_grid(ll.x, grid_offset.x, grid_size_x),
		    snap_to_grid(ll.y, grid_offset.y, grid_size_y));
	Vector ur_s(snap_to_grid(ur.x, grid_offset.x, grid_size_x),
		    snap_to_grid(ur.y, grid_offset.y, grid_size_y));
	Vector dll = ll_s - ll, dur = ur_s - ur;
	// the edge closest to a grid line wins
	Vector d(fabs(dll.x) < fabs(dur.x) ? dll.x : dur.x,
		 fabs(dll.y) < fabs(dur.y) ? dll.y : dur.y);
	move += d;
      }

      if(!locked) {
	for(i = selected.begin(); i != selected.end(); i++) {
	  Vector t = (*i)->get_matrix().tr();
	  t += move;
	  (*i)->set_translation(t);
	} 
      }
    } else if(reshaping) {
      for(i = selected.begin(); i != selected.end(); i++) {
	if(!(*i)->get_lock())
	  move_reshape_box(x, y);
      }
    }
    draw(true); // reshaping
  }

  return true;
}

void DocumentView::on_realize() {
  Gtk::DrawingArea::on_realize();

  // get_toplevel doesn't work in the constructor
  Gtk::Window *toplevel = dynamic_cast<Gtk::Window*>(get_toplevel());
  assert(toplevel != 0);
  // template_page_dialog is deleted in the destructor
  template_page_dialog.reset(new TemplatePageDialog(*toplevel, *this));

  win = get_window();
  gc = Gdk::GC::create(win);
  Glib::RefPtr<Gdk::Bitmap> bitmap;
  wait_pixmap = Gdk::Pixmap::create_from_xpm
    ((const Glib::RefPtr<const Gdk::Drawable>&)win, bitmap, white, wait_xpm); 
  missing_pixmap =
    Gdk::Pixmap::create_from_xpm
    ((const Glib::RefPtr<const Gdk::Drawable>&) win,
     bitmap, white, missing_xpm); 
  broken_pixmap =
    Gdk::Pixmap::create_from_xpm
    ((const Glib::RefPtr<const Gdk::Drawable>&) win,
     bitmap, white, broken_xpm); 
}

void DocumentView::draw(bool reshaping) {
  if(!is_realized())
    return; // will get lots of Gdk-CRITICALs otherwise

  Glib::RefPtr<Gtk::Style> style = get_style();
  IVector origin = get_origin();

  if(Page *page = get_page()) {
    // draw the actual page
    if(!pageview)
      pageview = core.getMeta("page")->create_viewent(*this, *page);
    pageview->draw_content();
    
//     // draw grid
//     if(snap_mode == snap::Grid) {
//       gc->set_line_attributes(1, Gdk::LINE_SOLID, Gdk::CAP_BUTT, 
// 			      Gdk::JOIN_MITER);
//       gc->set_foreground(get_color(Color::guide));
//       for(float x = grid_offset.x; x < page->get_width(); x += grid_size_x) {
// 	Vector v0(x, 0), v1(x, page->get_height());
// 	Gdk::Point p0 = pt2scr(v0), p1 = pt2scr(v1);
// 	win->draw_line(gc, p0.get_x(), p0.get_y(), p1.get_x(), p1.get_y());
//       }
//       for(float y = grid_offset.y; y < page->get_height(); y += grid_size_y) {
// 	Vector v0(0, y), v1(page->get_width(), y);
// 	Gdk::Point p0 = pt2scr(v0), p1 = pt2scr(v1);
// 	win->draw_line(gc, p0.get_x(), p0.get_y(), p1.get_x(), p1.get_y());
//       }
//       // draw diagonal (debug)
//       gc->set_foreground(get_color(Color::frame));
//       Vector ur, ll;
//       box_of_selected(ll, ur, *get_document());
//       ll = Vector(snap_to_grid(ll.x, grid_offset.x, grid_size_x),
// 		  snap_to_grid(ll.y, grid_offset.y, grid_size_y));
//       ur = Vector(snap_to_grid(ur.x, grid_offset.x, grid_size_x),
// 		  snap_to_grid(ur.y, grid_offset.y, grid_size_y));
//       Gdk::Point p0 = pt2scr(ll), p1 = pt2scr(ur);
//       win->draw_line(gc, p0.get_x(), p0.get_y(), p1.get_x(), p1.get_y());
//     }

    // draw resize handles
    gc->set_line_attributes(1, Gdk::LINE_SOLID, Gdk::CAP_BUTT, 
			    Gdk::JOIN_MITER);
    const Document::Selection &selection = document->selected();
    if(selection.size() == 1) {
      const Pagent &pagent = *selection.front();
      if(!pagent.get_lock() && &Page::containing(pagent) == page) {
	const int size = int(config.ReshapeBoxSize.values.front());
	ResizeHandles handles = get_resize_handles(pagent);
	for(ResizeHandles::const_iterator i = handles.begin(); 
	    i != handles.end(); i++) {
	  const Gdk::Point p = pt2scr(*i);
	  gc->set_foreground(get_color(Color::frame));
	  win->draw_rectangle(gc, false, 
			      p.get_x() - size/2, 
			      p.get_y() - size/2, 
			      size - 1, size - 1);
	  gc->set_foreground(get_color(Color::bg));
	  win->draw_rectangle(gc, true, p.get_x() - size/2 + 1, 
			      p.get_y() - size / 2 + 1, 
			      size - 2, size - 2);
	}
      }
    }

    // probably not necessary to draw background when reshaping
    //if(!reshaping) 
    {
      int w = page_width() + origin.x;
      int h = page_height() + origin.y;
      
      // draw widget background
      style->apply_default_background(win, true, Gtk::STATE_NORMAL,
				      Gdk::Rectangle(0, 0, 
						     get_width(), origin.y),
				      0, 0, get_width(), get_height());    
      style->apply_default_background(win, true, Gtk::STATE_NORMAL,
				      Gdk::Rectangle(0, 0, 
						     origin.x, get_height()),
				      0, 0, get_width(), get_height());    
      style->apply_default_background(win, true, Gtk::STATE_NORMAL,
				      Gdk::Rectangle(0, h, 
						     get_width(), 
						     get_height() - h),
				      0, 0, get_width(), get_height());    
      style->apply_default_background(win, true, Gtk::STATE_NORMAL,
				      Gdk::Rectangle(w, 0, 
						     get_width() - w,
						     get_height()),
				      0, 0, get_width(), get_height());    
    }
  } else if(document){
    // show that we have a document, albeit without pages
    Glib::ustring message = "No Pages";
    int x, y, w, h;

    w = page_width(); 
    h = page_height();

    Gdk::Rectangle rectangle(0, 0, get_width(), get_height());

    // draw widget background
    style->apply_default_background(win, true, Gtk::STATE_NORMAL,
				    rectangle,
				    0, 0, get_width(), get_height());    

    Glib::RefPtr<Pango::Context> context = get_pango_context();
    Glib::RefPtr<Pango::Layout> layout = Pango::Layout::create(context);
    layout->set_text(message);
    layout->get_pixel_size(x, y);
    x = (w - x) / 2 + origin.x; y = (h - y) / 2 + origin.y; 

    // draw box indicating paper size
    style->paint_box(win, Gtk::STATE_NORMAL, Gtk::SHADOW_ETCHED_IN,
		     rectangle, *this, "Foobar?",
		     origin.x, origin.y, 
		     w - 1, h - 1);

    // draw text
    style->paint_layout(win, Gtk::STATE_NORMAL, true,
			rectangle, *this, "Foobar?",
 			x, y,
 			layout);
  } else // just paint backgorund
    style->apply_default_background(win, true, Gtk::STATE_NORMAL,
				    Gdk::Rectangle(0, 0, 
						   get_width(), get_height()),
				    0, 0, get_width(), get_height());    
  show();
}

bool DocumentView::on_expose_event(GdkEventExpose *event) {
  draw();
  return false;
}

void DocumentView::on_drag_data_received
(const Glib::RefPtr<Gdk::DragContext>& context, 
 int x, int y, GtkSelectionData* selection_data, guint info, guint time)
{
  /// \todo unicode ???

  Gtk::SelectionData selection(selection_data);
  debug << "DocumentView::on_drop_drag_data_received:" << std::endl
	<< "x = " << x << ", y = " << y << std::endl
	<< selection.get_data_type() << " : \"" 
	<< selection.get_text() << "\"" << std::endl
	<< "\"" << selection.get_data_as_string() << "\"" << std::endl
	<< std::endl;

  if(!get_page()) {
    context->drag_finish(false, false, time);
    return;
  }

  // split into lines
  typedef std::vector<std::string> Strings;
  Strings strings;
  std::string data = selection.get_data_as_string();
  unsigned int start = 0;
  unsigned int stop;
  unsigned int length = data.length();
  do {
    unsigned int r = data.find('\r', start);
    unsigned int n = data.find('\n', start);
    stop = std::min(std::min(r, n), length);
    strings.push_back(data.substr(start, stop - start));
    start = stop;
    while(start < length 
	  && (data[start] == '\r' || data[start] == '\n'))
      start++;
  } while(start < length);
  
  bool success = false;

  for(Strings::const_iterator i = strings.begin();
      i != strings.end(); i++) {
    try {
      std::string filename = *i;
      if(filename.find("file://") == 0)
	filename.replace(0, 7, "");
      debug << "\"" << filename << "\"" << std::endl;
      new_image_frame(filename, false);
      success = true; // ok if at least one works
    } catch (const std::exception&) {
      debug << "failed to open" << std::endl;
    }
  }

  context->drag_finish(success, false, time);
}

const Gdk::Color& DocumentView::get_color(Color::Id color) const {
  switch(color) {
  case Color::bg:     return white;
  case Color::frame:  return black;
  case Color::locked: return gray;
  case Color::guide:  return red;
  case Color::empty:  return gray;
  default:  throw std::runtime_error("Unknown color id get_color");
  }
}
