///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "printdialog.h"
#include "widget/subpanel.h"
#include "widget/wmisc.h"
#include "widget/usererror.h"
#include <fstream>

#include <gtkmm/table.h>
#include <gtkmm/separator.h>
#include <gtkmm/stock.h>
#include <gtkmm/sizegroup.h>

#include "util/warning.h"
#include "util/filesys.h"
#include "util/os.h"
#include "util/tempfile.h"

#include "config.h"
#include "docview.h"
#include "document/document.h"

PrintDialog::PrintDialog(Gtk::Window &parent, 
			   DocumentView &_document_view):
  DialogWrap("Print", parent), 
  using_button("Print _using:", true), file_button("Print _to file:", true),
  all_button("_All", true), current_button("Cu_rrent", true), 
  from_button("_From:", true), 
  eps_button("Print as _EPS", true), fonts_button("_Include fonts", true),
  gray_button("_Grayscale", true),
  from_spinner(0, false), to_spinner(0, false),
  document_view(_document_view),
  file_entry("Print To File")
{
  set_resizable(false);

  {
    Gtk::RadioButton::Group group = using_button.get_group();
    file_button.set_group(group);
  }

  {
    Gtk::RadioButton::Group group = all_button.get_group();
    current_button.set_group(group);
    from_button.set_group(group);
  }
  
  Glib::RefPtr<Gtk::SizeGroup>  sizegroup = 
    Gtk::SizeGroup::create(Gtk::SIZE_GROUP_HORIZONTAL);
  sizegroup->add_widget(using_button);
  sizegroup->add_widget(file_button);

  Gtk::HBox *using_box = manage(new Gtk::HBox(false, double_space));
  using_box->pack_start(using_button, Gtk::PACK_SHRINK);
  using_box->pack_start(using_entry, Gtk::PACK_EXPAND_WIDGET);
    
  Gtk::HBox *file_box = manage(new Gtk::HBox(false, double_space));
  file_box->pack_start(file_button, Gtk::PACK_SHRINK);
  file_box->pack_start(file_entry, Gtk::PACK_EXPAND_WIDGET);

  Gtk::HBox *from_to_box = manage(new Gtk::HBox(false, double_space));
  from_to_box->pack_start(from_button, Gtk::PACK_SHRINK);
  from_to_box->pack_start(from_spinner, Gtk::PACK_SHRINK);
  from_to_box->pack_start(*manage(new Gtk::Label("to:")), 
			 Gtk::PACK_SHRINK);
  from_to_box->pack_start(to_spinner, Gtk::PACK_SHRINK);

  SubPanel *pages_box = manage(new SubPanel("Pages"));
  pages_box->pack_start(all_button, Gtk::PACK_SHRINK);
  pages_box->pack_start(current_button, Gtk::PACK_SHRINK);
  pages_box->pack_start(*from_to_box, Gtk::PACK_SHRINK);

  Gtk::VBox *format_box = manage(new Gtk::VBox(false, single_space));
  format_box->pack_start(eps_button, Gtk::PACK_SHRINK);
  format_box->pack_start(fonts_button, Gtk::PACK_SHRINK);
  format_box->pack_start(gray_button, Gtk::PACK_SHRINK);

  Gtk::HBox *foo_box = manage(new Gtk::HBox(false, double_space));
  foo_box->pack_start(*pages_box, Gtk::PACK_EXPAND_WIDGET);
  foo_box->pack_start(*manage(new Gtk::VSeparator()));
  foo_box->pack_start(*format_box, Gtk::PACK_SHRINK);

  Gtk::VBox *vbox = manage(new Gtk::VBox(false, double_space));
  vbox->set_border_width(border_width);
  //set_border_width(border_width);

  vbox->pack_start(*using_box, Gtk::PACK_SHRINK);
  vbox->pack_start(*file_box, Gtk::PACK_SHRINK);
  vbox->pack_start(*foo_box, Gtk::PACK_SHRINK, single_space);

  get_vbox()->pack_start(*vbox);
  get_vbox()->show_all();
  get_action_area()->show_all();

  using_entry.set_text(config.PrintCommand.values.front());
  using_button.set_active(true);

  using_button.signal_clicked().connect
    (slot(*this, &PrintDialog::update));
  // If there are only two buttons in a group, you only need to connect one.
  // If there are more, you need to connect them all. That is annoying.
  all_button.signal_clicked().connect
    (slot(*this, &PrintDialog::update));
  current_button.signal_clicked().connect
    (slot(*this, &PrintDialog::update));
  from_button.signal_clicked().connect
    (slot(*this, &PrintDialog::update));

  add_button(Gtk::Stock::CANCEL, 0);
  add_button(Gtk::Stock::PRINT, 1)->grab_default();
}

void PrintDialog::show_it() {
  DocRef document = document_view.get_document();
  if(!document)
    return;
  int first = document->get_first_page_num();
  int last = first + ((int) document->get_num_of_pages()) - 1;
  save_state(); 
  update();
  from_spinner.limits(first, last);  
  to_spinner.limits(first, last);
  from_spinner.set(first);
  to_spinner.set(last);
  show();
  // Gtk::Entry::set_position doesn't seem to work 
  // unless the entry is shown first
  const std::string &filename = 
    document_view.get_document_meta().get_filename();
  if(filename.empty())     // the document has not been saved 
    file_entry.entry.set_text("pptout.ps");
  else
    file_entry.entry.set_text(no_suffix(filename) + ".ps");
}

void PrintDialog::on_response(int response_id) {
  /// \todo setting the cursor has no effect until the function exits
  //  Glib::RefPtr<Gdk::Window> window = get_window();
  if(response_id == 0) {
    restore_state();
    hide();
  } else if(response_id == 1) { 
    try { 
      DocRef document = document_view.get_document();
      if(document) {
	int first = document->get_first_page_num();
	int last = first + document->get_num_of_pages() - 1;
	if(current_button.get_active()) {
	  first = last = document_view.get_current_page_num();
	} else if(from_button.get_active()) {
	  first = int(from_spinner.get());
	  last = int(to_spinner.get());
	  if(last < first)
	    throw UserError("Bad page interval",
			    "\"From\" page number must be lower\n"
			    "than \"To\" page number");
	  // We could make this error impossible to cause, 
	  // but I don't like putting leash and collar on the user.
	}
	// else all_button is active
	if(file_button.get_active()) {
	  std::ofstream out(file_entry.entry.get_text().c_str());
	  if(!out) throw UserError("Failed to open file for printing:\n"
				   + file_entry.entry.get_text(),
				   "Check that you have permission "
				   "to write to this file");
	  // 	  window->set_cursor(Gdk::WATCH);
	  document->print(out, first, last, 
			  eps_button.get_active(),
			  fonts_button.get_active(),
			  gray_button.get_active());
	  //	  window->set_cursor();
	} else {
	  Tempfile tempfile;
	  std::ofstream out(tempfile.get_filename().c_str());
	  if(!out) 
	    throw UserError("Failed to open temporary file for printing:\n"
			    + tempfile.get_filename(),
			    "Check that you have permission "
			    "to write to this file");
	  //	  window->set_cursor(Gdk::WATCH);
	  document->print(out, first, last, 
			  eps_button.get_active(),
			  fonts_button.get_active(),
			  gray_button.get_active());
	  //	  window->set_cursor();
	  std::string stdout_data, stderr_data;
	  int status;
	  std::string command = using_entry.get_text() 
	    + " < " + tempfile.get_filename();
	  command = "sh -c \"" + command + '\"';
	  try {
	    debug << command << std::endl;
	    Glib::spawn_command_line_sync(command,
					  &stdout_data, &stderr_data, &status);
	    debug << status << std::endl;
	    debug << stderr_data << std::endl;
	  } catch(const Glib::SpawnError &e) {
	    // Glib::SpawnError is not a std::exception
	    throw UserError("Failed to run: "
			    + using_entry.get_text(),
			    e.what());
	  }
	  if(status != 0)
	    throw UserError("Failed to print using: "
			    + using_entry.get_text(),
			    stderr_data);
	}
	hide();
      }
    } catch(...) {
      //      window->set_cursor(); // restore normal cursor
      throw;
    }
  }
}

void PrintDialog::save_state() {
  saved_using_text = using_entry.get_text();
  saved_file_text = file_entry.entry.get_text();
  saved_file_not_using = file_button.get_active();
  saved_eps = eps_button.get_active();
  saved_gray = gray_button.get_active();
  saved_fonts = fonts_button.get_active();
}

void PrintDialog::restore_state() {
  using_entry.set_text(saved_using_text);
  file_entry.entry.set_text(saved_file_text);
  file_button.set_active(saved_file_not_using);
  eps_button.set_active(saved_eps);
  gray_button.set_active(saved_gray);
  fonts_button.set_active(saved_fonts);
}

void PrintDialog::update() {
  using_entry.set_sensitive(using_button.get_active());
  file_entry.set_sensitive(file_button.get_active());
  eps_button.set_sensitive(current_button.get_active());
  // An EPS file may not have more than one page
  from_spinner.set_sensitive(from_button.get_active());
  to_spinner.set_sensitive(from_button.get_active());
}
