/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Quick Lounge Applet
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <libgnome/libgnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnome/gnome-desktop-item.h>
#include <libgnomeui/gnome-ditem-edit.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include "dlg-pick-applications.h"
#include "dlg-properties.h"
#include "file-utils.h"
#include "gtk-utils.h"
#include "quick-box.h"
#include "quick-separator.h"
#include "quick-lounge.h"


#define APPLET_IID         "OAFIID:GNOME_QuickLoungeApplet"
#define APPLET_FACTORY_IID "OAFIID:GNOME_QuickLoungeApplet_Factory"
#define APPLET_UI          "GNOME_QuickLoungeApplet.xml"

#define PREFS_LOCATION                 "location"
#define PREFS_ORDER                    "order"
#define PREFS_MIN_VISIBLE_COLS         "min_size"
#define PREFS_MAX_VISIBLE_COLS         "max_size"
#define PREFS_ICON_SIZE_FOLLOWS_PANEL  "icon_size_follows_panel"

#define BASE_DIR ".gnome2/quick-lounge"

#define DEFAULT_MIN_VISIBLE_COLS 4
#define DEFAULT_MAX_SIZE         20

#define ERROR_FORMAT "<span size=\"large\" weight=\"bold\">%s</span>\n\n%s"


enum {
        REVERT_BUTTON
};


static GtkTargetEntry target_table[] = {
	{ "text/uri-list", 0, 0 }
};
static guint n_targets = sizeof (target_table) / sizeof (target_table[0]);


static void
destroy_prop_dlg_cb (GtkWidget   *widget, 
		     QuickLounge *quick_lounge)
{
	quick_lounge->prop_dialog = NULL;
}


static void
properties_verb_cb (BonoboUIComponent *uic,
		    QuickLounge       *quick_lounge,
		    const gchar       *verbname)
{
	if (quick_lounge->prop_dialog == NULL) {
		quick_lounge->prop_dialog = dlg_properties (quick_lounge, 
							    NULL);
		g_signal_connect (G_OBJECT (quick_lounge->prop_dialog), 
				  "destroy",
				  G_CALLBACK (destroy_prop_dlg_cb),
				  quick_lounge);
	}

	if (quick_lounge->prop_dialog != NULL)
		gtk_window_present (GTK_WINDOW (quick_lounge->prop_dialog));
}


static void
help_verb_cb (BonoboUIComponent *uic,
	      QuickLounge       *quick_lounge,
	      const gchar       *verbname)
{
	GError *error = NULL;

	gnome_help_display_desktop (NULL, 
				    "quick-lounge-applet", 
				    NULL,
				    NULL, 
				    &error);
	
	if (error) {       
		_gtk_error_dialog_run (NULL, 
				       ERROR_FORMAT, 
				       _("Could not display help"), 
				       error->message);
		g_error_free (error);
	}
}


static void
about_verb_cb (BonoboUIComponent *uic,
	       QuickLounge       *quick_lounge,
	       const gchar       *verbname)
{
	GdkPixbuf  *pixbuf = NULL;
        char       *file;
	const char *authors[] = {
		"Paolo Bacchilega <paolo.bacch@tin.it>",
		NULL
	};
	const char *documenters [] = {
		NULL
	};
	const char *translator_credits = _("translator_credits");
	
	if (quick_lounge->about_dialog != NULL) {
		gtk_window_present (GTK_WINDOW (quick_lounge->about_dialog));
                return;
	}
	
	file = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_PIXMAP, "quick-lounge-applet.png", TRUE, NULL);
        pixbuf = gdk_pixbuf_new_from_file (file, NULL);
        g_free(file);

	quick_lounge->about_dialog = gnome_about_new (_("Quick Lounge"), 
						      VERSION,
						      _("Copyright (C) 2002 Free Software Foundation, Inc."),
						      _("Organize your favourite applications on the Panel"),
						      authors,
						      documenters,
						      strcmp (translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
						      pixbuf);

	gnome_window_icon_set_from_file (GTK_WINDOW (quick_lounge->about_dialog), GNOME_ICONDIR "quick-lounge-applet.png");

	g_signal_connect (G_OBJECT (quick_lounge->about_dialog), 
			  "destroy",
			  G_CALLBACK (gtk_widget_destroyed), 
			  &quick_lounge->about_dialog);

	gtk_widget_show_all (quick_lounge->about_dialog);
}


static const BonoboUIVerb menu_verbs [] = {
	BONOBO_UI_UNSAFE_VERB ("QLProperties",   properties_verb_cb),
	BONOBO_UI_UNSAFE_VERB ("QLHelp",         help_verb_cb),
	BONOBO_UI_UNSAFE_VERB ("QLAbout",        about_verb_cb),
	BONOBO_UI_VERB_END
};


#if 0

static void
remove_directory (const char *uri)
{
	GnomeVFSDirectoryHandle *h;
	GnomeVFSFileInfo        *file_info;
	GnomeVFSResult           r;

	/* this directory does not contain sub-directories so let's keep the
	 * function simple. */

	r = gnome_vfs_directory_open (&h, uri, GNOME_VFS_FILE_INFO_DEFAULT);
	if (r != GNOME_VFS_OK)
		return;

	file_info = gnome_vfs_file_info_new ();
	while (gnome_vfs_directory_read_next (h, file_info) == GNOME_VFS_OK) {
		char *file_uri;
		file_uri = g_strconcat (uri, "/", file_info->name, NULL);
		gnome_vfs_unlink (file_uri);
		g_free (file_uri);
	}
	gnome_vfs_file_info_unref (file_info);
	gnome_vfs_directory_close (h);

	gnome_vfs_remove_directory (uri);
}

#endif

static void
gh_desktop_item_unref  (gpointer  key,
			gpointer  value,
			gpointer  user_data)
{
	gnome_desktop_item_unref (value);
}


static void
applet_destroy (GtkWidget    *applet, 
		QuickLounge  *quick_lounge)
{
	/* free memory */

	if (quick_lounge->about_dialog != NULL)
                gtk_widget_destroy (quick_lounge->about_dialog);
	g_free (quick_lounge->location);

	if (quick_lounge->ditem_hash != NULL) {
                g_hash_table_foreach (quick_lounge->ditem_hash, 
                                      gh_desktop_item_unref,
                                      NULL);
                g_hash_table_destroy (quick_lounge->ditem_hash);
        }

	g_object_unref (quick_lounge->icon_theme);

	g_free (quick_lounge);
}


static void
applet_change_orient (PanelApplet       *applet,
                      PanelAppletOrient  orientation,
                      QuickLounge       *quick_lounge)
{
        if (quick_lounge->orientation == orientation)
                return;
	quick_lounge->orientation = orientation;
	quick_box_set_orient (quick_lounge->quick_box, orientation);
}


static void
applet_change_size (PanelApplet   *applet,
                    int            size,
                    QuickLounge   *quick_lounge)
{
	if (quick_lounge->size == size)
                return;
	quick_lounge->size = size;
	quick_box_set_size (quick_lounge->quick_box, size);
}


static void
applet_change_background (PanelApplet               *applet,
                          PanelAppletBackgroundType  type,
                          GdkColor                  *color,
                          const gchar               *pixmap,
                          QuickLounge               *quick_lounge)
{
	GtkWidget *widget = GTK_WIDGET (applet);

	if (type == PANEL_NO_BACKGROUND) {
		GtkRcStyle *rc_style = gtk_rc_style_new ();
		gtk_widget_modify_style (widget, rc_style);
		gtk_rc_style_unref (rc_style);

	} else if (type == PANEL_COLOR_BACKGROUND) 
		gtk_widget_modify_bg (widget, GTK_STATE_NORMAL, color);

	else if ((type == PANEL_PIXMAP_BACKGROUND) && (pixmap != NULL)) {
		/* FIXME */
	}
}


static void
applet_size_request (GtkWidget      *widget,
                     GtkRequisition *requisition,
                     QuickLounge    *quick_lounge)
{
	GtkRequisition  child_req;
        const int      *size_hints;
	int             len;

	gtk_widget_get_child_requisition (quick_lounge->applet, &child_req);
	size_hints = quick_box_get_size_hint_list (quick_lounge->quick_box, &len);
	panel_applet_set_size_hints (PANEL_APPLET (quick_lounge->applet),
                                     size_hints, 
                                     len, 
				     child_req.width);
}


/* -- Button Context Menu -- */


static gboolean
ditem_properties_apply_timeout (gpointer data)
{
	GtkWidget        *dialog = data;
	GnomeDItemEdit   *dedit;
        GError           *error = NULL;
	GnomeDesktopItem *ditem, *orig_ditem;
	QuickLounge      *quick_lounge;
	QuickButton      *button;
	QuickBox         *quick_box;
	GtkWidget        *revert_button;
	const char       *uri;

	dedit = g_object_get_data (G_OBJECT (dialog), "ditem_edit");

        g_object_set_data (G_OBJECT (dedit), "apply_timeout", NULL);

	/* save */

	quick_box = g_object_get_data (G_OBJECT (dialog), "quick_box");
	button = g_object_get_data (G_OBJECT (dialog), "button");
	orig_ditem = g_object_get_data (G_OBJECT (button), "ditem");
	uri = g_object_get_data (G_OBJECT (button), "uri");
	quick_lounge = g_object_get_data (G_OBJECT (quick_box), "quick_lounge");
	ditem = gnome_ditem_edit_get_ditem (dedit);
	gnome_desktop_item_set_location (ditem, gnome_desktop_item_get_location (orig_ditem));
	gnome_desktop_item_save (ditem, uri, TRUE, &error);
	
        /* save the error for later */

        if (error != NULL) {
                g_object_set_data_full (G_OBJECT (dedit), "SavingError",
                                        g_strdup (error->message),
                                        (GDestroyNotify) g_free);
                g_clear_error (&error);
        } else 
                g_object_set_data (G_OBJECT (dedit), "SavingError", NULL);

	/* update */

	quick_box_update_child (quick_box, button, ditem);
	dlg_properties_update (quick_lounge->prop_dialog);
	revert_button = g_object_get_data (G_OBJECT (dialog), "revert_button");
	gtk_widget_set_sensitive (revert_button, TRUE);

        return FALSE;
}


static void ditem_properties_changed (GtkWidget *dedit, gpointer   data);


static void
ditem_properties_clicked (GtkWidget *w, 
			  int        response, 
			  gpointer   data)
{
	GtkWidget *dialog = data;

        if (response == GTK_RESPONSE_HELP) {
		/* FIXME */

	} else if (response == REVERT_BUTTON) { 
		QuickButton      *button;
		GnomeDItemEdit   *dedit;
		GtkWidget        *revert_button;
		GnomeDesktopItem *revert_ditem, *ditem;

		dedit = g_object_get_data (G_OBJECT (dialog), "ditem_edit");
		button = g_object_get_data (G_OBJECT (dialog), "button");
		revert_ditem = g_object_get_data (G_OBJECT (dialog), "revert_ditem");

		g_signal_handlers_block_by_func (G_OBJECT (dedit),
						 ditem_properties_changed,
						 dialog);

		ditem = gnome_desktop_item_copy (revert_ditem);
		gnome_ditem_edit_set_ditem (GNOME_DITEM_EDIT (dedit), ditem);
		gnome_desktop_item_unref (ditem);

		g_signal_handlers_unblock_by_func (G_OBJECT (dedit),
						   ditem_properties_changed,
						   dialog);

		ditem_properties_apply_timeout (dialog);

		revert_button = g_object_get_data (G_OBJECT (dialog), "revert_button");
		gtk_widget_set_sensitive (revert_button, FALSE);

	} else if (response == GTK_RESPONSE_CLOSE) 
		gtk_widget_destroy (dialog);
}


static void
ditem_properties_changed (GtkWidget *dedit, 
			  gpointer   data)
{
	GtkWidget *dialog = data;
	gpointer   timeout_data;
	guint      timeout;

	timeout_data = g_object_get_data (G_OBJECT (dedit), "apply_timeout");
	timeout = GPOINTER_TO_UINT (timeout_data);

	g_object_set_data (G_OBJECT (dedit), "apply_timeout", NULL);

	if (timeout != 0)
		g_source_remove (timeout);

        /* Will delay save for after 0.2 seconds */
	timeout = g_timeout_add (2 * 100,
				 ditem_properties_apply_timeout,
				 dialog);

	g_object_set_data (G_OBJECT (dedit), 
			   "apply_timeout",
			   GUINT_TO_POINTER (timeout));
}


static void
ditem_properties_close (GtkWidget *dialog,
                        GtkWidget *data)
{
	GnomeDItemEdit   *dedit;
	const char       *saving_error;
	gpointer          timeout_data;
	guint             timeout;
	GnomeDesktopItem *revert_ditem;

	dedit = g_object_get_data (G_OBJECT (dialog), "ditem_edit");
	revert_ditem = g_object_get_data (G_OBJECT (dialog), "revert_ditem");

	g_signal_handlers_disconnect_by_func (G_OBJECT (dedit),
					      ditem_properties_changed,
					      dialog);

	timeout_data = g_object_get_data (G_OBJECT (dedit), "apply_timeout");
	timeout = GPOINTER_TO_UINT (timeout_data);

        g_object_set_data (G_OBJECT (dedit), "apply_timeout", NULL);

        /* If there was a timeout, then something changed after last save,
         * so we must save again now */
        if (timeout != 0) {
                g_source_remove (timeout);
                ditem_properties_apply_timeout (dialog);
        }

        saving_error = g_object_get_data (G_OBJECT (dedit), "SavingError");
        if (saving_error != NULL) 
		_gtk_error_dialog_run (NULL, 
				       _("Cannot save changes to launcher\n\n"
					 "Details: %s"), 
				       saving_error);

	gnome_ditem_edit_clear (dedit);
	gnome_desktop_item_unref (revert_ditem);
}


/* ---- 8< ---- 8< ---- */


void
quick_lounge_util__item_properties (QuickLounge *quick_lounge,
				    GtkWidget   *button)
{
	GtkWidget        *widget;
	GtkDialog        *dialog;
	GtkWidget        *dedit;
	GnomeDesktopItem *ditem;
	GnomeDesktopItem *revert_ditem;
	GtkWidget        *revert_button;

	widget = gtk_dialog_new_with_buttons (_("Launcher Properties"),
					      NULL, 0 /* flags */,
					      GTK_STOCK_HELP,
					      GTK_RESPONSE_HELP,
					      NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (widget), GNOME_ICONDIR "quick-lounge-applet.png");
	gtk_window_set_screen (GTK_WINDOW (widget), gtk_widget_get_screen (GTK_WIDGET (quick_lounge->applet)));

	dialog = GTK_DIALOG (widget);

	revert_button = gtk_dialog_add_button (dialog, GTK_STOCK_REVERT_TO_SAVED, REVERT_BUTTON);
	gtk_dialog_add_button (dialog, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);

	gtk_dialog_set_has_separator (dialog, FALSE);
	gtk_dialog_set_default_response (dialog, GTK_RESPONSE_CLOSE);

	ditem = g_object_get_data (G_OBJECT (button), "ditem");
	revert_ditem = gnome_desktop_item_copy (ditem);

	dedit = gnome_ditem_edit_new ();
	gnome_ditem_edit_set_ditem (GNOME_DITEM_EDIT (dedit), ditem);
        gtk_widget_show (dedit);

	gtk_box_pack_start (GTK_BOX (dialog->vbox), dedit, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
	gtk_container_set_border_width (GTK_CONTAINER (dedit), 6);
	gtk_box_set_spacing (GTK_BOX (dialog->vbox), 6);

        g_object_set_data (G_OBJECT (dialog), "quick_box", button->parent);
        g_object_set_data (G_OBJECT (dialog), "button", button);
	g_object_set_data (G_OBJECT (dialog), "ditem_edit", dedit);
	g_object_set_data (G_OBJECT (dialog), "revert_button", revert_button);
	g_object_set_data (G_OBJECT (dialog), "revert_ditem", revert_ditem);

	g_signal_connect (dedit, "changed",
                          G_CALLBACK (ditem_properties_changed), 
			  dialog);

	g_signal_connect (G_OBJECT (dialog), "response",
			  G_CALLBACK (ditem_properties_clicked),
			  dialog);
	g_signal_connect (G_OBJECT (dialog), "destroy",
                          G_CALLBACK (ditem_properties_close), 
			  dialog);

	gtk_widget_set_sensitive (revert_button, FALSE);
        gnome_ditem_edit_grab_focus (GNOME_DITEM_EDIT (dedit));
	gtk_widget_show (widget);
}


static void
item_remove_cb (GtkWidget *button)
{
	GtkWidget   *quick_box = button->parent;
	const char  *uri;
	QuickLounge *quick_lounge;

	quick_lounge = g_object_get_data (G_OBJECT (quick_box), 
					  "quick_lounge");

	uri = g_object_get_data (G_OBJECT (button), "uri");
	if (uri != NULL)
		gnome_vfs_unlink (uri);

	gtk_container_remove (GTK_CONTAINER (quick_box), button);
	quick_lounge_save_order (quick_lounge);
	dlg_properties_update (quick_lounge->prop_dialog);
}


static void
item_move_cb (GtkWidget *button)
{
	QuickBox    *quick_box = QUICK_BOX (button->parent);
	QuickLounge *quick_lounge;

	quick_lounge = g_object_get_data (G_OBJECT (quick_box), 
					  "quick_lounge");
	if (quick_lounge == NULL) 
		return;
	
	if (quick_lounge->prop_dialog != NULL)
		dlg_properties_select_button (quick_lounge->prop_dialog, 
					      button);
	else {
		quick_lounge->prop_dialog = dlg_properties (quick_lounge, 
							    button);
		g_signal_connect (G_OBJECT (quick_lounge->prop_dialog),
				  "destroy",
				  G_CALLBACK (destroy_prop_dlg_cb),
				  quick_lounge);
	}
}


/* -- show help on app -- */

/* -----8<----- cut & paste from the gnome-panel -----8<----- */

GQuark
panel_help_error_quark (void)
{
        static GQuark q = 0;
        if (q == 0)
                q = g_quark_from_static_string ("panel-help-error-quark");

        return q;
}

typedef enum {
        PANEL_HELP_ERROR_NO_DOCPATH /* No docpath sent */,
        PANEL_HELP_ERROR_NOT_FOUND /* Document not found */
} PanelHelpError;

#define PANEL_HELP_ERROR panel_help_error_quark ()

static gboolean
panel_show_gnome_help (const char *docpath, GError **error)
{
        char *app, *p, *path;
        gboolean retval;

        app = g_strdup (docpath);

        p = strchr (app, '/');

        if (p == NULL) {
                g_free (app);
                g_set_error (error,
                             PANEL_HELP_ERROR,
                             PANEL_HELP_ERROR_NOT_FOUND,
                             /* XXX: Not really a correct error,
                              * but just somewhat correct */
                             _("Help document not found"));
                return FALSE;
        }

        path = p+1;
        *p = '\0';

        retval = TRUE;

        if ( ! gnome_help_display_desktop (NULL /* program */,
                                           app /* doc_id */,
                                           path /* file_name */,
                                           NULL /* link_id */,
                                           error)) {
                retval = gnome_help_display_with_doc_id (NULL /* program */,
                                                         app /* doc_id */,
                                                         path /* file_name */,
                                                         NULL /* link_id */,
                                                         error);
        }

        g_free (app);

        return retval;
}


static gboolean
panel_show_kde_help (const char *docpath, GError **error)
{
        const GList *li;

        if ( ! g_file_test (KDE_DOCDIR, G_FILE_TEST_EXISTS)) {
                g_set_error (error,
                             PANEL_HELP_ERROR,
                             PANEL_HELP_ERROR_NOT_FOUND,
                             _("Help document not found"));
                return FALSE;
        }

        for (li = gnome_i18n_get_language_list ("LC_MESSAGES");
             li != NULL;
             li = li->next) {
                char *fullpath = g_strdup_printf ("%s/HTML/%s/%s",
                                                  KDE_DOCDIR,
                                                  (char *)li->data,
                                                  docpath);
                if (g_file_test (fullpath, G_FILE_TEST_EXISTS)) {
                        gboolean retval;
                        char *uri = g_strconcat ("ghelp:", fullpath, NULL);
                        g_free (fullpath);
                        retval = gnome_help_display_uri (uri, error);
                        g_free (uri);
                        return retval;
                }
                g_free (fullpath);
        }
	
	g_set_error (error,
                     PANEL_HELP_ERROR,
                     PANEL_HELP_ERROR_NOT_FOUND,
                     _("Help document not found"));

        return FALSE;
}


/* is url showable by gnome_url_show */
gboolean
panel_is_url (const char *url)
{
        if (strncmp (url, "http://", strlen ("http://")) == 0 ||
            strncmp (url, "https://", strlen ("https://")) == 0 ||
            strncmp (url, "gopher://", strlen ("gopher://")) == 0 ||
            strncmp (url, "ftp://", strlen ("ftp://")) == 0 ||
            strncmp (url, "file:", strlen ("file:")) == 0 ||
            strncmp (url, "ghelp:", strlen ("ghelp:")) == 0 ||
            strncmp (url, "help:", strlen ("help:")) == 0 ||
            strncmp (url, "man:", strlen ("man:")) == 0 ||
            strncmp (url, "info:", strlen ("info:")) == 0)
                return TRUE;
        else
                return FALSE;
}


/* -----8<----- cut & paste from the gnome-panel -----8<----- */


static void
item_help_on_app_cb (GtkWidget *button)
{
	GnomeDesktopItem *ditem;
	const char       *docpath;
	GError           *err = NULL;

	ditem = g_object_get_data (G_OBJECT (button), "ditem");
	docpath = gnome_desktop_item_get_localestring (ditem, "X-GNOME-DocPath");

	if (panel_is_url (docpath))
                gnome_help_display_uri (docpath, &err);
	else if (! panel_show_gnome_help (docpath, &err)) {
		g_clear_error (&err);
		panel_show_kde_help (docpath, &err);
	}
	
        if (err != NULL) {
		_gtk_error_dialog_run (NULL,
				       ERROR_FORMAT,
				       _("Could not display help"),
				       err->message);
                g_error_free (err);
        }
}


static int
get_child_position (QuickBox  *quick_box,
		    GtkWidget *widget)
{
	GList  *scan;
	int     pos = 0;

	for (scan = GTK_BOX (quick_box)->children; scan; scan = scan->next) {
		GtkBoxChild *child = scan->data;

		if (child->widget == widget)
			return pos;
		pos++;
	}
	return -1;
}


char *
quick_lounge_util__get_unique_uri (QuickLounge *quick_lounge)
{
	char *base_dir, *desktop_file, *uri;

	base_dir = quick_lounge_get_path (quick_lounge);
	desktop_file = get_unique_desktop_file (base_dir);
	uri = g_strconcat ("file://", desktop_file, NULL);
	g_free (desktop_file);
	g_free (base_dir);

	return uri;
}


void
quick_lounge_set_textdomain ()
{
	static gboolean domain_setted = FALSE;

	if (!domain_setted) {
		textdomain (GETTEXT_PACKAGE);
		domain_setted = TRUE;
	}
}


/* -- */


static void
new_launcher_dialog_response_cb (GtkWidget *dialog,
				 int        response,
				 GtkWidget *data)
{
	QuickLounge      *quick_lounge;
	GnomeDItemEdit   *dedit;
	gpointer          ppos;
	int               pos;

	quick_lounge = g_object_get_data (G_OBJECT (dialog), "quick_lounge");
	dedit = g_object_get_data (G_OBJECT (dialog), "ditem_edit");
	ppos = g_object_get_data (G_OBJECT (dialog), "position");
	pos = GPOINTER_TO_INT (ppos);

	if (response == GTK_RESPONSE_HELP) {
		/* FIXME */
		
	} else if (response == GTK_RESPONSE_OK) { 
		GnomeDesktopItem *ditem;
		char             *uri;

		uri = quick_lounge_util__get_unique_uri (quick_lounge);

		ditem = gnome_ditem_edit_get_ditem (dedit);
		gnome_desktop_item_set_location_file (ditem, uri + strlen ("file://"));

		if (! gnome_desktop_item_save (ditem, NULL, TRUE, NULL)) {
			_gtk_error_dialog_run (NULL, 
					       ERROR_FORMAT, 
					       _("Could not create the new launcher"),
					       errno_to_string ());
			g_free (uri);
			gtk_widget_destroy (dialog);
			return;
		}
		
		quick_box_add_button (quick_lounge->quick_box, uri, pos + 1);
		g_free (uri);

		quick_lounge_save_order (quick_lounge);
		dlg_properties_update (quick_lounge->prop_dialog);

		gtk_widget_destroy (dialog);

	} else if (response == GTK_RESPONSE_CANCEL) 
		gtk_widget_destroy (dialog);
}


void
quick_lounge_new_launcher (QuickLounge *quick_lounge,
			   int          pos)
{
	GtkWidget *widget;
	GtkDialog *dialog;
	GtkWidget *dedit;

	widget = gtk_dialog_new_with_buttons (_("New Launcher"),
					      NULL, 0 /* flags */,
					      GTK_STOCK_HELP, GTK_RESPONSE_HELP,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OK, GTK_RESPONSE_OK,
					      NULL);
	gnome_window_icon_set_from_file (GTK_WINDOW (widget), GNOME_ICONDIR "quick-lounge-applet.png");
	gtk_window_set_screen (GTK_WINDOW (dialog), gtk_widget_get_screen (GTK_WIDGET (quick_lounge->applet)));
	
	dialog = GTK_DIALOG (widget);

	gtk_dialog_set_has_separator (dialog, FALSE);
	gtk_dialog_set_default_response (dialog, GTK_RESPONSE_OK);
	
	dedit = gnome_ditem_edit_new ();
	gtk_widget_show (dedit);

	gtk_box_pack_start (GTK_BOX (dialog->vbox), dedit, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
	gtk_container_set_border_width (GTK_CONTAINER (dedit), 6);
	gtk_box_set_spacing (GTK_BOX (dialog->vbox), 6);

	g_object_set_data (G_OBJECT (dialog), "quick_lounge", quick_lounge);
	g_object_set_data (G_OBJECT (dialog), "ditem_edit", dedit);
	g_object_set_data (G_OBJECT (dialog), "position", GINT_TO_POINTER (pos));

	g_signal_connect (G_OBJECT (dialog), "response",
			  G_CALLBACK (new_launcher_dialog_response_cb),
			  dialog);

	gnome_ditem_edit_grab_focus (GNOME_DITEM_EDIT (dedit));
	gtk_widget_show (widget);
}


static void
add_launcher_cb (GtkWidget *button)
{
	QuickBox         *quick_box = QUICK_BOX (button->parent);
	QuickLounge      *quick_lounge;
	int               pos;

	quick_lounge = g_object_get_data (G_OBJECT (quick_box), "quick_lounge");
	if (quick_lounge == NULL) 
		return;

	pos = get_child_position (quick_box, button);

	quick_lounge_new_launcher (quick_lounge, pos);
}


static void
add_from_menu_cb (GtkWidget *button)
{
	QuickBox    *quick_box = QUICK_BOX (button->parent);
	QuickLounge *quick_lounge;
	int          pos;

	quick_lounge = g_object_get_data (G_OBJECT (quick_box), "quick_lounge");
	if (quick_lounge == NULL) 
		return;

	pos = get_child_position (quick_box, button);
	dlg_pick_applications (quick_lounge, pos);
}


static void
add_space_cb (GtkWidget *button)
{
	QuickBox    *quick_box = QUICK_BOX (button->parent);
	QuickLounge *quick_lounge;
	int          pos;

	quick_lounge = g_object_get_data (G_OBJECT (quick_box), "quick_lounge");
	if (quick_lounge == NULL) 
		return;

	pos = get_child_position (quick_box, button);
	quick_box_add_separator (quick_box, pos + 1);
	quick_lounge_save_order (quick_lounge);
	dlg_properties_update (quick_lounge->prop_dialog);
}


static void
activate_item_cb (GtkWidget *item,
		  GtkWidget *button)
{
	QuickBox    *quick_box;
	QuickLounge *quick_lounge;

	quick_box = (QuickBox*) button->parent;
	quick_lounge = g_object_get_data (G_OBJECT (quick_box), "quick_lounge");
	if (quick_lounge == NULL)
		return;

	quick_lounge_util__item_properties (quick_lounge, button);	
}

		  
static void 
populate_context_menu (GtkWidget *popup_menu, 
		       GtkWidget *button,
		       gpointer   data)
{
	GtkWidget        *item;
	GnomeDesktopItem *ditem;
	const char       *docpath;

	quick_lounge_set_textdomain ();

	/**/

	if (! QUICK_IS_SEPARATOR (button)) {
		item = gtk_image_menu_item_new_with_mnemonic (_("_Properties"));
		gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
		gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), gtk_image_new_from_stock (GTK_STOCK_PROPERTIES, GTK_ICON_SIZE_MENU));
		gtk_widget_show_all (item);
	
		g_signal_connect (G_OBJECT (item), 
				  "activate",
				  G_CALLBACK (activate_item_cb), 
				  button);
	}

	/**/

	item = gtk_image_menu_item_new_with_mnemonic (_("_Help"));
	gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), gtk_image_new_from_stock (GTK_STOCK_HELP, GTK_ICON_SIZE_MENU));
	gtk_widget_show_all (item);

	/**/

	ditem = g_object_get_data (G_OBJECT (button), "ditem");
	docpath = gnome_desktop_item_get_localestring (ditem, "X-GNOME-DocPath");
	if (docpath != NULL) {
		const char *app_name;
		char       *label;

		app_name = gnome_desktop_item_get_localestring (ditem, "Name");
		label = g_strdup_printf (_("Help on %s Application"), app_name);
		item = gtk_image_menu_item_new_with_mnemonic (label);
		g_free (label);

		gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
		gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), gtk_image_new_from_stock (GTK_STOCK_HELP, GTK_ICON_SIZE_MENU));
		gtk_widget_show_all (item);
		
		g_signal_connect_swapped (G_OBJECT (item), 
					  "activate",
					  G_CALLBACK (item_help_on_app_cb), 
					  button);
	}

	/**/

	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
	gtk_widget_show_all (item);

	/**/

	item = gtk_image_menu_item_new_with_mnemonic (_("Add _Launcher"));
	gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
	gtk_widget_show_all (item);

	g_signal_connect_swapped (G_OBJECT (item), 
				  "activate",
				  G_CALLBACK (add_launcher_cb), 
				  button);

	/**/

	item = gtk_image_menu_item_new_with_mnemonic (_("Add From _Menu"));
	gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
	gtk_widget_show_all (item);

	g_signal_connect_swapped (G_OBJECT (item), 
				  "activate",
				  G_CALLBACK (add_from_menu_cb), 
				  button);

	/**/

	item = gtk_image_menu_item_new_with_mnemonic (_("Add _Space"));
	gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
	gtk_widget_show_all (item);

	g_signal_connect_swapped (G_OBJECT (item), 
				  "activate",
				  G_CALLBACK (add_space_cb), 
				  button);

	/**/

	item = gtk_separator_menu_item_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
	gtk_widget_show_all (item);

	/**/

	item = gtk_image_menu_item_new_with_mnemonic (_("_Remove"));
	gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), gtk_image_new_from_stock (GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU));
	gtk_widget_show_all (item);

	g_signal_connect_swapped (G_OBJECT (item), 
				  "activate",
				  G_CALLBACK (item_remove_cb), 
				  button);

	/**/

	item = gtk_image_menu_item_new_with_mnemonic (_("_Move"));
	gtk_menu_shell_append (GTK_MENU_SHELL (popup_menu), item);
	gtk_widget_show_all (item);

	g_signal_connect_swapped (G_OBJECT (item), 
				  "activate",
				  G_CALLBACK (item_move_cb), 
				  button);

}


/* -- Drag & Drop -- */


static GtkWidget *
get_button_from_uri (QuickBox   *quick_box,
		     const char *uri1)
{
	GtkBox *box = GTK_BOX (quick_box);
	GList  *scan;

	for (scan = box->children; scan; scan = scan->next) {
		GtkBoxChild *child = scan->data;
		const char  *uri2;

		uri2 = g_object_get_data (G_OBJECT (child->widget), "uri");
		if (uri2 == NULL)
			continue;
		if (strcmp (uri1, uri2) == 0)
			return child->widget;
	}

	return NULL;
}


static void  
drag_data_received  (GtkWidget          *widget,
		     GdkDragContext     *context,
		     int                 x,
		     int                 y,
		     GtkSelectionData   *data,
		     guint               info,
		     guint               time,
		     gpointer            extra_data)
{
	QuickLounge *quick_lounge = extra_data;
	QuickBox    *quick_box = QUICK_BOX (widget);
        GList       *list, *scan;
	int          pos;

        if (! ((data->length >= 0) && (data->format == 8))) {
                gtk_drag_finish (context, FALSE, FALSE, time);
                return;
        }

	pos = quick_box_get_pointer_position (quick_box, x, y);

	list = get_file_list_from_url_list ((char *) data->data);

	for (scan = list; scan; scan = scan->next) {
		char      *src_uri = scan->data;
		char      *uri;
		GtkWidget *button;

		if (! is_desktop_file (src_uri))
			continue;

		uri = g_strconcat (quick_lounge->location,
				   "/",
				   file_name_from_path (src_uri),
				   NULL);

		if (strcmp (src_uri, uri) == 0) {
			button = get_button_from_uri (quick_box, uri);

			if (button != NULL) {
				int button_pos;
				button_pos = get_child_position (quick_box, button);
				if (button_pos != -1) 
					quick_box_reorder_child (quick_box, button, pos);
			}

			g_free (uri);
			continue;
		}

		g_free (uri);
		uri = quick_lounge_util__get_unique_uri (quick_lounge);

		if (! file_copy (src_uri, uri)) {
			g_free (uri);
			continue;
		}

		quick_box_add_button (quick_box, uri, ++pos);
		g_free (uri);
	}
	path_list_free (list);

	gtk_drag_finish (context, TRUE, FALSE, time);
	quick_lounge_save_order (quick_lounge);
	dlg_properties_update (quick_lounge->prop_dialog);
}


void
quick_lounge_save_order (QuickLounge *quick_lounge)
{
	GtkBox       *box = GTK_BOX (quick_lounge->quick_box);
	PanelApplet  *applet = PANEL_APPLET (quick_lounge->applet);
	GConfClient  *client;
	char         *full_key;
	GList        *scan;
	GSList       *uri_list = NULL;

	for (scan = box->children; scan; scan = scan->next) {
		GtkBoxChild      *child_info = scan->data;
		GtkWidget        *button = child_info->widget;
		GnomeDesktopItem *ditem;
		
		if (QUICK_IS_SEPARATOR (button)) {
			uri_list = g_slist_prepend (uri_list, g_strdup (""));
			continue;
		}

		ditem = g_object_get_data (G_OBJECT (button), "ditem");
		if (ditem != NULL) {
			const char *uri;
			uri = file_name_from_path (gnome_desktop_item_get_location (ditem));
			uri_list = g_slist_prepend (uri_list, g_strdup (uri));
		}
	}
	uri_list = g_slist_reverse (uri_list);

	full_key = panel_applet_gconf_get_full_key (applet, PREFS_ORDER);
	client = gconf_client_get_default ();
	gconf_client_set_list (client, full_key, GCONF_VALUE_STRING, uri_list,
			       NULL);
	g_free (full_key);

	g_slist_foreach (uri_list, (GFunc) g_free, NULL);
	g_slist_free (uri_list);
}


char *
quick_lounge_get_path (QuickLounge *quick_lounge)
{
	return gnome_vfs_get_local_path_from_uri (quick_lounge->location);
}


/* -- Load location -- */


static gboolean
path_in_list (GList *list, const char *path1)
{
	GList *scan;

	for (scan = list; scan; scan = scan->next) {
		char *path2 = scan->data;
		if (strcmp (path1, path2) == 0)
			return TRUE;
	}

	return FALSE;
}


static GList*
get_order_list (QuickLounge *quick_lounge)
{
	PanelApplet  *applet = PANEL_APPLET (quick_lounge->applet);
	GSList       *uri_list, *scan;
	GList        *result = NULL;
	GConfClient  *client;
	char         *full_key;

	full_key = panel_applet_gconf_get_full_key (applet, PREFS_ORDER);
	client = gconf_client_get_default ();
	uri_list = gconf_client_get_list (client, full_key, GCONF_VALUE_STRING,
					  NULL);
	g_free (full_key);

	/* GSList --> GList */

	for (scan = uri_list; scan; scan = scan->next) {
		const char *filename = scan->data;
		char       *uri;

		if (*filename == 0)
			uri = g_strdup ("");
		else
			uri = g_strconcat (quick_lounge->location, 
					   "/", 
					   filename, 
					   NULL);

		result = g_list_prepend (result, uri);
	}
	result = g_list_reverse (result);

	g_slist_foreach (uri_list, (GFunc) g_free, NULL);
	g_slist_free (uri_list);

	return result;
}


static void
load_uri__step2 (PathListData *pld, 
		 gpointer      data)
{
	QuickLounge *quick_lounge = data;
	QuickBox    *quick_box = quick_lounge->quick_box;
	GList       *folder_list = NULL;
	GList       *order_list = NULL;
	GList       *files, *scan;

	if (pld->result != GNOME_VFS_ERROR_EOF) {
		path_list_data_free (pld);
		return;
	}

	/**/

	order_list = get_order_list (quick_lounge);

	folder_list = pld->files;
	pld->files = NULL;

	for (scan = folder_list; order_list && scan;) {
		char *path = scan->data;
		
		if (path_in_list (order_list, path)) {
			folder_list = g_list_remove_link (folder_list, scan);
			g_list_free (scan);
			g_free (path);
			
			scan = folder_list;
		} else
			scan = scan->next;
	}

	files = g_list_concat (order_list, folder_list);

	gtk_widget_hide (GTK_WIDGET (quick_lounge->quick_box));

	for (scan = files; scan; scan = scan->next) {
		char *uri = scan->data;

		if (*uri == 0)  
			quick_box_add_separator (quick_box, -1);
		else if (is_desktop_file (uri))
			quick_box_add_button (quick_box, uri, -1);
	}

	path_list_free (files);
	path_list_handle_free (quick_lounge->handle);
	quick_lounge->handle = NULL;

	gtk_widget_show (GTK_WIDGET (quick_lounge->quick_box));

	quick_lounge_save_order (quick_lounge);
	dlg_properties_update (quick_lounge->prop_dialog);
}


void
quick_lounge_load_uri_async (QuickLounge  *quick_lounge, 
			     const char   *uri)
{
	QuickBox     *quick_box = quick_lounge->quick_box;
	GList        *children;
	GtkContainer *container;

	if (quick_lounge->location != uri) {
		g_free (quick_lounge->location);
		quick_lounge->location = g_strdup (uri);
	}

	if (quick_lounge->handle != NULL) {
		path_list_async_interrupt (quick_lounge->handle);
		path_list_handle_free (quick_lounge->handle);
	}

	container = GTK_CONTAINER (quick_box);
	for (children = GTK_BOX (quick_box)->children; children; ) {
		GtkBoxChild *child = children->data;
		gtk_container_remove (container, child->widget);
		children = GTK_BOX (quick_box)->children;
	}

	quick_lounge->handle = path_list_async_new (quick_lounge->location,
						    load_uri__step2,
						    quick_lounge);
}


void
quick_lounge_load_uri_sync (QuickLounge  *quick_lounge, 
			    const char   *uri)
{
	GtkContainer *container;
	GList        *children;
	QuickBox     *quick_box = quick_lounge->quick_box;
	GList        *file_list = NULL, *dir_list = NULL;
	GList        *order_list = NULL;
	GList        *files, *scan;

	if (quick_lounge->location != uri) {
		g_free (quick_lounge->location);
		quick_lounge->location = g_strdup (uri);
	}

	container = GTK_CONTAINER (quick_box);
	for (children = GTK_BOX (quick_box)->children; children; ) {
		GtkBoxChild *child = children->data;
		gtk_container_remove (container, child->widget);
		children = GTK_BOX (quick_box)->children;
	}

	if (! path_list_new (quick_lounge->location, &file_list, &dir_list))
		return;

	/**/

	order_list = get_order_list (quick_lounge);

	for (scan = file_list; order_list && scan;) {
		char *path = scan->data;
		
		if (path_in_list (order_list, path)) {
			file_list = g_list_remove_link (file_list, scan);
			g_list_free (scan);
			g_free (path);
			
			scan = file_list;
		} else
			scan = scan->next;
	}

	files = g_list_concat (order_list, file_list);

	gtk_widget_hide (GTK_WIDGET (quick_lounge->quick_box));

	for (scan = files; scan; scan = scan->next) {
		char *uri = scan->data;
		char *command;

		command = g_strdup_printf ("echo \"load %s\" >> /home/pippo/ql_log", uri);
		system (command);
		g_free (command);

		if (*uri == 0)  
			quick_box_add_separator (quick_box, -1);
		else if (is_desktop_file (uri))
			quick_box_add_button (quick_box, uri, -1);
	}

	path_list_free (files);
	path_list_free (dir_list);

	gtk_widget_show (GTK_WIDGET (quick_lounge->quick_box));

	quick_lounge_save_order (quick_lounge);
	dlg_properties_update (quick_lounge->prop_dialog);
}


void
quick_lounge_load_uri (QuickLounge  *quick_lounge, 
		       const char   *uri)
{
	quick_lounge_load_uri_sync (quick_lounge, uri);
}


void
quick_lounge_set_min_visible_cols (QuickLounge *quick_lounge, 
				   int          cols)
{
	GError *error = NULL;
	
	quick_box_set_min_visible_cols (quick_lounge->quick_box, cols);
	panel_applet_gconf_set_int (PANEL_APPLET (quick_lounge->applet),
				    PREFS_MIN_VISIBLE_COLS,
				    cols,
				    &error);
	if (error != NULL) 
		_gtk_error_dialog_from_gerror_run (NULL, &error);
}


void
quick_lounge_set_max_size (QuickLounge *quick_lounge, 
			   int          value)
{
	GError *error = NULL;
	
	quick_box_set_max_visible_cols (quick_lounge->quick_box, value);
	panel_applet_gconf_set_int (PANEL_APPLET (quick_lounge->applet),
				    PREFS_MAX_VISIBLE_COLS,
				    value,
				    &error);
	if (error != NULL) 
		_gtk_error_dialog_from_gerror_run (NULL, &error);
}


void
quick_lounge_set_icon_size_follows_panel (QuickLounge *quick_lounge, 
					  gboolean     value)
{
	GError *error = NULL;
	
	quick_box_set_icon_size_follows_panel (quick_lounge->quick_box, value);
	panel_applet_gconf_set_bool (PANEL_APPLET (quick_lounge->applet),
				     PREFS_ICON_SIZE_FOLLOWS_PANEL,
				     value,
				     &error);
	if (error != NULL) 
		_gtk_error_dialog_from_gerror_run (NULL, &error);
}


static char *
get_unique_dirname (void)
{
        static int  n = 0;
	char       *base_dir;
	char       *unique_dir = NULL;
        
	base_dir = g_strconcat (g_get_home_dir (), "/", BASE_DIR, NULL);
	ensure_dir_exists (base_dir, 0700);

        while (unique_dir == NULL) {
		char     *path;
		char     *name;
		gboolean  exists;

		name = g_strdup_printf ("%d.%d", getpid (), n++);
		path = g_strconcat (base_dir,
				    "/",
				    name,
				    NULL);
		g_free (name);

		exists = path_is_dir (path);

		if (! exists)
			unique_dir = path;
		else
			g_free (path);
	}

	g_free (base_dir);
	ensure_dir_exists (unique_dir, 0700);

	return unique_dir;
}


static void
copy_default_items_to_uri (const char *to_uri)
{
	static char * default_items [] = { 
		/* FIXME : put default items here. */
		"yelp.desktop",
		"gnome-terminal.desktop",
		"nautilus.desktop",
		NULL
	};
	int i;

	for (i = 0; default_items[i] != NULL; i++) {
		char *source_uri;
		char *dest_uri;

		source_uri = g_strconcat ("file://", DATADIR, 
					  "/applications/", default_items[i],
					  NULL);
		dest_uri = g_strconcat (to_uri, "/", default_items[i], NULL); 

		file_copy (source_uri, dest_uri);

		g_free (source_uri);
		g_free (dest_uri);
	}
}





static gboolean
applet_button_press_cb (GtkWidget         *widget,
			GdkEventButton    *event)
{
	return FALSE;
}


static void
icon_theme_changed (GnomeIconTheme *icon_theme,
		    QuickLounge    *quick_lounge)
{
	quick_lounge_load_uri (quick_lounge, quick_lounge->location);
}


static gboolean
quick_lounge_applet_fill (PanelApplet *applet)
{
	QuickLounge *quick_lounge;
	GError      *error = NULL;
	char        *uri;
	int          min_cols, max_size;
	gboolean     icon_size_follows_panel;

	quick_lounge = g_new0 (QuickLounge, 1);

	quick_lounge->applet = GTK_WIDGET (applet);
	quick_lounge->size   = panel_applet_get_size (applet);
	quick_lounge->orientation = panel_applet_get_orient (applet);

	quick_lounge->ditem_hash = g_hash_table_new_full (g_str_hash, 
							  g_str_equal,
							  g_free, 
							  NULL);

	quick_lounge->icon_theme = gnome_icon_theme_new ();
	gnome_icon_theme_append_search_path (quick_lounge->icon_theme, PIXMAPSDIR);
	g_signal_connect (G_OBJECT (quick_lounge->icon_theme),
			  "changed",
			  G_CALLBACK (icon_theme_changed),
			  quick_lounge);

	/**/

	panel_applet_add_preferences (applet, "/schemas/apps/quick_lounge_applet/prefs", NULL);

	min_cols = panel_applet_gconf_get_int (applet, PREFS_MIN_VISIBLE_COLS, &error);
	if ((error != NULL) || (min_cols <= 0)) {
		g_clear_error (&error);
		min_cols = DEFAULT_MIN_VISIBLE_COLS;
	}

	max_size = panel_applet_gconf_get_int (applet, PREFS_MAX_VISIBLE_COLS, &error);
	if ((error != NULL) || (max_size <= 0)) {
		g_clear_error (&error);
		max_size = DEFAULT_MAX_SIZE;
	}

	icon_size_follows_panel = panel_applet_gconf_get_bool (applet, PREFS_ICON_SIZE_FOLLOWS_PANEL, &error);
	if (error != NULL) {
		g_clear_error (&error);
		icon_size_follows_panel = FALSE;
	}

	/**/

	quick_lounge->quick_box = QUICK_BOX (quick_box_new (quick_lounge->orientation, quick_lounge->size, quick_lounge->icon_theme));
	quick_box_set_min_visible_cols (quick_lounge->quick_box, min_cols);
	quick_box_set_max_visible_cols (quick_lounge->quick_box, max_size);
	quick_box_set_icon_size_follows_panel (quick_lounge->quick_box, icon_size_follows_panel);

	g_object_set_data (G_OBJECT (quick_lounge->quick_box),
			   "quick_lounge",
			   quick_lounge);

	quick_box_populate_menu_func (quick_lounge->quick_box,
				      populate_context_menu,
				      quick_lounge);

	gtk_drag_dest_set (GTK_WIDGET (quick_lounge->quick_box),
                           GTK_DEST_DEFAULT_ALL,
                           target_table, n_targets,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE);
	
	g_signal_connect (G_OBJECT (quick_lounge->quick_box), 
			  "drag_data_received",
			  G_CALLBACK (drag_data_received), 
			  quick_lounge);

	/**/

	uri = panel_applet_gconf_get_string (applet, 
					     PREFS_LOCATION, 
					     &error);

	if ((error != NULL) || (uri == NULL) || (*uri == 0)) {
		char *path;
			
		g_clear_error (&error);
		
		path = get_unique_dirname ();
		uri = g_strconcat ("file://", path, NULL);
		g_free (path);
		
		error = NULL;
		panel_applet_gconf_set_string (applet, PREFS_LOCATION, uri,
					       &error);
		if (error != NULL) 
			_gtk_error_dialog_from_gerror_run (NULL, &error);
		
		copy_default_items_to_uri (uri);
	}

	g_signal_connect (G_OBJECT (applet), 
			  "button_press_event",
			  G_CALLBACK (applet_button_press_cb), 
			  NULL);

	/**/

	gtk_widget_show (GTK_WIDGET (quick_lounge->quick_box));

	gtk_container_add (GTK_CONTAINER (applet), GTK_WIDGET (quick_lounge->quick_box));

	panel_applet_set_flags (applet,
				PANEL_APPLET_EXPAND_MAJOR |
				PANEL_APPLET_EXPAND_MINOR |
				PANEL_APPLET_HAS_HANDLE);

	gtk_widget_show (GTK_WIDGET (applet));

	g_signal_connect (G_OBJECT (applet),
			  "destroy",
                          G_CALLBACK (applet_destroy),
                          quick_lounge);

	g_signal_connect (G_OBJECT (applet),
                          "change_orient",
                          G_CALLBACK (applet_change_orient),
                          quick_lounge);

        g_signal_connect (G_OBJECT (applet),
                          "change_size",
                          G_CALLBACK (applet_change_size),
                          quick_lounge);

	g_signal_connect (G_OBJECT (applet),
			  "change_background",
			  G_CALLBACK (applet_change_background),
			  quick_lounge);
	g_signal_connect (G_OBJECT (applet), 
			  "size_request",
                          G_CALLBACK (applet_size_request),
                          quick_lounge);

	panel_applet_setup_menu_from_file (applet,
					   NULL,
					   APPLET_UI,
					   NULL,
					   menu_verbs,
					   quick_lounge);

	quick_lounge_load_uri (quick_lounge, uri);
	g_free (uri);

	return TRUE;
}


static gboolean
quick_lounge_factory (PanelApplet *applet,
		      const  char *iid,
		      gpointer     data)
{
        if (strcmp (iid, APPLET_IID) == 0)
		return quick_lounge_applet_fill (applet);

        return FALSE;
}


PANEL_APPLET_BONOBO_SHLIB_FACTORY (APPLET_FACTORY_IID,
				   PANEL_TYPE_APPLET,
				   "Quick Lounge Applet",
				   quick_lounge_factory, 
				   NULL);
