/* valasemanticanalyzer.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valasemanticanalyzer.h"
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaattribute.h>
#include <vala/valastatement.h>
#include <vala/valacallback.h>
#include <vala/valaarray.h>
#include <vala/valalockable.h>
#include <vala/valatypeparameter.h>
#include <vala/valaliteral.h>
#include <vala/valaenumvalue.h>
#include <vala/valaflagsvalue.h>
#include <vala/valanamespacereference.h>
#include <vala/valapointer.h>
#include <vala/valaenum.h>
#include <vala/valainvokable.h>
#include <vala/valasignal.h>

struct _ValaSemanticAnalyzerPrivate {
	gboolean _memory_management;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaSourceFile* current_source_file;
	ValaTypeReference* current_return_type;
	ValaClass* current_class;
	ValaStruct* current_struct;
	GList* current_using_directives;
	ValaTypeReference* bool_type;
	ValaTypeReference* string_type;
	ValaTypeReference* int_type;
	ValaTypeReference* uint_type;
	ValaTypeReference* type_type;
	ValaDataType* pointer_type;
	ValaDataType* initially_unowned_type;
	ValaDataType* glist_type;
	ValaDataType* gslist_type;
	gint next_lambda_id;
};
#define VALA_SEMANTIC_ANALYZER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerPrivate))
enum  {
	VALA_SEMANTIC_ANALYZER_DUMMY_PROPERTY,
	VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT,
};
static void vala_semantic_analyzer_real_visit_begin_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_semantic_analyzer_real_visit_end_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_semantic_analyzer_real_visit_begin_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_semantic_analyzer_real_visit_end_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_semantic_analyzer_real_visit_begin_class (ValaCodeVisitor* base, ValaClass* cl);
static GList* vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface);
static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t);
static void vala_semantic_analyzer_real_visit_end_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_begin_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_semantic_analyzer_real_visit_end_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_semantic_analyzer_real_visit_begin_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_semantic_analyzer_real_visit_end_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_semantic_analyzer_real_visit_begin_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_end_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_semantic_analyzer_real_visit_begin_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_semantic_analyzer_real_visit_end_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_end_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_semantic_analyzer_real_visit_begin_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_semantic_analyzer_real_visit_end_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_semantic_analyzer_real_visit_begin_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_semantic_analyzer_real_visit_end_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_semantic_analyzer_real_visit_begin_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_semantic_analyzer_real_visit_end_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_semantic_analyzer_real_visit_begin_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_end_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_semantic_analyzer_real_visit_begin_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_end_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_semantic_analyzer_real_visit_begin_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_end_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static void vala_semantic_analyzer_real_visit_begin_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_end_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr);
static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr);
static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr);
static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr);
static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr);
static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr);
static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_node (ValaSemanticAnalyzer* self, ValaCodeNode* node);
static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static ValaDataType* vala_semantic_analyzer_find_parent_type (ValaSemanticAnalyzer* self, ValaSymbol* sym);
static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type);
static void vala_semantic_analyzer_real_visit_begin_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GList* params, GList* args);
static void vala_semantic_analyzer_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr);
static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr);
static void vala_semantic_analyzer_real_visit_end_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr);
static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr);
static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr);
static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr);
static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr);
static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type);
static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr);
static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GList* types);
static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr);
static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self);
static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self);
static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self);
static void vala_semantic_analyzer_real_visit_begin_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_semantic_analyzer_real_visit_begin_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static void vala_semantic_analyzer_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_semantic_analyzer_parent_class = NULL;
static void vala_semantic_analyzer_dispose (GObject * obj);


ValaSemanticAnalyzer* vala_semantic_analyzer_new (gboolean manage_memory)
{
	GParameter * __params = g_new0 (GParameter, 1);
	GParameter * __params_it = __params;
	(__params_it->name = "memory-management", g_value_init (&__params_it->value, G_TYPE_BOOLEAN), g_value_set_boolean (&__params_it->value, manage_memory), __params_it++);
	ValaSemanticAnalyzer * self = g_object_newv (VALA_TYPE_SEMANTIC_ANALYZER, __params_it - __params, __params);
	return self;
}


/**
 * Analyze and check code in the specified context.
 *
 * @param context a code context
 */
void vala_semantic_analyzer_analyze (ValaSemanticAnalyzer* self, ValaCodeContext* context)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	ValaSymbol* __temp0 = NULL;
	self->priv->root_symbol = (__temp0 = vala_code_context_get_root (context), (self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL))), __temp0);
	ValaTypeReference* __temp1 = NULL;
	self->priv->bool_type = (__temp1 = vala_type_reference_new (), (self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL))), __temp1);
	ValaSymbol* __temp2 = NULL;
	(vala_type_reference_set_data_type (self->priv->bool_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp2 = vala_symbol_lookup (self->priv->root_symbol, "bool"))))), vala_type_reference_get_data_type (self->priv->bool_type));
	(__temp2 == NULL ? NULL : (__temp2 = (g_object_unref (__temp2), NULL)));
	ValaTypeReference* __temp3 = NULL;
	self->priv->string_type = (__temp3 = vala_type_reference_new (), (self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL))), __temp3);
	ValaSymbol* __temp4 = NULL;
	(vala_type_reference_set_data_type (self->priv->string_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp4 = vala_symbol_lookup (self->priv->root_symbol, "string"))))), vala_type_reference_get_data_type (self->priv->string_type));
	(__temp4 == NULL ? NULL : (__temp4 = (g_object_unref (__temp4), NULL)));
	ValaDataType* __temp7 = NULL;
	ValaDataType* __temp6 = NULL;
	ValaSymbol* __temp5 = NULL;
	self->priv->pointer_type = (__temp7 = (__temp6 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp5 = vala_symbol_lookup (self->priv->root_symbol, "pointer")))), (__temp6 == NULL ? NULL : g_object_ref (__temp6))), (self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL))), __temp7);
	(__temp5 == NULL ? NULL : (__temp5 = (g_object_unref (__temp5), NULL)));
	ValaTypeReference* __temp8 = NULL;
	self->priv->int_type = (__temp8 = vala_type_reference_new (), (self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL))), __temp8);
	ValaSymbol* __temp9 = NULL;
	(vala_type_reference_set_data_type (self->priv->int_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp9 = vala_symbol_lookup (self->priv->root_symbol, "int"))))), vala_type_reference_get_data_type (self->priv->int_type));
	(__temp9 == NULL ? NULL : (__temp9 = (g_object_unref (__temp9), NULL)));
	ValaTypeReference* __temp10 = NULL;
	self->priv->uint_type = (__temp10 = vala_type_reference_new (), (self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL))), __temp10);
	ValaSymbol* __temp11 = NULL;
	(vala_type_reference_set_data_type (self->priv->uint_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp11 = vala_symbol_lookup (self->priv->root_symbol, "uint"))))), vala_type_reference_get_data_type (self->priv->uint_type));
	(__temp11 == NULL ? NULL : (__temp11 = (g_object_unref (__temp11), NULL)));
	/* TODO: don't require GLib namespace in semantic analyzer*/
	ValaSymbol* glib_ns = vala_symbol_lookup (self->priv->root_symbol, "GLib");
	if (glib_ns != NULL) {
		ValaDataType* __temp14 = NULL;
		ValaDataType* __temp13 = NULL;
		ValaSymbol* __temp12 = NULL;
		self->priv->initially_unowned_type = (__temp14 = (__temp13 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp12 = vala_symbol_lookup (glib_ns, "InitiallyUnowned")))), (__temp13 == NULL ? NULL : g_object_ref (__temp13))), (self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL))), __temp14);
		(__temp12 == NULL ? NULL : (__temp12 = (g_object_unref (__temp12), NULL)));
		ValaTypeReference* __temp15 = NULL;
		self->priv->type_type = (__temp15 = vala_type_reference_new (), (self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL))), __temp15);
		ValaSymbol* __temp16 = NULL;
		(vala_type_reference_set_data_type (self->priv->type_type, VALA_DATA_TYPE (vala_symbol_get_node ((__temp16 = vala_symbol_lookup (glib_ns, "Type"))))), vala_type_reference_get_data_type (self->priv->type_type));
		(__temp16 == NULL ? NULL : (__temp16 = (g_object_unref (__temp16), NULL)));
		ValaDataType* __temp19 = NULL;
		ValaDataType* __temp18 = NULL;
		ValaSymbol* __temp17 = NULL;
		self->priv->glist_type = (__temp19 = (__temp18 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp17 = vala_symbol_lookup (glib_ns, "List")))), (__temp18 == NULL ? NULL : g_object_ref (__temp18))), (self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL))), __temp19);
		(__temp17 == NULL ? NULL : (__temp17 = (g_object_unref (__temp17), NULL)));
		ValaDataType* __temp22 = NULL;
		ValaDataType* __temp21 = NULL;
		ValaSymbol* __temp20 = NULL;
		self->priv->gslist_type = (__temp22 = (__temp21 = VALA_DATA_TYPE (vala_symbol_get_node ((__temp20 = vala_symbol_lookup (glib_ns, "SList")))), (__temp21 == NULL ? NULL : g_object_ref (__temp21))), (self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL))), __temp22);
		(__temp20 == NULL ? NULL : (__temp20 = (g_object_unref (__temp20), NULL)));
	}
	ValaSymbol* __temp24 = NULL;
	ValaSymbol* __temp23 = NULL;
	self->priv->current_symbol = (__temp24 = (__temp23 = self->priv->root_symbol, (__temp23 == NULL ? NULL : g_object_ref (__temp23))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp24);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	(glib_ns == NULL ? NULL : (glib_ns = (g_object_unref (glib_ns), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_source_file (ValaCodeVisitor* base, ValaSourceFile* file)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	ValaSourceFile* __temp25 = NULL;
	self->priv->current_source_file = (__temp25 = g_object_ref (file), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), __temp25);
	GList* __temp26 = NULL;
	self->priv->current_using_directives = (__temp26 = vala_source_file_get_using_directives (file), (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL))), __temp26);
	self->priv->next_lambda_id = 0;
}


static void vala_semantic_analyzer_real_visit_end_source_file (ValaCodeVisitor* base, ValaSourceFile* file)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	GList* __temp27 = NULL;
	self->priv->current_using_directives = (__temp27 = NULL, (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL))), __temp27);
}


static void vala_semantic_analyzer_real_visit_begin_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	ValaSymbol* __temp29 = NULL;
	ValaSymbol* __temp28 = NULL;
	self->priv->current_symbol = (__temp29 = (__temp28 = vala_code_node_get_symbol (VALA_CODE_NODE (ns)), (__temp28 == NULL ? NULL : g_object_ref (__temp28))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp29);
}


static void vala_semantic_analyzer_real_visit_end_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	ValaSymbol* __temp31 = NULL;
	ValaSymbol* __temp30 = NULL;
	self->priv->current_symbol = (__temp31 = (__temp30 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp30 == NULL ? NULL : g_object_ref (__temp30))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp31);
}


static void vala_semantic_analyzer_real_visit_begin_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	ValaSymbol* __temp33 = NULL;
	ValaSymbol* __temp32 = NULL;
	self->priv->current_symbol = (__temp33 = (__temp32 = vala_code_node_get_symbol (VALA_CODE_NODE (cl)), (__temp32 == NULL ? NULL : g_object_ref (__temp32))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp33);
	ValaClass* __temp34 = NULL;
	self->priv->current_class = (__temp34 = g_object_ref (cl), (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), __temp34);
	if (vala_class_get_base_class (cl) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_class_get_base_class (cl))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
	}
	{
		GList* __temp35 = NULL;
		__temp35 = vala_class_get_base_types (cl);
		GList* base_type_reference_it;
		for (base_type_reference_it = __temp35; base_type_reference_it != NULL; base_type_reference_it = base_type_reference_it->next) {
			ValaTypeReference* base_type_reference = base_type_reference_it->data;
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type_reference))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			}
		}
		(__temp35 == NULL ? NULL : (__temp35 = (g_list_free (__temp35), NULL)));
	}
}


static GList* vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_INTERFACE (iface), NULL);
	GList* ret = NULL;
	{
		GList* __temp38 = NULL;
		__temp38 = vala_interface_get_prerequisites (iface);
		GList* prereq_it;
		for (prereq_it = __temp38; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaTypeReference* prereq = prereq_it->data;
			{
				ValaDataType* __temp36 = NULL;
				ValaDataType* type = (__temp36 = vala_type_reference_get_data_type (prereq), (__temp36 == NULL ? NULL : g_object_ref (__temp36)));
				/* skip on previous errors */
				if (type == NULL) {
					continue;
				}
				ValaDataType* __temp37 = NULL;
				ret = g_list_prepend (ret, (__temp37 = type, (__temp37 == NULL ? NULL : g_object_ref (__temp37))));
				if (VALA_IS_INTERFACE (type)) {
					ret = g_list_concat (ret, vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (type)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(__temp38 == NULL ? NULL : (__temp38 = (g_list_free (__temp38), NULL)));
	}
	ret = g_list_reverse (ret);
	GList* __temp39 = NULL;
	GList* __temp40 = NULL;
	return (__temp40 = (__temp39 = ret, ret = NULL, __temp39), (ret == NULL ? NULL : (ret = (g_list_foreach (ret, (GFunc) g_object_unref, NULL), g_list_free (ret), NULL))), __temp40);
	(ret == NULL ? NULL : (ret = (g_list_foreach (ret, (GFunc) g_object_unref, NULL), g_list_free (ret), NULL)));
}


static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_CLASS (cl), FALSE);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	if (VALA_DATA_TYPE (cl) == t) {
		return TRUE;
	}
	{
		GList* __temp44 = NULL;
		__temp44 = vala_class_get_base_types (cl);
		GList* base_type_it;
		for (base_type_it = __temp44; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type = base_type_it->data;
			{
				if (VALA_IS_CLASS (vala_type_reference_get_data_type (base_type))) {
					if (vala_semantic_analyzer_class_is_a (self, VALA_CLASS (vala_type_reference_get_data_type (base_type)), t)) {
						return TRUE;
					}
				} else {
					if (vala_type_reference_get_data_type (base_type) == t) {
						return TRUE;
					}
				}
			}
		}
		(__temp44 == NULL ? NULL : (__temp44 = (g_list_free (__temp44), NULL)));
	}
	return FALSE;
}


static void vala_semantic_analyzer_real_visit_end_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	/* gather all prerequisites */
	GList* prerequisites = NULL;
	{
		GList* __temp46 = NULL;
		__temp46 = vala_class_get_base_types (cl);
		GList* base_type_it;
		for (base_type_it = __temp46; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type = base_type_it->data;
			{
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					prerequisites = g_list_concat (prerequisites, vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (vala_type_reference_get_data_type (base_type))));
				}
			}
		}
		(__temp46 == NULL ? NULL : (__temp46 = (g_list_free (__temp46), NULL)));
	}
	/* check whether all prerequisites are met */
	GList* missing_prereqs = NULL;
	{
		GList* __temp47 = NULL;
		__temp47 = prerequisites;
		GList* prereq_it;
		for (prereq_it = __temp47; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaDataType* prereq = prereq_it->data;
			{
				if (!vala_semantic_analyzer_class_is_a (self, cl, prereq)) {
					missing_prereqs = g_list_prepend (missing_prereqs, vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prereq))));
				}
			}
		}
	}
	/* report any missing prerequisites */
	if (missing_prereqs != NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
		char* __temp48 = NULL;
		char* __temp49 = NULL;
		char* error_string = (__temp49 = g_strdup_printf ("%s: some prerequisites (", (__temp48 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl))))), (__temp48 = (g_free (__temp48), NULL)), __temp49);
		gboolean first = TRUE;
		{
			GList* __temp52 = NULL;
			__temp52 = missing_prereqs;
			GList* s_it;
			for (s_it = __temp52; s_it != NULL; s_it = s_it->next) {
				char* s = s_it->data;
				{
					if (first) {
						char* __temp50 = NULL;
						error_string = (__temp50 = g_strdup_printf ("%s`%s'", error_string, s), (error_string = (g_free (error_string), NULL)), __temp50);
						first = FALSE;
					} else {
						char* __temp51 = NULL;
						error_string = (__temp51 = g_strdup_printf ("%s, `%s'", error_string, s), (error_string = (g_free (error_string), NULL)), __temp51);
					}
				}
			}
		}
		char* __temp53 = NULL;
		error_string = (__temp53 = g_strconcat (error_string, (") are not met"), NULL), (error_string = (g_free (error_string), NULL)), __temp53);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), error_string);
		(error_string = (g_free (error_string), NULL));
	}
	{
		GList* __temp59 = NULL;
		__temp59 = vala_class_get_base_types (cl);
		GList* base_type_it;
		for (base_type_it = __temp59; base_type_it != NULL; base_type_it = base_type_it->next) {
			ValaTypeReference* base_type = base_type_it->data;
			{
				/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					ValaInterface* __temp54 = NULL;
					ValaInterface* iface = (__temp54 = VALA_INTERFACE (vala_type_reference_get_data_type (base_type)), (__temp54 == NULL ? NULL : g_object_ref (__temp54)));
					{
						GList* __temp58 = NULL;
						__temp58 = vala_interface_get_methods (iface);
						GList* m_it;
						for (m_it = __temp58; m_it != NULL; m_it = m_it->next) {
							ValaMethod* m = m_it->data;
							{
								/* We do not need to do expensive equality checking here since this is done
								 * already. We only need to guarantee the symbols are present.
								 
								 check methods */
								if (vala_method_get_is_abstract (m)) {
									ValaSymbol* sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
									if (sym == NULL || !(VALA_IS_METHOD (vala_symbol_get_node (sym))) || vala_method_get_base_interface_method ((VALA_METHOD (vala_symbol_get_node (sym)))) != m) {
										(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
										char* __temp57 = NULL;
										char* __temp56 = NULL;
										char* __temp55 = NULL;
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (__temp57 = g_strdup_printf ("`%s' does not implement interface method `%s'", (__temp55 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))), (__temp56 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
										(__temp57 = (g_free (__temp57), NULL));
										(__temp56 = (g_free (__temp56), NULL));
										(__temp55 = (g_free (__temp55), NULL));
									}
									(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								}
							}
						}
						(__temp58 == NULL ? NULL : (__temp58 = (g_list_free (__temp58), NULL)));
					}
					(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
				}
			}
		}
		(__temp59 == NULL ? NULL : (__temp59 = (g_list_free (__temp59), NULL)));
	}
	/* all abstract symbols defined in base classes have to be implemented in non-abstract classes
	 * VAPI classes don't have to specify overridden methods
	 */
	if (!vala_class_get_is_abstract (cl) && !vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl))))) {
		ValaClass* __temp60 = NULL;
		ValaClass* base_class = (__temp60 = vala_class_get_base_class (cl), (__temp60 == NULL ? NULL : g_object_ref (__temp60)));
		while (base_class != NULL && vala_class_get_is_abstract (base_class)) {
			{
				GList* __temp64 = NULL;
				__temp64 = vala_class_get_methods (base_class);
				GList* m_it;
				for (m_it = __temp64; m_it != NULL; m_it = m_it->next) {
					ValaMethod* m = m_it->data;
					{
						if (vala_method_get_is_abstract (m)) {
							ValaSymbol* sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
							if (sym == NULL || !(VALA_IS_METHOD (vala_symbol_get_node (sym))) || vala_method_get_base_method ((VALA_METHOD (vala_symbol_get_node (sym)))) != m) {
								(vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (cl)));
								char* __temp63 = NULL;
								char* __temp62 = NULL;
								char* __temp61 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (__temp63 = g_strdup_printf ("`%s' does not implement abstract method `%s'", (__temp61 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))), (__temp62 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
								(__temp63 = (g_free (__temp63), NULL));
								(__temp62 = (g_free (__temp62), NULL));
								(__temp61 = (g_free (__temp61), NULL));
							}
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
						}
					}
				}
				(__temp64 == NULL ? NULL : (__temp64 = (g_list_free (__temp64), NULL)));
			}
			ValaClass* __temp66 = NULL;
			ValaClass* __temp65 = NULL;
			base_class = (__temp66 = (__temp65 = vala_class_get_base_class (base_class), (__temp65 == NULL ? NULL : g_object_ref (__temp65))), (base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL))), __temp66);
		}
		(base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL)));
	}
	ValaSymbol* __temp68 = NULL;
	ValaSymbol* __temp67 = NULL;
	self->priv->current_symbol = (__temp68 = (__temp67 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp67 == NULL ? NULL : g_object_ref (__temp67))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp68);
	ValaClass* __temp69 = NULL;
	self->priv->current_class = (__temp69 = NULL, (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), __temp69);
	(prerequisites == NULL ? NULL : (prerequisites = (g_list_foreach (prerequisites, (GFunc) g_object_unref, NULL), g_list_free (prerequisites), NULL)));
	(missing_prereqs == NULL ? NULL : (missing_prereqs = (g_list_foreach (missing_prereqs, (GFunc) g_free, NULL), g_list_free (missing_prereqs), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	ValaSymbol* __temp71 = NULL;
	ValaSymbol* __temp70 = NULL;
	self->priv->current_symbol = (__temp71 = (__temp70 = vala_code_node_get_symbol (VALA_CODE_NODE (st)), (__temp70 == NULL ? NULL : g_object_ref (__temp70))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp71);
	ValaStruct* __temp72 = NULL;
	self->priv->current_struct = (__temp72 = g_object_ref (st), (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), __temp72);
}


static void vala_semantic_analyzer_real_visit_end_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	ValaSymbol* __temp74 = NULL;
	ValaSymbol* __temp73 = NULL;
	self->priv->current_symbol = (__temp74 = (__temp73 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp73 == NULL ? NULL : g_object_ref (__temp73))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp74);
	ValaStruct* __temp75 = NULL;
	self->priv->current_struct = (__temp75 = NULL, (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), __temp75);
}


static void vala_semantic_analyzer_real_visit_begin_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	ValaSymbol* __temp77 = NULL;
	ValaSymbol* __temp76 = NULL;
	self->priv->current_symbol = (__temp77 = (__temp76 = vala_code_node_get_symbol (VALA_CODE_NODE (iface)), (__temp76 == NULL ? NULL : g_object_ref (__temp76))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp77);
	{
		GList* __temp78 = NULL;
		__temp78 = vala_interface_get_prerequisites (iface);
		GList* prerequisite_reference_it;
		for (prerequisite_reference_it = __temp78; prerequisite_reference_it != NULL; prerequisite_reference_it = prerequisite_reference_it->next) {
			ValaTypeReference* prerequisite_reference = prerequisite_reference_it->data;
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (prerequisite_reference))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
			}
		}
		(__temp78 == NULL ? NULL : (__temp78 = (g_list_free (__temp78), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_end_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	/* check prerequisites */
	ValaClass* prereq_class = NULL;
	{
		GList* __temp86 = NULL;
		__temp86 = vala_interface_get_prerequisites (iface);
		GList* prereq_it;
		for (prereq_it = __temp86; prereq_it != NULL; prereq_it = prereq_it->next) {
			ValaTypeReference* prereq = prereq_it->data;
			{
				ValaDataType* __temp79 = NULL;
				ValaDataType* class_or_interface = (__temp79 = vala_type_reference_get_data_type (prereq), (__temp79 == NULL ? NULL : g_object_ref (__temp79)));
				/* skip on previous errors */
				if (class_or_interface == NULL) {
					(vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE), vala_code_node_get_error (VALA_CODE_NODE (iface)));
					continue;
				}
				/* interfaces are not allowed to have multiple instantiable prerequisites */
				if (VALA_IS_CLASS (class_or_interface)) {
					if (prereq_class != NULL) {
						(vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE), vala_code_node_get_error (VALA_CODE_NODE (iface)));
						char* __temp83 = NULL;
						char* __temp82 = NULL;
						char* __temp81 = NULL;
						char* __temp80 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (iface)), (__temp83 = g_strdup_printf ("%s: Interfaces cannot have multiple instantiable prerequisites (`%s' and `%s')", (__temp80 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (iface)))), (__temp81 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (class_or_interface)))), (__temp82 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prereq_class)))))));
						(__temp83 = (g_free (__temp83), NULL));
						(__temp82 = (g_free (__temp82), NULL));
						(__temp81 = (g_free (__temp81), NULL));
						(__temp80 = (g_free (__temp80), NULL));
						(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
						(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
						return;
					}
					ValaClass* __temp85 = NULL;
					ValaClass* __temp84 = NULL;
					prereq_class = (__temp85 = (__temp84 = VALA_CLASS (class_or_interface), (__temp84 == NULL ? NULL : g_object_ref (__temp84))), (prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL))), __temp85);
				}
				(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
			}
		}
		(__temp86 == NULL ? NULL : (__temp86 = (g_list_free (__temp86), NULL)));
	}
	ValaSymbol* __temp88 = NULL;
	ValaSymbol* __temp87 = NULL;
	self->priv->current_symbol = (__temp88 = (__temp87 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp87 == NULL ? NULL : g_object_ref (__temp87))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp88);
	(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
}


static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	if (!vala_source_file_get_pkg (self->priv->current_source_file)) {
		if (vala_constant_get_initializer (c) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (c), TRUE), vala_code_node_get_error (VALA_CODE_NODE (c)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (c)), "A const field requires a initializer to be provided");
		}
	}
}


static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	if (f->access != VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
	} else {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
}


static void vala_semantic_analyzer_real_visit_begin_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	ValaSymbol* __temp90 = NULL;
	ValaSymbol* __temp89 = NULL;
	self->priv->current_symbol = (__temp90 = (__temp89 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp89 == NULL ? NULL : g_object_ref (__temp89))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp90);
	ValaTypeReference* __temp92 = NULL;
	ValaTypeReference* __temp91 = NULL;
	self->priv->current_return_type = (__temp92 = (__temp91 = vala_method_get_return_type (m), (__temp91 == NULL ? NULL : g_object_ref (__temp91))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp92);
	ValaAttribute* init_attr = vala_code_node_get_attribute (VALA_CODE_NODE (m), "ModuleInit");
	if (init_attr != NULL) {
		(vala_code_context_set_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m)))), m), vala_code_context_get_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m))))));
	}
	if (vala_type_reference_get_data_type (vala_method_get_return_type (m)) != NULL) {
		/* is null if it is void or a reference to a type parameter */
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_method_get_return_type (m)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
	}
	(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
}


static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	ValaSymbol* sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_method_get_name (m));
	if (sym != NULL && VALA_IS_METHOD (vala_symbol_get_node (sym))) {
		ValaMethod* __temp93 = NULL;
		ValaMethod* base_method = (__temp93 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp93 == NULL ? NULL : g_object_ref (__temp93)));
		if (vala_method_get_is_abstract (base_method) || vala_method_get_is_virtual (base_method)) {
			if (!vala_method_equals (m, base_method)) {
				(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
				char* __temp96 = NULL;
				char* __temp95 = NULL;
				char* __temp94 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp96 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (__temp94 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp95 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_method)))))));
				(__temp96 = (g_free (__temp96), NULL));
				(__temp95 = (g_free (__temp95), NULL));
				(__temp94 = (g_free (__temp94), NULL));
				(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			(vala_method_set_base_method (m, base_method), vala_method_get_base_method (m));
			(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_method (self, m, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GList* __temp101 = NULL;
		__temp101 = vala_class_get_base_types (cl);
		GList* type_it;
		for (type_it = __temp101; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type = type_it->data;
			{
				/* FIXME report error if multiple possible base methods are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type))), vala_method_get_name (m));
					if (sym != NULL && VALA_IS_METHOD (vala_symbol_get_node (sym))) {
						ValaMethod* __temp97 = NULL;
						ValaMethod* base_method = (__temp97 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp97 == NULL ? NULL : g_object_ref (__temp97)));
						if (vala_method_get_is_abstract (base_method)) {
							if (!vala_method_equals (m, base_method)) {
								(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
								char* __temp100 = NULL;
								char* __temp99 = NULL;
								char* __temp98 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp100 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (__temp98 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp99 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_method)))))));
								(__temp100 = (g_free (__temp100), NULL));
								(__temp99 = (g_free (__temp99), NULL));
								(__temp98 = (g_free (__temp98), NULL));
								(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								return;
							}
							(vala_method_set_base_interface_method (m, base_method), vala_method_get_base_interface_method (m));
							(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							return;
						}
						(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
			}
		}
		(__temp101 == NULL ? NULL : (__temp101 = (g_list_free (__temp101), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_end_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	ValaSymbol* __temp103 = NULL;
	ValaSymbol* __temp102 = NULL;
	self->priv->current_symbol = (__temp103 = (__temp102 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp102 == NULL ? NULL : g_object_ref (__temp102))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp103);
	ValaTypeReference* __temp104 = NULL;
	self->priv->current_return_type = (__temp104 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp104);
	if (vala_symbol_get_parent_symbol (self->priv->current_symbol) != NULL && VALA_IS_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol)))) {
		/* lambda expressions produce nested methods */
		ValaMethod* __temp105 = NULL;
		ValaMethod* up_method = (__temp105 = VALA_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (self->priv->current_symbol))), (__temp105 == NULL ? NULL : g_object_ref (__temp105)));
		ValaTypeReference* __temp107 = NULL;
		ValaTypeReference* __temp106 = NULL;
		self->priv->current_return_type = (__temp107 = (__temp106 = vala_method_get_return_type (up_method), (__temp106 == NULL ? NULL : g_object_ref (__temp106))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp107);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (self->priv->current_symbol))) {
		if (!(VALA_IS_CREATION_METHOD (m))) {
			vala_semantic_analyzer_find_base_interface_method (self, m, VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
			if (vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				vala_semantic_analyzer_find_base_class_method (self, m, VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)));
				if (vala_method_get_base_method (m) == NULL) {
					char* __temp109 = NULL;
					char* __temp108 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp109 = g_strdup_printf ("%s: no suitable method found to override", (__temp108 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
					(__temp109 = (g_free (__temp109), NULL));
					(__temp108 = (g_free (__temp108), NULL));
				}
			}
		}
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_node (self->priv->current_symbol))) {
			if (vala_method_get_is_abstract (m) || vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				char* __temp111 = NULL;
				char* __temp110 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (__temp111 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", (__temp110 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))))));
				(__temp111 = (g_free (__temp111), NULL));
				(__temp110 = (g_free (__temp110), NULL));
				return;
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_begin_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	ValaTypeReference* __temp112 = NULL;
	(vala_method_set_return_type (VALA_METHOD (m), (__temp112 = vala_type_reference_new ())), vala_method_get_return_type (VALA_METHOD (m)));
	(__temp112 == NULL ? NULL : (__temp112 = (g_object_unref (__temp112), NULL)));
	(vala_type_reference_set_data_type (vala_method_get_return_type (VALA_METHOD (m)), VALA_DATA_TYPE (vala_symbol_get_node (self->priv->current_symbol))), vala_type_reference_get_data_type (vala_method_get_return_type (VALA_METHOD (m))));
	(vala_type_reference_set_transfers_ownership (vala_method_get_return_type (VALA_METHOD (m)), TRUE), vala_type_reference_get_transfers_ownership (vala_method_get_return_type (VALA_METHOD (m))));
	if (VALA_IS_CLASS (vala_symbol_get_node (self->priv->current_symbol))) {
		/* check for floating reference*/
		ValaClass* __temp113 = NULL;
		ValaClass* cl = (__temp113 = VALA_CLASS (vala_symbol_get_node (self->priv->current_symbol)), (__temp113 == NULL ? NULL : g_object_ref (__temp113)));
		while (cl != NULL) {
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				(vala_type_reference_set_floating_reference (vala_method_get_return_type (VALA_METHOD (m)), TRUE), vala_type_reference_get_floating_reference (vala_method_get_return_type (VALA_METHOD (m))));
				break;
			}
			ValaClass* __temp115 = NULL;
			ValaClass* __temp114 = NULL;
			cl = (__temp115 = (__temp114 = vala_class_get_base_class (cl), (__temp114 == NULL ? NULL : g_object_ref (__temp114))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp115);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
	ValaSymbol* __temp117 = NULL;
	ValaSymbol* __temp116 = NULL;
	self->priv->current_symbol = (__temp117 = (__temp116 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp116 == NULL ? NULL : g_object_ref (__temp116))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp117);
	ValaTypeReference* __temp119 = NULL;
	ValaTypeReference* __temp118 = NULL;
	self->priv->current_return_type = (__temp119 = (__temp118 = vala_method_get_return_type (VALA_METHOD (m)), (__temp118 == NULL ? NULL : g_object_ref (__temp118))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp119);
}


static void vala_semantic_analyzer_real_visit_end_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	vala_code_visitor_visit_end_method (VALA_CODE_VISITOR (self), VALA_METHOD (m));
	if (vala_method_get_body (VALA_METHOD (m)) != NULL && self->priv->current_class != NULL) {
		gint n_params = 0;
		{
			GList* __temp120 = NULL;
			__temp120 = vala_block_get_statements (vala_method_get_body (VALA_METHOD (m)));
			GList* stmt_it;
			for (stmt_it = __temp120; stmt_it != NULL; stmt_it = stmt_it->next) {
				ValaStatement* stmt = stmt_it->data;
				{
					gint params = vala_statement_get_number_of_set_construction_parameters (stmt);
					if (params == -1) {
						(vala_code_node_set_error (VALA_CODE_NODE (m), TRUE), vala_code_node_get_error (VALA_CODE_NODE (m)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "class creation methods only allow property assignment statements");
						return;
					}
					n_params = n_params + (params);
				}
			}
			(__temp120 == NULL ? NULL : (__temp120 = (g_list_free (__temp120), NULL)));
		}
		(vala_creation_method_set_n_construction_params (m, n_params), vala_creation_method_get_n_construction_params (m));
	}
}


static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (p));
	if (!vala_formal_parameter_get_ellipsis (p)) {
		if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
	/* special treatment for construct formal parameters used in creation methods */
	if (vala_formal_parameter_get_construct_parameter (p)) {
		if (!(VALA_IS_CREATION_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (p))))))) {
			(vala_code_node_set_error (VALA_CODE_NODE (p), TRUE), vala_code_node_get_error (VALA_CODE_NODE (p)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), "construct parameters are only allowed in type creation methods");
			return;
		}
		ValaBlock* __temp121 = NULL;
		ValaBlock* method_body = (__temp121 = vala_method_get_body (VALA_METHOD ((VALA_CREATION_METHOD (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (p)))))))), (__temp121 == NULL ? NULL : g_object_ref (__temp121)));
		ValaMemberAccess* left = vala_member_access_new_simple (vala_formal_parameter_get_name (p), NULL);
		ValaMemberAccess* right = vala_member_access_new_simple (vala_formal_parameter_get_name (p), NULL);
		/* try to lookup the requested property */
		ValaSymbol* prop_sym = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (self->priv->current_class)), vala_formal_parameter_get_name (p));
		if (!(VALA_IS_PROPERTY (vala_symbol_get_node (prop_sym)))) {
			(vala_code_node_set_error (VALA_CODE_NODE (p), TRUE), vala_code_node_get_error (VALA_CODE_NODE (p)));
			char* __temp123 = NULL;
			char* __temp122 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), (__temp123 = g_strdup_printf ("class `%s' does not contain a property named `%s'", (__temp122 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (self->priv->current_class)))), vala_formal_parameter_get_name (p))));
			(__temp123 = (g_free (__temp123), NULL));
			(__temp122 = (g_free (__temp122), NULL));
			(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
			(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
			(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
			(prop_sym == NULL ? NULL : (prop_sym = (g_object_unref (prop_sym), NULL)));
			return;
		}
		(vala_expression_set_symbol_reference (VALA_EXPRESSION (left), prop_sym), vala_expression_get_symbol_reference (VALA_EXPRESSION (left)));
		(vala_expression_set_symbol_reference (VALA_EXPRESSION (right), vala_code_node_get_symbol (VALA_CODE_NODE (p))), vala_expression_get_symbol_reference (VALA_EXPRESSION (right)));
		ValaExpressionStatement* __temp124 = NULL;
		vala_block_add_statement (method_body, VALA_STATEMENT ((__temp124 = vala_expression_statement_new (VALA_EXPRESSION (vala_assignment_new (VALA_EXPRESSION (left), VALA_EXPRESSION (right), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL)), NULL))));
		(__temp124 == NULL ? NULL : (__temp124 = (g_object_unref (__temp124), NULL)));
		(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
		(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
		(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
		(prop_sym == NULL ? NULL : (prop_sym = (g_object_unref (prop_sym), NULL)));
	}
}


static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	ValaSymbol* sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (cl)), vala_property_get_name (prop));
	if (sym != NULL && VALA_IS_PROPERTY (vala_symbol_get_node (sym))) {
		ValaProperty* __temp125 = NULL;
		ValaProperty* base_property = (__temp125 = VALA_PROPERTY (vala_symbol_get_node (sym)), (__temp125 == NULL ? NULL : g_object_ref (__temp125)));
		if (vala_property_get_is_abstract (base_property) || vala_property_get_is_virtual (base_property)) {
			if (!vala_property_equals (prop, base_property)) {
				(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
				char* __temp128 = NULL;
				char* __temp127 = NULL;
				char* __temp126 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp128 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (__temp126 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))), (__temp127 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_property)))))));
				(__temp128 = (g_free (__temp128), NULL));
				(__temp127 = (g_free (__temp127), NULL));
				(__temp126 = (g_free (__temp126), NULL));
				(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			(vala_property_set_base_property (prop, base_property), vala_property_get_base_property (prop));
			(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_property (self, prop, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GList* __temp133 = NULL;
		__temp133 = vala_class_get_base_types (cl);
		GList* type_it;
		for (type_it = __temp133; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type = type_it->data;
			{
				/* FIXME report error if multiple possible base properties are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym = vala_symbol_lookup (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type))), vala_property_get_name (prop));
					if (sym != NULL && VALA_IS_PROPERTY (vala_symbol_get_node (sym))) {
						ValaProperty* __temp129 = NULL;
						ValaProperty* base_property = (__temp129 = VALA_PROPERTY (vala_symbol_get_node (sym)), (__temp129 == NULL ? NULL : g_object_ref (__temp129)));
						if (vala_property_get_is_abstract (base_property)) {
							if (!vala_property_equals (prop, base_property)) {
								(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
								char* __temp132 = NULL;
								char* __temp131 = NULL;
								char* __temp130 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp132 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (__temp130 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))), (__temp131 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (base_property)))))));
								(__temp132 = (g_free (__temp132), NULL));
								(__temp131 = (g_free (__temp131), NULL));
								(__temp130 = (g_free (__temp130), NULL));
								(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								return;
							}
							(vala_property_set_base_interface_property (prop, base_property), vala_property_get_base_interface_property (prop));
							(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							return;
						}
						(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
			}
		}
		(__temp133 == NULL ? NULL : (__temp133 = (g_list_free (__temp133), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_end_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	if (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)) != NULL) {
		/* is null if it references a type parameter */
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))) {
		ValaClass* __temp134 = NULL;
		ValaClass* cl = (__temp134 = VALA_CLASS (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (prop))))), (__temp134 == NULL ? NULL : g_object_ref (__temp134)));
		vala_semantic_analyzer_find_base_interface_property (self, prop, cl);
		if (vala_property_get_is_virtual (prop) || vala_property_get_overrides (prop)) {
			vala_semantic_analyzer_find_base_class_property (self, prop, cl);
			if (vala_property_get_base_property (prop) == NULL) {
				(vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE), vala_code_node_get_error (VALA_CODE_NODE (prop)));
				char* __temp136 = NULL;
				char* __temp135 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (__temp136 = g_strdup_printf ("%s: no suitable property found to override", (__temp135 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))));
				(__temp136 = (g_free (__temp136), NULL));
				(__temp135 = (g_free (__temp135), NULL));
			}
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_begin_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
	ValaProperty* __temp137 = NULL;
	ValaProperty* prop = (__temp137 = VALA_PROPERTY (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (acc))))), (__temp137 == NULL ? NULL : g_object_ref (__temp137)));
	if (vala_property_accessor_get_readable (acc)) {
		ValaTypeReference* __temp138 = NULL;
		self->priv->current_return_type = (__temp138 = g_object_ref (vala_property_get_type_reference (prop)), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp138);
	} else {
		/* void*/
		ValaTypeReference* __temp139 = NULL;
		self->priv->current_return_type = (__temp139 = vala_type_reference_new (), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp139);
	}
	(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
}


static void vala_semantic_analyzer_real_visit_end_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
	ValaTypeReference* __temp140 = NULL;
	self->priv->current_return_type = (__temp140 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), __temp140);
}


static void vala_semantic_analyzer_real_visit_begin_constructor (ValaCodeVisitor* base, ValaConstructor* c)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
	ValaSymbol* __temp142 = NULL;
	ValaSymbol* __temp141 = NULL;
	self->priv->current_symbol = (__temp142 = (__temp141 = vala_code_node_get_symbol (VALA_CODE_NODE (c)), (__temp141 == NULL ? NULL : g_object_ref (__temp141))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp142);
}


static void vala_semantic_analyzer_real_visit_end_constructor (ValaCodeVisitor* base, ValaConstructor* c)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
	ValaSymbol* __temp144 = NULL;
	ValaSymbol* __temp143 = NULL;
	self->priv->current_symbol = (__temp144 = (__temp143 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp143 == NULL ? NULL : g_object_ref (__temp143))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp144);
}


static void vala_semantic_analyzer_real_visit_begin_destructor (ValaCodeVisitor* base, ValaDestructor* d)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
	ValaSymbol* __temp146 = NULL;
	ValaSymbol* __temp145 = NULL;
	self->priv->current_symbol = (__temp146 = (__temp145 = vala_code_node_get_symbol (VALA_CODE_NODE (d)), (__temp145 == NULL ? NULL : g_object_ref (__temp145))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp146);
}


static void vala_semantic_analyzer_real_visit_end_destructor (ValaCodeVisitor* base, ValaDestructor* d)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
	ValaSymbol* __temp148 = NULL;
	ValaSymbol* __temp147 = NULL;
	self->priv->current_symbol = (__temp148 = (__temp147 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp147 == NULL ? NULL : g_object_ref (__temp147))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp148);
}


static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (n));
}


static void vala_semantic_analyzer_real_visit_begin_block (ValaCodeVisitor* base, ValaBlock* b)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	ValaSymbol* __temp150 = NULL;
	ValaSymbol* __temp149 = NULL;
	self->priv->current_symbol = (__temp150 = (__temp149 = vala_code_node_get_symbol (VALA_CODE_NODE (b)), (__temp149 == NULL ? NULL : g_object_ref (__temp149))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp150);
}


static void vala_semantic_analyzer_real_visit_end_block (ValaCodeVisitor* base, ValaBlock* b)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	{
		GList* __temp151 = NULL;
		__temp151 = vala_block_get_local_variables (b);
		GList* decl_it;
		for (decl_it = __temp151; decl_it != NULL; decl_it = decl_it->next) {
			ValaVariableDeclarator* decl = decl_it->data;
			{
				(vala_symbol_set_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl)), FALSE), vala_symbol_get_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl))));
			}
		}
		(__temp151 == NULL ? NULL : (__temp151 = (g_list_free (__temp151), NULL)));
	}
	ValaSymbol* __temp153 = NULL;
	ValaSymbol* __temp152 = NULL;
	self->priv->current_symbol = (__temp153 = (__temp152 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (__temp152 == NULL ? NULL : g_object_ref (__temp152))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp153);
}


static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
	if (vala_variable_declarator_get_type_reference (decl) == NULL) {
		/* var type */
		if (vala_variable_declarator_get_initializer (decl) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed without initializer");
			return;
		}
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed with non-typed initializer");
			return;
		}
		ValaTypeReference* __temp154 = NULL;
		(vala_variable_declarator_set_type_reference (decl, (__temp154 = vala_type_reference_copy (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl))))), vala_variable_declarator_get_type_reference (decl));
		(__temp154 = (g_object_unref (__temp154), NULL));
		(vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (decl), (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) == NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))))), vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl)));
		(vala_type_reference_set_transfers_ownership (vala_variable_declarator_get_type_reference (decl), FALSE), vala_type_reference_get_transfers_ownership (vala_variable_declarator_get_type_reference (decl)));
	}
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			if (!(VALA_IS_MEMBER_ACCESS (vala_variable_declarator_get_initializer (decl)))) {
				(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
			if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
				ValaMethod* __temp155 = NULL;
				ValaMethod* m = (__temp155 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl)))), (__temp155 == NULL ? NULL : g_object_ref (__temp155)));
				ValaCallback* __temp156 = NULL;
				ValaCallback* cb = (__temp156 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp156 == NULL ? NULL : g_object_ref (__temp156)));
				/* check whether method matches callback type */
				if (!vala_callback_matches_method (cb, m)) {
					(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
					char* __temp159 = NULL;
					char* __temp158 = NULL;
					char* __temp157 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), (__temp159 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp157 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp158 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
					(__temp159 = (g_free (__temp159), NULL));
					(__temp158 = (g_free (__temp158), NULL));
					(__temp157 = (g_free (__temp157), NULL));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					return;
				}
				(vala_expression_set_static_type (vala_variable_declarator_get_initializer (decl), vala_variable_declarator_get_type_reference (decl)), vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			} else {
				(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
		}
		if (vala_semantic_analyzer_get_memory_management (self)) {
			if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))) {
				/* rhs transfers ownership of the expression */
				if (!vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
					/* lhs doesn't own the value */
					(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "Invalid assignment from owned expression to unowned variable");
					return;
				}
			}
		}
	}
	if (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	ValaSymbol* __temp160 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (decl), (__temp160 = vala_symbol_new (VALA_CODE_NODE (decl)))), vala_code_node_get_symbol (VALA_CODE_NODE (decl)));
	(__temp160 == NULL ? NULL : (__temp160 = (g_object_unref (__temp160), NULL)));
	vala_symbol_add (self->priv->current_symbol, vala_variable_declarator_get_name (decl), vala_code_node_get_symbol (VALA_CODE_NODE (decl)));
	ValaBlock* __temp161 = NULL;
	ValaBlock* block = (__temp161 = VALA_BLOCK (vala_symbol_get_node (self->priv->current_symbol)), (__temp161 == NULL ? NULL : g_object_ref (__temp161)));
	vala_block_add_local_variable (block, decl);
	(vala_symbol_set_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl)), TRUE), vala_symbol_get_active (vala_code_node_get_symbol (VALA_CODE_NODE (decl))));
	(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_begin_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
		/* initializer is used as array initializer */
		ValaArray* __temp162 = NULL;
		ValaArray* edt = (__temp162 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (__temp162 == NULL ? NULL : g_object_ref (__temp162)));
		GList* inits = vala_initializer_list_get_initializers (list);
		gint rank = vala_array_get_rank ((VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))));
		ValaTypeReference* child_type = vala_type_reference_copy (vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		if (rank > 1) {
			ValaArray* __temp163 = NULL;
			(vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((__temp163 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1)))), vala_type_reference_get_data_type (child_type));
			(__temp163 = (g_object_unref (__temp163), NULL));
		} else {
			(vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt)), vala_type_reference_get_data_type (child_type));
		}
		{
			GList* __temp165 = NULL;
			__temp165 = inits;
			GList* e_it;
			for (e_it = __temp165; e_it != NULL; e_it = e_it->next) {
				ValaExpression* e = e_it->data;
				{
					ValaTypeReference* __temp164 = NULL;
					(vala_expression_set_expected_type (e, (__temp164 = vala_type_reference_copy (child_type))), vala_expression_get_expected_type (e));
					(__temp164 = (g_object_unref (__temp164), NULL));
				}
			}
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_list_free (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_end_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
		ValaArray* __temp166 = NULL;
		ValaArray* edt = (__temp166 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (__temp166 == NULL ? NULL : g_object_ref (__temp166)));
		GList* inits = vala_initializer_list_get_initializers (list);
		gint rank = vala_array_get_rank (edt);
		ValaTypeReference* child_type = vala_type_reference_copy (vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		gboolean error = FALSE;
		if (rank > 1) {
			ValaArray* __temp167 = NULL;
			(vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((__temp167 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1)))), vala_type_reference_get_data_type (child_type));
			(__temp167 = (g_object_unref (__temp167), NULL));
			{
				GList* __temp169 = NULL;
				__temp169 = inits;
				GList* e_it;
				for (e_it = __temp169; e_it != NULL; e_it = e_it->next) {
					ValaExpression* e = e_it->data;
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							continue;
						}
						if (!(VALA_IS_INITIALIZER_LIST (e))) {
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Initializer list expected");
							continue;
						}
						if (!vala_type_reference_equals (vala_expression_get_static_type (e), child_type)) {
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							char* __temp168 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (__temp168 = g_strdup_printf ("Expected initializer list of type `%s' but got `%s'", vala_data_type_get_name (vala_type_reference_get_data_type (child_type)), vala_data_type_get_name (vala_type_reference_get_data_type (vala_expression_get_static_type (e))))));
							(__temp168 = (g_free (__temp168), NULL));
						}
					}
				}
			}
		} else {
			(vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt)), vala_type_reference_get_data_type (child_type));
			{
				GList* __temp171 = NULL;
				__temp171 = inits;
				GList* e_it;
				for (e_it = __temp171; e_it != NULL; e_it = e_it->next) {
					ValaExpression* e = e_it->data;
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							continue;
						}
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (e), child_type)) {
							error = TRUE;
							(vala_code_node_set_error (VALA_CODE_NODE (e), TRUE), vala_code_node_get_error (VALA_CODE_NODE (e)));
							char* __temp170 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (__temp170 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", vala_data_type_get_name (vala_type_reference_get_data_type (child_type)), vala_data_type_get_name (vala_type_reference_get_data_type (vala_expression_get_static_type (e))))));
							(__temp170 = (g_free (__temp170), NULL));
						}
					}
				}
			}
		}
		if (!error) {
			/* everything seems to be correct */
			(vala_expression_set_static_type (VALA_EXPRESSION (list), vala_expression_get_expected_type (VALA_EXPRESSION (list))), vala_expression_get_static_type (VALA_EXPRESSION (list)));
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_list_free (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_EXPRESSION_STATEMENT (stmt));
	if (vala_expression_get_static_type (vala_expression_statement_get_expression (stmt)) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_expression_statement_get_expression (stmt)))) {
		vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Short-living reference");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_IF_STATEMENT (stmt));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_if_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_if_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_if_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_WHILE_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_while_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_while_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOR_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_for_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_for_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_begin_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	ValaVariableDeclarator* __temp172 = NULL;
	(vala_foreach_statement_set_variable_declarator (stmt, (__temp172 = vala_variable_declarator_new (vala_foreach_statement_get_variable_name (stmt), NULL, NULL))), vala_foreach_statement_get_variable_declarator (stmt));
	(__temp172 == NULL ? NULL : (__temp172 = (g_object_unref (__temp172), NULL)));
	(vala_variable_declarator_set_type_reference (vala_foreach_statement_get_variable_declarator (stmt), vala_foreach_statement_get_type_reference (stmt)), vala_variable_declarator_get_type_reference (vala_foreach_statement_get_variable_declarator (stmt)));
	ValaSymbol* __temp173 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt)), (__temp173 = vala_symbol_new (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))));
	(__temp173 == NULL ? NULL : (__temp173 = (g_object_unref (__temp173), NULL)));
	vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_body (stmt))), vala_foreach_statement_get_variable_name (stmt), vala_code_node_get_symbol (VALA_CODE_NODE (vala_foreach_statement_get_variable_declarator (stmt))));
}


static void vala_semantic_analyzer_real_visit_end_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	ValaDataType* __temp174 = NULL;
	ValaDataType* collection_type = (__temp174 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt))), (__temp174 == NULL ? NULL : g_object_ref (__temp174)));
	if (!(VALA_IS_ARRAY (collection_type) || collection_type == self->priv->glist_type || collection_type == self->priv->gslist_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Collection not iterable");
		(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
		return;
	}
	(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
	if (self->priv->current_return_type == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return not allowed in this context");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) == NULL && vala_type_reference_get_data_type (self->priv->current_return_type) != NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return without value in non-void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_data_type (self->priv->current_return_type) == NULL && vala_type_reference_get_type_parameter (self->priv->current_return_type) == NULL) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return with value in void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)), self->priv->current_return_type)) {
		char* __temp177 = NULL;
		char* __temp176 = NULL;
		char* __temp175 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), (__temp177 = g_strdup_printf ("Return: Cannot convert from `%s' to `%s'", (__temp175 = vala_type_reference_to_string (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)))), (__temp176 = vala_type_reference_to_string (self->priv->current_return_type)))));
		(__temp177 = (g_free (__temp177), NULL));
		(__temp176 = (g_free (__temp176), NULL));
		(__temp175 = (g_free (__temp175), NULL));
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return value transfers ownership but method return type hasn't been declared to transfer ownership");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt)) != NULL && VALA_IS_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt)))) && vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Local variable with strong reference used as return value and method return type hasn't been declared to transfer ownership");
	}
}


/**
 * Visit operation called for lock statements.
 *
 * @param stmt a lock statement
 */
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LOCK_STATEMENT (stmt));
	/* resource must be a member access and denote a Lockable */
	if (!(VALA_IS_MEMBER_ACCESS (vala_lock_statement_get_resource (stmt)) && VALA_IS_LOCKABLE (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		(vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE), vala_code_node_get_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Expression is either not a member access or does not denote a lockable member");
		return;
	}
	/* parent symbol must be the current class */
	if (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))) != VALA_CODE_NODE (self->priv->current_class)) {
		(vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE), vala_code_node_get_error (VALA_CODE_NODE (stmt)));
		(vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE), vala_code_node_get_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Only members of the current class are lockable");
	}
	vala_lockable_set_lock_used ((VALA_LOCKABLE (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))))), TRUE);
}


static void vala_semantic_analyzer_real_visit_begin_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL) {
		ValaTypeReference* __temp178 = NULL;
		(vala_expression_set_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)), (__temp178 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr)))), vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))));
		(__temp178 = (g_object_unref (__temp178), NULL));
		ValaArray* __temp179 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))), VALA_DATA_TYPE ((__temp179 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)))), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr)))));
		(__temp179 = (g_object_unref (__temp179), NULL));
	}
}


/* FIXME: add element type to type_argument
*
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_semantic_analyzer_real_visit_end_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	gint i = 0;
	GList* size = vala_array_creation_expression_get_sizes (expr);
	/* check for errors in the size list */
	if (size != NULL) {
		{
			GList* __temp180 = NULL;
			__temp180 = size;
			GList* e_it;
			for (e_it = __temp180; e_it != NULL; e_it = e_it->next) {
				ValaExpression* e = e_it->data;
				{
					if (vala_expression_get_static_type (e) == NULL) {
						(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
						return;
					} else {
						/* return on previous error */
						if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
						}
					}
				}
			}
		}
		if (vala_code_node_get_error (VALA_CODE_NODE (expr))) {
			(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
			return;
		}
	}
	/* check for wrong elements inside the initializer */
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL && vala_expression_get_static_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))) == NULL) {
		(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
		return;
	}
	/* try to construct the type of the array */
	if (vala_array_creation_expression_get_element_type (expr) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Cannot determine the element type of the created array");
		(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
		return;
	}
	ValaTypeReference* __temp181 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp181 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr)))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp181 = (g_object_unref (__temp181), NULL));
	if (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)) != NULL) {
		ValaArray* __temp182 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp182 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
		(__temp182 = (g_object_unref (__temp182), NULL));
	} else {
		ValaArray* __temp183 = NULL;
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp183 = vala_type_parameter_get_array (vala_type_reference_get_type_parameter (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
		(__temp183 = (g_object_unref (__temp183), NULL));
	}
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	vala_type_reference_add_type_argument (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_array_creation_expression_get_element_type (expr));
	(size == NULL ? NULL : (size = (g_list_free (size), NULL)));
}


static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BOOLEAN_LITERAL (expr));
	(vala_literal_set_static_type (VALA_LITERAL (expr), self->priv->bool_type), vala_literal_get_static_type (VALA_LITERAL (expr)));
}


static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CHARACTER_LITERAL (expr));
	ValaTypeReference* __temp184 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp184 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp184 == NULL ? NULL : (__temp184 = (g_object_unref (__temp184), NULL)));
	ValaSymbol* __temp185 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp185 = vala_symbol_lookup (self->priv->root_symbol, "char"))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp185 == NULL ? NULL : (__temp185 = (g_object_unref (__temp185), NULL)));
}


static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTEGER_LITERAL (expr));
	ValaTypeReference* __temp186 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp186 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp186 == NULL ? NULL : (__temp186 = (g_object_unref (__temp186), NULL)));
	ValaSymbol* __temp188 = NULL;
	char* __temp187 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp188 = vala_symbol_lookup (self->priv->root_symbol, (__temp187 = vala_integer_literal_get_type_name (expr))))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp188 == NULL ? NULL : (__temp188 = (g_object_unref (__temp188), NULL)));
	(__temp187 = (g_free (__temp187), NULL));
}


static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_REAL_LITERAL (expr));
	ValaTypeReference* __temp189 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp189 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp189 == NULL ? NULL : (__temp189 = (g_object_unref (__temp189), NULL)));
	ValaSymbol* __temp191 = NULL;
	char* __temp190 = NULL;
	(vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), VALA_DATA_TYPE (vala_symbol_get_node ((__temp191 = vala_symbol_lookup (self->priv->root_symbol, (__temp190 = vala_real_literal_get_type_name (expr))))))), vala_type_reference_get_data_type (vala_literal_get_static_type (VALA_LITERAL (expr))));
	(__temp191 == NULL ? NULL : (__temp191 = (g_object_unref (__temp191), NULL)));
	(__temp190 = (g_free (__temp190), NULL));
}


static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRING_LITERAL (expr));
	ValaTypeReference* __temp192 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp192 = vala_type_reference_copy (self->priv->string_type))), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp192 = (g_object_unref (__temp192), NULL));
	(vala_type_reference_set_non_null (vala_literal_get_static_type (VALA_LITERAL (expr)), TRUE), vala_type_reference_get_non_null (vala_literal_get_static_type (VALA_LITERAL (expr))));
}


static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NULL_LITERAL (expr));
	/* empty TypeReference represents null */
	ValaTypeReference* __temp193 = NULL;
	(vala_literal_set_static_type (VALA_LITERAL (expr), (__temp193 = vala_type_reference_new ())), vala_literal_get_static_type (VALA_LITERAL (expr)));
	(__temp193 == NULL ? NULL : (__temp193 = (g_object_unref (__temp193), NULL)));
}


static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LITERAL_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_literal_get_static_type (vala_literal_expression_get_literal (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_node (ValaSemanticAnalyzer* self, ValaCodeNode* node)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_CODE_NODE (node), NULL);
	if (VALA_IS_FIELD (node)) {
		ValaField* __temp194 = NULL;
		ValaField* f = (__temp194 = VALA_FIELD (node), (__temp194 == NULL ? NULL : g_object_ref (__temp194)));
		ValaTypeReference* __temp195 = NULL;
		return (__temp195 = g_object_ref (vala_field_get_type_reference (f)), (f == NULL ? NULL : (f = (g_object_unref (f), NULL))), __temp195);
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	} else {
		if (VALA_IS_CONSTANT (node)) {
			ValaConstant* __temp196 = NULL;
			ValaConstant* c = (__temp196 = VALA_CONSTANT (node), (__temp196 == NULL ? NULL : g_object_ref (__temp196)));
			ValaTypeReference* __temp197 = NULL;
			return (__temp197 = g_object_ref (vala_constant_get_type_reference (c)), (c == NULL ? NULL : (c = (g_object_unref (c), NULL))), __temp197);
			(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
		} else {
			if (VALA_IS_PROPERTY (node)) {
				ValaProperty* __temp198 = NULL;
				ValaProperty* prop = (__temp198 = VALA_PROPERTY (node), (__temp198 == NULL ? NULL : g_object_ref (__temp198)));
				ValaTypeReference* type = vala_type_reference_copy (vala_property_get_type_reference (prop));
				(vala_type_reference_set_takes_ownership (type, FALSE), vala_type_reference_get_takes_ownership (type));
				ValaTypeReference* __temp199 = NULL;
				return (__temp199 = type, (prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL))), __temp199);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
				(type = (g_object_unref (type), NULL));
			} else {
				if (VALA_IS_FORMAL_PARAMETER (node)) {
					ValaFormalParameter* __temp200 = NULL;
					ValaFormalParameter* p = (__temp200 = VALA_FORMAL_PARAMETER (node), (__temp200 == NULL ? NULL : g_object_ref (__temp200)));
					ValaTypeReference* __temp201 = NULL;
					ValaTypeReference* __temp202 = NULL;
					return (__temp202 = (__temp201 = vala_formal_parameter_get_type_reference (p), (__temp201 == NULL ? NULL : g_object_ref (__temp201))), (p == NULL ? NULL : (p = (g_object_unref (p), NULL))), __temp202);
					(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				} else {
					if (VALA_IS_TYPE_REFERENCE (node)) {
						ValaTypeReference* __temp203 = NULL;
						return (__temp203 = VALA_TYPE_REFERENCE (node), (__temp203 == NULL ? NULL : g_object_ref (__temp203)));
					} else {
						if (VALA_IS_VARIABLE_DECLARATOR (node)) {
							ValaVariableDeclarator* __temp205 = NULL;
							ValaVariableDeclarator* decl = (__temp205 = VALA_VARIABLE_DECLARATOR (node), (__temp205 == NULL ? NULL : g_object_ref (__temp205)));
							ValaTypeReference* __temp206 = NULL;
							ValaTypeReference* __temp207 = NULL;
							return (__temp207 = (__temp206 = vala_variable_declarator_get_type_reference (decl), (__temp206 == NULL ? NULL : g_object_ref (__temp206))), (decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL))), __temp207);
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						} else {
							if (VALA_IS_ENUM_VALUE (node) || VALA_IS_FLAGS_VALUE (node)) {
								ValaTypeReference* type = vala_type_reference_new ();
								(vala_type_reference_set_data_type (type, VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (node))))), vala_type_reference_get_data_type (type));
								return type;
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							}
						}
					}
				}
			}
		}
	}
	return VALA_TYPE_REFERENCE (NULL);
}


ValaSymbol* vala_semantic_analyzer_symbol_lookup_inherited (ValaSymbol* sym, const char* name)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (sym), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	ValaSymbol* result = vala_symbol_lookup (sym, name);
	if (result != NULL) {
		return result;
	}
	if (VALA_IS_CLASS (vala_symbol_get_node (sym))) {
		ValaClass* __temp211 = NULL;
		ValaClass* cl = (__temp211 = VALA_CLASS (vala_symbol_get_node (sym)), (__temp211 == NULL ? NULL : g_object_ref (__temp211)));
		{
			GList* __temp214 = NULL;
			__temp214 = vala_class_get_base_types (cl);
			GList* base_type_it;
			for (base_type_it = __temp214; base_type_it != NULL; base_type_it = base_type_it->next) {
				ValaTypeReference* base_type = base_type_it->data;
				{
					ValaSymbol* __temp212 = NULL;
					result = (__temp212 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp212);
					if (result != NULL) {
						ValaSymbol* __temp213 = NULL;
						return (__temp213 = result, (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp213);
					}
				}
			}
			(__temp214 == NULL ? NULL : (__temp214 = (g_list_free (__temp214), NULL)));
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (vala_symbol_get_node (sym))) {
			ValaStruct* __temp215 = NULL;
			ValaStruct* st = (__temp215 = VALA_STRUCT (vala_symbol_get_node (sym)), (__temp215 == NULL ? NULL : g_object_ref (__temp215)));
			{
				GList* __temp218 = NULL;
				__temp218 = vala_struct_get_base_types (st);
				GList* base_type_it;
				for (base_type_it = __temp218; base_type_it != NULL; base_type_it = base_type_it->next) {
					ValaTypeReference* base_type = base_type_it->data;
					{
						ValaSymbol* __temp216 = NULL;
						result = (__temp216 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp216);
						if (result != NULL) {
							ValaSymbol* __temp217 = NULL;
							return (__temp217 = result, (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp217);
						}
					}
				}
				(__temp218 == NULL ? NULL : (__temp218 = (g_list_free (__temp218), NULL)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		} else {
			if (VALA_IS_INTERFACE (vala_symbol_get_node (sym))) {
				ValaInterface* __temp219 = NULL;
				ValaInterface* iface = (__temp219 = VALA_INTERFACE (vala_symbol_get_node (sym)), (__temp219 == NULL ? NULL : g_object_ref (__temp219)));
				{
					GList* __temp222 = NULL;
					__temp222 = vala_interface_get_prerequisites (iface);
					GList* prerequisite_it;
					for (prerequisite_it = __temp222; prerequisite_it != NULL; prerequisite_it = prerequisite_it->next) {
						ValaTypeReference* prerequisite = prerequisite_it->data;
						{
							ValaSymbol* __temp220 = NULL;
							result = (__temp220 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (prerequisite))), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp220);
							if (result != NULL) {
								ValaSymbol* __temp221 = NULL;
								return (__temp221 = result, (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), __temp221);
							}
						}
					}
					(__temp222 == NULL ? NULL : (__temp222 = (g_list_free (__temp222), NULL)));
				}
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	gpointer __temp223;
	return VALA_SYMBOL ((__temp223 = NULL, (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), __temp223));
	(result == NULL ? NULL : (result = (g_object_unref (result), NULL)));
}


static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PARENTHESIZED_EXPRESSION (expr));
	ValaTypeReference* __temp224 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp224 = vala_type_reference_copy (vala_expression_get_static_type (vala_parenthesized_expression_get_inner (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp224 = (g_object_unref (__temp224), NULL));
	/* don't call g_object_ref_sink on inner and outer expression*/
	(vala_type_reference_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE), vala_type_reference_get_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static ValaDataType* vala_semantic_analyzer_find_parent_type (ValaSemanticAnalyzer* self, ValaSymbol* sym)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (sym == NULL || VALA_IS_SYMBOL (sym), NULL);
	while (sym != NULL) {
		if (VALA_IS_DATA_TYPE (vala_symbol_get_node (sym))) {
			ValaDataType* __temp225 = NULL;
			return (__temp225 = VALA_DATA_TYPE (vala_symbol_get_node (sym)), (__temp225 == NULL ? NULL : g_object_ref (__temp225)));
		}
		sym = vala_symbol_get_parent_symbol (sym);
	}
	return VALA_DATA_TYPE (NULL);
}


static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (expr));
	ValaSymbol* base_symbol = NULL;
	if (vala_member_access_get_inner (expr) == NULL) {
		ValaSymbol* __temp229 = NULL;
		ValaSymbol* __temp228 = NULL;
		base_symbol = (__temp229 = (__temp228 = self->priv->current_symbol, (__temp228 == NULL ? NULL : g_object_ref (__temp228))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp229);
		ValaSymbol* __temp230 = NULL;
		ValaSymbol* sym = (__temp230 = self->priv->current_symbol, (__temp230 == NULL ? NULL : g_object_ref (__temp230)));
		while (sym != NULL && vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			ValaSymbol* __temp231 = NULL;
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp231 = vala_semantic_analyzer_symbol_lookup_inherited (sym, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			(__temp231 == NULL ? NULL : (__temp231 = (g_object_unref (__temp231), NULL)));
			ValaSymbol* __temp233 = NULL;
			ValaSymbol* __temp232 = NULL;
			sym = (__temp233 = (__temp232 = vala_symbol_get_parent_symbol (sym), (__temp232 == NULL ? NULL : g_object_ref (__temp232))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp233);
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			{
				GList* __temp237 = NULL;
				__temp237 = self->priv->current_using_directives;
				GList* ns_it;
				for (ns_it = __temp237; ns_it != NULL; ns_it = ns_it->next) {
					ValaNamespaceReference* ns = ns_it->data;
					{
						ValaSymbol* local_sym = vala_symbol_lookup (vala_namespace_reference_get_namespace_symbol (ns), vala_member_access_get_member_name (expr));
						if (local_sym != NULL) {
							if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
								(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
								char* __temp236 = NULL;
								char* __temp235 = NULL;
								char* __temp234 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp236 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", vala_member_access_get_member_name (expr), (__temp234 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))), (__temp235 = vala_symbol_get_full_name (local_sym)))));
								(__temp236 = (g_free (__temp236), NULL));
								(__temp235 = (g_free (__temp235), NULL));
								(__temp234 = (g_free (__temp234), NULL));
								(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
								return;
							}
							(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), local_sym), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
						}
						(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
					}
				}
			}
		}
		(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
	} else {
		if (vala_code_node_get_error (VALA_CODE_NODE (vala_member_access_get_inner (expr)))) {
			/* if there was an error in the inner expression, skip this check */
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			return;
		}
		if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (expr)) || VALA_IS_BASE_ACCESS (vala_member_access_get_inner (expr))) {
			ValaSymbol* __temp239 = NULL;
			ValaSymbol* __temp238 = NULL;
			base_symbol = (__temp239 = (__temp238 = vala_expression_get_symbol_reference (vala_member_access_get_inner (expr)), (__temp238 == NULL ? NULL : g_object_ref (__temp238))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp239);
			if (VALA_IS_NAMESPACE (vala_symbol_get_node (base_symbol)) || VALA_IS_DATA_TYPE (vala_symbol_get_node (base_symbol))) {
				ValaSymbol* __temp240 = NULL;
				(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp240 = vala_symbol_lookup (base_symbol, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
				(__temp240 == NULL ? NULL : (__temp240 = (g_object_unref (__temp240), NULL)));
			}
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_expression_get_static_type (vala_member_access_get_inner (expr)) != NULL) {
			ValaSymbol* __temp242 = NULL;
			ValaSymbol* __temp241 = NULL;
			base_symbol = (__temp242 = (__temp241 = vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_member_access_get_inner (expr))))), (__temp241 == NULL ? NULL : g_object_ref (__temp241))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), __temp242);
			ValaSymbol* __temp243 = NULL;
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (__temp243 = vala_semantic_analyzer_symbol_lookup_inherited (base_symbol, vala_member_access_get_member_name (expr)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			(__temp243 == NULL ? NULL : (__temp243 = (g_object_unref (__temp243), NULL)));
		}
	}
	if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		char* __temp245 = NULL;
		char* __temp244 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp245 = g_strdup_printf ("The name `%s' does not exist in the context of `%s'", vala_member_access_get_member_name (expr), (__temp244 = vala_symbol_get_full_name (base_symbol)))));
		(__temp245 = (g_free (__temp245), NULL));
		(__temp244 = (g_free (__temp244), NULL));
		(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
		return;
	}
	ValaCodeNode* __temp246 = NULL;
	ValaCodeNode* member = (__temp246 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr))), (__temp246 == NULL ? NULL : g_object_ref (__temp246)));
	ValaMemberAccessibility access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	if (VALA_IS_FIELD (member)) {
		access = (VALA_FIELD (member))->access;
	} else {
		if (VALA_IS_METHOD (member)) {
			access = (VALA_METHOD (member))->access;
		}
	}
	if (access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		ValaDataType* __temp247 = NULL;
		ValaDataType* target_type = (__temp247 = VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (member)))), (__temp247 == NULL ? NULL : g_object_ref (__temp247)));
		ValaDataType* this_type = vala_semantic_analyzer_find_parent_type (self, self->priv->current_symbol);
		if (target_type != this_type) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			char* __temp249 = NULL;
			char* __temp248 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp249 = g_strdup_printf ("Access to private member `%s' denied", (__temp248 = vala_symbol_get_full_name (vala_code_node_get_symbol (member))))));
			(__temp249 = (g_free (__temp249), NULL));
			(__temp248 = (g_free (__temp248), NULL));
			(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
			(this_type == NULL ? NULL : (this_type = (g_object_unref (this_type), NULL)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
			return;
		}
		(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
		(this_type == NULL ? NULL : (this_type = (g_object_unref (this_type), NULL)));
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	ValaTypeReference* __temp250 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp250 = vala_semantic_analyzer_get_static_type_for_node (self, vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp250 == NULL ? NULL : (__temp250 = (g_object_unref (__temp250), NULL)));
	(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
	(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
}


static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expression_type), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expected_type), FALSE);
	/* only null is compatible to null */
	if (vala_type_reference_get_data_type (expected_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL) {
		return (vala_type_reference_get_data_type (expression_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL);
	}
	if (vala_type_reference_get_data_type (expression_type) == NULL) {
		/* null can be cast to any reference or array type or pointer type */
		if (vala_type_reference_get_type_parameter (expected_type) != NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_reference_to_value_type (expected_type) || VALA_IS_POINTER (vala_type_reference_get_data_type (expected_type)) || VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type)) || VALA_IS_CALLBACK (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_data_type (expected_type) == self->priv->pointer_type) {
			return TRUE;
		}
		return FALSE;
	}
	/* null is not compatible with any other type (i.e. value types) 
	 temporarily ignore type parameters */
	if (vala_type_reference_get_type_parameter (expected_type) != NULL) {
		return TRUE;
	}
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (expression_type)) != VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type))) {
		return FALSE;
	}
	if (VALA_IS_ENUM (vala_type_reference_get_data_type (expression_type)) && vala_type_reference_get_data_type (expected_type) == vala_type_reference_get_data_type (self->priv->int_type)) {
		return TRUE;
	}
	if (vala_type_reference_get_data_type (expression_type) == vala_type_reference_get_data_type (expected_type)) {
		return TRUE;
	}
	if (VALA_IS_STRUCT (vala_type_reference_get_data_type (expression_type)) && VALA_IS_STRUCT (vala_type_reference_get_data_type (expected_type))) {
		ValaStruct* __temp258 = NULL;
		ValaStruct* expr_struct = (__temp258 = VALA_STRUCT (vala_type_reference_get_data_type (expression_type)), (__temp258 == NULL ? NULL : g_object_ref (__temp258)));
		ValaStruct* __temp259 = NULL;
		ValaStruct* expect_struct = (__temp259 = VALA_STRUCT (vala_type_reference_get_data_type (expected_type)), (__temp259 == NULL ? NULL : g_object_ref (__temp259)));
		/* integer types may be implicitly cast to floating point types */
		if (vala_struct_is_integer_type (expr_struct) && vala_struct_is_floating_type (expect_struct)) {
			gboolean __temp260;
			return (__temp260 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), __temp260);
		}
		if ((vala_struct_is_integer_type (expr_struct) && vala_struct_is_integer_type (expect_struct)) || (vala_struct_is_floating_type (expr_struct) && vala_struct_is_floating_type (expect_struct))) {
			if (vala_struct_get_rank (expr_struct) <= vala_struct_get_rank (expect_struct)) {
				gboolean __temp261;
				return (__temp261 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), __temp261);
			}
		}
		(expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL)));
		(expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL)));
	}
	return vala_data_type_is_subtype_of (vala_type_reference_get_data_type (expression_type), vala_type_reference_get_data_type (expected_type));
}


static void vala_semantic_analyzer_real_visit_begin_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_invocation_expression_get_call (expr)))) {
		/* if method resolving didn't succeed, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	ValaSymbol* __temp263 = NULL;
	ValaSymbol* msym = (__temp263 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp263 == NULL ? NULL : g_object_ref (__temp263)));
	if (msym == NULL) {
		/* if no symbol found, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		return;
	}
	GList* params = NULL;
	if (VALA_IS_INVOKABLE (vala_symbol_get_node (msym))) {
		ValaInvokable* __temp264 = NULL;
		ValaInvokable* m = (__temp264 = VALA_INVOKABLE (vala_symbol_get_node (msym)), (__temp264 == NULL ? NULL : g_object_ref (__temp264)));
		if (vala_invokable_is_invokable (m)) {
			GList* __temp265 = NULL;
			params = (__temp265 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp265);
		} else {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
			(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
			return;
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	} else {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
		return;
	}
	GList* args = vala_invocation_expression_get_argument_list (expr);
	GList* arg_it = args;
	{
		GList* __temp267 = NULL;
		__temp267 = params;
		GList* param_it;
		for (param_it = __temp267; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param = param_it->data;
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					break;
				}
				if (arg_it != NULL) {
					ValaExpression* __temp266 = NULL;
					ValaExpression* arg = (__temp266 = VALA_EXPRESSION (arg_it->data), (__temp266 == NULL ? NULL : g_object_ref (__temp266)));
					/* store expected type for callback parameters */
					(vala_expression_set_expected_type (arg, vala_formal_parameter_get_type_reference (param)), vala_expression_get_expected_type (arg));
					arg_it = arg_it->next;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
			}
		}
	}
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
	(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
}


static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GList* params, GList* args)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), FALSE);
	g_return_val_if_fail (VALA_IS_SYMBOL (msym), FALSE);
	GList* prev_arg_it = NULL;
	GList* arg_it = args;
	ValaAttribute* __temp268 = NULL;
	gboolean __temp269;
	gboolean diag = (__temp269 = ((__temp268 = vala_code_node_get_attribute (vala_symbol_get_node (msym), "Diagnostics")) != NULL), (__temp268 == NULL ? NULL : (__temp268 = (g_object_unref (__temp268), NULL))), __temp269);
	gboolean ellipsis = FALSE;
	gint i = 0;
	{
		GList* __temp278 = NULL;
		__temp278 = params;
		GList* param_it;
		for (param_it = __temp278; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param = param_it->data;
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					ellipsis = TRUE;
					break;
				}
				/* header file necessary if we need to cast argument */
				if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)) != NULL) {
					vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
				}
				if (arg_it == NULL) {
					if (vala_formal_parameter_get_default_expression (param) == NULL) {
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						char* __temp271 = NULL;
						char* __temp270 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp271 = g_strdup_printf ("Too few arguments, method `%s' does not take %d arguments", (__temp270 = vala_symbol_get_full_name (msym)), g_list_length (args))));
						(__temp271 = (g_free (__temp271), NULL));
						(__temp270 = (g_free (__temp270), NULL));
						return FALSE;
					}
				} else {
					ValaExpression* __temp273 = NULL;
					ValaExpression* arg = (__temp273 = VALA_EXPRESSION (arg_it->data), (__temp273 == NULL ? NULL : g_object_ref (__temp273)));
					if (vala_expression_get_static_type (arg) != NULL && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (arg), vala_formal_parameter_get_type_reference (param))) {
						/* if there was an error in the argument,
						 * i.e. arg.static_type == null, skip type check */
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						char* __temp276 = NULL;
						char* __temp275 = NULL;
						char* __temp274 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp276 = g_strdup_printf ("Argument %d: Cannot convert from `%s' to `%s'", i + 1, (__temp274 = vala_type_reference_to_string (vala_expression_get_static_type (arg))), (__temp275 = vala_type_reference_to_string (vala_formal_parameter_get_type_reference (param))))));
						(__temp276 = (g_free (__temp276), NULL));
						(__temp275 = (g_free (__temp275), NULL));
						(__temp274 = (g_free (__temp274), NULL));
						gboolean __temp277;
						return (__temp277 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), __temp277);
					}
					prev_arg_it = arg_it;
					arg_it = arg_it->next;
					i++;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
			}
		}
	}
	if (!ellipsis && arg_it != NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		char* __temp280 = NULL;
		char* __temp279 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp280 = g_strdup_printf ("Too many arguments, method `%s' does not take %d arguments", (__temp279 = vala_symbol_get_full_name (msym)), g_list_length (args))));
		(__temp280 = (g_free (__temp280), NULL));
		(__temp279 = (g_free (__temp279), NULL));
		return FALSE;
	}
	if (diag && prev_arg_it != NULL) {
		ValaExpression* __temp282 = NULL;
		ValaExpression* format_arg = (__temp282 = VALA_EXPRESSION (prev_arg_it->data), (__temp282 == NULL ? NULL : g_object_ref (__temp282)));
		if (VALA_IS_LITERAL_EXPRESSION (format_arg)) {
			ValaStringLiteral* __temp283 = NULL;
			ValaStringLiteral* format_lit = (__temp283 = VALA_STRING_LITERAL (vala_literal_expression_get_literal ((VALA_LITERAL_EXPRESSION (format_arg)))), (__temp283 == NULL ? NULL : g_object_ref (__temp283)));
			char* __temp284 = NULL;
			(vala_string_literal_set_value (format_lit, (__temp284 = g_strdup_printf ("\"%s:%d: %s", vala_source_file_get_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)))), vala_source_reference_get_first_line (vala_code_node_get_source_reference (VALA_CODE_NODE (expr))), g_utf8_offset_to_pointer (vala_string_literal_get_value (format_lit), 1)))), vala_string_literal_get_value (format_lit));
			(__temp284 = (g_free (__temp284), NULL));
			(format_lit == NULL ? NULL : (format_lit = (g_object_unref (format_lit), NULL)));
		}
		(format_arg == NULL ? NULL : (format_arg = (g_object_unref (format_arg), NULL)));
	}
	return TRUE;
}


static void vala_semantic_analyzer_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (expr))) {
		return;
	}
	ValaSymbol* __temp286 = NULL;
	ValaSymbol* msym = (__temp286 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp286 == NULL ? NULL : g_object_ref (__temp286)));
	ValaTypeReference* ret_type = NULL;
	GList* params = NULL;
	if (VALA_IS_INVOKABLE (vala_symbol_get_node (msym))) {
		ValaInvokable* __temp287 = NULL;
		ValaInvokable* m = (__temp287 = VALA_INVOKABLE (vala_symbol_get_node (msym)), (__temp287 == NULL ? NULL : g_object_ref (__temp287)));
		ValaTypeReference* __temp288 = NULL;
		ret_type = (__temp288 = vala_invokable_get_return_type (m), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), __temp288);
		GList* __temp289 = NULL;
		params = (__temp289 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp289);
		if (vala_type_reference_get_data_type (ret_type) == NULL && vala_type_reference_get_type_parameter (ret_type) == NULL) {
			/* void return type*/
			if (!(VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (VALA_CODE_NODE (expr))))) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation of void method not allowed as expression");
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			}
		}
		/* resolve generic return values*/
		if (vala_type_reference_get_type_parameter (ret_type) != NULL) {
			if (!(VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)))) {
				vala_report_error (vala_code_node_get_source_reference ((VALA_CODE_NODE (m))), "internal error: unsupported generic return value");
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			}
			ValaMemberAccess* __temp290 = NULL;
			ValaMemberAccess* ma = (__temp290 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (__temp290 == NULL ? NULL : g_object_ref (__temp290)));
			if (vala_member_access_get_inner (ma) == NULL) {
				/* TODO resolve generic return values within the type hierarchy if possible*/
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: resolving generic return values within type hierarchy not supported yet");
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
				return;
			} else {
				ValaTypeReference* __temp291 = NULL;
				ValaTypeReference* instance_type = (__temp291 = vala_expression_get_static_type (vala_member_access_get_inner (ma)), (__temp291 == NULL ? NULL : g_object_ref (__temp291)));
				while (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
					/* trace type arguments back to the datatype where the method has been declared*/
					GList* base_types = NULL;
					if (VALA_IS_CLASS (vala_type_reference_get_data_type (instance_type))) {
						ValaClass* __temp292 = NULL;
						ValaClass* cl = (__temp292 = VALA_CLASS (vala_type_reference_get_data_type (instance_type)), (__temp292 == NULL ? NULL : g_object_ref (__temp292)));
						GList* __temp293 = NULL;
						base_types = (__temp293 = vala_class_get_base_types (cl), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp293);
						(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
					} else {
						if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (instance_type))) {
							ValaInterface* __temp294 = NULL;
							ValaInterface* iface = (__temp294 = VALA_INTERFACE (vala_type_reference_get_data_type (instance_type)), (__temp294 == NULL ? NULL : g_object_ref (__temp294)));
							GList* __temp295 = NULL;
							base_types = (__temp295 = vala_interface_get_prerequisites (iface), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp295);
							(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
						} else {
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported generic type");
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
							(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
							(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
							(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
							return;
						}
					}
					{
						GList* __temp303 = NULL;
						__temp303 = base_types;
						GList* base_type_it;
						for (base_type_it = __temp303; base_type_it != NULL; base_type_it = base_type_it->next) {
							ValaTypeReference* base_type = base_type_it->data;
							{
								ValaSymbol* __temp296 = NULL;
								gboolean __temp297;
								if ((__temp297 = (__temp296 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), vala_symbol_get_name (msym))) != NULL, (__temp296 == NULL ? NULL : (__temp296 = (g_object_unref (__temp296), NULL))), __temp297)) {
									/* construct a new type reference for the base type with correctly linked type arguments*/
									ValaTypeReference* instance_base_type = vala_type_reference_new ();
									(vala_type_reference_set_data_type (instance_base_type, vala_type_reference_get_data_type (base_type)), vala_type_reference_get_data_type (instance_base_type));
									{
										GList* __temp300 = NULL;
										__temp300 = vala_type_reference_get_type_arguments (base_type);
										GList* type_arg_it;
										for (type_arg_it = __temp300; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
											ValaTypeReference* type_arg = type_arg_it->data;
											{
												if (vala_type_reference_get_type_parameter (type_arg) != NULL) {
													/* link to type argument of derived type*/
													gint param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)));
													if (param_index == -1) {
														char* __temp298 = NULL;
														vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp298 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)))));
														(__temp298 = (g_free (__temp298), NULL));
														(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
														(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
														(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
														(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
														(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
														(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
														(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
														(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
														(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
														return;
													}
													GList* __temp299 = NULL;
													type_arg = g_list_nth_data ((__temp299 = vala_type_reference_get_type_arguments (instance_type)), param_index);
													(__temp299 == NULL ? NULL : (__temp299 = (g_list_free (__temp299), NULL)));
												}
												vala_type_reference_add_type_argument (instance_base_type, type_arg);
											}
										}
										(__temp300 == NULL ? NULL : (__temp300 = (g_list_free (__temp300), NULL)));
									}
									ValaTypeReference* __temp302 = NULL;
									ValaTypeReference* __temp301 = NULL;
									instance_type = (__temp302 = (__temp301 = instance_base_type, (__temp301 == NULL ? NULL : g_object_ref (__temp301))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp302);
									(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
								}
							}
						}
					}
					(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
				}
				if (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: generic type parameter tracing not supported yet");
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				gint param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)));
				if (param_index == -1) {
					char* __temp304 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp304 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)))));
					(__temp304 = (g_free (__temp304), NULL));
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				ValaTypeReference* __temp307 = NULL;
				ValaTypeReference* __temp306 = NULL;
				GList* __temp305 = NULL;
				ret_type = (__temp307 = (__temp306 = VALA_TYPE_REFERENCE (g_list_nth_data ((__temp305 = vala_type_reference_get_type_arguments (instance_type)), param_index)), (__temp306 == NULL ? NULL : g_object_ref (__temp306))), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), __temp307);
				(__temp305 == NULL ? NULL : (__temp305 = (g_list_free (__temp305), NULL)));
				if (ret_type == NULL) {
					char* __temp308 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp308 = g_strdup_printf ("internal error: no actual argument found for type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (ret_type)))));
					(__temp308 = (g_free (__temp308), NULL));
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
					return;
				}
				(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
			}
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), ret_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	GList* __temp309 = NULL;
	vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), msym, params, (__temp309 = vala_invocation_expression_get_argument_list (expr)));
	(__temp309 == NULL ? NULL : (__temp309 = (g_list_free (__temp309), NULL)));
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
	(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
}


static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ELEMENT_ACCESS (expr));
	if (vala_expression_get_static_type (vala_element_access_get_container (expr)) == NULL) {
		/* don't proceed if a child expression failed */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	/* assign a static_type when possible */
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_element_access_get_container (expr))))) {
		GList* args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (expr)));
		if (g_list_length (args) != 1) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			char* __temp310 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp310 = g_strdup_printf ("internal error: array reference with %d type arguments, expected 1", g_list_length (args))));
			(__temp310 = (g_free (__temp310), NULL));
			(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
			return;
		}
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), VALA_TYPE_REFERENCE (args->data)), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
	} else {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		char* __temp312 = NULL;
		char* __temp311 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp312 = g_strdup_printf ("The expression `%s' does not denote an Array", (__temp311 = vala_type_reference_to_string (vala_expression_get_static_type (vala_element_access_get_container (expr)))))));
		(__temp312 = (g_free (__temp312), NULL));
		(__temp311 = (g_free (__temp311), NULL));
	}
	{
		GList* __temp313 = NULL;
		__temp313 = vala_element_access_get_indices (expr);
		GList* e_it;
		for (e_it = __temp313; e_it != NULL; e_it = e_it->next) {
			ValaExpression* e = e_it->data;
			{
				/* check if the index is of type integer 
				 don't proceed if a child expression failed */
				if (vala_expression_get_static_type (e) == NULL) {
					return;
				}
				/* check if the index is of type integer */
				if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
				}
			}
		}
		(__temp313 == NULL ? NULL : (__temp313 = (g_list_free (__temp313), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BASE_ACCESS (expr));
	if (self->priv->current_class == NULL) {
		if (self->priv->current_struct == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Base access invalid outside of class and struct");
			return;
		} else {
			GList* __temp314 = NULL;
			gboolean __temp315;
			if ((__temp315 = g_list_length ((__temp314 = vala_struct_get_base_types (self->priv->current_struct))) != 1, (__temp314 == NULL ? NULL : (__temp314 = (g_list_free (__temp314), NULL))), __temp315)) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				char* __temp317 = NULL;
				GList* __temp316 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp317 = g_strdup_printf ("Base access invalid without base type %d", g_list_length ((__temp316 = vala_struct_get_base_types (self->priv->current_struct))))));
				(__temp317 = (g_free (__temp317), NULL));
				(__temp316 == NULL ? NULL : (__temp316 = (g_list_free (__temp316), NULL)));
				return;
			}
		}
		GList* __temp318 = NULL;
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), g_list_first ((__temp318 = vala_struct_get_base_types (self->priv->current_struct)))->data), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(__temp318 == NULL ? NULL : (__temp318 = (g_list_free (__temp318), NULL)));
	} else {
		ValaTypeReference* __temp319 = NULL;
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp319 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(__temp319 == NULL ? NULL : (__temp319 = (g_object_unref (__temp319), NULL)));
		(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE (vala_class_get_base_class (self->priv->current_class))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	}
	(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)))))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_POSTFIX_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_postfix_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_end_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
	ValaDataType* type = NULL;
	if (vala_object_creation_expression_get_type_reference (expr) == NULL) {
		if (vala_object_creation_expression_get_member_name (expr) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Incomplete object creation expression");
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		ValaCodeNode* __temp320 = NULL;
		ValaCodeNode* constructor_node = (__temp320 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr)))), (__temp320 == NULL ? NULL : g_object_ref (__temp320)));
		ValaCodeNode* __temp321 = NULL;
		ValaCodeNode* type_node = (__temp321 = vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr)))), (__temp321 == NULL ? NULL : g_object_ref (__temp321)));
		GList* type_args = vala_member_access_get_type_arguments (vala_object_creation_expression_get_member_name (expr));
		if (VALA_IS_METHOD (constructor_node)) {
			ValaCodeNode* __temp323 = NULL;
			ValaCodeNode* __temp322 = NULL;
			type_node = (__temp323 = (__temp322 = vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (constructor_node))), (__temp322 == NULL ? NULL : g_object_ref (__temp322))), (type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL))), __temp323);
			ValaMethod* __temp324 = NULL;
			ValaMethod* constructor = (__temp324 = VALA_METHOD (constructor_node), (__temp324 == NULL ? NULL : g_object_ref (__temp324)));
			if (!(VALA_IS_CREATION_METHOD (constructor_node))) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				char* __temp326 = NULL;
				char* __temp325 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp326 = g_strdup_printf ("`%s' is not a creation method", (__temp325 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (constructor)))))));
				(__temp326 = (g_free (__temp326), NULL));
				(__temp325 = (g_free (__temp325), NULL));
				(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
				(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
				(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (constructor))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			GList* __temp327 = NULL;
			type_args = (__temp327 = vala_member_access_get_type_arguments ((VALA_MEMBER_ACCESS (vala_member_access_get_inner (vala_object_creation_expression_get_member_name (expr))))), (type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL))), __temp327);
			(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
		}
		if (VALA_IS_CLASS (type_node) || VALA_IS_STRUCT (type_node)) {
			ValaDataType* __temp329 = NULL;
			ValaDataType* __temp328 = NULL;
			type = (__temp329 = (__temp328 = VALA_DATA_TYPE (type_node), (__temp328 == NULL ? NULL : g_object_ref (__temp328))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp329);
		} else {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			char* __temp331 = NULL;
			char* __temp330 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp331 = g_strdup_printf ("`%s' is not a class or struct", (__temp330 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (type)))))));
			(__temp331 = (g_free (__temp331), NULL));
			(__temp330 = (g_free (__temp330), NULL));
			(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
			(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
			(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		ValaTypeReference* __temp332 = NULL;
		(vala_object_creation_expression_set_type_reference (expr, (__temp332 = vala_type_reference_new ())), vala_object_creation_expression_get_type_reference (expr));
		(__temp332 == NULL ? NULL : (__temp332 = (g_object_unref (__temp332), NULL)));
		(vala_type_reference_set_data_type (vala_object_creation_expression_get_type_reference (expr), type), vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr)));
		{
			GList* __temp333 = NULL;
			__temp333 = type_args;
			GList* type_arg_it;
			for (type_arg_it = __temp333; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
				ValaTypeReference* type_arg = type_arg_it->data;
				{
					vala_type_reference_add_type_argument (vala_object_creation_expression_get_type_reference (expr), type_arg);
				}
			}
		}
		(constructor_node == NULL ? NULL : (constructor_node = (g_object_unref (constructor_node), NULL)));
		(type_node == NULL ? NULL : (type_node = (g_object_unref (type_node), NULL)));
		(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
	} else {
		ValaDataType* __temp335 = NULL;
		ValaDataType* __temp334 = NULL;
		type = (__temp335 = (__temp334 = vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr)), (__temp334 == NULL ? NULL : g_object_ref (__temp334))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp335);
	}
	if (!vala_data_type_is_reference_type (type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		char* __temp337 = NULL;
		char* __temp336 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp337 = g_strdup_printf ("Can't create instance of value type `%s'", (__temp336 = vala_type_reference_to_string (vala_object_creation_expression_get_type_reference (expr))))));
		(__temp337 = (g_free (__temp337), NULL));
		(__temp336 = (g_free (__temp336), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (type)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	ValaTypeReference* __temp338 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp338 = vala_type_reference_copy (vala_object_creation_expression_get_type_reference (expr)))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp338 = (g_object_unref (__temp338), NULL));
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	if (VALA_IS_CLASS (type)) {
		ValaClass* __temp339 = NULL;
		ValaClass* cl = (__temp339 = VALA_CLASS (type), (__temp339 == NULL ? NULL : g_object_ref (__temp339)));
		if (vala_class_get_is_abstract (cl)) {
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			char* __temp341 = NULL;
			char* __temp340 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp341 = g_strdup_printf ("Can't create instance of abstract class `%s'", (__temp340 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cl)))))));
			(__temp341 = (g_free (__temp341), NULL));
			(__temp340 = (g_free (__temp340), NULL));
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_class_get_default_construction_method (cl) != NULL) {
			(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_class_get_default_construction_method (cl)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
		}
		while (cl != NULL) {
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				(vala_type_reference_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
				break;
			}
			ValaClass* __temp343 = NULL;
			ValaClass* __temp342 = NULL;
			cl = (__temp343 = (__temp342 = vala_class_get_base_class (cl), (__temp342 == NULL ? NULL : g_object_ref (__temp342))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), __temp343);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (type)) {
			ValaStruct* __temp344 = NULL;
			ValaStruct* st = (__temp344 = VALA_STRUCT (type), (__temp344 == NULL ? NULL : g_object_ref (__temp344)));
			if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_struct_get_default_construction_method (st) != NULL) {
				(vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (vala_struct_get_default_construction_method (st)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		}
	}
	GList* __temp345 = NULL;
	gboolean __temp346;
	if ((__temp346 = vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && g_list_length ((__temp345 = vala_object_creation_expression_get_argument_list (expr))) != 0, (__temp345 == NULL ? NULL : (__temp345 = (g_list_free (__temp345), NULL))), __temp346)) {
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		char* __temp348 = NULL;
		char* __temp347 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp348 = g_strdup_printf ("No arguments allowed when constructing type `%s'", (__temp347 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (type)))))));
		(__temp348 = (g_free (__temp348), NULL));
		(__temp347 = (g_free (__temp347), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
		ValaMethod* __temp349 = NULL;
		ValaMethod* m = (__temp349 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))), (__temp349 == NULL ? NULL : g_object_ref (__temp349)));
		GList* __temp351 = NULL;
		GList* __temp350 = NULL;
		vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp350 = vala_invokable_get_parameters (VALA_INVOKABLE (m))), (__temp351 = vala_object_creation_expression_get_argument_list (expr)));
		(__temp351 == NULL ? NULL : (__temp351 = (g_list_free (__temp351), NULL)));
		(__temp350 == NULL ? NULL : (__temp350 = (g_list_free (__temp350), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TYPEOF_EXPRESSION (expr));
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->type_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	ValaStruct* __temp353 = NULL;
	ValaStruct* st = (__temp353 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp353 == NULL ? NULL : g_object_ref (__temp353)));
	gboolean __temp354;
	return (__temp354 = vala_struct_is_integer_type (st) || vala_struct_is_floating_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp354);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	ValaStruct* __temp356 = NULL;
	ValaStruct* st = (__temp356 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp356 == NULL ? NULL : g_object_ref (__temp356)));
	gboolean __temp357;
	return (__temp357 = vala_struct_is_integer_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), __temp357);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_unary_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_PLUS || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_MINUS) {
		/* integer or floating point type*/
		if (!vala_semantic_analyzer_is_numeric_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			char* __temp359 = NULL;
			char* __temp358 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp359 = g_strdup_printf ("Operator not supported for `%s'", (__temp358 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
			(__temp359 = (g_free (__temp359), NULL));
			(__temp358 = (g_free (__temp358), NULL));
			return;
		}
		(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	} else {
		if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_LOGICAL_NEGATION) {
			/* boolean type*/
			if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				char* __temp361 = NULL;
				char* __temp360 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp361 = g_strdup_printf ("Operator not supported for `%s'", (__temp360 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
				(__temp361 = (g_free (__temp361), NULL));
				(__temp360 = (g_free (__temp360), NULL));
				return;
			}
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
		} else {
			if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT) {
				/* integer type*/
				if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					char* __temp363 = NULL;
					char* __temp362 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp363 = g_strdup_printf ("Operator not supported for `%s'", (__temp362 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
					(__temp363 = (g_free (__temp363), NULL));
					(__temp362 = (g_free (__temp362), NULL));
					return;
				}
				(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			} else {
				if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_DECREMENT) {
					/* integer type*/
					if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						char* __temp365 = NULL;
						char* __temp364 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp365 = g_strdup_printf ("Operator not supported for `%s'", (__temp364 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
						(__temp365 = (g_free (__temp365), NULL));
						(__temp364 = (g_free (__temp364), NULL));
						return;
					}
					ValaMemberAccess* ma = vala_semantic_analyzer_find_member_access (self, vala_unary_expression_get_inner (expr));
					if (ma == NULL) {
						(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Prefix operators not supported for this expression");
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					ValaMemberAccess* old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
					ValaBinaryExpression* bin = vala_binary_expression_new ((vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT ? VALA_BINARY_OPERATOR_PLUS : VALA_BINARY_OPERATOR_MINUS), VALA_EXPRESSION (old_value), VALA_EXPRESSION (vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("1", NULL)), NULL)), NULL);
					ValaAssignment* assignment = vala_assignment_new (VALA_EXPRESSION (ma), VALA_EXPRESSION (bin), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL);
					vala_code_node_replace (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (assignment));
					vala_code_node_accept (VALA_CODE_NODE (assignment), VALA_CODE_VISITOR (self));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
					return;
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment == NULL ? NULL : (assignment = (g_object_unref (assignment), NULL)));
				} else {
					if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_REF) {
						/* value type*/
						(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
					} else {
						if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_OUT) {
							/* reference type*/
							(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
						} else {
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported unary operator");
							return;
						}
					}
				}
			}
		}
	}
}


static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), NULL);
	if (VALA_IS_PARENTHESIZED_EXPRESSION (expr)) {
		ValaParenthesizedExpression* __temp366 = NULL;
		ValaParenthesizedExpression* pe = (__temp366 = VALA_PARENTHESIZED_EXPRESSION (expr), (__temp366 == NULL ? NULL : g_object_ref (__temp366)));
		ValaMemberAccess* __temp367 = NULL;
		return (__temp367 = vala_semantic_analyzer_find_member_access (self, vala_parenthesized_expression_get_inner (pe)), (pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL))), __temp367);
		(pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaMemberAccess* __temp368 = NULL;
		return (__temp368 = VALA_MEMBER_ACCESS (expr), (__temp368 == NULL ? NULL : g_object_ref (__temp368)));
	}
	return VALA_MEMBER_ACCESS (NULL);
}


static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CAST_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) == NULL && vala_type_reference_get_type_parameter (vala_cast_expression_get_type_reference (expr)) == NULL) {
		return;
	}
	/* if type resolving didn't succeed, skip this check 
	 FIXME: check whether cast is allowed*/
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_cast_expression_get_type_reference (expr)), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_POINTER_INDIRECTION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_pointer_indirection_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (!(VALA_IS_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Pointer indirection not supported for this expression");
		return;
	}
	ValaPointer* __temp371 = NULL;
	ValaPointer* pointer = (__temp371 = VALA_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))), (__temp371 == NULL ? NULL : g_object_ref (__temp371)));
	ValaTypeReference* __temp372 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp372 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp372 == NULL ? NULL : (__temp372 = (g_object_unref (__temp372), NULL)));
	(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_pointer_get_referent_type (pointer)), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(pointer == NULL ? NULL : (pointer = (g_object_unref (pointer), NULL)));
}


static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ADDRESSOF_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_addressof_expression_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))) == NULL) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Address-of operator not supported for this expression");
		return;
	}
	ValaTypeReference* __temp373 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp373 = vala_type_reference_new ())), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp373 == NULL ? NULL : (__temp373 = (g_object_unref (__temp373), NULL)));
	ValaPointer* __temp374 = NULL;
	(vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((__temp374 = vala_data_type_get_pointer (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))))))), vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(__temp374 = (g_object_unref (__temp374), NULL));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)))), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_REFERENCE_TRANSFER_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_reference_transfer_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (!(VALA_IS_MEMBER_ACCESS (vala_reference_transfer_expression_get_inner (expr)) || VALA_IS_ELEMENT_ACCESS (vala_reference_transfer_expression_get_inner (expr)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Reference transfer not supported for this expression");
		return;
	}
	if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "No reference to be transferred");
		return;
	}
	ValaTypeReference* __temp375 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp375 = vala_type_reference_copy (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp375 = (g_object_unref (__temp375), NULL));
	(vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
	(vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr))));
}


static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (left_type), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (right_type), NULL);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (left_type))) || !(VALA_IS_STRUCT (vala_type_reference_get_data_type (right_type)))) {
		return VALA_TYPE_REFERENCE (NULL);
	}
	/* at least one operand not struct*/
	ValaStruct* __temp377 = NULL;
	ValaStruct* left = (__temp377 = VALA_STRUCT (vala_type_reference_get_data_type (left_type)), (__temp377 == NULL ? NULL : g_object_ref (__temp377)));
	ValaStruct* __temp378 = NULL;
	ValaStruct* right = (__temp378 = VALA_STRUCT (vala_type_reference_get_data_type (right_type)), (__temp378 == NULL ? NULL : g_object_ref (__temp378)));
	if ((!vala_struct_is_floating_type (left) && !vala_struct_is_integer_type (left)) || (!vala_struct_is_floating_type (right) && !vala_struct_is_integer_type (right))) {
		gpointer __temp379;
		return VALA_TYPE_REFERENCE ((__temp379 = NULL, (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp379));
	}
	/* at least one operand not numeric*/
	if (vala_struct_is_floating_type (left) == vala_struct_is_floating_type (right)) {
		/* both operands integer or floating type*/
		if (vala_struct_get_rank (left) >= vala_struct_get_rank (right)) {
			ValaTypeReference* __temp380 = NULL;
			return (__temp380 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp380);
		} else {
			ValaTypeReference* __temp381 = NULL;
			return (__temp381 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp381);
		}
	} else {
		/* one integer and one floating type operand*/
		if (vala_struct_is_floating_type (left)) {
			ValaTypeReference* __temp382 = NULL;
			return (__temp382 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp382);
		} else {
			ValaTypeReference* __temp383 = NULL;
			return (__temp383 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), __temp383);
		}
	}
	(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
	(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
}


static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_left (expr))) || vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_right (expr)))) {
		/* if there were any errors in inner expressions, skip type check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS) {
		if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->string_type)) {
			(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be strings");
			return;
		}
		/* string concatenation: convert to a.concat (b) */
		ValaInvocationExpression* concat_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "concat", NULL)), NULL);
		vala_invocation_expression_add_argument (concat_call, vala_binary_expression_get_right (expr));
		vala_code_node_replace (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (concat_call));
		vala_code_node_accept (VALA_CODE_NODE (concat_call), VALA_CODE_VISITOR (self));
		(concat_call == NULL ? NULL : (concat_call = (g_object_unref (concat_call), NULL)));
	} else {
		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MINUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MUL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_DIV) {
			ValaTypeReference* __temp384 = NULL;
			(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp384 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
			(__temp384 == NULL ? NULL : (__temp384 = (g_object_unref (__temp384), NULL)));
			if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
				(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
				char* __temp387 = NULL;
				char* __temp386 = NULL;
				char* __temp385 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp387 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (__temp385 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp386 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
				(__temp387 = (g_free (__temp387), NULL));
				(__temp386 = (g_free (__temp386), NULL));
				(__temp385 = (g_free (__temp385), NULL));
				return;
			}
		} else {
			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MOD || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_LEFT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_RIGHT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_XOR) {
				ValaTypeReference* __temp388 = NULL;
				(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp388 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr))))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
				(__temp388 == NULL ? NULL : (__temp388 = (g_object_unref (__temp388), NULL)));
				if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
					(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
					char* __temp391 = NULL;
					char* __temp390 = NULL;
					char* __temp389 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp391 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (__temp389 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp390 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
					(__temp391 = (g_free (__temp391), NULL));
					(__temp390 = (g_free (__temp390), NULL));
					(__temp389 = (g_free (__temp389), NULL));
					return;
				}
			} else {
				if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL) {
					if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
						/* string comparison: convert to a.collate (b) OP 0 */
						ValaInvocationExpression* cmp_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL)), NULL);
						vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
						(vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call)), vala_binary_expression_get_left (expr));
						ValaLiteralExpression* __temp392 = NULL;
						(vala_binary_expression_set_right (expr, VALA_EXPRESSION ((__temp392 = vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("0", NULL)), NULL)))), vala_binary_expression_get_right (expr));
						(__temp392 == NULL ? NULL : (__temp392 = (g_object_unref (__temp392), NULL)));
						vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
						(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
					} else {
						ValaTypeReference* resulting_type = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)));
						if (resulting_type == NULL) {
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							char* __temp395 = NULL;
							char* __temp394 = NULL;
							char* __temp393 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp395 = g_strdup_printf ("Relational operation not supported for types `%s' and `%s'", (__temp393 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (__temp394 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
							(__temp395 = (g_free (__temp395), NULL));
							(__temp394 = (g_free (__temp394), NULL));
							(__temp393 = (g_free (__temp393), NULL));
							(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
							return;
						}
						(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
					}
					(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
				} else {
					if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY) {
						/* relational operation */
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_right (expr)), vala_expression_get_static_type (vala_binary_expression_get_left (expr))) && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))) {
							char* __temp398 = NULL;
							char* __temp397 = NULL;
							char* __temp396 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp398 = g_strdup_printf ("Equality operation: `%s' and `%s' are incompatible, comparison would always evaluate to false", (__temp396 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))), (__temp397 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))))));
							(__temp398 = (g_free (__temp398), NULL));
							(__temp397 = (g_free (__temp397), NULL));
							(__temp396 = (g_free (__temp396), NULL));
							(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
							return;
						}
						if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
							/* string comparison: convert to a.collate (b) OP 0 */
							ValaInvocationExpression* cmp_call = vala_invocation_expression_new (VALA_EXPRESSION (vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL)), NULL);
							vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
							(vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call)), vala_binary_expression_get_left (expr));
							ValaLiteralExpression* __temp399 = NULL;
							(vala_binary_expression_set_right (expr, VALA_EXPRESSION ((__temp399 = vala_literal_expression_new (VALA_LITERAL (vala_integer_literal_new ("0", NULL)), NULL)))), vala_binary_expression_get_right (expr));
							(__temp399 == NULL ? NULL : (__temp399 = (g_object_unref (__temp399), NULL)));
							vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
							(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
						}
						(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
					} else {
						if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_OR) {
							/* integer type or flags type*/
							(vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_binary_expression_get_left (expr))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
						} else {
							if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_OR) {
								if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) != vala_type_reference_get_data_type (self->priv->bool_type) || vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be boolean");
								}
								(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
							} else {
								g_assert_not_reached ();
							}
						}
					}
				}
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TYPE_CHECK (expr));
	if (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr)) == NULL) {
		/* if type resolving didn't succeed, skip this check */
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr)))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
}


static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GList* types)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	gboolean null_found = FALSE;
	gboolean class_or_iface_found = FALSE;
	gboolean type_param_found = FALSE;
	gboolean ref_struct_found = FALSE;
	gboolean val_struct_found = FALSE;
	gboolean enum_found = FALSE;
	gboolean callback_found = FALSE;
	ValaTypeReference* base_type = NULL;
	ValaTypeReference* last_type = NULL;
	{
		GList* __temp410 = NULL;
		__temp410 = types;
		GList* type_it;
		for (type_it = __temp410; type_it != NULL; type_it = type_it->next) {
			ValaTypeReference* type = type_it->data;
			{
				ValaTypeReference* __temp401 = NULL;
				ValaTypeReference* __temp400 = NULL;
				last_type = (__temp401 = (__temp400 = type, (__temp400 == NULL ? NULL : g_object_ref (__temp400))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp401);
				if (vala_code_node_get_error (VALA_CODE_NODE (type))) {
					ValaTypeReference* __temp402 = NULL;
					base_type = (__temp402 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp402);
					(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
					ValaTypeReference* __temp403 = NULL;
					return (__temp403 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp403);
				}
				if (vala_type_reference_get_data_type (type) == NULL && vala_type_reference_get_type_parameter (type) == NULL) {
					if (!null_found) {
						null_found = TRUE;
						if (val_struct_found || enum_found) {
							(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
							break;
						}
					}
				} else {
					if (VALA_IS_CLASS (vala_type_reference_get_data_type (type)) || VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
						if (!class_or_iface_found) {
							class_or_iface_found = TRUE;
							if (type_param_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
								(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
								break;
							}
						}
					} else {
						if (vala_type_reference_get_type_parameter (type) != NULL) {
							if (!type_param_found) {
								type_param_found = TRUE;
								if (class_or_iface_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
									(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
									break;
								}
							}
						} else {
							if (VALA_IS_STRUCT (vala_type_reference_get_data_type (type))) {
								ValaStruct* __temp404 = NULL;
								ValaStruct* st = (__temp404 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (__temp404 == NULL ? NULL : g_object_ref (__temp404)));
								if (vala_data_type_is_reference_type (VALA_DATA_TYPE (st))) {
									if (!ref_struct_found) {
										ref_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || val_struct_found || enum_found || callback_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								} else {
									if (!val_struct_found) {
										val_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || enum_found || callback_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								}
								(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
							} else {
								if (VALA_IS_ENUM (vala_type_reference_get_data_type (type))) {
									if (!enum_found) {
										enum_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found) {
											(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
											break;
										}
									}
								} else {
									if (VALA_IS_CALLBACK (vala_type_reference_get_data_type (type))) {
										if (!callback_found) {
											callback_found = TRUE;
											if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found || enum_found) {
												(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
												break;
											}
										}
									} else {
										ValaTypeReference* __temp405 = NULL;
										base_type = (__temp405 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp405);
										(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
										char* __temp407 = NULL;
										char* __temp406 = NULL;
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (type)), (__temp407 = g_strdup_printf ("internal error: unsupported type `%s'", (__temp406 = vala_type_reference_to_string (type)))));
										(__temp407 = (g_free (__temp407), NULL));
										(__temp406 = (g_free (__temp406), NULL));
										ValaTypeReference* __temp408 = NULL;
										return (__temp408 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp408);
									}
								}
							}
						}
					}
				}
				if (base_type == NULL) {
					ValaTypeReference* __temp409 = NULL;
					base_type = (__temp409 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), __temp409);
					(vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type)), vala_type_reference_get_data_type (base_type));
					(vala_type_reference_set_type_parameter (base_type, vala_type_reference_get_type_parameter (type)), vala_type_reference_get_type_parameter (base_type));
					(vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (type)), vala_type_reference_get_non_null (base_type));
					(vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (type)), vala_type_reference_get_is_null (base_type));
					(vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (type)), vala_type_reference_get_transfers_ownership (base_type));
				} else {
					if (vala_type_reference_get_data_type (base_type) != vala_type_reference_get_data_type (type)) {
						if (vala_semantic_analyzer_is_type_compatible (self, type, base_type)) {
						} else {
							if (vala_semantic_analyzer_is_type_compatible (self, base_type, type)) {
								(vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type)), vala_type_reference_get_data_type (base_type));
							} else {
								(vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE), vala_code_node_get_error (VALA_CODE_NODE (base_type)));
								break;
							}
						}
					}
					(vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (base_type) && vala_type_reference_get_non_null (type)), vala_type_reference_get_non_null (base_type));
					(vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (base_type) && vala_type_reference_get_is_null (type)), vala_type_reference_get_is_null (base_type));
					/* if one subexpression transfers ownership, all subexpressions must transfer ownership
					 FIXME add ref calls to subexpressions that don't transfer ownership*/
					(vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (base_type) || vala_type_reference_get_transfers_ownership (type)), vala_type_reference_get_transfers_ownership (base_type));
				}
			}
		}
	}
	if (base_type != NULL && vala_code_node_get_error (VALA_CODE_NODE (base_type))) {
		char* __temp413 = NULL;
		char* __temp412 = NULL;
		char* __temp411 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (last_type)), (__temp413 = g_strdup_printf ("`%s' is incompatible with `%s'", (__temp411 = vala_type_reference_to_string (last_type)), (__temp412 = vala_type_reference_to_string (base_type)))));
		(__temp413 = (g_free (__temp413), NULL));
		(__temp412 = (g_free (__temp412), NULL));
		(__temp411 = (g_free (__temp411), NULL));
	}
	ValaTypeReference* __temp414 = NULL;
	return (__temp414 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), __temp414);
	(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
	(last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONDITIONAL_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_conditional_expression_get_condition (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_conditional_expression_get_condition (expr))), "Condition must be boolean");
		return;
	}
	/* FIXME: support memory management */
	GList* types = NULL;
	ValaTypeReference* __temp415 = NULL;
	types = g_list_append (types, (__temp415 = vala_expression_get_static_type (vala_conditional_expression_get_true_expression (expr)), (__temp415 == NULL ? NULL : g_object_ref (__temp415))));
	ValaTypeReference* __temp416 = NULL;
	types = g_list_append (types, (__temp416 = vala_expression_get_static_type (vala_conditional_expression_get_false_expression (expr)), (__temp416 == NULL ? NULL : g_object_ref (__temp416))));
	ValaTypeReference* __temp417 = NULL;
	(vala_expression_set_static_type (VALA_EXPRESSION (expr), (__temp417 = vala_semantic_analyzer_compute_common_base_type (self, types))), vala_expression_get_static_type (VALA_EXPRESSION (expr)));
	(__temp417 == NULL ? NULL : (__temp417 = (g_object_unref (__temp417), NULL)));
	(types == NULL ? NULL : (types = (g_list_foreach (types, (GFunc) g_object_unref, NULL), g_list_free (types), NULL)));
}


static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	char* result = g_strdup_printf ("__lambda%d", self->priv->next_lambda_id);
	self->priv->next_lambda_id++;
	return result;
	(result = (g_free (result), NULL));
}


static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	ValaSymbol* __temp419 = NULL;
	ValaSymbol* sym = (__temp419 = self->priv->current_symbol, (__temp419 == NULL ? NULL : g_object_ref (__temp419)));
	while (sym != NULL) {
		if (VALA_IS_METHOD (vala_symbol_get_node (sym))) {
			ValaMethod* __temp420 = NULL;
			ValaMethod* __temp421 = NULL;
			return (__temp421 = (__temp420 = VALA_METHOD (vala_symbol_get_node (sym)), (__temp420 == NULL ? NULL : g_object_ref (__temp420))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp421);
		}
		ValaSymbol* __temp423 = NULL;
		ValaSymbol* __temp422 = NULL;
		sym = (__temp423 = (__temp422 = vala_symbol_get_parent_symbol (sym), (__temp422 == NULL ? NULL : g_object_ref (__temp422))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp423);
	}
	gpointer __temp424;
	return VALA_METHOD ((__temp424 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp424));
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	ValaSymbol* __temp425 = NULL;
	ValaSymbol* sym = (__temp425 = self->priv->current_symbol, (__temp425 == NULL ? NULL : g_object_ref (__temp425)));
	while (sym != NULL) {
		if (VALA_IS_CONSTRUCTOR (vala_symbol_get_node (sym))) {
			gboolean __temp426;
			return (__temp426 = TRUE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp426);
		}
		ValaSymbol* __temp428 = NULL;
		ValaSymbol* __temp427 = NULL;
		sym = (__temp428 = (__temp427 = vala_symbol_get_parent_symbol (sym), (__temp427 == NULL ? NULL : g_object_ref (__temp427))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp428);
	}
	gboolean __temp429;
	return (__temp429 = FALSE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), __temp429);
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (l));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (l)) == NULL || !(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (l)))))) {
		(vala_code_node_set_error (VALA_CODE_NODE (l), TRUE), vala_code_node_get_error (VALA_CODE_NODE (l)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression not allowed in this context");
		return;
	}
	gboolean in_instance_method = FALSE;
	ValaMethod* current_method = vala_semantic_analyzer_find_current_method (self);
	if (current_method != NULL) {
		in_instance_method = vala_method_get_instance (current_method);
	} else {
		in_instance_method = vala_semantic_analyzer_is_in_constructor (self);
	}
	ValaCallback* __temp430 = NULL;
	ValaCallback* cb = (__temp430 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (l)))), (__temp430 == NULL ? NULL : g_object_ref (__temp430)));
	ValaMethod* __temp431 = NULL;
	(vala_lambda_expression_set_method (l, (__temp431 = vala_method_new (vala_semantic_analyzer_get_lambda_name (self), vala_callback_get_return_type (cb), NULL))), vala_lambda_expression_get_method (l));
	(__temp431 == NULL ? NULL : (__temp431 = (g_object_unref (__temp431), NULL)));
	(vala_method_set_instance (vala_lambda_expression_get_method (l), vala_callback_get_instance (cb) && in_instance_method), vala_method_get_instance (vala_lambda_expression_get_method (l)));
	ValaSymbol* __temp432 = NULL;
	(vala_code_node_set_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)), (__temp432 = vala_symbol_new (VALA_CODE_NODE (vala_lambda_expression_get_method (l))))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))));
	(__temp432 == NULL ? NULL : (__temp432 = (g_object_unref (__temp432), NULL)));
	(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))), self->priv->current_symbol), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))));
	GList* lambda_params = vala_lambda_expression_get_parameters (l);
	GList* lambda_param_it = lambda_params;
	{
		GList* __temp435 = NULL;
		__temp435 = vala_callback_get_parameters (cb);
		GList* cb_param_it;
		for (cb_param_it = __temp435; cb_param_it != NULL; cb_param_it = cb_param_it->next) {
			ValaFormalParameter* cb_param = cb_param_it->data;
			{
				if (lambda_param_it == NULL) {
					break;
				}
				/* lambda expressions are allowed to have less parameters */
				const char* __temp433 = NULL;
				char* lambda_param = (__temp433 = ((char*) lambda_param_it->data), (__temp433 == NULL ? NULL : g_strdup (__temp433)));
				ValaFormalParameter* param = vala_formal_parameter_new (lambda_param, vala_formal_parameter_get_type_reference (cb_param), NULL);
				ValaSymbol* __temp434 = NULL;
				(vala_code_node_set_symbol (VALA_CODE_NODE (param), (__temp434 = vala_symbol_new (VALA_CODE_NODE (param)))), vala_code_node_get_symbol (VALA_CODE_NODE (param)));
				(__temp434 == NULL ? NULL : (__temp434 = (g_object_unref (__temp434), NULL)));
				vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l))), vala_formal_parameter_get_name (param), vala_code_node_get_symbol (VALA_CODE_NODE (param)));
				vala_method_add_parameter (vala_lambda_expression_get_method (l), param);
				lambda_param_it = lambda_param_it->next;
				(lambda_param = (g_free (lambda_param), NULL));
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(__temp435 == NULL ? NULL : (__temp435 = (g_list_free (__temp435), NULL)));
	}
	if (lambda_param_it != NULL) {
		/* lambda expressions may not expect more parameters */
		(vala_code_node_set_error (VALA_CODE_NODE (l), TRUE), vala_code_node_get_error (VALA_CODE_NODE (l)));
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression: too many parameters");
		(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
		(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
		(lambda_params == NULL ? NULL : (lambda_params = (g_list_free (lambda_params), NULL)));
		return;
	}
	if (vala_lambda_expression_get_expression_body (l) != NULL) {
		ValaBlock* block = vala_block_new (NULL);
		ValaSymbol* __temp436 = NULL;
		(vala_code_node_set_symbol (VALA_CODE_NODE (block), (__temp436 = vala_symbol_new (VALA_CODE_NODE (block)))), vala_code_node_get_symbol (VALA_CODE_NODE (block)));
		(__temp436 == NULL ? NULL : (__temp436 = (g_object_unref (__temp436), NULL)));
		(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (block)), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (block))));
		if (vala_type_reference_get_data_type (vala_method_get_return_type (vala_lambda_expression_get_method (l))) != NULL) {
			ValaReturnStatement* __temp437 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((__temp437 = vala_return_statement_new (vala_lambda_expression_get_expression_body (l), NULL))));
			(__temp437 == NULL ? NULL : (__temp437 = (g_object_unref (__temp437), NULL)));
		} else {
			ValaExpressionStatement* __temp438 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((__temp438 = vala_expression_statement_new (vala_lambda_expression_get_expression_body (l), NULL))));
			(__temp438 == NULL ? NULL : (__temp438 = (g_object_unref (__temp438), NULL)));
		}
		(vala_method_set_body (vala_lambda_expression_get_method (l), block), vala_method_get_body (vala_lambda_expression_get_method (l)));
		(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
	} else {
		(vala_method_set_body (vala_lambda_expression_get_method (l), vala_lambda_expression_get_statement_body (l)), vala_method_get_body (vala_lambda_expression_get_method (l)));
		(vala_symbol_set_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_method_get_body (vala_lambda_expression_get_method (l)))), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (vala_method_get_body (vala_lambda_expression_get_method (l))))));
	}
	/* lambda expressions should be usable like MemberAccess of a method */
	(vala_expression_set_symbol_reference (VALA_EXPRESSION (l), vala_code_node_get_symbol (VALA_CODE_NODE (vala_lambda_expression_get_method (l)))), vala_expression_get_symbol_reference (VALA_EXPRESSION (l)));
	(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
	(lambda_params == NULL ? NULL : (lambda_params = (g_list_free (lambda_params), NULL)));
}


static void vala_semantic_analyzer_real_visit_begin_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp439 = NULL;
		ValaMemberAccess* ma = (__temp439 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp439 == NULL ? NULL : g_object_ref (__temp439)));
		if (vala_code_node_get_error (VALA_CODE_NODE (ma)) || vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)) == NULL) {
			(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		/* if no symbol found, skip this check */
		if (VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
			ValaSignal* __temp440 = NULL;
			ValaSignal* sig = (__temp440 = VALA_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp440 == NULL ? NULL : g_object_ref (__temp440)));
			ValaTypeReference* __temp441 = NULL;
			(vala_expression_set_expected_type (vala_assignment_get_right (a), (__temp441 = vala_type_reference_new ())), vala_expression_get_expected_type (vala_assignment_get_right (a)));
			(__temp441 == NULL ? NULL : (__temp441 = (g_object_unref (__temp441), NULL)));
			ValaCallback* __temp442 = NULL;
			(vala_type_reference_set_data_type (vala_expression_get_expected_type (vala_assignment_get_right (a)), VALA_DATA_TYPE ((__temp442 = vala_signal_get_callback (sig)))), vala_type_reference_get_data_type (vala_expression_get_expected_type (vala_assignment_get_right (a))));
			(__temp442 = (g_object_unref (__temp442), NULL));
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
		} else {
			/* do nothing*/
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a))) {
			} else {
				/* do nothing*/
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "unsupported lvalue in assignment");
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (vala_code_node_get_error (VALA_CODE_NODE (a)) || vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_left (a))) || vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_right (a)))) {
		(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
		return;
	}
	if (vala_assignment_get_operator (a) != VALA_ASSIGNMENT_OPERATOR_SIMPLE && VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		ValaMemberAccess* __temp443 = NULL;
		ValaMemberAccess* ma = (__temp443 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp443 == NULL ? NULL : g_object_ref (__temp443)));
		if (!(VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))))) {
			ValaMemberAccess* old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			ValaBinaryExpression* bin = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, VALA_EXPRESSION (old_value), VALA_EXPRESSION (vala_parenthesized_expression_new (vala_assignment_get_right (a), vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))))), NULL);
			if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR), vala_binary_expression_get_operator (bin));
			} else {
				if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND), vala_binary_expression_get_operator (bin));
				} else {
					if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR), vala_binary_expression_get_operator (bin));
					} else {
						if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_ADD) {
							(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS), vala_binary_expression_get_operator (bin));
						} else {
							if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SUB) {
								(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS), vala_binary_expression_get_operator (bin));
							} else {
								if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_MUL) {
									(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL), vala_binary_expression_get_operator (bin));
								} else {
									if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_DIV) {
										(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV), vala_binary_expression_get_operator (bin));
									} else {
										if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD), vala_binary_expression_get_operator (bin));
										} else {
											if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT), vala_binary_expression_get_operator (bin));
											} else {
												if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													(vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT), vala_binary_expression_get_operator (bin));
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			(vala_assignment_set_right (a, VALA_EXPRESSION (bin)), vala_assignment_get_right (a));
			vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (a)), VALA_CODE_VISITOR (self));
			(vala_assignment_set_operator (a, VALA_ASSIGNMENT_OPERATOR_SIMPLE), vala_assignment_get_operator (a));
			(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
			(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* __temp444 = NULL;
		ValaMemberAccess* ma = (__temp444 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (__temp444 == NULL ? NULL : g_object_ref (__temp444)));
		if (VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
			ValaSignal* __temp445 = NULL;
			ValaSignal* sig = (__temp445 = VALA_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp445 == NULL ? NULL : g_object_ref (__temp445)));
			if (vala_expression_get_symbol_reference (vala_assignment_get_right (a)) == NULL) {
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "unsupported expression for signal handler");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			ValaMethod* __temp446 = NULL;
			ValaMethod* m = (__temp446 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_assignment_get_right (a)))), (__temp446 == NULL ? NULL : g_object_ref (__temp446)));
			if (vala_method_get_instance (m) && m->access != VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
				/* TODO: generate wrapper function */
				(vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE), vala_code_node_get_error (VALA_CODE_NODE (ma)));
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "public instance methods not yet supported as signal handlers");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			if (vala_method_get_instance (m)) {
				/* instance signal handlers must have the self
				 * parameter at the end
				 * do not use G_CONNECT_SWAPPED as this would
				 * rearrange the parameters for instance
				 * methods and non-instance methods
				 */
				(vala_method_set_instance_last (m, TRUE), vala_method_get_instance_last (m));
			}
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			if (VALA_IS_PROPERTY (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
				ValaProperty* __temp447 = NULL;
				ValaProperty* prop = (__temp447 = VALA_PROPERTY (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp447 == NULL ? NULL : g_object_ref (__temp447)));
				if (vala_property_get_set_accessor (prop) == NULL) {
					(vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE), vala_code_node_get_error (VALA_CODE_NODE (ma)));
					char* __temp449 = NULL;
					char* __temp448 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ma)), (__temp449 = g_strdup_printf ("Property `%s' is read-only", (__temp448 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (prop)))))));
					(__temp449 = (g_free (__temp449), NULL));
					(__temp448 = (g_free (__temp448), NULL));
					(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					return;
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			} else {
				if (VALA_IS_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
					ValaVariableDeclarator* __temp450 = NULL;
					ValaVariableDeclarator* decl = (__temp450 = VALA_VARIABLE_DECLARATOR (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp450 == NULL ? NULL : g_object_ref (__temp450)));
					ValaMemberAccess* __temp451 = NULL;
					ValaMemberAccess* right_ma = (__temp451 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (__temp451 == NULL ? NULL : g_object_ref (__temp451)));
					if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
						ValaMethod* __temp452 = NULL;
						ValaMethod* m = (__temp452 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))), (__temp452 == NULL ? NULL : g_object_ref (__temp452)));
						ValaCallback* __temp453 = NULL;
						ValaCallback* cb = (__temp453 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp453 == NULL ? NULL : g_object_ref (__temp453)));
						/* check whether method matches callback type */
						if (!vala_callback_matches_method (cb, m)) {
							(vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE), vala_code_node_get_error (VALA_CODE_NODE (decl)));
							char* __temp456 = NULL;
							char* __temp455 = NULL;
							char* __temp454 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp456 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp454 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp455 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
							(__temp456 = (g_free (__temp456), NULL));
							(__temp455 = (g_free (__temp455), NULL));
							(__temp454 = (g_free (__temp454), NULL));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(vala_expression_set_static_type (vala_assignment_get_right (a), vala_variable_declarator_get_type_reference (decl)), vala_expression_get_static_type (vala_assignment_get_right (a)));
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					} else {
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
						(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
					(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
				} else {
					if (VALA_IS_FIELD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
						ValaField* __temp457 = NULL;
						ValaField* f = (__temp457 = VALA_FIELD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))), (__temp457 == NULL ? NULL : g_object_ref (__temp457)));
						ValaMemberAccess* __temp458 = NULL;
						ValaMemberAccess* right_ma = (__temp458 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (__temp458 == NULL ? NULL : g_object_ref (__temp458)));
						if (VALA_IS_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))) {
							ValaMethod* __temp459 = NULL;
							ValaMethod* m = (__temp459 = VALA_METHOD (vala_symbol_get_node (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma)))), (__temp459 == NULL ? NULL : g_object_ref (__temp459)));
							ValaCallback* __temp460 = NULL;
							ValaCallback* cb = (__temp460 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), (__temp460 == NULL ? NULL : g_object_ref (__temp460)));
							/* check whether method matches callback type */
							if (!vala_callback_matches_method (cb, m)) {
								(vala_code_node_set_error (VALA_CODE_NODE (f), TRUE), vala_code_node_get_error (VALA_CODE_NODE (f)));
								char* __temp463 = NULL;
								char* __temp462 = NULL;
								char* __temp461 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp463 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (__temp461 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (m)))), (__temp462 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (cb)))))));
								(__temp463 = (g_free (__temp463), NULL));
								(__temp462 = (g_free (__temp462), NULL));
								(__temp461 = (g_free (__temp461), NULL));
								(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
								(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
								(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							(vala_expression_set_static_type (vala_assignment_get_right (a), vala_field_get_type_reference (f)), vala_expression_get_static_type (vala_assignment_get_right (a)));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						} else {
							(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
					} else {
						if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL && vala_expression_get_static_type (vala_assignment_get_right (a)) != NULL) {
							if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
								/* if there was an error on either side,
								 * i.e. a.{left|right}.static_type == null, skip type check */
								(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
								char* __temp466 = NULL;
								char* __temp465 = NULL;
								char* __temp464 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp466 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (__temp464 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (__temp465 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
								(__temp466 = (g_free (__temp466), NULL));
								(__temp465 = (g_free (__temp465), NULL));
								(__temp464 = (g_free (__temp464), NULL));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							if (vala_semantic_analyzer_get_memory_management (self)) {
								if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
									/* rhs transfers ownership of the expression */
									if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
										/* lhs doesn't own the value */
										(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
									}
								} else {
									if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
									}
								}
							}
						}
					}
				}
			}
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		/* lhs wants to own the value
		 * rhs doesn't transfer the ownership
		 * code generator needs to add reference
		 * increment calls */
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
			ValaElementAccess* __temp467 = NULL;
			ValaElementAccess* ea = (__temp467 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (a)), (__temp467 == NULL ? NULL : g_object_ref (__temp467)));
			if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
				/* if there was an error on either side,
				 * i.e. a.{left|right}.static_type == null, skip type check */
				(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
				char* __temp470 = NULL;
				char* __temp469 = NULL;
				char* __temp468 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (__temp470 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (__temp468 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (__temp469 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
				(__temp470 = (g_free (__temp470), NULL));
				(__temp469 = (g_free (__temp469), NULL));
				(__temp468 = (g_free (__temp468), NULL));
				(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
				return;
			}
			if (vala_semantic_analyzer_get_memory_management (self)) {
				if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
					/* rhs transfers ownership of the expression */
					GList* args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (ea)));
					if (g_list_length (args) != 1) {
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ea)), "internal error: array reference without type arguments");
						(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					ValaTypeReference* __temp471 = NULL;
					ValaTypeReference* element_type = (__temp471 = VALA_TYPE_REFERENCE (args->data), (__temp471 == NULL ? NULL : g_object_ref (__temp471)));
					if (!vala_type_reference_get_takes_ownership (element_type)) {
						/* lhs doesn't own the value */
						(vala_code_node_set_error (VALA_CODE_NODE (a), TRUE), vala_code_node_get_error (VALA_CODE_NODE (a)));
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
						(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
						(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					(args == NULL ? NULL : (args = (g_list_free (args), NULL)));
					(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
				} else {
					if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
					}
				}
			}
			(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
		} else {
			return;
		}
	}
	/* lhs wants to own the value
	 * rhs doesn't transfer the ownership
	 * code generator needs to add reference
	 * increment calls */
	(vala_expression_set_static_type (VALA_EXPRESSION (a), vala_expression_get_static_type (vala_assignment_get_left (a))), vala_expression_get_static_type (VALA_EXPRESSION (a)));
}


gboolean vala_semantic_analyzer_get_memory_management (ValaSemanticAnalyzer* self)
{
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	return self->priv->_memory_management;
}


void vala_semantic_analyzer_set_memory_management (ValaSemanticAnalyzer* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	self->priv->_memory_management = value;
}


static void vala_semantic_analyzer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		g_value_set_boolean (value, vala_semantic_analyzer_get_memory_management (self));
		break;
	}
}


static void vala_semantic_analyzer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		vala_semantic_analyzer_set_memory_management (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_semantic_analyzer_class_init (ValaSemanticAnalyzerClass * klass)
{
	vala_semantic_analyzer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSemanticAnalyzerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_semantic_analyzer_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_semantic_analyzer_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_semantic_analyzer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_source_file = vala_semantic_analyzer_real_visit_begin_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_source_file = vala_semantic_analyzer_real_visit_end_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_namespace = vala_semantic_analyzer_real_visit_begin_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_namespace = vala_semantic_analyzer_real_visit_end_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_class = vala_semantic_analyzer_real_visit_begin_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_class = vala_semantic_analyzer_real_visit_end_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_struct = vala_semantic_analyzer_real_visit_begin_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_struct = vala_semantic_analyzer_real_visit_end_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_interface = vala_semantic_analyzer_real_visit_begin_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_interface = vala_semantic_analyzer_real_visit_end_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_semantic_analyzer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_semantic_analyzer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_method = vala_semantic_analyzer_real_visit_begin_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_method = vala_semantic_analyzer_real_visit_end_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_creation_method = vala_semantic_analyzer_real_visit_begin_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_creation_method = vala_semantic_analyzer_real_visit_end_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_semantic_analyzer_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_property = vala_semantic_analyzer_real_visit_end_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_property_accessor = vala_semantic_analyzer_real_visit_begin_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_property_accessor = vala_semantic_analyzer_real_visit_end_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_constructor = vala_semantic_analyzer_real_visit_begin_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_constructor = vala_semantic_analyzer_real_visit_end_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_destructor = vala_semantic_analyzer_real_visit_begin_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_destructor = vala_semantic_analyzer_real_visit_end_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_semantic_analyzer_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_block = vala_semantic_analyzer_real_visit_begin_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_block = vala_semantic_analyzer_real_visit_end_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_semantic_analyzer_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_initializer_list = vala_semantic_analyzer_real_visit_begin_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_initializer_list = vala_semantic_analyzer_real_visit_end_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_semantic_analyzer_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_semantic_analyzer_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_semantic_analyzer_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_semantic_analyzer_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_foreach_statement = vala_semantic_analyzer_real_visit_begin_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_foreach_statement = vala_semantic_analyzer_real_visit_end_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_return_statement = vala_semantic_analyzer_real_visit_end_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_semantic_analyzer_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_array_creation_expression = vala_semantic_analyzer_real_visit_begin_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_array_creation_expression = vala_semantic_analyzer_real_visit_end_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_semantic_analyzer_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_semantic_analyzer_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_semantic_analyzer_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_semantic_analyzer_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_semantic_analyzer_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_semantic_analyzer_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_literal_expression = vala_semantic_analyzer_real_visit_literal_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_semantic_analyzer_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_semantic_analyzer_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_invocation_expression = vala_semantic_analyzer_real_visit_begin_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_invocation_expression = vala_semantic_analyzer_real_visit_end_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_semantic_analyzer_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_semantic_analyzer_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_semantic_analyzer_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_object_creation_expression = vala_semantic_analyzer_real_visit_end_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_semantic_analyzer_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_semantic_analyzer_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_semantic_analyzer_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_semantic_analyzer_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_semantic_analyzer_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_semantic_analyzer_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_semantic_analyzer_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_semantic_analyzer_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_semantic_analyzer_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_lambda_expression = vala_semantic_analyzer_real_visit_begin_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_assignment = vala_semantic_analyzer_real_visit_begin_assignment;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_assignment = vala_semantic_analyzer_real_visit_end_assignment;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT, g_param_spec_boolean ("memory-management", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_semantic_analyzer_init (ValaSemanticAnalyzer * self)
{
	self->priv = VALA_SEMANTIC_ANALYZER_GET_PRIVATE (self);
	self->priv->next_lambda_id = 0;
}


static void vala_semantic_analyzer_dispose (GObject * obj)
{
	ValaSemanticAnalyzer * self = VALA_SEMANTIC_ANALYZER (obj);
	(self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL)));
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL)));
	(self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL)));
	(self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL)));
	(self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_list_free (self->priv->current_using_directives), NULL)));
	(self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL)));
	(self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL)));
	(self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL)));
	(self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL)));
	(self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL)));
	(self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL)));
	(self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL)));
	(self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL)));
	(self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL)));
	ValaSemanticAnalyzerClass * klass;
	GObjectClass * parent_class;
	klass = VALA_SEMANTIC_ANALYZER_CLASS (g_type_class_peek (VALA_TYPE_SEMANTIC_ANALYZER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_semantic_analyzer_get_type ()
{
	static GType vala_semantic_analyzer_type_id = 0;
	if (G_UNLIKELY (vala_semantic_analyzer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSemanticAnalyzerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_semantic_analyzer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSemanticAnalyzer), 0, (GInstanceInitFunc) vala_semantic_analyzer_init };
		vala_semantic_analyzer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaSemanticAnalyzer", &g_define_type_info, 0);
	}
	return vala_semantic_analyzer_type_id;
}




