/* valaswitchsection.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaswitchsection.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valaswitchlabel.h>
#include <vala/valastatement.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaSwitchSectionPrivate {
	GeeList* labels;
	GeeList* statement_list;
};
#define VALA_SWITCH_SECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SWITCH_SECTION, ValaSwitchSectionPrivate))
enum  {
	VALA_SWITCH_SECTION_DUMMY_PROPERTY
};
static void vala_switch_section_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_switch_section_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_switch_section_parent_class = NULL;
static void vala_switch_section_dispose (GObject * obj);


/**
 * Creates a new switch section.
 *
 * @param source_reference reference to source code
 * @return                 newly created switch section
 */
ValaSwitchSection* vala_switch_section_new (ValaSourceReference* source_reference) {
	ValaSwitchSection * self;
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_SWITCH_SECTION, 0, NULL);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified label to the list of switch labels.
 *
 * @param label a switch label
 */
void vala_switch_section_add_label (ValaSwitchSection* self, ValaSwitchLabel* label) {
	g_return_if_fail (VALA_IS_SWITCH_SECTION (self));
	g_return_if_fail (VALA_IS_SWITCH_LABEL (label));
	gee_collection_add (GEE_COLLECTION (self->priv->labels), label);
}


/**
 * Returns a copy of the list of switch labels.
 *
 * @return switch label list
 */
GeeCollection* vala_switch_section_get_labels (ValaSwitchSection* self) {
	g_return_val_if_fail (VALA_IS_SWITCH_SECTION (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_SWITCH_LABEL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->labels)));
}


gboolean vala_switch_section_has_default_label (ValaSwitchSection* self) {
	g_return_val_if_fail (VALA_IS_SWITCH_SECTION (self), FALSE);
	{
		GeeList* label_collection;
		GeeIterator* label_it;
		label_collection = self->priv->labels;
		label_it = gee_iterable_iterator (GEE_ITERABLE (label_collection));
		while (gee_iterator_next (label_it)) {
			ValaSwitchLabel* label;
			label = gee_iterator_get (label_it);
			{
				if (vala_switch_label_get_expression (label) == NULL) {
					gboolean _tmp0;
					return (_tmp0 = TRUE, (label == NULL ? NULL : (label = (g_object_unref (label), NULL))), (label_it == NULL ? NULL : (label_it = (g_object_unref (label_it), NULL))), _tmp0);
				}
				(label == NULL ? NULL : (label = (g_object_unref (label), NULL)));
			}
		}
		(label_it == NULL ? NULL : (label_it = (g_object_unref (label_it), NULL)));
	}
	return FALSE;
}


/**
 * Appends the specified statement to this switch section.
 *
 * @param stmt a statement
 */
void vala_switch_section_add_statement (ValaSwitchSection* self, ValaStatement* stmt) {
	g_return_if_fail (VALA_IS_SWITCH_SECTION (self));
	g_return_if_fail (VALA_IS_STATEMENT (stmt));
	gee_collection_add (GEE_COLLECTION (self->priv->statement_list), stmt);
}


/**
 * Returns a copy of the list of statements.
 *
 * @return statement list
 */
GeeCollection* vala_switch_section_get_statements (ValaSwitchSection* self) {
	g_return_val_if_fail (VALA_IS_SWITCH_SECTION (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_STATEMENT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->statement_list)));
}


static void vala_switch_section_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSwitchSection * self;
	self = VALA_SWITCH_SECTION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_switch_section (visitor, self);
}


static void vala_switch_section_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaSwitchSection * self;
	self = VALA_SWITCH_SECTION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* label_collection;
		GeeIterator* label_it;
		label_collection = self->priv->labels;
		label_it = gee_iterable_iterator (GEE_ITERABLE (label_collection));
		while (gee_iterator_next (label_it)) {
			ValaSwitchLabel* label;
			label = gee_iterator_get (label_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (label), visitor);
				(label == NULL ? NULL : (label = (g_object_unref (label), NULL)));
			}
		}
		(label_it == NULL ? NULL : (label_it = (g_object_unref (label_it), NULL)));
	}
	{
		GeeList* st_collection;
		GeeIterator* st_it;
		st_collection = self->priv->statement_list;
		st_it = gee_iterable_iterator (GEE_ITERABLE (st_collection));
		while (gee_iterator_next (st_it)) {
			ValaStatement* st;
			st = gee_iterator_get (st_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (st), visitor);
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			}
		}
		(st_it == NULL ? NULL : (st_it = (g_object_unref (st_it), NULL)));
	}
}


static void vala_switch_section_class_init (ValaSwitchSectionClass * klass) {
	vala_switch_section_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSwitchSectionPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_switch_section_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_switch_section_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_switch_section_real_accept_children;
}


static void vala_switch_section_init (ValaSwitchSection * self) {
	self->priv = VALA_SWITCH_SECTION_GET_PRIVATE (self);
	self->priv->labels = GEE_LIST (gee_array_list_new (VALA_TYPE_SWITCH_LABEL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->statement_list = GEE_LIST (gee_array_list_new (VALA_TYPE_STATEMENT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_switch_section_dispose (GObject * obj) {
	ValaSwitchSection * self;
	self = VALA_SWITCH_SECTION (obj);
	(self->priv->labels == NULL ? NULL : (self->priv->labels = (g_object_unref (self->priv->labels), NULL)));
	(self->priv->statement_list == NULL ? NULL : (self->priv->statement_list = (g_object_unref (self->priv->statement_list), NULL)));
	G_OBJECT_CLASS (vala_switch_section_parent_class)->dispose (obj);
}


GType vala_switch_section_get_type (void) {
	static GType vala_switch_section_type_id = 0;
	if (G_UNLIKELY (vala_switch_section_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSwitchSectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_switch_section_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSwitchSection), 0, (GInstanceInitFunc) vala_switch_section_init };
		vala_switch_section_type_id = g_type_register_static (VALA_TYPE_BLOCK, "ValaSwitchSection", &g_define_type_info, 0);
	}
	return vala_switch_section_type_id;
}




