/* valadatatype.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valadatatype.h>
#include <gee/arraylist.h>
#include <gee/readonlylist.h>
#include <vala/valatypesymbol.h>
#include <vala/valatypeparameter.h>
#include <vala/valacodevisitor.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valaarraytype.h>
#include <vala/valasymbol.h>
#include <vala/valadelegatetype.h>
#include <vala/valapointertype.h>
#include <vala/valaenum.h>
#include <vala/valastruct.h>

struct _ValaDataTypePrivate {
	gboolean _transfers_ownership;
	gboolean _takes_ownership;
	gboolean _is_out;
	gboolean _nullable;
	gboolean _requires_null_check;
	ValaTypesymbol* _data_type;
	ValaTypeParameter* _type_parameter;
	gboolean _floating_reference;
	gboolean _is_ref;
	GeeList* type_argument_list;
};
#define VALA_DATA_TYPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DATA_TYPE, ValaDataTypePrivate))
enum  {
	VALA_DATA_TYPE_DUMMY_PROPERTY,
	VALA_DATA_TYPE_TRANSFERS_OWNERSHIP,
	VALA_DATA_TYPE_TAKES_OWNERSHIP,
	VALA_DATA_TYPE_IS_OUT,
	VALA_DATA_TYPE_NULLABLE,
	VALA_DATA_TYPE_REQUIRES_NULL_CHECK,
	VALA_DATA_TYPE_DATA_TYPE,
	VALA_DATA_TYPE_TYPE_PARAMETER,
	VALA_DATA_TYPE_FLOATING_REFERENCE,
	VALA_DATA_TYPE_IS_REF
};
static void vala_data_type_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_data_type_real_get_cname (ValaDataType* self, gboolean var_type, gboolean const_type);
static char* vala_data_type_real_get_lower_case_cname (ValaDataType* self, const char* infix);
static char* vala_data_type_real_to_string (ValaCodeNode* base);
static ValaDataType* vala_data_type_real_copy (ValaDataType* self);
static gboolean vala_data_type_real_equals (ValaDataType* self, ValaDataType* type2);
static gboolean vala_data_type_real_stricter (ValaDataType* self, ValaDataType* type2);
static void vala_data_type_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gboolean vala_data_type_real_compatible (ValaDataType* self, ValaDataType* target_type, gboolean enable_non_null);
static gboolean vala_data_type_real_is_invokable (ValaDataType* self);
static ValaDataType* vala_data_type_real_get_return_type (ValaDataType* self);
static GeeCollection* vala_data_type_real_get_parameters (ValaDataType* self);
static gboolean vala_data_type_real_is_reference_type_or_type_parameter (ValaDataType* self);
static gboolean vala_data_type_real_is_array (ValaDataType* self);
static GeeCollection* vala_data_type_real_get_symbols (ValaDataType* self);
static ValaSymbol* vala_data_type_real_get_pointer_member (ValaDataType* self, const char* member_name);
static gboolean vala_data_type_real_is_real_struct_type (ValaDataType* self);
static char* vala_data_type_real_get_type_id (ValaDataType* self);
static gpointer vala_data_type_parent_class = NULL;
static void vala_data_type_dispose (GObject * obj);


ValaDataType* vala_data_type_new (void) {
	ValaDataType * self;
	self = g_object_newv (VALA_TYPE_DATA_TYPE, 0, NULL);
	return self;
}


/**
 * Appends the specified type as generic type argument.
 *
 * @param arg a type reference
 */
void vala_data_type_add_type_argument (ValaDataType* self, ValaDataType* arg) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	g_return_if_fail (arg == NULL || VALA_IS_DATA_TYPE (arg));
	gee_collection_add (GEE_COLLECTION (self->priv->type_argument_list), arg);
	vala_code_node_set_parent_node (VALA_CODE_NODE (arg), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the list of generic type arguments.
 *
 * @return type argument list
 */
GeeList* vala_data_type_get_type_arguments (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, self->priv->type_argument_list));
}


/**
 * Removes all generic type arguments.
 */
void vala_data_type_remove_all_type_arguments (ValaDataType* self) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	gee_collection_clear (GEE_COLLECTION (self->priv->type_argument_list));
}


static void vala_data_type_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDataType * self;
	self = VALA_DATA_TYPE (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	if (gee_collection_get_size (GEE_COLLECTION (self->priv->type_argument_list)) > 0) {
		{
			GeeList* type_arg_collection;
			GeeIterator* type_arg_it;
			type_arg_collection = self->priv->type_argument_list;
			type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
			while (gee_iterator_next (type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = ((ValaDataType*) gee_iterator_get (type_arg_it));
				{
					vala_code_node_accept (VALA_CODE_NODE (type_arg), visitor);
					(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
				}
			}
			(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
		}
	}
	vala_code_visitor_visit_data_type (visitor, self);
}


/**
 * Returns the name and qualifiers of this type as it is used in C code.
 *
 * @return the type string to be used in C code
 */
static char* vala_data_type_real_get_cname (ValaDataType* self, gboolean var_type, gboolean const_type) {
	char* ptr;
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	if (vala_data_type_get_data_type (self) == NULL && vala_data_type_get_type_parameter (self) == NULL) {
		if (var_type) {
			return g_strdup ("gpointer");
		} else {
			return g_strdup ("void");
		}
	}
	ptr = NULL;
	if (vala_data_type_get_type_parameter (self) != NULL || (!vala_typesymbol_is_reference_type (vala_data_type_get_data_type (self)) && !vala_data_type_get_is_ref (self) && !vala_data_type_get_is_out (self))) {
		char* _tmp2;
		_tmp2 = NULL;
		ptr = (_tmp2 = g_strdup (""), (ptr = (g_free (ptr), NULL)), _tmp2);
	} else {
		if ((vala_typesymbol_is_reference_type (vala_data_type_get_data_type (self)) && !vala_data_type_get_is_ref (self) && !vala_data_type_get_is_out (self)) || (!vala_typesymbol_is_reference_type (vala_data_type_get_data_type (self)) && (vala_data_type_get_is_ref (self) || vala_data_type_get_is_out (self)))) {
			char* _tmp3;
			_tmp3 = NULL;
			ptr = (_tmp3 = g_strdup ("*"), (ptr = (g_free (ptr), NULL)), _tmp3);
		} else {
			char* _tmp4;
			_tmp4 = NULL;
			ptr = (_tmp4 = g_strdup ("**"), (ptr = (g_free (ptr), NULL)), _tmp4);
		}
	}
	if (vala_data_type_get_data_type (self) != NULL) {
		char* _tmp5;
		char* _tmp6;
		char* _tmp7;
		_tmp5 = NULL;
		_tmp6 = NULL;
		_tmp7 = NULL;
		return (_tmp7 = (_tmp6 = g_strconcat ((_tmp5 = vala_typesymbol_get_cname (vala_data_type_get_data_type (self), const_type)), ptr, NULL), (_tmp5 = (g_free (_tmp5), NULL)), _tmp6), (ptr = (g_free (ptr), NULL)), _tmp7);
	} else {
		if (vala_data_type_get_type_parameter (self) != NULL) {
			if (const_type) {
				char* _tmp8;
				_tmp8 = NULL;
				return (_tmp8 = g_strdup ("gconstpointer"), (ptr = (g_free (ptr), NULL)), _tmp8);
			} else {
				char* _tmp9;
				_tmp9 = NULL;
				return (_tmp9 = g_strdup ("gpointer"), (ptr = (g_free (ptr), NULL)), _tmp9);
			}
		} else {
			gpointer _tmp10;
			/* raise error */
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), "unresolved type reference");
			return (_tmp10 = NULL, (ptr = (g_free (ptr), NULL)), _tmp10);
		}
	}
	ptr = (g_free (ptr), NULL);
}


char* vala_data_type_get_cname (ValaDataType* self, gboolean var_type, gboolean const_type) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_cname (self, var_type, const_type);
}


/**
 * Returns the name and qualifiers of this type as it is used in C code
 * in a const declaration.
 *
 * @return the type string to be used in C code const declarations
 */
char* vala_data_type_get_const_cname (ValaDataType* self) {
	char* ptr;
	ValaTypesymbol* t;
	char* _tmp6;
	char* _tmp7;
	char* _tmp8;
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	ptr = NULL;
	t = NULL;
	/* FIXME: workaround to make constant arrays possible*/
	if (VALA_IS_ARRAY_TYPE (self)) {
		ValaTypesymbol* _tmp1;
		ValaTypesymbol* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		t = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (vala_array_type_get_element_type ((VALA_ARRAY_TYPE (self)))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), _tmp1);
	} else {
		ValaTypesymbol* _tmp3;
		ValaTypesymbol* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		t = (_tmp3 = (_tmp2 = vala_data_type_get_data_type (self), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), _tmp3);
	}
	if (!vala_typesymbol_is_reference_type (t)) {
		char* _tmp4;
		_tmp4 = NULL;
		ptr = (_tmp4 = g_strdup (""), (ptr = (g_free (ptr), NULL)), _tmp4);
	} else {
		char* _tmp5;
		_tmp5 = NULL;
		ptr = (_tmp5 = g_strdup ("*"), (ptr = (g_free (ptr), NULL)), _tmp5);
	}
	_tmp6 = NULL;
	_tmp7 = NULL;
	_tmp8 = NULL;
	return (_tmp8 = (_tmp7 = g_strdup_printf ("const %s%s", (_tmp6 = vala_typesymbol_get_cname (t, FALSE)), ptr), (_tmp6 = (g_free (_tmp6), NULL)), _tmp7), (ptr = (g_free (ptr), NULL)), (t == NULL ? NULL : (t = (g_object_unref (t), NULL))), _tmp8);
}


/**
 * Returns the C name of this data type in lower case. Words are
 * separated by underscores.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the lower case name to be used in C code
 */
static char* vala_data_type_real_get_lower_case_cname (ValaDataType* self, const char* infix) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return vala_symbol_get_lower_case_cname (VALA_SYMBOL (vala_data_type_get_data_type (self)), infix);
}


char* vala_data_type_get_lower_case_cname (ValaDataType* self, const char* infix) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_lower_case_cname (self, infix);
}


static char* vala_data_type_real_to_string (ValaCodeNode* base) {
	ValaDataType * self;
	char* s;
	GeeList* type_args;
	char* _tmp12;
	self = VALA_DATA_TYPE (base);
	s = NULL;
	if (vala_data_type_get_data_type (self) != NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		s = (_tmp0 = vala_symbol_get_full_name (VALA_SYMBOL (vala_data_type_get_data_type (self))), (s = (g_free (s), NULL)), _tmp0);
	} else {
		if (vala_data_type_get_type_parameter (self) != NULL) {
			char* _tmp2;
			const char* _tmp1;
			_tmp2 = NULL;
			_tmp1 = NULL;
			s = (_tmp2 = (_tmp1 = vala_symbol_get_name (VALA_SYMBOL (vala_data_type_get_type_parameter (self))), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (s = (g_free (s), NULL)), _tmp2);
		} else {
			char* _tmp3;
			_tmp3 = NULL;
			s = (_tmp3 = g_strdup ("null"), (s = (g_free (s), NULL)), _tmp3);
		}
	}
	type_args = vala_data_type_get_type_arguments (self);
	if (!(VALA_IS_ARRAY_TYPE (self)) && gee_collection_get_size (GEE_COLLECTION (type_args)) > 0) {
		char* _tmp4;
		gboolean first;
		char* _tmp10;
		_tmp4 = NULL;
		s = (_tmp4 = g_strconcat (s, ("<"), NULL), (s = (g_free (s), NULL)), _tmp4);
		first = TRUE;
		{
			GeeList* type_arg_collection;
			GeeIterator* type_arg_it;
			type_arg_collection = type_args;
			type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
			while (gee_iterator_next (type_arg_it)) {
				ValaDataType* type_arg;
				type_arg = ((ValaDataType*) gee_iterator_get (type_arg_it));
				{
					if (!first) {
						char* _tmp5;
						_tmp5 = NULL;
						s = (_tmp5 = g_strconcat (s, (","), NULL), (s = (g_free (s), NULL)), _tmp5);
					} else {
						first = FALSE;
					}
					if (!vala_data_type_get_takes_ownership (type_arg)) {
						char* _tmp6;
						_tmp6 = NULL;
						s = (_tmp6 = g_strconcat (s, ("weak "), NULL), (s = (g_free (s), NULL)), _tmp6);
					}
					if (vala_data_type_get_data_type (type_arg) != NULL) {
						char* _tmp8;
						char* _tmp7;
						_tmp8 = NULL;
						_tmp7 = NULL;
						s = (_tmp8 = g_strconcat (s, (_tmp7 = (vala_symbol_get_full_name (VALA_SYMBOL (vala_data_type_get_data_type (type_arg))))), NULL), (s = (g_free (s), NULL)), _tmp8);
						_tmp7 = (g_free (_tmp7), NULL);
					} else {
						char* _tmp9;
						_tmp9 = NULL;
						s = (_tmp9 = g_strconcat (s, (vala_symbol_get_name (VALA_SYMBOL (vala_data_type_get_type_parameter (type_arg)))), NULL), (s = (g_free (s), NULL)), _tmp9);
					}
					(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
				}
			}
			(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
		}
		_tmp10 = NULL;
		s = (_tmp10 = g_strconcat (s, (">"), NULL), (s = (g_free (s), NULL)), _tmp10);
	}
	if (vala_data_type_get_nullable (self)) {
		char* _tmp11;
		_tmp11 = NULL;
		s = (_tmp11 = g_strconcat (s, ("?"), NULL), (s = (g_free (s), NULL)), _tmp11);
	}
	_tmp12 = NULL;
	return (_tmp12 = s, (type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL))), _tmp12);
}


/**
 * Creates a shallow copy of this type reference.
 *
 * @return copy of this type reference
 */
static ValaDataType* vala_data_type_real_copy (ValaDataType* self) {
	ValaDataType* result;
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	result = vala_data_type_new ();
	vala_code_node_set_source_reference (VALA_CODE_NODE (result), vala_code_node_get_source_reference (VALA_CODE_NODE (self)));
	vala_data_type_set_transfers_ownership (result, vala_data_type_get_transfers_ownership (self));
	vala_data_type_set_takes_ownership (result, vala_data_type_get_takes_ownership (self));
	vala_data_type_set_is_out (result, vala_data_type_get_is_out (self));
	vala_data_type_set_nullable (result, vala_data_type_get_nullable (self));
	vala_data_type_set_requires_null_check (result, vala_data_type_get_requires_null_check (self));
	vala_data_type_set_data_type (result, vala_data_type_get_data_type (self));
	vala_data_type_set_type_parameter (result, vala_data_type_get_type_parameter (self));
	vala_data_type_set_floating_reference (result, vala_data_type_get_floating_reference (self));
	vala_data_type_set_is_ref (result, vala_data_type_get_is_ref (self));
	{
		GeeList* arg_collection;
		GeeIterator* arg_it;
		arg_collection = self->priv->type_argument_list;
		arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
		while (gee_iterator_next (arg_it)) {
			ValaDataType* arg;
			arg = ((ValaDataType*) gee_iterator_get (arg_it));
			{
				ValaDataType* _tmp0;
				_tmp0 = NULL;
				gee_collection_add (GEE_COLLECTION (result->priv->type_argument_list), (_tmp0 = vala_data_type_copy (arg)));
				(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
				(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
			}
		}
		(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	}
	return result;
}


ValaDataType* vala_data_type_copy (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->copy (self);
}


/**
 * Checks two type references for equality. May only be used with
 * resolved type references.
 *
 * @param type2 a type reference
 * @return      true if this type reference is equal to type2, false
 *              otherwise
 */
static gboolean vala_data_type_real_equals (ValaDataType* self, ValaDataType* type2) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	g_return_val_if_fail (type2 == NULL || VALA_IS_DATA_TYPE (type2), FALSE);
	if (vala_data_type_get_transfers_ownership (type2) != vala_data_type_get_transfers_ownership (self)) {
		return FALSE;
	}
	if (vala_data_type_get_takes_ownership (type2) != vala_data_type_get_takes_ownership (self)) {
		return FALSE;
	}
	if (vala_data_type_get_is_ref (type2) != vala_data_type_get_is_ref (self)) {
		return FALSE;
	}
	if (vala_data_type_get_is_out (type2) != vala_data_type_get_is_out (self)) {
		return FALSE;
	}
	if (vala_data_type_get_nullable (type2) != vala_data_type_get_nullable (self)) {
		return FALSE;
	}
	if (vala_data_type_get_data_type (type2) != vala_data_type_get_data_type (self)) {
		return FALSE;
	}
	if (vala_data_type_get_type_parameter (type2) != NULL || vala_data_type_get_type_parameter (self) != NULL) {
		if (vala_data_type_get_type_parameter (type2) == NULL || vala_data_type_get_type_parameter (self) == NULL) {
			return FALSE;
		}
		if (!vala_typeparameter_equals (vala_data_type_get_type_parameter (type2), vala_data_type_get_type_parameter (self))) {
			return FALSE;
		}
	}
	if (vala_data_type_get_floating_reference (type2) != vala_data_type_get_floating_reference (self)) {
		return FALSE;
	}
	return TRUE;
}


gboolean vala_data_type_equals (ValaDataType* self, ValaDataType* type2) {
	return VALA_DATA_TYPE_GET_CLASS (self)->equals (self, type2);
}


/**
 * Checks whether this type reference is at least as strict as the
 * specified type reference type2.
 *
 * @param type2 a type reference
 * @return      true if this type reference is stricter or equal
 */
static gboolean vala_data_type_real_stricter (ValaDataType* self, ValaDataType* type2) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	g_return_val_if_fail (type2 == NULL || VALA_IS_DATA_TYPE (type2), FALSE);
	if (vala_data_type_get_transfers_ownership (type2) != vala_data_type_get_transfers_ownership (self)) {
		return FALSE;
	}
	if (vala_data_type_get_takes_ownership (type2) != vala_data_type_get_takes_ownership (self)) {
		return FALSE;
	}
	if (vala_data_type_get_is_ref (type2) != vala_data_type_get_is_ref (self)) {
		return FALSE;
	}
	if (vala_data_type_get_is_out (type2) != vala_data_type_get_is_out (self)) {
		return FALSE;
	}
	if (!vala_data_type_get_nullable (type2) && vala_data_type_get_nullable (self)) {
		return FALSE;
	}
	if (vala_data_type_get_data_type (type2) != vala_data_type_get_data_type (self)) {
		return FALSE;
	}
	/* FIXME: allow this type reference to refer to a
	        subtype of the type type2 is referring to*/
	if (vala_data_type_get_type_parameter (type2) != vala_data_type_get_type_parameter (self)) {
		return FALSE;
	}
	if (vala_data_type_get_floating_reference (type2) != vala_data_type_get_floating_reference (self)) {
		return FALSE;
	}
	return TRUE;
}


gboolean vala_data_type_stricter (ValaDataType* self, ValaDataType* type2) {
	return VALA_DATA_TYPE_GET_CLASS (self)->stricter (self, type2);
}


static void vala_data_type_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaDataType * self;
	self = VALA_DATA_TYPE (base);
	g_return_if_fail (old_type == NULL || VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (new_type == NULL || VALA_IS_DATA_TYPE (new_type));
	{
		gint i;
		i = 0;
		for (i = 0; i < gee_collection_get_size (GEE_COLLECTION (self->priv->type_argument_list)); i++) {
			ValaDataType* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = ((ValaDataType*) gee_list_get (((GeeList*) self->priv->type_argument_list), i))) == old_type, (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1)) {
				gee_list_set (((GeeList*) self->priv->type_argument_list), i, new_type);
				return;
			}
		}
	}
}


static gboolean vala_data_type_real_compatible (ValaDataType* self, ValaDataType* target_type, gboolean enable_non_null) {
	ValaAttribute* _tmp2;
	gboolean _tmp3;
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	g_return_val_if_fail (target_type == NULL || VALA_IS_DATA_TYPE (target_type), FALSE);
	if (VALA_IS_DELEGATE_TYPE (target_type) && VALA_IS_DELEGATE_TYPE (self)) {
		return vala_delegate_type_get_delegate_symbol ((VALA_DELEGATE_TYPE (target_type))) == vala_delegate_type_get_delegate_symbol ((VALA_DELEGATE_TYPE (self)));
	}
	/* only null is compatible to null */
	if (!(VALA_IS_POINTER_TYPE (target_type)) && vala_data_type_get_data_type (target_type) == NULL && vala_data_type_get_type_parameter (target_type) == NULL) {
		return (vala_data_type_get_data_type (self) == NULL && vala_data_type_get_type_parameter (self) == NULL);
	}
	_tmp2 = NULL;
	if ((_tmp3 = VALA_IS_POINTER_TYPE (target_type) || (vala_data_type_get_data_type (target_type) != NULL && (_tmp2 = vala_code_node_get_attribute (VALA_CODE_NODE (vala_data_type_get_data_type (target_type)), "PointerType")) != NULL), (_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL))), _tmp3)) {
		ValaAttribute* _tmp4;
		gboolean _tmp5;
		/* any reference or array type or pointer type can be cast to a generic pointer */
		_tmp4 = NULL;
		if ((_tmp5 = vala_data_type_get_type_parameter (self) != NULL || (vala_data_type_get_data_type (self) != NULL && (vala_typesymbol_is_reference_type (vala_data_type_get_data_type (self)) || VALA_IS_ARRAY_TYPE (self) || VALA_IS_DELEGATE_TYPE (self) || (_tmp4 = vala_code_node_get_attribute (VALA_CODE_NODE (vala_data_type_get_data_type (self)), "PointerType")) != NULL)), (_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL))), _tmp5)) {
			return TRUE;
		}
		return FALSE;
	}
	/* temporarily ignore type parameters */
	if (vala_data_type_get_type_parameter (target_type) != NULL) {
		return TRUE;
	}
	if (VALA_IS_ARRAY_TYPE (self) != VALA_IS_ARRAY_TYPE (target_type)) {
		return FALSE;
	}
	if (VALA_IS_ENUM (vala_data_type_get_data_type (self)) && VALA_IS_STRUCT (vala_data_type_get_data_type (target_type)) && vala_struct_is_integer_type ((VALA_STRUCT (vala_data_type_get_data_type (target_type))))) {
		return TRUE;
	}
	if (vala_data_type_get_data_type (self) == vala_data_type_get_data_type (target_type)) {
		if (vala_data_type_get_requires_null_check (self) && !vala_data_type_get_nullable (target_type) && vala_data_type_get_data_type (self) != NULL && vala_typesymbol_is_reference_type (vala_data_type_get_data_type (self))) {
			return !enable_non_null;
		}
		return TRUE;
	}
	/* incompatibility between null and non-null types*/
	if (VALA_IS_STRUCT (vala_data_type_get_data_type (self)) && VALA_IS_STRUCT (vala_data_type_get_data_type (target_type))) {
		ValaStruct* _tmp13;
		ValaStruct* expr_struct;
		ValaStruct* _tmp14;
		ValaStruct* expect_struct;
		_tmp13 = NULL;
		expr_struct = (_tmp13 = VALA_STRUCT (vala_data_type_get_data_type (self)), (_tmp13 == NULL ? NULL : g_object_ref (_tmp13)));
		_tmp14 = NULL;
		expect_struct = (_tmp14 = VALA_STRUCT (vala_data_type_get_data_type (target_type)), (_tmp14 == NULL ? NULL : g_object_ref (_tmp14)));
		/* integer types may be implicitly cast to floating point types */
		if (vala_struct_is_integer_type (expr_struct) && vala_struct_is_floating_type (expect_struct)) {
			gboolean _tmp15;
			return (_tmp15 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), _tmp15);
		}
		if ((vala_struct_is_integer_type (expr_struct) && vala_struct_is_integer_type (expect_struct)) || (vala_struct_is_floating_type (expr_struct) && vala_struct_is_floating_type (expect_struct))) {
			if (vala_struct_get_rank (expr_struct) <= vala_struct_get_rank (expect_struct)) {
				gboolean _tmp16;
				return (_tmp16 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), _tmp16);
			}
		}
		(expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL)));
		(expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL)));
	}
	if (vala_data_type_get_data_type (self) != NULL && vala_data_type_get_data_type (target_type) != NULL && vala_typesymbol_is_subtype_of (vala_data_type_get_data_type (self), vala_data_type_get_data_type (target_type))) {
		if (vala_data_type_get_requires_null_check (self) && !vala_data_type_get_nullable (target_type) && vala_typesymbol_is_reference_type (vala_data_type_get_data_type (self))) {
			return !enable_non_null;
		}
		return TRUE;
	}
	return FALSE;
}


gboolean vala_data_type_compatible (ValaDataType* self, ValaDataType* target_type, gboolean enable_non_null) {
	return VALA_DATA_TYPE_GET_CLASS (self)->compatible (self, target_type, enable_non_null);
}


/* incompatibility between null and non-null types
*
 * Returns whether instances of this type are invokable.
 *
 * @return true if invokable, false otherwise
 */
static gboolean vala_data_type_real_is_invokable (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return FALSE;
}


gboolean vala_data_type_is_invokable (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_invokable (self);
}


/**
 * Returns the return type of this invokable.
 *
 * @return return type
 */
static ValaDataType* vala_data_type_real_get_return_type (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


ValaDataType* vala_data_type_get_return_type (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_return_type (self);
}


/**
 * Returns copy of the list of invocation parameters.
 *
 * @return parameter list
 */
static GeeCollection* vala_data_type_real_get_parameters (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


GeeCollection* vala_data_type_get_parameters (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_parameters (self);
}


static gboolean vala_data_type_real_is_reference_type_or_type_parameter (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return (vala_data_type_get_data_type (self) != NULL && vala_typesymbol_is_reference_type (vala_data_type_get_data_type (self))) || vala_data_type_get_type_parameter (self) != NULL;
}


gboolean vala_data_type_is_reference_type_or_type_parameter (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_reference_type_or_type_parameter (self);
}


static gboolean vala_data_type_real_is_array (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return FALSE;
}


gboolean vala_data_type_is_array (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_array (self);
}


/**
 * Returns a list of symbols that define this type.
 *
 * @return symbol list
 */
static GeeCollection* vala_data_type_real_get_symbols (ValaDataType* self) {
	GeeArrayList* symbols;
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	symbols = gee_array_list_new (VALA_TYPE_SYMBOL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal);
	if (vala_data_type_get_data_type (self) != NULL) {
		gee_collection_add (GEE_COLLECTION (symbols), vala_data_type_get_data_type (self));
	}
	return GEE_COLLECTION (symbols);
}


GeeCollection* vala_data_type_get_symbols (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_symbols (self);
}


static ValaSymbol* vala_data_type_real_get_pointer_member (ValaDataType* self, const char* member_name) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


ValaSymbol* vala_data_type_get_pointer_member (ValaDataType* self, const char* member_name) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_pointer_member (self, member_name);
}


/**
 * Checks whether this data type references a real struct. A real struct
 * is a struct which is not a simple (fundamental) type.
 */
static gboolean vala_data_type_real_is_real_struct_type (ValaDataType* self) {
	ValaStruct* _tmp1;
	ValaTypesymbol* _tmp0;
	ValaStruct* s;
	gboolean _tmp3;
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	_tmp1 = NULL;
	_tmp0 = NULL;
	s = (_tmp1 = (_tmp0 = vala_data_type_get_data_type (self), (VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
	if (s != NULL && !vala_struct_is_simple_type (s)) {
		gboolean _tmp2;
		return (_tmp2 = TRUE, (s == NULL ? NULL : (s = (g_object_unref (s), NULL))), _tmp2);
	}
	return (_tmp3 = FALSE, (s == NULL ? NULL : (s = (g_object_unref (s), NULL))), _tmp3);
}


gboolean vala_data_type_is_real_struct_type (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->is_real_struct_type (self);
}


static char* vala_data_type_real_get_type_id (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	if (vala_data_type_get_data_type (self) != NULL) {
		return vala_typesymbol_get_type_id (vala_data_type_get_data_type (self));
	} else {
		return NULL;
	}
}


char* vala_data_type_get_type_id (ValaDataType* self) {
	return VALA_DATA_TYPE_GET_CLASS (self)->get_type_id (self);
}


gboolean vala_data_type_get_transfers_ownership (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return self->priv->_transfers_ownership;
}


void vala_data_type_set_transfers_ownership (ValaDataType* self, gboolean value) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	self->priv->_transfers_ownership = value;
}


gboolean vala_data_type_get_takes_ownership (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return self->priv->_takes_ownership;
}


void vala_data_type_set_takes_ownership (ValaDataType* self, gboolean value) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	self->priv->_takes_ownership = value;
}


gboolean vala_data_type_get_is_out (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return self->priv->_is_out;
}


void vala_data_type_set_is_out (ValaDataType* self, gboolean value) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	self->priv->_is_out = value;
}


gboolean vala_data_type_get_nullable (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return self->priv->_nullable;
}


void vala_data_type_set_nullable (ValaDataType* self, gboolean value) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	self->priv->_nullable = value;
}


gboolean vala_data_type_get_requires_null_check (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return self->priv->_requires_null_check;
}


void vala_data_type_set_requires_null_check (ValaDataType* self, gboolean value) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	self->priv->_requires_null_check = value;
}


ValaTypesymbol* vala_data_type_get_data_type (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return self->priv->_data_type;
}


void vala_data_type_set_data_type (ValaDataType* self, ValaTypesymbol* value) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	self->priv->_data_type = value;
}


ValaTypeParameter* vala_data_type_get_type_parameter (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return self->priv->_type_parameter;
}


void vala_data_type_set_type_parameter (ValaDataType* self, ValaTypeParameter* value) {
	ValaTypeParameter* _tmp2;
	ValaTypeParameter* _tmp1;
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_type_parameter = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_type_parameter == NULL ? NULL : (self->priv->_type_parameter = (g_object_unref (self->priv->_type_parameter), NULL))), _tmp2);
}


gboolean vala_data_type_get_floating_reference (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return self->priv->_floating_reference;
}


void vala_data_type_set_floating_reference (ValaDataType* self, gboolean value) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	self->priv->_floating_reference = value;
}


gboolean vala_data_type_get_is_ref (ValaDataType* self) {
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return self->priv->_is_ref;
}


void vala_data_type_set_is_ref (ValaDataType* self, gboolean value) {
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	self->priv->_is_ref = value;
}


static void vala_data_type_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaDataType * self;
	self = VALA_DATA_TYPE (object);
	switch (property_id) {
		case VALA_DATA_TYPE_TRANSFERS_OWNERSHIP:
		g_value_set_boolean (value, vala_data_type_get_transfers_ownership (self));
		break;
		case VALA_DATA_TYPE_TAKES_OWNERSHIP:
		g_value_set_boolean (value, vala_data_type_get_takes_ownership (self));
		break;
		case VALA_DATA_TYPE_IS_OUT:
		g_value_set_boolean (value, vala_data_type_get_is_out (self));
		break;
		case VALA_DATA_TYPE_NULLABLE:
		g_value_set_boolean (value, vala_data_type_get_nullable (self));
		break;
		case VALA_DATA_TYPE_REQUIRES_NULL_CHECK:
		g_value_set_boolean (value, vala_data_type_get_requires_null_check (self));
		break;
		case VALA_DATA_TYPE_DATA_TYPE:
		g_value_set_object (value, vala_data_type_get_data_type (self));
		break;
		case VALA_DATA_TYPE_TYPE_PARAMETER:
		g_value_set_object (value, vala_data_type_get_type_parameter (self));
		break;
		case VALA_DATA_TYPE_FLOATING_REFERENCE:
		g_value_set_boolean (value, vala_data_type_get_floating_reference (self));
		break;
		case VALA_DATA_TYPE_IS_REF:
		g_value_set_boolean (value, vala_data_type_get_is_ref (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_data_type_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaDataType * self;
	self = VALA_DATA_TYPE (object);
	switch (property_id) {
		case VALA_DATA_TYPE_TRANSFERS_OWNERSHIP:
		vala_data_type_set_transfers_ownership (self, g_value_get_boolean (value));
		break;
		case VALA_DATA_TYPE_TAKES_OWNERSHIP:
		vala_data_type_set_takes_ownership (self, g_value_get_boolean (value));
		break;
		case VALA_DATA_TYPE_IS_OUT:
		vala_data_type_set_is_out (self, g_value_get_boolean (value));
		break;
		case VALA_DATA_TYPE_NULLABLE:
		vala_data_type_set_nullable (self, g_value_get_boolean (value));
		break;
		case VALA_DATA_TYPE_REQUIRES_NULL_CHECK:
		vala_data_type_set_requires_null_check (self, g_value_get_boolean (value));
		break;
		case VALA_DATA_TYPE_DATA_TYPE:
		vala_data_type_set_data_type (self, g_value_get_object (value));
		break;
		case VALA_DATA_TYPE_TYPE_PARAMETER:
		vala_data_type_set_type_parameter (self, g_value_get_object (value));
		break;
		case VALA_DATA_TYPE_FLOATING_REFERENCE:
		vala_data_type_set_floating_reference (self, g_value_get_boolean (value));
		break;
		case VALA_DATA_TYPE_IS_REF:
		vala_data_type_set_is_ref (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_data_type_class_init (ValaDataTypeClass * klass) {
	vala_data_type_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaDataTypePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_data_type_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_data_type_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_data_type_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_data_type_real_accept;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_data_type_real_get_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_data_type_real_get_lower_case_cname;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_data_type_real_to_string;
	VALA_DATA_TYPE_CLASS (klass)->copy = vala_data_type_real_copy;
	VALA_DATA_TYPE_CLASS (klass)->equals = vala_data_type_real_equals;
	VALA_DATA_TYPE_CLASS (klass)->stricter = vala_data_type_real_stricter;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_data_type_real_replace_type;
	VALA_DATA_TYPE_CLASS (klass)->compatible = vala_data_type_real_compatible;
	VALA_DATA_TYPE_CLASS (klass)->is_invokable = vala_data_type_real_is_invokable;
	VALA_DATA_TYPE_CLASS (klass)->get_return_type = vala_data_type_real_get_return_type;
	VALA_DATA_TYPE_CLASS (klass)->get_parameters = vala_data_type_real_get_parameters;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type_or_type_parameter = vala_data_type_real_is_reference_type_or_type_parameter;
	VALA_DATA_TYPE_CLASS (klass)->is_array = vala_data_type_real_is_array;
	VALA_DATA_TYPE_CLASS (klass)->get_symbols = vala_data_type_real_get_symbols;
	VALA_DATA_TYPE_CLASS (klass)->get_pointer_member = vala_data_type_real_get_pointer_member;
	VALA_DATA_TYPE_CLASS (klass)->is_real_struct_type = vala_data_type_real_is_real_struct_type;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_data_type_real_get_type_id;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_TRANSFERS_OWNERSHIP, g_param_spec_boolean ("transfers-ownership", "transfers-ownership", "transfers-ownership", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_TAKES_OWNERSHIP, g_param_spec_boolean ("takes-ownership", "takes-ownership", "takes-ownership", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_IS_OUT, g_param_spec_boolean ("is-out", "is-out", "is-out", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_NULLABLE, g_param_spec_boolean ("nullable", "nullable", "nullable", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_REQUIRES_NULL_CHECK, g_param_spec_boolean ("requires-null-check", "requires-null-check", "requires-null-check", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_DATA_TYPE, g_param_spec_object ("data-type", "data-type", "data-type", VALA_TYPE_TYPESYMBOL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_TYPE_PARAMETER, g_param_spec_object ("type-parameter", "type-parameter", "type-parameter", VALA_TYPE_TYPEPARAMETER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_FLOATING_REFERENCE, g_param_spec_boolean ("floating-reference", "floating-reference", "floating-reference", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_IS_REF, g_param_spec_boolean ("is-ref", "is-ref", "is-ref", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_data_type_init (ValaDataType * self) {
	self->priv = VALA_DATA_TYPE_GET_PRIVATE (self);
	self->priv->type_argument_list = GEE_LIST (gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_data_type_dispose (GObject * obj) {
	ValaDataType * self;
	self = VALA_DATA_TYPE (obj);
	(self->priv->_type_parameter == NULL ? NULL : (self->priv->_type_parameter = (g_object_unref (self->priv->_type_parameter), NULL)));
	(self->priv->type_argument_list == NULL ? NULL : (self->priv->type_argument_list = (g_object_unref (self->priv->type_argument_list), NULL)));
	G_OBJECT_CLASS (vala_data_type_parent_class)->dispose (obj);
}


GType vala_data_type_get_type (void) {
	static GType vala_data_type_type_id = 0;
	if (G_UNLIKELY (vala_data_type_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDataTypeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_data_type_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDataType), 0, (GInstanceInitFunc) vala_data_type_init };
		vala_data_type_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaDataType", &g_define_type_info, 0);
	}
	return vala_data_type_type_id;
}




