/* valanamespace.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valanamespace.h>
#include <gee/arraylist.h>
#include <gee/readonlycollection.h>
#include <vala/valaclass.h>
#include <vala/valainterface.h>
#include <vala/valastruct.h>
#include <vala/valaenum.h>
#include <vala/valaerrordomain.h>
#include <vala/valadelegate.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamethod.h>
#include <vala/valasourcereference.h>
#include <vala/valascope.h>
#include <vala/valadatatype.h>
#include <vala/valasourcefile.h>
#include <vala/valacreationmethod.h>
#include <vala/valareport.h>
#include <vala/valamember.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>




struct _ValaNamespacePrivate {
	GeeList* classes;
	GeeList* interfaces;
	GeeList* structs;
	GeeList* enums;
	GeeList* error_domains;
	GeeList* delegates;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* cprefixes;
	char* lower_case_cprefix;
	GeeList* cheader_filenames;
	GeeList* namespaces;
};

#define VALA_NAMESPACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_NAMESPACE, ValaNamespacePrivate))
enum  {
	VALA_NAMESPACE_DUMMY_PROPERTY
};
static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_namespace_real_get_cprefix (ValaSymbol* base);
static char* vala_namespace_real_get_lower_case_cprefix (ValaSymbol* base);
static GeeCollection* vala_namespace_real_get_cheader_filenames (ValaSymbol* base);
static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a);
static gpointer vala_namespace_parent_class = NULL;
static void vala_namespace_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new namespace.
 *
 * @param name             namespace name
 * @param source_reference reference to source code
 * @return                 newly created namespace
 */
ValaNamespace* vala_namespace_new (const char* name, ValaSourceReference* source_reference) {
	ValaNamespace * self;
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_NAMESPACE, 0, NULL);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_symbol_set_access (VALA_SYMBOL (self), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


/**
 * Adds the specified namespace to this source file.
 *
 * @param ns a namespace
 */
void vala_namespace_add_namespace (ValaNamespace* self, ValaNamespace* ns) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	if (VALA_IS_NAMESPACE (vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (ns))))) {
		ValaNamespace* old_ns;
		/* merge if namespace already exists*/
		old_ns = VALA_NAMESPACE (vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (ns))));
		if (vala_symbol_get_external_package (VALA_SYMBOL (old_ns)) && !vala_symbol_get_external_package (VALA_SYMBOL (ns))) {
			vala_code_node_set_source_reference (VALA_CODE_NODE (old_ns), vala_code_node_get_source_reference (VALA_CODE_NODE (ns)));
		}
		{
			GeeCollection* sub_ns_collection;
			GeeIterator* sub_ns_it;
			sub_ns_collection = vala_namespace_get_namespaces (ns);
			sub_ns_it = gee_iterable_iterator (GEE_ITERABLE (sub_ns_collection));
			while (gee_iterator_next (sub_ns_it)) {
				ValaNamespace* sub_ns;
				sub_ns = ((ValaNamespace*) gee_iterator_get (sub_ns_it));
				{
					vala_namespace_add_namespace (old_ns, sub_ns);
					(sub_ns == NULL ? NULL : (sub_ns = (g_object_unref (sub_ns), NULL)));
				}
			}
			(sub_ns_collection == NULL ? NULL : (sub_ns_collection = (g_object_unref (sub_ns_collection), NULL)));
			(sub_ns_it == NULL ? NULL : (sub_ns_it = (g_object_unref (sub_ns_it), NULL)));
		}
		{
			GeeCollection* cl_collection;
			GeeIterator* cl_it;
			cl_collection = vala_namespace_get_classes (ns);
			cl_it = gee_iterable_iterator (GEE_ITERABLE (cl_collection));
			while (gee_iterator_next (cl_it)) {
				ValaClass* cl;
				cl = ((ValaClass*) gee_iterator_get (cl_it));
				{
					vala_namespace_add_class (old_ns, cl);
					(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
				}
			}
			(cl_collection == NULL ? NULL : (cl_collection = (g_object_unref (cl_collection), NULL)));
			(cl_it == NULL ? NULL : (cl_it = (g_object_unref (cl_it), NULL)));
		}
		{
			GeeCollection* st_collection;
			GeeIterator* st_it;
			st_collection = vala_namespace_get_structs (ns);
			st_it = gee_iterable_iterator (GEE_ITERABLE (st_collection));
			while (gee_iterator_next (st_it)) {
				ValaStruct* st;
				st = ((ValaStruct*) gee_iterator_get (st_it));
				{
					vala_namespace_add_struct (old_ns, st);
					(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
				}
			}
			(st_collection == NULL ? NULL : (st_collection = (g_object_unref (st_collection), NULL)));
			(st_it == NULL ? NULL : (st_it = (g_object_unref (st_it), NULL)));
		}
		{
			GeeCollection* iface_collection;
			GeeIterator* iface_it;
			iface_collection = vala_namespace_get_interfaces (ns);
			iface_it = gee_iterable_iterator (GEE_ITERABLE (iface_collection));
			while (gee_iterator_next (iface_it)) {
				ValaInterface* iface;
				iface = ((ValaInterface*) gee_iterator_get (iface_it));
				{
					vala_namespace_add_interface (old_ns, iface);
					(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
				}
			}
			(iface_collection == NULL ? NULL : (iface_collection = (g_object_unref (iface_collection), NULL)));
			(iface_it == NULL ? NULL : (iface_it = (g_object_unref (iface_it), NULL)));
		}
		{
			GeeCollection* d_collection;
			GeeIterator* d_it;
			d_collection = vala_namespace_get_delegates (ns);
			d_it = gee_iterable_iterator (GEE_ITERABLE (d_collection));
			while (gee_iterator_next (d_it)) {
				ValaDelegate* d;
				d = ((ValaDelegate*) gee_iterator_get (d_it));
				{
					vala_namespace_add_delegate (old_ns, d);
					(d == NULL ? NULL : (d = (g_object_unref (d), NULL)));
				}
			}
			(d_collection == NULL ? NULL : (d_collection = (g_object_unref (d_collection), NULL)));
			(d_it == NULL ? NULL : (d_it = (g_object_unref (d_it), NULL)));
		}
		{
			GeeCollection* en_collection;
			GeeIterator* en_it;
			en_collection = vala_namespace_get_enums (ns);
			en_it = gee_iterable_iterator (GEE_ITERABLE (en_collection));
			while (gee_iterator_next (en_it)) {
				ValaEnum* en;
				en = ((ValaEnum*) gee_iterator_get (en_it));
				{
					vala_namespace_add_enum (old_ns, en);
					(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
				}
			}
			(en_collection == NULL ? NULL : (en_collection = (g_object_unref (en_collection), NULL)));
			(en_it == NULL ? NULL : (en_it = (g_object_unref (en_it), NULL)));
		}
		{
			GeeCollection* ed_collection;
			GeeIterator* ed_it;
			ed_collection = vala_namespace_get_error_domains (ns);
			ed_it = gee_iterable_iterator (GEE_ITERABLE (ed_collection));
			while (gee_iterator_next (ed_it)) {
				ValaErrorDomain* ed;
				ed = ((ValaErrorDomain*) gee_iterator_get (ed_it));
				{
					vala_namespace_add_error_domain (old_ns, ed);
					(ed == NULL ? NULL : (ed = (g_object_unref (ed), NULL)));
				}
			}
			(ed_collection == NULL ? NULL : (ed_collection = (g_object_unref (ed_collection), NULL)));
			(ed_it == NULL ? NULL : (ed_it = (g_object_unref (ed_it), NULL)));
		}
		{
			GeeCollection* c_collection;
			GeeIterator* c_it;
			c_collection = vala_namespace_get_constants (ns);
			c_it = gee_iterable_iterator (GEE_ITERABLE (c_collection));
			while (gee_iterator_next (c_it)) {
				ValaConstant* c;
				c = ((ValaConstant*) gee_iterator_get (c_it));
				{
					vala_namespace_add_constant (old_ns, c);
					(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
				}
			}
			(c_collection == NULL ? NULL : (c_collection = (g_object_unref (c_collection), NULL)));
			(c_it == NULL ? NULL : (c_it = (g_object_unref (c_it), NULL)));
		}
		{
			GeeCollection* f_collection;
			GeeIterator* f_it;
			f_collection = vala_namespace_get_fields (ns);
			f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
			while (gee_iterator_next (f_it)) {
				ValaField* f;
				f = ((ValaField*) gee_iterator_get (f_it));
				{
					vala_namespace_add_field (old_ns, f);
					(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
				}
			}
			(f_collection == NULL ? NULL : (f_collection = (g_object_unref (f_collection), NULL)));
			(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
		}
		{
			GeeCollection* m_collection;
			GeeIterator* m_it;
			m_collection = vala_namespace_get_methods (ns);
			m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
			while (gee_iterator_next (m_it)) {
				ValaMethod* m;
				m = ((ValaMethod*) gee_iterator_get (m_it));
				{
					vala_namespace_add_method (old_ns, m);
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				}
			}
			(m_collection == NULL ? NULL : (m_collection = (g_object_unref (m_collection), NULL)));
			(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
		}
		(old_ns == NULL ? NULL : (old_ns = (g_object_unref (old_ns), NULL)));
	} else {
		gee_collection_add (GEE_COLLECTION (self->priv->namespaces), ns);
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (ns)), VALA_SYMBOL (ns));
	}
}


/**
 * Returns a copy of the list of namespaces.
 *
 * @return namespace list
 */
GeeCollection* vala_namespace_get_namespaces (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_NAMESPACE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->namespaces)));
}


/**
 * Adds the specified class to this namespace.
 *
 * @param cl a class
 */
void vala_namespace_add_class (ValaNamespace* self, ValaClass* cl) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	if (VALA_IS_CLASS (vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (cl))))) {
		ValaClass* old_class;
		/* merge*/
		old_class = VALA_CLASS (vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (cl))));
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = vala_class_get_base_types (cl);
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				base_type = ((ValaDataType*) gee_iterator_get (base_type_it));
				{
					vala_class_add_base_type (old_class, base_type);
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		{
			GeeCollection* f_collection;
			GeeIterator* f_it;
			f_collection = vala_class_get_fields (cl);
			f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
			while (gee_iterator_next (f_it)) {
				ValaField* f;
				f = ((ValaField*) gee_iterator_get (f_it));
				{
					vala_class_add_field (old_class, f);
					(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
				}
			}
			(f_collection == NULL ? NULL : (f_collection = (g_object_unref (f_collection), NULL)));
			(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
		}
		{
			GeeCollection* m_collection;
			GeeIterator* m_it;
			m_collection = vala_class_get_methods (cl);
			m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
			while (gee_iterator_next (m_it)) {
				ValaMethod* m;
				m = ((ValaMethod*) gee_iterator_get (m_it));
				{
					if (m == vala_class_get_default_construction_method (cl) && vala_class_get_default_construction_method (old_class) != NULL) {
						/* ignore secondary default creation method*/
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
						continue;
					}
					vala_class_add_method (old_class, m);
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				}
			}
			(m_collection == NULL ? NULL : (m_collection = (g_object_unref (m_collection), NULL)));
			(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
		}
		if (vala_class_get_constructor (cl) != NULL) {
			vala_class_set_constructor (old_class, vala_class_get_constructor (cl));
		}
		vala_source_file_remove_node (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl))), VALA_CODE_NODE (cl));
		(old_class == NULL ? NULL : (old_class = (g_object_unref (old_class), NULL)));
	} else {
		gee_collection_add (GEE_COLLECTION (self->priv->classes), cl);
		vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (cl)), VALA_SYMBOL (cl));
	}
}


/**
 * Adds the specified interface to this namespace.
 *
 * @param iface an interface
 */
void vala_namespace_add_interface (ValaNamespace* self, ValaInterface* iface) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	gee_collection_add (GEE_COLLECTION (self->priv->interfaces), iface);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (iface)), VALA_SYMBOL (iface));
}


/**
 * Adds the specified struct to this namespace.
 *
 * @param st a struct
 */
void vala_namespace_add_struct (ValaNamespace* self, ValaStruct* st) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_STRUCT (st));
	gee_collection_add (GEE_COLLECTION (self->priv->structs), st);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (st)), VALA_SYMBOL (st));
}


/**
 * Adds the specified enum to this namespace.
 *
 * @param en an enum
 */
void vala_namespace_add_enum (ValaNamespace* self, ValaEnum* en) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_ENUM (en));
	gee_collection_add (GEE_COLLECTION (self->priv->enums), en);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (en)), VALA_SYMBOL (en));
}


/**
 * Adds the specified error domain to this namespace.
 *
 * @param edomain an error domain
 */
void vala_namespace_add_error_domain (ValaNamespace* self, ValaErrorDomain* edomain) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_ERROR_DOMAIN (edomain));
	gee_collection_add (GEE_COLLECTION (self->priv->error_domains), edomain);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (edomain)), VALA_SYMBOL (edomain));
}


/**
 * Adds the specified delegate to this namespace.
 *
 * @param d a delegate
 */
void vala_namespace_add_delegate (ValaNamespace* self, ValaDelegate* d) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_DELEGATE (d));
	gee_collection_add (GEE_COLLECTION (self->priv->delegates), d);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (d)), VALA_SYMBOL (d));
}


/**
 * Returns a copy of the list of structs.
 *
 * @return struct list
 */
GeeCollection* vala_namespace_get_structs (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_STRUCT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->structs)));
}


/**
 * Returns a copy of the list of classes.
 *
 * @return class list
 */
GeeCollection* vala_namespace_get_classes (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_CLASS, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->classes)));
}


/**
 * Returns a copy of the list of interfaces.
 *
 * @return interface list
 */
GeeCollection* vala_namespace_get_interfaces (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_INTERFACE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->interfaces)));
}


/**
 * Returns a copy of the list of enums.
 *
 * @return enum list
 */
GeeCollection* vala_namespace_get_enums (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_ENUM, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->enums)));
}


/**
 * Returns a copy of the list of error domains.
 *
 * @return error domain list
 */
GeeCollection* vala_namespace_get_error_domains (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_ERROR_DOMAIN, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->error_domains)));
}


/**
 * Returns a copy of the list of fields.
 *
 * @return field list
 */
GeeCollection* vala_namespace_get_fields (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->fields)));
}


/**
 * Returns a copy of the list of constants.
 *
 * @return constant list
 */
GeeCollection* vala_namespace_get_constants (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_CONSTANT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->constants)));
}


/**
 * Returns a copy of the list of delegates.
 *
 * @return delegate list
 */
GeeCollection* vala_namespace_get_delegates (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_DELEGATE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->delegates)));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return method list
 */
GeeCollection* vala_namespace_get_methods (ValaNamespace* self) {
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->methods)));
}


/**
 * Adds the specified constant to this namespace.
 *
 * @param constant a constant
 */
void vala_namespace_add_constant (ValaNamespace* self, ValaConstant* constant) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CONSTANT (constant));
	gee_collection_add (GEE_COLLECTION (self->priv->constants), constant);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (constant)), VALA_SYMBOL (constant));
}


/**
 * Adds the specified field to this namespace.
 *
 * @param f a field
 */
void vala_namespace_add_field (ValaNamespace* self, ValaField* f) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	gee_collection_add (GEE_COLLECTION (self->priv->fields), f);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (f)), VALA_SYMBOL (f));
}


/**
 * Adds the specified method to this namespace.
 *
 * @param m a method
 */
void vala_namespace_add_method (ValaNamespace* self, ValaMethod* m) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), "instance methods not allowed outside of data types");
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (m)), VALA_SYMBOL (m));
}


static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaNamespace * self;
	self = VALA_NAMESPACE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_namespace (visitor, self);
}


static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaNamespace * self;
	self = VALA_NAMESPACE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* ns_collection;
		GeeIterator* ns_it;
		ns_collection = self->priv->namespaces;
		ns_it = gee_iterable_iterator (GEE_ITERABLE (ns_collection));
		while (gee_iterator_next (ns_it)) {
			ValaNamespace* ns;
			ns = ((ValaNamespace*) gee_iterator_get (ns_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (ns), visitor);
				(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
			}
		}
		(ns_it == NULL ? NULL : (ns_it = (g_object_unref (ns_it), NULL)));
	}
	/* process enums first to avoid order problems in C code */
	{
		GeeList* en_collection;
		GeeIterator* en_it;
		en_collection = self->priv->enums;
		en_it = gee_iterable_iterator (GEE_ITERABLE (en_collection));
		while (gee_iterator_next (en_it)) {
			ValaEnum* en;
			en = ((ValaEnum*) gee_iterator_get (en_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (en), visitor);
				(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
			}
		}
		(en_it == NULL ? NULL : (en_it = (g_object_unref (en_it), NULL)));
	}
	{
		GeeList* edomain_collection;
		GeeIterator* edomain_it;
		edomain_collection = self->priv->error_domains;
		edomain_it = gee_iterable_iterator (GEE_ITERABLE (edomain_collection));
		while (gee_iterator_next (edomain_it)) {
			ValaErrorDomain* edomain;
			edomain = ((ValaErrorDomain*) gee_iterator_get (edomain_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (edomain), visitor);
				(edomain == NULL ? NULL : (edomain = (g_object_unref (edomain), NULL)));
			}
		}
		(edomain_it == NULL ? NULL : (edomain_it = (g_object_unref (edomain_it), NULL)));
	}
	{
		GeeList* cl_collection;
		GeeIterator* cl_it;
		cl_collection = self->priv->classes;
		cl_it = gee_iterable_iterator (GEE_ITERABLE (cl_collection));
		while (gee_iterator_next (cl_it)) {
			ValaClass* cl;
			cl = ((ValaClass*) gee_iterator_get (cl_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (cl), visitor);
				(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			}
		}
		(cl_it == NULL ? NULL : (cl_it = (g_object_unref (cl_it), NULL)));
	}
	{
		GeeList* iface_collection;
		GeeIterator* iface_it;
		iface_collection = self->priv->interfaces;
		iface_it = gee_iterable_iterator (GEE_ITERABLE (iface_collection));
		while (gee_iterator_next (iface_it)) {
			ValaInterface* iface;
			iface = ((ValaInterface*) gee_iterator_get (iface_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (iface), visitor);
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
		(iface_it == NULL ? NULL : (iface_it = (g_object_unref (iface_it), NULL)));
	}
	{
		GeeList* st_collection;
		GeeIterator* st_it;
		st_collection = self->priv->structs;
		st_it = gee_iterable_iterator (GEE_ITERABLE (st_collection));
		while (gee_iterator_next (st_it)) {
			ValaStruct* st;
			st = ((ValaStruct*) gee_iterator_get (st_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (st), visitor);
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			}
		}
		(st_it == NULL ? NULL : (st_it = (g_object_unref (st_it), NULL)));
	}
	{
		GeeList* d_collection;
		GeeIterator* d_it;
		d_collection = self->priv->delegates;
		d_it = gee_iterable_iterator (GEE_ITERABLE (d_collection));
		while (gee_iterator_next (d_it)) {
			ValaDelegate* d;
			d = ((ValaDelegate*) gee_iterator_get (d_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (d), visitor);
				(d == NULL ? NULL : (d = (g_object_unref (d), NULL)));
			}
		}
		(d_it == NULL ? NULL : (d_it = (g_object_unref (d_it), NULL)));
	}
	{
		GeeList* c_collection;
		GeeIterator* c_it;
		c_collection = self->priv->constants;
		c_it = gee_iterable_iterator (GEE_ITERABLE (c_collection));
		while (gee_iterator_next (c_it)) {
			ValaConstant* c;
			c = ((ValaConstant*) gee_iterator_get (c_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
				(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
			}
		}
		(c_it == NULL ? NULL : (c_it = (g_object_unref (c_it), NULL)));
	}
	{
		GeeList* f_collection;
		GeeIterator* f_it;
		f_collection = self->priv->fields;
		f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = ((ValaField*) gee_iterator_get (f_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
			}
		}
		(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
	}
	{
		GeeList* m_collection;
		GeeIterator* m_it;
		m_collection = self->priv->methods;
		m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = ((ValaMethod*) gee_iterator_get (m_it));
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
		(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
	}
}


static char* vala_namespace_real_get_cprefix (ValaSymbol* base) {
	ValaNamespace * self;
	self = VALA_NAMESPACE (base);
	if (gee_collection_get_size (GEE_COLLECTION (self->priv->cprefixes)) > 0) {
		return ((char*) gee_list_get (((GeeList*) self->priv->cprefixes), 0));
	} else {
		if (NULL != vala_symbol_get_name (VALA_SYMBOL (self))) {
			char* parent_prefix;
			char* _tmp3;
			parent_prefix = NULL;
			if (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)) == NULL) {
				char* _tmp1;
				_tmp1 = NULL;
				parent_prefix = (_tmp1 = g_strdup (""), (parent_prefix = (g_free (parent_prefix), NULL)), _tmp1);
			} else {
				char* _tmp2;
				_tmp2 = NULL;
				parent_prefix = (_tmp2 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))), (parent_prefix = (g_free (parent_prefix), NULL)), _tmp2);
			}
			_tmp3 = NULL;
			return (_tmp3 = g_strconcat (parent_prefix, vala_symbol_get_name (VALA_SYMBOL (self)), NULL), (parent_prefix = (g_free (parent_prefix), NULL)), _tmp3);
		} else {
			return g_strdup ("");
		}
	}
}


GeeList* vala_namespace_get_cprefixes (ValaNamespace* self) {
	GeeList* _tmp0;
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	if (0 == gee_collection_get_size (GEE_COLLECTION (self->priv->cprefixes)) && NULL != vala_symbol_get_name (VALA_SYMBOL (self))) {
		gee_collection_add (GEE_COLLECTION (self->priv->cprefixes), vala_symbol_get_name (VALA_SYMBOL (self)));
	}
	_tmp0 = NULL;
	return (_tmp0 = self->priv->cprefixes, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
}


/**
 * Adds a camel case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefixes the camel case prefixes used in C code
 */
void vala_namespace_add_cprefix (ValaNamespace* self, const char* cprefix) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (cprefix != NULL);
	g_return_if_fail (g_utf8_strlen (cprefix, -1) >= 1);
	gee_collection_add (GEE_COLLECTION (self->priv->cprefixes), cprefix);
}


/**
 * Returns the lower case string to be prepended to the name of members
 * of this namespace when used in C code.
 *
 * @return the lower case prefix to be used in C code
 */
static char* vala_namespace_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaNamespace * self;
	const char* _tmp5;
	self = VALA_NAMESPACE (base);
	if (self->priv->lower_case_cprefix == NULL) {
		if (vala_symbol_get_name (VALA_SYMBOL (self)) == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->lower_case_cprefix = (_tmp0 = g_strdup (""), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp0);
		} else {
			char* parent_prefix;
			char* _tmp4;
			char* _tmp3;
			parent_prefix = NULL;
			if (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)) == NULL) {
				char* _tmp1;
				_tmp1 = NULL;
				parent_prefix = (_tmp1 = g_strdup (""), (parent_prefix = (g_free (parent_prefix), NULL)), _tmp1);
			} else {
				char* _tmp2;
				_tmp2 = NULL;
				parent_prefix = (_tmp2 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self))), (parent_prefix = (g_free (parent_prefix), NULL)), _tmp2);
			}
			_tmp4 = NULL;
			_tmp3 = NULL;
			self->priv->lower_case_cprefix = (_tmp4 = g_strdup_printf ("%s%s_", parent_prefix, (_tmp3 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (VALA_SYMBOL (self))))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp4);
			_tmp3 = (g_free (_tmp3), NULL);
			parent_prefix = (g_free (parent_prefix), NULL);
		}
	}
	_tmp5 = NULL;
	return (_tmp5 = self->priv->lower_case_cprefix, (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
}


/**
 * Sets the lower case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefix the lower case prefix to be used in C code
 */
void vala_namespace_set_lower_case_cprefix (ValaNamespace* self, const char* cprefix) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (cprefix != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->lower_case_cprefix = (_tmp1 = (_tmp0 = cprefix, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), _tmp1);
}


static GeeCollection* vala_namespace_real_get_cheader_filenames (ValaSymbol* base) {
	ValaNamespace * self;
	self = VALA_NAMESPACE (base);
	return GEE_COLLECTION (gee_read_only_collection_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, GEE_COLLECTION (self->priv->cheader_filenames)));
}


/**
 * Returns the C header filename of this namespace.
 *
 * @return header filename
 */
char* vala_namespace_get_cheader_filename (ValaNamespace* self) {
	GString* s;
	gboolean first;
	const char* _tmp0;
	char* _tmp1;
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	s = g_string_new ("");
	first = TRUE;
	{
		GeeCollection* cheader_filename_collection;
		GeeIterator* cheader_filename_it;
		cheader_filename_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (self));
		cheader_filename_it = gee_iterable_iterator (GEE_ITERABLE (cheader_filename_collection));
		while (gee_iterator_next (cheader_filename_it)) {
			char* cheader_filename;
			cheader_filename = ((char*) gee_iterator_get (cheader_filename_it));
			{
				if (first) {
					first = FALSE;
				} else {
					g_string_append_c (s, ',');
				}
				g_string_append (s, cheader_filename);
				cheader_filename = (g_free (cheader_filename), NULL);
			}
		}
		(cheader_filename_collection == NULL ? NULL : (cheader_filename_collection = (g_object_unref (cheader_filename_collection), NULL)));
		(cheader_filename_it == NULL ? NULL : (cheader_filename_it = (g_object_unref (cheader_filename_it), NULL)));
	}
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = (_tmp0 = s->str, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (s == NULL ? NULL : (s = (g_string_free (s, TRUE), NULL))), _tmp1);
}


/**
 * Sets the C header filename of this namespace to the specified
 * filename.
 *
 * @param cheader_filename header filename
 */
void vala_namespace_set_cheader_filename (ValaNamespace* self, const char* cheader_filename) {
	GeeList* _tmp0;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (cheader_filename != NULL);
	_tmp0 = NULL;
	self->priv->cheader_filenames = (_tmp0 = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_equal)), (self->priv->cheader_filenames == NULL ? NULL : (self->priv->cheader_filenames = (g_object_unref (self->priv->cheader_filenames), NULL))), _tmp0);
	gee_collection_add (GEE_COLLECTION (self->priv->cheader_filenames), cheader_filename);
}


static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cprefix")) {
		{
			char* _tmp0;
			char** _tmp1;
			char** name_collection;
			char** name_it;
			int name_collection_length1;
			_tmp0 = NULL;
			_tmp1 = NULL;
			name_collection = (_tmp1 = g_strsplit ((_tmp0 = vala_attribute_get_string (a, "cprefix")), ",", 0), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
			name_collection_length1 = -1;
			for (name_it = name_collection; *name_it != NULL; name_it = name_it + 1) {
				const char* _tmp2;
				char* name;
				_tmp2 = NULL;
				name = (_tmp2 = *name_it, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
				{
					vala_namespace_add_cprefix (self, name);
					name = (g_free (name), NULL);
				}
			}
			name_collection = (_vala_array_free (name_collection, name_collection_length1, ((GDestroyNotify) g_free)), NULL);
		}
	}
	if (vala_attribute_has_argument (a, "lower_case_cprefix")) {
		char* _tmp3;
		_tmp3 = NULL;
		vala_namespace_set_lower_case_cprefix (self, (_tmp3 = vala_attribute_get_string (a, "lower_case_cprefix")));
		_tmp3 = (g_free (_tmp3), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			int filename_collection_length1;
			filename_collection = g_strsplit (val, ",", 0);
			filename_collection_length1 = -1;
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* _tmp4;
				char* filename;
				_tmp4 = NULL;
				filename = (_tmp4 = *filename_it, (_tmp4 == NULL ? NULL : g_strdup (_tmp4)));
				{
					gee_collection_add (GEE_COLLECTION (self->priv->cheader_filenames), filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, ((GDestroyNotify) g_free)), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_namespace_process_attributes (ValaNamespace* self) {
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = ((ValaAttribute*) a_it->data), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_namespace_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static void vala_namespace_class_init (ValaNamespaceClass * klass) {
	vala_namespace_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaNamespacePrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_namespace_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_namespace_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_namespace_real_accept_children;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_namespace_real_get_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_namespace_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_cheader_filenames = vala_namespace_real_get_cheader_filenames;
}


static void vala_namespace_init (ValaNamespace * self) {
	self->priv = VALA_NAMESPACE_GET_PRIVATE (self);
	self->priv->classes = GEE_LIST (gee_array_list_new (VALA_TYPE_CLASS, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->interfaces = GEE_LIST (gee_array_list_new (VALA_TYPE_INTERFACE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->structs = GEE_LIST (gee_array_list_new (VALA_TYPE_STRUCT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->enums = GEE_LIST (gee_array_list_new (VALA_TYPE_ENUM, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->error_domains = GEE_LIST (gee_array_list_new (VALA_TYPE_ERROR_DOMAIN, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->delegates = GEE_LIST (gee_array_list_new (VALA_TYPE_DELEGATE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->constants = GEE_LIST (gee_array_list_new (VALA_TYPE_CONSTANT, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->fields = GEE_LIST (gee_array_list_new (VALA_TYPE_FIELD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (VALA_TYPE_METHOD, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->cprefixes = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_equal));
	self->priv->cheader_filenames = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_equal));
	self->priv->namespaces = GEE_LIST (gee_array_list_new (VALA_TYPE_NAMESPACE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_namespace_dispose (GObject * obj) {
	ValaNamespace * self;
	self = VALA_NAMESPACE (obj);
	(self->priv->classes == NULL ? NULL : (self->priv->classes = (g_object_unref (self->priv->classes), NULL)));
	(self->priv->interfaces == NULL ? NULL : (self->priv->interfaces = (g_object_unref (self->priv->interfaces), NULL)));
	(self->priv->structs == NULL ? NULL : (self->priv->structs = (g_object_unref (self->priv->structs), NULL)));
	(self->priv->enums == NULL ? NULL : (self->priv->enums = (g_object_unref (self->priv->enums), NULL)));
	(self->priv->error_domains == NULL ? NULL : (self->priv->error_domains = (g_object_unref (self->priv->error_domains), NULL)));
	(self->priv->delegates == NULL ? NULL : (self->priv->delegates = (g_object_unref (self->priv->delegates), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_object_unref (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_object_unref (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	(self->priv->cprefixes == NULL ? NULL : (self->priv->cprefixes = (g_object_unref (self->priv->cprefixes), NULL)));
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	(self->priv->cheader_filenames == NULL ? NULL : (self->priv->cheader_filenames = (g_object_unref (self->priv->cheader_filenames), NULL)));
	(self->priv->namespaces == NULL ? NULL : (self->priv->namespaces = (g_object_unref (self->priv->namespaces), NULL)));
	G_OBJECT_CLASS (vala_namespace_parent_class)->dispose (obj);
}


GType vala_namespace_get_type (void) {
	static GType vala_namespace_type_id = 0;
	if (G_UNLIKELY (vala_namespace_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaNamespaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_namespace_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaNamespace), 0, (GInstanceInitFunc) vala_namespace_init };
		vala_namespace_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaNamespace", &g_define_type_info, 0);
	}
	return vala_namespace_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




