/* valablock.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valablock.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valalocalvariable.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaBlockPrivate {
	gboolean _contains_jump_statement;
	GeeList* statement_list;
	GeeList* local_variables;
};

#define VALA_BLOCK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_BLOCK, ValaBlockPrivate))
enum  {
	VALA_BLOCK_DUMMY_PROPERTY
};
static void vala_block_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_block_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_block_parent_class = NULL;
static ValaStatementIface* vala_block_vala_statement_parent_iface = NULL;
static void vala_block_finalize (ValaCodeNode* obj);



/**
 * Creates a new block.
 *
 * @param source reference to source code
 */
ValaBlock* vala_block_construct (GType object_type, ValaSourceReference* source_reference) {
	ValaBlock* self;
	g_return_val_if_fail (VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = ((ValaBlock*) (g_type_create_instance (object_type)));
	vala_symbol_init (VALA_SYMBOL (self));
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


ValaBlock* vala_block_new (ValaSourceReference* source_reference) {
	return vala_block_construct (VALA_TYPE_BLOCK, source_reference);
}


/**
 * Append a statement to this block.
 *
 * @param stmt a statement
 */
void vala_block_add_statement (ValaBlock* self, ValaStatement* stmt) {
	g_return_if_fail (VALA_IS_BLOCK (self));
	g_return_if_fail (VALA_IS_STATEMENT (stmt));
	gee_collection_add (GEE_COLLECTION (self->priv->statement_list), stmt);
}


/**
 * Returns a copy of the list of statements.
 *
 * @return statement list
 */
GeeList* vala_block_get_statements (ValaBlock* self) {
	g_return_val_if_fail (VALA_IS_BLOCK (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_STATEMENT, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->statement_list));
}


/**
 * Add a local variable to this block.
 *
 * @param decl a variable declarator
 */
void vala_block_add_local_variable (ValaBlock* self, ValaLocalVariable* local) {
	g_return_if_fail (VALA_IS_BLOCK (self));
	g_return_if_fail (VALA_IS_LOCAL_VARIABLE (local));
	gee_collection_add (GEE_COLLECTION (self->priv->local_variables), local);
}


/**
 * Returns a copy of the list of local variables.
 *
 * @return variable declarator list
 */
GeeList* vala_block_get_local_variables (ValaBlock* self) {
	g_return_val_if_fail (VALA_IS_BLOCK (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_LOCAL_VARIABLE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, self->priv->local_variables));
}


static void vala_block_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBlock * self;
	self = VALA_BLOCK (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_block (visitor, self);
}


static void vala_block_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaBlock * self;
	self = VALA_BLOCK (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* stmt_collection;
		int stmt_it;
		stmt_collection = self->priv->statement_list;
		for (stmt_it = 0; stmt_it < gee_collection_get_size (GEE_COLLECTION (stmt_collection)); stmt_it = stmt_it + 1) {
			ValaStatement* stmt;
			stmt = ((ValaStatement*) (gee_list_get (GEE_LIST (stmt_collection), stmt_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (stmt), visitor);
				(stmt == NULL ? NULL : (stmt = (vala_code_node_unref (stmt), NULL)));
			}
		}
	}
}


gboolean vala_block_get_contains_jump_statement (ValaBlock* self) {
	g_return_val_if_fail (VALA_IS_BLOCK (self), FALSE);
	return self->priv->_contains_jump_statement;
}


void vala_block_set_contains_jump_statement (ValaBlock* self, gboolean value) {
	g_return_if_fail (VALA_IS_BLOCK (self));
	self->priv->_contains_jump_statement = value;
}


static void vala_block_class_init (ValaBlockClass * klass) {
	vala_block_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_block_finalize;
	g_type_class_add_private (klass, sizeof (ValaBlockPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_block_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_block_real_accept_children;
}


static void vala_block_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_block_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_block_instance_init (ValaBlock * self) {
	self->priv = VALA_BLOCK_GET_PRIVATE (self);
	self->priv->statement_list = GEE_LIST (gee_array_list_new (VALA_TYPE_STATEMENT, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
	self->priv->local_variables = GEE_LIST (gee_array_list_new (VALA_TYPE_LOCAL_VARIABLE, ((GBoxedCopyFunc) (vala_code_node_ref)), vala_code_node_unref, g_direct_equal));
}


static void vala_block_finalize (ValaCodeNode* obj) {
	ValaBlock * self;
	self = VALA_BLOCK (obj);
	(self->priv->statement_list == NULL ? NULL : (self->priv->statement_list = (gee_collection_object_unref (self->priv->statement_list), NULL)));
	(self->priv->local_variables == NULL ? NULL : (self->priv->local_variables = (gee_collection_object_unref (self->priv->local_variables), NULL)));
	VALA_CODE_NODE_CLASS (vala_block_parent_class)->finalize (obj);
}


GType vala_block_get_type (void) {
	static GType vala_block_type_id = 0;
	if (vala_block_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaBlockClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_block_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaBlock), 0, (GInstanceInitFunc) vala_block_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_block_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_block_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaBlock", &g_define_type_info, 0);
		g_type_add_interface_static (vala_block_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_block_type_id;
}




