/* valaattribute.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaattribute.h>
#include <gee/hashmap.h>
#include <vala/valaexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valarealliteral.h>
#include <vala/valaintegerliteral.h>
#include <vala/valabooleanliteral.h>
#include <vala/valaunaryexpression.h>
#include <vala/valastringliteral.h>




struct _ValaAttributePrivate {
	char* _name;
};

#define VALA_ATTRIBUTE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ATTRIBUTE, ValaAttributePrivate))
enum  {
	VALA_ATTRIBUTE_DUMMY_PROPERTY
};
static gpointer vala_attribute_parent_class = NULL;
static void vala_attribute_finalize (ValaCodeNode* obj);



/**
 * Creates a new attribute.
 *
 * @param name             attribute type name
 * @param source_reference reference to source code
 * @return                 newly created attribute
 */
ValaAttribute* vala_attribute_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaAttribute* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaAttribute*) g_type_create_instance (object_type);
	vala_attribute_set_name (self, name);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaAttribute* vala_attribute_new (const char* name, ValaSourceReference* source_reference) {
	return vala_attribute_construct (VALA_TYPE_ATTRIBUTE, name, source_reference);
}


/**
 * Adds an attribute argument.
 *
 * @param arg named argument
 */
void vala_attribute_add_argument (ValaAttribute* self, const char* key, ValaExpression* value) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (key != NULL);
	g_return_if_fail (value != NULL);
	gee_map_set (self->args, key, value);
}


/**
 * Returns whether this attribute has the specified named argument.
 *
 * @param name argument name
 * @return     true if the argument has been found, false otherwise
 */
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	return gee_map_contains (self->args, name);
}


/**
 * Returns the string value of the specified named argument.
 *
 * @param name argument name
 * @return     string value
 */
char* vala_attribute_get_string (ValaAttribute* self, const char* name) {
	ValaExpression* _tmp0;
	ValaStringLiteral* lit;
	char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	_tmp0 = NULL;
	lit = (_tmp0 = (ValaExpression*) gee_map_get (self->args, name), VALA_IS_STRING_LITERAL (_tmp0) ? ((ValaStringLiteral*) _tmp0) : NULL);
	if (lit != NULL) {
		char* _tmp1;
		_tmp1 = NULL;
		return (_tmp1 = vala_string_literal_eval (lit), (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), _tmp1);
	}
	_tmp2 = NULL;
	return (_tmp2 = NULL, (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), _tmp2);
}


/**
 * Returns the integer value of the specified named argument.
 *
 * @param name argument name
 * @return     integer value
 */
gint vala_attribute_get_integer (ValaAttribute* self, const char* name) {
	ValaExpression* _tmp0;
	ValaIntegerLiteral* lit;
	gint _tmp2;
	g_return_val_if_fail (self != NULL, 0);
	g_return_val_if_fail (name != NULL, 0);
	_tmp0 = NULL;
	lit = (_tmp0 = (ValaExpression*) gee_map_get (self->args, name), VALA_IS_INTEGER_LITERAL (_tmp0) ? ((ValaIntegerLiteral*) _tmp0) : NULL);
	if (lit != NULL) {
		gint _tmp1;
		return (_tmp1 = atoi (vala_integer_literal_get_value (lit)), (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), _tmp1);
	}
	return (_tmp2 = 0, (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), _tmp2);
}


/**
 * Returns the double value of the specified named argument.
 *
 * @param name argument name
 * @return     double value
 */
double vala_attribute_get_double (ValaAttribute* self, const char* name) {
	ValaExpression* arg;
	double _tmp9;
	g_return_val_if_fail (self != NULL, 0.0);
	g_return_val_if_fail (name != NULL, 0.0);
	arg = (ValaExpression*) gee_map_get (self->args, name);
	if (VALA_IS_REAL_LITERAL (arg)) {
		ValaRealLiteral* _tmp0;
		ValaRealLiteral* lit;
		double _tmp1;
		_tmp0 = NULL;
		lit = (_tmp0 = VALA_REAL_LITERAL (arg), (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
		return (_tmp1 = strtod (vala_real_literal_get_value (lit), NULL), (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), (arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL)), _tmp1);
	} else {
		if (VALA_IS_INTEGER_LITERAL (arg)) {
			ValaIntegerLiteral* _tmp2;
			ValaIntegerLiteral* lit;
			double _tmp3;
			_tmp2 = NULL;
			lit = (_tmp2 = VALA_INTEGER_LITERAL (arg), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
			return (_tmp3 = (double) atoi (vala_integer_literal_get_value (lit)), (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), (arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL)), _tmp3);
		} else {
			if (VALA_IS_UNARY_EXPRESSION (arg)) {
				ValaUnaryExpression* _tmp4;
				ValaUnaryExpression* unary;
				_tmp4 = NULL;
				unary = (_tmp4 = VALA_UNARY_EXPRESSION (arg), (_tmp4 == NULL) ? NULL : vala_code_node_ref (_tmp4));
				if (vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_MINUS) {
					if (VALA_IS_REAL_LITERAL (vala_unary_expression_get_inner (unary))) {
						ValaRealLiteral* _tmp5;
						ValaRealLiteral* lit;
						double _tmp6;
						_tmp5 = NULL;
						lit = (_tmp5 = VALA_REAL_LITERAL (vala_unary_expression_get_inner (unary)), (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5));
						return (_tmp6 = -strtod (vala_real_literal_get_value (lit), NULL), (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), (unary == NULL) ? NULL : (unary = (vala_code_node_unref (unary), NULL)), (arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL)), _tmp6);
					} else {
						if (VALA_IS_INTEGER_LITERAL (vala_unary_expression_get_inner (unary))) {
							ValaIntegerLiteral* _tmp7;
							ValaIntegerLiteral* lit;
							double _tmp8;
							_tmp7 = NULL;
							lit = (_tmp7 = VALA_INTEGER_LITERAL (vala_unary_expression_get_inner (unary)), (_tmp7 == NULL) ? NULL : vala_code_node_ref (_tmp7));
							return (_tmp8 = (double) (-atoi (vala_integer_literal_get_value (lit))), (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), (unary == NULL) ? NULL : (unary = (vala_code_node_unref (unary), NULL)), (arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL)), _tmp8);
						}
					}
				}
				(unary == NULL) ? NULL : (unary = (vala_code_node_unref (unary), NULL));
			}
		}
	}
	return (_tmp9 = (double) 0, (arg == NULL) ? NULL : (arg = (vala_code_node_unref (arg), NULL)), _tmp9);
}


/**
 * Returns the boolean value of the specified named argument.
 *
 * @param name argument name
 * @return     boolean value
 */
gboolean vala_attribute_get_bool (ValaAttribute* self, const char* name) {
	ValaExpression* _tmp0;
	ValaBooleanLiteral* lit;
	gboolean _tmp2;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	_tmp0 = NULL;
	lit = (_tmp0 = (ValaExpression*) gee_map_get (self->args, name), VALA_IS_BOOLEAN_LITERAL (_tmp0) ? ((ValaBooleanLiteral*) _tmp0) : NULL);
	if (lit != NULL) {
		gboolean _tmp1;
		return (_tmp1 = vala_boolean_literal_get_value (lit), (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), _tmp1);
	}
	return (_tmp2 = FALSE, (lit == NULL) ? NULL : (lit = (vala_code_node_unref (lit), NULL)), _tmp2);
}


const char* vala_attribute_get_name (ValaAttribute* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_name;
}


void vala_attribute_set_name (ValaAttribute* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), self->priv->_name = (g_free (self->priv->_name), NULL), _tmp2);
}


static void vala_attribute_class_init (ValaAttributeClass * klass) {
	vala_attribute_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_attribute_finalize;
	g_type_class_add_private (klass, sizeof (ValaAttributePrivate));
}


static void vala_attribute_instance_init (ValaAttribute * self) {
	self->priv = VALA_ATTRIBUTE_GET_PRIVATE (self);
	self->args = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_str_hash, g_str_equal, g_direct_equal);
}


static void vala_attribute_finalize (ValaCodeNode* obj) {
	ValaAttribute * self;
	self = VALA_ATTRIBUTE (obj);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	(self->args == NULL) ? NULL : (self->args = (gee_collection_object_unref (self->args), NULL));
	VALA_CODE_NODE_CLASS (vala_attribute_parent_class)->finalize (obj);
}


GType vala_attribute_get_type (void) {
	static GType vala_attribute_type_id = 0;
	if (vala_attribute_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaAttributeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_attribute_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaAttribute), 0, (GInstanceInitFunc) vala_attribute_instance_init, NULL };
		vala_attribute_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaAttribute", &g_define_type_info, 0);
	}
	return vala_attribute_type_id;
}




