/* valamarkupreader.vala
 *
 * Copyright (C) 2008-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <vala.h>


#define VALA_TYPE_MARKUP_READER (vala_markup_reader_get_type ())
#define VALA_MARKUP_READER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MARKUP_READER, ValaMarkupReader))
#define VALA_MARKUP_READER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MARKUP_READER, ValaMarkupReaderClass))
#define VALA_IS_MARKUP_READER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MARKUP_READER))
#define VALA_IS_MARKUP_READER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MARKUP_READER))
#define VALA_MARKUP_READER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MARKUP_READER, ValaMarkupReaderClass))

typedef struct _ValaMarkupReader ValaMarkupReader;
typedef struct _ValaMarkupReaderClass ValaMarkupReaderClass;
typedef struct _ValaMarkupReaderPrivate ValaMarkupReaderPrivate;

#define VALA_TYPE_MARKUP_TOKEN_TYPE (vala_markup_token_type_get_type ())

/**
 * Simple reader for a subset of XML.
 */
struct _ValaMarkupReader {
	GObject parent_instance;
	ValaMarkupReaderPrivate * priv;
};

struct _ValaMarkupReaderClass {
	GObjectClass parent_class;
};

struct _ValaMarkupReaderPrivate {
	char* _filename;
	char* _name;
	GMappedFile* mapped_file;
	gchar* begin;
	gchar* current;
	gchar* end;
	gint line;
	gint column;
	GeeMap* attributes;
	gboolean empty_element;
};

typedef enum  {
	VALA_MARKUP_TOKEN_TYPE_NONE,
	VALA_MARKUP_TOKEN_TYPE_START_ELEMENT,
	VALA_MARKUP_TOKEN_TYPE_END_ELEMENT,
	VALA_MARKUP_TOKEN_TYPE_TEXT,
	VALA_MARKUP_TOKEN_TYPE_EOF
} ValaMarkupTokenType;



GType vala_markup_reader_get_type (void);
#define VALA_MARKUP_READER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MARKUP_READER, ValaMarkupReaderPrivate))
enum  {
	VALA_MARKUP_READER_DUMMY_PROPERTY,
	VALA_MARKUP_READER_FILENAME,
	VALA_MARKUP_READER_NAME
};
static void vala_markup_reader_set_filename (ValaMarkupReader* self, const char* value);
ValaMarkupReader* vala_markup_reader_new (const char* filename);
ValaMarkupReader* vala_markup_reader_construct (GType object_type, const char* filename);
ValaMarkupReader* vala_markup_reader_new (const char* filename);
char* vala_markup_reader_get_attribute (ValaMarkupReader* self, const char* attr);
static char* vala_markup_reader_read_name (ValaMarkupReader* self);
GType vala_markup_token_type_get_type (void);
static void vala_markup_reader_space (ValaMarkupReader* self);
ValaMarkupTokenType vala_markup_reader_read_token (ValaMarkupReader* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end);
static void vala_markup_reader_set_name (ValaMarkupReader* self, const char* value);
const char* vala_markup_reader_get_filename (ValaMarkupReader* self);
const char* vala_markup_reader_get_name (ValaMarkupReader* self);
static GObject * vala_markup_reader_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_markup_reader_parent_class = NULL;
static void vala_markup_reader_finalize (GObject* obj);
static void vala_markup_reader_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void vala_markup_reader_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
const char* vala_markup_token_type_to_string (ValaMarkupTokenType self);



ValaMarkupReader* vala_markup_reader_construct (GType object_type, const char* filename) {
	GParameter * __params;
	GParameter * __params_it;
	ValaMarkupReader * self;
	g_return_val_if_fail (filename != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "filename";
	g_value_init (&__params_it->value, G_TYPE_STRING);
	g_value_set_string (&__params_it->value, filename);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


ValaMarkupReader* vala_markup_reader_new (const char* filename) {
	return vala_markup_reader_construct (VALA_TYPE_MARKUP_READER, filename);
}


char* vala_markup_reader_get_attribute (ValaMarkupReader* self, const char* attr) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (attr != NULL, NULL);
	return (char*) gee_map_get (self->priv->attributes, attr);
}


static char* vala_markup_reader_read_name (ValaMarkupReader* self) {
	gchar* begin;
	g_return_val_if_fail (self != NULL, NULL);
	begin = self->priv->current;
	while (self->priv->current < self->priv->end) {
		gboolean _tmp0_;
		gboolean _tmp1_;
		gboolean _tmp2_;
		gunichar u;
		_tmp0_ = FALSE;
		_tmp1_ = FALSE;
		_tmp2_ = FALSE;
		if (self->priv->current[0] == ' ') {
			_tmp2_ = TRUE;
		} else {
			_tmp2_ = self->priv->current[0] == '>';
		}
		if (_tmp2_) {
			_tmp1_ = TRUE;
		} else {
			_tmp1_ = self->priv->current[0] == '/';
		}
		if (_tmp1_) {
			_tmp0_ = TRUE;
		} else {
			_tmp0_ = self->priv->current[0] == '=';
		}
		if (_tmp0_) {
			break;
		}
		u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
		if (u != ((gunichar) (-1))) {
			self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
		} else {
			vala_report_error (NULL, "invalid UTF-8 character");
		}
	}
	if (self->priv->current == begin) {
	}
	/* syntax error: invalid name*/
	return g_strndup ((const char*) begin, self->priv->current - begin);
}


ValaMarkupTokenType vala_markup_reader_read_token (ValaMarkupReader* self, ValaSourceLocation* token_begin, ValaSourceLocation* token_end) {
	ValaMarkupTokenType type;
	gchar* begin;
	g_return_val_if_fail (self != NULL, 0);
	gee_map_clear (self->priv->attributes);
	if (self->priv->empty_element) {
		self->priv->empty_element = FALSE;
		return VALA_MARKUP_TOKEN_TYPE_END_ELEMENT;
	}
	vala_markup_reader_space (self);
	type = VALA_MARKUP_TOKEN_TYPE_NONE;
	begin = self->priv->current;
	(*token_begin).pos = begin;
	(*token_begin).line = self->priv->line;
	(*token_begin).column = self->priv->column;
	if (self->priv->current >= self->priv->end) {
		type = VALA_MARKUP_TOKEN_TYPE_EOF;
	} else {
		if (self->priv->current[0] == '<') {
			self->priv->current++;
			if (self->priv->current >= self->priv->end) {
			} else {
				if (self->priv->current[0] == '?') {
				} else {
					if (self->priv->current[0] == '!') {
						gboolean _tmp1_;
						gboolean _tmp2_;
						/* comment or doctype*/
						self->priv->current++;
						_tmp1_ = FALSE;
						_tmp2_ = FALSE;
						if (self->priv->current < (self->priv->end - 1)) {
							_tmp2_ = self->priv->current[0] == '-';
						} else {
							_tmp2_ = FALSE;
						}
						if (_tmp2_) {
							_tmp1_ = self->priv->current[1] == '-';
						} else {
							_tmp1_ = FALSE;
						}
						if (_tmp1_) {
							/* comment*/
							self->priv->current = self->priv->current + 2;
							while (self->priv->current < (self->priv->end - 2)) {
								gboolean _tmp3_;
								gboolean _tmp4_;
								_tmp3_ = FALSE;
								_tmp4_ = FALSE;
								if (self->priv->current[0] == '-') {
									_tmp4_ = self->priv->current[1] == '-';
								} else {
									_tmp4_ = FALSE;
								}
								if (_tmp4_) {
									_tmp3_ = self->priv->current[2] == '>';
								} else {
									_tmp3_ = FALSE;
								}
								if (_tmp3_) {
									/* end of comment*/
									self->priv->current = self->priv->current + 3;
									break;
								}
								self->priv->current++;
							}
							/* ignore comment, read next token*/
							return vala_markup_reader_read_token (self, &(*token_begin), &(*token_end));
						}
					} else {
						if (self->priv->current[0] == '/') {
							char* _tmp6_;
							gboolean _tmp7_;
							type = VALA_MARKUP_TOKEN_TYPE_END_ELEMENT;
							self->priv->current++;
							_tmp6_ = NULL;
							vala_markup_reader_set_name (self, _tmp6_ = vala_markup_reader_read_name (self));
							_tmp6_ = (g_free (_tmp6_), NULL);
							_tmp7_ = FALSE;
							if (self->priv->current >= self->priv->end) {
								_tmp7_ = TRUE;
							} else {
								_tmp7_ = self->priv->current[0] != '>';
							}
							if (_tmp7_) {
							}
							/* error*/
							self->priv->current++;
						} else {
							char* _tmp8_;
							gboolean _tmp15_;
							type = VALA_MARKUP_TOKEN_TYPE_START_ELEMENT;
							_tmp8_ = NULL;
							vala_markup_reader_set_name (self, _tmp8_ = vala_markup_reader_read_name (self));
							_tmp8_ = (g_free (_tmp8_), NULL);
							vala_markup_reader_space (self);
							while (TRUE) {
								gboolean _tmp9_;
								gboolean _tmp10_;
								char* attr_name;
								gboolean _tmp11_;
								gboolean _tmp12_;
								gchar* attr_begin;
								char* attr_value;
								gboolean _tmp14_;
								_tmp9_ = FALSE;
								_tmp10_ = FALSE;
								if (self->priv->current < self->priv->end) {
									_tmp10_ = self->priv->current[0] != '>';
								} else {
									_tmp10_ = FALSE;
								}
								if (_tmp10_) {
									_tmp9_ = self->priv->current[0] != '/';
								} else {
									_tmp9_ = FALSE;
								}
								if (!_tmp9_) {
									break;
								}
								attr_name = vala_markup_reader_read_name (self);
								_tmp11_ = FALSE;
								if (self->priv->current >= self->priv->end) {
									_tmp11_ = TRUE;
								} else {
									_tmp11_ = self->priv->current[0] != '=';
								}
								if (_tmp11_) {
								}
								/* error*/
								self->priv->current++;
								_tmp12_ = FALSE;
								if (self->priv->current >= self->priv->end) {
									_tmp12_ = TRUE;
								} else {
									_tmp12_ = self->priv->current[0] != '"';
								}
								/* FIXME allow single quotes*/
								if (_tmp12_) {
								}
								/* error*/
								self->priv->current++;
								attr_begin = self->priv->current;
								while (TRUE) {
									gboolean _tmp13_;
									gunichar u;
									_tmp13_ = FALSE;
									if (self->priv->current < self->priv->end) {
										_tmp13_ = self->priv->current[0] != '"';
									} else {
										_tmp13_ = FALSE;
									}
									if (!_tmp13_) {
										break;
									}
									u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
									if (u != ((gunichar) (-1))) {
										self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
									} else {
										vala_report_error (NULL, "invalid UTF-8 character");
									}
								}
								/* TODO process &amp; &gt; &lt; &quot; &apos;*/
								attr_value = g_strndup ((const char*) attr_begin, self->priv->current - attr_begin);
								_tmp14_ = FALSE;
								if (self->priv->current >= self->priv->end) {
									_tmp14_ = TRUE;
								} else {
									_tmp14_ = self->priv->current[0] != '"';
								}
								if (_tmp14_) {
								}
								/* error*/
								self->priv->current++;
								gee_map_set (self->priv->attributes, attr_name, attr_value);
								vala_markup_reader_space (self);
								attr_name = (g_free (attr_name), NULL);
								attr_value = (g_free (attr_value), NULL);
							}
							if (self->priv->current[0] == '/') {
								self->priv->empty_element = TRUE;
								self->priv->current++;
								vala_markup_reader_space (self);
							} else {
								self->priv->empty_element = FALSE;
							}
							_tmp15_ = FALSE;
							if (self->priv->current >= self->priv->end) {
								_tmp15_ = TRUE;
							} else {
								_tmp15_ = self->priv->current[0] != '>';
							}
							if (_tmp15_) {
							}
							/* error*/
							self->priv->current++;
						}
					}
				}
			}
		} else {
			gchar* text_begin;
			vala_markup_reader_space (self);
			text_begin = self->priv->current;
			while (TRUE) {
				gboolean _tmp16_;
				gunichar u;
				_tmp16_ = FALSE;
				if (self->priv->current < self->priv->end) {
					_tmp16_ = self->priv->current[0] != '<';
				} else {
					_tmp16_ = FALSE;
				}
				if (!_tmp16_) {
					break;
				}
				u = g_utf8_get_char_validated ((const char*) self->priv->current, (glong) (self->priv->end - self->priv->current));
				if (u != ((gunichar) (-1))) {
					self->priv->current = self->priv->current + g_unichar_to_utf8 (u, NULL);
				} else {
					vala_report_error (NULL, "invalid UTF-8 character");
				}
			}
			if (text_begin == self->priv->current) {
				/* no text
				 read next token*/
				return vala_markup_reader_read_token (self, &(*token_begin), &(*token_end));
			}
			type = VALA_MARKUP_TOKEN_TYPE_TEXT;
		}
	}
	/* TODO process &amp; &gt; &lt; &quot; &apos;
	 string text = ((string) text_begin).ndup (current - text_begin);*/
	self->priv->column = self->priv->column + ((gint) (self->priv->current - begin));
	(*token_end).pos = self->priv->current;
	(*token_end).line = self->priv->line;
	(*token_end).column = self->priv->column - 1;
	return type;
}


static void vala_markup_reader_space (ValaMarkupReader* self) {
	g_return_if_fail (self != NULL);
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->current < self->priv->end) {
			_tmp0_ = g_ascii_isspace (self->priv->current[0]);
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		if (self->priv->current[0] == '\n') {
			self->priv->line++;
			self->priv->column = 0;
		}
		self->priv->current++;
		self->priv->column++;
	}
}


const char* vala_markup_reader_get_filename (ValaMarkupReader* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_filename;
}


static void vala_markup_reader_set_filename (ValaMarkupReader* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_filename = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_filename = (g_free (self->priv->_filename), NULL), _tmp2_);
	g_object_notify ((GObject *) self, "filename");
}


const char* vala_markup_reader_get_name (ValaMarkupReader* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_name;
}


static void vala_markup_reader_set_name (ValaMarkupReader* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_name = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_name = (g_free (self->priv->_name), NULL), _tmp2_);
	g_object_notify ((GObject *) self, "name");
}


static GObject * vala_markup_reader_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	ValaMarkupReaderClass * klass;
	GObjectClass * parent_class;
	ValaMarkupReader * self;
	GError * _inner_error_;
	klass = VALA_MARKUP_READER_CLASS (g_type_class_peek (VALA_TYPE_MARKUP_READER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = VALA_MARKUP_READER (obj);
	_inner_error_ = NULL;
	{
		{
			GMappedFile* _tmp0_;
			_tmp0_ = NULL;
			self->priv->mapped_file = (_tmp0_ = g_mapped_file_new (self->priv->_filename, FALSE, &_inner_error_), (self->priv->mapped_file == NULL) ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL)), _tmp0_);
			if (_inner_error_ != NULL) {
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch3_g_file_error;
				}
				goto __finally3;
			}
			self->priv->begin = g_mapped_file_get_contents (self->priv->mapped_file);
			self->priv->end = self->priv->begin + g_mapped_file_get_length (self->priv->mapped_file);
			self->priv->current = self->priv->begin;
			self->priv->line = 1;
			self->priv->column = 1;
		}
		goto __finally3;
		__catch3_g_file_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				char* _tmp1_;
				_tmp1_ = NULL;
				vala_report_error (NULL, _tmp1_ = g_strdup_printf ("Unable to map file `%s': %s", self->priv->_filename, e->message));
				_tmp1_ = (g_free (_tmp1_), NULL);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally3:
		if (_inner_error_ != NULL) {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
		}
	}
	return obj;
}


static void vala_markup_reader_class_init (ValaMarkupReaderClass * klass) {
	vala_markup_reader_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMarkupReaderPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_markup_reader_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_markup_reader_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_markup_reader_constructor;
	G_OBJECT_CLASS (klass)->finalize = vala_markup_reader_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MARKUP_READER_FILENAME, g_param_spec_string ("filename", "filename", "filename", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_MARKUP_READER_NAME, g_param_spec_string ("name", "name", "name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_markup_reader_instance_init (ValaMarkupReader * self) {
	self->priv = VALA_MARKUP_READER_GET_PRIVATE (self);
	self->priv->attributes = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
}


static void vala_markup_reader_finalize (GObject* obj) {
	ValaMarkupReader * self;
	self = VALA_MARKUP_READER (obj);
	self->priv->_filename = (g_free (self->priv->_filename), NULL);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	(self->priv->mapped_file == NULL) ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL));
	(self->priv->attributes == NULL) ? NULL : (self->priv->attributes = (gee_collection_object_unref (self->priv->attributes), NULL));
	G_OBJECT_CLASS (vala_markup_reader_parent_class)->finalize (obj);
}


GType vala_markup_reader_get_type (void) {
	static GType vala_markup_reader_type_id = 0;
	if (vala_markup_reader_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMarkupReaderClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_markup_reader_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMarkupReader), 0, (GInstanceInitFunc) vala_markup_reader_instance_init, NULL };
		vala_markup_reader_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaMarkupReader", &g_define_type_info, 0);
	}
	return vala_markup_reader_type_id;
}


static void vala_markup_reader_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaMarkupReader * self;
	gpointer boxed;
	self = VALA_MARKUP_READER (object);
	switch (property_id) {
		case VALA_MARKUP_READER_FILENAME:
		g_value_set_string (value, vala_markup_reader_get_filename (self));
		break;
		case VALA_MARKUP_READER_NAME:
		g_value_set_string (value, vala_markup_reader_get_name (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_markup_reader_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaMarkupReader * self;
	self = VALA_MARKUP_READER (object);
	switch (property_id) {
		case VALA_MARKUP_READER_FILENAME:
		vala_markup_reader_set_filename (self, g_value_get_string (value));
		break;
		case VALA_MARKUP_READER_NAME:
		vala_markup_reader_set_name (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


const char* vala_markup_token_type_to_string (ValaMarkupTokenType self) {
	switch (self) {
		case VALA_MARKUP_TOKEN_TYPE_START_ELEMENT:
		{
			return "start element";
		}
		case VALA_MARKUP_TOKEN_TYPE_END_ELEMENT:
		{
			return "end element";
		}
		case VALA_MARKUP_TOKEN_TYPE_TEXT:
		{
			return "text";
		}
		case VALA_MARKUP_TOKEN_TYPE_EOF:
		{
			return "end of file";
		}
		default:
		{
			return "unknown token type";
		}
	}
}



GType vala_markup_token_type_get_type (void) {
	static GType vala_markup_token_type_type_id = 0;
	if (G_UNLIKELY (vala_markup_token_type_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_MARKUP_TOKEN_TYPE_NONE, "VALA_MARKUP_TOKEN_TYPE_NONE", "none"}, {VALA_MARKUP_TOKEN_TYPE_START_ELEMENT, "VALA_MARKUP_TOKEN_TYPE_START_ELEMENT", "start-element"}, {VALA_MARKUP_TOKEN_TYPE_END_ELEMENT, "VALA_MARKUP_TOKEN_TYPE_END_ELEMENT", "end-element"}, {VALA_MARKUP_TOKEN_TYPE_TEXT, "VALA_MARKUP_TOKEN_TYPE_TEXT", "text"}, {VALA_MARKUP_TOKEN_TYPE_EOF, "VALA_MARKUP_TOKEN_TYPE_EOF", "eof"}, {0, NULL, NULL}};
		vala_markup_token_type_type_id = g_enum_register_static ("ValaMarkupTokenType", values);
	}
	return vala_markup_token_type_type_id;
}




