/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { type DetectedIde } from '../ide/detect-ide.js';
import type { DiffUpdateResult } from '../ide/ideContext.js';
export type IDEConnectionState = {
    status: IDEConnectionStatus;
    details?: string;
};
export declare enum IDEConnectionStatus {
    Connected = "connected",
    Disconnected = "disconnected",
    Connecting = "connecting"
}
/**
 * Manages the connection to and interaction with the IDE server.
 */
export declare class IdeClient {
    private static instancePromise;
    private client;
    private state;
    private currentIde;
    private currentIdeDisplayName;
    private ideProcessInfo;
    private diffResponses;
    private statusListeners;
    private trustChangeListeners;
    private constructor();
    static getInstance(): Promise<IdeClient>;
    addStatusChangeListener(listener: (state: IDEConnectionState) => void): void;
    removeStatusChangeListener(listener: (state: IDEConnectionState) => void): void;
    addTrustChangeListener(listener: (isTrusted: boolean) => void): void;
    removeTrustChangeListener(listener: (isTrusted: boolean) => void): void;
    connect(): Promise<void>;
    /**
     * A diff is accepted with any modifications if the user performs one of the
     * following actions:
     * - Clicks the checkbox icon in the IDE to accept
     * - Runs `command+shift+p` > "Gemini CLI: Accept Diff in IDE" to accept
     * - Selects "accept" in the CLI UI
     * - Saves the file via `ctrl/command+s`
     *
     * A diff is rejected if the user performs one of the following actions:
     * - Clicks the "x" icon in the IDE
     * - Runs "Gemini CLI: Close Diff in IDE"
     * - Selects "no" in the CLI UI
     * - Closes the file
     */
    openDiff(filePath: string, newContent?: string): Promise<DiffUpdateResult>;
    closeDiff(filePath: string, options?: {
        suppressNotification?: boolean;
    }): Promise<string | undefined>;
    resolveDiffFromCli(filePath: string, outcome: 'accepted' | 'rejected'): Promise<void>;
    disconnect(): Promise<void>;
    getCurrentIde(): DetectedIde | undefined;
    getConnectionStatus(): IDEConnectionState;
    getDetectedIdeDisplayName(): string | undefined;
    private setState;
    static validateWorkspacePath(ideWorkspacePath: string | undefined, currentIdeDisplayName: string | undefined, cwd: string): {
        isValid: boolean;
        error?: string;
    };
    private getPortFromEnv;
    private getStdioConfigFromEnv;
    private getConnectionConfigFromFile;
    private createProxyAwareFetch;
    private registerClientHandlers;
    private establishHttpConnection;
    private establishStdioConnection;
}
