/*
Copyright (c) 2019-2023,
Lawrence Livermore National Security, LLC;
See the top-level NOTICE for additional details. All rights reserved.
SPDX-License-Identifier: BSD-3-Clause
*/
#include "units.hpp"

#include <algorithm>
#include <array>
#include <cstring>
#include <tuple>

namespace UNITS_NAMESPACE {
using unitD = std::tuple<const char*, const char*, precise_unit>;
static UNITS_CPP14_CONSTEXPR_OBJECT std::array<unitD, 486> x12_units{{
    unitD{"03", "SECOND", precise::s},
    unitD{"05", "LIFT", precise::one},
    unitD{"07", "STRAND", precise::one},
    unitD{"09", "TIRE", precise::one},
    unitD{"10", "GROUP", precise::one},
    unitD{"11", "OUTFIT", precise::one},
    unitD{"12", "PACKET", precise::one},
    unitD{"13", "RATION", precise::one},
    unitD{"14", "SHOT", precise::one},
    unitD{"15", "STICK", precise::one},
    unitD{"17", "100 POUND DRUM", precise_unit(100.0, precise::lb)},
    unitD{"18", "55 GALLON DRUM", precise_unit(55.0, precise::lb)},
    unitD{"19", "TANK TRUCK", precise::one},
    unitD{"1N", "COUNT", precise::count},
    unitD{"1O", "SEASON", precise::one},
    unitD{"1P", "TANK CAR", precise::one},
    unitD{"1Q", "FRAME", precise::one},
    unitD{"1R", "TRANSACTION", precise::one},
    unitD{"1X", "QUARTER MILE", precise_unit(0.25, precise::mile)},
    unitD{"20", "20 FOOT CONTAINER", precise::one},
    unitD{"21", "40 FOOT CONTAINER", precise::one},
    unitD{"2D", "SHIP SET", precise::one},
    unitD{"2I", "BRITISH THERMAL UNITS/HOUR", precise::btu / precise::hr},
    unitD{"2N", "DECIBEL", precise::one},
    unitD{"2R", "KILOCURIE", precise::kilo* precise::cgs::curie},
    unitD{"2U", "MEGAGRAM", precise::mega* precise::g},
    unitD{"2W", "BIN", precise::one},
    unitD{"43", "SUPER BULK BAG (SAME SIZE AS PALLET)", precise::one},
    unitD{"44", "500 KILOGRAM BULK BAG", precise_unit(500.0, precise::kg)},
    unitD{"45", "300 KILOGRAM BULK BAG", precise_unit(300.0, precise::kg)},
    unitD{"46", "25 KILOGRAM BULK BAG", precise_unit(25.0, precise::kg)},
    unitD{"47", "50 POUND BAG", precise_unit(50.0, precise::kg)},
    unitD{"48", "BULK CAR LOAD", precise::one},
    unitD{"4A", "BOBBIN", precise::one},
    unitD{"4B", "CAP", precise::one},
    unitD{"4E", "20-PACK", precise_unit(20, precise::count)},
    unitD{"4F", "100-PACK", precise_unit(100, precise::count)},
    unitD{"4G", "MICROLITER", precise::micro* precise::L},
    unitD{"51", "ACTUAL TONNES", precise::one},
    unitD{"57", "MESH", precise::one},
    unitD{"58", "NET KILOGRAMS", precise::one},
    unitD{"5B", "BATCH", precise::one},
    unitD{"5D", "PROOF GALLONS", precise::one},
    unitD{"5G", "PUMP", precise::one},
    unitD{"5H", "STAGE", precise::one},
    unitD{"5I", "STANDARD CUBE FOOT", precise::one},
    unitD{"70", "VOLT", precise::volt},
    unitD{"7A", "LANDINGS", precise::one},
    unitD{"7C", "FLIGHT HOURS", precise::one},
    unitD{"8C", "CORD", precise::one},
    unitD{"8D", "DUTY", precise::one},
    unitD{"8P", "PROJECT", precise::one},
    unitD{"8R", "PROGRAM", precise::one},
    unitD{"8S", "SESSION", precise::one},
    unitD{"8U", "SQUARE KILOMETER", precise::km.pow(2)},
    unitD{"91", "STOKE", precise::one},
    unitD{"94", "CURL UNIT", precise::one},
    unitD{
        "95",
        "20,000 GALLON TANKCAR",
        precise_unit(20000, precise::gal) / precise::count},
    unitD{
        "96",
        "10,000 GALLON TANKCAR",
        precise_unit(10000, precise::gal) / precise::count},
    unitD{
        "97",
        "10 KILOGRAM DRUM",
        precise_unit(10, precise::kg) / precise::count},
    unitD{
        "98",
        "15 KILOGRAM DRUM",
        precise_unit(15, precise::kg) / precise::count},
    unitD{"99", "WATT", precise::watt},
    unitD{"9A", "APOTHECARY POUND", precise::one},
    unitD{"9B", "BRIQUET", precise::one},
    unitD{"9F", "FOLD", precise::one},
    unitD{"9O", "APOTHECARY OUNCE", precise::one},
    unitD{"9P", "PILLOW", precise::one},
    unitD{"9S", "SEAT", precise::one},
    unitD{"9Y", "FAMILY UNIT", precise::one},
    unitD{"9Z", "USP UNIT", precise::one},
    unitD{"AA", "BALL", precise::one},
    unitD{"AB", "BULK PACK", precise::one},
    unitD{"AC", "ACRE", precise::acre},
    unitD{"AD", "BYTE", precise::B},
    unitD{"AF", "CENTIGRAM", precise_unit(10.0, precise::mg)},
    unitD{"AJ", "COP (CYLINDRICAL OR CONICAL MASS)", precise::one},
    unitD{"AK", "FATHOM", precise::one},
    unitD{"AL", "ACCESS LINES", precise::one},
    unitD{"AM", "AMPOULE", precise::one},
    unitD{"AR", "SUPPOSITORY", precise::one},
    unitD{"AS", "ASSORTMENT", precise::one},
    unitD{"AV", "CAPSULE", precise::one},
    unitD{"AW", "POWDER-FILLED VIAL", precise::one},
    unitD{"AX", "TWENTY", precise_unit(20.0, precise::one)},
    unitD{"AY", "ASSEMBLY", precise::one},
    unitD{
        "B0",
        "BRITISH THERMAL UNITS PER CUBIC FOOT",
        precise::btu / (precise::ft.pow(3))},
    unitD{"B2", "BUNK", precise::one},
    unitD{"B3", "BATTING POUND", precise::one},
    unitD{"B4", "BARREL IMPERIAL", precise::one},
    unitD{"B5", "BILLET", precise::one},
    unitD{"B6", "BUN", precise::one},
    unitD{"B7", "CYCLE", precise::one},
    unitD{"B8", "BOARD", precise::one},
    unitD{"B9", "BATT", precise::one},
    unitD{"BA", "BALE", precise::one},
    unitD{"BB", "BASS BOX", precise::one},
    unitD{"BC", "BUCKET", precise::one},
    unitD{"BD", "BUNDLE", precise::one},
    unitD{"BE", "BEAM", precise::one},
    unitD{"BF", "BOARD FOOT", precise::one},
    unitD{"BG", "BAG", precise::one},
    unitD{"BH", "BRUSH", precise::one},
    unitD{"BI", "BAR", precise::one},
    unitD{"BJ", "BAND", precise::one},
    unitD{"BK", "BOOK", precise::one},
    unitD{"BL", "BLOCK", precise::one},
    unitD{"BM", "BOLT", precise::one},
    unitD{"BN", "BULK", precise::one},
    unitD{"BO", "BOTTLE", precise::one},
    unitD{"BP", "100 BOARD FEET", precise::one},
    unitD{"BR", "BARREL", precise::one},
    unitD{"BS", "BASKET", precise::one},
    unitD{"BT", "BELT", precise::one},
    unitD{"BU", "BUSHEL (32 DRY QUARTS)", precise::one},
    unitD{"BV", "BUSHEL DRY IMPERIAL", precise::one},
    unitD{"BW", "BASE WEIGHT", precise::one},
    unitD{"BX", "BOX", precise::one},
    unitD{"BZ", "MILLION BTUS/DEKATHERM", precise::one},
    unitD{"C0", "CALLS", precise::one},
    unitD{"C1", "COMPOSITE PRODUCT POUNDS (TOTAL WEIGHT)", precise::one},
    unitD{"C2", "CARSET", precise::one},
    unitD{"C4", "CARLOAD", precise::one},
    unitD{"C5", "COST", precise::currency},
    unitD{"C6", "CELL", precise::one},
    unitD{"C8", "CUBIC DECIMETER", precise::one},
    unitD{"C9", "COIL GROUP", precise::one},
    unitD{"CA", "CASE", precise::one},
    unitD{"CB", "CARBOY", precise::one},
    unitD{"CC", "CUBIC CENTIMETER", precise::cm.pow(3)},
    unitD{"CD", "KARAT (CARAT)", precise::one},
    unitD{"CF", "CUBIC FOOT", precise::ft.pow(3)},
    unitD{"CG", "CARD", precise::one},
    unitD{"CH", "CONTAINER", precise::one},
    unitD{"CI", "CUBIC INCH", precise::in.pow(3)},
    unitD{"CJ", "CONE", precise::one},
    unitD{"CK", "CONNECTOR", precise::one},
    unitD{"CL", "CYLINDER", precise::one},
    unitD{"CM", "CENTIMETER", precise::cm},
    unitD{"CN", "CAN", precise::one},
    unitD{"CP", "CRATE", precise::one},
    unitD{"CQ", "CARTRIDGE", precise::one},
    unitD{"CR", "CUBIC METER", precise::m.pow(3)},
    unitD{"CS", "CASSETTE", precise::one},
    unitD{"CT", "CARTON", precise::one},
    unitD{"CU", "CUP", precise::one},
    unitD{"CV", "COVER", precise::one},
    unitD{"CW", "HUNDRED POUNDS", precise_unit(100.0, precise::lb)},
    unitD{"CX", "COIL", precise::one},
    unitD{"CY", "CUBIC YARD", precise::yd.pow(3)},
    unitD{"CZ", "COMBO", precise::one},
    unitD{"DA", "DAY", precise::time::day},
    unitD{"DF", "DRAM", precise::one},
    unitD{"DG", "DECIGRAM", precise_unit(0.1, precise::kilogram)},
    unitD{"DH", "MILE", precise::mile},
    unitD{"DI", "DISPENSER", precise::one},
    unitD{"DJ", "DECAGRAM", precise_unit(10.0, precise::g)},
    unitD{"DK", "KILOMETER", precise::km},
    unitD{"DL", "DECILITER", precise::one},
    unitD{"DM", "DECIMETER", precise::one},
    unitD{"DO", "DOLLARS U.S.", precise::currency},
    unitD{"DP", "DOZEN PAIR", precise::one},
    unitD{"DQ", "DATA RECORD", precise::one},
    unitD{"DR", "DRUM", precise::one},
    unitD{"DS", "DISPLAY", precise::one},
    unitD{"DT", "DRY TON", precise::one},
    unitD{"DU", "DYNE", precise::cgs::dyn},
    unitD{"DV", "DEGREES, KELVIN", precise::K},
    unitD{"DW", "CALENDAR DAYS (NUMBER OF)", precise::time::day},
    unitD{"DY", "DIRECTORY BOOKS", precise::one},
    unitD{"DZ", "DOZEN", precise_unit(12, precise::one)},
    unitD{"E1", "HECTOMETER", precise_unit(100, precise::m)},
    unitD{"EA", "EACH", precise::count},
    unitD{"EB", "ELECTRONIC MAIL BOXES", precise::one},
    unitD{"EE", "EMPLOYEES", precise::one},
    unitD{"EH", "KNOTS", precise::nautical::knot},
    unitD{"EJ", "LOCATIONS", precise::one},
    unitD{"EP", "ELEVEN PACK", precise_unit(11, precise::one)},
    unitD{"EQ", "EQUIVALENT GALLONS", precise::one},
    unitD{"EV", "ENVELOPES", precise::one},
    unitD{"F2", "INTERNATIONAL UNIT", precise::one},
    unitD{"F4", "MINIM", precise::one},
    unitD{"F6", "PRICE PER SHARE", precise::currency / precise::count},
    unitD{"FA", "FAHRENHEIT", precise::F},
    unitD{"FB", "FIELDS", precise::one},
    unitD{"FC", "1000 CUBIC FEET", precise::kilo* precise::ft.pow(3)},
    unitD{"FE", "TRACK FOOT", precise::one},
    unitD{"FF", "HUNDRED CUBIC METERS", precise_unit(100.0, precise::m.pow(3))},
    unitD{"FG", "TRANSDERMAL PATCH", precise::one},
    unitD{"FJ", "SIZING FACTOR", precise::one},
    unitD{"FK", "FIBER", precise::one},
    unitD{"FL", "FLAKE TON", precise::one},
    unitD{"FM", "MILLION CUBIC FEET", precise::mega* precise::ft.pow(3)},
    unitD{"FO", "FLUID OUNCE", precise::us::floz},
    unitD{"FT", "FOOT", precise::ft},
    unitD{"FU", "FURLONG", precise::one},
    unitD{"G2", "GALLONS PER MINUTE", precise::gal / precise::min},
    unitD{"G5", "GILL (IMPERIAL)", precise::imp::gill},
    unitD{"GA", "GALLON", precise::gal},
    unitD{"GG", "GREAT GROSS", precise::one},
    unitD{"GI", "IMPERIAL GALLON", precise::one},
    unitD{"GR", "GRAM", precise::g},
    unitD{"GS", "GROSS", precise::one},
    unitD{"GX", "GRAIN", precise::one},
    unitD{"H2", "HALF LITER", precise::one},
    unitD{"H4", "HECTOLITER", precise::one},
    unitD{"HA", "HANK", precise::one},
    unitD{"HB", "HUNDRED BOXES", precise::one},
    unitD{"HC", "HUNDRED COUNT", precise::one},
    unitD{"HD", "HALF DOZEN", precise::one},
    unitD{"HF", "HUNDRED FEET", precise::one},
    unitD{"HG", "HECTOGRAM", precise::one},
    unitD{"HH", "HUNDRED CUBIC FEET", precise_unit(100.0, precise::ft.pow(3))},
    unitD{"HI", "HUNDRED SHEETS", precise::one},
    unitD{"HK", "HUNDREDS OF KILOGRAMS", precise::one},
    unitD{"HL", "HUNDRED FEET (LINEAR)", precise::one},
    unitD{"HO", "HUNDRED TROY OUNCES", precise::one},
    unitD{"HQ", "HECTARE", precise::one},
    unitD{"HR", "HOUR", precise::hr},
    unitD{"HS", "HUNDRED SQUARE FEET", precise_unit(100.0, precise::ft.pow(2))},
    unitD{"HT", "HALF HOUR", precise::one},
    unitD{"HU", "HUNDRED", precise::one},
    unitD{"HV", "HUNDRED WEIGHT - SHORT (HUNDRED WEIGHT)", precise::one},
    unitD{"HW", "HUNDRED WEIGHT (LONG)", precise::one},
    unitD{"HX", "HOSPITAL BEDS", precise::one},
    unitD{"HY", "HUNDRED YARDS", precise::one},
    unitD{"Na", "PERSONS, CAPACITY", precise::one},
    unitD{"I2", "PELLET", precise::one},
    unitD{"IE", "PERSON", precise::one},
    unitD{"IH", "INHALER", precise::one},
    unitD{"IM", "IMPRESSION", precise::one},
    unitD{"IN", "INCH", precise::in},
    unitD{"IP", "INSURANCE POLICY", precise::one},
    unitD{"IS", "STOP", precise::one},
    unitD{"JA", "JOB", precise::one},
    unitD{"JB", "JUMBO", precise::one},
    unitD{"JO", "JOINT", precise::one},
    unitD{"JR", "JAR", precise::one},
    unitD{"JU", "JUG", precise::one},
    unitD{"K6", "KILOLITER", precise::one},
    unitD{"K7", "KILOWATT", precise::electrical::kW},
    unitD{"K8", "KILOVOLT", precise::electrical::kV},
    unitD{"KA", "CAKE", precise::one},
    unitD{"KE", "KEG", precise::one},
    unitD{"KF", "KILOPACKET", precise::one},
    unitD{"KG", "KILOGRAM", precise::kg},
    unitD{"KH", "KILOWATT-HOUR", precise::kWh},
    unitD{"KK", "100 KILOGRAMS", precise_unit(100.0, precise::kg)},
    unitD{"KT", "KIT", precise::one},
    unitD{"KU", "TASK", precise::one},
    unitD{"KV", "KELVIN", precise::K},
    unitD{"L5", "LITERS AT 15 DEGREES CELSIUS", precise::one},
    unitD{"LB", "POUND (AVOIRDUPOIS)", precise::one},
    unitD{"LE", "LITE", precise::one},
    unitD{"LF", "LINEAR FOOT", precise::one},
    unitD{"LG", "LONG TON", precise::one},
    unitD{"LI", "LINEAR INCH", precise::one},
    unitD{"LJ", "LARGE SPRAY", precise::one},
    unitD{"LK", "LINK", precise::one},
    unitD{"LM", "LINEAR METER", precise::one},
    unitD{"LN", "LENGTH", precise::one},
    unitD{"LO", "LOT", precise::one},
    unitD{"LR", "LAYER", precise::one},
    unitD{"LS", "LUMP SUM", precise::one},
    unitD{"LT", "LITER", precise::one},
    unitD{"LY", "LINEAR YARD", precise::one},
    unitD{"M0", "MAGNETIC TAPE", precise::one},
    unitD{"M3", "MAT", precise::one},
    unitD{"MC", "MICROGRAM", precise::one},
    unitD{"MD", "AIR DRY METRIC TON", precise::one},
    unitD{"ME", "MILLIGRAM", precise::mg},
    unitD{"MG", "METRIC GROSS TON", precise::one},
    unitD{"MI", "METRIC", precise::one},
    unitD{"MJ", "MINUTE", precise::min},
    unitD{"ML", "MILLILITER", precise::milli* precise::L},
    unitD{"MM", "MILLIMETER", precise::mm},
    unitD{"MN", "METRIC NET TON", precise::one},
    unitD{"MO", "MONTH", precise::one},
    unitD{"MP", "METRIC TON", precise::mass::tonne},
    unitD{"MQ", "1000 METERS", precise::one},
    unitD{"MR", "METER", precise::m},
    unitD{"MS", "SQUARE MILLIMETER", precise::mm.pow(2)},
    unitD{"MT", "METRIC LONG TON", precise::one},
    unitD{"MX", "MIXED", precise::one},
    unitD{"N2", "NUMBER OF LINES", precise::one},
    unitD{"N7", "PART", precise::one},
    unitD{"N9", "CARTRIDGE NEEDLE", precise::one},
    unitD{"NB", "BARGE", precise::one},
    unitD{"NC", "CAR", precise::one},
    unitD{"ND", "NET BARRELS", precise::one},
    unitD{"NE", "NET LITERS", precise::one},
    unitD{"NF", "MESSAGES", precise::one},
    unitD{"NG", "NET GALLONS", precise::one},
    unitD{"NI", "NET IMPERIAL GALLONS", precise::one},
    unitD{"NJ", "NUMBER OF SCREENS", precise::one},
    unitD{"NK", "NIGHTS", precise::one},
    unitD{"NL", "LOAD", precise::one},
    unitD{"NM", "NAUTICAL MILE", precise::one},
    unitD{"NN", "TRAIN", precise::one},
    unitD{"NS", "SHORT TON", precise::one},
    unitD{"NT", "TRAILER", precise::one},
    unitD{"NV", "VEHICLE", precise::one},
    unitD{"NX", "PARTS PER THOUSAND", precise::one},
    unitD{"NZ", "THOUSAND GALLONS PER DAY", precise::one},
    unitD{"OA", "PANEL", precise::one},
    unitD{"OB", "OUTLET", precise::one},
    unitD{"OC", "BILLBOARD", precise::one},
    unitD{"OG", "APOTHECARY GRAIN", precise::one},
    unitD{"OP", "TWO PACK", precise::one},
    unitD{"OU", "OPERATING UNIT", precise::one},
    unitD{"OZ", "OUNCE - AV", precise::one},
    unitD{"P0", "PAGE - ELECTRONIC", precise::one},
    unitD{"P1", "PERCENT", precise::one},
    unitD{"P2", "POUNDS PER FOOT", precise::one},
    unitD{"P3", "THREE PACK", precise::one},
    unitD{"P4", "FOUR PACK", precise::one},
    unitD{"P5", "FIVE PACK", precise::one},
    unitD{"P6", "SIX PACK", precise::one},
    unitD{"P7", "SEVEN PACK", precise::one},
    unitD{"P8", "EIGHT PACK", precise::one},
    unitD{"P9", "NINE PACK", precise::one},
    unitD{"PA", "PAIL", precise::one},
    unitD{"PB", "PAIR INCHES", precise::one},
    unitD{"PC", "PIECE", precise::one},
    unitD{"PD", "PAD", precise::one},
    unitD{"PE", "POUNDS EQUIVALENT", precise::one},
    unitD{"PF", "PALLET (LIFT)", precise::one},
    unitD{"PG", "POUNDS GROSS", precise::one},
    unitD{"PH", "PACK (PAK)", precise::one},
    unitD{"PI", "PITCH", precise::one},
    unitD{"PK", "PACKAGE", precise::one},
    unitD{"PL", "PALLET/UNIT LOAD", precise::one},
    unitD{"PN", "POUNDS NET", precise::one},
    unitD{"PP", "PLATE", precise::one},
    unitD{"PR", "PAIR", precise::one},
    unitD{"PT", "PINT", precise::one},
    unitD{"PU", "MASS POUNDS", precise::one},
    unitD{"PV", "HALF PINT", precise::one},
    unitD{"PX", "PINT IMPERIAL", precise::one},
    unitD{"PY", "PECK DRY US", precise::one},
    unitD{"PZ", "PECK DRY IMPERIAL", precise::one},
    unitD{"Q3", "MEAL", precise::one},
    unitD{"Q4", "FIFTY", precise::one},
    unitD{"Q5", "TWENTY-FIVE", precise::one},
    unitD{"Q6", "THIRTY-SIX", precise::one},
    unitD{"Q7", "TWENTY-FOUR", precise::one},
    unitD{"Q9", "FIVE", precise::one},
    unitD{"QC", "CHANNEL", precise::one},
    unitD{"QD", "QUARTER DOZEN", precise::one},
    unitD{"QE", "PHOTOGRAPHS", precise::one},
    unitD{"QF", "QUARTER", precise::one},
    unitD{"QK", "QUARTER KILOGRAM", precise::one},
    unitD{"QR", "QUIRE", precise::one},
    unitD{"QS", "QUART DRY U.S.", precise::one},
    unitD{"QT", "QUART", precise::one},
    unitD{"QU", "QUART IMPERIAL", precise::one},
    unitD{"R4", "CALORIE", precise::one},
    unitD{"R5", "THOUSANDS OF DOLLARS", precise::one},
    unitD{"R9", "THOUSAND CUBIC METERS", precise::one},
    unitD{"RA", "RACK", precise::one},
    unitD{"RB", "RADIAN", precise::rad},
    unitD{"RC", "ROD (AREA) - 16.25 SQUARE YARDS", precise::one},
    unitD{"RD", "ROD (LENGTH) - 5.5 YARDS", precise::one},
    unitD{"RE", "REEL", precise::one},
    unitD{"RG", "RING", precise::one},
    unitD{"RH", "RUNNING OR OPERATING HOURS", precise::one},
    unitD{"RK", "ROLL-METRIC MEASURE", precise::one},
    unitD{"RL", "ROLL", precise::one},
    unitD{"RM", "REAM", precise::one},
    unitD{"RN", "REAM-METRIC MEASURE", precise::one},
    unitD{"RO", "ROUND", precise::one},
    unitD{"RP", "POUNDS PER REAM", precise::one},
    unitD{"RS", "RESETS", precise::one},
    unitD{"RT", "REVENUE TON MILES", precise::one},
    unitD{"RU", "RUN", precise::one},
    unitD{"RX", "THOUSAND ROUNDS", precise::one},
    unitD{
        "S5",
        "SIXTY-FOURTHS OF AN INCH",
        precise_unit(1.0 / 64.0, precise::in)},
    unitD{"S6", "SESSIONS", precise::one},
    unitD{"S7", "STORAGE UNITS", precise::one},
    unitD{"S8", "STANDARD ADVERTISING UNIT", precise::one},
    unitD{"S9", "SLIP SHEET", precise::one},
    unitD{"SA", "SANDWICH", precise::one},
    unitD{"SB", "SQUARE MILE", precise::mile.pow(2)},
    unitD{"SC", "SQUARE CENTIMETER", precise::cm.pow(2)},
    unitD{"SE", "SECTION (640 ACRES OR ONE SQUARE MILE)", precise::one},
    unitD{"SF", "SQUARE FOOT", precise::ft.pow(2)},
    unitD{"SG", "SEGMENT", precise::one},
    unitD{"SH", "SHEET", precise::one},
    unitD{"SI", "SQUARE INCH", precise::in.pow(2)},
    unitD{"SJ", "SACK", precise::one},
    unitD{"SK", "SPLIT TANKTRUCK", precise::one},
    unitD{"SL", "SLEEVE", precise::one},
    unitD{"SM", "SQUARE METER", precise::m.pow(2)},
    unitD{"SN", "SQUARE ROD", precise::one},
    unitD{"SO", "SPOOL", precise::one},
    unitD{"SP", "SHELF PACKAGE", precise::one},
    unitD{"SQ", "SQUARE", precise::one},
    unitD{"SR", "STRIP", precise::one},
    unitD{"SS", "SHEET-METRIC MEASURE", precise::one},
    unitD{"ST", "SET", precise::one},
    unitD{"SV", "SKID", precise::one},
    unitD{"SW", "SKEIN", precise::one},
    unitD{"SX", "SHIPMENT", precise::one},
    unitD{"SY", "SQUARE YARD", precise::yd.pow(2)},
    unitD{"SZ", "SYRINGE", precise::one},
    unitD{"T1", "THOUSAND POUNDS GROSS", precise::one},
    unitD{"T3", "THOUSAND PIECES", precise::one},
    unitD{"T4", "THOUSAND BAGS", precise::one},
    unitD{"T5", "THOUSAND CASINGS", precise::one},
    unitD{"T6", "THOUSAND GALLONS", precise::one},
    unitD{"T7", "THOUSAND IMPRESSIONS", precise::one},
    unitD{"T8", "THOUSAND LINEAR INCHES", precise::one},
    unitD{"T9", "THOUSAND KILOWATT HOURS/MEGAWATT-HOUR", precise::one},
    unitD{"TA", "TENTH CUBIC FOOT", precise::one},
    unitD{"TB", "TUBE", precise::one},
    unitD{"TC", "TRUCK LOAD", precise::one},
    unitD{"TG", "GROSS TON", precise::one},
    unitD{"TH", "THOUSAND", precise::kilo* precise::count},
    unitD{"TI", "THOUSAND SQUARE INCHES", precise::one},
    unitD{"TJ", "THOUSAND SQUARE CENTIMETERS", precise::one},
    unitD{"TK", "TANK", precise::one},
    unitD{"TL", "THOUSAND FEET", precise::one},
    unitD{"TM", "THOUSAND FEET (BOARD)", precise::one},
    unitD{"TN", "NET TON (2,000 POUNDS)", precise::one},
    unitD{"TO", "TROY OUNCE", precise::one},
    unitD{"TP", "TEN-PACK", precise::one},
    unitD{"TQ", "THOUSAND FEET", precise::one},
    unitD{"TR", "TEN SQUARE FEET", precise::one},
    unitD{"TS", "THOUSAND SQUARE FEET", precise::one},
    unitD{"TT", "THOUSAND LINEAR METERS", precise::one},
    unitD{"TU", "THOUSAND LINEAR YARDS", precise::one},
    unitD{"TV", "THOUSAND KILOGRAMS", precise::one},
    unitD{"TW", "THOUSAND SHEETS", precise::one},
    unitD{"TX", "TROY POUND", precise::one},
    unitD{"TY", "TRAY", precise::one},
    unitD{"TZ", "THOUSAND CUBIC FEET", precise::one},
    unitD{"U1", "TREATMENT", precise::one},
    unitD{"U2", "TABLET", precise::one},
    unitD{"U3", "TEN", precise::one},
    unitD{"U5", "TWO HUNDRED FIFTY", precise::one},
    unitD{"U6", "UNITED STATES GALLONS @ 60 DEGREES FAHRENHEIT", precise::one},
    unitD{"UH", "TEN THOUSAND YARDS", precise::one},
    unitD{"UL", "UNITLESS", precise::one},
    unitD{"UM", "MILLION UNITS", precise::one},
    unitD{"UN", "UNIT", precise::one},
    unitD{"UP", "TROCHE", precise::one},
    unitD{"UQ", "WAFER", precise::one},
    unitD{"UX", "APOTHECARY DRAM", precise::one},
    unitD{"V1", "FLAT", precise::one},
    unitD{"V2", "POUCH", precise::one},
    unitD{"VC", "FIVE HUNDRED", precise::one},
    unitD{"VI", "VIAL", precise::one},
    unitD{"VS", "VISIT", precise::one},
    unitD{"W2", "WET KILO", precise::one},
    unitD{"WB", "WET POUND", precise::one},
    unitD{"WD", "WORK DAY", precise::one},
    unitD{"WE", "WET TON", precise::one},
    unitD{"WG", "WINE GALLON", precise::one},
    unitD{"WH", "WHEEL", precise::one},
    unitD{"WK", "WEEK", precise::time::week},
    unitD{"WP", "PENNYWEIGHT", precise::one},
    unitD{"WR", "WRAP", precise::one},
    unitD{"X1", "CHAIN", precise::one},
    unitD{"X2", "BUNCH", precise::one},
    unitD{"X3", "CLOVE", precise::one},
    unitD{"X4", "DROP", precise::one},
    unitD{"X5", "HEAD", precise::one},
    unitD{"X6", "HEART", precise::one},
    unitD{"X7", "LEAF", precise::one},
    unitD{"X8", "LOAF", precise::one},
    unitD{"X9", "PORTION", precise::one},
    unitD{"Y1", "SLICE", precise::one},
    unitD{"Y2", "TABLESPOON", precise::us::tbsp},
    unitD{"Y3", "TEASPOON", precise::us::tsp},
    unitD{"Y4", "TUB", precise::one},
    unitD{"YD", "YARD", precise::yd},
    unitD{"YL", "100 LINEAL YARDS", precise::one},
    unitD{"YR", "YEAR", precise::one},
    unitD{"Z1", "LIFT VAN", precise::one},
    unitD{"Z2", "CHEST", precise::one},
    unitD{"Z3", "CASK", precise::one},
    unitD{"Z4", "HOGSHEAD", precise::one},
    unitD{"Z5", "LUG", precise::one},
    unitD{"ZP", "PAGE", precise::one},
    unitD{"ZW", "BARREL, DRY", precise::one},
    unitD{"ZX", "BARREL, LIQUID", precise::one},
    unitD{"ZY", "APOTHECARY SCRUPLE", precise::one},
}};

static UNITS_CPP14_CONSTEXPR_OBJECT std::array<unitD, 486> dod_units = {{
    unitD{"05", "LIFT", precise::one},
    unitD{"07", "STRAND", precise::one},
    unitD{"09", "TIRE", precise::one},
    unitD{"17", "100 POUND DRUM", precise::one},
    unitD{"18", "55 GALLON DRUM", precise::one},
    unitD{"19", "TANK TRUCK", precise::one},
    unitD{"1N", "COUNT", precise::one},
    unitD{"1O", "SEASON", precise::one},
    unitD{"1P", "TANK CAR", precise::one},
    unitD{"1R", "TRANSACTION", precise::one},
    unitD{"1X", "QUARTER MILE", precise::one},
    unitD{"20", "20 FOOT CONTAINER", precise::one},
    unitD{"21", "40 FOOT CONTAINER", precise::one},
    unitD{"2D", "SHIP SET", precise::one},
    unitD{"2U", "MEGAGRAM", precise::one},
    unitD{"2W", "BIN", precise::one},
    unitD{"43", "SUPER BULK BAG (SAME SIZE AS PALLET)", precise::one},
    unitD{"44", "500 KILOGRAM BULK BAG", precise::one},
    unitD{"45", "300 KILOGRAM BULK BAG", precise::one},
    unitD{"46", "25 KILOGRAM BULK BAG", precise::one},
    unitD{"47", "50 POUND BAG", precise::one},
    unitD{"48", "BULK CAR LOAD", precise::one},
    unitD{"4A", "BOBBIN", precise::one},
    unitD{"4B", "CAP", precise::one},
    unitD{"4E", "20-PACK", precise::one},
    unitD{"4F", "100-PACK", precise::one},
    unitD{"4G", "MICROLITER", precise::one},
    unitD{"51", "ACTUAL TONNES", precise::one},
    unitD{"57", "MESH", precise::one},
    unitD{"58", "NET KILOGRAMS", precise::one},
    unitD{"5B", "BATCH", precise::one},
    unitD{"5D", "PROOF GALLONS", precise::one},
    unitD{"5G", "PUMP", precise::one},
    unitD{"5H", "STAGE", precise::one},
    unitD{"5I", "STANDARD CUBE FOOT", precise::one},
    unitD{"7A", "LANDINGS", precise::one},
    unitD{"7C", "FLIGHT HOURS", precise::one},
    unitD{"8C", "CORD", precise::one},
    unitD{"8D", "DUTY", precise::one},
    unitD{"8P", "PROJECT", precise::one},
    unitD{"8R", "PROGRAM", precise::one},
    unitD{"8S", "SESSION", precise::one},
    unitD{"91", "STOKE", precise::one},
    unitD{"94", "CURL UNIT", precise::one},
    unitD{"95", "20,000 GALLON TANKCAR", precise::one},
    unitD{"96", "10,000 GALLON TANKCAR", precise::one},
    unitD{"97", "10 KILOGRAM DRUM", precise::one},
    unitD{"98", "15 KILOGRAM DRUM", precise::one},
    unitD{"9Y", "FAMILY UNIT", precise::one},
    unitD{"AB", "BULK PACK", precise::one},
    unitD{"AC", "ACRE", precise::one},
    unitD{"AD", "APOTHECARY DRAM", precise::one},
    unitD{"AJ", "COP (CYLINDRICAL OR CONICAL MASS)", precise::one},
    unitD{"AL", "ACCESS LINES", precise::one},
    unitD{"AM", "AMPOULE", precise::one},
    unitD{"AO", "APOTHECARY OUNCE", precise::one},
    unitD{"AP", "APOTHECARY POUND", precise::one},
    unitD{"AR", "SUPPOSITORY", precise::one},
    unitD{"AS", "APOTHECARY SCRUPLE", precise::one},
    unitD{"AT", "ASSORTMENT", precise::one},
    unitD{"AV", "CAPSULE", precise::one},
    unitD{"AW", "POWDER-FILLED VIAL", precise::one},
    unitD{"AX", "TWENTY", precise::one},
    unitD{"AY", "ASSEMBLY", precise::one},
    unitD{"B0", "BRITISH THERMAL UNITS PER CUBIC FOOT", precise::one},
    unitD{"B1", "BARREL, LIQUID", precise::one},
    unitD{"B2", "BARREL, DRY", precise::one},
    unitD{"B3", "BATTING POUND", precise::one},
    unitD{"B4", "BARREL IMPERIAL", precise::one},
    unitD{"B5", "BILLET", precise::one},
    unitD{"B6", "BUN", precise::one},
    unitD{"B7", "CYCLE", precise::one},
    unitD{"B8", "BOARD", precise::one},
    unitD{"B9", "BATT", precise::one},
    unitD{"BA", "BALL", precise::one},
    unitD{"BB", "BASS BOX", precise::one},
    unitD{"BC", "BUCKET", precise::one},
    unitD{"BD", "BUNDLE", precise::one},
    unitD{"BE", "BALE", precise::one},
    unitD{"BF", "BOARD FOOT", precise::one},
    unitD{"BG", "BAG", precise::one},
    unitD{"BH", "BRUSH", precise::one},
    unitD{"BI", "BELT", precise::one},
    unitD{"BJ", "BAND", precise::one},
    unitD{"BK", "BOOK", precise::one},
    unitD{"BL", "BARREL", precise::one},
    unitD{"BM", "BEAM", precise::one},
    unitD{"BN", "BULK", precise::one},
    unitD{"BO", "BOLT", precise::one},
    unitD{"BP", "100 BOARD FEET", precise::one},
    unitD{"BQ", "BRIQUET", precise::one},
    unitD{"BR", "BAR", precise::one},
    unitD{"BS", "BASKET", precise::one},
    unitD{"BT", "BOTTLE", precise::one},
    unitD{"BU", "BUSHEL (32 DRY QUARTS)", precise::one},
    unitD{"BV", "BUSHEL DRY IMPERIAL", precise::one},
    unitD{"BW", "BASE WEIGHT", precise::one},
    unitD{"BX", "BOX", precise::one},
    unitD{"BY", "BUNK", precise::one},
    unitD{"BZ", "BLOCK", precise::one},
    unitD{"C0", "CALLS", precise::one},
    unitD{"C1", "COMPOSITE PRODUCT POUNDS (TOTAL WEIGHT)", precise::one},
    unitD{"C2", "SQUARE CENTIMETER", precise::one},
    unitD{"C3", "COMBO", precise::one},
    unitD{"C4", "CARLOAD", precise::one},
    unitD{"C5", "COST", precise::one},
    unitD{"C6", "CELL", precise::one},
    unitD{"C7", "CARSET", precise::one},
    unitD{"C8", "CUBIC DECIMETER", precise::one},
    unitD{"C9", "COIL GROUP", precise::one},
    unitD{"CA", "CARTRIDGE", precise::one},
    unitD{"CB", "CARBOY", precise::one},
    unitD{"CC", "CUBIC CENTIMETER", precise::one},
    unitD{"CD", "CUBIC YARD", precise::one},
    unitD{"CE", "CONE", precise::one},
    unitD{"CF", "CUBIC FOOT", precise::one},
    unitD{"CG", "CENTIGRAM", precise::one},
    unitD{"CH", "CHAIN", precise::one},
    unitD{"CI", "CUBIC INCH", precise::one},
    unitD{"CK", "CAKE", precise::one},
    unitD{"CL", "COIL", precise::one},
    unitD{"CM", "CENTIMETER", precise::one},
    unitD{"CN", "CAN", precise::one},
    unitD{"CO", "CONTAINER", precise::one},
    unitD{"CP", "CRATE", precise::one},
    unitD{"CQ", "CARD", precise::one},
    unitD{"CR", "CONNECTOR", precise::one},
    unitD{"CS", "CASE", precise::one},
    unitD{"CT", "CARTON", precise::one},
    unitD{"CU", "CUP", precise::one},
    unitD{"CV", "COVER", precise::one},
    unitD{"CW", "HUNDRED WEIGHT (LONG)", precise::one},
    unitD{"CX", "CASSETTE", precise::one},
    unitD{"CY", "CYLINDER", precise::one},
    unitD{"CZ", "CUBIC METER", precise::one},
    unitD{"DA", "DAY", precise::one},
    unitD{"DB", "DECIBEL", precise::one},
    unitD{"DC", "DECAGRAM", precise::one},
    unitD{"DE", "DECIMETER", precise::one},
    unitD{"DF", "FAHRENHEIT", precise::one},
    unitD{"DG", "DECIGRAM", precise::one},
    unitD{"DI", "DISPENSER", precise::one},
    unitD{"DK", "DEGREES, KELVIN", precise::one},
    unitD{"DK", "KELVIN", precise::one},
    unitD{"DL", "DECILITER", precise::one},
    unitD{"DM", "DRAM", precise::one},
    unitD{"DO", "DOLLARS U.S.", precise::one},
    unitD{"DP", "DOZEN PAIR", precise::one},
    unitD{"DQ", "DATA RECORD", precise::one},
    unitD{"DR", "DRUM", precise::one},
    unitD{"DS", "DISPLAY", precise::one},
    unitD{"DT", "DRY TON", precise::one},
    unitD{"DU", "DYNE", precise::one},
    unitD{"DW", "PENNYWEIGHT", precise::one},
    unitD{"DX", "CALENDAR DAYS (NUMBER OF)", precise::one},
    unitD{"DY", "DIRECTORY BOOKS", precise::one},
    unitD{"DZ", "DOZEN", precise::one},
    unitD{"E1", "HECTOMETER", precise::one},
    unitD{"EA", "EACH", precise::one},
    unitD{"EB", "ELECTRONIC MAIL BOXES", precise::one},
    unitD{"EE", "EMPLOYEES", precise::one},
    unitD{"EH", "KNOTS", precise::one},
    unitD{"EJ", "LOCATIONS", precise::one},
    unitD{"EP", "ELEVEN PACK", precise::one},
    unitD{"EQ", "EQUIVALENT GALLONS", precise::one},
    unitD{"EV", "ENVELOPES", precise::one},
    unitD{"F4", "MINIM", precise::one},
    unitD{"F6", "PRICE PER SHARE", precise::one},
    unitD{"FA", "FATHOM", precise::one},
    unitD{"FB", "FIELDS", precise::one},
    unitD{"FC", "1000 CUBIC FEET", precise::one},
    unitD{"FD", "FOLD", precise::one},
    unitD{"FE", "TRACK FOOT", precise::one},
    unitD{"FF", "HUNDRED CUBIC METERS", precise::one},
    unitD{"FG", "TRANSDERMAL PATCH", precise::one},
    unitD{"FJ", "SIZING FACTOR", precise::one},
    unitD{"FK", "FIBER", precise::one},
    unitD{"FL", "FLAKE TON", precise::one},
    unitD{"FM", "MILLION CUBIC FEET", precise::one},
    unitD{"FO", "FLUID OUNCE", precise::one},
    unitD{"FR", "FRAME", precise::one},
    unitD{"FT", "FOOT", precise::one},
    unitD{"FU", "FURLONG", precise::one},
    unitD{"FV", "FIVE", precise::one},
    unitD{"FY", "FIFTY", precise::one},
    unitD{"GB", "GALLONS PER MINUTE", precise::one},
    unitD{"GG", "GREAT GROSS", precise::one},
    unitD{"GI", "GILL (IMPERIAL)", precise::one},
    unitD{"GL", "GALLON", precise::one},
    unitD{"GM", "GRAM", precise::one},
    unitD{"GN", "GRAIN", precise::one},
    unitD{"GP", "GROUP", precise::one},
    unitD{"GR", "GROSS", precise::one},
    unitD{"GT", "THOUSAND GALLONS PER DAY", precise::one},
    unitD{"GX", "APOTHECARY GRAIN", precise::one},
    unitD{"H2", "HALF LITER", precise::one},
    unitD{"H4", "HECTOLITER", precise::one},
    unitD{"HA", "HUNDRED CUBIC FEET", precise::one},
    unitD{"HB", "HOSPITAL BEDS", precise::one},
    unitD{"HC", "HUNDRED COUNT", precise::one},
    unitD{"HD", "HUNDRED", precise::one},
    unitD{"HF", "HUNDRED FEET", precise::one},
    unitD{"HG", "HECTOGRAM", precise::one},
    unitD{"HH", "HOGSHEAD", precise::one},
    unitD{"HI", "HUNDRED SHEETS", precise::one},
    unitD{"HK", "HANK", precise::one},
    unitD{"HL", "HUNDRED FEET (LINEAR)", precise::one},
    unitD{"HO", "HUNDRED TROY OUNCES", precise::one},
    unitD{"HP", "HUNDRED POUNDS", precise::one},
    unitD{"HQ", "HECTARE", precise::one},
    unitD{"HR", "HOUR", precise::one},
    unitD{"HS", "HUNDRED SQUARE FEET", precise::one},
    unitD{"HT", "HALF HOUR", precise::one},
    unitD{"HW", "HUNDRED WEIGHT - SHORT (HUNDRED WEIGHT)", precise::one},
    unitD{"HX", "HUNDRED BOXES", precise::one},
    unitD{"HY", "HUNDRED YARDS", precise::one},
    unitD{"HZ", "HALF DOZEN", precise::one},
    unitD{"I1", "PERSONS, CAPACITY", precise::one},
    unitD{"IG", "IMPERIAL GALLON", precise::one},
    unitD{"IH", "INHALER", precise::one},
    unitD{"IM", "IMPRESSION", precise::one},
    unitD{"IN", "INCH", precise::one},
    unitD{"IP", "INSURANCE POLICY", precise::one},
    unitD{"IS", "STOP", precise::one},
    unitD{"IU", "INTERNATIONAL UNIT", precise::one},
    unitD{"JB", "JOB", precise::one},
    unitD{"JG", "JUG", precise::one},
    unitD{"JO", "JOINT", precise::one},
    unitD{"JR", "JAR", precise::one},
    unitD{"JU", "JUMBO", precise::one},
    unitD{"K2", "SQUARE KILOMETER", precise::one},
    unitD{"K6", "KILOLITER", precise::one},
    unitD{"K7", "KILOWATT", precise::one},
    unitD{"KC", "KILOCURIE", precise::one},
    unitD{"KE", "KEG", precise::one},
    unitD{"KF", "KILOPACKET", precise::one},
    unitD{"KG", "KILOGRAM", precise::one},
    unitD{"KH", "HUNDREDS OF KILOGRAMS", precise::one},
    unitD{"KK", "100 KILOGRAMS", precise::one},
    unitD{"KM", "KILOMETER", precise::km},
    unitD{"KR", "KARAT (CARAT)", precise::one},
    unitD{"KT", "KIT", precise::one},
    unitD{"KU", "TASK", precise::one},
    unitD{"KV", "KILOVOLT", precise::electrical::kV},
    unitD{"KZ", "KILOWATT-HOUR", precise::kWh},
    unitD{"L5", "LITERS AT 15 DEGREES CELSIUS", precise::one},
    unitD{"LB", "POUND (AVOIRDUPOIS)", precise::one},
    unitD{"LE", "LITE", precise::one},
    unitD{"LF", "LINEAR FOOT", precise::one},
    unitD{"LG", "LENGTH", precise::one},
    unitD{"LI", "LITER", precise::one},
    unitD{"LJ", "LARGE SPRAY", precise::one},
    unitD{"LK", "LINK", precise::one},
    unitD{"LM", "LINEAR METER", precise::one},
    unitD{"LN", "LINEAR INCH", precise::one},
    unitD{"LO", "LONG TON", precise::one},
    unitD{"LR", "LAYER", precise::one},
    unitD{"LS", "LUMP SUM", precise::one},
    unitD{"LT", "LOT", precise::one},
    unitD{"LY", "LINEAR YARD", precise::one},
    unitD{"M0", "MAGNETIC TAPE", precise::one},
    unitD{"M2", "SQUARE MILE", precise::one},
    unitD{"M3", "MAT", precise::one},
    unitD{"M5", "MICROGRAM", precise::one},
    unitD{"M6", "METRIC TON", precise::one},
    unitD{"MA", "METRIC NET TON", precise::one},
    unitD{"MB", "BRITISH THERMAL UNITS/HOUR", precise::btu / precise::hr},
    unitD{"MC", "THOUSAND CUBIC FEET", precise::one},
    unitD{"MD", "AIR DRY METRIC TON", precise::one},
    unitD{"ME", "MEAL", precise::one},
    unitD{"MF", "THOUSAND FEET", precise::one},
    unitD{"MG", "MILLIGRAM", precise::mg},
    unitD{"MH", "METRIC", precise::one},
    unitD{"MI", "MILE", precise::one},
    unitD{"MJ", "METRIC GROSS TON", precise::one},
    unitD{"MK", "METRIC LONG TON", precise::one},
    unitD{"ML", "MILLILITER", precise::one},
    unitD{"MM", "MILLIMETER", precise::mm},
    unitD{"MO", "MONTH", precise::one},
    unitD{"MQ", "1000 METERS", precise::one},
    unitD{"MR", "METER", precise::one},
    unitD{"MS", "SQUARE MILLIMETER", precise::one},
    unitD{"MT", "MINUTE", precise::one},
    unitD{"MX", "THOUSAND", precise::one},
    unitD{"MZ", "MIXED", precise::one},
    unitD{"N2", "NUMBER OF LINES", precise::one},
    unitD{"N7", "PART", precise::one},
    unitD{"N9", "CARTRIDGE NEEDLE", precise::one},
    unitD{"NB", "BARGE", precise::one},
    unitD{"NC", "CAR", precise::one},
    unitD{"ND", "NET BARRELS", precise::one},
    unitD{"NE", "NET LITERS", precise::one},
    unitD{"NF", "MESSAGES", precise::one},
    unitD{"NG", "NET GALLONS", precise::one},
    unitD{"NI", "NET IMPERIAL GALLONS", precise::one},
    unitD{"NJ", "NUMBER OF SCREENS", precise::one},
    unitD{"NK", "NIGHTS", precise::one},
    unitD{"NL", "LOAD", precise::one},
    unitD{"NM", "NAUTICAL MILE", precise::one},
    unitD{"NN", "TRAIN", precise::one},
    unitD{"NS", "SHORT TON", precise::one},
    unitD{"NT", "TRAILER", precise::one},
    unitD{"NV", "VEHICLE", precise::one},
    unitD{"NX", "PARTS PER THOUSAND", precise::one},
    unitD{"OA", "PANEL", precise::one},
    unitD{"OC", "BILLBOARD", precise::one},
    unitD{"OL", "OUTLET", precise::one},
    unitD{"OP", "TWO PACK", precise::one},
    unitD{"OT", "OUTFIT", precise::one},
    unitD{"OU", "OPERATING UNIT", precise::one},
    unitD{"OZ", "OUNCE - AV", precise::one},
    unitD{"P0", "PAGE - ELECTRONIC", precise::one},
    unitD{"P1", "PERCENT", precise::one},
    unitD{"P2", "POUNDS PER FOOT", precise::one},
    unitD{"P3", "THREE PACK", precise::one},
    unitD{"P4", "FOUR PACK", precise::one},
    unitD{"P5", "FIVE PACK", precise::one},
    unitD{"P6", "SIX PACK", precise::one},
    unitD{"P7", "SEVEN PACK", precise::one},
    unitD{"P8", "EIGHT PACK", precise::one},
    unitD{"P9", "NINE PACK", precise::one},
    unitD{"PA", "PAGE", precise::one},
    unitD{"PB", "PAIR INCHES", precise::one},
    unitD{"PC", "PIECE", precise::one},
    unitD{"PD", "PAD", precise::one},
    unitD{"PE", "POUNDS EQUIVALENT", precise::one},
    unitD{"PF", "PALLET (LIFT)", precise::one},
    unitD{"PG", "PACKAGE", precise::one},
    unitD{"PH", "PACK (PAK)", precise::one},
    unitD{"PI", "PILLOW", precise::one},
    unitD{"PJ", "PALLET/UNIT LOAD", precise::one},
    unitD{"PK", "PECK DRY US", precise::one},
    unitD{"PL", "PAIL", precise::one},
    unitD{"PM", "PLATE", precise::one},
    unitD{"PN", "PERSON", precise::one},
    unitD{"PO", "POUNDS GROSS", precise::one},
    unitD{"PP", "PINT IMPERIAL", precise::one},
    unitD{"PQ", "PECK DRY IMPERIAL", precise::one},
    unitD{"PR", "PAIR", precise::one},
    unitD{"PS", "POUNDS NET", precise::one},
    unitD{"PT", "PINT", precise::one},
    unitD{"PU", "MASS POUNDS", precise::one},
    unitD{"PV", "HALF PINT", precise::one},
    unitD{"PX", "PELLET", precise::one},
    unitD{"PY", "PITCH", precise::one},
    unitD{"PZ", "PACKET", precise::one},
    unitD{"QC", "CHANNEL", precise::one},
    unitD{"QE", "PHOTOGRAPHS", precise::one},
    unitD{"QF", "QUARTER", precise::one},
    unitD{"QI", "QUART IMPERIAL", precise::one},
    unitD{"QK", "QUARTER KILOGRAM", precise::one},
    unitD{"QR", "QUIRE", precise::one},
    unitD{"QS", "QUART DRY U.S.", precise::one},
    unitD{"QT", "QUART", precise::one},
    unitD{"QU", "QUARTER DOZEN", precise::one},
    unitD{"R4", "CALORIE", precise::one},
    unitD{"R5", "THOUSANDS OF DOLLARS", precise::one},
    unitD{"R9", "THOUSAND CUBIC METERS", precise::one},
    unitD{"RA", "RATION", precise::one},
    unitD{"RB", "RADIAN", precise::one},
    unitD{"RC", "ROD (AREA) - 16.25 SQUARE YARDS", precise::one},
    unitD{"RD", "ROD (LENGTH) - 5.5 YARDS", precise::one},
    unitD{"RE", "ROUND", precise::one},
    unitD{"RG", "RING", precise::one},
    unitD{"RH", "RUNNING OR OPERATING HOURS", precise::one},
    unitD{"RK", "ROLL-METRIC MEASURE", precise::one},
    unitD{"RL", "REEL", precise::one},
    unitD{"RM", "REAM", precise::one},
    unitD{"RN", "REAM-METRIC MEASURE", precise::one},
    unitD{"RO", "ROLL", precise::one},
    unitD{"RP", "POUNDS PER REAM", precise::one},
    unitD{"RR", "RACK", precise::one},
    unitD{"RS", "RESETS", precise::one},
    unitD{"RT", "REVENUE TON MILES", precise::one},
    unitD{"RU", "RUN", precise::one},
    unitD{"RX", "THOUSAND ROUNDS", precise::one},
    unitD{"S5", "SIXTY-FOURTHS OF AN INCH", precise::one},
    unitD{"S6", "SESSIONS", precise::one},
    unitD{"S7", "STORAGE UNITS", precise::one},
    unitD{"S8", "SHELF PACKAGE", precise::one},
    unitD{"S9", "SLIP SHEET", precise::one},
    unitD{"SA", "SANDWICH", precise::one},
    unitD{"SB", "SHIPMENT", precise::one},
    unitD{"SC", "SECOND", precise::one},
    unitD{"SD", "SKID", precise::one},
    unitD{"SE", "SET", precise::one},
    unitD{"SF", "SQUARE FOOT", precise::one},
    unitD{"SG", "SEGMENT", precise::one},
    unitD{"SH", "SHEET", precise::one},
    unitD{"SI", "SQUARE INCH", precise::one},
    unitD{"SJ", "SACK", precise::one},
    unitD{"SK", "SKEIN", precise::one},
    unitD{"SL", "SPOOL", precise::one},
    unitD{"SM", "SQUARE METER", precise::one},
    unitD{"SN", "SECTION (640 ACRES OR ONE SQUARE MILE)", precise::one},
    unitD{"SO", "SHOT", precise::one},
    unitD{"SP", "STRIP", precise::one},
    unitD{"SQ", "SQUARE", precise::one},
    unitD{"SR", "SPLIT TANKTRUCK", precise::one},
    unitD{"SS", "SHEET-METRIC MEASURE", precise::one},
    unitD{"ST", "SEAT", precise::one},
    unitD{"SU", "SQUARE ROD", precise::one},
    unitD{"SV", "SLEEVE", precise::one},
    unitD{"SW", "STANDARD ADVERTISING UNIT", precise::one},
    unitD{"SX", "STICK", precise::one},
    unitD{"SY", "SQUARE YARD", precise::yd.pow(2)},
    unitD{"SZ", "SYRINGE", precise::one},
    unitD{"T1", "THOUSAND POUNDS GROSS", precise::one},
    unitD{"T2", "TEASPOON", precise::us::tsp},
    unitD{"T3", "THOUSAND PIECES", precise::one},
    unitD{"T4", "THOUSAND BAGS", precise::one},
    unitD{"T5", "THOUSAND CASINGS", precise::one},
    unitD{"T6", "THOUSAND GALLONS", precise::one},
    unitD{"T7", "THOUSAND IMPRESSIONS", precise::one},
    unitD{"T8", "THOUSAND LINEAR INCHES", precise::one},
    unitD{"T9", "THOUSAND KILOWATT HOURS/MEGAWATT-HOUR", precise::one},
    unitD{"TA", "TENTH CUBIC FOOT", precise::one},
    unitD{"TB", "TABLESPOON", precise::one},
    unitD{"TC", "TRUCK LOAD", precise::one},
    unitD{"TD", "TWENTY-FOUR", precise_unit(24.0, precise::one)},
    unitD{"TE", "TEN", precise::ten},
    unitD{"TF", "TWENTY-FIVE", precise_unit(25.0, precise::one)},
    unitD{"TG", "GROSS TON", precise::one},
    unitD{"TH", "THOUSAND KILOGRAMS", precise::kilo* precise::kilogram},
    unitD{"TI", "THOUSAND SQUARE INCHES", precise::one},
    unitD{"TJ", "THOUSAND SQUARE CENTIMETERS", precise::one},
    unitD{"TK", "TANK", precise::one},
    unitD{"TL", "THOUSAND LINEAR METERS", precise::one},
    unitD{"TM", "THOUSAND FEET (BOARD)", precise::one},
    unitD{"TN", "NET TON (2,000 POUNDS)", precise::one},
    unitD{"TO", "TROY OUNCE", precise::troy::oz},
    unitD{"TP", "TROY POUND", precise::troy::pound},
    unitD{"TQ", "THOUSAND FEET", precise::one},
    unitD{"TR", "TEN SQUARE FEET", precise::one},
    unitD{"TS", "THIRTY-SIX", precise::one},
    unitD{"TT", "TABLET", precise::one},
    unitD{"TU", "TUBE", precise::one},
    unitD{"TV", "TEN-PACK", precise::one},
    unitD{"TW", "THOUSAND SHEETS", precise::one},
    unitD{"TX", "THOUSAND LINEAR YARDS", precise::one},
    unitD{"TY", "TRAY", precise::one},
    unitD{"TZ", "THOUSAND SQUARE FEET", precise::one},
    unitD{"U1", "TREATMENT", precise::one},
    unitD{"U5", "TWO HUNDRED FIFTY", precise::one},
    unitD{"U6", "UNITED STATES GALLONS @ 60 DEGREES FAHRENHEIT", precise::one},
    unitD{"UH", "TEN THOUSAND YARDS", precise_unit(10000.0, precise::yd)},
    unitD{"UL", "UNITLESS", precise::one},
    unitD{"UM", "MILLION UNITS", precise::one},
    unitD{"UN", "UNIT", precise::one},
    unitD{"UP", "TROCHE", precise::one},
    unitD{"UQ", "WAFER", precise::one},
    unitD{"US", "USP UNIT", precise::one},
    unitD{"V1", "FLAT", precise::one},
    unitD{"V2", "POUCH", precise::one},
    unitD{"VC", "FIVE HUNDRED", precise::one},
    unitD{"VI", "VIAL", precise::one},
    unitD{"VS", "VISIT", precise::one},
    unitD{"VT", "VOLT", precise::one},
    unitD{"W2", "WET KILO", precise::one},
    unitD{"WB", "WET POUND", precise::one},
    unitD{"WD", "WORK DAY", precise::one},
    unitD{"WE", "WET TON", precise::one},
    unitD{"WG", "WINE GALLON", precise::one},
    unitD{"WH", "WHEEL", precise::one},
    unitD{"WK", "WEEK", precise::one},
    unitD{"WR", "WRAP", precise::one},
    unitD{"WT", "WATT", precise::one},
    unitD{"X2", "BUNCH", precise::one},
    unitD{"X3", "CLOVE", precise::one},
    unitD{"X4", "DROP", precise::one},
    unitD{"X5", "HEAD", precise::one},
    unitD{"X6", "HEART", precise::one},
    unitD{"X7", "LEAF", precise::one},
    unitD{"X8", "LOAF", precise::one},
    unitD{"X9", "PORTION", precise::one},
    unitD{"Y1", "SLICE", precise::one},
    unitD{"Y4", "TUB", precise::one},
    unitD{"YD", "YARD", precise::yd},
    unitD{"YL", "100 LINEAL YARDS", precise::one},
    unitD{"YR", "YEAR", precise::yr},
    unitD{"YT", "BYTE", precise::data::byte},
    unitD{"Z1", "LIFT VAN", precise::one},
    unitD{"Z2", "CHEST", precise::one},
    unitD{"Z3", "CASK", precise::one},
    unitD{"Z5", "LUG", precise::one},
    unitD{
        "ZF",
        "MILLION BTUS/DEKATHERM",
        precise::mega* precise::btu /
            precise_unit(10.0, precise::energy::therm_ec)},
}};

precise_unit x12_unit(const std::string& x12_string)
{
    // NOLINTNEXTLINE (readability-qualified-auto)
    auto ind = std::lower_bound(
        x12_units.begin(),
        x12_units.end(),
        x12_string,
        [](const unitD& u_set, const std::string& val) {
            return (strcmp(std::get<0>(u_set), val.c_str()) < 0);
        });
    if (strcmp(std::get<0>(*ind), x12_string.c_str()) == 0) {
        return std::get<2>(*ind);
    }
    return precise::error;
}

precise_unit dod_unit(const std::string& dod_string)
{
    // NOLINTNEXTLINE (readability-qualified-auto)
    auto ind = std::lower_bound(
        dod_units.begin(),
        dod_units.end(),
        dod_string,
        [](const unitD& u_set, const std::string& val) {
            return (strcmp(std::get<0>(u_set), val.c_str()) < 0);
        });
    if (strcmp(std::get<0>(*ind), dod_string.c_str()) == 0) {
        return std::get<2>(*ind);
    }
    return precise::error;
}

}  // namespace UNITS_NAMESPACE
