/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#pragma once

#include <arrow-glib/version.h>

/**
 * SECTION: version
 * @section_id: version-macros
 * @title: Version related macros
 * @include: arrow-flight-glib/arrow-flight-glib.h
 *
 * Apache Arrow Flight GLib provides macros that can be used by C pre-processor.
 * They are useful to check version related things at compile time.
 */

/**
 * GAFLIGHT_VERSION_MAJOR:
 *
 * The major version.
 *
 * Since: 17.0.0
 */
#define GAFLIGHT_VERSION_MAJOR (17)

/**
 * GAFLIGHT_VERSION_MINOR:
 *
 * The minor version.
 *
 * Since: 17.0.0
 */
#define GAFLIGHT_VERSION_MINOR (0)

/**
 * GAFLIGHT_VERSION_MICRO:
 *
 * The micro version.
 *
 * Since: 17.0.0
 */
#define GAFLIGHT_VERSION_MICRO (0)

/**
 * GAFLIGHT_VERSION_TAG:
 *
 * The version tag. Normally, it's an empty string. It's "SNAPSHOT"
 * for snapshot version.
 *
 * Since: 17.0.0
 */
#define GAFLIGHT_VERSION_TAG   ""

/**
 * GAFLIGHT_VERSION_CHECK:
 * @major: A major version to check for.
 * @minor: A minor version to check for.
 * @micro: A micro version to check for.
 *
 * You can use this macro in C pre-processor.
 *
 * Returns: %TRUE if the compile time Apache Arrow GLib version is the
 *   same as or newer than the passed version, %FALSE otherwise.
 *
 * Since: 17.0.0
 */
#define GAFLIGHT_VERSION_CHECK(major, minor, micro)       \
  (GAFLIGHT_VERSION_MAJOR > (major) ||                    \
   (GAFLIGHT_VERSION_MAJOR == (major) &&                  \
    GAFLIGHT_VERSION_MINOR > (minor)) ||                  \
   (GAFLIGHT_VERSION_MAJOR == (major) &&                  \
    GAFLIGHT_VERSION_MINOR == (minor) &&                  \
    GAFLIGHT_VERSION_MICRO >= (micro)))

/**
 * GAFLIGHT_DISABLE_DEPRECATION_WARNINGS:
 *
 * If this macro is defined, no deprecated warnings are produced.
 *
 * You must define this macro before including the
 * arrow-glib/arrow-glib.h header.
 *
 * Since: 17.0.0
 */

#ifdef GAFLIGHT_DISABLE_DEPRECATION_WARNINGS
#  define GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_FOR(function)
#  define GAFLIGHT_UNAVAILABLE(major, minor)
#else
#  define GAFLIGHT_DEPRECATED G_DEPRECATED
#  define GAFLIGHT_DEPRECATED_FOR(function) G_DEPRECATED_FOR(function)
#  define GAFLIGHT_UNAVAILABLE(major, minor) G_UNAVAILABLE(major, minor)
#endif

/**
 * GAFLIGHT_VERSION_17_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 17.0.0
 */
#define GAFLIGHT_VERSION_17_0 G_ENCODE_VERSION(17, 0)

/**
 * GAFLIGHT_VERSION_16_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 16.0.0
 */
#define GAFLIGHT_VERSION_16_0 G_ENCODE_VERSION(16, 0)

/**
 * GAFLIGHT_VERSION_15_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 15.0.0
 */
#define GAFLIGHT_VERSION_15_0 G_ENCODE_VERSION(15, 0)

/**
 * GAFLIGHT_VERSION_14_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 14.0.0
 */
#define GAFLIGHT_VERSION_14_0 G_ENCODE_VERSION(14, 0)

/**
 * GAFLIGHT_VERSION_13_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 13.0.0
 */
#define GAFLIGHT_VERSION_13_0 G_ENCODE_VERSION(13, 0)

/**
 * GAFLIGHT_VERSION_12_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 12.0.0
 */
#define GAFLIGHT_VERSION_12_0 G_ENCODE_VERSION(12, 0)

/**
 * GAFLIGHT_VERSION_11_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 11.0.0
 */
#define GAFLIGHT_VERSION_11_0 G_ENCODE_VERSION(11, 0)

/**
 * GAFLIGHT_VERSION_10_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 10.0.0
 */
#define GAFLIGHT_VERSION_10_0 G_ENCODE_VERSION(10, 0)

/**
 * GAFLIGHT_VERSION_9_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 9.0.0
 */
#define GAFLIGHT_VERSION_9_0 G_ENCODE_VERSION(9, 0)

/**
 * GAFLIGHT_VERSION_8_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 8.0.0
 */
#define GAFLIGHT_VERSION_8_0 G_ENCODE_VERSION(8, 0)

/**
 * GAFLIGHT_VERSION_7_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 7.0.0
 */
#define GAFLIGHT_VERSION_7_0 G_ENCODE_VERSION(7, 0)

/**
 * GAFLIGHT_VERSION_6_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 6.0.0
 */
#define GAFLIGHT_VERSION_6_0 G_ENCODE_VERSION(6, 0)

/**
 * GAFLIGHT_VERSION_5_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 5.0.0
 */
#define GAFLIGHT_VERSION_5_0 G_ENCODE_VERSION(5, 0)

/**
 * GAFLIGHT_VERSION_4_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 4.0.0
 */
#define GAFLIGHT_VERSION_4_0 G_ENCODE_VERSION(4, 0)

/**
 * GAFLIGHT_VERSION_3_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 3.0.0
 */
#define GAFLIGHT_VERSION_3_0 G_ENCODE_VERSION(3, 0)

/**
 * GAFLIGHT_VERSION_2_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 2.0.0
 */
#define GAFLIGHT_VERSION_2_0 G_ENCODE_VERSION(2, 0)

/**
 * GAFLIGHT_VERSION_1_0:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 1.0.0
 */
#define GAFLIGHT_VERSION_1_0 G_ENCODE_VERSION(1, 0)

/**
 * GAFLIGHT_VERSION_0_17:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.17.0
 */
#define GAFLIGHT_VERSION_0_17 G_ENCODE_VERSION(0, 17)

/**
 * GAFLIGHT_VERSION_0_16:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.16.0
 */
#define GAFLIGHT_VERSION_0_16 G_ENCODE_VERSION(0, 16)

/**
 * GAFLIGHT_VERSION_0_15:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.15.0
 */
#define GAFLIGHT_VERSION_0_15 G_ENCODE_VERSION(0, 15)

/**
 * GAFLIGHT_VERSION_0_14:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.14.0
 */
#define GAFLIGHT_VERSION_0_14 G_ENCODE_VERSION(0, 14)

/**
 * GAFLIGHT_VERSION_0_13:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.13.0
 */
#define GAFLIGHT_VERSION_0_13 G_ENCODE_VERSION(0, 13)

/**
 * GAFLIGHT_VERSION_0_12:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.12.0
 */
#define GAFLIGHT_VERSION_0_12 G_ENCODE_VERSION(0, 12)

/**
 * GAFLIGHT_VERSION_0_11:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.11.0
 */
#define GAFLIGHT_VERSION_0_11 G_ENCODE_VERSION(0, 11)

/**
 * GAFLIGHT_VERSION_0_10:
 *
 * You can use this macro value for compile time API version check.
 *
 * Since: 0.10.0
 */
#define GAFLIGHT_VERSION_0_10 G_ENCODE_VERSION(0, 10)

/**
 * GAFLIGHT_VERSION_MIN_REQUIRED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GAFLIGHT_VERSION_0_10.
 *
 * If you use any functions that is defined by newer version than
 * %GAFLIGHT_VERSION_MIN_REQUIRED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * arrow-flight-glib/arrow-flight-glib.h header.
 *
 * Since: 17.0.0
 */
#ifndef GAFLIGHT_VERSION_MIN_REQUIRED
#  define GAFLIGHT_VERSION_MIN_REQUIRED GARROW_VERSION_MIN_REQUIRED
#endif

/**
 * GAFLIGHT_VERSION_MAX_ALLOWED:
 *
 * You can use this macro for compile time API version check.
 *
 * This macro value must be one of the predefined version macros such
 * as %GAFLIGHT_VERSION_0_10.
 *
 * If you use any functions that is defined by newer version than
 * %GAFLIGHT_VERSION_MAX_ALLOWED, deprecated warnings are produced at
 * compile time.
 *
 * You must define this macro before including the
 * arrow-flight-glib/arrow-flight-glib.h header.
 *
 * Since: 17.0.0
 */
#ifndef GAFLIGHT_VERSION_MAX_ALLOWED
#  define GAFLIGHT_VERSION_MAX_ALLOWED GARROW_VERSION_MAX_ALLOWED
#endif

#if (defined(_WIN32) || defined(__CYGWIN__)) && defined(_MSC_VER) &&   !defined(GAFLIGHT_STATIC_COMPILATION)
#  define GAFLIGHT_EXPORT __declspec(dllexport)
#  define GAFLIGHT_IMPORT __declspec(dllimport)
#else
#  define GAFLIGHT_EXPORT
#  define GAFLIGHT_IMPORT
#endif

#ifdef GAFLIGHT_COMPILATION
#  define GAFLIGHT_API GAFLIGHT_EXPORT
#else
#  define GAFLIGHT_API GAFLIGHT_IMPORT
#endif

#define GAFLIGHT_EXTERN GAFLIGHT_API extern

#define GAFLIGHT_AVAILABLE_IN_ALL GAFLIGHT_EXTERN

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_17_0
#  define GAFLIGHT_DEPRECATED_IN_17_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_17_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_17_0
#  define GAFLIGHT_DEPRECATED_IN_17_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_17_0
#  define GAFLIGHT_AVAILABLE_IN_17_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(17, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_17_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_16_0
#  define GAFLIGHT_DEPRECATED_IN_16_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_16_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_16_0
#  define GAFLIGHT_DEPRECATED_IN_16_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_16_0
#  define GAFLIGHT_AVAILABLE_IN_16_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(16, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_16_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_15_0
#  define GAFLIGHT_DEPRECATED_IN_15_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_15_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_15_0
#  define GAFLIGHT_DEPRECATED_IN_15_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_15_0
#  define GAFLIGHT_AVAILABLE_IN_15_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(15, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_15_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_14_0
#  define GAFLIGHT_DEPRECATED_IN_14_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_14_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_14_0
#  define GAFLIGHT_DEPRECATED_IN_14_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_14_0
#  define GAFLIGHT_AVAILABLE_IN_14_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(14, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_14_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_13_0
#  define GAFLIGHT_DEPRECATED_IN_13_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_13_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_13_0
#  define GAFLIGHT_DEPRECATED_IN_13_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_13_0
#  define GAFLIGHT_AVAILABLE_IN_13_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(13, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_13_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_12_0
#  define GAFLIGHT_DEPRECATED_IN_12_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_12_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_12_0
#  define GAFLIGHT_DEPRECATED_IN_12_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_12_0
#  define GAFLIGHT_AVAILABLE_IN_12_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(12, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_12_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_11_0
#  define GAFLIGHT_DEPRECATED_IN_11_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_11_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_11_0
#  define GAFLIGHT_DEPRECATED_IN_11_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_11_0
#  define GAFLIGHT_AVAILABLE_IN_11_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(11, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_11_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_10_0
#  define GAFLIGHT_DEPRECATED_IN_10_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_10_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_10_0
#  define GAFLIGHT_DEPRECATED_IN_10_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_10_0
#  define GAFLIGHT_AVAILABLE_IN_10_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(10, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_10_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_9_0
#  define GAFLIGHT_DEPRECATED_IN_9_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_9_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_9_0
#  define GAFLIGHT_DEPRECATED_IN_9_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_9_0
#  define GAFLIGHT_AVAILABLE_IN_9_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(9, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_9_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_8_0
#  define GAFLIGHT_DEPRECATED_IN_8_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_8_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_8_0
#  define GAFLIGHT_DEPRECATED_IN_8_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_8_0
#  define GAFLIGHT_AVAILABLE_IN_8_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(8, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_8_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_7_0
#  define GAFLIGHT_DEPRECATED_IN_7_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_7_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_7_0
#  define GAFLIGHT_DEPRECATED_IN_7_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_7_0
#  define GAFLIGHT_AVAILABLE_IN_7_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(7, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_7_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_6_0
#  define GAFLIGHT_DEPRECATED_IN_6_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_6_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_6_0
#  define GAFLIGHT_DEPRECATED_IN_6_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_6_0
#  define GAFLIGHT_AVAILABLE_IN_6_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(6, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_6_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_5_0
#  define GAFLIGHT_DEPRECATED_IN_5_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_5_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_5_0
#  define GAFLIGHT_DEPRECATED_IN_5_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_5_0
#  define GAFLIGHT_AVAILABLE_IN_5_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(5, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_5_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_4_0
#  define GAFLIGHT_DEPRECATED_IN_4_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_4_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_4_0
#  define GAFLIGHT_DEPRECATED_IN_4_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_4_0
#  define GAFLIGHT_AVAILABLE_IN_4_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(4, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_4_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_3_0
#  define GAFLIGHT_DEPRECATED_IN_3_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_3_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_3_0
#  define GAFLIGHT_DEPRECATED_IN_3_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_3_0
#  define GAFLIGHT_AVAILABLE_IN_3_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(3, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_3_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_2_0
#  define GAFLIGHT_DEPRECATED_IN_2_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_2_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_2_0
#  define GAFLIGHT_DEPRECATED_IN_2_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_2_0
#  define GAFLIGHT_AVAILABLE_IN_2_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(2, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_2_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_1_0
#  define GAFLIGHT_DEPRECATED_IN_1_0               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_1_0_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_1_0
#  define GAFLIGHT_DEPRECATED_IN_1_0_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_1_0
#  define GAFLIGHT_AVAILABLE_IN_1_0 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(1, 0)
#else
#  define GAFLIGHT_AVAILABLE_IN_1_0 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_0_17
#  define GAFLIGHT_DEPRECATED_IN_0_17               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_0_17_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_0_17
#  define GAFLIGHT_DEPRECATED_IN_0_17_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_0_17
#  define GAFLIGHT_AVAILABLE_IN_0_17 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(0, 17)
#else
#  define GAFLIGHT_AVAILABLE_IN_0_17 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_0_16
#  define GAFLIGHT_DEPRECATED_IN_0_16               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_0_16_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_0_16
#  define GAFLIGHT_DEPRECATED_IN_0_16_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_0_16
#  define GAFLIGHT_AVAILABLE_IN_0_16 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(0, 16)
#else
#  define GAFLIGHT_AVAILABLE_IN_0_16 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_0_15
#  define GAFLIGHT_DEPRECATED_IN_0_15               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_0_15_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_0_15
#  define GAFLIGHT_DEPRECATED_IN_0_15_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_0_15
#  define GAFLIGHT_AVAILABLE_IN_0_15 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(0, 15)
#else
#  define GAFLIGHT_AVAILABLE_IN_0_15 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_0_14
#  define GAFLIGHT_DEPRECATED_IN_0_14               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_0_14_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_0_14
#  define GAFLIGHT_DEPRECATED_IN_0_14_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_0_14
#  define GAFLIGHT_AVAILABLE_IN_0_14 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(0, 14)
#else
#  define GAFLIGHT_AVAILABLE_IN_0_14 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_0_13
#  define GAFLIGHT_DEPRECATED_IN_0_13               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_0_13_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_0_13
#  define GAFLIGHT_DEPRECATED_IN_0_13_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_0_13
#  define GAFLIGHT_AVAILABLE_IN_0_13 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(0, 13)
#else
#  define GAFLIGHT_AVAILABLE_IN_0_13 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_0_12
#  define GAFLIGHT_DEPRECATED_IN_0_12               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_0_12_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_0_12
#  define GAFLIGHT_DEPRECATED_IN_0_12_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_0_12
#  define GAFLIGHT_AVAILABLE_IN_0_12 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(0, 12)
#else
#  define GAFLIGHT_AVAILABLE_IN_0_12 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_0_11
#  define GAFLIGHT_DEPRECATED_IN_0_11               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_0_11_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_0_11
#  define GAFLIGHT_DEPRECATED_IN_0_11_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_0_11
#  define GAFLIGHT_AVAILABLE_IN_0_11 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(0, 11)
#else
#  define GAFLIGHT_AVAILABLE_IN_0_11 GAFLIGHT_EXTERN
#endif

#if GAFLIGHT_VERSION_MIN_REQUIRED >= GAFLIGHT_VERSION_0_10
#  define GAFLIGHT_DEPRECATED_IN_0_10               GAFLIGHT_DEPRECATED
#  define GAFLIGHT_DEPRECATED_IN_0_10_FOR(function) GAFLIGHT_DEPRECATED_FOR(function)
#else
#  define GAFLIGHT_DEPRECATED_IN_0_10
#  define GAFLIGHT_DEPRECATED_IN_0_10_FOR(function)
#endif

#if GAFLIGHT_VERSION_MAX_ALLOWED < GAFLIGHT_VERSION_0_10
#  define GAFLIGHT_AVAILABLE_IN_0_10 GAFLIGHT_EXTERN GAFLIGHT_UNAVAILABLE(0, 10)
#else
#  define GAFLIGHT_AVAILABLE_IN_0_10 GAFLIGHT_EXTERN
#endif
