









with Ada.Containers;                  use Ada.Containers;
with Ada.Containers.Hashed_Maps;
with Ada.Containers.Vectors;
with Ada.Directories;
with Ada.Exceptions;
with Ada.Finalization;
with Ada.Strings.Unbounded;           use Ada.Strings.Unbounded;
with Ada.Strings.Wide_Wide_Unbounded; use Ada.Strings.Wide_Wide_Unbounded;

pragma Warnings (Off, "internal");
with Ada.Strings.Wide_Wide_Unbounded.Aux;
use Ada.Strings.Wide_Wide_Unbounded.Aux;
pragma Warnings (On, "internal");

with Ada.Text_IO;                     use Ada.Text_IO;
with Ada.Unchecked_Conversion;
with Ada.Unchecked_Deallocation;
with System;

with GNAT.Task_Lock;
with GNAT.Traceback.Symbolic;

with GNATCOLL.Traces;

with Liblktlang_Support.Adalog.Debug;
with Liblktlang_Support.Generic_API.Analysis;
with Liblktlang_Support.Generic_API.Introspection;
with Liblktlang_Support.Hashes; use Liblktlang_Support.Hashes;
with Liblktlang_Support.Images; use Liblktlang_Support.Images;
with Liblktlang_Support.Names;  use Liblktlang_Support.Names;
with Liblktlang_Support.Relative_Get;

with Liblktlang.Private_Converters;
use Liblktlang.Private_Converters;

pragma Warnings (Off, "referenced");

          with Liblktlang.Default_Provider;
          with Liblktlang.Implementation.Extensions;
            use Liblktlang.Implementation.Extensions;

pragma Warnings (On, "referenced");



package body Liblktlang.Implementation is

   use Precomputed_Symbols;

   pragma Warnings (Off, "has no effect");
   use Solver;
   pragma Warnings (On, "has no effect");

   package Context_Vectors is new Ada.Containers.Vectors
     (Index_Type   => Positive,
      Element_Type => Internal_Context);

   type Contexts_Destructor is limited
      new Ada.Finalization.Limited_Controlled with null record;
   overriding procedure Finalize (CD : in out Contexts_Destructor);
   --  Helper to destroy all contexts when terminating the process

   package Context_Pool is

      procedure Acquire (Context : out Internal_Context)
         with Post => Context /= null;
      --  If a context is free for reuse, increment its serial number and
      --  return it. Otherwise, allocate a new one. In any case, this does not
      --  initialize it, except for the Serial_Number field.

      procedure Release (Context : in out Internal_Context)
         with Pre  => Context /= null,
              Post => Context = null;
      --  Tag Context as free for reuse and set it to null

      procedure Free;
      --  Free all contexts in this pool. Intended to be called only when the
      --  process is terminating, to avoid reported memory leaks.

   private

      Available : Context_Vectors.Vector;
      --  List of allocated contexts that can be re-used right now

      CD : Contexts_Destructor with Unreferenced;
      --  Singleton whose only purpose is to free all contexts in Available
      --  when finalized.

   end Context_Pool;

   procedure Register_Destroyable_Helper
     (Unit    : Internal_Unit;
      Object  : System.Address;
      Destroy : Destroy_Procedure);
   --  Common underlying implementation for Register_Destroyable_Gen

   pragma Warnings (Off, "referenced");
   function Construct_Entity_Array
     (V : AST_Envs.Entity_Vectors.Vector) return Internal_Entity_Array_Access;
   pragma Warnings (On, "referenced");

   procedure Reset_Envs_Caches (Unit : Internal_Unit);
   --  Reset the env caches of all lexical environments created for ``Unit``

   procedure Destroy (Env : in out Lexical_Env_Access);

   function Snaps_At_Start (Self : Bare_Lkt_Node) return Boolean;
   function Snaps_At_End (Self : Bare_Lkt_Node) return Boolean;

   --  Those maps are used to give unique ids to lexical envs while pretty
   --  printing them.

   package Address_To_Id_Maps is new Ada.Containers.Hashed_Maps
     (Lexical_Env, Integer, Hash, "=");

   type Dump_Lexical_Env_State is record
      Env_Ids : Address_To_Id_Maps.Map;
      --  Mapping: Lexical_Env -> Integer, used to remember which unique Ids we
      --  assigned to the lexical environments we found.

      Next_Id : Positive := 1;
      --  Id to assign to the next unknown lexical environment

      Root_Env : Lexical_Env;
      --  Lexical environment we consider a root (this is the Root_Scope from
      --  the current analysis context), or null if unknown.
   end record;
   --  Holder for the state of lexical environment dumpers

   function Get_Env_Id
     (E : Lexical_Env; State : in out Dump_Lexical_Env_State) return String;
   --  If E is known, return its unique Id from State. Otherwise, assign it a
   --  new unique Id and return it.

   procedure Print
     (Node        : Liblktlang_Support.Generic_API.Analysis.Lk_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "");
   --  Helper for the public overload, but working on the generic API node type

   ------------------------
   -- Precomputed_Symbol --
   ------------------------

   pragma Warnings (Off, "referenced");
   function Precomputed_Symbol
     (Index : Precomputed_Symbol_Index) return Text_Type is
   pragma Warnings (On, "referenced");
   begin
         declare
            Raw_Text : constant Text_Type := (case Index is
            
               when Precomputed_Sym_Abstract => "abstract",
               when Precomputed_Sym_Analysisunit => "AnalysisUnit",
               when Precomputed_Sym_Array => "Array",
               when Precomputed_Sym_As => "as",
               when Precomputed_Sym_Astlist => "ASTList",
               when Precomputed_Sym_Bigint => "BigInt",
               when Precomputed_Sym_Bool => "Bool",
               when Precomputed_Sym_Builtin => "builtin",
               when Precomputed_Sym_Call => "__call__",
               when Precomputed_Sym_Char => "Char",
               when Precomputed_Sym_Dedent => "dedent",
               when Precomputed_Sym_Dont_Skip => "dont_skip",
               when Precomputed_Sym_Env_Spec => "env_spec",
               when Precomputed_Sym_Errornode => "ErrorNode",
               when Precomputed_Sym_External => "external",
               when Precomputed_Sym_Family => "family",
               when Precomputed_Sym_Indent => "indent",
               when Precomputed_Sym_Int => "Int",
               when Precomputed_Sym_Invalid => "invalid",
               when Precomputed_Sym_Iterator => "Iterator",
               when Precomputed_Sym_Keep => "keep",
               when Precomputed_Sym_List => "list",
               when Precomputed_Sym_Main_Rule => "main_rule",
               when Precomputed_Sym_Newline => "newline",
               when Precomputed_Sym_No_Case => "no_case",
               when Precomputed_Sym_Node => "Node",
               when Precomputed_Sym_Node_31 => "node",
               when Precomputed_Sym_Open => "open",
               when Precomputed_Sym_Out => "out",
               when Precomputed_Sym_Pick => "pick",
               when Precomputed_Sym_Previous_Token => "previous_token",
               when Precomputed_Sym_Property => "property",
               when Precomputed_Sym_Propertyerror => "PropertyError",
               when Precomputed_Sym_Regexp => "Regexp",
               when Precomputed_Sym_Self => "self",
               when Precomputed_Sym_Send => "send",
               when Precomputed_Sym_Skip => "skip",
               when Precomputed_Sym_Stop_Cut => "stop_cut",
               when Precomputed_Sym_String => "String",
               when Precomputed_Sym_Symbol => "Symbol",
               when Precomputed_Sym_Tokennode => "TokenNode",
               when Precomputed_Sym_When => "when",
               when Precomputed_Sym_With_Lexer => "with_lexer"
            );

            Symbol : constant Symbolization_Result :=
                  Create_Symbol (Raw_Text)
            ;
         begin
            if Symbol.Success then
               return Symbol.Symbol;
            else
               raise Program_Error with
                 "Cannot canonicalize symbol literal: " & Image (Raw_Text);
            end if;
         end;
   end Precomputed_Symbol;

   ----------------------------
   -- Construct_Entity_Array --
   ----------------------------

   function Construct_Entity_Array
     (V : AST_Envs.Entity_Vectors.Vector) return Internal_Entity_Array_Access
   is
      Ret : Internal_Entity_Array_Access :=
        Create_Internal_Entity_Array (V.Length);
   begin
      for J in V.First_Index .. V.Last_Index loop
         Ret.Items (J) := V.Get (J);
      end loop;

      declare
         Tmp : AST_Envs.Entity_Vectors.Vector := V;
      begin
         Tmp.Destroy;
      end;

      return Ret;
   end Construct_Entity_Array;

   -----------
   -- Image --
   -----------

   function Image (Self : Symbol_Type) return String_Type is
   begin
      return Create_String (Image (Self));
   end Image;

   ------------------
   -- Context_Pool --
   ------------------

   package body Context_Pool is

      -------------
      -- Acquire --
      -------------

      procedure Acquire (Context : out Internal_Context) is
      begin
         GNAT.Task_Lock.Lock;

         if Available.Is_Empty then
            Context := new Analysis_Context_Type;
            Context.Serial_Number := 1;
         else
            Context := Available.Last_Element;
            Available.Delete_Last;
         end if;

         GNAT.Task_Lock.Unlock;

         Context.Initialized := False;
         Context.Ref_Count := 1;

      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Acquire;

      -------------
      -- Release --
      -------------

      procedure Release (Context : in out Internal_Context) is
      begin
         GNAT.Task_Lock.Lock;

         Available.Append (Context);
         Context.Serial_Number := Context.Serial_Number + 1;
         Context := null;

         GNAT.Task_Lock.Unlock;

      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Release;

      ----------
      -- Free --
      ----------

      procedure Free is
      begin
         GNAT.Task_Lock.Lock;

         for C of Available loop
            Free (C);
         end loop;

         GNAT.Task_Lock.Unlock;

      exception
         when others =>
            GNAT.Task_Lock.Unlock;
            raise;
      end Free;

   end Context_Pool;

   --------------
   -- Finalize --
   --------------

   overriding procedure Finalize (CD : in out Contexts_Destructor) is
      pragma Unreferenced (CD);
   begin
      Context_Pool.Free;
   end Finalize;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (File_Reader : in out Internal_File_Reader_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_File_Reader'Class, Internal_File_Reader_Access);
   begin
      if File_Reader /= null and then File_Reader.all.Dec_Ref then
         Destroy (File_Reader);
      end if;
   end Dec_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Self : in out Internal_Event_Handler_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_Event_Handler'Class, Internal_Event_Handler_Access);
   begin
      if Self /= null and then Self.all.Dec_Ref then
         Destroy (Self);
      end if;
   end Dec_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Provider : in out Internal_Unit_Provider_Access) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Internal_Unit_Provider'Class, Internal_Unit_Provider_Access);
   begin
      if Provider /= null and then Provider.all.Dec_Ref then
         Destroy (Provider);
      end if;
   end Dec_Ref;

   ----------------
   -- Get_Env_Id --
   ----------------

   function Get_Env_Id
     (E : Lexical_Env; State : in out Dump_Lexical_Env_State) return String
   is
      C        : Address_To_Id_Maps.Cursor;
      Inserted : Boolean;
   begin
      if E = Null_Lexical_Env then
         return "$null";

      elsif E = State.Root_Env then
         --  Insert root env with a special Id so that we only print it once
         State.Env_Ids.Insert (E, -1, C, Inserted);
         return "$root";
      end if;

      State.Env_Ids.Insert (E, State.Next_Id, C, Inserted);
      if Inserted then
         State.Next_Id := State.Next_Id + 1;
      end if;

      return '@' & Stripped_Image (Address_To_Id_Maps.Element (C));
   end Get_Env_Id;

   pragma Warnings (Off, "referenced");
   function To_Lookup_Kind_Type (K : Lookup_Kind) return Lookup_Kind_Type
   is
     (Lookup_Kind_Type'Val (Lookup_Kind'Pos (K)));
   pragma Warnings (On, "referenced");

   ----------------------
   -- Allocate_Context --
   ----------------------

   function Allocate_Context return Internal_Context is
   begin
      return Context : Internal_Context do
         Context_Pool.Acquire (Context);
      end return;
   end Allocate_Context;

   ------------------------
   -- Initialize_Context --
   ------------------------

   procedure Initialize_Context
     (Context        : Internal_Context;
      Charset        : String;
      File_Reader    : Internal_File_Reader_Access;
      Unit_Provider  : Internal_Unit_Provider_Access;
      Event_Handler  : Internal_Event_Handler_Access;
      With_Trivia    : Boolean;
      Tab_Stop       : Positive)
   is
      Actual_Charset : constant String :=
        (if Charset = "" then Default_Charset else Charset);
      Symbols        : constant Precomputed_Symbol_Table
        := Create_Symbol_Table;
   begin
      Context.Initialized := True;
      Context.Symbols := Symbol_Table (Symbols);
      Context.Charset := To_Unbounded_String (Actual_Charset);
      Context.Tab_Stop := Tab_Stop;
      Context.With_Trivia := With_Trivia;
      Context.Root_Scope := Create_Static_Lexical_Env
        (Parent    => Null_Lexical_Env,
         Node      => null,
         Sym_Table => Context.Symbols);

      --  Create a new ownership share for Event_Handler so that it lives at
      --  least as long as this analysis context.
      Context.Event_Handler := Event_Handler;
      if Context.Event_Handler /= null then
         Context.Event_Handler.Inc_Ref;
      end if;

      --  Create a new ownership share for File_Reader so that it lives at
      --  least as long as this analysis context.
      Context.File_Reader := File_Reader;
      if Context.File_Reader /= null then
         Context.File_Reader.Inc_Ref;
      end if;

      --  Create a new ownership share for Unit_Provider so that it lives at
      --  least as long as this analysis context.
      Context.Unit_Provider := Unit_Provider;
      if Context.Unit_Provider /= null then
         Context.Unit_Provider.Inc_Ref;
      end if;

         if Context.Unit_Provider = null then
            Context.Unit_Provider := Liblktlang.Default_Provider.Create;
         end if;

      Initialize (Context.Parser);

      Context.Discard_Errors_In_Populate_Lexical_Env := True;
      Context.Logic_Resolution_Timeout :=
        Liblktlang_Support.Adalog.Default_Timeout_Ticks_Number;
      Context.In_Populate_Lexical_Env := False;
      Context.Cache_Version := 0;
      Context.Reparse_Cache_Version := 0;

      Context.Rewriting_Handle := No_Rewriting_Handle_Pointer;
      Context.Templates_Unit := No_Analysis_Unit;

      Context.Available_Rebindings := Env_Rebindings_Vectors.Empty_Vector;

      

   end Initialize_Context;

   -----------------
   -- Create_Unit --
   -----------------

   function Create_Unit
     (Context             : Internal_Context;
      Normalized_Filename : Virtual_File;
      Charset             : String;
      Rule                : Grammar_Rule) return Internal_Unit
   is
      use Units_Maps;

      Unit : Internal_Unit;
   begin
      Unit := Create_Special_Unit
        (Context, Normalized_Filename, Charset, Rule);
      Context.Units.Insert (Normalized_Filename, Unit);
      return Unit;
   end Create_Unit;

   --------------
   -- Get_Unit --
   --------------

   function Get_Unit
     (Context           : Internal_Context;
      Filename, Charset : String;
      Reparse           : Boolean;
      Input             : Internal_Lexer_Input;
      Rule              : Grammar_Rule;
      Is_Internal       : Boolean := False) return Internal_Unit
   is
      use Units_Maps;

      Normalized_Filename : constant GNATCOLL.VFS.Virtual_File :=
         Normalized_Unit_Filename (Context, Filename);

      Cur     : constant Cursor :=
         Context.Units.Find (Normalized_Filename);
      Created : constant Boolean := Cur = No_Element;
      Unit    : Internal_Unit;

      Actual_Charset : Unbounded_String;
      Refined_Input  : Internal_Lexer_Input := Input;

   begin
      --  Determine which encoding to use. Use the Charset parameter (if
      --  provided), otherwise use the context-wide default.

      Actual_Charset := (if Charset'Length /= 0
                         then To_Unbounded_String (Charset)
                         else Context.Charset);

      if Refined_Input.Kind = File then
         Refined_Input.Filename := Normalized_Filename;
      end if;

      if Refined_Input.Kind in File | Bytes_Buffer then
         Refined_Input.Charset := Actual_Charset;

         --  Unless the caller requested a specific charset for this unit,
         --  allow the lexer to automatically discover the source file encoding
         --  before defaulting to the context-specific one. We do this trying
         --  to match a byte order mark.

         Refined_Input.Read_BOM := Charset'Length = 0;
      end if;

      --  Create the Internal_Unit if needed

      Unit :=
        (if Created
         then Create_Unit (Context, Normalized_Filename,
                           To_String (Actual_Charset), Rule)
         else Element (Cur));

      --  If an internal unit is requested, set the corresponding flag.
      --  Otherwise, make sure that the unit we return isn't internal.

      if Is_Internal then
         Unit.Is_Internal := True;
      end if;

      --  (Re)parse it if needed

      if Created or else Reparse then

         --  It is illegal to reparse an internal unit for public API users.
         --  Since public APIs do not allow to pass True to Is_Internal, we can
         --  check here that only the implementation can ask to reparse an
         --  internal unit.

         if Unit.Is_Internal and then not Is_Internal then
            raise Precondition_Failure with "cannot reparse an internal unit";
         end if;

         declare
            Reparsed : Reparsed_Unit;
         begin
            Do_Parsing (Unit, Refined_Input, Reparsed);
            Update_After_Reparse (Unit, Reparsed);
         end;

         --  Now that we have removed reparsed the unit, update its current
         --  charset.

         Unit.Charset := Actual_Charset;
      end if;

      if Context.Event_Handler /= null then
         Context.Event_Handler.Unit_Parsed_Callback
           (Context  => Context,
            Unit     => Unit,
            Reparsed => not Created and then Reparse);
      end if;

      return Unit;
   end Get_Unit;

   --------------
   -- Has_Unit --
   --------------

   function Has_Unit
     (Context : Internal_Context; Unit_Filename : String) return Boolean is
   begin
      return Context.Units.Contains
        (Normalized_Unit_Filename (Context, Unit_Filename));
   end Has_Unit;

   -------------------
   -- Get_From_File --
   -------------------

   function Get_From_File
     (Context  : Internal_Context;
      Filename : String;
      Charset  : String;
      Reparse  : Boolean;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      Input : constant Internal_Lexer_Input :=
        (Kind     => File,
         Charset  => <>,
         Read_BOM => False,
         Filename => <>);
   begin
      if Reparse and then Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot reparse during tree rewriting";
      end if;

      return Get_Unit (Context, Filename, Charset, Reparse, Input, Rule);
   end Get_From_File;

   ---------------------
   -- Get_From_Buffer --
   ---------------------

   function Get_From_Buffer
     (Context  : Internal_Context;
      Filename : String;
      Charset  : String;
      Buffer   : String;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      Input : constant Internal_Lexer_Input :=
        (Kind        => Bytes_Buffer,
         Charset     => <>,
         Read_BOM    => False,
         Bytes       => Buffer'Address,
         Bytes_Count => Buffer'Length);
   begin
      if Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot parse from buffer during tree rewriting";

      elsif Context.File_Reader /= null then
         raise Precondition_Failure with
            "cannot parse from buffer with a file reader";
      end if;

      return Get_Unit (Context, Filename, Charset, True, Input, Rule);
   end Get_From_Buffer;

   --------------------
   -- Get_With_Error --
   --------------------

   function Get_With_Error
     (Context  : Internal_Context;
      Filename : String;
      Error    : Text_Type;
      Charset  : String;
      Rule     : Grammar_Rule) return Internal_Unit
   is
      use Units_Maps;

      Normalized_Filename : constant Virtual_File :=
         Normalized_Unit_Filename (Context, Filename);
      Cur                 : constant Cursor :=
         Context.Units.Find (Normalized_Filename);
   begin
      if Cur = No_Element then
         declare
            Unit : constant Internal_Unit := Create_Unit
              (Context, Normalized_Filename, Charset, Rule);
         begin
            Append (Unit.Diagnostics, No_Source_Location_Range, Error);
            return Unit;
         end;
      else
         return Element (Cur);
      end if;
   end Get_With_Error;


   -----------------------
   -- Get_From_Provider --
   -----------------------

   function Get_From_Provider
     (Context : Internal_Context;
      Name    : Text_Type;
      Kind    : Analysis_Unit_Kind;
      Charset : String;
      Reparse : Boolean) return Internal_Unit is
   begin
      if Reparse and then Has_Rewriting_Handle (Context) then
         raise Precondition_Failure with
            "cannot reparse during tree rewriting";
      end if;

      declare
         Result      : Internal_Unit;
         Dummy_Index : Positive;
      begin
         Context.Unit_Provider.Get_Unit_And_PLE_Root
           (Context, Name, Kind, Charset, Reparse, Result, Dummy_Index);
         return Result;
      exception
         when Property_Error =>
            raise Invalid_Unit_Name_Error with
               "Invalid unit name: " & Image (Name, With_Quotes => True)
               & " (" & Analysis_Unit_Kind'Image (Kind) & ")";
      end;
   end Get_From_Provider;


   -------------------
   -- Unit_Provider --
   -------------------

   function Unit_Provider
     (Context : Internal_Context) return Internal_Unit_Provider_Access
   is (Context.Unit_Provider);

   ------------------
   -- Resolve_Unit --
   ------------------

   procedure Resolve_Unit
     (Context : Internal_Context;
      Name    : Text_Type;
      Kind    : Analysis_Unit_Kind;
      Unit    : out Resolved_Unit)
   is
      --  Look for the cache entry corresponding to Unit; create one if needed

      Dummy    : Resolved_Unit_Array;
      Key      : constant Symbol_Type := Find (Context.Symbols, Name);
      Pos      : Unit_Provider_Cache_Maps.Cursor;
      Inserted : Boolean;
   begin
      Context.Unit_Provider_Cache.Insert (Key, Dummy, Pos, Inserted);
      declare
         Units : Resolved_Unit_Array renames
           Context.Unit_Provider_Cache.Reference (Pos);
         U     : Resolved_Unit renames Units (Kind);
      begin
         --  If the cache entry is not populated for the requested kind, run
         --  the query and save the result for later requests.

         if U.Filename = null then
            declare
               Provider : Internal_Unit_Provider'Class renames
                 Context.Unit_Provider.all;
               Filename : Unbounded_String;
            begin
               Provider.Get_Unit_Location
                 (Name           => Name,
                  Kind           => Kind,
                  Filename       => Filename,
                  PLE_Root_Index => U.PLE_Root_Index);
               Provider.Get_Unit_And_PLE_Root
                 (Context        => Context,
                  Name           => Name,
                  Kind           => Kind,
                  Unit           => U.Unit,
                  PLE_Root_Index => U.PLE_Root_Index);
               U.Filename := new String'(To_String (Filename));
            end;
         end if;

         Unit := U;
      end;
   end Resolve_Unit;

   -----------------------
   -- Get_Unit_Location --
   -----------------------

   procedure Get_Unit_Location
     (Context        : Internal_Context;
      Name           : Text_Type;
      Kind           : Analysis_Unit_Kind;
      Filename       : out String_Access;
      PLE_Root_Index : out Positive)
   is
      U : Resolved_Unit;
   begin
      Resolve_Unit (Context, Name, Kind, U);
      Filename := U.Filename;
      PLE_Root_Index := U.PLE_Root_Index;
   end Get_Unit_Location;

   ---------------------------
   -- Get_Unit_And_PLE_Root --
   ---------------------------

   procedure Get_Unit_And_PLE_Root
     (Context        : Internal_Context;
      Name           : Text_Type;
      Kind           : Analysis_Unit_Kind;
      Unit           : out Internal_Unit;
      PLE_Root_Index : out Positive)
   is
      U : Resolved_Unit;
   begin
      Resolve_Unit (Context, Name, Kind, U);
      Unit := U.Unit;
      PLE_Root_Index := U.PLE_Root_Index;
   end Get_Unit_And_PLE_Root;

   ----------
   -- Hash --
   ----------

   function Hash (Context : Internal_Context) return Hash_Type is
      function H is new Hash_Access (Analysis_Context_Type, Internal_Context);
   begin
      return H (Context);
   end Hash;

   ---------------------
   -- Has_With_Trivia --
   ---------------------

   function Has_With_Trivia (Context : Internal_Context) return Boolean is
   begin
      return Context.With_Trivia;
   end Has_With_Trivia;

   --------------------------------------------
   -- Discard_Errors_In_Populate_Lexical_Env --
   --------------------------------------------

   procedure Discard_Errors_In_Populate_Lexical_Env
     (Context : Internal_Context; Discard : Boolean) is
   begin
      Context.Discard_Errors_In_Populate_Lexical_Env := Discard;
   end Discard_Errors_In_Populate_Lexical_Env;

   ----------------------------------
   -- Set_Logic_Resolution_Timeout --
   ----------------------------------

   procedure Set_Logic_Resolution_Timeout
     (Context : Internal_Context; Timeout : Natural) is
   begin
      Context.Logic_Resolution_Timeout := Timeout;
   end Set_Logic_Resolution_Timeout;

   --------------------------
   -- Has_Rewriting_Handle --
   --------------------------

   function Has_Rewriting_Handle (Context : Internal_Context) return Boolean is
   begin
      return Context.Rewriting_Handle /= No_Rewriting_Handle_Pointer;
   end Has_Rewriting_Handle;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (Context : Internal_Context) is
   begin
      if Context /= null then
         Context.Ref_Count := Context.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Context : in out Internal_Context) is
   begin
      if Context /= null then
         Context.Ref_Count := Context.Ref_Count - 1;
         if Context.Ref_Count = 0 then

            --  If this context was not completely initialized, just release
            --  the allocated object. Do the full destruction otherwise.
            if Context.Initialized then
               Destroy (Context);
            end if;
            Context_Pool.Release (Context);
         end if;
      end if;
   end Dec_Ref;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Context : Internal_Context) is
   begin
      --  Destroy all named environment data structures
      for Desc of Context.Named_Envs loop
         for V of Desc.Foreign_Nodes loop
            V.Destroy;
         end loop;
         Destroy (Desc);
      end loop;
      Context.Named_Envs.Clear;

      --  If we are asked to free this context, it means that no one else have
      --  references to its analysis units, so it's safe to destroy these.
      for Unit of Context.Units loop
         Destroy (Unit);
      end loop;
      Context.Units := Units_Maps.Empty_Map;
      Context.Filenames := Virtual_File_Maps.Empty_Map;

      declare
         procedure Destroy is new Ada.Unchecked_Deallocation
           (Env_Rebindings_Type, Env_Rebindings);

         AR : Env_Rebindings_Vectors.Vector renames
            Context.Available_Rebindings;
         R  : Env_Rebindings;
      begin
         for I in AR.First_Index .. AR.Last_Index loop
            R := AR.Get (I);
            Destroy (R);
         end loop;
         AR.Destroy;
      end;

      for Pos in Context.Unit_Provider_Cache.Iterate loop
         declare
            Units : Resolved_Unit_Array renames
              Context.Unit_Provider_Cache.Reference (Pos);
         begin
            for U of Units loop
               Free (U.Filename);
            end loop;
         end;
      end loop;
      Context.Unit_Provider_Cache.Clear;

      Destroy (Context.Templates_Unit);
      AST_Envs.Destroy (Context.Root_Scope);
      Destroy (Context.Symbols);
      Destroy (Context.Parser);
      Dec_Ref (Context.File_Reader);
      Dec_Ref (Context.Unit_Provider);
      Dec_Ref (Context.Event_Handler);
   end Destroy;

   -------------
   -- Context --
   -------------

   function Context (Unit : Internal_Unit) return Internal_Context is
   begin
      return Unit.Context;
   end Context;

   ----------
   -- Hash --
   ----------

   function Hash (Unit : Internal_Unit) return Hash_Type is
      function H is new Hash_Access (Analysis_Unit_Type, Internal_Unit);
   begin
      return H (Unit);
   end Hash;

   -------------
   -- Reparse --
   -------------

   procedure Reparse (Unit : Internal_Unit; Charset : String) is
      Dummy : constant Internal_Unit := Get_From_File
        (Unit.Context, +Unit.Filename.Full_Name, Charset,
         Reparse => True,
         Rule    => Unit.Rule);
   begin
      null;
   end Reparse;

   -------------
   -- Reparse --
   -------------

   procedure Reparse (Unit : Internal_Unit; Charset : String; Buffer : String)
   is
      Dummy : constant Internal_Unit := Get_From_Buffer
        (Unit.Context, +Unit.Filename.Full_Name, Charset, Buffer, Unit.Rule);
   begin
      null;
   end Reparse;

   -----------------------
   -- Reset_Envs_Caches --
   -----------------------

   procedure Reset_Envs_Caches (Unit : Internal_Unit) is
      procedure Internal (Node : Bare_Lkt_Node);
      --  Reset env caches in ``Node`` and then in its children recursively

      Generic_Unit : constant Generic_Unit_Ptr := Convert_Unit (Unit);

      --------------
      -- Internal --
      --------------

      procedure Internal (Node : Bare_Lkt_Node) is
      begin
         if Node = null then
            return;
         end if;
         --  Make sure to only reset caches of envs belonging to this unit
         if Node.Self_Env.Owner = Generic_Unit then
            Reset_Caches (Node.Self_Env);
         end if;
         for I in 1 .. Children_Count (Node) loop
            Internal (Child (Node, I));
         end loop;
      end Internal;
   begin
      Internal (Unit.Ast_Root);
   end Reset_Envs_Caches;

   --------------------------
   -- Populate_Lexical_Env --
   --------------------------

   procedure Populate_Lexical_Env
     (Unit           : Internal_Unit;
      PLE_Root_Index : Positive
         := 1
   ) is
      Context : constant Internal_Context := Unit.Context;

      Saved_In_Populate_Lexical_Env : constant Boolean :=
        Context.In_Populate_Lexical_Env;

      Has_Errors : Boolean := False;
      --  Whether at least one Property_Error occurred during this PLE pass

   begin
      --  TODO??? Handle env invalidation when reparsing a unit and when a
      --  previous call raised a Property_Error.

      --  If we have already run PLE on this root, there is nothing to do.
      --  Otherwise, keep track of the fact that PLE was requested for it,
      --  possibly extending the vector if needed.

      if Unit.Env_Populated_Roots.Last_Index >= PLE_Root_Index
         and then Unit.Env_Populated_Roots.Get (PLE_Root_Index)
      then
         return;
      end if;
      for Dummy in Unit.Env_Populated_Roots.Last_Index + 1 .. PLE_Root_Index
      loop
         Unit.Env_Populated_Roots.Append (False);
      end loop;
      Unit.Env_Populated_Roots.Set (PLE_Root_Index, True);

      --  Create context for the PLE run: all exit points must call the Cleanup
      --  procedure above first to clean this context.

      Context.In_Populate_Lexical_Env := True;
      if Main_Trace.Active then
         Main_Trace.Trace
           ("Populating lexical envs for"
            & " unit: " & Basename (Unit));
         Main_Trace.Increase_Indent;
      end if;

      --  Fetch the node on which to run PLE: it's the unit root node, or one
      --  of its children if PLE roots are enabled and the unit has a list of
      --  PLE roots. Then run PLE itself.

      declare
         PLE_Root : Bare_Lkt_Node := Unit.Ast_Root;
      begin

         if PLE_Root /= null then
            Has_Errors := Populate_Lexical_Env (PLE_Root);
         end if;
      end;

      --  Restore the context for PLE run (undo what was done above)

      Context.In_Populate_Lexical_Env := Saved_In_Populate_Lexical_Env;
      if Main_Trace.Active then
         Main_Trace.Decrease_Indent;
         Main_Trace.Trace
           ("Finished populating lexical envs for"
            & " unit: " & Basename (Unit));
      end if;

      Reset_Envs_Caches (Unit);

      if Has_Errors and then not Context.Discard_Errors_In_Populate_Lexical_Env
      then
         raise Property_Error with
            "errors occurred in Populate_Lexical_Env";
      end if;
   end Populate_Lexical_Env;

   -----------------------------------
   -- Populate_Lexical_Env_For_Unit --
   -----------------------------------

   procedure Populate_Lexical_Env_For_Unit (Node : Bare_Lkt_Node) is
      Root  : Bare_Lkt_Node;
      Index : Natural;
   begin
      Lookup_PLE_Root (Node, Root, Index);
      if Index = 0 then
         Index := 1;
      end if;
      Populate_Lexical_Env (Node.Unit, Index);
   end Populate_Lexical_Env_For_Unit;

   ------------------
   -- Get_Filename --
   ------------------

   function Get_Filename (Unit : Internal_Unit) return String is
     (+Unit.Filename.Full_Name);

   -----------------
   -- Get_Charset --
   -----------------

   function Get_Charset (Unit : Internal_Unit) return String is
   begin
      return To_String (Unit.Charset);
   end Get_Charset;

   ---------------------
   -- Has_Diagnostics --
   ---------------------

   function Has_Diagnostics (Unit : Internal_Unit) return Boolean is
   begin
      return not Unit.Diagnostics.Is_Empty;
   end Has_Diagnostics;

   -----------------
   -- Diagnostics --
   -----------------

   function Diagnostics (Unit : Internal_Unit) return Diagnostics_Array is
      Result : Diagnostics_Array (1 .. Natural (Unit.Diagnostics.Length));
      I      : Natural := 1;
   begin
      for D of Unit.Diagnostics loop
         Result (I) := D;
         I := I + 1;
      end loop;
      return Result;
   end Diagnostics;

   ---------------------------
   -- Format_GNU_Diagnostic --
   ---------------------------

   function Format_GNU_Diagnostic
     (Unit : Internal_Unit; D : Diagnostic) return String
   is
      Filename : constant String := Basename (Unit);
      Sloc     : constant Source_Location := Start_Sloc (D.Sloc_Range);
      Msg      : constant String :=
         Image
           (Ada.Strings.Wide_Wide_Unbounded.To_Wide_Wide_String (D.Message));
   begin
      return (Filename
              & (if Sloc = No_Source_Location then "" else ":" & Image (Sloc))
              & ": " & Msg);
   end Format_GNU_Diagnostic;

   ----------
   -- Root --
   ----------

   function Root (Unit : Internal_Unit) return Bare_Lkt_Node is
     (Unit.Ast_Root);

   -----------------
   -- First_Token --
   -----------------

   function First_Token (Unit : Internal_Unit) return Token_Reference is
     (Wrap_Token_Reference (Unit.Context,
                            Unit.TDH'Access,
                            First_Token_Or_Trivia (Unit.TDH)));

   ----------------
   -- Last_Token --
   ----------------

   function Last_Token (Unit : Internal_Unit) return Token_Reference is
     (Wrap_Token_Reference (Unit.Context,
                            Unit.TDH'Access,
                            Last_Token_Or_Trivia (Unit.TDH)));

   -----------------
   -- Token_Count --
   -----------------

   function Token_Count (Unit : Internal_Unit) return Natural is
     (Unit.TDH.Tokens.Length);

   ------------------
   -- Trivia_Count --
   ------------------

   function Trivia_Count (Unit : Internal_Unit) return Natural is
     (Unit.TDH.Trivias.Length);

   ----------
   -- Text --
   ----------

   function Text (Unit : Internal_Unit) return Text_Type is
   begin
      return Text (First_Token (Unit), Last_Token (Unit));
   end Text;

   ------------------
   -- Lookup_Token --
   ------------------

   function Lookup_Token
     (Unit : Internal_Unit; Sloc : Source_Location) return Token_Reference
   is
      Result : constant Token_Or_Trivia_Index := Lookup_Token (Unit.TDH, Sloc);
   begin
      return Wrap_Token_Reference (Unit.Context, Unit.TDH'Access, Result);
   end Lookup_Token;

   ---------------------
   -- Lookup_PLE_Root --
   ---------------------

   procedure Lookup_PLE_Root
     (Node  : Bare_Lkt_Node;
      Root  : out Bare_Lkt_Node;
      Index : out Natural)
   is
      Unit : constant Internal_Unit := Node.Unit;
   begin
      --  If this unit does not contain a list of PLE roots, just return the
      --  unit root node.

      if Unit.PLE_Roots_Starting_Token.Is_Empty then
         Root := Unit.Ast_Root;
         Index := 0;
         return;
      end if;

      --  Otherwise, look for the last PLE root whose first token (in
      --  Unit.PLE_Roots_Starting_Token) appears before Node's (T). This vector
      --  is sorted by construction, so we can perform a binary search.

      declare
         T      : constant Token_Index := Node.Token_Start_Index;
         Tokens : Token_Index_Vectors.Vector renames
           Unit.PLE_Roots_Starting_Token;

         First : Positive := Tokens.First_Index;
         Last  : Positive := Tokens.Last_Index;
         I     : Positive;
      begin
         while First < Last loop

            --  Because we look for the "floor" (last element that is <= T), we
            --  need to look at the value in Last when there are only two
            --  elements left to look at. If we did not do that, then we would
            --  go into an infinite loop when Tokens[First] < T.

            I := (if First + 1 = Last
                  then Last
                  else (First + Last) / 2);
            declare
               I_T : constant Token_Index := Tokens.Get (I);
            begin
               if I_T <= T then
                  First := I;
               else
                  Last := I - 1;
               end if;
            end;
         end loop;

         Root := Child (Unit.Ast_Root, First);
         Index := First;
      end;
   end Lookup_PLE_Root;

   --------------
   -- Ple_Root --
   --------------

   function Ple_Root
     (Node : Bare_Lkt_Node) return Bare_Lkt_Node
   is
      Root        : Bare_Lkt_Node;
      Dummy_Index : Natural;
   begin
      if Node = null then
         raise Property_Error with "null node dereference";
      end if;
      Lookup_PLE_Root (Node, Root, Dummy_Index);
      return Root;
   end Ple_Root;

   ----------------------
   -- Dump_Lexical_Env --
   ----------------------

   procedure Dump_Lexical_Env (Unit : Internal_Unit) is
      Node     : constant Bare_Lkt_Node := Unit.Ast_Root;
      Root_Env : constant Lexical_Env := Unit.Context.Root_Scope;
      State    : Dump_Lexical_Env_State := (Root_Env => Root_Env, others => <>);

      function Get_Parent (Env : Lexical_Env) return Lexical_Env
      is (Unwrap (Env).Parent);

      --------------------------
      -- Explore_Parent_Chain --
      --------------------------

      procedure Explore_Parent_Chain (Env : Lexical_Env) is
         P : Lexical_Env;
      begin
         if Env /= Null_Lexical_Env then
            P := Get_Parent (Env);
            Dump_One_Lexical_Env
              (Env, Get_Env_Id (Env, State), Get_Env_Id (P, State));
            Explore_Parent_Chain (P);
         end if;
      end Explore_Parent_Chain;

      --------------
      -- Internal --
      --------------

      procedure Internal (Current : Bare_Lkt_Node) is
         Explore_Parent : Boolean := False;
         Env, Parent    : Lexical_Env;
      begin
         if Current = null then
            return;
         end if;

         --  We only dump environments that we haven't dumped before. This way
         --  we'll only dump environments at the site of their creation, and
         --  not in any subsequent link. We use the Env_Ids map to check which
         --  envs we have already seen or not.
         if not State.Env_Ids.Contains (Current.Self_Env) then
            Env := Current.Self_Env;
            Parent := Get_Parent (Env);
            Explore_Parent := not State.Env_Ids.Contains (Parent);

            Dump_One_Lexical_Env
              (Env, Get_Env_Id (Env, State), Get_Env_Id (Parent, State));

            if Explore_Parent then
               Explore_Parent_Chain (Parent);
            end if;
         end if;

         for Child of Internal_Bare_Lkt_Node_Array'(Children (Current))
         loop
            Internal (Child);
         end loop;
      end Internal;
      --  This procedure implements the main recursive logic of dumping the
      --  environments.
   begin
      Internal (Bare_Lkt_Node (Node));
   end Dump_Lexical_Env;

   --------------
   -- Get_Line --
   --------------

   function Get_Line
     (Unit : Internal_Unit; Line_Number : Positive) return Text_Type
   is
   begin
      return Get_Line (Unit.TDH, Line_Number);
   end Get_Line;

   -----------
   -- Print --
   -----------

   procedure Print (Unit : Internal_Unit; Show_Slocs : Boolean) is
   begin
      if Unit.Ast_Root = null then
         Put_Line ("<empty analysis unit>");
      else
         Print (Unit.Ast_Root, Show_Slocs);
      end if;
   end Print;

   ---------------
   -- PP_Trivia --
   ---------------

   procedure PP_Trivia (Unit : Internal_Unit) is

      procedure Process (Trivia : Token_Index) is
         Data : constant Stored_Token_Data :=
            Unit.TDH.Trivias.Get (Natural (Trivia)).T;
      begin
         Put_Line (Image (Text (Unit.TDH, Data)));
      end Process;

      Last_Token : constant Token_Index :=
         Token_Index (Token_Vectors.Last_Index (Unit.TDH.Tokens) - 1);
      --  Index for the last token in Unit excluding the Termination token
      --  (hence the -1).
   begin
      for Tok of Get_Leading_Trivias (Unit.TDH) loop
         Process (Tok);
      end loop;

      PP_Trivia (Unit.Ast_Root);

      for Tok of Get_Trivias (Unit.TDH, Last_Token) loop
         Process (Tok);
      end loop;
   end PP_Trivia;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Unit : in out Internal_Unit) is
   begin
      if Unit = No_Analysis_Unit then
         return;
      end if;

      --  Clear the env caches while the unit is still fully alive to make sure
      --  that what is accessed in ``Lexical_Env_Cache_Updated`` is still
      --  valid, as it will be called back by lexical envs that are being
      --  destroyed.
      Reset_Envs_Caches (Unit);

      Unit.PLE_Roots_Starting_Token.Destroy;
      Unit.Env_Populated_Roots.Destroy;

      Unit.Exiled_Entries.Destroy;
      Unit.Foreign_Nodes.Destroy;
      Unit.Exiled_Entries_In_NED.Destroy;
      Unit.Exiled_Envs.Destroy;
      Unit.Named_Envs.Destroy;
      Analysis_Unit_Sets.Destroy (Unit.Referenced_Units);

         Destroy (Unit.Memoization_Map);

      Destroy_Rebindings (Unit.Rebindings'Access);
      Unit.Rebindings.Destroy;

      if Unit.Ast_Root /= null then
         Destroy (Unit.Ast_Root);
      end if;

      Free (Unit.TDH);
      Free (Unit.Ast_Mem_Pool);
      Destroy_Unit_Destroyables (Unit);
      Destroyable_Vectors.Destroy (Unit.Destroyables);
      

      Free (Unit);
   end Destroy;

   -------------------
   -- Is_Token_Node --
   -------------------

   function Is_Token_Node (Node : Bare_Lkt_Node) return Boolean is
   begin
      return Is_Token_Node (Node.Kind);
   end Is_Token_Node;

   ------------------
   -- Is_Synthetic --
   ------------------

   function Is_Synthetic (Node : Bare_Lkt_Node) return Boolean is
   begin
      return Node.Kind in Synthetic_Nodes;
   end Is_Synthetic;

   ------------------------------
   -- Raise_Property_Exception --
   ------------------------------

   procedure Raise_Property_Exception
     (Node    : Bare_Lkt_Node;
      Exc     : Ada.Exceptions.Exception_Id;
      Message : String)
   is
      Sloc_Prefix : constant String :=
        (if Node = null
         then ""
         else Ada.Directories.Simple_Name (Get_Filename (Unit (Node)))
              & ":" & Image (Sloc_Range (Node)) & ": ");
   begin
      Ada.Exceptions.Raise_Exception (Exc, Sloc_Prefix & Message);
   end Raise_Property_Exception;

   ------------------------------
   -- Register_Destroyable_Gen --
   ------------------------------

   procedure Register_Destroyable_Gen
     (Unit : Internal_Unit; Object : T_Access)
   is
      function Convert is new Ada.Unchecked_Conversion
        (System.Address, Destroy_Procedure);
      procedure Destroy_Procedure (Object : in out T_Access) renames Destroy;
   begin
      Register_Destroyable_Helper
        (Unit,
         Object.all'Address,
         Convert (Destroy_Procedure'Address));
   end Register_Destroyable_Gen;

      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Bare_Lkt_Node_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Lkt_Node
   is
      function Absolute_Get
        (T : Bare_Lkt_Node_Array_Access; Index : Integer)
         return Bare_Lkt_Node
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Bare_Lkt_Node,
         Sequence_Type => Bare_Lkt_Node_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Bare_Lkt_Node;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Lkt_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Bare_Lkt_Node_Array_Access) return Bare_Lkt_Node_Array_Access is
      Ret : Bare_Lkt_Node_Array_Access := Create_Bare_Lkt_Node_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Bare_Lkt_Node_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Bare_Lkt_Node_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Bare_Lkt_Node_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Bare_Lkt_Node_Array (Items_Count : Natural) return Bare_Lkt_Node_Array_Access
   is (if Items_Count = 0
       then No_Bare_Lkt_Node_Array_Type
       else new Bare_Lkt_Node_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Bare_Lkt_Node_Array
     (Items : Internal_Bare_Lkt_Node_Array) return Bare_Lkt_Node_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Lkt_Node_Array_Type;
      end if;

      return new Bare_Lkt_Node_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Bare_Lkt_Node_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Bare_Lkt_Node_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Bare_Type_Decl_Vectors is new Liblktlang_Support.Vectors (Bare_Type_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Bare_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Type_Decl
   is
      function Absolute_Get
        (T : Bare_Type_Decl_Array_Access; Index : Integer)
         return Bare_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Bare_Type_Decl,
         Sequence_Type => Bare_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Bare_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Bare_Lkt_Node;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Bare_Type_Decl_Array_Access) return Bare_Type_Decl_Array_Access is
      Ret : Bare_Type_Decl_Array_Access := Create_Bare_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Bare_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Bare_Type_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Bare_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Bare_Type_Decl_Array (Items_Count : Natural) return Bare_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Bare_Type_Decl_Array_Type
       else new Bare_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Bare_Type_Decl_Array
     (Items : Internal_Bare_Type_Decl_Array) return Bare_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Bare_Type_Decl_Array_Type;
      end if;

      return new Bare_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Bare_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Bare_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;


      ----------
      -- Hash --
      ----------

      function Hash (R : Bare_Type_Decl_Array_Access) return Hash_Type is
         Result : Hash_Type := Initial_Hash;
      begin
         for I in R.Items'Range loop
            Result := Combine (Result, Hash (R.Items (I)));
         end loop;
         return Result;
      end Hash;


      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Boolean_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Boolean
   is
      function Absolute_Get
        (T : Boolean_Array_Access; Index : Integer)
         return Boolean
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Boolean,
         Sequence_Type => Boolean_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Boolean;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return False;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Boolean_Array_Access) return Boolean_Array_Access is
      Ret : Boolean_Array_Access := Create_Boolean_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Boolean_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Boolean_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Boolean_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Boolean_Array (Items_Count : Natural) return Boolean_Array_Access
   is (if Items_Count = 0
       then No_Boolean_Array_Type
       else new Boolean_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Boolean_Array
     (Items : Internal_Boolean_Array) return Boolean_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Boolean_Array_Type;
      end if;

      return new Boolean_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Boolean_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Boolean_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Integer_Vectors is new Liblktlang_Support.Vectors (Integer);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Integer_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Integer
   is
      function Absolute_Get
        (T : Integer_Array_Access; Index : Integer)
         return Integer
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Integer,
         Sequence_Type => Integer_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Integer;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return 0;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Integer_Array_Access) return Integer_Array_Access is
      Ret : Integer_Array_Access := Create_Integer_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Integer_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Integer_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Integer_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Integer_Array (Items_Count : Natural) return Integer_Array_Access
   is (if Items_Count = 0
       then No_Integer_Array_Type
       else new Integer_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Integer_Array
     (Items : Internal_Integer_Array) return Integer_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Integer_Array_Type;
      end if;

      return new Integer_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Integer_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Integer_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity
   is
      function Absolute_Get
        (T : Internal_Entity_Array_Access; Index : Integer)
         return Internal_Entity
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity,
         Sequence_Type => Internal_Entity_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Array_Access) return Internal_Entity_Array_Access is
      Ret : Internal_Entity_Array_Access := Create_Internal_Entity_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Array (Items_Count : Natural) return Internal_Entity_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Array_Type
       else new Internal_Entity_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));

   function Create_Internal_Entity_Array
     (Items : AST_Envs.Entity_Array) return Internal_Entity_Array_Access
   is (if Items'Length = 0
       then No_Internal_Entity_Array_Type
       else new Internal_Entity_Array_Record'
         (N         => Items'Length,
          Items     => Implementation.Internal_Internal_Entity_Array (Items),
          Ref_Count => 1));

   function Create_Internal_Entity_Array
     (Items : Internal_Internal_Entity_Array) return Internal_Entity_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Array_Type;
      end if;

      return new Internal_Entity_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Component_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Component_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Component_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Component_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Component_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Component_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Component_Decl,
         Sequence_Type => Internal_Entity_Component_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Component_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Component_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Component_Decl_Array_Access) return Internal_Entity_Component_Decl_Array_Access is
      Ret : Internal_Entity_Component_Decl_Array_Access := Create_Internal_Entity_Component_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Component_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Component_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Component_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Component_Decl_Array (Items_Count : Natural) return Internal_Entity_Component_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Component_Decl_Array_Type
       else new Internal_Entity_Component_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Component_Decl_Array
     (Items : Internal_Internal_Entity_Component_Decl_Array) return Internal_Entity_Component_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Component_Decl_Array_Type;
      end if;

      return new Internal_Entity_Component_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Component_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Component_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Enum_Class_Alt_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Enum_Class_Alt_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Enum_Class_Alt_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Enum_Class_Alt_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Enum_Class_Alt_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Enum_Class_Alt_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Enum_Class_Alt_Decl,
         Sequence_Type => Internal_Entity_Enum_Class_Alt_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Enum_Class_Alt_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Enum_Class_Alt_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) return Internal_Entity_Enum_Class_Alt_Decl_Array_Access is
      Ret : Internal_Entity_Enum_Class_Alt_Decl_Array_Access := Create_Internal_Entity_Enum_Class_Alt_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Enum_Class_Alt_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Enum_Class_Alt_Decl_Array (Items_Count : Natural) return Internal_Entity_Enum_Class_Alt_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Enum_Class_Alt_Decl_Array_Type
       else new Internal_Entity_Enum_Class_Alt_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Enum_Class_Alt_Decl_Array
     (Items : Internal_Internal_Entity_Enum_Class_Alt_Decl_Array) return Internal_Entity_Enum_Class_Alt_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Enum_Class_Alt_Decl_Array_Type;
      end if;

      return new Internal_Entity_Enum_Class_Alt_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Enum_Class_Alt_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Expr_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Expr);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Expr_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Expr
   is
      function Absolute_Get
        (T : Internal_Entity_Expr_Array_Access; Index : Integer)
         return Internal_Entity_Expr
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Expr,
         Sequence_Type => Internal_Entity_Expr_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Expr;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Expr;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Expr_Array_Access) return Internal_Entity_Expr_Array_Access is
      Ret : Internal_Entity_Expr_Array_Access := Create_Internal_Entity_Expr_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Expr_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Expr_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Expr_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Expr_Array (Items_Count : Natural) return Internal_Entity_Expr_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Expr_Array_Type
       else new Internal_Entity_Expr_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Expr_Array
     (Items : Internal_Internal_Entity_Expr_Array) return Internal_Entity_Expr_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Expr_Array_Type;
      end if;

      return new Internal_Entity_Expr_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Expr_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Expr_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Full_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Full_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Full_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Full_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Full_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Full_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Full_Decl,
         Sequence_Type => Internal_Entity_Full_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Full_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Full_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Full_Decl_Array_Access) return Internal_Entity_Full_Decl_Array_Access is
      Ret : Internal_Entity_Full_Decl_Array_Access := Create_Internal_Entity_Full_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Full_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Full_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Full_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Full_Decl_Array (Items_Count : Natural) return Internal_Entity_Full_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Full_Decl_Array_Type
       else new Internal_Entity_Full_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Full_Decl_Array
     (Items : Internal_Internal_Entity_Full_Decl_Array) return Internal_Entity_Full_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Full_Decl_Array_Type;
      end if;

      return new Internal_Entity_Full_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Full_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Full_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Generic_Formal_Type_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Generic_Formal_Type_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Generic_Formal_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Generic_Formal_Type_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Generic_Formal_Type_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Generic_Formal_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Generic_Formal_Type_Decl,
         Sequence_Type => Internal_Entity_Generic_Formal_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Generic_Formal_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Generic_Formal_Type_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Generic_Formal_Type_Decl_Array_Access) return Internal_Entity_Generic_Formal_Type_Decl_Array_Access is
      Ret : Internal_Entity_Generic_Formal_Type_Decl_Array_Access := Create_Internal_Entity_Generic_Formal_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Generic_Formal_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Generic_Formal_Type_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Generic_Formal_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Generic_Formal_Type_Decl_Array (Items_Count : Natural) return Internal_Entity_Generic_Formal_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Generic_Formal_Type_Decl_Array_Type
       else new Internal_Entity_Generic_Formal_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Generic_Formal_Type_Decl_Array
     (Items : Internal_Internal_Entity_Generic_Formal_Type_Decl_Array) return Internal_Entity_Generic_Formal_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Generic_Formal_Type_Decl_Array_Type;
      end if;

      return new Internal_Entity_Generic_Formal_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Generic_Formal_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Generic_Formal_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Param_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Param);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Param_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Param
   is
      function Absolute_Get
        (T : Internal_Entity_Param_Array_Access; Index : Integer)
         return Internal_Entity_Param
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Param,
         Sequence_Type => Internal_Entity_Param_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Param;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Param;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Param_Array_Access) return Internal_Entity_Param_Array_Access is
      Ret : Internal_Entity_Param_Array_Access := Create_Internal_Entity_Param_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Param_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Param_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Param_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Param_Array (Items_Count : Natural) return Internal_Entity_Param_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Param_Array_Type
       else new Internal_Entity_Param_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Param_Array
     (Items : Internal_Internal_Entity_Param_Array) return Internal_Entity_Param_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Param_Array_Type;
      end if;

      return new Internal_Entity_Param_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Param_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Param_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Entity_Type_Decl_Vectors is new Liblktlang_Support.Vectors (Internal_Entity_Type_Decl);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Entity_Type_Decl_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Entity_Type_Decl
   is
      function Absolute_Get
        (T : Internal_Entity_Type_Decl_Array_Access; Index : Integer)
         return Internal_Entity_Type_Decl
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Entity_Type_Decl,
         Sequence_Type => Internal_Entity_Type_Decl_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Entity_Type_Decl;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Entity_Type_Decl;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Entity_Type_Decl_Array_Access) return Internal_Entity_Type_Decl_Array_Access is
      Ret : Internal_Entity_Type_Decl_Array_Access := Create_Internal_Entity_Type_Decl_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Type_Decl_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Entity_Type_Decl_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Type_Decl_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Entity_Type_Decl_Array (Items_Count : Natural) return Internal_Entity_Type_Decl_Array_Access
   is (if Items_Count = 0
       then No_Internal_Entity_Type_Decl_Array_Type
       else new Internal_Entity_Type_Decl_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Entity_Type_Decl_Array
     (Items : Internal_Internal_Entity_Type_Decl_Array) return Internal_Entity_Type_Decl_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Entity_Type_Decl_Array_Type;
      end if;

      return new Internal_Entity_Type_Decl_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Entity_Type_Decl_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Type_Decl_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Env_Assoc_Vectors is new Liblktlang_Support.Vectors (Internal_Env_Assoc);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Env_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Env_Assoc
   is
      function Absolute_Get
        (T : Internal_Env_Assoc_Array_Access; Index : Integer)
         return Internal_Env_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Env_Assoc,
         Sequence_Type => Internal_Env_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Env_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Env_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Env_Assoc_Array_Access) return Internal_Env_Assoc_Array_Access is
      Ret : Internal_Env_Assoc_Array_Access := Create_Internal_Env_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Env_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Env_Assoc_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Env_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Env_Assoc_Array (Items_Count : Natural) return Internal_Env_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Env_Assoc_Array_Type
       else new Internal_Env_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Env_Assoc_Array
     (Items : Internal_Internal_Env_Assoc_Array) return Internal_Env_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Env_Assoc_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Env_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Env_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Env_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Env_Kv_Vectors is new Liblktlang_Support.Vectors (Internal_Env_Kv);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Env_Kv_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Env_Kv
   is
      function Absolute_Get
        (T : Internal_Env_Kv_Array_Access; Index : Integer)
         return Internal_Env_Kv
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Env_Kv,
         Sequence_Type => Internal_Env_Kv_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Env_Kv;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Env_Kv;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Env_Kv_Array_Access) return Internal_Env_Kv_Array_Access is
      Ret : Internal_Env_Kv_Array_Access := Create_Internal_Env_Kv_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Env_Kv_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Env_Kv_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Env_Kv_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Env_Kv_Array (Items_Count : Natural) return Internal_Env_Kv_Array_Access
   is (if Items_Count = 0
       then No_Internal_Env_Kv_Array_Type
       else new Internal_Env_Kv_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Env_Kv_Array
     (Items : Internal_Internal_Env_Kv_Array) return Internal_Env_Kv_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Env_Kv_Array_Type;
      end if;

      return new Internal_Env_Kv_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Env_Kv_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Env_Kv_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Formal_Param_Vectors is new Liblktlang_Support.Vectors (Internal_Formal_Param);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Formal_Param_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Formal_Param
   is
      function Absolute_Get
        (T : Internal_Formal_Param_Array_Access; Index : Integer)
         return Internal_Formal_Param
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Formal_Param,
         Sequence_Type => Internal_Formal_Param_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Formal_Param;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Formal_Param;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Formal_Param_Array_Access) return Internal_Formal_Param_Array_Access is
      Ret : Internal_Formal_Param_Array_Access := Create_Internal_Formal_Param_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Formal_Param_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Formal_Param_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Formal_Param_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Formal_Param_Array (Items_Count : Natural) return Internal_Formal_Param_Array_Access
   is (if Items_Count = 0
       then No_Internal_Formal_Param_Array_Type
       else new Internal_Formal_Param_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Formal_Param_Array
     (Items : Internal_Internal_Formal_Param_Array) return Internal_Formal_Param_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Formal_Param_Array_Type;
      end if;

      return new Internal_Formal_Param_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Formal_Param_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Formal_Param_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Generic_Param_Assoc_Vectors is new Liblktlang_Support.Vectors (Internal_Generic_Param_Assoc);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Generic_Param_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Generic_Param_Assoc
   is
      function Absolute_Get
        (T : Internal_Generic_Param_Assoc_Array_Access; Index : Integer)
         return Internal_Generic_Param_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Generic_Param_Assoc,
         Sequence_Type => Internal_Generic_Param_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Generic_Param_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Generic_Param_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Generic_Param_Assoc_Array_Access) return Internal_Generic_Param_Assoc_Array_Access is
      Ret : Internal_Generic_Param_Assoc_Array_Access := Create_Internal_Generic_Param_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Generic_Param_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Generic_Param_Assoc_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Generic_Param_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Generic_Param_Assoc_Array (Items_Count : Natural) return Internal_Generic_Param_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Generic_Param_Assoc_Array_Type
       else new Internal_Generic_Param_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Generic_Param_Assoc_Array
     (Items : Internal_Internal_Generic_Param_Assoc_Array) return Internal_Generic_Param_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Generic_Param_Assoc_Array_Type;
      end if;

      return new Internal_Generic_Param_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Generic_Param_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Generic_Param_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Infer_Instantiation_Vectors is new Liblktlang_Support.Vectors (Internal_Infer_Instantiation);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Infer_Instantiation_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Infer_Instantiation
   is
      function Absolute_Get
        (T : Internal_Infer_Instantiation_Array_Access; Index : Integer)
         return Internal_Infer_Instantiation
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Infer_Instantiation,
         Sequence_Type => Internal_Infer_Instantiation_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Infer_Instantiation;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Infer_Instantiation;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Infer_Instantiation_Array_Access) return Internal_Infer_Instantiation_Array_Access is
      Ret : Internal_Infer_Instantiation_Array_Access := Create_Internal_Infer_Instantiation_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Infer_Instantiation_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Infer_Instantiation_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Infer_Instantiation_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Infer_Instantiation_Array (Items_Count : Natural) return Internal_Infer_Instantiation_Array_Access
   is (if Items_Count = 0
       then No_Internal_Infer_Instantiation_Array_Type
       else new Internal_Infer_Instantiation_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Infer_Instantiation_Array
     (Items : Internal_Internal_Infer_Instantiation_Array) return Internal_Infer_Instantiation_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Infer_Instantiation_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Infer_Instantiation_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Infer_Instantiation_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Infer_Instantiation_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Inner_Env_Assoc_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Inner_Env_Assoc
   is
      function Absolute_Get
        (T : Internal_Inner_Env_Assoc_Array_Access; Index : Integer)
         return Internal_Inner_Env_Assoc
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Inner_Env_Assoc,
         Sequence_Type => Internal_Inner_Env_Assoc_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Inner_Env_Assoc;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Inner_Env_Assoc;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Inner_Env_Assoc_Array_Access) return Internal_Inner_Env_Assoc_Array_Access is
      Ret : Internal_Inner_Env_Assoc_Array_Access := Create_Internal_Inner_Env_Assoc_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Inner_Env_Assoc_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Inner_Env_Assoc_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Inner_Env_Assoc_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Inner_Env_Assoc_Array (Items_Count : Natural) return Internal_Inner_Env_Assoc_Array_Access
   is (if Items_Count = 0
       then No_Internal_Inner_Env_Assoc_Array_Type
       else new Internal_Inner_Env_Assoc_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Inner_Env_Assoc_Array
     (Items : Internal_Internal_Inner_Env_Assoc_Array) return Internal_Inner_Env_Assoc_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Inner_Env_Assoc_Array_Type;
      end if;

      return new Internal_Inner_Env_Assoc_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Inner_Env_Assoc_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Inner_Env_Assoc_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Logic_Context_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Logic_Context
   is
      function Absolute_Get
        (T : Internal_Logic_Context_Array_Access; Index : Integer)
         return Internal_Logic_Context
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Logic_Context,
         Sequence_Type => Internal_Logic_Context_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Logic_Context;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Logic_Context;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Logic_Context_Array_Access) return Internal_Logic_Context_Array_Access is
      Ret : Internal_Logic_Context_Array_Access := Create_Internal_Logic_Context_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Logic_Context_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Logic_Context_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Logic_Context_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Logic_Context_Array (Items_Count : Natural) return Internal_Logic_Context_Array_Access
   is (if Items_Count = 0
       then No_Internal_Logic_Context_Array_Type
       else new Internal_Logic_Context_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Logic_Context_Array
     (Items : Internal_Internal_Logic_Context_Array) return Internal_Logic_Context_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Logic_Context_Array_Type;
      end if;

      return new Internal_Logic_Context_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Logic_Context_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Logic_Context_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Param_Match_Vectors is new Liblktlang_Support.Vectors (Internal_Param_Match);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Param_Match_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Param_Match
   is
      function Absolute_Get
        (T : Internal_Param_Match_Array_Access; Index : Integer)
         return Internal_Param_Match
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Param_Match,
         Sequence_Type => Internal_Param_Match_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Param_Match;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Param_Match;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Param_Match_Array_Access) return Internal_Param_Match_Array_Access is
      Ret : Internal_Param_Match_Array_Access := Create_Internal_Param_Match_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Param_Match_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Param_Match_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Param_Match_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Param_Match_Array (Items_Count : Natural) return Internal_Param_Match_Array_Access
   is (if Items_Count = 0
       then No_Internal_Param_Match_Array_Type
       else new Internal_Param_Match_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Param_Match_Array
     (Items : Internal_Internal_Param_Match_Array) return Internal_Param_Match_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Param_Match_Array_Type;
      end if;

      return new Internal_Param_Match_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Param_Match_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Param_Match_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Semantic_Result_Vectors is new Liblktlang_Support.Vectors (Internal_Semantic_Result);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Semantic_Result_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Semantic_Result
   is
      function Absolute_Get
        (T : Internal_Semantic_Result_Array_Access; Index : Integer)
         return Internal_Semantic_Result
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Semantic_Result,
         Sequence_Type => Internal_Semantic_Result_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Semantic_Result;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Semantic_Result;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Semantic_Result_Array_Access) return Internal_Semantic_Result_Array_Access is
      Ret : Internal_Semantic_Result_Array_Access := Create_Internal_Semantic_Result_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Semantic_Result_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Semantic_Result_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Semantic_Result_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Semantic_Result_Array (Items_Count : Natural) return Internal_Semantic_Result_Array_Access
   is (if Items_Count = 0
       then No_Internal_Semantic_Result_Array_Type
       else new Internal_Semantic_Result_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Semantic_Result_Array
     (Items : Internal_Internal_Semantic_Result_Array) return Internal_Semantic_Result_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Semantic_Result_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Semantic_Result_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Semantic_Result_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Semantic_Result_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Solver_Diagnostic_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Solver_Diagnostic
   is
      function Absolute_Get
        (T : Internal_Solver_Diagnostic_Array_Access; Index : Integer)
         return Internal_Solver_Diagnostic
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Solver_Diagnostic,
         Sequence_Type => Internal_Solver_Diagnostic_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Solver_Diagnostic;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Solver_Diagnostic;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Solver_Diagnostic_Array_Access) return Internal_Solver_Diagnostic_Array_Access is
      Ret : Internal_Solver_Diagnostic_Array_Access := Create_Internal_Solver_Diagnostic_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Solver_Diagnostic_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Solver_Diagnostic_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Solver_Diagnostic_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Solver_Diagnostic_Array (Items_Count : Natural) return Internal_Solver_Diagnostic_Array_Access
   is (if Items_Count = 0
       then No_Internal_Solver_Diagnostic_Array_Type
       else new Internal_Solver_Diagnostic_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Solver_Diagnostic_Array
     (Items : Internal_Internal_Solver_Diagnostic_Array) return Internal_Solver_Diagnostic_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Solver_Diagnostic_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Solver_Diagnostic_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Solver_Diagnostic_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Solver_Diagnostic_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Internal_Tree_Semantic_Result_Vectors is new Liblktlang_Support.Vectors (Internal_Tree_Semantic_Result);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Internal_Tree_Semantic_Result_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Internal_Tree_Semantic_Result
   is
      function Absolute_Get
        (T : Internal_Tree_Semantic_Result_Array_Access; Index : Integer)
         return Internal_Tree_Semantic_Result
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Internal_Tree_Semantic_Result,
         Sequence_Type => Internal_Tree_Semantic_Result_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Internal_Tree_Semantic_Result;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return No_Tree_Semantic_Result;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Internal_Tree_Semantic_Result_Array_Access) return Internal_Tree_Semantic_Result_Array_Access is
      Ret : Internal_Tree_Semantic_Result_Array_Access := Create_Internal_Tree_Semantic_Result_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Tree_Semantic_Result_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Internal_Tree_Semantic_Result_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Tree_Semantic_Result_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Internal_Tree_Semantic_Result_Array (Items_Count : Natural) return Internal_Tree_Semantic_Result_Array_Access
   is (if Items_Count = 0
       then No_Internal_Tree_Semantic_Result_Array_Type
       else new Internal_Tree_Semantic_Result_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Internal_Tree_Semantic_Result_Array
     (Items : Internal_Internal_Tree_Semantic_Result_Array) return Internal_Tree_Semantic_Result_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Internal_Tree_Semantic_Result_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Internal_Tree_Semantic_Result_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Internal_Tree_Semantic_Result_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Tree_Semantic_Result_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package Lexical_Env_Vectors is new Liblktlang_Support.Vectors (Lexical_Env);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Lexical_Env_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Lexical_Env
   is
      function Absolute_Get
        (T : Lexical_Env_Array_Access; Index : Integer)
         return Lexical_Env
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Lexical_Env,
         Sequence_Type => Lexical_Env_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Lexical_Env;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Empty_Env;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Lexical_Env_Array_Access) return Lexical_Env_Array_Access is
      Ret : Lexical_Env_Array_Access := Create_Lexical_Env_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Lexical_Env_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Lexical_Env_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Lexical_Env_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Lexical_Env_Array (Items_Count : Natural) return Lexical_Env_Array_Access
   is (if Items_Count = 0
       then No_Lexical_Env_Array_Type
       else new Lexical_Env_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Lexical_Env_Array
     (Items : Internal_Lexical_Env_Array) return Lexical_Env_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Lexical_Env_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new Lexical_Env_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Lexical_Env_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Lexical_Env_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   

      package String_Type_Vectors is new Liblktlang_Support.Vectors (String_Type);

   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : String_Type_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return String_Type
   is
      function Absolute_Get
        (T : String_Type_Array_Access; Index : Integer)
         return String_Type
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => String_Type,
         Sequence_Type => String_Type_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : String_Type;
   begin
      if Relative_Get (T, Index, Result) then
            Inc_Ref (Result);
         return Result;
      elsif Or_Null then
         return Empty_String;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : String_Type_Array_Access) return String_Type_Array_Access is
      Ret : String_Type_Array_Access := Create_String_Type_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
         for Item of Ret.Items loop
            Inc_Ref (Item);
         end loop;
      return Ret;
   end Concat;

      ------------------
      -- Join_Strings --
      ------------------

      function Join_Strings
        (Separator : String_Type;
         Strings   : String_Type_Array_Access) return String_Type
      is
         Separator_Length : constant Natural := Separator.Length;
         Length           : Natural := 0;
         First            : Boolean;
      begin
         --  First, compute the length of the result: the sum of all string
         --  lengths in Strings.
         First := True;
         for S of Strings.Items loop
            if First then
               First := False;
            else
               Length := Length + Separator_Length;
            end if;
            Length := Length + S.Length;
         end loop;

         --  Create the result string with the correct length. Do not use our
         --  constructor and initialize the content in-place, to avoid extra
         --  copies.
         return Result : constant String_Type :=
            new String_Record (Length)
         do
            Result.Ref_Count := 1;

            --  Now copy the content of all strings into the result
            declare
               Last : Natural := 0;
            begin
               First := True;
               for S of Strings.Items loop
                  if First then
                     First := False;
                  else
                     Result.Content (Last + 1 .. Last + Separator_Length) :=
                        Separator.Content;
                     Last := Last + Separator_Length;
                  end if;
                  Result.Content (Last + 1 .. Last + S.Length) := S.Content;
                  Last := Last + S.Length;
               end loop;
            end;
         end return;
      end Join_Strings;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : String_Type_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : String_Type_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out String_Type_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
            for Item of T.Items loop
               Dec_Ref (Item);
            end loop;
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_String_Type_Array (Items_Count : Natural) return String_Type_Array_Access
   is (if Items_Count = 0
       then No_String_Type_Array_Type
       else new String_Type_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_String_Type_Array
     (Items : Internal_String_Type_Array) return String_Type_Array_Access is
   begin
      if Items'Length = 0 then
         return No_String_Type_Array_Type;
      end if;

         for El of Items loop
            Inc_Ref (El);
         end loop;
      return new String_Type_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : String_Type_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               not Equivalent (L.Items (I), R.Items (I))
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : String_Type_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;




      

   


   ---------
   -- Get --
   ---------

   function Get
     (Node    : Bare_Lkt_Node;
      T       : Symbol_Type_Array_Access;
      Index   : Integer;
      Or_Null : Boolean := False) return Symbol_Type
   is
      function Absolute_Get
        (T : Symbol_Type_Array_Access; Index : Integer)
         return Symbol_Type
      is
        (T.Items (Index + 1)); --  T.Items is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Symbol_Type,
         Sequence_Type => Symbol_Type_Array_Access,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Symbol_Type;
   begin
      if Relative_Get (T, Index, Result) then
         return Result;
      elsif Or_Null then
         return No_Symbol;
      else
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "out-of-bounds array access");
      end if;
   end Get;

   ------------
   -- Concat --
   ------------

   function Concat (L, R : Symbol_Type_Array_Access) return Symbol_Type_Array_Access is
      Ret : Symbol_Type_Array_Access := Create_Symbol_Type_Array (Length (L) + Length (R));
   begin
      Ret.Items := L.Items & R.Items;
      return Ret;
   end Concat;


   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Symbol_Type_Array_Access) is
   begin
      if T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   ------------
   -- Length --
   ------------

   function Length (T : Symbol_Type_Array_Access) return Natural is (T.N);

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Symbol_Type_Array_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

   function Create_Symbol_Type_Array (Items_Count : Natural) return Symbol_Type_Array_Access
   is (if Items_Count = 0
       then No_Symbol_Type_Array_Type
       else new Symbol_Type_Array_Record'(N => Items_Count, Ref_Count => 1, Items => <>));


   function Create_Symbol_Type_Array
     (Items : Internal_Symbol_Type_Array) return Symbol_Type_Array_Access is
   begin
      if Items'Length = 0 then
         return No_Symbol_Type_Array_Type;
      end if;

      return new Symbol_Type_Array_Record'
        (N => Items'Length, Ref_Count => 1, Items => Items);
   end;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (L, R : Symbol_Type_Array_Access) return Boolean is
   begin
      if L.N /= R.N then
         return False;
      end if;

      for I in L.Items'Range loop
         if
               L.Items (I) /= R.Items (I)
         then
            return False;
         end if;
      end loop;

      return True;
   end Equivalent;


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Symbol_Type_Array_Access) return String is
         Result : Unbounded_String;
      begin
         Append (Result, "[");
         for I in A.Items'Range loop
            if I > A.Items'First then
               Append (Result, ", ");
            end if;
            Append (Result, Trace_Image (A.Items (I)));
         end loop;
         Append (Result, "]");
         return To_String (Result);
      end Trace_Image;





         

   

   ----------
   -- Next --
   ----------

   function Next
     (T       : Bare_Lkt_Node_Iterator_Access;
      Element : out Bare_Lkt_Node) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);

      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Bare_Lkt_Node_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Bare_Lkt_Node_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Bare_Lkt_Node_Iterator_Access) return String is
      begin
         return "<Iterator of LktNode, index="
                & A.Index'Image & ">";
      end Trace_Image;


         

   

   ----------
   -- Next --
   ----------

   function Next
     (T       : Internal_Entity_Iterator_Access;
      Element : out Internal_Entity) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);

      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Entity_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Entity_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Entity_Iterator_Access) return String is
      begin
         return "<Iterator of LktNode.entity, index="
                & A.Index'Image & ">";
      end Trace_Image;


         

   

   ----------
   -- Next --
   ----------

   function Next
     (T       : Internal_Inner_Env_Assoc_Iterator_Access;
      Element : out Internal_Inner_Env_Assoc) return Boolean is
   begin
      if T = null then
         raise Property_Error with "null access dereference";
      end if;
      Check_Safety_Net (T.Safety_Net);

      if T.Index > T.Elements.Items'Last then
         return False;
      else
         Element := T.Elements.Items (T.Index);
         T.Index := T.Index + 1;
         return True;
      end if;
   end Next;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (T : Internal_Inner_Env_Assoc_Iterator_Access) is
   begin
      if T /= null and then T.Ref_Count >= 0 then
         T.Ref_Count := T.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (T : in out Internal_Inner_Env_Assoc_Iterator_Access) is
   begin
      if T = null or else T.Ref_Count < 0 then
         return;
      end if;

      if T.Ref_Count = 1 then
         Dec_Ref (T.Elements);
         Free (T);
      else
         T.Ref_Count := T.Ref_Count - 1;
         T := null;
      end if;
   end Dec_Ref;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (A : Internal_Inner_Env_Assoc_Iterator_Access) return String is
      begin
         return "<Iterator of InnerEnvAssoc, index="
                & A.Index'Image & ">";
      end Trace_Image;



   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Internal_Unit) return Boolean is
   begin
      return Left.Filename < Right.Filename;
   end "<";

      



function Hash (Key : Mmz_Key_Item) return Hash_Type;
function Equivalent (L, R : Mmz_Key_Item) return Boolean;
procedure Destroy (Key : in out Mmz_Key_Array_Access);

----------------
-- Equivalent --
----------------

function Equivalent (L, R : Mmz_Key_Item) return Boolean is
begin
   if L.Kind /= R.Kind then
      return False;
   end if;

   case L.Kind is
         when Mmz_Bare_Call_Expr =>
            
               return L.As_Bare_Call_Expr = R.As_Bare_Call_Expr;
         when Mmz_Bare_Generic_Decl =>
            
               return L.As_Bare_Generic_Decl = R.As_Bare_Generic_Decl;
         when Mmz_Bare_Instantiated_Generic_Type =>
            
               return L.As_Bare_Instantiated_Generic_Type = R.As_Bare_Instantiated_Generic_Type;
         when Mmz_Bare_Lambda_Arg_Decl =>
            
               return L.As_Bare_Lambda_Arg_Decl = R.As_Bare_Lambda_Arg_Decl;
         when Mmz_Bare_Langkit_Root =>
            
               return L.As_Bare_Langkit_Root = R.As_Bare_Langkit_Root;
         when Mmz_Bare_Lexer_Decl =>
            
               return L.As_Bare_Lexer_Decl = R.As_Bare_Lexer_Decl;
         when Mmz_Bare_Lkt_Node =>
            
               return L.As_Bare_Lkt_Node = R.As_Bare_Lkt_Node;
         when Mmz_Bare_Type_Decl =>
            
               return L.As_Bare_Type_Decl = R.As_Bare_Type_Decl;
         when Mmz_Bare_Type_Decl_Array_Access =>
            
               return Equivalent (L.As_Bare_Type_Decl_Array_Access, R.As_Bare_Type_Decl_Array_Access);
         when Mmz_Boolean =>
            
               return L.As_Boolean = R.As_Boolean;
         when Mmz_Env_Rebindings =>
            
               return L.As_Env_Rebindings = R.As_Env_Rebindings;
         when Mmz_Internal_Entity_Info =>
            
               return L.As_Internal_Entity_Info = R.As_Internal_Entity_Info;
         when Mmz_Internal_Metadata =>
            
               return L.As_Internal_Metadata = R.As_Internal_Metadata;
         when Mmz_Symbol_Type =>
            
               return L.As_Symbol_Type = R.As_Symbol_Type;
   end case;
end Equivalent;

----------
-- Hash --
----------

function Hash (Key : Mmz_Key_Item) return Hash_Type is
begin
   case Key.Kind is
         when Mmz_Bare_Call_Expr =>
            return Hash (Key.As_Bare_Call_Expr);
         when Mmz_Bare_Generic_Decl =>
            return Hash (Key.As_Bare_Generic_Decl);
         when Mmz_Bare_Instantiated_Generic_Type =>
            return Hash (Key.As_Bare_Instantiated_Generic_Type);
         when Mmz_Bare_Lambda_Arg_Decl =>
            return Hash (Key.As_Bare_Lambda_Arg_Decl);
         when Mmz_Bare_Langkit_Root =>
            return Hash (Key.As_Bare_Langkit_Root);
         when Mmz_Bare_Lexer_Decl =>
            return Hash (Key.As_Bare_Lexer_Decl);
         when Mmz_Bare_Lkt_Node =>
            return Hash (Key.As_Bare_Lkt_Node);
         when Mmz_Bare_Type_Decl =>
            return Hash (Key.As_Bare_Type_Decl);
         when Mmz_Bare_Type_Decl_Array_Access =>
            return Hash (Key.As_Bare_Type_Decl_Array_Access);
         when Mmz_Boolean =>
            return Hash (Key.As_Boolean);
         when Mmz_Env_Rebindings =>
            return Hash (Key.As_Env_Rebindings);
         when Mmz_Internal_Entity_Info =>
            return Hash (Key.As_Internal_Entity_Info);
         when Mmz_Internal_Metadata =>
            return Hash (Key.As_Internal_Metadata);
         when Mmz_Symbol_Type =>
            return Hash (Key.As_Symbol_Type);
   end case;
end Hash;

----------
-- Hash --
----------

function Hash (Key : Mmz_Key) return Hash_Type is
   Result : Hash_Type := Mmz_Property'Pos (Key.Property);
begin
   for K of Key.Items.all loop
      Result := Combine (Result, Hash (K));
   end loop;
   return Result;
end Hash;

----------------
-- Equivalent --
----------------

function Equivalent (L, R : Mmz_Key) return Boolean is
   L_Items : Mmz_Key_Array renames L.Items.all;
   R_Items : Mmz_Key_Array renames R.Items.all;
begin
   if L.Property /= R.Property or else L_Items'Length /= R_Items'Length then
      return False;
   end if;

   for I in L_Items'Range loop
      if not Equivalent (L_Items (I), R_Items (I)) then
         return False;
      end if;
   end loop;

   return True;
end Equivalent;

-------------
-- Destroy --
-------------

procedure Destroy (Map : in out Memoization_Maps.Map) is
   use Memoization_Maps;

   --  We need keys and values to be valid when clearing the memoization map,
   --  but on the other hand we need to free keys and values as well. To
   --  achieve both goals, we first copy key and values into arrays, then we
   --  clear the map, and then we free keys/values in the arrays. Allocate both
   --  arrays on the heap to avoid stack overflow, as they can be quite big.

   Length : constant Natural := Natural (Map.Length);

   type Key_Array is array (1 .. Length) of Mmz_Key_Array_Access;
   type Key_Array_Access is access Key_Array;
   procedure Free is new Ada.Unchecked_Deallocation
     (Key_Array, Key_Array_Access);

   type Value_Array is array (1 .. Length) of Mmz_Value;
   type Value_Array_Access is access Value_Array;
   procedure Free is new Ada.Unchecked_Deallocation
     (Value_Array, Value_Array_Access);

   Keys   : Key_Array_Access := new Key_Array;
   Values : Value_Array_Access := new Value_Array;
   I      : Positive := 1;
begin
   for Cur in Map.Iterate loop
      Keys (I) := Key (Cur).Items;
      Values (I) := Element (Cur);
      I := I + 1;
   end loop;

   Map.Clear;

   for K_Array of Keys.all loop
      Destroy (K_Array);
   end loop;

   
   for V of Values.all loop
      case V.Kind is
         when Mmz_Error =>
            Free_Memoized_Error (V.Exc_Id, V.Exc_Msg);

            when Mmz_Internal_Entity_Type_Decl_Array_Access =>
               Dec_Ref (V.As_Internal_Entity_Type_Decl_Array_Access);
            when Mmz_Internal_Env_Assoc_Array_Access =>
               Dec_Ref (V.As_Internal_Env_Assoc_Array_Access);
            when Mmz_Internal_Generic_Param_Assoc_Array_Access =>
               Dec_Ref (V.As_Internal_Generic_Param_Assoc_Array_Access);
            when Mmz_Internal_Param_Match_Array_Access =>
               Dec_Ref (V.As_Internal_Param_Match_Array_Access);
            when Mmz_Internal_Semantic_Result_Array_Access =>
               Dec_Ref (V.As_Internal_Semantic_Result_Array_Access);
            when Mmz_Lexical_Env =>
               Dec_Ref (V.As_Lexical_Env);

         when others => null;
      end case;
   end loop;

   Free (Keys);
   Free (Values);
end Destroy;

-------------
-- Destroy --
-------------

procedure Destroy (Key : in out Mmz_Key_Array_Access) is
   procedure Free is new Ada.Unchecked_Deallocation
     (Mmz_Key_Array, Mmz_Key_Array_Access);
begin
   

      for K of Key.all loop
         case K.Kind is
               when Mmz_Bare_Type_Decl_Array_Access =>
                  Dec_Ref (K.As_Bare_Type_Decl_Array_Access);

            when others => null;
         end case;
      end loop;
   Free (Key);
end Destroy;

-------------------------
-- Find_Memoized_Value --
-------------------------

function Find_Memoized_Value
  (Unit       : Internal_Unit;
   Handle     : out Memoization_Handle;
   Value      : out Mmz_Value;
   Create_Key : access function return Mmz_Key) return Boolean
is
   Inserted : Boolean;
begin
   --  Make sure that we don't lookup stale caches
   Reset_Caches (Unit);

   --  Initialize handle: create the key and create a cursor pointing to an
   --  existing entry.
   Handle.Key := Create_Key.all;
   Handle.Cache_Version := Unit.Cache_Version;
   Value := (Kind => Mmz_Evaluating);
   Unit.Memoization_Map.Insert (Handle.Key, Value, Handle.Cur, Inserted);

   --  No existing entry yet? The above just created one. Otherwise, destroy
   --  our key and reuse the existing entry's.
   if not Inserted then
      Destroy (Handle.Key.Items);
      Handle.Key := Memoization_Maps.Key (Handle.Cur);
      Value := Memoization_Maps.Element (Handle.Cur);
   end if;

   return not Inserted;
end Find_Memoized_Value;

------------------------
-- Add_Memoized_Value --
------------------------

procedure Add_Memoized_Value
  (Unit   : Internal_Unit;
   Handle : in out Memoization_Handle;
   Value  : Mmz_Value;
   Stored : out Boolean) is
begin
   --  If Handle was created using a memoization map that has been since then
   --  reset, do nothing: the result can be partly stale due to the event that
   --  triggered the memoization tables reset.

   Stored := Unit.Cache_Version <= Handle.Cache_Version;
   if Stored then
      Unit.Memoization_Map.Replace_Element (Handle.Cur, Value);
   end if;
end Add_Memoized_Value;

------------------------
-- Add_Memoized_Error --
------------------------

procedure Add_Memoized_Error
  (Unit   : Internal_Unit;
   Handle : in out Memoization_Handle;
   Exc    : Ada.Exceptions.Exception_Occurrence;
   Stored : out Boolean)
is
   Value : Mmz_Value (Kind => Mmz_Error);
begin
   Store_Memoized_Error (Exc, Value.Exc_Id, Value.Exc_Msg);
   Add_Memoized_Value (Unit, Handle, Value, Stored);
   if not Stored then
      Free_Memoized_Error (Value.Exc_Id, Value.Exc_Msg);
   end if;
end Add_Memoized_Error;

----------------------------
-- Reraise_Memoized_Error --
----------------------------

procedure Reraise_Memoized_Error (Value : Mmz_Value) is
begin
   Reraise_Memoized_Error (Value.Exc_Id, Value.Exc_Msg);
end Reraise_Memoized_Error;

--------------------------
-- Store_Memoized_Error --
--------------------------

procedure Store_Memoized_Error
  (Exc     : Ada.Exceptions.Exception_Occurrence;
   Exc_Id  : out Ada.Exceptions.Exception_Id;
   Exc_Msg : out String_Access) is
begin
   Exc_Id := Ada.Exceptions.Exception_Identity (Exc);
   Exc_Msg := new String'(Ada.Exceptions.Exception_Message (Exc));
end Store_Memoized_Error;

-------------------------
-- Free_Memoized_Error --
-------------------------

procedure Free_Memoized_Error
  (Exc_Id  : in out Ada.Exceptions.Exception_Id;
   Exc_Msg : in out String_Access)
is
   pragma Unreferenced (Exc_Id);
begin
   Free (Exc_Msg);
end Free_Memoized_Error;

----------------------------
-- Reraise_Memoized_Error --
----------------------------

procedure Reraise_Memoized_Error
  (Exc_Id  : Ada.Exceptions.Exception_Id;
   Exc_Msg : String_Access) is
begin
   Ada.Exceptions.Raise_Exception (Exc_Id, Exc_Msg.all & " (memoized)");
end Reraise_Memoized_Error;



   package Solver_Diagnostic_Vectors is new Liblktlang_Support.Vectors
     (Internal_Solver_Diagnostic);

   ----------------------------
   -- Allocate_Logic_Context --
   ----------------------------

   function Allocate_Logic_Context
     (Ctx : Internal_Logic_Context) return Internal_Logic_Context_Access
   is (new Internal_Logic_Context'(Ctx));

   -------------------------
   -- Trace_Logic_Context --
   -------------------------

   function Trace_Logic_Context
     (Ctx : Internal_Logic_Context_Access) return String
   is (Trace_Image (Ctx.all));

   -----------------
   -- Deep_Equals --
   -----------------

   function Deep_Equals
     (X, Y : Internal_Logic_Context_Access) return Boolean
   is (X.all = Y.all);

   ------------------------
   -- Free_Logic_Context --
   ------------------------

   procedure Free_Logic_Context
     (Ctx : in out Internal_Logic_Context_Access)
   is
      procedure Free is new Ada.Unchecked_Deallocation
        (Internal_Logic_Context, Internal_Logic_Context_Access);
   begin
      Free (Ctx);
   end Free_Logic_Context;

   -------------------
   -- Solve_Wrapper --
   -------------------

   function Solve_Wrapper
     (R            : Solver.Relation;
      Context_Node : Bare_Lkt_Node) return Boolean is
   begin
      if Context_Node /= null and then Liblktlang_Support.Adalog.Debug.Debug then
         Assign_Names_To_Logic_Vars (Context_Node);
      end if;

      begin
         return Solver.Solve_First
           (R, Timeout => Context_Node.Unit.Context.Logic_Resolution_Timeout);
      exception
         when Liblktlang_Support.Adalog.Early_Binding_Error =>
            Raise_Property_Exception
              (Context_Node,
               Property_Error'Identity,
               "invalid equation for logic resolution");
         when Liblktlang_Support.Adalog.Timeout_Error =>
            Raise_Property_Exception
              (Context_Node,
               Property_Error'Identity,
               "logic resolution timed out");
      end;
   end Solve_Wrapper;

   ----------------------------
   -- Solve_With_Diagnostics --
   ----------------------------

   function Solve_With_Diagnostics
     (R            : Solver.Relation;
      Context_Node : Bare_Lkt_Node) return Internal_Solver_Result
   is
      Ret : Internal_Solver_Result :=
        (True, No_Internal_Solver_Diagnostic_Array_Type);

      Acc : Solver_Diagnostic_Vectors.Vector;
      --  Vectors that will accumulate diagnostic emitted during resolution

      procedure Emit_Diagnostic (Diag : Internal_Solver_Diagnostic) is
      begin
         Acc.Append (Diag);
      end Emit_Diagnostic;
   begin
      Ret.Success := Solve_Wrapper (R, Context_Node);

      if not Ret.Success then
         Ret.Success := Solver.Solve_First
           (R,
            Solve_Options => (Report_Errors => True),
            Diag_Emitter => Emit_Diagnostic'Unrestricted_Access,
            Timeout => Context_Node.Unit.Context.Logic_Resolution_Timeout);
         Ret.Diagnostics := Create_Internal_Solver_Diagnostic_Array
           (Acc.Length);
         for I in 1 .. Acc.Length loop
            Ret.Diagnostics.Items (I) := Acc.Get (I);
         end loop;
         Acc.Destroy;
      end if;
      return Ret;
   end Solve_With_Diagnostics;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Env : in out Lexical_Env_Access) is
      Mutable_Env : Lexical_Env :=
        (Wrap (Env), 0, Env.Kind, No_Generic_Unit, 0);
   begin
      Destroy (Mutable_Env);
      Env := null;
   end Destroy;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize
     (Self              : Bare_Lkt_Node;
      Kind              : Lkt_Node_Kind_Type;
      Unit              : Internal_Unit;
      Token_Start_Index : Token_Index;
      Token_End_Index   : Token_Index;
      Parent            : Bare_Lkt_Node := null;
      Self_Env          : Lexical_Env := AST_Envs.Empty_Env) is
   begin
      pragma Unreferenced (Kind);
      Self.Parent := Parent;
      Self.Unit := Unit;

      Self.Token_Start_Index := Token_Start_Index;
      Self.Token_End_Index := Token_End_Index;

      Self.Self_Env := Self_Env;
      Self.Last_Attempted_Child := -1;

      

   end Initialize;

   --------------------
   -- Use_Direct_Env --
   --------------------

   procedure Use_Direct_Env (State : in out PLE_Node_State; Env : Lexical_Env)
   is
   begin
      State.Current_Env := Env;
      State.Current_NED := null;
   end Use_Direct_Env;

   -------------------
   -- Use_Named_Env --
   -------------------

   procedure Use_Named_Env
     (State   : in out PLE_Node_State;
      Context : Internal_Context;
      Name    : Symbol_Type) is
   begin
      State.Current_NED := Get_Named_Env_Descriptor (Context, Name);
      State.Current_Env := State.Current_NED.Env_With_Precedence;
   end Use_Named_Env;

   ---------------------
   -- Set_Initial_Env --
   ---------------------

   procedure Set_Initial_Env
     (Self         : Bare_Lkt_Node;
      State        : in out PLE_Node_State;
      Env          : Internal_Designated_Env;
      DSL_Location : String) is
   begin
      case Env.Kind is
         when None =>
            Use_Direct_Env (State, Empty_Env);

         when Current_Env =>
            null;

         when Named_Env =>
            Use_Named_Env (State, Self.Unit.Context, Env.Env_Name);

         when Direct_Env =>

            --  Sanitize this environment value: make sure it's a non-foreign
            --  and primary environment.

            if Env.Direct_Env.Kind /= Static_Primary then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "Cannot set an env that is not static-primary as the"
                  & " initial env");

            elsif Is_Foreign_Strict (Env.Direct_Env, Self) then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "unsound foreign environment in SetInitialEnv ("
                  & DSL_Location & ")");
            end if;
            Use_Direct_Env (State, Env.Direct_Env);
      end case;
   end Set_Initial_Env;

   ----------------
   -- Add_To_Env --
   ----------------

   procedure Add_To_Env
     (Self         : Bare_Lkt_Node;
      State        : PLE_Node_State;
      Key          : Symbol_Type;
      Value        : Bare_Lkt_Node;
      Md           : Internal_Metadata;
      Resolver     : Entity_Resolver;
      Dest_Env     : Internal_Designated_Env;
      DSL_Location : String)
   is
      Context    : constant Internal_Context := Self.Unit.Context;
      Root_Scope : Lexical_Env renames Context.Root_Scope;
      --  Shortcuts

      Actual_Dest_Env : Lexical_Env;
      Dest_NED        : Named_Env_Descriptor_Access;
      --  Description for the destination environment
   begin
      --  Skip the env addition if explicitly requested

      if Key = No_Symbol
         or else Value = null
         or else (case Dest_Env.Kind is
                  when None        => True,
                  when Current_Env => False,
                  when Named_Env   => Dest_Env.Env_Name = No_Symbol,
                  when Direct_Env  => Dest_Env.Direct_Env = Empty_Env)
      then
         return;
      end if;

      if Value.Unit /= Self.Unit then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add_to_env an AST node that comes from another analysis"
            & " unit");
      end if;

      

      --  Then determine the destination environment

      case Dest_Env.Kind is
         when None =>
            raise Program_Error with "unreachable code";

         when Current_Env =>
            --  Just use the current environment
            Dest_NED := State.Current_NED;
            Actual_Dest_Env := State.Current_Env;

         when Named_Env =>
            --  There is an environment name: just lookup the corresponding
            --  NED/env.
            Dest_NED := Get_Named_Env_Descriptor (Context, Dest_Env.Env_Name);
            Actual_Dest_Env := Dest_NED.Env_With_Precedence;

         when Direct_Env =>
            --  There is an explicit destination environment
            Dest_NED := null;
            Actual_Dest_Env := Dest_Env.Direct_Env;
      end case;

      --  Sanitize it

      if Actual_Dest_Env.Kind /= Static_Primary then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add elements to a lexical env that is not static-primary");

      elsif
         --  Since lexical envs need to sort the foreign nodes they contain,
         --  and that the total order on nodes is not defined for synthetic
         --  nodes, it is not possible to add a synthetic node to a foreign
         --  lexical environment.
         --
         --  This reasoning applies to environments that belong to foreign
         --  units, but also to the root environment.
         Is_Foreign (Actual_Dest_Env, Self) and then Is_Synthetic (Value)
      then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "Cannot add a synthetic node to a lexical env from another"
            & " analysis unit");

      elsif
         --  Reject direct references to foreign destination environments.
         --
         --  This is an attempt at identifying uses of the unsound relocation
         --  mechanism (as opposed to named environments), so this applies to
         --  all foreign environments (root scope included).
         DSL_Location'Length > 0
         and then Dest_Env.Kind = Direct_Env
         and then Is_Foreign_Strict (Actual_Dest_Env, Self)
      then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "unsound foreign environment in AddToEnv (" & DSL_Location & ")");
      end if;

      --  Now that everything is sanitized, we can proceed with the actual
      --  key/value pair addition. Note that this does nothing if
      --  Actual_Dest_Env ended up empty.
      Add (Actual_Dest_Env, Thin (Key), Value, Md, Resolver);

      --  If we're adding the element to an environment by env name, we must
      --  register this association in two places: in the target named env
      --  entry, and in Value's unit.
      if Dest_NED /= null then
         declare
            use NED_Assoc_Maps;

            FN    : Map renames Dest_NED.Foreign_Nodes;
            Dummy : Boolean;
            Cur   : Cursor;
         begin
            FN.Insert (Key      => Key,
                       New_Item => Internal_Map_Node_Vectors.Empty_Vector,
                       Position => Cur,
                       Inserted => Dummy);
            declare
               V : Internal_Map_Node_Vectors.Vector renames
                  FN.Reference (Cur);
            begin
               V.Append ((Value, null, Md, Resolver));
               --  Note that the rebindings field is unused by the relocation
               --  mechanism (since we don't even allow adding env entries
               --  with custom rebindings), hence we simply leave it to null.
            end;
         end;
         Value.Unit.Exiled_Entries_In_NED.Append ((Dest_NED, Key, Value));

      --  Otherwise, if we're adding the element to an environment that belongs
      --  to a different unit, or to the root scope, then...
      elsif Is_Foreign_Not_Empty (Actual_Dest_Env, Self) then
         --  Add the Key/Value association to the list of entries contained in
         --  other units, so we can remove them when reparsing Value's unit.
         Value.Unit.Exiled_Entries.Append ((Actual_Dest_Env, Key, Value));

         if Actual_Dest_Env /= Root_Scope then
            --  Add Val to the list of foreign nodes that Actual_Dest_Env's
            --  unit contains, so that when that unit is reparsed, we can call
            --  Add_To_Env again on those nodes.
            Convert_Unit (Actual_Dest_Env.Owner).Foreign_Nodes.Append
              ((Value, Self.Unit));
         end if;
      end if;
   end Add_To_Env;

   -------------
   -- Ref_Env --
   -------------

   procedure Ref_Env
     (Self                : Bare_Lkt_Node;
      Dest_Env            : Lexical_Env;
      Ref_Env_Nodes       : in out Bare_Lkt_Node_Array_Access;
      Resolver            : Lexical_Env_Resolver;
      Kind                : Ref_Kind;
      Cats                : Ref_Categories;
      Shed_Rebindings     : Boolean) is
   begin
      for N of Ref_Env_Nodes.Items loop
         if N /= null then
            if N.Unit /= Self.Unit then
               Raise_Property_Exception
                 (Self,
                  Property_Error'Identity,
                  "attempt to add a referenced environment to a foreign unit");
            end if;
            Reference (Dest_Env, N, Resolver, Kind, Cats, Shed_Rebindings);
         end if;
      end loop;
      Dec_Ref (Ref_Env_Nodes);
   end Ref_Env;

   -------------
   -- Add_Env --
   -------------

   procedure Add_Env
     (Self              : Bare_Lkt_Node;
      State             : in out PLE_Node_State;
      No_Parent         : Boolean;
      Transitive_Parent : Boolean;
      Names             : in out Symbol_Type_Array_Access)
   is
      Parent_From_Name : constant Boolean := State.Current_NED /= null;
      --  Does the parent environment comes from a named environment lookup?

      --  Determine the parent of this new environment:
      --
      --  (1) no parent if requested;
      --  (2) the current environment as the static parent if it comes from a
      --      named env lookup or if it is not foreign (or is the empty/root
      --      environment).
      Parent : constant Lexical_Env :=
        (if No_Parent
         then Null_Lexical_Env
         else State.Current_Env);
   begin
      --  Create the environment itself
      Self.Self_Env := Create_Static_Lexical_Env
        (Parent            => Parent,
         Node              => Self,
         Transitive_Parent => Transitive_Parent,
         Sym_Table         => Self.Unit.Context.Symbols);

      --  If the parent of this new environment comes from a named environment
      --  lookup, register this new environment so that its parent is updated
      --  when the precence for this named environment changes.
      if Parent_From_Name then
         declare
            NED : constant Named_Env_Descriptor_Access := State.Current_NED;
         begin
            Self.Unit.Exiled_Envs.Append ((NED, Self.Self_Env));
            NED.Foreign_Envs.Insert (Self, Self.Self_Env);
         end;
      end if;

      --  From now on, the current environment is Self.Self_Env, with a direct
      --  access to it. It does not go through the env naming scheme, since
      --  only this node and its children (i.e. non-foreign nodes) will access
      --  it as a "current" environment during PLE.
      Use_Direct_Env (State, Self.Self_Env);

      --  Register the environment we just created on all the requested names
      if Names /= null then
         declare
            Context   : constant Internal_Context := Self.Unit.Context;
            Env       : constant Lexical_Env := Self.Self_Env;
            NENU      : NED_Maps.Map renames
               State.Unit_State.Named_Envs_Needing_Update;
         begin
            for N of Names.Items loop
               Register_Named_Env (Context, N, Env, NENU);
            end loop;
            Dec_Ref (Names);
         end;
      end if;
   end Add_Env;

   ---------------------
   -- Pre_Env_Actions --
   ---------------------

   procedure Pre_Env_Actions
     (Self            : Bare_Lkt_Node;
      State           : in out PLE_Node_State;
      Add_To_Env_Only : Boolean := False) is
   begin

      
   

   case Self.Kind is
            when Lkt_Match_Branch =>
            
            Match_Branch_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Decl_Block =>
            
            Decl_Block_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Langkit_Root =>
            
            Langkit_Root_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Import =>
            
            Import_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Lambda_Expr =>
            
            Lambda_Expr_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Block_Expr =>
            
            Block_Expr_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Enum_Type_Decl =>
            
            Enum_Type_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Enum_Class_Decl =>
            
            Enum_Class_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Enum_Class_Alt_Decl =>
            
            null;
      
            when Lkt_Lexer_Decl =>
            
            Lexer_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Grammar_Decl =>
            
            Grammar_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Generic_Decl =>
            
            Generic_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Fun_Decl =>
            
            Fun_Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
            when Lkt_Enum_Lit_Decl =>
            
            null;
      
            when Lkt_Grammar_Rule_Decl .. Lkt_Self_Decl | Lkt_Field_Decl .. Lkt_Val_Decl | Lkt_Env_Spec_Decl | Lkt_Lexer_Family_Decl | Lkt_Function_Type .. Lkt_Class_Decl | Lkt_Struct_Decl .. Lkt_Trait_Decl =>
            
            Decl_Pre_Env_Actions (Self, State, Add_To_Env_Only);
      
      when others =>  null; 
   end case;


   end Pre_Env_Actions;

   ----------------------
   -- Post_Env_Actions --
   ----------------------

   procedure Post_Env_Actions
     (Self : Bare_Lkt_Node; State : in out PLE_Node_State) is
   begin
      
   

   case Self.Kind is
            when Lkt_Match_Branch =>
            
            null;
      
            when Lkt_Decl_Block =>
            
            null;
      
            when Lkt_Langkit_Root =>
            
            null;
      
            when Lkt_Import =>
            
            null;
      
            when Lkt_Lambda_Expr =>
            
            null;
      
            when Lkt_Block_Expr =>
            
            null;
      
            when Lkt_Enum_Type_Decl =>
            
            Enum_Type_Decl_Post_Env_Actions (Self, State);
      
            when Lkt_Enum_Class_Decl =>
            
            Enum_Class_Decl_Post_Env_Actions (Self, State);
      
            when Lkt_Enum_Class_Alt_Decl =>
            
            null;
      
            when Lkt_Lexer_Decl =>
            
            null;
      
            when Lkt_Grammar_Decl =>
            
            null;
      
            when Lkt_Generic_Decl =>
            
            null;
      
            when Lkt_Fun_Decl =>
            
            null;
      
            when Lkt_Enum_Lit_Decl =>
            
            null;
      
            when Lkt_Grammar_Rule_Decl .. Lkt_Self_Decl | Lkt_Field_Decl .. Lkt_Val_Decl | Lkt_Env_Spec_Decl | Lkt_Lexer_Family_Decl | Lkt_Function_Type .. Lkt_Class_Decl | Lkt_Struct_Decl .. Lkt_Trait_Decl =>
            
            null;
      
      when others =>  null; 
   end case;


   end Post_Env_Actions;

   ----------------
   -- Get_Symbol --
   ----------------

   function Get_Symbol
     (Node : Bare_Lkt_Node) return Symbol_Type is
   begin
      if Node = null then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "cannot get the symbol of a null node");
      end if;
      return Get_Symbol (Token (Node, Node.Token_Start_Index));
   end Get_Symbol;

   ----------
   -- Text --
   ----------

   function Text
     (Node : Bare_Lkt_Node) return Text_Type
   is
   begin
      if Node = null then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "cannot get the text of a null node");
      end if;

      declare
         Start_T : constant Token_Reference :=
            Token (Node, Node.Token_Start_Index);
         End_T   : constant Token_Reference :=
            Token (Node, Node.Token_End_Index);
      begin
         --  No text is associated to synthetic and ghost nodes

         if Is_Synthetic (Node) then
            return "";
         end if;

         if Is_Ghost (Node) then
            return "";
         end if;

         return Text (Start_T, End_T);
      end;
   end Text;

   ---------------------
   -- Is_Visible_From --
   ---------------------

   function Is_Visible_From
     (Self                     : Bare_Lkt_Node;
      Referenced_Env, Base_Env : Lexical_Env) return Boolean
   is
      Referenced_Unit : constant Internal_Unit :=
         Convert_Unit (Referenced_Env.Owner);
      Base_Unit       : constant Internal_Unit :=
         Convert_Unit (Base_Env.Owner);
   begin
      if Referenced_Unit = null then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "referenced environment does not belong to any analysis unit");
      elsif Base_Unit = null then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "base environment does not belong to any analysis unit");
      end if;
      return Is_Referenced_From (Referenced_Unit, Base_Unit);
   end Is_Visible_From;

   ----------
   -- Unit --
   ----------

   function Unit (Node : Bare_Lkt_Node) return Internal_Unit is
   begin
      return Node.Unit;
   end Unit;

   function Lookup_Internal
     (Node : Bare_Lkt_Node;
      Sloc : Source_Location) return Bare_Lkt_Node;
   procedure Lookup_Relative
     (Node       : Bare_Lkt_Node;
      Sloc       : Source_Location;
      Position   : out Relative_Position;
      Node_Found : out Bare_Lkt_Node);
   --  Implementation helpers for the looking up process

   -----------------
   -- Set_Parents --
   -----------------

   procedure Set_Parents
     (Node, Parent : Bare_Lkt_Node)
   is
   begin
      if Node = null then
         return;
      end if;

      Node.Parent := Bare_Lkt_Node (Parent);

      for I in 1 .. Children_Count (Node) loop
         Set_Parents (Child (Node, I), Node);
      end loop;
   end Set_Parents;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Node : Bare_Lkt_Node) is
   begin
      if Node = null then
         return;
      end if;

      Free_User_Fields (Node);
      for I in 1 .. Children_Count (Node) loop
         Destroy (Child (Node, I));
      end loop;
   end Destroy;

   -----------
   -- Child --
   -----------

   function Child (Node  : Bare_Lkt_Node;
                   Index : Positive) return Bare_Lkt_Node
   is
      Result          : Bare_Lkt_Node;
      Index_In_Bounds : Boolean;
   begin
      Get_Child (Node, Index, Index_In_Bounds, Result);
      return Result;
   end Child;

   --------------
   -- Traverse --
   --------------

   function Traverse
     (Node  : Bare_Lkt_Node;
      Visit : access function (Node : Bare_Lkt_Node)
              return Visit_Status)
     return Visit_Status
   is
      Status : Visit_Status := Into;

   begin
      if Node /= null then
         Status := Visit (Node);

         --  Skip processing the child nodes if the returned status is Over
         --  or Stop. In the former case the previous call to Visit has taken
         --  care of processing the needed childs, and in the latter case we
         --  must immediately stop processing the tree.

         if Status = Into then
            for I in 1 .. Children_Count (Node) loop
               declare
                  Cur_Child : constant Bare_Lkt_Node :=
                     Child (Node, I);

               begin
                  if Cur_Child /= null then
                     Status := Traverse (Cur_Child, Visit);
                     exit when Status /= Into;
                  end if;
               end;
            end loop;
         end if;
      end if;

      if Status = Stop then
         return Stop;

      --  At this stage the Over status has no sense and we just continue
      --  processing the tree.

      else
         return Into;
      end if;
   end Traverse;

   --------------
   -- Traverse --
   --------------

   procedure Traverse
     (Node  : Bare_Lkt_Node;
      Visit : access function (Node : Bare_Lkt_Node)
                               return Visit_Status)
   is
      Result_Status : Visit_Status;
      pragma Unreferenced (Result_Status);
   begin
      Result_Status := Traverse (Node, Visit);
   end Traverse;

   ------------------------
   -- Traverse_With_Data --
   ------------------------

   function Traverse_With_Data
     (Node  : Bare_Lkt_Node;
      Visit : access function (Node : Bare_Lkt_Node;
                               Data : in out Data_Type)
                               return Visit_Status;
      Data  : in out Data_Type)
      return Visit_Status
   is
      function Helper (Node : Bare_Lkt_Node) return Visit_Status;

      ------------
      -- Helper --
      ------------

      function Helper (Node : Bare_Lkt_Node) return Visit_Status is
      begin
         return Visit (Node, Data);
      end Helper;

      Saved_Data : Data_Type;
      Result     : Visit_Status;

   begin
      if Reset_After_Traversal then
         Saved_Data := Data;
      end if;
      Result := Traverse (Node, Helper'Access);
      if Reset_After_Traversal then
         Data := Saved_Data;
      end if;
      return Result;
   end Traverse_With_Data;

   ----------------
   -- Sloc_Range --
   ----------------

   function Sloc_Range
     (Node : Bare_Lkt_Node) return Source_Location_Range
   is
      type Token_Anchor is (T_Start, T_End);
      type Token_Pos is record
         Pos    : Token_Index;
         Anchor : Token_Anchor;
      end record;

      TDH                    : Token_Data_Handler renames Node.Unit.TDH;
      Token_Start, Token_End : Token_Pos;

      function Get (Index : Token_Index) return Stored_Token_Data is
        (Get_Token (TDH, Index));

      function Sloc (T : Token_Pos) return Source_Location is
        (if T.Anchor = T_Start
         then Sloc_Start (TDH, Get (T.Pos))
         else Sloc_End (TDH, Get (T.Pos)));

   begin
      if Is_Synthetic (Node) then
         return (if Node.Parent = null
                 then No_Source_Location_Range
                 else Sloc_Range (Node.Parent));
      end if;

      if Is_Ghost (Node) then
         Token_Start := (if Node.Token_Start_Index = 1
                         then (1, T_Start)
                         else (Node.Token_Start_Index - 1, T_End));
         Token_End := Token_Start;
      else
         Token_Start := (Node.Token_Start_Index, T_Start);
         Token_End := (Node.Token_End_Index, T_End);
      end if;

      if Snaps_At_Start (Node)
         and then not Is_Ghost (Node)
         and then Token_Start.Pos /= 1
      then
         Token_Start := (Token_Start.Pos - 1, T_End);
      end if;

      if Snaps_At_End (Node) and then Token_End.Pos /= Last_Token (TDH) then
         Token_End := (Token_End.Pos + 1, T_Start);
      end if;

      return Make_Range (Sloc (Token_Start), Sloc (Token_End));
   end Sloc_Range;

   ------------
   -- Lookup --
   ------------

   function Lookup
     (Node : Bare_Lkt_Node;
      Sloc : Source_Location) return Bare_Lkt_Node
   is
      Position : Relative_Position;
      Result   : Bare_Lkt_Node;
   begin
      if Sloc = No_Source_Location then
         return null;
      end if;

      Lookup_Relative
        (Bare_Lkt_Node (Node), Sloc, Position, Result);
      return Result;
   end Lookup;

   ---------------------
   -- Lookup_Internal --
   ---------------------

   function Lookup_Internal
     (Node : Bare_Lkt_Node;
      Sloc : Source_Location) return Bare_Lkt_Node
   is
      --  For this implementation helper (i.e. internal primitive), we can
      --  assume that all lookups fall into this node's sloc range.
      pragma Assert (Compare (Sloc_Range (Node), Sloc) = Inside);

      Children : constant Internal_Bare_Lkt_Node_Array :=
         Implementation.Children (Node);
      Pos      : Relative_Position;
      Result   : Bare_Lkt_Node;
   begin
      --  Look for a child node that contains Sloc (i.e. return the most
      --  precise result).

      for Child of Children loop
         --  Note that we assume here that child nodes are ordered so that the
         --  first one has a sloc range that is before the sloc range of the
         --  second child node, etc.

         if Child /= null then
            Lookup_Relative (Child, Sloc, Pos, Result);
            case Pos is
               when Before =>
                   --  If this is the first node, Sloc is before it, so we can
                   --  stop here.  Otherwise, Sloc is between the previous
                   --  child node and the next one...  so we can stop here,
                   --  too.
                   return Node;

               when Inside =>
                   return Result;

               when After =>
                   --  Sloc is after the current child node, so see with the
                   --  next one.
                   null;
            end case;
         end if;
      end loop;

      --  If we reach this point, we found no children that covers Sloc, but
      --  Node still covers it (see the assertion).
      return Node;
   end Lookup_Internal;

   -------------
   -- Compare --
   -------------

   function Compare
     (Node : Bare_Lkt_Node;
      Sloc : Source_Location) return Relative_Position is
   begin
      return Compare (Sloc_Range (Node), Sloc);
   end Compare;

   ---------------------
   -- Lookup_Relative --
   ---------------------

   procedure Lookup_Relative
     (Node       : Bare_Lkt_Node;
      Sloc       : Source_Location;
      Position   : out Relative_Position;
      Node_Found : out Bare_Lkt_Node)
   is
      Result : constant Relative_Position :=
        Compare (Node, Sloc);
   begin
      Position := Result;
      Node_Found := (if Result = Inside
                     then Lookup_Internal (Node, Sloc)
                     else null);
   end Lookup_Relative;

   -------------
   -- Compare --
   -------------

   function Compare
     (Self, Left, Right : Bare_Lkt_Node;
      Relation          : Comparison_Relation) return Boolean
   is
      LS, RS : Source_Location;
   begin
      if Left = null or else Right = null or else Left.Unit /= Right.Unit then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "invalid node comparison");
      end if;

      LS := Start_Sloc (Sloc_Range (Left));
      RS := Start_Sloc (Sloc_Range (Right));
      return (case Relation is
              when Liblktlang_Support.Types.Less_Than        => LS < RS,
              when Liblktlang_Support.Types.Less_Or_Equal    => LS <= RS,
              when Liblktlang_Support.Types.Greater_Than     => LS > RS,
              when Liblktlang_Support.Types.Greater_Or_Equal => LS >= RS);
   end Compare;

   --------------
   -- Children --
   --------------

   function Children
     (Node : Bare_Lkt_Node) return Internal_Bare_Lkt_Node_Array
   is
      First : constant Integer := Bare_Lkt_Node_Vectors.Index_Type'First;
      Last  : constant Integer := First + Children_Count (Node) - 1;
   begin
      return A : Internal_Bare_Lkt_Node_Array (First .. Last)
      do
         for I in First .. Last loop
            A (I) := Child (Node, I);
         end loop;
      end return;
   end Children;

   function Children
     (Node : Bare_Lkt_Node) return Bare_Lkt_Node_Array_Access
   is
      C : Internal_Bare_Lkt_Node_Array := Children (Node);
   begin
      return Ret : Bare_Lkt_Node_Array_Access :=
         Create_Bare_Lkt_Node_Array (C'Length)
      do
         Ret.Items := C;
      end return;
   end Children;

   ---------
   -- Get --
   ---------

   function Get
     (Self    : Bare_Lkt_Node;
      Node    : Bare_Lkt_Node_Base_List;
      Index   : Integer;
      Or_Null : Boolean := False) return Bare_Lkt_Node
   is
      function Length (Node : Bare_Lkt_Node_Base_List) return Natural
      is (Node.Count);
      --  Wrapper around the Length primitive to get the compiler happy for the
      --  the package instantiation below.

      function Absolute_Get
        (L     : Bare_Lkt_Node_Base_List;
         Index : Integer) return Bare_Lkt_Node
      is (L.Nodes.all (Index + 1));
      --  L.Nodes is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Bare_Lkt_Node,
         Sequence_Type => Bare_Lkt_Node_Base_List,
         Length        => Length,
         Get           => Absolute_Get);

      Result : Bare_Lkt_Node;
   begin
      if Node = null and then Or_Null then
         return null;
      elsif Relative_Get (Node, Index, Result) then
         return Result;
      elsif Or_Null then
         return null;
      else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out-of-bounds AST list access");
      end if;
   end Get;

   ---------------
   -- PP_Trivia --
   ---------------

   procedure PP_Trivia
     (Node        : Bare_Lkt_Node;
      Line_Prefix : String := "")
   is
      Children_Prefix : constant String := Line_Prefix & "|  ";
   begin
      if Node = null then
         Put_Line (Line_Prefix & "None");
         return;
      end if;
      Put_Line (Line_Prefix & Kind_Name (Node));
      for C of Children_And_Trivia (Node) loop
         case C.Kind is
            when Trivia =>
               Put_Line (Children_Prefix & Image (Text (C.Trivia)));
            when Child =>
               PP_Trivia (C.Node, Children_Prefix);
         end case;
      end loop;
   end PP_Trivia;

   --------------------------
   -- Populate_Lexical_Env --
   --------------------------

   function Populate_Lexical_Env (Node : Bare_Lkt_Node) return Boolean is

      Context    : constant Internal_Context := Node.Unit.Context;
      Unit_State : aliased PLE_Unit_State := (Named_Envs_Needing_Update => <>);
      Root_State : constant PLE_Node_State :=
        (Unit_State  => Unit_State'Unchecked_Access,
         Current_Env => Context.Root_Scope,
         Current_NED => null);

      function Populate_Internal
        (Node         : Bare_Lkt_Node;
         Parent_State : PLE_Node_State) return Boolean;
      --  Do the lexical env population on Node and recurse on its children

      procedure Register_Foreign_Env
        (Node : Bare_Lkt_Node; State : PLE_Node_State);
      --  Given a node and its PLE state, register Node.Self_Env as being
      --  initialized through the named environment mechanism, if that's indeed
      --  the case. Do nothing otherwise.

      -----------------------
      -- Populate_Internal --
      -----------------------

      function Populate_Internal
        (Node         : Bare_Lkt_Node;
         Parent_State : PLE_Node_State) return Boolean
      is
         Result : Boolean := False;
         State  : PLE_Node_State := Parent_State;
      begin
         if Node = null then
            return Result;
         end if;

         --  By default (i.e. unless env actions add a new env), the
         --  environment we store in Node is the current one.
         Node.Self_Env := State.Current_Env;

         --  Run pre/post actions, and run PLE on children in between. Make
         --  sure we register the potential foreign Node.Self_Env environment
         --  at the end, even when an exception interrupts PLE to keep the
         --  state consistent.
         begin
            Pre_Env_Actions (Node, State);
            if State.Current_Env /= Null_Lexical_Env then
               Node.Self_Env := State.Current_Env;
               Register_Foreign_Env (Node, State);
            end if;

            --  Call recursively on children
            for I in First_Child_Index (Node) .. Last_Child_Index (Node) loop
               Result := Populate_Internal
                 (Child (Node, I), State) or else Result;
            end loop;

            Post_Env_Actions (Node, State);
         exception
            when Exc : Property_Error =>
               if PLE_Errors_Trace.Is_Active then
                   GNATCOLL.Traces.Trace
                     (PLE_Errors_Trace,
                      "Exception raised during PLE "
                      & Ada.Exceptions.Exception_Name (Exc) & " : "
                      & Ada.Exceptions.Exception_Message (Exc));
                   GNATCOLL.Traces.Trace
                     (PLE_Errors_Trace,
                      GNAT.Traceback.Symbolic.Symbolic_Traceback (Exc));
               end if;
               Register_Foreign_Env (Node, State);
               return True;
         end;

         return Result;
      end Populate_Internal;

      --------------------------
      -- Register_Foreign_Env --
      --------------------------

      procedure Register_Foreign_Env
        (Node : Bare_Lkt_Node; State : PLE_Node_State) is
      begin
         if State.Current_NED /= null then
            State.Current_NED.Nodes_With_Foreign_Env.Insert (Node);
            Node.Unit.Nodes_With_Foreign_Env.Insert (Node, State.Current_NED);
         end if;
      end Register_Foreign_Env;

   begin
      --  This is intended to be called on the root node only (when there is no
      --  PLE root) or on a PLE root (child of the root node with a specific
      --  kind).
      if
         Node.Parent /= null
      then
         raise Program_Error;
      end if;

      return Result : constant Boolean :=
         Populate_Internal (Node, Root_State)
      do
         Update_Named_Envs (Context, Unit_State.Named_Envs_Needing_Update);
      end return;
   end Populate_Lexical_Env;

   ------------------------------
   -- AST_Envs_Node_Text_Image --
   ------------------------------

   function AST_Envs_Node_Text_Image
     (Node  : Bare_Lkt_Node;
      Short : Boolean := True) return Text_Type is
   begin
      if Short then
         if Node = null then
            return "null";
         end if;
         return To_Text (Basename (Node.Unit))
           & ":" & To_Text (Image (Start_Sloc (Sloc_Range (Node))));
      else
         return Short_Text_Image (Node);
      end if;
   end AST_Envs_Node_Text_Image;

   -------------------
   -- Is_Rebindable --
   -------------------

   function Is_Rebindable (Node : Bare_Lkt_Node) return Boolean is
   begin
      
         return Node.Kind in Lkt_Generic_Decl;
   end Is_Rebindable;

   -----------------------
   -- Acquire_Rebinding --
   -----------------------

   function Acquire_Rebinding
     (Node             : Bare_Lkt_Node;
      Parent           : Env_Rebindings;
      Old_Env, New_Env : Lexical_Env) return Env_Rebindings
   is
      Result    : Env_Rebindings;
      Available : Env_Rebindings_Vectors.Vector renames
         Node.Unit.Context.Available_Rebindings;
   begin
      --  Use an existing and available Env_Rebindings_Type record for Node's
      --  Context, otherwise allocate a new rebinding.
      Result := (if Available.Is_Empty
                 then new Env_Rebindings_Type'(Version => 0, others => <>)
                 else Available.Pop);

      Result.Parent := Parent;
      Result.Old_Env := Old_Env;
      Result.New_Env := New_Env;
      Result.Children := Env_Rebindings_Vectors.Empty_Vector;
      return Result;
   end Acquire_Rebinding;

   -----------------------
   -- Release_Rebinding --
   -----------------------

   procedure Release_Rebinding (Self : in out Env_Rebindings) is
      Available : Env_Rebindings_Vectors.Vector renames
         Unwrap (Self.Old_Env).Node.Unit.Context.Available_Rebindings;
   begin
      --  Bumping the version number, to invalidate existing references to
      --  Self.
      Self.Version := Self.Version + 1;

      Self.Children.Destroy;
      Available.Append (Self);
      Self := null;
   end Release_Rebinding;

   ------------------------
   -- Register_Rebinding --
   ------------------------

   procedure Register_Rebinding
     (Node : Bare_Lkt_Node; Rebinding : Env_Rebindings) is
   begin
      Node.Unit.Rebindings.Append (Rebinding);
   end Register_Rebinding;


   --------------------
   -- Element_Parent --
   --------------------

   function Element_Parent
     (Node : Bare_Lkt_Node) return Bare_Lkt_Node
   is (Node.Parent);

   ---------------
   -- Node_Unit --
   ---------------

   function Node_Unit (Node : Bare_Lkt_Node) return Generic_Unit_Ptr is
   begin
      return Convert_Unit (Node.Unit);
   end Node_Unit;

   ----------
   -- Hash --
   ----------

   function Hash (Node : Bare_Lkt_Node) return Hash_Type
   is
      function H is new Hash_Access
        (Root_Node_Record, Bare_Lkt_Node);
   begin
      return H (Node);
   end Hash;

      function Hash (B : Boolean) return Hash_Type is (Boolean'Pos (B));





   ------------------------
   -- Named environments --
   ------------------------

   ---------
   -- Add --
   ---------

   procedure Add
     (Self : in out NED_Assoc_Maps.Map;
      Key  : Symbol_Type;
      Node : AST_Envs.Internal_Map_Node)
   is
      use NED_Assoc_Maps;

      Pos   : Cursor;
      Dummy : Boolean;
   begin
      --  Make sure there is a vector entry for Key
      Self.Insert (Key, Internal_Map_Node_Vectors.Empty_Vector, Pos, Dummy);

      --  Append Node to that vector
      declare
         V : Internal_Map_Node_Vectors.Vector renames Self.Reference (Pos);
      begin
         V.Append (Node);
      end;
   end Add;

   ------------
   -- Remove --
   ------------

   procedure Remove
     (Self : in out NED_Assoc_Maps.Map;
      Key  : Symbol_Type;
      Node : Bare_Lkt_Node)
   is
      use NED_Assoc_Maps;

      V : Internal_Map_Node_Vectors.Vector renames Self.Reference (Key);
   begin
      --  Remove the (assumed unique) entry in V whose node is Node. The order
      --  of items in V is not significant, so we can use Pop for efficient
      --  removal. Do the traversal in reverse order for correctness.
      for I in reverse 1 .. V.Length loop
         if V.Get_Access (I).Node = Node then
            V.Pop (I);
            exit;
         end if;
      end loop;
   end Remove;

   ------------------------------
   -- Get_Named_Env_Descriptor --
   ------------------------------

   function Get_Named_Env_Descriptor
     (Context : Internal_Context;
      Name    : Symbol_Type) return Named_Env_Descriptor_Access
   is
      use NED_Maps;

      --  Look for an existing entry for Name
      Pos : constant Cursor := Context.Named_Envs.Find (Name);
   begin
      if Has_Element (Pos) then
         return Element (Pos);
      end if;

      --  There is no such entry: create one
      return Result : constant Named_Env_Descriptor_Access :=
         new Named_Env_Descriptor'
           (Name                   => Name,
            Envs                   => <>,
            Env_With_Precedence    => Empty_Env,
            Foreign_Nodes          => <>,
            Foreign_Envs           => <>,
            Nodes_With_Foreign_Env => <>)
      do
         Context.Named_Envs.Insert (Name, Result);
      end return;
   end Get_Named_Env_Descriptor;

   ------------------------
   -- Register_Named_Env --
   ------------------------

   procedure Register_Named_Env
     (Context                   : Internal_Context;
      Name                      : Symbol_Type;
      Env                       : Lexical_Env;
      Named_Envs_Needing_Update : in out NED_Maps.Map)
   is
      NED_Access : constant Named_Env_Descriptor_Access :=
         Get_Named_Env_Descriptor (Context, Name);
      NED        : Named_Env_Descriptor renames NED_Access.all;
      Node       : constant Bare_Lkt_Node := Env_Node (Env);
   begin
      NED.Envs.Insert (Node, Env);
      Node.Unit.Named_Envs.Append ((Name, Env));

      --  If that insertion must change the env that has precedence, signal
      --  that NED requires an update.

      if NED.Envs.First_Element /= NED.Env_With_Precedence then
         Named_Envs_Needing_Update.Include (Name, NED_Access);
      end if;
   end Register_Named_Env;

   ----------------------
   -- Update_Named_Env --
   ----------------------

   procedure Update_Named_Envs
     (Context : Internal_Context; Named_Envs : NED_Maps.Map)
   is
      Require_Cache_Reset : Boolean := False;
   begin
      for Cur in Named_Envs.Iterate loop
         declare
            NE      : Named_Env_Descriptor renames NED_Maps.Element (Cur).all;
            New_Env : constant Lexical_Env :=
              (if NE.Envs.Is_Empty
               then Empty_Env
               else NE.Envs.First_Element);
         begin
            --  If there was an environment with precedence, remove its foreign
            --  nodes.
            if NE.Env_With_Precedence /= Empty_Env then
               for Cur in NE.Foreign_Nodes.Iterate loop
                  declare
                     Key   : constant Symbol_Type :=
                        NED_Assoc_Maps.Key (Cur);
                     Nodes : Internal_Map_Node_Vectors.Vector renames
                        NE.Foreign_Nodes.Reference (Cur);
                  begin
                     for N of Nodes loop
                        Remove (NE.Env_With_Precedence, Thin (Key), N.Node);
                     end loop;
                  end;
               end loop;
            end if;

            --  Now, set the new environment that has precedence
            NE.Env_With_Precedence := New_Env;

            --  Add the foreign nodes to the new environment with precedence,
            --  if any.
            for Cur in NE.Foreign_Nodes.Iterate loop
               declare
                  Key   : constant Symbol_Type :=
                     NED_Assoc_Maps.Key (Cur);
                  Nodes : Internal_Map_Node_Vectors.Vector renames
                     NE.Foreign_Nodes.Reference (Cur);
               begin
                  for N of Nodes loop
                     Add (New_Env, Thin (Key), N.Node, N.Md, N.Resolver);
                  end loop;
               end;
            end loop;

            --  Set the parent environment of all foreign environments
            for Cur in NE.Foreign_Envs.Iterate loop
               declare
                  Env : Lexical_Env_Record renames
                     Unwrap (Sorted_Env_Maps.Element (Cur)).all;
               begin
                  Env.Parent := New_Env;

                  --  We have updated the lexical env hierarchy (e.g. an env
                  --  which had no parent may have one now), so the cached
                  --  entries for queries that traveresed the old env hierarchy
                  --  need to be invalidated.
                  Require_Cache_Reset := True;
               end;
            end loop;

            --  Update nodes whose environment was the old env with precedence
            for N of NE.Nodes_With_Foreign_Env loop
               N.Self_Env := New_Env;
            end loop;
         end;
      end loop;
      if Require_Cache_Reset then
         Invalidate_Caches (Context, Invalidate_Envs => True);
      end if;
   end Update_Named_Envs;

   --------------------------
   -- Big integers wrapper --
   --------------------------

   ------------------------
   -- Create_Big_Integer --
   ------------------------

   function Create_Big_Integer
     (Image : String; Base : Integer := 10) return Big_Integer_Type
   is
      use GNATCOLL.GMP;
      use GNATCOLL.GMP.Integers;
   begin
      return new Big_Integer_Record'(Value     => Make (Image, Int (Base)),
                                     Ref_Count => 1);
   end Create_Big_Integer;

   ------------------------
   -- Create_Big_Integer --
   ------------------------

   function Create_Big_Integer
     (Big_Int : GNATCOLL.GMP.Integers.Big_Integer) return Big_Integer_Type
   is
      Result : constant Big_Integer_Type :=
         new Big_Integer_Record'(Value     => <>,
                                 Ref_Count => 1);
   begin
      Result.Value.Set (Big_Int);
      return Result;
   end Create_Big_Integer;

   ------------------------
   -- Create_Big_Integer --
   ------------------------

   function Create_Big_Integer (Int : Integer) return Big_Integer_Type is
      Result : constant Big_Integer_Type :=
         new Big_Integer_Record'(Value     => <>,
                                 Ref_Count => 1);
   begin
      Result.Value.Set (GNATCOLL.GMP.Long (Int));
      return Result;
   end Create_Big_Integer;

   -------------------------------
   -- Create_Public_Big_Integer --
   -------------------------------

   function Create_Public_Big_Integer
     (Big_Int : Big_Integer_Type) return GNATCOLL.GMP.Integers.Big_Integer is
   begin
      return Result : GNATCOLL.GMP.Integers.Big_Integer do
         Result.Set (Big_Int.Value);
      end return;
   end Create_Public_Big_Integer;

   -----------------
   -- Trace_Image --
   -----------------

   function Trace_Image (I : Big_Integer_Type) return String is
   begin
      return GNATCOLL.GMP.Integers.Image (I.Value);
   end Trace_Image;

   ----------------
   -- To_Integer --
   ----------------

   function To_Integer
     (Self    : Bare_Lkt_Node;
      Big_Int : Big_Integer_Type) return Integer
   is
      Image : constant String := Big_Int.Value.Image;
   begin
      return Integer'Value (Image);
   exception
      when Constraint_Error =>
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out of range big integer");
   end To_Integer;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (Big_Int : Big_Integer_Type) is
   begin
      if Big_Int.Ref_Count /= -1 then
         Big_Int.Ref_Count := Big_Int.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Big_Int : in out Big_Integer_Type) is
      procedure Destroy is new Ada.Unchecked_Deallocation
        (Big_Integer_Record, Big_Integer_Type);
   begin
      if Big_Int = null or else Big_Int.Ref_Count = -1 then
         return;
      end if;

      Big_Int.Ref_Count := Big_Int.Ref_Count - 1;
      if Big_Int.Ref_Count = 0 then
         Destroy (Big_Int);
      end if;
   end Dec_Ref;

   ----------------
   -- Equivalent --
   ----------------

   function Equivalent (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value = Right.Value;
   end Equivalent;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value < Right.Value;
   end "<";

   ----------
   -- "<=" --
   ----------

   function "<=" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value <= Right.Value;
   end "<=";

   ---------
   -- ">" --
   ---------

   function ">" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value > Right.Value;
   end ">";

   ----------
   -- ">=" --
   ----------

   function ">=" (Left, Right : Big_Integer_Type) return Boolean is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Left.Value >= Right.Value;
   end ">=";

   ---------
   -- "+" --
   ---------

   function "+" (Left, Right : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (Left.Value + Right.Value);
   end "+";

   ---------
   -- "-" --
   ---------

   function "-" (Left, Right : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (Left.Value - Right.Value);
   end "-";

   ---------
   -- "-" --
   ---------

   function "-" (Value : Big_Integer_Type) return Big_Integer_Type is
      use type GNATCOLL.GMP.Integers.Big_Integer;
   begin
      return Create_Big_Integer (-Value.Value);
   end "-";

   ------------------
   -- Unit_Version --
   ------------------

   function Unit_Version (Unit : Generic_Unit_Ptr) return Version_Number is
   begin
      return Convert_Unit (Unit).Unit_Version;
   end Unit_Version;

   -------------------------
   -- Get_Context_Version --
   -------------------------

   function Get_Context_Version
     (Node : Bare_Lkt_Node) return Version_Number is
   begin
      return Node.Unit.Context.Cache_Version;
   end Get_Context_Version;

   ---------------
   --  Self_Env --
   ---------------

   function Self_Env (Node : Bare_Lkt_Node) return Lexical_Env is
   begin
      return Node.Self_Env;
   end Self_Env;

   --------------------------
   -- Properties_May_Raise --
   --------------------------

   function Properties_May_Raise
     (Exc : Ada.Exceptions.Exception_Occurrence) return Boolean is
   begin
      return Ada.Exceptions.Exception_Identity (Exc) in
            Property_Error'Identity
      ;
   end Properties_May_Raise;

   ----------------------
   -- Short_Text_Image --
   ----------------------

   function Short_Text_Image (Self : Bare_Lkt_Node) return Text_Type
   is
   begin
      if Self = null then
         return "None";
      end if;

      
   

   case Self.Kind is
            when Lkt_Ref_Id =>
            
         return Ref_Id_Short_Image (Self);
      
            when Lkt_Grammar_Rule_Decl .. Lkt_Trait_Decl =>
            
         return Decl_Short_Image (Self);
      
      when others => 
         return "<" & To_Text (Kind_Name (Self))
                & " "
                & To_Text
                  (Ada.Directories.Simple_Name
                     (Get_Filename (Unit (Self))))
                & ":" & To_Text (Image (Sloc_Range (Self))) & ">";
      
   end case;

   end Short_Text_Image;

   --------------------
   -- Snaps_At_Start --
   --------------------

   function Snaps_At_Start (Self : Bare_Lkt_Node) return Boolean is
   begin
      
   

   case Self.Kind is
      when others => 
         return False;
      
   end case;

   end Snaps_At_Start;

   ------------------
   -- Snaps_At_End --
   ------------------

   function Snaps_At_End (Self : Bare_Lkt_Node) return Boolean is
   begin
      
   

   case Self.Kind is
      when others => 
         return Is_Incomplete (Self);
      
   end case;

   end Snaps_At_End;

   -------------
   -- Parents --
   -------------

   function Parents
     (Node      : Bare_Lkt_Node;
      With_Self : Boolean := True)
      return Bare_Lkt_Node_Array_Access
   is
      Count : Natural := 0;
      Start : Bare_Lkt_Node :=
        (if With_Self then Node else Node.Parent);
      Cur   : Bare_Lkt_Node := Start;
   begin
      while Cur /= null loop
         Count := Count + 1;
         Cur := Cur.Parent;
      end loop;

      declare
         Result : constant Bare_Lkt_Node_Array_Access :=
            Create_Bare_Lkt_Node_Array (Count);
      begin
         Cur := Start;
         for I in Result.Items'Range loop
            Result.Items (I) := Cur;
            Cur := Cur.Parent;
         end loop;
         return Result;
      end;
   end Parents;

   -----------------------
   -- First_Child_Index --
   -----------------------

   function First_Child_Index (Node : Bare_Lkt_Node) return Natural
   is (1);

   ----------------------
   -- Last_Child_Index --
   ----------------------

   function Last_Child_Index (Node : Bare_Lkt_Node) return Natural
   is (Children_Count (Node));

   ---------------
   -- Get_Child --
   ---------------

   procedure Get_Child
     (Node            : Bare_Lkt_Node;
      Index           : Positive;
      Index_In_Bounds : out Boolean;
      Result          : out Bare_Lkt_Node)
   is
      K : constant Lkt_Node_Kind_Type := Node.Kind;
   begin
      

      Index_In_Bounds := True;
      Result := null;
      case Lkt_Lkt_Node (K) is
when Lkt_Lexer_Case_Rule_Cond_Alt_Range =>
declare
N_Bare_Lexer_Case_Rule_Cond_Alt : constant Bare_Lexer_Case_Rule_Cond_Alt := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Case_Rule_Cond_Alt.Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Case_Rule_Cond_Alt.Lexer_Case_Rule_Cond_Alt_F_Send;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Case_Rule_Default_Alt_Range =>
declare
N_Bare_Lexer_Case_Rule_Default_Alt : constant Bare_Lexer_Case_Rule_Default_Alt := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Case_Rule_Default_Alt.Lexer_Case_Rule_Default_Alt_F_Send;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Rule_Decl_Range =>
declare
N_Bare_Grammar_Rule_Decl : constant Bare_Grammar_Rule_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Rule_Decl.Grammar_Rule_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_Rule_Decl.Grammar_Rule_Decl_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Enum_Lit_Decl_Range =>
declare
N_Bare_Enum_Lit_Decl : constant Bare_Enum_Lit_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Enum_Lit_Decl.Enum_Lit_Decl_F_Syn_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Field_Decl_Range =>
declare
N_Bare_Field_Decl : constant Bare_Field_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Field_Decl.Field_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Field_Decl.Field_Decl_F_Decl_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Field_Decl.Field_Decl_F_Default_Val;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Fun_Arg_Decl_Range =>
declare
N_Bare_Fun_Arg_Decl : constant Bare_Fun_Arg_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Fun_Arg_Decl.Fun_Arg_Decl_F_Decl_Annotations;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Fun_Arg_Decl.Fun_Arg_Decl_F_Syn_Name;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Fun_Arg_Decl.Fun_Arg_Decl_F_Decl_Type;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Fun_Arg_Decl.Fun_Arg_Decl_F_Default_Val;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lambda_Arg_Decl_Range =>
declare
N_Bare_Lambda_Arg_Decl : constant Bare_Lambda_Arg_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lambda_Arg_Decl.Lambda_Arg_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lambda_Arg_Decl.Lambda_Arg_Decl_F_Decl_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Lambda_Arg_Decl.Lambda_Arg_Decl_F_Default_Val;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Dyn_Var_Decl_Range =>
declare
N_Bare_Dyn_Var_Decl : constant Bare_Dyn_Var_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Dyn_Var_Decl.Dyn_Var_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Dyn_Var_Decl.Dyn_Var_Decl_F_Decl_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Match_Val_Decl_Range =>
declare
N_Bare_Match_Val_Decl : constant Bare_Match_Val_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Match_Val_Decl.Match_Val_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Match_Val_Decl.Match_Val_Decl_F_Decl_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Val_Decl_Range =>
declare
N_Bare_Val_Decl : constant Bare_Val_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Val_Decl.Val_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Val_Decl.Val_Decl_F_Decl_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Val_Decl.Val_Decl_F_Val;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Fun_Decl_Range =>
declare
N_Bare_Fun_Decl : constant Bare_Fun_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Args;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Return_Type;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Fun_Decl.Fun_Decl_F_Body;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Env_Spec_Decl_Range =>
declare
N_Bare_Env_Spec_Decl : constant Bare_Env_Spec_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Env_Spec_Decl.Env_Spec_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Env_Spec_Decl.Env_Spec_Decl_F_Actions;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Generic_Decl_Range =>
declare
N_Bare_Generic_Decl : constant Bare_Generic_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Generic_Decl.Generic_Decl_F_Generic_Formal_Decls;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Generic_Decl.Generic_Decl_F_Decl;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Decl_Range =>
declare
N_Bare_Grammar_Decl : constant Bare_Grammar_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Decl.Grammar_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_Decl.Grammar_Decl_F_Rules;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Decl_Range =>
declare
N_Bare_Lexer_Decl : constant Bare_Lexer_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Decl.Lexer_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Decl.Lexer_Decl_F_Rules;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Family_Decl_Range =>
declare
N_Bare_Lexer_Family_Decl : constant Bare_Lexer_Family_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Family_Decl.Lexer_Family_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Family_Decl.Lexer_Family_Decl_F_Rules;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Enum_Class_Alt_Decl_Range =>
declare
N_Bare_Enum_Class_Alt_Decl : constant Bare_Enum_Class_Alt_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Enum_Class_Alt_Decl.Enum_Class_Alt_Decl_F_Syn_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Generic_Formal_Type_Decl_Range =>
declare
N_Bare_Generic_Formal_Type_Decl : constant Bare_Generic_Formal_Type_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Generic_Formal_Type_Decl.Generic_Formal_Type_Decl_F_Has_Class;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Generic_Formal_Type_Decl.Generic_Formal_Type_Decl_F_Syn_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Basic_Class_Decl =>
declare
N_Bare_Basic_Class_Decl : constant Bare_Basic_Class_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Basic_Class_Decl.Basic_Class_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Basic_Class_Decl.Basic_Class_Decl_F_Syn_Base_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Basic_Class_Decl.Basic_Class_Decl_F_Traits;
                            return;
                    

                        when others => null;
                    end case;
                
case Lkt_Basic_Class_Decl (K) is
when Lkt_Class_Decl_Range =>
declare
N_Bare_Class_Decl : constant Bare_Class_Decl := N_Bare_Basic_Class_Decl;
begin
case Index is

                        when 4 =>
                            Result := N_Bare_Class_Decl.Class_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Enum_Class_Decl_Range =>
declare
N_Bare_Enum_Class_Decl : constant Bare_Enum_Class_Decl := N_Bare_Basic_Class_Decl;
begin
case Index is

                        when 4 =>
                            Result := N_Bare_Enum_Class_Decl.Enum_Class_Decl_F_Branches;
                            return;
                    

                        when 5 =>
                            Result := N_Bare_Enum_Class_Decl.Enum_Class_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when others => null;
end case;
end;
when Lkt_Enum_Type_Decl_Range =>
declare
N_Bare_Enum_Type_Decl : constant Bare_Enum_Type_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Enum_Type_Decl.Enum_Type_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Enum_Type_Decl.Enum_Type_Decl_F_Traits;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Enum_Type_Decl.Enum_Type_Decl_F_Literals;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Enum_Type_Decl.Enum_Type_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Struct_Decl_Range =>
declare
N_Bare_Struct_Decl : constant Bare_Struct_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Struct_Decl.Struct_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Struct_Decl.Struct_Decl_F_Traits;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Struct_Decl.Struct_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Trait_Decl_Range =>
declare
N_Bare_Trait_Decl : constant Bare_Trait_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Trait_Decl.Trait_Decl_F_Syn_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Trait_Decl.Trait_Decl_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Decl_Annotation_Range =>
declare
N_Bare_Decl_Annotation : constant Bare_Decl_Annotation := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Decl_Annotation.Decl_Annotation_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Decl_Annotation.Decl_Annotation_F_Params;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Decl_Annotation_Params_Range =>
declare
N_Bare_Decl_Annotation_Params : constant Bare_Decl_Annotation_Params := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Decl_Annotation_Params.Decl_Annotation_Params_F_Params;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Elsif_Branch_Range =>
declare
N_Bare_Elsif_Branch : constant Bare_Elsif_Branch := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Elsif_Branch.Elsif_Branch_F_Cond_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Elsif_Branch.Elsif_Branch_F_Then_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Enum_Class_Case_Range =>
declare
N_Bare_Enum_Class_Case : constant Bare_Enum_Class_Case := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Enum_Class_Case.Enum_Class_Case_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Any_Of_Range =>
declare
N_Bare_Any_Of : constant Bare_Any_Of := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Any_Of.Any_Of_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Any_Of.Any_Of_F_Values;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Array_Literal_Range =>
declare
N_Bare_Array_Literal : constant Bare_Array_Literal := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Array_Literal.Array_Literal_F_Exprs;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Array_Literal.Array_Literal_F_Element_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Base_Dot_Expr =>
declare
N_Bare_Base_Dot_Expr : constant Bare_Base_Dot_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Base_Dot_Expr.Base_Dot_Expr_F_Prefix;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Base_Dot_Expr.Base_Dot_Expr_F_Suffix;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Bin_Op_Range =>
declare
N_Bare_Bin_Op : constant Bare_Bin_Op := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Left;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Op;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Bin_Op.Bin_Op_F_Right;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Block_Expr_Range =>
declare
N_Bare_Block_Expr : constant Bare_Block_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Block_Expr.Block_Expr_F_Val_Defs;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Block_Expr.Block_Expr_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Call_Expr_Range =>
declare
N_Bare_Call_Expr : constant Bare_Call_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Call_Expr.Call_Expr_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Call_Expr.Call_Expr_F_Args;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Cast_Expr_Range =>
declare
N_Bare_Cast_Expr : constant Bare_Cast_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Cast_Expr.Cast_Expr_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Cast_Expr.Cast_Expr_F_Excludes_Null;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Cast_Expr.Cast_Expr_F_Dest_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Error_On_Null_Range =>
declare
N_Bare_Error_On_Null : constant Bare_Error_On_Null := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Error_On_Null.Error_On_Null_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Generic_Instantiation_Range =>
declare
N_Bare_Generic_Instantiation : constant Bare_Generic_Instantiation := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Generic_Instantiation.Generic_Instantiation_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Generic_Instantiation.Generic_Instantiation_F_Args;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Discard_Range =>
declare
N_Bare_Grammar_Discard : constant Bare_Grammar_Discard := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Discard.Grammar_Discard_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Dont_Skip_Range =>
declare
N_Bare_Grammar_Dont_Skip : constant Bare_Grammar_Dont_Skip := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Dont_Skip.Grammar_Dont_Skip_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_Dont_Skip.Grammar_Dont_Skip_F_Dont_Skip;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_List_Range =>
declare
N_Bare_Grammar_List : constant Bare_Grammar_List := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_List.Grammar_List_F_List_Type;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_List.Grammar_List_F_Kind;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Grammar_List.Grammar_List_F_Expr;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_Grammar_List.Grammar_List_F_Sep;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Null_Range =>
declare
N_Bare_Grammar_Null : constant Bare_Grammar_Null := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Null.Grammar_Null_F_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Opt_Range =>
declare
N_Bare_Grammar_Opt : constant Bare_Grammar_Opt := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Opt.Grammar_Opt_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Opt_Error_Range =>
declare
N_Bare_Grammar_Opt_Error : constant Bare_Grammar_Opt_Error := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Opt_Error.Grammar_Opt_Error_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Opt_Error_Group_Range =>
declare
N_Bare_Grammar_Opt_Error_Group : constant Bare_Grammar_Opt_Error_Group := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Opt_Error_Group.Grammar_Opt_Error_Group_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Opt_Group_Range =>
declare
N_Bare_Grammar_Opt_Group : constant Bare_Grammar_Opt_Group := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Opt_Group.Grammar_Opt_Group_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Or_Expr_Range =>
declare
N_Bare_Grammar_Or_Expr : constant Bare_Grammar_Or_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Or_Expr.Grammar_Or_Expr_F_Sub_Exprs;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Pick_Range =>
declare
N_Bare_Grammar_Pick : constant Bare_Grammar_Pick := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Pick.Grammar_Pick_F_Exprs;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Predicate_Range =>
declare
N_Bare_Grammar_Predicate : constant Bare_Grammar_Predicate := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Predicate.Grammar_Predicate_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_Predicate.Grammar_Predicate_F_Prop_Ref;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Rule_Ref_Range =>
declare
N_Bare_Grammar_Rule_Ref : constant Bare_Grammar_Rule_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Rule_Ref.Grammar_Rule_Ref_F_Node_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Skip_Range =>
declare
N_Bare_Grammar_Skip : constant Bare_Grammar_Skip := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Skip.Grammar_Skip_F_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_Stop_Cut_Range =>
declare
N_Bare_Grammar_Stop_Cut : constant Bare_Grammar_Stop_Cut := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_Stop_Cut.Grammar_Stop_Cut_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Parse_Node_Expr_Range =>
declare
N_Bare_Parse_Node_Expr : constant Bare_Parse_Node_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Parse_Node_Expr.Parse_Node_Expr_F_Node_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Parse_Node_Expr.Parse_Node_Expr_F_Sub_Exprs;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Token_No_Case_Lit_Range =>
declare
N_Bare_Token_No_Case_Lit : constant Bare_Token_No_Case_Lit := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Token_No_Case_Lit.Token_No_Case_Lit_F_Lit;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Token_Ref_Range =>
declare
N_Bare_Token_Ref : constant Bare_Token_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Token_Ref.Token_Ref_F_Token_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Token_Ref.Token_Ref_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_If_Expr_Range =>
declare
N_Bare_If_Expr : constant Bare_If_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Cond_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Then_Expr;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Alternatives;
                            return;
                    

                        when 4 =>
                            Result := N_Bare_If_Expr.If_Expr_F_Else_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Isa_Range =>
declare
N_Bare_Isa : constant Bare_Isa := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Isa.Isa_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Isa.Isa_F_Dest_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Keep_Expr_Range =>
declare
N_Bare_Keep_Expr : constant Bare_Keep_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Keep_Expr.Keep_Expr_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Keep_Expr.Keep_Expr_F_Keep_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lambda_Expr_Range =>
declare
N_Bare_Lambda_Expr : constant Bare_Lambda_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lambda_Expr.Lambda_Expr_F_Params;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lambda_Expr.Lambda_Expr_F_Return_Type;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Lambda_Expr.Lambda_Expr_F_Body;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Null_Lit_Range =>
declare
N_Bare_Null_Lit : constant Bare_Null_Lit := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Null_Lit.Null_Lit_F_Dest_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Block_String_Lit_Range =>
declare
N_Bare_Block_String_Lit : constant Bare_Block_String_Lit := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Block_String_Lit.Block_String_Lit_F_Lines;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Logic_Expr_Range =>
declare
N_Bare_Logic_Expr : constant Bare_Logic_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Logic_Expr.Logic_Expr_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Match_Expr_Range =>
declare
N_Bare_Match_Expr : constant Bare_Match_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Match_Expr.Match_Expr_F_Match_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Match_Expr.Match_Expr_F_Branches;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Not_Expr_Range =>
declare
N_Bare_Not_Expr : constant Bare_Not_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Not_Expr.Not_Expr_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Paren_Expr_Range =>
declare
N_Bare_Paren_Expr : constant Bare_Paren_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Paren_Expr.Paren_Expr_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Raise_Expr_Range =>
declare
N_Bare_Raise_Expr : constant Bare_Raise_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Raise_Expr.Raise_Expr_F_Dest_Type;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Raise_Expr.Raise_Expr_F_Except_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Subscript_Expr_Range =>
declare
N_Bare_Subscript_Expr : constant Bare_Subscript_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Subscript_Expr.Subscript_Expr_F_Prefix;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Subscript_Expr.Subscript_Expr_F_Index;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Try_Expr_Range =>
declare
N_Bare_Try_Expr : constant Bare_Try_Expr := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Try_Expr.Try_Expr_F_Try_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Try_Expr.Try_Expr_F_Or_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Un_Op_Range =>
declare
N_Bare_Un_Op : constant Bare_Un_Op := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Un_Op.Un_Op_F_Op;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Un_Op.Un_Op_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Full_Decl_Range =>
declare
N_Bare_Full_Decl : constant Bare_Full_Decl := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Full_Decl.Full_Decl_F_Doc;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Full_Decl.Full_Decl_F_Decl_Annotations;
                            return;
                    

                        when 3 =>
                            Result := N_Bare_Full_Decl.Full_Decl_F_Decl;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Grammar_List_Sep_Range =>
declare
N_Bare_Grammar_List_Sep : constant Bare_Grammar_List_Sep := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Grammar_List_Sep.Grammar_List_Sep_F_Token;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Grammar_List_Sep.Grammar_List_Sep_F_Extra;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Import_Range =>
declare
N_Bare_Import : constant Bare_Import := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Import.Import_F_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Langkit_Root_Range =>
declare
N_Bare_Langkit_Root : constant Bare_Langkit_Root := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Langkit_Root.Langkit_Root_F_Imports;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Langkit_Root.Langkit_Root_F_Decls;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Case_Rule_Range =>
declare
N_Bare_Lexer_Case_Rule : constant Bare_Lexer_Case_Rule := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Case_Rule.Lexer_Case_Rule_F_Expr;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Case_Rule.Lexer_Case_Rule_F_Alts;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lexer_Case_Rule_Send_Range =>
declare
N_Bare_Lexer_Case_Rule_Send : constant Bare_Lexer_Case_Rule_Send := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Lexer_Case_Rule_Send.Lexer_Case_Rule_Send_F_Sent;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Lexer_Case_Rule_Send.Lexer_Case_Rule_Send_F_Match_Size;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Lkt_Node_Base_List =>
declare
N_Bare_Lkt_Node_Base_List : constant Bare_Lkt_Node_Base_List := Node;
begin

                    if Index > N_Bare_Lkt_Node_Base_List.Count then
                        Index_In_Bounds := False;
                    else
                        Result := N_Bare_Lkt_Node_Base_List.Nodes (Index);
                    end if;
                    return;
                
end;
when Lkt_Match_Branch_Range =>
declare
N_Bare_Match_Branch : constant Bare_Match_Branch := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Match_Branch.Match_Branch_F_Decl;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Match_Branch.Match_Branch_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Param_Range =>
declare
N_Bare_Param : constant Bare_Param := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Param.Param_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Param.Param_F_Value;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Function_Type_Ref_Range =>
declare
N_Bare_Function_Type_Ref : constant Bare_Function_Type_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Function_Type_Ref.Function_Type_Ref_F_Args_Types;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Function_Type_Ref.Function_Type_Ref_F_Return_Type;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Generic_Type_Ref_Range =>
declare
N_Bare_Generic_Type_Ref : constant Bare_Generic_Type_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Generic_Type_Ref.Generic_Type_Ref_F_Type_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Generic_Type_Ref.Generic_Type_Ref_F_Params;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Simple_Type_Ref_Range =>
declare
N_Bare_Simple_Type_Ref : constant Bare_Simple_Type_Ref := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Simple_Type_Ref.Simple_Type_Ref_F_Type_Name;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when Lkt_Var_Bind_Range =>
declare
N_Bare_Var_Bind : constant Bare_Var_Bind := Node;
begin
case Index is

                        when 1 =>
                            Result := N_Bare_Var_Bind.Var_Bind_F_Name;
                            return;
                    

                        when 2 =>
                            Result := N_Bare_Var_Bind.Var_Bind_F_Expr;
                            return;
                    

                        when others => null;
                    end case;
                
end;
when others => null;
end case;

      --  Execution should reach this point iff nothing matched this index, so
      --  we must be out of bounds.
      Index_In_Bounds := False;
   end Get_Child;

   -----------
   -- Print --
   -----------

   procedure Print
     (Node        : Liblktlang_Support.Generic_API.Analysis.Lk_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "")
   is
      use Liblktlang_Support.Generic_API.Analysis;
      use Liblktlang_Support.Generic_API.Introspection;

      T : Type_Ref;
   begin
      if Node.Is_Null then
         Put_Line ("None");
         return;
      end if;

      T := Type_Of (Node);
      Put (Line_Prefix & Image (Node_Type_Repr_Name (T)));
      if Show_Slocs then
         Put ("[" & Image (Node.Sloc_Range) & "]");
      end if;

      if Node.Is_Incomplete then
         Put (" <<INCOMPLETE>>");
      end if;

      if Node.Is_Token_Node then
         Put_Line (": " & Image (Node.Text));

      elsif Is_List_Node (Node) then

         --  List nodes are displayed in a special way (they have no field)

         declare
            Count : constant Natural := Node.Children_Count;
            Child : Lk_Node;
         begin
            if Count = 0 then
               Put_Line (": <empty list>");
               return;
            end if;
            New_Line;

            for I in 1 .. Count loop
               Child := Node.Child (I);
               if not Child.Is_Null then
                  Print (Child, Show_Slocs, Line_Prefix & "|  ");
               end if;
            end loop;
         end;

      else
         --  This is for regular nodes: display each syntax field (i.e.
         --  non-property member).

         declare
            Attr_Prefix     : constant String := Line_Prefix & "|";
            Children_Prefix : constant String := Line_Prefix & "|  ";
            M_List          : constant Struct_Member_Ref_Array := Members (T);
            Child           : Lk_Node;
         begin
            New_Line;
            for M of M_List loop
               if not Is_Property (M) and then not Is_Null_For (M, T) then
                  Child := As_Node (Eval_Node_Member (Node, M));
                  Put (Attr_Prefix
                       & Image (Format_Name (Member_Name (M), Lower)) & ":");
                  if Child.Is_Null then
                     Put_Line (" <null>");
                  else
                     New_Line;
                     Print (Child, Show_Slocs, Children_Prefix);
                  end if;
               end if;
            end loop;
         end;
      end if;
   end Print;

   -----------
   -- Print --
   -----------

   procedure Print
     (Node        : Bare_Lkt_Node;
      Show_Slocs  : Boolean;
      Line_Prefix : String := "")
   is
      Entity : constant Internal_Entity := (Node, No_Entity_Info);
   begin
      Print (To_Generic_Node (Entity), Show_Slocs, Line_Prefix);
   end Print;

   ------------
   -- Parent --
   ------------

   function Parent (Node : Bare_Lkt_Node) return Bare_Lkt_Node is
   begin
      return Node.Parent;
   end Parent;

   ------------------
   -- Stored_Token --
   ------------------

   function Stored_Token
     (Node  : Bare_Lkt_Node;
      Token : Token_Reference) return Token_Index
   is
      Index : constant Token_Or_Trivia_Index := Get_Token_Index (Token);
   begin
      if Node.Unit.TDH'Access /= Get_Token_TDH (Token) then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "Cannot associate a token and a node from different analysis"
            & " units");
      elsif Index.Trivia /= No_Token_Index then
         Raise_Property_Exception
           (Node,
            Property_Error'Identity,
            "A node cannot hold trivia");
      end if;

      return Index.Token;
   end Stored_Token;

   -------------------------
   -- Children_And_Trivia --
   -------------------------

   function Children_And_Trivia
     (Node : Bare_Lkt_Node) return Bare_Children_Vector
   is
      Ret_Vec : Bare_Children_Vector;
      Ctx     : Internal_Context renames Node.Unit.Context;
      TDH     : Token_Data_Handler renames Node.Unit.TDH;

      procedure Append_Trivias (First, Last : Token_Index);
      --  Append all the trivias of tokens between indices First and Last to
      --  the returned vector.

      function Filter_Children
        (Parent : Bare_Lkt_Node)
         return Internal_Bare_Lkt_Node_Array;
      --  Return an array for all children in Parent that are not null

      --------------------
      -- Append_Trivias --
      --------------------

      procedure Append_Trivias (First, Last : Token_Index) is
      begin
         for I in First .. Last loop
            for D of Get_Trivias (TDH, I) loop
               Ret_Vec.Append
                 (Bare_Child_Record'
                    (Kind   => Trivia,
                     Trivia => Wrap_Token_Reference
                                 (Ctx, TDH'Access, (I, D))));
            end loop;
         end loop;
      end Append_Trivias;

      ---------------------
      -- Filter_Children --
      ---------------------

      function Filter_Children
        (Parent : Bare_Lkt_Node)
         return Internal_Bare_Lkt_Node_Array
      is
         Children : constant Internal_Bare_Lkt_Node_Array :=
            Implementation.Children (Parent);
         Result   : Internal_Bare_Lkt_Node_Array (Children'Range);
         Next     : Integer := Result'First;
      begin
         for I in Children'Range loop
            if Children (I) /= null then
               Result (Next) := Children (I);
               Next := Next + 1;
            end if;
         end loop;
         return Result (Result'First .. Next - 1);
      end Filter_Children;

      First_Child : constant Positive := 1;
      N_Children  : constant Internal_Bare_Lkt_Node_Array :=
         Filter_Children (Node);
   begin
      if N_Children'Length > 0
        and then (Node.Token_Start_Index
                    /= N_Children (First_Child).Token_Start_Index)
      then
         Append_Trivias (Node.Token_Start_Index,
                         N_Children (First_Child).Token_Start_Index - 1);
      end if;

      --  Append each node to Ret_Vec, and append trivia that follow after each
      --  non-ghost nodes.
      for I in N_Children'Range loop
         Ret_Vec.Append (Bare_Child_Record'(Child, N_Children (I)));
         if not Is_Ghost (N_Children (I)) then
            Append_Trivias (N_Children (I).Token_End_Index,
                            (if I = N_Children'Last
                             then Node.Token_End_Index - 1
                             else N_Children (I + 1).Token_Start_Index - 1));
         end if;
      end loop;

      return Ret_Vec;
   end Children_And_Trivia;

   --------------
   -- Is_Ghost --
   --------------

   function Is_Ghost (Node : Bare_Lkt_Node) return Boolean
   is (Node.Token_End_Index = No_Token_Index);

   -------------------
   -- Is_Incomplete --
   -------------------

   function Is_Incomplete (Node : Bare_Lkt_Node) return Boolean
   is
      LGC : Bare_Lkt_Node;
   begin
     if Is_List_Node (Node.Kind) then
        LGC := (if Last_Child_Index (Node) /= 0
                then Child (Node, Last_Child_Index (Node))
                else null);
        return LGC /= null and then Is_Incomplete (LGC);
      else
         return Node.Last_Attempted_Child > -1;
      end if;
   end;

   -----------------
   -- Token_Start --
   -----------------

   function Token_Start (Node : Bare_Lkt_Node) return Token_Reference
   is (Token (Node, Node.Token_Start_Index));

   ---------------
   -- Token_End --
   ---------------

   function Token_End (Node : Bare_Lkt_Node) return Token_Reference
   is
     (if Node.Token_End_Index = No_Token_Index
      then Token_Start (Node)
      else Token (Node, Node.Token_End_Index));

   -----------
   -- Token --
   -----------

   function Token
     (Node  : Bare_Lkt_Node;
      Index : Token_Index) return Token_Reference
   is
      Unit    : constant Internal_Unit := Node.Unit;
      Context : constant Internal_Context := Unit.Context;
   begin
      return Wrap_Token_Reference
        (Context, Token_Data (Unit), (Index, No_Token_Index));
   end Token;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Bare_Lkt_Node) return Boolean is
   begin
      --  Reject invalid inputs
      if Left /= null and Is_Synthetic (Left) then
         raise Property_Error with "left node is synthetic";
      elsif Right /= null and Is_Synthetic (Right) then
         raise Property_Error with "right node is synthetic";
      end if;

      --  Null nodes come first
      if Left = null then
         return Right /= null;
      elsif Right = null then
         return False;
      end if;

      --  So we have two non-null nodes. Sort by unit filename
      if Left.Unit < Right.Unit then
         return True;
      elsif Left.Unit /= Right.Unit then
         return False;
      end if;

      --  Both nodes come from the same unit: compare their token indexes
      if Left.Token_Start_Index < Right.Token_Start_Index then
         return True;
      elsif Left.Token_Start_Index > Right.Token_Start_Index then
         return False;
      else
         return Left.Token_End_Index < Right.Token_End_Index;
      end if;
   end "<";

   -------------
   -- Is_Null --
   -------------

   function Is_Null (Node : Bare_Lkt_Node) return Boolean
   is (Node = null);

   ----------
   -- Kind --
   ----------

   function Kind (Node : Bare_Lkt_Node) return Lkt_Node_Kind_Type
   is (Node.Kind);

   -----------------
   -- Child_Index --
   -----------------

   function Child_Index (Node : Bare_Lkt_Node) return Integer
   is
      N : Bare_Lkt_Node := null;
   begin
      if Node.Parent = null then
         raise Property_Error with
            "Trying to get the child index of a root node";
      end if;

      for I in First_Child_Index (Node.Parent)
            .. Last_Child_Index (Node.Parent)
      loop
         N := Child (Node.Parent, I);
         if N = Node then
            return I - 1;
         end if;
      end loop;

      --  If we reach this point, then Node isn't a Child of Node.Parent. This
      --  is not supposed to happen.
      raise Program_Error;
   end Child_Index;

   -------------------
   -- Fetch_Sibling --
   -------------------

   function Fetch_Sibling
     (Node   : Bare_Lkt_Node;
      Offset : Integer) return Bare_Lkt_Node is
   begin
      --  Root nodes have no sibling: handle them now to avoid invalid requests
      --  in the code below.
      if Node.Parent = null then
         return null;
      end if;

      declare
         Node_Index : constant Positive := Child_Index (Node) + 1;
         --  Child_Index is 0-based, but the Child primitive expects a 1-based
         --  index.

         Sibling_Index : constant Integer := Node_Index + Offset;
      begin
         --  Child returns null for out-of-bound indexes

         return (if Sibling_Index >= 1
                 then Child (Node.Parent, Sibling_Index)
                 else null);
      end;
   end Fetch_Sibling;

   -------------------
   -- Fetch_Sibling --
   -------------------

   function Fetch_Sibling
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info;
      Offset : Integer) return Internal_Entity
   is
      Sibling : constant Bare_Lkt_Node := Fetch_Sibling (Node, Offset);
   begin
      --  Don't forget to clear entity info if the result is nul

      return (if Sibling = null
              then No_Entity
              else (Sibling, E_Info));
   end Fetch_Sibling;

   ----------------------
   -- Previous_Sibling --
   ----------------------

   function Previous_Sibling
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      return Fetch_Sibling (Node, E_Info, -1);
   end Previous_Sibling;

   ------------------
   -- Next_Sibling --
   ------------------

   function Next_Sibling
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      return Fetch_Sibling (Node, E_Info, 1);
   end Next_Sibling;

   -------------
   -- Inc_Ref --
   -------------

   procedure Inc_Ref (Self : Node_Builder_Type) is
   begin
      if Self.Ref_Count > 0 then
         Self.Ref_Count := Self.Ref_Count + 1;
      end if;
   end Inc_Ref;

   -------------
   -- Dec_Ref --
   -------------

   procedure Dec_Ref (Self : in out Node_Builder_Type) is
   begin
      if Self = null or else Self.Ref_Count < 0 then
         return;
      elsif Self.Ref_Count = 1 then
         Self.Release;
         Free (Self);
      else
         Self.Ref_Count := Self.Ref_Count - 1;
      end if;
   end Dec_Ref;

   ------------------------------
   -- Create_Copy_Node_Builder --
   ------------------------------

   function Create_Copy_Node_Builder
     (Value : Bare_Lkt_Node) return Node_Builder_Type is
   begin
      --  No need to allocate a new builder if in practice it cannot be
      --  distinguished from the "null" builder.

      if Value = null then
         return Null_Node_Builder;
      else
         return new Copy_Node_Builder_Record'(Ref_Count => 1, Value => Value);
      end if;
   end Create_Copy_Node_Builder;



   ----------------------
   -- Compare_Metadata --
   ----------------------

   --  Deactivate "not referenced" warnings because if the metadata struct has
   --  no fields, formals and temporaries won't be referenced in the two
   --  following functions.
   pragma Warnings (Off, "referenced");
   function Compare_Metadata (L, R : Internal_Metadata) return Boolean is
   begin
      return True;
   end Compare_Metadata;

   ----------
   -- Hash --
   ----------

   function Hash (Self : Internal_Metadata) return Hash_Type is
      Ret : Hash_Type := Liblktlang_Support.Hashes.Initial_Hash;
   begin
      return Ret;
   end Hash;
   pragma Warnings (On, "referenced");

   -------------
   -- Combine --
   -------------

   function Combine
     (L, R : Internal_Metadata) return Internal_Metadata
   is
      pragma Unreferenced (L, R);
      Ret : Internal_Metadata := No_Metadata;
   begin
      return Ret;
   end Combine;

   -------------------------------
   -- Create_Static_Lexical_Env --
   -------------------------------

   function Create_Static_Lexical_Env
     (Parent            : Lexical_Env;
      Node              : Bare_Lkt_Node;
      Sym_Table         : Symbol_Table;
      Transitive_Parent : Boolean := False) return Lexical_Env
   is
      Unit : constant Internal_Unit :=
        (if Node = null then null else Node.Unit);
   begin
      return Result : Lexical_Env := Create_Lexical_Env
        (Parent, Node, Transitive_Parent, Sym_Table, Convert_Unit (Unit))
      do
         if Unit /= null then
            Register_Destroyable (Unit, Unwrap (Result.Env));
         end if;
      end return;
   end Create_Static_Lexical_Env;

   ---------
   -- Get --
   ---------

   function Get
     (Self  : Bare_Lkt_Node;
      A     : AST_Envs.Entity_Array;
      Index : Integer) return Internal_Entity
   is
      function Length (A : AST_Envs.Entity_Array) return Natural
      is (A'Length);

      function Get
        (A     : AST_Envs.Entity_Array;
         Index : Integer) return Internal_Entity
      is (A (Index + 1)); --  A is 1-based but Index is 0-based

      function Relative_Get is new Liblktlang_Support.Relative_Get
        (Item_Type     => Entity,
         Sequence_Type => AST_Envs.Entity_Array,
         Length        => Length,
         Get           => Get);
      Result : Internal_Entity;
   begin
      if Relative_Get (A, Index, Result) then
         return Result;
      else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "out-of-bounds array access");
      end if;
   end Get;

   -----------
   -- Group --
   -----------

   function Group
     (Envs   : Lexical_Env_Array_Access;
      Env_Md : Internal_Metadata := No_Metadata) return Lexical_Env
   is (Group (Lexical_Env_Array (Envs.Items), Env_Md));

       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   

      --
      --  Untyped wrappers for Bare_Type_Decl
      --

         
function Type_Decl_P_Type_Base_Scope
  (E : Entity
  ) return Lexical_Env;



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   



       

   




   ------------------
   -- Children_Env --
   ------------------

   function Children_Env
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Lexical_Env
   is (Rebind_Env (Node.Self_Env, E_Info));

   --------------
   -- Node_Env --
   --------------

   function Node_Env
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Lexical_Env
   is
      function Get_Base_Env return Lexical_Env;
      --  Return the environment that we need to rebind before returning

      ------------------
      -- Get_Base_Env --
      ------------------

      function Get_Base_Env return Lexical_Env is
         pragma Warnings (Off, "referenced");
         function Get_Parent_Env return Lexical_Env;
         pragma Warnings (On, "referenced");

         --------------------
         -- Get_Parent_Env --
         --------------------

         function Get_Parent_Env return Lexical_Env is
            Parent : constant Lexical_Env := AST_Envs.Parent (Node.Self_Env);
         begin
            --  If Node is the root scope or the empty environment, Parent can
            --  be a wrapper around the null node. Turn this into the
            --  Empty_Env, as null envs are erroneous values in properties.
            return (if Unwrap (Parent) = null
                    then Empty_Env
                    else Parent);
         end Get_Parent_Env;

      begin
         
         return
           (if Node.Kind in Lkt_Fun_Decl | Lkt_Generic_Decl | Lkt_Grammar_Decl | Lkt_Lexer_Decl | Lkt_Block_Expr | Lkt_Lambda_Expr | Lkt_Decl_Block | Lkt_Match_Branch
            then Get_Parent_Env
            else Node.Self_Env);
      end Get_Base_Env;

      Base_Env : Lexical_Env := Get_Base_Env;
      Result   : constant Lexical_Env := Rebind_Env (Base_Env, E_Info);
   begin
      Dec_Ref (Base_Env);
      return Result;
   end Node_Env;

   ------------
   -- Parent --
   ------------

   function Parent
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity is
   begin
      --  TODO: shed entity information as appropriate
      return (Node.Parent, E_Info);
   end Parent;

   -------------
   -- Parents --
   -------------

   function Parents
     (Node      : Bare_Lkt_Node;
      With_Self : Boolean := True;
      E_Info    : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity_Array_Access
   is
      Bare_Parents : Bare_Lkt_Node_Array_Access := Parents (Node, With_Self);
      Result       : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (Bare_Parents.N);
   begin
      --  TODO: shed entity information as appropriate
      for I in Bare_Parents.Items'Range loop
         Result.Items (I) := (Bare_Parents.Items (I), E_Info);
      end loop;
      Dec_Ref (Bare_Parents);
      return Result;
   end Parents;

   --------------
   -- Children --
   --------------

   function Children
     (Node   : Bare_Lkt_Node;
      E_Info : Internal_Entity_Info := No_Entity_Info)
      return Internal_Entity_Array_Access
   is
      Bare_Children : Bare_Lkt_Node_Array_Access := Children (Node);
      Result        : Internal_Entity_Array_Access :=
         Create_Internal_Entity_Array (Bare_Children.N);
   begin
      --  TODO: shed entity information as appropriate
      for I in Bare_Children.Items'Range loop
         Result.Items (I) := (Bare_Children.Items (I), E_Info);
      end loop;
      Dec_Ref (Bare_Children);
      return Result;
   end Children;

   ---------------------
   -- New_Unit_String --
   ---------------------

   function New_Unit_String
     (Unit : Internal_Unit; Str : String) return String_Access
   is
      procedure Register_Destroyable_String is new Register_Destroyable_Gen
        (String, String_Access, Free);
   begin
      return Ret : String_Access := new String'(Str) do
         Register_Destroyable_String (Unit, Ret);
      end return;
   end New_Unit_String;

   --------------------------------
   -- Assign_Names_To_Logic_Vars --
   --------------------------------

   procedure Assign_Names_To_Logic_Vars (Node : Bare_Lkt_Node) is

      pragma Warnings (Off, "referenced");

      procedure Assign
        (Node  : Bare_Lkt_Node;
         LV    : in out Logic_Var_Record;
         Field : String);
      --  Assign a name to the LV logic variable. Node must be the node that
      --  owns LV, and Field must be the name of the field in Node that holds
      --  LV.

      ------------
      -- Assign --
      ------------

      procedure Assign
        (Node  : Bare_Lkt_Node;
         LV    : in out Logic_Var_Record;
         Field : String) is
      begin
         LV.Dbg_Name :=
           new String'(Image (Short_Text_Image (Node)) & "." & Field);
      end Assign;

      K : constant Lkt_Node_Kind_Type := Node.Kind;

      pragma Warnings (On, "referenced");

   begin
      
      pragma Unreferenced (K);
      for Child of Internal_Bare_Lkt_Node_Array'(Children (Node)) loop
         if Child /= null then
            Assign_Names_To_Logic_Vars (Child);
         end if;
      end loop;
   end Assign_Names_To_Logic_Vars;

   --------------------------
   -- Initialization_Error --
   --------------------------

   function Initialization_Error
     (Exc : Ada.Exceptions.Exception_Occurrence)
      return Error_Initialization_State
   is
      use Ada.Exceptions;
   begin
      if
         
            Exception_Identity (Exc) = Property_Error'Identity
         then
            return Raised_Property_Error;
      else
         raise Program_Error;
      end if;
   end Initialization_Error;

   ----------------------------------
   -- Reraise_Initialization_Error --
   ----------------------------------

   procedure Reraise_Initialization_Error
     (Node    : Bare_Lkt_Node;
      State   : Error_Initialization_State;
      Message : String)
   is
      Exc : Ada.Exceptions.Exception_Id;
   begin
      case State is
            when Raised_Property_Error =>
               Exc := Property_Error'Identity;
      end case;
      Raise_Property_Exception (Node, Exc, Message);
   end Reraise_Initialization_Error;

   ----------------
   -- Text_Image --
   ----------------

   function Text_Image (Ent : Internal_Entity) return Text_Type is
   begin
      if Ent.Node /= null then
         declare
            Node_Image : constant Text_Type := Short_Text_Image (Ent.Node);
         begin
            return
            (if Ent.Info.Rebindings /= null
             then "<| "
             & Node_Image (Node_Image'First + 1 .. Node_Image'Last - 1) & " "
             & AST_Envs.Text_Image (Ent.Info.Rebindings) & " |>"
             else Node_Image);
         end;
      else
         return "None";
      end if;
   end Text_Image;

   ---------------------
   -- Full_Sloc_Image --
   ---------------------

   function Full_Sloc_Image (Node : Bare_Lkt_Node) return String_Type
   is
      Res      : constant Text_Type :=
        To_Text
          (Ada.Directories.Simple_Name
             (Get_Filename (Unit (Node))))
           & ":" & To_Text (Image (Start_Sloc (Sloc_Range (Node)))) & ": ";
   begin
      return Create_String (Res);
   end Full_Sloc_Image;

   -----------
   -- Image --
   -----------

   function Image (Ent : Internal_Entity) return String is
      Result : constant Text_Type := Text_Image (Ent);
   begin
      return Image (Result);
   end Image;

   ---------------
   -- Can_Reach --
   ---------------

   function Can_Reach (El, From : Bare_Lkt_Node) return Boolean is
   begin
      return Lkt_Node_P_Can_Reach (El, From);
   end Can_Reach;

   -----------------
   -- Hash_Entity --
   -----------------

   function Hash_Entity (Self : Internal_Entity) return Hash_Type is
   begin
      return Combine
        ((Hash (Self.Node), Hash (Self.Info.Rebindings), Hash (Self.Info.Md)));
   end Hash_Entity;

   --------------------
   -- Compare_Entity --
   --------------------

   function Compare_Entity (Left, Right : Internal_Entity) return Boolean
   is
   begin
      return Left.Node = Right.Node
             and then Left.Info.Rebindings = Right.Info.Rebindings
             and then Compare_Metadata (Left.Info.Md, Right.Info.Md);
   end Compare_Entity;

   --------------------------------
   -- Create_Dynamic_Lexical_Env --
   --------------------------------

   function Create_Dynamic_Lexical_Env
     (Self              : Bare_Lkt_Node;
      Assocs_Getter     : Inner_Env_Assocs_Resolver;
      Assoc_Resolver    : Entity_Resolver;
      Transitive_Parent : Boolean;
      Sym_Table         : Symbol_Table) return Lexical_Env
   is
      Unit : constant Internal_Unit := Self.Unit;
   begin
      --  This restriction is necessary to avoid relocation issues when
      --  Self.Self_Env is terminated.
      if Is_Foreign_Strict (Self.Self_Env, Self) then
         Raise_Property_Exception
           (Self,
            Property_Error'Identity,
            "cannot create a dynamic lexical env when Self.Self_Env is"
            & " foreign");
      end if;

      return Result : constant Lexical_Env := Create_Dynamic_Lexical_Env
        (Parent            => Null_Lexical_Env,
         Node              => Self,
         Transitive_Parent => Transitive_Parent,
         Owner             => Convert_Unit (Unit),
         Assocs_Getter     => Assocs_Getter,
         Assoc_Resolver    => Assoc_Resolver,
         Sym_Table         => Sym_Table)
      do
         --  Since dynamic lexical environments can only be created in lazy
         --  field initializers, it is fine to tie Result's lifetime to the
         --  its owning unit's lifetime.
         Register_Destroyable (Unit, Unwrap (Result));
      end return;
   end Create_Dynamic_Lexical_Env;

   procedure Destroy_Synthetic_Node (Node : in out Bare_Lkt_Node);
   --  Helper for the Register_Destroyable above

   ------------
   -- Length --
   ------------

   function Length (Node : Bare_Lkt_Node_Base_List) return Natural
   is (if Node = null then 0 else Children_Count (Node));


      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (B : Boolean) return String is
      begin
         return (if B then "True" else "False");
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (I : Integer) return String is
      begin
         return Integer'Image (I);
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (S : Symbol_Type) return String is
      begin
         return (if S = No_Symbol
                 then "None"
                 else Image (S, With_Quotes => True));
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (C : Character_Type) return String is
         C_Str : constant Text_Type := (1 => C);
      begin
         return "'" & Image (C_Str) & "'";
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (S : String_Type) return String is
      begin
         return Image (S.Content, With_Quotes => True);
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Env : Lexical_Env) return String is
      begin
         case Env.Kind is
         when Static_Primary =>
            return "<LexicalEnv static-primary for "
                   & Trace_Image (Env_Node (Env)) & ">";
         when others =>
            return "<LexicalEnv synthetic>";
         end case;
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (R : Env_Rebindings) return String is
      begin
         return Image (Text_Image (R));
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Unit : Internal_Unit) return String is
      begin
         return "Internal_Unit (""" & Basename (Unit) & """)";
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Eq : Logic_Equation) return String is
         pragma Unreferenced (Eq);
      begin
         return "<LogicEquation>";
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Var : Logic_Var) return String is
         pragma Unreferenced (Var);
      begin
         return "<LogicVariable>";
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (K : Analysis_Unit_Kind) return String is
      begin
         return Analysis_Unit_Kind'Image (K);
      end Trace_Image;

      -----------------
      -- Trace_Image --
      -----------------

      function Trace_Image (Self : Ref_Categories) return String is
         Result : Unbounded_String;
         First  : Boolean := True;
      begin
         Append (Result, "RefCategories(");
         for C in Ref_Category loop
            if Self (C) then
               if First then
                  First := False;
               else
                  Append (Result, ", ");
               end if;
               Append (Result, C'Image);
            end if;
         end loop;
         Append (Result, ")");
         return To_String (Result);
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Decoded_Char_Value) is
      begin
               Inc_Ref (R.Error_Message);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Decoded_Char_Value) is
      begin
               Dec_Ref (R.Error_Message);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Decoded_Char_Value) return Boolean is
      begin
         return L.Value = R.Value and then L.Has_Error = R.Has_Error and then L.Error_Sloc = R.Error_Sloc and then Equivalent (L.Error_Message, R.Error_Message);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Decoded_Char_Value) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Has_Error => " & Trace_Image (R.Has_Error)
                        & ", "
                     & "Error_Sloc => " & Trace_Image (R.Error_Sloc)
                        & ", "
                     & "Error_Message => " & Trace_Image (R.Error_Message)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Decoded_String_Value) is
      begin
               Inc_Ref (R.Value);
               Inc_Ref (R.Error_Message);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Decoded_String_Value) is
      begin
               Dec_Ref (R.Value);
               Dec_Ref (R.Error_Message);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Decoded_String_Value) return Boolean is
      begin
         return Equivalent (L.Value, R.Value) and then L.Has_Error = R.Has_Error and then L.Error_Sloc = R.Error_Sloc and then Equivalent (L.Error_Message, R.Error_Message);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Decoded_String_Value) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Has_Error => " & Trace_Image (R.Has_Error)
                        & ", "
                     & "Error_Sloc => " & Trace_Image (R.Error_Sloc)
                        & ", "
                     & "Error_Message => " & Trace_Image (R.Error_Message)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Designated_Env) is
      begin
               Inc_Ref (R.Direct_Env);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Designated_Env) is
      begin
               Dec_Ref (R.Direct_Env);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Designated_Env) return Boolean is
      begin
         return L.Kind = R.Kind and then L.Env_Name = R.Env_Name and then Equivalent (L.Direct_Env, R.Direct_Env);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Designated_Env) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Kind => " & Trace_Image (R.Kind)
                        & ", "
                     & "Env_Name => " & Trace_Image (R.Env_Name)
                        & ", "
                     & "Direct_Env => " & Trace_Image (R.Direct_Env)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Metadata) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                  & "null record"
               & ")");
      end Trace_Image;


   

   




   

      ----------
      -- Hash --
      ----------

      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity_Info) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
         
            return Combine ((Hash (R.Md), Hash (R.Rebindings), Hash (R.From_Rebound)));
      end Hash;


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Info) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Md => " & Trace_Image (R.Md)
                        & ", "
                     & "Rebindings => " & Trace_Image (R.Rebindings)
                        & ", "
                     & "From_Rebound => " & Trace_Image (R.From_Rebound)
               & ")");
      end Trace_Image;


   

   



      function Create_Internal_Entity
        (Node : Bare_Lkt_Node; Info : Internal_Entity_Info)
         return Internal_Entity is
      begin
         if Node = null then
            return No_Entity;
         end if;
         return (Node => Node, Info => Info);
      end;



   

      ----------
      -- Hash --
      ----------

      pragma Warnings (Off, "referenced");
      function Hash (R : Internal_Entity) return Hash_Type is
         pragma Warnings (On, "referenced");
      begin
         
            return Combine
              (Hash (R.Node), Hash (R.Info));
      end Hash;


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Expr
        (Node : Bare_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Expr is
      begin
         if Node = null then
            return No_Entity_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Any_Of
        (Node : Bare_Any_Of; Info : Internal_Entity_Info)
         return Internal_Entity_Any_Of is
      begin
         if Node = null then
            return No_Entity_Any_Of;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Any_Of) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lkt_Node_Base_List
        (Node : Bare_Lkt_Node_Base_List; Info : Internal_Entity_Info)
         return Internal_Entity_Lkt_Node_Base_List is
      begin
         if Node = null then
            return No_Entity_Lkt_Node_Base_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lkt_Node_Base_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Expr_List
        (Node : Bare_Expr_List; Info : Internal_Entity_Info)
         return Internal_Entity_Expr_List is
      begin
         if Node = null then
            return No_Entity_Expr_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Expr_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Any_Of_List
        (Node : Bare_Any_Of_List; Info : Internal_Entity_Info)
         return Internal_Entity_Any_Of_List is
      begin
         if Node = null then
            return No_Entity_Any_Of_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Any_Of_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Array_Literal
        (Node : Bare_Array_Literal; Info : Internal_Entity_Info)
         return Internal_Entity_Array_Literal is
      begin
         if Node = null then
            return No_Entity_Array_Literal;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Array_Literal) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Dot_Expr
        (Node : Bare_Base_Dot_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Dot_Expr is
      begin
         if Node = null then
            return No_Entity_Base_Dot_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Dot_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl
        (Node : Bare_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Decl is
      begin
         if Node = null then
            return No_Entity_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Grammar_Rule_Decl
        (Node : Bare_Base_Grammar_Rule_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Grammar_Rule_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Grammar_Rule_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Grammar_Rule_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Lexer_Case_Rule_Alt
        (Node : Bare_Base_Lexer_Case_Rule_Alt; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Lexer_Case_Rule_Alt is
      begin
         if Node = null then
            return No_Entity_Base_Lexer_Case_Rule_Alt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Lexer_Case_Rule_Alt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Lexer_Case_Rule_Alt_List
        (Node : Bare_Base_Lexer_Case_Rule_Alt_List; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Lexer_Case_Rule_Alt_List is
      begin
         if Node = null then
            return No_Entity_Base_Lexer_Case_Rule_Alt_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Lexer_Case_Rule_Alt_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Base_Val_Decl
        (Node : Bare_Base_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Base_Val_Decl is
      begin
         if Node = null then
            return No_Entity_Base_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Base_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Type_Decl
        (Node : Bare_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Named_Type_Decl
        (Node : Bare_Named_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Named_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Named_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Named_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Basic_Class_Decl
        (Node : Bare_Basic_Class_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Basic_Class_Decl is
      begin
         if Node = null then
            return No_Entity_Basic_Class_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Basic_Class_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lit
        (Node : Bare_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Lit is
      begin
         if Node = null then
            return No_Entity_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Big_Num_Lit
        (Node : Bare_Big_Num_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Big_Num_Lit is
      begin
         if Node = null then
            return No_Entity_Big_Num_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Big_Num_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Bin_Op
        (Node : Bare_Bin_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Bin_Op is
      begin
         if Node = null then
            return No_Entity_Bin_Op;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Bin_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lkt_Node_List
        (Node : Bare_Lkt_Node_List; Info : Internal_Entity_Info)
         return Internal_Entity_Lkt_Node_List is
      begin
         if Node = null then
            return No_Entity_Lkt_Node_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lkt_Node_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_Decl_List
        (Node : Bare_Block_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Block_Decl_List is
      begin
         if Node = null then
            return No_Entity_Block_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_Expr
        (Node : Bare_Block_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Block_Expr is
      begin
         if Node = null then
            return No_Entity_Block_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_String_Line
        (Node : Bare_Block_String_Line; Info : Internal_Entity_Info)
         return Internal_Entity_Block_String_Line is
      begin
         if Node = null then
            return No_Entity_Block_String_Line;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_String_Line) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_String_Line_List
        (Node : Bare_Block_String_Line_List; Info : Internal_Entity_Info)
         return Internal_Entity_Block_String_Line_List is
      begin
         if Node = null then
            return No_Entity_Block_String_Line_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_String_Line_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_String_Lit
        (Node : Bare_String_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_String_Lit is
      begin
         if Node = null then
            return No_Entity_String_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_String_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Block_String_Lit
        (Node : Bare_Block_String_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Block_String_Lit is
      begin
         if Node = null then
            return No_Entity_Block_String_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Block_String_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Call_Expr
        (Node : Bare_Call_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Call_Expr is
      begin
         if Node = null then
            return No_Entity_Call_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Call_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Call_Expr_List
        (Node : Bare_Call_Expr_List; Info : Internal_Entity_Info)
         return Internal_Entity_Call_Expr_List is
      begin
         if Node = null then
            return No_Entity_Call_Expr_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Call_Expr_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Cast_Expr
        (Node : Bare_Cast_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Cast_Expr is
      begin
         if Node = null then
            return No_Entity_Cast_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Cast_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Char_Lit
        (Node : Bare_Char_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Char_Lit is
      begin
         if Node = null then
            return No_Entity_Char_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Char_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Class_Decl
        (Node : Bare_Class_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Class_Decl is
      begin
         if Node = null then
            return No_Entity_Class_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Class_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Class_Qualifier
        (Node : Bare_Class_Qualifier; Info : Internal_Entity_Info)
         return Internal_Entity_Class_Qualifier is
      begin
         if Node = null then
            return No_Entity_Class_Qualifier;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Class_Qualifier) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Class_Qualifier_Absent
        (Node : Bare_Class_Qualifier_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Class_Qualifier_Absent is
      begin
         if Node = null then
            return No_Entity_Class_Qualifier_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Class_Qualifier_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Class_Qualifier_Present
        (Node : Bare_Class_Qualifier_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Class_Qualifier_Present is
      begin
         if Node = null then
            return No_Entity_Class_Qualifier_Present;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Class_Qualifier_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_User_Val_Decl
        (Node : Bare_User_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_User_Val_Decl is
      begin
         if Node = null then
            return No_Entity_User_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_User_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Explicitly_Typed_Decl
        (Node : Bare_Explicitly_Typed_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Explicitly_Typed_Decl is
      begin
         if Node = null then
            return No_Entity_Explicitly_Typed_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Explicitly_Typed_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Component_Decl
        (Node : Bare_Component_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Component_Decl is
      begin
         if Node = null then
            return No_Entity_Component_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Component_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl_Annotation
        (Node : Bare_Decl_Annotation; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Annotation is
      begin
         if Node = null then
            return No_Entity_Decl_Annotation;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Annotation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl_Annotation_List
        (Node : Bare_Decl_Annotation_List; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Annotation_List is
      begin
         if Node = null then
            return No_Entity_Decl_Annotation_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Annotation_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl_Annotation_Params
        (Node : Bare_Decl_Annotation_Params; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Annotation_Params is
      begin
         if Node = null then
            return No_Entity_Decl_Annotation_Params;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Annotation_Params) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Full_Decl_List
        (Node : Bare_Full_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Full_Decl_List is
      begin
         if Node = null then
            return No_Entity_Full_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Full_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Decl_Block
        (Node : Bare_Decl_Block; Info : Internal_Entity_Info)
         return Internal_Entity_Decl_Block is
      begin
         if Node = null then
            return No_Entity_Decl_Block;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Decl_Block) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Id
        (Node : Bare_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Id is
      begin
         if Node = null then
            return No_Entity_Id;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Def_Id
        (Node : Bare_Def_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Def_Id is
      begin
         if Node = null then
            return No_Entity_Def_Id;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Def_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Type_Ref
        (Node : Bare_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Default_List_Type_Ref
        (Node : Bare_Default_List_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Default_List_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Default_List_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Default_List_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Dot_Expr
        (Node : Bare_Dot_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Dot_Expr is
      begin
         if Node = null then
            return No_Entity_Dot_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Dot_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Dyn_Var_Decl
        (Node : Bare_Dyn_Var_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Dyn_Var_Decl is
      begin
         if Node = null then
            return No_Entity_Dyn_Var_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Dyn_Var_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Elsif_Branch
        (Node : Bare_Elsif_Branch; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Branch is
      begin
         if Node = null then
            return No_Entity_Elsif_Branch;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Branch) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Elsif_Branch_List
        (Node : Bare_Elsif_Branch_List; Info : Internal_Entity_Info)
         return Internal_Entity_Elsif_Branch_List is
      begin
         if Node = null then
            return No_Entity_Elsif_Branch_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Elsif_Branch_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Alt_Decl
        (Node : Bare_Enum_Class_Alt_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Alt_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Alt_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Alt_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Alt_Decl_List
        (Node : Bare_Enum_Class_Alt_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Alt_Decl_List is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Alt_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Alt_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Case
        (Node : Bare_Enum_Class_Case; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Case is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Case;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Case) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Case_List
        (Node : Bare_Enum_Class_Case_List; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Case_List is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Case_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Case_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Class_Decl
        (Node : Bare_Enum_Class_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Class_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Class_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Class_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Lit_Decl
        (Node : Bare_Enum_Lit_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Lit_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Lit_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Lit_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Lit_Decl_List
        (Node : Bare_Enum_Lit_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Lit_Decl_List is
      begin
         if Node = null then
            return No_Entity_Enum_Lit_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Lit_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Enum_Type_Decl
        (Node : Bare_Enum_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Enum_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Enum_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Enum_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Env_Spec_Decl
        (Node : Bare_Env_Spec_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Env_Spec_Decl is
      begin
         if Node = null then
            return No_Entity_Env_Spec_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Env_Spec_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Error_On_Null
        (Node : Bare_Error_On_Null; Info : Internal_Entity_Info)
         return Internal_Entity_Error_On_Null is
      begin
         if Node = null then
            return No_Entity_Error_On_Null;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Error_On_Null) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Excludes_Null
        (Node : Bare_Excludes_Null; Info : Internal_Entity_Info)
         return Internal_Entity_Excludes_Null is
      begin
         if Node = null then
            return No_Entity_Excludes_Null;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Excludes_Null) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Excludes_Null_Absent
        (Node : Bare_Excludes_Null_Absent; Info : Internal_Entity_Info)
         return Internal_Entity_Excludes_Null_Absent is
      begin
         if Node = null then
            return No_Entity_Excludes_Null_Absent;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Excludes_Null_Absent) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Excludes_Null_Present
        (Node : Bare_Excludes_Null_Present; Info : Internal_Entity_Info)
         return Internal_Entity_Excludes_Null_Present is
      begin
         if Node = null then
            return No_Entity_Excludes_Null_Present;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Excludes_Null_Present) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Field_Decl
        (Node : Bare_Field_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Field_Decl is
      begin
         if Node = null then
            return No_Entity_Field_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Field_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Full_Decl
        (Node : Bare_Full_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Full_Decl is
      begin
         if Node = null then
            return No_Entity_Full_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Full_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Fun_Arg_Decl
        (Node : Bare_Fun_Arg_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Fun_Arg_Decl is
      begin
         if Node = null then
            return No_Entity_Fun_Arg_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Fun_Arg_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Fun_Arg_Decl_List
        (Node : Bare_Fun_Arg_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Fun_Arg_Decl_List is
      begin
         if Node = null then
            return No_Entity_Fun_Arg_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Fun_Arg_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Fun_Decl
        (Node : Bare_Fun_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Fun_Decl is
      begin
         if Node = null then
            return No_Entity_Fun_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Fun_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Function_Type
        (Node : Bare_Function_Type; Info : Internal_Entity_Info)
         return Internal_Entity_Function_Type is
      begin
         if Node = null then
            return No_Entity_Function_Type;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Function_Type) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Function_Type_Ref
        (Node : Bare_Function_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Function_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Function_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Function_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Decl
        (Node : Bare_Generic_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Formal_Decl_List
        (Node : Bare_Generic_Formal_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Decl_List is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Formal_Type_Decl
        (Node : Bare_Generic_Formal_Type_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Formal_Type_Decl is
      begin
         if Node = null then
            return No_Entity_Generic_Formal_Type_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Formal_Type_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Instantiation
        (Node : Bare_Generic_Instantiation; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Instantiation is
      begin
         if Node = null then
            return No_Entity_Generic_Instantiation;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Generic_Type_Ref
        (Node : Bare_Generic_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Generic_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Generic_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Generic_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Expr
        (Node : Bare_Grammar_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Expr is
      begin
         if Node = null then
            return No_Entity_Grammar_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Cut
        (Node : Bare_Grammar_Cut; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Cut is
      begin
         if Node = null then
            return No_Entity_Grammar_Cut;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Cut) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Decl
        (Node : Bare_Grammar_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Decl is
      begin
         if Node = null then
            return No_Entity_Grammar_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Discard
        (Node : Bare_Grammar_Discard; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Discard is
      begin
         if Node = null then
            return No_Entity_Grammar_Discard;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Discard) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Dont_Skip
        (Node : Bare_Grammar_Dont_Skip; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Dont_Skip is
      begin
         if Node = null then
            return No_Entity_Grammar_Dont_Skip;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Dont_Skip) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Expr_List
        (Node : Bare_Grammar_Expr_List; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Expr_List is
      begin
         if Node = null then
            return No_Entity_Grammar_Expr_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Expr_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Expr_List_List
        (Node : Bare_Grammar_Expr_List_List; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Expr_List_List is
      begin
         if Node = null then
            return No_Entity_Grammar_Expr_List_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Expr_List_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Pick
        (Node : Bare_Grammar_Pick; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Pick is
      begin
         if Node = null then
            return No_Entity_Grammar_Pick;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Pick) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Implicit_Pick
        (Node : Bare_Grammar_Implicit_Pick; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Implicit_Pick is
      begin
         if Node = null then
            return No_Entity_Grammar_Implicit_Pick;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Implicit_Pick) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_List
        (Node : Bare_Grammar_List; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_List is
      begin
         if Node = null then
            return No_Entity_Grammar_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_List_Sep
        (Node : Bare_Grammar_List_Sep; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_List_Sep is
      begin
         if Node = null then
            return No_Entity_Grammar_List_Sep;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_List_Sep) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Null
        (Node : Bare_Grammar_Null; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Null is
      begin
         if Node = null then
            return No_Entity_Grammar_Null;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Null) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Opt
        (Node : Bare_Grammar_Opt; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Opt is
      begin
         if Node = null then
            return No_Entity_Grammar_Opt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Opt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Opt_Error
        (Node : Bare_Grammar_Opt_Error; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Opt_Error is
      begin
         if Node = null then
            return No_Entity_Grammar_Opt_Error;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Opt_Error) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Opt_Error_Group
        (Node : Bare_Grammar_Opt_Error_Group; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Opt_Error_Group is
      begin
         if Node = null then
            return No_Entity_Grammar_Opt_Error_Group;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Opt_Error_Group) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Opt_Group
        (Node : Bare_Grammar_Opt_Group; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Opt_Group is
      begin
         if Node = null then
            return No_Entity_Grammar_Opt_Group;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Opt_Group) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Or_Expr
        (Node : Bare_Grammar_Or_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Or_Expr is
      begin
         if Node = null then
            return No_Entity_Grammar_Or_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Or_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Predicate
        (Node : Bare_Grammar_Predicate; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Predicate is
      begin
         if Node = null then
            return No_Entity_Grammar_Predicate;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Predicate) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Rule_Decl
        (Node : Bare_Grammar_Rule_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Rule_Decl is
      begin
         if Node = null then
            return No_Entity_Grammar_Rule_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Rule_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Rule_Ref
        (Node : Bare_Grammar_Rule_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Rule_Ref is
      begin
         if Node = null then
            return No_Entity_Grammar_Rule_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Rule_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Skip
        (Node : Bare_Grammar_Skip; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Skip is
      begin
         if Node = null then
            return No_Entity_Grammar_Skip;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Skip) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Grammar_Stop_Cut
        (Node : Bare_Grammar_Stop_Cut; Info : Internal_Entity_Info)
         return Internal_Entity_Grammar_Stop_Cut is
      begin
         if Node = null then
            return No_Entity_Grammar_Stop_Cut;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Grammar_Stop_Cut) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_If_Expr
        (Node : Bare_If_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_If_Expr is
      begin
         if Node = null then
            return No_Entity_If_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_If_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Import
        (Node : Bare_Import; Info : Internal_Entity_Info)
         return Internal_Entity_Import is
      begin
         if Node = null then
            return No_Entity_Import;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Import) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Import_List
        (Node : Bare_Import_List; Info : Internal_Entity_Info)
         return Internal_Entity_Import_List is
      begin
         if Node = null then
            return No_Entity_Import_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Import_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Instantiated_Generic_Type
        (Node : Bare_Instantiated_Generic_Type; Info : Internal_Entity_Info)
         return Internal_Entity_Instantiated_Generic_Type is
      begin
         if Node = null then
            return No_Entity_Instantiated_Generic_Type;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Instantiated_Generic_Type) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Isa
        (Node : Bare_Isa; Info : Internal_Entity_Info)
         return Internal_Entity_Isa is
      begin
         if Node = null then
            return No_Entity_Isa;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Isa) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Type_Ref_List
        (Node : Bare_Type_Ref_List; Info : Internal_Entity_Info)
         return Internal_Entity_Type_Ref_List is
      begin
         if Node = null then
            return No_Entity_Type_Ref_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Type_Ref_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Isa_List
        (Node : Bare_Isa_List; Info : Internal_Entity_Info)
         return Internal_Entity_Isa_List is
      begin
         if Node = null then
            return No_Entity_Isa_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Isa_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Keep_Expr
        (Node : Bare_Keep_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Keep_Expr is
      begin
         if Node = null then
            return No_Entity_Keep_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Keep_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lambda_Arg_Decl
        (Node : Bare_Lambda_Arg_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Lambda_Arg_Decl is
      begin
         if Node = null then
            return No_Entity_Lambda_Arg_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lambda_Arg_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lambda_Arg_Decl_List
        (Node : Bare_Lambda_Arg_Decl_List; Info : Internal_Entity_Info)
         return Internal_Entity_Lambda_Arg_Decl_List is
      begin
         if Node = null then
            return No_Entity_Lambda_Arg_Decl_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lambda_Arg_Decl_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lambda_Expr
        (Node : Bare_Lambda_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Lambda_Expr is
      begin
         if Node = null then
            return No_Entity_Lambda_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lambda_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Langkit_Root
        (Node : Bare_Langkit_Root; Info : Internal_Entity_Info)
         return Internal_Entity_Langkit_Root is
      begin
         if Node = null then
            return No_Entity_Langkit_Root;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Langkit_Root) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Case_Rule
        (Node : Bare_Lexer_Case_Rule; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Case_Rule is
      begin
         if Node = null then
            return No_Entity_Lexer_Case_Rule;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Case_Rule) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Case_Rule_Cond_Alt
        (Node : Bare_Lexer_Case_Rule_Cond_Alt; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Case_Rule_Cond_Alt is
      begin
         if Node = null then
            return No_Entity_Lexer_Case_Rule_Cond_Alt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Case_Rule_Cond_Alt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Case_Rule_Default_Alt
        (Node : Bare_Lexer_Case_Rule_Default_Alt; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Case_Rule_Default_Alt is
      begin
         if Node = null then
            return No_Entity_Lexer_Case_Rule_Default_Alt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Case_Rule_Default_Alt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Case_Rule_Send
        (Node : Bare_Lexer_Case_Rule_Send; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Case_Rule_Send is
      begin
         if Node = null then
            return No_Entity_Lexer_Case_Rule_Send;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Case_Rule_Send) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Decl
        (Node : Bare_Lexer_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Decl is
      begin
         if Node = null then
            return No_Entity_Lexer_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Lexer_Family_Decl
        (Node : Bare_Lexer_Family_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Lexer_Family_Decl is
      begin
         if Node = null then
            return No_Entity_Lexer_Family_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Lexer_Family_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_List_Kind
        (Node : Bare_List_Kind; Info : Internal_Entity_Info)
         return Internal_Entity_List_Kind is
      begin
         if Node = null then
            return No_Entity_List_Kind;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_List_Kind) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_List_Kind_One
        (Node : Bare_List_Kind_One; Info : Internal_Entity_Info)
         return Internal_Entity_List_Kind_One is
      begin
         if Node = null then
            return No_Entity_List_Kind_One;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_List_Kind_One) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_List_Kind_Zero
        (Node : Bare_List_Kind_Zero; Info : Internal_Entity_Info)
         return Internal_Entity_List_Kind_Zero is
      begin
         if Node = null then
            return No_Entity_List_Kind_Zero;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_List_Kind_Zero) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Logic_Expr
        (Node : Bare_Logic_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Logic_Expr is
      begin
         if Node = null then
            return No_Entity_Logic_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Logic_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Match_Branch
        (Node : Bare_Match_Branch; Info : Internal_Entity_Info)
         return Internal_Entity_Match_Branch is
      begin
         if Node = null then
            return No_Entity_Match_Branch;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Match_Branch) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Match_Branch_List
        (Node : Bare_Match_Branch_List; Info : Internal_Entity_Info)
         return Internal_Entity_Match_Branch_List is
      begin
         if Node = null then
            return No_Entity_Match_Branch_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Match_Branch_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Match_Expr
        (Node : Bare_Match_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Match_Expr is
      begin
         if Node = null then
            return No_Entity_Match_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Match_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Match_Val_Decl
        (Node : Bare_Match_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Match_Val_Decl is
      begin
         if Node = null then
            return No_Entity_Match_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Match_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Module_Ref_Id
        (Node : Bare_Module_Ref_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Module_Ref_Id is
      begin
         if Node = null then
            return No_Entity_Module_Ref_Id;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Module_Ref_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Node_Decl
        (Node : Bare_Node_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Node_Decl is
      begin
         if Node = null then
            return No_Entity_Node_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Node_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Not_Expr
        (Node : Bare_Not_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Not_Expr is
      begin
         if Node = null then
            return No_Entity_Not_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Not_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Null_Cond_Dotted_Name
        (Node : Bare_Null_Cond_Dotted_Name; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Cond_Dotted_Name is
      begin
         if Node = null then
            return No_Entity_Null_Cond_Dotted_Name;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Cond_Dotted_Name) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Subscript_Expr
        (Node : Bare_Subscript_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Subscript_Expr is
      begin
         if Node = null then
            return No_Entity_Subscript_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Subscript_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Null_Cond_Subscript_Expr
        (Node : Bare_Null_Cond_Subscript_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Cond_Subscript_Expr is
      begin
         if Node = null then
            return No_Entity_Null_Cond_Subscript_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Cond_Subscript_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Null_Lit
        (Node : Bare_Null_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Null_Lit is
      begin
         if Node = null then
            return No_Entity_Null_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Null_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Num_Lit
        (Node : Bare_Num_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Num_Lit is
      begin
         if Node = null then
            return No_Entity_Num_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Num_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op
        (Node : Bare_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Op is
      begin
         if Node = null then
            return No_Entity_Op;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Amp
        (Node : Bare_Op_Amp; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Amp is
      begin
         if Node = null then
            return No_Entity_Op_Amp;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Amp) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_And
        (Node : Bare_Op_And; Info : Internal_Entity_Info)
         return Internal_Entity_Op_And is
      begin
         if Node = null then
            return No_Entity_Op_And;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_And) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Div
        (Node : Bare_Op_Div; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Div is
      begin
         if Node = null then
            return No_Entity_Op_Div;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Div) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Eq
        (Node : Bare_Op_Eq; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Eq is
      begin
         if Node = null then
            return No_Entity_Op_Eq;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Eq) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Gt
        (Node : Bare_Op_Gt; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Gt is
      begin
         if Node = null then
            return No_Entity_Op_Gt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Gt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Gte
        (Node : Bare_Op_Gte; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Gte is
      begin
         if Node = null then
            return No_Entity_Op_Gte;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Gte) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Lt
        (Node : Bare_Op_Lt; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Lt is
      begin
         if Node = null then
            return No_Entity_Op_Lt;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Lt) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Lte
        (Node : Bare_Op_Lte; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Lte is
      begin
         if Node = null then
            return No_Entity_Op_Lte;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Lte) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Minus
        (Node : Bare_Op_Minus; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Minus is
      begin
         if Node = null then
            return No_Entity_Op_Minus;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Minus) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Mult
        (Node : Bare_Op_Mult; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Mult is
      begin
         if Node = null then
            return No_Entity_Op_Mult;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Mult) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Ne
        (Node : Bare_Op_Ne; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Ne is
      begin
         if Node = null then
            return No_Entity_Op_Ne;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Ne) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Or
        (Node : Bare_Op_Or; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Or is
      begin
         if Node = null then
            return No_Entity_Op_Or;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Or) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Or_Int
        (Node : Bare_Op_Or_Int; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Or_Int is
      begin
         if Node = null then
            return No_Entity_Op_Or_Int;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Or_Int) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Op_Plus
        (Node : Bare_Op_Plus; Info : Internal_Entity_Info)
         return Internal_Entity_Op_Plus is
      begin
         if Node = null then
            return No_Entity_Op_Plus;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Op_Plus) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Param
        (Node : Bare_Param; Info : Internal_Entity_Info)
         return Internal_Entity_Param is
      begin
         if Node = null then
            return No_Entity_Param;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Param) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Param_List
        (Node : Bare_Param_List; Info : Internal_Entity_Info)
         return Internal_Entity_Param_List is
      begin
         if Node = null then
            return No_Entity_Param_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Param_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Paren_Expr
        (Node : Bare_Paren_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Paren_Expr is
      begin
         if Node = null then
            return No_Entity_Paren_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Paren_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Parse_Node_Expr
        (Node : Bare_Parse_Node_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Parse_Node_Expr is
      begin
         if Node = null then
            return No_Entity_Parse_Node_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Parse_Node_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Single_Line_String_Lit
        (Node : Bare_Single_Line_String_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Single_Line_String_Lit is
      begin
         if Node = null then
            return No_Entity_Single_Line_String_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Single_Line_String_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Pattern_Single_Line_String_Lit
        (Node : Bare_Pattern_Single_Line_String_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Pattern_Single_Line_String_Lit is
      begin
         if Node = null then
            return No_Entity_Pattern_Single_Line_String_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Pattern_Single_Line_String_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Raise_Expr
        (Node : Bare_Raise_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Raise_Expr is
      begin
         if Node = null then
            return No_Entity_Raise_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Raise_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Ref_Id
        (Node : Bare_Ref_Id; Info : Internal_Entity_Info)
         return Internal_Entity_Ref_Id is
      begin
         if Node = null then
            return No_Entity_Ref_Id;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ref_Id) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Ref_Id_List
        (Node : Bare_Ref_Id_List; Info : Internal_Entity_Info)
         return Internal_Entity_Ref_Id_List is
      begin
         if Node = null then
            return No_Entity_Ref_Id_List;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Ref_Id_List) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Self_Decl
        (Node : Bare_Self_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Self_Decl is
      begin
         if Node = null then
            return No_Entity_Self_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Self_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Simple_Type_Ref
        (Node : Bare_Simple_Type_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Simple_Type_Ref is
      begin
         if Node = null then
            return No_Entity_Simple_Type_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Simple_Type_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Struct_Decl
        (Node : Bare_Struct_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Struct_Decl is
      begin
         if Node = null then
            return No_Entity_Struct_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Struct_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Synthetic_Lexer_Decl
        (Node : Bare_Synthetic_Lexer_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Synthetic_Lexer_Decl is
      begin
         if Node = null then
            return No_Entity_Synthetic_Lexer_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Synthetic_Lexer_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_Lit
        (Node : Bare_Token_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Token_Lit is
      begin
         if Node = null then
            return No_Entity_Token_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_No_Case_Lit
        (Node : Bare_Token_No_Case_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Token_No_Case_Lit is
      begin
         if Node = null then
            return No_Entity_Token_No_Case_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_No_Case_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_Pattern_Lit
        (Node : Bare_Token_Pattern_Lit; Info : Internal_Entity_Info)
         return Internal_Entity_Token_Pattern_Lit is
      begin
         if Node = null then
            return No_Entity_Token_Pattern_Lit;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_Pattern_Lit) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Token_Ref
        (Node : Bare_Token_Ref; Info : Internal_Entity_Info)
         return Internal_Entity_Token_Ref is
      begin
         if Node = null then
            return No_Entity_Token_Ref;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Token_Ref) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Trait_Decl
        (Node : Bare_Trait_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Trait_Decl is
      begin
         if Node = null then
            return No_Entity_Trait_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Trait_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Try_Expr
        (Node : Bare_Try_Expr; Info : Internal_Entity_Info)
         return Internal_Entity_Try_Expr is
      begin
         if Node = null then
            return No_Entity_Try_Expr;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Try_Expr) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Un_Op
        (Node : Bare_Un_Op; Info : Internal_Entity_Info)
         return Internal_Entity_Un_Op is
      begin
         if Node = null then
            return No_Entity_Un_Op;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Un_Op) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Val_Decl
        (Node : Bare_Val_Decl; Info : Internal_Entity_Info)
         return Internal_Entity_Val_Decl is
      begin
         if Node = null then
            return No_Entity_Val_Decl;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Val_Decl) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   



      function Create_Internal_Entity_Var_Bind
        (Node : Bare_Var_Bind; Info : Internal_Entity_Info)
         return Internal_Entity_Var_Bind is
      begin
         if Node = null then
            return No_Entity_Var_Bind;
         end if;
         return (Node => Node, Info => Info);
      end;



   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Entity_Var_Bind) return String is
         pragma Warnings (On, "referenced");
      begin
            return Image (Entity'(Node => R.Node, Info => R.Info));
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Env_Assoc) is
      begin
               Inc_Ref (R.Dest_Env);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Env_Assoc) is
      begin
               Dec_Ref (R.Dest_Env);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Env_Assoc) return Boolean is
      begin
         return L.Key = R.Key and then L.Value = R.Value and then Equivalent (L.Dest_Env, R.Dest_Env) and then L.Metadata = R.Metadata;
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Env_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Dest_Env => " & Trace_Image (R.Dest_Env)
                        & ", "
                     & "Metadata => " & Trace_Image (R.Metadata)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Env_Kv) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Formal_Param) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Formal_Name => " & Trace_Image (R.Formal_Name)
                        & ", "
                     & "Formal_Type => " & Trace_Image (R.Formal_Type)
                        & ", "
                     & "Default_Value => " & Trace_Image (R.Default_Value)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Generic_Param_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Formal => " & Trace_Image (R.Formal)
                        & ", "
                     & "Actual => " & Trace_Image (R.Actual)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Semantic_Result) is
      begin
               Inc_Ref (R.Error_Message);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Semantic_Result) is
      begin
               Dec_Ref (R.Error_Message);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Semantic_Result) return Boolean is
      begin
         return L.Node = R.Node and then L.Result_Type = R.Result_Type and then L.Result_Decl = R.Result_Decl and then L.Has_Error = R.Has_Error and then Equivalent (L.Error_Message, R.Error_Message) and then L.Exempt_Analysis = R.Exempt_Analysis;
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Semantic_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Node => " & Trace_Image (R.Node)
                        & ", "
                     & "Result_Type => " & Trace_Image (R.Result_Type)
                        & ", "
                     & "Result_Decl => " & Trace_Image (R.Result_Decl)
                        & ", "
                     & "Has_Error => " & Trace_Image (R.Has_Error)
                        & ", "
                     & "Error_Message => " & Trace_Image (R.Error_Message)
                        & ", "
                     & "Exempt_Analysis => " & Trace_Image (R.Exempt_Analysis)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Infer_Instantiation) is
      begin
               Inc_Ref (R.Result);
               Inc_Ref (R.Error);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Infer_Instantiation) is
      begin
               Dec_Ref (R.Result);
               Dec_Ref (R.Error);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Infer_Instantiation) return Boolean is
      begin
         return Equivalent (L.Result, R.Result) and then Equivalent (L.Error, R.Error);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Infer_Instantiation) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Result => " & Trace_Image (R.Result)
                        & ", "
                     & "Error => " & Trace_Image (R.Error)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Inner_Env_Assoc) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Key => " & Trace_Image (R.Key)
                        & ", "
                     & "Value => " & Trace_Image (R.Value)
                        & ", "
                     & "Rebindings => " & Trace_Image (R.Rebindings)
                        & ", "
                     & "Metadata => " & Trace_Image (R.Metadata)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Logic_Context) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Ref_Node => " & Trace_Image (R.Ref_Node)
                        & ", "
                     & "Decl_Node => " & Trace_Image (R.Decl_Node)
               & ")");
      end Trace_Image;


   

   




   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Param_Match) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Has_Matched => " & Trace_Image (R.Has_Matched)
                        & ", "
                     & "Actual => " & Trace_Image (R.Actual)
                        & ", "
                     & "Formal => " & Trace_Image (R.Formal)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Solver_Diagnostic) is
      begin
               Inc_Ref (R.Message_Template);
               Inc_Ref (R.Args);
               Inc_Ref (R.Contexts);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Solver_Diagnostic) is
      begin
               Dec_Ref (R.Message_Template);
               Dec_Ref (R.Args);
               Dec_Ref (R.Contexts);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Solver_Diagnostic) return Boolean is
      begin
         return Equivalent (L.Message_Template, R.Message_Template) and then Equivalent (L.Args, R.Args) and then L.Location = R.Location and then Equivalent (L.Contexts, R.Contexts) and then L.Round = R.Round;
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Solver_Diagnostic) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Message_Template => " & Trace_Image (R.Message_Template)
                        & ", "
                     & "Args => " & Trace_Image (R.Args)
                        & ", "
                     & "Location => " & Trace_Image (R.Location)
                        & ", "
                     & "Contexts => " & Trace_Image (R.Contexts)
                        & ", "
                     & "Round => " & Trace_Image (R.Round)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Solver_Result) is
      begin
               Inc_Ref (R.Diagnostics);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Solver_Result) is
      begin
               Dec_Ref (R.Diagnostics);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Solver_Result) return Boolean is
      begin
         return L.Success = R.Success and then Equivalent (L.Diagnostics, R.Diagnostics);
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Solver_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Success => " & Trace_Image (R.Success)
                        & ", "
                     & "Diagnostics => " & Trace_Image (R.Diagnostics)
               & ")");
      end Trace_Image;


   

   


      -------------
      -- Inc_Ref --
      -------------

      procedure Inc_Ref (R : Internal_Tree_Semantic_Result) is
      begin
               Inc_Ref (R.Results);
      end Inc_Ref;

      -------------
      -- Dec_Ref --
      -------------

      procedure Dec_Ref (R : in out Internal_Tree_Semantic_Result) is
      begin
               Dec_Ref (R.Results);
      end Dec_Ref;




      ----------------
      -- Equivalent --
      ----------------

      function Equivalent (L, R : Internal_Tree_Semantic_Result) return Boolean is
      begin
         return Equivalent (L.Results, R.Results) and then L.Has_Error = R.Has_Error;
      end Equivalent;


   


      -----------------
      -- Trace_Image --
      -----------------

      pragma Warnings (Off, "referenced");
      function Trace_Image (R : Internal_Tree_Semantic_Result) return String is
         pragma Warnings (On, "referenced");
      begin
            return
              ("("
                     & "Results => " & Trace_Image (R.Results)
                        & ", "
                     & "Has_Error => " & Trace_Image (R.Has_Error)
               & ")");
      end Trace_Image;



   
         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         

         



      

   --
   --  Primitives for Bare_Lkt_Node
   --

   





   







--# property-start LktNode.concat_tree_semantic_results D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:139
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Concat_Tree_Semantic_Results
  
  (Node : Bare_Lkt_Node
      ; Res : Internal_Tree_Semantic_Result_Array_Access
  )

   return Internal_Tree_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind res Res

   Property_Result : Internal_Tree_Semantic_Result;

      
            procedure Finalizer_Scope_6 with Inline_Always;
            procedure Finalizer_Scope_311 with Inline_Always;

      Fld : Internal_Semantic_Result_Array_Access;
Map_Result : Internal_Semantic_Result_Array_Access;
Fld_1 : Boolean;
Quantifier_Result : Boolean;
New_Struct : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Let_Result : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;

            procedure Finalizer_Scope_6 is
            begin
                     Dec_Ref (Map_Result);
                     Dec_Ref (New_Struct);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_6;
            procedure Finalizer_Scope_311 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_311;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 6 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 5 '<New TreeSemanticResult at parser.py:145>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:145
--# expr-start 2 '<AnyQuantifier at parser.py:147>' Quantifier_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:147








   

   

   
      Quantifier_Result := False;
   

   

      

      declare
         
         Collection : constant Internal_Tree_Semantic_Result_Array_Access := Res;
      begin
         for Item_1 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind r Item_1

            
      --# expr-start 1 '<FieldAccess for has_error at parser.py:147>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:147






   




Fld_1 := Item_1.Has_Error;
--# expr-done 1
      Quantifier_Result := Fld_1;
   

            
   --# end


               exit when Quantifier_Result;

         end loop;
      end;
   

   
   



--# expr-done 2
--# expr-start 4 '<Mapcat at parser.py:146>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:146








   

   

   
      declare
         Map_Result_Vec : Internal_Semantic_Result_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Tree_Semantic_Result_Array_Access := Res;
      begin
         for Item_0 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind r Item_0

            
         
         
      --# expr-start 3 '<FieldAccess for results at parser.py:146>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:146






   




Fld := Item_0.Results;
Inc_Ref (Fld);
--# expr-done 3
      

         for Item_To_Append of
               Fld.Items
         loop
               Inc_Ref (Item_To_Append);
            Internal_Semantic_Result_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end loop;

   
   
   

            
   --# end
      Finalizer_Scope_311;



         end loop;
      end;
   

   

         Map_Result := Create_Internal_Semantic_Result_Array
           (Items_Count => Natural (Internal_Semantic_Result_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Semantic_Result_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Semantic_Result_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Semantic_Result_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 4
Inc_Ref (Map_Result);
New_Struct := (Has_Error => Quantifier_Result, Results => Map_Result); 
--# expr-done 5
Let_Result := New_Struct; Inc_Ref (Let_Result);
--# end
--# expr-done 6

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_6;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_6;
                     Finalizer_Scope_311;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Concat_Tree_Semantic_Results;
--# end

   







--# property-start LktNode.new_tree_semantic_result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:150
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_New_Tree_Semantic_Result
  
  (Node : Bare_Lkt_Node
      ; Elements : Internal_Semantic_Result_Array_Access
  )

   return Internal_Tree_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind elements Elements

   Property_Result : Internal_Tree_Semantic_Result;

      
            procedure Finalizer_Scope_7 with Inline_Always;

      Fld : Boolean;
Quantifier_Result : Boolean;
New_Struct : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Let_Result : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;

            procedure Finalizer_Scope_7 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_7;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 10 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 9 '<New TreeSemanticResult at parser.py:152>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:152
--# expr-start 8 '<AnyQuantifier at parser.py:154>' Quantifier_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:154








   

   

   
      Quantifier_Result := False;
   

   

      

      declare
         
         Collection : constant Internal_Semantic_Result_Array_Access := Elements;
      begin
         for Item_2 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind e Item_2

            
      --# expr-start 7 '<FieldAccess for has_error at parser.py:154>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:154






   




Fld := Item_2.Has_Error;
--# expr-done 7
      Quantifier_Result := Fld;
   

            
   --# end


               exit when Quantifier_Result;

         end loop;
      end;
   

   
   



--# expr-done 8

Inc_Ref (Elements);
New_Struct := (Has_Error => Quantifier_Result, Results => Elements); 
--# expr-done 9
Let_Result := New_Struct; Inc_Ref (Let_Result);
--# end
--# expr-done 10

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_7;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_7;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_New_Tree_Semantic_Result;
--# end

   







--# property-start LktNode.root_get D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:157
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Root_Get
  
  (Node : Bare_Lkt_Node
      ; Entity_Name : Symbol_Type
  )

   return Internal_Entity_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind entity_name Entity_Name

   Property_Result : Internal_Entity_Decl;

      
            procedure Finalizer_Scope_8 with Inline_Always;

      Fld : Internal_Unit;
Fld_1 : Bare_Lkt_Node;
Fld_2 : Lexical_Env := Empty_Env;
Env_Get_Result : Internal_Entity;
Cast_Result : Internal_Entity_Decl;
Let_Result : Internal_Entity_Decl;

            procedure Finalizer_Scope_8 is
            begin
                     Dec_Ref (Fld_2);
            end Finalizer_Scope_8;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 16 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 15 '<Cast to Decl at parser.py:160>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:160





--# expr-start 14 '<EnvGet at parser.py:159>' Env_Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:159
--# expr-start 13 '<FieldAccess for node_env at parser.py:159>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:159



--# expr-start 12 '<FieldAccess for root at parser.py:159>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:159



--# expr-start 11 '<FieldAccess for unit at parser.py:159>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:159






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Unit (Node => Self);
--# expr-done 11


   

      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Fld.Ast_Root;
--# expr-done 12


   

      if Fld_1 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Liblktlang.Implementation.Node_Env (Node => Fld_1);
--# expr-done 13



Env_Get_Result := AST_Envs.Get_First (Self => Fld_2, Key => Thin (Entity_Name), Lookup_Kind => To_Lookup_Kind_Type (Recursive), Categories => (Nocat => True)); 
--# expr-done 14



   if Env_Get_Result.Node = null
      or else Env_Get_Result.Node.Kind in Lkt_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Decl
        (Node => Env_Get_Result.Node,
         Info => Env_Get_Result.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 15
Let_Result := Cast_Result; 
--# end
--# expr-done 16

         Property_Result := Let_Result;
         
   --# end
      Finalizer_Scope_8;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_8;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Root_Get;
--# end

   







--# property-start LktNode.get_builtin_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:162
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Get_Builtin_Type
  
  (Node : Bare_Lkt_Node
      ; Entity_Name : Symbol_Type
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind entity_name Entity_Name

   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Fld : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Named_Type_Decl;
Let_Result : Internal_Entity_Named_Type_Decl;


      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Get_Builtin_Type,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) := (Kind => Mmz_Symbol_Type,
                                      As_Symbol_Type => Entity_Name);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Internal_Entity_Named_Type_Decl;

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 19 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 18 '<Cast to NamedTypeDecl at parser.py:164>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:164





--# expr-start 17 '<FieldAccess for root_get at parser.py:164>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:164






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start LktNode.root_get
Fld := Liblktlang.Implementation.Lkt_Node_P_Root_Get (Node => Self, Entity_Name => Entity_Name);
--# end
--# expr-done 17



   if Fld.Node = null
      or else Fld.Node.Kind in Lkt_Named_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Named_Type_Decl
        (Node => Fld.Node,
         Info => Fld.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 18
Let_Result := Cast_Result; 
--# end
--# expr-done 19

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Internal_Entity_Named_Type_Decl,
                        As_Internal_Entity_Named_Type_Decl => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);

         end if;
      end if;



   return Property_Result;
end Lkt_Node_P_Get_Builtin_Type;
--# end

   







--# property-start LktNode.get_builtin_gen_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:166
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Get_Builtin_Gen_Decl
  
  (Node : Bare_Lkt_Node
      ; Entity_Name : Symbol_Type
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind entity_name Entity_Name

   Property_Result : Internal_Entity_Generic_Decl;

      

      Fld : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Generic_Decl;
Let_Result : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 22 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 21 '<Cast to GenericDecl at parser.py:168>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:168





--# expr-start 20 '<FieldAccess for root_get at parser.py:168>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:168






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start LktNode.root_get
Fld := Liblktlang.Implementation.Lkt_Node_P_Root_Get (Node => Self, Entity_Name => Entity_Name);
--# end
--# expr-done 20



   if Fld.Node = null
      or else Fld.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => Fld.Node,
         Info => Fld.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 21
Let_Result := Cast_Result; 
--# end
--# expr-done 22

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Get_Builtin_Gen_Decl;
--# end

   







--# property-start LktNode.node_gen_trait D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:170
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Node_Gen_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 24 '<FieldAccess for get_builtin_gen_decl at parser.py:171>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:171






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 23 '<Symbol Node at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Node); 
--# expr-done 23
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 24

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Node_Gen_Trait;
--# end

   







--# property-start LktNode.node_trait D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:175
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Node_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Trait_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Trait_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Trait_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 27 '<Cast to TraitDecl at parser.py:176>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:176





--# expr-start 26 '<FieldAccess for decl at parser.py:176>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:176



--# expr-start 25 '<FieldAccess for node_gen_trait at parser.py:176>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:176






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.node_gen_trait
Fld := Liblktlang.Implementation.Lkt_Node_P_Node_Gen_Trait (Node => Self);
--# end
--# expr-done 25


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 26



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Trait_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Trait_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Trait_Decl;
   end if;


--# expr-done 27

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Node_Trait;
--# end

   







--# property-start LktNode.token_node_trait D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:180
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Token_Node_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 29 '<FieldAccess for get_builtin_type at parser.py:181>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:181






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 28 '<Symbol TokenNode at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Tokennode); 
--# expr-done 28
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 29

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Token_Node_Trait;
--# end

   







--# property-start LktNode.error_node_trait D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:185
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Error_Node_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 31 '<FieldAccess for get_builtin_type at parser.py:186>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:186






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 30 '<Symbol ErrorNode at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Errornode); 
--# expr-done 30
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 31

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Error_Node_Trait;
--# end

   







--# property-start LktNode.char_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:190
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Char_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 33 '<FieldAccess for get_builtin_type at parser.py:191>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:191






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 32 '<Symbol Char at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Char); 
--# expr-done 32
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 33

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Char_Type;
--# end

   







--# property-start LktNode.int_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:195
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Int_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 35 '<FieldAccess for get_builtin_type at parser.py:196>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:196






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 34 '<Symbol Int at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Int); 
--# expr-done 34
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 35

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Int_Type;
--# end

   







--# property-start LktNode.bool_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:200
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Bool_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 37 '<FieldAccess for get_builtin_type at parser.py:201>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:201






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 36 '<Symbol Bool at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Bool); 
--# expr-done 36
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 37

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Bool_Type;
--# end

   







--# property-start LktNode.bigint_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:205
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Bigint_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 39 '<FieldAccess for get_builtin_type at parser.py:206>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:206






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 38 '<Symbol BigInt at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Bigint); 
--# expr-done 38
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 39

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Bigint_Type;
--# end

   







--# property-start LktNode.string_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:210
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_String_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 41 '<FieldAccess for get_builtin_type at parser.py:211>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:211






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 40 '<Symbol String at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_String); 
--# expr-done 40
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 41

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_String_Type;
--# end

   







--# property-start LktNode.symbol_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:215
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Symbol_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 43 '<FieldAccess for get_builtin_type at parser.py:216>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:216






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 42 '<Symbol Symbol at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Symbol); 
--# expr-done 42
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 43

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Symbol_Type;
--# end

   







--# property-start LktNode.property_error_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:220
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Property_Error_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 45 '<FieldAccess for get_builtin_type at parser.py:221>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:221






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 44 '<Symbol PropertyError at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Propertyerror); 
--# expr-done 44
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 45

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Property_Error_Type;
--# end

   







--# property-start LktNode.regexp_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:225
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Regexp_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 47 '<FieldAccess for get_builtin_type at parser.py:226>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:226






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 46 '<Symbol Regexp at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Regexp); 
--# expr-done 46
--# property-call-start LktNode.get_builtin_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Type (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 47

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Regexp_Type;
--# end

   







--# property-start LktNode.array_gen_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:230
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Array_Gen_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 49 '<FieldAccess for get_builtin_gen_decl at parser.py:231>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:231






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 48 '<Symbol Array at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Array); 
--# expr-done 48
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 49

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Array_Gen_Type;
--# end

   







--# property-start LktNode.array_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:235
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Array_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 52 '<Cast to NamedTypeDecl at parser.py:236>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:236





--# expr-start 51 '<FieldAccess for decl at parser.py:236>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:236



--# expr-start 50 '<FieldAccess for array_gen_type at parser.py:236>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:236






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.array_gen_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Array_Gen_Type (Node => Self);
--# end
--# expr-done 50


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 51



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Named_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Named_Type_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Named_Type_Decl;
   end if;


--# expr-done 52

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Array_Type;
--# end

   







--# property-start LktNode.astlist_gen_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:240
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Astlist_Gen_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 54 '<FieldAccess for get_builtin_gen_decl at parser.py:241>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:241






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 53 '<Symbol ASTList at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Astlist); 
--# expr-done 53
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 54

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Astlist_Gen_Type;
--# end

   







--# property-start LktNode.astlist_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:246
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Astlist_Type
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Named_Type_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Named_Type_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Named_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 57 '<Cast to NamedTypeDecl at parser.py:247>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:247





--# expr-start 56 '<FieldAccess for decl at parser.py:247>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:247



--# expr-start 55 '<FieldAccess for astlist_gen_type at parser.py:247>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:247






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.astlist_gen_type
Fld := Liblktlang.Implementation.Lkt_Node_P_Astlist_Gen_Type (Node => Self);
--# end
--# expr-done 55


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 56



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Named_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Named_Type_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Named_Type_Decl;
   end if;


--# expr-done 57

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Astlist_Type;
--# end

   







--# property-start LktNode.iterator_gen_trait D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:252
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Iterator_Gen_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 59 '<FieldAccess for get_builtin_gen_decl at parser.py:253>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:253






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 58 '<Symbol Iterator at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Iterator); 
--# expr-done 58
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 59

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Iterator_Gen_Trait;
--# end

   







--# property-start LktNode.iterator_trait D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:258
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Iterator_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Trait_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Trait_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Trait_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 62 '<Cast to TraitDecl at parser.py:259>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:259





--# expr-start 61 '<FieldAccess for decl at parser.py:259>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:259



--# expr-start 60 '<FieldAccess for iterator_gen_trait at parser.py:259>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:259






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.iterator_gen_trait
Fld := Liblktlang.Implementation.Lkt_Node_P_Iterator_Gen_Trait (Node => Self);
--# end
--# expr-done 60


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 61



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Trait_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Trait_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Trait_Decl;
   end if;


--# expr-done 62

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Iterator_Trait;
--# end

   







--# property-start LktNode.analysis_unit_gen_trait D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:264
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Analysis_Unit_Gen_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Sym : Symbol_Type;
Fld : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 64 '<FieldAccess for get_builtin_gen_decl at parser.py:265>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:265






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 63 '<Symbol AnalysisUnit at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Analysisunit); 
--# expr-done 63
--# property-call-start LktNode.get_builtin_gen_decl
Fld := Liblktlang.Implementation.Lkt_Node_P_Get_Builtin_Gen_Decl (Node => Self, Entity_Name => Sym);
--# end
--# expr-done 64

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Analysis_Unit_Gen_Trait;
--# end

   







--# property-start LktNode.analysis_unit_trait D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:270
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Analysis_Unit_Trait
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Entity_Trait_Decl
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Trait_Decl;

      

      Fld : Internal_Entity_Generic_Decl;
Fld_1 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Trait_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 67 '<Cast to TraitDecl at parser.py:271>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:271





--# expr-start 66 '<FieldAccess for decl at parser.py:271>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:271



--# expr-start 65 '<FieldAccess for analysis_unit_gen_trait at parser.py:271>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:271






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.analysis_unit_gen_trait
Fld := Liblktlang.Implementation.Lkt_Node_P_Analysis_Unit_Gen_Trait (Node => Self);
--# end
--# expr-done 65


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Create_Internal_Entity_Decl (Node => Fld.Node.Generic_Decl_F_Decl, Info => Fld.Info);
--# expr-done 66



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Trait_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Trait_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Cast_Result := No_Entity_Trait_Decl;
   end if;


--# expr-done 67

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Analysis_Unit_Trait;
--# end

   







--# property-start LktNode.function_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:297
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Function_Type
  
  (Node : Bare_Lkt_Node
      ; Args_Types : Bare_Type_Decl_Array_Access
      ; Return_Type : Bare_Type_Decl
  )

   return Bare_Function_Type
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind args_types Args_Types
   --# bind return_type Return_Type

   Property_Result : Bare_Function_Type;

      

      Fld : Internal_Unit;
Fld_1 : Bare_Lkt_Node;
Cast_Result : Bare_Langkit_Root;
Fld_2 : Bare_Function_Type;
Let_Result : Bare_Function_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 72 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 71 '<FieldAccess for function_type_helper at parser.py:310>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:310



--# expr-start 70 '<Cast to LangkitRoot at parser.py:310>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:310





--# expr-start 69 '<FieldAccess for root at parser.py:310>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:310



--# expr-start 68 '<FieldAccess for unit at parser.py:310>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:310






   

      if Return_Type = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Unit (Node => Return_Type);
--# expr-done 68


   

      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Fld.Ast_Root;
--# expr-done 69



   if Fld_1 = null
      or else Fld_1.Kind in Lkt_Langkit_Root_Range
   then
      
      Cast_Result := Fld_1;

   else
         Cast_Result := No_Bare_Lkt_Node;
   end if;


--# expr-done 70


   

      if Cast_Result = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;





--# property-call-start LangkitRoot.function_type_helper
Fld_2 := Liblktlang.Implementation.Langkit_Root_P_Function_Type_Helper (Node => Cast_Result, Args_Types => Args_Types, Return_Type => Return_Type);
--# end
--# expr-done 71
Let_Result := Fld_2; 
--# end
--# expr-done 72

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Function_Type;
--# end

   







--# property-start LktNode.exempt D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:314
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Exempt
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Semantic_Result;

      
            procedure Finalizer_Scope_34 with Inline_Always;

      New_Struct : Internal_Semantic_Result := No_Semantic_Result;
Let_Result : Internal_Semantic_Result := No_Semantic_Result;

            procedure Finalizer_Scope_34 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_34;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 74 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 73 '<New SemanticResult at parser.py:319>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:319






Inc_Ref (Empty_String);
New_Struct := (Error_Message => Empty_String, Exempt_Analysis => True, Has_Error => True, Node => Self, Result_Decl => No_Entity_Decl, Result_Type => No_Entity_Type_Decl); 
--# expr-done 73
Let_Result := New_Struct; Inc_Ref (Let_Result);
--# end
--# expr-done 74

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_34;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_34;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Exempt;
--# end

   







--# property-start LktNode.error D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:323
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Error
  
  (Node : Bare_Lkt_Node
      ; Message : String_Type
  )

   return Internal_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind message Message

   Property_Result : Internal_Semantic_Result;

      
            procedure Finalizer_Scope_35 with Inline_Always;

      New_Struct : Internal_Semantic_Result := No_Semantic_Result;
Let_Result : Internal_Semantic_Result := No_Semantic_Result;

            procedure Finalizer_Scope_35 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_35;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 76 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 75 '<New SemanticResult at parser.py:328>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:328






Inc_Ref (Message);
New_Struct := (Error_Message => Message, Exempt_Analysis => False, Has_Error => True, Node => Self, Result_Decl => No_Entity_Decl, Result_Type => No_Entity_Type_Decl); 
--# expr-done 75
Let_Result := New_Struct; Inc_Ref (Let_Result);
--# end
--# expr-done 76

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_35;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_35;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Error;
--# end

   







--# property-start LktNode.internal_error D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:332
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Internal_Error
  
  (Node : Bare_Lkt_Node
  )

   return Internal_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Semantic_Result;

      
            procedure Finalizer_Scope_36 with Inline_Always;

      Str : String_Type;
Fld : Internal_Semantic_Result := No_Semantic_Result;
Let_Result : Internal_Semantic_Result := No_Semantic_Result;

            procedure Finalizer_Scope_36 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Fld);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_36;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 79 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 78 '<FieldAccess for error at parser.py:337>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:337






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 77 '<String at parser.py:337>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:337
Str := Create_String (""); 
--# expr-done 77
--# property-call-start LktNode.error
Fld := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Self, Message => Str);
--# end
--# expr-done 78
Let_Result := Fld; Inc_Ref (Let_Result);
--# end
--# expr-done 79

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_36;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_36;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Internal_Error;
--# end

   







--# property-start LktNode.expected_type_error D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:339
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Expected_Type_Error
  
  (Node : Bare_Lkt_Node
      ; Expected : Internal_Entity_Type_Decl
      ; Got : String_Type
  )

   return Internal_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind expected Expected
   --# bind got Got

   Property_Result : Internal_Semantic_Result;

      
            procedure Finalizer_Scope_37 with Inline_Always;

      Str : String_Type;
Fld : String_Type;
Concat_Result : String_Type;
Str_1 : String_Type;
Concat_Result_1 : String_Type;
Concat_Result_2 : String_Type;
Fld_1 : Internal_Semantic_Result := No_Semantic_Result;
Let_Result : Internal_Semantic_Result := No_Semantic_Result;

            procedure Finalizer_Scope_37 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Fld);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Str_1);
                     Dec_Ref (Concat_Result_1);
                     Dec_Ref (Concat_Result_2);
                     Dec_Ref (Fld_1);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_37;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 84 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 83 '<FieldAccess for error at parser.py:348>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:348






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 80 '<String at parser.py:349>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:349
Str := Create_String ("Mismatched types: expected `"); 
--# expr-done 80
--# expr-start 81 '<FieldAccess for full_name at parser.py:350>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:350






   

      if Expected.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.full_name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Full_Name (Node => Expected.Node);
--# end
--# expr-done 81
Concat_Result := Concat_String (Str, Fld); 
--# expr-start 82 '<String at parser.py:351>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:351
Str_1 := Create_String ("`, got "); 
--# expr-done 82

Concat_Result_1 := Concat_String (Str_1, Got); 
Concat_Result_2 := Concat_String (Concat_Result, Concat_Result_1); 
--# property-call-start LktNode.error
Fld_1 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Self, Message => Concat_Result_2);
--# end
--# expr-done 83
Let_Result := Fld_1; Inc_Ref (Let_Result);
--# end
--# expr-done 84

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_37;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_37;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Expected_Type_Error;
--# end

   







--# property-start LktNode.type_mismatch_error D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:354
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Type_Mismatch_Error
  
  (Node : Bare_Lkt_Node
      ; Expected : Internal_Entity_Type_Decl
      ; Got : Internal_Entity_Type_Decl
  )

   return Internal_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind expected Expected
   --# bind got Got

   Property_Result : Internal_Semantic_Result;

      
            procedure Finalizer_Scope_38 with Inline_Always;

      Str : String_Type;
Fld : String_Type;
Concat_Result : String_Type;
Str_1 : String_Type;
Concat_Result_1 : String_Type;
Fld_1 : Internal_Semantic_Result := No_Semantic_Result;
Let_Result : Internal_Semantic_Result := No_Semantic_Result;

            procedure Finalizer_Scope_38 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Fld);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Str_1);
                     Dec_Ref (Concat_Result_1);
                     Dec_Ref (Fld_1);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_38;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 89 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 88 '<FieldAccess for expected_type_error at parser.py:364>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:364






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# expr-start 85 '<String at parser.py:366>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:366
Str := Create_String ("`"); 
--# expr-done 85
--# expr-start 86 '<FieldAccess for full_name at parser.py:366>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:366






   

      if Got.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.full_name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Full_Name (Node => Got.Node);
--# end
--# expr-done 86
Concat_Result := Concat_String (Str, Fld); 
--# expr-start 87 '<String at parser.py:366>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:366
Str_1 := Create_String ("`"); 
--# expr-done 87
Concat_Result_1 := Concat_String (Concat_Result, Str_1); 
--# property-call-start LktNode.expected_type_error
Fld_1 := Liblktlang.Implementation.Lkt_Node_P_Expected_Type_Error (Node => Self, Expected => Expected, Got => Concat_Result_1);
--# end
--# expr-done 88
Let_Result := Fld_1; Inc_Ref (Let_Result);
--# end
--# expr-done 89

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_38;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_38;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Type_Mismatch_Error;
--# end

   







--# property-start LktNode.check_legality_visitor D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:369
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Check_Legality_Visitor
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Semantic_Result_Array_Access;

      
            procedure Finalizer_Scope_39 with Inline_Always;
            procedure Finalizer_Scope_315 with Inline_Always;
            procedure Finalizer_Scope_316 with Inline_Always;

      Block_Var : Internal_Semantic_Result_Array_Access;
Var_Expr_1 : Internal_Entity;
Fld : Internal_Semantic_Result_Array_Access;
Fld_1 : Internal_Entity_Array_Access;
Fld_2 : Internal_Semantic_Result_Array_Access;
Scope_Result : Internal_Semantic_Result_Array_Access;
Result_Var : Internal_Semantic_Result_Array_Access;
Map_Result : Internal_Semantic_Result_Array_Access;
Concat_Result : Internal_Semantic_Result_Array_Access;
Let_Result : Internal_Semantic_Result_Array_Access;

            procedure Finalizer_Scope_39 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Block_Var);
                     Dec_Ref (Fld_1);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_39;
            procedure Finalizer_Scope_315 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
            end Finalizer_Scope_315;
            procedure Finalizer_Scope_316 is
            begin
                     Dec_Ref (Fld_2);
            end Finalizer_Scope_316;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 94 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 90 '<FieldAccess for check_legality at parser.py:375>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:375






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]LktNode.check_legality'
Fld := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Check_Legality (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 90
Block_Var := Fld; Inc_Ref (Block_Var);
--# bind own_results Block_Var

--# expr-start 93 '<Mapcat at parser.py:377>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:377








   

   --# expr-start 91 '<children at parser.py:377>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:377






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Liblktlang.Implementation.Children (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 91

   
      declare
         Map_Result_Vec : Internal_Semantic_Result_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Fld_1;
      begin
         for Item_3 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind c Item_3

            
         
         
      



Var_Expr_1 := Item_3;





if Var_Expr_1 /= No_Entity then
   



   --# scope-start


   

--# expr-start 92 '<FieldAccess for check_legality_visitor at parser.py:377>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:377






   

      if Var_Expr_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.check_legality_visitor
Fld_2 := Liblktlang.Implementation.Lkt_Node_P_Check_Legality_Visitor (Node => Var_Expr_1.Node, E_Info => Var_Expr_1.Info);
--# end
--# expr-done 92
Scope_Result := Fld_2;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_316;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var);


      

         for Item_To_Append of
               Result_Var.Items
         loop
               Inc_Ref (Item_To_Append);
            Internal_Semantic_Result_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end loop;

   
   
   

            
   --# end
      Finalizer_Scope_315;



         end loop;
      end;
   

   

         Map_Result := Create_Internal_Semantic_Result_Array
           (Items_Count => Natural (Internal_Semantic_Result_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Semantic_Result_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Semantic_Result_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Semantic_Result_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 93
Concat_Result := Concat (Block_Var, Map_Result); 
Let_Result := Concat_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 94

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_39;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_39;
                     Finalizer_Scope_315;
                     Finalizer_Scope_316;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Check_Legality_Visitor;
--# end

   







--# property-start LktNode.check_sem_recursive D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:380
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Check_Sem_Recursive
  
  (Node : Bare_Lkt_Node
      ; Exempted_Nodes : Bare_Lkt_Node_Array_Access
         := No_Bare_Lkt_Node_Array_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Tree_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind exempted_nodes Exempted_Nodes

   Property_Result : Internal_Tree_Semantic_Result;

      
            procedure Finalizer_Scope_40 with Inline_Always;
            procedure Finalizer_Scope_318 with Inline_Always;
            procedure Finalizer_Scope_319 with Inline_Always;
            procedure Finalizer_Scope_320 with Inline_Always;
            procedure Finalizer_Scope_322 with Inline_Always;

      Block_Var : Internal_Semantic_Result_Array_Access;
Block_Var_1 : Bare_Lkt_Node_Array_Access;
Block_Var_2 : Internal_Semantic_Result_Array_Access;
Block_Var_3 : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Var_Expr : Internal_Entity_Array_Access;
Var_Expr_2 : Internal_Entity;
Fld : Internal_Semantic_Result_Array_Access;
Fld_1 : Bare_Lkt_Node;
Fld_2 : Boolean;
Map_Result : Bare_Lkt_Node_Array_Access;
Concat_Result : Bare_Lkt_Node_Array_Access;
Fld_3 : Internal_Entity_Array_Access;
Fld_4 : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Scope_Result : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Result_Var : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Fld_5 : Bare_Lkt_Node;
Cast_Expr : Bare_Lkt_Node;
Cast_Result : Bare_Lkt_Node;
Is_Equal : Boolean;
Find_Result : Bare_Lkt_Node;
Is_Null : Boolean;
If_Result : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Map_Result_1 : Internal_Tree_Semantic_Result_Array_Access;
Scope_Result_1 : Internal_Tree_Semantic_Result_Array_Access;
Result_Var_1 : Internal_Tree_Semantic_Result_Array_Access;
Fld_6 : Internal_Semantic_Result_Array_Access;
Map_Result_2 : Internal_Semantic_Result_Array_Access;
Fld_7 : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Concat_Result_1 : Internal_Semantic_Result_Array_Access;
Fld_8 : Internal_Semantic_Result_Array_Access;
Concat_Result_2 : Internal_Semantic_Result_Array_Access;
Fld_9 : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Let_Result : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;

            procedure Finalizer_Scope_40 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Block_Var);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Block_Var_1);
                     Dec_Ref (Fld_3);
                     Dec_Ref (Scope_Result_1);
                     Dec_Ref (Result_Var_1);
                     Dec_Ref (Map_Result_2);
                     Dec_Ref (Block_Var_2);
                     Dec_Ref (Fld_7);
                     Dec_Ref (Block_Var_3);
                     Dec_Ref (Concat_Result_1);
                     Dec_Ref (Fld_8);
                     Dec_Ref (Concat_Result_2);
                     Dec_Ref (Fld_9);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_40;
            procedure Finalizer_Scope_318 is
            begin
                     Dec_Ref (Var_Expr);
                     Dec_Ref (Map_Result_1);
            end Finalizer_Scope_318;
            procedure Finalizer_Scope_319 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
                     Dec_Ref (If_Result);
            end Finalizer_Scope_319;
            procedure Finalizer_Scope_320 is
            begin
                     Dec_Ref (Fld_4);
            end Finalizer_Scope_320;
            procedure Finalizer_Scope_322 is
            begin
                     Dec_Ref (Fld_6);
            end Finalizer_Scope_322;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 112 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 95 '<FieldAccess for check_correctness_pre at parser.py:391>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:391






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]LktNode.check_correctness_pre'
Fld := Liblktlang.Implementation.Dispatcher_Lkt_Node_P_Check_Correctness_Pre (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 95
Block_Var := Fld; Inc_Ref (Block_Var);
--# bind pre_diagnostics Block_Var

--# expr-start 98 '<FilterMap at parser.py:392>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:392








   

   

   
      declare
         Map_Result_Vec : Bare_Lkt_Node_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Semantic_Result_Array_Access := Block_Var;
      begin
         for Item_4 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind d Item_4

            
         --# expr-start 96 '<FieldAccess for exempt_analysis at parser.py:393>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:393






   




Fld_2 := Item_4.Exempt_Analysis;
--# expr-done 96
         if Fld_2 then
            
         
      --# expr-start 97 '<FieldAccess for node at parser.py:393>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:393






   




Fld_1 := Item_4.Node;
--# expr-done 97
      

         declare
            Item_To_Append : constant Bare_Lkt_Node := Fld_1;
         begin
            Bare_Lkt_Node_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result := Create_Bare_Lkt_Node_Array
           (Items_Count => Natural (Bare_Lkt_Node_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Bare_Lkt_Node_Vectors.Get
              (Map_Result_Vec,
               I + Bare_Lkt_Node_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Bare_Lkt_Node_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 98
Concat_Result := Concat (Exempted_Nodes, Map_Result); 
Block_Var_1 := Concat_Result; Inc_Ref (Block_Var_1);
--# bind exemptions Block_Var_1
--# expr-start 108 '<Mapcat at parser.py:410>' Map_Result_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:410








   

   


--# expr-start 99 '<children at parser.py:397>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:397






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Liblktlang.Implementation.Children (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 99
Var_Expr := Fld_3;

      Inc_Ref (Var_Expr);




if Var_Expr /= No_Internal_Entity_Array_Type then
   



   --# scope-start


   --# bind children Var_Expr

--# expr-start 106 '<Map at parser.py:398>' Map_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:398








   

   

   
      declare
         Map_Result_1_Vec : Internal_Tree_Semantic_Result_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Var_Expr;
      begin
         for Item_6 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind child Item_6

            
         
         
      --# expr-start 105 '<If at parser.py:398>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:398



--# expr-start 103 '<IsNull at parser.py:406>' Is_Null D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:406
--# expr-start 102 '<Find at parser.py:406>' Find_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:406








   

   

   
      Find_Result := No_Bare_Lkt_Node;
   

   

      

      declare
         
         Collection : constant Bare_Lkt_Node_Array_Access := Block_Var_1;
      begin
         for Item_7 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind n Item_7

            
      --# expr-start 101 '<Eq at parser.py:406>' Is_Equal D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:406






Cast_Expr := Item_7; 



   
      Cast_Result := Cast_Expr;


--# expr-start 100 '<FieldAccess for node at parser.py:406>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:406






   




Fld_5 := Item_6.Node;
--# expr-done 100
Is_Equal := Cast_Result = Fld_5; 
--# expr-done 101
      if Is_Equal then
         Find_Result := Item_7; 
      end if;
   

            
   --# end


               exit when Is_Equal;

         end loop;
      end;
   

   
   



--# expr-done 102
Is_Null := Find_Result = null; 
--# expr-done 103
if Is_Null then
   



Var_Expr_2 := Item_6;





if Var_Expr_2 /= No_Entity then
   



   --# scope-start


   

--# expr-start 104 '<FieldAccess for check_sem_recursive at parser.py:407>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:407






   

      if Var_Expr_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start LktNode.check_sem_recursive
Fld_4 := Liblktlang.Implementation.Lkt_Node_P_Check_Sem_Recursive (Node => Var_Expr_2.Node, Exempted_Nodes => Block_Var_1, E_Info => Var_Expr_2.Info);
--# end
--# expr-done 104
Scope_Result := Fld_4;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_320;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Tree_Semantic_Result;
end if;

      Inc_Ref (Result_Var);


   If_Result := Result_Var;
else
   
   If_Result := No_Tree_Semantic_Result;
end if;

      Inc_Ref (If_Result);


--# expr-done 105
      

         declare
            Item_To_Append : constant Internal_Tree_Semantic_Result := If_Result;
         begin
               Inc_Ref (Item_To_Append);
            Internal_Tree_Semantic_Result_Vectors.Append (Map_Result_1_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end
      Finalizer_Scope_319;



         end loop;
      end;
   

   

         Map_Result_1 := Create_Internal_Tree_Semantic_Result_Array
           (Items_Count => Natural (Internal_Tree_Semantic_Result_Vectors.Length (Map_Result_1_Vec)));
         for I in Map_Result_1.Items'Range loop
            Map_Result_1.Items (I) := Internal_Tree_Semantic_Result_Vectors.Get
              (Map_Result_1_Vec,
               I + Internal_Tree_Semantic_Result_Vectors.Index_Type'First - Map_Result_1.Items'First);
         end loop;
         Internal_Tree_Semantic_Result_Vectors.Destroy (Map_Result_1_Vec);

      end;
   



--# expr-done 106
Scope_Result_1 := Map_Result_1;
   Inc_Ref (Scope_Result_1);


   --# end
      Finalizer_Scope_318;


   Result_Var_1 := Scope_Result_1;
else
   
   Result_Var_1 := No_Internal_Tree_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var_1);



   
      declare
         Map_Result_2_Vec : Internal_Semantic_Result_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Tree_Semantic_Result_Array_Access := Result_Var_1;
      begin
         for Item_5 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind res Item_5

            
         
         
      --# expr-start 107 '<FieldAccess for results at parser.py:410>' Fld_6 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:410






   




Fld_6 := Item_5.Results;
Inc_Ref (Fld_6);
--# expr-done 107
      

         for Item_To_Append of
               Fld_6.Items
         loop
               Inc_Ref (Item_To_Append);
            Internal_Semantic_Result_Vectors.Append (Map_Result_2_Vec, Item_To_Append);
         end loop;

   
   
   

            
   --# end
      Finalizer_Scope_322;



         end loop;
      end;
   

   

         Map_Result_2 := Create_Internal_Semantic_Result_Array
           (Items_Count => Natural (Internal_Semantic_Result_Vectors.Length (Map_Result_2_Vec)));
         for I in Map_Result_2.Items'Range loop
            Map_Result_2.Items (I) := Internal_Semantic_Result_Vectors.Get
              (Map_Result_2_Vec,
               I + Internal_Semantic_Result_Vectors.Index_Type'First - Map_Result_2.Items'First);
         end loop;
         Internal_Semantic_Result_Vectors.Destroy (Map_Result_2_Vec);

      end;
   



--# expr-done 108
Block_Var_2 := Map_Result_2; Inc_Ref (Block_Var_2);
--# bind children_results Block_Var_2
--# expr-start 109 '<FieldAccess for check_semantic_impl at parser.py:412>' Fld_7 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:412






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.check_semantic_impl
Fld_7 := Liblktlang.Implementation.Lkt_Node_P_Check_Semantic_Impl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 109
Block_Var_3 := Fld_7; Inc_Ref (Block_Var_3);
--# bind own_results Block_Var_3
--# expr-start 111 '<FieldAccess for new_tree_semantic_result at parser.py:420>' Fld_9 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:420






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;





Concat_Result_1 := Concat (Block_Var, Block_Var_2); 
--# expr-start 110 '<FieldAccess for results at parser.py:417>' Fld_8 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:417






   




Fld_8 := Block_Var_3.Results;
Inc_Ref (Fld_8);
--# expr-done 110
Concat_Result_2 := Concat (Concat_Result_1, Fld_8); 
--# property-call-start LktNode.new_tree_semantic_result
Fld_9 := Liblktlang.Implementation.Lkt_Node_P_New_Tree_Semantic_Result (Node => Ent.Node, Elements => Concat_Result_2);
--# end
--# expr-done 111
Let_Result := Fld_9; Inc_Ref (Let_Result);
--# end
--# expr-done 112

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_40;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_40;
                     Finalizer_Scope_318;
                     Finalizer_Scope_319;
                     Finalizer_Scope_320;
                     Finalizer_Scope_322;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Check_Sem_Recursive;
--# end

   







--# property-start '[dispatcher]LktNode.check_legality' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Lkt_Node_P_Check_Legality
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Semantic_Result_Array_Access;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Lkt_Node (Self.Kind) is
               when Lkt_Lexer_Case_Rule_Cond_Alt .. Lkt_Val_Decl | Lkt_Env_Spec_Decl .. Lkt_Generic_Decl | Lkt_Lexer_Decl .. Lkt_Instantiated_Generic_Type | Lkt_Decl_Annotation .. Lkt_Call_Expr | Lkt_Error_On_Null .. Lkt_Module_Ref_Id | Lkt_If_Expr .. Lkt_Paren_Expr | Lkt_Subscript_Expr .. Lkt_Var_Bind =>
                  --# property-call-start LktNode.check_legality
                  Property_Result := Lkt_Node_P_Check_Legality
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Fun_Decl =>
                  --# property-call-start FunDecl.check_legality
                  Property_Result := Fun_Decl_P_Check_Legality
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Grammar_Decl =>
                  --# property-call-start GrammarDecl.check_legality
                  Property_Result := Grammar_Decl_P_Check_Legality
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Type_Decl .. Lkt_Trait_Decl =>
                  --# property-call-start NamedTypeDecl.check_legality
                  Property_Result := Named_Type_Decl_P_Check_Legality
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Class_Decl .. Lkt_Enum_Class_Decl =>
                  --# property-call-start BasicClassDecl.check_legality
                  Property_Result := Basic_Class_Decl_P_Check_Legality
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Cast_Expr =>
                  --# property-call-start CastExpr.check_legality
                  Property_Result := Cast_Expr_P_Check_Legality
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Ref_Id =>
                  --# property-call-start RefId.check_legality
                  Property_Result := Ref_Id_P_Check_Legality
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Raise_Expr =>
                  --# property-call-start RaiseExpr.check_legality
                  Property_Result := Raise_Expr_P_Check_Legality
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Lkt_Node_P_Check_Legality;
--# end

   







--# property-start '[dispatcher]LktNode.check_correctness_pre' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Lkt_Node_P_Check_Correctness_Pre
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Semantic_Result_Array_Access;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Lkt_Node (Self.Kind) is
               when Lkt_Lexer_Case_Rule_Cond_Alt .. Lkt_Generic_Decl | Lkt_Lexer_Decl .. Lkt_Block_Expr | Lkt_Cast_Expr .. Lkt_Error_On_Null | Lkt_Grammar_Cut .. Lkt_Keep_Expr | Lkt_Big_Num_Lit .. Lkt_Langkit_Root | Lkt_Lexer_Case_Rule_Send .. Lkt_Function_Type_Ref | Lkt_Var_Bind =>
                  --# property-call-start LktNode.check_correctness_pre
                  Property_Result := Lkt_Node_P_Check_Correctness_Pre
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Grammar_Decl =>
                  --# property-call-start GrammarDecl.check_correctness_pre
                  Property_Result := Grammar_Decl_P_Check_Correctness_Pre
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Call_Expr =>
                  --# property-call-start CallExpr.check_correctness_pre
                  Property_Result := Call_Expr_P_Check_Correctness_Pre
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Instantiation =>
                  --# property-call-start GenericInstantiation.check_correctness_pre
                  Property_Result := Generic_Instantiation_P_Check_Correctness_Pre
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lambda_Expr =>
                  --# property-call-start LambdaExpr.check_correctness_pre
                  Property_Result := Lambda_Expr_P_Check_Correctness_Pre
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lexer_Case_Rule =>
                  --# property-call-start LexerCaseRule.check_correctness_pre
                  Property_Result := Lexer_Case_Rule_P_Check_Correctness_Pre
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Type_Ref =>
                  --# property-call-start GenericTypeRef.check_correctness_pre
                  Property_Result := Generic_Type_Ref_P_Check_Correctness_Pre
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Simple_Type_Ref =>
                  --# property-call-start SimpleTypeRef.check_correctness_pre
                  Property_Result := Simple_Type_Ref_P_Check_Correctness_Pre
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Lkt_Node_P_Check_Correctness_Pre;
--# end

   







--# property-start LktNode.check_semantic_impl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:452
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Check_Semantic_Impl
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Tree_Semantic_Result
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Tree_Semantic_Result;

      
            procedure Finalizer_Scope_43 with Inline_Always;
            procedure Finalizer_Scope_323 with Inline_Always;
            procedure Finalizer_Scope_324 with Inline_Always;
            procedure Finalizer_Scope_325 with Inline_Always;
            procedure Finalizer_Scope_326 with Inline_Always;

      Block_Var : Internal_Semantic_Result := No_Semantic_Result;
Block_Var_1 : Internal_Semantic_Result_Array_Access;
Match_0 : Internal_Entity_Ref_Id;
Match_1 : Internal_Entity_Expr;
Match_2 : Internal_Entity;
Rd : Internal_Semantic_Result := No_Semantic_Result;
Var_Expr_3 : Internal_Semantic_Result := No_Semantic_Result;
Var_Expr_4 : Internal_Semantic_Result := No_Semantic_Result;
Fld : Internal_Semantic_Result := No_Semantic_Result;
Fld_1 : Internal_Semantic_Result := No_Semantic_Result;
Try_Result : Internal_Semantic_Result := No_Semantic_Result;
Singleton : Internal_Semantic_Result_Array_Access;
Fld_2 : Internal_Semantic_Result := No_Semantic_Result;
Try_Result_1 : Internal_Semantic_Result := No_Semantic_Result;
Singleton_1 : Internal_Semantic_Result_Array_Access;
Scope_Result : Internal_Semantic_Result_Array_Access;
Result_Var : Internal_Semantic_Result_Array_Access;
Fld_3 : Boolean;
Not_Val : Boolean;
If_Result : Internal_Semantic_Result_Array_Access;
Concat_Result : Internal_Semantic_Result_Array_Access;
Let_Result : Internal_Semantic_Result_Array_Access;
Fld_4 : Internal_Semantic_Result := No_Semantic_Result;
Try_Result_2 : Internal_Semantic_Result := No_Semantic_Result;
Singleton_2 : Internal_Semantic_Result_Array_Access;
Scope_Result_1 : Internal_Semantic_Result_Array_Access;
Result_Var_1 : Internal_Semantic_Result_Array_Access;
Match_Prefix : Internal_Entity;
Cast_Result : Internal_Entity_Ref_Id;
Let_Result_1 : Internal_Semantic_Result_Array_Access;
Scope_Result_2 : Internal_Semantic_Result_Array_Access;
Cast_Result_1 : Internal_Entity_Expr;
Let_Result_2 : Internal_Semantic_Result_Array_Access;
Scope_Result_3 : Internal_Semantic_Result_Array_Access;
Cast_Result_2 : Internal_Entity;
Let_Result_3 : Internal_Semantic_Result_Array_Access;
Scope_Result_4 : Internal_Semantic_Result_Array_Access;
Match_Result : Internal_Semantic_Result_Array_Access;
Fld_5 : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;
Let_Result_4 : Internal_Tree_Semantic_Result := No_Tree_Semantic_Result;

            procedure Finalizer_Scope_43 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Block_Var);
                     Dec_Ref (Let_Result_1);
                     Dec_Ref (Scope_Result_2);
                     Dec_Ref (Let_Result_2);
                     Dec_Ref (Scope_Result_3);
                     Dec_Ref (Let_Result_3);
                     Dec_Ref (Scope_Result_4);
                     Dec_Ref (Match_Result);
                     Dec_Ref (Block_Var_1);
                     Dec_Ref (Fld_5);
                     Dec_Ref (Let_Result_4);
            end Finalizer_Scope_43;
            procedure Finalizer_Scope_323 is
            begin
                     Dec_Ref (Fld_1);
                     Dec_Ref (Try_Result);
                     Dec_Ref (Rd);
                     Dec_Ref (Singleton);
                     Dec_Ref (Fld_2);
                     Dec_Ref (Try_Result_1);
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
                     Dec_Ref (If_Result);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_323;
            procedure Finalizer_Scope_324 is
            begin
                     Dec_Ref (Var_Expr_3);
                     Dec_Ref (Singleton_1);
            end Finalizer_Scope_324;
            procedure Finalizer_Scope_325 is
            begin
                     Dec_Ref (Fld_4);
                     Dec_Ref (Try_Result_2);
                     Dec_Ref (Scope_Result_1);
                     Dec_Ref (Result_Var_1);
            end Finalizer_Scope_325;
            procedure Finalizer_Scope_326 is
            begin
                     Dec_Ref (Var_Expr_4);
                     Dec_Ref (Singleton_2);
            end Finalizer_Scope_326;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 126 '<Block at ???>' Let_Result_4 None
--# scope-start
--# expr-start 113 '<FieldAccess for internal_error at parser.py:476>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:476






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.internal_error
Fld := Liblktlang.Implementation.Lkt_Node_P_Internal_Error (Node => Self);
--# end
--# expr-done 113
Block_Var := Fld; Inc_Ref (Block_Var);
--# bind err Block_Var
--# expr-start 124 '<Match at parser.py:479>' Match_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:479









   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Match_Prefix := Ent; 

case Lkt_Lkt_Node (Match_Prefix.Node.Kind) is
         when Lkt_Ref_Id =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Ref_Id
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_0 := Cast_Result; 
--# bind r Match_0
--# expr-start 121 '<Let at parser.py:480>' Let_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:480
--# scope-start
--# expr-start 115 '<Try at parser.py:69>' Try_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:69



begin
   --# expr-start 114 '<FieldAccess for referenced_decl at parser.py:481>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:481






   

      if Match_0.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Expr.referenced_decl'
Fld_1 := Liblktlang.Implementation.Dispatcher_Expr_P_Referenced_Decl (Node => Match_0.Node, E_Info => Match_0.Info);
--# end
--# expr-done 114
   Try_Result := Fld_1;
exception
   when Property_Error =>
      
      Try_Result := Block_Var;
end;

      Inc_Ref (Try_Result);


--# expr-done 115
Rd := Try_Result; Inc_Ref (Rd);
--# bind rd Rd

                Singleton := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton.Items (1) := Rd;
                Inc_Ref (Singleton.Items (1));
            
--# expr-start 120 '<If at parser.py:483>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:483



--# expr-start 117 '<Not at parser.py:483>' Not_Val D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:483
--# expr-start 116 '<FieldAccess for has_error at parser.py:483>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:483






   




Fld_3 := Rd.Has_Error;
--# expr-done 116
Not_Val := not (Fld_3); 
--# expr-done 117
if Not_Val then
   


--# expr-start 119 '<Try at parser.py:69>' Try_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:69



begin
   --# expr-start 118 '<FieldAccess for expr_type at parser.py:484>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:484






   

      if Match_0.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Expr.expr_type
Fld_2 := Liblktlang.Implementation.Expr_P_Expr_Type (Node => Match_0.Node, E_Info => Match_0.Info);
--# end
--# expr-done 118
   Try_Result_1 := Fld_2;
exception
   when Property_Error =>
      
      Try_Result_1 := Block_Var;
end;

      Inc_Ref (Try_Result_1);


--# expr-done 119
Var_Expr_3 := Try_Result_1;

      Inc_Ref (Var_Expr_3);




if Var_Expr_3 /= No_Semantic_Result then
   



   --# scope-start


   


                Singleton_1 := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton_1.Items (1) := Var_Expr_3;
                Inc_Ref (Singleton_1.Items (1));
            
Scope_Result := Singleton_1;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_324;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var);


   If_Result := Result_Var;
else
   
   If_Result := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (If_Result);


--# expr-done 120
Concat_Result := Concat (Singleton, If_Result); 
Let_Result := Concat_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 121
Let_Result_1 := Let_Result; Inc_Ref (Let_Result_1);
--# end
Scope_Result_2 := Let_Result_1;
   Inc_Ref (Scope_Result_2);


   --# end
      Finalizer_Scope_323;


            Match_Result := Scope_Result_2; Inc_Ref (Match_Result);
         when Lkt_Any_Of .. Lkt_Module_Ref_Id | Lkt_If_Expr .. Lkt_Un_Op =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Expr
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_1 := Cast_Result_1; 
--# bind e Match_1



--# expr-start 123 '<Try at parser.py:69>' Try_Result_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:69



begin
   --# expr-start 122 '<FieldAccess for expr_type at parser.py:488>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:488






   

      if Match_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Expr.expr_type
Fld_4 := Liblktlang.Implementation.Expr_P_Expr_Type (Node => Match_1.Node, E_Info => Match_1.Info);
--# end
--# expr-done 122
   Try_Result_2 := Fld_4;
exception
   when Property_Error =>
      
      Try_Result_2 := Block_Var;
end;

      Inc_Ref (Try_Result_2);


--# expr-done 123
Var_Expr_4 := Try_Result_2;

      Inc_Ref (Var_Expr_4);




if Var_Expr_4 /= No_Semantic_Result then
   



   --# scope-start


   


                Singleton_2 := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton_2.Items (1) := Var_Expr_4;
                Inc_Ref (Singleton_2.Items (1));
            
Scope_Result_1 := Singleton_2;
   Inc_Ref (Scope_Result_1);


   --# end
      Finalizer_Scope_326;


   Result_Var_1 := Scope_Result_1;
else
   
   Result_Var_1 := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var_1);


Let_Result_2 := Result_Var_1; Inc_Ref (Let_Result_2);
--# end
Scope_Result_3 := Let_Result_2;
   Inc_Ref (Scope_Result_3);


   --# end
      Finalizer_Scope_325;


            Match_Result := Scope_Result_3; Inc_Ref (Match_Result);
         when Lkt_Lexer_Case_Rule_Cond_Alt .. Lkt_Excludes_Null_Present | Lkt_Full_Decl .. Lkt_Var_Bind =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_2 := Create_Internal_Entity
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_2 := Cast_Result_2; 
--# bind _ Match_2

Let_Result_3 := No_Internal_Semantic_Result_Array_Type; Inc_Ref (Let_Result_3);
--# end
Scope_Result_4 := Let_Result_3;
   Inc_Ref (Scope_Result_4);


   --# end


            Match_Result := Scope_Result_4; Inc_Ref (Match_Result);
end case;

--# expr-done 124
Block_Var_1 := Match_Result; Inc_Ref (Block_Var_1);
--# bind results Block_Var_1
--# expr-start 125 '<FieldAccess for new_tree_semantic_result at parser.py:492>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:492






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start LktNode.new_tree_semantic_result
Fld_5 := Liblktlang.Implementation.Lkt_Node_P_New_Tree_Semantic_Result (Node => Ent.Node, Elements => Block_Var_1);
--# end
--# expr-done 125
Let_Result_4 := Fld_5; Inc_Ref (Let_Result_4);
--# end
--# expr-done 126

         Property_Result := Let_Result_4;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_43;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_43;
                     Finalizer_Scope_323;
                     Finalizer_Scope_324;
                     Finalizer_Scope_325;
                     Finalizer_Scope_326;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Check_Semantic_Impl;
--# end

   







--# property-start LktNode.topmost_invalid_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:494
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Topmost_Invalid_Decl
  
  (Node : Bare_Lkt_Node
  )

   return Bare_Lkt_Node
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Bare_Lkt_Node;

      
            procedure Finalizer_Scope_44 with Inline_Always;

      Var_Expr : Bare_Full_Decl;
Node_Parents : Bare_Lkt_Node_Array_Access;
Cast_Expr : Bare_Lkt_Node;
Cast_Result : Bare_Full_Decl;
Sym : Symbol_Type;
Fld : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Map_Result : Bare_Lkt_Node_Array_Access;
Get_Result : Bare_Lkt_Node;
Let_Result : Bare_Lkt_Node;

            procedure Finalizer_Scope_44 is
            begin
                     Dec_Ref (Node_Parents);
                     Dec_Ref (Map_Result);
            end Finalizer_Scope_44;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 133 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 132 '<at at parser.py:503>' Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:503

--# expr-start 131 '<Filter at parser.py:500>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:500








   

   --# expr-start 127 '<FieldAccess for parents at parser.py:500>' Node_Parents D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:500






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




Node_Parents := Parents (Self, True); 
--# expr-done 127

   
      declare
         Map_Result_Vec : Bare_Lkt_Node_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Bare_Lkt_Node_Array_Access := Node_Parents;
      begin
         for Item_8 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind p Item_8

            
         


--# expr-start 128 '<Cast to FullDecl at parser.py:502>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:502






Cast_Expr := Item_8; 



   if Cast_Expr = null
      or else Cast_Expr.Kind in Lkt_Full_Decl_Range
   then
      
      Cast_Result := Cast_Expr;

   else
         Cast_Result := No_Bare_Lkt_Node;
   end if;


--# expr-done 128
Var_Expr := Cast_Result;





if Var_Expr /= No_Bare_Lkt_Node then
   



   --# scope-start


   --# bind fd Var_Expr

--# expr-start 130 '<FieldAccess for has_annotation at parser.py:502>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:502






   

      if Var_Expr = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 129 '<Symbol invalid at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Invalid); 
--# expr-done 129
--# property-call-start FullDecl.has_annotation
Fld := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Var_Expr, Name => Sym);
--# end
--# expr-done 130
Scope_Result := Fld;


   --# end


   Result_Var := Scope_Result;
else
   
   Result_Var := False;
end if;



         if Result_Var then
            
         
      
      

         declare
            Item_To_Append : constant Bare_Lkt_Node := Item_8;
         begin
            Bare_Lkt_Node_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result := Create_Bare_Lkt_Node_Array
           (Items_Count => Natural (Bare_Lkt_Node_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Bare_Lkt_Node_Vectors.Get
              (Map_Result_Vec,
               I + Bare_Lkt_Node_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Bare_Lkt_Node_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 131


Get_Result := Get (Self, Map_Result, -1, True); 
--# expr-done 132
Let_Result := Get_Result; 
--# end
--# expr-done 133

         Property_Result := Let_Result;
         
   --# end
      Finalizer_Scope_44;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_44;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Topmost_Invalid_Decl;
--# end

   







--# property-start LktNode.check_generic_inst_correctness_pre D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:505
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Check_Generic_Inst_Correctness_Pre
  
  (Node : Bare_Lkt_Node
      ; Prefix : Internal_Entity_Expr
      ; Params : Internal_Entity_Type_Ref_List
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
     Ent : Internal_Entity :=
       Internal_Entity'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind prefix Prefix
   --# bind params Params

   Property_Result : Internal_Semantic_Result_Array_Access;

      
            procedure Finalizer_Scope_45 with Inline_Always;

      Block_Var : Internal_Entity_Decl;
Block_Var_1 : Internal_Entity_Generic_Decl;
Block_Var_2 : Internal_Semantic_Result_Array_Access;
Block_Var_3 : Internal_Semantic_Result_Array_Access;
Fld : Internal_Semantic_Result := No_Semantic_Result;
Fld_1 : Internal_Entity_Decl;
Cast_Expr : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Generic_Decl;
Str : String_Type;
Fld_2 : Internal_Semantic_Result := No_Semantic_Result;
Singleton : Internal_Semantic_Result_Array_Access;
Is_Null : Boolean;
Is_Null_1 : Boolean;
Not_Val : Boolean;
If_Result : Boolean;
If_Result_1 : Internal_Semantic_Result_Array_Access;
Str_1 : String_Type;
Fld_3 : Internal_Semantic_Result := No_Semantic_Result;
Singleton_1 : Internal_Semantic_Result_Array_Access;
Is_Null_2 : Boolean;
Not_Val_1 : Boolean;
Fld_4 : Bare_Type_Ref_List;
Len : Integer;
Fld_5 : Internal_Entity_Generic_Formal_Decl_List;
Fld_6 : Bare_Generic_Formal_Decl_List;
Len_1 : Integer;
Is_Equal : Boolean;
Not_Val_2 : Boolean;
If_Result_2 : Boolean;
If_Result_3 : Internal_Semantic_Result_Array_Access;
Concat_Result : Internal_Semantic_Result_Array_Access;
Let_Result : Internal_Semantic_Result_Array_Access;

            procedure Finalizer_Scope_45 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Str);
                     Dec_Ref (Fld_2);
                     Dec_Ref (Singleton);
                     Dec_Ref (If_Result_1);
                     Dec_Ref (Block_Var_2);
                     Dec_Ref (Str_1);
                     Dec_Ref (Fld_3);
                     Dec_Ref (Singleton_1);
                     Dec_Ref (If_Result_3);
                     Dec_Ref (Block_Var_3);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_45;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 153 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 135 '<FieldAccess for result_decl at parser.py:516>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:516



--# expr-start 134 '<FieldAccess for referenced_decl at parser.py:516>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:516






   

      if Prefix.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Expr.referenced_decl'
Fld := Liblktlang.Implementation.Dispatcher_Expr_P_Referenced_Decl (Node => Prefix.Node, E_Info => Prefix.Info);
--# end
--# expr-done 134


   




Fld_1 := Fld.Result_Decl;
--# expr-done 135
Block_Var := Fld_1; 
--# bind prefix_decl Block_Var
--# expr-start 136 '<Cast to GenericDecl at parser.py:517>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:517






Cast_Expr := Block_Var; 



   if Cast_Expr.Node = null
      or else Cast_Expr.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);

   else
         Cast_Result := No_Entity_Generic_Decl;
   end if;


--# expr-done 136
Block_Var_1 := Cast_Result; 
--# bind gen_decl Block_Var_1
--# expr-start 142 '<If at parser.py:519>' If_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:519






--# expr-start 137 '<IsNull at parser.py:520>' Is_Null D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:520

Is_Null := Block_Var.Node = null; 
--# expr-done 137
if Is_Null then
   
   If_Result := True;
else
   --# expr-start 139 '<Not at parser.py:520>' Not_Val D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:520
--# expr-start 138 '<IsNull at parser.py:520>' Is_Null_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:520

Is_Null_1 := Block_Var_1.Node = null; 
--# expr-done 138
Not_Val := not (Is_Null_1); 
--# expr-done 139
   If_Result := Not_Val;
end if;



if If_Result then
   
   If_Result_1 := No_Internal_Semantic_Result_Array_Type;
else
   --# expr-start 141 '<FieldAccess for error at parser.py:522>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:522






   

      if Prefix.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 140 '<String at parser.py:522>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:522
Str := Create_String ("invalid reference to a generic"); 
--# expr-done 140
--# property-call-start LktNode.error
Fld_2 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Prefix.Node, Message => Str);
--# end
--# expr-done 141
                Singleton := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton.Items (1) := Fld_2;
                Inc_Ref (Singleton.Items (1));
            
   If_Result_1 := Singleton;
end if;

      Inc_Ref (If_Result_1);


--# expr-done 142
Block_Var_2 := If_Result_1; Inc_Ref (Block_Var_2);
--# bind invalid_decl Block_Var_2
--# expr-start 152 '<If at parser.py:530>' If_Result_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:530






--# expr-start 144 '<Not at parser.py:531>' Not_Val_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:531
--# expr-start 143 '<IsNull at parser.py:531>' Is_Null_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:531

Is_Null_2 := Block_Var_1.Node = null; 
--# expr-done 143
Not_Val_1 := not (Is_Null_2); 
--# expr-done 144
if Not_Val_1 then
   --# expr-start 149 '<Not at parser.py:532>' Not_Val_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:532
--# expr-start 148 '<Eq at parser.py:532>' Is_Equal D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:532
--# expr-start 145 '<length at parser.py:532>' Len D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:532






   




Fld_4 := Params.Node; 
Len := Length (Fld_4); 
--# expr-done 145
--# expr-start 147 '<length at parser.py:532>' Len_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:532



--# expr-start 146 '<FieldAccess for generic_formal_decls at parser.py:532>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:532






   

      if Block_Var_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_5 := Create_Internal_Entity_Generic_Formal_Decl_List (Node => Block_Var_1.Node.Generic_Decl_F_Generic_Formal_Decls, Info => Block_Var_1.Info);
--# expr-done 146


   




Fld_6 := Fld_5.Node; 
Len_1 := Length (Fld_6); 
--# expr-done 147
Is_Equal := Len = Len_1; 
--# expr-done 148
Not_Val_2 := not (Is_Equal); 
--# expr-done 149
   If_Result_2 := Not_Val_2;
else
   
   If_Result_2 := False;
end if;



if If_Result_2 then
   --# expr-start 151 '<FieldAccess for error at parser.py:534>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:534






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 150 '<String at parser.py:534>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:534
Str_1 := Create_String ("invalid number of generic parameters"); 
--# expr-done 150
--# property-call-start LktNode.error
Fld_3 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Ent.Node, Message => Str_1);
--# end
--# expr-done 151
                Singleton_1 := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton_1.Items (1) := Fld_3;
                Inc_Ref (Singleton_1.Items (1));
            
   If_Result_3 := Singleton_1;
else
   
   If_Result_3 := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (If_Result_3);


--# expr-done 152
Block_Var_3 := If_Result_3; Inc_Ref (Block_Var_3);
--# bind invalid_params Block_Var_3


Concat_Result := Concat (Block_Var_2, Block_Var_3); 
Let_Result := Concat_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 153

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_45;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_45;




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Check_Generic_Inst_Correctness_Pre;
--# end

   







--# property-start LktNode.can_reach D:/a/msys64/ucrt64/lib/python3.12/site-packages/langkit/compiled_types.py:0
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Can_Reach
  
  (Node : Bare_Lkt_Node
      ; From_Node : Bare_Lkt_Node
  )

   return Boolean
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   

   --# bind from_node From_Node

   Property_Result : Boolean;

      

      Cast_Expr : Bare_Lkt_Node;
Cast_Result : Bare_Lkt_Node;
Is_Equal : Boolean;
Fld : Internal_Unit;
Fld_1 : Internal_Unit;
Is_Equal_1 : Boolean;
Not_Val : Boolean;
If_Result : Boolean;
Node_Comp : Boolean;
If_Result_1 : Boolean;
Let_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 160 '<Block at ???>' Let_Result None
--# scope-start






--# expr-start 154 '<Eq at ???>' Is_Equal None






Cast_Expr := From_Node; 



   
      Cast_Result := Cast_Expr;



Is_Equal := Cast_Result = No_Bare_Lkt_Node; 
--# expr-done 154
if Is_Equal then
   
   If_Result := True;
else
   --# expr-start 158 '<Not at ???>' Not_Val None
--# expr-start 157 '<Eq at ???>' Is_Equal_1 None
--# expr-start 155 '<FieldAccess for unit at ???>' Fld None






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Unit (Node => Self);
--# expr-done 155
--# expr-start 156 '<FieldAccess for unit at ???>' Fld_1 None






   

      if From_Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Liblktlang.Implementation.Unit (Node => From_Node);
--# expr-done 156
Is_Equal_1 := Fld = Fld_1; 
--# expr-done 157
Not_Val := not (Is_Equal_1); 
--# expr-done 158
   If_Result := Not_Val;
end if;



if If_Result then
   
   If_Result_1 := True;
else
   --# expr-start 159 '<OrderingTest '"'"'lt'"'"' at ???>' Node_Comp None



Node_Comp := Compare (Self, Self, From_Node, Less_Than); 
--# expr-done 159
   If_Result_1 := Node_Comp;
end if;



Let_Result := If_Result_1; 
--# end
--# expr-done 160

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Lkt_Node_P_Can_Reach;
--# end

   







--# property-start LktNode.check_legality D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:422
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Check_Legality
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Semantic_Result_Array_Access;

      
            procedure Finalizer_Scope_41 with Inline_Always;

      Let_Result : Internal_Semantic_Result_Array_Access;

            procedure Finalizer_Scope_41 is
            begin
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_41;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Check_Legality,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);




      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Internal_Semantic_Result_Array_Access;
                     Inc_Ref (Property_Result);

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 161 '<Block at ???>' Let_Result None
--# scope-start

Let_Result := No_Internal_Semantic_Result_Array_Type; Inc_Ref (Let_Result);
--# end
--# expr-done 161

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_41;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_41;


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Internal_Semantic_Result_Array_Access,
                        As_Internal_Semantic_Result_Array_Access => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
               if Mmz_Stored then
                  Inc_Ref (Property_Result);
               end if;

         end if;
      end if;



   return Property_Result;
end Lkt_Node_P_Check_Legality;
--# end

   







--# property-start LktNode.check_correctness_pre D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:437
pragma Warnings (Off, "is not referenced");
 function Lkt_Node_P_Check_Correctness_Pre
  
  (Node : Bare_Lkt_Node
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Lkt_Node  := Bare_Lkt_Node (Node);
      --# bind self Self

   


   Property_Result : Internal_Semantic_Result_Array_Access;

      
            procedure Finalizer_Scope_42 with Inline_Always;

      Let_Result : Internal_Semantic_Result_Array_Access;

            procedure Finalizer_Scope_42 is
            begin
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_42;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lkt_Node_Lkt_Node_P_Check_Correctness_Pre,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lkt_Node,
                                As_Bare_Lkt_Node => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);




      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Internal_Semantic_Result_Array_Access;
                     Inc_Ref (Property_Result);

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 162 '<Block at ???>' Let_Result None
--# scope-start

Let_Result := No_Internal_Semantic_Result_Array_Type; Inc_Ref (Let_Result);
--# end
--# expr-done 162

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_42;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_42;


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Internal_Semantic_Result_Array_Access,
                        As_Internal_Semantic_Result_Array_Access => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
               if Mmz_Stored then
                  Inc_Ref (Property_Result);
               end if;

         end if;
      end if;



   return Property_Result;
end Lkt_Node_P_Check_Correctness_Pre;
--# end



   


      

   --
   --  Primitives for Bare_Base_Lexer_Case_Rule_Alt
   --

   




      
   function Base_Lexer_Case_Rule_Alt_F_Send
     (Node : Bare_Base_Lexer_Case_Rule_Alt) return Bare_Lexer_Case_Rule_Send
   is
      

         Kind : constant Lkt_Base_Lexer_Case_Rule_Alt := Node.Kind;
   begin
         case Kind is
               when Lkt_Lexer_Case_Rule_Cond_Alt =>
                     
         return Node.Lexer_Case_Rule_Cond_Alt_F_Send;
      
               when Lkt_Lexer_Case_Rule_Default_Alt =>
                     
         return Node.Lexer_Case_Rule_Default_Alt_F_Send;
      
         end case;
   end;




   


      

   --
   --  Primitives for Bare_Lexer_Case_Rule_Cond_Alt
   --

   



      
      procedure Initialize_Fields_For_Lexer_Case_Rule_Cond_Alt
        (Self : Bare_Lexer_Case_Rule_Cond_Alt
         ; Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs : Bare_Ref_Id_List
         ; Lexer_Case_Rule_Cond_Alt_F_Send : Bare_Lexer_Case_Rule_Send
        ) is
      begin

            Self.Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs := Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs;
            Self.Lexer_Case_Rule_Cond_Alt_F_Send := Lexer_Case_Rule_Cond_Alt_F_Send;
         

      end Initialize_Fields_For_Lexer_Case_Rule_Cond_Alt;

      
   function Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs
     (Node : Bare_Lexer_Case_Rule_Cond_Alt) return Bare_Ref_Id_List
   is
      

   begin
         
         return Node.Lexer_Case_Rule_Cond_Alt_F_Cond_Exprs;
      
   end;




   


      

   --
   --  Primitives for Bare_Lexer_Case_Rule_Default_Alt
   --

   



      
      procedure Initialize_Fields_For_Lexer_Case_Rule_Default_Alt
        (Self : Bare_Lexer_Case_Rule_Default_Alt
         ; Lexer_Case_Rule_Default_Alt_F_Send : Bare_Lexer_Case_Rule_Send
        ) is
      begin

            Self.Lexer_Case_Rule_Default_Alt_F_Send := Lexer_Case_Rule_Default_Alt_F_Send;
         

      end Initialize_Fields_For_Lexer_Case_Rule_Default_Alt;




   


      

   --
   --  Primitives for Bare_Block_String_Line
   --

   







   


      

   --
   --  Primitives for Bare_Class_Qualifier
   --

   





   







--# property-start '[dispatcher]ClassQualifier.as_bool' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Class_Qualifier_P_As_Bool
  
  (Node : Bare_Class_Qualifier
  )

   return Boolean
is
   Self : Bare_Class_Qualifier  := Bare_Class_Qualifier (Node);
      --# bind self Self

   


   Property_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Class_Qualifier (Self.Kind) is
               when Lkt_Class_Qualifier_Absent =>
                  --# property-call-start ClassQualifier.Absent.as_bool
                  Property_Result := Class_Qualifier_Absent_P_As_Bool
                    (Self
                    );
                  --# end
               when Lkt_Class_Qualifier_Present =>
                  --# property-call-start ClassQualifier.Present.as_bool
                  Property_Result := Class_Qualifier_Present_P_As_Bool
                    (Self
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Class_Qualifier_P_As_Bool;
--# end



   


      

   --
   --  Primitives for Bare_Class_Qualifier_Absent
   --

   





   







--# property-start ClassQualifier.Absent.as_bool D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1953
pragma Warnings (Off, "is not referenced");
 function Class_Qualifier_Absent_P_As_Bool
  
  (Node : Bare_Class_Qualifier_Absent
  )

   return Boolean
is
   Self : Bare_Class_Qualifier_Absent  := Bare_Class_Qualifier_Absent (Node);
      --# bind self Self

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := False;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Class_Qualifier_Absent_P_As_Bool;
--# end



   


      

   --
   --  Primitives for Bare_Class_Qualifier_Present
   --

   





   







--# property-start ClassQualifier.Present.as_bool D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1953
pragma Warnings (Off, "is not referenced");
 function Class_Qualifier_Present_P_As_Bool
  
  (Node : Bare_Class_Qualifier_Present
  )

   return Boolean
is
   Self : Bare_Class_Qualifier_Present  := Bare_Class_Qualifier_Present (Node);
      --# bind self Self

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := True;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Class_Qualifier_Present_P_As_Bool;
--# end



   


      

   --
   --  Primitives for Bare_Decl
   --

   



   

   

   

   

   

   

   


      procedure Decl_Pre_Env_Actions
        (Self            : Bare_Decl;
         State           : in out PLE_Node_State;
         Add_To_Env_Only : Boolean := False) is
      begin
            


   

      declare
         Resolver : constant Entity_Resolver :=
            null;

         

         Mapping : Internal_Env_Assoc :=
            Liblktlang.Implementation.Env_Mappings_2 (Node => 
Self);
      begin

         Add_To_Env
           (Self,
            State,
            Mapping.Key,
            Mapping.Value,
            Mapping.Metadata,
            Resolver,
            Mapping.Dest_Env,
            DSL_Location => "parser.py:739");
         Dec_Ref (Mapping.Dest_Env);

      end;
   
   
      end;




      
   function Decl_F_Syn_Name
     (Node : Bare_Decl) return Bare_Def_Id
   is
      

         Kind : constant Lkt_Decl := Node.Kind;
   begin
         case Kind is
               when Lkt_Lexer_Decl =>
                     
         return Node.Lexer_Decl_F_Syn_Name;
      
               when Lkt_Lexer_Family_Decl =>
                     
         return Node.Lexer_Family_Decl_F_Syn_Name;
      
               when Lkt_Grammar_Decl =>
                     
         return Node.Grammar_Decl_F_Syn_Name;
      
               when Lkt_Grammar_Rule_Decl =>
                     
         return Node.Grammar_Rule_Decl_F_Syn_Name;
      
               when Lkt_Synthetic_Lexer_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Function_Type =>
                     return No_Bare_Lkt_Node;
               when Lkt_Generic_Formal_Type_Decl =>
                     
         return Node.Generic_Formal_Type_Decl_F_Syn_Name;
      
               when Lkt_Generic_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Instantiated_Generic_Type =>
                     return No_Bare_Lkt_Node;
               when Lkt_Trait_Decl =>
                     
         return Node.Trait_Decl_F_Syn_Name;
      
               when Lkt_Enum_Type_Decl =>
                     
         return Node.Enum_Type_Decl_F_Syn_Name;
      
               when Lkt_Struct_Decl =>
                     
         return Node.Struct_Decl_F_Syn_Name;
      
               when Lkt_Class_Decl | Lkt_Enum_Class_Decl =>
                     
         return Node.Basic_Class_Decl_F_Syn_Name;
      
               when Lkt_Env_Spec_Decl =>
                     
         return Node.Env_Spec_Decl_F_Syn_Name;
      
               when Lkt_Enum_Class_Alt_Decl =>
                     
         return Node.Enum_Class_Alt_Decl_F_Syn_Name;
      
               when Lkt_Self_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Node_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Fun_Decl =>
                     
         return Node.Fun_Decl_F_Syn_Name;
      
               when Lkt_Enum_Lit_Decl =>
                     
         return Node.Enum_Lit_Decl_F_Syn_Name;
      
               when Lkt_Fun_Arg_Decl =>
                     
         return Node.Fun_Arg_Decl_F_Syn_Name;
      
               when Lkt_Lambda_Arg_Decl =>
                     
         return Node.Lambda_Arg_Decl_F_Syn_Name;
      
               when Lkt_Field_Decl =>
                     
         return Node.Field_Decl_F_Syn_Name;
      
               when Lkt_Match_Val_Decl =>
                     
         return Node.Match_Val_Decl_F_Syn_Name;
      
               when Lkt_Dyn_Var_Decl =>
                     
         return Node.Dyn_Var_Decl_F_Syn_Name;
      
               when Lkt_Val_Decl =>
                     
         return Node.Val_Decl_F_Syn_Name;
      
         end case;
   end;


   







--# property-start '[dispatcher]Decl.name' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Decl_P_Name
  
  (Node : Bare_Decl
  )

   return Symbol_Type
is
   Self : Bare_Decl  := Bare_Decl (Node);
      --# bind self Self

   


   Property_Result : Symbol_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Decl (Self.Kind) is
               when Lkt_Grammar_Rule_Decl | Lkt_Enum_Lit_Decl .. Lkt_Env_Spec_Decl | Lkt_Grammar_Decl .. Lkt_Generic_Formal_Type_Decl | Lkt_Class_Decl .. Lkt_Trait_Decl =>
                  --# property-call-start Decl.name
                  Property_Result := Decl_P_Name
                    (Self
                    );
                  --# end
               when Lkt_Synthetic_Lexer_Decl =>
                  --# property-call-start SyntheticLexerDecl.name
                  Property_Result := Synthetic_Lexer_Decl_P_Name
                    (Self
                    );
                  --# end
               when Lkt_Node_Decl =>
                  --# property-call-start NodeDecl.name
                  Property_Result := Node_Decl_P_Name
                    (Self
                    );
                  --# end
               when Lkt_Self_Decl =>
                  --# property-call-start SelfDecl.name
                  Property_Result := Self_Decl_P_Name
                    (Self
                    );
                  --# end
               when Lkt_Generic_Decl =>
                  --# property-call-start GenericDecl.name
                  Property_Result := Generic_Decl_P_Name
                    (Self
                    );
                  --# end
               when Lkt_Instantiated_Generic_Type =>
                  --# property-call-start InstantiatedGenericType.name
                  Property_Result := Instantiated_Generic_Type_P_Name
                    (Self
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Decl_P_Name;
--# end

   







--# property-start '[dispatcher]Decl.full_name' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Decl_P_Full_Name
  
  (Node : Bare_Decl
  )

   return String_Type
is
   Self : Bare_Decl  := Bare_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Decl (Self.Kind) is
               when Lkt_Grammar_Rule_Decl .. Lkt_Enum_Class_Alt_Decl | Lkt_Generic_Formal_Type_Decl | Lkt_Class_Decl .. Lkt_Trait_Decl =>
                  --# property-call-start Decl.full_name
                  Property_Result := Decl_P_Full_Name
                    (Self
                    );
                  --# end
               when Lkt_Function_Type =>
                  --# property-call-start FunctionType.full_name
                  Property_Result := Function_Type_P_Full_Name
                    (Self
                    );
                  --# end
               when Lkt_Instantiated_Generic_Type =>
                  --# property-call-start InstantiatedGenericType.full_name
                  Property_Result := Instantiated_Generic_Type_P_Full_Name
                    (Self
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Decl_P_Full_Name;
--# end

   







--# property-start '[dispatcher]Decl.decl_type_name' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Decl_P_Decl_Type_Name
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Decl  := Bare_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Decl (Self.Kind) is
               when Lkt_Grammar_Rule_Decl =>
                  --# property-call-start GrammarRuleDecl.decl_type_name
                  Property_Result := Grammar_Rule_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Synthetic_Lexer_Decl =>
                  --# property-call-start SyntheticLexerDecl.decl_type_name
                  Property_Result := Synthetic_Lexer_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Node_Decl =>
                  --# property-call-start NodeDecl.decl_type_name
                  Property_Result := Node_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Self_Decl =>
                  --# property-call-start SelfDecl.decl_type_name
                  Property_Result := Self_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Lit_Decl =>
                  --# property-call-start EnumLitDecl.decl_type_name
                  Property_Result := Enum_Lit_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Field_Decl =>
                  --# property-call-start FieldDecl.decl_type_name
                  Property_Result := Field_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Fun_Arg_Decl =>
                  --# property-call-start FunArgDecl.decl_type_name
                  Property_Result := Fun_Arg_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lambda_Arg_Decl =>
                  --# property-call-start LambdaArgDecl.decl_type_name
                  Property_Result := Lambda_Arg_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Dyn_Var_Decl =>
                  --# property-call-start DynVarDecl.decl_type_name
                  Property_Result := Dyn_Var_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Match_Val_Decl =>
                  --# property-call-start MatchValDecl.decl_type_name
                  Property_Result := Match_Val_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Val_Decl =>
                  --# property-call-start ValDecl.decl_type_name
                  Property_Result := Val_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Fun_Decl =>
                  --# property-call-start FunDecl.decl_type_name
                  Property_Result := Fun_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Env_Spec_Decl =>
                  --# property-call-start EnvSpecDecl.decl_type_name
                  Property_Result := Env_Spec_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Decl =>
                  --# property-call-start GenericDecl.decl_type_name
                  Property_Result := Generic_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Grammar_Decl =>
                  --# property-call-start GrammarDecl.decl_type_name
                  Property_Result := Grammar_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lexer_Decl =>
                  --# property-call-start LexerDecl.decl_type_name
                  Property_Result := Lexer_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Lexer_Family_Decl =>
                  --# property-call-start LexerFamilyDecl.decl_type_name
                  Property_Result := Lexer_Family_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Class_Alt_Decl =>
                  --# property-call-start EnumClassAltDecl.decl_type_name
                  Property_Result := Enum_Class_Alt_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Function_Type =>
                  --# property-call-start FunctionType.decl_type_name
                  Property_Result := Function_Type_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Formal_Type_Decl =>
                  --# property-call-start GenericFormalTypeDecl.decl_type_name
                  Property_Result := Generic_Formal_Type_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Instantiated_Generic_Type =>
                  --# property-call-start InstantiatedGenericType.decl_type_name
                  Property_Result := Instantiated_Generic_Type_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Class_Decl =>
                  --# property-call-start ClassDecl.decl_type_name
                  Property_Result := Class_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Class_Decl =>
                  --# property-call-start EnumClassDecl.decl_type_name
                  Property_Result := Enum_Class_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Type_Decl =>
                  --# property-call-start EnumTypeDecl.decl_type_name
                  Property_Result := Enum_Type_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Struct_Decl =>
                  --# property-call-start StructDecl.decl_type_name
                  Property_Result := Struct_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Trait_Decl =>
                  --# property-call-start TraitDecl.decl_type_name
                  Property_Result := Trait_Decl_P_Decl_Type_Name
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Decl_P_Decl_Type_Name;
--# end

   







--# property-start Decl.full_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:728
pragma Warnings (Off, "is not referenced");
 function Decl_P_Full_Decl
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Full_Decl
is
   Self : Bare_Decl  := Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Full_Decl;

      

      Match_0 : Internal_Entity_Full_Decl;
Match_1 : Internal_Entity_Generic_Decl;
Match_2 : Internal_Entity;
Fld : Internal_Entity;
Fld_1 : Internal_Entity_Full_Decl;
Match_Prefix : Internal_Entity;
Cast_Result : Internal_Entity_Full_Decl;
Let_Result : Internal_Entity_Full_Decl;
Scope_Result : Internal_Entity_Full_Decl;
Cast_Result_1 : Internal_Entity_Generic_Decl;
Let_Result_1 : Internal_Entity_Full_Decl;
Scope_Result_1 : Internal_Entity_Full_Decl;
Cast_Result_2 : Internal_Entity;
Let_Result_2 : Internal_Entity_Full_Decl;
Scope_Result_2 : Internal_Entity_Full_Decl;
Match_Result : Internal_Entity_Full_Decl;
Let_Result_3 : Internal_Entity_Full_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 166 '<Block at ???>' Let_Result_3 None
--# scope-start
--# expr-start 165 '<Match at parser.py:730>' Match_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:730






--# expr-start 163 '<parent at parser.py:730>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:730






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 163


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Match_Prefix := Fld; 

case Lkt_Lkt_Node (Match_Prefix.Node.Kind) is
         when Lkt_Full_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Full_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_0 := Cast_Result; 
--# bind fd Match_0

Let_Result := Match_0; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Generic_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Generic_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_1 := Cast_Result_1; 
--# bind gd Match_1
--# expr-start 164 '<FieldAccess for full_decl at parser.py:732>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:732






   

      if Match_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.full_decl
Fld_1 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Match_1.Node, E_Info => Match_1.Info);
--# end
--# expr-done 164
Let_Result_1 := Fld_1; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
         when Lkt_Lexer_Case_Rule_Cond_Alt .. Lkt_Env_Spec_Decl | Lkt_Grammar_Decl .. Lkt_Un_Op | Lkt_Grammar_List_Sep .. Lkt_Var_Bind =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_2 := Create_Internal_Entity
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_2 := Cast_Result_2; 
--# bind _ Match_2

Let_Result_2 := raise Property_Error with "should not happen"; 
--# end
Scope_Result_2 := Let_Result_2;


   --# end


            Match_Result := Scope_Result_2; 
end case;

--# expr-done 165
Let_Result_3 := Match_Result; 
--# end
--# expr-done 166

         Property_Result := Let_Result_3;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Full_Decl;
--# end

   







--# property-start Decl.quoted_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:737
pragma Warnings (Off, "is not referenced");
 function Decl_P_Quoted_Name
  
  (Node : Bare_Decl
  )

   return String_Type
is
   Self : Bare_Decl  := Bare_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_75 with Inline_Always;

      Str : String_Type;
Fld : String_Type;
Concat_Result : String_Type;
Str_1 : String_Type;
Concat_Result_1 : String_Type;

            procedure Finalizer_Scope_75 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Fld);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Str_1);
                     Dec_Ref (Concat_Result_1);
            end Finalizer_Scope_75;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 167 '<String at parser.py:737>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:737
Str := Create_String ("`"); 
--# expr-done 167
--# expr-start 168 '<FieldAccess for full_name at parser.py:737>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:737






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.full_name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Full_Name (Node => Self);
--# end
--# expr-done 168
Concat_Result := Concat_String (Str, Fld); 
--# expr-start 169 '<String at parser.py:737>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:737
Str_1 := Create_String ("`"); 
--# expr-done 169
Concat_Result_1 := Concat_String (Concat_Result, Str_1); 

         Property_Result := Concat_Result_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_75;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_75;




            raise;
      end;



   return Property_Result;
end Decl_P_Quoted_Name;
--# end

   







--# property-start Decl.assert_bare D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:744
pragma Warnings (Off, "is not referenced");
 function Decl_P_Assert_Bare
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Bare_Decl
is
   Self : Bare_Decl  := Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Bare_Decl;

      

      Fld : Bare_Decl;
Fld_1 : Internal_Entity_Info;
Fld_2 : Env_Rebindings;
Is_Equal : Boolean;
If_Result : Bare_Decl;
Let_Result : Bare_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 175 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 174 '<If at parser.py:746>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:746



--# expr-start 172 '<Eq at parser.py:747>' Is_Equal D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:747
--# expr-start 171 '<FieldAccess for rebindings at parser.py:747>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:747



--# expr-start 170 '<FieldAccess for info at parser.py:747>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:747






   




Fld_1 := Ent.Info;
--# expr-done 170


   




Fld_2 := Fld_1.Rebindings;
--# expr-done 171

Is_Equal := Fld_2 = null; 
--# expr-done 172
if Is_Equal then
   --# expr-start 173 '<FieldAccess for node at parser.py:748>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:748






   




Fld := Ent.Node;
--# expr-done 173
   If_Result := Fld;
else
   
   If_Result := raise Property_Error with "Type decl has rebindings but shouldn't";
end if;



--# expr-done 174
Let_Result := If_Result; 
--# end
--# expr-done 175

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Assert_Bare;
--# end

   







--# property-start Decl.as_bare_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:752
pragma Warnings (Off, "is not referenced");
 function Decl_P_As_Bare_Decl
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Decl
is
   Self : Bare_Decl  := Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Decl;

      

      Fld : Bare_Decl;
As_Entity : Internal_Entity_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Decl;
Let_Result : Internal_Entity_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 178 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 177 '<as_bare_entity at parser.py:761>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:761



--# expr-start 176 '<FieldAccess for node at parser.py:761>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:761






   




Fld := Ent.Node;
--# expr-done 176
Is_Null := Fld = null; 
if Is_Null then
   
   If_Result := No_Entity_Decl;
else
   

As_Entity := (Info => No_Entity_Info, Node => Fld); 
   If_Result := As_Entity;
end if;



--# expr-done 177
Let_Result := If_Result; 
--# end
--# expr-done 178

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_As_Bare_Decl;
--# end

   







--# property-start '[dispatcher]Decl.call_scope' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Decl_P_Call_Scope
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Decl  := Bare_Decl (Node);
      --# bind self Self

   


   Property_Result : Lexical_Env;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Decl (Self.Kind) is
               when Lkt_Grammar_Rule_Decl .. Lkt_Val_Decl | Lkt_Env_Spec_Decl .. Lkt_Lexer_Family_Decl =>
                  --# property-call-start Decl.call_scope
                  Property_Result := Decl_P_Call_Scope
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Fun_Decl =>
                  --# property-call-start FunDecl.call_scope
                  Property_Result := Fun_Decl_P_Call_Scope
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Class_Alt_Decl .. Lkt_Trait_Decl =>
                  --# property-call-start TypeDecl.call_scope
                  Property_Result := Type_Decl_P_Call_Scope
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Decl_P_Call_Scope;
--# end

   







--# property-start Decl.called_decl_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:772
pragma Warnings (Off, "is not referenced");
 function Decl_P_Called_Decl_Type
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Decl  := Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl;

      

      Match_0 : Internal_Entity_Type_Decl;
Match_1 : Internal_Entity_Fun_Decl;
Match_2 : Internal_Entity_Decl;
Fld : Internal_Entity_Type_Ref;
Fld_1 : Internal_Entity_Type_Decl;
Match_Prefix : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;
Cast_Result_1 : Internal_Entity_Fun_Decl;
Let_Result_1 : Internal_Entity_Type_Decl;
Scope_Result_1 : Internal_Entity_Type_Decl;
Cast_Result_2 : Internal_Entity_Decl;
Let_Result_2 : Internal_Entity_Type_Decl;
Scope_Result_2 : Internal_Entity_Type_Decl;
Match_Result : Internal_Entity_Type_Decl;
Let_Result_3 : Internal_Entity_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 182 '<Block at ???>' Let_Result_3 None
--# scope-start
--# expr-start 181 '<Match at parser.py:779>' Match_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:779









   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Match_Prefix := Ent; 

case Lkt_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Enum_Class_Alt_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_0 := Cast_Result; 
--# bind t Match_0

Let_Result := Match_0; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Fun_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Fun_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_1 := Cast_Result_1; 
--# bind f Match_1
--# expr-start 180 '<FieldAccess for designated_type at parser.py:781>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:781



--# expr-start 179 '<FieldAccess for return_type at parser.py:781>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:781






   

      if Match_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Type_Ref (Node => Match_1.Node.Fun_Decl_F_Return_Type, Info => Match_1.Info);
--# expr-done 179


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 180
Let_Result_1 := Fld_1; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
         when Lkt_Grammar_Rule_Decl .. Lkt_Val_Decl | Lkt_Env_Spec_Decl .. Lkt_Lexer_Family_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_2 := Create_Internal_Entity_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_2 := Cast_Result_2; 
--# bind _ Match_2

Let_Result_2 := raise Property_Error with "Should not happen"; 
--# end
Scope_Result_2 := Let_Result_2;


   --# end


            Match_Result := Scope_Result_2; 
end case;

--# expr-done 181
Let_Result_3 := Match_Result; 
--# end
--# expr-done 182

         Property_Result := Let_Result_3;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Called_Decl_Type;
--# end

   







--# property-start Decl.is_generic_fun_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:785
pragma Warnings (Off, "is not referenced");
 function Decl_P_Is_Generic_Fun_Decl
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Decl  := Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Boolean;

      

      Var_Expr_6 : Internal_Entity_Generic_Decl;
Cast_Expr : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Generic_Decl;
Fld : Internal_Entity_Decl;
Is_A : Boolean;
Scope_Result : Boolean;
Result_Var : Boolean;
Let_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 186 '<Block at ???>' Let_Result None
--# scope-start



--# expr-start 183 '<Cast to GenericDecl at parser.py:790>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:790






Cast_Expr := Ent; 



   if Cast_Expr.Node = null
      or else Cast_Expr.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);

   else
         Cast_Result := No_Entity_Generic_Decl;
   end if;


--# expr-done 183
Var_Expr_6 := Cast_Result;





if Var_Expr_6 /= No_Entity_Generic_Decl then
   



   --# scope-start


   

--# expr-start 185 '<IsA FunDecl at parser.py:790>' Is_A D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:790
--# expr-start 184 '<FieldAccess for decl at parser.py:790>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:790






   

      if Var_Expr_6.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Decl (Node => Var_Expr_6.Node.Generic_Decl_F_Decl, Info => Var_Expr_6.Info);
--# expr-done 184
Is_A := Fld.Node /= null 
and then Fld.Node.Kind in Lkt_Fun_Decl_Range; 
--# expr-done 185
Scope_Result := Is_A;


   --# end


   Result_Var := Scope_Result;
else
   
   Result_Var := False;
end if;



Let_Result := Result_Var; 
--# end
--# expr-done 186

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Is_Generic_Fun_Decl;
--# end

   







--# property-start Decl._env_mappings_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:739
pragma Warnings (Off, "is not referenced");
 function Env_Mappings_2
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Env_Assoc
is
   Self : Bare_Decl  := Bare_Decl (Node);
     Ent : Internal_Entity_Decl :=
       Internal_Entity_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Env_Assoc;

      
            procedure Finalizer_Scope_280 with Inline_Always;

      Fld : Symbol_Type;
New_Struct : Internal_Designated_Env := No_Designated_Env;
Cast_Expr : Bare_Decl;
Cast_Result : Bare_Lkt_Node;
New_Struct_1 : Internal_Env_Assoc := No_Env_Assoc;

            procedure Finalizer_Scope_280 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (New_Struct_1);
            end Finalizer_Scope_280;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 189 '<New EnvAssoc at parser.py:739>' New_Struct_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:739
--# expr-start 187 '<New DesignatedEnv at parser.py:739>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:739



Inc_Ref (Empty_Env);
New_Struct := (Direct_Env => Empty_Env, Env_Name => No_Symbol, Kind => Current_Env); 
--# expr-done 187
--# expr-start 188 '<FieldAccess for name at parser.py:740>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:740






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Ent.Node);
--# end
--# expr-done 188







Cast_Expr := Self; 



   
      Cast_Result := Cast_Expr;


Inc_Ref (New_Struct);
New_Struct_1 := (Dest_Env => New_Struct, Key => Fld, Metadata => No_Metadata, Value => Cast_Result); 
--# expr-done 189

         Property_Result := New_Struct_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_280;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_280;




            raise;
      end;



   return Property_Result;
end Env_Mappings_2;
--# end

   







--# property-start Decl.name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:706
pragma Warnings (Off, "is not referenced");
 function Decl_P_Name
  
  (Node : Bare_Decl
  )

   return Symbol_Type
is
   Self : Bare_Decl  := Bare_Decl (Node);
      --# bind self Self

   


   Property_Result : Symbol_Type;

      

      Var_Expr_5 : Bare_Def_Id;
Fld : Bare_Def_Id;
Sym : Symbol_Type;
Scope_Result : Symbol_Type;
Result_Var : Symbol_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         


--# expr-start 190 '<FieldAccess for syn_name at parser.py:706>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:706






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Implementation.Decl_F_Syn_Name (Self);
--# expr-done 190
Var_Expr_5 := Fld;





if Var_Expr_5 /= No_Bare_Lkt_Node then
   



   --# scope-start


   

--# expr-start 191 '<GetSymbol at parser.py:706>' Sym D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:706

Sym := Get_Symbol (Var_Expr_5); 
--# expr-done 191
Scope_Result := Sym;


   --# end


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Symbol;
end if;




         Property_Result := Result_Var;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Decl_P_Name;
--# end

   







--# property-start Decl.full_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:710
pragma Warnings (Off, "is not referenced");
 function Decl_P_Full_Name
  
  (Node : Bare_Decl
  )

   return String_Type
is
   Self : Bare_Decl  := Bare_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_72 with Inline_Always;

      Fld : Symbol_Type;
Fld_1 : String_Type;

            procedure Finalizer_Scope_72 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_72;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 193 '<FieldAccess for image at parser.py:711>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:711



--# expr-start 192 '<FieldAccess for name at parser.py:711>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:711






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Self);
--# end
--# expr-done 192


   




Fld_1 := Liblktlang.Implementation.Image (Self => Fld);
--# expr-done 193

         Property_Result := Fld_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_72;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_72;




            raise;
      end;



   return Property_Result;
end Decl_P_Full_Name;
--# end

   







--# property-start Decl.call_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:763
pragma Warnings (Off, "is not referenced");
 function Decl_P_Call_Scope
  
  (Node : Bare_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Decl  := Bare_Decl (Node);
      --# bind self Self

   


   Property_Result : Lexical_Env;

      
            procedure Finalizer_Scope_78 with Inline_Always;

      Let_Result : Lexical_Env := Empty_Env;

            procedure Finalizer_Scope_78 is
            begin
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_78;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 194 '<Block at ???>' Let_Result None
--# scope-start

Let_Result := Empty_Env; Inc_Ref (Let_Result);
--# end
--# expr-done 194

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_78;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_78;




            raise;
      end;



   return Property_Result;
end Decl_P_Call_Scope;
--# end



   


      

   --
   --  Primitives for Bare_Base_Grammar_Rule_Decl
   --

   




      
   function Base_Grammar_Rule_Decl_F_Expr
     (Node : Bare_Base_Grammar_Rule_Decl) return Bare_Grammar_Expr
   is
      

         Kind : constant Lkt_Base_Grammar_Rule_Decl := Node.Kind;
   begin
         case Kind is
               when Lkt_Grammar_Rule_Decl =>
                     
         return Node.Grammar_Rule_Decl_F_Expr;
      
               when Lkt_Synthetic_Lexer_Decl =>
                     return No_Bare_Lkt_Node;
         end case;
   end;




   


      

   --
   --  Primitives for Bare_Grammar_Rule_Decl
   --

   



      
      procedure Initialize_Fields_For_Grammar_Rule_Decl
        (Self : Bare_Grammar_Rule_Decl
         ; Grammar_Rule_Decl_F_Syn_Name : Bare_Def_Id
         ; Grammar_Rule_Decl_F_Expr : Bare_Grammar_Expr
        ) is
      begin

            Self.Grammar_Rule_Decl_F_Syn_Name := Grammar_Rule_Decl_F_Syn_Name;
            Self.Grammar_Rule_Decl_F_Expr := Grammar_Rule_Decl_F_Expr;
         

      end Initialize_Fields_For_Grammar_Rule_Decl;


   







--# property-start GrammarRuleDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1314
pragma Warnings (Off, "is not referenced");
 function Grammar_Rule_Decl_P_Decl_Type_Name
  
  (Node : Bare_Grammar_Rule_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Grammar_Rule_Decl  := Bare_Grammar_Rule_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_105 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_105 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_105;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 195 '<String at parser.py:1314>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1314
Str := Create_String ("grammar rule declaration"); 
--# expr-done 195

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_105;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_105;




            raise;
      end;



   return Property_Result;
end Grammar_Rule_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Synthetic_Lexer_Decl
   --

   



      
      procedure Initialize_Fields_For_Synthetic_Lexer_Decl
        (Self : Bare_Synthetic_Lexer_Decl
        ) is
      begin

         
      Self.Synthetic_Lexer_Decl_Sym := No_Symbol;

      end Initialize_Fields_For_Synthetic_Lexer_Decl;


   







--# property-start SyntheticLexerDecl.name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1322
pragma Warnings (Off, "is not referenced");
 function Synthetic_Lexer_Decl_P_Name
  
  (Node : Bare_Synthetic_Lexer_Decl
  )

   return Symbol_Type
is
   Self : Bare_Synthetic_Lexer_Decl  := Bare_Synthetic_Lexer_Decl (Node);
      --# bind self Self

   


   Property_Result : Symbol_Type;

      

      Fld : Symbol_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 196 '<FieldAccess for sym at parser.py:1322>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1322






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Self.Synthetic_Lexer_Decl_Sym;
--# expr-done 196

         Property_Result := Fld;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Synthetic_Lexer_Decl_P_Name;
--# end

   







--# property-start SyntheticLexerDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1325
pragma Warnings (Off, "is not referenced");
 function Synthetic_Lexer_Decl_P_Decl_Type_Name
  
  (Node : Bare_Synthetic_Lexer_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Synthetic_Lexer_Decl  := Bare_Synthetic_Lexer_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_107 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_107 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_107;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 197 '<String at parser.py:1325>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1325
Str := Create_String ("synthetic lexer declaration"); 
--# expr-done 197

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_107;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_107;




            raise;
      end;



   return Property_Result;
end Synthetic_Lexer_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Base_Val_Decl
   --

   





   







--# property-start '[dispatcher]BaseValDecl.get_type' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Base_Val_Decl_P_Get_Type
  
  (Node : Bare_Base_Val_Decl
      ; No_Inference : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Base_Val_Decl  := Bare_Base_Val_Decl (Node);
      --# bind self Self

   

   --# bind no_inference No_Inference

   Property_Result : Internal_Entity_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Base_Val_Decl (Self.Kind) is
               when Lkt_Node_Decl =>
                  --# property-call-start NodeDecl.get_type
                  Property_Result := Node_Decl_P_Get_Type
                    (Self
                        , No_Inference
                        , E_Info
                    );
                  --# end
               when Lkt_Self_Decl =>
                  --# property-call-start SelfDecl.get_type
                  Property_Result := Self_Decl_P_Get_Type
                    (Self
                        , No_Inference
                        , E_Info
                    );
                  --# end
               when Lkt_Enum_Lit_Decl =>
                  --# property-call-start EnumLitDecl.get_type
                  Property_Result := Enum_Lit_Decl_P_Get_Type
                    (Self
                        , No_Inference
                        , E_Info
                    );
                  --# end
               when Lkt_Field_Decl .. Lkt_Fun_Arg_Decl | Lkt_Dyn_Var_Decl .. Lkt_Match_Val_Decl =>
                  --# property-call-start ExplicitlyTypedDecl.get_type
                  Property_Result := Explicitly_Typed_Decl_P_Get_Type
                    (Self
                        , No_Inference
                        , E_Info
                    );
                  --# end
               when Lkt_Lambda_Arg_Decl =>
                  --# property-call-start LambdaArgDecl.get_type
                  Property_Result := Lambda_Arg_Decl_P_Get_Type
                    (Self
                        , No_Inference
                        , E_Info
                    );
                  --# end
               when Lkt_Val_Decl =>
                  --# property-call-start ValDecl.get_type
                  Property_Result := Val_Decl_P_Get_Type
                    (Self
                        , No_Inference
                        , E_Info
                    );
                  --# end
               when Lkt_Fun_Decl =>
                  --# property-call-start FunDecl.get_type
                  Property_Result := Fun_Decl_P_Get_Type
                    (Self
                        , No_Inference
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Base_Val_Decl_P_Get_Type;
--# end



   


      

   --
   --  Primitives for Bare_Node_Decl
   --

   





   







--# property-start NodeDecl.name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2461
pragma Warnings (Off, "is not referenced");
 function Node_Decl_P_Name
  
  (Node : Bare_Node_Decl
  )

   return Symbol_Type
is
   Self : Bare_Node_Decl  := Bare_Node_Decl (Node);
      --# bind self Self

   


   Property_Result : Symbol_Type;

      

      Sym : Symbol_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 198 '<Symbol node at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Node_31); 
--# expr-done 198

         Property_Result := Sym;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Node_Decl_P_Name;
--# end

   







--# property-start NodeDecl.get_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2463
pragma Warnings (Off, "is not referenced");
 function Node_Decl_P_Get_Type
  
  (Node : Bare_Node_Decl
      ; No_Inference : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Node_Decl  := Bare_Node_Decl (Node);
     Ent : Internal_Entity_Node_Decl :=
       Internal_Entity_Node_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind no_inference No_Inference

   Property_Result : Internal_Entity_Type_Decl;

      

      Fld : Internal_Entity;
Cast_Result : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 201 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 200 '<Cast to TypeDecl at parser.py:2466>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2466





--# expr-start 199 '<parent at parser.py:2466>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2466






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 199



   if Fld.Node = null
      or else Fld.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Fld.Node,
         Info => Fld.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 200
Let_Result := Cast_Result; 
--# end
--# expr-done 201

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Node_Decl_P_Get_Type;
--# end

   







--# property-start NodeDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2468
pragma Warnings (Off, "is not referenced");
 function Node_Decl_P_Decl_Type_Name
  
  (Node : Bare_Node_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Node_Decl  := Bare_Node_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_193 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_193 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_193;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 202 '<String at parser.py:2468>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2468
Str := Create_String ("node declaration"); 
--# expr-done 202

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_193;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_193;




            raise;
      end;



   return Property_Result;
end Node_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Self_Decl
   --

   





   







--# property-start SelfDecl.name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2443
pragma Warnings (Off, "is not referenced");
 function Self_Decl_P_Name
  
  (Node : Bare_Self_Decl
  )

   return Symbol_Type
is
   Self : Bare_Self_Decl  := Bare_Self_Decl (Node);
      --# bind self Self

   


   Property_Result : Symbol_Type;

      

      Sym : Symbol_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 203 '<Symbol self at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Self); 
--# expr-done 203

         Property_Result := Sym;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Self_Decl_P_Name;
--# end

   







--# property-start SelfDecl.get_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2445
pragma Warnings (Off, "is not referenced");
 function Self_Decl_P_Get_Type
  
  (Node : Bare_Self_Decl
      ; No_Inference : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Self_Decl  := Bare_Self_Decl (Node);
     Ent : Internal_Entity_Self_Decl :=
       Internal_Entity_Self_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind no_inference No_Inference

   Property_Result : Internal_Entity_Type_Decl;

      

      Fld : Internal_Entity;
Cast_Result : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 206 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 205 '<Cast to TypeDecl at parser.py:2448>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2448





--# expr-start 204 '<parent at parser.py:2448>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2448






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 204



   if Fld.Node = null
      or else Fld.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Fld.Node,
         Info => Fld.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 205
Let_Result := Cast_Result; 
--# end
--# expr-done 206

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Self_Decl_P_Get_Type;
--# end

   







--# property-start SelfDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2450
pragma Warnings (Off, "is not referenced");
 function Self_Decl_P_Decl_Type_Name
  
  (Node : Bare_Self_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Self_Decl  := Bare_Self_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_190 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_190 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_190;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 207 '<String at parser.py:2450>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2450
Str := Create_String ("self declaration"); 
--# expr-done 207

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_190;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_190;




            raise;
      end;



   return Property_Result;
end Self_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_User_Val_Decl
   --

   







   


      

   --
   --  Primitives for Bare_Enum_Lit_Decl
   --

   



      
      procedure Initialize_Fields_For_Enum_Lit_Decl
        (Self : Bare_Enum_Lit_Decl
         ; Enum_Lit_Decl_F_Syn_Name : Bare_Def_Id
        ) is
      begin

            Self.Enum_Lit_Decl_F_Syn_Name := Enum_Lit_Decl_F_Syn_Name;
         

      end Initialize_Fields_For_Enum_Lit_Decl;


   







--# property-start EnumLitDecl.get_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2582
pragma Warnings (Off, "is not referenced");
 function Enum_Lit_Decl_P_Get_Type
  
  (Node : Bare_Enum_Lit_Decl
      ; No_Inference : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Enum_Lit_Decl  := Bare_Enum_Lit_Decl (Node);
     Ent : Internal_Entity_Enum_Lit_Decl :=
       Internal_Entity_Enum_Lit_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind no_inference No_Inference

   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_200 with Inline_Always;

      Fld : Internal_Entity_Array_Access;
Is_A : Boolean;
Find_Result : Internal_Entity;
Cast_Result : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_200 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_200;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 212 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 211 '<Cast to TypeDecl at parser.py:2587>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2587





--# expr-start 210 '<Find at parser.py:2586>' Find_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2586








   

   --# expr-start 208 '<FieldAccess for parents at parser.py:2586>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2586






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




Fld := Liblktlang.Implementation.Parents (Node => Ent.Node, With_Self => True, E_Info => Ent.Info);
--# expr-done 208

   
      Find_Result := No_Entity;
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Fld;
      begin
         for Item_9 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind t Item_9

            
      --# expr-start 209 '<IsA EnumTypeDecl at parser.py:2586>' Is_A D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2586

Is_A := Item_9.Node /= null 
and then Item_9.Node.Kind in Lkt_Enum_Type_Decl_Range; 
--# expr-done 209
      if Is_A then
         Find_Result := Item_9; 
      end if;
   

            
   --# end


               exit when Is_A;

         end loop;
      end;
   

   
   



--# expr-done 210



   if Find_Result.Node = null
      or else Find_Result.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Find_Result.Node,
         Info => Find_Result.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 211
Let_Result := Cast_Result; 
--# end
--# expr-done 212

         Property_Result := Let_Result;
         
   --# end
      Finalizer_Scope_200;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_200;




            raise;
      end;



   return Property_Result;
end Enum_Lit_Decl_P_Get_Type;
--# end

   







--# property-start EnumLitDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2593
pragma Warnings (Off, "is not referenced");
 function Enum_Lit_Decl_P_Decl_Type_Name
  
  (Node : Bare_Enum_Lit_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Enum_Lit_Decl  := Bare_Enum_Lit_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_201 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_201 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_201;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 213 '<String at parser.py:2593>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2593
Str := Create_String ("enum literal declaration"); 
--# expr-done 213

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_201;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_201;




            raise;
      end;



   return Property_Result;
end Enum_Lit_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Explicitly_Typed_Decl
   --

   




      
   function Explicitly_Typed_Decl_F_Decl_Type
     (Node : Bare_Explicitly_Typed_Decl) return Bare_Type_Ref
   is
      

         Kind : constant Lkt_Explicitly_Typed_Decl := Node.Kind;
   begin
         case Kind is
               when Lkt_Fun_Arg_Decl =>
                     
         return Node.Fun_Arg_Decl_F_Decl_Type;
      
               when Lkt_Lambda_Arg_Decl =>
                     
         return Node.Lambda_Arg_Decl_F_Decl_Type;
      
               when Lkt_Field_Decl =>
                     
         return Node.Field_Decl_F_Decl_Type;
      
               when Lkt_Match_Val_Decl =>
                     
         return Node.Match_Val_Decl_F_Decl_Type;
      
               when Lkt_Dyn_Var_Decl =>
                     
         return Node.Dyn_Var_Decl_F_Decl_Type;
      
               when Lkt_Val_Decl =>
                     
         return Node.Val_Decl_F_Decl_Type;
      
         end case;
   end;


   







--# property-start ExplicitlyTypedDecl.get_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2604
pragma Warnings (Off, "is not referenced");
 function Explicitly_Typed_Decl_P_Get_Type
  
  (Node : Bare_Explicitly_Typed_Decl
      ; No_Inference : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Explicitly_Typed_Decl  := Bare_Explicitly_Typed_Decl (Node);
     Ent : Internal_Entity_Explicitly_Typed_Decl :=
       Internal_Entity_Explicitly_Typed_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind no_inference No_Inference

   Property_Result : Internal_Entity_Type_Decl;

      

      Fld : Internal_Entity_Type_Ref;
Fld_1 : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 216 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 215 '<FieldAccess for designated_type at parser.py:2607>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2607



--# expr-start 214 '<FieldAccess for decl_type at parser.py:2607>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2607






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Type_Ref (Node => Implementation.Explicitly_Typed_Decl_F_Decl_Type (Ent.Node), Info => Ent.Info);
--# expr-done 214


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 215
Let_Result := Fld_1; 
--# end
--# expr-done 216

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Explicitly_Typed_Decl_P_Get_Type;
--# end



   


      

   --
   --  Primitives for Bare_Component_Decl
   --

   




      
   function Component_Decl_F_Default_Val
     (Node : Bare_Component_Decl) return Bare_Expr
   is
      

         Kind : constant Lkt_Component_Decl := Node.Kind;
   begin
         case Kind is
               when Lkt_Fun_Arg_Decl =>
                     
         return Node.Fun_Arg_Decl_F_Default_Val;
      
               when Lkt_Lambda_Arg_Decl =>
                     
         return Node.Lambda_Arg_Decl_F_Default_Val;
      
               when Lkt_Field_Decl =>
                     
         return Node.Field_Decl_F_Default_Val;
      
         end case;
   end;


   







--# property-start ComponentDecl.to_formal_param D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2621
pragma Warnings (Off, "is not referenced");
 function Component_Decl_P_To_Formal_Param
  
  (Node : Bare_Component_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Formal_Param
is
   Self : Bare_Component_Decl  := Bare_Component_Decl (Node);
     Ent : Internal_Entity_Component_Decl :=
       Internal_Entity_Component_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Formal_Param;

      

      Fld : Symbol_Type;
Fld_1 : Internal_Entity_Type_Decl;
Fld_2 : Internal_Entity_Expr;
New_Struct : Internal_Formal_Param;
Let_Result : Internal_Formal_Param;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 221 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 220 '<New FormalParam at parser.py:2626>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2626
--# expr-start 217 '<FieldAccess for default_val at parser.py:2628>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2628






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Create_Internal_Entity_Expr (Node => Implementation.Component_Decl_F_Default_Val (Ent.Node), Info => Ent.Info);
--# expr-done 217
--# expr-start 218 '<FieldAccess for name at parser.py:2626>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2626






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Self);
--# end
--# expr-done 218
--# expr-start 219 '<FieldAccess for get_type at parser.py:2627>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2627






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]BaseValDecl.get_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Base_Val_Decl_P_Get_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 219
New_Struct := (Default_Value => Fld_2, Formal_Name => Fld, Formal_Type => Fld_1); 
--# expr-done 220
Let_Result := New_Struct; 
--# end
--# expr-done 221

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Component_Decl_P_To_Formal_Param;
--# end



   


      

   --
   --  Primitives for Bare_Field_Decl
   --

   



      
      procedure Initialize_Fields_For_Field_Decl
        (Self : Bare_Field_Decl
         ; Field_Decl_F_Syn_Name : Bare_Def_Id
         ; Field_Decl_F_Decl_Type : Bare_Type_Ref
         ; Field_Decl_F_Default_Val : Bare_Expr
        ) is
      begin

            Self.Field_Decl_F_Syn_Name := Field_Decl_F_Syn_Name;
            Self.Field_Decl_F_Decl_Type := Field_Decl_F_Decl_Type;
            Self.Field_Decl_F_Default_Val := Field_Decl_F_Default_Val;
         

      end Initialize_Fields_For_Field_Decl;


   







--# property-start FieldDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2693
pragma Warnings (Off, "is not referenced");
 function Field_Decl_P_Decl_Type_Name
  
  (Node : Bare_Field_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Field_Decl  := Bare_Field_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_209 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_209 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_209;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 222 '<String at parser.py:2693>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2693
Str := Create_String ("field declaration"); 
--# expr-done 222

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_209;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_209;




            raise;
      end;



   return Property_Result;
end Field_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Fun_Arg_Decl
   --

   



      
      procedure Initialize_Fields_For_Fun_Arg_Decl
        (Self : Bare_Fun_Arg_Decl
         ; Fun_Arg_Decl_F_Decl_Annotations : Bare_Decl_Annotation_List
         ; Fun_Arg_Decl_F_Syn_Name : Bare_Def_Id
         ; Fun_Arg_Decl_F_Decl_Type : Bare_Type_Ref
         ; Fun_Arg_Decl_F_Default_Val : Bare_Expr
        ) is
      begin

            Self.Fun_Arg_Decl_F_Decl_Annotations := Fun_Arg_Decl_F_Decl_Annotations;
            Self.Fun_Arg_Decl_F_Syn_Name := Fun_Arg_Decl_F_Syn_Name;
            Self.Fun_Arg_Decl_F_Decl_Type := Fun_Arg_Decl_F_Decl_Type;
            Self.Fun_Arg_Decl_F_Default_Val := Fun_Arg_Decl_F_Default_Val;
         

      end Initialize_Fields_For_Fun_Arg_Decl;

      
   function Fun_Arg_Decl_F_Decl_Annotations
     (Node : Bare_Fun_Arg_Decl) return Bare_Decl_Annotation_List
   is
      

   begin
         
         return Node.Fun_Arg_Decl_F_Decl_Annotations;
      
   end;


   







--# property-start FunArgDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2639
pragma Warnings (Off, "is not referenced");
 function Fun_Arg_Decl_P_Decl_Type_Name
  
  (Node : Bare_Fun_Arg_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Fun_Arg_Decl  := Bare_Fun_Arg_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_204 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_204 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_204;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 223 '<String at parser.py:2639>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2639
Str := Create_String ("fun arg declaration"); 
--# expr-done 223

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_204;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_204;




            raise;
      end;



   return Property_Result;
end Fun_Arg_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Lambda_Arg_Decl
   --

   



      
      procedure Initialize_Fields_For_Lambda_Arg_Decl
        (Self : Bare_Lambda_Arg_Decl
         ; Lambda_Arg_Decl_F_Syn_Name : Bare_Def_Id
         ; Lambda_Arg_Decl_F_Decl_Type : Bare_Type_Ref
         ; Lambda_Arg_Decl_F_Default_Val : Bare_Expr
        ) is
      begin

            Self.Lambda_Arg_Decl_F_Syn_Name := Lambda_Arg_Decl_F_Syn_Name;
            Self.Lambda_Arg_Decl_F_Decl_Type := Lambda_Arg_Decl_F_Decl_Type;
            Self.Lambda_Arg_Decl_F_Default_Val := Lambda_Arg_Decl_F_Default_Val;
         

      end Initialize_Fields_For_Lambda_Arg_Decl;


   







--# property-start LambdaArgDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2650
pragma Warnings (Off, "is not referenced");
 function Lambda_Arg_Decl_P_Decl_Type_Name
  
  (Node : Bare_Lambda_Arg_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Lambda_Arg_Decl  := Bare_Lambda_Arg_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_205 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_205 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_205;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 224 '<String at parser.py:2650>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2650
Str := Create_String ("lambda arg declaration"); 
--# expr-done 224

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_205;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_205;




            raise;
      end;



   return Property_Result;
end Lambda_Arg_Decl_P_Decl_Type_Name;
--# end

   







--# property-start LambdaArgDecl.owning_lambda D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2652
pragma Warnings (Off, "is not referenced");
 function Lambda_Arg_Decl_P_Owning_Lambda
  
  (Node : Bare_Lambda_Arg_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Lambda_Expr
is
   Self : Bare_Lambda_Arg_Decl  := Bare_Lambda_Arg_Decl (Node);
     Ent : Internal_Entity_Lambda_Arg_Decl :=
       Internal_Entity_Lambda_Arg_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Lambda_Expr;

      
            procedure Finalizer_Scope_206 with Inline_Always;

      Fld : Internal_Entity_Array_Access;
Is_A : Boolean;
Find_Result : Internal_Entity;
Cast_Result : Internal_Entity_Lambda_Expr;

            procedure Finalizer_Scope_206 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_206;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 228 '<Cast to LambdaExpr at parser.py:2653>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2653





--# expr-start 227 '<Find at parser.py:2653>' Find_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2653








   

   --# expr-start 225 '<FieldAccess for parents at parser.py:2653>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2653






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




Fld := Liblktlang.Implementation.Parents (Node => Ent.Node, With_Self => True, E_Info => Ent.Info);
--# expr-done 225

   
      Find_Result := No_Entity;
   

   

      

      declare
         
         Collection : constant Internal_Entity_Array_Access := Fld;
      begin
         for Item_10 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind p Item_10

            
      --# expr-start 226 '<IsA LambdaExpr at parser.py:2653>' Is_A D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2653

Is_A := Item_10.Node /= null 
and then Item_10.Node.Kind in Lkt_Lambda_Expr_Range; 
--# expr-done 226
      if Is_A then
         Find_Result := Item_10; 
      end if;
   

            
   --# end


               exit when Is_A;

         end loop;
      end;
   

   
   



--# expr-done 227



   if Find_Result.Node = null
      or else Find_Result.Node.Kind in Lkt_Lambda_Expr_Range
   then
      
      Cast_Result := Create_Internal_Entity_Lambda_Expr
        (Node => Find_Result.Node,
         Info => Find_Result.Info);

   else
         Cast_Result := No_Entity_Lambda_Expr;
   end if;


--# expr-done 228

         Property_Result := Cast_Result;
         
   --# end
      Finalizer_Scope_206;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_206;




            raise;
      end;



   return Property_Result;
end Lambda_Arg_Decl_P_Owning_Lambda;
--# end

   







--# property-start LambdaArgDecl.index D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2656
pragma Warnings (Off, "is not referenced");
 function Lambda_Arg_Decl_P_Index
  
  (Node : Bare_Lambda_Arg_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Integer
is
   Self : Bare_Lambda_Arg_Decl  := Bare_Lambda_Arg_Decl (Node);
     Ent : Internal_Entity_Lambda_Arg_Decl :=
       Internal_Entity_Lambda_Arg_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Integer;

      
            procedure Finalizer_Scope_207 with Inline_Always;

      Fld : Internal_Entity;
Cast_Result : Internal_Entity_Lambda_Arg_Decl_List;
Fld_1 : Bare_Lambda_Arg_Decl_List;
Fld_2 : Internal_Entity_Info;
Item_11 : Internal_Entity_Lambda_Arg_Decl;
Bare_Item_11 : Bare_Lambda_Arg_Decl;
Index_12 : Integer;
As_Entity : Internal_Entity_Lambda_Arg_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Lambda_Arg_Decl;
Cast_Expr : Internal_Entity_Lambda_Arg_Decl;
Cast_Result_1 : Internal_Entity;
Cast_Expr_1 : Internal_Entity_Lambda_Arg_Decl;
Cast_Result_2 : Internal_Entity;
Is_Equiv : Boolean;
Map_Result : Integer_Array_Access;
Get_Result : Integer;
Let_Result : Integer;

            procedure Finalizer_Scope_207 is
            begin
                     Dec_Ref (Map_Result);
            end Finalizer_Scope_207;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lambda_Arg_Decl_Lambda_Arg_Decl_P_Index,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lambda_Arg_Decl,
                                As_Bare_Lambda_Arg_Decl => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);




      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Integer;

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 234 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 233 '<at at parser.py:2664>' Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2664

--# expr-start 232 '<FilterMap at parser.py:2661>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2661








   

   --# expr-start 230 '<Cast to LambdaArgDecl.list at parser.py:2661>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2661





--# expr-start 229 '<parent at parser.py:2661>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2661






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 229



   if Fld.Node = null
      or else Fld.Node.Kind in Lkt_Lambda_Arg_Decl_List_Range
   then
      
      Cast_Result := Create_Internal_Entity_Lambda_Arg_Decl_List
        (Node => Fld.Node,
         Info => Fld.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 230






   




Fld_1 := Cast_Result.Node;

   
      declare
         Map_Result_Vec : Integer_Vectors.Vector;
      begin
   

   

      if Fld_1 = null then
         
      Map_Result := Create_Integer_Array (0);
   
      else
         
         Index_12 := 0;

      declare
         
         Collection : constant Bare_Lambda_Arg_Decl_List := Fld_1;
      begin
         for Untyped_Item_11 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_11 := Untyped_Item_11; 
                  



Is_Null := Bare_Item_11 = null; 
if Is_Null then
   
   If_Result := No_Entity_Lambda_Arg_Decl;
else
   





   




Fld_2 := Cast_Result.Info;

As_Entity := (Info => Fld_2, Node => Bare_Item_11); 
   If_Result := As_Entity;
end if;



                  Item_11 := If_Result; 

            
   --# scope-start


               --# bind arg Item_11
               --# bind i Index_12

            
         --# expr-start 231 '<Eq at parser.py:2663>' Is_Equiv D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2663






Cast_Expr := Ent; 



   
      Cast_Result_1 := Create_Internal_Entity
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);








Cast_Expr_1 := Item_11; 



   
      Cast_Result_2 := Create_Internal_Entity
        (Node => Cast_Expr_1.Node,
         Info => Cast_Expr_1.Info);


Is_Equiv := Equivalent (Cast_Result_1, Cast_Result_2); 
--# expr-done 231
         if Is_Equiv then
            
         
      
      

         declare
            Item_To_Append : constant Integer := Index_12;
         begin
            Integer_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end



               Index_12 := Index_12 + 1;
         end loop;
      end;
   
      end if;

   

         Map_Result := Create_Integer_Array
           (Items_Count => Natural (Integer_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Integer_Vectors.Get
              (Map_Result_Vec,
               I + Integer_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Integer_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 232


Get_Result := Get (Self, Map_Result, 0, True); 
--# expr-done 233
Let_Result := Get_Result; 
--# end
--# expr-done 234

         Property_Result := Let_Result;
         
   --# end
      Finalizer_Scope_207;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_207;


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Integer,
                        As_Integer => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);

         end if;
      end if;



   return Property_Result;
end Lambda_Arg_Decl_P_Index;
--# end

   







--# property-start LambdaArgDecl.get_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2666
pragma Warnings (Off, "is not referenced");
 function Lambda_Arg_Decl_P_Get_Type
  
  (Node : Bare_Lambda_Arg_Decl
      ; No_Inference : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Lambda_Arg_Decl  := Bare_Lambda_Arg_Decl (Node);
     Ent : Internal_Entity_Lambda_Arg_Decl :=
       Internal_Entity_Lambda_Arg_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind no_inference No_Inference

   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_342 with Inline_Always;

      Var_Expr : Internal_Entity_Function_Type;
Fld : Internal_Entity_Type_Ref;
Fld_1 : Internal_Entity_Type_Decl;
Fld_2 : Internal_Entity_Lambda_Expr;
Fld_3 : Internal_Entity_Type_Decl;
Cast_Result : Internal_Entity_Function_Type;
Fld_4 : Integer;
Fld_5 : Bare_Type_Decl_Array_Access;
Get_Result : Bare_Type_Decl;
As_Entity : Internal_Entity_Type_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;
Result_Var : Internal_Entity_Type_Decl;
If_Result_1 : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_342 is
            begin
                     Dec_Ref (Fld_5);
            end Finalizer_Scope_342;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 245 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 244 '<If at parser.py:2668>' If_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2668




if No_Inference then
   --# expr-start 236 '<FieldAccess for designated_type at parser.py:2672>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2672



--# expr-start 235 '<FieldAccess for decl_type at parser.py:2672>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2672






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Type_Ref (Node => Ent.Node.Lambda_Arg_Decl_F_Decl_Type, Info => Ent.Info);
--# expr-done 235


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 236
   If_Result_1 := Fld_1;
else
   


--# expr-start 239 '<Cast to FunctionType at parser.py:2679>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2679





--# expr-start 238 '<FieldAccess for expected_type at parser.py:2678>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2678



--# expr-start 237 '<FieldAccess for owning_lambda at parser.py:2678>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2678






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LambdaArgDecl.owning_lambda
Fld_2 := Liblktlang.Implementation.Lambda_Arg_Decl_P_Owning_Lambda (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 237


   

      if Fld_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Expr.expected_type
Fld_3 := Liblktlang.Implementation.Expr_P_Expected_Type (Node => Fld_2.Node, E_Info => Fld_2.Info);
--# end
--# expr-done 238



   if Fld_3.Node = null
      or else Fld_3.Node.Kind in Lkt_Function_Type_Range
   then
      
      Cast_Result := Create_Internal_Entity_Function_Type
        (Node => Fld_3.Node,
         Info => Fld_3.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 239
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Function_Type then
   



   --# scope-start


   --# bind ft Var_Expr

--# expr-start 243 '<as_entity at parser.py:2680>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2680



--# expr-start 242 '<at at parser.py:2680>' Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2680

--# expr-start 240 '<FieldAccess for args at parser.py:2680>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2680






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_5 := Var_Expr.Node.Function_Type_Args;
Inc_Ref (Fld_5);
--# expr-done 240
--# expr-start 241 '<FieldAccess for index at parser.py:2680>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2680






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LambdaArgDecl.index
Fld_4 := Liblktlang.Implementation.Lambda_Arg_Decl_P_Index (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 241

Get_Result := Get (Self, Fld_5, Fld_4, True); 
--# expr-done 242
Is_Null := Get_Result = null; 
if Is_Null then
   
   If_Result := No_Entity_Type_Decl;
else
   

As_Entity := (Info => E_Info, Node => Get_Result); 
   If_Result := As_Entity;
end if;



--# expr-done 243
Scope_Result := If_Result;


   --# end
      Finalizer_Scope_342;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Entity_Type_Decl;
end if;



   If_Result_1 := Result_Var;
end if;



--# expr-done 244
Let_Result := If_Result_1; 
--# end
--# expr-done 245

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_342;




            raise;
      end;



   return Property_Result;
end Lambda_Arg_Decl_P_Get_Type;
--# end



   


      

   --
   --  Primitives for Bare_Dyn_Var_Decl
   --

   



      
      procedure Initialize_Fields_For_Dyn_Var_Decl
        (Self : Bare_Dyn_Var_Decl
         ; Dyn_Var_Decl_F_Syn_Name : Bare_Def_Id
         ; Dyn_Var_Decl_F_Decl_Type : Bare_Type_Ref
        ) is
      begin

            Self.Dyn_Var_Decl_F_Syn_Name := Dyn_Var_Decl_F_Syn_Name;
            Self.Dyn_Var_Decl_F_Decl_Type := Dyn_Var_Decl_F_Decl_Type;
         

      end Initialize_Fields_For_Dyn_Var_Decl;


   







--# property-start DynVarDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3733
pragma Warnings (Off, "is not referenced");
 function Dyn_Var_Decl_P_Decl_Type_Name
  
  (Node : Bare_Dyn_Var_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Dyn_Var_Decl  := Bare_Dyn_Var_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_255 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_255 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_255;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 246 '<String at parser.py:3733>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3733
Str := Create_String ("dynamic variable declaration"); 
--# expr-done 246

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_255;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_255;




            raise;
      end;



   return Property_Result;
end Dyn_Var_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Match_Val_Decl
   --

   



      
      procedure Initialize_Fields_For_Match_Val_Decl
        (Self : Bare_Match_Val_Decl
         ; Match_Val_Decl_F_Syn_Name : Bare_Def_Id
         ; Match_Val_Decl_F_Decl_Type : Bare_Type_Ref
        ) is
      begin

            Self.Match_Val_Decl_F_Syn_Name := Match_Val_Decl_F_Syn_Name;
            Self.Match_Val_Decl_F_Decl_Type := Match_Val_Decl_F_Decl_Type;
         

      end Initialize_Fields_For_Match_Val_Decl;


   







--# property-start MatchValDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3686
pragma Warnings (Off, "is not referenced");
 function Match_Val_Decl_P_Decl_Type_Name
  
  (Node : Bare_Match_Val_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Match_Val_Decl  := Bare_Match_Val_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_253 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_253 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_253;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 247 '<String at parser.py:3686>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3686
Str := Create_String ("match value declaration"); 
--# expr-done 247

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_253;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_253;




            raise;
      end;



   return Property_Result;
end Match_Val_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Val_Decl
   --

   



      
      procedure Initialize_Fields_For_Val_Decl
        (Self : Bare_Val_Decl
         ; Val_Decl_F_Syn_Name : Bare_Def_Id
         ; Val_Decl_F_Decl_Type : Bare_Type_Ref
         ; Val_Decl_F_Val : Bare_Expr
        ) is
      begin

            Self.Val_Decl_F_Syn_Name := Val_Decl_F_Syn_Name;
            Self.Val_Decl_F_Decl_Type := Val_Decl_F_Decl_Type;
            Self.Val_Decl_F_Val := Val_Decl_F_Val;
         

      end Initialize_Fields_For_Val_Decl;

      
   function Val_Decl_F_Val
     (Node : Bare_Val_Decl) return Bare_Expr
   is
      

   begin
         
         return Node.Val_Decl_F_Val;
      
   end;


   







--# property-start ValDecl.get_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3744
pragma Warnings (Off, "is not referenced");
 function Val_Decl_P_Get_Type
  
  (Node : Bare_Val_Decl
      ; No_Inference : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Val_Decl  := Bare_Val_Decl (Node);
     Ent : Internal_Entity_Val_Decl :=
       Internal_Entity_Val_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind no_inference No_Inference

   Property_Result : Internal_Entity_Type_Decl;

      

      Var_Expr : Internal_Entity_Type_Ref;
Fld : Internal_Entity_Type_Ref;
Fld_1 : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;
Fld_2 : Internal_Entity_Expr;
Fld_3 : Internal_Entity_Type_Decl;
If_Result : Internal_Entity_Type_Decl;
Result_Var : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 253 '<Block at ???>' Let_Result None
--# scope-start



--# expr-start 248 '<FieldAccess for decl_type at parser.py:3748>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3748






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Type_Ref (Node => Ent.Node.Val_Decl_F_Decl_Type, Info => Ent.Info);
--# expr-done 248
Var_Expr := Fld;





if Var_Expr /= No_Entity_Type_Ref then
   



   --# scope-start


   --# bind tpe Var_Expr

--# expr-start 249 '<FieldAccess for designated_type at parser.py:3749>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3749






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 249
Scope_Result := Fld_1;


   --# end


   Result_Var := Scope_Result;
else
   --# expr-start 252 '<If at parser.py:3750>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3750




if No_Inference then
   
   If_Result := No_Entity_Type_Decl;
else
   --# expr-start 251 '<FieldAccess for check_expr_type at parser.py:3753>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3753



--# expr-start 250 '<FieldAccess for val at parser.py:3753>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3753






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Create_Internal_Entity_Expr (Node => Ent.Node.Val_Decl_F_Val, Info => Ent.Info);
--# expr-done 250


   

      if Fld_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Expr.check_expr_type
Fld_3 := Liblktlang.Implementation.Expr_P_Check_Expr_Type (Node => Fld_2.Node, E_Info => Fld_2.Info);
--# end
--# expr-done 251
   If_Result := Fld_3;
end if;



--# expr-done 252
   Result_Var := If_Result;
end if;



Let_Result := Result_Var; 
--# end
--# expr-done 253

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Val_Decl_P_Get_Type;
--# end

   







--# property-start ValDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3757
pragma Warnings (Off, "is not referenced");
 function Val_Decl_P_Decl_Type_Name
  
  (Node : Bare_Val_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Val_Decl  := Bare_Val_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_257 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_257 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_257;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 254 '<String at parser.py:3757>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:3757
Str := Create_String ("value declaration"); 
--# expr-done 254

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_257;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_257;




            raise;
      end;



   return Property_Result;
end Val_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Fun_Decl
   --

   



   

   

   

   

   

   

   


      procedure Fun_Decl_Pre_Env_Actions
        (Self            : Bare_Fun_Decl;
         State           : in out PLE_Node_State;
         Add_To_Env_Only : Boolean := False) is
      begin
            


   

      declare
         Resolver : constant Entity_Resolver :=
            null;

         

         Mapping : Internal_Env_Assoc :=
            Liblktlang.Implementation.Env_Mappings_17 (Node => 
Self);
      begin

         Add_To_Env
           (Self,
            State,
            Mapping.Key,
            Mapping.Value,
            Mapping.Metadata,
            Resolver,
            Mapping.Dest_Env,
            DSL_Location => "parser.py:2495");
         Dec_Ref (Mapping.Dest_Env);

      end;
   
   
            


   
      if Add_To_Env_Only then
         return;
      end if;

      declare
         No_Parent         : constant Boolean :=
            False;
         Transitive_Parent : constant Boolean :=
            Liblktlang.Implementation.Env_Trans_Parent_18 (Node => 
Self);
         Names             : Symbol_Type_Array_Access :=
            null;
      begin
         Add_Env (Self, State, No_Parent, Transitive_Parent, Names);
      end;
   
   
            


   

      declare
         Resolver : constant Entity_Resolver :=
            null;

         

         Mappings : Internal_Env_Assoc_Array_Access :=
            Liblktlang.Implementation.Env_Mappings_19 (Node => 
Self);
      begin
         for Mapping of Mappings.Items loop

         Add_To_Env
           (Self,
            State,
            Mapping.Key,
            Mapping.Value,
            Mapping.Metadata,
            Resolver,
            Mapping.Dest_Env,
            DSL_Location => "parser.py:2498");

         end loop;
         Dec_Ref (Mappings);
      end;
   
   
      end;



      
      procedure Initialize_Fields_For_Fun_Decl
        (Self : Bare_Fun_Decl
         ; Fun_Decl_F_Syn_Name : Bare_Def_Id
         ; Fun_Decl_F_Args : Bare_Fun_Arg_Decl_List
         ; Fun_Decl_F_Return_Type : Bare_Type_Ref
         ; Fun_Decl_F_Body : Bare_Expr
        ) is
      begin

            Self.Fun_Decl_F_Syn_Name := Fun_Decl_F_Syn_Name;
            Self.Fun_Decl_F_Args := Fun_Decl_F_Args;
            Self.Fun_Decl_F_Return_Type := Fun_Decl_F_Return_Type;
            Self.Fun_Decl_F_Body := Fun_Decl_F_Body;
         

      end Initialize_Fields_For_Fun_Decl;

      
   function Fun_Decl_F_Args
     (Node : Bare_Fun_Decl) return Bare_Fun_Arg_Decl_List
   is
      

   begin
         
         return Node.Fun_Decl_F_Args;
      
   end;

      
   function Fun_Decl_F_Return_Type
     (Node : Bare_Fun_Decl) return Bare_Type_Ref
   is
      

   begin
         
         return Node.Fun_Decl_F_Return_Type;
      
   end;

      
   function Fun_Decl_F_Body
     (Node : Bare_Fun_Decl) return Bare_Expr
   is
      

   begin
         
         return Node.Fun_Decl_F_Body;
      
   end;


   







--# property-start FunDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2488
pragma Warnings (Off, "is not referenced");
 function Fun_Decl_P_Decl_Type_Name
  
  (Node : Bare_Fun_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_194 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_194 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_194;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 255 '<String at parser.py:2488>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2488
Str := Create_String ("function declaration"); 
--# expr-done 255

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_194;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_194;




            raise;
      end;



   return Property_Result;
end Fun_Decl_P_Decl_Type_Name;
--# end

   







--# property-start FunDecl.owning_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2490
pragma Warnings (Off, "is not referenced");
 function Fun_Decl_P_Owning_Type
  
  (Node : Bare_Fun_Decl
  )

   return Bare_Type_Decl
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
      --# bind self Self

   


   Property_Result : Bare_Type_Decl;

      
            procedure Finalizer_Scope_195 with Inline_Always;

      Node_Parents : Bare_Lkt_Node_Array_Access;
Is_A : Boolean;
Find_Result : Bare_Lkt_Node;
Cast_Result : Bare_Type_Decl;

            procedure Finalizer_Scope_195 is
            begin
                     Dec_Ref (Node_Parents);
            end Finalizer_Scope_195;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 259 '<Cast to TypeDecl at parser.py:2491>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2491





--# expr-start 258 '<Find at parser.py:2491>' Find_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2491








   

   --# expr-start 256 '<FieldAccess for parents at parser.py:2491>' Node_Parents D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2491






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




Node_Parents := Parents (Self, True); 
--# expr-done 256

   
      Find_Result := No_Bare_Lkt_Node;
   

   

      

      declare
         
         Collection : constant Bare_Lkt_Node_Array_Access := Node_Parents;
      begin
         for Item_13 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind t Item_13

            
      --# expr-start 257 '<IsA TypeDecl at parser.py:2491>' Is_A D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2491

Is_A := Item_13 /= null 
and then Item_13.Kind in Lkt_Type_Decl; 
--# expr-done 257
      if Is_A then
         Find_Result := Item_13; 
      end if;
   

            
   --# end


               exit when Is_A;

         end loop;
      end;
   

   
   



--# expr-done 258



   if Find_Result = null
      or else Find_Result.Kind in Lkt_Type_Decl
   then
      
      Cast_Result := Find_Result;

   else
         Cast_Result := No_Bare_Lkt_Node;
   end if;


--# expr-done 259

         Property_Result := Cast_Result;
         
   --# end
      Finalizer_Scope_195;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_195;




            raise;
      end;



   return Property_Result;
end Fun_Decl_P_Owning_Type;
--# end

   







--# property-start FunDecl.call_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2504
pragma Warnings (Off, "is not referenced");
 function Fun_Decl_P_Call_Scope
  
  (Node : Bare_Fun_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
     Ent : Internal_Entity_Fun_Decl :=
       Internal_Entity_Fun_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Lexical_Env;

      
            procedure Finalizer_Scope_196 with Inline_Always;

      Fld : Lexical_Env := Empty_Env;

            procedure Finalizer_Scope_196 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_196;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 260 '<FieldAccess for children_env at parser.py:2504>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2504






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Children_Env (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 260

         Property_Result := Fld;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_196;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_196;




            raise;
      end;



   return Property_Result;
end Fun_Decl_P_Call_Scope;
--# end

   







--# property-start FunDecl.is_property D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2506
pragma Warnings (Off, "is not referenced");
 function Fun_Decl_P_Is_Property
  
  (Node : Bare_Fun_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
     Ent : Internal_Entity_Fun_Decl :=
       Internal_Entity_Fun_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Boolean;

      

      Fld : Internal_Entity_Full_Decl;
Sym : Symbol_Type;
Fld_1 : Boolean;
Let_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 264 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 263 '<FieldAccess for has_annotation at parser.py:2515>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2515



--# expr-start 261 '<FieldAccess for full_decl at parser.py:2515>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2515






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.full_decl
Fld := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 261


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 262 '<Symbol property at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Property); 
--# expr-done 262
--# property-call-start FullDecl.has_annotation
Fld_1 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Fld.Node, Name => Sym);
--# end
--# expr-done 263
Let_Result := Fld_1; 
--# end
--# expr-done 264

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Fun_Decl_P_Is_Property;
--# end

   







--# property-start FunDecl.get_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2517
pragma Warnings (Off, "is not referenced");
 function Fun_Decl_P_Get_Type
  
  (Node : Bare_Fun_Decl
      ; No_Inference : Boolean
         := False
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
     Ent : Internal_Entity_Fun_Decl :=
       Internal_Entity_Fun_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind no_inference No_Inference

   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_198 with Inline_Always;

      Fld : Internal_Entity_Type_Ref;
Fld_1 : Internal_Entity_Type_Decl;
Fld_2 : Internal_Entity_Fun_Arg_Decl_List;
Fld_3 : Bare_Fun_Arg_Decl_List;
Fld_4 : Internal_Entity_Info;
Item_14 : Internal_Entity_Fun_Arg_Decl;
Bare_Item_14 : Bare_Fun_Arg_Decl;
Fld_5 : Internal_Entity_Type_Ref;
Fld_6 : Internal_Entity_Type_Decl;
Fld_7 : Bare_Decl;
Cast_Result : Bare_Type_Decl;
As_Entity : Internal_Entity_Fun_Arg_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Fun_Arg_Decl;
Map_Result : Bare_Type_Decl_Array_Access;
Fld_8 : Internal_Entity_Type_Ref;
Fld_9 : Internal_Entity_Type_Decl;
Fld_10 : Bare_Decl;
Cast_Result_1 : Bare_Type_Decl;
Fld_11 : Bare_Function_Type;
As_Entity_1 : Internal_Entity_Function_Type;
Is_Null_1 : Boolean;
If_Result_1 : Internal_Entity_Function_Type;
Cast_Result_2 : Internal_Entity_Type_Decl;
Fld_12 : Boolean;
If_Result_2 : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_198 is
            begin
                     Dec_Ref (Map_Result);
            end Finalizer_Scope_198;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 281 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 280 '<If at parser.py:2520>' If_Result_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2520



--# expr-start 265 '<FieldAccess for is_property at parser.py:2521>' Fld_12 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2521






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start FunDecl.is_property
Fld_12 := Liblktlang.Implementation.Fun_Decl_P_Is_Property (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 265
if Fld_12 then
   --# expr-start 267 '<FieldAccess for designated_type at parser.py:2523>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2523



--# expr-start 266 '<FieldAccess for return_type at parser.py:2523>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2523






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Type_Ref (Node => Ent.Node.Fun_Decl_F_Return_Type, Info => Ent.Info);
--# expr-done 266


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 267
   If_Result_2 := Fld_1;
else
   




--# expr-start 279 '<as_bare_entity at parser.py:2539>' If_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2539



--# expr-start 278 '<FieldAccess for function_type at parser.py:2525>' Fld_11 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2525






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 273 '<Map at parser.py:2534>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2534








   

   --# expr-start 268 '<FieldAccess for args at parser.py:2534>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2534






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Create_Internal_Entity_Fun_Arg_Decl_List (Node => Ent.Node.Fun_Decl_F_Args, Info => Ent.Info);
--# expr-done 268






   




Fld_3 := Fld_2.Node;

   
      declare
         Map_Result_Vec : Bare_Type_Decl_Vectors.Vector;
      begin
   

   

      if Fld_3 = null then
         
      Map_Result := Create_Bare_Type_Decl_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Fun_Arg_Decl_List := Fld_3;
      begin
         for Untyped_Item_14 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_14 := Untyped_Item_14; 
                  



Is_Null := Bare_Item_14 = null; 
if Is_Null then
   
   If_Result := No_Entity_Fun_Arg_Decl;
else
   





   




Fld_4 := Fld_2.Info;

As_Entity := (Info => Fld_4, Node => Bare_Item_14); 
   If_Result := As_Entity;
end if;



                  Item_14 := If_Result; 

            
   --# scope-start


               --# bind a Item_14

            
         
         
      --# expr-start 272 '<Cast to TypeDecl at parser.py:2536>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2536





--# expr-start 271 '<FieldAccess for assert_bare at parser.py:2536>' Fld_7 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2536



--# expr-start 270 '<FieldAccess for designated_type at parser.py:2536>' Fld_6 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2536



--# expr-start 269 '<FieldAccess for decl_type at parser.py:2536>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2536






   

      if Item_14.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_5 := Create_Internal_Entity_Type_Ref (Node => Item_14.Node.Fun_Arg_Decl_F_Decl_Type, Info => Item_14.Info);
--# expr-done 269


   

      if Fld_5.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_6 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Fld_5.Node, E_Info => Fld_5.Info);
--# end
--# expr-done 270


   

      if Fld_6.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.assert_bare
Fld_7 := Liblktlang.Implementation.Decl_P_Assert_Bare (Node => Fld_6.Node, E_Info => Fld_6.Info);
--# end
--# expr-done 271



   if Fld_7 = null
      or else Fld_7.Kind in Lkt_Type_Decl
   then
      
      Cast_Result := Fld_7;

   else
         Cast_Result := No_Bare_Lkt_Node;
   end if;


--# expr-done 272
      

         declare
            Item_To_Append : constant Bare_Type_Decl := Cast_Result;
         begin
            Bare_Type_Decl_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   
      end if;

   

         Map_Result := Create_Bare_Type_Decl_Array
           (Items_Count => Natural (Bare_Type_Decl_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Bare_Type_Decl_Vectors.Get
              (Map_Result_Vec,
               I + Bare_Type_Decl_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Bare_Type_Decl_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 273
--# expr-start 277 '<Cast to TypeDecl at parser.py:2538>' Cast_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2538





--# expr-start 276 '<FieldAccess for assert_bare at parser.py:2538>' Fld_10 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2538



--# expr-start 275 '<FieldAccess for designated_type at parser.py:2538>' Fld_9 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2538



--# expr-start 274 '<FieldAccess for return_type at parser.py:2538>' Fld_8 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2538






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_8 := Create_Internal_Entity_Type_Ref (Node => Ent.Node.Fun_Decl_F_Return_Type, Info => Ent.Info);
--# expr-done 274


   

      if Fld_8.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_9 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Fld_8.Node, E_Info => Fld_8.Info);
--# end
--# expr-done 275


   

      if Fld_9.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.assert_bare
Fld_10 := Liblktlang.Implementation.Decl_P_Assert_Bare (Node => Fld_9.Node, E_Info => Fld_9.Info);
--# end
--# expr-done 276



   if Fld_10 = null
      or else Fld_10.Kind in Lkt_Type_Decl
   then
      
      Cast_Result_1 := Fld_10;

   else
         Cast_Result_1 := No_Bare_Lkt_Node;
   end if;


--# expr-done 277
--# property-call-start LktNode.function_type
Fld_11 := Liblktlang.Implementation.Lkt_Node_P_Function_Type (Node => Self, Args_Types => Map_Result, Return_Type => Cast_Result_1);
--# end
--# expr-done 278
Is_Null_1 := Fld_11 = null; 
if Is_Null_1 then
   
   If_Result_1 := No_Entity_Function_Type;
else
   

As_Entity_1 := (Info => No_Entity_Info, Node => Fld_11); 
   If_Result_1 := As_Entity_1;
end if;



--# expr-done 279



   
      Cast_Result_2 := Create_Internal_Entity_Type_Decl
        (Node => If_Result_1.Node,
         Info => If_Result_1.Info);


   If_Result_2 := Cast_Result_2;
end if;



--# expr-done 280
Let_Result := If_Result_2; 
--# end
--# expr-done 281

         Property_Result := Let_Result;
         
   --# end
      Finalizer_Scope_198;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_198;




            raise;
      end;



   return Property_Result;
end Fun_Decl_P_Get_Type;
--# end

   







--# property-start FunDecl.check_legality D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2542
pragma Warnings (Off, "is not referenced");
 function Fun_Decl_P_Check_Legality
  
  (Node : Bare_Fun_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
     Ent : Internal_Entity_Fun_Decl :=
       Internal_Entity_Fun_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Semantic_Result_Array_Access;

      
            procedure Finalizer_Scope_199 with Inline_Always;
            procedure Finalizer_Scope_346 with Inline_Always;
            procedure Finalizer_Scope_347 with Inline_Always;
            procedure Finalizer_Scope_348 with Inline_Always;
            procedure Finalizer_Scope_349 with Inline_Always;

      Block_Var : Boolean;
Block_Var_1 : Boolean;
Block_Var_2 : Boolean;
Block_Var_3 : Boolean;
Var_Expr : Boolean;
Var_Expr_1 : Boolean;
Var_Expr_2 : Boolean;
Var_Expr_7 : Internal_Semantic_Result := No_Semantic_Result;
Fld : Internal_Entity_Full_Decl;
Sym : Symbol_Type;
Fld_1 : Boolean;
Fld_2 : Internal_Entity_Full_Decl;
Sym_1 : Symbol_Type;
Fld_3 : Boolean;
Fld_4 : Internal_Entity_Full_Decl;
Sym_2 : Symbol_Type;
Fld_5 : Boolean;
Fld_6 : Internal_Entity_Expr;
Is_Null : Boolean;
Not_Val : Boolean;
Str : String_Type;
Fld_7 : Internal_Semantic_Result := No_Semantic_Result;
Scope_Result : Internal_Semantic_Result := No_Semantic_Result;
Result_Var : Internal_Semantic_Result := No_Semantic_Result;
Str_1 : String_Type;
Fld_8 : Internal_Semantic_Result := No_Semantic_Result;
Scope_Result_1 : Internal_Semantic_Result := No_Semantic_Result;
Result_Var_1 : Internal_Semantic_Result := No_Semantic_Result;
Str_2 : String_Type;
Fld_9 : Internal_Semantic_Result := No_Semantic_Result;
Scope_Result_2 : Internal_Semantic_Result := No_Semantic_Result;
Result_Var_2 : Internal_Semantic_Result := No_Semantic_Result;
Not_Val_1 : Boolean;
Str_3 : String_Type;
Fld_10 : Internal_Semantic_Result := No_Semantic_Result;
If_Result : Internal_Semantic_Result := No_Semantic_Result;
If_Result_1 : Internal_Semantic_Result := No_Semantic_Result;
If_Result_2 : Internal_Semantic_Result := No_Semantic_Result;
If_Result_3 : Internal_Semantic_Result := No_Semantic_Result;
Singleton : Internal_Semantic_Result_Array_Access;
Scope_Result_3 : Internal_Semantic_Result_Array_Access;
Result_Var_3 : Internal_Semantic_Result_Array_Access;
Let_Result : Internal_Semantic_Result_Array_Access;

            procedure Finalizer_Scope_199 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
                     Dec_Ref (Scope_Result_1);
                     Dec_Ref (Result_Var_1);
                     Dec_Ref (Scope_Result_2);
                     Dec_Ref (Result_Var_2);
                     Dec_Ref (Str_3);
                     Dec_Ref (Fld_10);
                     Dec_Ref (If_Result);
                     Dec_Ref (If_Result_1);
                     Dec_Ref (If_Result_2);
                     Dec_Ref (If_Result_3);
                     Dec_Ref (Scope_Result_3);
                     Dec_Ref (Result_Var_3);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_199;
            procedure Finalizer_Scope_346 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Fld_7);
            end Finalizer_Scope_346;
            procedure Finalizer_Scope_347 is
            begin
                     Dec_Ref (Str_1);
                     Dec_Ref (Fld_8);
            end Finalizer_Scope_347;
            procedure Finalizer_Scope_348 is
            begin
                     Dec_Ref (Str_2);
                     Dec_Ref (Fld_9);
            end Finalizer_Scope_348;
            procedure Finalizer_Scope_349 is
            begin
                     Dec_Ref (Var_Expr_7);
                     Dec_Ref (Singleton);
            end Finalizer_Scope_349;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 304 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 284 '<FieldAccess for has_annotation at parser.py:2546>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2546



--# expr-start 282 '<FieldAccess for full_decl at parser.py:2546>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2546






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.full_decl
Fld := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 282


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 283 '<Symbol abstract at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Abstract); 
--# expr-done 283
--# property-call-start FullDecl.has_annotation
Fld_1 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Fld.Node, Name => Sym);
--# end
--# expr-done 284
Block_Var := Fld_1; 
--# bind has_abstract Block_Var
--# expr-start 287 '<FieldAccess for has_annotation at parser.py:2547>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2547



--# expr-start 285 '<FieldAccess for full_decl at parser.py:2547>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2547






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.full_decl
Fld_2 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 285


   

      if Fld_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 286 '<Symbol builtin at ???>' Sym_1 None
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Builtin); 
--# expr-done 286
--# property-call-start FullDecl.has_annotation
Fld_3 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Fld_2.Node, Name => Sym_1);
--# end
--# expr-done 287
Block_Var_1 := Fld_3; 
--# bind has_builtin Block_Var_1
--# expr-start 290 '<FieldAccess for has_annotation at parser.py:2548>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2548



--# expr-start 288 '<FieldAccess for full_decl at parser.py:2548>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2548






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.full_decl
Fld_4 := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 288


   

      if Fld_4.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 289 '<Symbol external at ???>' Sym_2 None
Sym_2 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_External); 
--# expr-done 289
--# property-call-start FullDecl.has_annotation
Fld_5 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Fld_4.Node, Name => Sym_2);
--# end
--# expr-done 290
Block_Var_2 := Fld_5; 
--# bind has_external Block_Var_2
--# expr-start 293 '<Not at parser.py:2549>' Not_Val D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2549
--# expr-start 292 '<IsNull at parser.py:2549>' Is_Null D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2549
--# expr-start 291 '<FieldAccess for body at parser.py:2549>' Fld_6 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2549






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_6 := Create_Internal_Entity_Expr (Node => Ent.Node.Fun_Decl_F_Body, Info => Ent.Info);
--# expr-done 291
Is_Null := Fld_6.Node = null; 
--# expr-done 292
Not_Val := not (Is_Null); 
--# expr-done 293
Block_Var_3 := Not_Val; 
--# bind has_body Block_Var_3



--# expr-start 303 '<Cond at parser.py:2550>' If_Result_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2550




if Block_Var then
   



Var_Expr := Block_Var_3;





if Var_Expr /= False then
   



   --# scope-start


   --# bind _ Var_Expr

--# expr-start 295 '<FieldAccess for error at parser.py:2554>' Fld_7 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2554






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 294 '<String at parser.py:2554>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2554
Str := Create_String ("abstract functions cannot have a body"); 
--# expr-done 294
--# property-call-start LktNode.error
Fld_7 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Self, Message => Str);
--# end
--# expr-done 295
Scope_Result := Fld_7;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_346;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Semantic_Result;
end if;

      Inc_Ref (Result_Var);


   If_Result_3 := Result_Var;
else
   



if Block_Var_1 then
   



Var_Expr_1 := Block_Var_3;





if Var_Expr_1 /= False then
   



   --# scope-start


   --# bind _ Var_Expr_1

--# expr-start 297 '<FieldAccess for error at parser.py:2560>' Fld_8 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2560






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 296 '<String at parser.py:2560>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2560
Str_1 := Create_String ("builtin functions cannot have a body"); 
--# expr-done 296
--# property-call-start LktNode.error
Fld_8 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Self, Message => Str_1);
--# end
--# expr-done 297
Scope_Result_1 := Fld_8;
   Inc_Ref (Scope_Result_1);


   --# end
      Finalizer_Scope_347;


   Result_Var_1 := Scope_Result_1;
else
   
   Result_Var_1 := No_Semantic_Result;
end if;

      Inc_Ref (Result_Var_1);


   If_Result_2 := Result_Var_1;
else
   



if Block_Var_2 then
   



Var_Expr_2 := Block_Var_3;





if Var_Expr_2 /= False then
   



   --# scope-start


   --# bind _ Var_Expr_2

--# expr-start 299 '<FieldAccess for error at parser.py:2566>' Fld_9 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2566






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 298 '<String at parser.py:2566>' Str_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2566
Str_2 := Create_String ("external functions cannot have a body"); 
--# expr-done 298
--# property-call-start LktNode.error
Fld_9 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Self, Message => Str_2);
--# end
--# expr-done 299
Scope_Result_2 := Fld_9;
   Inc_Ref (Scope_Result_2);


   --# end
      Finalizer_Scope_348;


   Result_Var_2 := Scope_Result_2;
else
   
   Result_Var_2 := No_Semantic_Result;
end if;

      Inc_Ref (Result_Var_2);


   If_Result_1 := Result_Var_2;
else
   


--# expr-start 300 '<Not at parser.py:2569>' Not_Val_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2569

Not_Val_1 := not (Block_Var_3); 
--# expr-done 300
if Not_Val_1 then
   --# expr-start 302 '<FieldAccess for error at parser.py:2570>' Fld_10 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2570






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 301 '<String at parser.py:2570>' Str_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2570
Str_3 := Create_String ("regular functions must have a body"); 
--# expr-done 301
--# property-call-start LktNode.error
Fld_10 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Self, Message => Str_3);
--# end
--# expr-done 302
   If_Result := Fld_10;
else
   
   If_Result := No_Semantic_Result;
end if;

      Inc_Ref (If_Result);


   If_Result_1 := If_Result;
end if;

      Inc_Ref (If_Result_1);


   If_Result_2 := If_Result_1;
end if;

      Inc_Ref (If_Result_2);


   If_Result_3 := If_Result_2;
end if;

      Inc_Ref (If_Result_3);


--# expr-done 303
Var_Expr_7 := If_Result_3;

      Inc_Ref (Var_Expr_7);




if Var_Expr_7 /= No_Semantic_Result then
   



   --# scope-start


   


                Singleton := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton.Items (1) := Var_Expr_7;
                Inc_Ref (Singleton.Items (1));
            
Scope_Result_3 := Singleton;
   Inc_Ref (Scope_Result_3);


   --# end
      Finalizer_Scope_349;


   Result_Var_3 := Scope_Result_3;
else
   
   Result_Var_3 := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var_3);


Let_Result := Result_Var_3; Inc_Ref (Let_Result);
--# end
--# expr-done 304

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_199;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_199;
                     Finalizer_Scope_346;
                     Finalizer_Scope_347;
                     Finalizer_Scope_348;
                     Finalizer_Scope_349;




            raise;
      end;



   return Property_Result;
end Fun_Decl_P_Check_Legality;
--# end

   







--# property-start FunDecl._env_mappings_17 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2495
pragma Warnings (Off, "is not referenced");
 function Env_Mappings_17
  
  (Node : Bare_Fun_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Env_Assoc
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
     Ent : Internal_Entity_Fun_Decl :=
       Internal_Entity_Fun_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Env_Assoc;

      
            procedure Finalizer_Scope_297 with Inline_Always;

      Fld : Symbol_Type;
New_Struct : Internal_Designated_Env := No_Designated_Env;
Cast_Expr : Bare_Fun_Decl;
Cast_Result : Bare_Lkt_Node;
New_Struct_1 : Internal_Env_Assoc := No_Env_Assoc;

            procedure Finalizer_Scope_297 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (New_Struct_1);
            end Finalizer_Scope_297;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 307 '<New EnvAssoc at parser.py:2495>' New_Struct_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2495
--# expr-start 305 '<New DesignatedEnv at parser.py:2495>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2495



Inc_Ref (Empty_Env);
New_Struct := (Direct_Env => Empty_Env, Env_Name => No_Symbol, Kind => Current_Env); 
--# expr-done 305
--# expr-start 306 '<FieldAccess for name at parser.py:2495>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2495






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Ent.Node);
--# end
--# expr-done 306







Cast_Expr := Self; 



   
      Cast_Result := Cast_Expr;


Inc_Ref (New_Struct);
New_Struct_1 := (Dest_Env => New_Struct, Key => Fld, Metadata => No_Metadata, Value => Cast_Result); 
--# expr-done 307

         Property_Result := New_Struct_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_297;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_297;




            raise;
      end;



   return Property_Result;
end Env_Mappings_17;
--# end

   







--# property-start FunDecl._env_trans_parent_18 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2496
pragma Warnings (Off, "is not referenced");
 function Env_Trans_Parent_18
  
  (Node : Bare_Fun_Decl
  )

   return Boolean
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
      --# bind self Self

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := False;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Env_Trans_Parent_18;
--# end

   







--# property-start FunDecl._env_mappings_19 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2498
pragma Warnings (Off, "is not referenced");
 function Env_Mappings_19
  
  (Node : Bare_Fun_Decl
  )

   return Internal_Env_Assoc_Array_Access
is
   Self : Bare_Fun_Decl  := Bare_Fun_Decl (Node);
      --# bind self Self

   


   Property_Result : Internal_Env_Assoc_Array_Access;

      
            procedure Finalizer_Scope_299 with Inline_Always;
            procedure Finalizer_Scope_350 with Inline_Always;

      Var_Expr : Bare_Type_Decl;
Fld : Bare_Type_Decl;
Sym : Symbol_Type;
Fld_1 : Bare_Self_Decl;
New_Struct : Internal_Designated_Env := No_Designated_Env;
Cast_Result : Bare_Lkt_Node;
New_Struct_1 : Internal_Env_Assoc := No_Env_Assoc;
Sym_1 : Symbol_Type;
Fld_2 : Bare_Node_Decl;
New_Struct_2 : Internal_Designated_Env := No_Designated_Env;
Cast_Result_1 : Bare_Lkt_Node;
New_Struct_3 : Internal_Env_Assoc := No_Env_Assoc;
Array_Lit : Internal_Env_Assoc_Array_Access;
Scope_Result : Internal_Env_Assoc_Array_Access;
Result_Var : Internal_Env_Assoc_Array_Access;

            procedure Finalizer_Scope_299 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
            end Finalizer_Scope_299;
            procedure Finalizer_Scope_350 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (New_Struct_1);
                     Dec_Ref (New_Struct_2);
                     Dec_Ref (New_Struct_3);
                     Dec_Ref (Array_Lit);
            end Finalizer_Scope_350;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         


--# expr-start 308 '<FieldAccess for owning_type at parser.py:2498>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2498






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start FunDecl.owning_type
Fld := Liblktlang.Implementation.Fun_Decl_P_Owning_Type (Node => Self);
--# end
--# expr-done 308
Var_Expr := Fld;





if Var_Expr /= No_Bare_Lkt_Node then
   



   --# scope-start


   --# bind ot Var_Expr

--# expr-start 317 '<ArrayLiteral at ???>' Array_Lit None
--# expr-start 312 '<New EnvAssoc at parser.py:2499>' New_Struct_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2499
--# expr-start 309 '<New DesignatedEnv at parser.py:2499>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2499



Inc_Ref (Empty_Env);
New_Struct := (Direct_Env => Empty_Env, Env_Name => No_Symbol, Kind => Current_Env); 
--# expr-done 309
--# expr-start 310 '<Symbol self at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Self); 
--# expr-done 310






--# expr-start 311 '<FieldAccess for self_decl at parser.py:2499>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2499






   

      if Var_Expr = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.self_decl
Fld_1 := Liblktlang.Implementation.Type_Decl_P_Self_Decl (Node => Var_Expr);
--# end
--# expr-done 311



   
      Cast_Result := Fld_1;


Inc_Ref (New_Struct);
New_Struct_1 := (Dest_Env => New_Struct, Key => Sym, Metadata => No_Metadata, Value => Cast_Result); 
--# expr-done 312
--# expr-start 316 '<New EnvAssoc at parser.py:2500>' New_Struct_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2500
--# expr-start 313 '<New DesignatedEnv at parser.py:2500>' New_Struct_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2500



Inc_Ref (Empty_Env);
New_Struct_2 := (Direct_Env => Empty_Env, Env_Name => No_Symbol, Kind => Current_Env); 
--# expr-done 313
--# expr-start 314 '<Symbol node at ???>' Sym_1 None
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Node_31); 
--# expr-done 314






--# expr-start 315 '<FieldAccess for node_decl at parser.py:2500>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2500






   

      if Var_Expr = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.node_decl
Fld_2 := Liblktlang.Implementation.Type_Decl_P_Node_Decl (Node => Var_Expr);
--# end
--# expr-done 315



   
      Cast_Result_1 := Fld_2;


Inc_Ref (New_Struct_2);
New_Struct_3 := (Dest_Env => New_Struct_2, Key => Sym_1, Metadata => No_Metadata, Value => Cast_Result_1); 
--# expr-done 316
Array_Lit := Create_Internal_Env_Assoc_Array (Internal_Internal_Env_Assoc_Array'(1 => New_Struct_1, 2 => New_Struct_3)); 
--# expr-done 317
Scope_Result := Array_Lit;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_350;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Internal_Env_Assoc_Array_Type;
end if;

      Inc_Ref (Result_Var);



         Property_Result := Result_Var;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_299;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_299;
                     Finalizer_Scope_350;




            raise;
      end;



   return Property_Result;
end Env_Mappings_19;
--# end



   


      

   --
   --  Primitives for Bare_Env_Spec_Decl
   --

   



      
      procedure Initialize_Fields_For_Env_Spec_Decl
        (Self : Bare_Env_Spec_Decl
         ; Env_Spec_Decl_F_Syn_Name : Bare_Def_Id
         ; Env_Spec_Decl_F_Actions : Bare_Call_Expr_List
        ) is
      begin

            Self.Env_Spec_Decl_F_Syn_Name := Env_Spec_Decl_F_Syn_Name;
            Self.Env_Spec_Decl_F_Actions := Env_Spec_Decl_F_Actions;
         

      end Initialize_Fields_For_Env_Spec_Decl;

      
   function Env_Spec_Decl_F_Actions
     (Node : Bare_Env_Spec_Decl) return Bare_Call_Expr_List
   is
      

   begin
         
         return Node.Env_Spec_Decl_F_Actions;
      
   end;


   







--# property-start EnvSpecDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2360
pragma Warnings (Off, "is not referenced");
 function Env_Spec_Decl_P_Decl_Type_Name
  
  (Node : Bare_Env_Spec_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Env_Spec_Decl  := Bare_Env_Spec_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_180 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_180 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_180;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 318 '<String at parser.py:2360>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2360
Str := Create_String ("env spec declaration"); 
--# expr-done 318

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_180;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_180;




            raise;
      end;



   return Property_Result;
end Env_Spec_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Generic_Decl
   --

   



   

   

   

   

   

   

   


      procedure Generic_Decl_Pre_Env_Actions
        (Self            : Bare_Generic_Decl;
         State           : in out PLE_Node_State;
         Add_To_Env_Only : Boolean := False) is
      begin
            


   

      declare
         Resolver : constant Entity_Resolver :=
            null;

         

         Mapping : Internal_Env_Assoc :=
            Liblktlang.Implementation.Env_Mappings_10 (Node => 
Self);
      begin

         Add_To_Env
           (Self,
            State,
            Mapping.Key,
            Mapping.Value,
            Mapping.Metadata,
            Resolver,
            Mapping.Dest_Env,
            DSL_Location => "parser.py:2070");
         Dec_Ref (Mapping.Dest_Env);

      end;
   
   
            


   
      if Add_To_Env_Only then
         return;
      end if;

      declare
         No_Parent         : constant Boolean :=
            False;
         Transitive_Parent : constant Boolean :=
            Liblktlang.Implementation.Env_Trans_Parent_11 (Node => 
Self);
         Names             : Symbol_Type_Array_Access :=
            null;
      begin
         Add_Env (Self, State, No_Parent, Transitive_Parent, Names);
      end;
   
   
      end;



      
      procedure Initialize_Fields_For_Generic_Decl
        (Self : Bare_Generic_Decl
         ; Generic_Decl_F_Generic_Formal_Decls : Bare_Generic_Formal_Decl_List
         ; Generic_Decl_F_Decl : Bare_Decl
        ) is
      begin

            Self.Generic_Decl_F_Generic_Formal_Decls := Generic_Decl_F_Generic_Formal_Decls;
            Self.Generic_Decl_F_Decl := Generic_Decl_F_Decl;
         

      end Initialize_Fields_For_Generic_Decl;

      
   function Generic_Decl_F_Generic_Formal_Decls
     (Node : Bare_Generic_Decl) return Bare_Generic_Formal_Decl_List
   is
      

   begin
         
         return Node.Generic_Decl_F_Generic_Formal_Decls;
      
   end;

      
   function Generic_Decl_F_Decl
     (Node : Bare_Generic_Decl) return Bare_Decl
   is
      

   begin
         
         return Node.Generic_Decl_F_Decl;
      
   end;


   







--# property-start GenericDecl.name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2066
pragma Warnings (Off, "is not referenced");
 function Generic_Decl_P_Name
  
  (Node : Bare_Generic_Decl
  )

   return Symbol_Type
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
      --# bind self Self

   


   Property_Result : Symbol_Type;

      

      Fld : Bare_Decl;
Fld_1 : Symbol_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 320 '<FieldAccess for name at parser.py:2066>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2066



--# expr-start 319 '<FieldAccess for decl at parser.py:2066>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2066






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Self.Generic_Decl_F_Decl;
--# expr-done 319


   

      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld_1 := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Fld);
--# end
--# expr-done 320

         Property_Result := Fld_1;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Generic_Decl_P_Name;
--# end

   







--# property-start GenericDecl.generic_formals D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2074
pragma Warnings (Off, "is not referenced");
 function Generic_Decl_P_Generic_Formals
  
  (Node : Bare_Generic_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Generic_Formal_Type_Decl_Array_Access
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
     Ent : Internal_Entity_Generic_Decl :=
       Internal_Entity_Generic_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Generic_Formal_Type_Decl_Array_Access;

      
            procedure Finalizer_Scope_154 with Inline_Always;

      Fld : Internal_Entity_Generic_Formal_Decl_List;
Fld_1 : Bare_Generic_Formal_Decl_List;
Fld_2 : Internal_Entity_Info;
Item_15 : Internal_Entity_Full_Decl;
Bare_Item_15 : Bare_Full_Decl;
Fld_3 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Generic_Formal_Type_Decl;
As_Entity : Internal_Entity_Full_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Full_Decl;
Map_Result : Internal_Entity_Generic_Formal_Type_Decl_Array_Access;

            procedure Finalizer_Scope_154 is
            begin
                     Dec_Ref (Map_Result);
            end Finalizer_Scope_154;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 324 '<Map at parser.py:2074>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2074








   

   --# expr-start 321 '<FieldAccess for generic_formal_decls at parser.py:2074>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2074






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Generic_Formal_Decl_List (Node => Ent.Node.Generic_Decl_F_Generic_Formal_Decls, Info => Ent.Info);
--# expr-done 321






   




Fld_1 := Fld.Node;

   
      declare
         Map_Result_Vec : Internal_Entity_Generic_Formal_Type_Decl_Vectors.Vector;
      begin
   

   

      if Fld_1 = null then
         
      Map_Result := Create_Internal_Entity_Generic_Formal_Type_Decl_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Generic_Formal_Decl_List := Fld_1;
      begin
         for Untyped_Item_15 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_15 := Untyped_Item_15; 
                  



Is_Null := Bare_Item_15 = null; 
if Is_Null then
   
   If_Result := No_Entity_Full_Decl;
else
   





   




Fld_2 := Fld.Info;

As_Entity := (Info => Fld_2, Node => Bare_Item_15); 
   If_Result := As_Entity;
end if;



                  Item_15 := If_Result; 

            
   --# scope-start


               --# bind gfd Item_15

            
         
         
      --# expr-start 323 '<Cast to GenericFormalTypeDecl at parser.py:2075>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2075





--# expr-start 322 '<FieldAccess for decl at parser.py:2075>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2075






   

      if Item_15.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Create_Internal_Entity_Decl (Node => Item_15.Node.Full_Decl_F_Decl, Info => Item_15.Info);
--# expr-done 322



   if Fld_3.Node = null
      or else Fld_3.Node.Kind in Lkt_Generic_Formal_Type_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Generic_Formal_Type_Decl
        (Node => Fld_3.Node,
         Info => Fld_3.Info);

   else
         Cast_Result := No_Entity_Generic_Formal_Type_Decl;
   end if;


--# expr-done 323
      

         declare
            Item_To_Append : constant Internal_Entity_Generic_Formal_Type_Decl := Cast_Result;
         begin
            Internal_Entity_Generic_Formal_Type_Decl_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   
      end if;

   

         Map_Result := Create_Internal_Entity_Generic_Formal_Type_Decl_Array
           (Items_Count => Natural (Internal_Entity_Generic_Formal_Type_Decl_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Entity_Generic_Formal_Type_Decl_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Entity_Generic_Formal_Type_Decl_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Entity_Generic_Formal_Type_Decl_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 324

         Property_Result := Map_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_154;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_154;




            raise;
      end;



   return Property_Result;
end Generic_Decl_P_Generic_Formals;
--# end

   







--# property-start GenericDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2078
pragma Warnings (Off, "is not referenced");
 function Generic_Decl_P_Decl_Type_Name
  
  (Node : Bare_Generic_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
     Ent : Internal_Entity_Generic_Decl :=
       Internal_Entity_Generic_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_155 with Inline_Always;

      Str : String_Type;
Fld : Internal_Entity_Decl;
Fld_1 : String_Type;
Concat_Result : String_Type;

            procedure Finalizer_Scope_155 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Fld_1);
                     Dec_Ref (Concat_Result);
            end Finalizer_Scope_155;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 325 '<String at parser.py:2079>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2079
Str := Create_String ("generic "); 
--# expr-done 325
--# expr-start 327 '<FieldAccess for decl_type_name at parser.py:2079>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2079



--# expr-start 326 '<FieldAccess for decl at parser.py:2079>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2079






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Decl (Node => Ent.Node.Generic_Decl_F_Decl, Info => Ent.Info);
--# expr-done 326


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.decl_type_name'
Fld_1 := Liblktlang.Implementation.Dispatcher_Decl_P_Decl_Type_Name (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 327
Concat_Result := Concat_String (Str, Fld_1); 

         Property_Result := Concat_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_155;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_155;




            raise;
      end;



   return Property_Result;
end Generic_Decl_P_Decl_Type_Name;
--# end

   







--# property-start GenericDecl.instantiate D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2082
pragma Warnings (Off, "is not referenced");
 function Generic_Decl_P_Instantiate
  
  (Node : Bare_Generic_Decl
      ; Actuals : Bare_Type_Decl_Array_Access
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Decl
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
     Ent : Internal_Entity_Generic_Decl :=
       Internal_Entity_Generic_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind actuals Actuals

   Property_Result : Internal_Entity_Decl;

      
            procedure Finalizer_Scope_156 with Inline_Always;

      Block_Var : Bare_Fun_Decl;
Fld : Bare_Decl;
Cast_Result : Bare_Fun_Decl;
Fld_1 : Internal_Entity_Decl;
Is_A : Boolean;
Fld_2 : Bare_Instantiated_Generic_Type;
Cast_Result_1 : Bare_Decl;
As_Entity : Internal_Entity_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Decl;
Fld_3 : Internal_Entity_Decl;
Is_A_1 : Boolean;
Fld_4 : Internal_Entity_Info;
Fld_5 : Env_Rebindings;
Fld_6 : Lexical_Env := Empty_Env;
Fld_7 : Lexical_Env := Empty_Env;
Rebinding : Env_Rebindings;
New_Struct : Internal_Entity_Info;
New_Struct_1 : Internal_Entity_Fun_Decl;
Cast_Result_2 : Internal_Entity_Decl;
If_Result_1 : Internal_Entity_Decl;
If_Result_2 : Internal_Entity_Decl;
Let_Result : Internal_Entity_Decl;

            procedure Finalizer_Scope_156 is
            begin
                     Dec_Ref (Fld_6);
                     Dec_Ref (Fld_7);
            end Finalizer_Scope_156;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 346 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 329 '<Cast to FunDecl at parser.py:2087>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2087





--# expr-start 328 '<FieldAccess for decl at parser.py:2087>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2087






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Self.Generic_Decl_F_Decl;
--# expr-done 328



   if Fld = null
      or else Fld.Kind in Lkt_Fun_Decl_Range
   then
      
      Cast_Result := Fld;

   else
         Cast_Result := No_Bare_Lkt_Node;
   end if;


--# expr-done 329
Block_Var := Cast_Result; 
--# bind fd Block_Var
--# expr-start 345 '<Cond at parser.py:2088>' If_Result_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2088



--# expr-start 331 '<IsA TypeDecl at parser.py:2089>' Is_A D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2089
--# expr-start 330 '<FieldAccess for decl at parser.py:2089>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2089






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Create_Internal_Entity_Decl (Node => Ent.Node.Generic_Decl_F_Decl, Info => Ent.Info);
--# expr-done 330
Is_A := Fld_1.Node /= null 
and then Fld_1.Node.Kind in Lkt_Type_Decl; 
--# expr-done 331
if Is_A then
   --# expr-start 334 '<as_bare_entity at parser.py:2090>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2090



--# expr-start 333 '<Cast to Decl at parser.py:2090>' Cast_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2090





--# expr-start 332 '<FieldAccess for get_instantiated_type at parser.py:2090>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2090






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start GenericDecl.get_instantiated_type
Fld_2 := Liblktlang.Implementation.Generic_Decl_P_Get_Instantiated_Type (Node => Ent.Node, Actuals => Actuals);
--# end
--# expr-done 332



   
      Cast_Result_1 := Fld_2;


--# expr-done 333
Is_Null := Cast_Result_1 = null; 
if Is_Null then
   
   If_Result := No_Entity_Decl;
else
   

As_Entity := (Info => No_Entity_Info, Node => Cast_Result_1); 
   If_Result := As_Entity;
end if;



--# expr-done 334
   If_Result_2 := If_Result;
else
   


--# expr-start 336 '<IsA FunDecl at parser.py:2092>' Is_A_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2092
--# expr-start 335 '<FieldAccess for decl at parser.py:2092>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2092






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Create_Internal_Entity_Decl (Node => Ent.Node.Generic_Decl_F_Decl, Info => Ent.Info);
--# expr-done 335
Is_A_1 := Fld_3.Node /= null 
and then Fld_3.Node.Kind in Lkt_Fun_Decl_Range; 
--# expr-done 336
if Is_A_1 then
   --# expr-start 344 '<Cast to Decl.entity at ???>' Cast_Result_2 None





--# expr-start 343 '<New FunDecl.entity at ???>' New_Struct_1 None
--# expr-start 342 '<New EntityInfo at parser.py:2095>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2095


--# expr-start 341 '<append_rebinding at parser.py:2098>' Rebinding D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2098
--# expr-start 338 '<FieldAccess for rebindings at parser.py:2098>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2098



--# expr-start 337 '<FieldAccess for info at parser.py:2098>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2098






   




Fld_4 := Ent.Info;
--# expr-done 337


   




Fld_5 := Fld_4.Rebindings;
--# expr-done 338
--# expr-start 339 '<FieldAccess for children_env at parser.py:2099>' Fld_6 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2099






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_6 := Liblktlang.Implementation.Children_Env (Node => Self);
--# expr-done 339
--# expr-start 340 '<FieldAccess for get_inst_env at parser.py:2100>' Fld_7 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2100






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start GenericDecl.get_inst_env
Fld_7 := Liblktlang.Implementation.Generic_Decl_P_Get_Inst_Env (Node => Ent.Node, Actuals => Actuals, E_Info => Ent.Info);
--# end
--# expr-done 340
Rebinding := AST_Envs.Append_Rebinding (Fld_5, Fld_6, Fld_7); 
--# expr-done 341
New_Struct := (From_Rebound => False, Md => No_Metadata, Rebindings => Rebinding); 
--# expr-done 342

New_Struct_1 := (Info => New_Struct, Node => Block_Var); 
--# expr-done 343



   
      Cast_Result_2 := Create_Internal_Entity_Decl
        (Node => New_Struct_1.Node,
         Info => New_Struct_1.Info);


--# expr-done 344
   If_Result_1 := Cast_Result_2;
else
   
   If_Result_1 := raise Property_Error with "should not happen";
end if;



   If_Result_2 := If_Result_1;
end if;



--# expr-done 345
Let_Result := If_Result_2; 
--# end
--# expr-done 346

         Property_Result := Let_Result;
         
   --# end
      Finalizer_Scope_156;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_156;




            raise;
      end;



   return Property_Result;
end Generic_Decl_P_Instantiate;
--# end

   







--# property-start GenericDecl.get_instantiated_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2108
pragma Warnings (Off, "is not referenced");
 function Generic_Decl_P_Get_Instantiated_Type
  
  (Node : Bare_Generic_Decl
      ; Actuals : Bare_Type_Decl_Array_Access
  )

   return Bare_Instantiated_Generic_Type
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
      --# bind self Self

   

   --# bind actuals Actuals

   Property_Result : Bare_Instantiated_Generic_Type;

      

      Fld : Bare_Decl;
Cast_Result : Bare_Type_Decl;
New_Node : Bare_Instantiated_Generic_Type;
Let_Result : Bare_Instantiated_Generic_Type;


      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Generic_Decl_Generic_Decl_P_Get_Instantiated_Type,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Generic_Decl,
                                As_Bare_Generic_Decl => Self);
               Mmz_K.Items (2) := (Kind => Mmz_Bare_Type_Decl_Array_Access,
                                      As_Bare_Type_Decl_Array_Access => Actuals);
                  Inc_Ref (Mmz_K.Items (2).As_Bare_Type_Decl_Array_Access);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Bare_Instantiated_Generic_Type;

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 350 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 349 '<New InstantiatedGenericType at parser.py:2110>' New_Node D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2110

--# expr-start 348 '<Cast to TypeDecl at parser.py:2111>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2111





--# expr-start 347 '<FieldAccess for decl at parser.py:2111>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2111






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Self.Generic_Decl_F_Decl;
--# expr-done 347



   if Fld = null
      or else Fld.Kind in Lkt_Type_Decl
   then
      
      Cast_Result := Fld;

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 348
Inc_Ref (Actuals);


if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node := new Root_Node_Record
  (Lkt_Instantiated_Generic_Type);
Initialize
  (Self => New_Node,
   Kind => Lkt_Instantiated_Generic_Type,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node);


      New_Node.Instantiated_Generic_Type_Actuals :=
         Actuals;
      New_Node.Instantiated_Generic_Type_Inner_Type_Decl :=
         Cast_Result;

--# expr-done 349
Let_Result := New_Node; 
--# end
--# expr-done 350

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Bare_Instantiated_Generic_Type,
                        As_Bare_Instantiated_Generic_Type => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);

         end if;
      end if;



   return Property_Result;
end Generic_Decl_P_Get_Instantiated_Type;
--# end

   







--# property-start GenericDecl.get_assocs D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2115
pragma Warnings (Off, "is not referenced");
 function Generic_Decl_P_Get_Assocs
  
  (Node : Bare_Generic_Decl
      ; Actuals : Bare_Type_Decl_Array_Access
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Generic_Param_Assoc_Array_Access
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
     Ent : Internal_Entity_Generic_Decl :=
       Internal_Entity_Generic_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind actuals Actuals

   Property_Result : Internal_Generic_Param_Assoc_Array_Access;

      
            procedure Finalizer_Scope_158 with Inline_Always;

      Fld : Internal_Entity_Generic_Formal_Type_Decl_Array_Access;
Index_17 : Integer;
Fld_1 : Bare_Generic_Formal_Type_Decl;
Get_Result : Bare_Type_Decl;
New_Struct : Internal_Generic_Param_Assoc;
Map_Result : Internal_Generic_Param_Assoc_Array_Access;
Fld_2 : Internal_Entity_Generic_Formal_Type_Decl_Array_Access;
Len : Integer;
Len_1 : Integer;
Is_Equal : Boolean;
If_Result : Internal_Generic_Param_Assoc_Array_Access;
Let_Result : Internal_Generic_Param_Assoc_Array_Access;

            procedure Finalizer_Scope_158 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Fld_2);
                     Dec_Ref (If_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_158;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Generic_Decl_Generic_Decl_P_Get_Assocs,
            Items    => new Mmz_Key_Array (1 ..  3))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Generic_Decl,
                                As_Bare_Generic_Decl => Self);
               Mmz_K.Items (2) := (Kind => Mmz_Bare_Type_Decl_Array_Access,
                                      As_Bare_Type_Decl_Array_Access => Actuals);
                  Inc_Ref (Mmz_K.Items (2).As_Bare_Type_Decl_Array_Access);
               Mmz_K.Items (3) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);




      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Internal_Generic_Param_Assoc_Array_Access;
                     Inc_Ref (Property_Result);

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 361 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 360 '<If at parser.py:2121>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2121



--# expr-start 354 '<Eq at parser.py:2122>' Is_Equal D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2122
--# expr-start 352 '<length at parser.py:2122>' Len D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2122
--# expr-start 351 '<FieldAccess for generic_formals at parser.py:2122>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2122






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start GenericDecl.generic_formals
Fld_2 := Liblktlang.Implementation.Generic_Decl_P_Generic_Formals (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 351
Len := Length (Fld_2); 
--# expr-done 352
--# expr-start 353 '<length at parser.py:2122>' Len_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2122

Len_1 := Length (Actuals); 
--# expr-done 353
Is_Equal := Len = Len_1; 
--# expr-done 354
if Is_Equal then
   --# expr-start 359 '<Map at parser.py:2123>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2123








   

   --# expr-start 355 '<FieldAccess for generic_formals at parser.py:2123>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2123






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start GenericDecl.generic_formals
Fld := Liblktlang.Implementation.Generic_Decl_P_Generic_Formals (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 355

   
      declare
         Map_Result_Vec : Internal_Generic_Param_Assoc_Vectors.Vector;
      begin
   

   

      
         Index_17 := 0;

      declare
         
         Collection : constant Internal_Entity_Generic_Formal_Type_Decl_Array_Access := Fld;
      begin
         for Item_16 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind f Item_16
               --# bind idx Index_17

            
         
         
      --# expr-start 358 '<New GenericParamAssoc at parser.py:2124>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2124
--# expr-start 356 '<at at parser.py:2125>' Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2125




Get_Result := Get (Self, Actuals, Index_17, True); 
--# expr-done 356
--# expr-start 357 '<FieldAccess for node at parser.py:2125>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2125






   




Fld_1 := Item_16.Node;
--# expr-done 357
New_Struct := (Actual => Get_Result, Formal => Fld_1); 
--# expr-done 358
      

         declare
            Item_To_Append : constant Internal_Generic_Param_Assoc := New_Struct;
         begin
            Internal_Generic_Param_Assoc_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



               Index_17 := Index_17 + 1;
         end loop;
      end;
   

   

         Map_Result := Create_Internal_Generic_Param_Assoc_Array
           (Items_Count => Natural (Internal_Generic_Param_Assoc_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Generic_Param_Assoc_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Generic_Param_Assoc_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Generic_Param_Assoc_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 359
   If_Result := Map_Result;
else
   
   If_Result := raise Property_Error with "Actuals and formals don't match";
end if;

      Inc_Ref (If_Result);


--# expr-done 360
Let_Result := If_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 361

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_158;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_158;


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Internal_Generic_Param_Assoc_Array_Access,
                        As_Internal_Generic_Param_Assoc_Array_Access => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
               if Mmz_Stored then
                  Inc_Ref (Property_Result);
               end if;

         end if;
      end if;



   return Property_Result;
end Generic_Decl_P_Get_Assocs;
--# end

   







--# property-start GenericDecl.get_inst_env D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2132
pragma Warnings (Off, "is not referenced");
 function Generic_Decl_P_Get_Inst_Env
  
  (Node : Bare_Generic_Decl
      ; Actuals : Bare_Type_Decl_Array_Access
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
     Ent : Internal_Entity_Generic_Decl :=
       Internal_Entity_Generic_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind actuals Actuals

   Property_Result : Lexical_Env;

      
            procedure Finalizer_Scope_159 with Inline_Always;

      Fld : Internal_Generic_Param_Assoc_Array_Access;
Fld_1 : Bare_Generic_Formal_Type_Decl;
Fld_2 : Symbol_Type;
Fld_3 : Bare_Type_Decl;
Cast_Result : Bare_Lkt_Node;
New_Struct : Internal_Env_Kv;
Map_Result : Internal_Env_Kv_Array_Access;
Fld_4 : Lexical_Env := Empty_Env;
Let_Result : Lexical_Env := Empty_Env;

            procedure Finalizer_Scope_159 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Fld_4);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_159;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Generic_Decl_Generic_Decl_P_Get_Inst_Env,
            Items    => new Mmz_Key_Array (1 ..  3))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Generic_Decl,
                                As_Bare_Generic_Decl => Self);
               Mmz_K.Items (2) := (Kind => Mmz_Bare_Type_Decl_Array_Access,
                                      As_Bare_Type_Decl_Array_Access => Actuals);
                  Inc_Ref (Mmz_K.Items (2).As_Bare_Type_Decl_Array_Access);
               Mmz_K.Items (3) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Lexical_Env;
                     Inc_Ref (Property_Result);

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 369 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 368 '<FieldAccess for env_from_vals_internal at parser.py:2139>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2139






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 367 '<Map at parser.py:2140>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2140








   

   --# expr-start 362 '<FieldAccess for get_assocs at parser.py:2140>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2140






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start GenericDecl.get_assocs
Fld := Liblktlang.Implementation.Generic_Decl_P_Get_Assocs (Node => Ent.Node, Actuals => Actuals, E_Info => Ent.Info);
--# end
--# expr-done 362

   
      declare
         Map_Result_Vec : Internal_Env_Kv_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Generic_Param_Assoc_Array_Access := Fld;
      begin
         for Item_18 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind assoc Item_18

            
         
         
      --# expr-start 366 '<New EnvKv at parser.py:2141>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2141
--# expr-start 364 '<FieldAccess for name at parser.py:2142>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2142



--# expr-start 363 '<FieldAccess for formal at parser.py:2142>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2142






   




Fld_1 := Item_18.Formal;
--# expr-done 363


   

      if Fld_1 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld_2 := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Fld_1);
--# end
--# expr-done 364





--# expr-start 365 '<FieldAccess for actual at parser.py:2142>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2142






   




Fld_3 := Item_18.Actual;
--# expr-done 365



   
      Cast_Result := Fld_3;


New_Struct := (Key => Fld_2, Value => Cast_Result); 
--# expr-done 366
      

         declare
            Item_To_Append : constant Internal_Env_Kv := New_Struct;
         begin
            Internal_Env_Kv_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result := Create_Internal_Env_Kv_Array
           (Items_Count => Natural (Internal_Env_Kv_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Env_Kv_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Env_Kv_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Env_Kv_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 367
Fld_4 := Liblktlang.Implementation.Extensions.Lkt_Node_P_Env_From_Vals_Internal (Node => Ent.Node, Vals => Map_Result);
--# expr-done 368
Let_Result := Fld_4; Inc_Ref (Let_Result);
--# end
--# expr-done 369

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_159;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_159;


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Lexical_Env,
                        As_Lexical_Env => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
               if Mmz_Stored then
                  Inc_Ref (Property_Result);
               end if;

         end if;
      end if;



   return Property_Result;
end Generic_Decl_P_Get_Inst_Env;
--# end

   







--# property-start GenericDecl._env_mappings_10 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2070
pragma Warnings (Off, "is not referenced");
 function Env_Mappings_10
  
  (Node : Bare_Generic_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Env_Assoc
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
     Ent : Internal_Entity_Generic_Decl :=
       Internal_Entity_Generic_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Env_Assoc;

      
            procedure Finalizer_Scope_290 with Inline_Always;

      Fld : Symbol_Type;
New_Struct : Internal_Designated_Env := No_Designated_Env;
Cast_Expr : Bare_Generic_Decl;
Cast_Result : Bare_Lkt_Node;
New_Struct_1 : Internal_Env_Assoc := No_Env_Assoc;

            procedure Finalizer_Scope_290 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (New_Struct_1);
            end Finalizer_Scope_290;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 372 '<New EnvAssoc at parser.py:2070>' New_Struct_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2070
--# expr-start 370 '<New DesignatedEnv at parser.py:2070>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2070



Inc_Ref (Empty_Env);
New_Struct := (Direct_Env => Empty_Env, Env_Name => No_Symbol, Kind => Current_Env); 
--# expr-done 370
--# expr-start 371 '<FieldAccess for name at parser.py:2070>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2070






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Ent.Node);
--# end
--# expr-done 371







Cast_Expr := Self; 



   
      Cast_Result := Cast_Expr;


Inc_Ref (New_Struct);
New_Struct_1 := (Dest_Env => New_Struct, Key => Fld, Metadata => No_Metadata, Value => Cast_Result); 
--# expr-done 372

         Property_Result := New_Struct_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_290;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_290;




            raise;
      end;



   return Property_Result;
end Env_Mappings_10;
--# end

   







--# property-start GenericDecl._env_trans_parent_11 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2071
pragma Warnings (Off, "is not referenced");
 function Env_Trans_Parent_11
  
  (Node : Bare_Generic_Decl
  )

   return Boolean
is
   Self : Bare_Generic_Decl  := Bare_Generic_Decl (Node);
      --# bind self Self

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := False;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Env_Trans_Parent_11;
--# end



   


      

   --
   --  Primitives for Bare_Grammar_Decl
   --

   



   

   

   

   

   

   

   


      procedure Grammar_Decl_Pre_Env_Actions
        (Self            : Bare_Grammar_Decl;
         State           : in out PLE_Node_State;
         Add_To_Env_Only : Boolean := False) is
      begin
            


   

      declare
         Resolver : constant Entity_Resolver :=
            null;

         

         Mapping : Internal_Env_Assoc :=
            Liblktlang.Implementation.Env_Mappings_6 (Node => 
Self);
      begin

         Add_To_Env
           (Self,
            State,
            Mapping.Key,
            Mapping.Value,
            Mapping.Metadata,
            Resolver,
            Mapping.Dest_Env,
            DSL_Location => "parser.py:1222");
         Dec_Ref (Mapping.Dest_Env);

      end;
   
   
            


   
      if Add_To_Env_Only then
         return;
      end if;

      declare
         No_Parent         : constant Boolean :=
            False;
         Transitive_Parent : constant Boolean :=
            Liblktlang.Implementation.Env_Trans_Parent_7 (Node => 
Self);
         Names             : Symbol_Type_Array_Access :=
            null;
      begin
         Add_Env (Self, State, No_Parent, Transitive_Parent, Names);
      end;
   
   
      end;



      
      procedure Initialize_Fields_For_Grammar_Decl
        (Self : Bare_Grammar_Decl
         ; Grammar_Decl_F_Syn_Name : Bare_Def_Id
         ; Grammar_Decl_F_Rules : Bare_Full_Decl_List
        ) is
      begin

            Self.Grammar_Decl_F_Syn_Name := Grammar_Decl_F_Syn_Name;
            Self.Grammar_Decl_F_Rules := Grammar_Decl_F_Rules;
         

      end Initialize_Fields_For_Grammar_Decl;

      
   function Grammar_Decl_F_Rules
     (Node : Bare_Grammar_Decl) return Bare_Full_Decl_List
   is
      

   begin
         
         return Node.Grammar_Decl_F_Rules;
      
   end;


   







--# property-start GrammarDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1211
pragma Warnings (Off, "is not referenced");
 function Grammar_Decl_P_Decl_Type_Name
  
  (Node : Bare_Grammar_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Grammar_Decl  := Bare_Grammar_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_101 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_101 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_101;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 373 '<String at parser.py:1211>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1211
Str := Create_String ("grammar declaration"); 
--# expr-done 373

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_101;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_101;




            raise;
      end;



   return Property_Result;
end Grammar_Decl_P_Decl_Type_Name;
--# end

   







--# property-start GrammarDecl.lexer D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1213
pragma Warnings (Off, "is not referenced");
 function Grammar_Decl_P_Lexer
  
  (Node : Bare_Grammar_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Decl
is
   Self : Bare_Grammar_Decl  := Bare_Grammar_Decl (Node);
     Ent : Internal_Entity_Grammar_Decl :=
       Internal_Entity_Grammar_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Decl;

      

      Fld : Internal_Entity_Full_Decl;
Sym : Symbol_Type;
Fld_1 : Bare_Decl_Annotation;
Fld_2 : Bare_Decl_Annotation_Params;
Fld_3 : Bare_Param_List;
Get_Result : Bare_Param;
Fld_4 : Bare_Expr;
As_Entity : Internal_Entity_Expr;
Is_Null : Boolean;
If_Result : Internal_Entity_Expr;
Fld_5 : Internal_Entity_Decl;
Let_Result : Internal_Entity_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 383 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 382 '<FieldAccess for check_referenced_decl at parser.py:1219>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1219



--# expr-start 381 '<as_entity at parser.py:1219>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1219



--# expr-start 380 '<FieldAccess for value at parser.py:1219>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1219



--# expr-start 379 '<at at parser.py:1219>' Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1219

--# expr-start 378 '<FieldAccess for params at parser.py:1219>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1219



--# expr-start 377 '<FieldAccess for params at parser.py:1219>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1219



--# expr-start 376 '<FieldAccess for get_annotation at parser.py:1218>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1218



--# expr-start 374 '<FieldAccess for full_decl at parser.py:1218>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1218






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.full_decl
Fld := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 374


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 375 '<Symbol with_lexer at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_With_Lexer); 
--# expr-done 375
--# property-call-start FullDecl.get_annotation
Fld_1 := Liblktlang.Implementation.Full_Decl_P_Get_Annotation (Node => Fld.Node, Name => Sym);
--# end
--# expr-done 376


   

      if Fld_1 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Fld_1.Decl_Annotation_F_Params;
--# expr-done 377


   

      if Fld_2 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Fld_2.Decl_Annotation_Params_F_Params;
--# expr-done 378


Get_Result := Get (Self, Fld_3, 0, True); 
--# expr-done 379


   

      if Get_Result = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_4 := Get_Result.Param_F_Value;
--# expr-done 380
Is_Null := Fld_4 = null; 
if Is_Null then
   
   If_Result := No_Entity_Expr;
else
   

As_Entity := (Info => E_Info, Node => Fld_4); 
   If_Result := As_Entity;
end if;



--# expr-done 381


   

      if If_Result.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Expr.check_referenced_decl
Fld_5 := Liblktlang.Implementation.Expr_P_Check_Referenced_Decl (Node => If_Result.Node, E_Info => If_Result.Info);
--# end
--# expr-done 382
Let_Result := Fld_5; 
--# end
--# expr-done 383

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Grammar_Decl_P_Lexer;
--# end

   







--# property-start GrammarDecl.check_correctness_pre D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1226
pragma Warnings (Off, "is not referenced");
 function Grammar_Decl_P_Check_Correctness_Pre
  
  (Node : Bare_Grammar_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Grammar_Decl  := Bare_Grammar_Decl (Node);
     Ent : Internal_Entity_Grammar_Decl :=
       Internal_Entity_Grammar_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Semantic_Result_Array_Access;

      
            procedure Finalizer_Scope_103 with Inline_Always;
            procedure Finalizer_Scope_358 with Inline_Always;
            procedure Finalizer_Scope_359 with Inline_Always;
            procedure Finalizer_Scope_360 with Inline_Always;
            procedure Finalizer_Scope_363 with Inline_Always;
            procedure Finalizer_Scope_364 with Inline_Always;
            procedure Finalizer_Scope_365 with Inline_Always;
            procedure Finalizer_Scope_366 with Inline_Always;
            procedure Finalizer_Scope_367 with Inline_Always;

      Block_Var : Bare_Decl_Annotation;
Block_Var_1 : Bare_Param_List;
Block_Var_2 : Bare_Param;
Block_Var_3 : Internal_Entity_Lexer_Decl;
Var_Expr_8 : Bare_Decl_Annotation;
Var_Expr_9 : Bare_Decl_Annotation_Params;
Var_Expr_10 : Bare_Param;
Var_Expr_11 : Bare_Ref_Id;
Var_Expr : Boolean;
Var_Expr_1 : Internal_Semantic_Result_Array_Access;
Var_Expr_2 : Boolean;
Var_Expr_3 : Bare_Param_List;
Var_Expr_12 : Bare_Param;
Var_Expr_13 : Bare_Param;
Var_Expr_14 : Bare_Decl_Annotation;
Var_Expr_4 : Boolean;
Var_Expr_15 : Bare_Param;
Fld : Internal_Entity_Full_Decl;
Sym : Symbol_Type;
Fld_1 : Bare_Decl_Annotation;
Fld_2 : Bare_Decl_Annotation_Params;
Fld_3 : Bare_Param_List;
Scope_Result : Bare_Param_List;
Result_Var : Bare_Param_List;
Scope_Result_1 : Bare_Param_List;
Result_Var_1 : Bare_Param_List;
Get_Result : Bare_Param;
Fld_4 : Bare_Expr;
Cast_Result : Bare_Ref_Id;
As_Entity : Internal_Entity_Ref_Id;
Is_Null : Boolean;
If_Result : Internal_Entity_Ref_Id;
Fld_5 : Internal_Semantic_Result := No_Semantic_Result;
Fld_6 : Internal_Entity_Decl;
Cast_Result_1 : Internal_Entity_Lexer_Decl;
Scope_Result_2 : Internal_Entity_Lexer_Decl;
Result_Var_2 : Internal_Entity_Lexer_Decl;
Scope_Result_3 : Internal_Entity_Lexer_Decl;
Result_Var_3 : Internal_Entity_Lexer_Decl;
Is_Null_1 : Boolean;
Str : String_Type;
Fld_7 : Internal_Semantic_Result := No_Semantic_Result;
Singleton : Internal_Semantic_Result_Array_Access;
Scope_Result_4 : Internal_Semantic_Result_Array_Access;
Result_Var_4 : Internal_Semantic_Result_Array_Access;
Node_Children : Bare_Lkt_Node_Array_Access;
Len : Integer;
Is_Equal : Boolean;
Not_Val : Boolean;
Scope_Result_5 : Boolean;
Result_Var_5 : Boolean;
Fld_8 : Bare_Ref_Id;
Is_Null_2 : Boolean;
Scope_Result_6 : Boolean;
Result_Var_6 : Boolean;
Not_Val_1 : Boolean;
If_Result_1 : Boolean;
Fld_9 : Bare_Expr;
Is_Null_3 : Boolean;
Scope_Result_7 : Boolean;
Result_Var_7 : Boolean;
If_Result_2 : Boolean;
Str_1 : String_Type;
Fld_10 : Internal_Semantic_Result := No_Semantic_Result;
Singleton_1 : Internal_Semantic_Result_Array_Access;
Scope_Result_8 : Internal_Semantic_Result_Array_Access;
Result_Var_8 : Internal_Semantic_Result_Array_Access;
Scope_Result_9 : Internal_Semantic_Result_Array_Access;
Result_Var_9 : Internal_Semantic_Result_Array_Access;
Scope_Result_10 : Internal_Semantic_Result_Array_Access;
Is_Null_4 : Boolean;
Str_2 : String_Type;
Fld_11 : Internal_Semantic_Result := No_Semantic_Result;
Singleton_2 : Internal_Semantic_Result_Array_Access;
Scope_Result_11 : Internal_Semantic_Result_Array_Access;
Result_Var_10 : Internal_Semantic_Result_Array_Access;
Scope_Result_12 : Internal_Semantic_Result_Array_Access;
Result_Var_11 : Internal_Semantic_Result_Array_Access;
Result_Var_12 : Internal_Semantic_Result_Array_Access;
Concat_Result : Internal_Semantic_Result_Array_Access;
Let_Result : Internal_Semantic_Result_Array_Access;

            procedure Finalizer_Scope_103 is
            begin
                     Dec_Ref (Scope_Result_4);
                     Dec_Ref (Result_Var_4);
                     Dec_Ref (Scope_Result_9);
                     Dec_Ref (Result_Var_9);
                     Dec_Ref (Scope_Result_10);
                     Dec_Ref (Scope_Result_12);
                     Dec_Ref (Result_Var_11);
                     Dec_Ref (Result_Var_12);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_103;
            procedure Finalizer_Scope_358 is
            begin
                     Dec_Ref (Fld_5);
            end Finalizer_Scope_358;
            procedure Finalizer_Scope_359 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Fld_7);
                     Dec_Ref (Singleton);
            end Finalizer_Scope_359;
            procedure Finalizer_Scope_360 is
            begin
                     Dec_Ref (Node_Children);
            end Finalizer_Scope_360;
            procedure Finalizer_Scope_363 is
            begin
                     Dec_Ref (Scope_Result_8);
                     Dec_Ref (Result_Var_8);
            end Finalizer_Scope_363;
            procedure Finalizer_Scope_364 is
            begin
                     Dec_Ref (Str_1);
                     Dec_Ref (Fld_10);
                     Dec_Ref (Singleton_1);
            end Finalizer_Scope_364;
            procedure Finalizer_Scope_365 is
            begin
                     Dec_Ref (Var_Expr_1);
            end Finalizer_Scope_365;
            procedure Finalizer_Scope_366 is
            begin
                     Dec_Ref (Scope_Result_11);
                     Dec_Ref (Result_Var_10);
            end Finalizer_Scope_366;
            procedure Finalizer_Scope_367 is
            begin
                     Dec_Ref (Str_2);
                     Dec_Ref (Fld_11);
                     Dec_Ref (Singleton_2);
            end Finalizer_Scope_367;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 413 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 386 '<FieldAccess for get_annotation at parser.py:1231>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1231



--# expr-start 384 '<FieldAccess for full_decl at parser.py:1231>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1231






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.full_decl
Fld := Liblktlang.Implementation.Decl_P_Full_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 384


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 385 '<Symbol with_lexer at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_With_Lexer); 
--# expr-done 385
--# property-call-start FullDecl.get_annotation
Fld_1 := Liblktlang.Implementation.Full_Decl_P_Get_Annotation (Node => Fld.Node, Name => Sym);
--# end
--# expr-done 386
Block_Var := Fld_1; 
--# bind lexer_annot Block_Var




Var_Expr_8 := Block_Var;





if Var_Expr_8 /= No_Bare_Lkt_Node then
   



   --# scope-start


   




--# expr-start 387 '<FieldAccess for params at parser.py:1232>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1232






   

      if Var_Expr_8 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Var_Expr_8.Decl_Annotation_F_Params;
--# expr-done 387
Var_Expr_9 := Fld_2;





if Var_Expr_9 /= No_Bare_Lkt_Node then
   



   --# scope-start


   

--# expr-start 388 '<FieldAccess for params at parser.py:1232>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1232






   

      if Var_Expr_9 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Var_Expr_9.Decl_Annotation_Params_F_Params;
--# expr-done 388
Scope_Result := Fld_3;


   --# end


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Bare_Lkt_Node;
end if;



Scope_Result_1 := Result_Var;


   --# end


   Result_Var_1 := Scope_Result_1;
else
   
   Result_Var_1 := No_Bare_Lkt_Node;
end if;



Block_Var_1 := Result_Var_1; 
--# bind lexer_annot_params Block_Var_1
--# expr-start 389 '<at at parser.py:1233>' Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1233




Get_Result := Get (Self, Block_Var_1, 0, True); 
--# expr-done 389
Block_Var_2 := Get_Result; 
--# bind lexer_arg Block_Var_2




Var_Expr_10 := Block_Var_2;





if Var_Expr_10 /= No_Bare_Lkt_Node then
   



   --# scope-start


   




--# expr-start 391 '<Cast to RefId at parser.py:1236>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1236





--# expr-start 390 '<FieldAccess for value at parser.py:1236>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1236






   

      if Var_Expr_10 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_4 := Var_Expr_10.Param_F_Value;
--# expr-done 390



   if Fld_4 = null
      or else Fld_4.Kind in Lkt_Ref_Id_Range
   then
      
      Cast_Result := Fld_4;

   else
         Cast_Result := No_Bare_Lkt_Node;
   end if;


--# expr-done 391
Var_Expr_11 := Cast_Result;





if Var_Expr_11 /= No_Bare_Lkt_Node then
   



   --# scope-start


   

--# expr-start 395 '<Cast to LexerDecl at parser.py:1237>' Cast_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1237





--# expr-start 394 '<FieldAccess for result_decl at parser.py:1237>' Fld_6 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1237



--# expr-start 393 '<FieldAccess for referenced_decl at parser.py:1237>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1237



--# expr-start 392 '<as_entity at parser.py:1236>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1236




Is_Null := Var_Expr_11 = null; 
if Is_Null then
   
   If_Result := No_Entity_Ref_Id;
else
   

As_Entity := (Info => E_Info, Node => Var_Expr_11); 
   If_Result := As_Entity;
end if;



--# expr-done 392


   

      if If_Result.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Expr.referenced_decl'
Fld_5 := Liblktlang.Implementation.Dispatcher_Expr_P_Referenced_Decl (Node => If_Result.Node, E_Info => If_Result.Info);
--# end
--# expr-done 393


   




Fld_6 := Fld_5.Result_Decl;
--# expr-done 394



   if Fld_6.Node = null
      or else Fld_6.Node.Kind in Lkt_Lexer_Decl_Range
   then
      
      Cast_Result_1 := Create_Internal_Entity_Lexer_Decl
        (Node => Fld_6.Node,
         Info => Fld_6.Info);

   else
         Cast_Result_1 := No_Entity_Lexer_Decl;
   end if;


--# expr-done 395
Scope_Result_2 := Cast_Result_1;


   --# end
      Finalizer_Scope_358;


   Result_Var_2 := Scope_Result_2;
else
   
   Result_Var_2 := No_Entity_Lexer_Decl;
end if;



Scope_Result_3 := Result_Var_2;


   --# end


   Result_Var_3 := Scope_Result_3;
else
   
   Result_Var_3 := No_Entity_Lexer_Decl;
end if;



Block_Var_3 := Result_Var_3; 
--# bind lexer_decl Block_Var_3



--# expr-start 396 '<IsNull at parser.py:1242>' Is_Null_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1242

Is_Null_1 := Block_Var = null; 
--# expr-done 396
Var_Expr := Is_Null_1;





if Var_Expr /= False then
   



   --# scope-start


   --# bind _ Var_Expr

--# expr-start 398 '<FieldAccess for error at parser.py:1244>' Fld_7 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1244






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 397 '<String at parser.py:1244>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1244
Str := Create_String ("missing mandatory ""@with_lexer"" annotation"); 
--# expr-done 397
--# property-call-start LktNode.error
Fld_7 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Self, Message => Str);
--# end
--# expr-done 398
                Singleton := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton.Items (1) := Fld_7;
                Inc_Ref (Singleton.Items (1));
            
Scope_Result_4 := Singleton;
   Inc_Ref (Scope_Result_4);


   --# end
      Finalizer_Scope_359;


   Result_Var_4 := Scope_Result_4;
else
   
   Result_Var_4 := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var_4);


















Var_Expr_3 := Block_Var_1;





if Var_Expr_3 /= No_Bare_Lkt_Node then
   



   --# scope-start


   --# bind p Var_Expr_3

--# expr-start 402 '<Not at parser.py:1250>' Not_Val D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1250
--# expr-start 401 '<Eq at parser.py:1250>' Is_Equal D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1250
--# expr-start 400 '<length at parser.py:1250>' Len D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1250
--# expr-start 399 '<children at parser.py:1250>' Node_Children D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1250






   

      if Var_Expr_3 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Node_Children := Children (Var_Expr_3); 
--# expr-done 399
Len := Length (Node_Children); 
--# expr-done 400

Is_Equal := Len = 1; 
--# expr-done 401
Not_Val := not (Is_Equal); 
--# expr-done 402
Scope_Result_5 := Not_Val;


   --# end
      Finalizer_Scope_360;


   Result_Var_5 := Scope_Result_5;
else
   
   Result_Var_5 := False;
end if;



if Result_Var_5 then
   
   If_Result_1 := True;
else
   --# expr-start 405 '<Not at parser.py:1251>' Not_Val_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1251




Var_Expr_12 := Block_Var_2;





if Var_Expr_12 /= No_Bare_Lkt_Node then
   



   --# scope-start


   

--# expr-start 404 '<IsNull at parser.py:1251>' Is_Null_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1251
--# expr-start 403 '<FieldAccess for name at parser.py:1251>' Fld_8 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1251






   

      if Var_Expr_12 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_8 := Var_Expr_12.Param_F_Name;
--# expr-done 403
Is_Null_2 := Fld_8 = null; 
--# expr-done 404
Scope_Result_6 := Is_Null_2;


   --# end


   Result_Var_6 := Scope_Result_6;
else
   
   Result_Var_6 := False;
end if;



Not_Val_1 := not (Result_Var_6); 
--# expr-done 405
   If_Result_1 := Not_Val_1;
end if;



if If_Result_1 then
   
   If_Result_2 := True;
else
   



Var_Expr_13 := Block_Var_2;





if Var_Expr_13 /= No_Bare_Lkt_Node then
   



   --# scope-start


   

--# expr-start 407 '<IsNull at parser.py:1252>' Is_Null_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1252
--# expr-start 406 '<FieldAccess for value at parser.py:1252>' Fld_9 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1252






   

      if Var_Expr_13 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_9 := Var_Expr_13.Param_F_Value;
--# expr-done 406
Is_Null_3 := Fld_9 = null; 
--# expr-done 407
Scope_Result_7 := Is_Null_3;


   --# end


   Result_Var_7 := Scope_Result_7;
else
   
   Result_Var_7 := False;
end if;



   If_Result_2 := Result_Var_7;
end if;



Var_Expr_2 := If_Result_2;





if Var_Expr_2 /= False then
   



   --# scope-start


   --# bind _ Var_Expr_2





Var_Expr_14 := Block_Var;





if Var_Expr_14 /= No_Bare_Lkt_Node then
   



   --# scope-start


   

--# expr-start 409 '<FieldAccess for error at parser.py:1257>' Fld_10 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1257






   

      if Var_Expr_14 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 408 '<String at parser.py:1257>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1257
Str_1 := Create_String ("exactly 1 positional argument expected"); 
--# expr-done 408
--# property-call-start LktNode.error
Fld_10 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Var_Expr_14, Message => Str_1);
--# end
--# expr-done 409
                Singleton_1 := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton_1.Items (1) := Fld_10;
                Inc_Ref (Singleton_1.Items (1));
            
Scope_Result_8 := Singleton_1;
   Inc_Ref (Scope_Result_8);


   --# end
      Finalizer_Scope_364;


   Result_Var_8 := Scope_Result_8;
else
   
   Result_Var_8 := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var_8);


Scope_Result_9 := Result_Var_8;
   Inc_Ref (Scope_Result_9);


   --# end
      Finalizer_Scope_363;


   Result_Var_9 := Scope_Result_9;
else
   
   Result_Var_9 := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var_9);


Var_Expr_1 := Result_Var_9;

      Inc_Ref (Var_Expr_1);




if Var_Expr_1 /= No_Internal_Semantic_Result_Array_Type then
   



   --# scope-start


   --# bind e Var_Expr_1


Scope_Result_10 := Var_Expr_1;
   Inc_Ref (Scope_Result_10);


   --# end
      Finalizer_Scope_365;


   Result_Var_12 := Scope_Result_10;
else
   


--# expr-start 410 '<IsNull at parser.py:1261>' Is_Null_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1261

Is_Null_4 := Block_Var_3.Node = null; 
--# expr-done 410
Var_Expr_4 := Is_Null_4;





if Var_Expr_4 /= False then
   



   --# scope-start


   --# bind _ Var_Expr_4





Var_Expr_15 := Block_Var_2;





if Var_Expr_15 /= No_Bare_Lkt_Node then
   



   --# scope-start


   

--# expr-start 412 '<FieldAccess for error at parser.py:1263>' Fld_11 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1263






   

      if Var_Expr_15 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 411 '<String at parser.py:1263>' Str_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1263
Str_2 := Create_String ("reference to a lexer expected"); 
--# expr-done 411
--# property-call-start LktNode.error
Fld_11 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Var_Expr_15, Message => Str_2);
--# end
--# expr-done 412
                Singleton_2 := Create_Internal_Semantic_Result_Array (Items_Count => 1);
                Singleton_2.Items (1) := Fld_11;
                Inc_Ref (Singleton_2.Items (1));
            
Scope_Result_11 := Singleton_2;
   Inc_Ref (Scope_Result_11);


   --# end
      Finalizer_Scope_367;


   Result_Var_10 := Scope_Result_11;
else
   
   Result_Var_10 := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var_10);


Scope_Result_12 := Result_Var_10;
   Inc_Ref (Scope_Result_12);


   --# end
      Finalizer_Scope_366;


   Result_Var_11 := Scope_Result_12;
else
   
   Result_Var_11 := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var_11);


   Result_Var_12 := Result_Var_11;
end if;

      Inc_Ref (Result_Var_12);


Concat_Result := Concat (Result_Var_4, Result_Var_12); 
Let_Result := Concat_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 413

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_103;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_103;
                     Finalizer_Scope_358;
                     Finalizer_Scope_359;
                     Finalizer_Scope_360;
                     Finalizer_Scope_363;
                     Finalizer_Scope_364;
                     Finalizer_Scope_365;
                     Finalizer_Scope_366;
                     Finalizer_Scope_367;




            raise;
      end;



   return Property_Result;
end Grammar_Decl_P_Check_Correctness_Pre;
--# end

   







--# property-start GrammarDecl.check_legality D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1269
pragma Warnings (Off, "is not referenced");
 function Grammar_Decl_P_Check_Legality
  
  (Node : Bare_Grammar_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Grammar_Decl  := Bare_Grammar_Decl (Node);
     Ent : Internal_Entity_Grammar_Decl :=
       Internal_Entity_Grammar_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Semantic_Result_Array_Access;

      
            procedure Finalizer_Scope_104 with Inline_Always;
            procedure Finalizer_Scope_370 with Inline_Always;
            procedure Finalizer_Scope_371 with Inline_Always;
            procedure Finalizer_Scope_372 with Inline_Always;

      Block_Var : Internal_Entity_Full_Decl_Array_Access;
Var_Expr : Boolean;
Fld : Internal_Entity_Full_Decl_List;
Fld_1 : Bare_Full_Decl_List;
Fld_2 : Internal_Entity_Info;
Item_19 : Internal_Entity_Full_Decl;
Bare_Item_19 : Bare_Full_Decl;
As_Entity : Internal_Entity_Full_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Full_Decl;
Fld_3 : Internal_Entity_Decl;
Is_A : Boolean;
Sym : Symbol_Type;
Fld_4 : Boolean;
If_Result_1 : Boolean;
Map_Result : Internal_Entity_Full_Decl_Array_Access;
Fld_5 : Internal_Entity_Full_Decl_List;
Fld_6 : Bare_Full_Decl_List;
Fld_7 : Internal_Entity_Info;
Item_20 : Internal_Entity_Full_Decl;
Bare_Item_20 : Bare_Full_Decl;
Fld_8 : Internal_Entity_Decl;
Fld_9 : String_Type;
Str : String_Type;
Concat_Result : String_Type;
Fld_10 : String_Type;
Concat_Result_1 : String_Type;
Fld_11 : Internal_Semantic_Result := No_Semantic_Result;
As_Entity_1 : Internal_Entity_Full_Decl;
Is_Null_1 : Boolean;
If_Result_2 : Internal_Entity_Full_Decl;
Fld_12 : Internal_Entity_Decl;
Is_A_1 : Boolean;
Not_Val : Boolean;
Map_Result_1 : Internal_Semantic_Result_Array_Access;
Len : Integer;
Is_Equal : Boolean;
Str_1 : String_Type;
Fld_13 : Internal_Semantic_Result := No_Semantic_Result;
Array_Lit : Internal_Semantic_Result_Array_Access;
Scope_Result : Internal_Semantic_Result_Array_Access;
Result_Var : Internal_Semantic_Result_Array_Access;
Concat_Result_2 : Internal_Semantic_Result_Array_Access;
Index_22 : Integer;
Str_2 : String_Type;
Fld_14 : Internal_Semantic_Result := No_Semantic_Result;
Comp_Result : Boolean;
Map_Result_2 : Internal_Semantic_Result_Array_Access;
Concat_Result_3 : Internal_Semantic_Result_Array_Access;
Let_Result : Internal_Semantic_Result_Array_Access;

            procedure Finalizer_Scope_104 is
            begin
                     Dec_Ref (Map_Result);
                     Dec_Ref (Block_Var);
                     Dec_Ref (Map_Result_1);
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
                     Dec_Ref (Concat_Result_2);
                     Dec_Ref (Map_Result_2);
                     Dec_Ref (Concat_Result_3);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_104;
            procedure Finalizer_Scope_370 is
            begin
                     Dec_Ref (Fld_9);
                     Dec_Ref (Str);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Fld_10);
                     Dec_Ref (Concat_Result_1);
                     Dec_Ref (Fld_11);
            end Finalizer_Scope_370;
            procedure Finalizer_Scope_371 is
            begin
                     Dec_Ref (Str_1);
                     Dec_Ref (Fld_13);
                     Dec_Ref (Array_Lit);
            end Finalizer_Scope_371;
            procedure Finalizer_Scope_372 is
            begin
                     Dec_Ref (Str_2);
                     Dec_Ref (Fld_14);
            end Finalizer_Scope_372;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 438 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 419 '<Filter at parser.py:1272>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1272








   

   --# expr-start 414 '<FieldAccess for rules at parser.py:1272>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1272






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Full_Decl_List (Node => Ent.Node.Grammar_Decl_F_Rules, Info => Ent.Info);
--# expr-done 414






   




Fld_1 := Fld.Node;

   
      declare
         Map_Result_Vec : Internal_Entity_Full_Decl_Vectors.Vector;
      begin
   

   

      if Fld_1 = null then
         
      Map_Result := Create_Internal_Entity_Full_Decl_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Full_Decl_List := Fld_1;
      begin
         for Untyped_Item_19 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_19 := Untyped_Item_19; 
                  



Is_Null := Bare_Item_19 = null; 
if Is_Null then
   
   If_Result := No_Entity_Full_Decl;
else
   





   




Fld_2 := Fld.Info;

As_Entity := (Info => Fld_2, Node => Bare_Item_19); 
   If_Result := As_Entity;
end if;



                  Item_19 := If_Result; 

            
   --# scope-start


               --# bind d Item_19

            
         


--# expr-start 416 '<IsA GrammarRuleDecl at parser.py:1273>' Is_A D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1273
--# expr-start 415 '<FieldAccess for decl at parser.py:1273>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1273






   

      if Item_19.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Create_Internal_Entity_Decl (Node => Item_19.Node.Full_Decl_F_Decl, Info => Item_19.Info);
--# expr-done 415
Is_A := Fld_3.Node /= null 
and then Fld_3.Node.Kind in Lkt_Grammar_Rule_Decl_Range; 
--# expr-done 416
if Is_A then
   --# expr-start 418 '<FieldAccess for has_annotation at parser.py:1273>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1273






   

      if Item_19.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 417 '<Symbol main_rule at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Main_Rule); 
--# expr-done 417
--# property-call-start FullDecl.has_annotation
Fld_4 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Item_19.Node, Name => Sym);
--# end
--# expr-done 418
   If_Result_1 := Fld_4;
else
   
   If_Result_1 := False;
end if;



         if If_Result_1 then
            
         
      
      

         declare
            Item_To_Append : constant Internal_Entity_Full_Decl := Item_19;
         begin
            Internal_Entity_Full_Decl_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end



         end loop;
      end;
   
      end if;

   

         Map_Result := Create_Internal_Entity_Full_Decl_Array
           (Items_Count => Natural (Internal_Entity_Full_Decl_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Entity_Full_Decl_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Entity_Full_Decl_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Entity_Full_Decl_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 419
Block_Var := Map_Result; Inc_Ref (Block_Var);
--# bind main_rules Block_Var
--# expr-start 429 '<FilterMap at parser.py:1278>' Map_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1278








   

   --# expr-start 420 '<FieldAccess for rules at parser.py:1278>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1278






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_5 := Create_Internal_Entity_Full_Decl_List (Node => Ent.Node.Grammar_Decl_F_Rules, Info => Ent.Info);
--# expr-done 420






   




Fld_6 := Fld_5.Node;

   
      declare
         Map_Result_1_Vec : Internal_Semantic_Result_Vectors.Vector;
      begin
   

   

      if Fld_6 = null then
         
      Map_Result_1 := Create_Internal_Semantic_Result_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Full_Decl_List := Fld_6;
      begin
         for Untyped_Item_20 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_20 := Untyped_Item_20; 
                  



Is_Null_1 := Bare_Item_20 = null; 
if Is_Null_1 then
   
   If_Result_2 := No_Entity_Full_Decl;
else
   





   




Fld_7 := Fld_5.Info;

As_Entity_1 := (Info => Fld_7, Node => Bare_Item_20); 
   If_Result_2 := As_Entity_1;
end if;



                  Item_20 := If_Result_2; 

            
   --# scope-start


               --# bind d Item_20

            
         --# expr-start 423 '<Not at parser.py:1283>' Not_Val D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1283
--# expr-start 422 '<IsA GrammarRuleDecl at parser.py:1283>' Is_A_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1283
--# expr-start 421 '<FieldAccess for decl at parser.py:1283>' Fld_12 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1283






   

      if Item_20.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_12 := Create_Internal_Entity_Decl (Node => Item_20.Node.Full_Decl_F_Decl, Info => Item_20.Info);
--# expr-done 421
Is_A_1 := Fld_12.Node /= null 
and then Fld_12.Node.Kind in Lkt_Grammar_Rule_Decl_Range; 
--# expr-done 422
Not_Val := not (Is_A_1); 
--# expr-done 423
         if Not_Val then
            
         
      --# expr-start 428 '<FieldAccess for error at parser.py:1279>' Fld_11 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1279






   

      if Item_20.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 425 '<FieldAccess for decl_type_name at parser.py:1280>' Fld_9 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1280



--# expr-start 424 '<FieldAccess for decl at parser.py:1280>' Fld_8 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1280






   

      if Item_20.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_8 := Create_Internal_Entity_Decl (Node => Item_20.Node.Full_Decl_F_Decl, Info => Item_20.Info);
--# expr-done 424


   

      if Fld_8.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.decl_type_name'
Fld_9 := Liblktlang.Implementation.Dispatcher_Decl_P_Decl_Type_Name (Node => Fld_8.Node, E_Info => Fld_8.Info);
--# end
--# expr-done 425
--# expr-start 426 '<String at parser.py:1280>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1280
Str := Create_String (" forbidden in "); 
--# expr-done 426
Concat_Result := Concat_String (Fld_9, Str); 
--# expr-start 427 '<FieldAccess for decl_type_name at parser.py:1281>' Fld_10 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1281






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.decl_type_name'
Fld_10 := Liblktlang.Implementation.Dispatcher_Decl_P_Decl_Type_Name (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 427
Concat_Result_1 := Concat_String (Concat_Result, Fld_10); 
--# property-call-start LktNode.error
Fld_11 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Item_20.Node, Message => Concat_Result_1);
--# end
--# expr-done 428
      

         declare
            Item_To_Append : constant Internal_Semantic_Result := Fld_11;
         begin
               Inc_Ref (Item_To_Append);
            Internal_Semantic_Result_Vectors.Append (Map_Result_1_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end
      Finalizer_Scope_370;



         end loop;
      end;
   
      end if;

   

         Map_Result_1 := Create_Internal_Semantic_Result_Array
           (Items_Count => Natural (Internal_Semantic_Result_Vectors.Length (Map_Result_1_Vec)));
         for I in Map_Result_1.Items'Range loop
            Map_Result_1.Items (I) := Internal_Semantic_Result_Vectors.Get
              (Map_Result_1_Vec,
               I + Internal_Semantic_Result_Vectors.Index_Type'First - Map_Result_1.Items'First);
         end loop;
         Internal_Semantic_Result_Vectors.Destroy (Map_Result_1_Vec);

      end;
   



--# expr-done 429



--# expr-start 431 '<Eq at parser.py:1287>' Is_Equal D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1287
--# expr-start 430 '<length at parser.py:1287>' Len D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1287

Len := Length (Block_Var); 
--# expr-done 430

Is_Equal := Len = 0; 
--# expr-done 431
Var_Expr := Is_Equal;





if Var_Expr /= False then
   



   --# scope-start


   --# bind _ Var_Expr

--# expr-start 434 '<ArrayLiteral at ???>' Array_Lit None
--# expr-start 433 '<FieldAccess for error at parser.py:1289>' Fld_13 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1289






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 432 '<String at parser.py:1289>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1289
Str_1 := Create_String ("main rule missing (@main_rule annotation)"); 
--# expr-done 432
--# property-call-start LktNode.error
Fld_13 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Self, Message => Str_1);
--# end
--# expr-done 433
Array_Lit := Create_Internal_Semantic_Result_Array (Internal_Internal_Semantic_Result_Array'(1 => Fld_13)); 
--# expr-done 434
Scope_Result := Array_Lit;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_371;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Internal_Semantic_Result_Array_Type;
end if;

      Inc_Ref (Result_Var);


Concat_Result_2 := Concat (Map_Result_1, Result_Var); 
--# expr-start 437 '<FilterMap at parser.py:1293>' Map_Result_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1293








   

   

   
      declare
         Map_Result_2_Vec : Internal_Semantic_Result_Vectors.Vector;
      begin
   

   

      
         Index_22 := 0;

      declare
         
         Collection : constant Internal_Entity_Full_Decl_Array_Access := Block_Var;
      begin
         for Item_21 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind d Item_21
               --# bind i Index_22

            
         

Comp_Result := Index_22 > 0; 
         if Comp_Result then
            
         
      --# expr-start 436 '<FieldAccess for error at parser.py:1294>' Fld_14 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1294






   

      if Item_21.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 435 '<String at parser.py:1294>' Str_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1294
Str_2 := Create_String ("only one main rule allowed"); 
--# expr-done 435
--# property-call-start LktNode.error
Fld_14 := Liblktlang.Implementation.Lkt_Node_P_Error (Node => Item_21.Node, Message => Str_2);
--# end
--# expr-done 436
      

         declare
            Item_To_Append : constant Internal_Semantic_Result := Fld_14;
         begin
               Inc_Ref (Item_To_Append);
            Internal_Semantic_Result_Vectors.Append (Map_Result_2_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end
      Finalizer_Scope_372;



               Index_22 := Index_22 + 1;
         end loop;
      end;
   

   

         Map_Result_2 := Create_Internal_Semantic_Result_Array
           (Items_Count => Natural (Internal_Semantic_Result_Vectors.Length (Map_Result_2_Vec)));
         for I in Map_Result_2.Items'Range loop
            Map_Result_2.Items (I) := Internal_Semantic_Result_Vectors.Get
              (Map_Result_2_Vec,
               I + Internal_Semantic_Result_Vectors.Index_Type'First - Map_Result_2.Items'First);
         end loop;
         Internal_Semantic_Result_Vectors.Destroy (Map_Result_2_Vec);

      end;
   



--# expr-done 437
Concat_Result_3 := Concat (Concat_Result_2, Map_Result_2); 
Let_Result := Concat_Result_3; Inc_Ref (Let_Result);
--# end
--# expr-done 438

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_104;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_104;
                     Finalizer_Scope_370;
                     Finalizer_Scope_371;
                     Finalizer_Scope_372;




            raise;
      end;



   return Property_Result;
end Grammar_Decl_P_Check_Legality;
--# end

   







--# property-start GrammarDecl._env_mappings_6 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1222
pragma Warnings (Off, "is not referenced");
 function Env_Mappings_6
  
  (Node : Bare_Grammar_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Env_Assoc
is
   Self : Bare_Grammar_Decl  := Bare_Grammar_Decl (Node);
     Ent : Internal_Entity_Grammar_Decl :=
       Internal_Entity_Grammar_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Env_Assoc;

      
            procedure Finalizer_Scope_285 with Inline_Always;

      Fld : Symbol_Type;
New_Struct : Internal_Designated_Env := No_Designated_Env;
Cast_Expr : Bare_Grammar_Decl;
Cast_Result : Bare_Lkt_Node;
New_Struct_1 : Internal_Env_Assoc := No_Env_Assoc;

            procedure Finalizer_Scope_285 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (New_Struct_1);
            end Finalizer_Scope_285;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 441 '<New EnvAssoc at parser.py:1222>' New_Struct_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1222
--# expr-start 439 '<New DesignatedEnv at parser.py:1222>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1222



Inc_Ref (Empty_Env);
New_Struct := (Direct_Env => Empty_Env, Env_Name => No_Symbol, Kind => Current_Env); 
--# expr-done 439
--# expr-start 440 '<FieldAccess for name at parser.py:1222>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1222






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Ent.Node);
--# end
--# expr-done 440







Cast_Expr := Self; 



   
      Cast_Result := Cast_Expr;


Inc_Ref (New_Struct);
New_Struct_1 := (Dest_Env => New_Struct, Key => Fld, Metadata => No_Metadata, Value => Cast_Result); 
--# expr-done 441

         Property_Result := New_Struct_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_285;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_285;




            raise;
      end;



   return Property_Result;
end Env_Mappings_6;
--# end

   







--# property-start GrammarDecl._env_trans_parent_7 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1223
pragma Warnings (Off, "is not referenced");
 function Env_Trans_Parent_7
  
  (Node : Bare_Grammar_Decl
  )

   return Boolean
is
   Self : Bare_Grammar_Decl  := Bare_Grammar_Decl (Node);
      --# bind self Self

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := False;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Env_Trans_Parent_7;
--# end



   


      

   --
   --  Primitives for Bare_Lexer_Decl
   --

   



   

   

   

   

   

   

   


      procedure Lexer_Decl_Pre_Env_Actions
        (Self            : Bare_Lexer_Decl;
         State           : in out PLE_Node_State;
         Add_To_Env_Only : Boolean := False) is
      begin
            


   

      declare
         Resolver : constant Entity_Resolver :=
            null;

         

         Mapping : Internal_Env_Assoc :=
            Liblktlang.Implementation.Env_Mappings_3 (Node => 
Self);
      begin

         Add_To_Env
           (Self,
            State,
            Mapping.Key,
            Mapping.Value,
            Mapping.Metadata,
            Resolver,
            Mapping.Dest_Env,
            DSL_Location => "parser.py:1140");
         Dec_Ref (Mapping.Dest_Env);

      end;
   
   
            


   
      if Add_To_Env_Only then
         return;
      end if;

      declare
         No_Parent         : constant Boolean :=
            False;
         Transitive_Parent : constant Boolean :=
            Liblktlang.Implementation.Env_Trans_Parent_4 (Node => 
Self);
         Names             : Symbol_Type_Array_Access :=
            null;
      begin
         Add_Env (Self, State, No_Parent, Transitive_Parent, Names);
      end;
   
   
            


   

      declare
         Resolver : constant Entity_Resolver :=
            null;

         

         Mappings : Internal_Env_Assoc_Array_Access :=
            Liblktlang.Implementation.Env_Mappings_5 (Node => 
Self);
      begin
         for Mapping of Mappings.Items loop

         Add_To_Env
           (Self,
            State,
            Mapping.Key,
            Mapping.Value,
            Mapping.Metadata,
            Resolver,
            Mapping.Dest_Env,
            DSL_Location => "parser.py:1142");

         end loop;
         Dec_Ref (Mappings);
      end;
   
   
      end;



      
      procedure Initialize_Fields_For_Lexer_Decl
        (Self : Bare_Lexer_Decl
         ; Lexer_Decl_F_Syn_Name : Bare_Def_Id
         ; Lexer_Decl_F_Rules : Bare_Lkt_Node_List
        ) is
      begin

            Self.Lexer_Decl_F_Syn_Name := Lexer_Decl_F_Syn_Name;
            Self.Lexer_Decl_F_Rules := Lexer_Decl_F_Rules;
         

      end Initialize_Fields_For_Lexer_Decl;

      
   function Lexer_Decl_F_Rules
     (Node : Bare_Lexer_Decl) return Bare_Lkt_Node_List
   is
      

   begin
         
         return Node.Lexer_Decl_F_Rules;
      
   end;


   







--# property-start LexerDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1126
pragma Warnings (Off, "is not referenced");
 function Lexer_Decl_P_Decl_Type_Name
  
  (Node : Bare_Lexer_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Lexer_Decl  := Bare_Lexer_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_96 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_96 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_96;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 442 '<String at parser.py:1126>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1126
Str := Create_String ("lexer declaration"); 
--# expr-done 442

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_96;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_96;




            raise;
      end;



   return Property_Result;
end Lexer_Decl_P_Decl_Type_Name;
--# end

   







--# property-start LexerDecl.builtin_decls D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1128
pragma Warnings (Off, "is not referenced");
 function Lexer_Decl_P_Builtin_Decls
  
  (Node : Bare_Lexer_Decl
  )

   return Internal_Env_Assoc_Array_Access
is
   Self : Bare_Lexer_Decl  := Bare_Lexer_Decl (Node);
      --# bind self Self

   


   Property_Result : Internal_Env_Assoc_Array_Access;

      
            procedure Finalizer_Scope_98 with Inline_Always;
            procedure Finalizer_Scope_373 with Inline_Always;

      Sym : Symbol_Type;
Sym_1 : Symbol_Type;
Sym_2 : Symbol_Type;
Array_Lit : Symbol_Type_Array_Access;
New_Node : Bare_Synthetic_Lexer_Decl;
Fld : Lexical_Env := Empty_Env;
New_Struct : Internal_Designated_Env := No_Designated_Env;
Cast_Result : Bare_Lkt_Node;
New_Struct_1 : Internal_Env_Assoc := No_Env_Assoc;
Map_Result : Internal_Env_Assoc_Array_Access;
Let_Result : Internal_Env_Assoc_Array_Access;

            procedure Finalizer_Scope_98 is
            begin
                     Dec_Ref (Array_Lit);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_98;
            procedure Finalizer_Scope_373 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (New_Struct);
                     Dec_Ref (New_Struct_1);
            end Finalizer_Scope_373;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Lexer_Decl_Lexer_Decl_P_Builtin_Decls,
            Items    => new Mmz_Key_Array (1 ..  1))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Lexer_Decl,
                                As_Bare_Lexer_Decl => Self);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Internal_Env_Assoc_Array_Access;
                     Inc_Ref (Property_Result);

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 452 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 451 '<Map at parser.py:1130>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1130








   

   --# expr-start 446 '<ArrayLiteral at parser.py:1130>' Array_Lit D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1130
--# expr-start 443 '<Symbol newline at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Newline); 
--# expr-done 443
--# expr-start 444 '<Symbol indent at ???>' Sym_1 None
Sym_1 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Indent); 
--# expr-done 444
--# expr-start 445 '<Symbol dedent at ???>' Sym_2 None
Sym_2 := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Dedent); 
--# expr-done 445
Array_Lit := Create_Symbol_Type_Array (Internal_Symbol_Type_Array'(1 => Sym, 2 => Sym_1, 3 => Sym_2)); 
--# expr-done 446

   
      declare
         Map_Result_Vec : Internal_Env_Assoc_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Symbol_Type_Array_Access := Array_Lit;
      begin
         for Item_23 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind sym Item_23

            
         
         
      --# expr-start 450 '<New EnvAssoc at parser.py:1132>' New_Struct_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1132
--# expr-start 448 '<New DesignatedEnv at parser.py:1135>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1135
--# expr-start 447 '<FieldAccess for children_env at parser.py:1135>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1135






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Children_Env (Node => Self);
--# expr-done 447


Inc_Ref (Fld);
New_Struct := (Direct_Env => Fld, Env_Name => No_Symbol, Kind => Direct_Env); 
--# expr-done 448







--# expr-start 449 '<New SyntheticLexerDecl at parser.py:1134>' New_Node D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1134



if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node := new Root_Node_Record
  (Lkt_Synthetic_Lexer_Decl);
Initialize
  (Self => New_Node,
   Kind => Lkt_Synthetic_Lexer_Decl,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node);


      New_Node.Synthetic_Lexer_Decl_Sym :=
         Item_23;

--# expr-done 449



   
      Cast_Result := New_Node;


Inc_Ref (New_Struct);
New_Struct_1 := (Dest_Env => New_Struct, Key => Item_23, Metadata => No_Metadata, Value => Cast_Result); 
--# expr-done 450
      

         declare
            Item_To_Append : constant Internal_Env_Assoc := New_Struct_1;
         begin
               Inc_Ref (Item_To_Append);
            Internal_Env_Assoc_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end
      Finalizer_Scope_373;



         end loop;
      end;
   

   

         Map_Result := Create_Internal_Env_Assoc_Array
           (Items_Count => Natural (Internal_Env_Assoc_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Env_Assoc_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Env_Assoc_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Env_Assoc_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 451
Let_Result := Map_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 452

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_98;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_98;
                     Finalizer_Scope_373;


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Internal_Env_Assoc_Array_Access,
                        As_Internal_Env_Assoc_Array_Access => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
               if Mmz_Stored then
                  Inc_Ref (Property_Result);
               end if;

         end if;
      end if;



   return Property_Result;
end Lexer_Decl_P_Builtin_Decls;
--# end

   







--# property-start LexerDecl._env_mappings_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1140
pragma Warnings (Off, "is not referenced");
 function Env_Mappings_3
  
  (Node : Bare_Lexer_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Env_Assoc
is
   Self : Bare_Lexer_Decl  := Bare_Lexer_Decl (Node);
     Ent : Internal_Entity_Lexer_Decl :=
       Internal_Entity_Lexer_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Env_Assoc;

      
            procedure Finalizer_Scope_281 with Inline_Always;

      Fld : Symbol_Type;
New_Struct : Internal_Designated_Env := No_Designated_Env;
Cast_Expr : Bare_Lexer_Decl;
Cast_Result : Bare_Lkt_Node;
New_Struct_1 : Internal_Env_Assoc := No_Env_Assoc;

            procedure Finalizer_Scope_281 is
            begin
                     Dec_Ref (New_Struct);
                     Dec_Ref (New_Struct_1);
            end Finalizer_Scope_281;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 455 '<New EnvAssoc at parser.py:1140>' New_Struct_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1140
--# expr-start 453 '<New DesignatedEnv at parser.py:1140>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1140



Inc_Ref (Empty_Env);
New_Struct := (Direct_Env => Empty_Env, Env_Name => No_Symbol, Kind => Current_Env); 
--# expr-done 453
--# expr-start 454 '<FieldAccess for name at parser.py:1140>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1140






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Ent.Node);
--# end
--# expr-done 454







Cast_Expr := Self; 



   
      Cast_Result := Cast_Expr;


Inc_Ref (New_Struct);
New_Struct_1 := (Dest_Env => New_Struct, Key => Fld, Metadata => No_Metadata, Value => Cast_Result); 
--# expr-done 455

         Property_Result := New_Struct_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_281;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_281;




            raise;
      end;



   return Property_Result;
end Env_Mappings_3;
--# end

   







--# property-start LexerDecl._env_trans_parent_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1141
pragma Warnings (Off, "is not referenced");
 function Env_Trans_Parent_4
  
  (Node : Bare_Lexer_Decl
  )

   return Boolean
is
   Self : Bare_Lexer_Decl  := Bare_Lexer_Decl (Node);
      --# bind self Self

   


   Property_Result : Boolean;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := False;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Env_Trans_Parent_4;
--# end

   







--# property-start LexerDecl._env_mappings_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1142
pragma Warnings (Off, "is not referenced");
 function Env_Mappings_5
  
  (Node : Bare_Lexer_Decl
  )

   return Internal_Env_Assoc_Array_Access
is
   Self : Bare_Lexer_Decl  := Bare_Lexer_Decl (Node);
      --# bind self Self

   


   Property_Result : Internal_Env_Assoc_Array_Access;

      
            procedure Finalizer_Scope_284 with Inline_Always;

      Fld : Internal_Env_Assoc_Array_Access;

            procedure Finalizer_Scope_284 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_284;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 456 '<FieldAccess for builtin_decls at parser.py:1142>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1142






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LexerDecl.builtin_decls
Fld := Liblktlang.Implementation.Lexer_Decl_P_Builtin_Decls (Node => Self);
--# end
--# expr-done 456

         Property_Result := Fld;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_284;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_284;




            raise;
      end;



   return Property_Result;
end Env_Mappings_5;
--# end



   


      

   --
   --  Primitives for Bare_Lexer_Family_Decl
   --

   



      
      procedure Initialize_Fields_For_Lexer_Family_Decl
        (Self : Bare_Lexer_Family_Decl
         ; Lexer_Family_Decl_F_Syn_Name : Bare_Def_Id
         ; Lexer_Family_Decl_F_Rules : Bare_Full_Decl_List
        ) is
      begin

            Self.Lexer_Family_Decl_F_Syn_Name := Lexer_Family_Decl_F_Syn_Name;
            Self.Lexer_Family_Decl_F_Rules := Lexer_Family_Decl_F_Rules;
         

      end Initialize_Fields_For_Lexer_Family_Decl;

      
   function Lexer_Family_Decl_F_Rules
     (Node : Bare_Lexer_Family_Decl) return Bare_Full_Decl_List
   is
      

   begin
         
         return Node.Lexer_Family_Decl_F_Rules;
      
   end;


   







--# property-start LexerFamilyDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1152
pragma Warnings (Off, "is not referenced");
 function Lexer_Family_Decl_P_Decl_Type_Name
  
  (Node : Bare_Lexer_Family_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Lexer_Family_Decl  := Bare_Lexer_Family_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_99 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_99 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_99;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 457 '<String at parser.py:1152>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1152
Str := Create_String ("lexer family declaration"); 
--# expr-done 457

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_99;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_99;




            raise;
      end;



   return Property_Result;
end Lexer_Family_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Type_Decl
   --

   




      
   function Type_Decl_F_Traits
     (Node : Bare_Type_Decl) return Bare_Type_Ref_List
   is
      

         Kind : constant Lkt_Type_Decl := Node.Kind;
   begin
         case Kind is
               when Lkt_Function_Type =>
                     return No_Bare_Lkt_Node;
               when Lkt_Generic_Formal_Type_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Instantiated_Generic_Type =>
                     return No_Bare_Lkt_Node;
               when Lkt_Trait_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Enum_Type_Decl =>
                     
         return Node.Enum_Type_Decl_F_Traits;
      
               when Lkt_Struct_Decl =>
                     
         return Node.Struct_Decl_F_Traits;
      
               when Lkt_Class_Decl | Lkt_Enum_Class_Decl =>
                     
         return Node.Basic_Class_Decl_F_Traits;
      
               when Lkt_Enum_Class_Alt_Decl =>
                     return No_Bare_Lkt_Node;
         end case;
   end;

      
   function Type_Decl_F_Syn_Base_Type
     (Node : Bare_Type_Decl) return Bare_Type_Ref
   is
      

         Kind : constant Lkt_Type_Decl := Node.Kind;
   begin
         case Kind is
               when Lkt_Function_Type =>
                     return No_Bare_Lkt_Node;
               when Lkt_Generic_Formal_Type_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Instantiated_Generic_Type =>
                     return No_Bare_Lkt_Node;
               when Lkt_Trait_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Enum_Type_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Struct_Decl =>
                     return No_Bare_Lkt_Node;
               when Lkt_Class_Decl | Lkt_Enum_Class_Decl =>
                     
         return Node.Basic_Class_Decl_F_Syn_Base_Type;
      
               when Lkt_Enum_Class_Alt_Decl =>
                     return No_Bare_Lkt_Node;
         end case;
   end;


   







--# property-start TypeDecl.concrete_matches D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1732
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Concrete_Matches
  
  (Node : Bare_Type_Decl
      ; Other : Internal_Entity_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind other Other

   Property_Result : Boolean;

      

      Cast_Expr : Internal_Entity_Type_Decl;
Cast_Result : Internal_Entity;
Cast_Expr_1 : Internal_Entity_Type_Decl;
Cast_Result_1 : Internal_Entity;
Is_Equiv : Boolean;
Fld : Boolean;
If_Result : Boolean;
Let_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 460 '<Block at ???>' Let_Result None
--# scope-start



--# expr-start 458 '<Eq at parser.py:1737>' Is_Equiv D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1737






Cast_Expr := Ent; 



   
      Cast_Result := Create_Internal_Entity
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);








Cast_Expr_1 := Other; 



   
      Cast_Result_1 := Create_Internal_Entity
        (Node => Cast_Expr_1.Node,
         Info => Cast_Expr_1.Info);


Is_Equiv := Equivalent (Cast_Result, Cast_Result_1); 
--# expr-done 458
if Is_Equiv then
   
   If_Result := True;
else
   --# expr-start 459 '<FieldAccess for is_subtype at parser.py:1737>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1737






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start '[dispatcher]TypeDecl.is_subtype'
Fld := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Is_Subtype (Node => Ent.Node, Potential_Base => Other, E_Info => Ent.Info);
--# end
--# expr-done 459
   If_Result := Fld;
end if;



Let_Result := If_Result; 
--# end
--# expr-done 460

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Concrete_Matches;
--# end

   







--# property-start TypeDecl.matches D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1739
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Matches
  
  (Node : Bare_Type_Decl
      ; Other : Internal_Entity_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind other Other

   Property_Result : Boolean;

      
            procedure Finalizer_Scope_375 with Inline_Always;
            procedure Finalizer_Scope_376 with Inline_Always;

      Var_Expr : Internal_Entity_Instantiated_Generic_Type;
Var_Expr_1 : Internal_Entity_Instantiated_Generic_Type;
Formal : Internal_Entity_Full_Decl;
Actual_2 : Bare_Type_Decl;
Cast_Expr : Internal_Entity_Type_Decl;
Cast_Result : Internal_Entity_Instantiated_Generic_Type;
Cast_Expr_1 : Internal_Entity_Type_Decl;
Cast_Result_1 : Internal_Entity_Instantiated_Generic_Type;
Fld : Bare_Type_Decl;
As_Entity : Internal_Entity_Type_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Type_Decl;
Fld_1 : Bare_Type_Decl;
As_Entity_1 : Internal_Entity_Type_Decl;
Is_Null_1 : Boolean;
If_Result_1 : Internal_Entity_Type_Decl;
Fld_2 : Boolean;
Fld_3 : Bare_Type_Decl_Array_Access;
Index_25 : Integer;
Fld_4 : Internal_Entity_Generic_Decl;
Fld_5 : Internal_Entity_Generic_Formal_Decl_List;
Fld_6 : Bare_Generic_Formal_Decl_List;
Fld_7 : Internal_Entity_Info;
Get_Result : Bare_Full_Decl;
As_Entity_2 : Internal_Entity_Full_Decl;
Is_Null_2 : Boolean;
If_Result_2 : Internal_Entity_Full_Decl;
Fld_8 : Bare_Type_Decl_Array_Access;
Get_Result_1 : Bare_Type_Decl;
As_Entity_3 : Internal_Entity_Type_Decl;
Is_Null_3 : Boolean;
If_Result_3 : Internal_Entity_Type_Decl;
As_Entity_4 : Internal_Entity_Type_Decl;
Is_Null_4 : Boolean;
If_Result_4 : Internal_Entity_Type_Decl;
Fld_9 : Boolean;
Cast_Expr_2 : Bare_Type_Decl;
Cast_Result_2 : Bare_Type_Decl;
Is_Equal : Boolean;
Sym : Symbol_Type;
Fld_10 : Boolean;
If_Result_5 : Boolean;
Let_Result : Boolean;
Quantifier_Result : Boolean;
If_Result_6 : Boolean;
Scope_Result : Boolean;
Fld_11 : Boolean;
Result_Var : Boolean;
Scope_Result_1 : Boolean;
Fld_12 : Boolean;
Result_Var_1 : Boolean;
Let_Result_1 : Boolean;

            procedure Finalizer_Scope_375 is
            begin
                     Dec_Ref (Fld_3);
            end Finalizer_Scope_375;
            procedure Finalizer_Scope_376 is
            begin
                     Dec_Ref (Fld_8);
            end Finalizer_Scope_376;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 484 '<Block at ???>' Let_Result_1 None
--# scope-start



--# expr-start 461 '<Cast to InstantiatedGenericType at parser.py:1744>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1744






Cast_Expr := Ent; 



   if Cast_Expr.Node = null
      or else Cast_Expr.Node.Kind in Lkt_Instantiated_Generic_Type_Range
   then
      
      Cast_Result := Create_Internal_Entity_Instantiated_Generic_Type
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);

   else
         Cast_Result := No_Entity_Instantiated_Generic_Type;
   end if;


--# expr-done 461
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Instantiated_Generic_Type then
   



   --# scope-start


   --# bind e Var_Expr




--# expr-start 462 '<Cast to InstantiatedGenericType at parser.py:1745>' Cast_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1745






Cast_Expr_1 := Other; 



   if Cast_Expr_1.Node = null
      or else Cast_Expr_1.Node.Kind in Lkt_Instantiated_Generic_Type_Range
   then
      
      Cast_Result_1 := Create_Internal_Entity_Instantiated_Generic_Type
        (Node => Cast_Expr_1.Node,
         Info => Cast_Expr_1.Info);

   else
         Cast_Result_1 := No_Entity_Instantiated_Generic_Type;
   end if;


--# expr-done 462
Var_Expr_1 := Cast_Result_1;





if Var_Expr_1 /= No_Entity_Instantiated_Generic_Type then
   



   --# scope-start


   --# bind o Var_Expr_1




--# expr-start 467 '<FieldAccess for matches at parser.py:1753>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1753



--# expr-start 464 '<as_bare_entity at parser.py:1753>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1753



--# expr-start 463 '<FieldAccess for inner_type_decl at parser.py:1753>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1753






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Var_Expr.Node.Instantiated_Generic_Type_Inner_Type_Decl;
--# expr-done 463
Is_Null := Fld = null; 
if Is_Null then
   
   If_Result := No_Entity_Type_Decl;
else
   

As_Entity := (Info => No_Entity_Info, Node => Fld); 
   If_Result := As_Entity;
end if;



--# expr-done 464


   

      if If_Result.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 466 '<as_bare_entity at parser.py:1754>' If_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1754



--# expr-start 465 '<FieldAccess for inner_type_decl at parser.py:1754>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1754






   

      if Var_Expr_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Var_Expr_1.Node.Instantiated_Generic_Type_Inner_Type_Decl;
--# expr-done 465
Is_Null_1 := Fld_1 = null; 
if Is_Null_1 then
   
   If_Result_1 := No_Entity_Type_Decl;
else
   

As_Entity_1 := (Info => No_Entity_Info, Node => Fld_1); 
   If_Result_1 := As_Entity_1;
end if;



--# expr-done 466
--# property-call-start TypeDecl.matches
Fld_2 := Liblktlang.Implementation.Type_Decl_P_Matches (Node => If_Result.Node, Other => If_Result_1, E_Info => If_Result.Info);
--# end
--# expr-done 467
if Fld_2 then
   --# expr-start 481 '<AllQuantifier at parser.py:1758>' Quantifier_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1758








   

   --# expr-start 468 '<FieldAccess for actuals at parser.py:1758>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1758






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Var_Expr.Node.Instantiated_Generic_Type_Actuals;
Inc_Ref (Fld_3);
--# expr-done 468

   
      Quantifier_Result := True;
   

   

      
         Index_25 := 0;

      declare
         
         Collection : constant Bare_Type_Decl_Array_Access := Fld_3;
      begin
         for Item_24 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind actual_1 Item_24
               --# bind i Index_25

            
      --# expr-start 480 '<Let at parser.py:1758>' Let_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1758
--# scope-start
--# expr-start 470 '<FieldAccess for generic_formal_decls at parser.py:1760>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1760



--# expr-start 469 '<FieldAccess for generic_decl at parser.py:1760>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1760






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start InstantiatedGenericType.generic_decl
Fld_4 := Liblktlang.Implementation.Instantiated_Generic_Type_P_Generic_Decl (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 469


   

      if Fld_4.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_5 := Create_Internal_Entity_Generic_Formal_Decl_List (Node => Fld_4.Node.Generic_Decl_F_Generic_Formal_Decls, Info => Fld_4.Info);
--# expr-done 470










   




Fld_6 := Fld_5.Node;


Get_Result := Get (Self, Fld_6, Index_25, True); 
Is_Null_2 := Get_Result = null; 
if Is_Null_2 then
   
   If_Result_2 := No_Entity_Full_Decl;
else
   





   




Fld_7 := Fld_5.Info;

As_Entity_2 := (Info => Fld_7, Node => Get_Result); 
   If_Result_2 := As_Entity_2;
end if;



Formal := If_Result_2; 
--# bind formal Formal
--# expr-start 472 '<at at parser.py:1761>' Get_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1761

--# expr-start 471 '<FieldAccess for actuals at parser.py:1761>' Fld_8 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1761






   

      if Var_Expr_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_8 := Var_Expr_1.Node.Instantiated_Generic_Type_Actuals;
Inc_Ref (Fld_8);
--# expr-done 471


Get_Result_1 := Get (Self, Fld_8, Index_25, True); 
--# expr-done 472
Actual_2 := Get_Result_1; 
--# bind actual_2 Actual_2
--# expr-start 479 '<If at parser.py:1762>' If_Result_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1762



--# expr-start 474 '<FieldAccess for has_annotation at parser.py:1763>' Fld_10 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1763






   

      if Formal.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 473 '<Symbol out at ???>' Sym None
Sym := Precomputed_Symbol (Precomputed_Symbol_Table (Self.Unit.Context.Symbols), Precomputed_Sym_Out); 
--# expr-done 473
--# property-call-start FullDecl.has_annotation
Fld_10 := Liblktlang.Implementation.Full_Decl_P_Has_Annotation (Node => Formal.Node, Name => Sym);
--# end
--# expr-done 474
if Fld_10 then
   --# expr-start 477 '<FieldAccess for matches at parser.py:1768>' Fld_9 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1768



--# expr-start 475 '<as_bare_entity at parser.py:1768>' If_Result_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1768




Is_Null_3 := Item_24 = null; 
if Is_Null_3 then
   
   If_Result_3 := No_Entity_Type_Decl;
else
   

As_Entity_3 := (Info => No_Entity_Info, Node => Item_24); 
   If_Result_3 := As_Entity_3;
end if;



--# expr-done 475


   

      if If_Result_3.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 476 '<as_bare_entity at parser.py:1769>' If_Result_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1769




Is_Null_4 := Actual_2 = null; 
if Is_Null_4 then
   
   If_Result_4 := No_Entity_Type_Decl;
else
   

As_Entity_4 := (Info => No_Entity_Info, Node => Actual_2); 
   If_Result_4 := As_Entity_4;
end if;



--# expr-done 476
--# property-call-start TypeDecl.matches
Fld_9 := Liblktlang.Implementation.Type_Decl_P_Matches (Node => If_Result_3.Node, Other => If_Result_4, E_Info => If_Result_3.Info);
--# end
--# expr-done 477
   If_Result_5 := Fld_9;
else
   --# expr-start 478 '<Eq at parser.py:1774>' Is_Equal D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1774






Cast_Expr_2 := Item_24; 



   
      Cast_Result_2 := Cast_Expr_2;



Is_Equal := Cast_Result_2 = Actual_2; 
--# expr-done 478
   If_Result_5 := Is_Equal;
end if;



--# expr-done 479
Let_Result := If_Result_5; 
--# end
--# expr-done 480
      Quantifier_Result := Let_Result;
   

            
   --# end
      Finalizer_Scope_376;


               exit when not Quantifier_Result;

               Index_25 := Index_25 + 1;
         end loop;
      end;
   

   
   



--# expr-done 481
   If_Result_6 := Quantifier_Result;
else
   
   If_Result_6 := False;
end if;



Scope_Result := If_Result_6;


   --# end
      Finalizer_Scope_375;


   Result_Var := Scope_Result;
else
   --# expr-start 482 '<FieldAccess for concrete_matches at parser.py:1778>' Fld_11 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1778






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start TypeDecl.concrete_matches
Fld_11 := Liblktlang.Implementation.Type_Decl_P_Concrete_Matches (Node => Ent.Node, Other => Other, E_Info => Ent.Info);
--# end
--# expr-done 482
   Result_Var := Fld_11;
end if;



Scope_Result_1 := Result_Var;


   --# end


   Result_Var_1 := Scope_Result_1;
else
   --# expr-start 483 '<FieldAccess for concrete_matches at parser.py:1779>' Fld_12 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1779






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;




--# property-call-start TypeDecl.concrete_matches
Fld_12 := Liblktlang.Implementation.Type_Decl_P_Concrete_Matches (Node => Ent.Node, Other => Other, E_Info => Ent.Info);
--# end
--# expr-done 483
   Result_Var_1 := Fld_12;
end if;



Let_Result_1 := Result_Var_1; 
--# end
--# expr-done 484

         Property_Result := Let_Result_1;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_375;
                     Finalizer_Scope_376;




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Matches;
--# end

   







--# property-start TypeDecl.self_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1782
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Self_Decl
  
  (Node : Bare_Type_Decl
  )

   return Bare_Self_Decl
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
      --# bind self Self

   


   Property_Result : Bare_Self_Decl;

      

      New_Node : Bare_Self_Decl;
Let_Result : Bare_Self_Decl;


      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Type_Decl_Type_Decl_P_Self_Decl,
            Items    => new Mmz_Key_Array (1 ..  1))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Type_Decl,
                                As_Bare_Type_Decl => Self);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Bare_Self_Decl;

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 486 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 485 '<New SelfDecl at parser.py:1784>' New_Node D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1784



if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node := new Root_Node_Record
  (Lkt_Self_Decl);
Initialize
  (Self => New_Node,
   Kind => Lkt_Self_Decl,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node);



--# expr-done 485
Let_Result := New_Node; 
--# end
--# expr-done 486

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Bare_Self_Decl,
                        As_Bare_Self_Decl => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);

         end if;
      end if;



   return Property_Result;
end Type_Decl_P_Self_Decl;
--# end

   







--# property-start TypeDecl.node_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1786
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Node_Decl
  
  (Node : Bare_Type_Decl
  )

   return Bare_Node_Decl
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
      --# bind self Self

   


   Property_Result : Bare_Node_Decl;

      

      New_Node : Bare_Node_Decl;
Let_Result : Bare_Node_Decl;


      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Type_Decl_Type_Decl_P_Node_Decl,
            Items    => new Mmz_Key_Array (1 ..  1))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Type_Decl,
                                As_Bare_Type_Decl => Self);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Bare_Node_Decl;

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 488 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 487 '<New NodeDecl at parser.py:1788>' New_Node D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1788



if Is_Foreign_Strict (Self.Self_Env, Self) then
   Raise_Property_Exception
     (Self,
      Property_Error'Identity,
      "synthetic nodes cannot have foreign lexical envs");
end if;


New_Node := new Root_Node_Record
  (Lkt_Node_Decl);
Initialize
  (Self => New_Node,
   Kind => Lkt_Node_Decl,
   Unit => Self.Unit,

   Token_Start_Index => No_Token_Index,
   Token_End_Index   => No_Token_Index,

   Parent => Self,

   Self_Env => Self.Self_Env);
Register_Destroyable (Self.Unit, New_Node);



--# expr-done 487
Let_Result := New_Node; 
--# end
--# expr-done 488

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Bare_Node_Decl,
                        As_Bare_Node_Decl => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);

         end if;
      end if;



   return Property_Result;
end Type_Decl_P_Node_Decl;
--# end

   







--# property-start '[dispatcher]TypeDecl.type_scope' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Type_Decl_P_Type_Scope
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
      --# bind self Self

   


   Property_Result : Lexical_Env;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Type_Decl (Self.Kind) is
               when Lkt_Enum_Class_Alt_Decl =>
                  --# property-call-start EnumClassAltDecl.type_scope
                  Property_Result := Enum_Class_Alt_Decl_P_Type_Scope
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Function_Type =>
                  --# property-call-start FunctionType.type_scope
                  Property_Result := Function_Type_P_Type_Scope
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Generic_Formal_Type_Decl =>
                  --# property-call-start GenericFormalTypeDecl.type_scope
                  Property_Result := Generic_Formal_Type_Decl_P_Type_Scope
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Instantiated_Generic_Type =>
                  --# property-call-start InstantiatedGenericType.type_scope
                  Property_Result := Instantiated_Generic_Type_P_Type_Scope
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Class_Decl .. Lkt_Trait_Decl =>
                  --# property-call-start NamedTypeDecl.type_scope
                  Property_Result := Named_Type_Decl_P_Type_Scope
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Type_Decl_P_Type_Scope;
--# end

   







--# property-start TypeDecl.get_fun D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1797
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Get_Fun
  
  (Node : Bare_Type_Decl
      ; Name : Symbol_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Fun_Decl
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   

   --# bind name Name

   Property_Result : Internal_Entity_Fun_Decl;

      
            procedure Finalizer_Scope_126 with Inline_Always;

      Fld : Lexical_Env := Empty_Env;
Env_Get_Result : Internal_Entity;
Cast_Result : Internal_Entity_Fun_Decl;
Let_Result : Internal_Entity_Fun_Decl;

            procedure Finalizer_Scope_126 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_126;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 492 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 491 '<Cast to FunDecl at parser.py:1802>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1802





--# expr-start 490 '<EnvGet at parser.py:1802>' Env_Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1802
--# expr-start 489 '<FieldAccess for type_scope at parser.py:1802>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1802






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.type_scope'
Fld := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Type_Scope (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 489



Env_Get_Result := AST_Envs.Get_First (Self => Fld, Key => Thin (Name), Lookup_Kind => To_Lookup_Kind_Type (Recursive), Categories => (Nocat => True)); 
--# expr-done 490



   if Env_Get_Result.Node = null
      or else Env_Get_Result.Node.Kind in Lkt_Fun_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Fun_Decl
        (Node => Env_Get_Result.Node,
         Info => Env_Get_Result.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 491
Let_Result := Cast_Result; 
--# end
--# expr-done 492

         Property_Result := Let_Result;
         
   --# end
      Finalizer_Scope_126;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_126;




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Get_Fun;
--# end

   







--# property-start '[dispatcher]TypeDecl.fields' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Type_Decl_P_Fields
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Component_Decl_Array_Access
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Component_Decl_Array_Access;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Type_Decl (Self.Kind) is
               when Lkt_Enum_Class_Alt_Decl .. Lkt_Generic_Formal_Type_Decl =>
                  --# property-call-start TypeDecl.fields
                  Property_Result := Type_Decl_P_Fields
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Instantiated_Generic_Type =>
                  --# property-call-start InstantiatedGenericType.fields
                  Property_Result := Instantiated_Generic_Type_P_Fields
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Class_Decl .. Lkt_Trait_Decl =>
                  --# property-call-start NamedTypeDecl.fields
                  Property_Result := Named_Type_Decl_P_Fields
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Type_Decl_P_Fields;
--# end

   







--# property-start TypeDecl.ultimate_instantiated_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1811
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Ultimate_Instantiated_Type
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl;

      

      Match_0 : Internal_Entity_Instantiated_Generic_Type;
Match_1 : Internal_Entity_Type_Decl;
Fld : Internal_Entity_Type_Decl;
Fld_1 : Internal_Entity_Type_Decl;
Match_Prefix : Internal_Entity_Type_Decl;
Cast_Result : Internal_Entity_Instantiated_Generic_Type;
Let_Result : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;
Cast_Result_1 : Internal_Entity_Type_Decl;
Let_Result_1 : Internal_Entity_Type_Decl;
Scope_Result_1 : Internal_Entity_Type_Decl;
Match_Result : Internal_Entity_Type_Decl;
Let_Result_2 : Internal_Entity_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 496 '<Block at ???>' Let_Result_2 None
--# scope-start
--# expr-start 495 '<Match at parser.py:1820>' Match_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1820









   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Match_Prefix := Ent; 

case Lkt_Type_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Instantiated_Generic_Type =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Instantiated_Generic_Type
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_0 := Cast_Result; 
--# bind igt Match_0
--# expr-start 494 '<FieldAccess for ultimate_instantiated_type at parser.py:1822>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1822



--# expr-start 493 '<FieldAccess for get_instantiated_type at parser.py:1822>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1822






   

      if Match_0.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start InstantiatedGenericType.get_instantiated_type
Fld := Liblktlang.Implementation.Instantiated_Generic_Type_P_Get_Instantiated_Type (Node => Match_0.Node, E_Info => Match_0.Info);
--# end
--# expr-done 493


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.ultimate_instantiated_type
Fld_1 := Liblktlang.Implementation.Type_Decl_P_Ultimate_Instantiated_Type (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 494
Let_Result := Fld_1; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Enum_Class_Alt_Decl .. Lkt_Generic_Formal_Type_Decl | Lkt_Class_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_1 := Cast_Result_1; 
--# bind _ Match_1

Let_Result_1 := Ent; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
end case;

--# expr-done 495
Let_Result_2 := Match_Result; 
--# end
--# expr-done 496

         Property_Result := Let_Result_2;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Ultimate_Instantiated_Type;
--# end

   







--# property-start TypeDecl.is_class D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1826
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Is_Class
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Boolean;

      

      Match_0 : Internal_Entity_Class_Decl;
Match_1 : Internal_Entity_Generic_Formal_Type_Decl;
Match_2 : Internal_Entity_Type_Decl;
Fld : Internal_Entity_Type_Decl;
Fld_1 : Internal_Entity_Class_Qualifier;
Fld_2 : Boolean;
Match_Prefix : Internal_Entity_Type_Decl;
Cast_Result : Internal_Entity_Class_Decl;
Let_Result : Boolean;
Scope_Result : Boolean;
Cast_Result_1 : Internal_Entity_Generic_Formal_Type_Decl;
Let_Result_1 : Boolean;
Scope_Result_1 : Boolean;
Cast_Result_2 : Internal_Entity_Type_Decl;
Let_Result_2 : Boolean;
Scope_Result_2 : Boolean;
Match_Result : Boolean;
Let_Result_3 : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 501 '<Block at ???>' Let_Result_3 None
--# scope-start
--# expr-start 500 '<Match at parser.py:1831>' Match_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1831






--# expr-start 497 '<FieldAccess for ultimate_instantiated_type at parser.py:1831>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1831






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.ultimate_instantiated_type
Fld := Liblktlang.Implementation.Type_Decl_P_Ultimate_Instantiated_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 497


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Match_Prefix := Fld; 

case Lkt_Type_Decl (Match_Prefix.Node.Kind) is
         when Lkt_Class_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result := Create_Internal_Entity_Class_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_0 := Cast_Result; 
--# bind _ Match_0

Let_Result := True; 
--# end
Scope_Result := Let_Result;


   --# end


            Match_Result := Scope_Result; 
         when Lkt_Generic_Formal_Type_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_1 := Create_Internal_Entity_Generic_Formal_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_1 := Cast_Result_1; 
--# bind gft Match_1
--# expr-start 499 '<FieldAccess for as_bool at parser.py:1833>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1833



--# expr-start 498 '<FieldAccess for has_class at parser.py:1833>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1833






   

      if Match_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Create_Internal_Entity_Class_Qualifier (Node => Match_1.Node.Generic_Formal_Type_Decl_F_Has_Class, Info => Match_1.Info);
--# expr-done 498


   

      if Fld_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]ClassQualifier.as_bool'
Fld_2 := Liblktlang.Implementation.Dispatcher_Class_Qualifier_P_As_Bool (Node => Fld_1.Node);
--# end
--# expr-done 499
Let_Result_1 := Fld_2; 
--# end
Scope_Result_1 := Let_Result_1;


   --# end


            Match_Result := Scope_Result_1; 
         when Lkt_Enum_Class_Alt_Decl .. Lkt_Function_Type | Lkt_Instantiated_Generic_Type | Lkt_Enum_Class_Decl .. Lkt_Trait_Decl =>
            



   --# scope-start



--# scope-start









   
      Cast_Result_2 := Create_Internal_Entity_Type_Decl
        (Node => Match_Prefix.Node,
         Info => Match_Prefix.Info);


Match_2 := Cast_Result_2; 
--# bind _ Match_2

Let_Result_2 := False; 
--# end
Scope_Result_2 := Let_Result_2;


   --# end


            Match_Result := Scope_Result_2; 
end case;

--# expr-done 500
Let_Result_3 := Match_Result; 
--# end
--# expr-done 501

         Property_Result := Let_Result_3;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Is_Class;
--# end

   







--# property-start TypeDecl.implemented_traits D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1837
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Implemented_Traits
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl_Array_Access
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl_Array_Access;

      
            procedure Finalizer_Scope_130 with Inline_Always;
            procedure Finalizer_Scope_383 with Inline_Always;

      Fld : Internal_Entity_Type_Ref_List;
Fld_1 : Bare_Type_Ref_List;
Fld_2 : Internal_Entity_Info;
Item_26 : Internal_Entity_Type_Ref;
Bare_Item_26 : Bare_Type_Ref;
Fld_3 : Internal_Entity_Type_Decl;
As_Entity : Internal_Entity_Type_Ref;
Is_Null : Boolean;
If_Result : Internal_Entity_Type_Ref;
Map_Result : Internal_Entity_Type_Decl_Array_Access;
Fld_4 : Internal_Entity_Type_Decl_Array_Access;
Fld_5 : Internal_Entity_Type_Decl_Array_Access;
Map_Result_1 : Internal_Entity_Type_Decl_Array_Access;
Concat_Result : Internal_Entity_Type_Decl_Array_Access;
Let_Result : Internal_Entity_Type_Decl_Array_Access;

            procedure Finalizer_Scope_130 is
            begin
                     Dec_Ref (Map_Result);
                     Dec_Ref (Fld_4);
                     Dec_Ref (Map_Result_1);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_130;
            procedure Finalizer_Scope_383 is
            begin
                     Dec_Ref (Fld_5);
            end Finalizer_Scope_383;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      begin
         
   --# scope-start

         --# expr-start 508 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 504 '<Map at parser.py:1842>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1842








   

   --# expr-start 502 '<FieldAccess for traits at parser.py:1842>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1842






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Type_Ref_List (Node => Implementation.Type_Decl_F_Traits (Ent.Node), Info => Ent.Info);
--# expr-done 502






   




Fld_1 := Fld.Node;

   
      declare
         Map_Result_Vec : Internal_Entity_Type_Decl_Vectors.Vector;
      begin
   

   

      if Fld_1 = null then
         
      Map_Result := Create_Internal_Entity_Type_Decl_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Type_Ref_List := Fld_1;
      begin
         for Untyped_Item_26 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_26 := Untyped_Item_26; 
                  



Is_Null := Bare_Item_26 = null; 
if Is_Null then
   
   If_Result := No_Entity_Type_Ref;
else
   





   




Fld_2 := Fld.Info;

As_Entity := (Info => Fld_2, Node => Bare_Item_26); 
   If_Result := As_Entity;
end if;



                  Item_26 := If_Result; 

            
   --# scope-start


               --# bind t Item_26

            
         
         
      --# expr-start 503 '<FieldAccess for designated_type at parser.py:1842>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1842






   

      if Item_26.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_3 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Item_26.Node, E_Info => Item_26.Info);
--# end
--# expr-done 503
      

         declare
            Item_To_Append : constant Internal_Entity_Type_Decl := Fld_3;
         begin
            Internal_Entity_Type_Decl_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   
      end if;

   

         Map_Result := Create_Internal_Entity_Type_Decl_Array
           (Items_Count => Natural (Internal_Entity_Type_Decl_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Entity_Type_Decl_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Entity_Type_Decl_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Entity_Type_Decl_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 504
--# expr-start 507 '<Mapcat at parser.py:1843>' Map_Result_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1843








   

   --# expr-start 505 '<FieldAccess for base_types at parser.py:1843>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1843






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.base_types
Fld_4 := Liblktlang.Implementation.Type_Decl_P_Base_Types (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 505

   
      declare
         Map_Result_1_Vec : Internal_Entity_Type_Decl_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Type_Decl_Array_Access := Fld_4;
      begin
         for Item_27 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind bt Item_27

            
         
         
      --# expr-start 506 '<FieldAccess for implemented_traits at parser.py:1844>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1844






   

      if Item_27.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.implemented_traits
Fld_5 := Liblktlang.Implementation.Type_Decl_P_Implemented_Traits (Node => Item_27.Node, E_Info => Item_27.Info);
--# end
--# expr-done 506
      

         for Item_To_Append of
               Fld_5.Items
         loop
            Internal_Entity_Type_Decl_Vectors.Append (Map_Result_1_Vec, Item_To_Append);
         end loop;

   
   
   

            
   --# end
      Finalizer_Scope_383;



         end loop;
      end;
   

   

         Map_Result_1 := Create_Internal_Entity_Type_Decl_Array
           (Items_Count => Natural (Internal_Entity_Type_Decl_Vectors.Length (Map_Result_1_Vec)));
         for I in Map_Result_1.Items'Range loop
            Map_Result_1.Items (I) := Internal_Entity_Type_Decl_Vectors.Get
              (Map_Result_1_Vec,
               I + Internal_Entity_Type_Decl_Vectors.Index_Type'First - Map_Result_1.Items'First);
         end loop;
         Internal_Entity_Type_Decl_Vectors.Destroy (Map_Result_1_Vec);

      end;
   



--# expr-done 507
Concat_Result := Concat (Map_Result, Map_Result_1); 
Let_Result := Concat_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 508

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_130;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_130;
                     Finalizer_Scope_383;




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Implemented_Traits;
--# end

   







--# property-start '[dispatcher]TypeDecl.base_type' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Type_Decl_P_Base_Type
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Ref
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Ref;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Type_Decl (Self.Kind) is
               when Lkt_Enum_Class_Alt_Decl .. Lkt_Generic_Formal_Type_Decl | Lkt_Class_Decl .. Lkt_Trait_Decl =>
                  --# property-call-start TypeDecl.base_type
                  Property_Result := Type_Decl_P_Base_Type
                    (Self
                        , E_Info
                    );
                  --# end
               when Lkt_Instantiated_Generic_Type =>
                  --# property-call-start InstantiatedGenericType.base_type
                  Property_Result := Instantiated_Generic_Type_P_Base_Type
                    (Self
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Type_Decl_P_Base_Type;
--# end

   







--# property-start TypeDecl.base_types D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1857
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Base_Types
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl_Array_Access
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl_Array_Access;

      
            procedure Finalizer_Scope_132 with Inline_Always;
            procedure Finalizer_Scope_384 with Inline_Always;
            procedure Finalizer_Scope_385 with Inline_Always;

      Var_Expr : Internal_Entity_Type_Decl;
Var_Expr_16 : Internal_Entity_Type_Ref;
Fld : Internal_Entity_Type_Ref;
Fld_1 : Internal_Entity_Type_Decl;
Singleton : Internal_Entity_Type_Decl_Array_Access;
Fld_2 : Internal_Entity_Type_Decl_Array_Access;
Concat_Result : Internal_Entity_Type_Decl_Array_Access;
Scope_Result : Internal_Entity_Type_Decl_Array_Access;
Result_Var : Internal_Entity_Type_Decl_Array_Access;
Scope_Result_1 : Internal_Entity_Type_Decl_Array_Access;
Result_Var_1 : Internal_Entity_Type_Decl_Array_Access;
Let_Result : Internal_Entity_Type_Decl_Array_Access;

            procedure Finalizer_Scope_132 is
            begin
                     Dec_Ref (Scope_Result_1);
                     Dec_Ref (Result_Var_1);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_132;
            procedure Finalizer_Scope_384 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
            end Finalizer_Scope_384;
            procedure Finalizer_Scope_385 is
            begin
                     Dec_Ref (Singleton);
                     Dec_Ref (Fld_2);
                     Dec_Ref (Concat_Result);
            end Finalizer_Scope_385;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Type_Decl_Type_Decl_P_Base_Types,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Type_Decl,
                                As_Bare_Type_Decl => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Internal_Entity_Type_Decl_Array_Access;
                     Inc_Ref (Property_Result);

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 512 '<Block at ???>' Let_Result None
--# scope-start



--# expr-start 509 '<FieldAccess for base_type at parser.py:1863>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1863






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.base_type'
Fld := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Base_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 509
Var_Expr_16 := Fld;





if Var_Expr_16 /= No_Entity_Type_Ref then
   



   --# scope-start


   




--# expr-start 510 '<FieldAccess for designated_type at parser.py:1863>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1863






   

      if Var_Expr_16.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Var_Expr_16.Node, E_Info => Var_Expr_16.Info);
--# end
--# expr-done 510
Var_Expr := Fld_1;





if Var_Expr /= No_Entity_Type_Decl then
   



   --# scope-start


   --# bind b Var_Expr


                Singleton := Create_Internal_Entity_Type_Decl_Array (Items_Count => 1);
                Singleton.Items (1) := Var_Expr;
                
            
--# expr-start 511 '<FieldAccess for base_types at parser.py:1864>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1864






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.base_types
Fld_2 := Liblktlang.Implementation.Type_Decl_P_Base_Types (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 511
Concat_Result := Concat (Singleton, Fld_2); 
Scope_Result := Concat_Result;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_385;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Internal_Entity_Type_Decl_Array_Type;
end if;

      Inc_Ref (Result_Var);


Scope_Result_1 := Result_Var;
   Inc_Ref (Scope_Result_1);


   --# end
      Finalizer_Scope_384;


   Result_Var_1 := Scope_Result_1;
else
   
   Result_Var_1 := No_Internal_Entity_Type_Decl_Array_Type;
end if;

      Inc_Ref (Result_Var_1);


Let_Result := Result_Var_1; Inc_Ref (Let_Result);
--# end
--# expr-done 512

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_132;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_132;
                     Finalizer_Scope_384;
                     Finalizer_Scope_385;


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Internal_Entity_Type_Decl_Array_Access,
                        As_Internal_Entity_Type_Decl_Array_Access => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);
               if Mmz_Stored then
                  Inc_Ref (Property_Result);
               end if;

         end if;
      end if;



   return Property_Result;
end Type_Decl_P_Base_Types;
--# end

   







--# property-start TypeDecl.root_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1867
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Root_Type
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl;

      

      Var_Expr : Internal_Entity_Type_Ref;
Var_Expr_17 : Internal_Entity_Type_Decl;
Fld : Internal_Entity_Type_Ref;
Fld_1 : Internal_Entity_Type_Decl;
Fld_2 : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;
Result_Var : Internal_Entity_Type_Decl;
Scope_Result_1 : Internal_Entity_Type_Decl;
Result_Var_1 : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;


      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Type_Decl_Type_Decl_P_Root_Type,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Type_Decl,
                                As_Bare_Type_Decl => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Internal_Entity_Type_Decl;

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 516 '<Block at ???>' Let_Result None
--# scope-start



--# expr-start 513 '<FieldAccess for base_type at parser.py:1873>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1873






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.base_type'
Fld := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Base_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 513
Var_Expr := Fld;





if Var_Expr /= No_Entity_Type_Ref then
   



   --# scope-start


   --# bind bd Var_Expr




--# expr-start 514 '<FieldAccess for designated_type at parser.py:1874>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1874






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 514
Var_Expr_17 := Fld_1;





if Var_Expr_17 /= No_Entity_Type_Decl then
   



   --# scope-start


   

--# expr-start 515 '<FieldAccess for root_type at parser.py:1874>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1874






   

      if Var_Expr_17.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.root_type
Fld_2 := Liblktlang.Implementation.Type_Decl_P_Root_Type (Node => Var_Expr_17.Node, E_Info => Var_Expr_17.Info);
--# end
--# expr-done 515
Scope_Result := Fld_2;


   --# end


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Entity_Type_Decl;
end if;



Scope_Result_1 := Result_Var;


   --# end


   Result_Var_1 := Scope_Result_1;
else
   
   Result_Var_1 := Ent;
end if;



Let_Result := Result_Var_1; 
--# end
--# expr-done 516

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Internal_Entity_Type_Decl,
                        As_Internal_Entity_Type_Decl => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);

         end if;
      end if;



   return Property_Result;
end Type_Decl_P_Root_Type;
--# end

   







--# property-start '[dispatcher]TypeDecl.is_subtype' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Type_Decl_P_Is_Subtype
  
  (Node : Bare_Type_Decl
      ; Potential_Base : Internal_Entity_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
      --# bind self Self

   

   --# bind potential_base Potential_Base

   Property_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;


      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Type_Decl (Self.Kind) is
               when Lkt_Enum_Class_Alt_Decl .. Lkt_Instantiated_Generic_Type | Lkt_Enum_Class_Decl .. Lkt_Trait_Decl =>
                  --# property-call-start TypeDecl.is_subtype
                  Property_Result := Type_Decl_P_Is_Subtype
                    (Self
                        , Potential_Base
                        , E_Info
                    );
                  --# end
               when Lkt_Class_Decl =>
                  --# property-call-start ClassDecl.is_subtype
                  Property_Result := Class_Decl_P_Is_Subtype
                    (Self
                        , Potential_Base
                        , E_Info
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Type_Decl_P_Is_Subtype;
--# end

   







--# property-start TypeDecl.type_base_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1886
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Type_Base_Scope
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Lexical_Env;

      
            procedure Finalizer_Scope_135 with Inline_Always;
            procedure Finalizer_Scope_388 with Inline_Always;
            procedure Finalizer_Scope_389 with Inline_Always;

      Var_Expr_18 : Internal_Entity_Type_Ref;
Fld : Internal_Entity_Type_Decl_Array_Access;
Fld_1 : Lexical_Env := Empty_Env;
Map_Result : Lexical_Env_Array_Access;
Fld_2 : Internal_Entity_Type_Ref;
Fld_3 : Internal_Entity_Type_Decl;
Fld_4 : Lexical_Env := Empty_Env;
Scope_Result : Lexical_Env := Empty_Env;
Result_Var : Lexical_Env := Empty_Env;
Try_Result : Lexical_Env := Empty_Env;
Singleton : Lexical_Env_Array_Access;
Concat_Result : Lexical_Env_Array_Access;
Group_Env : Lexical_Env := Empty_Env;
Let_Result : Lexical_Env := Empty_Env;

            procedure Finalizer_Scope_135 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
                     Dec_Ref (Try_Result);
                     Dec_Ref (Singleton);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Group_Env);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_135;
            procedure Finalizer_Scope_388 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_388;
            procedure Finalizer_Scope_389 is
            begin
                     Dec_Ref (Fld_4);
            end Finalizer_Scope_389;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 525 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 524 '<env_group at parser.py:1904>' Group_Env D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1904
--# expr-start 519 '<Map at parser.py:1894>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1894








   

   --# expr-start 517 '<FieldAccess for implemented_traits at parser.py:1894>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1894






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start TypeDecl.implemented_traits
Fld := Liblktlang.Implementation.Type_Decl_P_Implemented_Traits (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 517

   
      declare
         Map_Result_Vec : Lexical_Env_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Internal_Entity_Type_Decl_Array_Access := Fld;
      begin
         for Item_28 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind td Item_28

            
         
         
      --# expr-start 518 '<FieldAccess for type_scope at parser.py:1895>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1895






   

      if Item_28.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.type_scope'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Type_Scope (Node => Item_28.Node, E_Info => Item_28.Info);
--# end
--# expr-done 518
      

         declare
            Item_To_Append : constant Lexical_Env := Fld_1;
         begin
               Inc_Ref (Item_To_Append);
            Lexical_Env_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end
      Finalizer_Scope_388;



         end loop;
      end;
   

   

         Map_Result := Create_Lexical_Env_Array
           (Items_Count => Natural (Lexical_Env_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Lexical_Env_Vectors.Get
              (Map_Result_Vec,
               I + Lexical_Env_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Lexical_Env_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 519
--# expr-start 523 '<Try at parser.py:69>' Try_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:69



begin
   


--# expr-start 520 '<FieldAccess for base_type at parser.py:1900>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1900






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.base_type'
Fld_2 := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Base_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 520
Var_Expr_18 := Fld_2;





if Var_Expr_18 /= No_Entity_Type_Ref then
   



   --# scope-start


   

--# expr-start 522 '<FieldAccess for type_scope at parser.py:1900>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1900



--# expr-start 521 '<FieldAccess for designated_type at parser.py:1900>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1900






   

      if Var_Expr_18.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeRef.designated_type'
Fld_3 := Liblktlang.Implementation.Dispatcher_Type_Ref_P_Designated_Type (Node => Var_Expr_18.Node, E_Info => Var_Expr_18.Info);
--# end
--# expr-done 521


   

      if Fld_3.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.type_scope'
Fld_4 := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Type_Scope (Node => Fld_3.Node, E_Info => Fld_3.Info);
--# end
--# expr-done 522
Scope_Result := Fld_4;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_389;


   Result_Var := Scope_Result;
else
   
   Result_Var := Empty_Env;
end if;

      Inc_Ref (Result_Var);


   Try_Result := Result_Var;
exception
   when Property_Error =>
      
      Try_Result := Empty_Env;
end;

      Inc_Ref (Try_Result);


--# expr-done 523
                Singleton := Create_Lexical_Env_Array (Items_Count => 1);
                Singleton.Items (1) := Try_Result;
                Inc_Ref (Singleton.Items (1));
            
Concat_Result := Concat (Map_Result, Singleton); 

Group_Env := Group (Concat_Result, No_Metadata); 
--# expr-done 524
Let_Result := Group_Env; Inc_Ref (Let_Result);
--# end
--# expr-done 525

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_135;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_135;
                     Finalizer_Scope_388;
                     Finalizer_Scope_389;




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Type_Base_Scope;
--# end

   







--# property-start TypeDecl.call_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1907
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Call_Scope
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Lexical_Env;

      
            procedure Finalizer_Scope_136 with Inline_Always;

      Fld : Lexical_Env := Empty_Env;

            procedure Finalizer_Scope_136 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_136;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 526 '<FieldAccess for type_scope at parser.py:1907>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1907






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.type_scope'
Fld := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Type_Scope (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 526

         Property_Result := Fld;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_136;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_136;




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Call_Scope;
--# end

   







--# property-start TypeDecl.is_generic D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1909
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Is_Generic
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Boolean;

      

      Is_A : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 527 '<IsA InstantiatedGenericType at parser.py:1910>' Is_A D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1910

Is_A := Ent.Node /= null 
and then Ent.Node.Kind in Lkt_Instantiated_Generic_Type_Range; 
--# expr-done 527

         Property_Result := Is_A;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Is_Generic;
--# end

   







--# property-start TypeDecl.array_element_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1914
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Array_Element_Type
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_390 with Inline_Always;

      Var_Expr : Internal_Entity_Instantiated_Generic_Type;
Cast_Expr : Internal_Entity_Type_Decl;
Cast_Result : Internal_Entity_Instantiated_Generic_Type;
Fld : Internal_Entity_Type_Decl_Array_Access;
Get_Result : Internal_Entity_Type_Decl;
Fld_1 : Bare_Type_Decl;
Fld_2 : Internal_Entity_Named_Type_Decl;
Fld_3 : Bare_Decl;
Cast_Result_1 : Bare_Decl;
Is_Equal : Boolean;
If_Result : Internal_Entity_Type_Decl;
Scope_Result : Internal_Entity_Type_Decl;
Result_Var : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_390 is
            begin
                     Dec_Ref (Fld);
            end Finalizer_Scope_390;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 536 '<Block at ???>' Let_Result None
--# scope-start



--# expr-start 528 '<Cast to InstantiatedGenericType at parser.py:1922>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1922






Cast_Expr := Ent; 



   if Cast_Expr.Node = null
      or else Cast_Expr.Node.Kind in Lkt_Instantiated_Generic_Type_Range
   then
      
      Cast_Result := Create_Internal_Entity_Instantiated_Generic_Type
        (Node => Cast_Expr.Node,
         Info => Cast_Expr.Info);

   else
         Cast_Result := No_Entity_Instantiated_Generic_Type;
   end if;


--# expr-done 528
Var_Expr := Cast_Result;





if Var_Expr /= No_Entity_Instantiated_Generic_Type then
   



   --# scope-start


   --# bind array Var_Expr

--# expr-start 535 '<If at parser.py:1923>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1923



--# expr-start 532 '<Eq at parser.py:1924>' Is_Equal D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1924





--# expr-start 529 '<FieldAccess for get_inner_type at parser.py:1924>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1924






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start InstantiatedGenericType.get_inner_type
Fld_1 := Liblktlang.Implementation.Instantiated_Generic_Type_P_Get_Inner_Type (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 529



   
      Cast_Result_1 := Fld_1;


--# expr-start 531 '<FieldAccess for assert_bare at parser.py:1924>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1924



--# expr-start 530 '<FieldAccess for array_type at parser.py:1924>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1924






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start LktNode.array_type
Fld_2 := Liblktlang.Implementation.Lkt_Node_P_Array_Type (Node => Ent.Node);
--# end
--# expr-done 530


   

      if Fld_2.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start Decl.assert_bare
Fld_3 := Liblktlang.Implementation.Decl_P_Assert_Bare (Node => Fld_2.Node, E_Info => Fld_2.Info);
--# end
--# expr-done 531
Is_Equal := Cast_Result_1 = Fld_3; 
--# expr-done 532
if Is_Equal then
   --# expr-start 534 '<at at parser.py:1925>' Get_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1925

--# expr-start 533 '<FieldAccess for get_actuals at parser.py:1925>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1925






   

      if Var_Expr.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start InstantiatedGenericType.get_actuals
Fld := Liblktlang.Implementation.Instantiated_Generic_Type_P_Get_Actuals (Node => Var_Expr.Node, E_Info => Var_Expr.Info);
--# end
--# expr-done 533


Get_Result := Get (Self, Fld, 0, True); 
--# expr-done 534
   If_Result := Get_Result;
else
   
   If_Result := No_Entity_Type_Decl;
end if;



--# expr-done 535
Scope_Result := If_Result;


   --# end
      Finalizer_Scope_390;


   Result_Var := Scope_Result;
else
   
   Result_Var := No_Entity_Type_Decl;
end if;



Let_Result := Result_Var; 
--# end
--# expr-done 536

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_390;




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Array_Element_Type;
--# end

   







--# property-start TypeDecl.fields D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1804
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Fields
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Component_Decl_Array_Access
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
      --# bind self Self

   


   Property_Result : Internal_Entity_Component_Decl_Array_Access;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := No_Internal_Entity_Component_Decl_Array_Type;
            Inc_Ref (Property_Result);
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Fields;
--# end

   







--# property-start TypeDecl.base_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1850
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Base_Type
  
  (Node : Bare_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Ref
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
     Ent : Internal_Entity_Type_Decl :=
       Internal_Entity_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Ref;

      

      Fld : Internal_Entity_Type_Ref;
Let_Result : Internal_Entity_Type_Ref;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 538 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 537 '<FieldAccess for syn_base_type at parser.py:1855>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1855






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Type_Ref (Node => Implementation.Type_Decl_F_Syn_Base_Type (Ent.Node), Info => Ent.Info);
--# expr-done 537
Let_Result := Fld; 
--# end
--# expr-done 538

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Base_Type;
--# end

   







--# property-start TypeDecl.is_subtype D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1878
pragma Warnings (Off, "is not referenced");
 function Type_Decl_P_Is_Subtype
  
  (Node : Bare_Type_Decl
      ; Potential_Base : Internal_Entity_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Boolean
is
   Self : Bare_Type_Decl  := Bare_Type_Decl (Node);
      --# bind self Self

   

   --# bind potential_base Potential_Base

   Property_Result : Boolean;

      

      Let_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 539 '<Block at ???>' Let_Result None
--# scope-start

Let_Result := False; 
--# end
--# expr-done 539

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Type_Decl_P_Is_Subtype;
--# end


   
function Type_Decl_P_Type_Base_Scope
  (E : Entity
  ) return Lexical_Env
is
   

      E_Info : Internal_Entity_Info :=
         Shed_Rebindings (E.Info, Children_Env (E.Node));
      
begin
   
      if E.Node /= null
         and then E.Node.Kind not in Lkt_Type_Decl
      then
         Raise_Property_Exception
           (E.Node,
            Property_Error'Identity,
            "mismatching node type");
      end if;

   declare
      Result : constant Lexical_Env :=
         Type_Decl_P_Type_Base_Scope (E.Node, E_Info);
   begin
         return Result;
   end;
end;


   


      

   --
   --  Primitives for Bare_Enum_Class_Alt_Decl
   --

   



      
      procedure Initialize_Fields_For_Enum_Class_Alt_Decl
        (Self : Bare_Enum_Class_Alt_Decl
         ; Enum_Class_Alt_Decl_F_Syn_Name : Bare_Def_Id
        ) is
      begin

            Self.Enum_Class_Alt_Decl_F_Syn_Name := Enum_Class_Alt_Decl_F_Syn_Name;
         

      end Initialize_Fields_For_Enum_Class_Alt_Decl;


   







--# property-start EnumClassAltDecl.parent_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2371
pragma Warnings (Off, "is not referenced");
 function Enum_Class_Alt_Decl_P_Parent_Type
  
  (Node : Bare_Enum_Class_Alt_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Enum_Class_Alt_Decl  := Bare_Enum_Class_Alt_Decl (Node);
     Ent : Internal_Entity_Enum_Class_Alt_Decl :=
       Internal_Entity_Enum_Class_Alt_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl;

      

      Fld : Internal_Entity;
Fld_1 : Internal_Entity;
Cast_Result : Internal_Entity_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 542 '<Cast to TypeDecl at parser.py:2371>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2371





--# expr-start 541 '<parent at parser.py:2371>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2371



--# expr-start 540 '<parent at parser.py:2371>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2371






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 540


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Liblktlang.Implementation.Parent (Node => Fld.Node, E_Info => Fld.Info);
--# expr-done 541



   if Fld_1.Node = null
      or else Fld_1.Node.Kind in Lkt_Type_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Type_Decl
        (Node => Fld_1.Node,
         Info => Fld_1.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 542

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Enum_Class_Alt_Decl_P_Parent_Type;
--# end

   







--# property-start EnumClassAltDecl.type_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2373
pragma Warnings (Off, "is not referenced");
 function Enum_Class_Alt_Decl_P_Type_Scope
  
  (Node : Bare_Enum_Class_Alt_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Enum_Class_Alt_Decl  := Bare_Enum_Class_Alt_Decl (Node);
     Ent : Internal_Entity_Enum_Class_Alt_Decl :=
       Internal_Entity_Enum_Class_Alt_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Lexical_Env;

      
            procedure Finalizer_Scope_182 with Inline_Always;

      Fld : Internal_Entity_Type_Decl;
Fld_1 : Lexical_Env := Empty_Env;

            procedure Finalizer_Scope_182 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_182;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 544 '<FieldAccess for type_scope at parser.py:2373>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2373



--# expr-start 543 '<FieldAccess for parent_type at parser.py:2373>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2373






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start EnumClassAltDecl.parent_type
Fld := Liblktlang.Implementation.Enum_Class_Alt_Decl_P_Parent_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 543


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.type_scope'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Type_Scope (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 544

         Property_Result := Fld_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_182;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_182;




            raise;
      end;



   return Property_Result;
end Enum_Class_Alt_Decl_P_Type_Scope;
--# end

   







--# property-start EnumClassAltDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2378
pragma Warnings (Off, "is not referenced");
 function Enum_Class_Alt_Decl_P_Decl_Type_Name
  
  (Node : Bare_Enum_Class_Alt_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Enum_Class_Alt_Decl  := Bare_Enum_Class_Alt_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_183 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_183 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_183;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 545 '<String at parser.py:2378>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2378
Str := Create_String ("enum class alt declaration"); 
--# expr-done 545

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_183;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_183;




            raise;
      end;



   return Property_Result;
end Enum_Class_Alt_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Function_Type
   --

   



      
      procedure Initialize_Fields_For_Function_Type
        (Self : Bare_Function_Type
        ) is
      begin

         
      Self.Function_Type_Args := No_Bare_Type_Decl_Array_Type;
      Self.Function_Type_Return_Type := No_Bare_Lkt_Node;

      end Initialize_Fields_For_Function_Type;


   







--# property-start FunctionType.full_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1942
pragma Warnings (Off, "is not referenced");
 function Function_Type_P_Full_Name
  
  (Node : Bare_Function_Type
  )

   return String_Type
is
   Self : Bare_Function_Type  := Bare_Function_Type (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_139 with Inline_Always;
            procedure Finalizer_Scope_393 with Inline_Always;

      Str : String_Type;
Str_1 : String_Type;
Fld : Bare_Type_Decl_Array_Access;
Fld_1 : String_Type;
Map_Result : String_Type_Array_Access;
Join_Result : String_Type;
Concat_Result : String_Type;
Str_2 : String_Type;
Concat_Result_1 : String_Type;
Fld_2 : Bare_Type_Decl;
Fld_3 : String_Type;
Concat_Result_2 : String_Type;

            procedure Finalizer_Scope_139 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Str_1);
                     Dec_Ref (Fld);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Join_Result);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Str_2);
                     Dec_Ref (Concat_Result_1);
                     Dec_Ref (Fld_3);
                     Dec_Ref (Concat_Result_2);
            end Finalizer_Scope_139;
            procedure Finalizer_Scope_393 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_393;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 546 '<String at parser.py:1943>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1943
Str := Create_String ("("); 
--# expr-done 546
--# expr-start 551 '<Join at parser.py:1944>' Join_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1944
--# expr-start 547 '<String at parser.py:1944>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1944
Str_1 := Create_String (", "); 
--# expr-done 547
--# expr-start 550 '<Map at parser.py:1944>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1944








   

   --# expr-start 548 '<FieldAccess for args at parser.py:1944>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1944






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Self.Function_Type_Args;
Inc_Ref (Fld);
--# expr-done 548

   
      declare
         Map_Result_Vec : String_Type_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Bare_Type_Decl_Array_Access := Fld;
      begin
         for Item_29 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind t Item_29

            
         
         
      --# expr-start 549 '<FieldAccess for full_name at parser.py:1944>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1944






   

      if Item_29 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.full_name'
Fld_1 := Liblktlang.Implementation.Dispatcher_Decl_P_Full_Name (Node => Item_29);
--# end
--# expr-done 549
      

         declare
            Item_To_Append : constant String_Type := Fld_1;
         begin
               Inc_Ref (Item_To_Append);
            String_Type_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end
      Finalizer_Scope_393;



         end loop;
      end;
   

   

         Map_Result := Create_String_Type_Array
           (Items_Count => Natural (String_Type_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := String_Type_Vectors.Get
              (Map_Result_Vec,
               I + String_Type_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         String_Type_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 550
Join_Result := Join_Strings (Str_1, Map_Result); 
--# expr-done 551
Concat_Result := Concat_String (Str, Join_Result); 
--# expr-start 552 '<String at parser.py:1945>' Str_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1945
Str_2 := Create_String (") -> "); 
--# expr-done 552
Concat_Result_1 := Concat_String (Concat_Result, Str_2); 
--# expr-start 554 '<FieldAccess for full_name at parser.py:1945>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1945



--# expr-start 553 '<FieldAccess for return_type at parser.py:1945>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1945






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Self.Function_Type_Return_Type;
--# expr-done 553


   

      if Fld_2 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.full_name'
Fld_3 := Liblktlang.Implementation.Dispatcher_Decl_P_Full_Name (Node => Fld_2);
--# end
--# expr-done 554
Concat_Result_2 := Concat_String (Concat_Result_1, Fld_3); 

         Property_Result := Concat_Result_2;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_139;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_139;
                     Finalizer_Scope_393;




            raise;
      end;



   return Property_Result;
end Function_Type_P_Full_Name;
--# end

   







--# property-start FunctionType.type_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1948
pragma Warnings (Off, "is not referenced");
 function Function_Type_P_Type_Scope
  
  (Node : Bare_Function_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Function_Type  := Bare_Function_Type (Node);
      --# bind self Self

   


   Property_Result : Lexical_Env;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := Empty_Env;
            Inc_Ref (Property_Result);
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Function_Type_P_Type_Scope;
--# end

   







--# property-start FunctionType.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1950
pragma Warnings (Off, "is not referenced");
 function Function_Type_P_Decl_Type_Name
  
  (Node : Bare_Function_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Function_Type  := Bare_Function_Type (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_141 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_141 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_141;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 555 '<String at parser.py:1950>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1950
Str := Create_String ("function type"); 
--# expr-done 555

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_141;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_141;




            raise;
      end;



   return Property_Result;
end Function_Type_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Generic_Formal_Type_Decl
   --

   



      
      procedure Initialize_Fields_For_Generic_Formal_Type_Decl
        (Self : Bare_Generic_Formal_Type_Decl
         ; Generic_Formal_Type_Decl_F_Has_Class : Bare_Class_Qualifier
         ; Generic_Formal_Type_Decl_F_Syn_Name : Bare_Def_Id
        ) is
      begin

            Self.Generic_Formal_Type_Decl_F_Has_Class := Generic_Formal_Type_Decl_F_Has_Class;
            Self.Generic_Formal_Type_Decl_F_Syn_Name := Generic_Formal_Type_Decl_F_Syn_Name;
         

      end Initialize_Fields_For_Generic_Formal_Type_Decl;

      
   function Generic_Formal_Type_Decl_F_Has_Class
     (Node : Bare_Generic_Formal_Type_Decl) return Bare_Class_Qualifier
   is
      

   begin
         
         return Node.Generic_Formal_Type_Decl_F_Has_Class;
      
   end;


   







--# property-start GenericFormalTypeDecl.type_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1969
pragma Warnings (Off, "is not referenced");
 function Generic_Formal_Type_Decl_P_Type_Scope
  
  (Node : Bare_Generic_Formal_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Generic_Formal_Type_Decl  := Bare_Generic_Formal_Type_Decl (Node);
      --# bind self Self

   


   Property_Result : Lexical_Env;

      

      



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         

         Property_Result := Empty_Env;
            Inc_Ref (Property_Result);
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Generic_Formal_Type_Decl_P_Type_Scope;
--# end

   







--# property-start GenericFormalTypeDecl.generic_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1971
pragma Warnings (Off, "is not referenced");
 function Generic_Formal_Type_Decl_P_Generic_Decl
  
  (Node : Bare_Generic_Formal_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Generic_Formal_Type_Decl  := Bare_Generic_Formal_Type_Decl (Node);
     Ent : Internal_Entity_Generic_Formal_Type_Decl :=
       Internal_Entity_Generic_Formal_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Fld : Internal_Entity;
Fld_1 : Internal_Entity;
Fld_2 : Internal_Entity;
Cast_Result : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 559 '<Cast to GenericDecl at parser.py:1972>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1972





--# expr-start 558 '<parent at parser.py:1972>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1972



--# expr-start 557 '<parent at parser.py:1972>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1972



--# expr-start 556 '<parent at parser.py:1972>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1972






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Liblktlang.Implementation.Parent (Node => Ent.Node, E_Info => Ent.Info);
--# expr-done 556


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Liblktlang.Implementation.Parent (Node => Fld.Node, E_Info => Fld.Info);
--# expr-done 557


   

      if Fld_1.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Liblktlang.Implementation.Parent (Node => Fld_1.Node, E_Info => Fld_1.Info);
--# expr-done 558



   if Fld_2.Node = null
      or else Fld_2.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => Fld_2.Node,
         Info => Fld_2.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 559

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Generic_Formal_Type_Decl_P_Generic_Decl;
--# end

   







--# property-start GenericFormalTypeDecl.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1975
pragma Warnings (Off, "is not referenced");
 function Generic_Formal_Type_Decl_P_Decl_Type_Name
  
  (Node : Bare_Generic_Formal_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Generic_Formal_Type_Decl  := Bare_Generic_Formal_Type_Decl (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_147 with Inline_Always;

      Str : String_Type;
Str_1 : String_Type;
Fld : Bare_Class_Qualifier;
Fld_1 : Boolean;
If_Result : String_Type;
Let_Result : String_Type;

            procedure Finalizer_Scope_147 is
            begin
                     Dec_Ref (Str);
                     Dec_Ref (Str_1);
                     Dec_Ref (If_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_147;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 565 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 564 '<If at parser.py:1977>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1977



--# expr-start 561 '<FieldAccess for as_bool at parser.py:1978>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1978



--# expr-start 560 '<FieldAccess for has_class at parser.py:1978>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1978






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Self.Generic_Formal_Type_Decl_F_Has_Class;
--# expr-done 560


   

      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]ClassQualifier.as_bool'
Fld_1 := Liblktlang.Implementation.Dispatcher_Class_Qualifier_P_As_Bool (Node => Fld);
--# end
--# expr-done 561
if Fld_1 then
   --# expr-start 562 '<String at parser.py:1979>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1979
Str := Create_String ("generic formal class type declaration"); 
--# expr-done 562
   If_Result := Str;
else
   --# expr-start 563 '<String at parser.py:1980>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1980
Str_1 := Create_String ("generic formal type declaration"); 
--# expr-done 563
   If_Result := Str_1;
end if;

      Inc_Ref (If_Result);


--# expr-done 564
Let_Result := If_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 565

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_147;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_147;




            raise;
      end;



   return Property_Result;
end Generic_Formal_Type_Decl_P_Decl_Type_Name;
--# end



   


      

   --
   --  Primitives for Bare_Instantiated_Generic_Type
   --

   



      
      procedure Initialize_Fields_For_Instantiated_Generic_Type
        (Self : Bare_Instantiated_Generic_Type
        ) is
      begin

         
      Self.Instantiated_Generic_Type_Inner_Type_Decl := No_Bare_Lkt_Node;
      Self.Instantiated_Generic_Type_Actuals := No_Bare_Type_Decl_Array_Type;

      end Initialize_Fields_For_Instantiated_Generic_Type;


   







--# property-start InstantiatedGenericType.name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2158
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Name
  
  (Node : Bare_Instantiated_Generic_Type
  )

   return Symbol_Type
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
      --# bind self Self

   


   Property_Result : Symbol_Type;

      

      Fld : Bare_Type_Decl;
Fld_1 : Symbol_Type;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 567 '<FieldAccess for name at parser.py:2158>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2158



--# expr-start 566 '<FieldAccess for inner_type_decl at parser.py:2158>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2158






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Self.Instantiated_Generic_Type_Inner_Type_Decl;
--# expr-done 566


   

      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld_1 := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Fld);
--# end
--# expr-done 567

         Property_Result := Fld_1;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Name;
--# end

   







--# property-start InstantiatedGenericType.full_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2160
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Full_Name
  
  (Node : Bare_Instantiated_Generic_Type
  )

   return String_Type
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_161 with Inline_Always;
            procedure Finalizer_Scope_394 with Inline_Always;

      Fld : Symbol_Type;
Fld_1 : String_Type;
Str : String_Type;
Concat_Result : String_Type;
Str_1 : String_Type;
Fld_2 : Bare_Type_Decl_Array_Access;
Fld_3 : String_Type;
Map_Result : String_Type_Array_Access;
Join_Result : String_Type;
Concat_Result_1 : String_Type;
Str_2 : String_Type;
Concat_Result_2 : String_Type;

            procedure Finalizer_Scope_161 is
            begin
                     Dec_Ref (Fld_1);
                     Dec_Ref (Str);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Str_1);
                     Dec_Ref (Fld_2);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Join_Result);
                     Dec_Ref (Concat_Result_1);
                     Dec_Ref (Str_2);
                     Dec_Ref (Concat_Result_2);
            end Finalizer_Scope_161;
            procedure Finalizer_Scope_394 is
            begin
                     Dec_Ref (Fld_3);
            end Finalizer_Scope_394;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 569 '<FieldAccess for image at parser.py:2161>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2161



--# expr-start 568 '<FieldAccess for name at parser.py:2161>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2161






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.name'
Fld := Liblktlang.Implementation.Dispatcher_Decl_P_Name (Node => Self);
--# end
--# expr-done 568


   




Fld_1 := Liblktlang.Implementation.Image (Self => Fld);
--# expr-done 569
--# expr-start 570 '<String at parser.py:2161>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2161
Str := Create_String ("["); 
--# expr-done 570
Concat_Result := Concat_String (Fld_1, Str); 
--# expr-start 575 '<Join at parser.py:2162>' Join_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2162
--# expr-start 571 '<String at parser.py:2162>' Str_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2162
Str_1 := Create_String (", "); 
--# expr-done 571
--# expr-start 574 '<Map at parser.py:2162>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2162








   

   --# expr-start 572 '<FieldAccess for actuals at parser.py:2162>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2162






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_2 := Self.Instantiated_Generic_Type_Actuals;
Inc_Ref (Fld_2);
--# expr-done 572

   
      declare
         Map_Result_Vec : String_Type_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Bare_Type_Decl_Array_Access := Fld_2;
      begin
         for Item_30 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind t Item_30

            
         
         
      --# expr-start 573 '<FieldAccess for full_name at parser.py:2162>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2162






   

      if Item_30 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.full_name'
Fld_3 := Liblktlang.Implementation.Dispatcher_Decl_P_Full_Name (Node => Item_30);
--# end
--# expr-done 573
      

         declare
            Item_To_Append : constant String_Type := Fld_3;
         begin
               Inc_Ref (Item_To_Append);
            String_Type_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end
      Finalizer_Scope_394;



         end loop;
      end;
   

   

         Map_Result := Create_String_Type_Array
           (Items_Count => Natural (String_Type_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := String_Type_Vectors.Get
              (Map_Result_Vec,
               I + String_Type_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         String_Type_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 574
Join_Result := Join_Strings (Str_1, Map_Result); 
--# expr-done 575
Concat_Result_1 := Concat_String (Concat_Result, Join_Result); 
--# expr-start 576 '<String at parser.py:2163>' Str_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2163
Str_2 := Create_String ("]"); 
--# expr-done 576
Concat_Result_2 := Concat_String (Concat_Result_1, Str_2); 

         Property_Result := Concat_Result_2;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_161;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_161;
                     Finalizer_Scope_394;




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Full_Name;
--# end

   







--# property-start InstantiatedGenericType.generic_decl D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2166
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Generic_Decl
  
  (Node : Bare_Instantiated_Generic_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Generic_Decl
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
     Ent : Internal_Entity_Instantiated_Generic_Type :=
       Internal_Entity_Instantiated_Generic_Type'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Generic_Decl;

      

      Fld : Bare_Type_Decl;
Fld_1 : Bare_Lkt_Node;
As_Entity : Internal_Entity;
Is_Null : Boolean;
If_Result : Internal_Entity;
Cast_Result : Internal_Entity_Generic_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 580 '<Cast to GenericDecl at parser.py:2167>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2167





--# expr-start 579 '<as_entity at parser.py:2167>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2167



--# expr-start 578 '<parent at parser.py:2167>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2167



--# expr-start 577 '<FieldAccess for inner_type_decl at parser.py:2167>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2167






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Ent.Node.Instantiated_Generic_Type_Inner_Type_Decl;
--# expr-done 577


   

      if Fld = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Fld.Parent; 
--# expr-done 578
Is_Null := Fld_1 = null; 
if Is_Null then
   
   If_Result := No_Entity;
else
   

As_Entity := (Info => E_Info, Node => Fld_1); 
   If_Result := As_Entity;
end if;



--# expr-done 579



   if If_Result.Node = null
      or else If_Result.Node.Kind in Lkt_Generic_Decl_Range
   then
      
      Cast_Result := Create_Internal_Entity_Generic_Decl
        (Node => If_Result.Node,
         Info => If_Result.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 580

         Property_Result := Cast_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Generic_Decl;
--# end

   







--# property-start InstantiatedGenericType.decl_type_name D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2170
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Decl_Type_Name
  
  (Node : Bare_Instantiated_Generic_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return String_Type
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
      --# bind self Self

   


   Property_Result : String_Type;

      
            procedure Finalizer_Scope_163 with Inline_Always;

      Str : String_Type;

            procedure Finalizer_Scope_163 is
            begin
                     Dec_Ref (Str);
            end Finalizer_Scope_163;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 581 '<String at parser.py:2170>' Str D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2170
Str := Create_String ("instantiated generic type"); 
--# expr-done 581

         Property_Result := Str;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_163;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_163;




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Decl_Type_Name;
--# end

   







--# property-start InstantiatedGenericType.base_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2172
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Base_Type
  
  (Node : Bare_Instantiated_Generic_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Ref
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
     Ent : Internal_Entity_Instantiated_Generic_Type :=
       Internal_Entity_Instantiated_Generic_Type'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Ref;

      

      Fld : Internal_Entity_Type_Decl;
Fld_1 : Internal_Entity_Type_Ref;
Let_Result : Internal_Entity_Type_Ref;



begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 584 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 583 '<FieldAccess for base_type at parser.py:2174>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2174



--# expr-start 582 '<FieldAccess for get_instantiated_type at parser.py:2174>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2174






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start InstantiatedGenericType.get_instantiated_type
Fld := Liblktlang.Implementation.Instantiated_Generic_Type_P_Get_Instantiated_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 582


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.base_type'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Base_Type (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 583
Let_Result := Fld_1; 
--# end
--# expr-done 584

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Base_Type;
--# end

   







--# property-start InstantiatedGenericType.get_inner_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2176
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Get_Inner_Type
  
  (Node : Bare_Instantiated_Generic_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Bare_Type_Decl
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
     Ent : Internal_Entity_Instantiated_Generic_Type :=
       Internal_Entity_Instantiated_Generic_Type'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Bare_Type_Decl;

      

      Fld : Bare_Type_Decl;
Let_Result : Bare_Type_Decl;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 586 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 585 '<FieldAccess for inner_type_decl at parser.py:2181>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2181






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Ent.Node.Instantiated_Generic_Type_Inner_Type_Decl;
--# expr-done 585
Let_Result := Fld; 
--# end
--# expr-done 586

         Property_Result := Let_Result;
         
   --# end


      exception
         when Exc : Property_Error =>




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Get_Inner_Type;
--# end

   







--# property-start InstantiatedGenericType.get_actuals D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2183
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Get_Actuals
  
  (Node : Bare_Instantiated_Generic_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl_Array_Access
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
     Ent : Internal_Entity_Instantiated_Generic_Type :=
       Internal_Entity_Instantiated_Generic_Type'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl_Array_Access;

      
            procedure Finalizer_Scope_166 with Inline_Always;

      Fld : Bare_Type_Decl_Array_Access;
As_Entity : Internal_Entity_Type_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Type_Decl;
Map_Result : Internal_Entity_Type_Decl_Array_Access;
Let_Result : Internal_Entity_Type_Decl_Array_Access;

            procedure Finalizer_Scope_166 is
            begin
                     Dec_Ref (Fld);
                     Dec_Ref (Map_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_166;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 590 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 589 '<Map at parser.py:2189>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2189








   

   --# expr-start 587 '<FieldAccess for actuals at parser.py:2189>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2189






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Ent.Node.Instantiated_Generic_Type_Actuals;
Inc_Ref (Fld);
--# expr-done 587

   
      declare
         Map_Result_Vec : Internal_Entity_Type_Decl_Vectors.Vector;
      begin
   

   

      

      declare
         
         Collection : constant Bare_Type_Decl_Array_Access := Fld;
      begin
         for Item_31 of
               Collection.Items
         loop

            
   --# scope-start


               --# bind a Item_31

            
         
         
      --# expr-start 588 '<as_entity at parser.py:2189>' If_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2189




Is_Null := Item_31 = null; 
if Is_Null then
   
   If_Result := No_Entity_Type_Decl;
else
   

As_Entity := (Info => E_Info, Node => Item_31); 
   If_Result := As_Entity;
end if;



--# expr-done 588
      

         declare
            Item_To_Append : constant Internal_Entity_Type_Decl := If_Result;
         begin
            Internal_Entity_Type_Decl_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
   

            
   --# end



         end loop;
      end;
   

   

         Map_Result := Create_Internal_Entity_Type_Decl_Array
           (Items_Count => Natural (Internal_Entity_Type_Decl_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Entity_Type_Decl_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Entity_Type_Decl_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Entity_Type_Decl_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 589
Let_Result := Map_Result; Inc_Ref (Let_Result);
--# end
--# expr-done 590

         Property_Result := Let_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_166;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_166;




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Get_Actuals;
--# end

   







--# property-start InstantiatedGenericType.get_instantiated_type D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2191
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Get_Instantiated_Type
  
  (Node : Bare_Instantiated_Generic_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Type_Decl
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
     Ent : Internal_Entity_Instantiated_Generic_Type :=
       Internal_Entity_Instantiated_Generic_Type'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Type_Decl;

      
            procedure Finalizer_Scope_167 with Inline_Always;

      Fld : Bare_Type_Decl;
Fld_1 : Internal_Entity_Info;
Fld_2 : Env_Rebindings;
Fld_3 : Bare_Type_Decl;
Fld_4 : Bare_Lkt_Node;
Fld_5 : Lexical_Env := Empty_Env;
Fld_6 : Internal_Entity_Generic_Decl;
Fld_7 : Bare_Type_Decl_Array_Access;
Fld_8 : Lexical_Env := Empty_Env;
Rebinding : Env_Rebindings;
New_Struct : Internal_Entity_Info;
New_Struct_1 : Internal_Entity_Type_Decl;
Let_Result : Internal_Entity_Type_Decl;

            procedure Finalizer_Scope_167 is
            begin
                     Dec_Ref (Fld_5);
                     Dec_Ref (Fld_7);
                     Dec_Ref (Fld_8);
            end Finalizer_Scope_167;

      
      Mmz_Handle : Memoization_Handle;
      Mmz_Val    : Mmz_Value;

      Mmz_Stored : Boolean;
      --  Whether the memoization couple was actually stored. Used to determine
      --  whether to inc-ref the memoized value.

      function Create_Mmz_Key return Mmz_Key;
      --  Create a memoization key for this property call and return it

      --------------------
      -- Create_Mmz_Key --
      --------------------

      function Create_Mmz_Key return Mmz_Key is
      begin
         return Mmz_K : Mmz_Key :=
           (Property => Mmz_Bare_Instantiated_Generic_Type_Instantiated_Generic_Type_P_Get_Instantiated_Type,
            Items    => new Mmz_Key_Array (1 ..  2))
         do
            Mmz_K.Items (1) := (Kind => Mmz_Bare_Instantiated_Generic_Type,
                                As_Bare_Instantiated_Generic_Type => Self);
               Mmz_K.Items (2) :=
                 (Kind => Mmz_Internal_Entity_Info,
                  As_Internal_Entity_Info => E_Info);
         end return;
      end Create_Mmz_Key;

begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

            Populate_Lexical_Env_For_Unit (Self);
      end if;

      if Self /= null then

         if not Self.Unit.Context.In_Populate_Lexical_Env then

            if Find_Memoized_Value
              (Self.Unit, Mmz_Handle, Mmz_Val, Create_Mmz_Key'Access)
            then
               --# memoization-lookup

               if Mmz_Val.Kind = Mmz_Evaluating then
                  --# memoization-return
                  Raise_Property_Exception
                    (Self,
                     Property_Error'Identity,
                     "Infinite recursion detected");

               elsif Mmz_Val.Kind = Mmz_Error then
                  --# memoization-return
                  Reraise_Memoized_Error (Mmz_Val);

               else
                  Property_Result := Mmz_Val.As_Internal_Entity_Type_Decl;

                  --# memoization-return
                  return Property_Result;
               end if;
               --# end
            end if;

         end if;
      end if;

      begin
         
   --# scope-start

         --# expr-start 603 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 602 '<New TypeDecl.entity at ???>' New_Struct_1 None
--# expr-start 600 '<New EntityInfo at parser.py:2199>' New_Struct D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2199


--# expr-start 599 '<append_rebinding at parser.py:2202>' Rebinding D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2202
--# expr-start 592 '<FieldAccess for rebindings at parser.py:2202>' Fld_2 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2202



--# expr-start 591 '<FieldAccess for info at parser.py:2202>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2202






   




Fld_1 := Ent.Info;
--# expr-done 591


   




Fld_2 := Fld_1.Rebindings;
--# expr-done 592
--# expr-start 595 '<FieldAccess for children_env at parser.py:2203>' Fld_5 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2203



--# expr-start 594 '<parent at parser.py:2203>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2203



--# expr-start 593 '<FieldAccess for inner_type_decl at parser.py:2203>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2203






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Self.Instantiated_Generic_Type_Inner_Type_Decl;
--# expr-done 593


   

      if Fld_3 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_4 := Fld_3.Parent; 
--# expr-done 594


   

      if Fld_4 = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_5 := Liblktlang.Implementation.Children_Env (Node => Fld_4);
--# expr-done 595
--# expr-start 598 '<FieldAccess for get_inst_env at parser.py:2204>' Fld_8 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2204



--# expr-start 596 '<FieldAccess for generic_decl at parser.py:2204>' Fld_6 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2204






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start InstantiatedGenericType.generic_decl
Fld_6 := Liblktlang.Implementation.Instantiated_Generic_Type_P_Generic_Decl (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 596


   

      if Fld_6.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 597 '<FieldAccess for actuals at parser.py:2204>' Fld_7 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2204






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_7 := Self.Instantiated_Generic_Type_Actuals;
Inc_Ref (Fld_7);
--# expr-done 597
--# property-call-start GenericDecl.get_inst_env
Fld_8 := Liblktlang.Implementation.Generic_Decl_P_Get_Inst_Env (Node => Fld_6.Node, Actuals => Fld_7, E_Info => Fld_6.Info);
--# end
--# expr-done 598
Rebinding := AST_Envs.Append_Rebinding (Fld_2, Fld_5, Fld_8); 
--# expr-done 599
New_Struct := (From_Rebound => False, Md => No_Metadata, Rebindings => Rebinding); 
--# expr-done 600
--# expr-start 601 '<FieldAccess for inner_type_decl at parser.py:2198>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2198






   

      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Self.Instantiated_Generic_Type_Inner_Type_Decl;
--# expr-done 601
New_Struct_1 := (Info => New_Struct, Node => Fld); 
--# expr-done 602
Let_Result := New_Struct_1; 
--# end
--# expr-done 603

         Property_Result := Let_Result;
         
   --# end
      Finalizer_Scope_167;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_167;


               if Self /= null then
                     if not Self.Unit.Context.In_Populate_Lexical_Env then
                        Add_Memoized_Error
                          (Self.Unit, Mmz_Handle, Exc, Mmz_Stored);
                     end if;
               end if;


            raise;
      end;

      if Self /= null then
         if not Self.Unit.Context.In_Populate_Lexical_Env then

            Mmz_Val := (Kind => Mmz_Internal_Entity_Type_Decl,
                        As_Internal_Entity_Type_Decl => Property_Result);
            Add_Memoized_Value (Self.Unit, Mmz_Handle, Mmz_Val, Mmz_Stored);

         end if;
      end if;



   return Property_Result;
end Instantiated_Generic_Type_P_Get_Instantiated_Type;
--# end

   







--# property-start InstantiatedGenericType.type_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2209
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Type_Scope
  
  (Node : Bare_Instantiated_Generic_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
     Ent : Internal_Entity_Instantiated_Generic_Type :=
       Internal_Entity_Instantiated_Generic_Type'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Lexical_Env;

      
            procedure Finalizer_Scope_168 with Inline_Always;

      Fld : Internal_Entity_Type_Decl;
Fld_1 : Lexical_Env := Empty_Env;

            procedure Finalizer_Scope_168 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_168;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 605 '<FieldAccess for type_scope at parser.py:2209>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2209



--# expr-start 604 '<FieldAccess for get_instantiated_type at parser.py:2209>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2209






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start InstantiatedGenericType.get_instantiated_type
Fld := Liblktlang.Implementation.Instantiated_Generic_Type_P_Get_Instantiated_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 604


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.type_scope'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Type_Scope (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 605

         Property_Result := Fld_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_168;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_168;




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Type_Scope;
--# end

   







--# property-start InstantiatedGenericType.fields D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2211
pragma Warnings (Off, "is not referenced");
 function Instantiated_Generic_Type_P_Fields
  
  (Node : Bare_Instantiated_Generic_Type
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Component_Decl_Array_Access
is
   Self : Bare_Instantiated_Generic_Type  := Bare_Instantiated_Generic_Type (Node);
     Ent : Internal_Entity_Instantiated_Generic_Type :=
       Internal_Entity_Instantiated_Generic_Type'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Component_Decl_Array_Access;

      
            procedure Finalizer_Scope_169 with Inline_Always;

      Fld : Internal_Entity_Type_Decl;
Fld_1 : Internal_Entity_Component_Decl_Array_Access;

            procedure Finalizer_Scope_169 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_169;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         --# expr-start 607 '<FieldAccess for fields at parser.py:2211>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2211



--# expr-start 606 '<FieldAccess for get_instantiated_type at parser.py:2211>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2211






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start InstantiatedGenericType.get_instantiated_type
Fld := Liblktlang.Implementation.Instantiated_Generic_Type_P_Get_Instantiated_Type (Node => Ent.Node, E_Info => Ent.Info);
--# end
--# expr-done 606


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]TypeDecl.fields'
Fld_1 := Liblktlang.Implementation.Dispatcher_Type_Decl_P_Fields (Node => Fld.Node, E_Info => Fld.Info);
--# end
--# expr-done 607

         Property_Result := Fld_1;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_169;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_169;




            raise;
      end;



   return Property_Result;
end Instantiated_Generic_Type_P_Fields;
--# end



   


      

   --
   --  Primitives for Bare_Named_Type_Decl
   --

   




      
   function Named_Type_Decl_F_Decls
     (Node : Bare_Named_Type_Decl) return Bare_Decl_Block
   is
      

         Kind : constant Lkt_Named_Type_Decl := Node.Kind;
   begin
         case Kind is
               when Lkt_Trait_Decl =>
                     
         return Node.Trait_Decl_F_Decls;
      
               when Lkt_Enum_Type_Decl =>
                     
         return Node.Enum_Type_Decl_F_Decls;
      
               when Lkt_Struct_Decl =>
                     
         return Node.Struct_Decl_F_Decls;
      
               when Lkt_Class_Decl =>
                     
         return Node.Class_Decl_F_Decls;
      
               when Lkt_Enum_Class_Decl =>
                     
         return Node.Enum_Class_Decl_F_Decls;
      
         end case;
   end;


   







--# property-start NamedTypeDecl.type_scope D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1991
pragma Warnings (Off, "is not referenced");
 function Named_Type_Decl_P_Type_Scope
  
  (Node : Bare_Named_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Lexical_Env
is
   Self : Bare_Named_Type_Decl  := Bare_Named_Type_Decl (Node);
     Ent : Internal_Entity_Named_Type_Decl :=
       Internal_Entity_Named_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Lexical_Env;

      
            procedure Finalizer_Scope_148 with Inline_Always;
            procedure Finalizer_Scope_395 with Inline_Always;

      Var_Expr_19 : Internal_Entity_Named_Type_Decl;
Fld : Internal_Entity_Decl_Block;
Fld_1 : Lexical_Env := Empty_Env;
Scope_Result : Lexical_Env := Empty_Env;
Result_Var : Lexical_Env := Empty_Env;

            procedure Finalizer_Scope_148 is
            begin
                     Dec_Ref (Scope_Result);
                     Dec_Ref (Result_Var);
            end Finalizer_Scope_148;
            procedure Finalizer_Scope_395 is
            begin
                     Dec_Ref (Fld_1);
            end Finalizer_Scope_395;


begin
   --# property-body-start

   pragma Assert (Self = Node);



      if Self /= null then
         Reset_Caches (Self.Unit);

      end if;


      begin
         
   --# scope-start

         



Var_Expr_19 := Ent;





if Var_Expr_19 /= No_Entity_Named_Type_Decl then
   



   --# scope-start


   

--# expr-start 609 '<FieldAccess for children_env at parser.py:1991>' Fld_1 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1991



--# expr-start 608 '<FieldAccess for decls at parser.py:1991>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1991






   

      if Var_Expr_19.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Decl_Block (Node => Implementation.Named_Type_Decl_F_Decls (Var_Expr_19.Node), Info => Var_Expr_19.Info);
--# expr-done 608


   

      if Fld.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_1 := Liblktlang.Implementation.Children_Env (Node => Fld.Node, E_Info => Fld.Info);
--# expr-done 609
Scope_Result := Fld_1;
   Inc_Ref (Scope_Result);


   --# end
      Finalizer_Scope_395;


   Result_Var := Scope_Result;
else
   
   Result_Var := Empty_Env;
end if;

      Inc_Ref (Result_Var);



         Property_Result := Result_Var;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_148;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_148;
                     Finalizer_Scope_395;




            raise;
      end;



   return Property_Result;
end Named_Type_Decl_P_Type_Scope;
--# end

   







--# property-start NamedTypeDecl.fields D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1993
pragma Warnings (Off, "is not referenced");
 function Named_Type_Decl_P_Fields
  
  (Node : Bare_Named_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Entity_Component_Decl_Array_Access
is
   Self : Bare_Named_Type_Decl  := Bare_Named_Type_Decl (Node);
     Ent : Internal_Entity_Named_Type_Decl :=
       Internal_Entity_Named_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Entity_Component_Decl_Array_Access;

      
            procedure Finalizer_Scope_149 with Inline_Always;

      Fld : Internal_Entity_Decl_Block;
Fld_1 : Bare_Decl_Block;
Fld_2 : Internal_Entity_Info;
Item_32 : Internal_Entity_Full_Decl;
Bare_Item_32 : Bare_Full_Decl;
Fld_3 : Internal_Entity_Decl;
Cast_Result : Internal_Entity_Component_Decl;
As_Entity : Internal_Entity_Full_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Full_Decl;
Fld_4 : Internal_Entity_Decl;
Is_A : Boolean;
Map_Result : Internal_Entity_Component_Decl_Array_Access;

            procedure Finalizer_Scope_149 is
            begin
                     Dec_Ref (Map_Result);
            end Finalizer_Scope_149;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 615 '<FilterMap at parser.py:1993>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1993








   

   --# expr-start 610 '<FieldAccess for decls at parser.py:1993>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1993






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Decl_Block (Node => Implementation.Named_Type_Decl_F_Decls (Ent.Node), Info => Ent.Info);
--# expr-done 610






   




Fld_1 := Fld.Node;

   
      declare
         Map_Result_Vec : Internal_Entity_Component_Decl_Vectors.Vector;
      begin
   

   

      if Fld_1 = null then
         
      Map_Result := Create_Internal_Entity_Component_Decl_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Decl_Block := Fld_1;
      begin
         for Untyped_Item_32 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_32 := Untyped_Item_32; 
                  



Is_Null := Bare_Item_32 = null; 
if Is_Null then
   
   If_Result := No_Entity_Full_Decl;
else
   





   




Fld_2 := Fld.Info;

As_Entity := (Info => Fld_2, Node => Bare_Item_32); 
   If_Result := As_Entity;
end if;



                  Item_32 := If_Result; 

            
   --# scope-start


               --# bind d Item_32

            
         --# expr-start 612 '<IsA ComponentDecl at parser.py:1995>' Is_A D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1995
--# expr-start 611 '<FieldAccess for decl at parser.py:1995>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1995






   

      if Item_32.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_4 := Create_Internal_Entity_Decl (Node => Item_32.Node.Full_Decl_F_Decl, Info => Item_32.Info);
--# expr-done 611
Is_A := Fld_4.Node /= null 
and then Fld_4.Node.Kind in Lkt_Component_Decl; 
--# expr-done 612
         if Is_A then
            
         
      --# expr-start 614 '<Cast to ComponentDecl at parser.py:1994>' Cast_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1994





--# expr-start 613 '<FieldAccess for decl at parser.py:1994>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:1994






   

      if Item_32.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Create_Internal_Entity_Decl (Node => Item_32.Node.Full_Decl_F_Decl, Info => Item_32.Info);
--# expr-done 613



   if Fld_3.Node = null
      or else Fld_3.Node.Kind in Lkt_Component_Decl
   then
      
      Cast_Result := Create_Internal_Entity_Component_Decl
        (Node => Fld_3.Node,
         Info => Fld_3.Info);

   else
         Raise_Property_Exception
           (Self, Property_Error'Identity, "invalid object cast");
   end if;


--# expr-done 614
      

         declare
            Item_To_Append : constant Internal_Entity_Component_Decl := Cast_Result;
         begin
            Internal_Entity_Component_Decl_Vectors.Append (Map_Result_Vec, Item_To_Append);
         end;
   
   
         end if;
   

            
   --# end



         end loop;
      end;
   
      end if;

   

         Map_Result := Create_Internal_Entity_Component_Decl_Array
           (Items_Count => Natural (Internal_Entity_Component_Decl_Vectors.Length (Map_Result_Vec)));
         for I in Map_Result.Items'Range loop
            Map_Result.Items (I) := Internal_Entity_Component_Decl_Vectors.Get
              (Map_Result_Vec,
               I + Internal_Entity_Component_Decl_Vectors.Index_Type'First - Map_Result.Items'First);
         end loop;
         Internal_Entity_Component_Decl_Vectors.Destroy (Map_Result_Vec);

      end;
   



--# expr-done 615

         Property_Result := Map_Result;
            Inc_Ref (Property_Result);
         
   --# end
      Finalizer_Scope_149;


      exception
         when Exc : Property_Error =>
                     Finalizer_Scope_149;




            raise;
      end;



   return Property_Result;
end Named_Type_Decl_P_Fields;
--# end

   







--# property-start '[dispatcher]NamedTypeDecl.is_authorized_decl' dispatcher
pragma Warnings (Off, "is not referenced");
 function Dispatcher_Named_Type_Decl_P_Is_Authorized_Decl
  
  (Node : Bare_Named_Type_Decl
      ; Decl : Internal_Entity_Decl
  )

   return Boolean
is
   Self : Bare_Named_Type_Decl  := Bare_Named_Type_Decl (Node);
      --# bind self Self

   

   --# bind decl Decl

   Property_Result : Boolean;



begin
   --# property-body-start

   pragma Assert (Self = Node);





      if Self = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dispatching on null node");
      end if;

      case Lkt_Named_Type_Decl (Self.Kind) is
               when Lkt_Class_Decl =>
                  --# property-call-start ClassDecl.is_authorized_decl
                  Property_Result := Class_Decl_P_Is_Authorized_Decl
                    (Self
                        , Decl
                    );
                  --# end
               when Lkt_Enum_Class_Decl =>
                  --# property-call-start EnumClassDecl.is_authorized_decl
                  Property_Result := Enum_Class_Decl_P_Is_Authorized_Decl
                    (Self
                        , Decl
                    );
                  --# end
               when Lkt_Enum_Type_Decl =>
                  --# property-call-start EnumTypeDecl.is_authorized_decl
                  Property_Result := Enum_Type_Decl_P_Is_Authorized_Decl
                    (Self
                        , Decl
                    );
                  --# end
               when Lkt_Struct_Decl =>
                  --# property-call-start StructDecl.is_authorized_decl
                  Property_Result := Struct_Decl_P_Is_Authorized_Decl
                    (Self
                        , Decl
                    );
                  --# end
               when Lkt_Trait_Decl =>
                  --# property-call-start TraitDecl.is_authorized_decl
                  Property_Result := Trait_Decl_P_Is_Authorized_Decl
                    (Self
                        , Decl
                    );
                  --# end
      end case;




   return Property_Result;
end Dispatcher_Named_Type_Decl_P_Is_Authorized_Decl;
--# end

   







--# property-start NamedTypeDecl.check_legality D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2007
pragma Warnings (Off, "is not referenced");
 function Named_Type_Decl_P_Check_Legality
  
  (Node : Bare_Named_Type_Decl
   ; E_Info : Internal_Entity_Info :=
      No_Entity_Info
  )

   return Internal_Semantic_Result_Array_Access
is
   Self : Bare_Named_Type_Decl  := Bare_Named_Type_Decl (Node);
     Ent : Internal_Entity_Named_Type_Decl :=
       Internal_Entity_Named_Type_Decl'(Node => Self, Info => E_Info);
      --# bind entity Ent

   


   Property_Result : Internal_Semantic_Result_Array_Access;

      
            procedure Finalizer_Scope_151 with Inline_Always;
            procedure Finalizer_Scope_397 with Inline_Always;

      Fld : Internal_Entity_Decl_Block;
Fld_1 : Bare_Decl_Block;
Fld_2 : Internal_Entity_Info;
Item_33 : Internal_Entity_Full_Decl;
Bare_Item_33 : Bare_Full_Decl;
Fld_3 : Internal_Entity_Decl;
Fld_4 : String_Type;
Str : String_Type;
Concat_Result : String_Type;
Fld_5 : String_Type;
Concat_Result_1 : String_Type;
Fld_6 : Internal_Semantic_Result := No_Semantic_Result;
As_Entity : Internal_Entity_Full_Decl;
Is_Null : Boolean;
If_Result : Internal_Entity_Full_Decl;
Fld_7 : Internal_Entity_Decl;
Fld_8 : Boolean;
Not_Val : Boolean;
Map_Result : Internal_Semantic_Result_Array_Access;
Let_Result : Internal_Semantic_Result_Array_Access;

            procedure Finalizer_Scope_151 is
            begin
                     Dec_Ref (Map_Result);
                     Dec_Ref (Let_Result);
            end Finalizer_Scope_151;
            procedure Finalizer_Scope_397 is
            begin
                     Dec_Ref (Fld_4);
                     Dec_Ref (Str);
                     Dec_Ref (Concat_Result);
                     Dec_Ref (Fld_5);
                     Dec_Ref (Concat_Result_1);
                     Dec_Ref (Fld_6);
            end Finalizer_Scope_397;


begin
   --# property-body-start

   pragma Assert (Self = Node);





      begin
         
   --# scope-start

         --# expr-start 626 '<Block at ???>' Let_Result None
--# scope-start
--# expr-start 625 '<FilterMap at parser.py:2019>' Map_Result D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2019








   

   --# expr-start 616 '<FieldAccess for decls at parser.py:2019>' Fld D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2019






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld := Create_Internal_Entity_Decl_Block (Node => Implementation.Named_Type_Decl_F_Decls (Ent.Node), Info => Ent.Info);
--# expr-done 616






   




Fld_1 := Fld.Node;

   
      declare
         Map_Result_Vec : Internal_Semantic_Result_Vectors.Vector;
      begin
   

   

      if Fld_1 = null then
         
      Map_Result := Create_Internal_Semantic_Result_Array (0);
   
      else
         

      declare
         
         Collection : constant Bare_Decl_Block := Fld_1;
      begin
         for Untyped_Item_33 of
               Collection.Nodes (1 .. Children_Count (Collection))
         loop
                  
                  Bare_Item_33 := Untyped_Item_33; 
                  



Is_Null := Bare_Item_33 = null; 
if Is_Null then
   
   If_Result := No_Entity_Full_Decl;
else
   





   




Fld_2 := Fld.Info;

As_Entity := (Info => Fld_2, Node => Bare_Item_33); 
   If_Result := As_Entity;
end if;



                  Item_33 := If_Result; 

            
   --# scope-start


               --# bind d Item_33

            
         --# expr-start 619 '<Not at parser.py:2025>' Not_Val D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2025
--# expr-start 618 '<FieldAccess for is_authorized_decl at parser.py:2025>' Fld_8 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2025






   

      if Ent.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 617 '<FieldAccess for decl at parser.py:2025>' Fld_7 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2025






   

      if Item_33.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_7 := Create_Internal_Entity_Decl (Node => Item_33.Node.Full_Decl_F_Decl, Info => Item_33.Info);
--# expr-done 617
--# property-call-start '[dispatcher]NamedTypeDecl.is_authorized_decl'
Fld_8 := Liblktlang.Implementation.Dispatcher_Named_Type_Decl_P_Is_Authorized_Decl (Node => Ent.Node, Decl => Fld_7);
--# end
--# expr-done 618
Not_Val := not (Fld_8); 
--# expr-done 619
         if Not_Val then
            
         
      --# expr-start 624 '<FieldAccess for error at parser.py:2020>' Fld_6 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2020






   

      if Item_33.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# expr-start 621 '<FieldAccess for decl_type_name at parser.py:2021>' Fld_4 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2021



--# expr-start 620 '<FieldAccess for decl at parser.py:2021>' Fld_3 D:/M/B/src/build-UCRT64/contrib/lkt/language/parser.py:2021






   

      if Item_33.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



Fld_3 := Create_Internal_Entity_Decl (Node => Item_33.Node.Full_Decl_F_Decl, Info => Item_33.Info);
--# expr-done 620


   

      if Fld_3.Node = null then
         Raise_Property_Exception
           (Self, Property_Error'Identity, "dereferencing a null access");
      end if;



--# property-call-start '[dispatcher]Decl.d