/* 
   Unix SMB/Netbios implementation.
   Version 1.7.
   Copyright (C) Karl Auer 1993,1994
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *  Load parameters.
 *
 *  This module provides suitable callback functions for the params
 *  module. It builds the internal table of service details which is
 *  then used by the rest of the server.
 *  
 *  To do:
 *
 *     Support more global parameters.
 *
 *     Move the table of service details and the enquiry functions to
 *     a separate module. This business of global arrays irritates me.
 *     This would also permit the structure to change - perhaps to a
 *     list with entries generated at run time as needed.
 *
 *     I'm reasonably sure that the 'available' flag in the service
 *     is an artifact of the old storage method. However, just in case
 *     it is designed for some future feature, I've maintained it. If
 *     it does turn out to be an artifact, it could still be useful -
 *     we could add a parameter that lets you 'turn off' services 
 *     without having to comment them out. 'available = yes/no'.
 *
 *     General cleaning up - this lot is very messy and inefficient. In
 *     particular, I'd like to make the Hungarian notation consistent
 *     again (multi-programmer hazard, that).
 *
 * To add a parameter:
 *
 *   1: Add an enum value for it in e_parms.
 *   2: Add the its string-to-enum mapping to ParmMap.
 *   3: If appropriate, add a flag for it to the copymap typedef.
 *   4: If appropriate, add suitable processing to copy_service().
 *   5: If appropriate, add suitable processing to service_ok().
 *   6  Add suitable processing to dump_globals(), dump_a_service() and
 *      dump_copy_map().
 *   7: Add a case for the new enum to do_parameter
 *   8: Initialise the value and the copy flag in do_section
 *
 * Notes:
 *   The configuration file is processed sequentially for speed. It is NOT
 *   accessed randomly as happens in 'real' Windows. For this reason, there
 *   is a fair bit of sequence-dependent code here - ie., code which assumes
 *   that certain things happen before others. In particular, the code which
 *   happens at the boundary between sections is delicately poised, so be
 *   careful!
 *
 */

#include "includes.h"

#include "smb.h"
#include "params.h"
#include "loadparm.h"

extern int DEBUGLEVEL;

#ifndef GLOBAL_NAME
#define GLOBAL_NAME "global"
#endif

#ifndef PRINTCAP_NAME
#define PRINTCAP_NAME "/etc/printcap"
#endif

#ifndef LOCK_DIR
#define LOCK_DIR "/tmp/samba"
#endif

#ifndef PRINTERS_NAME
#define PRINTERS_NAME "printers"
#endif

#ifndef HOMES_NAME
#define HOMES_NAME "homes"
#endif

/* some helpful bits */
#define NUMELS(x) (sizeof(x) / sizeof((x)[0]))
#define LP_SNUM_OK(iService) (((iService) >= 0) && ((iService) < iNumServices))
#define BOOLSTR(b) (b) ? "Yes" : "No"

/* these are the classes of parameter we have */
typedef enum
{
   E_NULLTYPE,
   E_GLOBAL,
   E_SERVICE,
   E_BOTH
} parmtype;

/* this is the list of numbers we use to identify parameters */
typedef enum
{
   E_NULLPARM,
   E_AVAILABLE,
   E_COPY,
   E_PATH,
   E_USERNAME,
   E_PRINTCOMMAND,
   E_LPQCOMMAND,
   E_PRINTERNAME,
   E_PRINTCAPNAME,
   E_LOCKDIR,
   E_GUESTACCOUNT,
   E_READONLY,
   E_WRITEOK,
   E_NOSETDIR,
   E_GUESTONLY,
   E_GUESTOK,
   E_PRINTOK,
   E_MAPSYSTEM,
   E_MAPHIDDEN,
   E_LOCKING,
   E_ONLYUSER,
   E_WIDELINKS,
   E_MANGLED,
   E_GETWDCACHE,
   E_READPREDICTION,
   E_READRAW,
   E_WRITERAW,
   E_CREATEMODE,
   E_MAXCONNECTIONS,
   E_ROOTDIR,
   E_DEFSERVICE,
   E_DFREE,
   E_MANGLEDSTACK,
   E_MAXXMIT,
   E_MAXPACKET,
   E_PWORDLEVEL,
   E_KEEPALIVE,
   E_DEADTIME,
   E_DEBUGLEVEL,
   E_MAXPROTOCOL,
   E_DONTDESCEND,
   E_HOSTSALLOW,
   E_HOSTSDENY,
   E_SECURITY,
   E_MAGICSCRIPT,
   E_MAGICOUTPUT
} parmnum;

/* 
 * This structure describes global (ie., server-wide) parameters.
 */
typedef struct
{
   char *szPrintcommand;
   char *szLpqcommand;
   char *szPrintername;
   char *szPrintcapname;
   char *szLockDir;
   char *szGuestaccount;
   char *szRootdir;
   char *szDefaultService;
   char *szDfree;
   char *szHostsallow;
   char *szHostsdeny;
   char *szMagicScript;
   char *szMagicOutput;
   int mangled_stack;
   int max_xmit;
   int max_packet;
   int keepalive;
   int pwordlevel;
   int deadtime;
   int maxprotocol;
   int iCreate_mode;
   int security;
   BOOL bLocking;
   BOOL bOnlyUser;
   BOOL bWidelinks;
   BOOL bReadRaw;
   BOOL bWriteRaw;
   BOOL bMangledNames;
   BOOL bGetwdCache;
   BOOL bReadPrediction;
} global;

/* 
 * This structure contains details required for a 'copy'. We make it a
 * separate structure so that we can allocate and deallocate as needed.
 * How it works: As each field is set, the 'use source' flag for that field
 * is set to False. If a 'copy' command is encountered anywhere in the
 * section, these flags will be used to decide what fields should be taken
 * from the source service - fields where 'use source' == True will be taken 
 * from the source service. If no 'copy' command is encountered, all this
 * information is discarded. When all sections have been processed from 
 * the file, a final pass through the services array is made to resolve any
 * references to services (well, not yet, but later maybe :-))
 */
typedef struct
{
   char    *szSourceService;
   BOOL    bCopyEncountered;
   BOOL    bUS_available;
   BOOL    bUS_path;
   BOOL    bUS_hosts_allow;
   BOOL    bUS_hosts_deny;
   BOOL    bUS_username;
   BOOL    bUS_printcommand;
   BOOL    bUS_lpqcommand;
   BOOL    bUS_printername;
   BOOL    bUS_read_only;
   BOOL    bUS_no_set_dir;
   BOOL    bUS_guest_only;
   BOOL    bUS_guest_ok;
   BOOL    bUS_print_ok;
   BOOL    bUS_map_system;
   BOOL    bUS_map_hidden;
   BOOL    bUS_locking;
   BOOL    bUS_only_user;
   BOOL    bUS_widelinks;
   BOOL    bUS_create_mode;
   BOOL    bUS_max_connections;
   BOOL    bUS_dont_descend;
   BOOL    bUS_magic_script;
   BOOL    bUS_magic_output;
   char    dummy[6]; /* dummy is here to ensure we get the numbers of
			elements in the default copymap right */
} copymap;

/* 
 * This structure describes a single service. 
 */
typedef struct
{
   char *szService;
   char *szPath;
   char *szUsername;
   char *szPrintcommand;
   char *szLpqcommand;
   char *szPrintername;
   char *szDontdescend;
   char *szHostsallow;
   char *szHostsdeny;
   char *szMagicScript;
   char *szMagicOutput;
   int  iCreate_mode;
   int  iMaxConnections;
   BOOL bAvailable;
   BOOL bRead_only;
   BOOL bNo_set_dir;
   BOOL bGuest_only;
   BOOL bGuest_ok;
   BOOL bPrint_ok;
   BOOL bMap_system;
   BOOL bMap_hidden;
   BOOL bLocking;
   BOOL bOnlyUser;
   BOOL bWidelinks;
   char dummy[3]; /* for alignment */
   copymap sCopyMap;
} service;


#define COPYMAPS(i) (pServices[i].sCopyMap)

/* 
 * This structure maps parameter names to parameter numbers
 * Use this rather than an array of strings so that elements are
 * not position dependent.
 * 
 * Fairly obviously, if you want to have synonyms (eg., "user" = "username")
 * you can just add entries...
 *
 */
typedef struct
{
   char        *pszName;
   parmnum     epNumber;
   parmtype    etNumber;
} parmmap;


/* This is a default service used to prime a services structure */
static service DefaultService = 
{
  "",    /* szService */
  "",    /* szPath */
  "",    /* szUsername */
  "",    /* szPrintcommand */
  "",    /* szLpqcommand */
  "",    /* szPrintername */
  "",    /* szDontdescend */
  "",    /* szHostsallow */
  "",    /* szHostsdeny */
  "",    /* szMagicScript */
  "",    /* szMagicOutput */
  DEF_CREATE_MASK,   /* iCreate_mode */
  0,     /* iMaxConnections */
  True,  /* bAvailable */
  True,  /* bRead_only */
  True,  /* bNo_set_dir */
  False, /* bGuest_only */
  False, /* bGuest_ok */
  False, /* bPrint_ok */
  False, /* bMap_system */
  False, /* bMap_hidden */
  True,  /* bLocking */
  False, /* bOnlyUser */
  True,  /* bWidelinks */
  "",    /* dummy */
  {"",False,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,True,"DUMMY" }, /* sCopyMap */
};


/* local variables */
static global Globals;
static service *pServices = NULL;
static int iNumServices = 0;
static int iServiceIndex = 0;
static BOOL bInGlobalSection = False;
static BOOL bDoneGlobalSection = False;

static parmmap aParmMaps[] =
{
   {"",              E_NULLPARM,     E_NULLTYPE},   /* this must be first */
   {"available",     E_AVAILABLE,    E_SERVICE },
   {"copy",          E_COPY,         E_SERVICE },
   {"path",          E_PATH,         E_SERVICE },
   {"directory",     E_PATH,         E_SERVICE },
   {"username",      E_USERNAME,     E_SERVICE },
   {"user",          E_USERNAME,     E_SERVICE },
   {"users",         E_USERNAME,     E_SERVICE },
   {"read only",     E_READONLY,     E_SERVICE },
   {"write ok",      E_WRITEOK,      E_SERVICE },
   {"writable",      E_WRITEOK,      E_SERVICE },
   {"set directory", E_NOSETDIR,     E_SERVICE },
   {"guest only",    E_GUESTONLY,    E_SERVICE },
   {"only guest",    E_GUESTONLY,    E_SERVICE },
   {"guest ok",      E_GUESTOK,      E_SERVICE },
   {"public",        E_GUESTOK,      E_SERVICE },
   {"print ok",      E_PRINTOK,      E_SERVICE },
   {"printable",     E_PRINTOK,      E_SERVICE },
   {"map system",    E_MAPSYSTEM,    E_SERVICE },
   {"map hidden",    E_MAPHIDDEN,    E_SERVICE },
   {"locking",       E_LOCKING,      E_BOTH },
   {"only user",     E_ONLYUSER,     E_BOTH },
   {"wide links",    E_WIDELINKS,    E_BOTH },
   {"mangled names", E_MANGLED,      E_GLOBAL },
   {"getwd cache",   E_GETWDCACHE,   E_GLOBAL },
   {"read prediction",E_READPREDICTION,E_GLOBAL },
   {"read raw",      E_READRAW,      E_GLOBAL },
   {"write raw",     E_WRITERAW,     E_GLOBAL },
   {"print command", E_PRINTCOMMAND, E_BOTH    },
   {"lpq command",   E_LPQCOMMAND,   E_BOTH    },
   {"printer",       E_PRINTERNAME,  E_BOTH    },
   {"printer name",  E_PRINTERNAME,  E_BOTH    },
   {"printcap name", E_PRINTCAPNAME, E_GLOBAL  },
   {"lock dir",      E_LOCKDIR,      E_GLOBAL  },
   {"lock directory",E_LOCKDIR,      E_GLOBAL  },
   {"hosts allow",   E_HOSTSALLOW,   E_BOTH    },
   {"allow hosts",   E_HOSTSALLOW,   E_BOTH    },
   {"hosts deny",    E_HOSTSDENY,    E_BOTH    },
   {"deny hosts",    E_HOSTSDENY,    E_BOTH    },
   {"guest account", E_GUESTACCOUNT, E_GLOBAL  },
   {"max connections",E_MAXCONNECTIONS,E_BOTH },
   {"create mask",   E_CREATEMODE,   E_BOTH },
   {"create mode",   E_CREATEMODE,   E_BOTH },
   {"root directory",E_ROOTDIR,      E_GLOBAL  },
   {"root dir",      E_ROOTDIR,      E_GLOBAL  },
   {"root",          E_ROOTDIR,      E_GLOBAL  },
   {"default service", E_DEFSERVICE, E_GLOBAL  },
   {"default",       E_DEFSERVICE,   E_GLOBAL  },
   {"dfree command", E_DFREE,        E_GLOBAL  },
   {"mangled stack", E_MANGLEDSTACK, E_GLOBAL  },
   {"max xmit",      E_MAXXMIT,      E_GLOBAL  },
   {"max packet",    E_MAXPACKET,    E_GLOBAL  },
   {"packet size",   E_MAXPACKET,    E_GLOBAL  },
   {"password level",E_PWORDLEVEL,   E_GLOBAL  },
   {"keepalive",     E_KEEPALIVE,    E_GLOBAL  },
   {"deadtime",      E_DEADTIME,     E_GLOBAL  },
   {"debuglevel",    E_DEBUGLEVEL,   E_GLOBAL  },
   {"protocol",      E_MAXPROTOCOL,  E_GLOBAL  },
   {"security",      E_SECURITY,     E_GLOBAL  },
   {"dont descend",  E_DONTDESCEND,  E_SERVICE  },
   {"magic script",  E_MAGICSCRIPT,  E_BOTH  },
   {"magic output",  E_MAGICOUTPUT,  E_BOTH  },
};

/* local prototypes */
static void   init_globals(void);
static int    strwicmp( char *psz1, char *psz2 );
static void   map_parameter( char *pszParmName, parmmap *pparmmap);
static BOOL   set_boolean( BOOL *pb, char *pszParmValue );
static int    getservicebyname(char *pszServiceName, service *pserviceDest);
static void   copy_service( service *pserviceDest, 
                            service *pserviceSource,
                            copymap *pcopymapDest );
static BOOL   service_ok(int iService);
static BOOL   finalise_service(int iService);
static BOOL   do_global_parameter(parmmap *pparmmap, char *pszParmValue);
static BOOL   do_service_parameter(parmmap *pparmmap, char *pszSectionName);
static BOOL   do_parameter(char *pszParmName, char *pszParmValue);
static BOOL   do_section(char *pszSectionName);
static void   dump_globals(void);
static void   dump_a_service(service *pService);
static void   dump_copy_map(copymap *pcopymap);

/***************************************************************************
initialise a service to the defaults
***************************************************************************/
static void init_service(service *pservice)
{
  memset((char *)pservice,0,sizeof(service));
  copy_service(pservice,&DefaultService,NULL);
}

/***************************************************************************
add a new service to the services array initialising it with the given 
service
***************************************************************************/
static int add_a_service(service *pservice)
{
  service tservice = *pservice;

  if (iNumServices == 0)
    pServices = (service *)malloc(sizeof(service));
  else
    {
      service *ptmpservice = 
	(service *)realloc(pServices,sizeof(service)*(iNumServices+1));
      if (!ptmpservice)
	{
	  DEBUG(0,("Failed to realloc to %d services\n",(iNumServices+1)));
	  return(False);
	}
      pServices = ptmpservice;
    }
  
  if (pServices)
    {
      init_service(&pServices[iNumServices]);
      copy_service(&pServices[iNumServices],&tservice,NULL);
      iNumServices++;
    }

  if (!pServices)
    return(-1);

  return(iNumServices - 1);
}

/***************************************************************************
add a new home service, with the specified home directory, defaults coming 
from service ifrom
***************************************************************************/
BOOL lp_add_home(char *pszHomename, int iDefaultService, char *pszHomedir)
{
  int i = add_a_service(&pServices[iDefaultService]);

  if (i < 0)
    return(False);

  string_set(&pServices[i].szService,pszHomename);
  string_set(&pServices[i].szPath,pszHomedir);
  pServices[i].bAvailable = True;

  if (!pServices[i].bGuest_ok)
    {
      pstring s;
      strcpy(s,pszHomename);
      strcat(s," ");
      if (pServices[i].szUsername)
	strcat(s,pServices[i].szUsername);
      string_set(&pServices[i].szUsername,s);
    }

  DEBUG(2,("adding home directory %s at %s\n", pszHomename, pszHomedir));

  return(True);
}


/***************************************************************************
add a new printer service, with defaults coming from service iFrom
***************************************************************************/
BOOL lp_add_printer(char *pszPrintername, int iDefaultService)
{
   int i = add_a_service(&pServices[iDefaultService]);

   if (i < 0)
      return(False);

   string_set(&pServices[i].szService,pszPrintername);

   /* note that we do NOT default the availability flag to True - */
   /* we take it from the default service passed. This allows all */
   /* dynamic printers to be disabled by disabling the [printers] */
   /* entry (if/when the 'available' keyword is implemented!).    */

   /* if there's no default username and the printer is not guest access, */
   /* make the username the service name. */
   if (pServices[i].szUsername[0] == '\0')
      if (!pServices[i].bGuest_ok)
         string_set(&pServices[i].szUsername,pszPrintername);

   /* if the default printer name is not set, the printer name is */
   /* set to the service name too. */
   if (pServices[i].szPrintername[0] == '\0')
      string_set(&pServices[i].szPrintername,pszPrintername);

   DEBUG(2,("adding printer service %s\n",pszPrintername));

   return(True);
}


/***************************************************************************
Initialise the global parameter structure.
***************************************************************************/
static void init_globals(void)
{
   string_init(&Globals.szPrintcommand, PRINT_COMMAND);
   string_init(&Globals.szLpqcommand, LPQ_COMMAND);
   string_init(&Globals.szPrintcapname, PRINTCAP_NAME);
   string_init(&Globals.szLockDir, LOCK_DIR);
   string_init(&Globals.szPrintername, "");
   string_init(&Globals.szGuestaccount, GUEST_ACCOUNT);
   string_init(&Globals.szRootdir, "/");
   string_init(&Globals.szDefaultService, "");
   string_init(&Globals.szDfree, "");
   string_init(&Globals.szHostsallow, "");
   string_init(&Globals.szHostsdeny, "");
   string_init(&Globals.szMagicScript, "");
   string_init(&Globals.szMagicOutput, "");
   Globals.max_packet = 65535;
   Globals.mangled_stack = 50;
   Globals.max_xmit = Globals.max_packet;
   Globals.keepalive = 0;
   Globals.pwordlevel = 0;
   Globals.deadtime = 0;
   Globals.maxprotocol = PROT_LANMAN1;
   Globals.security = SEC_SHARE;
   Globals.iCreate_mode = DEF_CREATE_MASK;
   Globals.bLocking = True;
   Globals.bOnlyUser = False;
   Globals.bWidelinks = True;
   Globals.bReadRaw = True;
   Globals.bWriteRaw = False;
   Globals.bMangledNames = True;
   Globals.bGetwdCache = False;
   Globals.bReadPrediction = True;
}

/***************************************************************************
Do a case-insensitive, whitespace-ignoring string compare.
***************************************************************************/
static int strwicmp(char *psz1, char *psz2)
{
   /* if BOTH strings are NULL, return TRUE, if ONE is NULL return */
   /* appropriate value. */
   if (psz1 == psz2)
      return (0);
   else
      if (psz1 == NULL)
         return (-1);
      else
          if (psz2 == NULL)
              return (1);

   /* sync the strings on first non-whitespace */
   while (1)
   {
      while (isspace(*psz1))
         psz1++;
      while (isspace(*psz2))
         psz2++;
      if (toupper(*psz1) != toupper(*psz2) || *psz1 == '\0' || *psz2 == '\0')
         break;
      psz1++;
      psz2++;
   }
   return (*psz1 - *psz2);
}

/***************************************************************************
Map a parameter's string representation to something we can use. 
Returns False if the parameter string is not recognised, else TRUE.
***************************************************************************/
static void map_parameter(char *pszParmName, parmmap *pparmmap)
{
   int iIndex;

   for (iIndex = NUMELS(aParmMaps) - 1; iIndex > 0; iIndex--) 
      if (strwicmp(aParmMaps[iIndex].pszName, pszParmName) == 0)
         break;

   if (iIndex < 1)
      DEBUG(0,( "Unknown parameter encountered: \"%s\"\n", pszParmName));

   *pparmmap = aParmMaps[iIndex];
}


/***************************************************************************
Set a boolean variable from the text value stored in the passed string.
Returns True in success, False if the passed string does not correctly 
represent a boolean.
***************************************************************************/
static BOOL set_boolean(BOOL *pb, char *pszParmValue)
{
   BOOL bRetval;

   bRetval = True;
   if (strwicmp(pszParmValue, "yes") == 0 ||
       strwicmp(pszParmValue, "true") == 0 ||
       strwicmp(pszParmValue, "1") == 0)
      *pb = True;
   else
      if (strwicmp(pszParmValue, "no") == 0 ||
          strwicmp(pszParmValue, "False") == 0 ||
          strwicmp(pszParmValue, "0") == 0)
         *pb = False;
      else
      {
         DEBUG(0,( "Badly formed boolean in configuration file: \"%s\".\n",
               pszParmValue));
         bRetval = False;
      }
   return (bRetval);
}

/***************************************************************************
Find a service by name. Otherwise works like get_service.
Note that this works from iServiceIndex because it is only called while
loading services, during which time iNumServices does not reflect the number
of correct and complete services.
***************************************************************************/
static int getservicebyname(char *pszServiceName, service *pserviceDest)
{
   int iService;

   for (iService = iServiceIndex - 1; iService >= 0; iService--)
      if (strwicmp(pServices[iService].szService, pszServiceName) == 0) 
      {
         if (pserviceDest != NULL)
	   copy_service(pserviceDest, &pServices[iService], NULL);
         break;
      }

   return (iService);
}

/***************************************************************************
free the dynamically allocated parts of the globals structure.
***************************************************************************/
static void free_globals(void)
{
   string_free(&Globals.szPrintcommand);
   string_free(&Globals.szLpqcommand);
   string_free(&Globals.szPrintcapname);
   string_free(&Globals.szLockDir);
   string_free(&Globals.szPrintername);
   string_free(&Globals.szGuestaccount);
   string_free(&Globals.szRootdir);
   string_free(&Globals.szDefaultService);
   string_free(&Globals.szDfree);
   string_free(&Globals.szHostsallow);
   string_free(&Globals.szHostsdeny);
   string_free(&Globals.szMagicScript);
   string_free(&Globals.szMagicOutput);
}

/***************************************************************************
free the dynamically allocated parts of a service struct
***************************************************************************/
static void free_service(service *pservice)
{
  if (!pservice)
     return;

  string_free(&pservice->szService);
  string_free(&pservice->szPath);
  string_free(&pservice->szUsername);
  string_free(&pservice->szPrintcommand);
  string_free(&pservice->szLpqcommand);
  string_free(&pservice->szPrintername);
  string_free(&pservice->szDontdescend);
  string_free(&pservice->sCopyMap.szSourceService);
  string_free(&pservice->szHostsallow);
  string_free(&pservice->szHostsdeny);
  string_free(&pservice->szMagicScript);
  string_free(&pservice->szMagicOutput);
}

/***************************************************************************
Copy a service structure to another, paying attention to the 'use source'
flags. If a flag is True, it means that the source SHOULD be copied. 

If pcopymapDest is NULL then copy all fields
***************************************************************************/
static void copy_service(service *pserviceDest, 
                         service *pserviceSource,
                         copymap *pcopymapDest)
{
   BOOL bcopyall = (pcopymapDest == NULL);

   if (bcopyall)
      string_set(&pserviceDest->szService, pserviceSource->szService);

   if (bcopyall || pcopymapDest->bUS_available)
      pserviceDest->bAvailable = pserviceSource->bAvailable;
 
   if (bcopyall || pcopymapDest->bUS_path)
      string_set(&pserviceDest->szPath, pserviceSource->szPath);

   if (bcopyall || pcopymapDest->bUS_magic_script)
      string_set(&pserviceDest->szMagicScript, pserviceSource->szMagicScript);

   if (bcopyall || pcopymapDest->bUS_magic_output)
      string_set(&pserviceDest->szMagicOutput, pserviceSource->szMagicOutput);

   if (bcopyall || pcopymapDest->bUS_dont_descend)
      string_set(&pserviceDest->szDontdescend, pserviceSource->szDontdescend);

   if (bcopyall || pcopymapDest->bUS_username)
      string_set(&pserviceDest->szUsername, pserviceSource->szUsername);

   if (bcopyall || pcopymapDest->bUS_printcommand)
      string_set(&pserviceDest->szPrintcommand, pserviceSource->szPrintcommand);

   if (bcopyall || pcopymapDest->bUS_lpqcommand)
      string_set(&pserviceDest->szLpqcommand, pserviceSource->szLpqcommand);

   if (bcopyall || pcopymapDest->bUS_printername)
      string_set(&pserviceDest->szPrintername, pserviceSource->szPrintername);

   if (bcopyall || pcopymapDest->bUS_hosts_allow)
      string_set(&pserviceDest->szHostsallow, pserviceSource->szHostsallow);

   if (bcopyall || pcopymapDest->bUS_hosts_deny)
      string_set(&pserviceDest->szHostsdeny, pserviceSource->szHostsdeny);

   if (bcopyall || pcopymapDest->bUS_read_only)
      pserviceDest->bRead_only =
             pserviceSource->bRead_only;

   if (bcopyall || pcopymapDest->bUS_no_set_dir)
      pserviceDest->bNo_set_dir =
             pserviceSource->bNo_set_dir;

   if (bcopyall || pcopymapDest->bUS_guest_ok)
      pserviceDest->bGuest_ok =
             pserviceSource->bGuest_ok;

   if (bcopyall || pcopymapDest->bUS_print_ok)
      pserviceDest->bPrint_ok =
             pserviceSource->bPrint_ok;

   if (bcopyall || pcopymapDest->bUS_map_system)
      pserviceDest->bMap_system =
             pserviceSource->bMap_system;

   if (bcopyall || pcopymapDest->bUS_map_hidden)
      pserviceDest->bMap_hidden =
             pserviceSource->bMap_hidden;

   if (bcopyall || pcopymapDest->bUS_locking)
      pserviceDest->bLocking =
             pserviceSource->bLocking;

   if (bcopyall || pcopymapDest->bUS_only_user)
      pserviceDest->bOnlyUser =
             pserviceSource->bOnlyUser;

   if (bcopyall || pcopymapDest->bUS_widelinks)
      pserviceDest->bWidelinks =
             pserviceSource->bWidelinks;

   if (bcopyall || pcopymapDest->bUS_create_mode)
      pserviceDest->iCreate_mode =
             pserviceSource->iCreate_mode;

   if (bcopyall || pcopymapDest->bUS_max_connections)
      pserviceDest->iMaxConnections =
             pserviceSource->iMaxConnections;

  pserviceDest->sCopyMap = pserviceSource->sCopyMap;
  pserviceDest->sCopyMap.szSourceService = NULL;

  if (bcopyall)
    {
      string_set(&pserviceDest->sCopyMap.szSourceService,
		 pserviceSource->sCopyMap.szSourceService);
    }
  else
    pserviceDest->sCopyMap.bCopyEncountered = False;    

}

/***************************************************************************
Check a service for consistency. Return False if the service is in any way
incomplete or faulty, else True.
***************************************************************************/
static BOOL service_ok(int iService)
{
   BOOL bRetval;

   bRetval = True;
   if (pServices[iService].szService[0] == '\0')
   {
      DEBUG(0,( "The following message indicates an internal error:\n"));
      DEBUG(0,( "No service name in service entry.\n"));
      bRetval = False;
   }

   /* A printer name in the [printers] entry will override the dynamic */
   /* printer names. Good for pointing all output to a single printer, */
   /* but probably not what you wanted. */
   if (strwicmp(pServices[iService].szService,PRINTERS_NAME) == 0)
      if (pServices[iService].szPrintername[0] != '\0')
         DEBUG(0,( "WARNING! Printer name set in [%s] service!\n",
               pServices[iService].szService));

   /* The [printers] entry MUST be printable. I'm all for flexibility, but */
   /* I can't see why you'd want a non-printable printer service...        */
   if (strwicmp(pServices[iService].szService,PRINTERS_NAME) == 0)
      if (!pServices[iService].bPrint_ok)
      {
         DEBUG(0,( "[%s] service MUST be printable!\n",
               pServices[iService].szService));
         bRetval = False;
      }

   if (pServices[iService].szPath[0] == '\0' &&
       strwicmp(pServices[iService].szService,HOMES_NAME) != 0)
   {
      DEBUG(0,( "No path in service %s\n", pServices[iService].szService));
      bRetval = False;
   }

   if (!pServices[iService].bGuest_ok && 
       (pServices[iService].szUsername[0] == '\0') &&
       strwicmp(pServices[iService].szService,HOMES_NAME) != 0)
   {
      DEBUG(0,( "No user specified for non-guest service %s\n",
            pServices[iService].szService));
      bRetval = False;
   }

   if (!pServices[iService].bPrint_ok && 
       (pServices[iService].szPrintcommand[0] != '\0'))
   {
      DEBUG(0,( "Print command specified for non-print service %s\n",
            pServices[iService].szService));
      bRetval = False;
   }

   /* If a service is flagged unavailable, log the fact at level 0. */
   if (!pServices[iService].bAvailable) 
      DEBUG(0,( "NOTE: Service %s is flagged unavailable.\n",
            pServices[iService].szService));

   return (bRetval);
}

/***************************************************************************
Finalise a service section.
***************************************************************************/
static BOOL finalise_service(int iService)
{
   DEBUG(3,( "Finalising service %s\n", pServices[iService].szService));

#if 0
   /* The effect of the code here has been gained by having the */
   /* function lp_username() return the service name in cases where */
   /* the username is empty and service allows guest access. */


   /* make the user name the service name if guest OK and no other */
   /* username provided */
   if (pServices[iService].szUsername[0] == '\0')
      if (pServices[iService].bGuest_ok)
         string_set(&pServices[iService].szUsername,
                pServices[iService].szService);
#endif

   /* finally see if our new section is complete and correct */
   return (service_ok(iService));
}

/***************************************************************************
Process a global. The parameter has already been mapped, so we just have to 
interpret it here. Returns True if successful, False if there is something
wrong with the parameter's value (out of range, wrong type).
***************************************************************************/
static BOOL do_global_parameter(parmmap *pparmmap, char *pszParmValue)
{
   BOOL bRetval;

   bRetval = True;
   switch (pparmmap->epNumber)
   {
      case E_PRINTCOMMAND:
         string_set(&Globals.szPrintcommand, pszParmValue);
         break;

      case E_LPQCOMMAND:
         string_set(&Globals.szLpqcommand, pszParmValue);
         break;

      case E_PRINTERNAME:
         string_set(&Globals.szPrintername, pszParmValue);
         break;

      case E_PRINTCAPNAME:
         string_set(&Globals.szPrintcapname, pszParmValue);
         break;

      case E_LOCKDIR:
         string_set(&Globals.szLockDir, pszParmValue);
         break;

      case E_HOSTSALLOW:
         string_set(&Globals.szHostsallow, pszParmValue);
         break;

      case E_HOSTSDENY:
         string_set(&Globals.szHostsdeny, pszParmValue);
         break;

      case E_MAGICSCRIPT:
         string_set(&Globals.szMagicScript, pszParmValue);
         break;

      case E_MAGICOUTPUT:
         string_set(&Globals.szMagicOutput, pszParmValue);
         break;

      case E_CREATEMODE:
	 sscanf(pszParmValue,"%o",&Globals.iCreate_mode);
	 DefaultService.iCreate_mode = Globals.iCreate_mode;
         break;

      case E_MAXCONNECTIONS:
	 DefaultService.iMaxConnections = atoi(pszParmValue);
         break;

      case E_GUESTACCOUNT:
         string_set(&Globals.szGuestaccount, pszParmValue);
         break;

      case E_ROOTDIR:
         string_set(&Globals.szRootdir, pszParmValue);
         break;

       case E_DEFSERVICE:
         string_set(&Globals.szDefaultService, pszParmValue);
         break;

      case E_DFREE:
         string_set(&Globals.szDfree, pszParmValue);
         break;

       case E_MAXXMIT:
         Globals.max_xmit = atoi(pszParmValue);
         break;

       case E_MANGLEDSTACK:
         Globals.mangled_stack = atoi(pszParmValue);
         break;

       case E_MAXPACKET:
         Globals.max_packet = atoi(pszParmValue);
         break;

       case E_KEEPALIVE:
         Globals.keepalive = atoi(pszParmValue);
         break;

       case E_PWORDLEVEL:
         Globals.pwordlevel = atoi(pszParmValue);
         break;

       case E_DEADTIME:
         Globals.deadtime = atoi(pszParmValue);
         break;

       case E_DEBUGLEVEL:
	 {
	   extern int DEBUGLEVEL;
	   DEBUGLEVEL = atoi(pszParmValue);
	   DEBUG(2,("Switched to debug level %d\n",DEBUGLEVEL));
	 }
         break;

       case E_MAXPROTOCOL:
         Globals.maxprotocol = interpret_protocol(pszParmValue,
						  Globals.maxprotocol);
         break;

       case E_SECURITY:
         Globals.security = interpret_security(pszParmValue,
						  Globals.security);
         break;

       case E_LOCKING:
	 set_boolean(&Globals.bLocking,pszParmValue);
	 DefaultService.bLocking = Globals.bLocking;
         break;

       case E_ONLYUSER:
	 set_boolean(&Globals.bOnlyUser,pszParmValue);
	 DefaultService.bOnlyUser = Globals.bOnlyUser;
         break;

       case E_WIDELINKS:
	 set_boolean(&Globals.bWidelinks,pszParmValue);
	 DefaultService.bWidelinks = Globals.bWidelinks;
         break;

       case E_READRAW:
	 set_boolean(&Globals.bReadRaw,pszParmValue);
         break;
 
       case E_MANGLED:
	 set_boolean(&Globals.bMangledNames,pszParmValue);
         break;
 
       case E_GETWDCACHE:
	 set_boolean(&Globals.bGetwdCache,pszParmValue);
         break;
 
       case E_READPREDICTION:
	 set_boolean(&Globals.bReadPrediction,pszParmValue);
         break;
 
       case E_WRITERAW:
	 set_boolean(&Globals.bWriteRaw,pszParmValue);
         break;
 
      default:
         DEBUG(0,( "Unsupported global parameter.\n"));
         bRetval = False;
         break;
   }
   return (bRetval);
}

/***************************************************************************
Process a service parameter. This assumes that iServiceIndex has been set 
correctly. The parameter has already been mapped, so we just have to 
interpret it here. Returns True if successful, False if there is something
wrong with the parameter's value (out of range, wrong type).

Note that copy at this stage requires that the source service be present.
This may change in the future - which explains the mysterious presence
of things like bCopyEncountered.

Note also that the space for the services and copy map arrays is assumed
to have been allocated!
***************************************************************************/
static BOOL do_service_parameter(parmmap *pparmmap, char *pszParmValue)
{
   BOOL bRetval;
   BOOL bTemp;
   int iTemp;
   service serviceTemp;

   init_service(&serviceTemp);

   bRetval = True;
   switch (pparmmap->epNumber)
   {
      case E_COPY:
         bRetval = False;
         string_set(&COPYMAPS(iServiceIndex).szSourceService, pszParmValue);
         COPYMAPS(iServiceIndex).bCopyEncountered = True;
	 DEBUG(3,("Copying service from service %s\n",pszParmValue));
         if ((iTemp = getservicebyname(pszParmValue, &serviceTemp)) >= 0)
         {
            if (iTemp == iServiceIndex)
               DEBUG(0,( "Can't copy service %s- unable to copy self!\n",
                     pszParmValue));
            else
               if (COPYMAPS(iTemp).bCopyEncountered)
                  DEBUG(0,( "Can't copy service %s - source uses copy too!\n",
                        pszParmValue));
               else
               {
		 copy_service(&pServices[iServiceIndex], 
                               &serviceTemp,
                               &COPYMAPS(iServiceIndex));
		 COPYMAPS(iServiceIndex).bCopyEncountered = False;
		 bRetval = True;
               }
         }
         else
            DEBUG(0,( "Unable to copy service - source not found: %s\n",
                  pszParmValue));
         break;

      case E_AVAILABLE:
         bRetval = set_boolean(&pServices[iServiceIndex].bAvailable,
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_available = False;
         break;

      case E_PATH:
         string_set(&pServices[iServiceIndex].szPath, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_path = False;
         break;

      case E_MAGICSCRIPT:
         string_set(&pServices[iServiceIndex].szMagicScript, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_magic_script = False;
         break;

      case E_MAGICOUTPUT:
         string_set(&pServices[iServiceIndex].szMagicOutput, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_magic_output = False;
         break;

      case E_DONTDESCEND:
         string_set(&pServices[iServiceIndex].szDontdescend, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_dont_descend = False;
         break;

      case E_USERNAME:
         string_set(&pServices[iServiceIndex].szUsername, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_username = False;
         break;

      case E_PRINTCOMMAND:
         string_set(&pServices[iServiceIndex].szPrintcommand, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_printcommand = False;
         break;

      case E_LPQCOMMAND:
         string_set(&pServices[iServiceIndex].szLpqcommand, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_lpqcommand = False;
         break;

      case E_PRINTERNAME:
         string_set(&pServices[iServiceIndex].szPrintername, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_printername = False;
         break;

      case E_HOSTSALLOW:
         string_set(&pServices[iServiceIndex].szHostsallow, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_hosts_allow = False;
         break;

      case E_HOSTSDENY:
         string_set(&pServices[iServiceIndex].szHostsdeny, pszParmValue);
         COPYMAPS(iServiceIndex).bUS_hosts_deny = False;
         break;

      case E_READONLY:
         bRetval = set_boolean(&pServices[iServiceIndex].bRead_only,
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_read_only = False;
         break;

      case E_WRITEOK:
         bRetval = set_boolean(&bTemp, pszParmValue);
         pServices[iServiceIndex].bRead_only = !bTemp;
         COPYMAPS(iServiceIndex).bUS_read_only = False;
         break;

      case E_NOSETDIR:
         bRetval = set_boolean(&pServices[iServiceIndex].bNo_set_dir, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_no_set_dir = False;
         break;

      case E_GUESTOK:
         bRetval = set_boolean(&pServices[iServiceIndex].bGuest_ok, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_guest_ok = False;
         break;

      case E_GUESTONLY:
         bRetval = set_boolean(&pServices[iServiceIndex].bGuest_only, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_guest_only = False;
         break;

      case E_PRINTOK:
         bRetval = set_boolean(&pServices[iServiceIndex].bPrint_ok, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_print_ok = False;
         break;

      case E_MAPSYSTEM:
         bRetval = set_boolean(&pServices[iServiceIndex].bMap_system, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_map_system = False;
         break;

      case E_MAPHIDDEN:
         bRetval = set_boolean(&pServices[iServiceIndex].bMap_hidden, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_map_hidden = False;
         break;

      case E_LOCKING:
         bRetval = set_boolean(&pServices[iServiceIndex].bLocking, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_locking = False;
         break;

      case E_ONLYUSER:
         bRetval = set_boolean(&pServices[iServiceIndex].bOnlyUser, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_only_user = False;
         break;

      case E_WIDELINKS:
         bRetval = set_boolean(&pServices[iServiceIndex].bWidelinks, 
                                pszParmValue);
         COPYMAPS(iServiceIndex).bUS_widelinks = False;
         break;

      case E_CREATEMODE:
         bRetval = 
	   (sscanf(pszParmValue,"%o",&pServices[iServiceIndex].iCreate_mode)==1);
         COPYMAPS(iServiceIndex).bUS_create_mode = False;
         break;

      case E_MAXCONNECTIONS:
	 pServices[iServiceIndex].iMaxConnections = atoi(pszParmValue);
         COPYMAPS(iServiceIndex).bUS_max_connections = False;
         break;

      case E_NULLPARM:
         DEBUG(0,( "Unknown parameter ignored.\n"));
         break;

      default:
         DEBUG(0,( "The following message indicates an internal error.\n"));
         DEBUG(0,( "Unknown parameter %d ignored.\n", pparmmap->epNumber));
         break;
   }

   free_service(&serviceTemp);
   return (bRetval);
}

/***************************************************************************
Process a parameter. This function just splits the processing between global
and service functions - the real action happens elsewhere. However, this
function does check that parameters are allowable in context.
***************************************************************************/
static BOOL do_parameter(char *pszParmName, char *pszParmValue)
{
   BOOL bRetval;
   parmmap parmmapTemp;

   bRetval = False;
   map_parameter(pszParmName, &parmmapTemp);
   switch (parmmapTemp.etNumber)
   {
      case E_SERVICE:
         if (bInGlobalSection)
            DEBUG(0,( "Service parameter %s found in global section!\n",
                  pszParmName));
         else
            bRetval = do_service_parameter(&parmmapTemp, pszParmValue);
         break;

      case E_GLOBAL:
         if (!bInGlobalSection)
            DEBUG(0,( "Global parameter %s found in service section!\n",
                  pszParmName));
         else
            bRetval = do_global_parameter(&parmmapTemp, pszParmValue);
         break;

      case E_BOTH:
         if (bInGlobalSection)
            bRetval = do_global_parameter(&parmmapTemp, pszParmValue);
         else
            bRetval = do_service_parameter(&parmmapTemp, pszParmValue);
         break;

      case E_NULLTYPE:
         DEBUG(0,( "Ignoring unknown parameter \"%s\"\n", pszParmName));
         bRetval = True;
         break;
 
      default:
         DEBUG(0,( "The following two messages indicate an internal error:\n"));
         DEBUG(0,( "Unknown parameter \"%s\"\n", pszParmName));
         DEBUG(0,( "Unknown parameter type %d\n", parmmapTemp.etNumber));
         break;
   }
 
   return (bRetval);
}

/***************************************************************************
Process a new section (service). At this stage all sections are services.
Later we'll have special sections that permit server parameters to be set.
Returns True on success, False on failure.
***************************************************************************/
static BOOL do_section(char *pszSectionName)
{
   BOOL bRetval;

   bRetval = False;

   /* if we've just struck a global section, note the fact. */
   bInGlobalSection = (strwicmp(pszSectionName, GLOBAL_NAME) == 0);

   /* check for multiple global sections */
   if (bInGlobalSection)
   {
      /* if found, bug out with error */
      if (bDoneGlobalSection)
         DEBUG(0,( "Multiple global sections found in configuration file!\n"));
      else
      {
         /* otherwise there is nothing more to do here */
         DEBUG(3,( "Processing section \"[%s]\"\n", pszSectionName));
         bDoneGlobalSection = True;
         bRetval = True;
      }
      return (bRetval);
   }

   /*
    * Following processing occurs only for service sections, not for global
    */

   /* check that service name is unique */
   if (getservicebyname(pszSectionName, NULL) >= 0)
   {
      DEBUG(0,( "The service name \"%s\" is not unique.\n", pszSectionName));
      return (bRetval);
   }

   if (iServiceIndex >= iNumServices)
   {
      DEBUG(0,( "Maximum service count (%d) exceeded at service \"%s\"\n",
            iNumServices, pszSectionName));
   }
   else
   {
      /* if we have a current service, tidy it up before moving on */
      bRetval = True;

      if (iServiceIndex >= 0)
         bRetval = finalise_service(iServiceIndex);

      /* if all is still well, move to the next record in the services array */
      if (bRetval)
      {
         /* We put this here to avoid an odd message order if messages are */
         /* issued by the post-processing of a previous section. */
         DEBUG(3,( "Processing section \"[%s]\"\n", pszSectionName));

         iServiceIndex++;

	 if (add_a_service(&DefaultService) < 0)
	   {
	     DEBUG(0,("Failed to add a new service\n"));
	     return(False);
	   }

         string_init(&pServices[iServiceIndex].szService, pszSectionName); 
      }
   }
   return (bRetval);
}

/***************************************************************************
Display the contents of the global structure.
***************************************************************************/
static void dump_globals(void)
{
    
      printf("Global parameters:\n");
      printf("\tPrint command: %s\n",  Globals.szPrintcommand);
      printf("\tLpq command  : %s\n",  Globals.szLpqcommand);
      printf("\tPrinter name : %s\n",  Globals.szPrintername);
      printf("\tPrintcap name: %s\n",  Globals.szPrintcapname);
      printf("\tLock directory: %s\n", Globals.szLockDir);
      printf("\tGuest account: %s\n",  Globals.szGuestaccount);
      printf("\tRoot dir     : %s\n",  Globals.szRootdir);
      printf("\tDefault Service : %s\n",  Globals.szDefaultService);
      printf("\tDfree command: %s\n",  Globals.szDfree);
      printf("\tMagic Script : %s\n",  Globals.szMagicScript);
      printf("\tMagic Output : %s\n",  Globals.szMagicOutput);
      printf("\tHosts allowed: %s\n",  Globals.szHostsallow);
      printf("\tHosts denied : %s\n",  Globals.szHostsdeny);
      printf("\tMax Xmit     : %d\n",  Globals.max_xmit);
      printf("\tMangled Stack: %d\n",  Globals.mangled_stack);
      printf("\tMax Packet   : %d\n",  Globals.max_packet);
      printf("\tKeepalive    : %d\n",  Globals.keepalive);
      printf("\tPassword level: %d\n",  Globals.pwordlevel);
      printf("\tDeadtime     : %d\n",  Globals.deadtime);
      printf("\tMaxprotocol  : %d\n",  Globals.maxprotocol);
      printf("\tCreate mode  : %o\n",  Globals.iCreate_mode);
      printf("\tSecurity     : %d\n",  Globals.security);
}

/***************************************************************************
Display the contents of a single services record.
***************************************************************************/
static void dump_a_service(service *pService)
{
      printf("Service name: %s\n",  pService->szService);
      printf("\tavailable     : %s\n", BOOLSTR(pService->bAvailable));
      printf("\tpath          : %s\n", pService->szPath);
      printf("\tusername      : %s\n", pService->szUsername);
      printf("\tprint command : %s\n", pService->szPrintcommand);
      printf("\tlpq command   : %s\n", pService->szLpqcommand);
      printf("\tprinter name  : %s\n", pService->szPrintername);
      printf("\thosts allow   : %s\n", pService->szHostsallow);
      printf("\thosts deny    : %s\n", pService->szHostsdeny);
      printf("\tdont descend  : %s\n", pService->szDontdescend);
      printf("\tmagic script  : %s\n", pService->szMagicScript);
      printf("\tmagic output  : %s\n", pService->szMagicOutput);
      printf("\tread_only     : %s\n", BOOLSTR(pService->bRead_only));
      printf("\tno_set_dir    : %s\n", BOOLSTR(pService->bNo_set_dir));
      printf("\tguest_only    : %s\n", BOOLSTR(pService->bGuest_only));
      printf("\tguest_ok      : %s\n", BOOLSTR(pService->bGuest_ok));
      printf("\tprint_ok      : %s\n", BOOLSTR(pService->bPrint_ok));
      printf("\tmap_system    : %s\n", BOOLSTR(pService->bMap_system));
      printf("\tmap_hidden    : %s\n", BOOLSTR(pService->bMap_hidden));
      printf("\tlocking       : %s\n", BOOLSTR(pService->bLocking));
      printf("\tonly user     : %s\n", BOOLSTR(pService->bOnlyUser));
      printf("\twidelinks     : %s\n", BOOLSTR(pService->bWidelinks));
      printf("\tcreate_mode   : 0%o\n", pService->iCreate_mode);
      printf("\tmax cons      : %d\n", pService->iMaxConnections);
}

/***************************************************************************
Display the contents of a single copy structure.
***************************************************************************/
static void dump_copy_map(copymap *pcopymap)
{
  if (!pcopymap)
    {
      DEBUG(0,("attempt to dump null pcopymap\n"));
      return;
    }

      if (!pcopymap->bCopyEncountered)
         printf("\tNo copy details needed.\n");
      else
      {
         printf("\tCopy details:\n");
         printf("\t\tsource      : %s\n", pcopymap->szSourceService);
         printf("\t\tavailable   : %s\n", BOOLSTR(pcopymap->bUS_available));
         printf("\t\tpath        : %s\n", BOOLSTR(pcopymap->bUS_path));
         printf("\t\tusername    : %s\n", BOOLSTR(pcopymap->bUS_username));
         printf("\t\tprintcommand: %s\n", BOOLSTR(pcopymap->bUS_printcommand));
         printf("\t\tlpqcommand  : %s\n", BOOLSTR(pcopymap->bUS_lpqcommand));
         printf("\t\tprinter name: %s\n", BOOLSTR(pcopymap->bUS_printername));
         printf("\t\thosts allow : %s\n", BOOLSTR(pcopymap->bUS_hosts_allow));
         printf("\t\thosts deny  : %s\n", BOOLSTR(pcopymap->bUS_hosts_deny));
         printf("\t\tread_only   : %s\n", BOOLSTR(pcopymap->bUS_read_only));
         printf("\t\tno_set_dir  : %s\n", BOOLSTR(pcopymap->bUS_no_set_dir));
         printf("\t\tguest_only  : %s\n", BOOLSTR(pcopymap->bUS_guest_only));
         printf("\t\tguest_ok    : %s\n", BOOLSTR(pcopymap->bUS_guest_ok));
         printf("\t\tprint_ok    : %s\n", BOOLSTR(pcopymap->bUS_print_ok));
         printf("\t\tmap_system  : %s\n", BOOLSTR(pcopymap->bUS_map_system));
         printf("\t\tmap_hidden  : %s\n", BOOLSTR(pcopymap->bUS_map_hidden));
         printf("\t\tlocking     : %s\n", BOOLSTR(pcopymap->bUS_locking));
         printf("\t\tonly user   : %s\n", BOOLSTR(pcopymap->bUS_only_user));
         printf("\t\twidelinks   : %s\n", BOOLSTR(pcopymap->bUS_widelinks));
         printf("\t\tcreate_mode : %s\n", BOOLSTR(pcopymap->bUS_create_mode));
         printf("\t\tmagic script: %s\n", BOOLSTR(pcopymap->bUS_magic_script));
         printf("\t\tmagic output: %s\n", BOOLSTR(pcopymap->bUS_magic_output));
      }
}

/***************************************************************************
Return TRUE if the passed service number is within range.
***************************************************************************/
BOOL lp_snum_ok(int iService)
{
   return (LP_SNUM_OK(iService) && pServices[iService].bAvailable);
}

/***************************************************************************
Return a pointer to the guest user account name. It would be MOST unwise 
to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_guestaccount(void)
{
   return (&(Globals.szGuestaccount[0]));
}

/***************************************************************************
Return a pointer to the global printcap name. It would be MOST unwise 
to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_printcapname(void)
{
   return (&(Globals.szPrintcapname[0]));
}

/***************************************************************************
Return a pointer to the global lock directory. It would be MOST unwise 
to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_lockdir(void)
{
   return (&(Globals.szLockDir[0]));
}

/***************************************************************************
Return a pointer to the root directory. It would be MOST unwise 
to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_rootdir(void)
{
   return (&(Globals.szRootdir[0]));
}

/***************************************************************************
Return a pointer to the default service. It would be MOST unwise 
to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_defaultservice(void)
{
   return (&(Globals.szDefaultService[0]));
}

/***************************************************************************
Return a pointer to the dfree command. It would be MOST unwise 
to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_dfree_command(void)
{
   return (&(Globals.szDfree[0]));
}

/***************************************************************************
Return whether mangled names are supported
***************************************************************************/
BOOL lp_manglednames(void)
{
   return (Globals.bMangledNames);
}

/***************************************************************************
Return whether the getwd cache should be enabled
***************************************************************************/
BOOL lp_getwdcache(void)
{
   return (Globals.bGetwdCache);
}

/***************************************************************************
Return whether read prediction should be enabled
***************************************************************************/
BOOL lp_readprediction(void)
{
   return (Globals.bReadPrediction);
}

/***************************************************************************
Return whether readraw is supported
***************************************************************************/
BOOL lp_readraw(void)
{
   return (Globals.bReadRaw);
}

/***************************************************************************
Return whether writeraw is supported
***************************************************************************/
BOOL lp_writeraw(void)
{
   return (Globals.bWriteRaw);
}

/***************************************************************************
Return a the max xmit packet size
***************************************************************************/
int lp_mangledstack(void)
{
   return (Globals.mangled_stack);
}

/***************************************************************************
Return a the max xmit packet size
***************************************************************************/
int lp_maxxmit(void)
{
   return (Globals.max_xmit);
}

/***************************************************************************
Return a the max packet size (used by readbraw)
***************************************************************************/
int lp_maxpacket(void)
{
   return (Globals.max_packet);
}

/***************************************************************************
Return the keepalive time
***************************************************************************/
int lp_keepalive(void)
{
   return (Globals.keepalive);
}

/***************************************************************************
Return the password level
***************************************************************************/
int lp_passwordlevel(void)
{
   return (Globals.pwordlevel);
}

/***************************************************************************
Return the deadtime time
***************************************************************************/
int lp_deadtime(void)
{
   return (Globals.deadtime);
}

/***************************************************************************
Return the maximum allowable protocol level
***************************************************************************/
int lp_maxprotocol(void)
{
   return (Globals.maxprotocol);
}

/***************************************************************************
Return the security level
***************************************************************************/
int lp_security(void)
{
   return (Globals.security);
}

/***************************************************************************
Return a pointer to the service name of a specified service. It would be 
MOST unwise to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_servicename(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].szService : NULL);
}

/***************************************************************************
Return a pointer to the path name of a specified service. It would be 
MOST unwise to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_pathname(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].szPath : NULL);
}

/***************************************************************************
Return a pointer to the "dont descend" list of a specified service. 
It would be MOST unwise to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_dontdescend(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].szDontdescend : NULL);
}

/***************************************************************************
Return a pointer to the username of a specified service.

NOTE: This bit below is the old behaviour - it has changed! 
#if 0
If no username is specified in the service entry then if the service does NOT
allow guest access, the service name is returned.
#endif

It would be MOST unwise to treat the pointer as anything but read-only!
***************************************************************************/
char *lp_username(int iService)
{
   char *pszTemp;

   pszTemp = NULL;
   if (LP_SNUM_OK(iService))
   {
      pszTemp = pServices[iService].szUsername;

/* NOTE: Guest account handling is now done explicitly in server.c,
   so returning NULL from this routine is quite possible */
#if 0
      if (pszTemp[0] == '\0')
        if (pServices[iService].bGuest_ok)
	  pszTemp = Globals.szGuestaccount;
        else
	  pszTemp = pServices[iService].szService;
#endif
   }
   return (pszTemp);
}

/***************************************************************************
Return a pointer to the printcommand of a specified service. It would be 
MOST unwise to treat the pointer as anything but read-only! Note that the
pointer will point to the global printcommand if no service-specific print
command has been specified. A NULL pointer indicates an error and should
always be checked for.
***************************************************************************/
char *lp_printcommand(int iService)
{
   char *pszTemp;

   pszTemp = NULL;
   if (LP_SNUM_OK(iService))
   {
      pszTemp = Globals.szPrintcommand;
      if (pszTemp[0] == '\0')
         pszTemp = NULL;

      if (pServices[iService].szPrintcommand[0] != '\0')
         pszTemp = pServices[iService].szPrintcommand;
   }
      
   return (pszTemp);
}


/***************************************************************************
Return a pointer to the lpqcommand of a specified service.
Default to the global one.
***************************************************************************/
char *lp_lpqcommand(int iService)
{
   char *pszTemp;

   pszTemp = Globals.szLpqcommand;
   if (pszTemp[0] == '\0')
     pszTemp = NULL;
   if (LP_SNUM_OK(iService))
     {
       if (pServices[iService].szLpqcommand[0] != '\0')
         pszTemp = pServices[iService].szLpqcommand;
     }
      
   return (pszTemp);
}

/***************************************************************************
Return a pointer to the printer name of a specified service. It would be 
MOST unwise to treat the pointer as anything but read-only! Note that the
pointer will point to the global printer name if no service-specific printer
name has been specified. A NULL pointer indicates an error and should
always be checked for.
***************************************************************************/
char *lp_printername(int iService)
{
   char *pszTemp;

   pszTemp = NULL;
   if (LP_SNUM_OK(iService))
   {
      pszTemp = Globals.szPrintername;
      if (pszTemp[0] == '\0')
         pszTemp = NULL;

      if (pServices[iService].szPrintername[0] != '\0')
         pszTemp = pServices[iService].szPrintername;
   }
      
   return (pszTemp);
}

/***************************************************************************
Return a pointer to the "hosts allow" list of a specified service. It would be 
MOST unwise to treat the pointer as anything but read-only! Note that the
pointer will point to the global hosts allow if no service-specific hosts allow
command has been specified. A NULL pointer indicates an error and should
always be checked for.
***************************************************************************/
char *lp_hostsallow(int iService)
{
   char *pszTemp;

   pszTemp = NULL;
   if (LP_SNUM_OK(iService))
   {
      pszTemp = Globals.szHostsallow;
      if (pszTemp[0] == '\0')
         pszTemp = NULL;

      if (pServices[iService].szHostsallow[0] != '\0')
         pszTemp = pServices[iService].szHostsallow;
   }
      
   return (pszTemp);
}

/***************************************************************************
Return a pointer to the "hosts deny" list of a specified service. It would be 
MOST unwise to treat the pointer as anything but read-only! Note that the
pointer will point to the global hosts deny if no service-specific hosts deny
command has been specified. A NULL pointer indicates an error and should
always be checked for.
***************************************************************************/
char *lp_hostsdeny(int iService)
{
   char *pszTemp;

   pszTemp = NULL;
   if (LP_SNUM_OK(iService))
   {
      pszTemp = Globals.szHostsdeny;
      if (pszTemp[0] == '\0')
         pszTemp = NULL;

      if (pServices[iService].szHostsdeny[0] != '\0')
         pszTemp = pServices[iService].szHostsdeny;
   }
      
   return (pszTemp);
}

/***************************************************************************
Return a pointer to the "magic script" list of a specified service.
***************************************************************************/
char *lp_magicscript(int iService)
{
   char *pszTemp;

   pszTemp = NULL;
   if (LP_SNUM_OK(iService))
   {
      pszTemp = Globals.szMagicScript;
      if (pszTemp[0] == '\0')
         pszTemp = NULL;

      if (pServices[iService].szMagicScript[0] != '\0')
         pszTemp = pServices[iService].szMagicScript;
   }
      
   return (pszTemp);
}

/***************************************************************************
Return a pointer to the "magic output" list of a specified service.
***************************************************************************/
char *lp_magicoutput(int iService)
{
   char *pszTemp;

   pszTemp = NULL;
   if (LP_SNUM_OK(iService))
   {
      pszTemp = Globals.szMagicOutput;
      if (pszTemp[0] == '\0')
         pszTemp = NULL;

      if (pServices[iService].szMagicOutput[0] != '\0')
         pszTemp = pServices[iService].szMagicOutput;
   }
      
   return (pszTemp);
}

/***************************************************************************
Return the bRead_only flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return TRUE here...
***************************************************************************/
BOOL lp_readonly(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bRead_only : True);
}

/***************************************************************************
Return the iCreate_mode flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return the default here...
***************************************************************************/
int lp_create_mode(int iService)
{
  return (LP_SNUM_OK(iService) ? pServices[iService].iCreate_mode : Globals.iCreate_mode);
}

/***************************************************************************
Return the max connections
***************************************************************************/
int lp_max_connections(int iService)
{
  return(LP_SNUM_OK(iService) ? pServices[iService].iMaxConnections : 0);
}

/***************************************************************************
Return the bNo_set_dir flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return TRUE here...
***************************************************************************/
BOOL lp_no_set_dir(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bNo_set_dir : True);
}

/***************************************************************************
Return the bGuest_ok flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return FALSE here...
***************************************************************************/
BOOL lp_guest_ok(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bGuest_ok : False);
}

/***************************************************************************
Return the bGuest_only flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return FALSE here...
***************************************************************************/
BOOL lp_guest_only(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bGuest_only : False);
}

/***************************************************************************
Return the bPrint_ok flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return FALSE here...
***************************************************************************/
BOOL lp_print_ok(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bPrint_ok : False);
}

/***************************************************************************
Return the bMap_hidden flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return FALSE here...
***************************************************************************/
BOOL lp_map_hidden(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bMap_hidden : False);
}

/***************************************************************************
Return the bLocking flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return the Global value here
***************************************************************************/
BOOL lp_locking(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bLocking : Globals.bLocking);
}

/***************************************************************************
Return the bOnlyUser flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return the Global value here
***************************************************************************/
BOOL lp_onlyuser(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bOnlyUser : Globals.bOnlyUser);
}

/***************************************************************************
Return the bWidelinks flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return the Global value here
***************************************************************************/
BOOL lp_widelinks(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bWidelinks : Globals.bWidelinks);
}

/***************************************************************************
Return the bMap_system flag from a specified service. If the passed service
number is not valid the results should be treated as undefined, but in a
(probably) vain hope of avoiding catastrophe, we return FALSE here...
***************************************************************************/
BOOL lp_map_system(int iService)
{
   return (LP_SNUM_OK(iService) ? pServices[iService].bMap_system : False);
}

/***************************************************************************
Discard the services array. Used internally, but available for external use
by the conscientious. Also unloads the globals.
***************************************************************************/
void lp_unload(void)
{
   bDoneGlobalSection = False;
   bInGlobalSection = False;

   free_globals();
   if (pServices != NULL)
   {
     int i;
     for (i=0;i<iNumServices;i++)
       free_service(&pServices[i]);
      free (pServices);
      pServices = NULL;
   }

   iNumServices = 0;
}

/***************************************************************************
Load the services array from the services file. Return True on success, 
False on failure.
***************************************************************************/
BOOL lp_load(char *pszFname)
{
  BOOL bRetval;
  
  bRetval = False;
  
  /* throw away any existing service details */
  lp_unload();
  init_globals();
  
  /* We get sections first, so have to start 'behind' to make up */
  iServiceIndex = -1;
  bRetval = pm_process(pszFname, do_section, do_parameter);
  
  /* finish up the last section */
  DEBUG(3,( "pm_process() returned %s\n", BOOLSTR(bRetval)));
  if (bRetval)
    if (iServiceIndex >= 0)
      bRetval = finalise_service(iServiceIndex);	   
  
  return (bRetval);
}

/***************************************************************************
Display the contents of the services array in human-readable form.
***************************************************************************/
void lp_dump(void)
{
   int iService;

   dump_globals();
   for (iService = 0; iService < iNumServices; iService++)
   {
      if (pServices[iService].szService[0] == '\0')
         break;
      dump_a_service(pServices + iService);      
      dump_copy_map(&COPYMAPS(iService));
   }
}

/***************************************************************************
Return the number of the service with the given name, or -1 if it doesn't
exist. Note that this is a DIFFERENT ANIMAL from the internal function
getservicebyname()! This works ONLY if all services have been loaded, and
does not copy the found service.
***************************************************************************/
int lp_servicenumber(char *pszServiceName)
{
   int iService;

   for (iService = iNumServices - 1; iService >= 0; iService--)
      if (strwicmp(pServices[iService].szService, pszServiceName) == 0) 
         break;

   if (iService < 0)
     DEBUG(3,("lp_servicenumber: couldn't find %s\n",pszServiceName));
   
   return (iService);
}
