/////////////////////////////////////////////////////////////////////////////// 
// 
// Monitor.cc 
// ---------- 
// Curses based monitor implementation 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2017 Bjoern Lemke 
// 
// IMPLEMENTATION MODULE 
// 
// Class: Monitor 
// 
// Description: Cursor based monitor implementation 
// 
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 


#include <stdio.h> 
#include <stdlib.h> 
#include <unistd.h> 

#include "Monitor.h"
#include "Tokenizer.h"
#include "Sleeper.h" 

#define COL_HEADER_PASSIVE 1 
#define COL_HEADER_ACTIVE 2 
#define COL_TITLE 3 
#define COL_CONTENT 4
#define COL_SELECTED 5

// advanced color map
#define COL_BLUE 6
#define COL_GREEN 7
#define COL_YELLOW 8
#define COL_RED 9
#define COL_INPUT 10

#define LEFTOFFSET 1 
#define TOPOFFSET 1 

#define KEY_TAB 9 
#define KEY_RETURN 10 
#define KEY_ESC 27 
#define KEY_DEL 127 

#define INPUT_TYPE_STRING "S" 
#define INPUT_TYPE_ID "I" 
#define INPUT_TYPE_NUMBER "N" 
#define INPUT_TYPE_PWD "P"
#define INPUT_TYPE_MENU "M" 

#define CUR_ROW_OFFSET 3 
#define CUR_COL_OFFSET 20 
#define VAL_MAX_LEN 100 
#define VAL_MAX_NUM 20 
#define OK_COL_POS 6 
#define ABORT_COL_POS 16 

#define MSEC_SLEEPDELAY 300

Monitor::Monitor(const Chain& attrSepToken, const Chain& listSepToken, const Chain& valSepToken) : SigHandler() 
{ 

    _attrSepToken = attrSepToken;
    _listSepToken = listSepToken;
    _valSepToken = valSepToken;
    
    _menuSelected = 0;
    
#ifndef LFCNOCURSES

    /*  Initialize ncurses  */ 
    initscr();
    curs_set(0); 
    start_color(); 

    init_pair(COL_HEADER_PASSIVE,  COLOR_CYAN,     COLOR_BLACK); 
    init_pair(COL_HEADER_ACTIVE,  COLOR_YELLOW,     COLOR_BLACK); 
    init_pair(COL_TITLE,  COLOR_YELLOW,     COLOR_BLACK); 
    init_pair(COL_CONTENT,  COLOR_WHITE,     COLOR_BLACK);
    init_pair(COL_SELECTED,  COLOR_GREEN,     COLOR_BLACK);

    init_pair(COL_INPUT,  COLOR_GREEN,  COLOR_BLACK);

    init_pair(COL_BLUE,  COLOR_BLUE,     COLOR_BLACK);
    init_pair(COL_RED,  COLOR_RED,     COLOR_BLACK);
    init_pair(COL_YELLOW,  COLOR_YELLOW,     COLOR_BLACK);
    init_pair(COL_GREEN,  COLOR_GREEN,     COLOR_BLACK); 
    
    noecho();                  /*  Turn off key echoing                 */ 
    keypad(stdscr, TRUE);
    // keypad(_mainwin, TRUE);     /*  Enable the keypad for non-char keys  */ 

    // getmaxyx(_mainwin, _mainy, _mainx); 
    timeout(0); 

    init();
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

#ifndef HAVE_MINGW 
    install(SIGWINCH); 
#endif
    
} 

Monitor::~Monitor() 
{
#ifndef LFCNOCURSES
    delwin(stdscr); 
    endwin(); 
    refresh();
#endif
} 


void Monitor::sigCatch(int sig) 
{ 

    try 
    {	
#ifndef LFCNOCURSES
	
        clear();
	
	endwin();	
	initscr();

	// Sleeper::milliSleep(MSEC_SLEEPDELAY);
	
	// Needs to be called after an endwin() so ncurses will initialize 
	// itself with the new terminal dimensions. 
	// refresh(); 
	clear();        		
	refresh();

#endif
	// we nee to add a delay, otherwise signal handling can lead to core dumps
	

	// refreshHeader(); 

#ifndef HAVE_MINGW 
	install(SIGWINCH);
#endif
	
    } 
    catch ( Exception e) 
    { 
        Chain msg; 
        e.pop(msg); 
        cout << msg << endl; 
    }

    
}; 


void Monitor::regMenu(const Chain& name, int id) 
{ 
    _menu.Insert(MenuItem(name, id)); 
} 

void Monitor::regShortCut(char c, int code) 
{ 
    _scList.Insert(ShortCut(c, code)); 
}

int Monitor::showHeader() 
{
#ifndef LFCNOCURSES
    clear(); 

    
    bool isSelected=false; 

    while( isSelected == false ) 
    { 
	bool doRefresh = true;

	wtimeout(stdscr, MSEC_SLEEPDELAY);
        int c = wgetch(stdscr); 

        switch(c) 
        { 
        case KEY_RIGHT: 
            if ( _menuSelected < _menu.Size()-1 ) 
                _menuSelected++; 
            else 
                _menuSelected = 0;
            break; 
        case KEY_LEFT: 
            if ( _menuSelected > 0  ) 
                _menuSelected--; 
            else 
                _menuSelected = _menu.Size()-1; 
            break; 
        case KEY_RETURN: 
        { 
            isSelected = true; 
            break; 
        }
	case ERR:
	{
	    // mvwprintw(stdscr, 1, 1, "Timeout %d", c); 
	    // timeout
	    break;
	}
	default:
	{
	    doRefresh=false;
	    break;
        }
        } 

	if ( doRefresh )
	    refreshHeader(); 
        
    }
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return _menu[_menuSelected].getId(); 
} 

void Monitor::refreshHeader() 
{
    
#ifndef LFCNOCURSES

    
    int colno=LEFTOFFSET; 
    attroff(A_REVERSE); 

    MenuItem *pMI = _menu.First(); 
    int idx=0; 
    while ( pMI ) 
    {
	
        if ( _menuSelected == idx ) 
        { 
            attron(A_REVERSE); 
            color_set(COL_HEADER_ACTIVE, NULL); 
        } 
        else 
        { 
            color_set(COL_HEADER_PASSIVE, NULL); 
        }
	// mvwprintw(stdscr, 1, 1, (char*)Chain(COLS)) ; 
        mvprintw(0, colno, (char*)pMI->getName()); 
        colno += pMI->getName().length() + 1; 
        attroff(A_REVERSE); 
	
        idx++; 
        pMI=_menu.Next(); 
    } 
    refresh();
    
#endif
} 

void Monitor::showInfoBox(const Chain& title, const Chain& msg, int width) 
{ 
#ifndef LFCNOCURSES
    
    int height; 
    Chain formatedMsg; 
    formatMsg(msg, formatedMsg, width-2, height); 

    height+=4; 

    int c; 

    int startx = (COLS - width) / 2; 
    int starty = (LINES - height) / 2; 

    WINDOW *infowin = newwin(height, width, starty, startx); 

    noecho(); 
    
    keypad(infowin, TRUE); 

    wattron(infowin, A_BOLD); 
    mvwprintw(infowin, 1, 1, "%s", (char*)title); 
    wattroff(infowin, A_BOLD); 
    box(infowin, 0, 0); 

    Tokenizer t(formatedMsg, Chain("\n")); 
    Chain l; 
    int lno=3; 
    while ( t.nextToken(l)) 
    { 
       mvwprintw(infowin, lno, 1, "%s", (char*)l); 
       lno++; 
    } 

    // wrefresh(infowin); 

    // any input to continue 

    c = wgetch(infowin); 
    switch(c) 
    { 
    case 'Y': 
        break; 
    case 'N': 
        break; 
    } 

    delwin(infowin);
    
    clear();    
    refreshHeader();
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif

} 

int Monitor::showSelectBox(const Chain& title, const ListT<Chain>& selectList, int posx, int posy)
{
#ifndef LFCNOCURSES
    int width = title.length() + 5;
    Chain *pS = selectList.First();
    while ( pS )
    {
	width = pS->length() + 1 > width ? pS->length() + 1 : width;   
	pS = selectList.Next();
    }
    int height = selectList.Size() + 4;
 
    int highlight=1;
    int choice = 0;
    int c;
    
	
    bool showIt=true;

    WINDOW* menuwin = 0;
    
    while( showIt )
    {

	int startx = posx; 
	int starty = posy; 
	if ( startx == 0 && starty == 0 ) 
	{
	    if ( COLS >= width ) 
		startx = (COLS - width) / 2;
	    if ( LINES >= height ) 
		starty = (LINES - height) / 2; 
	}
	
	if ( menuwin == 0 )
	    menuwin = newwin(height, width, starty, startx);

	keypad(menuwin, TRUE);
	printMenu(menuwin, title, selectList, highlight);
    
	noecho(); 

	wtimeout(menuwin, MSEC_SLEEPDELAY); 	
	c = wgetch(menuwin);
	switch(c)
	{	
	case KEY_UP:
	{
	    if (highlight == 1)
		highlight = selectList.Size();
	    else
		--highlight;
	    break;
	}
	case KEY_DOWN:
	{
	    if(highlight == selectList.Size() )
		highlight = 1;
	    else 
		++highlight;
	    break;
	}
	case KEY_RESIZE:
	{
	    delwin(menuwin);
	    refreshHeader();
	    menuwin=0;
	    break;  
	}

	case KEY_RETURN:
	{
	    showIt = false;
	    choice = highlight;
	    break;
	}
	case KEY_ESC:
	{
	    showIt = false;
	    break;
	}

	}

	// printf("You chose choice %d", choice);
	printMenu(menuwin, title, selectList, highlight);
    }

    if ( menuwin )
	delwin(menuwin);

    clear();
    refresh();
    refreshHeader();
    
    if ( choice > 0 )
    {
	Chain key, value;
	getKeyValue(selectList[highlight-1], key, value);
	return value.asInteger();
    }

#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return 0;	
}

#ifndef LFCNOCURSES

void Monitor::printMenu(WINDOW* win, const Chain& title, const ListT<Chain>& menuList, int highlight)
{
    int x, y, i;	
    wattron(win, A_BOLD);
    mvwprintw(win, 1, 1, "%s", (char*)title);
    wattroff(win, A_BOLD);
    x = 2;
    y = 3;
    box(win, 0, 0);
    for(i = 0; i < menuList.Size(); ++i)
    {	
	Chain key, value;
	getKeyValue(menuList[i], key, value);
	
	if (highlight == i + 1) /* High light the present choice */
	{	
	    wattron(win, A_REVERSE);
	    mvwprintw(win, y, x, "%s", (char*)key);
	    wattroff(win, A_REVERSE);
	}
	else
	    mvwprintw(win, y, x, "%s", (char*)key);
	++y;
    }

    wrefresh(win);
}

#endif

int Monitor::showFormBox(const Chain& title, const ListT<Chain>& attrList, ListT<Chain>& valList)
{
    int returnVal = 0;
    
#ifndef LFCNOCURSES

    int height = attrList.Size() + CUR_ROW_OFFSET + 3; 

    int c; 
    char inputArray[VAL_MAX_NUM][VAL_MAX_LEN]; 

    // init array
    for(int i = 0; i < VAL_MAX_NUM; i++) 
	for(int j = 0; j < VAL_MAX_LEN; j++) 
	    inputArray[i][j]=0; 
    

    int width = 0;
    for(int i = 0; i < attrList.Size(); ++i) 
    {
	
	Chain attr, type, value;
	int maxLen;
	getAttrTypeValue(attrList[i], attr, type, maxLen, value);
	    
	if ( (char*)value ) 
	{

	    if ( type == Chain(INPUT_TYPE_MENU) ) 
	    { 
		Tokenizer t1(value, _listSepToken);	
		Chain menuItem;
		if ( t1.nextToken(menuItem))
		{
		    Tokenizer t2(menuItem, _valSepToken);
		    Chain v;
		    if ( t2.nextToken(v))
			value = v;
		}
	    }
	    
            for(int j = 0; j < value.length(); j++) 
                inputArray[i][j]=value[j];	    
            for(int j = value.length(); j < VAL_MAX_LEN; j++) 
                inputArray[i][j]=0; 
	} 
	else 
	{ 
	    inputArray[i][0]=0; 
	}

	if ( width < maxLen + attr.length() )
	    width = maxLen + attr.length();
    } 

    width+=20;
    
    curs_set(1);
    
    // printf("\e]12;%s\a", "white"); 

    int curRow = CUR_ROW_OFFSET; 
    int curCol = CUR_COL_OFFSET; 

    bool showMask=true;

    WINDOW *maskwin = 0;
    
    while( showMask ) 
    { 

	int startx = 0, starty = 0;
	if ( COLS >= width ) 
	    startx = (COLS - width) / 2;
	if ( LINES >= height ) 
	    starty = (LINES - height) / 2; 

	if ( maskwin == 0 )
	    maskwin = newwin(height, width, starty, startx);
	
	noecho(); 

	wcolor_set(maskwin, COL_CONTENT, NULL);
	keypad(maskwin, TRUE); 
	
	wattron(maskwin, A_BOLD); 
	mvwprintw(maskwin, 1, 1, "%s", (char*)title); 
	wattroff(maskwin, A_BOLD); 
	int x = 2; 
	int y = CUR_ROW_OFFSET; 
	box(maskwin, 0, 0); 
	for(int i = 0; i < attrList.Size(); ++i) 
	{
	    
	    Chain attr, type, value;
	    int maxLen;
	    getAttrTypeValue(attrList[i], attr, type, maxLen, value);
	    
	    mvwprintw(maskwin, y, x, "%s", (char*)attr);
	    mvwprintw(maskwin, y, CUR_COL_OFFSET - 2, ":"); 

	    if ( type == Chain(INPUT_TYPE_MENU) ) 
	    { 
		wcolor_set(maskwin, COL_TITLE, NULL); 

		wattron(maskwin, A_BOLD);		

		// cleanup first
		int j = 0;
		while ( j < maxLen )
		{
		    mvwprintw(maskwin, y, CUR_COL_OFFSET + j, "%c", ' ');
		    j++;
		}

		mvwprintw(maskwin, y, CUR_COL_OFFSET, (char*)inputArray[i]);

		wattroff(maskwin, A_BOLD); 
		wcolor_set(maskwin, COL_CONTENT, NULL); 
	    } 
	    else 
	    { 
		
		wattron(maskwin, A_UNDERLINE); 
		wcolor_set(maskwin, COL_INPUT, NULL);
		for(int j = 0; j < maxLen; ++j)
		    mvwprintw(maskwin, y, CUR_COL_OFFSET + j, " ");	    
		if ( inputArray[i] )
		{
		    if ( type == Chain(INPUT_TYPE_PWD) )
		    {
			int j=0;
			while ( inputArray[i][j] != 0 )
			{
			    mvwprintw(maskwin, y, CUR_COL_OFFSET + j, "%c", '*');
			    j++;
			}
		    }
		    else
			mvwprintw(maskwin, y, CUR_COL_OFFSET, (char*)inputArray[i]);
		}
		
		wattroff(maskwin, A_UNDERLINE); 
		wcolor_set(maskwin, COL_CONTENT, NULL);
	    }
	    
	    ++y; 
	} 
	y++; 
	wattron(maskwin, A_BOLD); 
	mvwprintw(maskwin, y, x, "Ok [ ] Abort [ ]"); 
	wattroff(maskwin, A_BOLD); 
	
	wmove(maskwin, curRow, curCol); 

	wtimeout(maskwin, MSEC_SLEEPDELAY);	
        c = wgetch(maskwin); 
        switch(c) 
        { 
        case KEY_UP: 
            int prevCurRow, prevCurCol; 
            prevCursorPos(attrList.Size(), curRow, curCol, prevCurRow, prevCurCol); 
            curRow = prevCurRow; 
            curCol = prevCurCol; 
            wmove(maskwin, curRow, curCol); 
            break; 
        case KEY_DOWN: 
        case KEY_TAB: 
            int nextCurRow, nextCurCol; 
            nextCursorPos(attrList.Size(), curRow, curCol, nextCurRow, nextCurCol); 
            curRow = nextCurRow; 
            curCol = nextCurCol; 
            wmove(maskwin, curRow, curCol); 
            break; 
        case KEY_LEFT: 
            if ( curRow < attrList.Size() + CUR_ROW_OFFSET 
                 && curCol > CUR_COL_OFFSET ) 
                curCol--; 
            wmove(maskwin, curRow, curCol); 
            break; 
        case KEY_RIGHT: 
            if ( curRow < attrList.Size() + CUR_ROW_OFFSET 
                 && curCol < CUR_COL_OFFSET + VAL_MAX_LEN ) 
		curCol++; 
            wmove(maskwin, curRow, curCol); 
            break; 
        case KEY_RETURN: 
        { 
            if ( curRow == attrList.Size() + CUR_ROW_OFFSET + 1 ) 
            { 
                if ( curCol == OK_COL_POS ) 
                    returnVal = 1; 
                else if ( curCol == ABORT_COL_POS ) 
                    returnVal = 0; 
                showMask=false; 
            }
	    else 
            { 
                Chain attr, type, value;
		int maxLen;
                getAttrTypeValue(attrList[curRow-CUR_ROW_OFFSET], attr, type, maxLen, value); 
                if ( type == Chain(INPUT_TYPE_MENU)) 
                {
		    
		    Tokenizer t(value, _listSepToken);

		    ListT<Chain> menu; 
		    Chain menuItem;
		    while ( t.nextToken(menuItem))
		    {
			menu.Insert(menuItem); 
		    }

		    int r = showSelectBox("Selection", menu, startx + CUR_COL_OFFSET, starty + curRow); 

		    Chain selectedKey;
		    getListKey(value, Chain(r), selectedKey);
		    
		    
		    if ( (char*)selectedKey ) 
		    { 
			for(int j = 0; j < selectedKey.length(); j++) 
			    inputArray[curRow-CUR_ROW_OFFSET][j]=selectedKey[j];
			for(int j = selectedKey.length(); j < VAL_MAX_LEN; j++) 
			    inputArray[curRow-CUR_ROW_OFFSET][j]=0; 
		    } 
		    else 
		    { 
			inputArray[curRow-CUR_ROW_OFFSET][0]=0; 
		    }
                } 
            } 
            break; 
        } 
        case KEY_DEL: 
        {
	    Chain attr, type, value;
	    int maxLen;
	    getAttrTypeValue(attrList[curRow-CUR_ROW_OFFSET], attr, type, maxLen, value); 
	    if ( type != Chain(INPUT_TYPE_MENU))
	    {
		if ( curCol > CUR_COL_OFFSET ) 
		    curCol--; 
		mvwprintw(maskwin, curRow, curCol, "%c", ' ');
		for ( int i=curCol-CUR_COL_OFFSET; i<VAL_MAX_LEN; i++)
		    inputArray[curRow-CUR_ROW_OFFSET][i] = 0; 
		wmove(maskwin, curRow, curCol);
	    }
	    break; 
	} 
        case KEY_ESC: 
        { 
            returnVal = 0; 
            showMask=false; 
            break; 
        }
	case KEY_RESIZE:
	{
	    delwin(maskwin);
	    refreshHeader();
	    maskwin=0;
	    break;  
	}
	case ERR:
	{
	    // timeout
	    break;
	}
        default: 
        { 	    
            if ( curRow < attrList.Size() + CUR_ROW_OFFSET ) 
            { 
		
		Chain attr, type, value;
		int maxLen;
		getAttrTypeValue(attrList[curRow-CUR_ROW_OFFSET], attr, type, maxLen, value);
		
                bool inputAccepted=false; 

		if ( curCol-CUR_COL_OFFSET < maxLen )
		{	       		
		    if ( type == Chain(INPUT_TYPE_STRING) 
			 && ( isalnum (c) 
			      || c == '_' 
			      || c == '-' 
			      || c == '/' 
			      || c == '.' 
			      || c == '*' 
			      || c == '[' 
			      || c == ']' )) 
			inputAccepted=true; 
		    else if ( type == Chain(INPUT_TYPE_ID) 
			      && ( isalnum (c) 
				   || c == '_' )) 
			inputAccepted=true; 
		    else if ( type == Chain(INPUT_TYPE_NUMBER) && isdigit (c) ) 
			inputAccepted=true; 
		    else if ( type == Chain(INPUT_TYPE_PWD) ) 
			inputAccepted=true;
		}
				
                if ( inputAccepted ) 
                {
		    
                    if ( type == Chain(INPUT_TYPE_PWD) ) 
                        mvwprintw(maskwin, curRow, curCol, "%c", '*'); 
                    else 
                        mvwprintw(maskwin, curRow, curCol, "%c", c); 
                    inputArray[curRow-CUR_ROW_OFFSET][curCol-CUR_COL_OFFSET] = c; 
                    curCol++; 
                    wmove(maskwin, curRow, curCol); 
                } 
            } 
        } 
        } 
        wrefresh(maskwin); 

        // printf("You chose choice %d", choice); 
        // printMask(maskwin, title, attrList, valList);

	// curs_set(0);
	
	// mvwprintw(stdscr, 2, 2, "Char = %d", c);
	// refresh();
	// refreshHeader();
		
    } 

    if ( maskwin )
	delwin(maskwin);

    clear();
    refresh();
    refreshHeader();

    // store values in valList 

    valList.Empty();
    for(int i = 0; i < attrList.Size(); ++i) 
    {
	Chain attr, type, value;
	int maxLen;
	getAttrTypeValue(attrList[i], attr, type, maxLen, value); 

	if ( type == Chain(INPUT_TYPE_MENU)) 
	{
	    Chain menuValue;
	    getListValue(value, Chain(inputArray[i]), menuValue);
	    valList.Insert(menuValue);		
	}
	else
	{		    
	    if ( inputArray[i][0] ) 
		valList.Insert(Chain(inputArray[i]).cutTrailing(" ")); 
	    else 
		valList.Insert(Chain()); 	    
	}
    }
    
#else
    throw Exception(EXLOC, "Curses not supported");
#endif
    
    return returnVal; 
} 

void Monitor::setSelectedRow(int row)
{
    _rowSelected = row;
}

int Monitor::getSelectedRow() const
{
    return _rowSelected;
}

int Monitor::showTableBox(const Chain& title, ListT<Chain>& schema,  const ListT<ListT<Chain> >& table, int timeout, const ListT<TableColor>& colorMap,  bool doInit, bool isSelectable )
{

    int ret = MON_TIMEOUT;
    
#ifndef LFCNOCURSES
    int c = 0;

    bool modFlag = false;
        
    bool showTable = true;

    if ( doInit || isSelectable == false)
	_rowSelected=1;
    
    int numCol=0;
    Chain *pA = schema.First();
    while ( pA )
    {
	Chain attr, attrLen;
	getKeyValue(*pA, attr, attrLen);
	numCol = numCol + attrLen.asInteger();	    
	pA = schema.Next();
    }

    WINDOW* tabwin = 0;
    
    tabwin = newwin(table.Size() + 3, numCol + 4, 2, 2);
    noecho();		
    keypad(tabwin, TRUE);
    
    box(tabwin, 0, 0);
    
    while ( showTable )
    {      	    	   	    
	switch(c)
	{	
	case KEY_UP:
	{
	    if ( isSelectable )
	    {
		if (_rowSelected == 1)
		    _rowSelected = table.Size();
		else
		    --_rowSelected;
	    }
	    break;
	}
	case KEY_DOWN:
	{
	    if ( isSelectable )
	    {
		if( _rowSelected == table.Size() )
		    _rowSelected = 1;
		else 
		    ++_rowSelected;
	    }
	    break;
	}
	case KEY_RETURN:
	{
	    modFlag=true;
	    showTable=false;
	    ret = MON_SELECT;
	    break;
	}
	case KEY_ESC:
	{
	    modFlag=true;
	    showTable=false;
	    clear();
	    ret = MON_LEAVE;
	    break;
	}
	default:
	{
	    ShortCut* pSC = _scList.First();
	    while ( pSC && modFlag == false)
	    {
		if ( c == pSC->getShortCut() )
		{
		    modFlag=true;
		    showTable=false;
		    ret = pSC->getCode();		    
		}
		pSC = _scList.Next();
	    }
	}
	}

	
	int rowno=1;
	int colno=2;

	wcolor_set(tabwin, COL_TITLE, NULL);	
	wattron(tabwin, A_BOLD);

	mvwprintw(tabwin, 0, 2, "%s", (char*)title);
		
	pA = schema.First();
	while ( pA )
	{
	    Chain attr, attrLen;
	    getKeyValue(*pA, attr, attrLen);

	    mvwprintw(tabwin, rowno, colno, "%s", (char*)attr);

	    colno = colno + attrLen.asInteger();
	    
	    pA = schema.Next();
	}

	rowno++;

	wattroff(tabwin, A_BOLD);
	wcolor_set(tabwin, COL_CONTENT, NULL);
	
	ListT<Chain> *pRow = table.First();
		
	int tabRow=1;
	while ( pRow )
	{
	    
	    colno=2;

	    int rowColor = COL_CONTENT;
	    
	    if ( _rowSelected == tabRow && isSelectable )
	    {
		wcolor_set(tabwin, COL_SELECTED, NULL);
		wattron(tabwin, A_REVERSE);
		rowColor = COL_SELECTED;
		// wattron(tabwin, A_BOLD);
	    }

	    Chain *pA = schema.First();
	    Chain *pVal = pRow->First();
	    while ( pA && pVal )
	    {		       

		TableColor *pTC = colorMap.Find(TableColor(*pVal));	       
		if ( pTC )
		{
		    switch ( pTC->getCode() )
		    {
		    case MON_BLUE:
			wcolor_set(tabwin, COL_BLUE, NULL);
			break;
		    case MON_GREEN:
			wcolor_set(tabwin, COL_GREEN, NULL);
			break;
		    case MON_RED:
			wcolor_set(tabwin, COL_RED, NULL);
			break;
		    case MON_YELLOW:
			wcolor_set(tabwin, COL_YELLOW, NULL);
			break;			
		    }
		}
		
		mvwprintw(tabwin, rowno, colno, "%s", (char*)*pVal);
		
		Chain attr, attrLen;
		getKeyValue(*pA, attr, attrLen);

		colno = colno + attrLen.asInteger();
		
		wcolor_set(tabwin, rowColor, NULL);
		
		pVal = pRow->Next();
		pA = schema.Next();
	    }

	    if ( _rowSelected == tabRow && isSelectable )
	    {		
		wcolor_set(tabwin, COL_CONTENT, NULL);
		wattroff(tabwin, A_REVERSE); 		
		// wattroff(tabwin, A_BOLD);
	    }
	    
	    tabRow++;
	    rowno++;
	    pRow = table.Next();
	}
	
	wrefresh(tabwin);

	if ( modFlag == false )
	{
	    wtimeout(tabwin, timeout);
	    c = wgetch(tabwin);
	    if ( c == -1 )
		showTable=false;	   
	}
	else
	{
	    c = 0;
	    modFlag = false;
	}
    }

    delwin(tabwin);

    // clear();
    refresh();
    refreshHeader();

#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return ret;
}

int Monitor::showAttributeBox(int keyLen, int valLen, const ListT< ListT<Chain> >& keyValueSpec, int timeout)
{

    int ret = MON_TIMEOUT;
    
#ifndef LFCNOCURSES
    int c = 0;

    bool modFlag = false;
        
    bool showTable = true;

    clear();
    refreshHeader();

    while ( showTable )
    {

	WINDOW* tabwin[10];
           	    	   	    
	switch(c)
	{	
	case KEY_RETURN:
	case KEY_ESC:
	{
	    modFlag=true;
	    showTable=false;
	    ret = MON_LEAVE;
	    break;
	}
	default:
	{
	    ShortCut* pSC = _scList.First();
	    while ( pSC && modFlag == false)
	    {
		if ( c == pSC->getShortCut() )
		{
		    modFlag=true;
		    showTable=false;
		    ret = pSC->getCode();		    
		}
		pSC = _scList.Next();
	    }
	}
	}

	ListT<Chain> *pKVL = keyValueSpec.First();

	int widx=0;
	int woff=2;
	while ( pKVL )
	{
	    tabwin[widx] = newwin(pKVL->Size() + 2, keyLen + valLen, woff, 2);

	    woff += pKVL->Size() + 2;
	    
	    noecho();		
	    keypad(tabwin[widx], TRUE);
	
	    box(tabwin[widx], 0, 0);
	
	    wcolor_set(tabwin[widx], COL_TITLE, NULL);
	
	    // wattron(tabwin, A_BOLD);

	    int rowno=1;
	    Chain *pKV = pKVL->First();
	    while ( pKV )
	    {
		Chain key, value;
		getKeyValue(*pKV, key, value);

		int colno=1;
		
		wcolor_set(tabwin[widx], COL_TITLE, NULL);
		mvwprintw(tabwin[widx], rowno, colno, "%s", (char*)key);

		colno += keyLen;
		
		wcolor_set(tabwin[widx], COL_CONTENT, NULL);
		mvwprintw(tabwin[widx], rowno, colno, "%s", (char*)value);
		
		rowno++;
		pKV = pKVL->Next();
	    }
	
	    wrefresh(tabwin[widx]);
	    widx++;
	    pKVL = keyValueSpec.Next();
	}
	    	
	if ( modFlag == false )
	{
	    wtimeout(tabwin[0], timeout);
	    c = wgetch(tabwin[0]);
	    if ( c == -1 )
		showTable=false;	  
	}
	else
	{
	    c = 0;
	    modFlag = false;
	}

	// wtimeout(tabwin, timeout);
	// c = wgetch(tabwin);

	for ( int i=0; i<=widx; i++)
	    delwin(tabwin[widx]);

	clear();
	refreshHeader();

    }

#else
    throw Exception(EXLOC, "Curses not supported");
#endif

    return ret;
}



void Monitor::prevCursorPos(int attrListSize, int curRow, int curCol, int& prevCurRow, int& prevCurCol) 
{ 
    int okRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int okCol = OK_COL_POS; 

    int abortRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int abortCol = ABORT_COL_POS; 

    if ( curRow == CUR_ROW_OFFSET ) 
    { 
        prevCurRow = abortRow; 
        prevCurCol = abortCol; 
    } 
    else if ( curRow == okRow && curCol == okCol ) 
    { 
        prevCurRow = attrListSize + CUR_ROW_OFFSET - 1; 
        prevCurCol = CUR_COL_OFFSET; 
    } 
    else if ( curRow == abortRow && curCol == abortCol ) 
    { 
        prevCurRow = okRow; 
        prevCurCol = okCol; 
    } 
    else 
    { 
        prevCurRow = curRow-1; 
        prevCurCol = CUR_COL_OFFSET; 
    } 
} 


void Monitor::nextCursorPos(int attrListSize, int curRow, int curCol, int& nextCurRow, int& nextCurCol) 
{ 
    int okRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int okCol = OK_COL_POS; 

    int abortRow = attrListSize + CUR_ROW_OFFSET + 1; 
    int abortCol = ABORT_COL_POS; 

    if ( curRow == attrListSize + CUR_ROW_OFFSET - 1) 
    { 
        nextCurRow = okRow; 
        nextCurCol = okCol; 
    } 
    else if ( curRow == okRow && curCol == okCol ) 
    { 
        nextCurRow = abortRow; 
        nextCurCol = abortCol; 
    } 
    else if ( curRow == abortRow && curCol == abortCol ) 
    { 
        nextCurRow = CUR_ROW_OFFSET; 
        nextCurCol = CUR_COL_OFFSET; 
    } 
    else 
    { 
        nextCurRow = curRow+1; 
        nextCurCol=CUR_COL_OFFSET; 
    } 
} 

void Monitor::getKeyValue(const Chain& s, Chain& key, Chain& value)
{
    Tokenizer t(s, _valSepToken);
    Chain v;
    t.nextToken(key);
    t.nextToken(value);
}

void Monitor::getAttrTypeValue(const Chain& s, Chain& attr, Chain& type, int& maxLen, Chain& value)
{
    Tokenizer t(s, _attrSepToken);
    Chain v;
    t.nextToken(attr);
    t.nextToken(type);

    Chain maxLenStr;
    t.nextToken(maxLenStr);
    maxLen = maxLenStr.asInteger();
    
    t.nextToken(value);
}


void Monitor::getListValue(const Chain& s, const Chain& key, Chain& value)
{    
    Tokenizer t1(s, _listSepToken);	
    Chain menuItem;
    while ( t1.nextToken(menuItem))
    {
	Tokenizer t2(menuItem, _valSepToken);
	Chain menuKey;
	while ( t2.nextToken(menuKey))
	{
	    if ( menuKey == key )
	    {
		t2.nextToken(value);
		return;
	    }
	}
    }
}

void Monitor::getListKey(const Chain& s, const Chain& value, Chain& key)
{
    Tokenizer t1(s, _listSepToken);	
    Chain menuItem;
    while ( t1.nextToken(menuItem))
    {
	Tokenizer t2(menuItem, _valSepToken);
	Chain menuKey;
	Chain menuValue;
	
	if ( t2.nextToken(key) )
	{
	    if ( t2.nextToken(menuValue) )
	    {		
		if ( menuValue == value )
		{
		    return;
		}	
	    }
	}
    }
}


void Monitor::formatMsg(const Chain& msg, Chain& formatedMsg, int width, int& height) 
{ 
   Tokenizer t(msg, Chain(" ")); 
   Chain tok; 

   int curLen=0; 
   height=1; 
   while ( t.nextToken(tok) ) 
   { 
      curLen += tok.length(); 
      if ( curLen > width ) 
      { 
          formatedMsg += Chain("\n"); 
          curLen=tok.length(); 
          height++; 
      } 
      formatedMsg += tok + Chain(" "); 
   } 
} 
