/////////////////////////////////////////////////////////////////////////////// 
// 
// Pager.cc 
// ---------- 
// Curses based pager implementation 
// 
// Design and Implementation by Bjoern Lemke 
// 
// (C)opyright 2017 Bjoern Lemke 
// 
// IMPLEMENTATION MODULE 
// 
// Class: Monitor 
// 
// Description: Cursor based pager implementation 
// 
// Status: CLEAN 
// 
/////////////////////////////////////////////////////////////////////////////// 

#include <stdio.h> 
#include <stdlib.h> 
#include <unistd.h> 

#include "Pager.h" 
#include "Tokenizer.h" 

#define COL_HEADER_PASSIVE 1 
#define COL_HEADER_ACTIVE 2 
#define COL_TITLE 3 
#define COL_CONTENT 4 
#define COL_SELECTED 5

#define LEFTOFFSET 1 
#define TOPOFFSET 1 

#define KEY_TAB 9 
#define KEY_RETURN 10 
#define KEY_ESC 27 
#define KEY_DEL 127 

#define CUR_ROW_OFFSET 3 
#define CUR_COL_OFFSET 20 
#define VAL_MAX_LEN 30 
#define VAL_MAX_NUM 20 
#define OK_COL_POS 6 
#define ABORT_COL_POS 16 

#define MSEC_SLEEPDELAY 300

Pager::Pager() : SigHandler() 
{
    
#ifndef LFCNOCURSES
    /*  Initialize ncurses  */ 
    initscr(); 
    curs_set(0); 
    start_color(); 

    init_pair(COL_HEADER_PASSIVE,  COLOR_CYAN,     COLOR_WHITE); 
    init_pair(COL_HEADER_ACTIVE,  COLOR_YELLOW,     COLOR_WHITE); 
    init_pair(COL_TITLE,  COLOR_BLACK,     COLOR_WHITE); 
    init_pair(COL_CONTENT,  COLOR_BLACK,     COLOR_WHITE); 
    init_pair(COL_SELECTED,  COLOR_BLACK,     COLOR_WHITE);

    noecho();                  /*  Turn off key echoing                 */ 
    keypad(stdscr, TRUE); 
    timeout(0); 

    wbkgd(stdscr, COLOR_PAIR(COL_CONTENT));
    
    init(); 
#ifndef HAVE_MINGW 
    install(SIGWINCH);
#endif

#else
    throw Exception(EXLOC, "Curses not supported");
#endif

} 

Pager::~Pager() 
{ 
#ifndef LFCNOCURSES 
#endif 
} 


void Pager::sigCatch(int sig) 
{ 

    try 
    { 
#ifndef LFCNOCURSES 

        clear(); 

        endwin(); 
        initscr(); 

	_rows = LINES;
	_cols = COLS;
    
        clear(); 
        refresh(); 
#endif

#ifndef HAVE_MINGW 
        install(SIGWINCH); 
#endif
	
    } 
    catch ( Exception e) 
    { 
        Chain msg; 
        e.pop(msg); 
        cout << msg << endl; 
    } 


}; 


void Pager::managePage(const Chain& title, const ListT< ListT<Chain> >& outData, const ListT<Chain>& schema) 
{

    _schema = schema;
    _title = title;
    
#ifndef LFCNOCURSES 
        
    _rows = LINES;
    _cols = COLS;
	
    int i, ch = 0; 
    unsigned int start = 0; 
    unsigned int headerSize = 1; 
    unsigned int footerSize = 1; 
    
    do 
    { 	
        switch (ch) 
        { 
	case KEY_UP :
	{
	    if ( start > 0 ) 
		start--; 
	    break;
	}
	case KEY_DOWN :
	{
	    if ( outData.Size() >= (_rows - headerSize - footerSize) && start < (outData.Size() - (_rows - headerSize - footerSize))) 
		start++; 
               break;
	}
	case 'n' : 
	case ' ' :
	{
	    if(start < (outData.Size() - (_rows - headerSize - footerSize))) start+=(_rows-headerSize-footerSize); 
	    break;
	}
	case 'b' :
	{
	    if ( start > _rows ) 
		start-=_rows; 
	    else 
		start=0; 
	    break;
	}
	case KEY_RESIZE:
	{
	    _rows = LINES;
	    _cols = COLS;

	    writeHeader();
	    refresh();
	    
	    break;  
	}
	default:
	{
	    break;
	}
	
	} 
	
	writeHeader(); 
	wcolor_set(stdscr, COL_CONTENT, NULL);
	
	for(i = 0; (i< _rows - headerSize-footerSize) && (i + start < outData.Size()); i++) 
	{ 
	    int col=0;
	    Chain *pAttr = _schema.First();
	    int j=0;
	    while ( pAttr ) 
	    {
		Chain attr;
		int len;
		getAttrSpec(*pAttr, attr, len); 
		
		mvprintw(i+headerSize, col, "%s", (char*)(outData[i+start][j])); 
		col += len;
		j++;
		pAttr = _schema.Next();
	    } 	    
	} 
	
	writeFooter(start + 1, start + i, outData.Size()); 
	refresh(); 
	
	wtimeout(stdscr, MSEC_SLEEPDELAY); 
	ch = wgetch(stdscr);
	
    } 
    while( ch != 'q' );             // continue until user hits q 
        
    refresh(); 
    clear();
    endwin();

#else 
    cout << "Curses not enabled" << endl; 
#endif 
    return; 
} 

void Pager::writeHeader() 
{ 
#ifndef LFCNOCURSES 

    attron(A_BOLD); 
    wcolor_set(stdscr, COL_TITLE, NULL);
    
    move(0,0); 

    int col=0;
    Chain *pAttr = _schema.First();
    while ( pAttr ) 
    {
	Chain attr;
	int len;
        getAttrSpec(*pAttr, attr, len); 
        mvprintw(0, col, "%s", (char*)attr); 
        for(int j = attr.length(); j < len; j++) 
        { 
            addch(' '); 
        }                       // write a black bar across the top 
        col += len;
	pAttr = _schema.Next();
    } 

    /*
    col=0; 
    for ( int i=0; i < 3; i++ ) 
    { 
        int maxLen = 10; 
        mvprintw(1, col, "%s", (char*)("YYY")); 
        for(int j = 5; j < maxLen; j++) 
        { 
            addch(' '); 
        } 
        col +=maxLen; 
    } 
    */
    attroff(A_BOLD); 
#else 
    throw Exception(EXLOC, "Curses not supported"); 
#endif 

} 

void Pager::writeFooter(int start, int end, int totalRow) 
{ 
#ifndef LFCNOCURSES 

    attron(A_REVERSE);    
    move(_rows-1,0); 
    for(int i = 0; i < _cols; i++) 
    { 
        addch(' '); 
    } 
    mvprintw(_rows-1,0, "up,down,n,b,q ?"); 
    mvprintw(_rows-1,20, "%s", (char*)_title); 
    mvprintw(_rows-1,_cols-27,"%04d - %04d ( %04d total)", start, end, totalRow); 
    attroff(A_REVERSE); 
#endif 
} 


void Pager::getAttrSpec(const Chain& s, Chain& attr, int& len)
{    
    Tokenizer t1(s, ":");	

    t1.nextToken(attr);
    Chain attrLen;
    t1.nextToken(attrLen);
    len = attrLen.asInteger();
}

