///////////////////////////////////////////////////////////////////////////////
//                                                         
// SigHandler.cc
// -------------
// Signal handler class implementation
//                                               
// Design and Implementation by Bjoern Lemke               
//                                                         
// (C)opyright 2000-2016 Bjoern Lemke
//                                                         
// IMPLEMENTATION MODULE
//
// Class: SigHandler
// 
// Description: Signal handler utility class
//
// Status: CLEAN
// 
///////////////////////////////////////////////////////////////////////////////

// INCLUDES
#include "SigHandler.h"
#include "Exception.h"

#ifdef HAVE_MINGW
#include <windows.h>
#endif

#include <signal.h>

#define MAXSIGHANDLER 10

static SigHandler *_pSC[MAXSIGHANDLER];
static int _sigArray[MAXSIGHANDLER];

SigHandler::SigHandler()
{
}

SigHandler::~SigHandler()
{
    int i = 0;
    while ( i< MAXSIGHANDLER )
    {
	if ( _pSC[i] == this )
	{
	    _pSC[i] = 0;
	    _sigArray[i] = 0;
	}
	i++;
    }    
}

void SigHandler::init()
{
    for ( int i=0; i< MAXSIGHANDLER; i++)
    {
	_pSC[i] = 0;
	_sigArray[i] = 0;
    }
}

void SigHandler::install(int sig)
{

#ifdef HAVE_MINGW

    signal(sig, SigHandler::handleSig);

#else

    struct sigaction sa;

    sa.sa_handler = SigHandler::handleSig;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART; /* Restart functions if interrupted by handler */

    if (sigaction(sig, &sa, NULL) == -1)
    {
	throw Exception(EXLOC, "signal system error"); 	
    }

#endif
	
    bool notSet = true;
    int i = 0;
    while ( i< MAXSIGHANDLER )
    {
	// is signal is already registered a signal handler is already active ( idempotent operation ) 
	if ( _sigArray[i] == sig )
	    return;
	i++;
    }
    
    i = 0;
    while ( i< MAXSIGHANDLER && notSet )
    {
	if ( _pSC[i] == 0 )
	{
	    _pSC[i] = this;
	    _sigArray[i] = sig;
	    notSet = false;
	}
	i++;
    }
    
    if ( notSet )
    {
	throw Exception(EXLOC, "No more signal slots available"); 	
    }
}

void SigHandler::handleSig(int sig)
{
    int i = 0;
    while ( i< MAXSIGHANDLER )
    {
	if ( _sigArray[i] == sig )
	{
	    
	    SigHandler *pSig = _pSC[i];
	    
	    _sigArray[i] = 0;
	    _pSC[i] = 0;
	    
	    pSig->sigCatch(sig);
	    return;
	}
	i++;
    }
}
