// Copyright (c) Microsoft Corporation.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception

#ifndef DOUBLE_SCIENTIFIC_PRECISION_TO_CHARS_TEST_CASES_4_HPP
#define DOUBLE_SCIENTIFIC_PRECISION_TO_CHARS_TEST_CASES_4_HPP

#include <charconv>

#include "test.hpp"
using namespace std;

inline constexpr DoublePrecisionToCharsTestCase double_scientific_precision_to_chars_test_cases_4[] = {
    // Test the maximum mantissa, which generates the most digits for each exponent.
    {0x0.fffffffffffffp-1022, chars_format::scientific, 766,
        "2."
        "2250738585072008890245868760858598876504231122409594654935248025624400092282356951787758888037591552642309"
        "7809504343120858773871583572918219930202943792242235598198275012420417889695713117910822610439719796040004"
        "5489739193807919893608152561311337614984204327175103362739154978273159414382813627511383860409424946494228"
        "6316695429105080201815926642134996606517803095075913058719846423906068637102005108723282784678843631944515"
        "8661350412234790147923695852083215976210663754016137365830441936037147783553066828345356340050740730401356"
        "0296804637591858316312422452159926254649430083685186171942241764645513713542013221703137049658321015465406"
        "8035397417906022589503023501937519773030945763173210852507299305089761582519159720757232455434770912461317"
        "493580281734466552734375e-308"},
    {0x1.fffffffffffffp-1022, chars_format::scientific, 766,
        "4."
        "4501477170144022721148195934182639518696390927032912960468522194496444440421538910330590478162701758282983"
        "1782607924221374017287738918929105531441481564124348675997628212653465850710457376274429802596224490290377"
        "9698114444614570510266311510031828794952795966823603998647925096578034214163701381261333311989876551545144"
        "0315261253813266652951306000184917766328660755595837392240989947807556594098101021612198814605258742579179"
        "0000716759993441450860872056815779154359230189103349648694206140521828924314457976051636509036065141403772"
        "1744226256159024466852576737244643007551333245007965068671949137768847800530996396770975896584413789443379"
        "6621993967316936280457084866613206797017728916080020698679408551343728867675409720757232455434770912461317"
        "493580281734466552734375e-308"},
    {0x1.fffffffffffffp-1021, chars_format::scientific, 765,
        "8."
        "9002954340288045442296391868365279037392781854065825920937044388992888880843077820661180956325403516565966"
        "3565215848442748034575477837858211062882963128248697351995256425306931701420914752548859605192448980580755"
        "9396228889229141020532623020063657589905591933647207997295850193156068428327402762522666623979753103090288"
        "0630522507626533305902612000369835532657321511191674784481979895615113188196202043224397629210517485158358"
        "0001433519986882901721744113631558308718460378206699297388412281043657848628915952103273018072130282807544"
        "3488452512318048933705153474489286015102666490015930137343898275537695601061992793541951793168827578886759"
        "3243987934633872560914169733226413594035457832160041397358817102687457735350819441514464910869541824922634"
        "98716056346893310546875e-308"},
    {0x1.fffffffffffffp-1020, chars_format::scientific, 765,
        "1."
        "7800590868057609088459278373673055807478556370813165184187408877798577776168615564132236191265080703313193"
        "2713043169688549606915095567571642212576592625649739470399051285061386340284182950509771921038489796116151"
        "1879245777845828204106524604012731517981118386729441599459170038631213685665480552504533324795950620618057"
        "6126104501525306661180522400073967106531464302238334956896395979123022637639240408644879525842103497031671"
        "6000286703997376580344348822726311661743692075641339859477682456208731569725783190420654603614426056561508"
        "8697690502463609786741030694897857203020533298003186027468779655107539120212398558708390358633765515777351"
        "8648797586926774512182833946645282718807091566432008279471763420537491547070163888302892982173908364984526"
        "99743211269378662109375e-307"},
    {0x1.fffffffffffffp-1019, chars_format::scientific, 764,
        "3."
        "5601181736115218176918556747346111614957112741626330368374817755597155552337231128264472382530161406626386"
        "5426086339377099213830191135143284425153185251299478940798102570122772680568365901019543842076979592232302"
        "3758491555691656408213049208025463035962236773458883198918340077262427371330961105009066649591901241236115"
        "2252209003050613322361044800147934213062928604476669913792791958246045275278480817289759051684206994063343"
        "2000573407994753160688697645452623323487384151282679718955364912417463139451566380841309207228852113123017"
        "7395381004927219573482061389795714406041066596006372054937559310215078240424797117416780717267531031554703"
        "7297595173853549024365667893290565437614183132864016558943526841074983094140327776605785964347816729969053"
        "9948642253875732421875e-307"},
    {0x1.fffffffffffffp-1018, chars_format::scientific, 763,
        "7."
        "1202363472230436353837113494692223229914225483252660736749635511194311104674462256528944765060322813252773"
        "0852172678754198427660382270286568850306370502598957881596205140245545361136731802039087684153959184464604"
        "7516983111383312816426098416050926071924473546917766397836680154524854742661922210018133299183802482472230"
        "4504418006101226644722089600295868426125857208953339827585583916492090550556961634579518103368413988126686"
        "4001146815989506321377395290905246646974768302565359437910729824834926278903132761682618414457704226246035"
        "4790762009854439146964122779591428812082133192012744109875118620430156480849594234833561434535062063109407"
        "4595190347707098048731335786581130875228366265728033117887053682149966188280655553211571928695633459938107"
        "989728450775146484375e-307"},
    {0x1.fffffffffffffp-1017, chars_format::scientific, 763,
        "1."
        "4240472694446087270767422698938444645982845096650532147349927102238862220934892451305788953012064562650554"
        "6170434535750839685532076454057313770061274100519791576319241028049109072227346360407817536830791836892920"
        "9503396622276662563285219683210185214384894709383553279567336030904970948532384442003626659836760496494446"
        "0900883601220245328944417920059173685225171441790667965517116783298418110111392326915903620673682797625337"
        "2800229363197901264275479058181049329394953660513071887582145964966985255780626552336523682891540845249207"
        "0958152401970887829392824555918285762416426638402548821975023724086031296169918846966712286907012412621881"
        "4919038069541419609746267157316226175045673253145606623577410736429993237656131110642314385739126691987621"
        "597945690155029296875e-306"},
    {0x1.fffffffffffffp-1016, chars_format::scientific, 762,
        "2."
        "8480945388892174541534845397876889291965690193301064294699854204477724441869784902611577906024129125301109"
        "2340869071501679371064152908114627540122548201039583152638482056098218144454692720815635073661583673785841"
        "9006793244553325126570439366420370428769789418767106559134672061809941897064768884007253319673520992988892"
        "1801767202440490657888835840118347370450342883581335931034233566596836220222784653831807241347365595250674"
        "5600458726395802528550958116362098658789907321026143775164291929933970511561253104673047365783081690498414"
        "1916304803941775658785649111836571524832853276805097643950047448172062592339837693933424573814024825243762"
        "9838076139082839219492534314632452350091346506291213247154821472859986475312262221284628771478253383975243"
        "19589138031005859375e-306"},
    {0x1.fffffffffffffp-1015, chars_format::scientific, 761,
        "5."
        "6961890777784349083069690795753778583931380386602128589399708408955448883739569805223155812048258250602218"
        "4681738143003358742128305816229255080245096402079166305276964112196436288909385441631270147323167347571683"
        "8013586489106650253140878732840740857539578837534213118269344123619883794129537768014506639347041985977784"
        "3603534404880981315777671680236694740900685767162671862068467133193672440445569307663614482694731190501349"
        "1200917452791605057101916232724197317579814642052287550328583859867941023122506209346094731566163380996828"
        "3832609607883551317571298223673143049665706553610195287900094896344125184679675387866849147628049650487525"
        "9676152278165678438985068629264904700182693012582426494309642945719972950624524442569257542956506767950486"
        "3917827606201171875e-306"},
    {0x1.fffffffffffffp-1014, chars_format::scientific, 761,
        "1."
        "1392378155556869816613938159150755716786276077320425717879941681791089776747913961044631162409651650120443"
        "6936347628600671748425661163245851016049019280415833261055392822439287257781877088326254029464633469514336"
        "7602717297821330050628175746568148171507915767506842623653868824723976758825907553602901327869408397195556"
        "8720706880976196263155534336047338948180137153432534372413693426638734488089113861532722896538946238100269"
        "8240183490558321011420383246544839463515962928410457510065716771973588204624501241869218946313232676199365"
        "6766521921576710263514259644734628609933141310722039057580018979268825036935935077573369829525609930097505"
        "1935230455633135687797013725852980940036538602516485298861928589143994590124904888513851508591301353590097"
        "2783565521240234375e-305"},
    {0x1.fffffffffffffp-1013, chars_format::scientific, 760,
        "2."
        "2784756311113739633227876318301511433572552154640851435759883363582179553495827922089262324819303300240887"
        "3872695257201343496851322326491702032098038560831666522110785644878574515563754176652508058929266939028673"
        "5205434595642660101256351493136296343015831535013685247307737649447953517651815107205802655738816794391113"
        "7441413761952392526311068672094677896360274306865068744827386853277468976178227723065445793077892476200539"
        "6480366981116642022840766493089678927031925856820915020131433543947176409249002483738437892626465352398731"
        "3533043843153420527028519289469257219866282621444078115160037958537650073871870155146739659051219860195010"
        "3870460911266271375594027451705961880073077205032970597723857178287989180249809777027703017182602707180194"
        "556713104248046875e-305"},
    {0x1.fffffffffffffp-1012, chars_format::scientific, 759,
        "4."
        "5569512622227479266455752636603022867145104309281702871519766727164359106991655844178524649638606600481774"
        "7745390514402686993702644652983404064196077121663333044221571289757149031127508353305016117858533878057347"
        "0410869191285320202512702986272592686031663070027370494615475298895907035303630214411605311477633588782227"
        "4882827523904785052622137344189355792720548613730137489654773706554937952356455446130891586155784952401079"
        "2960733962233284045681532986179357854063851713641830040262867087894352818498004967476875785252930704797462"
        "7066087686306841054057038578938514439732565242888156230320075917075300147743740310293479318102439720390020"
        "7740921822532542751188054903411923760146154410065941195447714356575978360499619554055406034365205414360389"
        "11342620849609375e-305"},
    {0x1.fffffffffffffp-1011, chars_format::scientific, 758,
        "9."
        "1139025244454958532911505273206045734290208618563405743039533454328718213983311688357049299277213200963549"
        "5490781028805373987405289305966808128392154243326666088443142579514298062255016706610032235717067756114694"
        "0821738382570640405025405972545185372063326140054740989230950597791814070607260428823210622955267177564454"
        "9765655047809570105244274688378711585441097227460274979309547413109875904712910892261783172311569904802158"
        "5921467924466568091363065972358715708127703427283660080525734175788705636996009934953751570505861409594925"
        "4132175372613682108114077157877028879465130485776312460640151834150600295487480620586958636204879440780041"
        "5481843645065085502376109806823847520292308820131882390895428713151956720999239108110812068730410828720778"
        "2268524169921875e-305"},
    {0x1.fffffffffffffp-1010, chars_format::scientific, 758,
        "1."
        "8227805048890991706582301054641209146858041723712681148607906690865743642796662337671409859855442640192709"
        "9098156205761074797481057861193361625678430848665333217688628515902859612451003341322006447143413551222938"
        "8164347676514128081005081194509037074412665228010948197846190119558362814121452085764642124591053435512890"
        "9953131009561914021048854937675742317088219445492054995861909482621975180942582178452356634462313980960431"
        "7184293584893313618272613194471743141625540685456732016105146835157741127399201986990750314101172281918985"
        "0826435074522736421622815431575405775893026097155262492128030366830120059097496124117391727240975888156008"
        "3096368729013017100475221961364769504058461764026376478179085742630391344199847821622162413746082165744155"
        "6453704833984375e-304"},
    {0x1.fffffffffffffp-1009, chars_format::scientific, 757,
        "3."
        "6455610097781983413164602109282418293716083447425362297215813381731487285593324675342819719710885280385419"
        "8196312411522149594962115722386723251356861697330666435377257031805719224902006682644012894286827102445877"
        "6328695353028256162010162389018074148825330456021896395692380239116725628242904171529284249182106871025781"
        "9906262019123828042097709875351484634176438890984109991723818965243950361885164356904713268924627961920863"
        "4368587169786627236545226388943486283251081370913464032210293670315482254798403973981500628202344563837970"
        "1652870149045472843245630863150811551786052194310524984256060733660240118194992248234783454481951776312016"
        "6192737458026034200950443922729539008116923528052752956358171485260782688399695643244324827492164331488311"
        "290740966796875e-304"},
    {0x1.fffffffffffffp-1008, chars_format::scientific, 756,
        "7."
        "2911220195563966826329204218564836587432166894850724594431626763462974571186649350685639439421770560770839"
        "6392624823044299189924231444773446502713723394661332870754514063611438449804013365288025788573654204891755"
        "2657390706056512324020324778036148297650660912043792791384760478233451256485808343058568498364213742051563"
        "9812524038247656084195419750702969268352877781968219983447637930487900723770328713809426537849255923841726"
        "8737174339573254473090452777886972566502162741826928064420587340630964509596807947963001256404689127675940"
        "3305740298090945686491261726301623103572104388621049968512121467320480236389984496469566908963903552624033"
        "2385474916052068401900887845459078016233847056105505912716342970521565376799391286488649654984328662976622"
        "58148193359375e-304"},
    {0x1.fffffffffffffp-1007, chars_format::scientific, 756,
        "1."
        "4582244039112793365265840843712967317486433378970144918886325352692594914237329870137127887884354112154167"
        "9278524964608859837984846288954689300542744678932266574150902812722287689960802673057605157714730840978351"
        "0531478141211302464804064955607229659530132182408758558276952095646690251297161668611713699672842748410312"
        "7962504807649531216839083950140593853670575556393643996689527586097580144754065742761885307569851184768345"
        "3747434867914650894618090555577394513300432548365385612884117468126192901919361589592600251280937825535188"
        "0661148059618189137298252345260324620714420877724209993702424293464096047277996899293913381792780710524806"
        "6477094983210413680380177569091815603246769411221101182543268594104313075359878257297729930996865732595324"
        "51629638671875e-303"},
    {0x1.fffffffffffffp-1006, chars_format::scientific, 755,
        "2."
        "9164488078225586730531681687425934634972866757940289837772650705385189828474659740274255775768708224308335"
        "8557049929217719675969692577909378601085489357864533148301805625444575379921605346115210315429461681956702"
        "1062956282422604929608129911214459319060264364817517116553904191293380502594323337223427399345685496820625"
        "5925009615299062433678167900281187707341151112787287993379055172195160289508131485523770615139702369536690"
        "7494869735829301789236181111154789026600865096730771225768234936252385803838723179185200502561875651070376"
        "1322296119236378274596504690520649241428841755448419987404848586928192094555993798587826763585561421049613"
        "2954189966420827360760355138183631206493538822442202365086537188208626150719756514595459861993731465190649"
        "0325927734375e-303"},
    {0x1.fffffffffffffp-1005, chars_format::scientific, 754,
        "5."
        "8328976156451173461063363374851869269945733515880579675545301410770379656949319480548511551537416448616671"
        "7114099858435439351939385155818757202170978715729066296603611250889150759843210692230420630858923363913404"
        "2125912564845209859216259822428918638120528729635034233107808382586761005188646674446854798691370993641251"
        "1850019230598124867356335800562375414682302225574575986758110344390320579016262971047541230279404739073381"
        "4989739471658603578472362222309578053201730193461542451536469872504771607677446358370401005123751302140752"
        "2644592238472756549193009381041298482857683510896839974809697173856384189111987597175653527171122842099226"
        "5908379932841654721520710276367262412987077644884404730173074376417252301439513029190919723987462930381298"
        "065185546875e-303"},
    {0x1.fffffffffffffp-1004, chars_format::scientific, 754,
        "1."
        "1665795231290234692212672674970373853989146703176115935109060282154075931389863896109702310307483289723334"
        "3422819971687087870387877031163751440434195743145813259320722250177830151968642138446084126171784672782680"
        "8425182512969041971843251964485783727624105745927006846621561676517352201037729334889370959738274198728250"
        "2370003846119624973471267160112475082936460445114915197351622068878064115803252594209508246055880947814676"
        "2997947894331720715694472444461915610640346038692308490307293974500954321535489271674080201024750260428150"
        "4528918447694551309838601876208259696571536702179367994961939434771276837822397519435130705434224568419845"
        "3181675986568330944304142055273452482597415528976880946034614875283450460287902605838183944797492586076259"
        "613037109375e-302"},
    {0x1.fffffffffffffp-1003, chars_format::scientific, 753,
        "2."
        "3331590462580469384425345349940747707978293406352231870218120564308151862779727792219404620614966579446668"
        "6845639943374175740775754062327502880868391486291626518641444500355660303937284276892168252343569345565361"
        "6850365025938083943686503928971567455248211491854013693243123353034704402075458669778741919476548397456500"
        "4740007692239249946942534320224950165872920890229830394703244137756128231606505188419016492111761895629352"
        "5995895788663441431388944888923831221280692077384616980614587949001908643070978543348160402049500520856300"
        "9057836895389102619677203752416519393143073404358735989923878869542553675644795038870261410868449136839690"
        "6363351973136661888608284110546904965194831057953761892069229750566900920575805211676367889594985172152519"
        "22607421875e-302"},
    {0x1.fffffffffffffp-1002, chars_format::scientific, 752,
        "4."
        "6663180925160938768850690699881495415956586812704463740436241128616303725559455584438809241229933158893337"
        "3691279886748351481551508124655005761736782972583253037282889000711320607874568553784336504687138691130723"
        "3700730051876167887373007857943134910496422983708027386486246706069408804150917339557483838953096794913000"
        "9480015384478499893885068640449900331745841780459660789406488275512256463213010376838032984223523791258705"
        "1991791577326882862777889777847662442561384154769233961229175898003817286141957086696320804099001041712601"
        "8115673790778205239354407504833038786286146808717471979847757739085107351289590077740522821736898273679381"
        "2726703946273323777216568221093809930389662115907523784138459501133801841151610423352735779189970344305038"
        "4521484375e-302"},
    {0x1.fffffffffffffp-1001, chars_format::scientific, 751,
        "9."
        "3326361850321877537701381399762990831913173625408927480872482257232607451118911168877618482459866317786674"
        "7382559773496702963103016249310011523473565945166506074565778001422641215749137107568673009374277382261446"
        "7401460103752335774746015715886269820992845967416054772972493412138817608301834679114967677906193589826001"
        "8960030768956999787770137280899800663491683560919321578812976551024512926426020753676065968447047582517410"
        "3983583154653765725555779555695324885122768309538467922458351796007634572283914173392641608198002083425203"
        "6231347581556410478708815009666077572572293617434943959695515478170214702579180155481045643473796547358762"
        "5453407892546647554433136442187619860779324231815047568276919002267603682303220846705471558379940688610076"
        "904296875e-302"},
    {0x1.fffffffffffffp-1000, chars_format::scientific, 751,
        "1."
        "8665272370064375507540276279952598166382634725081785496174496451446521490223782233775523696491973263557334"
        "9476511954699340592620603249862002304694713189033301214913155600284528243149827421513734601874855476452289"
        "3480292020750467154949203143177253964198569193483210954594498682427763521660366935822993535581238717965200"
        "3792006153791399957554027456179960132698336712183864315762595310204902585285204150735213193689409516503482"
        "0796716630930753145111155911139064977024553661907693584491670359201526914456782834678528321639600416685040"
        "7246269516311282095741763001933215514514458723486988791939103095634042940515836031096209128694759309471752"
        "5090681578509329510886627288437523972155864846363009513655383800453520736460644169341094311675988137722015"
        "380859375e-301"},
    {0x1.fffffffffffffp-999, chars_format::scientific, 750,
        "3."
        "7330544740128751015080552559905196332765269450163570992348992902893042980447564467551047392983946527114669"
        "8953023909398681185241206499724004609389426378066602429826311200569056486299654843027469203749710952904578"
        "6960584041500934309898406286354507928397138386966421909188997364855527043320733871645987071162477435930400"
        "7584012307582799915108054912359920265396673424367728631525190620409805170570408301470426387378819033006964"
        "1593433261861506290222311822278129954049107323815387168983340718403053828913565669357056643279200833370081"
        "4492539032622564191483526003866431029028917446973977583878206191268085881031672062192418257389518618943505"
        "0181363157018659021773254576875047944311729692726019027310767600907041472921288338682188623351976275444030"
        "76171875e-301"},
    {0x1.fffffffffffffp-998, chars_format::scientific, 749,
        "7."
        "4661089480257502030161105119810392665530538900327141984697985805786085960895128935102094785967893054229339"
        "7906047818797362370482412999448009218778852756133204859652622401138112972599309686054938407499421905809157"
        "3921168083001868619796812572709015856794276773932843818377994729711054086641467743291974142324954871860801"
        "5168024615165599830216109824719840530793346848735457263050381240819610341140816602940852774757638066013928"
        "3186866523723012580444623644556259908098214647630774337966681436806107657827131338714113286558401666740162"
        "8985078065245128382967052007732862058057834893947955167756412382536171762063344124384836514779037237887010"
        "0362726314037318043546509153750095888623459385452038054621535201814082945842576677364377246703952550888061"
        "5234375e-301"},
    {0x1.fffffffffffffp-997, chars_format::scientific, 749,
        "1."
        "4932217896051500406032221023962078533106107780065428396939597161157217192179025787020418957193578610845867"
        "9581209563759472474096482599889601843755770551226640971930524480227622594519861937210987681499884381161831"
        "4784233616600373723959362514541803171358855354786568763675598945942210817328293548658394828464990974372160"
        "3033604923033119966043221964943968106158669369747091452610076248163922068228163320588170554951527613202785"
        "6637373304744602516088924728911251981619642929526154867593336287361221531565426267742822657311680333348032"
        "5797015613049025676593410401546572411611566978789591033551282476507234352412668824876967302955807447577402"
        "0072545262807463608709301830750019177724691877090407610924307040362816589168515335472875449340790510177612"
        "3046875e-300"},
    {0x1.fffffffffffffp-996, chars_format::scientific, 748,
        "2."
        "9864435792103000812064442047924157066212215560130856793879194322314434384358051574040837914387157221691735"
        "9162419127518944948192965199779203687511541102453281943861048960455245189039723874421975362999768762323662"
        "9568467233200747447918725029083606342717710709573137527351197891884421634656587097316789656929981948744320"
        "6067209846066239932086443929887936212317338739494182905220152496327844136456326641176341109903055226405571"
        "3274746609489205032177849457822503963239285859052309735186672574722443063130852535485645314623360666696065"
        "1594031226098051353186820803093144823223133957579182067102564953014468704825337649753934605911614895154804"
        "0145090525614927217418603661500038355449383754180815221848614080725633178337030670945750898681581020355224"
        "609375e-300"},
    {0x1.fffffffffffffp-995, chars_format::scientific, 747,
        "5."
        "9728871584206001624128884095848314132424431120261713587758388644628868768716103148081675828774314443383471"
        "8324838255037889896385930399558407375023082204906563887722097920910490378079447748843950725999537524647325"
        "9136934466401494895837450058167212685435421419146275054702395783768843269313174194633579313859963897488641"
        "2134419692132479864172887859775872424634677478988365810440304992655688272912653282352682219806110452811142"
        "6549493218978410064355698915645007926478571718104619470373345149444886126261705070971290629246721333392130"
        "3188062452196102706373641606186289646446267915158364134205129906028937409650675299507869211823229790309608"
        "0290181051229854434837207323000076710898767508361630443697228161451266356674061341891501797363162040710449"
        "21875e-300"},
    {0x1.fffffffffffffp-994, chars_format::scientific, 747,
        "1."
        "1945774316841200324825776819169662826484886224052342717551677728925773753743220629616335165754862888676694"
        "3664967651007577979277186079911681475004616440981312777544419584182098075615889549768790145199907504929465"
        "1827386893280298979167490011633442537087084283829255010940479156753768653862634838926715862771992779497728"
        "2426883938426495972834577571955174484926935495797673162088060998531137654582530656470536443961222090562228"
        "5309898643795682012871139783129001585295714343620923894074669029888977225252341014194258125849344266678426"
        "0637612490439220541274728321237257929289253583031672826841025981205787481930135059901573842364645958061921"
        "6058036210245970886967441464600015342179753501672326088739445632290253271334812268378300359472632408142089"
        "84375e-299"},
    {0x1.fffffffffffffp-993, chars_format::scientific, 746,
        "2."
        "3891548633682400649651553638339325652969772448104685435103355457851547507486441259232670331509725777353388"
        "7329935302015155958554372159823362950009232881962625555088839168364196151231779099537580290399815009858930"
        "3654773786560597958334980023266885074174168567658510021880958313507537307725269677853431725543985558995456"
        "4853767876852991945669155143910348969853870991595346324176121997062275309165061312941072887922444181124457"
        "0619797287591364025742279566258003170591428687241847788149338059777954450504682028388516251698688533356852"
        "1275224980878441082549456642474515858578507166063345653682051962411574963860270119803147684729291916123843"
        "2116072420491941773934882929200030684359507003344652177478891264580506542669624536756600718945264816284179"
        "6875e-299"},
    {0x1.fffffffffffffp-992, chars_format::scientific, 745,
        "4."
        "7783097267364801299303107276678651305939544896209370870206710915703095014972882518465340663019451554706777"
        "4659870604030311917108744319646725900018465763925251110177678336728392302463558199075160580799630019717860"
        "7309547573121195916669960046533770148348337135317020043761916627015074615450539355706863451087971117990912"
        "9707535753705983891338310287820697939707741983190692648352243994124550618330122625882145775844888362248914"
        "1239594575182728051484559132516006341182857374483695576298676119555908901009364056777032503397377066713704"
        "2550449961756882165098913284949031717157014332126691307364103924823149927720540239606295369458583832247686"
        "4232144840983883547869765858400061368719014006689304354957782529161013085339249073513201437890529632568359"
        "375e-299"},
    {0x1.fffffffffffffp-991, chars_format::scientific, 744,
        "9."
        "5566194534729602598606214553357302611879089792418741740413421831406190029945765036930681326038903109413554"
        "9319741208060623834217488639293451800036931527850502220355356673456784604927116398150321161599260039435721"
        "4619095146242391833339920093067540296696674270634040087523833254030149230901078711413726902175942235981825"
        "9415071507411967782676620575641395879415483966381385296704487988249101236660245251764291551689776724497828"
        "2479189150365456102969118265032012682365714748967391152597352239111817802018728113554065006794754133427408"
        "5100899923513764330197826569898063434314028664253382614728207849646299855441080479212590738917167664495372"
        "8464289681967767095739531716800122737438028013378608709915565058322026170678498147026402875781059265136718"
        "75e-299"},
    {0x1.fffffffffffffp-990, chars_format::scientific, 744,
        "1."
        "9113238906945920519721242910671460522375817958483748348082684366281238005989153007386136265207780621882710"
        "9863948241612124766843497727858690360007386305570100444071071334691356920985423279630064232319852007887144"
        "2923819029248478366667984018613508059339334854126808017504766650806029846180215742282745380435188447196365"
        "1883014301482393556535324115128279175883096793276277059340897597649820247332049050352858310337955344899565"
        "6495837830073091220593823653006402536473142949793478230519470447822363560403745622710813001358950826685481"
        "7020179984702752866039565313979612686862805732850676522945641569929259971088216095842518147783433532899074"
        "5692857936393553419147906343360024547487605602675721741983113011664405234135699629405280575156211853027343"
        "75e-298"},
    {0x1.fffffffffffffp-989, chars_format::scientific, 743,
        "3."
        "8226477813891841039442485821342921044751635916967496696165368732562476011978306014772272530415561243765421"
        "9727896483224249533686995455717380720014772611140200888142142669382713841970846559260128464639704015774288"
        "5847638058496956733335968037227016118678669708253616035009533301612059692360431484565490760870376894392730"
        "3766028602964787113070648230256558351766193586552554118681795195299640494664098100705716620675910689799131"
        "2991675660146182441187647306012805072946285899586956461038940895644727120807491245421626002717901653370963"
        "4040359969405505732079130627959225373725611465701353045891283139858519942176432191685036295566867065798149"
        "1385715872787106838295812686720049094975211205351443483966226023328810468271399258810561150312423706054687"
        "5e-298"},
    {0x1.fffffffffffffp-988, chars_format::scientific, 742,
        "7."
        "6452955627783682078884971642685842089503271833934993392330737465124952023956612029544545060831122487530843"
        "9455792966448499067373990911434761440029545222280401776284285338765427683941693118520256929279408031548577"
        "1695276116993913466671936074454032237357339416507232070019066603224119384720862969130981521740753788785460"
        "7532057205929574226141296460513116703532387173105108237363590390599280989328196201411433241351821379598262"
        "5983351320292364882375294612025610145892571799173912922077881791289454241614982490843252005435803306741926"
        "8080719938811011464158261255918450747451222931402706091782566279717039884352864383370072591133734131596298"
        "2771431745574213676591625373440098189950422410702886967932452046657620936542798517621122300624847412109375"
        "e-298"},
    {0x1.fffffffffffffp-987, chars_format::scientific, 742,
        "1."
        "5290591125556736415776994328537168417900654366786998678466147493024990404791322405908909012166224497506168"
        "7891158593289699813474798182286952288005909044456080355256857067753085536788338623704051385855881606309715"
        "4339055223398782693334387214890806447471467883301446414003813320644823876944172593826196304348150757757092"
        "1506411441185914845228259292102623340706477434621021647472718078119856197865639240282286648270364275919652"
        "5196670264058472976475058922405122029178514359834782584415576358257890848322996498168650401087160661348385"
        "3616143987762202292831652251183690149490244586280541218356513255943407976870572876674014518226746826319259"
        "6554286349114842735318325074688019637990084482140577393586490409331524187308559703524224460124969482421875"
        "e-297"},
    {0x1.fffffffffffffp-986, chars_format::scientific, 741,
        "3."
        "0581182251113472831553988657074336835801308733573997356932294986049980809582644811817818024332448995012337"
        "5782317186579399626949596364573904576011818088912160710513714135506171073576677247408102771711763212619430"
        "8678110446797565386668774429781612894942935766602892828007626641289647753888345187652392608696301515514184"
        "3012822882371829690456518584205246681412954869242043294945436156239712395731278480564573296540728551839305"
        "0393340528116945952950117844810244058357028719669565168831152716515781696645992996337300802174321322696770"
        "7232287975524404585663304502367380298980489172561082436713026511886815953741145753348029036453493652638519"
        "310857269822968547063665014937603927598016896428115478717298081866304837461711940704844892024993896484375e"
        "-297"},
    {0x1.fffffffffffffp-985, chars_format::scientific, 740,
        "6."
        "1162364502226945663107977314148673671602617467147994713864589972099961619165289623635636048664897990024675"
        "1564634373158799253899192729147809152023636177824321421027428271012342147153354494816205543423526425238861"
        "7356220893595130773337548859563225789885871533205785656015253282579295507776690375304785217392603031028368"
        "6025645764743659380913037168410493362825909738484086589890872312479424791462556961129146593081457103678610"
        "0786681056233891905900235689620488116714057439339130337662305433031563393291985992674601604348642645393541"
        "4464575951048809171326609004734760597960978345122164873426053023773631907482291506696058072906987305277038"
        "62171453964593709412733002987520785519603379285623095743459616373260967492342388140968978404998779296875e-"
        "297"},
    {0x1.fffffffffffffp-984, chars_format::scientific, 740,
        "1."
        "2232472900445389132621595462829734734320523493429598942772917994419992323833057924727127209732979598004935"
        "0312926874631759850779838545829561830404727235564864284205485654202468429430670898963241108684705285047772"
        "3471244178719026154667509771912645157977174306641157131203050656515859101555338075060957043478520606205673"
        "7205129152948731876182607433682098672565181947696817317978174462495884958292511392225829318616291420735722"
        "0157336211246778381180047137924097623342811487867826067532461086606312678658397198534920320869728529078708"
        "2892915190209761834265321800946952119592195669024432974685210604754726381496458301339211614581397461055407"
        "72434290792918741882546600597504157103920675857124619148691923274652193498468477628193795680999755859375e-"
        "296"},
    {0x1.fffffffffffffp-983, chars_format::scientific, 739,
        "2."
        "4464945800890778265243190925659469468641046986859197885545835988839984647666115849454254419465959196009870"
        "0625853749263519701559677091659123660809454471129728568410971308404936858861341797926482217369410570095544"
        "6942488357438052309335019543825290315954348613282314262406101313031718203110676150121914086957041212411347"
        "4410258305897463752365214867364197345130363895393634635956348924991769916585022784451658637232582841471444"
        "0314672422493556762360094275848195246685622975735652135064922173212625357316794397069840641739457058157416"
        "5785830380419523668530643601893904239184391338048865949370421209509452762992916602678423229162794922110815"
        "4486858158583748376509320119500831420784135171424923829738384654930438699693695525638759136199951171875e-"
        "296"},
    {0x1.fffffffffffffp-982, chars_format::scientific, 738,
        "4."
        "8929891601781556530486381851318938937282093973718395771091671977679969295332231698908508838931918392019740"
        "1251707498527039403119354183318247321618908942259457136821942616809873717722683595852964434738821140191089"
        "3884976714876104618670039087650580631908697226564628524812202626063436406221352300243828173914082424822694"
        "8820516611794927504730429734728394690260727790787269271912697849983539833170045568903317274465165682942888"
        "0629344844987113524720188551696390493371245951471304270129844346425250714633588794139681283478914116314833"
        "1571660760839047337061287203787808478368782676097731898740842419018905525985833205356846458325589844221630"
        "897371631716749675301864023900166284156827034284984765947676930986087739938739105127751827239990234375e-"
        "296"},
    {0x1.fffffffffffffp-981, chars_format::scientific, 737,
        "9."
        "7859783203563113060972763702637877874564187947436791542183343955359938590664463397817017677863836784039480"
        "2503414997054078806238708366636494643237817884518914273643885233619747435445367191705928869477642280382178"
        "7769953429752209237340078175301161263817394453129257049624405252126872812442704600487656347828164849645389"
        "7641033223589855009460859469456789380521455581574538543825395699967079666340091137806634548930331365885776"
        "1258689689974227049440377103392780986742491902942608540259688692850501429267177588279362566957828232629666"
        "3143321521678094674122574407575616956737565352195463797481684838037811051971666410713692916651179688443261"
        "79474326343349935060372804780033256831365406856996953189535386197217547987747821025550365447998046875e-"
        "296"},
    {0x1.fffffffffffffp-980, chars_format::scientific, 737,
        "1."
        "9571956640712622612194552740527575574912837589487358308436668791071987718132892679563403535572767356807896"
        "0500682999410815761247741673327298928647563576903782854728777046723949487089073438341185773895528456076435"
        "7553990685950441847468015635060232252763478890625851409924881050425374562488540920097531269565632969929077"
        "9528206644717971001892171893891357876104291116314907708765079139993415933268018227561326909786066273177155"
        "2251737937994845409888075420678556197348498380588521708051937738570100285853435517655872513391565646525933"
        "2628664304335618934824514881515123391347513070439092759496336967607562210394333282142738583330235937688652"
        "35894865268669987012074560956006651366273081371399390637907077239443509597549564205110073089599609375e-"
        "295"},
    {0x1.fffffffffffffp-979, chars_format::scientific, 736,
        "3."
        "9143913281425245224389105481055151149825675178974716616873337582143975436265785359126807071145534713615792"
        "1001365998821631522495483346654597857295127153807565709457554093447898974178146876682371547791056912152871"
        "5107981371900883694936031270120464505526957781251702819849762100850749124977081840195062539131265939858155"
        "9056413289435942003784343787782715752208582232629815417530158279986831866536036455122653819572132546354310"
        "4503475875989690819776150841357112394696996761177043416103875477140200571706871035311745026783131293051866"
        "5257328608671237869649029763030246782695026140878185518992673935215124420788666564285477166660471875377304"
        "7178973053733997402414912191201330273254616274279878127581415447888701919509912841022014617919921875e-"
        "295"},
    {0x1.fffffffffffffp-978, chars_format::scientific, 735,
        "7."
        "8287826562850490448778210962110302299651350357949433233746675164287950872531570718253614142291069427231584"
        "2002731997643263044990966693309195714590254307615131418915108186895797948356293753364743095582113824305743"
        "0215962743801767389872062540240929011053915562503405639699524201701498249954163680390125078262531879716311"
        "8112826578871884007568687575565431504417164465259630835060316559973663733072072910245307639144265092708620"
        "9006951751979381639552301682714224789393993522354086832207750954280401143413742070623490053566262586103733"
        "0514657217342475739298059526060493565390052281756371037985347870430248841577333128570954333320943750754609"
        "435794610746799480482982438240266054650923254855975625516283089577740383901982568204402923583984375e-295"},
    {0x1.fffffffffffffp-977, chars_format::scientific, 735,
        "1."
        "5657565312570098089755642192422060459930270071589886646749335032857590174506314143650722828458213885446316"
        "8400546399528652608998193338661839142918050861523026283783021637379159589671258750672948619116422764861148"
        "6043192548760353477974412508048185802210783112500681127939904840340299649990832736078025015652506375943262"
        "3622565315774376801513737515113086300883432893051926167012063311994732746614414582049061527828853018541724"
        "1801390350395876327910460336542844957878798704470817366441550190856080228682748414124698010713252517220746"
        "6102931443468495147859611905212098713078010456351274207597069574086049768315466625714190866664188750150921"
        "887158922149359896096596487648053210930184650971195125103256617915548076780396513640880584716796875e-294"},
    {0x1.fffffffffffffp-976, chars_format::scientific, 734,
        "3."
        "1315130625140196179511284384844120919860540143179773293498670065715180349012628287301445656916427770892633"
        "6801092799057305217996386677323678285836101723046052567566043274758319179342517501345897238232845529722297"
        "2086385097520706955948825016096371604421566225001362255879809680680599299981665472156050031305012751886524"
        "7245130631548753603027475030226172601766865786103852334024126623989465493228829164098123055657706037083448"
        "3602780700791752655820920673085689915757597408941634732883100381712160457365496828249396021426505034441493"
        "2205862886936990295719223810424197426156020912702548415194139148172099536630933251428381733328377500301843"
        "77431784429871979219319297529610642186036930194239025020651323583109615356079302728176116943359375e-294"},
    {0x1.fffffffffffffp-975, chars_format::scientific, 733,
        "6."
        "2630261250280392359022568769688241839721080286359546586997340131430360698025256574602891313832855541785267"
        "3602185598114610435992773354647356571672203446092105135132086549516638358685035002691794476465691059444594"
        "4172770195041413911897650032192743208843132450002724511759619361361198599963330944312100062610025503773049"
        "4490261263097507206054950060452345203533731572207704668048253247978930986457658328196246111315412074166896"
        "7205561401583505311641841346171379831515194817883269465766200763424320914730993656498792042853010068882986"
        "4411725773873980591438447620848394852312041825405096830388278296344199073261866502856763466656755000603687"
        "5486356885974395843863859505922128437207386038847805004130264716621923071215860545635223388671875e-294"},
    {0x1.fffffffffffffp-974, chars_format::scientific, 733,
        "1."
        "2526052250056078471804513753937648367944216057271909317399468026286072139605051314920578262766571108357053"
        "4720437119622922087198554670929471314334440689218421027026417309903327671737007000538358895293138211888918"
        "8834554039008282782379530006438548641768626490000544902351923872272239719992666188862420012522005100754609"
        "8898052252619501441210990012090469040706746314441540933609650649595786197291531665639249222263082414833379"
        "3441112280316701062328368269234275966303038963576653893153240152684864182946198731299758408570602013776597"
        "2882345154774796118287689524169678970462408365081019366077655659268839814652373300571352693331351000120737"
        "5097271377194879168772771901184425687441477207769561000826052943324384614243172109127044677734375e-293"},
    {0x1.fffffffffffffp-973, chars_format::scientific, 732,
        "2."
        "5052104500112156943609027507875296735888432114543818634798936052572144279210102629841156525533142216714106"
        "9440874239245844174397109341858942628668881378436842054052834619806655343474014001076717790586276423777837"
        "7669108078016565564759060012877097283537252980001089804703847744544479439985332377724840025044010201509219"
        "7796104505239002882421980024180938081413492628883081867219301299191572394583063331278498444526164829666758"
        "6882224560633402124656736538468551932606077927153307786306480305369728365892397462599516817141204027553194"
        "5764690309549592236575379048339357940924816730162038732155311318537679629304746601142705386662702000241475"
        "019454275438975833754554380236885137488295441553912200165210588664876922848634421825408935546875e-293"},
    {0x1.fffffffffffffp-972, chars_format::scientific, 731,
        "5."
        "0104209000224313887218055015750593471776864229087637269597872105144288558420205259682313051066284433428213"
        "8881748478491688348794218683717885257337762756873684108105669239613310686948028002153435581172552847555675"
        "5338216156033131129518120025754194567074505960002179609407695489088958879970664755449680050088020403018439"
        "5592209010478005764843960048361876162826985257766163734438602598383144789166126662556996889052329659333517"
        "3764449121266804249313473076937103865212155854306615572612960610739456731784794925199033634282408055106389"
        "1529380619099184473150758096678715881849633460324077464310622637075359258609493202285410773325404000482950"
        "03890855087795166750910876047377027497659088310782440033042117732975384569726884365081787109375e-293"},
    {0x1.fffffffffffffp-971, chars_format::scientific, 731,
        "1."
        "0020841800044862777443611003150118694355372845817527453919574421028857711684041051936462610213256886685642"
        "7776349695698337669758843736743577051467552551374736821621133847922662137389605600430687116234510569511135"
        "1067643231206626225903624005150838913414901192000435921881539097817791775994132951089936010017604080603687"
        "9118441802095601152968792009672375232565397051553232746887720519676628957833225332511399377810465931866703"
        "4752889824253360849862694615387420773042431170861323114522592122147891346356958985039806726856481611021277"
        "8305876123819836894630151619335743176369926692064815492862124527415071851721898640457082154665080800096590"
        "00778171017559033350182175209475405499531817662156488006608423546595076913945376873016357421875e-292"},
    {0x1.fffffffffffffp-970, chars_format::scientific, 730,
        "2."
        "0041683600089725554887222006300237388710745691635054907839148842057715423368082103872925220426513773371285"
        "5552699391396675339517687473487154102935105102749473643242267695845324274779211200861374232469021139022270"
        "2135286462413252451807248010301677826829802384000871843763078195635583551988265902179872020035208161207375"
        "8236883604191202305937584019344750465130794103106465493775441039353257915666450665022798755620931863733406"
        "9505779648506721699725389230774841546084862341722646229045184244295782692713917970079613453712963222042555"
        "6611752247639673789260303238671486352739853384129630985724249054830143703443797280914164309330161600193180"
        "0155634203511806670036435041895081099906363532431297601321684709319015382789075374603271484375e-292"},
    {0x1.fffffffffffffp-969, chars_format::scientific, 729,
        "4."
        "0083367200179451109774444012600474777421491383270109815678297684115430846736164207745850440853027546742571"
        "1105398782793350679035374946974308205870210205498947286484535391690648549558422401722748464938042278044540"
        "4270572924826504903614496020603355653659604768001743687526156391271167103976531804359744040070416322414751"
        "6473767208382404611875168038689500930261588206212930987550882078706515831332901330045597511241863727466813"
        "9011559297013443399450778461549683092169724683445292458090368488591565385427835940159226907425926444085111"
        "3223504495279347578520606477342972705479706768259261971448498109660287406887594561828328618660323200386360"
        "031126840702361334007287008379016219981272706486259520264336941863803076557815074920654296875e-292"},
    {0x1.fffffffffffffp-968, chars_format::scientific, 728,
        "8."
        "0166734400358902219548888025200949554842982766540219631356595368230861693472328415491700881706055093485142"
        "2210797565586701358070749893948616411740420410997894572969070783381297099116844803445496929876084556089080"
        "8541145849653009807228992041206711307319209536003487375052312782542334207953063608719488080140832644829503"
        "2947534416764809223750336077379001860523176412425861975101764157413031662665802660091195022483727454933627"
        "8023118594026886798901556923099366184339449366890584916180736977183130770855671880318453814851852888170222"
        "6447008990558695157041212954685945410959413536518523942896996219320574813775189123656657237320646400772720"
        "06225368140472266801457401675803243996254541297251904052867388372760615311563014984130859375e-292"},
    {0x1.fffffffffffffp-967, chars_format::scientific, 728,
        "1."
        "6033346880071780443909777605040189910968596553308043926271319073646172338694465683098340176341211018697028"
        "4442159513117340271614149978789723282348084082199578914593814156676259419823368960689099385975216911217816"
        "1708229169930601961445798408241342261463841907200697475010462556508466841590612721743897616028166528965900"
        "6589506883352961844750067215475800372104635282485172395020352831482606332533160532018239004496745490986725"
        "5604623718805377359780311384619873236867889873378116983236147395436626154171134376063690762970370577634044"
        "5289401798111739031408242590937189082191882707303704788579399243864114962755037824731331447464129280154544"
        "01245073628094453360291480335160648799250908259450380810573477674552123062312602996826171875e-291"},
    {0x1.fffffffffffffp-966, chars_format::scientific, 727,
        "3."
        "2066693760143560887819555210080379821937193106616087852542638147292344677388931366196680352682422037394056"
        "8884319026234680543228299957579446564696168164399157829187628313352518839646737921378198771950433822435632"
        "3416458339861203922891596816482684522927683814401394950020925113016933683181225443487795232056333057931801"
        "3179013766705923689500134430951600744209270564970344790040705662965212665066321064036478008993490981973451"
        "1209247437610754719560622769239746473735779746756233966472294790873252308342268752127381525940741155268089"
        "0578803596223478062816485181874378164383765414607409577158798487728229925510075649462662894928258560309088"
        "0249014725618890672058296067032129759850181651890076162114695534910424612462520599365234375e-291"},
    {0x1.fffffffffffffp-965, chars_format::scientific, 726,
        "6."
        "4133387520287121775639110420160759643874386213232175705085276294584689354777862732393360705364844074788113"
        "7768638052469361086456599915158893129392336328798315658375256626705037679293475842756397543900867644871264"
        "6832916679722407845783193632965369045855367628802789900041850226033867366362450886975590464112666115863602"
        "6358027533411847379000268861903201488418541129940689580081411325930425330132642128072956017986981963946902"
        "2418494875221509439121245538479492947471559493512467932944589581746504616684537504254763051881482310536178"
        "1157607192446956125632970363748756328767530829214819154317596975456459851020151298925325789856517120618176"
        "049802945123778134411659213406425951970036330378015232422939106982084922492504119873046875e-291"},
    {0x1.fffffffffffffp-964, chars_format::scientific, 726,
        "1."
        "2826677504057424355127822084032151928774877242646435141017055258916937870955572546478672141072968814957622"
        "7553727610493872217291319983031778625878467265759663131675051325341007535858695168551279508780173528974252"
        "9366583335944481569156638726593073809171073525760557980008370045206773473272490177395118092822533223172720"
        "5271605506682369475800053772380640297683708225988137916016282265186085066026528425614591203597396392789380"
        "4483698975044301887824249107695898589494311898702493586588917916349300923336907500850952610376296462107235"
        "6231521438489391225126594072749751265753506165842963830863519395091291970204030259785065157971303424123635"
        "209960589024755626882331842681285190394007266075603046484587821396416984498500823974609375e-290"},
    {0x1.fffffffffffffp-963, chars_format::scientific, 725,
        "2."
        "5653355008114848710255644168064303857549754485292870282034110517833875741911145092957344282145937629915245"
        "5107455220987744434582639966063557251756934531519326263350102650682015071717390337102559017560347057948505"
        "8733166671888963138313277453186147618342147051521115960016740090413546946544980354790236185645066446345441"
        "0543211013364738951600107544761280595367416451976275832032564530372170132053056851229182407194792785578760"
        "8967397950088603775648498215391797178988623797404987173177835832698601846673815001701905220752592924214471"
        "2463042876978782450253188145499502531507012331685927661727038790182583940408060519570130315942606848247270"
        "41992117804951125376466368536257038078801453215120609296917564279283396899700164794921875e-290"},
    {0x1.fffffffffffffp-962, chars_format::scientific, 724,
        "5."
        "1306710016229697420511288336128607715099508970585740564068221035667751483822290185914688564291875259830491"
        "0214910441975488869165279932127114503513869063038652526700205301364030143434780674205118035120694115897011"
        "7466333343777926276626554906372295236684294103042231920033480180827093893089960709580472371290132892690882"
        "1086422026729477903200215089522561190734832903952551664065129060744340264106113702458364814389585571157521"
        "7934795900177207551296996430783594357977247594809974346355671665397203693347630003403810441505185848428942"
        "4926085753957564900506376290999005063014024663371855323454077580365167880816121039140260631885213696494540"
        "8398423560990225075293273707251407615760290643024121859383512855856679379940032958984375e-290"},
    {0x1.fffffffffffffp-961, chars_format::scientific, 724,
        "1."
        "0261342003245939484102257667225721543019901794117148112813644207133550296764458037182937712858375051966098"
        "2042982088395097773833055986425422900702773812607730505340041060272806028686956134841023607024138823179402"
        "3493266668755585255325310981274459047336858820608446384006696036165418778617992141916094474258026578538176"
        "4217284405345895580640043017904512238146966580790510332813025812148868052821222740491672962877917114231504"
        "3586959180035441510259399286156718871595449518961994869271134333079440738669526000680762088301037169685788"
        "4985217150791512980101275258199801012602804932674371064690815516073033576163224207828052126377042739298908"
        "1679684712198045015058654741450281523152058128604824371876702571171335875988006591796875e-289"},
    {0x1.fffffffffffffp-960, chars_format::scientific, 723,
        "2."
        "0522684006491878968204515334451443086039803588234296225627288414267100593528916074365875425716750103932196"
        "4085964176790195547666111972850845801405547625215461010680082120545612057373912269682047214048277646358804"
        "6986533337511170510650621962548918094673717641216892768013392072330837557235984283832188948516053157076352"
        "8434568810691791161280086035809024476293933161581020665626051624297736105642445480983345925755834228463008"
        "7173918360070883020518798572313437743190899037923989738542268666158881477339052001361524176602074339371576"
        "9970434301583025960202550516399602025205609865348742129381631032146067152326448415656104252754085478597816"
        "335936942439609003011730948290056304630411625720964874375340514234267175197601318359375e-289"},
    {0x1.fffffffffffffp-959, chars_format::scientific, 722,
        "4."
        "1045368012983757936409030668902886172079607176468592451254576828534201187057832148731750851433500207864392"
        "8171928353580391095332223945701691602811095250430922021360164241091224114747824539364094428096555292717609"
        "3973066675022341021301243925097836189347435282433785536026784144661675114471968567664377897032106314152705"
        "6869137621383582322560172071618048952587866323162041331252103248595472211284890961966691851511668456926017"
        "4347836720141766041037597144626875486381798075847979477084537332317762954678104002723048353204148678743153"
        "9940868603166051920405101032799204050411219730697484258763262064292134304652896831312208505508170957195632"
        "67187388487921800602346189658011260926082325144192974875068102846853435039520263671875e-289"},
    {0x1.fffffffffffffp-958, chars_format::scientific, 721,
        "8."
        "2090736025967515872818061337805772344159214352937184902509153657068402374115664297463501702867000415728785"
        "6343856707160782190664447891403383205622190500861844042720328482182448229495649078728188856193110585435218"
        "7946133350044682042602487850195672378694870564867571072053568289323350228943937135328755794064212628305411"
        "3738275242767164645120344143236097905175732646324082662504206497190944422569781923933383703023336913852034"
        "8695673440283532082075194289253750972763596151695958954169074664635525909356208005446096706408297357486307"
        "9881737206332103840810202065598408100822439461394968517526524128584268609305793662624417011016341914391265"
        "3437477697584360120469237931602252185216465028838594975013620569370687007904052734375e-289"},
    {0x1.fffffffffffffp-957, chars_format::scientific, 721,
        "1."
        "6418147205193503174563612267561154468831842870587436980501830731413680474823132859492700340573400083145757"
        "1268771341432156438132889578280676641124438100172368808544065696436489645899129815745637771238622117087043"
        "7589226670008936408520497570039134475738974112973514214410713657864670045788787427065751158812842525661082"
        "2747655048553432929024068828647219581035146529264816532500841299438188884513956384786676740604667382770406"
        "9739134688056706416415038857850750194552719230339191790833814932927105181871241601089219341281659471497261"
        "5976347441266420768162040413119681620164487892278993703505304825716853721861158732524883402203268382878253"
        "0687495539516872024093847586320450437043293005767718995002724113874137401580810546875e-288"},
    {0x1.fffffffffffffp-956, chars_format::scientific, 720,
        "3."
        "2836294410387006349127224535122308937663685741174873961003661462827360949646265718985400681146800166291514"
        "2537542682864312876265779156561353282248876200344737617088131392872979291798259631491275542477244234174087"
        "5178453340017872817040995140078268951477948225947028428821427315729340091577574854131502317625685051322164"
        "5495310097106865858048137657294439162070293058529633065001682598876377769027912769573353481209334765540813"
        "9478269376113412832830077715701500389105438460678383581667629865854210363742483202178438682563318942994523"
        "1952694882532841536324080826239363240328975784557987407010609651433707443722317465049766804406536765756506"
        "137499107903374404818769517264090087408658601153543799000544822774827480316162109375e-288"},
    {0x1.fffffffffffffp-955, chars_format::scientific, 719,
        "6."
        "5672588820774012698254449070244617875327371482349747922007322925654721899292531437970801362293600332583028"
        "5075085365728625752531558313122706564497752400689475234176262785745958583596519262982551084954488468348175"
        "0356906680035745634081990280156537902955896451894056857642854631458680183155149708263004635251370102644329"
        "0990620194213731716096275314588878324140586117059266130003365197752755538055825539146706962418669531081627"
        "8956538752226825665660155431403000778210876921356767163335259731708420727484966404356877365126637885989046"
        "3905389765065683072648161652478726480657951569115974814021219302867414887444634930099533608813073531513012"
        "27499821580674880963753903452818017481731720230708759800108964554965496063232421875e-288"},
    {0x1.fffffffffffffp-954, chars_format::scientific, 719,
        "1."
        "3134517764154802539650889814048923575065474296469949584401464585130944379858506287594160272458720066516605"
        "7015017073145725150506311662624541312899550480137895046835252557149191716719303852596510216990897693669635"
        "0071381336007149126816398056031307580591179290378811371528570926291736036631029941652600927050274020528865"
        "8198124038842746343219255062917775664828117223411853226000673039550551107611165107829341392483733906216325"
        "5791307750445365133132031086280600155642175384271353432667051946341684145496993280871375473025327577197809"
        "2781077953013136614529632330495745296131590313823194962804243860573482977488926986019906721762614706302602"
        "45499964316134976192750780690563603496346344046141751960021792910993099212646484375e-287"},
    {0x1.fffffffffffffp-953, chars_format::scientific, 718,
        "2."
        "6269035528309605079301779628097847150130948592939899168802929170261888759717012575188320544917440133033211"
        "4030034146291450301012623325249082625799100960275790093670505114298383433438607705193020433981795387339270"
        "0142762672014298253632796112062615161182358580757622743057141852583472073262059883305201854100548041057731"
        "6396248077685492686438510125835551329656234446823706452001346079101102215222330215658682784967467812432651"
        "1582615500890730266264062172561200311284350768542706865334103892683368290993986561742750946050655154395618"
        "5562155906026273229059264660991490592263180627646389925608487721146965954977853972039813443525229412605204"
        "9099992863226995238550156138112720699269268809228350392004358582198619842529296875e-287"},
    {0x1.fffffffffffffp-952, chars_format::scientific, 717,
        "5."
        "2538071056619210158603559256195694300261897185879798337605858340523777519434025150376641089834880266066422"
        "8060068292582900602025246650498165251598201920551580187341010228596766866877215410386040867963590774678540"
        "0285525344028596507265592224125230322364717161515245486114283705166944146524119766610403708201096082115463"
        "2792496155370985372877020251671102659312468893647412904002692158202204430444660431317365569934935624865302"
        "3165231001781460532528124345122400622568701537085413730668207785366736581987973123485501892101310308791237"
        "1124311812052546458118529321982981184526361255292779851216975442293931909955707944079626887050458825210409"
        "819998572645399047710031227622544139853853761845670078400871716439723968505859375e-287"},
    {0x1.fffffffffffffp-951, chars_format::scientific, 717,
        "1."
        "0507614211323842031720711851239138860052379437175959667521171668104755503886805030075328217966976053213284"
        "5612013658516580120405049330099633050319640384110316037468202045719353373375443082077208173592718154935708"
        "0057105068805719301453118444825046064472943432303049097222856741033388829304823953322080741640219216423092"
        "6558499231074197074575404050334220531862493778729482580800538431640440886088932086263473113986987124973060"
        "4633046200356292106505624869024480124513740307417082746133641557073347316397594624697100378420262061758247"
        "4224862362410509291623705864396596236905272251058555970243395088458786381991141588815925377410091765042081"
        "963999714529079809542006245524508827970770752369134015680174343287944793701171875e-286"},
    {0x1.fffffffffffffp-950, chars_format::scientific, 716,
        "2."
        "1015228422647684063441423702478277720104758874351919335042343336209511007773610060150656435933952106426569"
        "1224027317033160240810098660199266100639280768220632074936404091438706746750886164154416347185436309871416"
        "0114210137611438602906236889650092128945886864606098194445713482066777658609647906644161483280438432846185"
        "3116998462148394149150808100668441063724987557458965161601076863280881772177864172526946227973974249946120"
        "9266092400712584213011249738048960249027480614834165492267283114146694632795189249394200756840524123516494"
        "8449724724821018583247411728793192473810544502117111940486790176917572763982283177631850754820183530084163"
        "92799942905815961908401249104901765594154150473826803136034868657588958740234375e-286"},
    {0x1.fffffffffffffp-949, chars_format::scientific, 715,
        "4."
        "2030456845295368126882847404956555440209517748703838670084686672419022015547220120301312871867904212853138"
        "2448054634066320481620197320398532201278561536441264149872808182877413493501772328308832694370872619742832"
        "0228420275222877205812473779300184257891773729212196388891426964133555317219295813288322966560876865692370"
        "6233996924296788298301616201336882127449975114917930323202153726561763544355728345053892455947948499892241"
        "8532184801425168426022499476097920498054961229668330984534566228293389265590378498788401513681048247032989"
        "6899449449642037166494823457586384947621089004234223880973580353835145527964566355263701509640367060168327"
        "8559988581163192381680249820980353118830830094765360627206973731517791748046875e-286"},
    {0x1.fffffffffffffp-948, chars_format::scientific, 714,
        "8."
        "4060913690590736253765694809913110880419035497407677340169373344838044031094440240602625743735808425706276"
        "4896109268132640963240394640797064402557123072882528299745616365754826987003544656617665388741745239485664"
        "0456840550445754411624947558600368515783547458424392777782853928267110634438591626576645933121753731384741"
        "2467993848593576596603232402673764254899950229835860646404307453123527088711456690107784911895896999784483"
        "7064369602850336852044998952195840996109922459336661969069132456586778531180756997576803027362096494065979"
        "3798898899284074332989646915172769895242178008468447761947160707670291055929132710527403019280734120336655"
        "711997716232638476336049964196070623766166018953072125441394746303558349609375e-286"},
    {0x1.fffffffffffffp-947, chars_format::scientific, 714,
        "1."
        "6812182738118147250753138961982622176083807099481535468033874668967608806218888048120525148747161685141255"
        "2979221853626528192648078928159412880511424614576505659949123273150965397400708931323533077748349047897132"
        "8091368110089150882324989511720073703156709491684878555556570785653422126887718325315329186624350746276948"
        "2493598769718715319320646480534752850979990045967172129280861490624705417742291338021556982379179399956896"
        "7412873920570067370408999790439168199221984491867332393813826491317355706236151399515360605472419298813195"
        "8759779779856814866597929383034553979048435601693689552389432141534058211185826542105480603856146824067331"
        "142399543246527695267209992839214124753233203790614425088278949260711669921875e-285"},
    {0x1.fffffffffffffp-946, chars_format::scientific, 713,
        "3."
        "3624365476236294501506277923965244352167614198963070936067749337935217612437776096241050297494323370282510"
        "5958443707253056385296157856318825761022849229153011319898246546301930794801417862647066155496698095794265"
        "6182736220178301764649979023440147406313418983369757111113141571306844253775436650630658373248701492553896"
        "4987197539437430638641292961069505701959980091934344258561722981249410835484582676043113964758358799913793"
        "4825747841140134740817999580878336398443968983734664787627652982634711412472302799030721210944838597626391"
        "7519559559713629733195858766069107958096871203387379104778864283068116422371653084210961207712293648134662"
        "28479908649305539053441998567842824950646640758122885017655789852142333984375e-285"},
    {0x1.fffffffffffffp-945, chars_format::scientific, 712,
        "6."
        "7248730952472589003012555847930488704335228397926141872135498675870435224875552192482100594988646740565021"
        "1916887414506112770592315712637651522045698458306022639796493092603861589602835725294132310993396191588531"
        "2365472440356603529299958046880294812626837966739514222226283142613688507550873301261316746497402985107792"
        "9974395078874861277282585922139011403919960183868688517123445962498821670969165352086227929516717599827586"
        "9651495682280269481635999161756672796887937967469329575255305965269422824944605598061442421889677195252783"
        "5039119119427259466391717532138215916193742406774758209557728566136232844743306168421922415424587296269324"
        "5695981729861107810688399713568564990129328151624577003531157970428466796875e-285"},
    {0x1.fffffffffffffp-944, chars_format::scientific, 712,
        "1."
        "3449746190494517800602511169586097740867045679585228374427099735174087044975110438496420118997729348113004"
        "2383377482901222554118463142527530304409139691661204527959298618520772317920567145058826462198679238317706"
        "2473094488071320705859991609376058962525367593347902844445256628522737701510174660252263349299480597021558"
        "5994879015774972255456517184427802280783992036773737703424689192499764334193833070417245585903343519965517"
        "3930299136456053896327199832351334559377587593493865915051061193053884564988921119612288484377935439050556"
        "7007823823885451893278343506427643183238748481354951641911545713227246568948661233684384483084917459253864"
        "9139196345972221562137679942713712998025865630324915400706231594085693359375e-284"},
    {0x1.fffffffffffffp-943, chars_format::scientific, 711,
        "2."
        "6899492380989035601205022339172195481734091359170456748854199470348174089950220876992840237995458696226008"
        "4766754965802445108236926285055060608818279383322409055918597237041544635841134290117652924397358476635412"
        "4946188976142641411719983218752117925050735186695805688890513257045475403020349320504526698598961194043117"
        "1989758031549944510913034368855604561567984073547475406849378384999528668387666140834491171806687039931034"
        "7860598272912107792654399664702669118755175186987731830102122386107769129977842239224576968755870878101113"
        "4015647647770903786556687012855286366477496962709903283823091426454493137897322467368768966169834918507729"
        "827839269194444312427535988542742599605173126064983080141246318817138671875e-284"},
    {0x1.fffffffffffffp-942, chars_format::scientific, 710,
        "5."
        "3798984761978071202410044678344390963468182718340913497708398940696348179900441753985680475990917392452016"
        "9533509931604890216473852570110121217636558766644818111837194474083089271682268580235305848794716953270824"
        "9892377952285282823439966437504235850101470373391611377781026514090950806040698641009053397197922388086234"
        "3979516063099889021826068737711209123135968147094950813698756769999057336775332281668982343613374079862069"
        "5721196545824215585308799329405338237510350373975463660204244772215538259955684478449153937511741756202226"
        "8031295295541807573113374025710572732954993925419806567646182852908986275794644934737537932339669837015459"
        "65567853838888862485507197708548519921034625212996616028249263763427734375e-284"},
    {0x1.fffffffffffffp-941, chars_format::scientific, 710,
        "1."
        "0759796952395614240482008935668878192693636543668182699541679788139269635980088350797136095198183478490403"
        "3906701986320978043294770514022024243527311753328963622367438894816617854336453716047061169758943390654164"
        "9978475590457056564687993287500847170020294074678322275556205302818190161208139728201810679439584477617246"
        "8795903212619977804365213747542241824627193629418990162739751353999811467355066456333796468722674815972413"
        "9144239309164843117061759865881067647502070074795092732040848954443107651991136895689830787502348351240445"
        "3606259059108361514622674805142114546590998785083961313529236570581797255158928986947507586467933967403091"
        "93113570767777772497101439541709703984206925042599323205649852752685546875e-283"},
    {0x1.fffffffffffffp-940, chars_format::scientific, 709,
        "2."
        "1519593904791228480964017871337756385387273087336365399083359576278539271960176701594272190396366956980806"
        "7813403972641956086589541028044048487054623506657927244734877789633235708672907432094122339517886781308329"
        "9956951180914113129375986575001694340040588149356644551112410605636380322416279456403621358879168955234493"
        "7591806425239955608730427495084483649254387258837980325479502707999622934710132912667592937445349631944827"
        "8288478618329686234123519731762135295004140149590185464081697908886215303982273791379661575004696702480890"
        "7212518118216723029245349610284229093181997570167922627058473141163594510317857973895015172935867934806183"
        "8622714153555554499420287908341940796841385008519864641129970550537109375e-283"},
    {0x1.fffffffffffffp-939, chars_format::scientific, 708,
        "4."
        "3039187809582456961928035742675512770774546174672730798166719152557078543920353403188544380792733913961613"
        "5626807945283912173179082056088096974109247013315854489469755579266471417345814864188244679035773562616659"
        "9913902361828226258751973150003388680081176298713289102224821211272760644832558912807242717758337910468987"
        "5183612850479911217460854990168967298508774517675960650959005415999245869420265825335185874890699263889655"
        "6576957236659372468247039463524270590008280299180370928163395817772430607964547582759323150009393404961781"
        "4425036236433446058490699220568458186363995140335845254116946282327189020635715947790030345871735869612367"
        "724542830711110899884057581668388159368277001703972928225994110107421875e-283"},
    {0x1.fffffffffffffp-938, chars_format::scientific, 707,
        "8."
        "6078375619164913923856071485351025541549092349345461596333438305114157087840706806377088761585467827923227"
        "1253615890567824346358164112176193948218494026631708978939511158532942834691629728376489358071547125233319"
        "9827804723656452517503946300006777360162352597426578204449642422545521289665117825614485435516675820937975"
        "0367225700959822434921709980337934597017549035351921301918010831998491738840531650670371749781398527779311"
        "3153914473318744936494078927048541180016560598360741856326791635544861215929095165518646300018786809923562"
        "8850072472866892116981398441136916372727990280671690508233892564654378041271431895580060691743471739224735"
        "44908566142222179976811516333677631873655400340794585645198822021484375e-283"},
    {0x1.fffffffffffffp-937, chars_format::scientific, 707,
        "1."
        "7215675123832982784771214297070205108309818469869092319266687661022831417568141361275417752317093565584645"
        "4250723178113564869271632822435238789643698805326341795787902231706588566938325945675297871614309425046663"
        "9965560944731290503500789260001355472032470519485315640889928484509104257933023565122897087103335164187595"
        "0073445140191964486984341996067586919403509807070384260383602166399698347768106330134074349956279705555862"
        "2630782894663748987298815785409708236003312119672148371265358327108972243185819033103729260003757361984712"
        "5770014494573378423396279688227383274545598056134338101646778512930875608254286379116012138348694347844947"
        "08981713228444435995362303266735526374731080068158917129039764404296875e-282"},
    {0x1.fffffffffffffp-936, chars_format::scientific, 706,
        "3."
        "4431350247665965569542428594140410216619636939738184638533375322045662835136282722550835504634187131169290"
        "8501446356227129738543265644870477579287397610652683591575804463413177133876651891350595743228618850093327"
        "9931121889462581007001578520002710944064941038970631281779856969018208515866047130245794174206670328375190"
        "0146890280383928973968683992135173838807019614140768520767204332799396695536212660268148699912559411111724"
        "5261565789327497974597631570819416472006624239344296742530716654217944486371638066207458520007514723969425"
        "1540028989146756846792559376454766549091196112268676203293557025861751216508572758232024276697388695689894"
        "1796342645688887199072460653347105274946216013631783425807952880859375e-282"},
    {0x1.fffffffffffffp-935, chars_format::scientific, 705,
        "6."
        "8862700495331931139084857188280820433239273879476369277066750644091325670272565445101671009268374262338581"
        "7002892712454259477086531289740955158574795221305367183151608926826354267753303782701191486457237700186655"
        "9862243778925162014003157040005421888129882077941262563559713938036417031732094260491588348413340656750380"
        "0293780560767857947937367984270347677614039228281537041534408665598793391072425320536297399825118822223449"
        "0523131578654995949195263141638832944013248478688593485061433308435888972743276132414917040015029447938850"
        "3080057978293513693585118752909533098182392224537352406587114051723502433017145516464048553394777391379788"
        "359268529137777439814492130669421054989243202726356685161590576171875e-282"},
    {0x1.fffffffffffffp-934, chars_format::scientific, 705,
        "1."
        "3772540099066386227816971437656164086647854775895273855413350128818265134054513089020334201853674852467716"
        "3400578542490851895417306257948191031714959044261073436630321785365270853550660756540238297291447540037331"
        "1972448755785032402800631408001084377625976415588252512711942787607283406346418852098317669682668131350076"
        "0058756112153571589587473596854069535522807845656307408306881733119758678214485064107259479965023764444689"
        "8104626315730999189839052628327766588802649695737718697012286661687177794548655226482983408003005889587770"
        "0616011595658702738717023750581906619636478444907470481317422810344700486603429103292809710678955478275957"
        "671853705827555487962898426133884210997848640545271337032318115234375e-281"},
    {0x1.fffffffffffffp-933, chars_format::scientific, 704,
        "2."
        "7545080198132772455633942875312328173295709551790547710826700257636530268109026178040668403707349704935432"
        "6801157084981703790834612515896382063429918088522146873260643570730541707101321513080476594582895080074662"
        "3944897511570064805601262816002168755251952831176505025423885575214566812692837704196635339365336262700152"
        "0117512224307143179174947193708139071045615691312614816613763466239517356428970128214518959930047528889379"
        "6209252631461998379678105256655533177605299391475437394024573323374355589097310452965966816006011779175540"
        "1232023191317405477434047501163813239272956889814940962634845620689400973206858206585619421357910956551915"
        "34370741165511097592579685226776842199569728109054267406463623046875e-281"},
    {0x1.fffffffffffffp-932, chars_format::scientific, 703,
        "5."
        "5090160396265544911267885750624656346591419103581095421653400515273060536218052356081336807414699409870865"
        "3602314169963407581669225031792764126859836177044293746521287141461083414202643026160953189165790160149324"
        "7889795023140129611202525632004337510503905662353010050847771150429133625385675408393270678730672525400304"
        "0235024448614286358349894387416278142091231382625229633227526932479034712857940256429037919860095057778759"
        "2418505262923996759356210513311066355210598782950874788049146646748711178194620905931933632012023558351080"
        "2464046382634810954868095002327626478545913779629881925269691241378801946413716413171238842715821913103830"
        "6874148233102219518515937045355368439913945621810853481292724609375e-281"},
    {0x1.fffffffffffffp-931, chars_format::scientific, 703,
        "1."
        "1018032079253108982253577150124931269318283820716219084330680103054612107243610471216267361482939881974173"
        "0720462833992681516333845006358552825371967235408858749304257428292216682840528605232190637833158032029864"
        "9577959004628025922240505126400867502100781132470602010169554230085826725077135081678654135746134505080060"
        "8047004889722857271669978877483255628418246276525045926645505386495806942571588051285807583972019011555751"
        "8483701052584799351871242102662213271042119756590174957609829329349742235638924181186386726402404711670216"
        "0492809276526962190973619000465525295709182755925976385053938248275760389282743282634247768543164382620766"
        "1374829646620443903703187409071073687982789124362170696258544921875e-280"},
    {0x1.fffffffffffffp-930, chars_format::scientific, 702,
        "2."
        "2036064158506217964507154300249862538636567641432438168661360206109224214487220942432534722965879763948346"
        "1440925667985363032667690012717105650743934470817717498608514856584433365681057210464381275666316064059729"
        "9155918009256051844481010252801735004201562264941204020339108460171653450154270163357308271492269010160121"
        "6094009779445714543339957754966511256836492553050091853291010772991613885143176102571615167944038023111503"
        "6967402105169598703742484205324426542084239513180349915219658658699484471277848362372773452804809423340432"
        "0985618553053924381947238000931050591418365511851952770107876496551520778565486565268495537086328765241532"
        "274965929324088780740637481814214737596557824872434139251708984375e-280"},
    {0x1.fffffffffffffp-929, chars_format::scientific, 701,
        "4."
        "4072128317012435929014308600499725077273135282864876337322720412218448428974441884865069445931759527896692"
        "2881851335970726065335380025434211301487868941635434997217029713168866731362114420928762551332632128119459"
        "8311836018512103688962020505603470008403124529882408040678216920343306900308540326714616542984538020320243"
        "2188019558891429086679915509933022513672985106100183706582021545983227770286352205143230335888076046223007"
        "3934804210339197407484968410648853084168479026360699830439317317398968942555696724745546905609618846680864"
        "1971237106107848763894476001862101182836731023703905540215752993103041557130973130536991074172657530483064"
        "54993185864817756148127496362842947519311564974486827850341796875e-280"},
    {0x1.fffffffffffffp-928, chars_format::scientific, 700,
        "8."
        "8144256634024871858028617200999450154546270565729752674645440824436896857948883769730138891863519055793384"
        "5763702671941452130670760050868422602975737883270869994434059426337733462724228841857525102665264256238919"
        "6623672037024207377924041011206940016806249059764816081356433840686613800617080653429233085969076040640486"
        "4376039117782858173359831019866045027345970212200367413164043091966455540572704410286460671776152092446014"
        "7869608420678394814969936821297706168336958052721399660878634634797937885111393449491093811219237693361728"
        "3942474212215697527788952003724202365673462047407811080431505986206083114261946261073982148345315060966129"
        "0998637172963551229625499272568589503862312994897365570068359375e-280"},
    {0x1.fffffffffffffp-927, chars_format::scientific, 700,
        "1."
        "7628851326804974371605723440199890030909254113145950534929088164887379371589776753946027778372703811158676"
        "9152740534388290426134152010173684520595147576654173998886811885267546692544845768371505020533052851247783"
        "9324734407404841475584808202241388003361249811952963216271286768137322760123416130685846617193815208128097"
        "2875207823556571634671966203973209005469194042440073482632808618393291108114540882057292134355230418489202"
        "9573921684135678962993987364259541233667391610544279932175726926959587577022278689898218762243847538672345"
        "6788494842443139505557790400744840473134692409481562216086301197241216622852389252214796429669063012193225"
        "8199727434592710245925099854513717900772462598979473114013671875e-279"},
    {0x1.fffffffffffffp-926, chars_format::scientific, 699,
        "3."
        "5257702653609948743211446880399780061818508226291901069858176329774758743179553507892055556745407622317353"
        "8305481068776580852268304020347369041190295153308347997773623770535093385089691536743010041066105702495567"
        "8649468814809682951169616404482776006722499623905926432542573536274645520246832261371693234387630416256194"
        "5750415647113143269343932407946418010938388084880146965265617236786582216229081764114584268710460836978405"
        "9147843368271357925987974728519082467334783221088559864351453853919175154044557379796437524487695077344691"
        "3576989684886279011115580801489680946269384818963124432172602394482433245704778504429592859338126024386451"
        "639945486918542049185019970902743580154492519795894622802734375e-279"},
    {0x1.fffffffffffffp-925, chars_format::scientific, 698,
        "7."
        "0515405307219897486422893760799560123637016452583802139716352659549517486359107015784111113490815244634707"
        "6610962137553161704536608040694738082380590306616695995547247541070186770179383073486020082132211404991135"
        "7298937629619365902339232808965552013444999247811852865085147072549291040493664522743386468775260832512389"
        "1500831294226286538687864815892836021876776169760293930531234473573164432458163528229168537420921673956811"
        "8295686736542715851975949457038164934669566442177119728702907707838350308089114759592875048975390154689382"
        "7153979369772558022231161602979361892538769637926248864345204788964866491409557008859185718676252048772903"
        "27989097383708409837003994180548716030898503959178924560546875e-279"},
    {0x1.fffffffffffffp-924, chars_format::scientific, 698,
        "1."
        "4103081061443979497284578752159912024727403290516760427943270531909903497271821403156822222698163048926941"
        "5322192427510632340907321608138947616476118061323339199109449508214037354035876614697204016426442280998227"
        "1459787525923873180467846561793110402688999849562370573017029414509858208098732904548677293755052166502477"
        "8300166258845257307737572963178567204375355233952058786106246894714632886491632705645833707484184334791362"
        "3659137347308543170395189891407632986933913288435423945740581541567670061617822951918575009795078030937876"
        "5430795873954511604446232320595872378507753927585249772869040957792973298281911401771837143735250409754580"
        "65597819476741681967400798836109743206179700791835784912109375e-278"},
    {0x1.fffffffffffffp-923, chars_format::scientific, 697,
        "2."
        "8206162122887958994569157504319824049454806581033520855886541063819806994543642806313644445396326097853883"
        "0644384855021264681814643216277895232952236122646678398218899016428074708071753229394408032852884561996454"
        "2919575051847746360935693123586220805377999699124741146034058829019716416197465809097354587510104333004955"
        "6600332517690514615475145926357134408750710467904117572212493789429265772983265411291667414968368669582724"
        "7318274694617086340790379782815265973867826576870847891481163083135340123235645903837150019590156061875753"
        "0861591747909023208892464641191744757015507855170499545738081915585946596563822803543674287470500819509161"
        "3119563895348336393480159767221948641235940158367156982421875e-278"},
    {0x1.fffffffffffffp-922, chars_format::scientific, 696,
        "5."
        "6412324245775917989138315008639648098909613162067041711773082127639613989087285612627288890792652195707766"
        "1288769710042529363629286432555790465904472245293356796437798032856149416143506458788816065705769123992908"
        "5839150103695492721871386247172441610755999398249482292068117658039432832394931618194709175020208666009911"
        "3200665035381029230950291852714268817501420935808235144424987578858531545966530822583334829936737339165449"
        "4636549389234172681580759565630531947735653153741695782962326166270680246471291807674300039180312123751506"
        "1723183495818046417784929282383489514031015710340999091476163831171893193127645607087348574941001639018322"
        "623912779069667278696031953444389728247188031673431396484375e-278"},
    {0x1.fffffffffffffp-921, chars_format::scientific, 696,
        "1."
        "1282464849155183597827663001727929619781922632413408342354616425527922797817457122525457778158530439141553"
        "2257753942008505872725857286511158093180894449058671359287559606571229883228701291757763213141153824798581"
        "7167830020739098544374277249434488322151199879649896458413623531607886566478986323638941835004041733201982"
        "2640133007076205846190058370542853763500284187161647028884997515771706309193306164516666965987347467833089"
        "8927309877846834536316151913126106389547130630748339156592465233254136049294258361534860007836062424750301"
        "2344636699163609283556985856476697902806203142068199818295232766234378638625529121417469714988200327803664"
        "524782555813933455739206390688877945649437606334686279296875e-277"},
    {0x1.fffffffffffffp-920, chars_format::scientific, 695,
        "2."
        "2564929698310367195655326003455859239563845264826816684709232851055845595634914245050915556317060878283106"
        "4515507884017011745451714573022316186361788898117342718575119213142459766457402583515526426282307649597163"
        "4335660041478197088748554498868976644302399759299792916827247063215773132957972647277883670008083466403964"
        "5280266014152411692380116741085707527000568374323294057769995031543412618386612329033333931974694935666179"
        "7854619755693669072632303826252212779094261261496678313184930466508272098588516723069720015672124849500602"
        "4689273398327218567113971712953395805612406284136399636590465532468757277251058242834939429976400655607329"
        "04956511162786691147841278137775589129887521266937255859375e-277"},
    {0x1.fffffffffffffp-919, chars_format::scientific, 694,
        "4."
        "5129859396620734391310652006911718479127690529653633369418465702111691191269828490101831112634121756566212"
        "9031015768034023490903429146044632372723577796234685437150238426284919532914805167031052852564615299194326"
        "8671320082956394177497108997737953288604799518599585833654494126431546265915945294555767340016166932807929"
        "0560532028304823384760233482171415054001136748646588115539990063086825236773224658066667863949389871332359"
        "5709239511387338145264607652504425558188522522993356626369860933016544197177033446139440031344249699001204"
        "9378546796654437134227943425906791611224812568272799273180931064937514554502116485669878859952801311214658"
        "0991302232557338229568255627555117825977504253387451171875e-277"},
    {0x1.fffffffffffffp-918, chars_format::scientific, 693,
        "9."
        "0259718793241468782621304013823436958255381059307266738836931404223382382539656980203662225268243513132425"
        "8062031536068046981806858292089264745447155592469370874300476852569839065829610334062105705129230598388653"
        "7342640165912788354994217995475906577209599037199171667308988252863092531831890589111534680032333865615858"
        "1121064056609646769520466964342830108002273497293176231079980126173650473546449316133335727898779742664719"
        "1418479022774676290529215305008851116377045045986713252739721866033088394354066892278880062688499398002409"
        "8757093593308874268455886851813583222449625136545598546361862129875029109004232971339757719905602622429316"
        "198260446511467645913651125511023565195500850677490234375e-277"},
    {0x1.fffffffffffffp-917, chars_format::scientific, 693,
        "1."
        "8051943758648293756524260802764687391651076211861453347767386280844676476507931396040732445053648702626485"
        "1612406307213609396361371658417852949089431118493874174860095370513967813165922066812421141025846119677730"
        "7468528033182557670998843599095181315441919807439834333461797650572618506366378117822306936006466773123171"
        "6224212811321929353904093392868566021600454699458635246215996025234730094709289863226667145579755948532943"
        "8283695804554935258105843061001770223275409009197342650547944373206617678870813378455776012537699879600481"
        "9751418718661774853691177370362716644489925027309119709272372425975005821800846594267951543981120524485863"
        "239652089302293529182730225102204713039100170135498046875e-276"},
    {0x1.fffffffffffffp-916, chars_format::scientific, 692,
        "3."
        "6103887517296587513048521605529374783302152423722906695534772561689352953015862792081464890107297405252970"
        "3224812614427218792722743316835705898178862236987748349720190741027935626331844133624842282051692239355461"
        "4937056066365115341997687198190362630883839614879668666923595301145237012732756235644613872012933546246343"
        "2448425622643858707808186785737132043200909398917270492431992050469460189418579726453334291159511897065887"
        "6567391609109870516211686122003540446550818018394685301095888746413235357741626756911552025075399759200963"
        "9502837437323549707382354740725433288979850054618239418544744851950011643601693188535903087962241048971726"
        "47930417860458705836546045020440942607820034027099609375e-276"},
    {0x1.fffffffffffffp-915, chars_format::scientific, 691,
        "7."
        "2207775034593175026097043211058749566604304847445813391069545123378705906031725584162929780214594810505940"
        "6449625228854437585445486633671411796357724473975496699440381482055871252663688267249684564103384478710922"
        "9874112132730230683995374396380725261767679229759337333847190602290474025465512471289227744025867092492686"
        "4896851245287717415616373571474264086401818797834540984863984100938920378837159452906668582319023794131775"
        "3134783218219741032423372244007080893101636036789370602191777492826470715483253513823104050150799518401927"
        "9005674874647099414764709481450866577959700109236478837089489703900023287203386377071806175924482097943452"
        "9586083572091741167309209004088188521564006805419921875e-276"},
    {0x1.fffffffffffffp-914, chars_format::scientific, 691,
        "1."
        "4441555006918635005219408642211749913320860969489162678213909024675741181206345116832585956042918962101188"
        "1289925045770887517089097326734282359271544894795099339888076296411174250532737653449936912820676895742184"
        "5974822426546046136799074879276145052353535845951867466769438120458094805093102494257845548805173418498537"
        "2979370249057543483123274714294852817280363759566908196972796820187784075767431890581333716463804758826355"
        "0626956643643948206484674448801416178620327207357874120438355498565294143096650702764620810030159903680385"
        "5801134974929419882952941896290173315591940021847295767417897940780004657440677275414361235184896419588690"
        "5917216714418348233461841800817637704312801361083984375e-275"},
    {0x1.fffffffffffffp-913, chars_format::scientific, 690,
        "2."
        "8883110013837270010438817284423499826641721938978325356427818049351482362412690233665171912085837924202376"
        "2579850091541775034178194653468564718543089789590198679776152592822348501065475306899873825641353791484369"
        "1949644853092092273598149758552290104707071691903734933538876240916189610186204988515691097610346836997074"
        "5958740498115086966246549428589705634560727519133816393945593640375568151534863781162667432927609517652710"
        "1253913287287896412969348897602832357240654414715748240876710997130588286193301405529241620060319807360771"
        "1602269949858839765905883792580346631183880043694591534835795881560009314881354550828722470369792839177381"
        "183443342883669646692368360163527540862560272216796875e-275"},
    {0x1.fffffffffffffp-912, chars_format::scientific, 689,
        "5."
        "7766220027674540020877634568846999653283443877956650712855636098702964724825380467330343824171675848404752"
        "5159700183083550068356389306937129437086179579180397359552305185644697002130950613799747651282707582968738"
        "3899289706184184547196299517104580209414143383807469867077752481832379220372409977031382195220693673994149"
        "1917480996230173932493098857179411269121455038267632787891187280751136303069727562325334865855219035305420"
        "2507826574575792825938697795205664714481308829431496481753421994261176572386602811058483240120639614721542"
        "3204539899717679531811767585160693262367760087389183069671591763120018629762709101657444940739585678354762"
        "36688668576733929338473672032705508172512054443359375e-275"},
    {0x1.fffffffffffffp-911, chars_format::scientific, 689,
        "1."
        "1553244005534908004175526913769399930656688775591330142571127219740592944965076093466068764834335169680950"
        "5031940036616710013671277861387425887417235915836079471910461037128939400426190122759949530256541516593747"
        "6779857941236836909439259903420916041882828676761493973415550496366475844074481995406276439044138734798829"
        "8383496199246034786498619771435882253824291007653526557578237456150227260613945512465066973171043807061084"
        "0501565314915158565187739559041132942896261765886299296350684398852235314477320562211696648024127922944308"
        "4640907979943535906362353517032138652473552017477836613934318352624003725952541820331488988147917135670952"
        "47337733715346785867694734406541101634502410888671875e-274"},
    {0x1.fffffffffffffp-910, chars_format::scientific, 688,
        "2."
        "3106488011069816008351053827538799861313377551182660285142254439481185889930152186932137529668670339361901"
        "0063880073233420027342555722774851774834471831672158943820922074257878800852380245519899060513083033187495"
        "3559715882473673818878519806841832083765657353522987946831100992732951688148963990812552878088277469597659"
        "6766992398492069572997239542871764507648582015307053115156474912300454521227891024930133946342087614122168"
        "1003130629830317130375479118082265885792523531772598592701368797704470628954641124423393296048255845888616"
        "9281815959887071812724707034064277304947104034955673227868636705248007451905083640662977976295834271341904"
        "9467546743069357173538946881308220326900482177734375e-274"},
    {0x1.fffffffffffffp-909, chars_format::scientific, 687,
        "4."
        "6212976022139632016702107655077599722626755102365320570284508878962371779860304373864275059337340678723802"
        "0127760146466840054685111445549703549668943663344317887641844148515757601704760491039798121026166066374990"
        "7119431764947347637757039613683664167531314707045975893662201985465903376297927981625105756176554939195319"
        "3533984796984139145994479085743529015297164030614106230312949824600909042455782049860267892684175228244336"
        "2006261259660634260750958236164531771585047063545197185402737595408941257909282248846786592096511691777233"
        "8563631919774143625449414068128554609894208069911346455737273410496014903810167281325955952591668542683809"
        "893509348613871434707789376261644065380096435546875e-274"},
    {0x1.fffffffffffffp-908, chars_format::scientific, 686,
        "9."
        "2425952044279264033404215310155199445253510204730641140569017757924743559720608747728550118674681357447604"
        "0255520292933680109370222891099407099337887326688635775283688297031515203409520982079596242052332132749981"
        "4238863529894695275514079227367328335062629414091951787324403970931806752595855963250211512353109878390638"
        "7067969593968278291988958171487058030594328061228212460625899649201818084911564099720535785368350456488672"
        "4012522519321268521501916472329063543170094127090394370805475190817882515818564497693573184193023383554467"
        "7127263839548287250898828136257109219788416139822692911474546820992029807620334562651911905183337085367619"
        "78701869722774286941557875252328813076019287109375e-274"},
    {0x1.fffffffffffffp-907, chars_format::scientific, 686,
        "1."
        "8485190408855852806680843062031039889050702040946128228113803551584948711944121749545710023734936271489520"
        "8051104058586736021874044578219881419867577465337727155056737659406303040681904196415919248410466426549996"
        "2847772705978939055102815845473465667012525882818390357464880794186361350519171192650042302470621975678127"
        "7413593918793655658397791634297411606118865612245642492125179929840363616982312819944107157073670091297734"
        "4802504503864253704300383294465812708634018825418078874161095038163576503163712899538714636838604676710893"
        "5425452767909657450179765627251421843957683227964538582294909364198405961524066912530382381036667417073523"
        "95740373944554857388311575050465762615203857421875e-273"},
    {0x1.fffffffffffffp-906, chars_format::scientific, 685,
        "3."
        "6970380817711705613361686124062079778101404081892256456227607103169897423888243499091420047469872542979041"
        "6102208117173472043748089156439762839735154930675454310113475318812606081363808392831838496820932853099992"
        "5695545411957878110205631690946931334025051765636780714929761588372722701038342385300084604941243951356255"
        "4827187837587311316795583268594823212237731224491284984250359859680727233964625639888214314147340182595468"
        "9605009007728507408600766588931625417268037650836157748322190076327153006327425799077429273677209353421787"
        "0850905535819314900359531254502843687915366455929077164589818728396811923048133825060764762073334834147047"
        "9148074788910971477662315010093152523040771484375e-273"},
    {0x1.fffffffffffffp-905, chars_format::scientific, 684,
        "7."
        "3940761635423411226723372248124159556202808163784512912455214206339794847776486998182840094939745085958083"
        "2204416234346944087496178312879525679470309861350908620226950637625212162727616785663676993641865706199985"
        "1391090823915756220411263381893862668050103531273561429859523176745445402076684770600169209882487902712510"
        "9654375675174622633591166537189646424475462448982569968500719719361454467929251279776428628294680365190937"
        "9210018015457014817201533177863250834536075301672315496644380152654306012654851598154858547354418706843574"
        "1701811071638629800719062509005687375830732911858154329179637456793623846096267650121529524146669668294095"
        "829614957782194295532463002018630504608154296875e-273"},
    {0x1.fffffffffffffp-904, chars_format::scientific, 684,
        "1."
        "4788152327084682245344674449624831911240561632756902582491042841267958969555297399636568018987949017191616"
        "6440883246869388817499235662575905135894061972270181724045390127525042432545523357132735398728373141239997"
        "0278218164783151244082252676378772533610020706254712285971904635349089080415336954120033841976497580542502"
        "1930875135034924526718233307437929284895092489796513993700143943872290893585850255955285725658936073038187"
        "5842003603091402963440306635572650166907215060334463099328876030530861202530970319630971709470883741368714"
        "8340362214327725960143812501801137475166146582371630865835927491358724769219253530024305904829333933658819"
        "165922991556438859106492600403726100921630859375e-272"},
    {0x1.fffffffffffffp-903, chars_format::scientific, 683,
        "2."
        "9576304654169364490689348899249663822481123265513805164982085682535917939110594799273136037975898034383233"
        "2881766493738777634998471325151810271788123944540363448090780255050084865091046714265470797456746282479994"
        "0556436329566302488164505352757545067220041412509424571943809270698178160830673908240067683952995161085004"
        "3861750270069849053436466614875858569790184979593027987400287887744581787171700511910571451317872146076375"
        "1684007206182805926880613271145300333814430120668926198657752061061722405061940639261943418941767482737429"
        "6680724428655451920287625003602274950332293164743261731671854982717449538438507060048611809658667867317638"
        "33184598311287771821298520080745220184326171875e-272"},
    {0x1.fffffffffffffp-902, chars_format::scientific, 682,
        "5."
        "9152609308338728981378697798499327644962246531027610329964171365071835878221189598546272075951796068766466"
        "5763532987477555269996942650303620543576247889080726896181560510100169730182093428530941594913492564959988"
        "1112872659132604976329010705515090134440082825018849143887618541396356321661347816480135367905990322170008"
        "7723500540139698106872933229751717139580369959186055974800575775489163574343401023821142902635744292152750"
        "3368014412365611853761226542290600667628860241337852397315504122123444810123881278523886837883534965474859"
        "3361448857310903840575250007204549900664586329486523463343709965434899076877014120097223619317335734635276"
        "6636919662257554364259704016149044036865234375e-272"},
    {0x1.fffffffffffffp-901, chars_format::scientific, 682,
        "1."
        "1830521861667745796275739559699865528992449306205522065992834273014367175644237919709254415190359213753293"
        "3152706597495511053999388530060724108715249577816145379236312102020033946036418685706188318982698512991997"
        "6222574531826520995265802141103018026888016565003769828777523708279271264332269563296027073581198064434001"
        "7544700108027939621374586645950343427916073991837211194960115155097832714868680204764228580527148858430550"
        "0673602882473122370752245308458120133525772048267570479463100824424688962024776255704777367576706993094971"
        "8672289771462180768115050001440909980132917265897304692668741993086979815375402824019444723863467146927055"
        "3327383932451510872851940803229808807373046875e-271"},
    {0x1.fffffffffffffp-900, chars_format::scientific, 681,
        "2."
        "3661043723335491592551479119399731057984898612411044131985668546028734351288475839418508830380718427506586"
        "6305413194991022107998777060121448217430499155632290758472624204040067892072837371412376637965397025983995"
        "2445149063653041990531604282206036053776033130007539657555047416558542528664539126592054147162396128868003"
        "5089400216055879242749173291900686855832147983674422389920230310195665429737360409528457161054297716861100"
        "1347205764946244741504490616916240267051544096535140958926201648849377924049552511409554735153413986189943"
        "7344579542924361536230100002881819960265834531794609385337483986173959630750805648038889447726934293854110"
        "665476786490302174570388160645961761474609375e-271"},
    {0x1.fffffffffffffp-899, chars_format::scientific, 680,
        "4."
        "7322087446670983185102958238799462115969797224822088263971337092057468702576951678837017660761436855013173"
        "2610826389982044215997554120242896434860998311264581516945248408080135784145674742824753275930794051967990"
        "4890298127306083981063208564412072107552066260015079315110094833117085057329078253184108294324792257736007"
        "0178800432111758485498346583801373711664295967348844779840460620391330859474720819056914322108595433722200"
        "2694411529892489483008981233832480534103088193070281917852403297698755848099105022819109470306827972379887"
        "4689159085848723072460200005763639920531669063589218770674967972347919261501611296077778895453868587708221"
        "33095357298060434914077632129192352294921875e-271"},
    {0x1.fffffffffffffp-898, chars_format::scientific, 679,
        "9."
        "4644174893341966370205916477598924231939594449644176527942674184114937405153903357674035321522873710026346"
        "5221652779964088431995108240485792869721996622529163033890496816160271568291349485649506551861588103935980"
        "9780596254612167962126417128824144215104132520030158630220189666234170114658156506368216588649584515472014"
        "0357600864223516970996693167602747423328591934697689559680921240782661718949441638113828644217190867444400"
        "5388823059784978966017962467664961068206176386140563835704806595397511696198210045638218940613655944759774"
        "9378318171697446144920400011527279841063338127178437541349935944695838523003222592155557790907737175416442"
        "6619071459612086982815526425838470458984375e-271"},
    {0x1.fffffffffffffp-897, chars_format::scientific, 679,
        "1."
        "8928834978668393274041183295519784846387918889928835305588534836822987481030780671534807064304574742005269"
        "3044330555992817686399021648097158573944399324505832606778099363232054313658269897129901310372317620787196"
        "1956119250922433592425283425764828843020826504006031726044037933246834022931631301273643317729916903094402"
        "8071520172844703394199338633520549484665718386939537911936184248156532343789888327622765728843438173488880"
        "1077764611956995793203592493532992213641235277228112767140961319079502339239642009127643788122731188951954"
        "9875663634339489228984080002305455968212667625435687508269987188939167704600644518431111558181547435083288"
        "5323814291922417396563105285167694091796875e-270"},
    {0x1.fffffffffffffp-896, chars_format::scientific, 678,
        "3."
        "7857669957336786548082366591039569692775837779857670611177069673645974962061561343069614128609149484010538"
        "6088661111985635372798043296194317147888798649011665213556198726464108627316539794259802620744635241574392"
        "3912238501844867184850566851529657686041653008012063452088075866493668045863262602547286635459833806188805"
        "6143040345689406788398677267041098969331436773879075823872368496313064687579776655245531457686876346977760"
        "2155529223913991586407184987065984427282470554456225534281922638159004678479284018255287576245462377903909"
        "9751327268678978457968160004610911936425335250871375016539974377878335409201289036862223116363094870166577"
        "064762858384483479312621057033538818359375e-270"},
    {0x1.fffffffffffffp-895, chars_format::scientific, 677,
        "7."
        "5715339914673573096164733182079139385551675559715341222354139347291949924123122686139228257218298968021077"
        "2177322223971270745596086592388634295777597298023330427112397452928217254633079588519605241489270483148784"
        "7824477003689734369701133703059315372083306016024126904176151732987336091726525205094573270919667612377611"
        "2286080691378813576797354534082197938662873547758151647744736992626129375159553310491062915373752693955520"
        "4311058447827983172814369974131968854564941108912451068563845276318009356958568036510575152490924755807819"
        "9502654537357956915936320009221823872850670501742750033079948755756670818402578073724446232726189740333154"
        "12952571676896695862524211406707763671875e-270"},
    {0x1.fffffffffffffp-894, chars_format::scientific, 677,
        "1."
        "5143067982934714619232946636415827877110335111943068244470827869458389984824624537227845651443659793604215"
        "4435464444794254149119217318477726859155519459604666085422479490585643450926615917703921048297854096629756"
        "9564895400737946873940226740611863074416661203204825380835230346597467218345305041018914654183933522475522"
        "2457216138275762715359470906816439587732574709551630329548947398525225875031910662098212583074750538791104"
        "0862211689565596634562873994826393770912988221782490213712769055263601871391713607302115030498184951161563"
        "9900530907471591383187264001844364774570134100348550006615989751151334163680515614744889246545237948066630"
        "82590514335379339172504842281341552734375e-269"},
    {0x1.fffffffffffffp-893, chars_format::scientific, 676,
        "3."
        "0286135965869429238465893272831655754220670223886136488941655738916779969649249074455691302887319587208430"
        "8870928889588508298238434636955453718311038919209332170844958981171286901853231835407842096595708193259513"
        "9129790801475893747880453481223726148833322406409650761670460693194934436690610082037829308367867044951044"
        "4914432276551525430718941813632879175465149419103260659097894797050451750063821324196425166149501077582208"
        "1724423379131193269125747989652787541825976443564980427425538110527203742783427214604230060996369902323127"
        "9801061814943182766374528003688729549140268200697100013231979502302668327361031229489778493090475896133261"
        "6518102867075867834500968456268310546875e-269"},
    {0x1.fffffffffffffp-892, chars_format::scientific, 675,
        "6."
        "0572271931738858476931786545663311508441340447772272977883311477833559939298498148911382605774639174416861"
        "7741857779177016596476869273910907436622077838418664341689917962342573803706463670815684193191416386519027"
        "8259581602951787495760906962447452297666644812819301523340921386389868873381220164075658616735734089902088"
        "9828864553103050861437883627265758350930298838206521318195789594100903500127642648392850332299002155164416"
        "3448846758262386538251495979305575083651952887129960854851076221054407485566854429208460121992739804646255"
        "9602123629886365532749056007377459098280536401394200026463959004605336654722062458979556986180951792266523"
        "303620573415173566900193691253662109375e-269"},
    {0x1.fffffffffffffp-891, chars_format::scientific, 675,
        "1."
        "2114454386347771695386357309132662301688268089554454595576662295566711987859699629782276521154927834883372"
        "3548371555835403319295373854782181487324415567683732868337983592468514760741292734163136838638283277303805"
        "5651916320590357499152181392489490459533328962563860304668184277277973774676244032815131723347146817980417"
        "7965772910620610172287576725453151670186059767641304263639157918820180700025528529678570066459800431032883"
        "2689769351652477307650299195861115016730390577425992170970215244210881497113370885841692024398547960929251"
        "1920424725977273106549811201475491819656107280278840005292791800921067330944412491795911397236190358453304"
        "660724114683034713380038738250732421875e-268"},
    {0x1.fffffffffffffp-890, chars_format::scientific, 674,
        "2."
        "4228908772695543390772714618265324603376536179108909191153324591133423975719399259564553042309855669766744"
        "7096743111670806638590747709564362974648831135367465736675967184937029521482585468326273677276566554607611"
        "1303832641180714998304362784978980919066657925127720609336368554555947549352488065630263446694293635960835"
        "5931545821241220344575153450906303340372119535282608527278315837640361400051057059357140132919600862065766"
        "5379538703304954615300598391722230033460781154851984341940430488421762994226741771683384048797095921858502"
        "3840849451954546213099622402950983639312214560557680010585583601842134661888824983591822794472380716906609"
        "32144822936606942676007747650146484375e-268"},
    {0x1.fffffffffffffp-889, chars_format::scientific, 673,
        "4."
        "8457817545391086781545429236530649206753072358217818382306649182266847951438798519129106084619711339533489"
        "4193486223341613277181495419128725949297662270734931473351934369874059042965170936652547354553133109215222"
        "2607665282361429996608725569957961838133315850255441218672737109111895098704976131260526893388587271921671"
        "1863091642482440689150306901812606680744239070565217054556631675280722800102114118714280265839201724131533"
        "0759077406609909230601196783444460066921562309703968683880860976843525988453483543366768097594191843717004"
        "7681698903909092426199244805901967278624429121115360021171167203684269323777649967183645588944761433813218"
        "6428964587321388535201549530029296875e-268"},
    {0x1.fffffffffffffp-888, chars_format::scientific, 672,
        "9."
        "6915635090782173563090858473061298413506144716435636764613298364533695902877597038258212169239422679066978"
        "8386972446683226554362990838257451898595324541469862946703868739748118085930341873305094709106266218430444"
        "5215330564722859993217451139915923676266631700510882437345474218223790197409952262521053786777174543843342"
        "3726183284964881378300613803625213361488478141130434109113263350561445600204228237428560531678403448263066"
        "1518154813219818461202393566888920133843124619407937367761721953687051976906967086733536195188383687434009"
        "5363397807818184852398489611803934557248858242230720042342334407368538647555299934367291177889522867626437"
        "285792917464277707040309906005859375e-268"},
    {0x1.fffffffffffffp-887, chars_format::scientific, 672,
        "1."
        "9383127018156434712618171694612259682701228943287127352922659672906739180575519407651642433847884535813395"
        "7677394489336645310872598167651490379719064908293972589340773747949623617186068374661018941821253243686088"
        "9043066112944571998643490227983184735253326340102176487469094843644758039481990452504210757355434908768668"
        "4745236656992976275660122760725042672297695628226086821822652670112289120040845647485712106335680689652613"
        "2303630962643963692240478713377784026768624923881587473552344390737410395381393417346707239037676737486801"
        "9072679561563636970479697922360786911449771648446144008468466881473707729511059986873458235577904573525287"
        "457158583492855541408061981201171875e-267"},
    {0x1.fffffffffffffp-886, chars_format::scientific, 671,
        "3."
        "8766254036312869425236343389224519365402457886574254705845319345813478361151038815303284867695769071626791"
        "5354788978673290621745196335302980759438129816587945178681547495899247234372136749322037883642506487372177"
        "8086132225889143997286980455966369470506652680204352974938189687289516078963980905008421514710869817537336"
        "9490473313985952551320245521450085344595391256452173643645305340224578240081691294971424212671361379305226"
        "4607261925287927384480957426755568053537249847763174947104688781474820790762786834693414478075353474973603"
        "8145359123127273940959395844721573822899543296892288016936933762947415459022119973746916471155809147050574"
        "91431716698571108281612396240234375e-267"},
    {0x1.fffffffffffffp-885, chars_format::scientific, 670,
        "7."
        "7532508072625738850472686778449038730804915773148509411690638691626956722302077630606569735391538143253583"
        "0709577957346581243490392670605961518876259633175890357363094991798494468744273498644075767285012974744355"
        "6172264451778287994573960911932738941013305360408705949876379374579032157927961810016843029421739635074673"
        "8980946627971905102640491042900170689190782512904347287290610680449156480163382589942848425342722758610452"
        "9214523850575854768961914853511136107074499695526349894209377562949641581525573669386828956150706949947207"
        "6290718246254547881918791689443147645799086593784576033873867525894830918044239947493832942311618294101149"
        "8286343339714221656322479248046875e-267"},
    {0x1.fffffffffffffp-884, chars_format::scientific, 670,
        "1."
        "5506501614525147770094537355689807746160983154629701882338127738325391344460415526121313947078307628650716"
        "6141915591469316248698078534121192303775251926635178071472618998359698893748854699728815153457002594948871"
        "1234452890355657598914792182386547788202661072081741189975275874915806431585592362003368605884347927014934"
        "7796189325594381020528098208580034137838156502580869457458122136089831296032676517988569685068544551722090"
        "5842904770115170953792382970702227221414899939105269978841875512589928316305114733877365791230141389989441"
        "5258143649250909576383758337888629529159817318756915206774773505178966183608847989498766588462323658820229"
        "9657268667942844331264495849609375e-266"},
    {0x1.fffffffffffffp-883, chars_format::scientific, 669,
        "3."
        "1013003229050295540189074711379615492321966309259403764676255476650782688920831052242627894156615257301433"
        "2283831182938632497396157068242384607550503853270356142945237996719397787497709399457630306914005189897742"
        "2468905780711315197829584364773095576405322144163482379950551749831612863171184724006737211768695854029869"
        "5592378651188762041056196417160068275676313005161738914916244272179662592065353035977139370137089103444181"
        "1685809540230341907584765941404454442829799878210539957683751025179856632610229467754731582460282779978883"
        "0516287298501819152767516675777259058319634637513830413549547010357932367217695978997533176924647317640459"
        "931453733588568866252899169921875e-266"},
    {0x1.fffffffffffffp-882, chars_format::scientific, 668,
        "6."
        "2026006458100591080378149422759230984643932618518807529352510953301565377841662104485255788313230514602866"
        "4567662365877264994792314136484769215101007706540712285890475993438795574995418798915260613828010379795484"
        "4937811561422630395659168729546191152810644288326964759901103499663225726342369448013474423537391708059739"
        "1184757302377524082112392834320136551352626010323477829832488544359325184130706071954278740274178206888362"
        "3371619080460683815169531882808908885659599756421079915367502050359713265220458935509463164920565559957766"
        "1032574597003638305535033351554518116639269275027660827099094020715864734435391957995066353849294635280919"
        "86290746717713773250579833984375e-266"},
    {0x1.fffffffffffffp-881, chars_format::scientific, 668,
        "1."
        "2405201291620118216075629884551846196928786523703761505870502190660313075568332420897051157662646102920573"
        "2913532473175452998958462827296953843020201541308142457178095198687759114999083759783052122765602075959096"
        "8987562312284526079131833745909238230562128857665392951980220699932645145268473889602694884707478341611947"
        "8236951460475504816422478566864027310270525202064695565966497708871865036826141214390855748054835641377672"
        "4674323816092136763033906376561781777131919951284215983073500410071942653044091787101892632984113111991553"
        "2206514919400727661107006670310903623327853855005532165419818804143172946887078391599013270769858927056183"
        "97258149343542754650115966796875e-265"},
    {0x1.fffffffffffffp-880, chars_format::scientific, 667,
        "2."
        "4810402583240236432151259769103692393857573047407523011741004381320626151136664841794102315325292205841146"
        "5827064946350905997916925654593907686040403082616284914356190397375518229998167519566104245531204151918193"
        "7975124624569052158263667491818476461124257715330785903960441399865290290536947779205389769414956683223895"
        "6473902920951009632844957133728054620541050404129391131932995417743730073652282428781711496109671282755344"
        "9348647632184273526067812753123563554263839902568431966147000820143885306088183574203785265968226223983106"
        "4413029838801455322214013340621807246655707710011064330839637608286345893774156783198026541539717854112367"
        "9451629868708550930023193359375e-265"},
    {0x1.fffffffffffffp-879, chars_format::scientific, 666,
        "4."
        "9620805166480472864302519538207384787715146094815046023482008762641252302273329683588204630650584411682293"
        "1654129892701811995833851309187815372080806165232569828712380794751036459996335039132208491062408303836387"
        "5950249249138104316527334983636952922248515430661571807920882799730580581073895558410779538829913366447791"
        "2947805841902019265689914267456109241082100808258782263865990835487460147304564857563422992219342565510689"
        "8697295264368547052135625506247127108527679805136863932294001640287770612176367148407570531936452447966212"
        "8826059677602910644428026681243614493311415420022128661679275216572691787548313566396053083079435708224735"
        "890325973741710186004638671875e-265"},
    {0x1.fffffffffffffp-878, chars_format::scientific, 665,
        "9."
        "9241610332960945728605039076414769575430292189630092046964017525282504604546659367176409261301168823364586"
        "3308259785403623991667702618375630744161612330465139657424761589502072919992670078264416982124816607672775"
        "1900498498276208633054669967273905844497030861323143615841765599461161162147791116821559077659826732895582"
        "5895611683804038531379828534912218482164201616517564527731981670974920294609129715126845984438685131021379"
        "7394590528737094104271251012494254217055359610273727864588003280575541224352734296815141063872904895932425"
        "7652119355205821288856053362487228986622830840044257323358550433145383575096627132792106166158871416449471"
        "78065194748342037200927734375e-265"},
    {0x1.fffffffffffffp-877, chars_format::scientific, 665,
        "1."
        "9848322066592189145721007815282953915086058437926018409392803505056500920909331873435281852260233764672917"
        "2661651957080724798333540523675126148832322466093027931484952317900414583998534015652883396424963321534555"
        "0380099699655241726610933993454781168899406172264628723168353119892232232429558223364311815531965346579116"
        "5179122336760807706275965706982443696432840323303512905546396334194984058921825943025369196887737026204275"
        "9478918105747418820854250202498850843411071922054745572917600656115108244870546859363028212774580979186485"
        "1530423871041164257771210672497445797324566168008851464671710086629076715019325426558421233231774283289894"
        "35613038949668407440185546875e-264"},
    {0x1.fffffffffffffp-876, chars_format::scientific, 664,
        "3."
        "9696644133184378291442015630565907830172116875852036818785607010113001841818663746870563704520467529345834"
        "5323303914161449596667081047350252297664644932186055862969904635800829167997068031305766792849926643069110"
        "0760199399310483453221867986909562337798812344529257446336706239784464464859116446728623631063930693158233"
        "0358244673521615412551931413964887392865680646607025811092792668389968117843651886050738393775474052408551"
        "8957836211494837641708500404997701686822143844109491145835201312230216489741093718726056425549161958372970"
        "3060847742082328515542421344994891594649132336017702929343420173258153430038650853116842466463548566579788"
        "7122607789933681488037109375e-264"},
    {0x1.fffffffffffffp-875, chars_format::scientific, 663,
        "7."
        "9393288266368756582884031261131815660344233751704073637571214020226003683637327493741127409040935058691669"
        "0646607828322899193334162094700504595329289864372111725939809271601658335994136062611533585699853286138220"
        "1520398798620966906443735973819124675597624689058514892673412479568928929718232893457247262127861386316466"
        "0716489347043230825103862827929774785731361293214051622185585336779936235687303772101476787550948104817103"
        "7915672422989675283417000809995403373644287688218982291670402624460432979482187437452112851098323916745940"
        "6121695484164657031084842689989783189298264672035405858686840346516306860077301706233684932927097133159577"
        "424521557986736297607421875e-264"},
    {0x1.fffffffffffffp-874, chars_format::scientific, 663,
        "1."
        "5878657653273751316576806252226363132068846750340814727514242804045200736727465498748225481808187011738333"
        "8129321565664579838666832418940100919065857972874422345187961854320331667198827212522306717139970657227644"
        "0304079759724193381288747194763824935119524937811702978534682495913785785943646578691449452425572277263293"
        "2143297869408646165020772565585954957146272258642810324437117067355987247137460754420295357510189620963420"
        "7583134484597935056683400161999080674728857537643796458334080524892086595896437487490422570219664783349188"
        "1224339096832931406216968537997956637859652934407081171737368069303261372015460341246736986585419426631915"
        "484904311597347259521484375e-263"},
    {0x1.fffffffffffffp-873, chars_format::scientific, 662,
        "3."
        "1757315306547502633153612504452726264137693500681629455028485608090401473454930997496450963616374023476667"
        "6258643131329159677333664837880201838131715945748844690375923708640663334397654425044613434279941314455288"
        "0608159519448386762577494389527649870239049875623405957069364991827571571887293157382898904851144554526586"
        "4286595738817292330041545131171909914292544517285620648874234134711974494274921508840590715020379241926841"
        "5166268969195870113366800323998161349457715075287592916668161049784173191792874974980845140439329566698376"
        "2448678193665862812433937075995913275719305868814162343474736138606522744030920682493473973170838853263830"
        "96980862319469451904296875e-263"},
    {0x1.fffffffffffffp-872, chars_format::scientific, 661,
        "6."
        "3514630613095005266307225008905452528275387001363258910056971216180802946909861994992901927232748046953335"
        "2517286262658319354667329675760403676263431891497689380751847417281326668795308850089226868559882628910576"
        "1216319038896773525154988779055299740478099751246811914138729983655143143774586314765797809702289109053172"
        "8573191477634584660083090262343819828585089034571241297748468269423948988549843017681181430040758483853683"
        "0332537938391740226733600647996322698915430150575185833336322099568346383585749949961690280878659133396752"
        "4897356387331725624867874151991826551438611737628324686949472277213045488061841364986947946341677706527661"
        "9396172463893890380859375e-263"},
    {0x1.fffffffffffffp-871, chars_format::scientific, 661,
        "1."
        "2702926122619001053261445001781090505655077400272651782011394243236160589381972398998580385446549609390667"
        "0503457252531663870933465935152080735252686378299537876150369483456265333759061770017845373711976525782115"
        "2243263807779354705030997755811059948095619950249362382827745996731028628754917262953159561940457821810634"
        "5714638295526916932016618052468763965717017806914248259549693653884789797709968603536236286008151696770736"
        "6066507587678348045346720129599264539783086030115037166667264419913669276717149989992338056175731826679350"
        "4979471277466345124973574830398365310287722347525664937389894455442609097612368272997389589268335541305532"
        "3879234492778778076171875e-262"},
    {0x1.fffffffffffffp-870, chars_format::scientific, 660,
        "2."
        "5405852245238002106522890003562181011310154800545303564022788486472321178763944797997160770893099218781334"
        "1006914505063327741866931870304161470505372756599075752300738966912530667518123540035690747423953051564230"
        "4486527615558709410061995511622119896191239900498724765655491993462057257509834525906319123880915643621269"
        "1429276591053833864033236104937527931434035613828496519099387307769579595419937207072472572016303393541473"
        "2133015175356696090693440259198529079566172060230074333334528839827338553434299979984676112351463653358700"
        "9958942554932690249947149660796730620575444695051329874779788910885218195224736545994779178536671082611064"
        "775846898555755615234375e-262"},
    {0x1.fffffffffffffp-869, chars_format::scientific, 659,
        "5."
        "0811704490476004213045780007124362022620309601090607128045576972944642357527889595994321541786198437562668"
        "2013829010126655483733863740608322941010745513198151504601477933825061335036247080071381494847906103128460"
        "8973055231117418820123991023244239792382479800997449531310983986924114515019669051812638247761831287242538"
        "2858553182107667728066472209875055862868071227656993038198774615539159190839874414144945144032606787082946"
        "4266030350713392181386880518397058159132344120460148666669057679654677106868599959969352224702927306717401"
        "9917885109865380499894299321593461241150889390102659749559577821770436390449473091989558357073342165222129"
        "55169379711151123046875e-262"},
    {0x1.fffffffffffffp-868, chars_format::scientific, 659,
        "1."
        "0162340898095200842609156001424872404524061920218121425609115394588928471505577919198864308357239687512533"
        "6402765802025331096746772748121664588202149102639630300920295586765012267007249416014276298969581220625692"
        "1794611046223483764024798204648847958476495960199489906262196797384822903003933810362527649552366257448507"
        "6571710636421533545613294441975011172573614245531398607639754923107831838167974882828989028806521357416589"
        "2853206070142678436277376103679411631826468824092029733333811535930935421373719991993870444940585461343480"
        "3983577021973076099978859864318692248230177878020531949911915564354087278089894618397911671414668433044425"
        "91033875942230224609375e-261"},
    {0x1.fffffffffffffp-867, chars_format::scientific, 658,
        "2."
        "0324681796190401685218312002849744809048123840436242851218230789177856943011155838397728616714479375025067"
        "2805531604050662193493545496243329176404298205279260601840591173530024534014498832028552597939162441251384"
        "3589222092446967528049596409297695916952991920398979812524393594769645806007867620725055299104732514897015"
        "3143421272843067091226588883950022345147228491062797215279509846215663676335949765657978057613042714833178"
        "5706412140285356872554752207358823263652937648184059466667623071861870842747439983987740889881170922686960"
        "7967154043946152199957719728637384496460355756041063899823831128708174556179789236795823342829336866088851"
        "8206775188446044921875e-261"},
    {0x1.fffffffffffffp-866, chars_format::scientific, 657,
        "4."
        "0649363592380803370436624005699489618096247680872485702436461578355713886022311676795457233428958750050134"
        "5611063208101324386987090992486658352808596410558521203681182347060049068028997664057105195878324882502768"
        "7178444184893935056099192818595391833905983840797959625048787189539291612015735241450110598209465029794030"
        "6286842545686134182453177767900044690294456982125594430559019692431327352671899531315956115226085429666357"
        "1412824280570713745109504414717646527305875296368118933335246143723741685494879967975481779762341845373921"
        "5934308087892304399915439457274768992920711512082127799647662257416349112359578473591646685658673732177703"
        "641355037689208984375e-261"},
    {0x1.fffffffffffffp-865, chars_format::scientific, 656,
        "8."
        "1298727184761606740873248011398979236192495361744971404872923156711427772044623353590914466857917500100269"
        "1222126416202648773974181984973316705617192821117042407362364694120098136057995328114210391756649765005537"
        "4356888369787870112198385637190783667811967681595919250097574379078583224031470482900221196418930059588061"
        "2573685091372268364906355535800089380588913964251188861118039384862654705343799062631912230452170859332714"
        "2825648561141427490219008829435293054611750592736237866670492287447483370989759935950963559524683690747843"
        "1868616175784608799830878914549537985841423024164255599295324514832698224719156947183293371317347464355407"
        "28271007537841796875e-261"},
    {0x1.fffffffffffffp-864, chars_format::scientific, 656,
        "1."
        "6259745436952321348174649602279795847238499072348994280974584631342285554408924670718182893371583500020053"
        "8244425283240529754794836396994663341123438564223408481472472938824019627211599065622842078351329953001107"
        "4871377673957574022439677127438156733562393536319183850019514875815716644806294096580044239283786011917612"
        "2514737018274453672981271107160017876117782792850237772223607876972530941068759812526382446090434171866542"
        "8565129712228285498043801765887058610922350118547247573334098457489496674197951987190192711904936738149568"
        "6373723235156921759966175782909907597168284604832851119859064902966539644943831389436658674263469492871081"
        "45654201507568359375e-260"},
    {0x1.fffffffffffffp-863, chars_format::scientific, 655,
        "3."
        "2519490873904642696349299204559591694476998144697988561949169262684571108817849341436365786743167000040107"
        "6488850566481059509589672793989326682246877128446816962944945877648039254423198131245684156702659906002214"
        "9742755347915148044879354254876313467124787072638367700039029751631433289612588193160088478567572023835224"
        "5029474036548907345962542214320035752235565585700475544447215753945061882137519625052764892180868343733085"
        "7130259424456570996087603531774117221844700237094495146668196914978993348395903974380385423809873476299137"
        "2747446470313843519932351565819815194336569209665702239718129805933079289887662778873317348526938985742162"
        "9130840301513671875e-260"},
    {0x1.fffffffffffffp-862, chars_format::scientific, 654,
        "6."
        "5038981747809285392698598409119183388953996289395977123898338525369142217635698682872731573486334000080215"
        "2977701132962119019179345587978653364493754256893633925889891755296078508846396262491368313405319812004429"
        "9485510695830296089758708509752626934249574145276735400078059503262866579225176386320176957135144047670449"
        "0058948073097814691925084428640071504471131171400951088894431507890123764275039250105529784361736687466171"
        "4260518848913141992175207063548234443689400474188990293336393829957986696791807948760770847619746952598274"
        "5494892940627687039864703131639630388673138419331404479436259611866158579775325557746634697053877971484325"
        "826168060302734375e-260"},
    {0x1.fffffffffffffp-861, chars_format::scientific, 654,
        "1."
        "3007796349561857078539719681823836677790799257879195424779667705073828443527139736574546314697266800016043"
        "0595540226592423803835869117595730672898750851378726785177978351059215701769279252498273662681063962400885"
        "9897102139166059217951741701950525386849914829055347080015611900652573315845035277264035391427028809534089"
        "8011789614619562938385016885728014300894226234280190217778886301578024752855007850021105956872347337493234"
        "2852103769782628398435041412709646888737880094837798058667278765991597339358361589752154169523949390519654"
        "9098978588125537407972940626327926077734627683866280895887251922373231715955065111549326939410775594296865"
        "165233612060546875e-259"},
    {0x1.fffffffffffffp-860, chars_format::scientific, 653,
        "2."
        "6015592699123714157079439363647673355581598515758390849559335410147656887054279473149092629394533600032086"
        "1191080453184847607671738235191461345797501702757453570355956702118431403538558504996547325362127924801771"
        "9794204278332118435903483403901050773699829658110694160031223801305146631690070554528070782854057619068179"
        "6023579229239125876770033771456028601788452468560380435557772603156049505710015700042211913744694674986468"
        "5704207539565256796870082825419293777475760189675596117334557531983194678716723179504308339047898781039309"
        "8197957176251074815945881252655852155469255367732561791774503844746463431910130223098653878821551188593730"
        "33046722412109375e-259"},
    {0x1.fffffffffffffp-859, chars_format::scientific, 652,
        "5."
        "2031185398247428314158878727295346711163197031516781699118670820295313774108558946298185258789067200064172"
        "2382160906369695215343476470382922691595003405514907140711913404236862807077117009993094650724255849603543"
        "9588408556664236871806966807802101547399659316221388320062447602610293263380141109056141565708115238136359"
        "2047158458478251753540067542912057203576904937120760871115545206312099011420031400084423827489389349972937"
        "1408415079130513593740165650838587554951520379351192234669115063966389357433446359008616678095797562078619"
        "6395914352502149631891762505311704310938510735465123583549007689492926863820260446197307757643102377187460"
        "6609344482421875e-259"},
    {0x1.fffffffffffffp-858, chars_format::scientific, 652,
        "1."
        "0406237079649485662831775745459069342232639406303356339823734164059062754821711789259637051757813440012834"
        "4476432181273939043068695294076584538319000681102981428142382680847372561415423401998618930144851169920708"
        "7917681711332847374361393361560420309479931863244277664012489520522058652676028221811228313141623047627271"
        "8409431691695650350708013508582411440715380987424152174223109041262419802284006280016884765497877869994587"
        "4281683015826102718748033130167717510990304075870238446933823012793277871486689271801723335619159512415723"
        "9279182870500429926378352501062340862187702147093024716709801537898585372764052089239461551528620475437492"
        "1321868896484375e-258"},
    {0x1.fffffffffffffp-857, chars_format::scientific, 651,
        "2."
        "0812474159298971325663551490918138684465278812606712679647468328118125509643423578519274103515626880025668"
        "8952864362547878086137390588153169076638001362205962856284765361694745122830846803997237860289702339841417"
        "5835363422665694748722786723120840618959863726488555328024979041044117305352056443622456626283246095254543"
        "6818863383391300701416027017164822881430761974848304348446218082524839604568012560033769530995755739989174"
        "8563366031652205437496066260335435021980608151740476893867646025586555742973378543603446671238319024831447"
        "8558365741000859852756705002124681724375404294186049433419603075797170745528104178478923103057240950874984"
        "264373779296875e-258"},
    {0x1.fffffffffffffp-856, chars_format::scientific, 650,
        "4."
        "1624948318597942651327102981836277368930557625213425359294936656236251019286847157038548207031253760051337"
        "7905728725095756172274781176306338153276002724411925712569530723389490245661693607994475720579404679682835"
        "1670726845331389497445573446241681237919727452977110656049958082088234610704112887244913252566492190509087"
        "3637726766782601402832054034329645762861523949696608696892436165049679209136025120067539061991511479978349"
        "7126732063304410874992132520670870043961216303480953787735292051173111485946757087206893342476638049662895"
        "7116731482001719705513410004249363448750808588372098866839206151594341491056208356957846206114481901749968"
        "52874755859375e-258"},
    {0x1.fffffffffffffp-855, chars_format::scientific, 649,
        "8."
        "3249896637195885302654205963672554737861115250426850718589873312472502038573694314077096414062507520102675"
        "5811457450191512344549562352612676306552005448823851425139061446778980491323387215988951441158809359365670"
        "3341453690662778994891146892483362475839454905954221312099916164176469221408225774489826505132984381018174"
        "7275453533565202805664108068659291525723047899393217393784872330099358418272050240135078123983022959956699"
        "4253464126608821749984265041341740087922432606961907575470584102346222971893514174413786684953276099325791"
        "4233462964003439411026820008498726897501617176744197733678412303188682982112416713915692412228963803499937"
        "0574951171875e-258"},
    {0x1.fffffffffffffp-854, chars_format::scientific, 649,
        "1."
        "6649979327439177060530841192734510947572223050085370143717974662494500407714738862815419282812501504020535"
        "1162291490038302468909912470522535261310401089764770285027812289355796098264677443197790288231761871873134"
        "0668290738132555798978229378496672495167890981190844262419983232835293844281645154897965301026596876203634"
        "9455090706713040561132821613731858305144609579878643478756974466019871683654410048027015624796604591991339"
        "8850692825321764349996853008268348017584486521392381515094116820469244594378702834882757336990655219865158"
        "2846692592800687882205364001699745379500323435348839546735682460637736596422483342783138482445792760699987"
        "4114990234375e-257"},
    {0x1.fffffffffffffp-853, chars_format::scientific, 648,
        "3."
        "3299958654878354121061682385469021895144446100170740287435949324989000815429477725630838565625003008041070"
        "2324582980076604937819824941045070522620802179529540570055624578711592196529354886395580576463523743746268"
        "1336581476265111597956458756993344990335781962381688524839966465670587688563290309795930602053193752407269"
        "8910181413426081122265643227463716610289219159757286957513948932039743367308820096054031249593209183982679"
        "7701385650643528699993706016536696035168973042784763030188233640938489188757405669765514673981310439730316"
        "5693385185601375764410728003399490759000646870697679093471364921275473192844966685566276964891585521399974"
        "822998046875e-257"},
    {0x1.fffffffffffffp-852, chars_format::scientific, 647,
        "6."
        "6599917309756708242123364770938043790288892200341480574871898649978001630858955451261677131250006016082140"
        "4649165960153209875639649882090141045241604359059081140111249157423184393058709772791161152927047487492536"
        "2673162952530223195912917513986689980671563924763377049679932931341175377126580619591861204106387504814539"
        "7820362826852162244531286454927433220578438319514573915027897864079486734617640192108062499186418367965359"
        "5402771301287057399987412033073392070337946085569526060376467281876978377514811339531029347962620879460633"
        "1386770371202751528821456006798981518001293741395358186942729842550946385689933371132553929783171042799949"
        "64599609375e-257"},
    {0x1.fffffffffffffp-851, chars_format::scientific, 647,
        "1."
        "3319983461951341648424672954187608758057778440068296114974379729995600326171791090252335426250001203216428"
        "0929833192030641975127929976418028209048320871811816228022249831484636878611741954558232230585409497498507"
        "2534632590506044639182583502797337996134312784952675409935986586268235075425316123918372240821277500962907"
        "9564072565370432448906257290985486644115687663902914783005579572815897346923528038421612499837283673593071"
        "9080554260257411479997482406614678414067589217113905212075293456375395675502962267906205869592524175892126"
        "6277354074240550305764291201359796303600258748279071637388545968510189277137986674226510785956634208559989"
        "92919921875e-256"},
    {0x1.fffffffffffffp-850, chars_format::scientific, 646,
        "2."
        "6639966923902683296849345908375217516115556880136592229948759459991200652343582180504670852500002406432856"
        "1859666384061283950255859952836056418096641743623632456044499662969273757223483909116464461170818994997014"
        "5069265181012089278365167005594675992268625569905350819871973172536470150850632247836744481642555001925815"
        "9128145130740864897812514581970973288231375327805829566011159145631794693847056076843224999674567347186143"
        "8161108520514822959994964813229356828135178434227810424150586912750791351005924535812411739185048351784253"
        "2554708148481100611528582402719592607200517496558143274777091937020378554275973348453021571913268417119979"
        "8583984375e-256"},
    {0x1.fffffffffffffp-849, chars_format::scientific, 645,
        "5."
        "3279933847805366593698691816750435032231113760273184459897518919982401304687164361009341705000004812865712"
        "3719332768122567900511719905672112836193283487247264912088999325938547514446967818232928922341637989994029"
        "0138530362024178556730334011189351984537251139810701639743946345072940301701264495673488963285110003851631"
        "8256290261481729795625029163941946576462750655611659132022318291263589387694112153686449999349134694372287"
        "6322217041029645919989929626458713656270356868455620848301173825501582702011849071624823478370096703568506"
        "5109416296962201223057164805439185214401034993116286549554183874040757108551946696906043143826536834239959"
        "716796875e-256"},
    {0x1.fffffffffffffp-848, chars_format::scientific, 645,
        "1."
        "0655986769561073318739738363350087006446222752054636891979503783996480260937432872201868341000000962573142"
        "4743866553624513580102343981134422567238656697449452982417799865187709502889393563646585784468327597998805"
        "8027706072404835711346066802237870396907450227962140327948789269014588060340252899134697792657022000770326"
        "3651258052296345959125005832788389315292550131122331826404463658252717877538822430737289999869826938874457"
        "5264443408205929183997985925291742731254071373691124169660234765100316540402369814324964695674019340713701"
        "3021883259392440244611432961087837042880206998623257309910836774808151421710389339381208628765307366847991"
        "943359375e-255"},
    {0x1.fffffffffffffp-847, chars_format::scientific, 644,
        "2."
        "1311973539122146637479476726700174012892445504109273783959007567992960521874865744403736682000001925146284"
        "9487733107249027160204687962268845134477313394898905964835599730375419005778787127293171568936655195997611"
        "6055412144809671422692133604475740793814900455924280655897578538029176120680505798269395585314044001540652"
        "7302516104592691918250011665576778630585100262244663652808927316505435755077644861474579999739653877748915"
        "0528886816411858367995971850583485462508142747382248339320469530200633080804739628649929391348038681427402"
        "6043766518784880489222865922175674085760413997246514619821673549616302843420778678762417257530614733695983"
        "88671875e-255"},
    {0x1.fffffffffffffp-846, chars_format::scientific, 643,
        "4."
        "2623947078244293274958953453400348025784891008218547567918015135985921043749731488807473364000003850292569"
        "8975466214498054320409375924537690268954626789797811929671199460750838011557574254586343137873310391995223"
        "2110824289619342845384267208951481587629800911848561311795157076058352241361011596538791170628088003081305"
        "4605032209185383836500023331153557261170200524489327305617854633010871510155289722949159999479307755497830"
        "1057773632823716735991943701166970925016285494764496678640939060401266161609479257299858782696077362854805"
        "2087533037569760978445731844351348171520827994493029239643347099232605686841557357524834515061229467391967"
        "7734375e-255"},
    {0x1.fffffffffffffp-845, chars_format::scientific, 642,
        "8."
        "5247894156488586549917906906800696051569782016437095135836030271971842087499462977614946728000007700585139"
        "7950932428996108640818751849075380537909253579595623859342398921501676023115148509172686275746620783990446"
        "4221648579238685690768534417902963175259601823697122623590314152116704482722023193077582341256176006162610"
        "9210064418370767673000046662307114522340401048978654611235709266021743020310579445898319998958615510995660"
        "2115547265647433471983887402333941850032570989528993357281878120802532323218958514599717565392154725709610"
        "4175066075139521956891463688702696343041655988986058479286694198465211373683114715049669030122458934783935"
        "546875e-255"},
    {0x1.fffffffffffffp-844, chars_format::scientific, 642,
        "1."
        "7049578831297717309983581381360139210313956403287419027167206054394368417499892595522989345600001540117027"
        "9590186485799221728163750369815076107581850715919124771868479784300335204623029701834537255149324156798089"
        "2844329715847737138153706883580592635051920364739424524718062830423340896544404638615516468251235201232522"
        "1842012883674153534600009332461422904468080209795730922247141853204348604062115889179663999791723102199132"
        "0423109453129486694396777480466788370006514197905798671456375624160506464643791702919943513078430945141922"
        "0835013215027904391378292737740539268608331197797211695857338839693042274736622943009933806024491786956787"
        "109375e-254"},
    {0x1.fffffffffffffp-843, chars_format::scientific, 641,
        "3."
        "4099157662595434619967162762720278420627912806574838054334412108788736834999785191045978691200003080234055"
        "9180372971598443456327500739630152215163701431838249543736959568600670409246059403669074510298648313596178"
        "5688659431695474276307413767161185270103840729478849049436125660846681793088809277231032936502470402465044"
        "3684025767348307069200018664922845808936160419591461844494283706408697208124231778359327999583446204398264"
        "0846218906258973388793554960933576740013028395811597342912751248321012929287583405839887026156861890283844"
        "1670026430055808782756585475481078537216662395594423391714677679386084549473245886019867612048983573913574"
        "21875e-254"},
    {0x1.fffffffffffffp-842, chars_format::scientific, 640,
        "6."
        "8198315325190869239934325525440556841255825613149676108668824217577473669999570382091957382400006160468111"
        "8360745943196886912655001479260304430327402863676499087473919137201340818492118807338149020597296627192357"
        "1377318863390948552614827534322370540207681458957698098872251321693363586177618554462065873004940804930088"
        "7368051534696614138400037329845691617872320839182923688988567412817394416248463556718655999166892408796528"
        "1692437812517946777587109921867153480026056791623194685825502496642025858575166811679774052313723780567688"
        "3340052860111617565513170950962157074433324791188846783429355358772169098946491772039735224097967147827148"
        "4375e-254"},
    {0x1.fffffffffffffp-841, chars_format::scientific, 640,
        "1."
        "3639663065038173847986865105088111368251165122629935221733764843515494733999914076418391476480001232093622"
        "3672149188639377382531000295852060886065480572735299817494783827440268163698423761467629804119459325438471"
        "4275463772678189710522965506864474108041536291791539619774450264338672717235523710892413174600988160986017"
        "7473610306939322827680007465969138323574464167836584737797713482563478883249692711343731199833378481759305"
        "6338487562503589355517421984373430696005211358324638937165100499328405171715033362335954810462744756113537"
        "6668010572022323513102634190192431414886664958237769356685871071754433819789298354407947044819593429565429"
        "6875e-253"},
    {0x1.fffffffffffffp-840, chars_format::scientific, 639,
        "2."
        "7279326130076347695973730210176222736502330245259870443467529687030989467999828152836782952960002464187244"
        "7344298377278754765062000591704121772130961145470599634989567654880536327396847522935259608238918650876942"
        "8550927545356379421045931013728948216083072583583079239548900528677345434471047421784826349201976321972035"
        "4947220613878645655360014931938276647148928335673169475595426965126957766499385422687462399666756963518611"
        "2676975125007178711034843968746861392010422716649277874330200998656810343430066724671909620925489512227075"
        "3336021144044647026205268380384862829773329916475538713371742143508867639578596708815894089639186859130859"
        "375e-253"},
    {0x1.fffffffffffffp-839, chars_format::scientific, 638,
        "5."
        "4558652260152695391947460420352445473004660490519740886935059374061978935999656305673565905920004928374489"
        "4688596754557509530124001183408243544261922290941199269979135309761072654793695045870519216477837301753885"
        "7101855090712758842091862027457896432166145167166158479097801057354690868942094843569652698403952643944070"
        "9894441227757291310720029863876553294297856671346338951190853930253915532998770845374924799333513927037222"
        "5353950250014357422069687937493722784020845433298555748660401997313620686860133449343819241850979024454150"
        "6672042288089294052410536760769725659546659832951077426743484287017735279157193417631788179278373718261718"
        "75e-253"},
    {0x1.fffffffffffffp-838, chars_format::scientific, 638,
        "1."
        "0911730452030539078389492084070489094600932098103948177387011874812395787199931261134713181184000985674897"
        "8937719350911501906024800236681648708852384458188239853995827061952214530958739009174103843295567460350777"
        "1420371018142551768418372405491579286433229033433231695819560211470938173788418968713930539680790528788814"
        "1978888245551458262144005972775310658859571334269267790238170786050783106599754169074984959866702785407444"
        "5070790050002871484413937587498744556804169086659711149732080399462724137372026689868763848370195804890830"
        "1334408457617858810482107352153945131909331966590215485348696857403547055831438683526357635855674743652343"
        "75e-252"},
    {0x1.fffffffffffffp-837, chars_format::scientific, 637,
        "2."
        "1823460904061078156778984168140978189201864196207896354774023749624791574399862522269426362368001971349795"
        "7875438701823003812049600473363297417704768916376479707991654123904429061917478018348207686591134920701554"
        "2840742036285103536836744810983158572866458066866463391639120422941876347576837937427861079361581057577628"
        "3957776491102916524288011945550621317719142668538535580476341572101566213199508338149969919733405570814889"
        "0141580100005742968827875174997489113608338173319422299464160798925448274744053379737527696740391609781660"
        "2668816915235717620964214704307890263818663933180430970697393714807094111662877367052715271711349487304687"
        "5e-252"},
    {0x1.fffffffffffffp-836, chars_format::scientific, 636,
        "4."
        "3646921808122156313557968336281956378403728392415792709548047499249583148799725044538852724736003942699591"
        "5750877403646007624099200946726594835409537832752959415983308247808858123834956036696415373182269841403108"
        "5681484072570207073673489621966317145732916133732926783278240845883752695153675874855722158723162115155256"
        "7915552982205833048576023891101242635438285337077071160952683144203132426399016676299939839466811141629778"
        "0283160200011485937655750349994978227216676346638844598928321597850896549488106759475055393480783219563320"
        "5337633830471435241928429408615780527637327866360861941394787429614188223325754734105430543422698974609375"
        "e-252"},
    {0x1.fffffffffffffp-835, chars_format::scientific, 635,
        "8."
        "7293843616244312627115936672563912756807456784831585419096094998499166297599450089077705449472007885399183"
        "1501754807292015248198401893453189670819075665505918831966616495617716247669912073392830746364539682806217"
        "1362968145140414147346979243932634291465832267465853566556481691767505390307351749711444317446324230310513"
        "5831105964411666097152047782202485270876570674154142321905366288406264852798033352599879678933622283259556"
        "0566320400022971875311500699989956454433352693277689197856643195701793098976213518950110786961566439126641"
        "067526766094287048385685881723156105527465573272172388278957485922837644665150946821086108684539794921875e"
        "-252"},
    {0x1.fffffffffffffp-834, chars_format::scientific, 635,
        "1."
        "7458768723248862525423187334512782551361491356966317083819218999699833259519890017815541089894401577079836"
        "6300350961458403049639680378690637934163815133101183766393323299123543249533982414678566149272907936561243"
        "4272593629028082829469395848786526858293166453493170713311296338353501078061470349942288863489264846062102"
        "7166221192882333219430409556440497054175314134830828464381073257681252970559606670519975935786724456651911"
        "2113264080004594375062300139997991290886670538655537839571328639140358619795242703790022157392313287825328"
        "213505353218857409677137176344631221105493114654434477655791497184567528933030189364217221736907958984375e"
        "-251"},
    {0x1.fffffffffffffp-833, chars_format::scientific, 634,
        "3."
        "4917537446497725050846374669025565102722982713932634167638437999399666519039780035631082179788803154159673"
        "2600701922916806099279360757381275868327630266202367532786646598247086499067964829357132298545815873122486"
        "8545187258056165658938791697573053716586332906986341426622592676707002156122940699884577726978529692124205"
        "4332442385764666438860819112880994108350628269661656928762146515362505941119213341039951871573448913303822"
        "4226528160009188750124600279995982581773341077311075679142657278280717239590485407580044314784626575650656"
        "42701070643771481935427435268926244221098622930886895531158299436913505786606037872843444347381591796875e-"
        "251"},
    {0x1.fffffffffffffp-832, chars_format::scientific, 633,
        "6."
        "9835074892995450101692749338051130205445965427865268335276875998799333038079560071262164359577606308319346"
        "5201403845833612198558721514762551736655260532404735065573293196494172998135929658714264597091631746244973"
        "7090374516112331317877583395146107433172665813972682853245185353414004312245881399769155453957059384248410"
        "8664884771529332877721638225761988216701256539323313857524293030725011882238426682079903743146897826607644"
        "8453056320018377500249200559991965163546682154622151358285314556561434479180970815160088629569253151301312"
        "8540214128754296387085487053785248844219724586177379106231659887382701157321207574568688869476318359375e-"
        "251"},
    {0x1.fffffffffffffp-831, chars_format::scientific, 633,
        "1."
        "3967014978599090020338549867610226041089193085573053667055375199759866607615912014252432871915521261663869"
        "3040280769166722439711744302952510347331052106480947013114658639298834599627185931742852919418326349248994"
        "7418074903222466263575516679029221486634533162794536570649037070682800862449176279953831090791411876849682"
        "1732976954305866575544327645152397643340251307864662771504858606145002376447685336415980748629379565321528"
        "9690611264003675500049840111998393032709336430924430271657062911312286895836194163032017725913850630260262"
        "5708042825750859277417097410757049768843944917235475821246331977476540231464241514913737773895263671875e-"
        "250"},
    {0x1.fffffffffffffp-830, chars_format::scientific, 632,
        "2."
        "7934029957198180040677099735220452082178386171146107334110750399519733215231824028504865743831042523327738"
        "6080561538333444879423488605905020694662104212961894026229317278597669199254371863485705838836652698497989"
        "4836149806444932527151033358058442973269066325589073141298074141365601724898352559907662181582823753699364"
        "3465953908611733151088655290304795286680502615729325543009717212290004752895370672831961497258759130643057"
        "9381222528007351000099680223996786065418672861848860543314125822624573791672388326064035451827701260520525"
        "141608565150171855483419482151409953768788983447095164249266395495308046292848302982747554779052734375e-"
        "250"},
    {0x1.fffffffffffffp-829, chars_format::scientific, 631,
        "5."
        "5868059914396360081354199470440904164356772342292214668221500799039466430463648057009731487662085046655477"
        "2161123076666889758846977211810041389324208425923788052458634557195338398508743726971411677673305396995978"
        "9672299612889865054302066716116885946538132651178146282596148282731203449796705119815324363165647507398728"
        "6931907817223466302177310580609590573361005231458651086019434424580009505790741345663922994517518261286115"
        "8762445056014702000199360447993572130837345723697721086628251645249147583344776652128070903655402521041050"
        "28321713030034371096683896430281990753757796689419032849853279099061609258569660596549510955810546875e-"
        "250"},
    {0x1.fffffffffffffp-828, chars_format::scientific, 631,
        "1."
        "1173611982879272016270839894088180832871354468458442933644300159807893286092729611401946297532417009331095"
        "4432224615333377951769395442362008277864841685184757610491726911439067679701748745394282335534661079399195"
        "7934459922577973010860413343223377189307626530235629256519229656546240689959341023963064872633129501479745"
        "7386381563444693260435462116121918114672201046291730217203886884916001901158148269132784598903503652257223"
        "1752489011202940400039872089598714426167469144739544217325650329049829516668955330425614180731080504208210"
        "05664342606006874219336779286056398150751559337883806569970655819812321851713932119309902191162109375e-"
        "249"},
    {0x1.fffffffffffffp-827, chars_format::scientific, 630,
        "2."
        "2347223965758544032541679788176361665742708936916885867288600319615786572185459222803892595064834018662190"
        "8864449230666755903538790884724016555729683370369515220983453822878135359403497490788564671069322158798391"
        "5868919845155946021720826686446754378615253060471258513038459313092481379918682047926129745266259002959491"
        "4772763126889386520870924232243836229344402092583460434407773769832003802316296538265569197807007304514446"
        "3504978022405880800079744179197428852334938289479088434651300658099659033337910660851228361462161008416420"
        "1132868521201374843867355857211279630150311867576761313994131163962464370342786423861980438232421875e-"
        "249"},
    {0x1.fffffffffffffp-826, chars_format::scientific, 629,
        "4."
        "4694447931517088065083359576352723331485417873833771734577200639231573144370918445607785190129668037324381"
        "7728898461333511807077581769448033111459366740739030441966907645756270718806994981577129342138644317596783"
        "1737839690311892043441653372893508757230506120942517026076918626184962759837364095852259490532518005918982"
        "9545526253778773041741848464487672458688804185166920868815547539664007604632593076531138395614014609028892"
        "7009956044811761600159488358394857704669876578958176869302601316199318066675821321702456722924322016832840"
        "226573704240274968773471171442255926030062373515352262798826232792492874068557284772396087646484375e-249"},
    {0x1.fffffffffffffp-825, chars_format::scientific, 628,
        "8."
        "9388895863034176130166719152705446662970835747667543469154401278463146288741836891215570380259336074648763"
        "5457796922667023614155163538896066222918733481478060883933815291512541437613989963154258684277288635193566"
        "3475679380623784086883306745787017514461012241885034052153837252369925519674728191704518981065036011837965"
        "9091052507557546083483696928975344917377608370333841737631095079328015209265186153062276791228029218057785"
        "4019912089623523200318976716789715409339753157916353738605202632398636133351642643404913445848644033665680"
        "45314740848054993754694234288451185206012474703070452559765246558498574813711456954479217529296875e-249"},
    {0x1.fffffffffffffp-824, chars_format::scientific, 628,
        "1."
        "7877779172606835226033343830541089332594167149533508693830880255692629257748367378243114076051867214929752"
        "7091559384533404722831032707779213244583746696295612176786763058302508287522797992630851736855457727038713"
        "2695135876124756817376661349157403502892202448377006810430767450473985103934945638340903796213007202367593"
        "1818210501511509216696739385795068983475521674066768347526219015865603041853037230612455358245605843611557"
        "0803982417924704640063795343357943081867950631583270747721040526479727226670328528680982689169728806733136"
        "09062948169610998750938846857690237041202494940614090511953049311699714962742291390895843505859375e-248"},
    {0x1.fffffffffffffp-823, chars_format::scientific, 627,
        "3."
        "5755558345213670452066687661082178665188334299067017387661760511385258515496734756486228152103734429859505"
        "4183118769066809445662065415558426489167493392591224353573526116605016575045595985261703473710915454077426"
        "5390271752249513634753322698314807005784404896754013620861534900947970207869891276681807592426014404735186"
        "3636421003023018433393478771590137966951043348133536695052438031731206083706074461224910716491211687223114"
        "1607964835849409280127590686715886163735901263166541495442081052959454453340657057361965378339457613466272"
        "1812589633922199750187769371538047408240498988122818102390609862339942992548458278179168701171875e-248"},
    {0x1.fffffffffffffp-822, chars_format::scientific, 626,
        "7."
        "1511116690427340904133375322164357330376668598134034775323521022770517030993469512972456304207468859719010"
        "8366237538133618891324130831116852978334986785182448707147052233210033150091191970523406947421830908154853"
        "0780543504499027269506645396629614011568809793508027241723069801895940415739782553363615184852028809470372"
        "7272842006046036866786957543180275933902086696267073390104876063462412167412148922449821432982423374446228"
        "3215929671698818560255181373431772327471802526333082990884162105918908906681314114723930756678915226932544"
        "362517926784439950037553874307609481648099797624563620478121972467988598509691655635833740234375e-248"},
    {0x1.fffffffffffffp-821, chars_format::scientific, 626,
        "1."
        "4302223338085468180826675064432871466075333719626806955064704204554103406198693902594491260841493771943802"
        "1673247507626723778264826166223370595666997357036489741429410446642006630018238394104681389484366181630970"
        "6156108700899805453901329079325922802313761958701605448344613960379188083147956510672723036970405761894074"
        "5454568401209207373357391508636055186780417339253414678020975212692482433482429784489964286596484674889245"
        "6643185934339763712051036274686354465494360505266616598176832421183781781336262822944786151335783045386508"
        "872503585356887990007510774861521896329619959524912724095624394493597719701938331127166748046875e-247"},
    {0x1.fffffffffffffp-820, chars_format::scientific, 625,
        "2."
        "8604446676170936361653350128865742932150667439253613910129408409108206812397387805188982521682987543887604"
        "3346495015253447556529652332446741191333994714072979482858820893284013260036476788209362778968732363261941"
        "2312217401799610907802658158651845604627523917403210896689227920758376166295913021345446073940811523788149"
        "0909136802418414746714783017272110373560834678506829356041950425384964866964859568979928573192969349778491"
        "3286371868679527424102072549372708930988721010533233196353664842367563562672525645889572302671566090773017"
        "74500717071377598001502154972304379265923991904982544819124878898719543940387666225433349609375e-247"},
    {0x1.fffffffffffffp-819, chars_format::scientific, 624,
        "5."
        "7208893352341872723306700257731485864301334878507227820258816818216413624794775610377965043365975087775208"
        "6692990030506895113059304664893482382667989428145958965717641786568026520072953576418725557937464726523882"
        "4624434803599221815605316317303691209255047834806421793378455841516752332591826042690892147881623047576298"
        "1818273604836829493429566034544220747121669357013658712083900850769929733929719137959857146385938699556982"
        "6572743737359054848204145098745417861977442021066466392707329684735127125345051291779144605343132181546035"
        "4900143414275519600300430994460875853184798380996508963824975779743908788077533245086669921875e-247"},
    {0x1.fffffffffffffp-818, chars_format::scientific, 624,
        "1."
        "1441778670468374544661340051546297172860266975701445564051763363643282724958955122075593008673195017555041"
        "7338598006101379022611860932978696476533597885629191793143528357313605304014590715283745111587492945304776"
        "4924886960719844363121063263460738241851009566961284358675691168303350466518365208538178429576324609515259"
        "6363654720967365898685913206908844149424333871402731742416780170153985946785943827591971429277187739911396"
        "5314548747471810969640829019749083572395488404213293278541465936947025425069010258355828921068626436309207"
        "0980028682855103920060086198892175170636959676199301792764995155948781757615506649017333984375e-246"},
    {0x1.fffffffffffffp-817, chars_format::scientific, 623,
        "2."
        "2883557340936749089322680103092594345720533951402891128103526727286565449917910244151186017346390035110083"
        "4677196012202758045223721865957392953067195771258383586287056714627210608029181430567490223174985890609552"
        "9849773921439688726242126526921476483702019133922568717351382336606700933036730417076356859152649219030519"
        "2727309441934731797371826413817688298848667742805463484833560340307971893571887655183942858554375479822793"
        "0629097494943621939281658039498167144790976808426586557082931873894050850138020516711657842137252872618414"
        "196005736571020784012017239778435034127391935239860358552999031189756351523101329803466796875e-246"},
    {0x1.fffffffffffffp-816, chars_format::scientific, 622,
        "4."
        "5767114681873498178645360206185188691441067902805782256207053454573130899835820488302372034692780070220166"
        "9354392024405516090447443731914785906134391542516767172574113429254421216058362861134980446349971781219105"
        "9699547842879377452484253053842952967404038267845137434702764673213401866073460834152713718305298438061038"
        "5454618883869463594743652827635376597697335485610926969667120680615943787143775310367885717108750959645586"
        "1258194989887243878563316078996334289581953616853173114165863747788101700276041033423315684274505745236828"
        "39201147314204156802403447955687006825478387047972071710599806237951270304620265960693359375e-246"},
    {0x1.fffffffffffffp-815, chars_format::scientific, 621,
        "9."
        "1534229363746996357290720412370377382882135805611564512414106909146261799671640976604744069385560140440333"
        "8708784048811032180894887463829571812268783085033534345148226858508842432116725722269960892699943562438211"
        "9399095685758754904968506107685905934808076535690274869405529346426803732146921668305427436610596876122077"
        "0909237767738927189487305655270753195394670971221853939334241361231887574287550620735771434217501919291172"
        "2516389979774487757126632157992668579163907233706346228331727495576203400552082066846631368549011490473656"
        "7840229462840831360480689591137401365095677409594414342119961247590254060924053192138671875e-246"},
    {0x1.fffffffffffffp-814, chars_format::scientific, 621,
        "1."
        "8306845872749399271458144082474075476576427161122312902482821381829252359934328195320948813877112028088066"
        "7741756809762206436178977492765914362453756617006706869029645371701768486423345144453992178539988712487642"
        "3879819137151750980993701221537181186961615307138054973881105869285360746429384333661085487322119375224415"
        "4181847553547785437897461131054150639078934194244370787866848272246377514857510124147154286843500383858234"
        "4503277995954897551425326431598533715832781446741269245666345499115240680110416413369326273709802298094731"
        "3568045892568166272096137918227480273019135481918882868423992249518050812184810638427734375e-245"},
    {0x1.fffffffffffffp-813, chars_format::scientific, 620,
        "3."
        "6613691745498798542916288164948150953152854322244625804965642763658504719868656390641897627754224056176133"
        "5483513619524412872357954985531828724907513234013413738059290743403536972846690288907984357079977424975284"
        "7759638274303501961987402443074362373923230614276109947762211738570721492858768667322170974644238750448830"
        "8363695107095570875794922262108301278157868388488741575733696544492755029715020248294308573687000767716468"
        "9006555991909795102850652863197067431665562893482538491332690998230481360220832826738652547419604596189462"
        "713609178513633254419227583645496054603827096383776573684798449903610162436962127685546875e-245"},
    {0x1.fffffffffffffp-812, chars_format::scientific, 619,
        "7."
        "3227383490997597085832576329896301906305708644489251609931285527317009439737312781283795255508448112352267"
        "0967027239048825744715909971063657449815026468026827476118581486807073945693380577815968714159954849950569"
        "5519276548607003923974804886148724747846461228552219895524423477141442985717537334644341949288477500897661"
        "6727390214191141751589844524216602556315736776977483151467393088985510059430040496588617147374001535432937"
        "8013111983819590205701305726394134863331125786965076982665381996460962720441665653477305094839209192378925"
        "42721835702726650883845516729099210920765419276755314736959689980722032487392425537109375e-245"},
    {0x1.fffffffffffffp-811, chars_format::scientific, 619,
        "1."
        "4645476698199519417166515265979260381261141728897850321986257105463401887947462556256759051101689622470453"
        "4193405447809765148943181994212731489963005293605365495223716297361414789138676115563193742831990969990113"
        "9103855309721400784794960977229744949569292245710443979104884695428288597143507466928868389857695500179532"
        "3345478042838228350317968904843320511263147355395496630293478617797102011886008099317723429474800307086587"
        "5602622396763918041140261145278826972666225157393015396533076399292192544088333130695461018967841838475785"
        "08544367140545330176769103345819842184153083855351062947391937996144406497478485107421875e-244"},
    {0x1.fffffffffffffp-810, chars_format::scientific, 618,
        "2."
        "9290953396399038834333030531958520762522283457795700643972514210926803775894925112513518102203379244940906"
        "8386810895619530297886363988425462979926010587210730990447432594722829578277352231126387485663981939980227"
        "8207710619442801569589921954459489899138584491420887958209769390856577194287014933857736779715391000359064"
        "6690956085676456700635937809686641022526294710790993260586957235594204023772016198635446858949600614173175"
        "1205244793527836082280522290557653945332450314786030793066152798584385088176666261390922037935683676951570"
        "1708873428109066035353820669163968436830616771070212589478387599228881299495697021484375e-244"},
    {0x1.fffffffffffffp-809, chars_format::scientific, 617,
        "5."
        "8581906792798077668666061063917041525044566915591401287945028421853607551789850225027036204406758489881813"
        "6773621791239060595772727976850925959852021174421461980894865189445659156554704462252774971327963879960455"
        "6415421238885603139179843908918979798277168982841775916419538781713154388574029867715473559430782000718129"
        "3381912171352913401271875619373282045052589421581986521173914471188408047544032397270893717899201228346350"
        "2410489587055672164561044581115307890664900629572061586132305597168770176353332522781844075871367353903140"
        "341774685621813207070764133832793687366123354214042517895677519845776259899139404296875e-244"},
    {0x1.fffffffffffffp-808, chars_format::scientific, 617,
        "1."
        "1716381358559615533733212212783408305008913383118280257589005684370721510357970045005407240881351697976362"
        "7354724358247812119154545595370185191970404234884292396178973037889131831310940892450554994265592775992091"
        "1283084247777120627835968781783795959655433796568355183283907756342630877714805973543094711886156400143625"
        "8676382434270582680254375123874656409010517884316397304234782894237681609508806479454178743579840245669270"
        "0482097917411134432912208916223061578132980125914412317226461119433754035270666504556368815174273470780628"
        "068354937124362641414152826766558737473224670842808503579135503969155251979827880859375e-243"},
    {0x1.fffffffffffffp-807, chars_format::scientific, 616,
        "2."
        "3432762717119231067466424425566816610017826766236560515178011368741443020715940090010814481762703395952725"
        "4709448716495624238309091190740370383940808469768584792357946075778263662621881784901109988531185551984182"
        "2566168495554241255671937563567591919310867593136710366567815512685261755429611947086189423772312800287251"
        "7352764868541165360508750247749312818021035768632794608469565788475363219017612958908357487159680491338540"
        "0964195834822268865824417832446123156265960251828824634452922238867508070541333009112737630348546941561256"
        "13670987424872528282830565353311747494644934168561700715827100793831050395965576171875e-243"},
    {0x1.fffffffffffffp-806, chars_format::scientific, 615,
        "4."
        "6865525434238462134932848851133633220035653532473121030356022737482886041431880180021628963525406791905450"
        "9418897432991248476618182381480740767881616939537169584715892151556527325243763569802219977062371103968364"
        "5132336991108482511343875127135183838621735186273420733135631025370523510859223894172378847544625600574503"
        "4705529737082330721017500495498625636042071537265589216939131576950726438035225917816714974319360982677080"
        "1928391669644537731648835664892246312531920503657649268905844477735016141082666018225475260697093883122512"
        "2734197484974505656566113070662349498928986833712340143165420158766210079193115234375e-243"},
    {0x1.fffffffffffffp-805, chars_format::scientific, 614,
        "9."
        "3731050868476924269865697702267266440071307064946242060712045474965772082863760360043257927050813583810901"
        "8837794865982496953236364762961481535763233879074339169431784303113054650487527139604439954124742207936729"
        "0264673982216965022687750254270367677243470372546841466271262050741047021718447788344757695089251201149006"
        "9411059474164661442035000990997251272084143074531178433878263153901452876070451835633429948638721965354160"
        "3856783339289075463297671329784492625063841007315298537811688955470032282165332036450950521394187766245024"
        "546839496994901131313222614132469899785797366742468028633084031753242015838623046875e-243"},
    {0x1.fffffffffffffp-804, chars_format::scientific, 614,
        "1."
        "8746210173695384853973139540453453288014261412989248412142409094993154416572752072008651585410162716762180"
        "3767558973196499390647272952592296307152646775814867833886356860622610930097505427920887990824948441587345"
        "8052934796443393004537550050854073535448694074509368293254252410148209404343689557668951539017850240229801"
        "3882211894832932288407000198199450254416828614906235686775652630780290575214090367126685989727744393070832"
        "0771356667857815092659534265956898525012768201463059707562337791094006456433066407290190104278837553249004"
        "909367899398980226262644522826493979957159473348493605726616806350648403167724609375e-242"},
    {0x1.fffffffffffffp-803, chars_format::scientific, 613,
        "3."
        "7492420347390769707946279080906906576028522825978496824284818189986308833145504144017303170820325433524360"
        "7535117946392998781294545905184592614305293551629735667772713721245221860195010855841775981649896883174691"
        "6105869592886786009075100101708147070897388149018736586508504820296418808687379115337903078035700480459602"
        "7764423789665864576814000396398900508833657229812471373551305261560581150428180734253371979455488786141664"
        "1542713335715630185319068531913797050025536402926119415124675582188012912866132814580380208557675106498009"
        "81873579879796045252528904565298795991431894669698721145323361270129680633544921875e-242"},
    {0x1.fffffffffffffp-802, chars_format::scientific, 612,
        "7."
        "4984840694781539415892558161813813152057045651956993648569636379972617666291008288034606341640650867048721"
        "5070235892785997562589091810369185228610587103259471335545427442490443720390021711683551963299793766349383"
        "2211739185773572018150200203416294141794776298037473173017009640592837617374758230675806156071400960919205"
        "5528847579331729153628000792797801017667314459624942747102610523121162300856361468506743958910977572283328"
        "3085426671431260370638137063827594100051072805852238830249351164376025825732265629160760417115350212996019"
        "6374715975959209050505780913059759198286378933939744229064672254025936126708984375e-242"},
    {0x1.fffffffffffffp-801, chars_format::scientific, 612,
        "1."
        "4996968138956307883178511632362762630411409130391398729713927275994523533258201657606921268328130173409744"
        "3014047178557199512517818362073837045722117420651894267109085488498088744078004342336710392659958753269876"
        "6442347837154714403630040040683258828358955259607494634603401928118567523474951646135161231214280192183841"
        "1105769515866345830725600158559560203533462891924988549420522104624232460171272293701348791782195514456665"
        "6617085334286252074127627412765518820010214561170447766049870232875205165146453125832152083423070042599203"
        "9274943195191841810101156182611951839657275786787948845812934450805187225341796875e-241"},
    {0x1.fffffffffffffp-800, chars_format::scientific, 611,
        "2."
        "9993936277912615766357023264725525260822818260782797459427854551989047066516403315213842536656260346819488"
        "6028094357114399025035636724147674091444234841303788534218170976996177488156008684673420785319917506539753"
        "2884695674309428807260080081366517656717910519214989269206803856237135046949903292270322462428560384367682"
        "2211539031732691661451200317119120407066925783849977098841044209248464920342544587402697583564391028913331"
        "3234170668572504148255254825531037640020429122340895532099740465750410330292906251664304166846140085198407"
        "854988639038368362020231236522390367931455157357589769162586890161037445068359375e-241"},
    {0x1.fffffffffffffp-799, chars_format::scientific, 610,
        "5."
        "9987872555825231532714046529451050521645636521565594918855709103978094133032806630427685073312520693638977"
        "2056188714228798050071273448295348182888469682607577068436341953992354976312017369346841570639835013079506"
        "5769391348618857614520160162733035313435821038429978538413607712474270093899806584540644924857120768735364"
        "4423078063465383322902400634238240814133851567699954197682088418496929840685089174805395167128782057826662"
        "6468341337145008296510509651062075280040858244681791064199480931500820660585812503328608333692280170396815"
        "70997727807673672404046247304478073586291031471517953832517378032207489013671875e-241"},
    {0x1.fffffffffffffp-798, chars_format::scientific, 610,
        "1."
        "1997574511165046306542809305890210104329127304313118983771141820795618826606561326085537014662504138727795"
        "4411237742845759610014254689659069636577693936521515413687268390798470995262403473869368314127967002615901"
        "3153878269723771522904032032546607062687164207685995707682721542494854018779961316908128984971424153747072"
        "8884615612693076664580480126847648162826770313539990839536417683699385968137017834961079033425756411565332"
        "5293668267429001659302101930212415056008171648936358212839896186300164132117162500665721666738456034079363"
        "14199545561534734480809249460895614717258206294303590766503475606441497802734375e-240"},
    {0x1.fffffffffffffp-797, chars_format::scientific, 609,
        "2."
        "3995149022330092613085618611780420208658254608626237967542283641591237653213122652171074029325008277455590"
        "8822475485691519220028509379318139273155387873043030827374536781596941990524806947738736628255934005231802"
        "6307756539447543045808064065093214125374328415371991415365443084989708037559922633816257969942848307494145"
        "7769231225386153329160960253695296325653540627079981679072835367398771936274035669922158066851512823130665"
        "0587336534858003318604203860424830112016343297872716425679792372600328264234325001331443333476912068158726"
        "2839909112306946896161849892179122943451641258860718153300695121288299560546875e-240"},
    {0x1.fffffffffffffp-796, chars_format::scientific, 608,
        "4."
        "7990298044660185226171237223560840417316509217252475935084567283182475306426245304342148058650016554911181"
        "7644950971383038440057018758636278546310775746086061654749073563193883981049613895477473256511868010463605"
        "2615513078895086091616128130186428250748656830743982830730886169979416075119845267632515939885696614988291"
        "5538462450772306658321920507390592651307081254159963358145670734797543872548071339844316133703025646261330"
        "1174673069716006637208407720849660224032686595745432851359584745200656528468650002662886666953824136317452"
        "567981822461389379232369978435824588690328251772143630660139024257659912109375e-240"},
    {0x1.fffffffffffffp-795, chars_format::scientific, 607,
        "9."
        "5980596089320370452342474447121680834633018434504951870169134566364950612852490608684296117300033109822363"
        "5289901942766076880114037517272557092621551492172123309498147126387767962099227790954946513023736020927210"
        "5231026157790172183232256260372856501497313661487965661461772339958832150239690535265031879771393229976583"
        "1076924901544613316643841014781185302614162508319926716291341469595087745096142679688632267406051292522660"
        "2349346139432013274416815441699320448065373191490865702719169490401313056937300005325773333907648272634905"
        "13596364492277875846473995687164917738065650354428726132027804851531982421875e-240"},
    {0x1.fffffffffffffp-794, chars_format::scientific, 607,
        "1."
        "9196119217864074090468494889424336166926603686900990374033826913272990122570498121736859223460006621964472"
        "7057980388553215376022807503454511418524310298434424661899629425277553592419845558190989302604747204185442"
        "1046205231558034436646451252074571300299462732297593132292354467991766430047938107053006375954278645995316"
        "6215384980308922663328768202956237060522832501663985343258268293919017549019228535937726453481210258504532"
        "0469869227886402654883363088339864089613074638298173140543833898080262611387460001065154666781529654526981"
        "02719272898455575169294799137432983547613130070885745226405560970306396484375e-239"},
    {0x1.fffffffffffffp-793, chars_format::scientific, 606,
        "3."
        "8392238435728148180936989778848672333853207373801980748067653826545980245140996243473718446920013243928945"
        "4115960777106430752045615006909022837048620596868849323799258850555107184839691116381978605209494408370884"
        "2092410463116068873292902504149142600598925464595186264584708935983532860095876214106012751908557291990633"
        "2430769960617845326657536405912474121045665003327970686516536587838035098038457071875452906962420517009064"
        "0939738455772805309766726176679728179226149276596346281087667796160525222774920002130309333563059309053962"
        "0543854579691115033858959827486596709522626014177149045281112194061279296875e-239"},
    {0x1.fffffffffffffp-792, chars_format::scientific, 605,
        "7."
        "6784476871456296361873979557697344667706414747603961496135307653091960490281992486947436893840026487857890"
        "8231921554212861504091230013818045674097241193737698647598517701110214369679382232763957210418988816741768"
        "4184820926232137746585805008298285201197850929190372529169417871967065720191752428212025503817114583981266"
        "4861539921235690653315072811824948242091330006655941373033073175676070196076914143750905813924841034018128"
        "1879476911545610619533452353359456358452298553192692562175335592321050445549840004260618667126118618107924"
        "108770915938223006771791965497319341904525202835429809056222438812255859375e-239"},
    {0x1.fffffffffffffp-791, chars_format::scientific, 605,
        "1."
        "5356895374291259272374795911539468933541282949520792299227061530618392098056398497389487378768005297571578"
        "1646384310842572300818246002763609134819448238747539729519703540222042873935876446552791442083797763348353"
        "6836964185246427549317161001659657040239570185838074505833883574393413144038350485642405100763422916796253"
        "2972307984247138130663014562364989648418266001331188274606614635135214039215382828750181162784968206803625"
        "6375895382309122123906690470671891271690459710638538512435067118464210089109968000852123733425223723621584"
        "821754183187644601354358393099463868380905040567085961811244487762451171875e-238"},
    {0x1.fffffffffffffp-790, chars_format::scientific, 604,
        "3."
        "0713790748582518544749591823078937867082565899041584598454123061236784196112796994778974757536010595143156"
        "3292768621685144601636492005527218269638896477495079459039407080444085747871752893105582884167595526696707"
        "3673928370492855098634322003319314080479140371676149011667767148786826288076700971284810201526845833592506"
        "5944615968494276261326029124729979296836532002662376549213229270270428078430765657500362325569936413607251"
        "2751790764618244247813380941343782543380919421277077024870134236928420178219936001704247466850447447243169"
        "64350836637528920270871678619892773676181008113417192362248897552490234375e-238"},
    {0x1.fffffffffffffp-789, chars_format::scientific, 603,
        "6."
        "1427581497165037089499183646157875734165131798083169196908246122473568392225593989557949515072021190286312"
        "6585537243370289203272984011054436539277792954990158918078814160888171495743505786211165768335191053393414"
        "7347856740985710197268644006638628160958280743352298023335534297573652576153401942569620403053691667185013"
        "1889231936988552522652058249459958593673064005324753098426458540540856156861531315000724651139872827214502"
        "5503581529236488495626761882687565086761838842554154049740268473856840356439872003408494933700894894486339"
        "2870167327505784054174335723978554735236201622683438472449779510498046875e-238"},
    {0x1.fffffffffffffp-788, chars_format::scientific, 603,
        "1."
        "2285516299433007417899836729231575146833026359616633839381649224494713678445118797911589903014404238057262"
        "5317107448674057840654596802210887307855558590998031783615762832177634299148701157242233153667038210678682"
        "9469571348197142039453728801327725632191656148670459604667106859514730515230680388513924080610738333437002"
        "6377846387397710504530411649891991718734612801064950619685291708108171231372306263000144930227974565442900"
        "5100716305847297699125352376537513017352367768510830809948053694771368071287974400681698986740178978897267"
        "8574033465501156810834867144795710947047240324536687694489955902099609375e-237"},
    {0x1.fffffffffffffp-787, chars_format::scientific, 602,
        "2."
        "4571032598866014835799673458463150293666052719233267678763298448989427356890237595823179806028808476114525"
        "0634214897348115681309193604421774615711117181996063567231525664355268598297402314484466307334076421357365"
        "8939142696394284078907457602655451264383312297340919209334213719029461030461360777027848161221476666874005"
        "2755692774795421009060823299783983437469225602129901239370583416216342462744612526000289860455949130885801"
        "0201432611694595398250704753075026034704735537021661619896107389542736142575948801363397973480357957794535"
        "714806693100231362166973428959142189409448064907337538897991180419921875e-237"},
    {0x1.fffffffffffffp-786, chars_format::scientific, 601,
        "4."
        "9142065197732029671599346916926300587332105438466535357526596897978854713780475191646359612057616952229050"
        "1268429794696231362618387208843549231422234363992127134463051328710537196594804628968932614668152842714731"
        "7878285392788568157814915205310902528766624594681838418668427438058922060922721554055696322442953333748010"
        "5511385549590842018121646599567966874938451204259802478741166832432684925489225052000579720911898261771602"
        "0402865223389190796501409506150052069409471074043323239792214779085472285151897602726795946960715915589071"
        "42961338620046272433394685791828437881889612981467507779598236083984375e-237"},
    {0x1.fffffffffffffp-785, chars_format::scientific, 600,
        "9."
        "8284130395464059343198693833852601174664210876933070715053193795957709427560950383292719224115233904458100"
        "2536859589392462725236774417687098462844468727984254268926102657421074393189609257937865229336305685429463"
        "5756570785577136315629830410621805057533249189363676837336854876117844121845443108111392644885906667496021"
        "1022771099181684036243293199135933749876902408519604957482333664865369850978450104001159441823796523543204"
        "0805730446778381593002819012300104138818942148086646479584429558170944570303795205453591893921431831178142"
        "8592267724009254486678937158365687576377922596293501555919647216796875e-237"},
    {0x1.fffffffffffffp-784, chars_format::scientific, 600,
        "1."
        "9656826079092811868639738766770520234932842175386614143010638759191541885512190076658543844823046780891620"
        "0507371917878492545047354883537419692568893745596850853785220531484214878637921851587573045867261137085892"
        "7151314157115427263125966082124361011506649837872735367467370975223568824369088621622278528977181333499204"
        "2204554219836336807248658639827186749975380481703920991496466732973073970195690020800231888364759304708640"
        "8161146089355676318600563802460020827763788429617329295916885911634188914060759041090718378784286366235628"
        "5718453544801850897335787431673137515275584519258700311183929443359375e-236"},
    {0x1.fffffffffffffp-783, chars_format::scientific, 599,
        "3."
        "9313652158185623737279477533541040469865684350773228286021277518383083771024380153317087689646093561783240"
        "1014743835756985090094709767074839385137787491193701707570441062968429757275843703175146091734522274171785"
        "4302628314230854526251932164248722023013299675745470734934741950447137648738177243244557057954362666998408"
        "4409108439672673614497317279654373499950760963407841982992933465946147940391380041600463776729518609417281"
        "6322292178711352637201127604920041655527576859234658591833771823268377828121518082181436757568572732471257"
        "143690708960370179467157486334627503055116903851740062236785888671875e-236"},
    {0x1.fffffffffffffp-782, chars_format::scientific, 598,
        "7."
        "8627304316371247474558955067082080939731368701546456572042555036766167542048760306634175379292187123566480"
        "2029487671513970180189419534149678770275574982387403415140882125936859514551687406350292183469044548343570"
        "8605256628461709052503864328497444046026599351490941469869483900894275297476354486489114115908725333996816"
        "8818216879345347228994634559308746999901521926815683965985866931892295880782760083200927553459037218834563"
        "2644584357422705274402255209840083311055153718469317183667543646536755656243036164362873515137145464942514"
        "28738141792074035893431497266925500611023380770348012447357177734375e-236"},
    {0x1.fffffffffffffp-781, chars_format::scientific, 598,
        "1."
        "5725460863274249494911791013416416187946273740309291314408511007353233508409752061326835075858437424713296"
        "0405897534302794036037883906829935754055114996477480683028176425187371902910337481270058436693808909668714"
        "1721051325692341810500772865699488809205319870298188293973896780178855059495270897297822823181745066799363"
        "3763643375869069445798926911861749399980304385363136793197173386378459176156552016640185510691807443766912"
        "6528916871484541054880451041968016662211030743693863436733508729307351131248607232872574703027429092988502"
        "85747628358414807178686299453385100122204676154069602489471435546875e-235"},
    {0x1.fffffffffffffp-780, chars_format::scientific, 597,
        "3."
        "1450921726548498989823582026832832375892547480618582628817022014706467016819504122653670151716874849426592"
        "0811795068605588072075767813659871508110229992954961366056352850374743805820674962540116873387617819337428"
        "3442102651384683621001545731398977618410639740596376587947793560357710118990541794595645646363490133598726"
        "7527286751738138891597853823723498799960608770726273586394346772756918352313104033280371021383614887533825"
        "3057833742969082109760902083936033324422061487387726873467017458614702262497214465745149406054858185977005"
        "7149525671682961435737259890677020024440935230813920497894287109375e-235"},
    {0x1.fffffffffffffp-779, chars_format::scientific, 596,
        "6."
        "2901843453096997979647164053665664751785094961237165257634044029412934033639008245307340303433749698853184"
        "1623590137211176144151535627319743016220459985909922732112705700749487611641349925080233746775235638674856"
        "6884205302769367242003091462797955236821279481192753175895587120715420237981083589191291292726980267197453"
        "5054573503476277783195707647446997599921217541452547172788693545513836704626208066560742042767229775067650"
        "6115667485938164219521804167872066648844122974775453746934034917229404524994428931490298812109716371954011"
        "429905134336592287147451978135404004888187046162784099578857421875e-235"},
    {0x1.fffffffffffffp-778, chars_format::scientific, 596,
        "1."
        "2580368690619399595929432810733132950357018992247433051526808805882586806727801649061468060686749939770636"
        "8324718027442235228830307125463948603244091997181984546422541140149897522328269985016046749355047127734971"
        "3376841060553873448400618292559591047364255896238550635179117424143084047596216717838258258545396053439490"
        "7010914700695255556639141529489399519984243508290509434557738709102767340925241613312148408553445955013530"
        "1223133497187632843904360833574413329768824594955090749386806983445880904998885786298059762421943274390802"
        "285981026867318457429490395627080800977637409232556819915771484375e-234"},
    {0x1.fffffffffffffp-777, chars_format::scientific, 595,
        "2."
        "5160737381238799191858865621466265900714037984494866103053617611765173613455603298122936121373499879541273"
        "6649436054884470457660614250927897206488183994363969092845082280299795044656539970032093498710094255469942"
        "6753682121107746896801236585119182094728511792477101270358234848286168095192433435676516517090792106878981"
        "4021829401390511113278283058978799039968487016581018869115477418205534681850483226624296817106891910027060"
        "2446266994375265687808721667148826659537649189910181498773613966891761809997771572596119524843886548781604"
        "57196205373463691485898079125416160195527481846511363983154296875e-234"},
    {0x1.fffffffffffffp-776, chars_format::scientific, 594,
        "5."
        "0321474762477598383717731242932531801428075968989732206107235223530347226911206596245872242746999759082547"
        "3298872109768940915321228501855794412976367988727938185690164560599590089313079940064186997420188510939885"
        "3507364242215493793602473170238364189457023584954202540716469696572336190384866871353033034181584213757962"
        "8043658802781022226556566117957598079936974033162037738230954836411069363700966453248593634213783820054120"
        "4892533988750531375617443334297653319075298379820362997547227933783523619995543145192239049687773097563209"
        "1439241074692738297179615825083232039105496369302272796630859375e-234"},
    {0x1.fffffffffffffp-775, chars_format::scientific, 594,
        "1."
        "0064294952495519676743546248586506360285615193797946441221447044706069445382241319249174448549399951816509"
        "4659774421953788183064245700371158882595273597745587637138032912119918017862615988012837399484037702187977"
        "0701472848443098758720494634047672837891404716990840508143293939314467238076973374270606606836316842751592"
        "5608731760556204445311313223591519615987394806632407547646190967282213872740193290649718726842756764010824"
        "0978506797750106275123488666859530663815059675964072599509445586756704723999108629038447809937554619512641"
        "8287848214938547659435923165016646407821099273860454559326171875e-233"},
    {0x1.fffffffffffffp-774, chars_format::scientific, 593,
        "2."
        "0128589904991039353487092497173012720571230387595892882442894089412138890764482638498348897098799903633018"
        "9319548843907576366128491400742317765190547195491175274276065824239836035725231976025674798968075404375954"
        "1402945696886197517440989268095345675782809433981681016286587878628934476153946748541213213672633685503185"
        "1217463521112408890622626447183039231974789613264815095292381934564427745480386581299437453685513528021648"
        "1957013595500212550246977333719061327630119351928145199018891173513409447998217258076895619875109239025283"
        "657569642987709531887184633003329281564219854772090911865234375e-233"},
    {0x1.fffffffffffffp-773, chars_format::scientific, 592,
        "4."
        "0257179809982078706974184994346025441142460775191785764885788178824277781528965276996697794197599807266037"
        "8639097687815152732256982801484635530381094390982350548552131648479672071450463952051349597936150808751908"
        "2805891393772395034881978536190691351565618867963362032573175757257868952307893497082426427345267371006370"
        "2434927042224817781245252894366078463949579226529630190584763869128855490960773162598874907371027056043296"
        "3914027191000425100493954667438122655260238703856290398037782347026818895996434516153791239750218478050567"
        "31513928597541906377436926600665856312843970954418182373046875e-233"},
    {0x1.fffffffffffffp-772, chars_format::scientific, 591,
        "8."
        "0514359619964157413948369988692050882284921550383571529771576357648555563057930553993395588395199614532075"
        "7278195375630305464513965602969271060762188781964701097104263296959344142900927904102699195872301617503816"
        "5611782787544790069763957072381382703131237735926724065146351514515737904615786994164852854690534742012740"
        "4869854084449635562490505788732156927899158453059260381169527738257710981921546325197749814742054112086592"
        "7828054382000850200987909334876245310520477407712580796075564694053637791992869032307582479500436956101134"
        "6302785719508381275487385320133171262568794190883636474609375e-233"},
    {0x1.fffffffffffffp-771, chars_format::scientific, 591,
        "1."
        "6102871923992831482789673997738410176456984310076714305954315271529711112611586110798679117679039922906415"
        "1455639075126061092902793120593854212152437756392940219420852659391868828580185580820539839174460323500763"
        "3122356557508958013952791414476276540626247547185344813029270302903147580923157398832970570938106948402548"
        "0973970816889927112498101157746431385579831690611852076233905547651542196384309265039549962948410822417318"
        "5565610876400170040197581866975249062104095481542516159215112938810727558398573806461516495900087391220226"
        "9260557143901676255097477064026634252513758838176727294921875e-232"},
    {0x1.fffffffffffffp-770, chars_format::scientific, 590,
        "3."
        "2205743847985662965579347995476820352913968620153428611908630543059422225223172221597358235358079845812830"
        "2911278150252122185805586241187708424304875512785880438841705318783737657160371161641079678348920647001526"
        "6244713115017916027905582828952553081252495094370689626058540605806295161846314797665941141876213896805096"
        "1947941633779854224996202315492862771159663381223704152467811095303084392768618530079099925896821644834637"
        "1131221752800340080395163733950498124208190963085032318430225877621455116797147612923032991800174782440453"
        "852111428780335251019495412805326850502751767635345458984375e-232"},
    {0x1.fffffffffffffp-769, chars_format::scientific, 589,
        "6."
        "4411487695971325931158695990953640705827937240306857223817261086118844450446344443194716470716159691625660"
        "5822556300504244371611172482375416848609751025571760877683410637567475314320742323282159356697841294003053"
        "2489426230035832055811165657905106162504990188741379252117081211612590323692629595331882283752427793610192"
        "3895883267559708449992404630985725542319326762447408304935622190606168785537237060158199851793643289669274"
        "2262443505600680160790327467900996248416381926170064636860451755242910233594295225846065983600349564880907"
        "70422285756067050203899082561065370100550353527069091796875e-232"},
    {0x1.fffffffffffffp-768, chars_format::scientific, 589,
        "1."
        "2882297539194265186231739198190728141165587448061371444763452217223768890089268888638943294143231938325132"
        "1164511260100848874322234496475083369721950205114352175536682127513495062864148464656431871339568258800610"
        "6497885246007166411162233131581021232500998037748275850423416242322518064738525919066376456750485558722038"
        "4779176653511941689998480926197145108463865352489481660987124438121233757107447412031639970358728657933854"
        "8452488701120136032158065493580199249683276385234012927372090351048582046718859045169213196720069912976181"
        "54084457151213410040779816512213074020110070705413818359375e-231"},
    {0x1.fffffffffffffp-767, chars_format::scientific, 588,
        "2."
        "5764595078388530372463478396381456282331174896122742889526904434447537780178537777277886588286463876650264"
        "2329022520201697748644468992950166739443900410228704351073364255026990125728296929312863742679136517601221"
        "2995770492014332822324466263162042465001996075496551700846832484645036129477051838132752913500971117444076"
        "9558353307023883379996961852394290216927730704978963321974248876242467514214894824063279940717457315867709"
        "6904977402240272064316130987160398499366552770468025854744180702097164093437718090338426393440139825952363"
        "0816891430242682008155963302442614804022014141082763671875e-231"},
    {0x1.fffffffffffffp-766, chars_format::scientific, 587,
        "5."
        "1529190156777060744926956792762912564662349792245485779053808868895075560357075554555773176572927753300528"
        "4658045040403395497288937985900333478887800820457408702146728510053980251456593858625727485358273035202442"
        "5991540984028665644648932526324084930003992150993103401693664969290072258954103676265505827001942234888153"
        "9116706614047766759993923704788580433855461409957926643948497752484935028429789648126559881434914631735419"
        "3809954804480544128632261974320796998733105540936051709488361404194328186875436180676852786880279651904726"
        "163378286048536401631192660488522960804402828216552734375e-231"},
    {0x1.fffffffffffffp-765, chars_format::scientific, 587,
        "1."
        "0305838031355412148985391358552582512932469958449097155810761773779015112071415110911154635314585550660105"
        "6931609008080679099457787597180066695777560164091481740429345702010796050291318771725145497071654607040488"
        "5198308196805733128929786505264816986000798430198620680338732993858014451790820735253101165400388446977630"
        "7823341322809553351998784740957716086771092281991585328789699550496987005685957929625311976286982926347083"
        "8761990960896108825726452394864159399746621108187210341897672280838865637375087236135370557376055930380945"
        "232675657209707280326238532097704592160880565643310546875e-230"},
    {0x1.fffffffffffffp-764, chars_format::scientific, 586,
        "2."
        "0611676062710824297970782717105165025864939916898194311621523547558030224142830221822309270629171101320211"
        "3863218016161358198915575194360133391555120328182963480858691404021592100582637543450290994143309214080977"
        "0396616393611466257859573010529633972001596860397241360677465987716028903581641470506202330800776893955261"
        "5646682645619106703997569481915432173542184563983170657579399100993974011371915859250623952573965852694167"
        "7523981921792217651452904789728318799493242216374420683795344561677731274750174472270741114752111860761890"
        "46535131441941456065247706419540918432176113128662109375e-230"},
    {0x1.fffffffffffffp-763, chars_format::scientific, 585,
        "4."
        "1223352125421648595941565434210330051729879833796388623243047095116060448285660443644618541258342202640422"
        "7726436032322716397831150388720266783110240656365926961717382808043184201165275086900581988286618428161954"
        "0793232787222932515719146021059267944003193720794482721354931975432057807163282941012404661601553787910523"
        "1293365291238213407995138963830864347084369127966341315158798201987948022743831718501247905147931705388335"
        "5047963843584435302905809579456637598986484432748841367590689123355462549500348944541482229504223721523780"
        "9307026288388291213049541283908183686435222625732421875e-230"},
    {0x1.fffffffffffffp-762, chars_format::scientific, 584,
        "8."
        "2446704250843297191883130868420660103459759667592777246486094190232120896571320887289237082516684405280845"
        "5452872064645432795662300777440533566220481312731853923434765616086368402330550173801163976573236856323908"
        "1586465574445865031438292042118535888006387441588965442709863950864115614326565882024809323203107575821046"
        "2586730582476426815990277927661728694168738255932682630317596403975896045487663437002495810295863410776671"
        "0095927687168870605811619158913275197972968865497682735181378246710925099000697889082964459008447443047561"
        "861405257677658242609908256781636737287044525146484375e-230"},
    {0x1.fffffffffffffp-761, chars_format::scientific, 584,
        "1."
        "6489340850168659438376626173684132020691951933518555449297218838046424179314264177457847416503336881056169"
        "1090574412929086559132460155488106713244096262546370784686953123217273680466110034760232795314647371264781"
        "6317293114889173006287658408423707177601277488317793088541972790172823122865313176404961864640621515164209"
        "2517346116495285363198055585532345738833747651186536526063519280795179209097532687400499162059172682155334"
        "2019185537433774121162323831782655039594593773099536547036275649342185019800139577816592891801689488609512"
        "372281051535531648521981651356327347457408905029296875e-229"},
    {0x1.fffffffffffffp-760, chars_format::scientific, 583,
        "3."
        "2978681700337318876753252347368264041383903867037110898594437676092848358628528354915694833006673762112338"
        "2181148825858173118264920310976213426488192525092741569373906246434547360932220069520465590629294742529563"
        "2634586229778346012575316816847414355202554976635586177083945580345646245730626352809923729281243030328418"
        "5034692232990570726396111171064691477667495302373073052127038561590358418195065374800998324118345364310668"
        "4038371074867548242324647663565310079189187546199073094072551298684370039600279155633185783603378977219024"
        "74456210307106329704396330271265469491481781005859375e-229"},
    {0x1.fffffffffffffp-759, chars_format::scientific, 582,
        "6."
        "5957363400674637753506504694736528082767807734074221797188875352185696717257056709831389666013347524224676"
        "4362297651716346236529840621952426852976385050185483138747812492869094721864440139040931181258589485059126"
        "5269172459556692025150633633694828710405109953271172354167891160691292491461252705619847458562486060656837"
        "0069384465981141452792222342129382955334990604746146104254077123180716836390130749601996648236690728621336"
        "8076742149735096484649295327130620158378375092398146188145102597368740079200558311266371567206757954438049"
        "4891242061421265940879266054253093898296356201171875e-229"},
    {0x1.fffffffffffffp-758, chars_format::scientific, 582,
        "1."
        "3191472680134927550701300938947305616553561546814844359437775070437139343451411341966277933202669504844935"
        "2872459530343269247305968124390485370595277010037096627749562498573818944372888027808186236251717897011825"
        "3053834491911338405030126726738965742081021990654234470833578232138258498292250541123969491712497212131367"
        "4013876893196228290558444468425876591066998120949229220850815424636143367278026149920399329647338145724267"
        "3615348429947019296929859065426124031675675018479629237629020519473748015840111662253274313441351590887609"
        "8978248412284253188175853210850618779659271240234375e-228"},
    {0x1.fffffffffffffp-757, chars_format::scientific, 581,
        "2."
        "6382945360269855101402601877894611233107123093629688718875550140874278686902822683932555866405339009689870"
        "5744919060686538494611936248780970741190554020074193255499124997147637888745776055616372472503435794023650"
        "6107668983822676810060253453477931484162043981308468941667156464276516996584501082247938983424994424262734"
        "8027753786392456581116888936851753182133996241898458441701630849272286734556052299840798659294676291448534"
        "7230696859894038593859718130852248063351350036959258475258041038947496031680223324506548626882703181775219"
        "795649682456850637635170642170123755931854248046875e-228"},
    {0x1.fffffffffffffp-756, chars_format::scientific, 580,
        "5."
        "2765890720539710202805203755789222466214246187259377437751100281748557373805645367865111732810678019379741"
        "1489838121373076989223872497561941482381108040148386510998249994295275777491552111232744945006871588047301"
        "2215337967645353620120506906955862968324087962616937883334312928553033993169002164495877966849988848525469"
        "6055507572784913162233777873703506364267992483796916883403261698544573469112104599681597318589352582897069"
        "4461393719788077187719436261704496126702700073918516950516082077894992063360446649013097253765406363550439"
        "59129936491370127527034128434024751186370849609375e-228"},
    {0x1.fffffffffffffp-755, chars_format::scientific, 580,
        "1."
        "0553178144107942040561040751157844493242849237451875487550220056349711474761129073573022346562135603875948"
        "2297967624274615397844774499512388296476221608029677302199649998859055155498310422246548989001374317609460"
        "2443067593529070724024101381391172593664817592523387576666862585710606798633800432899175593369997769705093"
        "9211101514556982632446755574740701272853598496759383376680652339708914693822420919936319463717870516579413"
        "8892278743957615437543887252340899225340540014783703390103216415578998412672089329802619450753081272710087"
        "91825987298274025505406825686804950237274169921875e-227"},
    {0x1.fffffffffffffp-754, chars_format::scientific, 579,
        "2."
        "1106356288215884081122081502315688986485698474903750975100440112699422949522258147146044693124271207751896"
        "4595935248549230795689548999024776592952443216059354604399299997718110310996620844493097978002748635218920"
        "4886135187058141448048202762782345187329635185046775153333725171421213597267600865798351186739995539410187"
        "8422203029113965264893511149481402545707196993518766753361304679417829387644841839872638927435741033158827"
        "7784557487915230875087774504681798450681080029567406780206432831157996825344178659605238901506162545420175"
        "8365197459654805101081365137360990047454833984375e-227"},
    {0x1.fffffffffffffp-753, chars_format::scientific, 578,
        "4."
        "2212712576431768162244163004631377972971396949807501950200880225398845899044516294292089386248542415503792"
        "9191870497098461591379097998049553185904886432118709208798599995436220621993241688986195956005497270437840"
        "9772270374116282896096405525564690374659270370093550306667450342842427194535201731596702373479991078820375"
        "6844406058227930529787022298962805091414393987037533506722609358835658775289683679745277854871482066317655"
        "5569114975830461750175549009363596901362160059134813560412865662315993650688357319210477803012325090840351"
        "673039491930961020216273027472198009490966796875e-227"},
    {0x1.fffffffffffffp-752, chars_format::scientific, 577,
        "8."
        "4425425152863536324488326009262755945942793899615003900401760450797691798089032588584178772497084831007585"
        "8383740994196923182758195996099106371809772864237418417597199990872441243986483377972391912010994540875681"
        "9544540748232565792192811051129380749318540740187100613334900685684854389070403463193404746959982157640751"
        "3688812116455861059574044597925610182828787974075067013445218717671317550579367359490555709742964132635311"
        "1138229951660923500351098018727193802724320118269627120825731324631987301376714638420955606024650181680703"
        "34607898386192204043254605494439601898193359375e-227"},
    {0x1.fffffffffffffp-751, chars_format::scientific, 577,
        "1."
        "6885085030572707264897665201852551189188558779923000780080352090159538359617806517716835754499416966201517"
        "1676748198839384636551639199219821274361954572847483683519439998174488248797296675594478382402198908175136"
        "3908908149646513158438562210225876149863708148037420122666980137136970877814080692638680949391996431528150"
        "2737762423291172211914808919585122036565757594815013402689043743534263510115873471898111141948592826527062"
        "2227645990332184700070219603745438760544864023653925424165146264926397460275342927684191121204930036336140"
        "66921579677238440808650921098887920379638671875e-226"},
    {0x1.fffffffffffffp-750, chars_format::scientific, 576,
        "3."
        "3770170061145414529795330403705102378377117559846001560160704180319076719235613035433671508998833932403034"
        "3353496397678769273103278398439642548723909145694967367038879996348976497594593351188956764804397816350272"
        "7817816299293026316877124420451752299727416296074840245333960274273941755628161385277361898783992863056300"
        "5475524846582344423829617839170244073131515189630026805378087487068527020231746943796222283897185653054124"
        "4455291980664369400140439207490877521089728047307850848330292529852794920550685855368382242409860072672281"
        "3384315935447688161730184219777584075927734375e-226"},
    {0x1.fffffffffffffp-749, chars_format::scientific, 575,
        "6."
        "7540340122290829059590660807410204756754235119692003120321408360638153438471226070867343017997667864806068"
        "6706992795357538546206556796879285097447818291389934734077759992697952995189186702377913529608795632700545"
        "5635632598586052633754248840903504599454832592149680490667920548547883511256322770554723797567985726112601"
        "0951049693164688847659235678340488146263030379260053610756174974137054040463493887592444567794371306108248"
        "8910583961328738800280878414981755042179456094615701696660585059705589841101371710736764484819720145344562"
        "676863187089537632346036843955516815185546875e-226"},
    {0x1.fffffffffffffp-748, chars_format::scientific, 575,
        "1."
        "3508068024458165811918132161482040951350847023938400624064281672127630687694245214173468603599533572961213"
        "7341398559071507709241311359375857019489563658277986946815551998539590599037837340475582705921759126540109"
        "1127126519717210526750849768180700919890966518429936098133584109709576702251264554110944759513597145222520"
        "2190209938632937769531847135668097629252606075852010722151234994827410808092698777518488913558874261221649"
        "7782116792265747760056175682996351008435891218923140339332117011941117968220274342147352896963944029068912"
        "535372637417907526469207368791103363037109375e-225"},
    {0x1.fffffffffffffp-747, chars_format::scientific, 574,
        "2."
        "7016136048916331623836264322964081902701694047876801248128563344255261375388490428346937207199067145922427"
        "4682797118143015418482622718751714038979127316555973893631103997079181198075674680951165411843518253080218"
        "2254253039434421053501699536361401839781933036859872196267168219419153404502529108221889519027194290445040"
        "4380419877265875539063694271336195258505212151704021444302469989654821616185397555036977827117748522443299"
        "5564233584531495520112351365992702016871782437846280678664234023882235936440548684294705793927888058137825"
        "07074527483581505293841473758220672607421875e-225"},
    {0x1.fffffffffffffp-746, chars_format::scientific, 573,
        "5."
        "4032272097832663247672528645928163805403388095753602496257126688510522750776980856693874414398134291844854"
        "9365594236286030836965245437503428077958254633111947787262207994158362396151349361902330823687036506160436"
        "4508506078868842107003399072722803679563866073719744392534336438838306809005058216443779038054388580890080"
        "8760839754531751078127388542672390517010424303408042888604939979309643232370795110073955654235497044886599"
        "1128467169062991040224702731985404033743564875692561357328468047764471872881097368589411587855776116275650"
        "1414905496716301058768294751644134521484375e-225"},
    {0x1.fffffffffffffp-745, chars_format::scientific, 573,
        "1."
        "0806454419566532649534505729185632761080677619150720499251425337702104550155396171338774882879626858368970"
        "9873118847257206167393049087500685615591650926622389557452441598831672479230269872380466164737407301232087"
        "2901701215773768421400679814544560735912773214743948878506867287767661361801011643288755807610877716178016"
        "1752167950906350215625477708534478103402084860681608577720987995861928646474159022014791130847099408977319"
        "8225693433812598208044940546397080806748712975138512271465693609552894374576219473717882317571155223255130"
        "0282981099343260211753658950328826904296875e-224"},
    {0x1.fffffffffffffp-744, chars_format::scientific, 572,
        "2."
        "1612908839133065299069011458371265522161355238301440998502850675404209100310792342677549765759253716737941"
        "9746237694514412334786098175001371231183301853244779114904883197663344958460539744760932329474814602464174"
        "5803402431547536842801359629089121471825546429487897757013734575535322723602023286577511615221755432356032"
        "3504335901812700431250955417068956206804169721363217155441975991723857292948318044029582261694198817954639"
        "6451386867625196416089881092794161613497425950277024542931387219105788749152438947435764635142310446510260"
        "056596219868652042350731790065765380859375e-224"},
    {0x1.fffffffffffffp-743, chars_format::scientific, 571,
        "4."
        "3225817678266130598138022916742531044322710476602881997005701350808418200621584685355099531518507433475883"
        "9492475389028824669572196350002742462366603706489558229809766395326689916921079489521864658949629204928349"
        "1606804863095073685602719258178242943651092858975795514027469151070645447204046573155023230443510864712064"
        "7008671803625400862501910834137912413608339442726434310883951983447714585896636088059164523388397635909279"
        "2902773735250392832179762185588323226994851900554049085862774438211577498304877894871529270284620893020520"
        "11319243973730408470146358013153076171875e-224"},
    {0x1.fffffffffffffp-742, chars_format::scientific, 570,
        "8."
        "6451635356532261196276045833485062088645420953205763994011402701616836401243169370710199063037014866951767"
        "8984950778057649339144392700005484924733207412979116459619532790653379833842158979043729317899258409856698"
        "3213609726190147371205438516356485887302185717951591028054938302141290894408093146310046460887021729424129"
        "4017343607250801725003821668275824827216678885452868621767903966895429171793272176118329046776795271818558"
        "5805547470500785664359524371176646453989703801108098171725548876423154996609755789743058540569241786041040"
        "2263848794746081694029271602630615234375e-224"},
    {0x1.fffffffffffffp-741, chars_format::scientific, 570,
        "1."
        "7290327071306452239255209166697012417729084190641152798802280540323367280248633874142039812607402973390353"
        "5796990155611529867828878540001096984946641482595823291923906558130675966768431795808745863579851681971339"
        "6642721945238029474241087703271297177460437143590318205610987660428258178881618629262009292177404345884825"
        "8803468721450160345000764333655164965443335777090573724353580793379085834358654435223665809355359054363711"
        "7161109494100157132871904874235329290797940760221619634345109775284630999321951157948611708113848357208208"
        "0452769758949216338805854320526123046875e-223"},
    {0x1.fffffffffffffp-740, chars_format::scientific, 569,
        "3."
        "4580654142612904478510418333394024835458168381282305597604561080646734560497267748284079625214805946780707"
        "1593980311223059735657757080002193969893282965191646583847813116261351933536863591617491727159703363942679"
        "3285443890476058948482175406542594354920874287180636411221975320856516357763237258524018584354808691769651"
        "7606937442900320690001528667310329930886671554181147448707161586758171668717308870447331618710718108727423"
        "4322218988200314265743809748470658581595881520443239268690219550569261998643902315897223416227696714416416"
        "090553951789843267761170864105224609375e-223"},
    {0x1.fffffffffffffp-739, chars_format::scientific, 568,
        "6."
        "9161308285225808957020836666788049670916336762564611195209122161293469120994535496568159250429611893561414"
        "3187960622446119471315514160004387939786565930383293167695626232522703867073727183234983454319406727885358"
        "6570887780952117896964350813085188709841748574361272822443950641713032715526474517048037168709617383539303"
        "5213874885800641380003057334620659861773343108362294897414323173516343337434617740894663237421436217454846"
        "8644437976400628531487619496941317163191763040886478537380439101138523997287804631794446832455393428832832"
        "18110790357968653552234172821044921875e-223"},
    {0x1.fffffffffffffp-738, chars_format::scientific, 568,
        "1."
        "3832261657045161791404167333357609934183267352512922239041824432258693824198907099313631850085922378712282"
        "8637592124489223894263102832000877587957313186076658633539125246504540773414745436646996690863881345577071"
        "7314177556190423579392870162617037741968349714872254564488790128342606543105294903409607433741923476707860"
        "7042774977160128276000611466924131972354668621672458979482864634703268667486923548178932647484287243490969"
        "3728887595280125706297523899388263432638352608177295707476087820227704799457560926358889366491078685766566"
        "43622158071593730710446834564208984375e-222"},
    {0x1.fffffffffffffp-737, chars_format::scientific, 567,
        "2."
        "7664523314090323582808334666715219868366534705025844478083648864517387648397814198627263700171844757424565"
        "7275184248978447788526205664001755175914626372153317267078250493009081546829490873293993381727762691154143"
        "4628355112380847158785740325234075483936699429744509128977580256685213086210589806819214867483846953415721"
        "4085549954320256552001222933848263944709337243344917958965729269406537334973847096357865294968574486981938"
        "7457775190560251412595047798776526865276705216354591414952175640455409598915121852717778732982157371533132"
        "8724431614318746142089366912841796875e-222"},
    {0x1.fffffffffffffp-736, chars_format::scientific, 566,
        "5."
        "5329046628180647165616669333430439736733069410051688956167297729034775296795628397254527400343689514849131"
        "4550368497956895577052411328003510351829252744306634534156500986018163093658981746587986763455525382308286"
        "9256710224761694317571480650468150967873398859489018257955160513370426172421179613638429734967693906831442"
        "8171099908640513104002445867696527889418674486689835917931458538813074669947694192715730589937148973963877"
        "4915550381120502825190095597553053730553410432709182829904351280910819197830243705435557465964314743066265"
        "744886322863749228417873382568359375e-222"},
    {0x1.fffffffffffffp-735, chars_format::scientific, 566,
        "1."
        "1065809325636129433123333866686087947346613882010337791233459545806955059359125679450905480068737902969826"
        "2910073699591379115410482265600702070365850548861326906831300197203632618731796349317597352691105076461657"
        "3851342044952338863514296130093630193574679771897803651591032102674085234484235922727685946993538781366288"
        "5634219981728102620800489173539305577883734897337967183586291707762614933989538838543146117987429794792775"
        "4983110076224100565038019119510610746110682086541836565980870256182163839566048741087111493192862948613253"
        "148977264572749845683574676513671875e-221"},
    {0x1.fffffffffffffp-734, chars_format::scientific, 565,
        "2."
        "2131618651272258866246667733372175894693227764020675582466919091613910118718251358901810960137475805939652"
        "5820147399182758230820964531201404140731701097722653813662600394407265237463592698635194705382210152923314"
        "7702684089904677727028592260187260387149359543795607303182064205348170468968471845455371893987077562732577"
        "1268439963456205241600978347078611155767469794675934367172583415525229867979077677086292235974859589585550"
        "9966220152448201130076038239021221492221364173083673131961740512364327679132097482174222986385725897226506"
        "29795452914549969136714935302734375e-221"},
    {0x1.fffffffffffffp-733, chars_format::scientific, 564,
        "4."
        "4263237302544517732493335466744351789386455528041351164933838183227820237436502717803621920274951611879305"
        "1640294798365516461641929062402808281463402195445307627325200788814530474927185397270389410764420305846629"
        "5405368179809355454057184520374520774298719087591214606364128410696340937936943690910743787974155125465154"
        "2536879926912410483201956694157222311534939589351868734345166831050459735958155354172584471949719179171101"
        "9932440304896402260152076478042442984442728346167346263923481024728655358264194964348445972771451794453012"
        "5959090582909993827342987060546875e-221"},
    {0x1.fffffffffffffp-732, chars_format::scientific, 563,
        "8."
        "8526474605089035464986670933488703578772911056082702329867676366455640474873005435607243840549903223758610"
        "3280589596731032923283858124805616562926804390890615254650401577629060949854370794540778821528840611693259"
        "0810736359618710908114369040749041548597438175182429212728256821392681875873887381821487575948310250930308"
        "5073759853824820966403913388314444623069879178703737468690333662100919471916310708345168943899438358342203"
        "9864880609792804520304152956084885968885456692334692527846962049457310716528389928696891945542903588906025"
        "191818116581998765468597412109375e-221"},
    {0x1.fffffffffffffp-731, chars_format::scientific, 563,
        "1."
        "7705294921017807092997334186697740715754582211216540465973535273291128094974601087121448768109980644751722"
        "0656117919346206584656771624961123312585360878178123050930080315525812189970874158908155764305768122338651"
        "8162147271923742181622873808149808309719487635036485842545651364278536375174777476364297515189662050186061"
        "7014751970764964193280782677662888924613975835740747493738066732420183894383262141669033788779887671668440"
        "7972976121958560904060830591216977193777091338466938505569392409891462143305677985739378389108580717781205"
        "038363623316399753093719482421875e-220"},
    {0x1.fffffffffffffp-730, chars_format::scientific, 562,
        "3."
        "5410589842035614185994668373395481431509164422433080931947070546582256189949202174242897536219961289503444"
        "1312235838692413169313543249922246625170721756356246101860160631051624379941748317816311528611536244677303"
        "6324294543847484363245747616299616619438975270072971685091302728557072750349554952728595030379324100372123"
        "4029503941529928386561565355325777849227951671481494987476133464840367788766524283338067577559775343336881"
        "5945952243917121808121661182433954387554182676933877011138784819782924286611355971478756778217161435562410"
        "07672724663279950618743896484375e-220"},
    {0x1.fffffffffffffp-729, chars_format::scientific, 561,
        "7."
        "0821179684071228371989336746790962863018328844866161863894141093164512379898404348485795072439922579006888"
        "2624471677384826338627086499844493250341443512712492203720321262103248759883496635632623057223072489354607"
        "2648589087694968726491495232599233238877950540145943370182605457114145500699109905457190060758648200744246"
        "8059007883059856773123130710651555698455903342962989974952266929680735577533048566676135155119550686673763"
        "1891904487834243616243322364867908775108365353867754022277569639565848573222711942957513556434322871124820"
        "1534544932655990123748779296875e-220"},
    {0x1.fffffffffffffp-728, chars_format::scientific, 561,
        "1."
        "4164235936814245674397867349358192572603665768973232372778828218632902475979680869697159014487984515801377"
        "6524894335476965267725417299968898650068288702542498440744064252420649751976699327126524611444614497870921"
        "4529717817538993745298299046519846647775590108029188674036521091422829100139821981091438012151729640148849"
        "3611801576611971354624626142130311139691180668592597994990453385936147115506609713335227031023910137334752"
        "6378380897566848723248664472973581755021673070773550804455513927913169714644542388591502711286864574224964"
        "0306908986531198024749755859375e-219"},
    {0x1.fffffffffffffp-727, chars_format::scientific, 560,
        "2."
        "8328471873628491348795734698716385145207331537946464745557656437265804951959361739394318028975969031602755"
        "3049788670953930535450834599937797300136577405084996881488128504841299503953398654253049222889228995741842"
        "9059435635077987490596598093039693295551180216058377348073042182845658200279643962182876024303459280297698"
        "7223603153223942709249252284260622279382361337185195989980906771872294231013219426670454062047820274669505"
        "2756761795133697446497328945947163510043346141547101608911027855826339429289084777183005422573729148449928"
        "061381797306239604949951171875e-219"},
    {0x1.fffffffffffffp-726, chars_format::scientific, 559,
        "5."
        "6656943747256982697591469397432770290414663075892929491115312874531609903918723478788636057951938063205510"
        "6099577341907861070901669199875594600273154810169993762976257009682599007906797308506098445778457991483685"
        "8118871270155974981193196186079386591102360432116754696146084365691316400559287924365752048606918560595397"
        "4447206306447885418498504568521244558764722674370391979961813543744588462026438853340908124095640549339010"
        "5513523590267394892994657891894327020086692283094203217822055711652678858578169554366010845147458296899856"
        "12276359461247920989990234375e-219"},
    {0x1.fffffffffffffp-725, chars_format::scientific, 559,
        "1."
        "1331388749451396539518293879486554058082932615178585898223062574906321980783744695757727211590387612641102"
        "1219915468381572214180333839975118920054630962033998752595251401936519801581359461701219689155691598296737"
        "1623774254031194996238639237215877318220472086423350939229216873138263280111857584873150409721383712119079"
        "4889441261289577083699700913704248911752944534874078395992362708748917692405287770668181624819128109867802"
        "1102704718053478978598931578378865404017338456618840643564411142330535771715633910873202169029491659379971"
        "22455271892249584197998046875e-218"},
    {0x1.fffffffffffffp-724, chars_format::scientific, 558,
        "2."
        "2662777498902793079036587758973108116165865230357171796446125149812643961567489391515454423180775225282204"
        "2439830936763144428360667679950237840109261924067997505190502803873039603162718923402439378311383196593474"
        "3247548508062389992477278474431754636440944172846701878458433746276526560223715169746300819442767424238158"
        "9778882522579154167399401827408497823505889069748156791984725417497835384810575541336363249638256219735604"
        "2205409436106957957197863156757730808034676913237681287128822284661071543431267821746404338058983318759942"
        "4491054378449916839599609375e-218"},
    {0x1.fffffffffffffp-723, chars_format::scientific, 557,
        "4."
        "5325554997805586158073175517946216232331730460714343592892250299625287923134978783030908846361550450564408"
        "4879661873526288856721335359900475680218523848135995010381005607746079206325437846804878756622766393186948"
        "6495097016124779984954556948863509272881888345693403756916867492553053120447430339492601638885534848476317"
        "9557765045158308334798803654816995647011778139496313583969450834995670769621151082672726499276512439471208"
        "4410818872213915914395726313515461616069353826475362574257644569322143086862535643492808676117966637519884"
        "898210875689983367919921875e-218"},
    {0x1.fffffffffffffp-722, chars_format::scientific, 556,
        "9."
        "0651109995611172316146351035892432464663460921428687185784500599250575846269957566061817692723100901128816"
        "9759323747052577713442670719800951360437047696271990020762011215492158412650875693609757513245532786373897"
        "2990194032249559969909113897727018545763776691386807513833734985106106240894860678985203277771069696952635"
        "9115530090316616669597607309633991294023556278992627167938901669991341539242302165345452998553024878942416"
        "8821637744427831828791452627030923232138707652950725148515289138644286173725071286985617352235933275039769"
        "79642175137996673583984375e-218"},
    {0x1.fffffffffffffp-721, chars_format::scientific, 556,
        "1."
        "8130221999122234463229270207178486492932692184285737437156900119850115169253991513212363538544620180225763"
        "3951864749410515542688534143960190272087409539254398004152402243098431682530175138721951502649106557274779"
        "4598038806449911993981822779545403709152755338277361502766746997021221248178972135797040655554213939390527"
        "1823106018063323333919521461926798258804711255798525433587780333998268307848460433069090599710604975788483"
        "3764327548885566365758290525406184646427741530590145029703057827728857234745014257397123470447186655007953"
        "95928435027599334716796875e-217"},
    {0x1.fffffffffffffp-720, chars_format::scientific, 555,
        "3."
        "6260443998244468926458540414356972985865384368571474874313800239700230338507983026424727077089240360451526"
        "7903729498821031085377068287920380544174819078508796008304804486196863365060350277443903005298213114549558"
        "9196077612899823987963645559090807418305510676554723005533493994042442496357944271594081311108427878781054"
        "3646212036126646667839042923853596517609422511597050867175560667996536615696920866138181199421209951576966"
        "7528655097771132731516581050812369292855483061180290059406115655457714469490028514794246940894373310015907"
        "9185687005519866943359375e-217"},
    {0x1.fffffffffffffp-719, chars_format::scientific, 554,
        "7."
        "2520887996488937852917080828713945971730768737142949748627600479400460677015966052849454154178480720903053"
        "5807458997642062170754136575840761088349638157017592016609608972393726730120700554887806010596426229099117"
        "8392155225799647975927291118181614836611021353109446011066987988084884992715888543188162622216855757562108"
        "7292424072253293335678085847707193035218845023194101734351121335993073231393841732276362398842419903153933"
        "5057310195542265463033162101624738585710966122360580118812231310915428938980057029588493881788746620031815"
        "837137401103973388671875e-217"},
    {0x1.fffffffffffffp-718, chars_format::scientific, 554,
        "1."
        "4504177599297787570583416165742789194346153747428589949725520095880092135403193210569890830835696144180610"
        "7161491799528412434150827315168152217669927631403518403321921794478745346024140110977561202119285245819823"
        "5678431045159929595185458223636322967322204270621889202213397597616976998543177708637632524443371151512421"
        "7458484814450658667135617169541438607043769004638820346870224267198614646278768346455272479768483980630786"
        "7011462039108453092606632420324947717142193224472116023762446262183085787796011405917698776357749324006363"
        "167427480220794677734375e-216"},
    {0x1.fffffffffffffp-717, chars_format::scientific, 553,
        "2."
        "9008355198595575141166832331485578388692307494857179899451040191760184270806386421139781661671392288361221"
        "4322983599056824868301654630336304435339855262807036806643843588957490692048280221955122404238570491639647"
        "1356862090319859190370916447272645934644408541243778404426795195233953997086355417275265048886742303024843"
        "4916969628901317334271234339082877214087538009277640693740448534397229292557536692910544959536967961261573"
        "4022924078216906185213264840649895434284386448944232047524892524366171575592022811835397552715498648012726"
        "33485496044158935546875e-216"},
    {0x1.fffffffffffffp-716, chars_format::scientific, 552,
        "5."
        "8016710397191150282333664662971156777384614989714359798902080383520368541612772842279563323342784576722442"
        "8645967198113649736603309260672608870679710525614073613287687177914981384096560443910244808477140983279294"
        "2713724180639718380741832894545291869288817082487556808853590390467907994172710834550530097773484606049686"
        "9833939257802634668542468678165754428175076018555281387480897068794458585115073385821089919073935922523146"
        "8045848156433812370426529681299790868568772897888464095049785048732343151184045623670795105430997296025452"
        "6697099208831787109375e-216"},
    {0x1.fffffffffffffp-715, chars_format::scientific, 552,
        "1."
        "1603342079438230056466732932594231355476922997942871959780416076704073708322554568455912664668556915344488"
        "5729193439622729947320661852134521774135942105122814722657537435582996276819312088782048961695428196655858"
        "8542744836127943676148366578909058373857763416497511361770718078093581598834542166910106019554696921209937"
        "3966787851560526933708493735633150885635015203711056277496179413758891717023014677164217983814787184504629"
        "3609169631286762474085305936259958173713754579577692819009957009746468630236809124734159021086199459205090"
        "5339419841766357421875e-215"},
    {0x1.fffffffffffffp-714, chars_format::scientific, 551,
        "2."
        "3206684158876460112933465865188462710953845995885743919560832153408147416645109136911825329337113830688977"
        "1458386879245459894641323704269043548271884210245629445315074871165992553638624177564097923390856393311717"
        "7085489672255887352296733157818116747715526832995022723541436156187163197669084333820212039109393842419874"
        "7933575703121053867416987471266301771270030407422112554992358827517783434046029354328435967629574369009258"
        "7218339262573524948170611872519916347427509159155385638019914019492937260473618249468318042172398918410181"
        "067883968353271484375e-215"},
    {0x1.fffffffffffffp-713, chars_format::scientific, 550,
        "4."
        "6413368317752920225866931730376925421907691991771487839121664306816294833290218273823650658674227661377954"
        "2916773758490919789282647408538087096543768420491258890630149742331985107277248355128195846781712786623435"
        "4170979344511774704593466315636233495431053665990045447082872312374326395338168667640424078218787684839749"
        "5867151406242107734833974942532603542540060814844225109984717655035566868092058708656871935259148738018517"
        "4436678525147049896341223745039832694855018318310771276039828038985874520947236498936636084344797836820362"
        "13576793670654296875e-215"},
    {0x1.fffffffffffffp-712, chars_format::scientific, 549,
        "9."
        "2826736635505840451733863460753850843815383983542975678243328613632589666580436547647301317348455322755908"
        "5833547516981839578565294817076174193087536840982517781260299484663970214554496710256391693563425573246870"
        "8341958689023549409186932631272466990862107331980090894165744624748652790676337335280848156437575369679499"
        "1734302812484215469667949885065207085080121629688450219969435310071133736184117417313743870518297476037034"
        "8873357050294099792682447490079665389710036636621542552079656077971749041894472997873272168689595673640724"
        "2715358734130859375e-215"},
    {0x1.fffffffffffffp-711, chars_format::scientific, 549,
        "1."
        "8565347327101168090346772692150770168763076796708595135648665722726517933316087309529460263469691064551181"
        "7166709503396367915713058963415234838617507368196503556252059896932794042910899342051278338712685114649374"
        "1668391737804709881837386526254493398172421466396018178833148924949730558135267467056169631287515073935899"
        "8346860562496843093933589977013041417016024325937690043993887062014226747236823483462748774103659495207406"
        "9774671410058819958536489498015933077942007327324308510415931215594349808378894599574654433737919134728144"
        "8543071746826171875e-214"},
    {0x1.fffffffffffffp-710, chars_format::scientific, 548,
        "3."
        "7130694654202336180693545384301540337526153593417190271297331445453035866632174619058920526939382129102363"
        "4333419006792735831426117926830469677235014736393007112504119793865588085821798684102556677425370229298748"
        "3336783475609419763674773052508986796344842932792036357666297849899461116270534934112339262575030147871799"
        "6693721124993686187867179954026082834032048651875380087987774124028453494473646966925497548207318990414813"
        "9549342820117639917072978996031866155884014654648617020831862431188699616757789199149308867475838269456289"
        "708614349365234375e-214"},
    {0x1.fffffffffffffp-709, chars_format::scientific, 547,
        "7."
        "4261389308404672361387090768603080675052307186834380542594662890906071733264349238117841053878764258204726"
        "8666838013585471662852235853660939354470029472786014225008239587731176171643597368205113354850740458597496"
        "6673566951218839527349546105017973592689685865584072715332595699798922232541069868224678525150060295743599"
        "3387442249987372375734359908052165668064097303750760175975548248056906988947293933850995096414637980829627"
        "9098685640235279834145957992063732311768029309297234041663724862377399233515578398298617734951676538912579"
        "41722869873046875e-214"},
    {0x1.fffffffffffffp-708, chars_format::scientific, 547,
        "1."
        "4852277861680934472277418153720616135010461437366876108518932578181214346652869847623568210775752851640945"
        "3733367602717094332570447170732187870894005894557202845001647917546235234328719473641022670970148091719499"
        "3334713390243767905469909221003594718537937173116814543066519139959784446508213973644935705030012059148719"
        "8677488449997474475146871981610433133612819460750152035195109649611381397789458786770199019282927596165925"
        "5819737128047055966829191598412746462353605861859446808332744972475479846703115679659723546990335307782515"
        "88344573974609375e-213"},
    {0x1.fffffffffffffp-707, chars_format::scientific, 546,
        "2."
        "9704555723361868944554836307441232270020922874733752217037865156362428693305739695247136421551505703281890"
        "7466735205434188665140894341464375741788011789114405690003295835092470468657438947282045341940296183438998"
        "6669426780487535810939818442007189437075874346233629086133038279919568893016427947289871410060024118297439"
        "7354976899994948950293743963220866267225638921500304070390219299222762795578917573540398038565855192331851"
        "1639474256094111933658383196825492924707211723718893616665489944950959693406231359319447093980670615565031"
        "7668914794921875e-213"},
    {0x1.fffffffffffffp-706, chars_format::scientific, 545,
        "5."
        "9409111446723737889109672614882464540041845749467504434075730312724857386611479390494272843103011406563781"
        "4933470410868377330281788682928751483576023578228811380006591670184940937314877894564090683880592366877997"
        "3338853560975071621879636884014378874151748692467258172266076559839137786032855894579742820120048236594879"
        "4709953799989897900587487926441732534451277843000608140780438598445525591157835147080796077131710384663702"
        "3278948512188223867316766393650985849414423447437787233330979889901919386812462718638894187961341231130063"
        "533782958984375e-213"},
    {0x1.fffffffffffffp-705, chars_format::scientific, 545,
        "1."
        "1881822289344747577821934522976492908008369149893500886815146062544971477322295878098854568620602281312756"
        "2986694082173675466056357736585750296715204715645762276001318334036988187462975578912818136776118473375599"
        "4667770712195014324375927376802875774830349738493451634453215311967827557206571178915948564024009647318975"
        "8941990759997979580117497585288346506890255568600121628156087719689105118231567029416159215426342076932740"
        "4655789702437644773463353278730197169882884689487557446666195977980383877362492543727778837592268246226012"
        "706756591796875e-212"},
    {0x1.fffffffffffffp-704, chars_format::scientific, 544,
        "2."
        "3763644578689495155643869045952985816016738299787001773630292125089942954644591756197709137241204562625512"
        "5973388164347350932112715473171500593430409431291524552002636668073976374925951157825636273552236946751198"
        "9335541424390028648751854753605751549660699476986903268906430623935655114413142357831897128048019294637951"
        "7883981519995959160234995170576693013780511137200243256312175439378210236463134058832318430852684153865480"
        "9311579404875289546926706557460394339765769378975114893332391955960767754724985087455557675184536492452025"
        "41351318359375e-212"},
    {0x1.fffffffffffffp-703, chars_format::scientific, 543,
        "4."
        "7527289157378990311287738091905971632033476599574003547260584250179885909289183512395418274482409125251025"
        "1946776328694701864225430946343001186860818862583049104005273336147952749851902315651272547104473893502397"
        "8671082848780057297503709507211503099321398953973806537812861247871310228826284715663794256096038589275903"
        "5767963039991918320469990341153386027561022274400486512624350878756420472926268117664636861705368307730961"
        "8623158809750579093853413114920788679531538757950229786664783911921535509449970174911115350369072984904050"
        "8270263671875e-212"},
    {0x1.fffffffffffffp-702, chars_format::scientific, 542,
        "9."
        "5054578314757980622575476183811943264066953199148007094521168500359771818578367024790836548964818250502050"
        "3893552657389403728450861892686002373721637725166098208010546672295905499703804631302545094208947787004795"
        "7342165697560114595007419014423006198642797907947613075625722495742620457652569431327588512192077178551807"
        "1535926079983836640939980682306772055122044548800973025248701757512840945852536235329273723410736615461923"
        "7246317619501158187706826229841577359063077515900459573329567823843071018899940349822230700738145969808101"
        "654052734375e-212"},
    {0x1.fffffffffffffp-701, chars_format::scientific, 542,
        "1."
        "9010915662951596124515095236762388652813390639829601418904233700071954363715673404958167309792963650100410"
        "0778710531477880745690172378537200474744327545033219641602109334459181099940760926260509018841789557400959"
        "1468433139512022919001483802884601239728559581589522615125144499148524091530513886265517702438415435710361"
        "4307185215996767328187996136461354411024408909760194605049740351502568189170507247065854744682147323092384"
        "7449263523900231637541365245968315471812615503180091914665913564768614203779988069964446140147629193961620"
        "330810546875e-211"},
    {0x1.fffffffffffffp-700, chars_format::scientific, 541,
        "3."
        "8021831325903192249030190473524777305626781279659202837808467400143908727431346809916334619585927300200820"
        "1557421062955761491380344757074400949488655090066439283204218668918362199881521852521018037683579114801918"
        "2936866279024045838002967605769202479457119163179045230250288998297048183061027772531035404876830871420722"
        "8614370431993534656375992272922708822048817819520389210099480703005136378341014494131709489364294646184769"
        "4898527047800463275082730491936630943625231006360183829331827129537228407559976139928892280295258387923240"
        "66162109375e-211"},
    {0x1.fffffffffffffp-699, chars_format::scientific, 540,
        "7."
        "6043662651806384498060380947049554611253562559318405675616934800287817454862693619832669239171854600401640"
        "3114842125911522982760689514148801898977310180132878566408437337836724399763043705042036075367158229603836"
        "5873732558048091676005935211538404958914238326358090460500577996594096366122055545062070809753661742841445"
        "7228740863987069312751984545845417644097635639040778420198961406010272756682028988263418978728589292369538"
        "9797054095600926550165460983873261887250462012720367658663654259074456815119952279857784560590516775846481"
        "3232421875e-211"},
    {0x1.fffffffffffffp-698, chars_format::scientific, 540,
        "1."
        "5208732530361276899612076189409910922250712511863681135123386960057563490972538723966533847834370920080328"
        "0622968425182304596552137902829760379795462036026575713281687467567344879952608741008407215073431645920767"
        "3174746511609618335201187042307680991782847665271618092100115599318819273224411109012414161950732348568289"
        "1445748172797413862550396909169083528819527127808155684039792281202054551336405797652683795745717858473907"
        "7959410819120185310033092196774652377450092402544073531732730851814891363023990455971556912118103355169296"
        "2646484375e-210"},
    {0x1.fffffffffffffp-697, chars_format::scientific, 539,
        "3."
        "0417465060722553799224152378819821844501425023727362270246773920115126981945077447933067695668741840160656"
        "1245936850364609193104275805659520759590924072053151426563374935134689759905217482016814430146863291841534"
        "6349493023219236670402374084615361983565695330543236184200231198637638546448822218024828323901464697136578"
        "2891496345594827725100793818338167057639054255616311368079584562404109102672811595305367591491435716947815"
        "5918821638240370620066184393549304754900184805088147063465461703629782726047980911943113824236206710338592"
        "529296875e-210"},
    {0x1.fffffffffffffp-696, chars_format::scientific, 538,
        "6."
        "0834930121445107598448304757639643689002850047454724540493547840230253963890154895866135391337483680321312"
        "2491873700729218386208551611319041519181848144106302853126749870269379519810434964033628860293726583683069"
        "2698986046438473340804748169230723967131390661086472368400462397275277092897644436049656647802929394273156"
        "5782992691189655450201587636676334115278108511232622736159169124808218205345623190610735182982871433895631"
        "1837643276480741240132368787098609509800369610176294126930923407259565452095961823886227648472413420677185"
        "05859375e-210"},
    {0x1.fffffffffffffp-695, chars_format::scientific, 538,
        "1."
        "2166986024289021519689660951527928737800570009490944908098709568046050792778030979173227078267496736064262"
        "4498374740145843677241710322263808303836369628821260570625349974053875903962086992806725772058745316736613"
        "8539797209287694668160949633846144793426278132217294473680092479455055418579528887209931329560585878854631"
        "3156598538237931090040317527335266823055621702246524547231833824961643641069124638122147036596574286779126"
        "2367528655296148248026473757419721901960073922035258825386184681451913090419192364777245529694482684135437"
        "01171875e-209"},
    {0x1.fffffffffffffp-694, chars_format::scientific, 537,
        "2."
        "4333972048578043039379321903055857475601140018981889816197419136092101585556061958346454156534993472128524"
        "8996749480291687354483420644527616607672739257642521141250699948107751807924173985613451544117490633473227"
        "7079594418575389336321899267692289586852556264434588947360184958910110837159057774419862659121171757709262"
        "6313197076475862180080635054670533646111243404493049094463667649923287282138249276244294073193148573558252"
        "4735057310592296496052947514839443803920147844070517650772369362903826180838384729554491059388965368270874"
        "0234375e-209"},
    {0x1.fffffffffffffp-693, chars_format::scientific, 536,
        "4."
        "8667944097156086078758643806111714951202280037963779632394838272184203171112123916692908313069986944257049"
        "7993498960583374708966841289055233215345478515285042282501399896215503615848347971226903088234981266946455"
        "4159188837150778672643798535384579173705112528869177894720369917820221674318115548839725318242343515418525"
        "2626394152951724360161270109341067292222486808986098188927335299846574564276498552488588146386297147116504"
        "9470114621184592992105895029678887607840295688141035301544738725807652361676769459108982118777930736541748"
        "046875e-209"},
    {0x1.fffffffffffffp-692, chars_format::scientific, 535,
        "9."
        "7335888194312172157517287612223429902404560075927559264789676544368406342224247833385816626139973888514099"
        "5986997921166749417933682578110466430690957030570084565002799792431007231696695942453806176469962533892910"
        "8318377674301557345287597070769158347410225057738355789440739835640443348636231097679450636484687030837050"
        "5252788305903448720322540218682134584444973617972196377854670599693149128552997104977176292772594294233009"
        "8940229242369185984211790059357775215680591376282070603089477451615304723353538918217964237555861473083496"
        "09375e-209"},
    {0x1.fffffffffffffp-691, chars_format::scientific, 535,
        "1."
        "9467177638862434431503457522444685980480912015185511852957935308873681268444849566677163325227994777702819"
        "9197399584233349883586736515622093286138191406114016913000559958486201446339339188490761235293992506778582"
        "1663675534860311469057519414153831669482045011547671157888147967128088669727246219535890127296937406167410"
        "1050557661180689744064508043736426916888994723594439275570934119938629825710599420995435258554518858846601"
        "9788045848473837196842358011871555043136118275256414120617895490323060944670707783643592847511172294616699"
        "21875e-208"},
    {0x1.fffffffffffffp-690, chars_format::scientific, 534,
        "3."
        "8934355277724868863006915044889371960961824030371023705915870617747362536889699133354326650455989555405639"
        "8394799168466699767173473031244186572276382812228033826001119916972402892678678376981522470587985013557164"
        "3327351069720622938115038828307663338964090023095342315776295934256177339454492439071780254593874812334820"
        "2101115322361379488129016087472853833777989447188878551141868239877259651421198841990870517109037717693203"
        "9576091696947674393684716023743110086272236550512828241235790980646121889341415567287185695022344589233398"
        "4375e-208"},
    {0x1.fffffffffffffp-689, chars_format::scientific, 533,
        "7."
        "7868710555449737726013830089778743921923648060742047411831741235494725073779398266708653300911979110811279"
        "6789598336933399534346946062488373144552765624456067652002239833944805785357356753963044941175970027114328"
        "6654702139441245876230077656615326677928180046190684631552591868512354678908984878143560509187749624669640"
        "4202230644722758976258032174945707667555978894377757102283736479754519302842397683981741034218075435386407"
        "9152183393895348787369432047486220172544473101025656482471581961292243778682831134574371390044689178466796"
        "875e-208"},
    {0x1.fffffffffffffp-688, chars_format::scientific, 533,
        "1."
        "5573742111089947545202766017955748784384729612148409482366348247098945014755879653341730660182395822162255"
        "9357919667386679906869389212497674628910553124891213530400447966788961157071471350792608988235194005422865"
        "7330940427888249175246015531323065335585636009238136926310518373702470935781796975628712101837549924933928"
        "0840446128944551795251606434989141533511195778875551420456747295950903860568479536796348206843615087077281"
        "5830436678779069757473886409497244034508894620205131296494316392258448755736566226914874278008937835693359"
        "375e-207"},
    {0x1.fffffffffffffp-687, chars_format::scientific, 532,
        "3."
        "1147484222179895090405532035911497568769459224296818964732696494197890029511759306683461320364791644324511"
        "8715839334773359813738778424995349257821106249782427060800895933577922314142942701585217976470388010845731"
        "4661880855776498350492031062646130671171272018476273852621036747404941871563593951257424203675099849867856"
        "1680892257889103590503212869978283067022391557751102840913494591901807721136959073592696413687230174154563"
        "1660873357558139514947772818994488069017789240410262592988632784516897511473132453829748556017875671386718"
        "75e-207"},
    {0x1.fffffffffffffp-686, chars_format::scientific, 531,
        "6."
        "2294968444359790180811064071822995137538918448593637929465392988395780059023518613366922640729583288649023"
        "7431678669546719627477556849990698515642212499564854121601791867155844628285885403170435952940776021691462"
        "9323761711552996700984062125292261342342544036952547705242073494809883743127187902514848407350199699735712"
        "3361784515778207181006425739956566134044783115502205681826989183803615442273918147185392827374460348309126"
        "3321746715116279029895545637988976138035578480820525185977265569033795022946264907659497112035751342773437"
        "5e-207"},
    {0x1.fffffffffffffp-685, chars_format::scientific, 531,
        "1."
        "2458993688871958036162212814364599027507783689718727585893078597679156011804703722673384528145916657729804"
        "7486335733909343925495511369998139703128442499912970824320358373431168925657177080634087190588155204338292"
        "5864752342310599340196812425058452268468508807390509541048414698961976748625437580502969681470039939947142"
        "4672356903155641436201285147991313226808956623100441136365397836760723088454783629437078565474892069661825"
        "2664349343023255805979109127597795227607115696164105037195453113806759004589252981531899422407150268554687"
        "5e-206"},
    {0x1.fffffffffffffp-684, chars_format::scientific, 530,
        "2."
        "4917987377743916072324425628729198055015567379437455171786157195358312023609407445346769056291833315459609"
        "4972671467818687850991022739996279406256884999825941648640716746862337851314354161268174381176310408676585"
        "1729504684621198680393624850116904536937017614781019082096829397923953497250875161005939362940079879894284"
        "9344713806311282872402570295982626453617913246200882272730795673521446176909567258874157130949784139323650"
        "5328698686046511611958218255195590455214231392328210074390906227613518009178505963063798844814300537109375"
        "e-206"},
    {0x1.fffffffffffffp-683, chars_format::scientific, 529,
        "4."
        "9835974755487832144648851257458396110031134758874910343572314390716624047218814890693538112583666630919218"
        "9945342935637375701982045479992558812513769999651883297281433493724675702628708322536348762352620817353170"
        "3459009369242397360787249700233809073874035229562038164193658795847906994501750322011878725880159759788569"
        "8689427612622565744805140591965252907235826492401764545461591347042892353819134517748314261899568278647301"
        "065739737209302322391643651039118091042846278465642014878181245522703601835701192612759768962860107421875e"
        "-206"},
    {0x1.fffffffffffffp-682, chars_format::scientific, 528,
        "9."
        "9671949510975664289297702514916792220062269517749820687144628781433248094437629781387076225167333261838437"
        "9890685871274751403964090959985117625027539999303766594562866987449351405257416645072697524705241634706340"
        "6918018738484794721574499400467618147748070459124076328387317591695813989003500644023757451760319519577139"
        "7378855225245131489610281183930505814471652984803529090923182694085784707638269035496628523799136557294602"
        "13147947441860464478328730207823618208569255693128402975636249104540720367140238522551953792572021484375e-"
        "206"},
    {0x1.fffffffffffffp-681, chars_format::scientific, 528,
        "1."
        "9934389902195132857859540502983358444012453903549964137428925756286649618887525956277415245033466652367687"
        "5978137174254950280792818191997023525005507999860753318912573397489870281051483329014539504941048326941268"
        "1383603747696958944314899880093523629549614091824815265677463518339162797800700128804751490352063903915427"
        "9475771045049026297922056236786101162894330596960705818184636538817156941527653807099325704759827311458920"
        "42629589488372092895665746041564723641713851138625680595127249820908144073428047704510390758514404296875e-"
        "205"},
    {0x1.fffffffffffffp-680, chars_format::scientific, 527,
        "3."
        "9868779804390265715719081005966716888024907807099928274857851512573299237775051912554830490066933304735375"
        "1956274348509900561585636383994047050011015999721506637825146794979740562102966658029079009882096653882536"
        "2767207495393917888629799760187047259099228183649630531354927036678325595601400257609502980704127807830855"
        "8951542090098052595844112473572202325788661193921411636369273077634313883055307614198651409519654622917840"
        "8525917897674418579133149208312944728342770227725136119025449964181628814685609540902078151702880859375e-"
        "205"},
    {0x1.fffffffffffffp-679, chars_format::scientific, 526,
        "7."
        "9737559608780531431438162011933433776049815614199856549715703025146598475550103825109660980133866609470750"
        "3912548697019801123171272767988094100022031999443013275650293589959481124205933316058158019764193307765072"
        "5534414990787835777259599520374094518198456367299261062709854073356651191202800515219005961408255615661711"
        "7903084180196105191688224947144404651577322387842823272738546155268627766110615228397302819039309245835681"
        "705183579534883715826629841662588945668554045545027223805089992836325762937121908180415630340576171875e-"
        "205"},
    {0x1.fffffffffffffp-678, chars_format::scientific, 526,
        "1."
        "5947511921756106286287632402386686755209963122839971309943140605029319695110020765021932196026773321894150"
        "0782509739403960224634254553597618820004406399888602655130058717991896224841186663211631603952838661553014"
        "5106882998157567155451919904074818903639691273459852212541970814671330238240560103043801192281651123132342"
        "3580616836039221038337644989428880930315464477568564654547709231053725553222123045679460563807861849167136"
        "341036715906976743165325968332517789133710809109005444761017998567265152587424381636083126068115234375e-"
        "204"},
    {0x1.fffffffffffffp-677, chars_format::scientific, 525,
        "3."
        "1895023843512212572575264804773373510419926245679942619886281210058639390220041530043864392053546643788300"
        "1565019478807920449268509107195237640008812799777205310260117435983792449682373326423263207905677323106029"
        "0213765996315134310903839808149637807279382546919704425083941629342660476481120206087602384563302246264684"
        "7161233672078442076675289978857761860630928955137129309095418462107451106444246091358921127615723698334272"
        "68207343181395348633065193666503557826742161821801088952203599713453030517484876327216625213623046875e-"
        "204"},
    {0x1.fffffffffffffp-676, chars_format::scientific, 524,
        "6."
        "3790047687024425145150529609546747020839852491359885239772562420117278780440083060087728784107093287576600"
        "3130038957615840898537018214390475280017625599554410620520234871967584899364746652846526415811354646212058"
        "0427531992630268621807679616299275614558765093839408850167883258685320952962240412175204769126604492529369"
        "4322467344156884153350579957715523721261857910274258618190836924214902212888492182717842255231447396668545"
        "3641468636279069726613038733300711565348432364360217790440719942690606103496975265443325042724609375e-"
        "204"},
    {0x1.fffffffffffffp-675, chars_format::scientific, 524,
        "1."
        "2758009537404885029030105921909349404167970498271977047954512484023455756088016612017545756821418657515320"
        "0626007791523168179707403642878095056003525119910882124104046974393516979872949330569305283162270929242411"
        "6085506398526053724361535923259855122911753018767881770033576651737064190592448082435040953825320898505873"
        "8864493468831376830670115991543104744252371582054851723638167384842980442577698436543568451046289479333709"
        "0728293727255813945322607746660142313069686472872043558088143988538121220699395053088665008544921875e-"
        "203"},
    {0x1.fffffffffffffp-674, chars_format::scientific, 523,
        "2."
        "5516019074809770058060211843818698808335940996543954095909024968046911512176033224035091513642837315030640"
        "1252015583046336359414807285756190112007050239821764248208093948787033959745898661138610566324541858484823"
        "2171012797052107448723071846519710245823506037535763540067153303474128381184896164870081907650641797011747"
        "7728986937662753661340231983086209488504743164109703447276334769685960885155396873087136902092578958667418"
        "145658745451162789064521549332028462613937294574408711617628797707624244139879010617733001708984375e-203"},
    {0x1.fffffffffffffp-673, chars_format::scientific, 522,
        "5."
        "1032038149619540116120423687637397616671881993087908191818049936093823024352066448070183027285674630061280"
        "2504031166092672718829614571512380224014100479643528496416187897574067919491797322277221132649083716969646"
        "4342025594104214897446143693039420491647012075071527080134306606948256762369792329740163815301283594023495"
        "5457973875325507322680463966172418977009486328219406894552669539371921770310793746174273804185157917334836"
        "29131749090232557812904309866405692522787458914881742323525759541524848827975802123546600341796875e-203"},
    {0x1.fffffffffffffp-672, chars_format::scientific, 522,
        "1."
        "0206407629923908023224084737527479523334376398617581638363609987218764604870413289614036605457134926012256"
        "0500806233218534543765922914302476044802820095928705699283237579514813583898359464455444226529816743393929"
        "2868405118820842979489228738607884098329402415014305416026861321389651352473958465948032763060256718804699"
        "1091594775065101464536092793234483795401897265643881378910533907874384354062158749234854760837031583466967"
        "25826349818046511562580861973281138504557491782976348464705151908304969765595160424709320068359375e-202"},
    {0x1.fffffffffffffp-671, chars_format::scientific, 521,
        "2."
        "0412815259847816046448169475054959046668752797235163276727219974437529209740826579228073210914269852024512"
        "1001612466437069087531845828604952089605640191857411398566475159029627167796718928910888453059633486787858"
        "5736810237641685958978457477215768196658804830028610832053722642779302704947916931896065526120513437609398"
        "2183189550130202929072185586468967590803794531287762757821067815748768708124317498469709521674063166933934"
        "5165269963609302312516172394656227700911498356595269692941030381660993953119032084941864013671875e-202"},
    {0x1.fffffffffffffp-670, chars_format::scientific, 520,
        "4."
        "0825630519695632092896338950109918093337505594470326553454439948875058419481653158456146421828539704049024"
        "2003224932874138175063691657209904179211280383714822797132950318059254335593437857821776906119266973575717"
        "1473620475283371917956914954431536393317609660057221664107445285558605409895833863792131052241026875218796"
        "4366379100260405858144371172937935181607589062575525515642135631497537416248634996939419043348126333867869"
        "033053992721860462503234478931245540182299671319053938588206076332198790623806416988372802734375e-202"},
    {0x1.fffffffffffffp-669, chars_format::scientific, 519,
        "8."
        "1651261039391264185792677900219836186675011188940653106908879897750116838963306316912292843657079408098048"
        "4006449865748276350127383314419808358422560767429645594265900636118508671186875715643553812238533947151434"
        "2947240950566743835913829908863072786635219320114443328214890571117210819791667727584262104482053750437592"
        "8732758200520811716288742345875870363215178125151051031284271262995074832497269993878838086696252667735738"
        "06610798544372092500646895786249108036459934263810787717641215266439758124761283397674560546875e-202"},
    {0x1.fffffffffffffp-668, chars_format::scientific, 519,
        "1."
        "6330252207878252837158535580043967237335002237788130621381775979550023367792661263382458568731415881619609"
        "6801289973149655270025476662883961671684512153485929118853180127223701734237375143128710762447706789430286"
        "8589448190113348767182765981772614557327043864022888665642978114223442163958333545516852420896410750087518"
        "5746551640104162343257748469175174072643035625030210206256854252599014966499453998775767617339250533547147"
        "61322159708874418500129379157249821607291986852762157543528243053287951624952256679534912109375e-201"},
    {0x1.fffffffffffffp-667, chars_format::scientific, 518,
        "3."
        "2660504415756505674317071160087934474670004475576261242763551959100046735585322526764917137462831763239219"
        "3602579946299310540050953325767923343369024306971858237706360254447403468474750286257421524895413578860573"
        "7178896380226697534365531963545229114654087728045777331285956228446884327916667091033704841792821500175037"
        "1493103280208324686515496938350348145286071250060420412513708505198029932998907997551535234678501067094295"
        "2264431941774883700025875831449964321458397370552431508705648610657590324990451335906982421875e-201"},
    {0x1.fffffffffffffp-666, chars_format::scientific, 517,
        "6."
        "5321008831513011348634142320175868949340008951152522485527103918200093471170645053529834274925663526478438"
        "7205159892598621080101906651535846686738048613943716475412720508894806936949500572514843049790827157721147"
        "4357792760453395068731063927090458229308175456091554662571912456893768655833334182067409683585643000350074"
        "2986206560416649373030993876700696290572142500120840825027417010396059865997815995103070469357002134188590"
        "452886388354976740005175166289992864291679474110486301741129722131518064998090267181396484375e-201"},
    {0x1.fffffffffffffp-665, chars_format::scientific, 517,
        "1."
        "3064201766302602269726828464035173789868001790230504497105420783640018694234129010705966854985132705295687"
        "7441031978519724216020381330307169337347609722788743295082544101778961387389900114502968609958165431544229"
        "4871558552090679013746212785418091645861635091218310932514382491378753731166666836413481936717128600070014"
        "8597241312083329874606198775340139258114428500024168165005483402079211973199563199020614093871400426837718"
        "090577277670995348001035033257998572858335894822097260348225944426303612999618053436279296875e-200"},
    {0x1.fffffffffffffp-664, chars_format::scientific, 516,
        "2."
        "6128403532605204539453656928070347579736003580461008994210841567280037388468258021411933709970265410591375"
        "4882063957039448432040762660614338674695219445577486590165088203557922774779800229005937219916330863088458"
        "9743117104181358027492425570836183291723270182436621865028764982757507462333333672826963873434257200140029"
        "7194482624166659749212397550680278516228857000048336330010966804158423946399126398041228187742800853675436"
        "18115455534199069600207006651599714571667178964419452069645188885260722599923610687255859375e-200"},
    {0x1.fffffffffffffp-663, chars_format::scientific, 515,
        "5."
        "2256807065210409078907313856140695159472007160922017988421683134560074776936516042823867419940530821182750"
        "9764127914078896864081525321228677349390438891154973180330176407115845549559600458011874439832661726176917"
        "9486234208362716054984851141672366583446540364873243730057529965515014924666667345653927746868514400280059"
        "4388965248333319498424795101360557032457714000096672660021933608316847892798252796082456375485601707350872"
        "3623091106839813920041401330319942914333435792883890413929037777052144519984722137451171875e-200"},
    {0x1.fffffffffffffp-662, chars_format::scientific, 515,
        "1."
        "0451361413042081815781462771228139031894401432184403597684336626912014955387303208564773483988106164236550"
        "1952825582815779372816305064245735469878087778230994636066035281423169109911920091602374887966532345235383"
        "5897246841672543210996970228334473316689308072974648746011505993103002984933333469130785549373702880056011"
        "8877793049666663899684959020272111406491542800019334532004386721663369578559650559216491275097120341470174"
        "4724618221367962784008280266063988582866687158576778082785807555410428903996944427490234375e-199"},
    {0x1.fffffffffffffp-661, chars_format::scientific, 514,
        "2."
        "0902722826084163631562925542456278063788802864368807195368673253824029910774606417129546967976212328473100"
        "3905651165631558745632610128491470939756175556461989272132070562846338219823840183204749775933064690470767"
        "1794493683345086421993940456668946633378616145949297492023011986206005969866666938261571098747405760112023"
        "7755586099333327799369918040544222812983085600038669064008773443326739157119301118432982550194240682940348"
        "944923644273592556801656053212797716573337431715355616557161511082085780799388885498046875e-199"},
    {0x1.fffffffffffffp-660, chars_format::scientific, 513,
        "4."
        "1805445652168327263125851084912556127577605728737614390737346507648059821549212834259093935952424656946200"
        "7811302331263117491265220256982941879512351112923978544264141125692676439647680366409499551866129380941534"
        "3588987366690172843987880913337893266757232291898594984046023972412011939733333876523142197494811520224047"
        "5511172198666655598739836081088445625966171200077338128017546886653478314238602236865965100388481365880697"
        "88984728854718511360331210642559543314667486343071123311432302216417156159877777099609375e-199"},
    {0x1.fffffffffffffp-659, chars_format::scientific, 512,
        "8."
        "3610891304336654526251702169825112255155211457475228781474693015296119643098425668518187871904849313892401"
        "5622604662526234982530440513965883759024702225847957088528282251385352879295360732818999103732258761883068"
        "7177974733380345687975761826675786533514464583797189968092047944824023879466667753046284394989623040448095"
        "1022344397333311197479672162176891251932342400154676256035093773306956628477204473731930200776962731761395"
        "7796945770943702272066242128511908662933497268614224662286460443283431231975555419921875e-199"},
    {0x1.fffffffffffffp-658, chars_format::scientific, 512,
        "1."
        "6722178260867330905250340433965022451031042291495045756294938603059223928619685133703637574380969862778480"
        "3124520932505246996506088102793176751804940445169591417705656450277070575859072146563799820746451752376613"
        "7435594946676069137595152365335157306702892916759437993618409588964804775893333550609256878997924608089619"
        "0204468879466662239495934432435378250386468480030935251207018754661391325695440894746386040155392546352279"
        "1559389154188740454413248425702381732586699453722844932457292088656686246395111083984375e-198"},
    {0x1.fffffffffffffp-657, chars_format::scientific, 511,
        "3."
        "3444356521734661810500680867930044902062084582990091512589877206118447857239370267407275148761939725556960"
        "6249041865010493993012176205586353503609880890339182835411312900554141151718144293127599641492903504753227"
        "4871189893352138275190304730670314613405785833518875987236819177929609551786667101218513757995849216179238"
        "0408937758933324478991868864870756500772936960061870502414037509322782651390881789492772080310785092704558"
        "311877830837748090882649685140476346517339890744568986491458417731337249279022216796875e-198"},
    {0x1.fffffffffffffp-656, chars_format::scientific, 510,
        "6."
        "6888713043469323621001361735860089804124169165980183025179754412236895714478740534814550297523879451113921"
        "2498083730020987986024352411172707007219761780678365670822625801108282303436288586255199282985807009506454"
        "9742379786704276550380609461340629226811571667037751974473638355859219103573334202437027515991698432358476"
        "0817875517866648957983737729741513001545873920123741004828075018645565302781763578985544160621570185409116"
        "62375566167549618176529937028095269303467978148913797298291683546267449855804443359375e-198"},
    {0x1.fffffffffffffp-655, chars_format::scientific, 510,
        "1."
        "3377742608693864724200272347172017960824833833196036605035950882447379142895748106962910059504775890222784"
        "2499616746004197597204870482234541401443952356135673134164525160221656460687257717251039856597161401901290"
        "9948475957340855310076121892268125845362314333407550394894727671171843820714666840487405503198339686471695"
        "2163575103573329791596747545948302600309174784024748200965615003729113060556352715797108832124314037081823"
        "32475113233509923635305987405619053860693595629782759459658336709253489971160888671875e-197"},
    {0x1.fffffffffffffp-654, chars_format::scientific, 509,
        "2."
        "6755485217387729448400544694344035921649667666392073210071901764894758285791496213925820119009551780445568"
        "4999233492008395194409740964469082802887904712271346268329050320443312921374515434502079713194322803802581"
        "9896951914681710620152243784536251690724628666815100789789455342343687641429333680974811006396679372943390"
        "4327150207146659583193495091896605200618349568049496401931230007458226121112705431594217664248628074163646"
        "6495022646701984727061197481123810772138719125956551891931667341850697994232177734375e-197"},
    {0x1.fffffffffffffp-653, chars_format::scientific, 508,
        "5."
        "3510970434775458896801089388688071843299335332784146420143803529789516571582992427851640238019103560891136"
        "9998466984016790388819481928938165605775809424542692536658100640886625842749030869004159426388645607605163"
        "9793903829363421240304487569072503381449257333630201579578910684687375282858667361949622012793358745886780"
        "8654300414293319166386990183793210401236699136098992803862460014916452242225410863188435328497256148327293"
        "299004529340396945412239496224762154427743825191310378386333468370139598846435546875e-197"},
    {0x1.fffffffffffffp-652, chars_format::scientific, 508,
        "1."
        "0702194086955091779360217877737614368659867066556829284028760705957903314316598485570328047603820712178227"
        "3999693396803358077763896385787633121155161884908538507331620128177325168549806173800831885277729121521032"
        "7958780765872684248060897513814500676289851466726040315915782136937475056571733472389924402558671749177356"
        "1730860082858663833277398036758642080247339827219798560772492002983290448445082172637687065699451229665458"
        "659800905868079389082447899244952430885548765038262075677266693674027919769287109375e-196"},
    {0x1.fffffffffffffp-651, chars_format::scientific, 507,
        "2."
        "1404388173910183558720435755475228737319734133113658568057521411915806628633196971140656095207641424356454"
        "7999386793606716155527792771575266242310323769817077014663240256354650337099612347601663770555458243042065"
        "5917561531745368496121795027629001352579702933452080631831564273874950113143466944779848805117343498354712"
        "3461720165717327666554796073517284160494679654439597121544984005966580896890164345275374131398902459330917"
        "31960181173615877816489579848990486177109753007652415135453338734805583953857421875e-196"},
    {0x1.fffffffffffffp-650, chars_format::scientific, 506,
        "4."
        "2808776347820367117440871510950457474639468266227317136115042823831613257266393942281312190415282848712909"
        "5998773587213432311055585543150532484620647539634154029326480512709300674199224695203327541110916486084131"
        "1835123063490736992243590055258002705159405866904161263663128547749900226286933889559697610234686996709424"
        "6923440331434655333109592147034568320989359308879194243089968011933161793780328690550748262797804918661834"
        "6392036234723175563297915969798097235421950601530483027090667746961116790771484375e-196"},
    {0x1.fffffffffffffp-649, chars_format::scientific, 505,
        "8."
        "5617552695640734234881743021900914949278936532454634272230085647663226514532787884562624380830565697425819"
        "1997547174426864622111171086301064969241295079268308058652961025418601348398449390406655082221832972168262"
        "3670246126981473984487180110516005410318811733808322527326257095499800452573867779119395220469373993418849"
        "3846880662869310666219184294069136641978718617758388486179936023866323587560657381101496525595609837323669"
        "278407246944635112659583193959619447084390120306096605418133549392223358154296875e-196"},
    {0x1.fffffffffffffp-648, chars_format::scientific, 505,
        "1."
        "7123510539128146846976348604380182989855787306490926854446017129532645302906557576912524876166113139485163"
        "8399509434885372924422234217260212993848259015853661611730592205083720269679689878081331016444366594433652"
        "4734049225396294796897436022103201082063762346761664505465251419099960090514773555823879044093874798683769"
        "8769376132573862133243836858813827328395743723551677697235987204773264717512131476220299305119121967464733"
        "855681449388927022531916638791923889416878024061219321083626709878444671630859375e-195"},
    {0x1.fffffffffffffp-647, chars_format::scientific, 504,
        "3."
        "4247021078256293693952697208760365979711574612981853708892034259065290605813115153825049752332226278970327"
        "6799018869770745848844468434520425987696518031707323223461184410167440539359379756162662032888733188867304"
        "9468098450792589593794872044206402164127524693523329010930502838199920181029547111647758088187749597367539"
        "7538752265147724266487673717627654656791487447103355394471974409546529435024262952440598610238243934929467"
        "71136289877785404506383327758384777883375604812243864216725341975688934326171875e-195"},
    {0x1.fffffffffffffp-646, chars_format::scientific, 503,
        "6."
        "8494042156512587387905394417520731959423149225963707417784068518130581211626230307650099504664452557940655"
        "3598037739541491697688936869040851975393036063414646446922368820334881078718759512325324065777466377734609"
        "8936196901585179187589744088412804328255049387046658021861005676399840362059094223295516176375499194735079"
        "5077504530295448532975347435255309313582974894206710788943948819093058870048525904881197220476487869858935"
        "4227257975557080901276665551676955576675120962448772843345068395137786865234375e-195"},
    {0x1.fffffffffffffp-645, chars_format::scientific, 503,
        "1."
        "3698808431302517477581078883504146391884629845192741483556813703626116242325246061530019900932890511588131"
        "0719607547908298339537787373808170395078607212682929289384473764066976215743751902465064813155493275546921"
        "9787239380317035837517948817682560865651009877409331604372201135279968072411818844659103235275099838947015"
        "9015500906059089706595069487051061862716594978841342157788789763818611774009705180976239444095297573971787"
        "0845451595111416180255333110335391115335024192489754568669013679027557373046875e-194"},
    {0x1.fffffffffffffp-644, chars_format::scientific, 502,
        "2."
        "7397616862605034955162157767008292783769259690385482967113627407252232484650492123060039801865781023176262"
        "1439215095816596679075574747616340790157214425365858578768947528133952431487503804930129626310986551093843"
        "9574478760634071675035897635365121731302019754818663208744402270559936144823637689318206470550199677894031"
        "8031001812118179413190138974102123725433189957682684315577579527637223548019410361952478888190595147943574"
        "169090319022283236051066622067078223067004838497950913733802735805511474609375e-194"},
    {0x1.fffffffffffffp-643, chars_format::scientific, 501,
        "5."
        "4795233725210069910324315534016585567538519380770965934227254814504464969300984246120079603731562046352524"
        "2878430191633193358151149495232681580314428850731717157537895056267904862975007609860259252621973102187687"
        "9148957521268143350071795270730243462604039509637326417488804541119872289647275378636412941100399355788063"
        "6062003624236358826380277948204247450866379915365368631155159055274447096038820723904957776381190295887148"
        "33818063804456647210213324413415644613400967699590182746760547161102294921875e-194"},
    {0x1.fffffffffffffp-642, chars_format::scientific, 501,
        "1."
        "0959046745042013982064863106803317113507703876154193186845450962900892993860196849224015920746312409270504"
        "8575686038326638671630229899046536316062885770146343431507579011253580972595001521972051850524394620437537"
        "5829791504253628670014359054146048692520807901927465283497760908223974457929455075727282588220079871157612"
        "7212400724847271765276055589640849490173275983073073726231031811054889419207764144780991555276238059177429"
        "66763612760891329442042664882683128922680193539918036549352109432220458984375e-193"},
    {0x1.fffffffffffffp-641, chars_format::scientific, 500,
        "2."
        "1918093490084027964129726213606634227015407752308386373690901925801785987720393698448031841492624818541009"
        "7151372076653277343260459798093072632125771540292686863015158022507161945190003043944103701048789240875075"
        "1659583008507257340028718108292097385041615803854930566995521816447948915858910151454565176440159742315225"
        "4424801449694543530552111179281698980346551966146147452462063622109778838415528289561983110552476118354859"
        "3352722552178265888408532976536625784536038707983607309870421886444091796875e-193"},
    {0x1.fffffffffffffp-640, chars_format::scientific, 499,
        "4."
        "3836186980168055928259452427213268454030815504616772747381803851603571975440787396896063682985249637082019"
        "4302744153306554686520919596186145264251543080585373726030316045014323890380006087888207402097578481750150"
        "3319166017014514680057436216584194770083231607709861133991043632895897831717820302909130352880319484630450"
        "8849602899389087061104222358563397960693103932292294904924127244219557676831056579123966221104952236709718"
        "670544510435653177681706595307325156907207741596721461974084377288818359375e-193"},
    {0x1.fffffffffffffp-639, chars_format::scientific, 498,
        "8."
        "7672373960336111856518904854426536908061631009233545494763607703207143950881574793792127365970499274164038"
        "8605488306613109373041839192372290528503086161170747452060632090028647780760012175776414804195156963500300"
        "6638332034029029360114872433168389540166463215419722267982087265791795663435640605818260705760638969260901"
        "7699205798778174122208444717126795921386207864584589809848254488439115353662113158247932442209904473419437"
        "34108902087130635536341319061465031381441548319344292394816875457763671875e-193"},
    {0x1.fffffffffffffp-638, chars_format::scientific, 498,
        "1."
        "7534474792067222371303780970885307381612326201846709098952721540641428790176314958758425473194099854832807"
        "7721097661322621874608367838474458105700617232234149490412126418005729556152002435155282960839031392700060"
        "1327666406805805872022974486633677908033292643083944453596417453158359132687128121163652141152127793852180"
        "3539841159755634824441688943425359184277241572916917961969650897687823070732422631649586488441980894683887"
        "46821780417426127107268263812293006276288309663868858478963375091552734375e-192"},
    {0x1.fffffffffffffp-637, chars_format::scientific, 497,
        "3."
        "5068949584134444742607561941770614763224652403693418197905443081282857580352629917516850946388199709665615"
        "5442195322645243749216735676948916211401234464468298980824252836011459112304004870310565921678062785400120"
        "2655332813611611744045948973267355816066585286167888907192834906316718265374256242327304282304255587704360"
        "7079682319511269648883377886850718368554483145833835923939301795375646141464845263299172976883961789367774"
        "9364356083485225421453652762458601255257661932773771695792675018310546875e-192"},
    {0x1.fffffffffffffp-636, chars_format::scientific, 496,
        "7."
        "0137899168268889485215123883541229526449304807386836395810886162565715160705259835033701892776399419331231"
        "0884390645290487498433471353897832422802468928936597961648505672022918224608009740621131843356125570800240"
        "5310665627223223488091897946534711632133170572335777814385669812633436530748512484654608564608511175408721"
        "4159364639022539297766755773701436737108966291667671847878603590751292282929690526598345953767923578735549"
        "872871216697045084290730552491720251051532386554754339158535003662109375e-192"},
    {0x1.fffffffffffffp-635, chars_format::scientific, 496,
        "1."
        "4027579833653777897043024776708245905289860961477367279162177232513143032141051967006740378555279883866246"
        "2176878129058097499686694270779566484560493785787319592329701134404583644921601948124226368671225114160048"
        "1062133125444644697618379589306942326426634114467155562877133962526687306149702496930921712921702235081744"
        "2831872927804507859553351154740287347421793258333534369575720718150258456585938105319669190753584715747109"
        "974574243339409016858146110498344050210306477310950867831707000732421875e-191"},
    {0x1.fffffffffffffp-634, chars_format::scientific, 495,
        "2."
        "8055159667307555794086049553416491810579721922954734558324354465026286064282103934013480757110559767732492"
        "4353756258116194999373388541559132969120987571574639184659402268809167289843203896248452737342450228320096"
        "2124266250889289395236759178613884652853268228934311125754267925053374612299404993861843425843404470163488"
        "5663745855609015719106702309480574694843586516667068739151441436300516913171876210639338381507169431494219"
        "94914848667881803371629222099668810042061295462190173566341400146484375e-191"},
    {0x1.fffffffffffffp-633, chars_format::scientific, 494,
        "5."
        "6110319334615111588172099106832983621159443845909469116648708930052572128564207868026961514221119535464984"
        "8707512516232389998746777083118265938241975143149278369318804537618334579686407792496905474684900456640192"
        "4248532501778578790473518357227769305706536457868622251508535850106749224598809987723686851686808940326977"
        "1327491711218031438213404618961149389687173033334137478302882872601033826343752421278676763014338862988439"
        "8982969733576360674325844419933762008412259092438034713268280029296875e-191"},
    {0x1.fffffffffffffp-632, chars_format::scientific, 494,
        "1."
        "1222063866923022317634419821366596724231888769181893823329741786010514425712841573605392302844223907092996"
        "9741502503246477999749355416623653187648395028629855673863760907523666915937281558499381094936980091328038"
        "4849706500355715758094703671445553861141307291573724450301707170021349844919761997544737370337361788065395"
        "4265498342243606287642680923792229877937434606666827495660576574520206765268750484255735352602867772597687"
        "9796593946715272134865168883986752401682451818487606942653656005859375e-190"},
    {0x1.fffffffffffffp-631, chars_format::scientific, 493,
        "2."
        "2444127733846044635268839642733193448463777538363787646659483572021028851425683147210784605688447814185993"
        "9483005006492955999498710833247306375296790057259711347727521815047333831874563116998762189873960182656076"
        "9699413000711431516189407342891107722282614583147448900603414340042699689839523995089474740674723576130790"
        "8530996684487212575285361847584459755874869213333654991321153149040413530537500968511470705205735545195375"
        "959318789343054426973033776797350480336490363697521388530731201171875e-190"},
    {0x1.fffffffffffffp-630, chars_format::scientific, 492,
        "4."
        "4888255467692089270537679285466386896927555076727575293318967144042057702851366294421569211376895628371987"
        "8966010012985911998997421666494612750593580114519422695455043630094667663749126233997524379747920365312153"
        "9398826001422863032378814685782215444565229166294897801206828680085399379679047990178949481349447152261581"
        "7061993368974425150570723695168919511749738426667309982642306298080827061075001937022941410411471090390751"
        "91863757868610885394606755359470096067298072739504277706146240234375e-190"},
    {0x1.fffffffffffffp-629, chars_format::scientific, 491,
        "8."
        "9776510935384178541075358570932773793855110153455150586637934288084115405702732588843138422753791256743975"
        "7932020025971823997994843332989225501187160229038845390910087260189335327498252467995048759495840730624307"
        "8797652002845726064757629371564430889130458332589795602413657360170798759358095980357898962698894304523163"
        "4123986737948850301141447390337839023499476853334619965284612596161654122150003874045882820822942180781503"
        "8372751573722177078921351071894019213459614547900855541229248046875e-190"},
    {0x1.fffffffffffffp-628, chars_format::scientific, 491,
        "1."
        "7955302187076835708215071714186554758771022030691030117327586857616823081140546517768627684550758251348795"
        "1586404005194364799598968666597845100237432045807769078182017452037867065499650493599009751899168146124861"
        "5759530400569145212951525874312886177826091666517959120482731472034159751871619196071579792539778860904632"
        "6824797347589770060228289478067567804699895370666923993056922519232330824430000774809176564164588436156300"
        "7674550314744435415784270214378803842691922909580171108245849609375e-189"},
    {0x1.fffffffffffffp-627, chars_format::scientific, 490,
        "3."
        "5910604374153671416430143428373109517542044061382060234655173715233646162281093035537255369101516502697590"
        "3172808010388729599197937333195690200474864091615538156364034904075734130999300987198019503798336292249723"
        "1519060801138290425903051748625772355652183333035918240965462944068319503743238392143159585079557721809265"
        "3649594695179540120456578956135135609399790741333847986113845038464661648860001549618353128329176872312601"
        "534910062948887083156854042875760768538384581916034221649169921875e-189"},
    {0x1.fffffffffffffp-626, chars_format::scientific, 489,
        "7."
        "1821208748307342832860286856746219035084088122764120469310347430467292324562186071074510738203033005395180"
        "6345616020777459198395874666391380400949728183231076312728069808151468261998601974396039007596672584499446"
        "3038121602276580851806103497251544711304366666071836481930925888136639007486476784286319170159115443618530"
        "7299189390359080240913157912270271218799581482667695972227690076929323297720003099236706256658353744625203"
        "06982012589777416631370808575152153707676916383206844329833984375e-189"},
    {0x1.fffffffffffffp-625, chars_format::scientific, 489,
        "1."
        "4364241749661468566572057371349243807016817624552824093862069486093458464912437214214902147640606601079036"
        "1269123204155491839679174933278276080189945636646215262545613961630293652399720394879207801519334516899889"
        "2607624320455316170361220699450308942260873333214367296386185177627327801497295356857263834031823088723706"
        "1459837878071816048182631582454054243759916296533539194445538015385864659544000619847341251331670748925040"
        "61396402517955483326274161715030430741535383276641368865966796875e-188"},
    {0x1.fffffffffffffp-624, chars_format::scientific, 488,
        "2."
        "8728483499322937133144114742698487614033635249105648187724138972186916929824874428429804295281213202158072"
        "2538246408310983679358349866556552160379891273292430525091227923260587304799440789758415603038669033799778"
        "5215248640910632340722441398900617884521746666428734592772370355254655602994590713714527668063646177447412"
        "2919675756143632096365263164908108487519832593067078388891076030771729319088001239694682502663341497850081"
        "2279280503591096665254832343006086148307076655328273773193359375e-188"},
    {0x1.fffffffffffffp-623, chars_format::scientific, 487,
        "5."
        "7456966998645874266288229485396975228067270498211296375448277944373833859649748856859608590562426404316144"
        "5076492816621967358716699733113104320759782546584861050182455846521174609598881579516831206077338067599557"
        "0430497281821264681444882797801235769043493332857469185544740710509311205989181427429055336127292354894824"
        "5839351512287264192730526329816216975039665186134156777782152061543458638176002479389365005326682995700162"
        "455856100718219333050966468601217229661415331065654754638671875e-188"},
    {0x1.fffffffffffffp-622, chars_format::scientific, 487,
        "1."
        "1491393399729174853257645897079395045613454099642259275089655588874766771929949771371921718112485280863228"
        "9015298563324393471743339946622620864151956509316972210036491169304234921919776315903366241215467613519911"
        "4086099456364252936288976559560247153808698666571493837108948142101862241197836285485811067225458470978964"
        "9167870302457452838546105265963243395007933037226831355556430412308691727635200495877873001065336599140032"
        "491171220143643866610193293720243445932283066213130950927734375e-187"},
    {0x1.fffffffffffffp-621, chars_format::scientific, 486,
        "2."
        "2982786799458349706515291794158790091226908199284518550179311177749533543859899542743843436224970561726457"
        "8030597126648786943486679893245241728303913018633944420072982338608469843839552631806732482430935227039822"
        "8172198912728505872577953119120494307617397333142987674217896284203724482395672570971622134450916941957929"
        "8335740604914905677092210531926486790015866074453662711112860824617383455270400991755746002130673198280064"
        "98234244028728773322038658744048689186456613242626190185546875e-187"},
    {0x1.fffffffffffffp-620, chars_format::scientific, 485,
        "4."
        "5965573598916699413030583588317580182453816398569037100358622355499067087719799085487686872449941123452915"
        "6061194253297573886973359786490483456607826037267888840145964677216939687679105263613464964861870454079645"
        "6344397825457011745155906238240988615234794666285975348435792568407448964791345141943244268901833883915859"
        "6671481209829811354184421063852973580031732148907325422225721649234766910540801983511492004261346396560129"
        "9646848805745754664407731748809737837291322648525238037109375e-187"},
    {0x1.fffffffffffffp-619, chars_format::scientific, 484,
        "9."
        "1931147197833398826061167176635160364907632797138074200717244710998134175439598170975373744899882246905831"
        "2122388506595147773946719572980966913215652074535777680291929354433879375358210527226929929723740908159291"
        "2688795650914023490311812476481977230469589332571950696871585136814897929582690283886488537803667767831719"
        "3342962419659622708368842127705947160063464297814650844451443298469533821081603967022984008522692793120259"
        "929369761149150932881546349761947567458264529705047607421875e-187"},
    {0x1.fffffffffffffp-618, chars_format::scientific, 484,
        "1."
        "8386229439566679765212233435327032072981526559427614840143448942199626835087919634195074748979976449381166"
        "2424477701319029554789343914596193382643130414907155536058385870886775875071642105445385985944748181631858"
        "2537759130182804698062362495296395446093917866514390139374317027362979585916538056777297707560733553566343"
        "8668592483931924541673768425541189432012692859562930168890288659693906764216320793404596801704538558624051"
        "985873952229830186576309269952389513491652905941009521484375e-186"},
    {0x1.fffffffffffffp-617, chars_format::scientific, 483,
        "3."
        "6772458879133359530424466870654064145963053118855229680286897884399253670175839268390149497959952898762332"
        "4848955402638059109578687829192386765286260829814311072116771741773551750143284210890771971889496363263716"
        "5075518260365609396124724990592790892187835733028780278748634054725959171833076113554595415121467107132687"
        "7337184967863849083347536851082378864025385719125860337780577319387813528432641586809193603409077117248103"
        "97174790445966037315261853990477902698330581188201904296875e-186"},
    {0x1.fffffffffffffp-616, chars_format::scientific, 482,
        "7."
        "3544917758266719060848933741308128291926106237710459360573795768798507340351678536780298995919905797524664"
        "9697910805276118219157375658384773530572521659628622144233543483547103500286568421781543943778992726527433"
        "0151036520731218792249449981185581784375671466057560557497268109451918343666152227109190830242934214265375"
        "4674369935727698166695073702164757728050771438251720675561154638775627056865283173618387206818154234496207"
        "9434958089193207463052370798095580539666116237640380859375e-186"},
    {0x1.fffffffffffffp-615, chars_format::scientific, 482,
        "1."
        "4708983551653343812169786748261625658385221247542091872114759153759701468070335707356059799183981159504932"
        "9939582161055223643831475131676954706114504331925724428846708696709420700057313684356308788755798545305486"
        "6030207304146243758449889996237116356875134293211512111499453621890383668733230445421838166048586842853075"
        "0934873987145539633339014740432951545610154287650344135112230927755125411373056634723677441363630846899241"
        "5886991617838641492610474159619116107933223247528076171875e-185"},
    {0x1.fffffffffffffp-614, chars_format::scientific, 481,
        "2."
        "9417967103306687624339573496523251316770442495084183744229518307519402936140671414712119598367962319009865"
        "9879164322110447287662950263353909412229008663851448857693417393418841400114627368712617577511597090610973"
        "2060414608292487516899779992474232713750268586423024222998907243780767337466460890843676332097173685706150"
        "1869747974291079266678029480865903091220308575300688270224461855510250822746113269447354882727261693798483"
        "177398323567728298522094831923823221586644649505615234375e-185"},
    {0x1.fffffffffffffp-613, chars_format::scientific, 480,
        "5."
        "8835934206613375248679146993046502633540884990168367488459036615038805872281342829424239196735924638019731"
        "9758328644220894575325900526707818824458017327702897715386834786837682800229254737425235155023194181221946"
        "4120829216584975033799559984948465427500537172846048445997814487561534674932921781687352664194347371412300"
        "3739495948582158533356058961731806182440617150601376540448923711020501645492226538894709765454523387596966"
        "35479664713545659704418966384764644317328929901123046875e-185"},
    {0x1.fffffffffffffp-612, chars_format::scientific, 480,
        "1."
        "1767186841322675049735829398609300526708176998033673497691807323007761174456268565884847839347184927603946"
        "3951665728844178915065180105341563764891603465540579543077366957367536560045850947485047031004638836244389"
        "2824165843316995006759911996989693085500107434569209689199562897512306934986584356337470532838869474282460"
        "0747899189716431706671211792346361236488123430120275308089784742204100329098445307778941953090904677519393"
        "27095932942709131940883793276952928863465785980224609375e-184"},
    {0x1.fffffffffffffp-611, chars_format::scientific, 479,
        "2."
        "3534373682645350099471658797218601053416353996067346995383614646015522348912537131769695678694369855207892"
        "7903331457688357830130360210683127529783206931081159086154733914735073120091701894970094062009277672488778"
        "5648331686633990013519823993979386171000214869138419378399125795024613869973168712674941065677738948564920"
        "1495798379432863413342423584692722472976246860240550616179569484408200658196890615557883906181809355038786"
        "5419186588541826388176758655390585772693157196044921875e-184"},
    {0x1.fffffffffffffp-610, chars_format::scientific, 478,
        "4."
        "7068747365290700198943317594437202106832707992134693990767229292031044697825074263539391357388739710415785"
        "5806662915376715660260720421366255059566413862162318172309467829470146240183403789940188124018555344977557"
        "1296663373267980027039647987958772342000429738276838756798251590049227739946337425349882131355477897129840"
        "2991596758865726826684847169385444945952493720481101232359138968816401316393781231115767812363618710077573"
        "083837317708365277635351731078117154538631439208984375e-184"},
    {0x1.fffffffffffffp-609, chars_format::scientific, 477,
        "9."
        "4137494730581400397886635188874404213665415984269387981534458584062089395650148527078782714777479420831571"
        "1613325830753431320521440842732510119132827724324636344618935658940292480366807579880376248037110689955114"
        "2593326746535960054079295975917544684000859476553677513596503180098455479892674850699764262710955794259680"
        "5983193517731453653369694338770889891904987440962202464718277937632802632787562462231535624727237420155146"
        "16767463541673055527070346215623430907726287841796875e-184"},
    {0x1.fffffffffffffp-608, chars_format::scientific, 477,
        "1."
        "8827498946116280079577327037774880842733083196853877596306891716812417879130029705415756542955495884166314"
        "2322665166150686264104288168546502023826565544864927268923787131788058496073361515976075249607422137991022"
        "8518665349307192010815859195183508936800171895310735502719300636019691095978534970139952852542191158851936"
        "1196638703546290730673938867754177978380997488192440492943655587526560526557512492446307124945447484031029"
        "23353492708334611105414069243124686181545257568359375e-183"},
    {0x1.fffffffffffffp-607, chars_format::scientific, 476,
        "3."
        "7654997892232560159154654075549761685466166393707755192613783433624835758260059410831513085910991768332628"
        "4645330332301372528208576337093004047653131089729854537847574263576116992146723031952150499214844275982045"
        "7037330698614384021631718390367017873600343790621471005438601272039382191957069940279905705084382317703872"
        "2393277407092581461347877735508355956761994976384880985887311175053121053115024984892614249890894968062058"
        "4670698541666922221082813848624937236309051513671875e-183"},
    {0x1.fffffffffffffp-606, chars_format::scientific, 475,
        "7."
        "5309995784465120318309308151099523370932332787415510385227566867249671516520118821663026171821983536665256"
        "9290660664602745056417152674186008095306262179459709075695148527152233984293446063904300998429688551964091"
        "4074661397228768043263436780734035747200687581242942010877202544078764383914139880559811410168764635407744"
        "4786554814185162922695755471016711913523989952769761971774622350106242106230049969785228499781789936124116"
        "934139708333384444216562769724987447261810302734375e-183"},
    {0x1.fffffffffffffp-605, chars_format::scientific, 475,
        "1."
        "5061999156893024063661861630219904674186466557483102077045513373449934303304023764332605234364396707333051"
        "3858132132920549011283430534837201619061252435891941815139029705430446796858689212780860199685937710392818"
        "2814932279445753608652687356146807149440137516248588402175440508815752876782827976111962282033752927081548"
        "8957310962837032584539151094203342382704797990553952394354924470021248421246009993957045699956357987224823"
        "386827941666676888843312553944997489452362060546875e-182"},
    {0x1.fffffffffffffp-604, chars_format::scientific, 474,
        "3."
        "0123998313786048127323723260439809348372933114966204154091026746899868606608047528665210468728793414666102"
        "7716264265841098022566861069674403238122504871783883630278059410860893593717378425561720399371875420785636"
        "5629864558891507217305374712293614298880275032497176804350881017631505753565655952223924564067505854163097"
        "7914621925674065169078302188406684765409595981107904788709848940042496842492019987914091399912715974449646"
        "77365588333335377768662510788999497890472412109375e-182"},
    {0x1.fffffffffffffp-603, chars_format::scientific, 473,
        "6."
        "0247996627572096254647446520879618696745866229932408308182053493799737213216095057330420937457586829332205"
        "5432528531682196045133722139348806476245009743567767260556118821721787187434756851123440798743750841571273"
        "1259729117783014434610749424587228597760550064994353608701762035263011507131311904447849128135011708326195"
        "5829243851348130338156604376813369530819191962215809577419697880084993684984039975828182799825431948899293"
        "5473117666667075553732502157799899578094482421875e-182"},
    {0x1.fffffffffffffp-602, chars_format::scientific, 473,
        "1."
        "2049599325514419250929489304175923739349173245986481661636410698759947442643219011466084187491517365866441"
        "1086505706336439209026744427869761295249001948713553452111223764344357437486951370224688159748750168314254"
        "6251945823556602886922149884917445719552110012998870721740352407052602301426262380889569825627002341665239"
        "1165848770269626067631320875362673906163838392443161915483939576016998736996807995165636559965086389779858"
        "7094623533333415110746500431559979915618896484375e-181"},
    {0x1.fffffffffffffp-601, chars_format::scientific, 472,
        "2."
        "4099198651028838501858978608351847478698346491972963323272821397519894885286438022932168374983034731732882"
        "2173011412672878418053488855739522590498003897427106904222447528688714874973902740449376319497500336628509"
        "2503891647113205773844299769834891439104220025997741443480704814105204602852524761779139651254004683330478"
        "2331697540539252135262641750725347812327676784886323830967879152033997473993615990331273119930172779559717"
        "418924706666683022149300086311995983123779296875e-181"},
    {0x1.fffffffffffffp-600, chars_format::scientific, 471,
        "4."
        "8198397302057677003717957216703694957396692983945926646545642795039789770572876045864336749966069463465764"
        "4346022825345756836106977711479045180996007794854213808444895057377429749947805480898752638995000673257018"
        "5007783294226411547688599539669782878208440051995482886961409628210409205705049523558279302508009366660956"
        "4663395081078504270525283501450695624655353569772647661935758304067994947987231980662546239860345559119434"
        "83784941333336604429860017262399196624755859375e-181"},
    {0x1.fffffffffffffp-599, chars_format::scientific, 470,
        "9."
        "6396794604115354007435914433407389914793385967891853293091285590079579541145752091728673499932138926931528"
        "8692045650691513672213955422958090361992015589708427616889790114754859499895610961797505277990001346514037"
        "0015566588452823095377199079339565756416880103990965773922819256420818411410099047116558605016018733321912"
        "9326790162157008541050567002901391249310707139545295323871516608135989895974463961325092479720691118238869"
        "6756988266667320885972003452479839324951171875e-181"},
    {0x1.fffffffffffffp-598, chars_format::scientific, 470,
        "1."
        "9279358920823070801487182886681477982958677193578370658618257118015915908229150418345734699986427785386305"
        "7738409130138302734442791084591618072398403117941685523377958022950971899979122192359501055598000269302807"
        "4003113317690564619075439815867913151283376020798193154784563851284163682282019809423311721003203746664382"
        "5865358032431401708210113400580278249862141427909059064774303321627197979194892792265018495944138223647773"
        "9351397653333464177194400690495967864990234375e-180"},
    {0x1.fffffffffffffp-597, chars_format::scientific, 469,
        "3."
        "8558717841646141602974365773362955965917354387156741317236514236031831816458300836691469399972855570772611"
        "5476818260276605468885582169183236144796806235883371046755916045901943799958244384719002111196000538605614"
        "8006226635381129238150879631735826302566752041596386309569127702568327364564039618846623442006407493328765"
        "1730716064862803416420226801160556499724282855818118129548606643254395958389785584530036991888276447295547"
        "870279530666692835438880138099193572998046875e-180"},
    {0x1.fffffffffffffp-596, chars_format::scientific, 468,
        "7."
        "7117435683292283205948731546725911931834708774313482634473028472063663632916601673382938799945711141545223"
        "0953636520553210937771164338366472289593612471766742093511832091803887599916488769438004222392001077211229"
        "6012453270762258476301759263471652605133504083192772619138255405136654729128079237693246884012814986657530"
        "3461432129725606832840453602321112999448565711636236259097213286508791916779571169060073983776552894591095"
        "74055906133338567087776027619838714599609375e-180"},
    {0x1.fffffffffffffp-595, chars_format::scientific, 468,
        "1."
        "5423487136658456641189746309345182386366941754862696526894605694412732726583320334676587759989142228309044"
        "6190727304110642187554232867673294457918722494353348418702366418360777519983297753887600844478400215442245"
        "9202490654152451695260351852694330521026700816638554523827651081027330945825615847538649376802562997331506"
        "0692286425945121366568090720464222599889713142327247251819442657301758383355914233812014796755310578918219"
        "14811181226667713417555205523967742919921875e-179"},
    {0x1.fffffffffffffp-594, chars_format::scientific, 467,
        "3."
        "0846974273316913282379492618690364772733883509725393053789211388825465453166640669353175519978284456618089"
        "2381454608221284375108465735346588915837444988706696837404732836721555039966595507775201688956800430884491"
        "8404981308304903390520703705388661042053401633277109047655302162054661891651231695077298753605125994663012"
        "1384572851890242733136181440928445199779426284654494503638885314603516766711828467624029593510621157836438"
        "2962236245333542683511041104793548583984375e-179"},
    {0x1.fffffffffffffp-593, chars_format::scientific, 466,
        "6."
        "1693948546633826564758985237380729545467767019450786107578422777650930906333281338706351039956568913236178"
        "4762909216442568750216931470693177831674889977413393674809465673443110079933191015550403377913600861768983"
        "6809962616609806781041407410777322084106803266554218095310604324109323783302463390154597507210251989326024"
        "2769145703780485466272362881856890399558852569308989007277770629207033533423656935248059187021242315672876"
        "592447249066708536702208220958709716796875e-179"},
    {0x1.fffffffffffffp-592, chars_format::scientific, 466,
        "1."
        "2338789709326765312951797047476145909093553403890157221515684555530186181266656267741270207991313782647235"
        "6952581843288513750043386294138635566334977995482678734961893134688622015986638203110080675582720172353796"
        "7361992523321961356208281482155464416821360653310843619062120864821864756660492678030919501442050397865204"
        "8553829140756097093254472576371378079911770513861797801455554125841406706684731387049611837404248463134575"
        "318489449813341707340441644191741943359375e-178"},
    {0x1.fffffffffffffp-591, chars_format::scientific, 465,
        "2."
        "4677579418653530625903594094952291818187106807780314443031369111060372362533312535482540415982627565294471"
        "3905163686577027500086772588277271132669955990965357469923786269377244031973276406220161351165440344707593"
        "4723985046643922712416562964310928833642721306621687238124241729643729513320985356061839002884100795730409"
        "7107658281512194186508945152742756159823541027723595602911108251682813413369462774099223674808496926269150"
        "63697889962668341468088328838348388671875e-178"},
    {0x1.fffffffffffffp-590, chars_format::scientific, 464,
        "4."
        "9355158837307061251807188189904583636374213615560628886062738222120744725066625070965080831965255130588942"
        "7810327373154055000173545176554542265339911981930714939847572538754488063946552812440322702330880689415186"
        "9447970093287845424833125928621857667285442613243374476248483459287459026641970712123678005768201591460819"
        "4215316563024388373017890305485512319647082055447191205822216503365626826738925548198447349616993852538301"
        "2739577992533668293617665767669677734375e-178"},
    {0x1.fffffffffffffp-589, chars_format::scientific, 463,
        "9."
        "8710317674614122503614376379809167272748427231121257772125476444241489450133250141930161663930510261177885"
        "5620654746308110000347090353109084530679823963861429879695145077508976127893105624880645404661761378830373"
        "8895940186575690849666251857243715334570885226486748952496966918574918053283941424247356011536403182921638"
        "8430633126048776746035780610971024639294164110894382411644433006731253653477851096396894699233987705076602"
        "547915598506733658723533153533935546875e-178"},
    {0x1.fffffffffffffp-588, chars_format::scientific, 463,
        "1."
        "9742063534922824500722875275961833454549685446224251554425095288848297890026650028386032332786102052235577"
        "1124130949261622000069418070621816906135964792772285975939029015501795225578621124976129080932352275766074"
        "7779188037315138169933250371448743066914177045297349790499393383714983610656788284849471202307280636584327"
        "7686126625209755349207156122194204927858832822178876482328886601346250730695570219279378939846797541015320"
        "509583119701346731744706630706787109375e-177"},
    {0x1.fffffffffffffp-587, chars_format::scientific, 462,
        "3."
        "9484127069845649001445750551923666909099370892448503108850190577696595780053300056772064665572204104471154"
        "2248261898523244000138836141243633812271929585544571951878058031003590451157242249952258161864704551532149"
        "5558376074630276339866500742897486133828354090594699580998786767429967221313576569698942404614561273168655"
        "5372253250419510698414312244388409855717665644357752964657773202692501461391140438558757879693595082030641"
        "01916623940269346348941326141357421875e-177"},
    {0x1.fffffffffffffp-586, chars_format::scientific, 461,
        "7."
        "8968254139691298002891501103847333818198741784897006217700381155393191560106600113544129331144408208942308"
        "4496523797046488000277672282487267624543859171089143903756116062007180902314484499904516323729409103064299"
        "1116752149260552679733001485794972267656708181189399161997573534859934442627153139397884809229122546337311"
        "0744506500839021396828624488776819711435331288715505929315546405385002922782280877117515759387190164061282"
        "0383324788053869269788265228271484375e-177"},
    {0x1.fffffffffffffp-585, chars_format::scientific, 461,
        "1."
        "5793650827938259600578300220769466763639748356979401243540076231078638312021320022708825866228881641788461"
        "6899304759409297600055534456497453524908771834217828780751223212401436180462896899980903264745881820612859"
        "8223350429852110535946600297158994453531341636237879832399514706971986888525430627879576961845824509267462"
        "2148901300167804279365724897755363942287066257743101185863109281077000584556456175423503151877438032812256"
        "4076664957610773853957653045654296875e-176"},
    {0x1.fffffffffffffp-584, chars_format::scientific, 460,
        "3."
        "1587301655876519201156600441538933527279496713958802487080152462157276624042640045417651732457763283576923"
        "3798609518818595200111068912994907049817543668435657561502446424802872360925793799961806529491763641225719"
        "6446700859704221071893200594317988907062683272475759664799029413943973777050861255759153923691649018534924"
        "4297802600335608558731449795510727884574132515486202371726218562154001169112912350847006303754876065624512"
        "815332991522154770791530609130859375e-176"},
    {0x1.fffffffffffffp-583, chars_format::scientific, 459,
        "6."
        "3174603311753038402313200883077867054558993427917604974160304924314553248085280090835303464915526567153846"
        "7597219037637190400222137825989814099635087336871315123004892849605744721851587599923613058983527282451439"
        "2893401719408442143786401188635977814125366544951519329598058827887947554101722511518307847383298037069848"
        "8595605200671217117462899591021455769148265030972404743452437124308002338225824701694012607509752131249025"
        "63066598304430954158306121826171875e-176"},
    {0x1.fffffffffffffp-582, chars_format::scientific, 459,
        "1."
        "2634920662350607680462640176615573410911798685583520994832060984862910649617056018167060692983105313430769"
        "3519443807527438080044427565197962819927017467374263024600978569921148944370317519984722611796705456490287"
        "8578680343881688428757280237727195562825073308990303865919611765577589510820344502303661569476659607413969"
        "7719121040134243423492579918204291153829653006194480948690487424861600467645164940338802521501950426249805"
        "12613319660886190831661224365234375e-175"},
    {0x1.fffffffffffffp-581, chars_format::scientific, 458,
        "2."
        "5269841324701215360925280353231146821823597371167041989664121969725821299234112036334121385966210626861538"
        "7038887615054876160088855130395925639854034934748526049201957139842297888740635039969445223593410912980575"
        "7157360687763376857514560475454391125650146617980607731839223531155179021640689004607323138953319214827939"
        "5438242080268486846985159836408582307659306012388961897380974849723200935290329880677605043003900852499610"
        "2522663932177238166332244873046875e-175"},
    {0x1.fffffffffffffp-580, chars_format::scientific, 457,
        "5."
        "0539682649402430721850560706462293643647194742334083979328243939451642598468224072668242771932421253723077"
        "4077775230109752320177710260791851279708069869497052098403914279684595777481270079938890447186821825961151"
        "4314721375526753715029120950908782251300293235961215463678447062310358043281378009214646277906638429655879"
        "0876484160536973693970319672817164615318612024777923794761949699446401870580659761355210086007801704999220"
        "504532786435447633266448974609375e-175"},
    {0x1.fffffffffffffp-579, chars_format::scientific, 457,
        "1."
        "0107936529880486144370112141292458728729438948466816795865648787890328519693644814533648554386484250744615"
        "4815555046021950464035542052158370255941613973899410419680782855936919155496254015987778089437364365192230"
        "2862944275105350743005824190181756450260058647192243092735689412462071608656275601842929255581327685931175"
        "8175296832107394738794063934563432923063722404955584758952389939889280374116131952271042017201560340999844"
        "100906557287089526653289794921875e-174"},
    {0x1.fffffffffffffp-578, chars_format::scientific, 456,
        "2."
        "0215873059760972288740224282584917457458877896933633591731297575780657039387289629067297108772968501489230"
        "9631110092043900928071084104316740511883227947798820839361565711873838310992508031975556178874728730384460"
        "5725888550210701486011648380363512900520117294384486185471378824924143217312551203685858511162655371862351"
        "6350593664214789477588127869126865846127444809911169517904779879778560748232263904542084034403120681999688"
        "20181311457417905330657958984375e-174"},
    {0x1.fffffffffffffp-577, chars_format::scientific, 455,
        "4."
        "0431746119521944577480448565169834914917755793867267183462595151561314078774579258134594217545937002978461"
        "9262220184087801856142168208633481023766455895597641678723131423747676621985016063951112357749457460768921"
        "1451777100421402972023296760727025801040234588768972370942757649848286434625102407371717022325310743724703"
        "2701187328429578955176255738253731692254889619822339035809559759557121496464527809084168068806241363999376"
        "4036262291483581066131591796875e-174"},
    {0x1.fffffffffffffp-576, chars_format::scientific, 454,
        "8."
        "0863492239043889154960897130339669829835511587734534366925190303122628157549158516269188435091874005956923"
        "8524440368175603712284336417266962047532911791195283357446262847495353243970032127902224715498914921537842"
        "2903554200842805944046593521454051602080469177537944741885515299696572869250204814743434044650621487449406"
        "5402374656859157910352511476507463384509779239644678071619119519114242992929055618168336137612482727998752"
        "807252458296716213226318359375e-174"},
    {0x1.fffffffffffffp-575, chars_format::scientific, 454,
        "1."
        "6172698447808777830992179426067933965967102317546906873385038060624525631509831703253837687018374801191384"
        "7704888073635120742456867283453392409506582358239056671489252569499070648794006425580444943099782984307568"
        "4580710840168561188809318704290810320416093835507588948377103059939314573850040962948686808930124297489881"
        "3080474931371831582070502295301492676901955847928935614323823903822848598585811123633667227522496545599750"
        "561450491659343242645263671875e-173"},
    {0x1.fffffffffffffp-574, chars_format::scientific, 453,
        "3."
        "2345396895617555661984358852135867931934204635093813746770076121249051263019663406507675374036749602382769"
        "5409776147270241484913734566906784819013164716478113342978505138998141297588012851160889886199565968615136"
        "9161421680337122377618637408581620640832187671015177896754206119878629147700081925897373617860248594979762"
        "6160949862743663164141004590602985353803911695857871228647647807645697197171622247267334455044993091199501"
        "12290098331868648529052734375e-173"},
    {0x1.fffffffffffffp-573, chars_format::scientific, 452,
        "6."
        "4690793791235111323968717704271735863868409270187627493540152242498102526039326813015350748073499204765539"
        "0819552294540482969827469133813569638026329432956226685957010277996282595176025702321779772399131937230273"
        "8322843360674244755237274817163241281664375342030355793508412239757258295400163851794747235720497189959525"
        "2321899725487326328282009181205970707607823391715742457295295615291394394343244494534668910089986182399002"
        "2458019666373729705810546875e-173"},
    {0x1.fffffffffffffp-572, chars_format::scientific, 452,
        "1."
        "2938158758247022264793743540854347172773681854037525498708030448499620505207865362603070149614699840953107"
        "8163910458908096593965493826762713927605265886591245337191402055599256519035205140464355954479826387446054"
        "7664568672134848951047454963432648256332875068406071158701682447951451659080032770358949447144099437991905"
        "0464379945097465265656401836241194141521564678343148491459059123058278878868648898906933782017997236479800"
        "4491603933274745941162109375e-172"},
    {0x1.fffffffffffffp-571, chars_format::scientific, 451,
        "2."
        "5876317516494044529587487081708694345547363708075050997416060896999241010415730725206140299229399681906215"
        "6327820917816193187930987653525427855210531773182490674382804111198513038070410280928711908959652774892109"
        "5329137344269697902094909926865296512665750136812142317403364895902903318160065540717898894288198875983810"
        "0928759890194930531312803672482388283043129356686296982918118246116557757737297797813867564035994472959600"
        "898320786654949188232421875e-172"},
    {0x1.fffffffffffffp-570, chars_format::scientific, 450,
        "5."
        "1752635032988089059174974163417388691094727416150101994832121793998482020831461450412280598458799363812431"
        "2655641835632386375861975307050855710421063546364981348765608222397026076140820561857423817919305549784219"
        "0658274688539395804189819853730593025331500273624284634806729791805806636320131081435797788576397751967620"
        "1857519780389861062625607344964776566086258713372593965836236492233115515474595595627735128071988945919201"
        "79664157330989837646484375e-172"},
    {0x1.fffffffffffffp-569, chars_format::scientific, 450,
        "1."
        "0350527006597617811834994832683477738218945483230020398966424358799696404166292290082456119691759872762486"
        "2531128367126477275172395061410171142084212709272996269753121644479405215228164112371484763583861109956843"
        "8131654937707879160837963970746118605066300054724856926961345958361161327264026216287159557715279550393524"
        "0371503956077972212525121468992955313217251742674518793167247298446623103094919119125547025614397789183840"
        "35932831466197967529296875e-171"},
    {0x1.fffffffffffffp-568, chars_format::scientific, 449,
        "2."
        "0701054013195235623669989665366955476437890966460040797932848717599392808332584580164912239383519745524972"
        "5062256734252954550344790122820342284168425418545992539506243288958810430456328224742969527167722219913687"
        "6263309875415758321675927941492237210132600109449713853922691916722322654528052432574319115430559100787048"
        "0743007912155944425050242937985910626434503485349037586334494596893246206189838238251094051228795578367680"
        "7186566293239593505859375e-171"},
    {0x1.fffffffffffffp-567, chars_format::scientific, 448,
        "4."
        "1402108026390471247339979330733910952875781932920081595865697435198785616665169160329824478767039491049945"
        "0124513468505909100689580245640684568336850837091985079012486577917620860912656449485939054335444439827375"
        "2526619750831516643351855882984474420265200218899427707845383833444645309056104865148638230861118201574096"
        "1486015824311888850100485875971821252869006970698075172668989193786492412379676476502188102457591156735361"
        "437313258647918701171875e-171"},
    {0x1.fffffffffffffp-566, chars_format::scientific, 447,
        "8."
        "2804216052780942494679958661467821905751563865840163191731394870397571233330338320659648957534078982099890"
        "0249026937011818201379160491281369136673701674183970158024973155835241721825312898971878108670888879654750"
        "5053239501663033286703711765968948840530400437798855415690767666889290618112209730297276461722236403148192"
        "2972031648623777700200971751943642505738013941396150345337978387572984824759352953004376204915182313470722"
        "87462651729583740234375e-171"},
    {0x1.fffffffffffffp-565, chars_format::scientific, 447,
        "1."
        "6560843210556188498935991732293564381150312773168032638346278974079514246666067664131929791506815796419978"
        "0049805387402363640275832098256273827334740334836794031604994631167048344365062579794375621734177775930950"
        "1010647900332606657340742353193789768106080087559771083138153533377858123622441946059455292344447280629638"
        "4594406329724755540040194350388728501147602788279230069067595677514596964951870590600875240983036462694144"
        "57492530345916748046875e-170"},
    {0x1.fffffffffffffp-564, chars_format::scientific, 446,
        "3."
        "3121686421112376997871983464587128762300625546336065276692557948159028493332135328263859583013631592839956"
        "0099610774804727280551664196512547654669480669673588063209989262334096688730125159588751243468355551861900"
        "2021295800665213314681484706387579536212160175119542166276307066755716247244883892118910584688894561259276"
        "9188812659449511080080388700777457002295205576558460138135191355029193929903741181201750481966072925388289"
        "1498506069183349609375e-170"},
    {0x1.fffffffffffffp-563, chars_format::scientific, 445,
        "6."
        "6243372842224753995743966929174257524601251092672130553385115896318056986664270656527719166027263185679912"
        "0199221549609454561103328393025095309338961339347176126419978524668193377460250319177502486936711103723800"
        "4042591601330426629362969412775159072424320350239084332552614133511432494489767784237821169377789122518553"
        "8377625318899022160160777401554914004590411153116920276270382710058387859807482362403500963932145850776578"
        "299701213836669921875e-170"},
    {0x1.fffffffffffffp-562, chars_format::scientific, 445,
        "1."
        "3248674568444950799148793385834851504920250218534426110677023179263611397332854131305543833205452637135982"
        "4039844309921890912220665678605019061867792267869435225283995704933638675492050063835500497387342220744760"
        "0808518320266085325872593882555031814484864070047816866510522826702286498897953556847564233875557824503710"
        "7675525063779804432032155480310982800918082230623384055254076542011677571961496472480700192786429170155315"
        "659940242767333984375e-169"},
    {0x1.fffffffffffffp-561, chars_format::scientific, 444,
        "2."
        "6497349136889901598297586771669703009840500437068852221354046358527222794665708262611087666410905274271964"
        "8079688619843781824441331357210038123735584535738870450567991409867277350984100127671000994774684441489520"
        "1617036640532170651745187765110063628969728140095633733021045653404572997795907113695128467751115649007421"
        "5351050127559608864064310960621965601836164461246768110508153084023355143922992944961400385572858340310631"
        "31988048553466796875e-169"},
    {0x1.fffffffffffffp-560, chars_format::scientific, 443,
        "5."
        "2994698273779803196595173543339406019681000874137704442708092717054445589331416525222175332821810548543929"
        "6159377239687563648882662714420076247471169071477740901135982819734554701968200255342001989549368882979040"
        "3234073281064341303490375530220127257939456280191267466042091306809145995591814227390256935502231298014843"
        "0702100255119217728128621921243931203672328922493536221016306168046710287845985889922800771145716680621262"
        "6397609710693359375e-169"},
    {0x1.fffffffffffffp-559, chars_format::scientific, 443,
        "1."
        "0598939654755960639319034708667881203936200174827540888541618543410889117866283305044435066564362109708785"
        "9231875447937512729776532542884015249494233814295548180227196563946910940393640051068400397909873776595808"
        "0646814656212868260698075106044025451587891256038253493208418261361829199118362845478051387100446259602968"
        "6140420051023843545625724384248786240734465784498707244203261233609342057569197177984560154229143336124252"
        "5279521942138671875e-168"},
    {0x1.fffffffffffffp-558, chars_format::scientific, 442,
        "2."
        "1197879309511921278638069417335762407872400349655081777083237086821778235732566610088870133128724219417571"
        "8463750895875025459553065085768030498988467628591096360454393127893821880787280102136800795819747553191616"
        "1293629312425736521396150212088050903175782512076506986416836522723658398236725690956102774200892519205937"
        "2280840102047687091251448768497572481468931568997414488406522467218684115138394355969120308458286672248505"
        "055904388427734375e-168"},
    {0x1.fffffffffffffp-557, chars_format::scientific, 441,
        "4."
        "2395758619023842557276138834671524815744800699310163554166474173643556471465133220177740266257448438835143"
        "6927501791750050919106130171536060997976935257182192720908786255787643761574560204273601591639495106383232"
        "2587258624851473042792300424176101806351565024153013972833673045447316796473451381912205548401785038411874"
        "4561680204095374182502897536995144962937863137994828976813044934437368230276788711938240616916573344497010"
        "11180877685546875e-168"},
    {0x1.fffffffffffffp-556, chars_format::scientific, 440,
        "8."
        "4791517238047685114552277669343049631489601398620327108332948347287112942930266440355480532514896877670287"
        "3855003583500101838212260343072121995953870514364385441817572511575287523149120408547203183278990212766464"
        "5174517249702946085584600848352203612703130048306027945667346090894633592946902763824411096803570076823748"
        "9123360408190748365005795073990289925875726275989657953626089868874736460553577423876481233833146688994020"
        "2236175537109375e-168"},
    {0x1.fffffffffffffp-555, chars_format::scientific, 440,
        "1."
        "6958303447609537022910455533868609926297920279724065421666589669457422588586053288071096106502979375534057"
        "4771000716700020367642452068614424399190774102872877088363514502315057504629824081709440636655798042553292"
        "9034903449940589217116920169670440722540626009661205589133469218178926718589380552764882219360714015364749"
        "7824672081638149673001159014798057985175145255197931590725217973774947292110715484775296246766629337798804"
        "0447235107421875e-167"},
    {0x1.fffffffffffffp-554, chars_format::scientific, 439,
        "3."
        "3916606895219074045820911067737219852595840559448130843333179338914845177172106576142192213005958751068114"
        "9542001433400040735284904137228848798381548205745754176727029004630115009259648163418881273311596085106585"
        "8069806899881178434233840339340881445081252019322411178266938436357853437178761105529764438721428030729499"
        "5649344163276299346002318029596115970350290510395863181450435947549894584221430969550592493533258675597608"
        "089447021484375e-167"},
    {0x1.fffffffffffffp-553, chars_format::scientific, 438,
        "6."
        "7833213790438148091641822135474439705191681118896261686666358677829690354344213152284384426011917502136229"
        "9084002866800081470569808274457697596763096411491508353454058009260230018519296326837762546623192170213171"
        "6139613799762356868467680678681762890162504038644822356533876872715706874357522211059528877442856061458999"
        "1298688326552598692004636059192231940700581020791726362900871895099789168442861939101184987066517351195216"
        "17889404296875e-167"},
    {0x1.fffffffffffffp-552, chars_format::scientific, 438,
        "1."
        "3566642758087629618328364427094887941038336223779252337333271735565938070868842630456876885202383500427245"
        "9816800573360016294113961654891539519352619282298301670690811601852046003703859265367552509324638434042634"
        "3227922759952471373693536135736352578032500807728964471306775374543141374871504442211905775488571212291799"
        "8259737665310519738400927211838446388140116204158345272580174379019957833688572387820236997413303470239043"
        "23577880859375e-166"},
    {0x1.fffffffffffffp-551, chars_format::scientific, 437,
        "2."
        "7133285516175259236656728854189775882076672447558504674666543471131876141737685260913753770404767000854491"
        "9633601146720032588227923309783079038705238564596603341381623203704092007407718530735105018649276868085268"
        "6455845519904942747387072271472705156065001615457928942613550749086282749743008884423811550977142424583599"
        "6519475330621039476801854423676892776280232408316690545160348758039915667377144775640473994826606940478086"
        "4715576171875e-166"},
    {0x1.fffffffffffffp-550, chars_format::scientific, 436,
        "5."
        "4266571032350518473313457708379551764153344895117009349333086942263752283475370521827507540809534001708983"
        "9267202293440065176455846619566158077410477129193206682763246407408184014815437061470210037298553736170537"
        "2911691039809885494774144542945410312130003230915857885227101498172565499486017768847623101954284849167199"
        "3038950661242078953603708847353785552560464816633381090320697516079831334754289551280947989653213880956172"
        "943115234375e-166"},
    {0x1.fffffffffffffp-549, chars_format::scientific, 436,
        "1."
        "0853314206470103694662691541675910352830668979023401869866617388452750456695074104365501508161906800341796"
        "7853440458688013035291169323913231615482095425838641336552649281481636802963087412294042007459710747234107"
        "4582338207961977098954828908589082062426000646183171577045420299634513099897203553769524620390856969833439"
        "8607790132248415790720741769470757110512092963326676218064139503215966266950857910256189597930642776191234"
        "588623046875e-165"},
    {0x1.fffffffffffffp-548, chars_format::scientific, 435,
        "2."
        "1706628412940207389325383083351820705661337958046803739733234776905500913390148208731003016323813600683593"
        "5706880917376026070582338647826463230964190851677282673105298562963273605926174824588084014919421494468214"
        "9164676415923954197909657817178164124852001292366343154090840599269026199794407107539049240781713939666879"
        "7215580264496831581441483538941514221024185926653352436128279006431932533901715820512379195861285552382469"
        "17724609375e-165"},
    {0x1.fffffffffffffp-547, chars_format::scientific, 434,
        "4."
        "3413256825880414778650766166703641411322675916093607479466469553811001826780296417462006032647627201367187"
        "1413761834752052141164677295652926461928381703354565346210597125926547211852349649176168029838842988936429"
        "8329352831847908395819315634356328249704002584732686308181681198538052399588814215078098481563427879333759"
        "4431160528993663162882967077883028442048371853306704872256558012863865067803431641024758391722571104764938"
        "3544921875e-165"},
    {0x1.fffffffffffffp-546, chars_format::scientific, 433,
        "8."
        "6826513651760829557301532333407282822645351832187214958932939107622003653560592834924012065295254402734374"
        "2827523669504104282329354591305852923856763406709130692421194251853094423704699298352336059677685977872859"
        "6658705663695816791638631268712656499408005169465372616363362397076104799177628430156196963126855758667518"
        "8862321057987326325765934155766056884096743706613409744513116025727730135606863282049516783445142209529876"
        "708984375e-165"},
    {0x1.fffffffffffffp-545, chars_format::scientific, 433,
        "1."
        "7365302730352165911460306466681456564529070366437442991786587821524400730712118566984802413059050880546874"
        "8565504733900820856465870918261170584771352681341826138484238850370618884740939859670467211935537195574571"
        "9331741132739163358327726253742531299881601033893074523272672479415220959835525686031239392625371151733503"
        "7772464211597465265153186831153211376819348741322681948902623205145546027121372656409903356689028441905975"
        "341796875e-164"},
    {0x1.fffffffffffffp-544, chars_format::scientific, 432,
        "3."
        "4730605460704331822920612933362913129058140732874885983573175643048801461424237133969604826118101761093749"
        "7131009467801641712931741836522341169542705362683652276968477700741237769481879719340934423871074391149143"
        "8663482265478326716655452507485062599763202067786149046545344958830441919671051372062478785250742303467007"
        "5544928423194930530306373662306422753638697482645363897805246410291092054242745312819806713378056883811950"
        "68359375e-164"},
    {0x1.fffffffffffffp-543, chars_format::scientific, 431,
        "6."
        "9461210921408663645841225866725826258116281465749771967146351286097602922848474267939209652236203522187499"
        "4262018935603283425863483673044682339085410725367304553936955401482475538963759438681868847742148782298287"
        "7326964530956653433310905014970125199526404135572298093090689917660883839342102744124957570501484606934015"
        "1089856846389861060612747324612845507277394965290727795610492820582184108485490625639613426756113767623901"
        "3671875e-164"},
    {0x1.fffffffffffffp-542, chars_format::scientific, 431,
        "1."
        "3892242184281732729168245173345165251623256293149954393429270257219520584569694853587841930447240704437499"
        "8852403787120656685172696734608936467817082145073460910787391080296495107792751887736373769548429756459657"
        "5465392906191330686662181002994025039905280827114459618618137983532176767868420548824991514100296921386803"
        "0217971369277972212122549464922569101455478993058145559122098564116436821697098125127922685351222753524780"
        "2734375e-163"},
    {0x1.fffffffffffffp-541, chars_format::scientific, 430,
        "2."
        "7784484368563465458336490346690330503246512586299908786858540514439041169139389707175683860894481408874999"
        "7704807574241313370345393469217872935634164290146921821574782160592990215585503775472747539096859512919315"
        "0930785812382661373324362005988050079810561654228919237236275967064353535736841097649983028200593842773606"
        "0435942738555944424245098929845138202910957986116291118244197128232873643394196250255845370702445507049560"
        "546875e-163"},
    {0x1.fffffffffffffp-540, chars_format::scientific, 429,
        "5."
        "5568968737126930916672980693380661006493025172599817573717081028878082338278779414351367721788962817749999"
        "5409615148482626740690786938435745871268328580293843643149564321185980431171007550945495078193719025838630"
        "1861571624765322746648724011976100159621123308457838474472551934128707071473682195299966056401187685547212"
        "0871885477111888848490197859690276405821915972232582236488394256465747286788392500511690741404891014099121"
        "09375e-163"},
    {0x1.fffffffffffffp-539, chars_format::scientific, 429,
        "1."
        "1113793747425386183334596138676132201298605034519963514743416205775616467655755882870273544357792563549999"
        "9081923029696525348138157387687149174253665716058768728629912864237196086234201510189099015638743805167726"
        "0372314324953064549329744802395220031924224661691567694894510386825741414294736439059993211280237537109442"
        "4174377095422377769698039571938055281164383194446516447297678851293149457357678500102338148280978202819824"
        "21875e-162"},
    {0x1.fffffffffffffp-538, chars_format::scientific, 428,
        "2."
        "2227587494850772366669192277352264402597210069039927029486832411551232935311511765740547088715585127099999"
        "8163846059393050696276314775374298348507331432117537457259825728474392172468403020378198031277487610335452"
        "0744628649906129098659489604790440063848449323383135389789020773651482828589472878119986422560475074218884"
        "8348754190844755539396079143876110562328766388893032894595357702586298914715357000204676296561956405639648"
        "4375e-162"},
    {0x1.fffffffffffffp-537, chars_format::scientific, 427,
        "4."
        "4455174989701544733338384554704528805194420138079854058973664823102465870623023531481094177431170254199999"
        "6327692118786101392552629550748596697014662864235074914519651456948784344936806040756396062554975220670904"
        "1489257299812258197318979209580880127696898646766270779578041547302965657178945756239972845120950148437769"
        "6697508381689511078792158287752221124657532777786065789190715405172597829430714000409352593123912811279296"
        "875e-162"},
    {0x1.fffffffffffffp-536, chars_format::scientific, 426,
        "8."
        "8910349979403089466676769109409057610388840276159708117947329646204931741246047062962188354862340508399999"
        "2655384237572202785105259101497193394029325728470149829039302913897568689873612081512792125109950441341808"
        "2978514599624516394637958419161760255393797293532541559156083094605931314357891512479945690241900296875539"
        "3395016763379022157584316575504442249315065555572131578381430810345195658861428000818705186247825622558593"
        "75e-162"},
    {0x1.fffffffffffffp-535, chars_format::scientific, 426,
        "1."
        "7782069995880617893335353821881811522077768055231941623589465929240986348249209412592437670972468101679999"
        "8531076847514440557021051820299438678805865145694029965807860582779513737974722416302558425021990088268361"
        "6595702919924903278927591683832352051078759458706508311831216618921186262871578302495989138048380059375107"
        "8679003352675804431516863315100888449863013111114426315676286162069039131772285600163741037249565124511718"
        "75e-161"},
    {0x1.fffffffffffffp-534, chars_format::scientific, 425,
        "3."
        "5564139991761235786670707643763623044155536110463883247178931858481972696498418825184875341944936203359999"
        "7062153695028881114042103640598877357611730291388059931615721165559027475949444832605116850043980176536723"
        "3191405839849806557855183367664704102157518917413016623662433237842372525743156604991978276096760118750215"
        "7358006705351608863033726630201776899726026222228852631352572324138078263544571200327482074499130249023437"
        "5e-161"},
    {0x1.fffffffffffffp-533, chars_format::scientific, 424,
        "7."
        "1128279983522471573341415287527246088311072220927766494357863716963945392996837650369750683889872406719999"
        "4124307390057762228084207281197754715223460582776119863231442331118054951898889665210233700087960353073446"
        "6382811679699613115710366735329408204315037834826033247324866475684745051486313209983956552193520237500431"
        "4716013410703217726067453260403553799452052444457705262705144648276156527089142400654964148998260498046875"
        "e-161"},
    {0x1.fffffffffffffp-532, chars_format::scientific, 424,
        "1."
        "4225655996704494314668283057505449217662214444185553298871572743392789078599367530073950136777974481343999"
        "8824861478011552445616841456239550943044692116555223972646288466223610990379777933042046740017592070614689"
        "3276562335939922623142073347065881640863007566965206649464973295136949010297262641996791310438704047500086"
        "2943202682140643545213490652080710759890410488891541052541028929655231305417828480130992829799652099609375"
        "e-160"},
    {0x1.fffffffffffffp-531, chars_format::scientific, 423,
        "2."
        "8451311993408988629336566115010898435324428888371106597743145486785578157198735060147900273555948962687999"
        "7649722956023104891233682912479101886089384233110447945292576932447221980759555866084093480035184141229378"
        "6553124671879845246284146694131763281726015133930413298929946590273898020594525283993582620877408095000172"
        "588640536428128709042698130416142151978082097778308210508205785931046261083565696026198565959930419921875e"
        "-160"},
    {0x1.fffffffffffffp-530, chars_format::scientific, 422,
        "5."
        "6902623986817977258673132230021796870648857776742213195486290973571156314397470120295800547111897925375999"
        "5299445912046209782467365824958203772178768466220895890585153864894443961519111732168186960070368282458757"
        "3106249343759690492568293388263526563452030267860826597859893180547796041189050567987165241754816190000345"
        "17728107285625741808539626083228430395616419555661642101641157186209252216713139205239713191986083984375e-"
        "160"},
    {0x1.fffffffffffffp-529, chars_format::scientific, 422,
        "1."
        "1380524797363595451734626446004359374129771555348442639097258194714231262879494024059160109422379585075199"
        "9059889182409241956493473164991640754435753693244179178117030772978888792303822346433637392014073656491751"
        "4621249868751938098513658677652705312690406053572165319571978636109559208237810113597433048350963238000069"
        "03545621457125148361707925216645686079123283911132328420328231437241850443342627841047942638397216796875e-"
        "159"},
    {0x1.fffffffffffffp-528, chars_format::scientific, 421,
        "2."
        "2761049594727190903469252892008718748259543110696885278194516389428462525758988048118320218844759170150399"
        "8119778364818483912986946329983281508871507386488358356234061545957777584607644692867274784028147312983502"
        "9242499737503876197027317355305410625380812107144330639143957272219118416475620227194866096701926476000138"
        "0709124291425029672341585043329137215824656782226465684065646287448370088668525568209588527679443359375e-"
        "159"},
    {0x1.fffffffffffffp-527, chars_format::scientific, 420,
        "4."
        "5522099189454381806938505784017437496519086221393770556389032778856925051517976096236640437689518340300799"
        "6239556729636967825973892659966563017743014772976716712468123091915555169215289385734549568056294625967005"
        "8484999475007752394054634710610821250761624214288661278287914544438236832951240454389732193403852952000276"
        "141824858285005934468317008665827443164931356445293136813129257489674017733705113641917705535888671875e-"
        "159"},
    {0x1.fffffffffffffp-526, chars_format::scientific, 419,
        "9."
        "1044198378908763613877011568034874993038172442787541112778065557713850103035952192473280875379036680601599"
        "2479113459273935651947785319933126035486029545953433424936246183831110338430578771469099136112589251934011"
        "6969998950015504788109269421221642501523248428577322556575829088876473665902480908779464386807705904000552"
        "28364971657001186893663401733165488632986271289058627362625851497934803546741022728383541107177734375e-"
        "159"},
    {0x1.fffffffffffffp-525, chars_format::scientific, 419,
        "1."
        "8208839675781752722775402313606974998607634488557508222555613111542770020607190438494656175075807336120319"
        "8495822691854787130389557063986625207097205909190686684987249236766222067686115754293819827222517850386802"
        "3393999790003100957621853884244328500304649685715464511315165817775294733180496181755892877361541180800110"
        "45672994331400237378732680346633097726597254257811725472525170299586960709348204545676708221435546875e-"
        "158"},
    {0x1.fffffffffffffp-524, chars_format::scientific, 418,
        "3."
        "6417679351563505445550804627213949997215268977115016445111226223085540041214380876989312350151614672240639"
        "6991645383709574260779114127973250414194411818381373369974498473532444135372231508587639654445035700773604"
        "6787999580006201915243707768488657000609299371430929022630331635550589466360992363511785754723082361600220"
        "9134598866280047475746536069326619545319450851562345094505034059917392141869640909135341644287109375e-"
        "158"},
    {0x1.fffffffffffffp-523, chars_format::scientific, 417,
        "7."
        "2835358703127010891101609254427899994430537954230032890222452446171080082428761753978624700303229344481279"
        "3983290767419148521558228255946500828388823636762746739948996947064888270744463017175279308890071401547209"
        "3575999160012403830487415536977314001218598742861858045260663271101178932721984727023571509446164723200441"
        "826919773256009495149307213865323909063890170312469018901006811983478428373928181827068328857421875e-158"},
    {0x1.fffffffffffffp-522, chars_format::scientific, 417,
        "1."
        "4567071740625402178220321850885579998886107590846006578044490489234216016485752350795724940060645868896255"
        "8796658153483829704311645651189300165677764727352549347989799389412977654148892603435055861778014280309441"
        "8715199832002480766097483107395462800243719748572371609052132654220235786544396945404714301889232944640088"
        "365383954651201899029861442773064781812778034062493803780201362396695685674785636365413665771484375e-157"},
    {0x1.fffffffffffffp-521, chars_format::scientific, 416,
        "2."
        "9134143481250804356440643701771159997772215181692013156088980978468432032971504701591449880121291737792511"
        "7593316306967659408623291302378600331355529454705098695979598778825955308297785206870111723556028560618883"
        "7430399664004961532194966214790925600487439497144743218104265308440471573088793890809428603778465889280176"
        "73076790930240379805972288554612956362555606812498760756040272479339137134957127273082733154296875e-157"},
    {0x1.fffffffffffffp-520, chars_format::scientific, 415,
        "5."
        "8268286962501608712881287403542319995544430363384026312177961956936864065943009403182899760242583475585023"
        "5186632613935318817246582604757200662711058909410197391959197557651910616595570413740223447112057121237767"
        "4860799328009923064389932429581851200974878994289486436208530616880943146177587781618857207556931778560353"
        "4615358186048075961194457710922591272511121362499752151208054495867827426991425454616546630859375e-157"},
    {0x1.fffffffffffffp-519, chars_format::scientific, 415,
        "1."
        "1653657392500321742576257480708463999108886072676805262435592391387372813188601880636579952048516695117004"
        "7037326522787063763449316520951440132542211781882039478391839511530382123319114082748044689422411424247553"
        "4972159865601984612877986485916370240194975798857897287241706123376188629235517556323771441511386355712070"
        "6923071637209615192238891542184518254502224272499950430241610899173565485398285090923309326171875e-156"},
    {0x1.fffffffffffffp-518, chars_format::scientific, 414,
        "2."
        "3307314785000643485152514961416927998217772145353610524871184782774745626377203761273159904097033390234009"
        "4074653045574127526898633041902880265084423563764078956783679023060764246638228165496089378844822848495106"
        "9944319731203969225755972971832740480389951597715794574483412246752377258471035112647542883022772711424141"
        "384614327441923038447778308436903650900444854499990086048322179834713097079657018184661865234375e-156"},
    {0x1.fffffffffffffp-517, chars_format::scientific, 413,
        "4."
        "6614629570001286970305029922833855996435544290707221049742369565549491252754407522546319808194066780468018"
        "8149306091148255053797266083805760530168847127528157913567358046121528493276456330992178757689645696990213"
        "9888639462407938451511945943665480960779903195431589148966824493504754516942070225295085766045545422848282"
        "76922865488384607689555661687380730180088970899998017209664435966942619415931403636932373046875e-156"},
    {0x1.fffffffffffffp-516, chars_format::scientific, 412,
        "9."
        "3229259140002573940610059845667711992871088581414442099484739131098982505508815045092639616388133560936037"
        "6298612182296510107594532167611521060337694255056315827134716092243056986552912661984357515379291393980427"
        "9777278924815876903023891887330961921559806390863178297933648987009509033884140450590171532091090845696565"
        "5384573097676921537911132337476146036017794179999603441932887193388523883186280727386474609375e-156"},
    {0x1.fffffffffffffp-515, chars_format::scientific, 412,
        "1."
        "8645851828000514788122011969133542398574217716282888419896947826219796501101763009018527923277626712187207"
        "5259722436459302021518906433522304212067538851011263165426943218448611397310582532396871503075858278796085"
        "5955455784963175380604778377466192384311961278172635659586729797401901806776828090118034306418218169139313"
        "1076914619535384307582226467495229207203558835999920688386577438677704776637256145477294921875e-155"},
    {0x1.fffffffffffffp-514, chars_format::scientific, 411,
        "3."
        "7291703656001029576244023938267084797148435432565776839793895652439593002203526018037055846555253424374415"
        "0519444872918604043037812867044608424135077702022526330853886436897222794621165064793743006151716557592171"
        "1910911569926350761209556754932384768623922556345271319173459594803803613553656180236068612836436338278626"
        "215382923907076861516445293499045841440711767199984137677315487735540955327451229095458984375e-155"},
    {0x1.fffffffffffffp-513, chars_format::scientific, 410,
        "7."
        "4583407312002059152488047876534169594296870865131553679587791304879186004407052036074111693110506848748830"
        "1038889745837208086075625734089216848270155404045052661707772873794445589242330129587486012303433115184342"
        "3821823139852701522419113509864769537247845112690542638346919189607607227107312360472137225672872676557252"
        "43076584781415372303289058699809168288142353439996827535463097547108191065490245819091796875e-155"},
    {0x1.fffffffffffffp-512, chars_format::scientific, 410,
        "1."
        "4916681462400411830497609575306833918859374173026310735917558260975837200881410407214822338622101369749766"
        "0207777949167441617215125146817843369654031080809010532341554574758889117848466025917497202460686623036868"
        "4764364627970540304483822701972953907449569022538108527669383837921521445421462472094427445134574535311450"
        "48615316956283074460657811739961833657628470687999365507092619509421638213098049163818359375e-154"},
    {0x1.fffffffffffffp-511, chars_format::scientific, 409,
        "2."
        "9833362924800823660995219150613667837718748346052621471835116521951674401762820814429644677244202739499532"
        "0415555898334883234430250293635686739308062161618021064683109149517778235696932051834994404921373246073736"
        "9528729255941080608967645403945907814899138045076217055338767675843042890842924944188854890269149070622900"
        "9723063391256614892131562347992366731525694137599873101418523901884327642619609832763671875e-154"},
    {0x1.fffffffffffffp-510, chars_format::scientific, 408,
        "5."
        "9666725849601647321990438301227335675437496692105242943670233043903348803525641628859289354488405478999064"
        "0831111796669766468860500587271373478616124323236042129366218299035556471393864103669988809842746492147473"
        "9057458511882161217935290807891815629798276090152434110677535351686085781685849888377709780538298141245801"
        "944612678251322978426312469598473346305138827519974620283704780376865528523921966552734375e-154"},
    {0x1.fffffffffffffp-509, chars_format::scientific, 408,
        "1."
        "1933345169920329464398087660245467135087499338421048588734046608780669760705128325771857870897681095799812"
        "8166222359333953293772100117454274695723224864647208425873243659807111294278772820733997761968549298429494"
        "7811491702376432243587058161578363125959655218030486822135507070337217156337169977675541956107659628249160"
        "388922535650264595685262493919694669261027765503994924056740956075373105704784393310546875e-153"},
    {0x1.fffffffffffffp-508, chars_format::scientific, 407,
        "2."
        "3866690339840658928796175320490934270174998676842097177468093217561339521410256651543715741795362191599625"
        "6332444718667906587544200234908549391446449729294416851746487319614222588557545641467995523937098596858989"
        "5622983404752864487174116323156726251919310436060973644271014140674434312674339955351083912215319256498320"
        "77784507130052919137052498783938933852205553100798984811348191215074621140956878662109375e-153"},
    {0x1.fffffffffffffp-507, chars_format::scientific, 406,
        "4."
        "7733380679681317857592350640981868540349997353684194354936186435122679042820513303087431483590724383199251"
        "2664889437335813175088400469817098782892899458588833703492974639228445177115091282935991047874197193717979"
        "1245966809505728974348232646313452503838620872121947288542028281348868625348679910702167824430638512996641"
        "5556901426010583827410499756787786770441110620159796962269638243014924228191375732421875e-153"},
    {0x1.fffffffffffffp-506, chars_format::scientific, 405,
        "9."
        "5466761359362635715184701281963737080699994707368388709872372870245358085641026606174862967181448766398502"
        "5329778874671626350176800939634197565785798917177667406985949278456890354230182565871982095748394387435958"
        "2491933619011457948696465292626905007677241744243894577084056562697737250697359821404335648861277025993283"
        "111380285202116765482099951357557354088222124031959392453927648602984845638275146484375e-153"},
    {0x1.fffffffffffffp-505, chars_format::scientific, 405,
        "1."
        "9093352271872527143036940256392747416139998941473677741974474574049071617128205321234972593436289753279700"
        "5065955774934325270035360187926839513157159783435533481397189855691378070846036513174396419149678877487191"
        "6498386723802291589739293058525381001535448348848778915416811312539547450139471964280867129772255405198656"
        "622276057040423353096419990271511470817644424806391878490785529720596969127655029296875e-152"},
    {0x1.fffffffffffffp-504, chars_format::scientific, 404,
        "3."
        "8186704543745054286073880512785494832279997882947355483948949148098143234256410642469945186872579506559401"
        "0131911549868650540070720375853679026314319566871066962794379711382756141692073026348792838299357754974383"
        "2996773447604583179478586117050762003070896697697557830833622625079094900278943928561734259544510810397313"
        "24455211408084670619283998054302294163528884961278375698157105944119393825531005859375e-152"},
    {0x1.fffffffffffffp-503, chars_format::scientific, 403,
        "7."
        "6373409087490108572147761025570989664559995765894710967897898296196286468512821284939890373745159013118802"
        "0263823099737301080141440751707358052628639133742133925588759422765512283384146052697585676598715509948766"
        "5993546895209166358957172234101524006141793395395115661667245250158189800557887857123468519089021620794626"
        "4891042281616934123856799610860458832705776992255675139631421188823878765106201171875e-152"},
    {0x1.fffffffffffffp-502, chars_format::scientific, 403,
        "1."
        "5274681817498021714429552205114197932911999153178942193579579659239257293702564256987978074749031802623760"
        "4052764619947460216028288150341471610525727826748426785117751884553102456676829210539517135319743101989753"
        "3198709379041833271791434446820304801228358679079023132333449050031637960111577571424693703817804324158925"
        "2978208456323386824771359922172091766541155398451135027926284237764775753021240234375e-151"},
    {0x1.fffffffffffffp-501, chars_format::scientific, 402,
        "3."
        "0549363634996043428859104410228395865823998306357884387159159318478514587405128513975956149498063605247520"
        "8105529239894920432056576300682943221051455653496853570235503769106204913353658421079034270639486203979506"
        "6397418758083666543582868893640609602456717358158046264666898100063275920223155142849387407635608648317850"
        "595641691264677364954271984434418353308231079690227005585256847552955150604248046875e-151"},
    {0x1.fffffffffffffp-500, chars_format::scientific, 401,
        "6."
        "1098727269992086857718208820456791731647996612715768774318318636957029174810257027951912298996127210495041"
        "6211058479789840864113152601365886442102911306993707140471007538212409826707316842158068541278972407959013"
        "2794837516167333087165737787281219204913434716316092529333796200126551840446310285698774815271217296635701"
        "19128338252935472990854396886883670661646215938045401117051369510591030120849609375e-151"},
    {0x1.fffffffffffffp-499, chars_format::scientific, 401,
        "1."
        "2219745453998417371543641764091358346329599322543153754863663727391405834962051405590382459799225442099008"
        "3242211695957968172822630520273177288420582261398741428094201507642481965341463368431613708255794481591802"
        "6558967503233466617433147557456243840982686943263218505866759240025310368089262057139754963054243459327140"
        "23825667650587094598170879377376734132329243187609080223410273902118206024169921875e-150"},
    {0x1.fffffffffffffp-498, chars_format::scientific, 400,
        "2."
        "4439490907996834743087283528182716692659198645086307509727327454782811669924102811180764919598450884198016"
        "6484423391915936345645261040546354576841164522797482856188403015284963930682926736863227416511588963183605"
        "3117935006466933234866295114912487681965373886526437011733518480050620736178524114279509926108486918654280"
        "4765133530117418919634175875475346826465848637521816044682054780423641204833984375e-150"},
    {0x1.fffffffffffffp-497, chars_format::scientific, 399,
        "4."
        "8878981815993669486174567056365433385318397290172615019454654909565623339848205622361529839196901768396033"
        "2968846783831872691290522081092709153682329045594965712376806030569927861365853473726454833023177926367210"
        "6235870012933866469732590229824975363930747773052874023467036960101241472357048228559019852216973837308560"
        "953026706023483783926835175095069365293169727504363208936410956084728240966796875e-150"},
    {0x1.fffffffffffffp-496, chars_format::scientific, 398,
        "9."
        "7757963631987338972349134112730866770636794580345230038909309819131246679696411244723059678393803536792066"
        "5937693567663745382581044162185418307364658091189931424753612061139855722731706947452909666046355852734421"
        "2471740025867732939465180459649950727861495546105748046934073920202482944714096457118039704433947674617121"
        "90605341204696756785367035019013873058633945500872641787282191216945648193359375e-150"},
    {0x1.fffffffffffffp-495, chars_format::scientific, 398,
        "1."
        "9551592726397467794469826822546173354127358916069046007781861963826249335939282248944611935678760707358413"
        "3187538713532749076516208832437083661472931618237986284950722412227971144546341389490581933209271170546884"
        "2494348005173546587893036091929990145572299109221149609386814784040496588942819291423607940886789534923424"
        "38121068240939351357073407003802774611726789100174528357456438243389129638671875e-149"},
    {0x1.fffffffffffffp-494, chars_format::scientific, 397,
        "3."
        "9103185452794935588939653645092346708254717832138092015563723927652498671878564497889223871357521414716826"
        "6375077427065498153032417664874167322945863236475972569901444824455942289092682778981163866418542341093768"
        "4988696010347093175786072183859980291144598218442299218773629568080993177885638582847215881773579069846848"
        "7624213648187870271414681400760554922345357820034905671491287648677825927734375e-149"},
    {0x1.fffffffffffffp-493, chars_format::scientific, 396,
        "7."
        "8206370905589871177879307290184693416509435664276184031127447855304997343757128995778447742715042829433653"
        "2750154854130996306064835329748334645891726472951945139802889648911884578185365557962327732837084682187536"
        "9977392020694186351572144367719960582289196436884598437547259136161986355771277165694431763547158139693697"
        "524842729637574054282936280152110984469071564006981134298257529735565185546875e-149"},
    {0x1.fffffffffffffp-492, chars_format::scientific, 396,
        "1."
        "5641274181117974235575861458036938683301887132855236806225489571060999468751425799155689548543008565886730"
        "6550030970826199261212967065949666929178345294590389027960577929782376915637073111592465546567416936437507"
        "3995478404138837270314428873543992116457839287376919687509451827232397271154255433138886352709431627938739"
        "504968545927514810856587256030422196893814312801396226859651505947113037109375e-148"},
    {0x1.fffffffffffffp-491, chars_format::scientific, 395,
        "3."
        "1282548362235948471151722916073877366603774265710473612450979142121998937502851598311379097086017131773461"
        "3100061941652398522425934131899333858356690589180778055921155859564753831274146223184931093134833872875014"
        "7990956808277674540628857747087984232915678574753839375018903654464794542308510866277772705418863255877479"
        "00993709185502962171317451206084439378762862560279245371930301189422607421875e-148"},
    {0x1.fffffffffffffp-490, chars_format::scientific, 394,
        "6."
        "2565096724471896942303445832147754733207548531420947224901958284243997875005703196622758194172034263546922"
        "6200123883304797044851868263798667716713381178361556111842311719129507662548292446369862186269667745750029"
        "5981913616555349081257715494175968465831357149507678750037807308929589084617021732555545410837726511754958"
        "0198741837100592434263490241216887875752572512055849074386060237884521484375e-148"},
    {0x1.fffffffffffffp-489, chars_format::scientific, 394,
        "1."
        "2513019344894379388460689166429550946641509706284189444980391656848799575001140639324551638834406852709384"
        "5240024776660959408970373652759733543342676235672311222368462343825901532509658489273972437253933549150005"
        "9196382723311069816251543098835193693166271429901535750007561461785917816923404346511109082167545302350991"
        "6039748367420118486852698048243377575150514502411169814877212047576904296875e-147"},
    {0x1.fffffffffffffp-488, chars_format::scientific, 393,
        "2."
        "5026038689788758776921378332859101893283019412568378889960783313697599150002281278649103277668813705418769"
        "0480049553321918817940747305519467086685352471344622444736924687651803065019316978547944874507867098300011"
        "8392765446622139632503086197670387386332542859803071500015122923571835633846808693022218164335090604701983"
        "207949673484023697370539609648675515030102900482233962975442409515380859375e-147"},
    {0x1.fffffffffffffp-487, chars_format::scientific, 392,
        "5."
        "0052077379577517553842756665718203786566038825136757779921566627395198300004562557298206555337627410837538"
        "0960099106643837635881494611038934173370704942689244889473849375303606130038633957095889749015734196600023"
        "6785530893244279265006172395340774772665085719606143000030245847143671267693617386044436328670181209403966"
        "41589934696804739474107921929735103006020580096446792595088481903076171875e-147"},
    {0x1.fffffffffffffp-486, chars_format::scientific, 392,
        "1."
        "0010415475915503510768551333143640757313207765027351555984313325479039660000912511459641311067525482167507"
        "6192019821328767527176298922207786834674140988537848977894769875060721226007726791419177949803146839320004"
        "7357106178648855853001234479068154954533017143921228600006049169428734253538723477208887265734036241880793"
        "28317986939360947894821584385947020601204116019289358519017696380615234375e-146"},
    {0x1.fffffffffffffp-485, chars_format::scientific, 391,
        "2."
        "0020830951831007021537102666287281514626415530054703111968626650958079320001825022919282622135050964335015"
        "2384039642657535054352597844415573669348281977075697955789539750121442452015453582838355899606293678640009"
        "4714212357297711706002468958136309909066034287842457200012098338857468507077446954417774531468072483761586"
        "5663597387872189578964316877189404120240823203857871703803539276123046875e-146"},
    {0x1.fffffffffffffp-484, chars_format::scientific, 390,
        "4."
        "0041661903662014043074205332574563029252831060109406223937253301916158640003650045838565244270101928670030"
        "4768079285315070108705195688831147338696563954151395911579079500242884904030907165676711799212587357280018"
        "9428424714595423412004937916272619818132068575684914400024196677714937014154893908835549062936144967523173"
        "132719477574437915792863375437880824048164640771574340760707855224609375e-146"},
    {0x1.fffffffffffffp-483, chars_format::scientific, 389,
        "8."
        "0083323807324028086148410665149126058505662120218812447874506603832317280007300091677130488540203857340060"
        "9536158570630140217410391377662294677393127908302791823158159000485769808061814331353423598425174714560037"
        "8856849429190846824009875832545239636264137151369828800048393355429874028309787817671098125872289935046346"
        "26543895514887583158572675087576164809632928154314868152141571044921875e-146"},
    {0x1.fffffffffffffp-482, chars_format::scientific, 389,
        "1."
        "6016664761464805617229682133029825211701132424043762489574901320766463456001460018335426097708040771468012"
        "1907231714126028043482078275532458935478625581660558364631631800097153961612362866270684719685034942912007"
        "5771369885838169364801975166509047927252827430273965760009678671085974805661957563534219625174457987009269"
        "25308779102977516631714535017515232961926585630862973630428314208984375e-145"},
    {0x1.fffffffffffffp-481, chars_format::scientific, 388,
        "3."
        "2033329522929611234459364266059650423402264848087524979149802641532926912002920036670852195416081542936024"
        "3814463428252056086964156551064917870957251163321116729263263600194307923224725732541369439370069885824015"
        "1542739771676338729603950333018095854505654860547931520019357342171949611323915127068439250348915974018538"
        "5061755820595503326342907003503046592385317126172594726085662841796875e-145"},
    {0x1.fffffffffffffp-480, chars_format::scientific, 387,
        "6."
        "4066659045859222468918728532119300846804529696175049958299605283065853824005840073341704390832163085872048"
        "7628926856504112173928313102129835741914502326642233458526527200388615846449451465082738878740139771648030"
        "3085479543352677459207900666036191709011309721095863040038714684343899222647830254136878500697831948037077"
        "012351164119100665268581400700609318477063425234518945217132568359375e-145"},
    {0x1.fffffffffffffp-479, chars_format::scientific, 387,
        "1."
        "2813331809171844493783745706423860169360905939235009991659921056613170764801168014668340878166432617174409"
        "7525785371300822434785662620425967148382900465328446691705305440077723169289890293016547775748027954329606"
        "0617095908670535491841580133207238341802261944219172608007742936868779844529566050827375700139566389607415"
        "402470232823820133053716280140121863695412685046903789043426513671875e-144"},
    {0x1.fffffffffffffp-478, chars_format::scientific, 386,
        "2."
        "5626663618343688987567491412847720338721811878470019983319842113226341529602336029336681756332865234348819"
        "5051570742601644869571325240851934296765800930656893383410610880155446338579780586033095551496055908659212"
        "1234191817341070983683160266414476683604523888438345216015485873737559689059132101654751400279132779214830"
        "80494046564764026610743256028024372739082537009380757808685302734375e-144"},
    {0x1.fffffffffffffp-477, chars_format::scientific, 385,
        "5."
        "1253327236687377975134982825695440677443623756940039966639684226452683059204672058673363512665730468697639"
        "0103141485203289739142650481703868593531601861313786766821221760310892677159561172066191102992111817318424"
        "2468383634682141967366320532828953367209047776876690432030971747475119378118264203309502800558265558429661"
        "6098809312952805322148651205604874547816507401876151561737060546875e-144"},
    {0x1.fffffffffffffp-476, chars_format::scientific, 385,
        "1."
        "0250665447337475595026996565139088135488724751388007993327936845290536611840934411734672702533146093739527"
        "8020628297040657947828530096340773718706320372262757353364244352062178535431912234413238220598422363463684"
        "8493676726936428393473264106565790673441809555375338086406194349495023875623652840661900560111653111685932"
        "3219761862590561064429730241120974909563301480375230312347412109375e-143"},
    {0x1.fffffffffffffp-475, chars_format::scientific, 384,
        "2."
        "0501330894674951190053993130278176270977449502776015986655873690581073223681868823469345405066292187479055"
        "6041256594081315895657060192681547437412640744525514706728488704124357070863824468826476441196844726927369"
        "6987353453872856786946528213131581346883619110750676172812388698990047751247305681323801120223306223371864"
        "643952372518112212885946048224194981912660296075046062469482421875e-143"},
    {0x1.fffffffffffffp-474, chars_format::scientific, 383,
        "4."
        "1002661789349902380107986260556352541954899005552031973311747381162146447363737646938690810132584374958111"
        "2082513188162631791314120385363094874825281489051029413456977408248714141727648937652952882393689453854739"
        "3974706907745713573893056426263162693767238221501352345624777397980095502494611362647602240446612446743729"
        "28790474503622442577189209644838996382532059215009212493896484375e-143"},
    {0x1.fffffffffffffp-473, chars_format::scientific, 382,
        "8."
        "2005323578699804760215972521112705083909798011104063946623494762324292894727475293877381620265168749916222"
        "4165026376325263582628240770726189749650562978102058826913954816497428283455297875305905764787378907709478"
        "7949413815491427147786112852526325387534476443002704691249554795960191004989222725295204480893224893487458"
        "5758094900724488515437841928967799276506411843001842498779296875e-143"},
    {0x1.fffffffffffffp-472, chars_format::scientific, 382,
        "1."
        "6401064715739960952043194504222541016781959602220812789324698952464858578945495058775476324053033749983244"
        "4833005275265052716525648154145237949930112595620411765382790963299485656691059575061181152957475781541895"
        "7589882763098285429557222570505265077506895288600540938249910959192038200997844545059040896178644978697491"
        "7151618980144897703087568385793559855301282368600368499755859375e-142"},
    {0x1.fffffffffffffp-471, chars_format::scientific, 381,
        "3."
        "2802129431479921904086389008445082033563919204441625578649397904929717157890990117550952648106067499966488"
        "9666010550530105433051296308290475899860225191240823530765581926598971313382119150122362305914951563083791"
        "5179765526196570859114445141010530155013790577201081876499821918384076401995689090118081792357289957394983"
        "430323796028979540617513677158711971060256473720073699951171875e-142"},
    {0x1.fffffffffffffp-470, chars_format::scientific, 380,
        "6."
        "5604258862959843808172778016890164067127838408883251157298795809859434315781980235101905296212134999932977"
        "9332021101060210866102592616580951799720450382481647061531163853197942626764238300244724611829903126167583"
        "0359531052393141718228890282021060310027581154402163752999643836768152803991378180236163584714579914789966"
        "86064759205795908123502735431742394212051294744014739990234375e-142"},
    {0x1.fffffffffffffp-469, chars_format::scientific, 380,
        "1."
        "3120851772591968761634555603378032813425567681776650231459759161971886863156396047020381059242426999986595"
        "5866404220212042173220518523316190359944090076496329412306232770639588525352847660048944922365980625233516"
        "6071906210478628343645778056404212062005516230880432750599928767353630560798275636047232716942915982957993"
        "37212951841159181624700547086348478842410258948802947998046875e-141"},
    {0x1.fffffffffffffp-468, chars_format::scientific, 379,
        "2."
        "6241703545183937523269111206756065626851135363553300462919518323943773726312792094040762118484853999973191"
        "1732808440424084346441037046632380719888180152992658824612465541279177050705695320097889844731961250467033"
        "2143812420957256687291556112808424124011032461760865501199857534707261121596551272094465433885831965915986"
        "7442590368231836324940109417269695768482051789760589599609375e-141"},
    {0x1.fffffffffffffp-467, chars_format::scientific, 378,
        "5."
        "2483407090367875046538222413512131253702270727106600925839036647887547452625584188081524236969707999946382"
        "3465616880848168692882074093264761439776360305985317649224931082558354101411390640195779689463922500934066"
        "4287624841914513374583112225616848248022064923521731002399715069414522243193102544188930867771663931831973"
        "488518073646367264988021883453939153696410357952117919921875e-141"},
    {0x1.fffffffffffffp-466, chars_format::scientific, 378,
        "1."
        "0496681418073575009307644482702426250740454145421320185167807329577509490525116837616304847393941599989276"
        "4693123376169633738576414818652952287955272061197063529844986216511670820282278128039155937892784500186813"
        "2857524968382902674916622445123369649604412984704346200479943013882904448638620508837786173554332786366394"
        "697703614729273452997604376690787830739282071590423583984375e-140"},
    {0x1.fffffffffffffp-465, chars_format::scientific, 377,
        "2."
        "0993362836147150018615288965404852501480908290842640370335614659155018981050233675232609694787883199978552"
        "9386246752339267477152829637305904575910544122394127059689972433023341640564556256078311875785569000373626"
        "5715049936765805349833244890246739299208825969408692400959886027765808897277241017675572347108665572732789"
        "39540722945854690599520875338157566147856414318084716796875e-140"},
    {0x1.fffffffffffffp-464, chars_format::scientific, 376,
        "4."
        "1986725672294300037230577930809705002961816581685280740671229318310037962100467350465219389575766399957105"
        "8772493504678534954305659274611809151821088244788254119379944866046683281129112512156623751571138000747253"
        "1430099873531610699666489780493478598417651938817384801919772055531617794554482035351144694217331145465578"
        "7908144589170938119904175067631513229571282863616943359375e-140"},
    {0x1.fffffffffffffp-463, chars_format::scientific, 375,
        "8."
        "3973451344588600074461155861619410005923633163370561481342458636620075924200934700930438779151532799914211"
        "7544987009357069908611318549223618303642176489576508238759889732093366562258225024313247503142276001494506"
        "2860199747063221399332979560986957196835303877634769603839544111063235589108964070702289388434662290931157"
        "581628917834187623980835013526302645914256572723388671875e-140"},
    {0x1.fffffffffffffp-462, chars_format::scientific, 375,
        "1."
        "6794690268917720014892231172323882001184726632674112296268491727324015184840186940186087755830306559982842"
        "3508997401871413981722263709844723660728435297915301647751977946418673312451645004862649500628455200298901"
        "2572039949412644279866595912197391439367060775526953920767908822212647117821792814140457877686932458186231"
        "516325783566837524796167002705260529182851314544677734375e-139"},
    {0x1.fffffffffffffp-461, chars_format::scientific, 374,
        "3."
        "3589380537835440029784462344647764002369453265348224592536983454648030369680373880372175511660613119965684"
        "7017994803742827963444527419689447321456870595830603295503955892837346624903290009725299001256910400597802"
        "5144079898825288559733191824394782878734121551053907841535817644425294235643585628280915755373864916372463"
        "03265156713367504959233400541052105836570262908935546875e-139"},
    {0x1.fffffffffffffp-460, chars_format::scientific, 373,
        "6."
        "7178761075670880059568924689295528004738906530696449185073966909296060739360747760744351023321226239931369"
        "4035989607485655926889054839378894642913741191661206591007911785674693249806580019450598002513820801195605"
        "0288159797650577119466383648789565757468243102107815683071635288850588471287171256561831510747729832744926"
        "0653031342673500991846680108210421167314052581787109375e-139"},
    {0x1.fffffffffffffp-459, chars_format::scientific, 373,
        "1."
        "3435752215134176011913784937859105600947781306139289837014793381859212147872149552148870204664245247986273"
        "8807197921497131185377810967875778928582748238332241318201582357134938649961316003890119600502764160239121"
        "0057631959530115423893276729757913151493648620421563136614327057770117694257434251312366302149545966548985"
        "2130606268534700198369336021642084233462810516357421875e-138"},
    {0x1.fffffffffffffp-458, chars_format::scientific, 372,
        "2."
        "6871504430268352023827569875718211201895562612278579674029586763718424295744299104297740409328490495972547"
        "7614395842994262370755621935751557857165496476664482636403164714269877299922632007780239201005528320478242"
        "0115263919060230847786553459515826302987297240843126273228654115540235388514868502624732604299091933097970"
        "426121253706940039673867204328416846692562103271484375e-138"},
    {0x1.fffffffffffffp-457, chars_format::scientific, 371,
        "5."
        "3743008860536704047655139751436422403791125224557159348059173527436848591488598208595480818656980991945095"
        "5228791685988524741511243871503115714330992953328965272806329428539754599845264015560478402011056640956484"
        "0230527838120461695573106919031652605974594481686252546457308231080470777029737005249465208598183866195940"
        "85224250741388007934773440865683369338512420654296875e-138"},
    {0x1.fffffffffffffp-456, chars_format::scientific, 371,
        "1."
        "0748601772107340809531027950287284480758225044911431869611834705487369718297719641719096163731396198389019"
        "1045758337197704948302248774300623142866198590665793054561265885707950919969052803112095680402211328191296"
        "8046105567624092339114621383806330521194918896337250509291461646216094155405947401049893041719636773239188"
        "17044850148277601586954688173136673867702484130859375e-137"},
    {0x1.fffffffffffffp-455, chars_format::scientific, 370,
        "2."
        "1497203544214681619062055900574568961516450089822863739223669410974739436595439283438192327462792396778038"
        "2091516674395409896604497548601246285732397181331586109122531771415901839938105606224191360804422656382593"
        "6092211135248184678229242767612661042389837792674501018582923292432188310811894802099786083439273546478376"
        "3408970029655520317390937634627334773540496826171875e-137"},
    {0x1.fffffffffffffp-454, chars_format::scientific, 369,
        "4."
        "2994407088429363238124111801149137923032900179645727478447338821949478873190878566876384654925584793556076"
        "4183033348790819793208995097202492571464794362663172218245063542831803679876211212448382721608845312765187"
        "2184422270496369356458485535225322084779675585349002037165846584864376621623789604199572166878547092956752"
        "681794005931104063478187526925466954708099365234375e-137"},
    {0x1.fffffffffffffp-453, chars_format::scientific, 368,
        "8."
        "5988814176858726476248223602298275846065800359291454956894677643898957746381757133752769309851169587112152"
        "8366066697581639586417990194404985142929588725326344436490127085663607359752422424896765443217690625530374"
        "4368844540992738712916971070450644169559351170698004074331693169728753243247579208399144333757094185913505"
        "36358801186220812695637505385093390941619873046875e-137"},
    {0x1.fffffffffffffp-452, chars_format::scientific, 368,
        "1."
        "7197762835371745295249644720459655169213160071858290991378935528779791549276351426750553861970233917422430"
        "5673213339516327917283598038880997028585917745065268887298025417132721471950484484979353088643538125106074"
        "8873768908198547742583394214090128833911870234139600814866338633945750648649515841679828866751418837182701"
        "07271760237244162539127501077018678188323974609375e-136"},
    {0x1.fffffffffffffp-451, chars_format::scientific, 367,
        "3."
        "4395525670743490590499289440919310338426320143716581982757871057559583098552702853501107723940467834844861"
        "1346426679032655834567196077761994057171835490130537774596050834265442943900968969958706177287076250212149"
        "7747537816397095485166788428180257667823740468279201629732677267891501297299031683359657733502837674365402"
        "1454352047448832507825500215403735637664794921875e-136"},
    {0x1.fffffffffffffp-450, chars_format::scientific, 366,
        "6."
        "8791051341486981180998578881838620676852640287433163965515742115119166197105405707002215447880935669689722"
        "2692853358065311669134392155523988114343670980261075549192101668530885887801937939917412354574152500424299"
        "5495075632794190970333576856360515335647480936558403259465354535783002594598063366719315467005675348730804"
        "290870409489766501565100043080747127532958984375e-136"},
    {0x1.fffffffffffffp-449, chars_format::scientific, 366,
        "1."
        "3758210268297396236199715776367724135370528057486632793103148423023833239421081141400443089576187133937944"
        "4538570671613062333826878431104797622868734196052215109838420333706177177560387587983482470914830500084859"
        "9099015126558838194066715371272103067129496187311680651893070907156600518919612673343863093401135069746160"
        "858174081897953300313020008616149425506591796875e-135"},
    {0x1.fffffffffffffp-448, chars_format::scientific, 365,
        "2."
        "7516420536594792472399431552735448270741056114973265586206296846047666478842162282800886179152374267875888"
        "9077141343226124667653756862209595245737468392104430219676840667412354355120775175966964941829661000169719"
        "8198030253117676388133430742544206134258992374623361303786141814313201037839225346687726186802270139492321"
        "71634816379590660062604001723229885101318359375e-135"},
    {0x1.fffffffffffffp-447, chars_format::scientific, 364,
        "5."
        "5032841073189584944798863105470896541482112229946531172412593692095332957684324565601772358304748535751777"
        "8154282686452249335307513724419190491474936784208860439353681334824708710241550351933929883659322000339439"
        "6396060506235352776266861485088412268517984749246722607572283628626402075678450693375452373604540278984643"
        "4326963275918132012520800344645977020263671875e-135"},
    {0x1.fffffffffffffp-446, chars_format::scientific, 364,
        "1."
        "1006568214637916988959772621094179308296422445989306234482518738419066591536864913120354471660949707150355"
        "5630856537290449867061502744883838098294987356841772087870736266964941742048310070386785976731864400067887"
        "9279212101247070555253372297017682453703596949849344521514456725725280415135690138675090474720908055796928"
        "6865392655183626402504160068929195404052734375e-134"},
    {0x1.fffffffffffffp-445, chars_format::scientific, 363,
        "2."
        "2013136429275833977919545242188358616592844891978612468965037476838133183073729826240708943321899414300711"
        "1261713074580899734123005489767676196589974713683544175741472533929883484096620140773571953463728800135775"
        "8558424202494141110506744594035364907407193899698689043028913451450560830271380277350180949441816111593857"
        "373078531036725280500832013785839080810546875e-134"},
    {0x1.fffffffffffffp-444, chars_format::scientific, 362,
        "4."
        "4026272858551667955839090484376717233185689783957224937930074953676266366147459652481417886643798828601422"
        "2523426149161799468246010979535352393179949427367088351482945067859766968193240281547143906927457600271551"
        "7116848404988282221013489188070729814814387799397378086057826902901121660542760554700361898883632223187714"
        "74615706207345056100166402757167816162109375e-134"},
    {0x1.fffffffffffffp-443, chars_format::scientific, 361,
        "8."
        "8052545717103335911678180968753434466371379567914449875860149907352532732294919304962835773287597657202844"
        "5046852298323598936492021959070704786359898854734176702965890135719533936386480563094287813854915200543103"
        "4233696809976564442026978376141459629628775598794756172115653805802243321085521109400723797767264446375429"
        "4923141241469011220033280551433563232421875e-134"},
    {0x1.fffffffffffffp-442, chars_format::scientific, 361,
        "1."
        "7610509143420667182335636193750686893274275913582889975172029981470506546458983860992567154657519531440568"
        "9009370459664719787298404391814140957271979770946835340593178027143906787277296112618857562770983040108620"
        "6846739361995312888405395675228291925925755119758951234423130761160448664217104221880144759553452889275085"
        "8984628248293802244006656110286712646484375e-133"},
    {0x1.fffffffffffffp-441, chars_format::scientific, 360,
        "3."
        "5221018286841334364671272387501373786548551827165779950344059962941013092917967721985134309315039062881137"
        "8018740919329439574596808783628281914543959541893670681186356054287813574554592225237715125541966080217241"
        "3693478723990625776810791350456583851851510239517902468846261522320897328434208443760289519106905778550171"
        "796925649658760448801331222057342529296875e-133"},
    {0x1.fffffffffffffp-440, chars_format::scientific, 359,
        "7."
        "0442036573682668729342544775002747573097103654331559900688119925882026185835935443970268618630078125762275"
        "6037481838658879149193617567256563829087919083787341362372712108575627149109184450475430251083932160434482"
        "7386957447981251553621582700913167703703020479035804937692523044641794656868416887520579038213811557100343"
        "59385129931752089760266244411468505859375e-133"},
    {0x1.fffffffffffffp-439, chars_format::scientific, 359,
        "1."
        "4088407314736533745868508955000549514619420730866311980137623985176405237167187088794053723726015625152455"
        "1207496367731775829838723513451312765817583816757468272474542421715125429821836890095086050216786432086896"
        "5477391489596250310724316540182633540740604095807160987538504608928358931373683377504115807642762311420068"
        "71877025986350417952053248882293701171875e-132"},
    {0x1.fffffffffffffp-438, chars_format::scientific, 358,
        "2."
        "8176814629473067491737017910001099029238841461732623960275247970352810474334374177588107447452031250304910"
        "2414992735463551659677447026902625531635167633514936544949084843430250859643673780190172100433572864173793"
        "0954782979192500621448633080365267081481208191614321975077009217856717862747366755008231615285524622840137"
        "4375405197270083590410649776458740234375e-132"},
    {0x1.fffffffffffffp-437, chars_format::scientific, 357,
        "5."
        "6353629258946134983474035820002198058477682923465247920550495940705620948668748355176214894904062500609820"
        "4829985470927103319354894053805251063270335267029873089898169686860501719287347560380344200867145728347586"
        "1909565958385001242897266160730534162962416383228643950154018435713435725494733510016463230571049245680274"
        "875081039454016718082129955291748046875e-132"},
    {0x1.fffffffffffffp-436, chars_format::scientific, 357,
        "1."
        "1270725851789226996694807164000439611695536584693049584110099188141124189733749671035242978980812500121964"
        "0965997094185420663870978810761050212654067053405974617979633937372100343857469512076068840173429145669517"
        "2381913191677000248579453232146106832592483276645728790030803687142687145098946702003292646114209849136054"
        "975016207890803343616425991058349609375e-131"},
    {0x1.fffffffffffffp-435, chars_format::scientific, 356,
        "2."
        "2541451703578453993389614328000879223391073169386099168220198376282248379467499342070485957961625000243928"
        "1931994188370841327741957621522100425308134106811949235959267874744200687714939024152137680346858291339034"
        "4763826383354000497158906464292213665184966553291457580061607374285374290197893404006585292228419698272109"
        "95003241578160668723285198211669921875e-131"},
    {0x1.fffffffffffffp-434, chars_format::scientific, 355,
        "4."
        "5082903407156907986779228656001758446782146338772198336440396752564496758934998684140971915923250000487856"
        "3863988376741682655483915243044200850616268213623898471918535749488401375429878048304275360693716582678068"
        "9527652766708000994317812928584427330369933106582915160123214748570748580395786808013170584456839396544219"
        "9000648315632133744657039642333984375e-131"},
    {0x1.fffffffffffffp-433, chars_format::scientific, 354,
        "9."
        "0165806814313815973558457312003516893564292677544396672880793505128993517869997368281943831846500000975712"
        "7727976753483365310967830486088401701232536427247796943837071498976802750859756096608550721387433165356137"
        "9055305533416001988635625857168854660739866213165830320246429497141497160791573616026341168913678793088439"
        "800129663126426748931407928466796875e-131"},
    {0x1.fffffffffffffp-432, chars_format::scientific, 354,
        "1."
        "8033161362862763194711691462400703378712858535508879334576158701025798703573999473656388766369300000195142"
        "5545595350696673062193566097217680340246507285449559388767414299795360550171951219321710144277486633071227"
        "5811061106683200397727125171433770932147973242633166064049285899428299432158314723205268233782735758617687"
        "960025932625285349786281585693359375e-130"},
    {0x1.fffffffffffffp-431, chars_format::scientific, 353,
        "3."
        "6066322725725526389423382924801406757425717071017758669152317402051597407147998947312777532738600000390285"
        "1091190701393346124387132194435360680493014570899118777534828599590721100343902438643420288554973266142455"
        "1622122213366400795454250342867541864295946485266332128098571798856598864316629446410536467565471517235375"
        "92005186525057069957256317138671875e-130"},
    {0x1.fffffffffffffp-430, chars_format::scientific, 352,
        "7."
        "2132645451451052778846765849602813514851434142035517338304634804103194814295997894625555065477200000780570"
        "2182381402786692248774264388870721360986029141798237555069657199181442200687804877286840577109946532284910"
        "3244244426732801590908500685735083728591892970532664256197143597713197728633258892821072935130943034470751"
        "8401037305011413991451263427734375e-130"},
    {0x1.fffffffffffffp-429, chars_format::scientific, 352,
        "1."
        "4426529090290210555769353169920562702970286828407103467660926960820638962859199578925111013095440000156114"
        "0436476280557338449754852877774144272197205828359647511013931439836288440137560975457368115421989306456982"
        "0648848885346560318181700137147016745718378594106532851239428719542639545726651778564214587026188606894150"
        "3680207461002282798290252685546875e-129"},
    {0x1.fffffffffffffp-428, chars_format::scientific, 351,
        "2."
        "8853058180580421111538706339841125405940573656814206935321853921641277925718399157850222026190880000312228"
        "0872952561114676899509705755548288544394411656719295022027862879672576880275121950914736230843978612913964"
        "1297697770693120636363400274294033491436757188213065702478857439085279091453303557128429174052377213788300"
        "736041492200456559658050537109375e-129"},
    {0x1.fffffffffffffp-427, chars_format::scientific, 350,
        "5."
        "7706116361160842223077412679682250811881147313628413870643707843282555851436798315700444052381760000624456"
        "1745905122229353799019411511096577088788823313438590044055725759345153760550243901829472461687957225827928"
        "2595395541386241272726800548588066982873514376426131404957714878170558182906607114256858348104754427576601"
        "47208298440091311931610107421875e-129"},
    {0x1.fffffffffffffp-426, chars_format::scientific, 350,
        "1."
        "1541223272232168444615482535936450162376229462725682774128741568656511170287359663140088810476352000124891"
        "2349181024445870759803882302219315417757764662687718008811145151869030752110048780365894492337591445165585"
        "6519079108277248254545360109717613396574702875285226280991542975634111636581321422851371669620950885515320"
        "29441659688018262386322021484375e-128"},
    {0x1.fffffffffffffp-425, chars_format::scientific, 349,
        "2."
        "3082446544464336889230965071872900324752458925451365548257483137313022340574719326280177620952704000249782"
        "4698362048891741519607764604438630835515529325375436017622290303738061504220097560731788984675182890331171"
        "3038158216554496509090720219435226793149405750570452561983085951268223273162642845702743339241901771030640"
        "5888331937603652477264404296875e-128"},
    {0x1.fffffffffffffp-424, chars_format::scientific, 348,
        "4."
        "6164893088928673778461930143745800649504917850902731096514966274626044681149438652560355241905408000499564"
        "9396724097783483039215529208877261671031058650750872035244580607476123008440195121463577969350365780662342"
        "6076316433108993018181440438870453586298811501140905123966171902536446546325285691405486678483803542061281"
        "177666387520730495452880859375e-128"},
    {0x1.fffffffffffffp-423, chars_format::scientific, 347,
        "9."
        "2329786177857347556923860287491601299009835701805462193029932549252089362298877305120710483810816000999129"
        "8793448195566966078431058417754523342062117301501744070489161214952246016880390242927155938700731561324685"
        "2152632866217986036362880877740907172597623002281810247932343805072893092650571382810973356967607084122562"
        "35533277504146099090576171875e-128"},
    {0x1.fffffffffffffp-422, chars_format::scientific, 347,
        "1."
        "8465957235571469511384772057498320259801967140361092438605986509850417872459775461024142096762163200199825"
        "9758689639113393215686211683550904668412423460300348814097832242990449203376078048585431187740146312264937"
        "0430526573243597207272576175548181434519524600456362049586468761014578618530114276562194671393521416824512"
        "47106655500829219818115234375e-127"},
    {0x1.fffffffffffffp-421, chars_format::scientific, 346,
        "3."
        "6931914471142939022769544114996640519603934280722184877211973019700835744919550922048284193524326400399651"
        "9517379278226786431372423367101809336824846920600697628195664485980898406752156097170862375480292624529874"
        "0861053146487194414545152351096362869039049200912724099172937522029157237060228553124389342787042833649024"
        "9421331100165843963623046875e-127"},
    {0x1.fffffffffffffp-420, chars_format::scientific, 345,
        "7."
        "3863828942285878045539088229993281039207868561444369754423946039401671489839101844096568387048652800799303"
        "9034758556453572862744846734203618673649693841201395256391328971961796813504312194341724750960585249059748"
        "1722106292974388829090304702192725738078098401825448198345875044058314474120457106248778685574085667298049"
        "884266220033168792724609375e-127"},
    {0x1.fffffffffffffp-419, chars_format::scientific, 345,
        "1."
        "4772765788457175609107817645998656207841573712288873950884789207880334297967820368819313677409730560159860"
        "7806951711290714572548969346840723734729938768240279051278265794392359362700862438868344950192117049811949"
        "6344421258594877765818060940438545147615619680365089639669175008811662894824091421249755737114817133459609"
        "976853244006633758544921875e-126"},
    {0x1.fffffffffffffp-418, chars_format::scientific, 344,
        "2."
        "9545531576914351218215635291997312415683147424577747901769578415760668595935640737638627354819461120319721"
        "5613903422581429145097938693681447469459877536480558102556531588784718725401724877736689900384234099623899"
        "2688842517189755531636121880877090295231239360730179279338350017623325789648182842499511474229634266919219"
        "95370648801326751708984375e-126"},
    {0x1.fffffffffffffp-417, chars_format::scientific, 343,
        "5."
        "9091063153828702436431270583994624831366294849155495803539156831521337191871281475277254709638922240639443"
        "1227806845162858290195877387362894938919755072961116205113063177569437450803449755473379800768468199247798"
        "5377685034379511063272243761754180590462478721460358558676700035246651579296365684999022948459268533838439"
        "9074129760265350341796875e-126"},
    {0x1.fffffffffffffp-416, chars_format::scientific, 343,
        "1."
        "1818212630765740487286254116798924966273258969831099160707831366304267438374256295055450941927784448127888"
        "6245561369032571658039175477472578987783951014592223241022612635513887490160689951094675960153693639849559"
        "7075537006875902212654448752350836118092495744292071711735340007049330315859273136999804589691853706767687"
        "9814825952053070068359375e-125"},
    {0x1.fffffffffffffp-415, chars_format::scientific, 342,
        "2."
        "3636425261531480974572508233597849932546517939662198321415662732608534876748512590110901883855568896255777"
        "2491122738065143316078350954945157975567902029184446482045225271027774980321379902189351920307387279699119"
        "4151074013751804425308897504701672236184991488584143423470680014098660631718546273999609179383707413535375"
        "962965190410614013671875e-125"},
    {0x1.fffffffffffffp-414, chars_format::scientific, 341,
        "4."
        "7272850523062961949145016467195699865093035879324396642831325465217069753497025180221803767711137792511554"
        "4982245476130286632156701909890315951135804058368892964090450542055549960642759804378703840614774559398238"
        "8302148027503608850617795009403344472369982977168286846941360028197321263437092547999218358767414827070751"
        "92593038082122802734375e-125"},
    {0x1.fffffffffffffp-413, chars_format::scientific, 340,
        "9."
        "4545701046125923898290032934391399730186071758648793285662650930434139506994050360443607535422275585023108"
        "9964490952260573264313403819780631902271608116737785928180901084111099921285519608757407681229549118796477"
        "6604296055007217701235590018806688944739965954336573693882720056394642526874185095998436717534829654141503"
        "8518607616424560546875e-125"},
    {0x1.fffffffffffffp-412, chars_format::scientific, 340,
        "1."
        "8909140209225184779658006586878279946037214351729758657132530186086827901398810072088721507084455117004621"
        "7992898190452114652862680763956126380454321623347557185636180216822219984257103921751481536245909823759295"
        "5320859211001443540247118003761337788947993190867314738776544011278928505374837019199687343506965930828300"
        "7703721523284912109375e-124"},
    {0x1.fffffffffffffp-411, chars_format::scientific, 339,
        "3."
        "7818280418450369559316013173756559892074428703459517314265060372173655802797620144177443014168910234009243"
        "5985796380904229305725361527912252760908643246695114371272360433644439968514207843502963072491819647518591"
        "0641718422002887080494236007522675577895986381734629477553088022557857010749674038399374687013931861656601"
        "540744304656982421875e-124"},
    {0x1.fffffffffffffp-410, chars_format::scientific, 338,
        "7."
        "5636560836900739118632026347513119784148857406919034628530120744347311605595240288354886028337820468018487"
        "1971592761808458611450723055824505521817286493390228742544720867288879937028415687005926144983639295037182"
        "1283436844005774160988472015045351155791972763469258955106176045115714021499348076798749374027863723313203"
        "08148860931396484375e-124"},
    {0x1.fffffffffffffp-409, chars_format::scientific, 338,
        "1."
        "5127312167380147823726405269502623956829771481383806925706024148869462321119048057670977205667564093603697"
        "4394318552361691722290144611164901104363457298678045748508944173457775987405683137401185228996727859007436"
        "4256687368801154832197694403009070231158394552693851791021235209023142804299869615359749874805572744662640"
        "61629772186279296875e-123"},
    {0x1.fffffffffffffp-408, chars_format::scientific, 337,
        "3."
        "0254624334760295647452810539005247913659542962767613851412048297738924642238096115341954411335128187207394"
        "8788637104723383444580289222329802208726914597356091497017888346915551974811366274802370457993455718014872"
        "8513374737602309664395388806018140462316789105387703582042470418046285608599739230719499749611145489325281"
        "2325954437255859375e-123"},
    {0x1.fffffffffffffp-407, chars_format::scientific, 336,
        "6."
        "0509248669520591294905621078010495827319085925535227702824096595477849284476192230683908822670256374414789"
        "7577274209446766889160578444659604417453829194712182994035776693831103949622732549604740915986911436029745"
        "7026749475204619328790777612036280924633578210775407164084940836092571217199478461438999499222290978650562"
        "465190887451171875e-123"},
    {0x1.fffffffffffffp-406, chars_format::scientific, 336,
        "1."
        "2101849733904118258981124215602099165463817185107045540564819319095569856895238446136781764534051274882957"
        "9515454841889353377832115688931920883490765838942436598807155338766220789924546509920948183197382287205949"
        "1405349895040923865758155522407256184926715642155081432816988167218514243439895692287799899844458195730112"
        "493038177490234375e-122"},
    {0x1.fffffffffffffp-405, chars_format::scientific, 335,
        "2."
        "4203699467808236517962248431204198330927634370214091081129638638191139713790476892273563529068102549765915"
        "9030909683778706755664231377863841766981531677884873197614310677532441579849093019841896366394764574411898"
        "2810699790081847731516311044814512369853431284310162865633976334437028486879791384575599799688916391460224"
        "98607635498046875e-122"},
    {0x1.fffffffffffffp-404, chars_format::scientific, 334,
        "4."
        "8407398935616473035924496862408396661855268740428182162259277276382279427580953784547127058136205099531831"
        "8061819367557413511328462755727683533963063355769746395228621355064883159698186039683792732789529148823796"
        "5621399580163695463032622089629024739706862568620325731267952668874056973759582769151199599377832782920449"
        "9721527099609375e-122"},
    {0x1.fffffffffffffp-403, chars_format::scientific, 333,
        "9."
        "6814797871232946071848993724816793323710537480856364324518554552764558855161907569094254116272410199063663"
        "6123638735114827022656925511455367067926126711539492790457242710129766319396372079367585465579058297647593"
        "1242799160327390926065244179258049479413725137240651462535905337748113947519165538302399198755665565840899"
        "944305419921875e-122"},
    {0x1.fffffffffffffp-402, chars_format::scientific, 333,
        "1."
        "9362959574246589214369798744963358664742107496171272864903710910552911771032381513818850823254482039812732"
        "7224727747022965404531385102291073413585225342307898558091448542025953263879274415873517093115811659529518"
        "6248559832065478185213048835851609895882745027448130292507181067549622789503833107660479839751133113168179"
        "988861083984375e-121"},
    {0x1.fffffffffffffp-401, chars_format::scientific, 332,
        "3."
        "8725919148493178428739597489926717329484214992342545729807421821105823542064763027637701646508964079625465"
        "4449455494045930809062770204582146827170450684615797116182897084051906527758548831747034186231623319059037"
        "2497119664130956370426097671703219791765490054896260585014362135099245579007666215320959679502266226336359"
        "97772216796875e-121"},
    {0x1.fffffffffffffp-400, chars_format::scientific, 331,
        "7."
        "7451838296986356857479194979853434658968429984685091459614843642211647084129526055275403293017928159250930"
        "8898910988091861618125540409164293654340901369231594232365794168103813055517097663494068372463246638118074"
        "4994239328261912740852195343406439583530980109792521170028724270198491158015332430641919359004532452672719"
        "9554443359375e-121"},
    {0x1.fffffffffffffp-399, chars_format::scientific, 331,
        "1."
        "5490367659397271371495838995970686931793685996937018291922968728442329416825905211055080658603585631850186"
        "1779782197618372323625108081832858730868180273846318846473158833620762611103419532698813674492649327623614"
        "8998847865652382548170439068681287916706196021958504234005744854039698231603066486128383871800906490534543"
        "9910888671875e-120"},
    {0x1.fffffffffffffp-398, chars_format::scientific, 330,
        "3."
        "0980735318794542742991677991941373863587371993874036583845937456884658833651810422110161317207171263700372"
        "3559564395236744647250216163665717461736360547692637692946317667241525222206839065397627348985298655247229"
        "7997695731304765096340878137362575833412392043917008468011489708079396463206132972256767743601812981069087"
        "982177734375e-120"},
    {0x1.fffffffffffffp-397, chars_format::scientific, 329,
        "6."
        "1961470637589085485983355983882747727174743987748073167691874913769317667303620844220322634414342527400744"
        "7119128790473489294500432327331434923472721095385275385892635334483050444413678130795254697970597310494459"
        "5995391462609530192681756274725151666824784087834016936022979416158792926412265944513535487203625962138175"
        "96435546875e-120"},
    {0x1.fffffffffffffp-396, chars_format::scientific, 329,
        "1."
        "2392294127517817097196671196776549545434948797549614633538374982753863533460724168844064526882868505480148"
        "9423825758094697858900086465466286984694544219077055077178527066896610088882735626159050939594119462098891"
        "9199078292521906038536351254945030333364956817566803387204595883231758585282453188902707097440725192427635"
        "19287109375e-119"},
    {0x1.fffffffffffffp-395, chars_format::scientific, 328,
        "2."
        "4784588255035634194393342393553099090869897595099229267076749965507727066921448337688129053765737010960297"
        "8847651516189395717800172930932573969389088438154110154357054133793220177765471252318101879188238924197783"
        "8398156585043812077072702509890060666729913635133606774409191766463517170564906377805414194881450384855270"
        "3857421875e-119"},
    {0x1.fffffffffffffp-394, chars_format::scientific, 327,
        "4."
        "9569176510071268388786684787106198181739795190198458534153499931015454133842896675376258107531474021920595"
        "7695303032378791435600345861865147938778176876308220308714108267586440355530942504636203758376477848395567"
        "6796313170087624154145405019780121333459827270267213548818383532927034341129812755610828389762900769710540"
        "771484375e-119"},
    {0x1.fffffffffffffp-393, chars_format::scientific, 326,
        "9."
        "9138353020142536777573369574212396363479590380396917068306999862030908267685793350752516215062948043841191"
        "5390606064757582871200691723730295877556353752616440617428216535172880711061885009272407516752955696791135"
        "3592626340175248308290810039560242666919654540534427097636767065854068682259625511221656779525801539421081"
        "54296875e-119"},
    {0x1.fffffffffffffp-392, chars_format::scientific, 326,
        "1."
        "9827670604028507355514673914842479272695918076079383413661399972406181653537158670150503243012589608768238"
        "3078121212951516574240138344746059175511270750523288123485643307034576142212377001854481503350591139358227"
        "0718525268035049661658162007912048533383930908106885419527353413170813736451925102244331355905160307884216"
        "30859375e-118"},
    {0x1.fffffffffffffp-391, chars_format::scientific, 325,
        "3."
        "9655341208057014711029347829684958545391836152158766827322799944812363307074317340301006486025179217536476"
        "6156242425903033148480276689492118351022541501046576246971286614069152284424754003708963006701182278716454"
        "1437050536070099323316324015824097066767861816213770839054706826341627472903850204488662711810320615768432"
        "6171875e-118"},
    {0x1.fffffffffffffp-390, chars_format::scientific, 324,
        "7."
        "9310682416114029422058695659369917090783672304317533654645599889624726614148634680602012972050358435072953"
        "2312484851806066296960553378984236702045083002093152493942573228138304568849508007417926013402364557432908"
        "2874101072140198646632648031648194133535723632427541678109413652683254945807700408977325423620641231536865"
        "234375e-118"},
    {0x1.fffffffffffffp-389, chars_format::scientific, 324,
        "1."
        "5862136483222805884411739131873983418156734460863506730929119977924945322829726936120402594410071687014590"
        "6462496970361213259392110675796847340409016600418630498788514645627660913769901601483585202680472911486581"
        "6574820214428039729326529606329638826707144726485508335621882730536650989161540081795465084724128246307373"
        "046875e-117"},
    {0x1.fffffffffffffp-388, chars_format::scientific, 323,
        "3."
        "1724272966445611768823478263747966836313468921727013461858239955849890645659453872240805188820143374029181"
        "2924993940722426518784221351593694680818033200837260997577029291255321827539803202967170405360945822973163"
        "3149640428856079458653059212659277653414289452971016671243765461073301978323080163590930169448256492614746"
        "09375e-117"},
    {0x1.fffffffffffffp-387, chars_format::scientific, 322,
        "6."
        "3448545932891223537646956527495933672626937843454026923716479911699781291318907744481610377640286748058362"
        "5849987881444853037568442703187389361636066401674521995154058582510643655079606405934340810721891645946326"
        "6299280857712158917306118425318555306828578905942033342487530922146603956646160327181860338896512985229492"
        "1875e-117"},
    {0x1.fffffffffffffp-386, chars_format::scientific, 322,
        "1."
        "2689709186578244707529391305499186734525387568690805384743295982339956258263781548896322075528057349611672"
        "5169997576288970607513688540637477872327213280334904399030811716502128731015921281186868162144378329189265"
        "3259856171542431783461223685063711061365715781188406668497506184429320791329232065436372067779302597045898"
        "4375e-116"},
    {0x1.fffffffffffffp-385, chars_format::scientific, 321,
        "2."
        "5379418373156489415058782610998373469050775137381610769486591964679912516527563097792644151056114699223345"
        "0339995152577941215027377081274955744654426560669808798061623433004257462031842562373736324288756658378530"
        "6519712343084863566922447370127422122731431562376813336995012368858641582658464130872744135558605194091796"
        "875e-116"},
    {0x1.fffffffffffffp-384, chars_format::scientific, 320,
        "5."
        "0758836746312978830117565221996746938101550274763221538973183929359825033055126195585288302112229398446690"
        "0679990305155882430054754162549911489308853121339617596123246866008514924063685124747472648577513316757061"
        "3039424686169727133844894740254844245462863124753626673990024737717283165316928261745488271117210388183593"
        "75e-116"},
    {0x1.fffffffffffffp-383, chars_format::scientific, 320,
        "1."
        "0151767349262595766023513044399349387620310054952644307794636785871965006611025239117057660422445879689338"
        "0135998061031176486010950832509982297861770624267923519224649373201702984812737024949494529715502663351412"
        "2607884937233945426768978948050968849092572624950725334798004947543456633063385652349097654223442077636718"
        "75e-115"},
    {0x1.fffffffffffffp-382, chars_format::scientific, 319,
        "2."
        "0303534698525191532047026088798698775240620109905288615589273571743930013222050478234115320844891759378676"
        "0271996122062352972021901665019964595723541248535847038449298746403405969625474049898989059431005326702824"
        "5215769874467890853537957896101937698185145249901450669596009895086913266126771304698195308446884155273437"
        "5e-115"},
    {0x1.fffffffffffffp-381, chars_format::scientific, 318,
        "4."
        "0607069397050383064094052177597397550481240219810577231178547143487860026444100956468230641689783518757352"
        "0543992244124705944043803330039929191447082497071694076898597492806811939250948099797978118862010653405649"
        "0431539748935781707075915792203875396370290499802901339192019790173826532253542609396390616893768310546875"
        "e-115"},
    {0x1.fffffffffffffp-380, chars_format::scientific, 317,
        "8."
        "1214138794100766128188104355194795100962480439621154462357094286975720052888201912936461283379567037514704"
        "1087984488249411888087606660079858382894164994143388153797194985613623878501896199595956237724021306811298"
        "086307949787156341415183158440775079274058099960580267838403958034765306450708521879278123378753662109375e"
        "-115"},
    {0x1.fffffffffffffp-379, chars_format::scientific, 317,
        "1."
        "6242827758820153225637620871038959020192496087924230892471418857395144010577640382587292256675913407502940"
        "8217596897649882377617521332015971676578832998828677630759438997122724775700379239919191247544804261362259"
        "617261589957431268283036631688155015854811619992116053567680791606953061290141704375855624675750732421875e"
        "-114"},
    {0x1.fffffffffffffp-378, chars_format::scientific, 316,
        "3."
        "2485655517640306451275241742077918040384992175848461784942837714790288021155280765174584513351826815005881"
        "6435193795299764755235042664031943353157665997657355261518877994245449551400758479838382495089608522724519"
        "23452317991486253656607326337631003170962323998423210713536158321390612258028340875171124935150146484375e-"
        "114"},
    {0x1.fffffffffffffp-377, chars_format::scientific, 315,
        "6."
        "4971311035280612902550483484155836080769984351696923569885675429580576042310561530349169026703653630011763"
        "2870387590599529510470085328063886706315331995314710523037755988490899102801516959676764990179217045449038"
        "4690463598297250731321465267526200634192464799684642142707231664278122451605668175034224987030029296875e-"
        "114"},
    {0x1.fffffffffffffp-376, chars_format::scientific, 315,
        "1."
        "2994262207056122580510096696831167216153996870339384713977135085916115208462112306069833805340730726002352"
        "6574077518119905902094017065612777341263066399062942104607551197698179820560303391935352998035843409089807"
        "6938092719659450146264293053505240126838492959936928428541446332855624490321133635006844997406005859375e-"
        "113"},
    {0x1.fffffffffffffp-375, chars_format::scientific, 314,
        "2."
        "5988524414112245161020193393662334432307993740678769427954270171832230416924224612139667610681461452004705"
        "3148155036239811804188034131225554682526132798125884209215102395396359641120606783870705996071686818179615"
        "387618543931890029252858610701048025367698591987385685708289266571124898064226727001368999481201171875e-"
        "113"},
    {0x1.fffffffffffffp-374, chars_format::scientific, 313,
        "5."
        "1977048828224490322040386787324668864615987481357538855908540343664460833848449224279335221362922904009410"
        "6296310072479623608376068262451109365052265596251768418430204790792719282241213567741411992143373636359230"
        "77523708786378005850571722140209605073539718397477137141657853314224979612845345400273799896240234375e-"
        "113"},
    {0x1.fffffffffffffp-373, chars_format::scientific, 313,
        "1."
        "0395409765644898064408077357464933772923197496271507771181708068732892166769689844855867044272584580801882"
        "1259262014495924721675213652490221873010453119250353683686040958158543856448242713548282398428674727271846"
        "15504741757275601170114344428041921014707943679495427428331570662844995922569069080054759979248046875e-"
        "112"},
    {0x1.fffffffffffffp-372, chars_format::scientific, 312,
        "2."
        "0790819531289796128816154714929867545846394992543015542363416137465784333539379689711734088545169161603764"
        "2518524028991849443350427304980443746020906238500707367372081916317087712896485427096564796857349454543692"
        "3100948351455120234022868885608384202941588735899085485666314132568999184513813816010951995849609375e-"
        "112"},
    {0x1.fffffffffffffp-371, chars_format::scientific, 311,
        "4."
        "1581639062579592257632309429859735091692789985086031084726832274931568667078759379423468177090338323207528"
        "5037048057983698886700854609960887492041812477001414734744163832634175425792970854193129593714698909087384"
        "620189670291024046804573777121676840588317747179817097133262826513799836902762763202190399169921875e-112"},
    {0x1.fffffffffffffp-370, chars_format::scientific, 310,
        "8."
        "3163278125159184515264618859719470183385579970172062169453664549863137334157518758846936354180676646415057"
        "0074096115967397773401709219921774984083624954002829469488327665268350851585941708386259187429397818174769"
        "24037934058204809360914755424335368117663549435963419426652565302759967380552552640438079833984375e-112"},
    {0x1.fffffffffffffp-369, chars_format::scientific, 310,
        "1."
        "6632655625031836903052923771943894036677115994034412433890732909972627466831503751769387270836135329283011"
        "4014819223193479554680341843984354996816724990800565893897665533053670170317188341677251837485879563634953"
        "84807586811640961872182951084867073623532709887192683885330513060551993476110510528087615966796875e-111"},
    {0x1.fffffffffffffp-368, chars_format::scientific, 309,
        "3."
        "3265311250063673806105847543887788073354231988068824867781465819945254933663007503538774541672270658566022"
        "8029638446386959109360683687968709993633449981601131787795331066107340340634376683354503674971759127269907"
        "6961517362328192374436590216973414724706541977438536777066102612110398695222102105617523193359375e-111"},
    {0x1.fffffffffffffp-367, chars_format::scientific, 308,
        "6."
        "6530622500127347612211695087775576146708463976137649735562931639890509867326015007077549083344541317132045"
        "6059276892773918218721367375937419987266899963202263575590662132214680681268753366709007349943518254539815"
        "392303472465638474887318043394682944941308395487707355413220522422079739044420421123504638671875e-111"},
    {0x1.fffffffffffffp-366, chars_format::scientific, 308,
        "1."
        "3306124500025469522442339017555115229341692795227529947112586327978101973465203001415509816668908263426409"
        "1211855378554783643744273475187483997453379992640452715118132426442936136253750673341801469988703650907963"
        "078460694493127694977463608678936588988261679097541471082644104484415947808884084224700927734375e-110"},
    {0x1.fffffffffffffp-365, chars_format::scientific, 307,
        "2."
        "6612249000050939044884678035110230458683385590455059894225172655956203946930406002831019633337816526852818"
        "2423710757109567287488546950374967994906759985280905430236264852885872272507501346683602939977407301815926"
        "15692138898625538995492721735787317797652335819508294216528820896883189561776816844940185546875e-110"},
    {0x1.fffffffffffffp-364, chars_format::scientific, 306,
        "5."
        "3224498000101878089769356070220460917366771180910119788450345311912407893860812005662039266675633053705636"
        "4847421514219134574977093900749935989813519970561810860472529705771744545015002693367205879954814603631852"
        "3138427779725107799098544347157463559530467163901658843305764179376637912355363368988037109375e-110"},
    {0x1.fffffffffffffp-363, chars_format::scientific, 306,
        "1."
        "0644899600020375617953871214044092183473354236182023957690069062382481578772162401132407853335126610741127"
        "2969484302843826914995418780149987197962703994112362172094505941154348909003000538673441175990962920726370"
        "4627685555945021559819708869431492711906093432780331768661152835875327582471072673797607421875e-109"},
    {0x1.fffffffffffffp-362, chars_format::scientific, 305,
        "2."
        "1289799200040751235907742428088184366946708472364047915380138124764963157544324802264815706670253221482254"
        "5938968605687653829990837560299974395925407988224724344189011882308697818006001077346882351981925841452740"
        "925537111189004311963941773886298542381218686556066353732230567175065516494214534759521484375e-109"},
    {0x1.fffffffffffffp-361, chars_format::scientific, 304,
        "4."
        "2579598400081502471815484856176368733893416944728095830760276249529926315088649604529631413340506442964509"
        "1877937211375307659981675120599948791850815976449448688378023764617395636012002154693764703963851682905481"
        "85107422237800862392788354777259708476243737311213270746446113435013103298842906951904296875e-109"},
    {0x1.fffffffffffffp-360, chars_format::scientific, 303,
        "8."
        "5159196800163004943630969712352737467786833889456191661520552499059852630177299209059262826681012885929018"
        "3755874422750615319963350241199897583701631952898897376756047529234791272024004309387529407927703365810963"
        "7021484447560172478557670955451941695248747462242654149289222687002620659768581390380859375e-109"},
    {0x1.fffffffffffffp-359, chars_format::scientific, 303,
        "1."
        "7031839360032600988726193942470547493557366777891238332304110499811970526035459841811852565336202577185803"
        "6751174884550123063992670048239979516740326390579779475351209505846958254404800861877505881585540673162192"
        "7404296889512034495711534191090388339049749492448530829857844537400524131953716278076171875e-108"},
    {0x1.fffffffffffffp-358, chars_format::scientific, 302,
        "3."
        "4063678720065201977452387884941094987114733555782476664608220999623941052070919683623705130672405154371607"
        "3502349769100246127985340096479959033480652781159558950702419011693916508809601723755011763171081346324385"
        "480859377902406899142306838218077667809949898489706165971568907480104826390743255615234375e-108"},
    {0x1.fffffffffffffp-357, chars_format::scientific, 301,
        "6."
        "8127357440130403954904775769882189974229467111564953329216441999247882104141839367247410261344810308743214"
        "7004699538200492255970680192959918066961305562319117901404838023387833017619203447510023526342162692648770"
        "96171875580481379828461367643615533561989979697941233194313781496020965278148651123046875e-108"},
    {0x1.fffffffffffffp-356, chars_format::scientific, 301,
        "1."
        "3625471488026080790980955153976437994845893422312990665843288399849576420828367873449482052268962061748642"
        "9400939907640098451194136038591983613392261112463823580280967604677566603523840689502004705268432538529754"
        "19234375116096275965692273528723106712397995939588246638862756299204193055629730224609375e-107"},
    {0x1.fffffffffffffp-355, chars_format::scientific, 300,
        "2."
        "7250942976052161581961910307952875989691786844625981331686576799699152841656735746898964104537924123497285"
        "8801879815280196902388272077183967226784522224927647160561935209355133207047681379004009410536865077059508"
        "3846875023219255193138454705744621342479599187917649327772551259840838611125946044921875e-107"},
    {0x1.fffffffffffffp-354, chars_format::scientific, 299,
        "5."
        "4501885952104323163923820615905751979383573689251962663373153599398305683313471493797928209075848246994571"
        "7603759630560393804776544154367934453569044449855294321123870418710266414095362758008018821073730154119016"
        "769375004643851038627690941148924268495919837583529865554510251968167722225189208984375e-107"},
    {0x1.fffffffffffffp-353, chars_format::scientific, 299,
        "1."
        "0900377190420864632784764123181150395876714737850392532674630719879661136662694298759585641815169649398914"
        "3520751926112078760955308830873586890713808889971058864224774083742053282819072551601603764214746030823803"
        "353875000928770207725538188229784853699183967516705973110902050393633544445037841796875e-106"},
    {0x1.fffffffffffffp-352, chars_format::scientific, 298,
        "2."
        "1800754380841729265569528246362300791753429475700785065349261439759322273325388597519171283630339298797828"
        "7041503852224157521910617661747173781427617779942117728449548167484106565638145103203207528429492061647606"
        "70775000185754041545107637645956970739836793503341194622180410078726708889007568359375e-106"},
    {0x1.fffffffffffffp-351, chars_format::scientific, 297,
        "4."
        "3601508761683458531139056492724601583506858951401570130698522879518644546650777195038342567260678597595657"
        "4083007704448315043821235323494347562855235559884235456899096334968213131276290206406415056858984123295213"
        "4155000037150808309021527529191394147967358700668238924436082015745341777801513671875e-106"},
    {0x1.fffffffffffffp-350, chars_format::scientific, 296,
        "8."
        "7203017523366917062278112985449203167013717902803140261397045759037289093301554390076685134521357195191314"
        "8166015408896630087642470646988695125710471119768470913798192669936426262552580412812830113717968246590426"
        "831000007430161661804305505838278829593471740133647784887216403149068355560302734375e-106"},
    {0x1.fffffffffffffp-349, chars_format::scientific, 296,
        "1."
        "7440603504673383412455622597089840633402743580560628052279409151807457818660310878015337026904271439038262"
        "9633203081779326017528494129397739025142094223953694182759638533987285252510516082562566022743593649318085"
        "366200001486032332360861101167655765918694348026729556977443280629813671112060546875e-105"},
    {0x1.fffffffffffffp-348, chars_format::scientific, 295,
        "3."
        "4881207009346766824911245194179681266805487161121256104558818303614915637320621756030674053808542878076525"
        "9266406163558652035056988258795478050284188447907388365519277067974570505021032165125132045487187298636170"
        "73240000297206466472172220233531153183738869605345911395488656125962734222412109375e-105"},
    {0x1.fffffffffffffp-347, chars_format::scientific, 294,
        "6."
        "9762414018693533649822490388359362533610974322242512209117636607229831274641243512061348107617085756153051"
        "8532812327117304070113976517590956100568376895814776731038554135949141010042064330250264090974374597272341"
        "4648000059441293294434444046706230636747773921069182279097731225192546844482421875e-105"},
    {0x1.fffffffffffffp-346, chars_format::scientific, 294,
        "1."
        "3952482803738706729964498077671872506722194864448502441823527321445966254928248702412269621523417151230610"
        "3706562465423460814022795303518191220113675379162955346207710827189828202008412866050052818194874919454468"
        "2929600011888258658886888809341246127349554784213836455819546245038509368896484375e-104"},
    {0x1.fffffffffffffp-345, chars_format::scientific, 293,
        "2."
        "7904965607477413459928996155343745013444389728897004883647054642891932509856497404824539243046834302461220"
        "7413124930846921628045590607036382440227350758325910692415421654379656404016825732100105636389749838908936"
        "585920002377651731777377761868249225469910956842767291163909249007701873779296875e-104"},
    {0x1.fffffffffffffp-344, chars_format::scientific, 292,
        "5."
        "5809931214954826919857992310687490026888779457794009767294109285783865019712994809649078486093668604922441"
        "4826249861693843256091181214072764880454701516651821384830843308759312808033651464200211272779499677817873"
        "17184000475530346355475552373649845093982191368553458232781849801540374755859375e-104"},
    {0x1.fffffffffffffp-343, chars_format::scientific, 292,
        "1."
        "1161986242990965383971598462137498005377755891558801953458821857156773003942598961929815697218733720984488"
        "2965249972338768651218236242814552976090940303330364276966168661751862561606730292840042254555899935563574"
        "63436800095106069271095110474729969018796438273710691646556369960308074951171875e-103"},
    {0x1.fffffffffffffp-342, chars_format::scientific, 291,
        "2."
        "2323972485981930767943196924274996010755511783117603906917643714313546007885197923859631394437467441968976"
        "5930499944677537302436472485629105952181880606660728553932337323503725123213460585680084509111799871127149"
        "2687360019021213854219022094945993803759287654742138329311273992061614990234375e-103"},
    {0x1.fffffffffffffp-341, chars_format::scientific, 290,
        "4."
        "4647944971963861535886393848549992021511023566235207813835287428627092015770395847719262788874934883937953"
        "1860999889355074604872944971258211904363761213321457107864674647007450246426921171360169018223599742254298"
        "537472003804242770843804418989198760751857530948427665862254798412322998046875e-103"},
    {0x1.fffffffffffffp-340, chars_format::scientific, 289,
        "8."
        "9295889943927723071772787697099984043022047132470415627670574857254184031540791695438525577749869767875906"
        "3721999778710149209745889942516423808727522426642914215729349294014900492853842342720338036447199484508597"
        "07494400760848554168760883797839752150371506189685533172450959682464599609375e-103"},
    {0x1.fffffffffffffp-339, chars_format::scientific, 289,
        "1."
        "7859177988785544614354557539419996808604409426494083125534114971450836806308158339087705115549973953575181"
        "2744399955742029841949177988503284761745504485328582843145869858802980098570768468544067607289439896901719"
        "41498880152169710833752176759567950430074301237937106634490191936492919921875e-102"},
    {0x1.fffffffffffffp-338, chars_format::scientific, 288,
        "3."
        "5718355977571089228709115078839993617208818852988166251068229942901673612616316678175410231099947907150362"
        "5488799911484059683898355977006569523491008970657165686291739717605960197141536937088135214578879793803438"
        "8299776030433942166750435351913590086014860247587421326898038387298583984375e-102"},
    {0x1.fffffffffffffp-337, chars_format::scientific, 287,
        "7."
        "1436711955142178457418230157679987234417637705976332502136459885803347225232633356350820462199895814300725"
        "0977599822968119367796711954013139046982017941314331372583479435211920394283073874176270429157759587606877"
        "659955206086788433350087070382718017202972049517484265379607677459716796875e-102"},
    {0x1.fffffffffffffp-336, chars_format::scientific, 287,
        "1."
        "4287342391028435691483646031535997446883527541195266500427291977160669445046526671270164092439979162860145"
        "0195519964593623873559342390802627809396403588262866274516695887042384078856614774835254085831551917521375"
        "531991041217357686670017414076543603440594409903496853075921535491943359375e-101"},
    {0x1.fffffffffffffp-335, chars_format::scientific, 286,
        "2."
        "8574684782056871382967292063071994893767055082390533000854583954321338890093053342540328184879958325720290"
        "0391039929187247747118684781605255618792807176525732549033391774084768157713229549670508171663103835042751"
        "06398208243471537334003482815308720688118881980699370615184307098388671875e-101"},
    {0x1.fffffffffffffp-334, chars_format::scientific, 285,
        "5."
        "7149369564113742765934584126143989787534110164781066001709167908642677780186106685080656369759916651440580"
        "0782079858374495494237369563210511237585614353051465098066783548169536315426459099341016343326207670085502"
        "1279641648694307466800696563061744137623776396139874123036861419677734375e-101"},
    {0x1.fffffffffffffp-333, chars_format::scientific, 285,
        "1."
        "1429873912822748553186916825228797957506822032956213200341833581728535556037221337016131273951983330288116"
        "0156415971674899098847473912642102247517122870610293019613356709633907263085291819868203268665241534017100"
        "4255928329738861493360139312612348827524755279227974824607372283935546875e-100"},
    {0x1.fffffffffffffp-332, chars_format::scientific, 284,
        "2."
        "2859747825645497106373833650457595915013644065912426400683667163457071112074442674032262547903966660576232"
        "0312831943349798197694947825284204495034245741220586039226713419267814526170583639736406537330483068034200"
        "851185665947772298672027862522469765504951055845594964921474456787109375e-100"},
    {0x1.fffffffffffffp-331, chars_format::scientific, 283,
        "4."
        "5719495651290994212747667300915191830027288131824852801367334326914142224148885348064525095807933321152464"
        "0625663886699596395389895650568408990068491482441172078453426838535629052341167279472813074660966136068401"
        "70237133189554459734405572504493953100990211169118992984294891357421875e-100"},
    {0x1.fffffffffffffp-330, chars_format::scientific, 282,
        "9."
        "1438991302581988425495334601830383660054576263649705602734668653828284448297770696129050191615866642304928"
        "1251327773399192790779791301136817980136982964882344156906853677071258104682334558945626149321932272136803"
        "4047426637910891946881114500898790620198042233823798596858978271484375e-100"},
    {0x1.fffffffffffffp-329, chars_format::scientific, 282,
        "1."
        "8287798260516397685099066920366076732010915252729941120546933730765656889659554139225810038323173328460985"
        "6250265554679838558155958260227363596027396592976468831381370735414251620936466911789125229864386454427360"
        "6809485327582178389376222900179758124039608446764759719371795654296875e-99"},
    {0x1.fffffffffffffp-328, chars_format::scientific, 281,
        "3."
        "6575596521032795370198133840732153464021830505459882241093867461531313779319108278451620076646346656921971"
        "2500531109359677116311916520454727192054793185952937662762741470828503241872933823578250459728772908854721"
        "361897065516435677875244580035951624807921689352951943874359130859375e-99"},
    {0x1.fffffffffffffp-327, chars_format::scientific, 280,
        "7."
        "3151193042065590740396267681464306928043661010919764482187734923062627558638216556903240153292693313843942"
        "5001062218719354232623833040909454384109586371905875325525482941657006483745867647156500919457545817709442"
        "72379413103287135575048916007190324961584337870590388774871826171875e-99"},
    {0x1.fffffffffffffp-326, chars_format::scientific, 280,
        "1."
        "4630238608413118148079253536292861385608732202183952896437546984612525511727643311380648030658538662768788"
        "5000212443743870846524766608181890876821917274381175065105096588331401296749173529431300183891509163541888"
        "54475882620657427115009783201438064992316867574118077754974365234375e-98"},
    {0x1.fffffffffffffp-325, chars_format::scientific, 279,
        "2."
        "9260477216826236296158507072585722771217464404367905792875093969225051023455286622761296061317077325537577"
        "0000424887487741693049533216363781753643834548762350130210193176662802593498347058862600367783018327083777"
        "0895176524131485423001956640287612998463373514823615550994873046875e-98"},
    {0x1.fffffffffffffp-324, chars_format::scientific, 278,
        "5."
        "8520954433652472592317014145171445542434928808735811585750187938450102046910573245522592122634154651075154"
        "0000849774975483386099066432727563507287669097524700260420386353325605186996694117725200735566036654167554"
        "179035304826297084600391328057522599692674702964723110198974609375e-98"},
    {0x1.fffffffffffffp-323, chars_format::scientific, 278,
        "1."
        "1704190886730494518463402829034289108486985761747162317150037587690020409382114649104518424526830930215030"
        "8000169954995096677219813286545512701457533819504940052084077270665121037399338823545040147113207330833510"
        "835807060965259416920078265611504519938534940592944622039794921875e-97"},
    {0x1.fffffffffffffp-322, chars_format::scientific, 277,
        "2."
        "3408381773460989036926805658068578216973971523494324634300075175380040818764229298209036849053661860430061"
        "6000339909990193354439626573091025402915067639009880104168154541330242074798677647090080294226414661667021"
        "67161412193051883384015653122300903987706988118588924407958984375e-97"},
    {0x1.fffffffffffffp-321, chars_format::scientific, 276,
        "4."
        "6816763546921978073853611316137156433947943046988649268600150350760081637528458596418073698107323720860123"
        "2000679819980386708879253146182050805830135278019760208336309082660484149597355294180160588452829323334043"
        "3432282438610376676803130624460180797541397623717784881591796875e-97"},
    {0x1.fffffffffffffp-320, chars_format::scientific, 275,
        "9."
        "3633527093843956147707222632274312867895886093977298537200300701520163275056917192836147396214647441720246"
        "4001359639960773417758506292364101611660270556039520416672618165320968299194710588360321176905658646668086"
        "686456487722075335360626124892036159508279524743556976318359375e-97"},
    {0x1.fffffffffffffp-319, chars_format::scientific, 275,
        "1."
        "8726705418768791229541444526454862573579177218795459707440060140304032655011383438567229479242929488344049"
        "2800271927992154683551701258472820322332054111207904083334523633064193659838942117672064235381131729333617"
        "337291297544415067072125224978407231901655904948711395263671875e-96"},
    {0x1.fffffffffffffp-318, chars_format::scientific, 274,
        "3."
        "7453410837537582459082889052909725147158354437590919414880120280608065310022766877134458958485858976688098"
        "5600543855984309367103402516945640644664108222415808166669047266128387319677884235344128470762263458667234"
        "67458259508883013414425044995681446380331180989742279052734375e-96"},
    {0x1.fffffffffffffp-317, chars_format::scientific, 273,
        "7."
        "4906821675075164918165778105819450294316708875181838829760240561216130620045533754268917916971717953376197"
        "1201087711968618734206805033891281289328216444831616333338094532256774639355768470688256941524526917334469"
        "3491651901776602682885008999136289276066236197948455810546875e-96"},
    {0x1.fffffffffffffp-316, chars_format::scientific, 273,
        "1."
        "4981364335015032983633155621163890058863341775036367765952048112243226124009106750853783583394343590675239"
        "4240217542393723746841361006778256257865643288966323266667618906451354927871153694137651388304905383466893"
        "8698330380355320536577001799827257855213247239589691162109375e-95"},
    {0x1.fffffffffffffp-315, chars_format::scientific, 272,
        "2."
        "9962728670030065967266311242327780117726683550072735531904096224486452248018213501707567166788687181350478"
        "8480435084787447493682722013556512515731286577932646533335237812902709855742307388275302776609810766933787"
        "739666076071064107315400359965451571042649447917938232421875e-95"},
    {0x1.fffffffffffffp-314, chars_format::scientific, 271,
        "5."
        "9925457340060131934532622484655560235453367100145471063808192448972904496036427003415134333577374362700957"
        "6960870169574894987365444027113025031462573155865293066670475625805419711484614776550605553219621533867575"
        "47933215214212821463080071993090314208529889583587646484375e-95"},
    {0x1.fffffffffffffp-313, chars_format::scientific, 271,
        "1."
        "1985091468012026386906524496931112047090673420029094212761638489794580899207285400683026866715474872540191"
        "5392174033914978997473088805422605006292514631173058613334095125161083942296922955310121110643924306773515"
        "09586643042842564292616014398618062841705977916717529296875e-94"},
    {0x1.fffffffffffffp-312, chars_format::scientific, 270,
        "2."
        "3970182936024052773813048993862224094181346840058188425523276979589161798414570801366053733430949745080383"
        "0784348067829957994946177610845210012585029262346117226668190250322167884593845910620242221287848613547030"
        "1917328608568512858523202879723612568341195583343505859375e-94"},
    {0x1.fffffffffffffp-311, chars_format::scientific, 269,
        "4."
        "7940365872048105547626097987724448188362693680116376851046553959178323596829141602732107466861899490160766"
        "1568696135659915989892355221690420025170058524692234453336380500644335769187691821240484442575697227094060"
        "383465721713702571704640575944722513668239116668701171875e-94"},
    {0x1.fffffffffffffp-310, chars_format::scientific, 268,
        "9."
        "5880731744096211095252195975448896376725387360232753702093107918356647193658283205464214933723798980321532"
        "3137392271319831979784710443380840050340117049384468906672761001288671538375383642480968885151394454188120"
        "76693144342740514340928115188944502733647823333740234375e-94"},
    {0x1.fffffffffffffp-309, chars_format::scientific, 268,
        "1."
        "9176146348819242219050439195089779275345077472046550740418621583671329438731656641092842986744759796064306"
        "4627478454263966395956942088676168010068023409876893781334552200257734307675076728496193777030278890837624"
        "15338628868548102868185623037788900546729564666748046875e-93"},
    {0x1.fffffffffffffp-308, chars_format::scientific, 267,
        "3."
        "8352292697638484438100878390179558550690154944093101480837243167342658877463313282185685973489519592128612"
        "9254956908527932791913884177352336020136046819753787562669104400515468615350153456992387554060557781675248"
        "3067725773709620573637124607557780109345912933349609375e-93"},
    {0x1.fffffffffffffp-307, chars_format::scientific, 266,
        "7."
        "6704585395276968876201756780359117101380309888186202961674486334685317754926626564371371946979039184257225"
        "8509913817055865583827768354704672040272093639507575125338208801030937230700306913984775108121115563350496"
        "613545154741924114727424921511556021869182586669921875e-93"},
    {0x1.fffffffffffffp-306, chars_format::scientific, 266,
        "1."
        "5340917079055393775240351356071823420276061977637240592334897266937063550985325312874274389395807836851445"
        "1701982763411173116765553670940934408054418727901515025067641760206187446140061382796955021624223112670099"
        "322709030948384822945484984302311204373836517333984375e-92"},
    {0x1.fffffffffffffp-305, chars_format::scientific, 265,
        "3."
        "0681834158110787550480702712143646840552123955274481184669794533874127101970650625748548778791615673702890"
        "3403965526822346233531107341881868816108837455803030050135283520412374892280122765593910043248446225340198"
        "64541806189676964589096996860462240874767303466796875e-92"},
    {0x1.fffffffffffffp-304, chars_format::scientific, 264,
        "6."
        "1363668316221575100961405424287293681104247910548962369339589067748254203941301251497097557583231347405780"
        "6807931053644692467062214683763737632217674911606060100270567040824749784560245531187820086496892450680397"
        "2908361237935392917819399372092448174953460693359375e-92"},
    {0x1.fffffffffffffp-303, chars_format::scientific, 264,
        "1."
        "2272733663244315020192281084857458736220849582109792473867917813549650840788260250299419511516646269481156"
        "1361586210728938493412442936752747526443534982321212020054113408164949956912049106237564017299378490136079"
        "4581672247587078583563879874418489634990692138671875e-91"},
    {0x1.fffffffffffffp-302, chars_format::scientific, 263,
        "2."
        "4545467326488630040384562169714917472441699164219584947735835627099301681576520500598839023033292538962312"
        "2723172421457876986824885873505495052887069964642424040108226816329899913824098212475128034598756980272158"
        "916334449517415716712775974883697926998138427734375e-91"},
    {0x1.fffffffffffffp-301, chars_format::scientific, 262,
        "4."
        "9090934652977260080769124339429834944883398328439169895471671254198603363153041001197678046066585077924624"
        "5446344842915753973649771747010990105774139929284848080216453632659799827648196424950256069197513960544317"
        "83266889903483143342555194976739585399627685546875e-91"},
    {0x1.fffffffffffffp-300, chars_format::scientific, 261,
        "9."
        "8181869305954520161538248678859669889766796656878339790943342508397206726306082002395356092133170155849249"
        "0892689685831507947299543494021980211548279858569696160432907265319599655296392849900512138395027921088635"
        "6653377980696628668511038995347917079925537109375e-91"},
    {0x1.fffffffffffffp-299, chars_format::scientific, 261,
        "1."
        "9636373861190904032307649735771933977953359331375667958188668501679441345261216400479071218426634031169849"
        "8178537937166301589459908698804396042309655971713939232086581453063919931059278569980102427679005584217727"
        "1330675596139325733702207799069583415985107421875e-90"},
    {0x1.fffffffffffffp-298, chars_format::scientific, 260,
        "3."
        "9272747722381808064615299471543867955906718662751335916377337003358882690522432800958142436853268062339699"
        "6357075874332603178919817397608792084619311943427878464173162906127839862118557139960204855358011168435454"
        "266135119227865146740441559813916683197021484375e-90"},
    {0x1.fffffffffffffp-297, chars_format::scientific, 259,
        "7."
        "8545495444763616129230598943087735911813437325502671832754674006717765381044865601916284873706536124679399"
        "2714151748665206357839634795217584169238623886855756928346325812255679724237114279920409710716022336870908"
        "53227023845573029348088311962783336639404296875e-90"},
    {0x1.fffffffffffffp-296, chars_format::scientific, 259,
        "1."
        "5709099088952723225846119788617547182362687465100534366550934801343553076208973120383256974741307224935879"
        "8542830349733041271567926959043516833847724777371151385669265162451135944847422855984081942143204467374181"
        "70645404769114605869617662392556667327880859375e-89"},
    {0x1.fffffffffffffp-295, chars_format::scientific, 258,
        "3."
        "1418198177905446451692239577235094364725374930201068733101869602687106152417946240766513949482614449871759"
        "7085660699466082543135853918087033667695449554742302771338530324902271889694845711968163884286408934748363"
        "4129080953822921173923532478511333465576171875e-89"},
    {0x1.fffffffffffffp-294, chars_format::scientific, 257,
        "6."
        "2836396355810892903384479154470188729450749860402137466203739205374212304835892481533027898965228899743519"
        "4171321398932165086271707836174067335390899109484605542677060649804543779389691423936327768572817869496726"
        "825816190764584234784706495702266693115234375e-89"},
    {0x1.fffffffffffffp-293, chars_format::scientific, 257,
        "1."
        "2567279271162178580676895830894037745890149972080427493240747841074842460967178496306605579793045779948703"
        "8834264279786433017254341567234813467078179821896921108535412129960908755877938284787265553714563573899345"
        "365163238152916846956941299140453338623046875e-88"},
    {0x1.fffffffffffffp-292, chars_format::scientific, 256,
        "2."
        "5134558542324357161353791661788075491780299944160854986481495682149684921934356992613211159586091559897407"
        "7668528559572866034508683134469626934156359643793842217070824259921817511755876569574531107429127147798690"
        "73032647630583369391388259828090667724609375e-88"},
    {0x1.fffffffffffffp-291, chars_format::scientific, 255,
        "5."
        "0269117084648714322707583323576150983560599888321709972962991364299369843868713985226422319172183119794815"
        "5337057119145732069017366268939253868312719287587684434141648519843635023511753139149062214858254295597381"
        "4606529526116673878277651965618133544921875e-88"},
    {0x1.fffffffffffffp-290, chars_format::scientific, 255,
        "1."
        "0053823416929742864541516664715230196712119977664341994592598272859873968773742797045284463834436623958963"
        "1067411423829146413803473253787850773662543857517536886828329703968727004702350627829812442971650859119476"
        "2921305905223334775655530393123626708984375e-87"},
    {0x1.fffffffffffffp-289, chars_format::scientific, 254,
        "2."
        "0107646833859485729083033329430460393424239955328683989185196545719747937547485594090568927668873247917926"
        "2134822847658292827606946507575701547325087715035073773656659407937454009404701255659624885943301718238952"
        "584261181044666955131106078624725341796875e-87"},
    {0x1.fffffffffffffp-288, chars_format::scientific, 253,
        "4."
        "0215293667718971458166066658860920786848479910657367978370393091439495875094971188181137855337746495835852"
        "4269645695316585655213893015151403094650175430070147547313318815874908018809402511319249771886603436477905"
        "16852236208933391026221215724945068359375e-87"},
    {0x1.fffffffffffffp-287, chars_format::scientific, 252,
        "8."
        "0430587335437942916332133317721841573696959821314735956740786182878991750189942376362275710675492991671704"
        "8539291390633171310427786030302806189300350860140295094626637631749816037618805022638499543773206872955810"
        "3370447241786678205244243144989013671875e-87"},
    {0x1.fffffffffffffp-286, chars_format::scientific, 252,
        "1."
        "6086117467087588583266426663544368314739391964262947191348157236575798350037988475272455142135098598334340"
        "9707858278126634262085557206060561237860070172028059018925327526349963207523761004527699908754641374591162"
        "0674089448357335641048848628997802734375e-86"},
    {0x1.fffffffffffffp-285, chars_format::scientific, 251,
        "3."
        "2172234934175177166532853327088736629478783928525894382696314473151596700075976950544910284270197196668681"
        "9415716556253268524171114412121122475720140344056118037850655052699926415047522009055399817509282749182324"
        "134817889671467128209769725799560546875e-86"},
    {0x1.fffffffffffffp-284, chars_format::scientific, 250,
        "6."
        "4344469868350354333065706654177473258957567857051788765392628946303193400151953901089820568540394393337363"
        "8831433112506537048342228824242244951440280688112236075701310105399852830095044018110799635018565498364648"
        "26963577934293425641953945159912109375e-86"},
    {0x1.fffffffffffffp-283, chars_format::scientific, 250,
        "1."
        "2868893973670070866613141330835494651791513571410357753078525789260638680030390780217964113708078878667472"
        "7766286622501307409668445764848448990288056137622447215140262021079970566019008803622159927003713099672929"
        "65392715586858685128390789031982421875e-85"},
    {0x1.fffffffffffffp-282, chars_format::scientific, 249,
        "2."
        "5737787947340141733226282661670989303583027142820715506157051578521277360060781560435928227416157757334945"
        "5532573245002614819336891529696897980576112275244894430280524042159941132038017607244319854007426199345859"
        "3078543117371737025678157806396484375e-85"},
    {0x1.fffffffffffffp-281, chars_format::scientific, 248,
        "5."
        "1475575894680283466452565323341978607166054285641431012314103157042554720121563120871856454832315514669891"
        "1065146490005229638673783059393795961152224550489788860561048084319882264076035214488639708014852398691718"
        "615708623474347405135631561279296875e-85"},
    {0x1.fffffffffffffp-280, chars_format::scientific, 248,
        "1."
        "0295115178936056693290513064668395721433210857128286202462820631408510944024312624174371290966463102933978"
        "2213029298001045927734756611878759192230444910097957772112209616863976452815207042897727941602970479738343"
        "723141724694869481027126312255859375e-84"},
    {0x1.fffffffffffffp-279, chars_format::scientific, 247,
        "2."
        "0590230357872113386581026129336791442866421714256572404925641262817021888048625248348742581932926205867956"
        "4426058596002091855469513223757518384460889820195915544224419233727952905630414085795455883205940959476687"
        "44628344938973896205425262451171875e-84"},
    {0x1.fffffffffffffp-278, chars_format::scientific, 246,
        "4."
        "1180460715744226773162052258673582885732843428513144809851282525634043776097250496697485163865852411735912"
        "8852117192004183710939026447515036768921779640391831088448838467455905811260828171590911766411881918953374"
        "8925668987794779241085052490234375e-84"},
    {0x1.fffffffffffffp-277, chars_format::scientific, 245,
        "8."
        "2360921431488453546324104517347165771465686857026289619702565051268087552194500993394970327731704823471825"
        "7704234384008367421878052895030073537843559280783662176897676934911811622521656343181823532823763837906749"
        "785133797558955848217010498046875e-84"},
    {0x1.fffffffffffffp-276, chars_format::scientific, 245,
        "1."
        "6472184286297690709264820903469433154293137371405257923940513010253617510438900198678994065546340964694365"
        "1540846876801673484375610579006014707568711856156732435379535386982362324504331268636364706564752767581349"
        "957026759511791169643402099609375e-83"},
    {0x1.fffffffffffffp-275, chars_format::scientific, 244,
        "3."
        "2944368572595381418529641806938866308586274742810515847881026020507235020877800397357988131092681929388730"
        "3081693753603346968751221158012029415137423712313464870759070773964724649008662537272729413129505535162699"
        "91405351902358233928680419921875e-83"},
    {0x1.fffffffffffffp-274, chars_format::scientific, 243,
        "6."
        "5888737145190762837059283613877732617172549485621031695762052041014470041755600794715976262185363858777460"
        "6163387507206693937502442316024058830274847424626929741518141547929449298017325074545458826259011070325399"
        "8281070380471646785736083984375e-83"},
    {0x1.fffffffffffffp-273, chars_format::scientific, 243,
        "1."
        "3177747429038152567411856722775546523434509897124206339152410408202894008351120158943195252437072771755492"
        "1232677501441338787500488463204811766054969484925385948303628309585889859603465014909091765251802214065079"
        "9656214076094329357147216796875e-82"},
    {0x1.fffffffffffffp-272, chars_format::scientific, 242,
        "2."
        "6355494858076305134823713445551093046869019794248412678304820816405788016702240317886390504874145543510984"
        "2465355002882677575000976926409623532109938969850771896607256619171779719206930029818183530503604428130159"
        "931242815218865871429443359375e-82"},
    {0x1.fffffffffffffp-271, chars_format::scientific, 241,
        "5."
        "2710989716152610269647426891102186093738039588496825356609641632811576033404480635772781009748291087021968"
        "4930710005765355150001953852819247064219877939701543793214513238343559438413860059636367061007208856260319"
        "86248563043773174285888671875e-82"},
    {0x1.fffffffffffffp-270, chars_format::scientific, 241,
        "1."
        "0542197943230522053929485378220437218747607917699365071321928326562315206680896127154556201949658217404393"
        "6986142001153071030000390770563849412843975587940308758642902647668711887682772011927273412201441771252063"
        "97249712608754634857177734375e-81"},
    {0x1.fffffffffffffp-269, chars_format::scientific, 240,
        "2."
        "1084395886461044107858970756440874437495215835398730142643856653124630413361792254309112403899316434808787"
        "3972284002306142060000781541127698825687951175880617517285805295337423775365544023854546824402883542504127"
        "9449942521750926971435546875e-81"},
    {0x1.fffffffffffffp-268, chars_format::scientific, 239,
        "4."
        "2168791772922088215717941512881748874990431670797460285287713306249260826723584508618224807798632869617574"
        "7944568004612284120001563082255397651375902351761235034571610590674847550731088047709093648805767085008255"
        "889988504350185394287109375e-81"},
    {0x1.fffffffffffffp-267, chars_format::scientific, 238,
        "8."
        "4337583545844176431435883025763497749980863341594920570575426612498521653447169017236449615597265739235149"
        "5889136009224568240003126164510795302751804703522470069143221181349695101462176095418187297611534170016511"
        "77997700870037078857421875e-81"},
    {0x1.fffffffffffffp-266, chars_format::scientific, 238,
        "1."
        "6867516709168835286287176605152699549996172668318984114115085322499704330689433803447289923119453147847029"
        "9177827201844913648000625232902159060550360940704494013828644236269939020292435219083637459522306834003302"
        "35599540174007415771484375e-80"},
    {0x1.fffffffffffffp-265, chars_format::scientific, 237,
        "3."
        "3735033418337670572574353210305399099992345336637968228230170644999408661378867606894579846238906295694059"
        "8355654403689827296001250465804318121100721881408988027657288472539878040584870438167274919044613668006604"
        "7119908034801483154296875e-80"},
    {0x1.fffffffffffffp-264, chars_format::scientific, 236,
        "6."
        "7470066836675341145148706420610798199984690673275936456460341289998817322757735213789159692477812591388119"
        "6711308807379654592002500931608636242201443762817976055314576945079756081169740876334549838089227336013209"
        "423981606960296630859375e-80"},
    {0x1.fffffffffffffp-263, chars_format::scientific, 236,
        "1."
        "3494013367335068229029741284122159639996938134655187291292068257999763464551547042757831938495562518277623"
        "9342261761475930918400500186321727248440288752563595211062915389015951216233948175266909967617845467202641"
        "884796321392059326171875e-79"},
    {0x1.fffffffffffffp-262, chars_format::scientific, 235,
        "2."
        "6988026734670136458059482568244319279993876269310374582584136515999526929103094085515663876991125036555247"
        "8684523522951861836801000372643454496880577505127190422125830778031902432467896350533819935235690934405283"
        "76959264278411865234375e-79"},
    {0x1.fffffffffffffp-261, chars_format::scientific, 234,
        "5."
        "3976053469340272916118965136488638559987752538620749165168273031999053858206188171031327753982250073110495"
        "7369047045903723673602000745286908993761155010254380844251661556063804864935792701067639870471381868810567"
        "5391852855682373046875e-79"},
    {0x1.fffffffffffffp-260, chars_format::scientific, 234,
        "1."
        "0795210693868054583223793027297727711997550507724149833033654606399810771641237634206265550796450014622099"
        "1473809409180744734720400149057381798752231002050876168850332311212760972987158540213527974094276373762113"
        "5078370571136474609375e-78"},
    {0x1.fffffffffffffp-259, chars_format::scientific, 233,
        "2."
        "1590421387736109166447586054595455423995101015448299666067309212799621543282475268412531101592900029244198"
        "2947618818361489469440800298114763597504462004101752337700664622425521945974317080427055948188552747524227"
        "015674114227294921875e-78"},
    {0x1.fffffffffffffp-258, chars_format::scientific, 232,
        "4."
        "3180842775472218332895172109190910847990202030896599332134618425599243086564950536825062203185800058488396"
        "5895237636722978938881600596229527195008924008203504675401329244851043891948634160854111896377105495048454"
        "03134822845458984375e-78"},
    {0x1.fffffffffffffp-257, chars_format::scientific, 231,
        "8."
        "6361685550944436665790344218381821695980404061793198664269236851198486173129901073650124406371600116976793"
        "1790475273445957877763201192459054390017848016407009350802658489702087783897268321708223792754210990096908"
        "0626964569091796875e-78"},
    {0x1.fffffffffffffp-256, chars_format::scientific, 231,
        "1."
        "7272337110188887333158068843676364339196080812358639732853847370239697234625980214730024881274320023395358"
        "6358095054689191575552640238491810878003569603281401870160531697940417556779453664341644758550842198019381"
        "6125392913818359375e-77"},
    {0x1.fffffffffffffp-255, chars_format::scientific, 230,
        "3."
        "4544674220377774666316137687352728678392161624717279465707694740479394469251960429460049762548640046790717"
        "2716190109378383151105280476983621756007139206562803740321063395880835113558907328683289517101684396038763"
        "225078582763671875e-77"},
    {0x1.fffffffffffffp-254, chars_format::scientific, 229,
        "6."
        "9089348440755549332632275374705457356784323249434558931415389480958788938503920858920099525097280093581434"
        "5432380218756766302210560953967243512014278413125607480642126791761670227117814657366579034203368792077526"
        "45015716552734375e-77"},
    {0x1.fffffffffffffp-253, chars_format::scientific, 229,
        "1."
        "3817869688151109866526455074941091471356864649886911786283077896191757787700784171784019905019456018716286"
        "9086476043751353260442112190793448702402855682625121496128425358352334045423562931473315806840673758415505"
        "29003143310546875e-76"},
    {0x1.fffffffffffffp-252, chars_format::scientific, 228,
        "2."
        "7635739376302219733052910149882182942713729299773823572566155792383515575401568343568039810038912037432573"
        "8172952087502706520884224381586897404805711365250242992256850716704668090847125862946631613681347516831010"
        "5800628662109375e-76"},
    {0x1.fffffffffffffp-251, chars_format::scientific, 227,
        "5."
        "5271478752604439466105820299764365885427458599547647145132311584767031150803136687136079620077824074865147"
        "6345904175005413041768448763173794809611422730500485984513701433409336181694251725893263227362695033662021"
        "160125732421875e-76"},
    {0x1.fffffffffffffp-250, chars_format::scientific, 227,
        "1."
        "1054295750520887893221164059952873177085491719909529429026462316953406230160627337427215924015564814973029"
        "5269180835001082608353689752634758961922284546100097196902740286681867236338850345178652645472539006732404"
        "232025146484375e-75"},
    {0x1.fffffffffffffp-249, chars_format::scientific, 226,
        "2."
        "2108591501041775786442328119905746354170983439819058858052924633906812460321254674854431848031129629946059"
        "0538361670002165216707379505269517923844569092200194393805480573363734472677700690357305290945078013464808"
        "46405029296875e-75"},
    {0x1.fffffffffffffp-248, chars_format::scientific, 225,
        "4."
        "4217183002083551572884656239811492708341966879638117716105849267813624920642509349708863696062259259892118"
        "1076723340004330433414759010539035847689138184400388787610961146727468945355401380714610581890156026929616"
        "9281005859375e-75"},
    {0x1.fffffffffffffp-247, chars_format::scientific, 224,
        "8."
        "8434366004167103145769312479622985416683933759276235432211698535627249841285018699417727392124518519784236"
        "2153446680008660866829518021078071695378276368800777575221922293454937890710802761429221163780312053859233"
        "856201171875e-75"},
    {0x1.fffffffffffffp-246, chars_format::scientific, 224,
        "1."
        "7686873200833420629153862495924597083336786751855247086442339707125449968257003739883545478424903703956847"
        "2430689336001732173365903604215614339075655273760155515044384458690987578142160552285844232756062410771846"
        "771240234375e-74"},
    {0x1.fffffffffffffp-245, chars_format::scientific, 223,
        "3."
        "5373746401666841258307724991849194166673573503710494172884679414250899936514007479767090956849807407913694"
        "4861378672003464346731807208431228678151310547520311030088768917381975156284321104571688465512124821543693"
        "54248046875e-74"},
    {0x1.fffffffffffffp-244, chars_format::scientific, 222,
        "7."
        "0747492803333682516615449983698388333347147007420988345769358828501799873028014959534181913699614815827388"
        "9722757344006928693463614416862457356302621095040622060177537834763950312568642209143376931024249643087387"
        "0849609375e-74"},
    {0x1.fffffffffffffp-243, chars_format::scientific, 222,
        "1."
        "4149498560666736503323089996739677666669429401484197669153871765700359974605602991906836382739922963165477"
        "7944551468801385738692722883372491471260524219008124412035507566952790062513728441828675386204849928617477"
        "4169921875e-73"},
    {0x1.fffffffffffffp-242, chars_format::scientific, 221,
        "2."
        "8298997121333473006646179993479355333338858802968395338307743531400719949211205983813672765479845926330955"
        "5889102937602771477385445766744982942521048438016248824071015133905580125027456883657350772409699857234954"
        "833984375e-73"},
    {0x1.fffffffffffffp-241, chars_format::scientific, 220,
        "5."
        "6597994242666946013292359986958710666677717605936790676615487062801439898422411967627345530959691852661911"
        "1778205875205542954770891533489965885042096876032497648142030267811160250054913767314701544819399714469909"
        "66796875e-73"},
    {0x1.fffffffffffffp-240, chars_format::scientific, 220,
        "1."
        "1319598848533389202658471997391742133335543521187358135323097412560287979684482393525469106191938370532382"
        "2355641175041108590954178306697993177008419375206499529628406053562232050010982753462940308963879942893981"
        "93359375e-72"},
    {0x1.fffffffffffffp-239, chars_format::scientific, 219,
        "2."
        "2639197697066778405316943994783484266671087042374716270646194825120575959368964787050938212383876741064764"
        "4711282350082217181908356613395986354016838750412999059256812107124464100021965506925880617927759885787963"
        "8671875e-72"},
    {0x1.fffffffffffffp-238, chars_format::scientific, 218,
        "4."
        "5278395394133556810633887989566968533342174084749432541292389650241151918737929574101876424767753482129528"
        "9422564700164434363816713226791972708033677500825998118513624214248928200043931013851761235855519771575927"
        "734375e-72"},
    {0x1.fffffffffffffp-237, chars_format::scientific, 217,
        "9."
        "0556790788267113621267775979133937066684348169498865082584779300482303837475859148203752849535506964259057"
        "8845129400328868727633426453583945416067355001651996237027248428497856400087862027703522471711039543151855"
        "46875e-72"},
    {0x1.fffffffffffffp-236, chars_format::scientific, 217,
        "1."
        "8111358157653422724253555195826787413336869633899773016516955860096460767495171829640750569907101392851811"
        "5769025880065773745526685290716789083213471000330399247405449685699571280017572405540704494342207908630371"
        "09375e-71"},
    {0x1.fffffffffffffp-235, chars_format::scientific, 216,
        "3."
        "6222716315306845448507110391653574826673739267799546033033911720192921534990343659281501139814202785703623"
        "1538051760131547491053370581433578166426942000660798494810899371399142560035144811081408988684415817260742"
        "1875e-71"},
    {0x1.fffffffffffffp-234, chars_format::scientific, 215,
        "7."
        "2445432630613690897014220783307149653347478535599092066067823440385843069980687318563002279628405571407246"
        "3076103520263094982106741162867156332853884001321596989621798742798285120070289622162817977368831634521484"
        "375e-71"},
    {0x1.fffffffffffffp-233, chars_format::scientific, 215,
        "1."
        "4489086526122738179402844156661429930669495707119818413213564688077168613996137463712600455925681114281449"
        "2615220704052618996421348232573431266570776800264319397924359748559657024014057924432563595473766326904296"
        "875e-70"},
    {0x1.fffffffffffffp-232, chars_format::scientific, 214,
        "2."
        "8978173052245476358805688313322859861338991414239636826427129376154337227992274927425200911851362228562898"
        "5230441408105237992842696465146862533141553600528638795848719497119314048028115848865127190947532653808593"
        "75e-70"},
    {0x1.fffffffffffffp-231, chars_format::scientific, 213,
        "5."
        "7956346104490952717611376626645719722677982828479273652854258752308674455984549854850401823702724457125797"
        "0460882816210475985685392930293725066283107201057277591697438994238628096056231697730254381895065307617187"
        "5e-70"},
    {0x1.fffffffffffffp-230, chars_format::scientific, 213,
        "1."
        "1591269220898190543522275325329143944535596565695854730570851750461734891196909970970080364740544891425159"
        "4092176563242095197137078586058745013256621440211455518339487798847725619211246339546050876379013061523437"
        "5e-69"},
    {0x1.fffffffffffffp-229, chars_format::scientific, 212,
        "2."
        "3182538441796381087044550650658287889071193131391709461141703500923469782393819941940160729481089782850318"
        "8184353126484190394274157172117490026513242880422911036678975597695451238422492679092101752758026123046875"
        "e-69"},
    {0x1.fffffffffffffp-228, chars_format::scientific, 211,
        "4."
        "6365076883592762174089101301316575778142386262783418922283407001846939564787639883880321458962179565700637"
        "636870625296838078854831434423498005302648576084582207335795119539090247684498535818420350551605224609375e"
        "-69"},
    {0x1.fffffffffffffp-227, chars_format::scientific, 210,
        "9."
        "2730153767185524348178202602633151556284772525566837844566814003693879129575279767760642917924359131401275"
        "27374125059367615770966286884699601060529715216916441467159023907818049536899707163684070110321044921875e-"
        "69"},
    {0x1.fffffffffffffp-226, chars_format::scientific, 210,
        "1."
        "8546030753437104869635640520526630311256954505113367568913362800738775825915055953552128583584871826280255"
        "05474825011873523154193257376939920212105943043383288293431804781563609907379941432736814022064208984375e-"
        "68"},
    {0x1.fffffffffffffp-225, chars_format::scientific, 209,
        "3."
        "7092061506874209739271281041053260622513909010226735137826725601477551651830111907104257167169743652560510"
        "1094965002374704630838651475387984042421188608676657658686360956312721981475988286547362804412841796875e-"
        "68"},
    {0x1.fffffffffffffp-224, chars_format::scientific, 208,
        "7."
        "4184123013748419478542562082106521245027818020453470275653451202955103303660223814208514334339487305121020"
        "218993000474940926167730295077596808484237721735331531737272191262544396295197657309472560882568359375e-"
        "68"},
    {0x1.fffffffffffffp-223, chars_format::scientific, 208,
        "1."
        "4836824602749683895708512416421304249005563604090694055130690240591020660732044762841702866867897461024204"
        "043798600094988185233546059015519361696847544347066306347454438252508879259039531461894512176513671875e-"
        "67"},
    {0x1.fffffffffffffp-222, chars_format::scientific, 207,
        "2."
        "9673649205499367791417024832842608498011127208181388110261380481182041321464089525683405733735794922048408"
        "08759720018997637046709211803103872339369508869413261269490887650501775851807906292378902435302734375e-"
        "67"},
    {0x1.fffffffffffffp-221, chars_format::scientific, 206,
        "5."
        "9347298410998735582834049665685216996022254416362776220522760962364082642928179051366811467471589844096816"
        "1751944003799527409341842360620774467873901773882652253898177530100355170361581258475780487060546875e-67"},
    {0x1.fffffffffffffp-220, chars_format::scientific, 206,
        "1."
        "1869459682199747116566809933137043399204450883272555244104552192472816528585635810273362293494317968819363"
        "2350388800759905481868368472124154893574780354776530450779635506020071034072316251695156097412109375e-66"},
    {0x1.fffffffffffffp-219, chars_format::scientific, 205,
        "2."
        "3738919364399494233133619866274086798408901766545110488209104384945633057171271620546724586988635937638726"
        "470077760151981096373673694424830978714956070955306090155927101204014206814463250339031219482421875e-66"},
    {0x1.fffffffffffffp-218, chars_format::scientific, 204,
        "4."
        "7477838728798988466267239732548173596817803533090220976418208769891266114342543241093449173977271875277452"
        "94015552030396219274734738884966195742991214191061218031185420240802841362892650067806243896484375e-66"},
    {0x1.fffffffffffffp-217, chars_format::scientific, 203,
        "9."
        "4955677457597976932534479465096347193635607066180441952836417539782532228685086482186898347954543750554905"
        "8803110406079243854946947776993239148598242838212243606237084048160568272578530013561248779296875e-66"},
    {0x1.fffffffffffffp-216, chars_format::scientific, 203,
        "1."
        "8991135491519595386506895893019269438727121413236088390567283507956506445737017296437379669590908750110981"
        "1760622081215848770989389555398647829719648567642448721247416809632113654515706002712249755859375e-65"},
    {0x1.fffffffffffffp-215, chars_format::scientific, 202,
        "3."
        "7982270983039190773013791786038538877454242826472176781134567015913012891474034592874759339181817500221962"
        "352124416243169754197877911079729565943929713528489744249483361926422730903141200542449951171875e-65"},
    {0x1.fffffffffffffp-214, chars_format::scientific, 201,
        "7."
        "5964541966078381546027583572077077754908485652944353562269134031826025782948069185749518678363635000443924"
        "70424883248633950839575582215945913188785942705697948849896672385284546180628240108489990234375e-65"},
    {0x1.fffffffffffffp-213, chars_format::scientific, 201,
        "1."
        "5192908393215676309205516714415415550981697130588870712453826806365205156589613837149903735672727000088784"
        "94084976649726790167915116443189182637757188541139589769979334477056909236125648021697998046875e-64"},
    {0x1.fffffffffffffp-212, chars_format::scientific, 200,
        "3."
        "0385816786431352618411033428830831101963394261177741424907653612730410313179227674299807471345454000177569"
        "8816995329945358033583023288637836527551437708227917953995866895411381847225129604339599609375e-64"},
    {0x1.fffffffffffffp-211, chars_format::scientific, 199,
        "6."
        "0771633572862705236822066857661662203926788522355482849815307225460820626358455348599614942690908000355139"
        "763399065989071606716604657727567305510287541645583590799173379082276369445025920867919921875e-64"},
    {0x1.fffffffffffffp-210, chars_format::scientific, 199,
        "1."
        "2154326714572541047364413371532332440785357704471096569963061445092164125271691069719922988538181600071027"
        "952679813197814321343320931545513461102057508329116718159834675816455273889005184173583984375e-63"},
    {0x1.fffffffffffffp-209, chars_format::scientific, 198,
        "2."
        "4308653429145082094728826743064664881570715408942193139926122890184328250543382139439845977076363200142055"
        "90535962639562864268664186309102692220411501665823343631966935163291054777801036834716796875e-63"},
    {0x1.fffffffffffffp-208, chars_format::scientific, 197,
        "4."
        "8617306858290164189457653486129329763141430817884386279852245780368656501086764278879691954152726400284111"
        "8107192527912572853732837261820538444082300333164668726393387032658210955560207366943359375e-63"},
    {0x1.fffffffffffffp-207, chars_format::scientific, 196,
        "9."
        "7234613716580328378915306972258659526282861635768772559704491560737313002173528557759383908305452800568223"
        "621438505582514570746567452364107688816460066632933745278677406531642191112041473388671875e-63"},
    {0x1.fffffffffffffp-206, chars_format::scientific, 196,
        "1."
        "9446922743316065675783061394451731905256572327153754511940898312147462600434705711551876781661090560113644"
        "724287701116502914149313490472821537763292013326586749055735481306328438222408294677734375e-62"},
    {0x1.fffffffffffffp-205, chars_format::scientific, 195,
        "3."
        "8893845486632131351566122788903463810513144654307509023881796624294925200869411423103753563322181120227289"
        "44857540223300582829862698094564307552658402665317349811147096261265687644481658935546875e-62"},
    {0x1.fffffffffffffp-204, chars_format::scientific, 194,
        "7."
        "7787690973264262703132245577806927621026289308615018047763593248589850401738822846207507126644362240454578"
        "8971508044660116565972539618912861510531680533063469962229419252253137528896331787109375e-62"},
    {0x1.fffffffffffffp-203, chars_format::scientific, 194,
        "1."
        "5557538194652852540626449115561385524205257861723003609552718649717970080347764569241501425328872448090915"
        "7794301608932023313194507923782572302106336106612693992445883850450627505779266357421875e-61"},
    {0x1.fffffffffffffp-202, chars_format::scientific, 193,
        "3."
        "1115076389305705081252898231122771048410515723446007219105437299435940160695529138483002850657744896181831"
        "558860321786404662638901584756514460421267221322538798489176770090125501155853271484375e-61"},
    {0x1.fffffffffffffp-201, chars_format::scientific, 192,
        "6."
        "2230152778611410162505796462245542096821031446892014438210874598871880321391058276966005701315489792363663"
        "11772064357280932527780316951302892084253444264507759697835354018025100231170654296875e-61"},
    {0x1.fffffffffffffp-200, chars_format::scientific, 192,
        "1."
        "2446030555722282032501159292449108419364206289378402887642174919774376064278211655393201140263097958472732"
        "62354412871456186505556063390260578416850688852901551939567070803605020046234130859375e-60"},
    {0x1.fffffffffffffp-199, chars_format::scientific, 191,
        "2."
        "4892061111444564065002318584898216838728412578756805775284349839548752128556423310786402280526195916945465"
        "2470882574291237301111212678052115683370137770580310387913414160721004009246826171875e-60"},
    {0x1.fffffffffffffp-198, chars_format::scientific, 190,
        "4."
        "9784122222889128130004637169796433677456825157513611550568699679097504257112846621572804561052391833890930"
        "494176514858247460222242535610423136674027554116062077582682832144200801849365234375e-60"},
    {0x1.fffffffffffffp-197, chars_format::scientific, 189,
        "9."
        "9568244445778256260009274339592867354913650315027223101137399358195008514225693243145609122104783667781860"
        "98835302971649492044448507122084627334805510823212415516536566428840160369873046875e-60"},
    {0x1.fffffffffffffp-196, chars_format::scientific, 189,
        "1."
        "9913648889155651252001854867918573470982730063005444620227479871639001702845138648629121824420956733556372"
        "19767060594329898408889701424416925466961102164642483103307313285768032073974609375e-59"},
    {0x1.fffffffffffffp-195, chars_format::scientific, 188,
        "3."
        "9827297778311302504003709735837146941965460126010889240454959743278003405690277297258243648841913467112744"
        "3953412118865979681777940284883385093392220432928496620661462657153606414794921875e-59"},
    {0x1.fffffffffffffp-194, chars_format::scientific, 187,
        "7."
        "9654595556622605008007419471674293883930920252021778480909919486556006811380554594516487297683826934225488"
        "790682423773195936355588056976677018678444086585699324132292531430721282958984375e-59"},
    {0x1.fffffffffffffp-193, chars_format::scientific, 187,
        "1."
        "5930919111324521001601483894334858776786184050404355696181983897311201362276110918903297459536765386845097"
        "758136484754639187271117611395335403735688817317139864826458506286144256591796875e-58"},
    {0x1.fffffffffffffp-192, chars_format::scientific, 186,
        "3."
        "1861838222649042003202967788669717553572368100808711392363967794622402724552221837806594919073530773690195"
        "51627296950927837454223522279067080747137763463427972965291701257228851318359375e-58"},
    {0x1.fffffffffffffp-191, chars_format::scientific, 185,
        "6."
        "3723676445298084006405935577339435107144736201617422784727935589244805449104443675613189838147061547380391"
        "0325459390185567490844704455813416149427552692685594593058340251445770263671875e-58"},
    {0x1.fffffffffffffp-190, chars_format::scientific, 185,
        "1."
        "2744735289059616801281187115467887021428947240323484556945587117848961089820888735122637967629412309476078"
        "2065091878037113498168940891162683229885510538537118918611668050289154052734375e-57"},
    {0x1.fffffffffffffp-189, chars_format::scientific, 184,
        "2."
        "5489470578119233602562374230935774042857894480646969113891174235697922179641777470245275935258824618952156"
        "413018375607422699633788178232536645977102107707423783722333610057830810546875e-57"},
    {0x1.fffffffffffffp-188, chars_format::scientific, 183,
        "5."
        "0978941156238467205124748461871548085715788961293938227782348471395844359283554940490551870517649237904312"
        "82603675121484539926757635646507329195420421541484756744466722011566162109375e-57"},
    {0x1.fffffffffffffp-187, chars_format::scientific, 183,
        "1."
        "0195788231247693441024949692374309617143157792258787645556469694279168871856710988098110374103529847580862"
        "56520735024296907985351527129301465839084084308296951348893344402313232421875e-56"},
    {0x1.fffffffffffffp-186, chars_format::scientific, 182,
        "2."
        "0391576462495386882049899384748619234286315584517575291112939388558337743713421976196220748207059695161725"
        "1304147004859381597070305425860293167816816861659390269778668880462646484375e-56"},
    {0x1.fffffffffffffp-185, chars_format::scientific, 181,
        "4."
        "0783152924990773764099798769497238468572631169035150582225878777116675487426843952392441496414119390323450"
        "260829400971876319414061085172058633563363372331878053955733776092529296875e-56"},
    {0x1.fffffffffffffp-184, chars_format::scientific, 180,
        "8."
        "1566305849981547528199597538994476937145262338070301164451757554233350974853687904784882992828238780646900"
        "52165880194375263882812217034411726712672674466375610791146755218505859375e-56"},
    {0x1.fffffffffffffp-183, chars_format::scientific, 180,
        "1."
        "6313261169996309505639919507798895387429052467614060232890351510846670194970737580956976598565647756129380"
        "10433176038875052776562443406882345342534534893275122158229351043701171875e-55"},
    {0x1.fffffffffffffp-182, chars_format::scientific, 179,
        "3."
        "2626522339992619011279839015597790774858104935228120465780703021693340389941475161913953197131295512258760"
        "2086635207775010555312488681376469068506906978655024431645870208740234375e-55"},
    {0x1.fffffffffffffp-181, chars_format::scientific, 178,
        "6."
        "5253044679985238022559678031195581549716209870456240931561406043386680779882950323827906394262591024517520"
        "417327041555002111062497736275293813701381395731004886329174041748046875e-55"},
    {0x1.fffffffffffffp-180, chars_format::scientific, 178,
        "1."
        "3050608935997047604511935606239116309943241974091248186312281208677336155976590064765581278852518204903504"
        "083465408311000422212499547255058762740276279146200977265834808349609375e-54"},
    {0x1.fffffffffffffp-179, chars_format::scientific, 177,
        "2."
        "6101217871994095209023871212478232619886483948182496372624562417354672311953180129531162557705036409807008"
        "16693081662200084442499909451011752548055255829240195453166961669921875e-54"},
    {0x1.fffffffffffffp-178, chars_format::scientific, 176,
        "5."
        "2202435743988190418047742424956465239772967896364992745249124834709344623906360259062325115410072819614016"
        "3338616332440016888499981890202350509611051165848039090633392333984375e-54"},
    {0x1.fffffffffffffp-177, chars_format::scientific, 176,
        "1."
        "0440487148797638083609548484991293047954593579272998549049824966941868924781272051812465023082014563922803"
        "2667723266488003377699996378040470101922210233169607818126678466796875e-53"},
    {0x1.fffffffffffffp-176, chars_format::scientific, 175,
        "2."
        "0880974297595276167219096969982586095909187158545997098099649933883737849562544103624930046164029127845606"
        "533544653297600675539999275608094020384442046633921563625335693359375e-53"},
    {0x1.fffffffffffffp-175, chars_format::scientific, 174,
        "4."
        "1761948595190552334438193939965172191818374317091994196199299867767475699125088207249860092328058255691213"
        "06708930659520135107999855121618804076888409326784312725067138671875e-53"},
    {0x1.fffffffffffffp-174, chars_format::scientific, 173,
        "8."
        "3523897190381104668876387879930344383636748634183988392398599735534951398250176414499720184656116511382426"
        "1341786131904027021599971024323760815377681865356862545013427734375e-53"},
    {0x1.fffffffffffffp-173, chars_format::scientific, 173,
        "1."
        "6704779438076220933775277575986068876727349726836797678479719947106990279650035282899944036931223302276485"
        "2268357226380805404319994204864752163075536373071372509002685546875e-52"},
    {0x1.fffffffffffffp-172, chars_format::scientific, 172,
        "3."
        "3409558876152441867550555151972137753454699453673595356959439894213980559300070565799888073862446604552970"
        "453671445276161080863998840972950432615107274614274501800537109375e-52"},
    {0x1.fffffffffffffp-171, chars_format::scientific, 171,
        "6."
        "6819117752304883735101110303944275506909398907347190713918879788427961118600141131599776147724893209105940"
        "90734289055232216172799768194590086523021454922854900360107421875e-52"},
    {0x1.fffffffffffffp-170, chars_format::scientific, 171,
        "1."
        "3363823550460976747020222060788855101381879781469438142783775957685592223720028226319955229544978641821188"
        "18146857811046443234559953638918017304604290984570980072021484375e-51"},
    {0x1.fffffffffffffp-169, chars_format::scientific, 170,
        "2."
        "6727647100921953494040444121577710202763759562938876285567551915371184447440056452639910459089957283642376"
        "3629371562209288646911990727783603460920858196914196014404296875e-51"},
    {0x1.fffffffffffffp-168, chars_format::scientific, 169,
        "5."
        "3455294201843906988080888243155420405527519125877752571135103830742368894880112905279820918179914567284752"
        "725874312441857729382398145556720692184171639382839202880859375e-51"},
    {0x1.fffffffffffffp-167, chars_format::scientific, 169,
        "1."
        "0691058840368781397616177648631084081105503825175550514227020766148473778976022581055964183635982913456950"
        "545174862488371545876479629111344138436834327876567840576171875e-50"},
    {0x1.fffffffffffffp-166, chars_format::scientific, 168,
        "2."
        "1382117680737562795232355297262168162211007650351101028454041532296947557952045162111928367271965826913901"
        "09034972497674309175295925822268827687366865575313568115234375e-50"},
    {0x1.fffffffffffffp-165, chars_format::scientific, 167,
        "4."
        "2764235361475125590464710594524336324422015300702202056908083064593895115904090324223856734543931653827802"
        "1806994499534861835059185164453765537473373115062713623046875e-50"},
    {0x1.fffffffffffffp-164, chars_format::scientific, 166,
        "8."
        "5528470722950251180929421189048672648844030601404404113816166129187790231808180648447713469087863307655604"
        "361398899906972367011837032890753107494674623012542724609375e-50"},
    {0x1.fffffffffffffp-163, chars_format::scientific, 166,
        "1."
        "7105694144590050236185884237809734529768806120280880822763233225837558046361636129689542693817572661531120"
        "872279779981394473402367406578150621498934924602508544921875e-49"},
    {0x1.fffffffffffffp-162, chars_format::scientific, 165,
        "3."
        "4211388289180100472371768475619469059537612240561761645526466451675116092723272259379085387635145323062241"
        "74455955996278894680473481315630124299786984920501708984375e-49"},
    {0x1.fffffffffffffp-161, chars_format::scientific, 164,
        "6."
        "8422776578360200944743536951238938119075224481123523291052932903350232185446544518758170775270290646124483"
        "4891191199255778936094696263126024859957396984100341796875e-49"},
    {0x1.fffffffffffffp-160, chars_format::scientific, 164,
        "1."
        "3684555315672040188948707390247787623815044896224704658210586580670046437089308903751634155054058129224896"
        "6978238239851155787218939252625204971991479396820068359375e-48"},
    {0x1.fffffffffffffp-159, chars_format::scientific, 163,
        "2."
        "7369110631344080377897414780495575247630089792449409316421173161340092874178617807503268310108116258449793"
        "395647647970231157443787850525040994398295879364013671875e-48"},
    {0x1.fffffffffffffp-158, chars_format::scientific, 162,
        "5."
        "4738221262688160755794829560991150495260179584898818632842346322680185748357235615006536620216232516899586"
        "79129529594046231488757570105008198879659175872802734375e-48"},
    {0x1.fffffffffffffp-157, chars_format::scientific, 162,
        "1."
        "0947644252537632151158965912198230099052035916979763726568469264536037149671447123001307324043246503379917"
        "35825905918809246297751514021001639775931835174560546875e-47"},
    {0x1.fffffffffffffp-156, chars_format::scientific, 161,
        "2."
        "1895288505075264302317931824396460198104071833959527453136938529072074299342894246002614648086493006759834"
        "7165181183761849259550302804200327955186367034912109375e-47"},
    {0x1.fffffffffffffp-155, chars_format::scientific, 160,
        "4."
        "3790577010150528604635863648792920396208143667919054906273877058144148598685788492005229296172986013519669"
        "433036236752369851910060560840065591037273406982421875e-47"},
    {0x1.fffffffffffffp-154, chars_format::scientific, 159,
        "8."
        "7581154020301057209271727297585840792416287335838109812547754116288297197371576984010458592345972027039338"
        "86607247350473970382012112168013118207454681396484375e-47"},
    {0x1.fffffffffffffp-153, chars_format::scientific, 159,
        "1."
        "7516230804060211441854345459517168158483257467167621962509550823257659439474315396802091718469194405407867"
        "77321449470094794076402422433602623641490936279296875e-46"},
    {0x1.fffffffffffffp-152, chars_format::scientific, 158,
        "3."
        "5032461608120422883708690919034336316966514934335243925019101646515318878948630793604183436938388810815735"
        "5464289894018958815280484486720524728298187255859375e-46"},
    {0x1.fffffffffffffp-151, chars_format::scientific, 157,
        "7."
        "0064923216240845767417381838068672633933029868670487850038203293030637757897261587208366873876777621631471"
        "092857978803791763056096897344104945659637451171875e-46"},
    {0x1.fffffffffffffp-150, chars_format::scientific, 157,
        "1."
        "4012984643248169153483476367613734526786605973734097570007640658606127551579452317441673374775355524326294"
        "218571595760758352611219379468820989131927490234375e-45"},
    {0x1.fffffffffffffp-149, chars_format::scientific, 156,
        "2."
        "8025969286496338306966952735227469053573211947468195140015281317212255103158904634883346749550711048652588"
        "43714319152151670522243875893764197826385498046875e-45"},
    {0x1.fffffffffffffp-148, chars_format::scientific, 155,
        "5."
        "6051938572992676613933905470454938107146423894936390280030562634424510206317809269766693499101422097305176"
        "8742863830430334104448775178752839565277099609375e-45"},
    {0x1.fffffffffffffp-147, chars_format::scientific, 155,
        "1."
        "1210387714598535322786781094090987621429284778987278056006112526884902041263561853953338699820284419461035"
        "3748572766086066820889755035750567913055419921875e-44"},
    {0x1.fffffffffffffp-146, chars_format::scientific, 154,
        "2."
        "2420775429197070645573562188181975242858569557974556112012225053769804082527123707906677399640568838922070"
        "749714553217213364177951007150113582611083984375e-44"},
    {0x1.fffffffffffffp-145, chars_format::scientific, 153,
        "4."
        "4841550858394141291147124376363950485717139115949112224024450107539608165054247415813354799281137677844141"
        "49942910643442672835590201430022716522216796875e-44"},
    {0x1.fffffffffffffp-144, chars_format::scientific, 152,
        "8."
        "9683101716788282582294248752727900971434278231898224448048900215079216330108494831626709598562275355688282"
        "9988582128688534567118040286004543304443359375e-44"},
    {0x1.fffffffffffffp-143, chars_format::scientific, 152,
        "1."
        "7936620343357656516458849750545580194286855646379644889609780043015843266021698966325341919712455071137656"
        "5997716425737706913423608057200908660888671875e-43"},
    {0x1.fffffffffffffp-142, chars_format::scientific, 151,
        "3."
        "5873240686715313032917699501091160388573711292759289779219560086031686532043397932650683839424910142275313"
        "199543285147541382684721611440181732177734375e-43"},
    {0x1.fffffffffffffp-141, chars_format::scientific, 150,
        "7."
        "1746481373430626065835399002182320777147422585518579558439120172063373064086795865301367678849820284550626"
        "39908657029508276536944322288036346435546875e-43"},
    {0x1.fffffffffffffp-140, chars_format::scientific, 150,
        "1."
        "4349296274686125213167079800436464155429484517103715911687824034412674612817359173060273535769964056910125"
        "27981731405901655307388864457607269287109375e-42"},
    {0x1.fffffffffffffp-139, chars_format::scientific, 149,
        "2."
        "8698592549372250426334159600872928310858969034207431823375648068825349225634718346120547071539928113820250"
        "5596346281180331061477772891521453857421875e-42"},
    {0x1.fffffffffffffp-138, chars_format::scientific, 148,
        "5."
        "7397185098744500852668319201745856621717938068414863646751296137650698451269436692241094143079856227640501"
        "119269256236066212295554578304290771484375e-42"},
    {0x1.fffffffffffffp-137, chars_format::scientific, 148,
        "1."
        "1479437019748900170533663840349171324343587613682972729350259227530139690253887338448218828615971245528100"
        "223853851247213242459110915660858154296875e-41"},
    {0x1.fffffffffffffp-136, chars_format::scientific, 147,
        "2."
        "2958874039497800341067327680698342648687175227365945458700518455060279380507774676896437657231942491056200"
        "44770770249442648491822183132171630859375e-41"},
    {0x1.fffffffffffffp-135, chars_format::scientific, 146,
        "4."
        "5917748078995600682134655361396685297374350454731890917401036910120558761015549353792875314463884982112400"
        "8954154049888529698364436626434326171875e-41"},
    {0x1.fffffffffffffp-134, chars_format::scientific, 145,
        "9."
        "1835496157991201364269310722793370594748700909463781834802073820241117522031098707585750628927769964224801"
        "790830809977705939672887325286865234375e-41"},
    {0x1.fffffffffffffp-133, chars_format::scientific, 145,
        "1."
        "8367099231598240272853862144558674118949740181892756366960414764048223504406219741517150125785553992844960"
        "358166161995541187934577465057373046875e-40"},
    {0x1.fffffffffffffp-132, chars_format::scientific, 144,
        "3."
        "6734198463196480545707724289117348237899480363785512733920829528096447008812439483034300251571107985689920"
        "71633232399108237586915493011474609375e-40"},
    {0x1.fffffffffffffp-131, chars_format::scientific, 143,
        "7."
        "3468396926392961091415448578234696475798960727571025467841659056192894017624878966068600503142215971379841"
        "4326646479821647517383098602294921875e-40"},
    {0x1.fffffffffffffp-130, chars_format::scientific, 143,
        "1."
        "4693679385278592218283089715646939295159792145514205093568331811238578803524975793213720100628443194275968"
        "2865329295964329503476619720458984375e-39"},
    {0x1.fffffffffffffp-129, chars_format::scientific, 142,
        "2."
        "9387358770557184436566179431293878590319584291028410187136663622477157607049951586427440201256886388551936"
        "573065859192865900695323944091796875e-39"},
    {0x1.fffffffffffffp-128, chars_format::scientific, 141,
        "5."
        "8774717541114368873132358862587757180639168582056820374273327244954315214099903172854880402513772777103873"
        "14613171838573180139064788818359375e-39"},
    {0x1.fffffffffffffp-127, chars_format::scientific, 141,
        "1."
        "1754943508222873774626471772517551436127833716411364074854665448990863042819980634570976080502754555420774"
        "62922634367714636027812957763671875e-38"},
    {0x1.fffffffffffffp-126, chars_format::scientific, 140,
        "2."
        "3509887016445747549252943545035102872255667432822728149709330897981726085639961269141952161005509110841549"
        "2584526873542927205562591552734375e-38"},
    {0x1.fffffffffffffp-125, chars_format::scientific, 139,
        "4."
        "7019774032891495098505887090070205744511334865645456299418661795963452171279922538283904322011018221683098"
        "516905374708585441112518310546875e-38"},
    {0x1.fffffffffffffp-124, chars_format::scientific, 138,
        "9."
        "4039548065782990197011774180140411489022669731290912598837323591926904342559845076567808644022036443366197"
        "03381074941717088222503662109375e-38"},
    {0x1.fffffffffffffp-123, chars_format::scientific, 138,
        "1."
        "8807909613156598039402354836028082297804533946258182519767464718385380868511969015313561728804407288673239"
        "40676214988343417644500732421875e-37"},
    {0x1.fffffffffffffp-122, chars_format::scientific, 137,
        "3."
        "7615819226313196078804709672056164595609067892516365039534929436770761737023938030627123457608814577346478"
        "8135242997668683528900146484375e-37"},
    {0x1.fffffffffffffp-121, chars_format::scientific, 136,
        "7."
        "5231638452626392157609419344112329191218135785032730079069858873541523474047876061254246915217629154692957"
        "627048599533736705780029296875e-37"},
    {0x1.fffffffffffffp-120, chars_format::scientific, 136,
        "1."
        "5046327690525278431521883868822465838243627157006546015813971774708304694809575212250849383043525830938591"
        "525409719906747341156005859375e-36"},
    {0x1.fffffffffffffp-119, chars_format::scientific, 135,
        "3."
        "0092655381050556863043767737644931676487254314013092031627943549416609389619150424501698766087051661877183"
        "05081943981349468231201171875e-36"},
    {0x1.fffffffffffffp-118, chars_format::scientific, 134,
        "6."
        "0185310762101113726087535475289863352974508628026184063255887098833218779238300849003397532174103323754366"
        "1016388796269893646240234375e-36"},
    {0x1.fffffffffffffp-117, chars_format::scientific, 134,
        "1."
        "2037062152420222745217507095057972670594901725605236812651177419766643755847660169800679506434820664750873"
        "2203277759253978729248046875e-35"},
    {0x1.fffffffffffffp-116, chars_format::scientific, 133,
        "2."
        "4074124304840445490435014190115945341189803451210473625302354839533287511695320339601359012869641329501746"
        "440655551850795745849609375e-35"},
    {0x1.fffffffffffffp-115, chars_format::scientific, 132,
        "4."
        "8148248609680890980870028380231890682379606902420947250604709679066575023390640679202718025739282659003492"
        "88131110370159149169921875e-35"},
    {0x1.fffffffffffffp-114, chars_format::scientific, 131,
        "9."
        "6296497219361781961740056760463781364759213804841894501209419358133150046781281358405436051478565318006985"
        "7626222074031829833984375e-35"},
    {0x1.fffffffffffffp-113, chars_format::scientific, 131,
        "1."
        "9259299443872356392348011352092756272951842760968378900241883871626630009356256271681087210295713063601397"
        "1525244414806365966796875e-34"},
    {0x1.fffffffffffffp-112, chars_format::scientific, 130,
        "3."
        "8518598887744712784696022704185512545903685521936757800483767743253260018712512543362174420591426127202794"
        "305048882961273193359375e-34"},
    {0x1.fffffffffffffp-111, chars_format::scientific, 129,
        "7."
        "7037197775489425569392045408371025091807371043873515600967535486506520037425025086724348841182852254405588"
        "61009776592254638671875e-34"},
    {0x1.fffffffffffffp-110, chars_format::scientific, 129,
        "1."
        "5407439555097885113878409081674205018361474208774703120193507097301304007485005017344869768236570450881117"
        "72201955318450927734375e-33"},
    {0x1.fffffffffffffp-109, chars_format::scientific, 128,
        "3."
        "0814879110195770227756818163348410036722948417549406240387014194602608014970010034689739536473140901762235"
        "4440391063690185546875e-33"},
    {0x1.fffffffffffffp-108, chars_format::scientific, 127,
        "6."
        "1629758220391540455513636326696820073445896835098812480774028389205216029940020069379479072946281803524470"
        "888078212738037109375e-33"},
    {0x1.fffffffffffffp-107, chars_format::scientific, 127,
        "1."
        "2325951644078308091102727265339364014689179367019762496154805677841043205988004013875895814589256360704894"
        "177615642547607421875e-32"},
    {0x1.fffffffffffffp-106, chars_format::scientific, 126,
        "2."
        "4651903288156616182205454530678728029378358734039524992309611355682086411976008027751791629178512721409788"
        "35523128509521484375e-32"},
    {0x1.fffffffffffffp-105, chars_format::scientific, 125,
        "4."
        "9303806576313232364410909061357456058756717468079049984619222711364172823952016055503583258357025442819576"
        "7104625701904296875e-32"},
    {0x1.fffffffffffffp-104, chars_format::scientific, 124,
        "9."
        "8607613152626464728821818122714912117513434936158099969238445422728345647904032111007166516714050885639153"
        "420925140380859375e-32"},
    {0x1.fffffffffffffp-103, chars_format::scientific, 124,
        "1."
        "9721522630525292945764363624542982423502686987231619993847689084545669129580806422201433303342810177127830"
        "684185028076171875e-31"},
    {0x1.fffffffffffffp-102, chars_format::scientific, 123,
        "3."
        "9443045261050585891528727249085964847005373974463239987695378169091338259161612844402866606685620354255661"
        "36837005615234375e-31"},
    {0x1.fffffffffffffp-101, chars_format::scientific, 122,
        "7."
        "8886090522101171783057454498171929694010747948926479975390756338182676518323225688805733213371240708511322"
        "7367401123046875e-31"},
    {0x1.fffffffffffffp-100, chars_format::scientific, 122,
        "1."
        "5777218104420234356611490899634385938802149589785295995078151267636535303664645137761146642674248141702264"
        "5473480224609375e-30"},
    {0x1.fffffffffffffp-99, chars_format::scientific, 121,
        "3."
        "1554436208840468713222981799268771877604299179570591990156302535273070607329290275522293285348496283404529"
        "094696044921875e-30"},
    {0x1.fffffffffffffp-98, chars_format::scientific, 120,
        "6."
        "3108872417680937426445963598537543755208598359141183980312605070546141214658580551044586570696992566809058"
        "18939208984375e-30"},
    {0x1.fffffffffffffp-97, chars_format::scientific, 120,
        "1."
        "2621774483536187485289192719707508751041719671828236796062521014109228242931716110208917314139398513361811"
        "63787841796875e-29"},
    {0x1.fffffffffffffp-96, chars_format::scientific, 119,
        "2."
        "5243548967072374970578385439415017502083439343656473592125042028218456485863432220417834628278797026723623"
        "2757568359375e-29"},
    {0x1.fffffffffffffp-95, chars_format::scientific, 118,
        "5."
        "0487097934144749941156770878830035004166878687312947184250084056436912971726864440835669256557594053447246"
        "551513671875e-29"},
    {0x1.fffffffffffffp-94, chars_format::scientific, 118,
        "1."
        "0097419586828949988231354175766007000833375737462589436850016811287382594345372888167133851311518810689449"
        "310302734375e-28"},
    {0x1.fffffffffffffp-93, chars_format::scientific, 117,
        "2."
        "0194839173657899976462708351532014001666751474925178873700033622574765188690745776334267702623037621378898"
        "62060546875e-28"},
    {0x1.fffffffffffffp-92, chars_format::scientific, 116,
        "4."
        "0389678347315799952925416703064028003333502949850357747400067245149530377381491552668535405246075242757797"
        "2412109375e-28"},
    {0x1.fffffffffffffp-91, chars_format::scientific, 115,
        "8."
        "0779356694631599905850833406128056006667005899700715494800134490299060754762983105337070810492150485515594"
        "482421875e-28"},
    {0x1.fffffffffffffp-90, chars_format::scientific, 115,
        "1."
        "6155871338926319981170166681225611201333401179940143098960026898059812150952596621067414162098430097103118"
        "896484375e-27"},
    {0x1.fffffffffffffp-89, chars_format::scientific, 114,
        "3."
        "2311742677852639962340333362451222402666802359880286197920053796119624301905193242134828324196860194206237"
        "79296875e-27"},
    {0x1.fffffffffffffp-88, chars_format::scientific, 113,
        "6."
        "4623485355705279924680666724902444805333604719760572395840107592239248603810386484269656648393720388412475"
        "5859375e-27"},
    {0x1.fffffffffffffp-87, chars_format::scientific, 113,
        "1."
        "2924697071141055984936133344980488961066720943952114479168021518447849720762077296853931329678744077682495"
        "1171875e-26"},
    {0x1.fffffffffffffp-86, chars_format::scientific, 112,
        "2."
        "5849394142282111969872266689960977922133441887904228958336043036895699441524154593707862659357488155364990"
        "234375e-26"},
    {0x1.fffffffffffffp-85, chars_format::scientific, 111,
        "5."
        "1698788284564223939744533379921955844266883775808457916672086073791398883048309187415725318714976310729980"
        "46875e-26"},
    {0x1.fffffffffffffp-84, chars_format::scientific, 111,
        "1."
        "0339757656912844787948906675984391168853376755161691583334417214758279776609661837483145063742995262145996"
        "09375e-25"},
    {0x1.fffffffffffffp-83, chars_format::scientific, 110,
        "2."
        "0679515313825689575897813351968782337706753510323383166668834429516559553219323674966290127485990524291992"
        "1875e-25"},
    {0x1.fffffffffffffp-82, chars_format::scientific, 109,
        "4."
        "1359030627651379151795626703937564675413507020646766333337668859033119106438647349932580254971981048583984"
        "375e-25"},
    {0x1.fffffffffffffp-81, chars_format::scientific, 108,
        "8."
        "2718061255302758303591253407875129350827014041293532666675337718066238212877294699865160509943962097167968"
        "75e-25"},
    {0x1.fffffffffffffp-80, chars_format::scientific, 108,
        "1."
        "6543612251060551660718250681575025870165402808258706533335067543613247642575458939973032101988792419433593"
        "75e-24"},
    {0x1.fffffffffffffp-79, chars_format::scientific, 107,
        "3."
        "3087224502121103321436501363150051740330805616517413066670135087226495285150917879946064203977584838867187"
        "5e-24"},
    {0x1.fffffffffffffp-78, chars_format::scientific, 106,
        "6."
        "6174449004242206642873002726300103480661611233034826133340270174452990570301835759892128407955169677734375"
        "e-24"},
    {0x1.fffffffffffffp-77, chars_format::scientific, 106,
        "1."
        "3234889800848441328574600545260020696132322246606965226668054034890598114060367151978425681591033935546875"
        "e-23"},
    {0x1.fffffffffffffp-76, chars_format::scientific, 105,
        "2."
        "646977960169688265714920109052004139226464449321393045333610806978119622812073430395685136318206787109375e"
        "-23"},
    {0x1.fffffffffffffp-75, chars_format::scientific, 104,
        "5."
        "29395592033937653142984021810400827845292889864278609066722161395623924562414686079137027263641357421875e-"
        "23"},
    {0x1.fffffffffffffp-74, chars_format::scientific, 104,
        "1."
        "05879118406787530628596804362080165569058577972855721813344432279124784912482937215827405452728271484375e-"
        "22"},
    {0x1.fffffffffffffp-73, chars_format::scientific, 103,
        "2."
        "1175823681357506125719360872416033113811715594571144362668886455824956982496587443165481090545654296875e-"
        "22"},
    {0x1.fffffffffffffp-72, chars_format::scientific, 102,
        "4.235164736271501225143872174483206622762343118914228872533777291164991396499317488633096218109130859375e-"
        "22"},
    {0x1.fffffffffffffp-71, chars_format::scientific, 101,
        "8.47032947254300245028774434896641324552468623782845774506755458232998279299863497726619243621826171875e-"
        "22"},
    {0x1.fffffffffffffp-70, chars_format::scientific, 101,
        "1.69406589450860049005754886979328264910493724756569154901351091646599655859972699545323848724365234375e-"
        "21"},
    {0x1.fffffffffffffp-69, chars_format::scientific, 100,
        "3.3881317890172009801150977395865652982098744951313830980270218329319931171994539909064769744873046875e-"
        "21"},
    {0x1.fffffffffffffp-68, chars_format::scientific, 99,
        "6.776263578034401960230195479173130596419748990262766196054043665863986234398907981812953948974609375e-"
        "21"},
    {0x1.fffffffffffffp-67, chars_format::scientific, 99,
        "1.355252715606880392046039095834626119283949798052553239210808733172797246879781596362590789794921875e-"
        "20"},
    {0x1.fffffffffffffp-66, chars_format::scientific, 98,
        "2.71050543121376078409207819166925223856789959610510647842161746634559449375956319272518157958984375e-20"},
    {0x1.fffffffffffffp-65, chars_format::scientific, 97,
        "5.4210108624275215681841563833385044771357991922102129568432349326911889875191263854503631591796875e-20"},
    {0x1.fffffffffffffp-64, chars_format::scientific, 97,
        "1.0842021724855043136368312766677008954271598384420425913686469865382377975038252770900726318359375e-19"},
    {0x1.fffffffffffffp-63, chars_format::scientific, 96,
        "2.168404344971008627273662553335401790854319676884085182737293973076475595007650554180145263671875e-19"},
    {0x1.fffffffffffffp-62, chars_format::scientific, 95,
        "4.33680868994201725454732510667080358170863935376817036547458794615295119001530110836029052734375e-19"},
    {0x1.fffffffffffffp-61, chars_format::scientific, 94,
        "8.6736173798840345090946502133416071634172787075363407309491758923059023800306022167205810546875e-19"},
    {0x1.fffffffffffffp-60, chars_format::scientific, 94,
        "1.7347234759768069018189300426683214326834557415072681461898351784611804760061204433441162109375e-18"},
    {0x1.fffffffffffffp-59, chars_format::scientific, 93,
        "3.469446951953613803637860085336642865366911483014536292379670356922360952012240886688232421875e-18"},
    {0x1.fffffffffffffp-58, chars_format::scientific, 92,
        "6.93889390390722760727572017067328573073382296602907258475934071384472190402448177337646484375e-18"},
    {0x1.fffffffffffffp-57, chars_format::scientific, 92,
        "1.38777878078144552145514403413465714614676459320581451695186814276894438080489635467529296875e-17"},
    {0x1.fffffffffffffp-56, chars_format::scientific, 91,
        "2.7755575615628910429102880682693142922935291864116290339037362855378887616097927093505859375e-17"},
    {0x1.fffffffffffffp-55, chars_format::scientific, 90,
        "5.551115123125782085820576136538628584587058372823258067807472571075777523219585418701171875e-17"},
    {0x1.fffffffffffffp-54, chars_format::scientific, 90,
        "1.110223024625156417164115227307725716917411674564651613561494514215155504643917083740234375e-16"},
    {0x1.fffffffffffffp-53, chars_format::scientific, 89,
        "2.22044604925031283432823045461545143383482334912930322712298902843031100928783416748046875e-16"},
    {0x1.fffffffffffffp-52, chars_format::scientific, 88,
        "4.4408920985006256686564609092309028676696466982586064542459780568606220185756683349609375e-16"},
    {0x1.fffffffffffffp-51, chars_format::scientific, 87,
        "8.881784197001251337312921818461805735339293396517212908491956113721244037151336669921875e-16"},
    {0x1.fffffffffffffp-50, chars_format::scientific, 87,
        "1.776356839400250267462584363692361147067858679303442581698391222744248807430267333984375e-15"},
    {0x1.fffffffffffffp-49, chars_format::scientific, 86,
        "3.55271367880050053492516872738472229413571735860688516339678244548849761486053466796875e-15"},
    {0x1.fffffffffffffp-48, chars_format::scientific, 85,
        "7.1054273576010010698503374547694445882714347172137703267935648909769952297210693359375e-15"},
    {0x1.fffffffffffffp-47, chars_format::scientific, 85,
        "1.4210854715202002139700674909538889176542869434427540653587129781953990459442138671875e-14"},
    {0x1.fffffffffffffp-46, chars_format::scientific, 84,
        "2.842170943040400427940134981907777835308573886885508130717425956390798091888427734375e-14"},
    {0x1.fffffffffffffp-45, chars_format::scientific, 83,
        "5.68434188608080085588026996381555567061714777377101626143485191278159618377685546875e-14"},
    {0x1.fffffffffffffp-44, chars_format::scientific, 83,
        "1.13686837721616017117605399276311113412342955475420325228697038255631923675537109375e-13"},
    {0x1.fffffffffffffp-43, chars_format::scientific, 82,
        "2.2737367544323203423521079855262222682468591095084065045739407651126384735107421875e-13"},
    {0x1.fffffffffffffp-42, chars_format::scientific, 81,
        "4.547473508864640684704215971052444536493718219016813009147881530225276947021484375e-13"},
    {0x1.fffffffffffffp-41, chars_format::scientific, 80,
        "9.09494701772928136940843194210488907298743643803362601829576306045055389404296875e-13"},
    {0x1.fffffffffffffp-40, chars_format::scientific, 80,
        "1.81898940354585627388168638842097781459748728760672520365915261209011077880859375e-12"},
    {0x1.fffffffffffffp-39, chars_format::scientific, 79,
        "3.6379788070917125477633727768419556291949745752134504073183052241802215576171875e-12"},
    {0x1.fffffffffffffp-38, chars_format::scientific, 78,
        "7.275957614183425095526745553683911258389949150426900814636610448360443115234375e-12"},
    {0x1.fffffffffffffp-37, chars_format::scientific, 78,
        "1.455191522836685019105349110736782251677989830085380162927322089672088623046875e-11"},
    {0x1.fffffffffffffp-36, chars_format::scientific, 77,
        "2.91038304567337003821069822147356450335597966017076032585464417934417724609375e-11"},
    {0x1.fffffffffffffp-35, chars_format::scientific, 76,
        "5.8207660913467400764213964429471290067119593203415206517092883586883544921875e-11"},
    {0x1.fffffffffffffp-34, chars_format::scientific, 76,
        "1.1641532182693480152842792885894258013423918640683041303418576717376708984375e-10"},
    {0x1.fffffffffffffp-33, chars_format::scientific, 75,
        "2.328306436538696030568558577178851602684783728136608260683715343475341796875e-10"},
    {0x1.fffffffffffffp-32, chars_format::scientific, 74,
        "4.65661287307739206113711715435770320536956745627321652136743068695068359375e-10"},
    {0x1.fffffffffffffp-31, chars_format::scientific, 73,
        "9.3132257461547841222742343087154064107391349125464330427348613739013671875e-10"},
    {0x1.fffffffffffffp-30, chars_format::scientific, 73,
        "1.8626451492309568244548468617430812821478269825092866085469722747802734375e-09"},
    {0x1.fffffffffffffp-29, chars_format::scientific, 72,
        "3.725290298461913648909693723486162564295653965018573217093944549560546875e-09"},
    {0x1.fffffffffffffp-28, chars_format::scientific, 71,
        "7.45058059692382729781938744697232512859130793003714643418788909912109375e-09"},
    {0x1.fffffffffffffp-27, chars_format::scientific, 71,
        "1.49011611938476545956387748939446502571826158600742928683757781982421875e-08"},
    {0x1.fffffffffffffp-26, chars_format::scientific, 70,
        "2.9802322387695309191277549787889300514365231720148585736751556396484375e-08"},
    {0x1.fffffffffffffp-25, chars_format::scientific, 69,
        "5.960464477539061838255509957577860102873046344029717147350311279296875e-08"},
    {0x1.fffffffffffffp-24, chars_format::scientific, 69,
        "1.192092895507812367651101991515572020574609268805943429470062255859375e-07"},
    {0x1.fffffffffffffp-23, chars_format::scientific, 68,
        "2.38418579101562473530220398303114404114921853761188685894012451171875e-07"},
    {0x1.fffffffffffffp-22, chars_format::scientific, 67,
        "4.7683715820312494706044079660622880822984370752237737178802490234375e-07"},
    {0x1.fffffffffffffp-21, chars_format::scientific, 66,
        "9.536743164062498941208815932124576164596874150447547435760498046875e-07"},
    {0x1.fffffffffffffp-20, chars_format::scientific, 66,
        "1.907348632812499788241763186424915232919374830089509487152099609375e-06"},
    {0x1.fffffffffffffp-19, chars_format::scientific, 65,
        "3.81469726562499957648352637284983046583874966017901897430419921875e-06"},
    {0x1.fffffffffffffp-18, chars_format::scientific, 64,
        "7.6293945312499991529670527456996609316774993203580379486083984375e-06"},
    {0x1.fffffffffffffp-17, chars_format::scientific, 64,
        "1.5258789062499998305934105491399321863354998640716075897216796875e-05"},
    {0x1.fffffffffffffp-16, chars_format::scientific, 63,
        "3.051757812499999661186821098279864372670999728143215179443359375e-05"},
    {0x1.fffffffffffffp-15, chars_format::scientific, 62,
        "6.10351562499999932237364219655972874534199945628643035888671875e-05"},
    {0x1.fffffffffffffp-14, chars_format::scientific, 62,
        "1.22070312499999986447472843931194574906839989125728607177734375e-04"},
    {0x1.fffffffffffffp-13, chars_format::scientific, 61,
        "2.4414062499999997289494568786238914981367997825145721435546875e-04"},
    {0x1.fffffffffffffp-12, chars_format::scientific, 60,
        "4.882812499999999457898913757247782996273599565029144287109375e-04"},
    {0x1.fffffffffffffp-11, chars_format::scientific, 59,
        "9.76562499999999891579782751449556599254719913005828857421875e-04"},
    {0x1.fffffffffffffp-10, chars_format::scientific, 59,
        "1.95312499999999978315956550289911319850943982601165771484375e-03"},
    {0x1.fffffffffffffp-9, chars_format::scientific, 58,
        "3.9062499999999995663191310057982263970188796520233154296875e-03"},
    {0x1.fffffffffffffp-8, chars_format::scientific, 57,
        "7.812499999999999132638262011596452794037759304046630859375e-03"},
    {0x1.fffffffffffffp-7, chars_format::scientific, 57,
        "1.562499999999999826527652402319290558807551860809326171875e-02"},
    {0x1.fffffffffffffp-6, chars_format::scientific, 56,
        "3.12499999999999965305530480463858111761510372161865234375e-02"},
    {0x1.fffffffffffffp-5, chars_format::scientific, 55,
        "6.2499999999999993061106096092771622352302074432373046875e-02"},
    {0x1.fffffffffffffp-4, chars_format::scientific, 55,
        "1.2499999999999998612221219218554324470460414886474609375e-01"},
    {0x1.fffffffffffffp-3, chars_format::scientific, 54,
        "2.499999999999999722444243843710864894092082977294921875e-01"},
    {0x1.fffffffffffffp-2, chars_format::scientific, 53, "4.99999999999999944488848768742172978818416595458984375e-01"},
    {0x1.fffffffffffffp-1, chars_format::scientific, 52, "9.9999999999999988897769753748434595763683319091796875e-01"},
    {0x1.fffffffffffffp+0, chars_format::scientific, 52, "1.9999999999999997779553950749686919152736663818359375e+00"},
    {0x1.fffffffffffffp+1, chars_format::scientific, 51, "3.999999999999999555910790149937383830547332763671875e+00"},
    {0x1.fffffffffffffp+2, chars_format::scientific, 50, "7.99999999999999911182158029987476766109466552734375e+00"},
    {0x1.fffffffffffffp+3, chars_format::scientific, 50, "1.59999999999999982236431605997495353221893310546875e+01"},
    {0x1.fffffffffffffp+4, chars_format::scientific, 49, "3.1999999999999996447286321199499070644378662109375e+01"},
    {0x1.fffffffffffffp+5, chars_format::scientific, 48, "6.399999999999999289457264239899814128875732421875e+01"},
    {0x1.fffffffffffffp+6, chars_format::scientific, 48, "1.279999999999999857891452847979962825775146484375e+02"},
    {0x1.fffffffffffffp+7, chars_format::scientific, 47, "2.55999999999999971578290569595992565155029296875e+02"},
    {0x1.fffffffffffffp+8, chars_format::scientific, 46, "5.1199999999999994315658113919198513031005859375e+02"},
    {0x1.fffffffffffffp+9, chars_format::scientific, 46, "1.0239999999999998863131622783839702606201171875e+03"},
    {0x1.fffffffffffffp+10, chars_format::scientific, 45, "2.047999999999999772626324556767940521240234375e+03"},
    {0x1.fffffffffffffp+11, chars_format::scientific, 44, "4.09599999999999954525264911353588104248046875e+03"},
    {0x1.fffffffffffffp+12, chars_format::scientific, 43, "8.1919999999999990905052982270717620849609375e+03"},
    {0x1.fffffffffffffp+13, chars_format::scientific, 43, "1.6383999999999998181010596454143524169921875e+04"},
    {0x1.fffffffffffffp+14, chars_format::scientific, 42, "3.276799999999999636202119290828704833984375e+04"},
    {0x1.fffffffffffffp+15, chars_format::scientific, 41, "6.55359999999999927240423858165740966796875e+04"},
    {0x1.fffffffffffffp+16, chars_format::scientific, 41, "1.31071999999999985448084771633148193359375e+05"},
    {0x1.fffffffffffffp+17, chars_format::scientific, 40, "2.6214399999999997089616954326629638671875e+05"},
    {0x1.fffffffffffffp+18, chars_format::scientific, 39, "5.242879999999999417923390865325927734375e+05"},
    {0x1.fffffffffffffp+19, chars_format::scientific, 39, "1.048575999999999883584678173065185546875e+06"},
    {0x1.fffffffffffffp+20, chars_format::scientific, 38, "2.09715199999999976716935634613037109375e+06"},
    {0x1.fffffffffffffp+21, chars_format::scientific, 37, "4.1943039999999995343387126922607421875e+06"},
    {0x1.fffffffffffffp+22, chars_format::scientific, 36, "8.388607999999999068677425384521484375e+06"},
    {0x1.fffffffffffffp+23, chars_format::scientific, 36, "1.677721599999999813735485076904296875e+07"},
    {0x1.fffffffffffffp+24, chars_format::scientific, 35, "3.35544319999999962747097015380859375e+07"},
    {0x1.fffffffffffffp+25, chars_format::scientific, 34, "6.7108863999999992549419403076171875e+07"},
    {0x1.fffffffffffffp+26, chars_format::scientific, 34, "1.3421772799999998509883880615234375e+08"},
    {0x1.fffffffffffffp+27, chars_format::scientific, 33, "2.684354559999999701976776123046875e+08"},
    {0x1.fffffffffffffp+28, chars_format::scientific, 32, "5.36870911999999940395355224609375e+08"},
    {0x1.fffffffffffffp+29, chars_format::scientific, 32, "1.07374182399999988079071044921875e+09"},
    {0x1.fffffffffffffp+30, chars_format::scientific, 31, "2.1474836479999997615814208984375e+09"},
    {0x1.fffffffffffffp+31, chars_format::scientific, 30, "4.294967295999999523162841796875e+09"},
    {0x1.fffffffffffffp+32, chars_format::scientific, 29, "8.58993459199999904632568359375e+09"},
    {0x1.fffffffffffffp+33, chars_format::scientific, 29, "1.71798691839999980926513671875e+10"},
    {0x1.fffffffffffffp+34, chars_format::scientific, 28, "3.4359738367999996185302734375e+10"},
    {0x1.fffffffffffffp+35, chars_format::scientific, 27, "6.871947673599999237060546875e+10"},
    {0x1.fffffffffffffp+36, chars_format::scientific, 27, "1.374389534719999847412109375e+11"},
    {0x1.fffffffffffffp+37, chars_format::scientific, 26, "2.74877906943999969482421875e+11"},
    {0x1.fffffffffffffp+38, chars_format::scientific, 25, "5.4975581388799993896484375e+11"},
    {0x1.fffffffffffffp+39, chars_format::scientific, 25, "1.0995116277759998779296875e+12"},
    {0x1.fffffffffffffp+40, chars_format::scientific, 24, "2.199023255551999755859375e+12"},
    {0x1.fffffffffffffp+41, chars_format::scientific, 23, "4.39804651110399951171875e+12"},
    {0x1.fffffffffffffp+42, chars_format::scientific, 22, "8.7960930222079990234375e+12"},
    {0x1.fffffffffffffp+43, chars_format::scientific, 22, "1.7592186044415998046875e+13"},
    {0x1.fffffffffffffp+44, chars_format::scientific, 21, "3.518437208883199609375e+13"},
    {0x1.fffffffffffffp+45, chars_format::scientific, 20, "7.03687441776639921875e+13"},
    {0x1.fffffffffffffp+46, chars_format::scientific, 20, "1.40737488355327984375e+14"},
    {0x1.fffffffffffffp+47, chars_format::scientific, 19, "2.8147497671065596875e+14"},
    {0x1.fffffffffffffp+48, chars_format::scientific, 18, "5.629499534213119375e+14"},
    {0x1.fffffffffffffp+49, chars_format::scientific, 18, "1.125899906842623875e+15"},
    {0x1.fffffffffffffp+50, chars_format::scientific, 17, "2.25179981368524775e+15"},
    {0x1.fffffffffffffp+51, chars_format::scientific, 16, "4.5035996273704955e+15"},
    {0x1.fffffffffffffp+52, chars_format::scientific, 15, "9.007199254740991e+15"},
    {0x1.fffffffffffffp+53, chars_format::scientific, 16, "1.8014398509481982e+16"},
    {0x1.fffffffffffffp+54, chars_format::scientific, 16, "3.6028797018963964e+16"},
    {0x1.fffffffffffffp+55, chars_format::scientific, 16, "7.2057594037927928e+16"},
    {0x1.fffffffffffffp+56, chars_format::scientific, 17, "1.44115188075855856e+17"},
    {0x1.fffffffffffffp+57, chars_format::scientific, 17, "2.88230376151711712e+17"},
    {0x1.fffffffffffffp+58, chars_format::scientific, 17, "5.76460752303423424e+17"},
    {0x1.fffffffffffffp+59, chars_format::scientific, 18, "1.152921504606846848e+18"},
    {0x1.fffffffffffffp+60, chars_format::scientific, 18, "2.305843009213693696e+18"},
    {0x1.fffffffffffffp+61, chars_format::scientific, 18, "4.611686018427387392e+18"},
    {0x1.fffffffffffffp+62, chars_format::scientific, 18, "9.223372036854774784e+18"},
    {0x1.fffffffffffffp+63, chars_format::scientific, 19, "1.8446744073709549568e+19"},
    {0x1.fffffffffffffp+64, chars_format::scientific, 19, "3.6893488147419099136e+19"},
    {0x1.fffffffffffffp+65, chars_format::scientific, 19, "7.3786976294838198272e+19"},
    {0x1.fffffffffffffp+66, chars_format::scientific, 20, "1.47573952589676396544e+20"},
    {0x1.fffffffffffffp+67, chars_format::scientific, 20, "2.95147905179352793088e+20"},
    {0x1.fffffffffffffp+68, chars_format::scientific, 20, "5.90295810358705586176e+20"},
    {0x1.fffffffffffffp+69, chars_format::scientific, 21, "1.180591620717411172352e+21"},
    {0x1.fffffffffffffp+70, chars_format::scientific, 21, "2.361183241434822344704e+21"},
    {0x1.fffffffffffffp+71, chars_format::scientific, 21, "4.722366482869644689408e+21"},
    {0x1.fffffffffffffp+72, chars_format::scientific, 21, "9.444732965739289378816e+21"},
    {0x1.fffffffffffffp+73, chars_format::scientific, 22, "1.8889465931478578757632e+22"},
    {0x1.fffffffffffffp+74, chars_format::scientific, 22, "3.7778931862957157515264e+22"},
    {0x1.fffffffffffffp+75, chars_format::scientific, 22, "7.5557863725914315030528e+22"},
    {0x1.fffffffffffffp+76, chars_format::scientific, 23, "1.51115727451828630061056e+23"},
    {0x1.fffffffffffffp+77, chars_format::scientific, 23, "3.02231454903657260122112e+23"},
    {0x1.fffffffffffffp+78, chars_format::scientific, 23, "6.04462909807314520244224e+23"},
    {0x1.fffffffffffffp+79, chars_format::scientific, 24, "1.208925819614629040488448e+24"},
    {0x1.fffffffffffffp+80, chars_format::scientific, 24, "2.417851639229258080976896e+24"},
    {0x1.fffffffffffffp+81, chars_format::scientific, 24, "4.835703278458516161953792e+24"},
    {0x1.fffffffffffffp+82, chars_format::scientific, 24, "9.671406556917032323907584e+24"},
    {0x1.fffffffffffffp+83, chars_format::scientific, 25, "1.9342813113834064647815168e+25"},
    {0x1.fffffffffffffp+84, chars_format::scientific, 25, "3.8685626227668129295630336e+25"},
    {0x1.fffffffffffffp+85, chars_format::scientific, 25, "7.7371252455336258591260672e+25"},
    {0x1.fffffffffffffp+86, chars_format::scientific, 26, "1.54742504910672517182521344e+26"},
    {0x1.fffffffffffffp+87, chars_format::scientific, 26, "3.09485009821345034365042688e+26"},
    {0x1.fffffffffffffp+88, chars_format::scientific, 26, "6.18970019642690068730085376e+26"},
    {0x1.fffffffffffffp+89, chars_format::scientific, 27, "1.237940039285380137460170752e+27"},
    {0x1.fffffffffffffp+90, chars_format::scientific, 27, "2.475880078570760274920341504e+27"},
    {0x1.fffffffffffffp+91, chars_format::scientific, 27, "4.951760157141520549840683008e+27"},
    {0x1.fffffffffffffp+92, chars_format::scientific, 27, "9.903520314283041099681366016e+27"},
    {0x1.fffffffffffffp+93, chars_format::scientific, 28, "1.9807040628566082199362732032e+28"},
    {0x1.fffffffffffffp+94, chars_format::scientific, 28, "3.9614081257132164398725464064e+28"},
    {0x1.fffffffffffffp+95, chars_format::scientific, 28, "7.9228162514264328797450928128e+28"},
    {0x1.fffffffffffffp+96, chars_format::scientific, 29, "1.58456325028528657594901856256e+29"},
    {0x1.fffffffffffffp+97, chars_format::scientific, 29, "3.16912650057057315189803712512e+29"},
    {0x1.fffffffffffffp+98, chars_format::scientific, 29, "6.33825300114114630379607425024e+29"},
    {0x1.fffffffffffffp+99, chars_format::scientific, 30, "1.267650600228229260759214850048e+30"},
    {0x1.fffffffffffffp+100, chars_format::scientific, 30, "2.535301200456458521518429700096e+30"},
    {0x1.fffffffffffffp+101, chars_format::scientific, 30, "5.070602400912917043036859400192e+30"},
    {0x1.fffffffffffffp+102, chars_format::scientific, 31, "1.0141204801825834086073718800384e+31"},
    {0x1.fffffffffffffp+103, chars_format::scientific, 31, "2.0282409603651668172147437600768e+31"},
    {0x1.fffffffffffffp+104, chars_format::scientific, 31, "4.0564819207303336344294875201536e+31"},
    {0x1.fffffffffffffp+105, chars_format::scientific, 31, "8.1129638414606672688589750403072e+31"},
    {0x1.fffffffffffffp+106, chars_format::scientific, 32, "1.62259276829213345377179500806144e+32"},
    {0x1.fffffffffffffp+107, chars_format::scientific, 32, "3.24518553658426690754359001612288e+32"},
    {0x1.fffffffffffffp+108, chars_format::scientific, 32, "6.49037107316853381508718003224576e+32"},
    {0x1.fffffffffffffp+109, chars_format::scientific, 33, "1.298074214633706763017436006449152e+33"},
    {0x1.fffffffffffffp+110, chars_format::scientific, 33, "2.596148429267413526034872012898304e+33"},
    {0x1.fffffffffffffp+111, chars_format::scientific, 33, "5.192296858534827052069744025796608e+33"},
    {0x1.fffffffffffffp+112, chars_format::scientific, 34, "1.0384593717069654104139488051593216e+34"},
    {0x1.fffffffffffffp+113, chars_format::scientific, 34, "2.0769187434139308208278976103186432e+34"},
    {0x1.fffffffffffffp+114, chars_format::scientific, 34, "4.1538374868278616416557952206372864e+34"},
    {0x1.fffffffffffffp+115, chars_format::scientific, 34, "8.3076749736557232833115904412745728e+34"},
    {0x1.fffffffffffffp+116, chars_format::scientific, 35, "1.66153499473114465666231808825491456e+35"},
    {0x1.fffffffffffffp+117, chars_format::scientific, 35, "3.32306998946228931332463617650982912e+35"},
    {0x1.fffffffffffffp+118, chars_format::scientific, 35, "6.64613997892457862664927235301965824e+35"},
    {0x1.fffffffffffffp+119, chars_format::scientific, 36, "1.329227995784915725329854470603931648e+36"},
    {0x1.fffffffffffffp+120, chars_format::scientific, 36, "2.658455991569831450659708941207863296e+36"},
    {0x1.fffffffffffffp+121, chars_format::scientific, 36, "5.316911983139662901319417882415726592e+36"},
    {0x1.fffffffffffffp+122, chars_format::scientific, 37, "1.0633823966279325802638835764831453184e+37"},
    {0x1.fffffffffffffp+123, chars_format::scientific, 37, "2.1267647932558651605277671529662906368e+37"},
    {0x1.fffffffffffffp+124, chars_format::scientific, 37, "4.2535295865117303210555343059325812736e+37"},
    {0x1.fffffffffffffp+125, chars_format::scientific, 37, "8.5070591730234606421110686118651625472e+37"},
    {0x1.fffffffffffffp+126, chars_format::scientific, 38, "1.70141183460469212842221372237303250944e+38"},
    {0x1.fffffffffffffp+127, chars_format::scientific, 38, "3.40282366920938425684442744474606501888e+38"},
    {0x1.fffffffffffffp+128, chars_format::scientific, 38, "6.80564733841876851368885488949213003776e+38"},
    {0x1.fffffffffffffp+129, chars_format::scientific, 39, "1.361129467683753702737770977898426007552e+39"},
    {0x1.fffffffffffffp+130, chars_format::scientific, 39, "2.722258935367507405475541955796852015104e+39"},
    {0x1.fffffffffffffp+131, chars_format::scientific, 39, "5.444517870735014810951083911593704030208e+39"},
    {0x1.fffffffffffffp+132, chars_format::scientific, 40, "1.0889035741470029621902167823187408060416e+40"},
    {0x1.fffffffffffffp+133, chars_format::scientific, 40, "2.1778071482940059243804335646374816120832e+40"},
    {0x1.fffffffffffffp+134, chars_format::scientific, 40, "4.3556142965880118487608671292749632241664e+40"},
    {0x1.fffffffffffffp+135, chars_format::scientific, 40, "8.7112285931760236975217342585499264483328e+40"},
    {0x1.fffffffffffffp+136, chars_format::scientific, 41, "1.74224571863520473950434685170998528966656e+41"},
    {0x1.fffffffffffffp+137, chars_format::scientific, 41, "3.48449143727040947900869370341997057933312e+41"},
    {0x1.fffffffffffffp+138, chars_format::scientific, 41, "6.96898287454081895801738740683994115866624e+41"},
    {0x1.fffffffffffffp+139, chars_format::scientific, 42, "1.393796574908163791603477481367988231733248e+42"},
    {0x1.fffffffffffffp+140, chars_format::scientific, 42, "2.787593149816327583206954962735976463466496e+42"},
    {0x1.fffffffffffffp+141, chars_format::scientific, 42, "5.575186299632655166413909925471952926932992e+42"},
    {0x1.fffffffffffffp+142, chars_format::scientific, 43, "1.1150372599265310332827819850943905853865984e+43"},
    {0x1.fffffffffffffp+143, chars_format::scientific, 43, "2.2300745198530620665655639701887811707731968e+43"},
    {0x1.fffffffffffffp+144, chars_format::scientific, 43, "4.4601490397061241331311279403775623415463936e+43"},
    {0x1.fffffffffffffp+145, chars_format::scientific, 43, "8.9202980794122482662622558807551246830927872e+43"},
    {0x1.fffffffffffffp+146, chars_format::scientific, 44, "1.78405961588244965325245117615102493661855744e+44"},
    {0x1.fffffffffffffp+147, chars_format::scientific, 44, "3.56811923176489930650490235230204987323711488e+44"},
    {0x1.fffffffffffffp+148, chars_format::scientific, 44, "7.13623846352979861300980470460409974647422976e+44"},
    {0x1.fffffffffffffp+149, chars_format::scientific, 45, "1.427247692705959722601960940920819949294845952e+45"},
    {0x1.fffffffffffffp+150, chars_format::scientific, 45, "2.854495385411919445203921881841639898589691904e+45"},
    {0x1.fffffffffffffp+151, chars_format::scientific, 45, "5.708990770823838890407843763683279797179383808e+45"},
    {0x1.fffffffffffffp+152, chars_format::scientific, 46, "1.1417981541647677780815687527366559594358767616e+46"},
    {0x1.fffffffffffffp+153, chars_format::scientific, 46, "2.2835963083295355561631375054733119188717535232e+46"},
    {0x1.fffffffffffffp+154, chars_format::scientific, 46, "4.5671926166590711123262750109466238377435070464e+46"},
    {0x1.fffffffffffffp+155, chars_format::scientific, 46, "9.1343852333181422246525500218932476754870140928e+46"},
    {0x1.fffffffffffffp+156, chars_format::scientific, 47, "1.82687704666362844493051000437864953509740281856e+47"},
    {0x1.fffffffffffffp+157, chars_format::scientific, 47, "3.65375409332725688986102000875729907019480563712e+47"},
    {0x1.fffffffffffffp+158, chars_format::scientific, 47, "7.30750818665451377972204001751459814038961127424e+47"},
    {0x1.fffffffffffffp+159, chars_format::scientific, 48, "1.461501637330902755944408003502919628077922254848e+48"},
    {0x1.fffffffffffffp+160, chars_format::scientific, 48, "2.923003274661805511888816007005839256155844509696e+48"},
    {0x1.fffffffffffffp+161, chars_format::scientific, 48, "5.846006549323611023777632014011678512311689019392e+48"},
    {0x1.fffffffffffffp+162, chars_format::scientific, 49, "1.1692013098647222047555264028023357024623378038784e+49"},
    {0x1.fffffffffffffp+163, chars_format::scientific, 49, "2.3384026197294444095110528056046714049246756077568e+49"},
    {0x1.fffffffffffffp+164, chars_format::scientific, 49, "4.6768052394588888190221056112093428098493512155136e+49"},
    {0x1.fffffffffffffp+165, chars_format::scientific, 49, "9.3536104789177776380442112224186856196987024310272e+49"},
    {0x1.fffffffffffffp+166, chars_format::scientific, 50, "1.87072209578355552760884224448373712393974048620544e+50"},
    {0x1.fffffffffffffp+167, chars_format::scientific, 50, "3.74144419156711105521768448896747424787948097241088e+50"},
    {0x1.fffffffffffffp+168, chars_format::scientific, 50, "7.48288838313422211043536897793494849575896194482176e+50"},
    {0x1.fffffffffffffp+169, chars_format::scientific, 51, "1.496577676626844422087073795586989699151792388964352e+51"},
    {0x1.fffffffffffffp+170, chars_format::scientific, 51, "2.993155353253688844174147591173979398303584777928704e+51"},
    {0x1.fffffffffffffp+171, chars_format::scientific, 51, "5.986310706507377688348295182347958796607169555857408e+51"},
    {0x1.fffffffffffffp+172, chars_format::scientific, 52,
        "1.1972621413014755376696590364695917593214339111714816e+52"},
    {0x1.fffffffffffffp+173, chars_format::scientific, 52,
        "2.3945242826029510753393180729391835186428678223429632e+52"},
    {0x1.fffffffffffffp+174, chars_format::scientific, 52,
        "4.7890485652059021506786361458783670372857356446859264e+52"},
    {0x1.fffffffffffffp+175, chars_format::scientific, 52,
        "9.5780971304118043013572722917567340745714712893718528e+52"},
    {0x1.fffffffffffffp+176, chars_format::scientific, 53,
        "1.91561942608236086027145445835134681491429425787437056e+53"},
    {0x1.fffffffffffffp+177, chars_format::scientific, 53,
        "3.83123885216472172054290891670269362982858851574874112e+53"},
    {0x1.fffffffffffffp+178, chars_format::scientific, 53,
        "7.66247770432944344108581783340538725965717703149748224e+53"},
    {0x1.fffffffffffffp+179, chars_format::scientific, 54,
        "1.532495540865888688217163566681077451931435406299496448e+54"},
    {0x1.fffffffffffffp+180, chars_format::scientific, 54,
        "3.064991081731777376434327133362154903862870812598992896e+54"},
    {0x1.fffffffffffffp+181, chars_format::scientific, 54,
        "6.129982163463554752868654266724309807725741625197985792e+54"},
    {0x1.fffffffffffffp+182, chars_format::scientific, 55,
        "1.2259964326927109505737308533448619615451483250395971584e+55"},
    {0x1.fffffffffffffp+183, chars_format::scientific, 55,
        "2.4519928653854219011474617066897239230902966500791943168e+55"},
    {0x1.fffffffffffffp+184, chars_format::scientific, 55,
        "4.9039857307708438022949234133794478461805933001583886336e+55"},
    {0x1.fffffffffffffp+185, chars_format::scientific, 55,
        "9.8079714615416876045898468267588956923611866003167772672e+55"},
    {0x1.fffffffffffffp+186, chars_format::scientific, 56,
        "1.96159429230833752091796936535177913847223732006335545344e+56"},
    {0x1.fffffffffffffp+187, chars_format::scientific, 56,
        "3.92318858461667504183593873070355827694447464012671090688e+56"},
    {0x1.fffffffffffffp+188, chars_format::scientific, 56,
        "7.84637716923335008367187746140711655388894928025342181376e+56"},
    {0x1.fffffffffffffp+189, chars_format::scientific, 57,
        "1.569275433846670016734375492281423310777789856050684362752e+57"},
    {0x1.fffffffffffffp+190, chars_format::scientific, 57,
        "3.138550867693340033468750984562846621555579712101368725504e+57"},
    {0x1.fffffffffffffp+191, chars_format::scientific, 57,
        "6.277101735386680066937501969125693243111159424202737451008e+57"},
    {0x1.fffffffffffffp+192, chars_format::scientific, 58,
        "1.2554203470773360133875003938251386486222318848405474902016e+58"},
    {0x1.fffffffffffffp+193, chars_format::scientific, 58,
        "2.5108406941546720267750007876502772972444637696810949804032e+58"},
    {0x1.fffffffffffffp+194, chars_format::scientific, 58,
        "5.0216813883093440535500015753005545944889275393621899608064e+58"},
    {0x1.fffffffffffffp+195, chars_format::scientific, 59,
        "1.00433627766186881071000031506011091889778550787243799216128e+59"},
    {0x1.fffffffffffffp+196, chars_format::scientific, 59,
        "2.00867255532373762142000063012022183779557101574487598432256e+59"},
    {0x1.fffffffffffffp+197, chars_format::scientific, 59,
        "4.01734511064747524284000126024044367559114203148975196864512e+59"},
    {0x1.fffffffffffffp+198, chars_format::scientific, 59,
        "8.03469022129495048568000252048088735118228406297950393729024e+59"},
    {0x1.fffffffffffffp+199, chars_format::scientific, 60,
        "1.606938044258990097136000504096177470236456812595900787458048e+60"},
    {0x1.fffffffffffffp+200, chars_format::scientific, 60,
        "3.213876088517980194272001008192354940472913625191801574916096e+60"},
    {0x1.fffffffffffffp+201, chars_format::scientific, 60,
        "6.427752177035960388544002016384709880945827250383603149832192e+60"},
    {0x1.fffffffffffffp+202, chars_format::scientific, 61,
        "1.2855504354071920777088004032769419761891654500767206299664384e+61"},
    {0x1.fffffffffffffp+203, chars_format::scientific, 61,
        "2.5711008708143841554176008065538839523783309001534412599328768e+61"},
    {0x1.fffffffffffffp+204, chars_format::scientific, 61,
        "5.1422017416287683108352016131077679047566618003068825198657536e+61"},
    {0x1.fffffffffffffp+205, chars_format::scientific, 62,
        "1.02844034832575366216704032262155358095133236006137650397315072e+62"},
    {0x1.fffffffffffffp+206, chars_format::scientific, 62,
        "2.05688069665150732433408064524310716190266472012275300794630144e+62"},
    {0x1.fffffffffffffp+207, chars_format::scientific, 62,
        "4.11376139330301464866816129048621432380532944024550601589260288e+62"},
    {0x1.fffffffffffffp+208, chars_format::scientific, 62,
        "8.22752278660602929733632258097242864761065888049101203178520576e+62"},
    {0x1.fffffffffffffp+209, chars_format::scientific, 63,
        "1.645504557321205859467264516194485729522131776098202406357041152e+63"},
    {0x1.fffffffffffffp+210, chars_format::scientific, 63,
        "3.291009114642411718934529032388971459044263552196404812714082304e+63"},
    {0x1.fffffffffffffp+211, chars_format::scientific, 63,
        "6.582018229284823437869058064777942918088527104392809625428164608e+63"},
    {0x1.fffffffffffffp+212, chars_format::scientific, 64,
        "1.3164036458569646875738116129555885836177054208785619250856329216e+64"},
    {0x1.fffffffffffffp+213, chars_format::scientific, 64,
        "2.6328072917139293751476232259111771672354108417571238501712658432e+64"},
    {0x1.fffffffffffffp+214, chars_format::scientific, 64,
        "5.2656145834278587502952464518223543344708216835142477003425316864e+64"},
    {0x1.fffffffffffffp+215, chars_format::scientific, 65,
        "1.05312291668557175005904929036447086689416433670284954006850633728e+65"},
    {0x1.fffffffffffffp+216, chars_format::scientific, 65,
        "2.10624583337114350011809858072894173378832867340569908013701267456e+65"},
    {0x1.fffffffffffffp+217, chars_format::scientific, 65,
        "4.21249166674228700023619716145788346757665734681139816027402534912e+65"},
    {0x1.fffffffffffffp+218, chars_format::scientific, 65,
        "8.42498333348457400047239432291576693515331469362279632054805069824e+65"},
    {0x1.fffffffffffffp+219, chars_format::scientific, 66,
        "1.684996666696914800094478864583153387030662938724559264109610139648e+66"},
    {0x1.fffffffffffffp+220, chars_format::scientific, 66,
        "3.369993333393829600188957729166306774061325877449118528219220279296e+66"},
    {0x1.fffffffffffffp+221, chars_format::scientific, 66,
        "6.739986666787659200377915458332613548122651754898237056438440558592e+66"},
    {0x1.fffffffffffffp+222, chars_format::scientific, 67,
        "1.3479973333575318400755830916665227096245303509796474112876881117184e+67"},
    {0x1.fffffffffffffp+223, chars_format::scientific, 67,
        "2.6959946667150636801511661833330454192490607019592948225753762234368e+67"},
    {0x1.fffffffffffffp+224, chars_format::scientific, 67,
        "5.3919893334301273603023323666660908384981214039185896451507524468736e+67"},
    {0x1.fffffffffffffp+225, chars_format::scientific, 68,
        "1.07839786668602547206046647333321816769962428078371792903015048937472e+68"},
    {0x1.fffffffffffffp+226, chars_format::scientific, 68,
        "2.15679573337205094412093294666643633539924856156743585806030097874944e+68"},
    {0x1.fffffffffffffp+227, chars_format::scientific, 68,
        "4.31359146674410188824186589333287267079849712313487171612060195749888e+68"},
    {0x1.fffffffffffffp+228, chars_format::scientific, 68,
        "8.62718293348820377648373178666574534159699424626974343224120391499776e+68"},
    {0x1.fffffffffffffp+229, chars_format::scientific, 69,
        "1.725436586697640755296746357333149068319398849253948686448240782999552e+69"},
    {0x1.fffffffffffffp+230, chars_format::scientific, 69,
        "3.450873173395281510593492714666298136638797698507897372896481565999104e+69"},
    {0x1.fffffffffffffp+231, chars_format::scientific, 69,
        "6.901746346790563021186985429332596273277595397015794745792963131998208e+69"},
    {0x1.fffffffffffffp+232, chars_format::scientific, 70,
        "1.3803492693581126042373970858665192546555190794031589491585926263996416e+70"},
    {0x1.fffffffffffffp+233, chars_format::scientific, 70,
        "2.7606985387162252084747941717330385093110381588063178983171852527992832e+70"},
    {0x1.fffffffffffffp+234, chars_format::scientific, 70,
        "5.5213970774324504169495883434660770186220763176126357966343705055985664e+70"},
    {0x1.fffffffffffffp+235, chars_format::scientific, 71,
        "1.10427941548649008338991766869321540372441526352252715932687410111971328e+71"},
    {0x1.fffffffffffffp+236, chars_format::scientific, 71,
        "2.20855883097298016677983533738643080744883052704505431865374820223942656e+71"},
    {0x1.fffffffffffffp+237, chars_format::scientific, 71,
        "4.41711766194596033355967067477286161489766105409010863730749640447885312e+71"},
    {0x1.fffffffffffffp+238, chars_format::scientific, 71,
        "8.83423532389192066711934134954572322979532210818021727461499280895770624e+71"},
    {0x1.fffffffffffffp+239, chars_format::scientific, 72,
        "1.766847064778384133423868269909144645959064421636043454922998561791541248e+72"},
    {0x1.fffffffffffffp+240, chars_format::scientific, 72,
        "3.533694129556768266847736539818289291918128843272086909845997123583082496e+72"},
    {0x1.fffffffffffffp+241, chars_format::scientific, 72,
        "7.067388259113536533695473079636578583836257686544173819691994247166164992e+72"},
    {0x1.fffffffffffffp+242, chars_format::scientific, 73,
        "1.4134776518227073067390946159273157167672515373088347639383988494332329984e+73"},
    {0x1.fffffffffffffp+243, chars_format::scientific, 73,
        "2.8269553036454146134781892318546314335345030746176695278767976988664659968e+73"},
    {0x1.fffffffffffffp+244, chars_format::scientific, 73,
        "5.6539106072908292269563784637092628670690061492353390557535953977329319936e+73"},
    {0x1.fffffffffffffp+245, chars_format::scientific, 74,
        "1.13078212145816584539127569274185257341380122984706781115071907954658639872e+74"},
    {0x1.fffffffffffffp+246, chars_format::scientific, 74,
        "2.26156424291633169078255138548370514682760245969413562230143815909317279744e+74"},
    {0x1.fffffffffffffp+247, chars_format::scientific, 74,
        "4.52312848583266338156510277096741029365520491938827124460287631818634559488e+74"},
    {0x1.fffffffffffffp+248, chars_format::scientific, 74,
        "9.04625697166532676313020554193482058731040983877654248920575263637269118976e+74"},
    {0x1.fffffffffffffp+249, chars_format::scientific, 75,
        "1.809251394333065352626041108386964117462081967755308497841150527274538237952e+75"},
    {0x1.fffffffffffffp+250, chars_format::scientific, 75,
        "3.618502788666130705252082216773928234924163935510616995682301054549076475904e+75"},
    {0x1.fffffffffffffp+251, chars_format::scientific, 75,
        "7.237005577332261410504164433547856469848327871021233991364602109098152951808e+75"},
    {0x1.fffffffffffffp+252, chars_format::scientific, 76,
        "1.4474011154664522821008328867095712939696655742042467982729204218196305903616e+76"},
    {0x1.fffffffffffffp+253, chars_format::scientific, 76,
        "2.8948022309329045642016657734191425879393311484084935965458408436392611807232e+76"},
    {0x1.fffffffffffffp+254, chars_format::scientific, 76,
        "5.7896044618658091284033315468382851758786622968169871930916816872785223614464e+76"},
    {0x1.fffffffffffffp+255, chars_format::scientific, 77,
        "1.15792089237316182568066630936765703517573245936339743861833633745570447228928e+77"},
    {0x1.fffffffffffffp+256, chars_format::scientific, 77,
        "2.31584178474632365136133261873531407035146491872679487723667267491140894457856e+77"},
    {0x1.fffffffffffffp+257, chars_format::scientific, 77,
        "4.63168356949264730272266523747062814070292983745358975447334534982281788915712e+77"},
    {0x1.fffffffffffffp+258, chars_format::scientific, 77,
        "9.26336713898529460544533047494125628140585967490717950894669069964563577831424e+77"},
    {0x1.fffffffffffffp+259, chars_format::scientific, 78,
        "1.852673427797058921089066094988251256281171934981435901789338139929127155662848e+78"},
    {0x1.fffffffffffffp+260, chars_format::scientific, 78,
        "3.705346855594117842178132189976502512562343869962871803578676279858254311325696e+78"},
    {0x1.fffffffffffffp+261, chars_format::scientific, 78,
        "7.410693711188235684356264379953005025124687739925743607157352559716508622651392e+78"},
    {0x1.fffffffffffffp+262, chars_format::scientific, 79,
        "1.4821387422376471368712528759906010050249375479851487214314705119433017245302784e+79"},
    {0x1.fffffffffffffp+263, chars_format::scientific, 79,
        "2.9642774844752942737425057519812020100498750959702974428629410238866034490605568e+79"},
    {0x1.fffffffffffffp+264, chars_format::scientific, 79,
        "5.9285549689505885474850115039624040200997501919405948857258820477732068981211136e+79"},
    {0x1.fffffffffffffp+265, chars_format::scientific, 80,
        "1.18571099379011770949700230079248080401995003838811897714517640955464137962422272e+80"},
    {0x1.fffffffffffffp+266, chars_format::scientific, 80,
        "2.37142198758023541899400460158496160803990007677623795429035281910928275924844544e+80"},
    {0x1.fffffffffffffp+267, chars_format::scientific, 80,
        "4.74284397516047083798800920316992321607980015355247590858070563821856551849689088e+80"},
    {0x1.fffffffffffffp+268, chars_format::scientific, 80,
        "9.48568795032094167597601840633984643215960030710495181716141127643713103699378176e+80"},
    {0x1.fffffffffffffp+269, chars_format::scientific, 81,
        "1.897137590064188335195203681267969286431920061420990363432282255287426207398756352e+81"},
    {0x1.fffffffffffffp+270, chars_format::scientific, 81,
        "3.794275180128376670390407362535938572863840122841980726864564510574852414797512704e+81"},
    {0x1.fffffffffffffp+271, chars_format::scientific, 81,
        "7.588550360256753340780814725071877145727680245683961453729129021149704829595025408e+81"},
    {0x1.fffffffffffffp+272, chars_format::scientific, 82,
        "1.5177100720513506681561629450143754291455360491367922907458258042299409659190050816e+82"},
    {0x1.fffffffffffffp+273, chars_format::scientific, 82,
        "3.0354201441027013363123258900287508582910720982735845814916516084598819318380101632e+82"},
    {0x1.fffffffffffffp+274, chars_format::scientific, 82,
        "6.0708402882054026726246517800575017165821441965471691629833032169197638636760203264e+82"},
    {0x1.fffffffffffffp+275, chars_format::scientific, 83,
        "1.21416805764108053452493035601150034331642883930943383259666064338395277273520406528e+83"},
    {0x1.fffffffffffffp+276, chars_format::scientific, 83,
        "2.42833611528216106904986071202300068663285767861886766519332128676790554547040813056e+83"},
    {0x1.fffffffffffffp+277, chars_format::scientific, 83,
        "4.85667223056432213809972142404600137326571535723773533038664257353581109094081626112e+83"},
    {0x1.fffffffffffffp+278, chars_format::scientific, 83,
        "9.71334446112864427619944284809200274653143071447547066077328514707162218188163252224e+83"},
    {0x1.fffffffffffffp+279, chars_format::scientific, 84,
        "1.942668892225728855239888569618400549306286142895094132154657029414324436376326504448e+84"},
    {0x1.fffffffffffffp+280, chars_format::scientific, 84,
        "3.885337784451457710479777139236801098612572285790188264309314058828648872752653008896e+84"},
    {0x1.fffffffffffffp+281, chars_format::scientific, 84,
        "7.770675568902915420959554278473602197225144571580376528618628117657297745505306017792e+84"},
    {0x1.fffffffffffffp+282, chars_format::scientific, 85,
        "1.5541351137805830841919108556947204394450289143160753057237256235314595491010612035584e+85"},
    {0x1.fffffffffffffp+283, chars_format::scientific, 85,
        "3.1082702275611661683838217113894408788900578286321506114474512470629190982021224071168e+85"},
    {0x1.fffffffffffffp+284, chars_format::scientific, 85,
        "6.2165404551223323367676434227788817577801156572643012228949024941258381964042448142336e+85"},
    {0x1.fffffffffffffp+285, chars_format::scientific, 86,
        "1.24330809102446646735352868455577635155602313145286024457898049882516763928084896284672e+86"},
    {0x1.fffffffffffffp+286, chars_format::scientific, 86,
        "2.48661618204893293470705736911155270311204626290572048915796099765033527856169792569344e+86"},
    {0x1.fffffffffffffp+287, chars_format::scientific, 86,
        "4.97323236409786586941411473822310540622409252581144097831592199530067055712339585138688e+86"},
    {0x1.fffffffffffffp+288, chars_format::scientific, 86,
        "9.94646472819573173882822947644621081244818505162288195663184399060134111424679170277376e+86"},
    {0x1.fffffffffffffp+289, chars_format::scientific, 87,
        "1.989292945639146347765645895289242162489637010324576391326368798120268222849358340554752e+87"},
    {0x1.fffffffffffffp+290, chars_format::scientific, 87,
        "3.978585891278292695531291790578484324979274020649152782652737596240536445698716681109504e+87"},
    {0x1.fffffffffffffp+291, chars_format::scientific, 87,
        "7.957171782556585391062583581156968649958548041298305565305475192481072891397433362219008e+87"},
    {0x1.fffffffffffffp+292, chars_format::scientific, 88,
        "1.5914343565113170782125167162313937299917096082596611130610950384962145782794866724438016e+88"},
    {0x1.fffffffffffffp+293, chars_format::scientific, 88,
        "3.1828687130226341564250334324627874599834192165193222261221900769924291565589733448876032e+88"},
    {0x1.fffffffffffffp+294, chars_format::scientific, 88,
        "6.3657374260452683128500668649255749199668384330386444522443801539848583131179466897752064e+88"},
    {0x1.fffffffffffffp+295, chars_format::scientific, 89,
        "1.27314748520905366257001337298511498399336768660772889044887603079697166262358933795504128e+89"},
    {0x1.fffffffffffffp+296, chars_format::scientific, 89,
        "2.54629497041810732514002674597022996798673537321545778089775206159394332524717867591008256e+89"},
    {0x1.fffffffffffffp+297, chars_format::scientific, 89,
        "5.09258994083621465028005349194045993597347074643091556179550412318788665049435735182016512e+89"},
    {0x1.fffffffffffffp+298, chars_format::scientific, 90,
        "1.018517988167242930056010698388091987194694149286183112359100824637577330098871470364033024e+90"},
    {0x1.fffffffffffffp+299, chars_format::scientific, 90,
        "2.037035976334485860112021396776183974389388298572366224718201649275154660197742940728066048e+90"},
    {0x1.fffffffffffffp+300, chars_format::scientific, 90,
        "4.074071952668971720224042793552367948778776597144732449436403298550309320395485881456132096e+90"},
    {0x1.fffffffffffffp+301, chars_format::scientific, 90,
        "8.148143905337943440448085587104735897557553194289464898872806597100618640790971762912264192e+90"},
    {0x1.fffffffffffffp+302, chars_format::scientific, 91,
        "1.6296287810675886880896171174209471795115106388578929797745613194201237281581943525824528384e+91"},
    {0x1.fffffffffffffp+303, chars_format::scientific, 91,
        "3.2592575621351773761792342348418943590230212777157859595491226388402474563163887051649056768e+91"},
    {0x1.fffffffffffffp+304, chars_format::scientific, 91,
        "6.5185151242703547523584684696837887180460425554315719190982452776804949126327774103298113536e+91"},
    {0x1.fffffffffffffp+305, chars_format::scientific, 92,
        "1.30370302485407095047169369393675774360920851108631438381964905553609898252655548206596227072e+92"},
    {0x1.fffffffffffffp+306, chars_format::scientific, 92,
        "2.60740604970814190094338738787351548721841702217262876763929811107219796505311096413192454144e+92"},
    {0x1.fffffffffffffp+307, chars_format::scientific, 92,
        "5.21481209941628380188677477574703097443683404434525753527859622214439593010622192826384908288e+92"},
    {0x1.fffffffffffffp+308, chars_format::scientific, 93,
        "1.042962419883256760377354955149406194887366808869051507055719244428879186021244385652769816576e+93"},
    {0x1.fffffffffffffp+309, chars_format::scientific, 93,
        "2.085924839766513520754709910298812389774733617738103014111438488857758372042488771305539633152e+93"},
    {0x1.fffffffffffffp+310, chars_format::scientific, 93,
        "4.171849679533027041509419820597624779549467235476206028222876977715516744084977542611079266304e+93"},
    {0x1.fffffffffffffp+311, chars_format::scientific, 93,
        "8.343699359066054083018839641195249559098934470952412056445753955431033488169955085222158532608e+93"},
    {0x1.fffffffffffffp+312, chars_format::scientific, 94,
        "1.6687398718132108166037679282390499118197868941904824112891507910862066976339910170444317065216e+94"},
    {0x1.fffffffffffffp+313, chars_format::scientific, 94,
        "3.3374797436264216332075358564780998236395737883809648225783015821724133952679820340888634130432e+94"},
    {0x1.fffffffffffffp+314, chars_format::scientific, 94,
        "6.6749594872528432664150717129561996472791475767619296451566031643448267905359640681777268260864e+94"},
    {0x1.fffffffffffffp+315, chars_format::scientific, 95,
        "1.33499189745056865328301434259123992945582951535238592903132063286896535810719281363554536521728e+95"},
    {0x1.fffffffffffffp+316, chars_format::scientific, 95,
        "2.66998379490113730656602868518247985891165903070477185806264126573793071621438562727109073043456e+95"},
    {0x1.fffffffffffffp+317, chars_format::scientific, 95,
        "5.33996758980227461313205737036495971782331806140954371612528253147586143242877125454218146086912e+95"},
    {0x1.fffffffffffffp+318, chars_format::scientific, 96,
        "1.067993517960454922626411474072991943564663612281908743225056506295172286485754250908436292173824e+96"},
    {0x1.fffffffffffffp+319, chars_format::scientific, 96,
        "2.135987035920909845252822948145983887129327224563817486450113012590344572971508501816872584347648e+96"},
    {0x1.fffffffffffffp+320, chars_format::scientific, 96,
        "4.271974071841819690505645896291967774258654449127634972900226025180689145943017003633745168695296e+96"},
    {0x1.fffffffffffffp+321, chars_format::scientific, 96,
        "8.543948143683639381011291792583935548517308898255269945800452050361378291886034007267490337390592e+96"},
    {0x1.fffffffffffffp+322, chars_format::scientific, 97,
        "1.7087896287367278762022583585167871097034617796510539891600904100722756583772068014534980674781184e+97"},
    {0x1.fffffffffffffp+323, chars_format::scientific, 97,
        "3.4175792574734557524045167170335742194069235593021079783201808201445513167544136029069961349562368e+97"},
    {0x1.fffffffffffffp+324, chars_format::scientific, 97,
        "6.8351585149469115048090334340671484388138471186042159566403616402891026335088272058139922699124736e+97"},
    {0x1.fffffffffffffp+325, chars_format::scientific, 98,
        "1.36703170298938230096180668681342968776276942372084319132807232805782052670176544116279845398249472e+98"},
    {0x1.fffffffffffffp+326, chars_format::scientific, 98,
        "2.73406340597876460192361337362685937552553884744168638265614465611564105340353088232559690796498944e+98"},
    {0x1.fffffffffffffp+327, chars_format::scientific, 98,
        "5.46812681195752920384722674725371875105107769488337276531228931223128210680706176465119381592997888e+98"},
    {0x1.fffffffffffffp+328, chars_format::scientific, 99,
        "1.093625362391505840769445349450743750210215538976674553062457862446256421361412352930238763185995776e+"
        "99"},
    {0x1.fffffffffffffp+329, chars_format::scientific, 99,
        "2.187250724783011681538890698901487500420431077953349106124915724892512842722824705860477526371991552e+"
        "99"},
    {0x1.fffffffffffffp+330, chars_format::scientific, 99,
        "4.374501449566023363077781397802975000840862155906698212249831449785025685445649411720955052743983104e+"
        "99"},
    {0x1.fffffffffffffp+331, chars_format::scientific, 99,
        "8.749002899132046726155562795605950001681724311813396424499662899570051370891298823441910105487966208e+"
        "99"},
    {0x1.fffffffffffffp+332, chars_format::scientific, 100,
        "1.7498005798264093452311125591211900003363448623626792848999325799140102741782597646883820210975932416e+"
        "100"},
    {0x1.fffffffffffffp+333, chars_format::scientific, 100,
        "3.4996011596528186904622251182423800006726897247253585697998651598280205483565195293767640421951864832e+"
        "100"},
    {0x1.fffffffffffffp+334, chars_format::scientific, 100,
        "6.9992023193056373809244502364847600013453794494507171395997303196560410967130390587535280843903729664e+"
        "100"},
    {0x1.fffffffffffffp+335, chars_format::scientific, 101,
        "1.39984046386112747618489004729695200026907588989014342791994606393120821934260781175070561687807459328e+"
        "101"},
    {0x1.fffffffffffffp+336, chars_format::scientific, 101,
        "2.79968092772225495236978009459390400053815177978028685583989212786241643868521562350141123375614918656e+"
        "101"},
    {0x1.fffffffffffffp+337, chars_format::scientific, 101,
        "5.59936185544450990473956018918780800107630355956057371167978425572483287737043124700282246751229837312e+"
        "101"},
    {0x1.fffffffffffffp+338, chars_format::scientific, 102,
        "1.119872371088901980947912037837561600215260711912114742335956851144966575474086249400564493502459674624e+"
        "102"},
    {0x1.fffffffffffffp+339, chars_format::scientific, 102,
        "2.239744742177803961895824075675123200430521423824229484671913702289933150948172498801128987004919349248e+"
        "102"},
    {0x1.fffffffffffffp+340, chars_format::scientific, 102,
        "4.479489484355607923791648151350246400861042847648458969343827404579866301896344997602257974009838698496e+"
        "102"},
    {0x1.fffffffffffffp+341, chars_format::scientific, 102,
        "8.958978968711215847583296302700492801722085695296917938687654809159732603792689995204515948019677396992e+"
        "102"},
    {0x1.fffffffffffffp+342, chars_format::scientific, 103,
        "1."
        "7917957937422431695166592605400985603444171390593835877375309618319465207585379990409031896039354793984e+"
        "103"},
    {0x1.fffffffffffffp+343, chars_format::scientific, 103,
        "3."
        "5835915874844863390333185210801971206888342781187671754750619236638930415170759980818063792078709587968e+"
        "103"},
    {0x1.fffffffffffffp+344, chars_format::scientific, 103,
        "7."
        "1671831749689726780666370421603942413776685562375343509501238473277860830341519961636127584157419175936e+"
        "103"},
    {0x1.fffffffffffffp+345, chars_format::scientific, 104,
        "1."
        "43343663499379453561332740843207884827553371124750687019002476946555721660683039923272255168314838351872e+"
        "104"},
    {0x1.fffffffffffffp+346, chars_format::scientific, 104,
        "2."
        "86687326998758907122665481686415769655106742249501374038004953893111443321366079846544510336629676703744e+"
        "104"},
    {0x1.fffffffffffffp+347, chars_format::scientific, 104,
        "5."
        "73374653997517814245330963372831539310213484499002748076009907786222886642732159693089020673259353407488e+"
        "104"},
    {0x1.fffffffffffffp+348, chars_format::scientific, 105,
        "1."
        "146749307995035628490661926745663078620426968998005496152019815572445773285464319386178041346518706814976e"
        "+105"},
    {0x1.fffffffffffffp+349, chars_format::scientific, 105,
        "2."
        "293498615990071256981323853491326157240853937996010992304039631144891546570928638772356082693037413629952e"
        "+105"},
    {0x1.fffffffffffffp+350, chars_format::scientific, 105,
        "4."
        "586997231980142513962647706982652314481707875992021984608079262289783093141857277544712165386074827259904e"
        "+105"},
    {0x1.fffffffffffffp+351, chars_format::scientific, 105,
        "9."
        "173994463960285027925295413965304628963415751984043969216158524579566186283714555089424330772149654519808e"
        "+105"},
    {0x1.fffffffffffffp+352, chars_format::scientific, 106,
        "1."
        "8347988927920570055850590827930609257926831503968087938432317049159132372567429110178848661544299309039616"
        "e+106"},
    {0x1.fffffffffffffp+353, chars_format::scientific, 106,
        "3."
        "6695977855841140111701181655861218515853663007936175876864634098318264745134858220357697323088598618079232"
        "e+106"},
    {0x1.fffffffffffffp+354, chars_format::scientific, 106,
        "7."
        "3391955711682280223402363311722437031707326015872351753729268196636529490269716440715394646177197236158464"
        "e+106"},
    {0x1.fffffffffffffp+355, chars_format::scientific, 107,
        "1."
        "4678391142336456044680472662344487406341465203174470350745853639327305898053943288143078929235439447231692"
        "8e+107"},
    {0x1.fffffffffffffp+356, chars_format::scientific, 107,
        "2."
        "9356782284672912089360945324688974812682930406348940701491707278654611796107886576286157858470878894463385"
        "6e+107"},
    {0x1.fffffffffffffp+357, chars_format::scientific, 107,
        "5."
        "8713564569345824178721890649377949625365860812697881402983414557309223592215773152572315716941757788926771"
        "2e+107"},
    {0x1.fffffffffffffp+358, chars_format::scientific, 108,
        "1."
        "1742712913869164835744378129875589925073172162539576280596682911461844718443154630514463143388351557785354"
        "24e+108"},
    {0x1.fffffffffffffp+359, chars_format::scientific, 108,
        "2."
        "3485425827738329671488756259751179850146344325079152561193365822923689436886309261028926286776703115570708"
        "48e+108"},
    {0x1.fffffffffffffp+360, chars_format::scientific, 108,
        "4."
        "6970851655476659342977512519502359700292688650158305122386731645847378873772618522057852573553406231141416"
        "96e+108"},
    {0x1.fffffffffffffp+361, chars_format::scientific, 108,
        "9."
        "3941703310953318685955025039004719400585377300316610244773463291694757747545237044115705147106812462282833"
        "92e+108"},
    {0x1.fffffffffffffp+362, chars_format::scientific, 109,
        "1."
        "8788340662190663737191005007800943880117075460063322048954692658338951549509047408823141029421362492456566"
        "784e+109"},
    {0x1.fffffffffffffp+363, chars_format::scientific, 109,
        "3."
        "7576681324381327474382010015601887760234150920126644097909385316677903099018094817646282058842724984913133"
        "568e+109"},
    {0x1.fffffffffffffp+364, chars_format::scientific, 109,
        "7."
        "5153362648762654948764020031203775520468301840253288195818770633355806198036189635292564117685449969826267"
        "136e+109"},
    {0x1.fffffffffffffp+365, chars_format::scientific, 110,
        "1."
        "5030672529752530989752804006240755104093660368050657639163754126671161239607237927058512823537089993965253"
        "4272e+110"},
    {0x1.fffffffffffffp+366, chars_format::scientific, 110,
        "3."
        "0061345059505061979505608012481510208187320736101315278327508253342322479214475854117025647074179987930506"
        "8544e+110"},
    {0x1.fffffffffffffp+367, chars_format::scientific, 110,
        "6."
        "0122690119010123959011216024963020416374641472202630556655016506684644958428951708234051294148359975861013"
        "7088e+110"},
    {0x1.fffffffffffffp+368, chars_format::scientific, 111,
        "1."
        "2024538023802024791802243204992604083274928294440526111331003301336928991685790341646810258829671995172202"
        "74176e+111"},
    {0x1.fffffffffffffp+369, chars_format::scientific, 111,
        "2."
        "4049076047604049583604486409985208166549856588881052222662006602673857983371580683293620517659343990344405"
        "48352e+111"},
    {0x1.fffffffffffffp+370, chars_format::scientific, 111,
        "4."
        "8098152095208099167208972819970416333099713177762104445324013205347715966743161366587241035318687980688810"
        "96704e+111"},
    {0x1.fffffffffffffp+371, chars_format::scientific, 111,
        "9."
        "6196304190416198334417945639940832666199426355524208890648026410695431933486322733174482070637375961377621"
        "93408e+111"},
    {0x1.fffffffffffffp+372, chars_format::scientific, 112,
        "1."
        "9239260838083239666883589127988166533239885271104841778129605282139086386697264546634896414127475192275524"
        "386816e+112"},
    {0x1.fffffffffffffp+373, chars_format::scientific, 112,
        "3."
        "8478521676166479333767178255976333066479770542209683556259210564278172773394529093269792828254950384551048"
        "773632e+112"},
    {0x1.fffffffffffffp+374, chars_format::scientific, 112,
        "7."
        "6957043352332958667534356511952666132959541084419367112518421128556345546789058186539585656509900769102097"
        "547264e+112"},
    {0x1.fffffffffffffp+375, chars_format::scientific, 113,
        "1."
        "5391408670466591733506871302390533226591908216883873422503684225711269109357811637307917131301980153820419"
        "5094528e+113"},
    {0x1.fffffffffffffp+376, chars_format::scientific, 113,
        "3."
        "0782817340933183467013742604781066453183816433767746845007368451422538218715623274615834262603960307640839"
        "0189056e+113"},
    {0x1.fffffffffffffp+377, chars_format::scientific, 113,
        "6."
        "1565634681866366934027485209562132906367632867535493690014736902845076437431246549231668525207920615281678"
        "0378112e+113"},
    {0x1.fffffffffffffp+378, chars_format::scientific, 114,
        "1."
        "2313126936373273386805497041912426581273526573507098738002947380569015287486249309846333705041584123056335"
        "60756224e+114"},
    {0x1.fffffffffffffp+379, chars_format::scientific, 114,
        "2."
        "4626253872746546773610994083824853162547053147014197476005894761138030574972498619692667410083168246112671"
        "21512448e+114"},
    {0x1.fffffffffffffp+380, chars_format::scientific, 114,
        "4."
        "9252507745493093547221988167649706325094106294028394952011789522276061149944997239385334820166336492225342"
        "43024896e+114"},
    {0x1.fffffffffffffp+381, chars_format::scientific, 114,
        "9."
        "8505015490986187094443976335299412650188212588056789904023579044552122299889994478770669640332672984450684"
        "86049792e+114"},
    {0x1.fffffffffffffp+382, chars_format::scientific, 115,
        "1."
        "9701003098197237418888795267059882530037642517611357980804715808910424459977998895754133928066534596890136"
        "972099584e+115"},
    {0x1.fffffffffffffp+383, chars_format::scientific, 115,
        "3."
        "9402006196394474837777590534119765060075285035222715961609431617820848919955997791508267856133069193780273"
        "944199168e+115"},
    {0x1.fffffffffffffp+384, chars_format::scientific, 115,
        "7."
        "8804012392788949675555181068239530120150570070445431923218863235641697839911995583016535712266138387560547"
        "888398336e+115"},
    {0x1.fffffffffffffp+385, chars_format::scientific, 116,
        "1."
        "5760802478557789935111036213647906024030114014089086384643772647128339567982399116603307142453227677512109"
        "5776796672e+116"},
    {0x1.fffffffffffffp+386, chars_format::scientific, 116,
        "3."
        "1521604957115579870222072427295812048060228028178172769287545294256679135964798233206614284906455355024219"
        "1553593344e+116"},
    {0x1.fffffffffffffp+387, chars_format::scientific, 116,
        "6."
        "3043209914231159740444144854591624096120456056356345538575090588513358271929596466413228569812910710048438"
        "3107186688e+116"},
    {0x1.fffffffffffffp+388, chars_format::scientific, 117,
        "1."
        "2608641982846231948088828970918324819224091211271269107715018117702671654385919293282645713962582142009687"
        "66214373376e+117"},
    {0x1.fffffffffffffp+389, chars_format::scientific, 117,
        "2."
        "5217283965692463896177657941836649638448182422542538215430036235405343308771838586565291427925164284019375"
        "32428746752e+117"},
    {0x1.fffffffffffffp+390, chars_format::scientific, 117,
        "5."
        "0434567931384927792355315883673299276896364845085076430860072470810686617543677173130582855850328568038750"
        "64857493504e+117"},
    {0x1.fffffffffffffp+391, chars_format::scientific, 118,
        "1."
        "0086913586276985558471063176734659855379272969017015286172014494162137323508735434626116571170065713607750"
        "129714987008e+118"},
    {0x1.fffffffffffffp+392, chars_format::scientific, 118,
        "2."
        "0173827172553971116942126353469319710758545938034030572344028988324274647017470869252233142340131427215500"
        "259429974016e+118"},
    {0x1.fffffffffffffp+393, chars_format::scientific, 118,
        "4."
        "0347654345107942233884252706938639421517091876068061144688057976648549294034941738504466284680262854431000"
        "518859948032e+118"},
    {0x1.fffffffffffffp+394, chars_format::scientific, 118,
        "8."
        "0695308690215884467768505413877278843034183752136122289376115953297098588069883477008932569360525708862001"
        "037719896064e+118"},
    {0x1.fffffffffffffp+395, chars_format::scientific, 119,
        "1."
        "6139061738043176893553701082775455768606836750427224457875223190659419717613976695401786513872105141772400"
        "2075439792128e+119"},
    {0x1.fffffffffffffp+396, chars_format::scientific, 119,
        "3."
        "2278123476086353787107402165550911537213673500854448915750446381318839435227953390803573027744210283544800"
        "4150879584256e+119"},
    {0x1.fffffffffffffp+397, chars_format::scientific, 119,
        "6."
        "4556246952172707574214804331101823074427347001708897831500892762637678870455906781607146055488420567089600"
        "8301759168512e+119"},
    {0x1.fffffffffffffp+398, chars_format::scientific, 120,
        "1."
        "2911249390434541514842960866220364614885469400341779566300178552527535774091181356321429211097684113417920"
        "16603518337024e+120"},
    {0x1.fffffffffffffp+399, chars_format::scientific, 120,
        "2."
        "5822498780869083029685921732440729229770938800683559132600357105055071548182362712642858422195368226835840"
        "33207036674048e+120"},
    {0x1.fffffffffffffp+400, chars_format::scientific, 120,
        "5."
        "1644997561738166059371843464881458459541877601367118265200714210110143096364725425285716844390736453671680"
        "66414073348096e+120"},
    {0x1.fffffffffffffp+401, chars_format::scientific, 121,
        "1."
        "0328999512347633211874368692976291691908375520273423653040142842022028619272945085057143368878147290734336"
        "132828146696192e+121"},
    {0x1.fffffffffffffp+402, chars_format::scientific, 121,
        "2."
        "0657999024695266423748737385952583383816751040546847306080285684044057238545890170114286737756294581468672"
        "265656293392384e+121"},
    {0x1.fffffffffffffp+403, chars_format::scientific, 121,
        "4."
        "1315998049390532847497474771905166767633502081093694612160571368088114477091780340228573475512589162937344"
        "531312586784768e+121"},
    {0x1.fffffffffffffp+404, chars_format::scientific, 121,
        "8."
        "2631996098781065694994949543810333535267004162187389224321142736176228954183560680457146951025178325874689"
        "062625173569536e+121"},
    {0x1.fffffffffffffp+405, chars_format::scientific, 122,
        "1."
        "6526399219756213138998989908762066707053400832437477844864228547235245790836712136091429390205035665174937"
        "8125250347139072e+122"},
    {0x1.fffffffffffffp+406, chars_format::scientific, 122,
        "3."
        "3052798439512426277997979817524133414106801664874955689728457094470491581673424272182858780410071330349875"
        "6250500694278144e+122"},
    {0x1.fffffffffffffp+407, chars_format::scientific, 122,
        "6."
        "6105596879024852555995959635048266828213603329749911379456914188940983163346848544365717560820142660699751"
        "2501001388556288e+122"},
    {0x1.fffffffffffffp+408, chars_format::scientific, 123,
        "1."
        "3221119375804970511199191927009653365642720665949982275891382837788196632669369708873143512164028532139950"
        "25002002777112576e+123"},
    {0x1.fffffffffffffp+409, chars_format::scientific, 123,
        "2."
        "6442238751609941022398383854019306731285441331899964551782765675576393265338739417746287024328057064279900"
        "50004005554225152e+123"},
    {0x1.fffffffffffffp+410, chars_format::scientific, 123,
        "5."
        "2884477503219882044796767708038613462570882663799929103565531351152786530677478835492574048656114128559801"
        "00008011108450304e+123"},
    {0x1.fffffffffffffp+411, chars_format::scientific, 124,
        "1."
        "0576895500643976408959353541607722692514176532759985820713106270230557306135495767098514809731222825711960"
        "200016022216900608e+124"},
    {0x1.fffffffffffffp+412, chars_format::scientific, 124,
        "2."
        "1153791001287952817918707083215445385028353065519971641426212540461114612270991534197029619462445651423920"
        "400032044433801216e+124"},
    {0x1.fffffffffffffp+413, chars_format::scientific, 124,
        "4."
        "2307582002575905635837414166430890770056706131039943282852425080922229224541983068394059238924891302847840"
        "800064088867602432e+124"},
    {0x1.fffffffffffffp+414, chars_format::scientific, 124,
        "8."
        "4615164005151811271674828332861781540113412262079886565704850161844458449083966136788118477849782605695681"
        "600128177735204864e+124"},
    {0x1.fffffffffffffp+415, chars_format::scientific, 125,
        "1."
        "6923032801030362254334965666572356308022682452415977313140970032368891689816793227357623695569956521139136"
        "3200256355470409728e+125"},
    {0x1.fffffffffffffp+416, chars_format::scientific, 125,
        "3."
        "3846065602060724508669931333144712616045364904831954626281940064737783379633586454715247391139913042278272"
        "6400512710940819456e+125"},
    {0x1.fffffffffffffp+417, chars_format::scientific, 125,
        "6."
        "7692131204121449017339862666289425232090729809663909252563880129475566759267172909430494782279826084556545"
        "2801025421881638912e+125"},
    {0x1.fffffffffffffp+418, chars_format::scientific, 126,
        "1."
        "3538426240824289803467972533257885046418145961932781850512776025895113351853434581886098956455965216911309"
        "05602050843763277824e+126"},
    {0x1.fffffffffffffp+419, chars_format::scientific, 126,
        "2."
        "7076852481648579606935945066515770092836291923865563701025552051790226703706869163772197912911930433822618"
        "11204101687526555648e+126"},
    {0x1.fffffffffffffp+420, chars_format::scientific, 126,
        "5."
        "4153704963297159213871890133031540185672583847731127402051104103580453407413738327544395825823860867645236"
        "22408203375053111296e+126"},
    {0x1.fffffffffffffp+421, chars_format::scientific, 127,
        "1."
        "0830740992659431842774378026606308037134516769546225480410220820716090681482747665508879165164772173529047"
        "244816406750106222592e+127"},
    {0x1.fffffffffffffp+422, chars_format::scientific, 127,
        "2."
        "1661481985318863685548756053212616074269033539092450960820441641432181362965495331017758330329544347058094"
        "489632813500212445184e+127"},
    {0x1.fffffffffffffp+423, chars_format::scientific, 127,
        "4."
        "3322963970637727371097512106425232148538067078184901921640883282864362725930990662035516660659088694116188"
        "979265627000424890368e+127"},
    {0x1.fffffffffffffp+424, chars_format::scientific, 127,
        "8."
        "6645927941275454742195024212850464297076134156369803843281766565728725451861981324071033321318177388232377"
        "958531254000849780736e+127"},
    {0x1.fffffffffffffp+425, chars_format::scientific, 128,
        "1."
        "7329185588255090948439004842570092859415226831273960768656353313145745090372396264814206664263635477646475"
        "5917062508001699561472e+128"},
    {0x1.fffffffffffffp+426, chars_format::scientific, 128,
        "3."
        "4658371176510181896878009685140185718830453662547921537312706626291490180744792529628413328527270955292951"
        "1834125016003399122944e+128"},
    {0x1.fffffffffffffp+427, chars_format::scientific, 128,
        "6."
        "9316742353020363793756019370280371437660907325095843074625413252582980361489585059256826657054541910585902"
        "3668250032006798245888e+128"},
    {0x1.fffffffffffffp+428, chars_format::scientific, 129,
        "1."
        "3863348470604072758751203874056074287532181465019168614925082650516596072297917011851365331410908382117180"
        "47336500064013596491776e+129"},
    {0x1.fffffffffffffp+429, chars_format::scientific, 129,
        "2."
        "7726696941208145517502407748112148575064362930038337229850165301033192144595834023702730662821816764234360"
        "94673000128027192983552e+129"},
    {0x1.fffffffffffffp+430, chars_format::scientific, 129,
        "5."
        "5453393882416291035004815496224297150128725860076674459700330602066384289191668047405461325643633528468721"
        "89346000256054385967104e+129"},
    {0x1.fffffffffffffp+431, chars_format::scientific, 130,
        "1."
        "1090678776483258207000963099244859430025745172015334891940066120413276857838333609481092265128726705693744"
        "378692000512108771934208e+130"},
    {0x1.fffffffffffffp+432, chars_format::scientific, 130,
        "2."
        "2181357552966516414001926198489718860051490344030669783880132240826553715676667218962184530257453411387488"
        "757384001024217543868416e+130"},
    {0x1.fffffffffffffp+433, chars_format::scientific, 130,
        "4."
        "4362715105933032828003852396979437720102980688061339567760264481653107431353334437924369060514906822774977"
        "514768002048435087736832e+130"},
    {0x1.fffffffffffffp+434, chars_format::scientific, 130,
        "8."
        "8725430211866065656007704793958875440205961376122679135520528963306214862706668875848738121029813645549955"
        "029536004096870175473664e+130"},
    {0x1.fffffffffffffp+435, chars_format::scientific, 131,
        "1."
        "7745086042373213131201540958791775088041192275224535827104105792661242972541333775169747624205962729109991"
        "0059072008193740350947328e+131"},
    {0x1.fffffffffffffp+436, chars_format::scientific, 131,
        "3."
        "5490172084746426262403081917583550176082384550449071654208211585322485945082667550339495248411925458219982"
        "0118144016387480701894656e+131"},
    {0x1.fffffffffffffp+437, chars_format::scientific, 131,
        "7."
        "0980344169492852524806163835167100352164769100898143308416423170644971890165335100678990496823850916439964"
        "0236288032774961403789312e+131"},
    {0x1.fffffffffffffp+438, chars_format::scientific, 132,
        "1."
        "4196068833898570504961232767033420070432953820179628661683284634128994378033067020135798099364770183287992"
        "80472576065549922807578624e+132"},
    {0x1.fffffffffffffp+439, chars_format::scientific, 132,
        "2."
        "8392137667797141009922465534066840140865907640359257323366569268257988756066134040271596198729540366575985"
        "60945152131099845615157248e+132"},
    {0x1.fffffffffffffp+440, chars_format::scientific, 132,
        "5."
        "6784275335594282019844931068133680281731815280718514646733138536515977512132268080543192397459080733151971"
        "21890304262199691230314496e+132"},
    {0x1.fffffffffffffp+441, chars_format::scientific, 133,
        "1."
        "1356855067118856403968986213626736056346363056143702929346627707303195502426453616108638479491816146630394"
        "243780608524399382460628992e+133"},
    {0x1.fffffffffffffp+442, chars_format::scientific, 133,
        "2."
        "2713710134237712807937972427253472112692726112287405858693255414606391004852907232217276958983632293260788"
        "487561217048798764921257984e+133"},
    {0x1.fffffffffffffp+443, chars_format::scientific, 133,
        "4."
        "5427420268475425615875944854506944225385452224574811717386510829212782009705814464434553917967264586521576"
        "975122434097597529842515968e+133"},
    {0x1.fffffffffffffp+444, chars_format::scientific, 133,
        "9."
        "0854840536950851231751889709013888450770904449149623434773021658425564019411628928869107835934529173043153"
        "950244868195195059685031936e+133"},
    {0x1.fffffffffffffp+445, chars_format::scientific, 134,
        "1."
        "8170968107390170246350377941802777690154180889829924686954604331685112803882325785773821567186905834608630"
        "7900489736390390119370063872e+134"},
    {0x1.fffffffffffffp+446, chars_format::scientific, 134,
        "3."
        "6341936214780340492700755883605555380308361779659849373909208663370225607764651571547643134373811669217261"
        "5800979472780780238740127744e+134"},
    {0x1.fffffffffffffp+447, chars_format::scientific, 134,
        "7."
        "2683872429560680985401511767211110760616723559319698747818417326740451215529303143095286268747623338434523"
        "1601958945561560477480255488e+134"},
    {0x1.fffffffffffffp+448, chars_format::scientific, 135,
        "1."
        "4536774485912136197080302353442222152123344711863939749563683465348090243105860628619057253749524667686904"
        "63203917891123120954960510976e+135"},
    {0x1.fffffffffffffp+449, chars_format::scientific, 135,
        "2."
        "9073548971824272394160604706884444304246689423727879499127366930696180486211721257238114507499049335373809"
        "26407835782246241909921021952e+135"},
    {0x1.fffffffffffffp+450, chars_format::scientific, 135,
        "5."
        "8147097943648544788321209413768888608493378847455758998254733861392360972423442514476229014998098670747618"
        "52815671564492483819842043904e+135"},
    {0x1.fffffffffffffp+451, chars_format::scientific, 136,
        "1."
        "1629419588729708957664241882753777721698675769491151799650946772278472194484688502895245802999619734149523"
        "705631343128984967639684087808e+136"},
    {0x1.fffffffffffffp+452, chars_format::scientific, 136,
        "2."
        "3258839177459417915328483765507555443397351538982303599301893544556944388969377005790491605999239468299047"
        "411262686257969935279368175616e+136"},
    {0x1.fffffffffffffp+453, chars_format::scientific, 136,
        "4."
        "6517678354918835830656967531015110886794703077964607198603787089113888777938754011580983211998478936598094"
        "822525372515939870558736351232e+136"},
    {0x1.fffffffffffffp+454, chars_format::scientific, 136,
        "9."
        "3035356709837671661313935062030221773589406155929214397207574178227777555877508023161966423996957873196189"
        "645050745031879741117472702464e+136"},
    {0x1.fffffffffffffp+455, chars_format::scientific, 137,
        "1."
        "8607071341967534332262787012406044354717881231185842879441514835645555511175501604632393284799391574639237"
        "9290101490063759482234945404928e+137"},
    {0x1.fffffffffffffp+456, chars_format::scientific, 137,
        "3."
        "7214142683935068664525574024812088709435762462371685758883029671291111022351003209264786569598783149278475"
        "8580202980127518964469890809856e+137"},
    {0x1.fffffffffffffp+457, chars_format::scientific, 137,
        "7."
        "4428285367870137329051148049624177418871524924743371517766059342582222044702006418529573139197566298556951"
        "7160405960255037928939781619712e+137"},
    {0x1.fffffffffffffp+458, chars_format::scientific, 138,
        "1."
        "4885657073574027465810229609924835483774304984948674303553211868516444408940401283705914627839513259711390"
        "34320811920510075857879563239424e+138"},
    {0x1.fffffffffffffp+459, chars_format::scientific, 138,
        "2."
        "9771314147148054931620459219849670967548609969897348607106423737032888817880802567411829255679026519422780"
        "68641623841020151715759126478848e+138"},
    {0x1.fffffffffffffp+460, chars_format::scientific, 138,
        "5."
        "9542628294296109863240918439699341935097219939794697214212847474065777635761605134823658511358053038845561"
        "37283247682040303431518252957696e+138"},
    {0x1.fffffffffffffp+461, chars_format::scientific, 139,
        "1."
        "1908525658859221972648183687939868387019443987958939442842569494813155527152321026964731702271610607769112"
        "274566495364080606863036505915392e+139"},
    {0x1.fffffffffffffp+462, chars_format::scientific, 139,
        "2."
        "3817051317718443945296367375879736774038887975917878885685138989626311054304642053929463404543221215538224"
        "549132990728161213726073011830784e+139"},
    {0x1.fffffffffffffp+463, chars_format::scientific, 139,
        "4."
        "7634102635436887890592734751759473548077775951835757771370277979252622108609284107858926809086442431076449"
        "098265981456322427452146023661568e+139"},
    {0x1.fffffffffffffp+464, chars_format::scientific, 139,
        "9."
        "5268205270873775781185469503518947096155551903671515542740555958505244217218568215717853618172884862152898"
        "196531962912644854904292047323136e+139"},
    {0x1.fffffffffffffp+465, chars_format::scientific, 140,
        "1."
        "9053641054174755156237093900703789419231110380734303108548111191701048843443713643143570723634576972430579"
        "6393063925825289709808584094646272e+140"},
    {0x1.fffffffffffffp+466, chars_format::scientific, 140,
        "3."
        "8107282108349510312474187801407578838462220761468606217096222383402097686887427286287141447269153944861159"
        "2786127851650579419617168189292544e+140"},
    {0x1.fffffffffffffp+467, chars_format::scientific, 140,
        "7."
        "6214564216699020624948375602815157676924441522937212434192444766804195373774854572574282894538307889722318"
        "5572255703301158839234336378585088e+140"},
    {0x1.fffffffffffffp+468, chars_format::scientific, 141,
        "1."
        "5242912843339804124989675120563031535384888304587442486838488953360839074754970914514856578907661577944463"
        "71144511406602317678468672757170176e+141"},
    {0x1.fffffffffffffp+469, chars_format::scientific, 141,
        "3."
        "0485825686679608249979350241126063070769776609174884973676977906721678149509941829029713157815323155888927"
        "42289022813204635356937345514340352e+141"},
    {0x1.fffffffffffffp+470, chars_format::scientific, 141,
        "6."
        "0971651373359216499958700482252126141539553218349769947353955813443356299019883658059426315630646311777854"
        "84578045626409270713874691028680704e+141"},
    {0x1.fffffffffffffp+471, chars_format::scientific, 142,
        "1."
        "2194330274671843299991740096450425228307910643669953989470791162688671259803976731611885263126129262355570"
        "969156091252818541427749382057361408e+142"},
    {0x1.fffffffffffffp+472, chars_format::scientific, 142,
        "2."
        "4388660549343686599983480192900850456615821287339907978941582325377342519607953463223770526252258524711141"
        "938312182505637082855498764114722816e+142"},
    {0x1.fffffffffffffp+473, chars_format::scientific, 142,
        "4."
        "8777321098687373199966960385801700913231642574679815957883164650754685039215906926447541052504517049422283"
        "876624365011274165710997528229445632e+142"},
    {0x1.fffffffffffffp+474, chars_format::scientific, 142,
        "9."
        "7554642197374746399933920771603401826463285149359631915766329301509370078431813852895082105009034098844567"
        "753248730022548331421995056458891264e+142"},
    {0x1.fffffffffffffp+475, chars_format::scientific, 143,
        "1."
        "9510928439474949279986784154320680365292657029871926383153265860301874015686362770579016421001806819768913"
        "5506497460045096662843990112917782528e+143"},
    {0x1.fffffffffffffp+476, chars_format::scientific, 143,
        "3."
        "9021856878949898559973568308641360730585314059743852766306531720603748031372725541158032842003613639537827"
        "1012994920090193325687980225835565056e+143"},
    {0x1.fffffffffffffp+477, chars_format::scientific, 143,
        "7."
        "8043713757899797119947136617282721461170628119487705532613063441207496062745451082316065684007227279075654"
        "2025989840180386651375960451671130112e+143"},
    {0x1.fffffffffffffp+478, chars_format::scientific, 144,
        "1."
        "5608742751579959423989427323456544292234125623897541106522612688241499212549090216463213136801445455815130"
        "84051979680360773302751920903342260224e+144"},
    {0x1.fffffffffffffp+479, chars_format::scientific, 144,
        "3."
        "1217485503159918847978854646913088584468251247795082213045225376482998425098180432926426273602890911630261"
        "68103959360721546605503841806684520448e+144"},
    {0x1.fffffffffffffp+480, chars_format::scientific, 144,
        "6."
        "2434971006319837695957709293826177168936502495590164426090450752965996850196360865852852547205781823260523"
        "36207918721443093211007683613369040896e+144"},
    {0x1.fffffffffffffp+481, chars_format::scientific, 145,
        "1."
        "2486994201263967539191541858765235433787300499118032885218090150593199370039272173170570509441156364652104"
        "672415837442886186422015367226738081792e+145"},
    {0x1.fffffffffffffp+482, chars_format::scientific, 145,
        "2."
        "4973988402527935078383083717530470867574600998236065770436180301186398740078544346341141018882312729304209"
        "344831674885772372844030734453476163584e+145"},
    {0x1.fffffffffffffp+483, chars_format::scientific, 145,
        "4."
        "9947976805055870156766167435060941735149201996472131540872360602372797480157088692682282037764625458608418"
        "689663349771544745688061468906952327168e+145"},
    {0x1.fffffffffffffp+484, chars_format::scientific, 145,
        "9."
        "9895953610111740313532334870121883470298403992944263081744721204745594960314177385364564075529250917216837"
        "379326699543089491376122937813904654336e+145"},
    {0x1.fffffffffffffp+485, chars_format::scientific, 146,
        "1."
        "9979190722022348062706466974024376694059680798588852616348944240949118992062835477072912815105850183443367"
        "4758653399086178982752245875627809308672e+146"},
    {0x1.fffffffffffffp+486, chars_format::scientific, 146,
        "3."
        "9958381444044696125412933948048753388119361597177705232697888481898237984125670954145825630211700366886734"
        "9517306798172357965504491751255618617344e+146"},
    {0x1.fffffffffffffp+487, chars_format::scientific, 146,
        "7."
        "9916762888089392250825867896097506776238723194355410465395776963796475968251341908291651260423400733773469"
        "9034613596344715931008983502511237234688e+146"},
    {0x1.fffffffffffffp+488, chars_format::scientific, 147,
        "1."
        "5983352577617878450165173579219501355247744638871082093079155392759295193650268381658330252084680146754693"
        "98069227192689431862017967005022474469376e+147"},
    {0x1.fffffffffffffp+489, chars_format::scientific, 147,
        "3."
        "1966705155235756900330347158439002710495489277742164186158310785518590387300536763316660504169360293509387"
        "96138454385378863724035934010044948938752e+147"},
    {0x1.fffffffffffffp+490, chars_format::scientific, 147,
        "6."
        "3933410310471513800660694316878005420990978555484328372316621571037180774601073526633321008338720587018775"
        "92276908770757727448071868020089897877504e+147"},
    {0x1.fffffffffffffp+491, chars_format::scientific, 148,
        "1."
        "2786682062094302760132138863375601084198195711096865674463324314207436154920214705326664201667744117403755"
        "184553817541515454896143736040179795755008e+148"},
    {0x1.fffffffffffffp+492, chars_format::scientific, 148,
        "2."
        "5573364124188605520264277726751202168396391422193731348926648628414872309840429410653328403335488234807510"
        "369107635083030909792287472080359591510016e+148"},
    {0x1.fffffffffffffp+493, chars_format::scientific, 148,
        "5."
        "1146728248377211040528555453502404336792782844387462697853297256829744619680858821306656806670976469615020"
        "738215270166061819584574944160719183020032e+148"},
    {0x1.fffffffffffffp+494, chars_format::scientific, 149,
        "1."
        "0229345649675442208105711090700480867358556568877492539570659451365948923936171764261331361334195293923004"
        "1476430540332123639169149888321438366040064e+149"},
    {0x1.fffffffffffffp+495, chars_format::scientific, 149,
        "2."
        "0458691299350884416211422181400961734717113137754985079141318902731897847872343528522662722668390587846008"
        "2952861080664247278338299776642876732080128e+149"},
    {0x1.fffffffffffffp+496, chars_format::scientific, 149,
        "4."
        "0917382598701768832422844362801923469434226275509970158282637805463795695744687057045325445336781175692016"
        "5905722161328494556676599553285753464160256e+149"},
    {0x1.fffffffffffffp+497, chars_format::scientific, 149,
        "8."
        "1834765197403537664845688725603846938868452551019940316565275610927591391489374114090650890673562351384033"
        "1811444322656989113353199106571506928320512e+149"},
    {0x1.fffffffffffffp+498, chars_format::scientific, 150,
        "1."
        "6366953039480707532969137745120769387773690510203988063313055122185518278297874822818130178134712470276806"
        "63622888645313978226706398213143013856641024e+150"},
    {0x1.fffffffffffffp+499, chars_format::scientific, 150,
        "3."
        "2733906078961415065938275490241538775547381020407976126626110244371036556595749645636260356269424940553613"
        "27245777290627956453412796426286027713282048e+150"},
    {0x1.fffffffffffffp+500, chars_format::scientific, 150,
        "6."
        "5467812157922830131876550980483077551094762040815952253252220488742073113191499291272520712538849881107226"
        "54491554581255912906825592852572055426564096e+150"},
    {0x1.fffffffffffffp+501, chars_format::scientific, 151,
        "1."
        "3093562431584566026375310196096615510218952408163190450650444097748414622638299858254504142507769976221445"
        "308983109162511825813651185705144110853128192e+151"},
    {0x1.fffffffffffffp+502, chars_format::scientific, 151,
        "2."
        "6187124863169132052750620392193231020437904816326380901300888195496829245276599716509008285015539952442890"
        "617966218325023651627302371410288221706256384e+151"},
    {0x1.fffffffffffffp+503, chars_format::scientific, 151,
        "5."
        "2374249726338264105501240784386462040875809632652761802601776390993658490553199433018016570031079904885781"
        "235932436650047303254604742820576443412512768e+151"},
    {0x1.fffffffffffffp+504, chars_format::scientific, 152,
        "1."
        "0474849945267652821100248156877292408175161926530552360520355278198731698110639886603603314006215980977156"
        "2471864873300094606509209485641152886825025536e+152"},
    {0x1.fffffffffffffp+505, chars_format::scientific, 152,
        "2."
        "0949699890535305642200496313754584816350323853061104721040710556397463396221279773207206628012431961954312"
        "4943729746600189213018418971282305773650051072e+152"},
    {0x1.fffffffffffffp+506, chars_format::scientific, 152,
        "4."
        "1899399781070611284400992627509169632700647706122209442081421112794926792442559546414413256024863923908624"
        "9887459493200378426036837942564611547300102144e+152"},
    {0x1.fffffffffffffp+507, chars_format::scientific, 152,
        "8."
        "3798799562141222568801985255018339265401295412244418884162842225589853584885119092828826512049727847817249"
        "9774918986400756852073675885129223094600204288e+152"},
    {0x1.fffffffffffffp+508, chars_format::scientific, 153,
        "1."
        "6759759912428244513760397051003667853080259082448883776832568445117970716977023818565765302409945569563449"
        "99549837972801513704147351770258446189200408576e+153"},
    {0x1.fffffffffffffp+509, chars_format::scientific, 153,
        "3."
        "3519519824856489027520794102007335706160518164897767553665136890235941433954047637131530604819891139126899"
        "99099675945603027408294703540516892378400817152e+153"},
    {0x1.fffffffffffffp+510, chars_format::scientific, 153,
        "6."
        "7039039649712978055041588204014671412321036329795535107330273780471882867908095274263061209639782278253799"
        "98199351891206054816589407081033784756801634304e+153"},
    {0x1.fffffffffffffp+511, chars_format::scientific, 154,
        "1."
        "3407807929942595611008317640802934282464207265959107021466054756094376573581619054852612241927956455650759"
        "996398703782412109633178814162067569513603268608e+154"},
    {0x1.fffffffffffffp+512, chars_format::scientific, 154,
        "2."
        "6815615859885191222016635281605868564928414531918214042932109512188753147163238109705224483855912911301519"
        "992797407564824219266357628324135139027206537216e+154"},
    {0x1.fffffffffffffp+513, chars_format::scientific, 154,
        "5."
        "3631231719770382444033270563211737129856829063836428085864219024377506294326476219410448967711825822603039"
        "985594815129648438532715256648270278054413074432e+154"},
    {0x1.fffffffffffffp+514, chars_format::scientific, 155,
        "1."
        "0726246343954076488806654112642347425971365812767285617172843804875501258865295243882089793542365164520607"
        "9971189630259296877065430513296540556108826148864e+155"},
    {0x1.fffffffffffffp+515, chars_format::scientific, 155,
        "2."
        "1452492687908152977613308225284694851942731625534571234345687609751002517730590487764179587084730329041215"
        "9942379260518593754130861026593081112217652297728e+155"},
    {0x1.fffffffffffffp+516, chars_format::scientific, 155,
        "4."
        "2904985375816305955226616450569389703885463251069142468691375219502005035461180975528359174169460658082431"
        "9884758521037187508261722053186162224435304595456e+155"},
    {0x1.fffffffffffffp+517, chars_format::scientific, 155,
        "8."
        "5809970751632611910453232901138779407770926502138284937382750439004010070922361951056718348338921316164863"
        "9769517042074375016523444106372324448870609190912e+155"},
    {0x1.fffffffffffffp+518, chars_format::scientific, 156,
        "1."
        "7161994150326522382090646580227755881554185300427656987476550087800802014184472390211343669667784263232972"
        "79539034084148750033046888212744648897741218381824e+156"},
    {0x1.fffffffffffffp+519, chars_format::scientific, 156,
        "3."
        "4323988300653044764181293160455511763108370600855313974953100175601604028368944780422687339335568526465945"
        "59078068168297500066093776425489297795482436763648e+156"},
    {0x1.fffffffffffffp+520, chars_format::scientific, 156,
        "6."
        "8647976601306089528362586320911023526216741201710627949906200351203208056737889560845374678671137052931891"
        "18156136336595000132187552850978595590964873527296e+156"},
    {0x1.fffffffffffffp+521, chars_format::scientific, 157,
        "1."
        "3729595320261217905672517264182204705243348240342125589981240070240641611347577912169074935734227410586378"
        "236312272673190000264375105701957191181929747054592e+157"},
    {0x1.fffffffffffffp+522, chars_format::scientific, 157,
        "2."
        "7459190640522435811345034528364409410486696480684251179962480140481283222695155824338149871468454821172756"
        "472624545346380000528750211403914382363859494109184e+157"},
    {0x1.fffffffffffffp+523, chars_format::scientific, 157,
        "5."
        "4918381281044871622690069056728818820973392961368502359924960280962566445390311648676299742936909642345512"
        "945249090692760001057500422807828764727718988218368e+157"},
    {0x1.fffffffffffffp+524, chars_format::scientific, 158,
        "1."
        "0983676256208974324538013811345763764194678592273700471984992056192513289078062329735259948587381928469102"
        "5890498181385520002115000845615657529455437976436736e+158"},
    {0x1.fffffffffffffp+525, chars_format::scientific, 158,
        "2."
        "1967352512417948649076027622691527528389357184547400943969984112385026578156124659470519897174763856938205"
        "1780996362771040004230001691231315058910875952873472e+158"},
    {0x1.fffffffffffffp+526, chars_format::scientific, 158,
        "4."
        "3934705024835897298152055245383055056778714369094801887939968224770053156312249318941039794349527713876410"
        "3561992725542080008460003382462630117821751905746944e+158"},
    {0x1.fffffffffffffp+527, chars_format::scientific, 158,
        "8."
        "7869410049671794596304110490766110113557428738189603775879936449540106312624498637882079588699055427752820"
        "7123985451084160016920006764925260235643503811493888e+158"},
    {0x1.fffffffffffffp+528, chars_format::scientific, 159,
        "1."
        "7573882009934358919260822098153222022711485747637920755175987289908021262524899727576415917739811085550564"
        "14247970902168320033840013529850520471287007622987776e+159"},
    {0x1.fffffffffffffp+529, chars_format::scientific, 159,
        "3."
        "5147764019868717838521644196306444045422971495275841510351974579816042525049799455152831835479622171101128"
        "28495941804336640067680027059701040942574015245975552e+159"},
    {0x1.fffffffffffffp+530, chars_format::scientific, 159,
        "7."
        "0295528039737435677043288392612888090845942990551683020703949159632085050099598910305663670959244342202256"
        "56991883608673280135360054119402081885148030491951104e+159"},
    {0x1.fffffffffffffp+531, chars_format::scientific, 160,
        "1."
        "4059105607947487135408657678522577618169188598110336604140789831926417010019919782061132734191848868440451"
        "313983767217346560270720108238804163770296060983902208e+160"},
    {0x1.fffffffffffffp+532, chars_format::scientific, 160,
        "2."
        "8118211215894974270817315357045155236338377196220673208281579663852834020039839564122265468383697736880902"
        "627967534434693120541440216477608327540592121967804416e+160"},
    {0x1.fffffffffffffp+533, chars_format::scientific, 160,
        "5."
        "6236422431789948541634630714090310472676754392441346416563159327705668040079679128244530936767395473761805"
        "255935068869386241082880432955216655081184243935608832e+160"},
    {0x1.fffffffffffffp+534, chars_format::scientific, 161,
        "1."
        "1247284486357989708326926142818062094535350878488269283312631865541133608015935825648906187353479094752361"
        "0511870137738772482165760865910433310162368487871217664e+161"},
    {0x1.fffffffffffffp+535, chars_format::scientific, 161,
        "2."
        "2494568972715979416653852285636124189070701756976538566625263731082267216031871651297812374706958189504722"
        "1023740275477544964331521731820866620324736975742435328e+161"},
    {0x1.fffffffffffffp+536, chars_format::scientific, 161,
        "4."
        "4989137945431958833307704571272248378141403513953077133250527462164534432063743302595624749413916379009444"
        "2047480550955089928663043463641733240649473951484870656e+161"},
    {0x1.fffffffffffffp+537, chars_format::scientific, 161,
        "8."
        "9978275890863917666615409142544496756282807027906154266501054924329068864127486605191249498827832758018888"
        "4094961101910179857326086927283466481298947902969741312e+161"},
    {0x1.fffffffffffffp+538, chars_format::scientific, 162,
        "1."
        "7995655178172783533323081828508899351256561405581230853300210984865813772825497321038249899765566551603777"
        "68189922203820359714652173854566932962597895805939482624e+162"},
    {0x1.fffffffffffffp+539, chars_format::scientific, 162,
        "3."
        "5991310356345567066646163657017798702513122811162461706600421969731627545650994642076499799531133103207555"
        "36379844407640719429304347709133865925195791611878965248e+162"},
    {0x1.fffffffffffffp+540, chars_format::scientific, 162,
        "7."
        "1982620712691134133292327314035597405026245622324923413200843939463255091301989284152999599062266206415110"
        "72759688815281438858608695418267731850391583223757930496e+162"},
    {0x1.fffffffffffffp+541, chars_format::scientific, 163,
        "1."
        "4396524142538226826658465462807119481005249124464984682640168787892651018260397856830599919812453241283022"
        "145519377630562877717217390836535463700783166447515860992e+163"},
    {0x1.fffffffffffffp+542, chars_format::scientific, 163,
        "2."
        "8793048285076453653316930925614238962010498248929969365280337575785302036520795713661199839624906482566044"
        "291038755261125755434434781673070927401566332895031721984e+163"},
    {0x1.fffffffffffffp+543, chars_format::scientific, 163,
        "5."
        "7586096570152907306633861851228477924020996497859938730560675151570604073041591427322399679249812965132088"
        "582077510522251510868869563346141854803132665790063443968e+163"},
    {0x1.fffffffffffffp+544, chars_format::scientific, 164,
        "1."
        "1517219314030581461326772370245695584804199299571987746112135030314120814608318285464479935849962593026417"
        "7164155021044503021737739126692283709606265331580126887936e+164"},
    {0x1.fffffffffffffp+545, chars_format::scientific, 164,
        "2."
        "3034438628061162922653544740491391169608398599143975492224270060628241629216636570928959871699925186052835"
        "4328310042089006043475478253384567419212530663160253775872e+164"},
    {0x1.fffffffffffffp+546, chars_format::scientific, 164,
        "4."
        "6068877256122325845307089480982782339216797198287950984448540121256483258433273141857919743399850372105670"
        "8656620084178012086950956506769134838425061326320507551744e+164"},
    {0x1.fffffffffffffp+547, chars_format::scientific, 164,
        "9."
        "2137754512244651690614178961965564678433594396575901968897080242512966516866546283715839486799700744211341"
        "7313240168356024173901913013538269676850122652641015103488e+164"},
    {0x1.fffffffffffffp+548, chars_format::scientific, 165,
        "1."
        "8427550902448930338122835792393112935686718879315180393779416048502593303373309256743167897359940148842268"
        "34626480336712048347803826027076539353700245305282030206976e+165"},
    {0x1.fffffffffffffp+549, chars_format::scientific, 165,
        "3."
        "6855101804897860676245671584786225871373437758630360787558832097005186606746618513486335794719880297684536"
        "69252960673424096695607652054153078707400490610564060413952e+165"},
    {0x1.fffffffffffffp+550, chars_format::scientific, 165,
        "7."
        "3710203609795721352491343169572451742746875517260721575117664194010373213493237026972671589439760595369073"
        "38505921346848193391215304108306157414800981221128120827904e+165"},
    {0x1.fffffffffffffp+551, chars_format::scientific, 166,
        "1."
        "4742040721959144270498268633914490348549375103452144315023532838802074642698647405394534317887952119073814"
        "677011842693696386782430608216612314829601962442256241655808e+166"},
    {0x1.fffffffffffffp+552, chars_format::scientific, 166,
        "2."
        "9484081443918288540996537267828980697098750206904288630047065677604149285397294810789068635775904238147629"
        "354023685387392773564861216433224629659203924884512483311616e+166"},
    {0x1.fffffffffffffp+553, chars_format::scientific, 166,
        "5."
        "8968162887836577081993074535657961394197500413808577260094131355208298570794589621578137271551808476295258"
        "708047370774785547129722432866449259318407849769024966623232e+166"},
    {0x1.fffffffffffffp+554, chars_format::scientific, 167,
        "1."
        "1793632577567315416398614907131592278839500082761715452018826271041659714158917924315627454310361695259051"
        "7416094741549571094259444865732898518636815699538049933246464e+167"},
    {0x1.fffffffffffffp+555, chars_format::scientific, 167,
        "2."
        "3587265155134630832797229814263184557679000165523430904037652542083319428317835848631254908620723390518103"
        "4832189483099142188518889731465797037273631399076099866492928e+167"},
    {0x1.fffffffffffffp+556, chars_format::scientific, 167,
        "4."
        "7174530310269261665594459628526369115358000331046861808075305084166638856635671697262509817241446781036206"
        "9664378966198284377037779462931594074547262798152199732985856e+167"},
    {0x1.fffffffffffffp+557, chars_format::scientific, 167,
        "9."
        "4349060620538523331188919257052738230716000662093723616150610168333277713271343394525019634482893562072413"
        "9328757932396568754075558925863188149094525596304399465971712e+167"},
    {0x1.fffffffffffffp+558, chars_format::scientific, 168,
        "1."
        "8869812124107704666237783851410547646143200132418744723230122033666655542654268678905003926896578712414482"
        "78657515864793137508151117851726376298189051192608798931943424e+168"},
    {0x1.fffffffffffffp+559, chars_format::scientific, 168,
        "3."
        "7739624248215409332475567702821095292286400264837489446460244067333311085308537357810007853793157424828965"
        "57315031729586275016302235703452752596378102385217597863886848e+168"},
    {0x1.fffffffffffffp+560, chars_format::scientific, 168,
        "7."
        "5479248496430818664951135405642190584572800529674978892920488134666622170617074715620015707586314849657931"
        "14630063459172550032604471406905505192756204770435195727773696e+168"},
    {0x1.fffffffffffffp+561, chars_format::scientific, 169,
        "1."
        "5095849699286163732990227081128438116914560105934995778584097626933324434123414943124003141517262969931586"
        "229260126918345100065208942813811010385512409540870391455547392e+169"},
    {0x1.fffffffffffffp+562, chars_format::scientific, 169,
        "3."
        "0191699398572327465980454162256876233829120211869991557168195253866648868246829886248006283034525939863172"
        "458520253836690200130417885627622020771024819081740782911094784e+169"},
    {0x1.fffffffffffffp+563, chars_format::scientific, 169,
        "6."
        "0383398797144654931960908324513752467658240423739983114336390507733297736493659772496012566069051879726344"
        "917040507673380400260835771255244041542049638163481565822189568e+169"},
    {0x1.fffffffffffffp+564, chars_format::scientific, 170,
        "1."
        "2076679759428930986392181664902750493531648084747996622867278101546659547298731954499202513213810375945268"
        "9834081015346760800521671542510488083084099276326963131644379136e+170"},
    {0x1.fffffffffffffp+565, chars_format::scientific, 170,
        "2."
        "4153359518857861972784363329805500987063296169495993245734556203093319094597463908998405026427620751890537"
        "9668162030693521601043343085020976166168198552653926263288758272e+170"},
    {0x1.fffffffffffffp+566, chars_format::scientific, 170,
        "4."
        "8306719037715723945568726659611001974126592338991986491469112406186638189194927817996810052855241503781075"
        "9336324061387043202086686170041952332336397105307852526577516544e+170"},
    {0x1.fffffffffffffp+567, chars_format::scientific, 170,
        "9."
        "6613438075431447891137453319222003948253184677983972982938224812373276378389855635993620105710483007562151"
        "8672648122774086404173372340083904664672794210615705053155033088e+170"},
    {0x1.fffffffffffffp+568, chars_format::scientific, 171,
        "1."
        "9322687615086289578227490663844400789650636935596794596587644962474655275677971127198724021142096601512430"
        "37345296245548172808346744680167809329345588421231410106310066176e+171"},
    {0x1.fffffffffffffp+569, chars_format::scientific, 171,
        "3."
        "8645375230172579156454981327688801579301273871193589193175289924949310551355942254397448042284193203024860"
        "74690592491096345616693489360335618658691176842462820212620132352e+171"},
    {0x1.fffffffffffffp+570, chars_format::scientific, 171,
        "7."
        "7290750460345158312909962655377603158602547742387178386350579849898621102711884508794896084568386406049721"
        "49381184982192691233386978720671237317382353684925640425240264704e+171"},
    {0x1.fffffffffffffp+571, chars_format::scientific, 172,
        "1."
        "5458150092069031662581992531075520631720509548477435677270115969979724220542376901758979216913677281209944"
        "298762369964385382466773957441342474634764707369851280850480529408e+172"},
    {0x1.fffffffffffffp+572, chars_format::scientific, 172,
        "3."
        "0916300184138063325163985062151041263441019096954871354540231939959448441084753803517958433827354562419888"
        "597524739928770764933547914882684949269529414739702561700961058816e+172"},
    {0x1.fffffffffffffp+573, chars_format::scientific, 172,
        "6."
        "1832600368276126650327970124302082526882038193909742709080463879918896882169507607035916867654709124839777"
        "195049479857541529867095829765369898539058829479405123401922117632e+172"},
    {0x1.fffffffffffffp+574, chars_format::scientific, 173,
        "1."
        "2366520073655225330065594024860416505376407638781948541816092775983779376433901521407183373530941824967955"
        "4390098959715083059734191659530739797078117658958810246803844235264e+173"},
    {0x1.fffffffffffffp+575, chars_format::scientific, 173,
        "2."
        "4733040147310450660131188049720833010752815277563897083632185551967558752867803042814366747061883649935910"
        "8780197919430166119468383319061479594156235317917620493607688470528e+173"},
    {0x1.fffffffffffffp+576, chars_format::scientific, 173,
        "4."
        "9466080294620901320262376099441666021505630555127794167264371103935117505735606085628733494123767299871821"
        "7560395838860332238936766638122959188312470635835240987215376941056e+173"},
    {0x1.fffffffffffffp+577, chars_format::scientific, 173,
        "9."
        "8932160589241802640524752198883332043011261110255588334528742207870235011471212171257466988247534599743643"
        "5120791677720664477873533276245918376624941271670481974430753882112e+173"},
    {0x1.fffffffffffffp+578, chars_format::scientific, 174,
        "1."
        "9786432117848360528104950439776666408602252222051117666905748441574047002294242434251493397649506919948728"
        "70241583355441328955747066552491836753249882543340963948861507764224e+174"},
    {0x1.fffffffffffffp+579, chars_format::scientific, 174,
        "3."
        "9572864235696721056209900879553332817204504444102235333811496883148094004588484868502986795299013839897457"
        "40483166710882657911494133104983673506499765086681927897723015528448e+174"},
    {0x1.fffffffffffffp+580, chars_format::scientific, 174,
        "7."
        "9145728471393442112419801759106665634409008888204470667622993766296188009176969737005973590598027679794914"
        "80966333421765315822988266209967347012999530173363855795446031056896e+174"},
    {0x1.fffffffffffffp+581, chars_format::scientific, 175,
        "1."
        "5829145694278688422483960351821333126881801777640894133524598753259237601835393947401194718119605535958982"
        "961932666843530631645976532419934694025999060346727711590892062113792e+175"},
    {0x1.fffffffffffffp+582, chars_format::scientific, 175,
        "3."
        "1658291388557376844967920703642666253763603555281788267049197506518475203670787894802389436239211071917965"
        "923865333687061263291953064839869388051998120693455423181784124227584e+175"},
    {0x1.fffffffffffffp+583, chars_format::scientific, 175,
        "6."
        "3316582777114753689935841407285332507527207110563576534098395013036950407341575789604778872478422143835931"
        "847730667374122526583906129679738776103996241386910846363568248455168e+175"},
    {0x1.fffffffffffffp+584, chars_format::scientific, 176,
        "1."
        "2663316555422950737987168281457066501505441422112715306819679002607390081468315157920955774495684428767186"
        "3695461334748245053167812259359477552207992482773821692727136496910336e+176"},
    {0x1.fffffffffffffp+585, chars_format::scientific, 176,
        "2."
        "5326633110845901475974336562914133003010882844225430613639358005214780162936630315841911548991368857534372"
        "7390922669496490106335624518718955104415984965547643385454272993820672e+176"},
    {0x1.fffffffffffffp+586, chars_format::scientific, 176,
        "5."
        "0653266221691802951948673125828266006021765688450861227278716010429560325873260631683823097982737715068745"
        "4781845338992980212671249037437910208831969931095286770908545987641344e+176"},
    {0x1.fffffffffffffp+587, chars_format::scientific, 177,
        "1."
        "0130653244338360590389734625165653201204353137690172245455743202085912065174652126336764619596547543013749"
        "09563690677985960425342498074875820417663939862190573541817091975282688e+177"},
    {0x1.fffffffffffffp+588, chars_format::scientific, 177,
        "2."
        "0261306488676721180779469250331306402408706275380344490911486404171824130349304252673529239193095086027498"
        "19127381355971920850684996149751640835327879724381147083634183950565376e+177"},
    {0x1.fffffffffffffp+589, chars_format::scientific, 177,
        "4."
        "0522612977353442361558938500662612804817412550760688981822972808343648260698608505347058478386190172054996"
        "38254762711943841701369992299503281670655759448762294167268367901130752e+177"},
    {0x1.fffffffffffffp+590, chars_format::scientific, 177,
        "8."
        "1045225954706884723117877001325225609634825101521377963645945616687296521397217010694116956772380344109992"
        "76509525423887683402739984599006563341311518897524588334536735802261504e+177"},
    {0x1.fffffffffffffp+591, chars_format::scientific, 178,
        "1."
        "6209045190941376944623575400265045121926965020304275592729189123337459304279443402138823391354476068821998"
        "553019050847775366805479969198013126682623037795049176669073471604523008e+178"},
    {0x1.fffffffffffffp+592, chars_format::scientific, 178,
        "3."
        "2418090381882753889247150800530090243853930040608551185458378246674918608558886804277646782708952137643997"
        "106038101695550733610959938396026253365246075590098353338146943209046016e+178"},
    {0x1.fffffffffffffp+593, chars_format::scientific, 178,
        "6."
        "4836180763765507778494301601060180487707860081217102370916756493349837217117773608555293565417904275287994"
        "212076203391101467221919876792052506730492151180196706676293886418092032e+178"},
    {0x1.fffffffffffffp+594, chars_format::scientific, 179,
        "1."
        "2967236152753101555698860320212036097541572016243420474183351298669967443423554721711058713083580855057598"
        "8424152406782202934443839753584105013460984302360393413352587772836184064e+179"},
    {0x1.fffffffffffffp+595, chars_format::scientific, 179,
        "2."
        "5934472305506203111397720640424072195083144032486840948366702597339934886847109443422117426167161710115197"
        "6848304813564405868887679507168210026921968604720786826705175545672368128e+179"},
    {0x1.fffffffffffffp+596, chars_format::scientific, 179,
        "5."
        "1868944611012406222795441280848144390166288064973681896733405194679869773694218886844234852334323420230395"
        "3696609627128811737775359014336420053843937209441573653410351091344736256e+179"},
    {0x1.fffffffffffffp+597, chars_format::scientific, 180,
        "1."
        "0373788922202481244559088256169628878033257612994736379346681038935973954738843777368846970466864684046079"
        "07393219254257623475550718028672840107687874418883147306820702182689472512e+180"},
    {0x1.fffffffffffffp+598, chars_format::scientific, 180,
        "2."
        "0747577844404962489118176512339257756066515225989472758693362077871947909477687554737693940933729368092158"
        "14786438508515246951101436057345680215375748837766294613641404365378945024e+180"},
    {0x1.fffffffffffffp+599, chars_format::scientific, 180,
        "4."
        "1495155688809924978236353024678515512133030451978945517386724155743895818955375109475387881867458736184316"
        "29572877017030493902202872114691360430751497675532589227282808730757890048e+180"},
    {0x1.fffffffffffffp+600, chars_format::scientific, 180,
        "8."
        "2990311377619849956472706049357031024266060903957891034773448311487791637910750218950775763734917472368632"
        "59145754034060987804405744229382720861502995351065178454565617461515780096e+180"},
    {0x1.fffffffffffffp+601, chars_format::scientific, 181,
        "1."
        "6598062275523969991294541209871406204853212180791578206954689662297558327582150043790155152746983494473726"
        "518291508068121975608811488458765441723005990702130356909131234923031560192e+181"},
    {0x1.fffffffffffffp+602, chars_format::scientific, 181,
        "3."
        "3196124551047939982589082419742812409706424361583156413909379324595116655164300087580310305493966988947453"
        "036583016136243951217622976917530883446011981404260713818262469846063120384e+181"},
    {0x1.fffffffffffffp+603, chars_format::scientific, 181,
        "6."
        "6392249102095879965178164839485624819412848723166312827818758649190233310328600175160620610987933977894906"
        "073166032272487902435245953835061766892023962808521427636524939692126240768e+181"},
    {0x1.fffffffffffffp+604, chars_format::scientific, 182,
        "1."
        "3278449820419175993035632967897124963882569744633262565563751729838046662065720035032124122197586795578981"
        "2146332064544975804870491907670123533784047925617042855273049879384252481536e+182"},
    {0x1.fffffffffffffp+605, chars_format::scientific, 182,
        "2."
        "6556899640838351986071265935794249927765139489266525131127503459676093324131440070064248244395173591157962"
        "4292664129089951609740983815340247067568095851234085710546099758768504963072e+182"},
    {0x1.fffffffffffffp+606, chars_format::scientific, 182,
        "5."
        "3113799281676703972142531871588499855530278978533050262255006919352186648262880140128496488790347182315924"
        "8585328258179903219481967630680494135136191702468171421092199517537009926144e+182"},
    {0x1.fffffffffffffp+607, chars_format::scientific, 183,
        "1."
        "0622759856335340794428506374317699971106055795706610052451001383870437329652576028025699297758069436463184"
        "97170656516359806438963935261360988270272383404936342842184399035074019852288e+183"},
    {0x1.fffffffffffffp+608, chars_format::scientific, 183,
        "2."
        "1245519712670681588857012748635399942212111591413220104902002767740874659305152056051398595516138872926369"
        "94341313032719612877927870522721976540544766809872685684368798070148039704576e+183"},
    {0x1.fffffffffffffp+609, chars_format::scientific, 183,
        "4."
        "2491039425341363177714025497270799884424223182826440209804005535481749318610304112102797191032277745852739"
        "88682626065439225755855741045443953081089533619745371368737596140296079409152e+183"},
    {0x1.fffffffffffffp+610, chars_format::scientific, 183,
        "8."
        "4982078850682726355428050994541599768848446365652880419608011070963498637220608224205594382064555491705479"
        "77365252130878451511711482090887906162179067239490742737475192280592158818304e+183"},
    {0x1.fffffffffffffp+611, chars_format::scientific, 184,
        "1."
        "6996415770136545271085610198908319953769689273130576083921602214192699727444121644841118876412911098341095"
        "954730504261756903023422964181775812324358134478981485474950384561184317636608e+184"},
    {0x1.fffffffffffffp+612, chars_format::scientific, 184,
        "3."
        "3992831540273090542171220397816639907539378546261152167843204428385399454888243289682237752825822196682191"
        "909461008523513806046845928363551624648716268957962970949900769122368635273216e+184"},
    {0x1.fffffffffffffp+613, chars_format::scientific, 184,
        "6."
        "7985663080546181084342440795633279815078757092522304335686408856770798909776486579364475505651644393364383"
        "818922017047027612093691856727103249297432537915925941899801538244737270546432e+184"},
    {0x1.fffffffffffffp+614, chars_format::scientific, 185,
        "1."
        "3597132616109236216868488159126655963015751418504460867137281771354159781955297315872895101130328878672876"
        "7637844034094055224187383713454206498594865075831851883799603076489474541092864e+185"},
    {0x1.fffffffffffffp+615, chars_format::scientific, 185,
        "2."
        "7194265232218472433736976318253311926031502837008921734274563542708319563910594631745790202260657757345753"
        "5275688068188110448374767426908412997189730151663703767599206152978949082185728e+185"},
    {0x1.fffffffffffffp+616, chars_format::scientific, 185,
        "5."
        "4388530464436944867473952636506623852063005674017843468549127085416639127821189263491580404521315514691507"
        "0551376136376220896749534853816825994379460303327407535198412305957898164371456e+185"},
    {0x1.fffffffffffffp+617, chars_format::scientific, 186,
        "1."
        "0877706092887388973494790527301324770412601134803568693709825417083327825564237852698316080904263102938301"
        "41102752272752441793499069707633651988758920606654815070396824611915796328742912e+186"},
    {0x1.fffffffffffffp+618, chars_format::scientific, 186,
        "2."
        "1755412185774777946989581054602649540825202269607137387419650834166655651128475705396632161808526205876602"
        "82205504545504883586998139415267303977517841213309630140793649223831592657485824e+186"},
    {0x1.fffffffffffffp+619, chars_format::scientific, 186,
        "4."
        "3510824371549555893979162109205299081650404539214274774839301668333311302256951410793264323617052411753205"
        "64411009091009767173996278830534607955035682426619260281587298447663185314971648e+186"},
    {0x1.fffffffffffffp+620, chars_format::scientific, 186,
        "8."
        "7021648743099111787958324218410598163300809078428549549678603336666622604513902821586528647234104823506411"
        "28822018182019534347992557661069215910071364853238520563174596895326370629943296e+186"},
    {0x1.fffffffffffffp+621, chars_format::scientific, 187,
        "1."
        "7404329748619822357591664843682119632660161815685709909935720667333324520902780564317305729446820964701282"
        "257644036364039068695985115322138431820142729706477041126349193790652741259886592e+187"},
    {0x1.fffffffffffffp+622, chars_format::scientific, 187,
        "3."
        "4808659497239644715183329687364239265320323631371419819871441334666649041805561128634611458893641929402564"
        "515288072728078137391970230644276863640285459412954082252698387581305482519773184e+187"},
    {0x1.fffffffffffffp+623, chars_format::scientific, 187,
        "6."
        "9617318994479289430366659374728478530640647262742839639742882669333298083611122257269222917787283858805129"
        "030576145456156274783940461288553727280570918825908164505396775162610965039546368e+187"},
    {0x1.fffffffffffffp+624, chars_format::scientific, 188,
        "1."
        "3923463798895857886073331874945695706128129452548567927948576533866659616722224451453844583557456771761025"
        "8061152290912312549567880922577107454561141837651816329010793550325221930079092736e+188"},
    {0x1.fffffffffffffp+625, chars_format::scientific, 188,
        "2."
        "7846927597791715772146663749891391412256258905097135855897153067733319233444448902907689167114913543522051"
        "6122304581824625099135761845154214909122283675303632658021587100650443860158185472e+188"},
    {0x1.fffffffffffffp+626, chars_format::scientific, 188,
        "5."
        "5693855195583431544293327499782782824512517810194271711794306135466638466888897805815378334229827087044103"
        "2244609163649250198271523690308429818244567350607265316043174201300887720316370944e+188"},
    {0x1.fffffffffffffp+627, chars_format::scientific, 189,
        "1."
        "1138771039116686308858665499956556564902503562038854342358861227093327693377779561163075666845965417408820"
        "64489218327298500396543047380616859636489134701214530632086348402601775440632741888e+189"},
    {0x1.fffffffffffffp+628, chars_format::scientific, 189,
        "2."
        "2277542078233372617717330999913113129805007124077708684717722454186655386755559122326151333691930834817641"
        "28978436654597000793086094761233719272978269402429061264172696805203550881265483776e+189"},
    {0x1.fffffffffffffp+629, chars_format::scientific, 189,
        "4."
        "4555084156466745235434661999826226259610014248155417369435444908373310773511118244652302667383861669635282"
        "57956873309194001586172189522467438545956538804858122528345393610407101762530967552e+189"},
    {0x1.fffffffffffffp+630, chars_format::scientific, 189,
        "8."
        "9110168312933490470869323999652452519220028496310834738870889816746621547022236489304605334767723339270565"
        "15913746618388003172344379044934877091913077609716245056690787220814203525061935104e+189"},
    {0x1.fffffffffffffp+631, chars_format::scientific, 190,
        "1."
        "7822033662586698094173864799930490503844005699262166947774177963349324309404447297860921066953544667854113"
        "031827493236776006344688758089869754183826155219432490113381574441628407050123870208e+190"},
    {0x1.fffffffffffffp+632, chars_format::scientific, 190,
        "3."
        "5644067325173396188347729599860981007688011398524333895548355926698648618808894595721842133907089335708226"
        "063654986473552012689377516179739508367652310438864980226763148883256814100247740416e+190"},
    {0x1.fffffffffffffp+633, chars_format::scientific, 190,
        "7."
        "1288134650346792376695459199721962015376022797048667791096711853397297237617789191443684267814178671416452"
        "127309972947104025378755032359479016735304620877729960453526297766513628200495480832e+190"},
    {0x1.fffffffffffffp+634, chars_format::scientific, 191,
        "1."
        "4257626930069358475339091839944392403075204559409733558219342370679459447523557838288736853562835734283290"
        "4254619945894208050757510064718958033470609241755459920907052595533027256400990961664e+191"},
    {0x1.fffffffffffffp+635, chars_format::scientific, 191,
        "2."
        "8515253860138716950678183679888784806150409118819467116438684741358918895047115676577473707125671468566580"
        "8509239891788416101515020129437916066941218483510919841814105191066054512801981923328e+191"},
    {0x1.fffffffffffffp+636, chars_format::scientific, 191,
        "5."
        "7030507720277433901356367359777569612300818237638934232877369482717837790094231353154947414251342937133161"
        "7018479783576832203030040258875832133882436967021839683628210382132109025603963846656e+191"},
    {0x1.fffffffffffffp+637, chars_format::scientific, 192,
        "1."
        "1406101544055486780271273471955513922460163647527786846575473896543567558018846270630989482850268587426632"
        "34036959567153664406060080517751664267764873934043679367256420764264218051207927693312e+192"},
    {0x1.fffffffffffffp+638, chars_format::scientific, 192,
        "2."
        "2812203088110973560542546943911027844920327295055573693150947793087135116037692541261978965700537174853264"
        "68073919134307328812120161035503328535529747868087358734512841528528436102415855386624e+192"},
    {0x1.fffffffffffffp+639, chars_format::scientific, 192,
        "4."
        "5624406176221947121085093887822055689840654590111147386301895586174270232075385082523957931401074349706529"
        "36147838268614657624240322071006657071059495736174717469025683057056872204831710773248e+192"},
    {0x1.fffffffffffffp+640, chars_format::scientific, 192,
        "9."
        "1248812352443894242170187775644111379681309180222294772603791172348540464150770165047915862802148699413058"
        "72295676537229315248480644142013314142118991472349434938051366114113744409663421546496e+192"},
    {0x1.fffffffffffffp+641, chars_format::scientific, 193,
        "1."
        "8249762470488778848434037555128822275936261836044458954520758234469708092830154033009583172560429739882611"
        "744591353074458630496961288284026628284237982944698869876102732228227488819326843092992e+193"},
    {0x1.fffffffffffffp+642, chars_format::scientific, 193,
        "3."
        "6499524940977557696868075110257644551872523672088917909041516468939416185660308066019166345120859479765223"
        "489182706148917260993922576568053256568475965889397739752205464456454977638653686185984e+193"},
    {0x1.fffffffffffffp+643, chars_format::scientific, 193,
        "7."
        "2999049881955115393736150220515289103745047344177835818083032937878832371320616132038332690241718959530446"
        "978365412297834521987845153136106513136951931778795479504410928912909955277307372371968e+193"},
    {0x1.fffffffffffffp+644, chars_format::scientific, 194,
        "1."
        "4599809976391023078747230044103057820749009468835567163616606587575766474264123226407666538048343791906089"
        "3956730824595669043975690306272213026273903863557590959008821857825819910554614744743936e+194"},
    {0x1.fffffffffffffp+645, chars_format::scientific, 194,
        "2."
        "9199619952782046157494460088206115641498018937671134327233213175151532948528246452815333076096687583812178"
        "7913461649191338087951380612544426052547807727115181918017643715651639821109229489487872e+194"},
    {0x1.fffffffffffffp+646, chars_format::scientific, 194,
        "5."
        "8399239905564092314988920176412231282996037875342268654466426350303065897056492905630666152193375167624357"
        "5826923298382676175902761225088852105095615454230363836035287431303279642218458978975744e+194"},
    {0x1.fffffffffffffp+647, chars_format::scientific, 195,
        "1."
        "1679847981112818462997784035282446256599207575068453730893285270060613179411298581126133230438675033524871"
        "51653846596765352351805522450177704210191230908460727672070574862606559284436917957951488e+195"},
    {0x1.fffffffffffffp+648, chars_format::scientific, 195,
        "2."
        "3359695962225636925995568070564892513198415150136907461786570540121226358822597162252266460877350067049743"
        "03307693193530704703611044900355408420382461816921455344141149725213118568873835915902976e+195"},
    {0x1.fffffffffffffp+649, chars_format::scientific, 195,
        "4."
        "6719391924451273851991136141129785026396830300273814923573141080242452717645194324504532921754700134099486"
        "06615386387061409407222089800710816840764923633842910688282299450426237137747671831805952e+195"},
    {0x1.fffffffffffffp+650, chars_format::scientific, 195,
        "9."
        "3438783848902547703982272282259570052793660600547629847146282160484905435290388649009065843509400268198972"
        "13230772774122818814444179601421633681529847267685821376564598900852474275495343663611904e+195"},
    {0x1.fffffffffffffp+651, chars_format::scientific, 196,
        "1."
        "8687756769780509540796454456451914010558732120109525969429256432096981087058077729801813168701880053639794"
        "426461545548245637628888359202843267363059694535371642753129197801704948550990687327223808e+196"},
    {0x1.fffffffffffffp+652, chars_format::scientific, 196,
        "3."
        "7375513539561019081592908912903828021117464240219051938858512864193962174116155459603626337403760107279588"
        "852923091096491275257776718405686534726119389070743285506258395603409897101981374654447616e+196"},
    {0x1.fffffffffffffp+653, chars_format::scientific, 196,
        "7."
        "4751027079122038163185817825807656042234928480438103877717025728387924348232310919207252674807520214559177"
        "705846182192982550515553436811373069452238778141486571012516791206819794203962749308895232e+196"},
    {0x1.fffffffffffffp+654, chars_format::scientific, 197,
        "1."
        "4950205415824407632637163565161531208446985696087620775543405145677584869646462183841450534961504042911835"
        "5411692364385965101031106873622746138904477556282973142025033582413639588407925498617790464e+197"},
    {0x1.fffffffffffffp+655, chars_format::scientific, 197,
        "2."
        "9900410831648815265274327130323062416893971392175241551086810291355169739292924367682901069923008085823671"
        "0823384728771930202062213747245492277808955112565946284050067164827279176815850997235580928e+197"},
    {0x1.fffffffffffffp+656, chars_format::scientific, 197,
        "5."
        "9800821663297630530548654260646124833787942784350483102173620582710339478585848735365802139846016171647342"
        "1646769457543860404124427494490984555617910225131892568100134329654558353631701994471161856e+197"},
    {0x1.fffffffffffffp+657, chars_format::scientific, 198,
        "1."
        "1960164332659526106109730852129224966757588556870096620434724116542067895717169747073160427969203234329468"
        "43293538915087720808248854988981969111235820450263785136200268659309116707263403988942323712e+198"},
    {0x1.fffffffffffffp+658, chars_format::scientific, 198,
        "2."
        "3920328665319052212219461704258449933515177113740193240869448233084135791434339494146320855938406468658936"
        "86587077830175441616497709977963938222471640900527570272400537318618233414526807977884647424e+198"},
    {0x1.fffffffffffffp+659, chars_format::scientific, 198,
        "4."
        "7840657330638104424438923408516899867030354227480386481738896466168271582868678988292641711876812937317873"
        "73174155660350883232995419955927876444943281801055140544801074637236466829053615955769294848e+198"},
    {0x1.fffffffffffffp+660, chars_format::scientific, 198,
        "9."
        "5681314661276208848877846817033799734060708454960772963477792932336543165737357976585283423753625874635747"
        "46348311320701766465990839911855752889886563602110281089602149274472933658107231911538589696e+198"},
    {0x1.fffffffffffffp+661, chars_format::scientific, 199,
        "1."
        "9136262932255241769775569363406759946812141690992154592695558586467308633147471595317056684750725174927149"
        "492696622641403532931981679823711505779773127204220562179204298548945867316214463823077179392e+199"},
    {0x1.fffffffffffffp+662, chars_format::scientific, 199,
        "3."
        "8272525864510483539551138726813519893624283381984309185391117172934617266294943190634113369501450349854298"
        "985393245282807065863963359647423011559546254408441124358408597097891734632428927646154358784e+199"},
    {0x1.fffffffffffffp+663, chars_format::scientific, 199,
        "7."
        "6545051729020967079102277453627039787248566763968618370782234345869234532589886381268226739002900699708597"
        "970786490565614131727926719294846023119092508816882248716817194195783469264857855292308717568e+199"},
    {0x1.fffffffffffffp+664, chars_format::scientific, 200,
        "1."
        "5309010345804193415820455490725407957449713352793723674156446869173846906517977276253645347800580139941719"
        "5941572981131228263455853438589692046238185017633764497433634388391566938529715710584617435136e+200"},
    {0x1.fffffffffffffp+665, chars_format::scientific, 200,
        "3."
        "0618020691608386831640910981450815914899426705587447348312893738347693813035954552507290695601160279883439"
        "1883145962262456526911706877179384092476370035267528994867268776783133877059431421169234870272e+200"},
    {0x1.fffffffffffffp+666, chars_format::scientific, 200,
        "6."
        "1236041383216773663281821962901631829798853411174894696625787476695387626071909105014581391202320559766878"
        "3766291924524913053823413754358768184952740070535057989734537553566267754118862842338469740544e+200"},
    {0x1.fffffffffffffp+667, chars_format::scientific, 201,
        "1."
        "2247208276643354732656364392580326365959770682234978939325157495339077525214381821002916278240464111953375"
        "67532583849049826107646827508717536369905480141070115979469075107132535508237725684676939481088e+201"},
    {0x1.fffffffffffffp+668, chars_format::scientific, 201,
        "2."
        "4494416553286709465312728785160652731919541364469957878650314990678155050428763642005832556480928223906751"
        "35065167698099652215293655017435072739810960282140231958938150214265071016475451369353878962176e+201"},
    {0x1.fffffffffffffp+669, chars_format::scientific, 201,
        "4."
        "8988833106573418930625457570321305463839082728939915757300629981356310100857527284011665112961856447813502"
        "70130335396199304430587310034870145479621920564280463917876300428530142032950902738707757924352e+201"},
    {0x1.fffffffffffffp+670, chars_format::scientific, 201,
        "9."
        "7977666213146837861250915140642610927678165457879831514601259962712620201715054568023330225923712895627005"
        "40260670792398608861174620069740290959243841128560927835752600857060284065901805477415515848704e+201"},
    {0x1.fffffffffffffp+671, chars_format::scientific, 202,
        "1."
        "9595533242629367572250183028128522185535633091575966302920251992542524040343010913604666045184742579125401"
        "080521341584797217722349240139480581918487682257121855671505201714120568131803610954831031697408e+202"},
    {0x1.fffffffffffffp+672, chars_format::scientific, 202,
        "3."
        "9191066485258735144500366056257044371071266183151932605840503985085048080686021827209332090369485158250802"
        "161042683169594435444698480278961163836975364514243711343010403428241136263607221909662063394816e+202"},
    {0x1.fffffffffffffp+673, chars_format::scientific, 202,
        "7."
        "8382132970517470289000732112514088742142532366303865211681007970170096161372043654418664180738970316501604"
        "322085366339188870889396960557922327673950729028487422686020806856482272527214443819324126789632e+202"},
    {0x1.fffffffffffffp+674, chars_format::scientific, 203,
        "1."
        "5676426594103494057800146422502817748428506473260773042336201594034019232274408730883732836147794063300320"
        "8644170732678377741778793921115844655347901458056974845372041613712964545054428887638648253579264e+203"},
    {0x1.fffffffffffffp+675, chars_format::scientific, 203,
        "3."
        "1352853188206988115600292845005635496857012946521546084672403188068038464548817461767465672295588126600641"
        "7288341465356755483557587842231689310695802916113949690744083227425929090108857775277296507158528e+203"},
    {0x1.fffffffffffffp+676, chars_format::scientific, 203,
        "6."
        "2705706376413976231200585690011270993714025893043092169344806376136076929097634923534931344591176253201283"
        "4576682930713510967115175684463378621391605832227899381488166454851858180217715550554593014317056e+203"},
    {0x1.fffffffffffffp+677, chars_format::scientific, 204,
        "1."
        "2541141275282795246240117138002254198742805178608618433868961275227215385819526984706986268918235250640256"
        "69153365861427021934230351368926757242783211664455798762976332909703716360435431101109186028634112e+204"},
    {0x1.fffffffffffffp+678, chars_format::scientific, 204,
        "2."
        "5082282550565590492480234276004508397485610357217236867737922550454430771639053969413972537836470501280513"
        "38306731722854043868460702737853514485566423328911597525952665819407432720870862202218372057268224e+204"},
    {0x1.fffffffffffffp+679, chars_format::scientific, 204,
        "5."
        "0164565101131180984960468552009016794971220714434473735475845100908861543278107938827945075672941002561026"
        "76613463445708087736921405475707028971132846657823195051905331638814865441741724404436744114536448e+204"},
    {0x1.fffffffffffffp+680, chars_format::scientific, 205,
        "1."
        "0032913020226236196992093710401803358994244142886894747095169020181772308655621587765589015134588200512205"
        "353226926891416175473842810951414057942265693315646390103810663277629730883483448808873488229072896e+205"},
    {0x1.fffffffffffffp+681, chars_format::scientific, 205,
        "2."
        "0065826040452472393984187420803606717988488285773789494190338040363544617311243175531178030269176401024410"
        "706453853782832350947685621902828115884531386631292780207621326555259461766966897617746976458145792e+205"},
    {0x1.fffffffffffffp+682, chars_format::scientific, 205,
        "4."
        "0131652080904944787968374841607213435976976571547578988380676080727089234622486351062356060538352802048821"
        "412907707565664701895371243805656231769062773262585560415242653110518923533933795235493952916291584e+205"},
    {0x1.fffffffffffffp+683, chars_format::scientific, 205,
        "8."
        "0263304161809889575936749683214426871953953143095157976761352161454178469244972702124712121076705604097642"
        "825815415131329403790742487611312463538125546525171120830485306221037847067867590470987905832583168e+205"},
    {0x1.fffffffffffffp+684, chars_format::scientific, 206,
        "1."
        "6052660832361977915187349936642885374390790628619031595352270432290835693848994540424942424215341120819528"
        "5651630830262658807581484975222624927076251093050342241660970612442075694135735180941975811665166336e+"
        "206"},
    {0x1.fffffffffffffp+685, chars_format::scientific, 206,
        "3."
        "2105321664723955830374699873285770748781581257238063190704540864581671387697989080849884848430682241639057"
        "1303261660525317615162969950445249854152502186100684483321941224884151388271470361883951623330332672e+"
        "206"},
    {0x1.fffffffffffffp+686, chars_format::scientific, 206,
        "6."
        "4210643329447911660749399746571541497563162514476126381409081729163342775395978161699769696861364483278114"
        "2606523321050635230325939900890499708305004372201368966643882449768302776542940723767903246660665344e+"
        "206"},
    {0x1.fffffffffffffp+687, chars_format::scientific, 207,
        "1."
        "2842128665889582332149879949314308299512632502895225276281816345832668555079195632339953939372272896655622"
        "85213046642101270460651879801780999416610008744402737933287764899536605553085881447535806493321330688e+"
        "207"},
    {0x1.fffffffffffffp+688, chars_format::scientific, 207,
        "2."
        "5684257331779164664299759898628616599025265005790450552563632691665337110158391264679907878744545793311245"
        "70426093284202540921303759603561998833220017488805475866575529799073211106171762895071612986642661376e+"
        "207"},
    {0x1.fffffffffffffp+689, chars_format::scientific, 207,
        "5."
        "1368514663558329328599519797257233198050530011580901105127265383330674220316782529359815757489091586622491"
        "40852186568405081842607519207123997666440034977610951733151059598146422212343525790143225973285322752e+"
        "207"},
    {0x1.fffffffffffffp+690, chars_format::scientific, 208,
        "1."
        "0273702932711665865719903959451446639610106002316180221025453076666134844063356505871963151497818317324498"
        "281704373136810163685215038414247995332880069955221903466302119196292844424687051580286451946570645504e+"
        "208"},
    {0x1.fffffffffffffp+691, chars_format::scientific, 208,
        "2."
        "0547405865423331731439807918902893279220212004632360442050906153332269688126713011743926302995636634648996"
        "563408746273620327370430076828495990665760139910443806932604238392585688849374103160572903893141291008e+"
        "208"},
    {0x1.fffffffffffffp+692, chars_format::scientific, 208,
        "4."
        "1094811730846663462879615837805786558440424009264720884101812306664539376253426023487852605991273269297993"
        "126817492547240654740860153656991981331520279820887613865208476785171377698748206321145807786282582016e+"
        "208"},
    {0x1.fffffffffffffp+693, chars_format::scientific, 208,
        "8."
        "2189623461693326925759231675611573116880848018529441768203624613329078752506852046975705211982546538595986"
        "253634985094481309481720307313983962663040559641775227730416953570342755397496412642291615572565164032e+"
        "208"},
    {0x1.fffffffffffffp+694, chars_format::scientific, 209,
        "1."
        "6437924692338665385151846335122314623376169603705888353640724922665815750501370409395141042396509307719197"
        "2507269970188962618963440614627967925326081119283550455460833907140685510794992825284583231145130328064e+"
        "209"},
    {0x1.fffffffffffffp+695, chars_format::scientific, 209,
        "3."
        "2875849384677330770303692670244629246752339207411776707281449845331631501002740818790282084793018615438394"
        "5014539940377925237926881229255935850652162238567100910921667814281371021589985650569166462290260656128e+"
        "209"},
    {0x1.fffffffffffffp+696, chars_format::scientific, 209,
        "6."
        "5751698769354661540607385340489258493504678414823553414562899690663263002005481637580564169586037230876789"
        "0029079880755850475853762458511871701304324477134201821843335628562742043179971301138332924580521312256e+"
        "209"},
    {0x1.fffffffffffffp+697, chars_format::scientific, 210,
        "1."
        "3150339753870932308121477068097851698700935682964710682912579938132652600401096327516112833917207446175357"
        "80058159761511700951707524917023743402608648954268403643686671257125484086359942602276665849161042624512e+"
        "210"},
    {0x1.fffffffffffffp+698, chars_format::scientific, 210,
        "2."
        "6300679507741864616242954136195703397401871365929421365825159876265305200802192655032225667834414892350715"
        "60116319523023401903415049834047486805217297908536807287373342514250968172719885204553331698322085249024e+"
        "210"},
    {0x1.fffffffffffffp+699, chars_format::scientific, 210,
        "5."
        "2601359015483729232485908272391406794803742731858842731650319752530610401604385310064451335668829784701431"
        "20232639046046803806830099668094973610434595817073614574746685028501936345439770409106663396644170498048e+"
        "210"},
    {0x1.fffffffffffffp+700, chars_format::scientific, 211,
        "1."
        "0520271803096745846497181654478281358960748546371768546330063950506122080320877062012890267133765956940286"
        "240465278092093607613660199336189947220869191634147229149493370057003872690879540818213326793288340996096e"
        "+211"},
    {0x1.fffffffffffffp+701, chars_format::scientific, 211,
        "2."
        "1040543606193491692994363308956562717921497092743537092660127901012244160641754124025780534267531913880572"
        "480930556184187215227320398672379894441738383268294458298986740114007745381759081636426653586576681992192e"
        "+211"},
    {0x1.fffffffffffffp+702, chars_format::scientific, 211,
        "4."
        "2081087212386983385988726617913125435842994185487074185320255802024488321283508248051561068535063827761144"
        "961861112368374430454640797344759788883476766536588916597973480228015490763518163272853307173153363984384e"
        "+211"},
    {0x1.fffffffffffffp+703, chars_format::scientific, 211,
        "8."
        "4162174424773966771977453235826250871685988370974148370640511604048976642567016496103122137070127655522289"
        "923722224736748860909281594689519577766953533073177833195946960456030981527036326545706614346306727968768e"
        "+211"},
    {0x1.fffffffffffffp+704, chars_format::scientific, 212,
        "1."
        "6832434884954793354395490647165250174337197674194829674128102320809795328513403299220624427414025531104457"
        "9847444449473497721818563189379039155533907066146355666391893920912061963054072653091413228692613455937536"
        "e+212"},
    {0x1.fffffffffffffp+705, chars_format::scientific, 212,
        "3."
        "3664869769909586708790981294330500348674395348389659348256204641619590657026806598441248854828051062208915"
        "9694888898946995443637126378758078311067814132292711332783787841824123926108145306182826457385226911875072"
        "e+212"},
    {0x1.fffffffffffffp+706, chars_format::scientific, 212,
        "6."
        "7329739539819173417581962588661000697348790696779318696512409283239181314053613196882497709656102124417831"
        "9389777797893990887274252757516156622135628264585422665567575683648247852216290612365652914770453823750144"
        "e+212"},
    {0x1.fffffffffffffp+707, chars_format::scientific, 213,
        "1."
        "3465947907963834683516392517732200139469758139355863739302481856647836262810722639376499541931220424883566"
        "3877955559578798177454850551503231324427125652917084533113515136729649570443258122473130582954090764750028"
        "8e+213"},
    {0x1.fffffffffffffp+708, chars_format::scientific, 213,
        "2."
        "6931895815927669367032785035464400278939516278711727478604963713295672525621445278752999083862440849767132"
        "7755911119157596354909701103006462648854251305834169066227030273459299140886516244946261165908181529500057"
        "6e+213"},
    {0x1.fffffffffffffp+709, chars_format::scientific, 213,
        "5."
        "3863791631855338734065570070928800557879032557423454957209927426591345051242890557505998167724881699534265"
        "5511822238315192709819402206012925297708502611668338132454060546918598281773032489892522331816363059000115"
        "2e+213"},
    {0x1.fffffffffffffp+710, chars_format::scientific, 214,
        "1."
        "0772758326371067746813114014185760111575806511484690991441985485318269010248578111501199633544976339906853"
        "1102364447663038541963880441202585059541700522333667626490812109383719656354606497978504466363272611800023"
        "04e+214"},
    {0x1.fffffffffffffp+711, chars_format::scientific, 214,
        "2."
        "1545516652742135493626228028371520223151613022969381982883970970636538020497156223002399267089952679813706"
        "2204728895326077083927760882405170119083401044667335252981624218767439312709212995957008932726545223600046"
        "08e+214"},
    {0x1.fffffffffffffp+712, chars_format::scientific, 214,
        "4."
        "3091033305484270987252456056743040446303226045938763965767941941273076040994312446004798534179905359627412"
        "4409457790652154167855521764810340238166802089334670505963248437534878625418425991914017865453090447200092"
        "16e+214"},
    {0x1.fffffffffffffp+713, chars_format::scientific, 214,
        "8."
        "6182066610968541974504912113486080892606452091877527931535883882546152081988624892009597068359810719254824"
        "8818915581304308335711043529620680476333604178669341011926496875069757250836851983828035730906180894400184"
        "32e+214"},
    {0x1.fffffffffffffp+714, chars_format::scientific, 215,
        "1."
        "7236413322193708394900982422697216178521290418375505586307176776509230416397724978401919413671962143850964"
        "9763783116260861667142208705924136095266720835733868202385299375013951450167370396765607146181236178880036"
        "864e+215"},
    {0x1.fffffffffffffp+715, chars_format::scientific, 215,
        "3."
        "4472826644387416789801964845394432357042580836751011172614353553018460832795449956803838827343924287701929"
        "9527566232521723334284417411848272190533441671467736404770598750027902900334740793531214292362472357760073"
        "728e+215"},
    {0x1.fffffffffffffp+716, chars_format::scientific, 215,
        "6."
        "8945653288774833579603929690788864714085161673502022345228707106036921665590899913607677654687848575403859"
        "9055132465043446668568834823696544381066883342935472809541197500055805800669481587062428584724944715520147"
        "456e+215"},
    {0x1.fffffffffffffp+717, chars_format::scientific, 216,
        "1."
        "3789130657754966715920785938157772942817032334700404469045741421207384333118179982721535530937569715080771"
        "9811026493008689333713766964739308876213376668587094561908239500011161160133896317412485716944988943104029"
        "4912e+216"},
    {0x1.fffffffffffffp+718, chars_format::scientific, 216,
        "2."
        "7578261315509933431841571876315545885634064669400808938091482842414768666236359965443071061875139430161543"
        "9622052986017378667427533929478617752426753337174189123816479000022322320267792634824971433889977886208058"
        "9824e+216"},
    {0x1.fffffffffffffp+719, chars_format::scientific, 216,
        "5."
        "5156522631019866863683143752631091771268129338801617876182965684829537332472719930886142123750278860323087"
        "9244105972034757334855067858957235504853506674348378247632958000044644640535585269649942867779955772416117"
        "9648e+216"},
    {0x1.fffffffffffffp+720, chars_format::scientific, 217,
        "1."
        "1031304526203973372736628750526218354253625867760323575236593136965907466494543986177228424750055772064617"
        "5848821194406951466971013571791447100970701334869675649526591600008928928107117053929988573555991154483223"
        "59296e+217"},
    {0x1.fffffffffffffp+721, chars_format::scientific, 217,
        "2."
        "2062609052407946745473257501052436708507251735520647150473186273931814932989087972354456849500111544129235"
        "1697642388813902933942027143582894201941402669739351299053183200017857856214234107859977147111982308966447"
        "18592e+217"},
    {0x1.fffffffffffffp+722, chars_format::scientific, 217,
        "4."
        "4125218104815893490946515002104873417014503471041294300946372547863629865978175944708913699000223088258470"
        "3395284777627805867884054287165788403882805339478702598106366400035715712428468215719954294223964617932894"
        "37184e+217"},
    {0x1.fffffffffffffp+723, chars_format::scientific, 217,
        "8."
        "8250436209631786981893030004209746834029006942082588601892745095727259731956351889417827398000446176516940"
        "6790569555255611735768108574331576807765610678957405196212732800071431424856936431439908588447929235865788"
        "74368e+217"},
    {0x1.fffffffffffffp+724, chars_format::scientific, 218,
        "1."
        "7650087241926357396378606000841949366805801388416517720378549019145451946391270377883565479600089235303388"
        "1358113911051122347153621714866315361553122135791481039242546560014286284971387286287981717689585847173157"
        "748736e+218"},
    {0x1.fffffffffffffp+725, chars_format::scientific, 218,
        "3."
        "5300174483852714792757212001683898733611602776833035440757098038290903892782540755767130959200178470606776"
        "2716227822102244694307243429732630723106244271582962078485093120028572569942774572575963435379171694346315"
        "497472e+218"},
    {0x1.fffffffffffffp+726, chars_format::scientific, 218,
        "7."
        "0600348967705429585514424003367797467223205553666070881514196076581807785565081511534261918400356941213552"
        "5432455644204489388614486859465261446212488543165924156970186240057145139885549145151926870758343388692630"
        "994944e+218"},
    {0x1.fffffffffffffp+727, chars_format::scientific, 219,
        "1."
        "4120069793541085917102884800673559493444641110733214176302839215316361557113016302306852383680071388242710"
        "5086491128840897877722897371893052289242497708633184831394037248011429027977109829030385374151668677738526"
        "1989888e+219"},
    {0x1.fffffffffffffp+728, chars_format::scientific, 219,
        "2."
        "8240139587082171834205769601347118986889282221466428352605678430632723114226032604613704767360142776485421"
        "0172982257681795755445794743786104578484995417266369662788074496022858055954219658060770748303337355477052"
        "3979776e+219"},
    {0x1.fffffffffffffp+729, chars_format::scientific, 219,
        "5."
        "6480279174164343668411539202694237973778564442932856705211356861265446228452065209227409534720285552970842"
        "0345964515363591510891589487572209156969990834532739325576148992045716111908439316121541496606674710954104"
        "7959552e+219"},
    {0x1.fffffffffffffp+730, chars_format::scientific, 220,
        "1."
        "1296055834832868733682307840538847594755712888586571341042271372253089245690413041845481906944057110594168"
        "4069192903072718302178317897514441831393998166906547865115229798409143222381687863224308299321334942190820"
        "95919104e+220"},
    {0x1.fffffffffffffp+731, chars_format::scientific, 220,
        "2."
        "2592111669665737467364615681077695189511425777173142682084542744506178491380826083690963813888114221188336"
        "8138385806145436604356635795028883662787996333813095730230459596818286444763375726448616598642669884381641"
        "91838208e+220"},
    {0x1.fffffffffffffp+732, chars_format::scientific, 220,
        "4."
        "5184223339331474934729231362155390379022851554346285364169085489012356982761652167381927627776228442376673"
        "6276771612290873208713271590057767325575992667626191460460919193636572889526751452897233197285339768763283"
        "83676416e+220"},
    {0x1.fffffffffffffp+733, chars_format::scientific, 220,
        "9."
        "0368446678662949869458462724310780758045703108692570728338170978024713965523304334763855255552456884753347"
        "2553543224581746417426543180115534651151985335252382920921838387273145779053502905794466394570679537526567"
        "67352832e+220"},
    {0x1.fffffffffffffp+734, chars_format::scientific, 221,
        "1."
        "8073689335732589973891692544862156151609140621738514145667634195604942793104660866952771051110491376950669"
        "4510708644916349283485308636023106930230397067050476584184367677454629155810700581158893278914135907505313"
        "534705664e+221"},
    {0x1.fffffffffffffp+735, chars_format::scientific, 221,
        "3."
        "6147378671465179947783385089724312303218281243477028291335268391209885586209321733905542102220982753901338"
        "9021417289832698566970617272046213860460794134100953168368735354909258311621401162317786557828271815010627"
        "069411328e+221"},
    {0x1.fffffffffffffp+736, chars_format::scientific, 221,
        "7."
        "2294757342930359895566770179448624606436562486954056582670536782419771172418643467811084204441965507802677"
        "8042834579665397133941234544092427720921588268201906336737470709818516623242802324635573115656543630021254"
        "138822656e+221"},
    {0x1.fffffffffffffp+737, chars_format::scientific, 222,
        "1."
        "4458951468586071979113354035889724921287312497390811316534107356483954234483728693562216840888393101560535"
        "5608566915933079426788246908818485544184317653640381267347494141963703324648560464927114623131308726004250"
        "8277645312e+222"},
    {0x1.fffffffffffffp+738, chars_format::scientific, 222,
        "2."
        "8917902937172143958226708071779449842574624994781622633068214712967908468967457387124433681776786203121071"
        "1217133831866158853576493817636971088368635307280762534694988283927406649297120929854229246262617452008501"
        "6555290624e+222"},
    {0x1.fffffffffffffp+739, chars_format::scientific, 222,
        "5."
        "7835805874344287916453416143558899685149249989563245266136429425935816937934914774248867363553572406242142"
        "2434267663732317707152987635273942176737270614561525069389976567854813298594241859708458492525234904017003"
        "3110581248e+222"},
    {0x1.fffffffffffffp+740, chars_format::scientific, 223,
        "1."
        "1567161174868857583290683228711779937029849997912649053227285885187163387586982954849773472710714481248428"
        "4486853532746463541430597527054788435347454122912305013877995313570962659718848371941691698505046980803400"
        "66221162496e+223"},
    {0x1.fffffffffffffp+741, chars_format::scientific, 223,
        "2."
        "3134322349737715166581366457423559874059699995825298106454571770374326775173965909699546945421428962496856"
        "8973707065492927082861195054109576870694908245824610027755990627141925319437696743883383397010093961606801"
        "32442324992e+223"},
    {0x1.fffffffffffffp+742, chars_format::scientific, 223,
        "4."
        "6268644699475430333162732914847119748119399991650596212909143540748653550347931819399093890842857924993713"
        "7947414130985854165722390108219153741389816491649220055511981254283850638875393487766766794020187923213602"
        "64884649984e+223"},
    {0x1.fffffffffffffp+743, chars_format::scientific, 223,
        "9."
        "2537289398950860666325465829694239496238799983301192425818287081497307100695863638798187781685715849987427"
        "5894828261971708331444780216438307482779632983298440111023962508567701277750786975533533588040375846427205"
        "29769299968e+223"},
    {0x1.fffffffffffffp+744, chars_format::scientific, 224,
        "1."
        "8507457879790172133265093165938847899247759996660238485163657416299461420139172727759637556337143169997485"
        "5178965652394341666288956043287661496555926596659688022204792501713540255550157395106706717608075169285441"
        "059538599936e+224"},
    {0x1.fffffffffffffp+745, chars_format::scientific, 224,
        "3."
        "7014915759580344266530186331877695798495519993320476970327314832598922840278345455519275112674286339994971"
        "0357931304788683332577912086575322993111853193319376044409585003427080511100314790213413435216150338570882"
        "119077199872e+224"},
    {0x1.fffffffffffffp+746, chars_format::scientific, 224,
        "7."
        "4029831519160688533060372663755391596991039986640953940654629665197845680556690911038550225348572679989942"
        "0715862609577366665155824173150645986223706386638752088819170006854161022200629580426826870432300677141764"
        "238154399744e+224"},
    {0x1.fffffffffffffp+747, chars_format::scientific, 225,
        "1."
        "4805966303832137706612074532751078319398207997328190788130925933039569136111338182207710045069714535997988"
        "4143172521915473333031164834630129197244741277327750417763834001370832204440125916085365374086460135428352"
        "8476308799488e+225"},
    {0x1.fffffffffffffp+748, chars_format::scientific, 225,
        "2."
        "9611932607664275413224149065502156638796415994656381576261851866079138272222676364415420090139429071995976"
        "8286345043830946666062329669260258394489482554655500835527668002741664408880251832170730748172920270856705"
        "6952617598976e+225"},
    {0x1.fffffffffffffp+749, chars_format::scientific, 225,
        "5."
        "9223865215328550826448298131004313277592831989312763152523703732158276544445352728830840180278858143991953"
        "6572690087661893332124659338520516788978965109311001671055336005483328817760503664341461496345840541713411"
        "3905235197952e+225"},
    {0x1.fffffffffffffp+750, chars_format::scientific, 226,
        "1."
        "1844773043065710165289659626200862655518566397862552630504740746431655308889070545766168036055771628798390"
        "7314538017532378666424931867704103357795793021862200334211067201096665763552100732868292299269168108342682"
        "27810470395904e+226"},
    {0x1.fffffffffffffp+751, chars_format::scientific, 226,
        "2."
        "3689546086131420330579319252401725311037132795725105261009481492863310617778141091532336072111543257596781"
        "4629076035064757332849863735408206715591586043724400668422134402193331527104201465736584598538336216685364"
        "55620940791808e+226"},
    {0x1.fffffffffffffp+752, chars_format::scientific, 226,
        "4."
        "7379092172262840661158638504803450622074265591450210522018962985726621235556282183064672144223086515193562"
        "9258152070129514665699727470816413431183172087448801336844268804386663054208402931473169197076672433370729"
        "11241881583616e+226"},
    {0x1.fffffffffffffp+753, chars_format::scientific, 226,
        "9."
        "4758184344525681322317277009606901244148531182900421044037925971453242471112564366129344288446173030387125"
        "8516304140259029331399454941632826862366344174897602673688537608773326108416805862946338394153344866741458"
        "22483763167232e+226"},
    {0x1.fffffffffffffp+754, chars_format::scientific, 227,
        "1."
        "8951636868905136264463455401921380248829706236580084208807585194290648494222512873225868857689234606077425"
        "1703260828051805866279890988326565372473268834979520534737707521754665221683361172589267678830668973348291"
        "644967526334464e+227"},
    {0x1.fffffffffffffp+755, chars_format::scientific, 227,
        "3."
        "7903273737810272528926910803842760497659412473160168417615170388581296988445025746451737715378469212154850"
        "3406521656103611732559781976653130744946537669959041069475415043509330443366722345178535357661337946696583"
        "289935052668928e+227"},
    {0x1.fffffffffffffp+756, chars_format::scientific, 227,
        "7."
        "5806547475620545057853821607685520995318824946320336835230340777162593976890051492903475430756938424309700"
        "6813043312207223465119563953306261489893075339918082138950830087018660886733444690357070715322675893393166"
        "579870105337856e+227"},
    {0x1.fffffffffffffp+757, chars_format::scientific, 228,
        "1."
        "5161309495124109011570764321537104199063764989264067367046068155432518795378010298580695086151387684861940"
        "1362608662441444693023912790661252297978615067983616427790166017403732177346688938071414143064535178678633"
        "3159740210675712e+228"},
    {0x1.fffffffffffffp+758, chars_format::scientific, 228,
        "3."
        "0322618990248218023141528643074208398127529978528134734092136310865037590756020597161390172302775369723880"
        "2725217324882889386047825581322504595957230135967232855580332034807464354693377876142828286129070357357266"
        "6319480421351424e+228"},
    {0x1.fffffffffffffp+759, chars_format::scientific, 228,
        "6."
        "0645237980496436046283057286148416796255059957056269468184272621730075181512041194322780344605550739447760"
        "5450434649765778772095651162645009191914460271934465711160664069614928709386755752285656572258140714714533"
        "2638960842702848e+228"},
    {0x1.fffffffffffffp+760, chars_format::scientific, 229,
        "1."
        "2129047596099287209256611457229683359251011991411253893636854524346015036302408238864556068921110147889552"
        "1090086929953155754419130232529001838382892054386893142232132813922985741877351150457131314451628142942906"
        "65277921685405696e+229"},
    {0x1.fffffffffffffp+761, chars_format::scientific, 229,
        "2."
        "4258095192198574418513222914459366718502023982822507787273709048692030072604816477729112137842220295779104"
        "2180173859906311508838260465058003676765784108773786284464265627845971483754702300914262628903256285885813"
        "30555843370811392e+229"},
    {0x1.fffffffffffffp+762, chars_format::scientific, 229,
        "4."
        "8516190384397148837026445828918733437004047965645015574547418097384060145209632955458224275684440591558208"
        "4360347719812623017676520930116007353531568217547572568928531255691942967509404601828525257806512571771626"
        "61111686741622784e+229"},
    {0x1.fffffffffffffp+763, chars_format::scientific, 229,
        "9."
        "7032380768794297674052891657837466874008095931290031149094836194768120290419265910916448551368881183116416"
        "8720695439625246035353041860232014707063136435095145137857062511383885935018809203657050515613025143543253"
        "22223373483245568e+229"},
    {0x1.fffffffffffffp+764, chars_format::scientific, 230,
        "1."
        "9406476153758859534810578331567493374801619186258006229818967238953624058083853182183289710273776236623283"
        "3744139087925049207070608372046402941412627287019029027571412502276777187003761840731410103122605028708650"
        "644446746966491136e+230"},
    {0x1.fffffffffffffp+765, chars_format::scientific, 230,
        "3."
        "8812952307517719069621156663134986749603238372516012459637934477907248116167706364366579420547552473246566"
        "7488278175850098414141216744092805882825254574038058055142825004553554374007523681462820206245210057417301"
        "288893493932982272e+230"},
    {0x1.fffffffffffffp+766, chars_format::scientific, 230,
        "7."
        "7625904615035438139242313326269973499206476745032024919275868955814496232335412728733158841095104946493133"
        "4976556351700196828282433488185611765650509148076116110285650009107108748015047362925640412490420114834602"
        "577786987865964544e+230"},
    {0x1.fffffffffffffp+767, chars_format::scientific, 231,
        "1."
        "5525180923007087627848462665253994699841295349006404983855173791162899246467082545746631768219020989298626"
        "6995311270340039365656486697637122353130101829615223222057130001821421749603009472585128082498084022966920"
        "5155573975731929088e+231"},
    {0x1.fffffffffffffp+768, chars_format::scientific, 231,
        "3."
        "1050361846014175255696925330507989399682590698012809967710347582325798492934165091493263536438041978597253"
        "3990622540680078731312973395274244706260203659230446444114260003642843499206018945170256164996168045933841"
        "0311147951463858176e+231"},
    {0x1.fffffffffffffp+769, chars_format::scientific, 231,
        "6."
        "2100723692028350511393850661015978799365181396025619935420695164651596985868330182986527072876083957194506"
        "7981245081360157462625946790548489412520407318460892888228520007285686998412037890340512329992336091867682"
        "0622295902927716352e+231"},
    {0x1.fffffffffffffp+770, chars_format::scientific, 232,
        "1."
        "2420144738405670102278770132203195759873036279205123987084139032930319397173666036597305414575216791438901"
        "3596249016272031492525189358109697882504081463692178577645704001457137399682407578068102465998467218373536"
        "41244591805855432704e+232"},
    {0x1.fffffffffffffp+771, chars_format::scientific, 232,
        "2."
        "4840289476811340204557540264406391519746072558410247974168278065860638794347332073194610829150433582877802"
        "7192498032544062985050378716219395765008162927384357155291408002914274799364815156136204931996934436747072"
        "82489183611710865408e+232"},
    {0x1.fffffffffffffp+772, chars_format::scientific, 232,
        "4."
        "9680578953622680409115080528812783039492145116820495948336556131721277588694664146389221658300867165755605"
        "4384996065088125970100757432438791530016325854768714310582816005828549598729630312272409863993868873494145"
        "64978367223421730816e+232"},
    {0x1.fffffffffffffp+773, chars_format::scientific, 232,
        "9."
        "9361157907245360818230161057625566078984290233640991896673112263442555177389328292778443316601734331511210"
        "8769992130176251940201514864877583060032651709537428621165632011657099197459260624544819727987737746988291"
        "29956734446843461632e+232"},
    {0x1.fffffffffffffp+774, chars_format::scientific, 233,
        "1."
        "9872231581449072163646032211525113215796858046728198379334622452688511035477865658555688663320346866302242"
        "1753998426035250388040302972975516612006530341907485724233126402331419839491852124908963945597547549397658"
        "259913468893686923264e+233"},
    {0x1.fffffffffffffp+775, chars_format::scientific, 233,
        "3."
        "9744463162898144327292064423050226431593716093456396758669244905377022070955731317111377326640693732604484"
        "3507996852070500776080605945951033224013060683814971448466252804662839678983704249817927891195095098795316"
        "519826937787373846528e+233"},
    {0x1.fffffffffffffp+776, chars_format::scientific, 233,
        "7."
        "9488926325796288654584128846100452863187432186912793517338489810754044141911462634222754653281387465208968"
        "7015993704141001552161211891902066448026121367629942896932505609325679357967408499635855782390190197590633"
        "039653875574747693056e+233"},
    {0x1.fffffffffffffp+777, chars_format::scientific, 234,
        "1."
        "5897785265159257730916825769220090572637486437382558703467697962150808828382292526844550930656277493041793"
        "7403198740828200310432242378380413289605224273525988579386501121865135871593481699927171156478038039518126"
        "6079307751149495386112e+234"},
    {0x1.fffffffffffffp+778, chars_format::scientific, 234,
        "3."
        "1795570530318515461833651538440181145274972874765117406935395924301617656764585053689101861312554986083587"
        "4806397481656400620864484756760826579210448547051977158773002243730271743186963399854342312956076079036253"
        "2158615502298990772224e+234"},
    {0x1.fffffffffffffp+779, chars_format::scientific, 234,
        "6."
        "3591141060637030923667303076880362290549945749530234813870791848603235313529170107378203722625109972167174"
        "9612794963312801241728969513521653158420897094103954317546004487460543486373926799708684625912152158072506"
        "4317231004597981544448e+234"},
    {0x1.fffffffffffffp+780, chars_format::scientific, 235,
        "1."
        "2718228212127406184733460615376072458109989149906046962774158369720647062705834021475640744525021994433434"
        "9922558992662560248345793902704330631684179418820790863509200897492108697274785359941736925182430431614501"
        "28634462009195963088896e+235"},
    {0x1.fffffffffffffp+781, chars_format::scientific, 235,
        "2."
        "5436456424254812369466921230752144916219978299812093925548316739441294125411668042951281489050043988866869"
        "9845117985325120496691587805408661263368358837641581727018401794984217394549570719883473850364860863229002"
        "57268924018391926177792e+235"},
    {0x1.fffffffffffffp+782, chars_format::scientific, 235,
        "5."
        "0872912848509624738933842461504289832439956599624187851096633478882588250823336085902562978100087977733739"
        "9690235970650240993383175610817322526736717675283163454036803589968434789099141439766947700729721726458005"
        "14537848036783852355584e+235"},
    {0x1.fffffffffffffp+783, chars_format::scientific, 236,
        "1."
        "0174582569701924947786768492300857966487991319924837570219326695776517650164667217180512595620017595546747"
        "9938047194130048198676635122163464505347343535056632690807360717993686957819828287953389540145944345291601"
        "029075696073567704711168e+236"},
    {0x1.fffffffffffffp+784, chars_format::scientific, 236,
        "2."
        "0349165139403849895573536984601715932975982639849675140438653391553035300329334434361025191240035191093495"
        "9876094388260096397353270244326929010694687070113265381614721435987373915639656575906779080291888690583202"
        "058151392147135409422336e+236"},
    {0x1.fffffffffffffp+785, chars_format::scientific, 236,
        "4."
        "0698330278807699791147073969203431865951965279699350280877306783106070600658668868722050382480070382186991"
        "9752188776520192794706540488653858021389374140226530763229442871974747831279313151813558160583777381166404"
        "116302784294270818844672e+236"},
    {0x1.fffffffffffffp+786, chars_format::scientific, 236,
        "8."
        "1396660557615399582294147938406863731903930559398700561754613566212141201317337737444100764960140764373983"
        "9504377553040385589413080977307716042778748280453061526458885743949495662558626303627116321167554762332808"
        "232605568588541637689344e+236"},
    {0x1.fffffffffffffp+787, chars_format::scientific, 237,
        "1."
        "6279332111523079916458829587681372746380786111879740112350922713242428240263467547488820152992028152874796"
        "7900875510608077117882616195461543208555749656090612305291777148789899132511725260725423264233510952466561"
        "6465211137177083275378688e+237"},
    {0x1.fffffffffffffp+788, chars_format::scientific, 237,
        "3."
        "2558664223046159832917659175362745492761572223759480224701845426484856480526935094977640305984056305749593"
        "5801751021216154235765232390923086417111499312181224610583554297579798265023450521450846528467021904933123"
        "2930422274354166550757376e+237"},
    {0x1.fffffffffffffp+789, chars_format::scientific, 237,
        "6."
        "5117328446092319665835318350725490985523144447518960449403690852969712961053870189955280611968112611499187"
        "1603502042432308471530464781846172834222998624362449221167108595159596530046901042901693056934043809866246"
        "5860844548708333101514752e+237"},
    {0x1.fffffffffffffp+790, chars_format::scientific, 238,
        "1."
        "3023465689218463933167063670145098197104628889503792089880738170593942592210774037991056122393622522299837"
        "4320700408486461694306092956369234566844599724872489844233421719031919306009380208580338611386808761973249"
        "31721689097416666203029504e+238"},
    {0x1.fffffffffffffp+791, chars_format::scientific, 238,
        "2."
        "6046931378436927866334127340290196394209257779007584179761476341187885184421548075982112244787245044599674"
        "8641400816972923388612185912738469133689199449744979688466843438063838612018760417160677222773617523946498"
        "63443378194833332406059008e+238"},
    {0x1.fffffffffffffp+792, chars_format::scientific, 238,
        "5."
        "2093862756873855732668254680580392788418515558015168359522952682375770368843096151964224489574490089199349"
        "7282801633945846777224371825476938267378398899489959376933686876127677224037520834321354445547235047892997"
        "26886756389666664812118016e+238"},
    {0x1.fffffffffffffp+793, chars_format::scientific, 239,
        "1."
        "0418772551374771146533650936116078557683703111603033671904590536475154073768619230392844897914898017839869"
        "9456560326789169355444874365095387653475679779897991875386737375225535444807504166864270889109447009578599"
        "453773512779333329624236032e+239"},
    {0x1.fffffffffffffp+794, chars_format::scientific, 239,
        "2."
        "0837545102749542293067301872232157115367406223206067343809181072950308147537238460785689795829796035679739"
        "8913120653578338710889748730190775306951359559795983750773474750451070889615008333728541778218894019157198"
        "907547025558666659248472064e+239"},
    {0x1.fffffffffffffp+795, chars_format::scientific, 239,
        "4."
        "1675090205499084586134603744464314230734812446412134687618362145900616295074476921571379591659592071359479"
        "7826241307156677421779497460381550613902719119591967501546949500902141779230016667457083556437788038314397"
        "815094051117333318496944128e+239"},
    {0x1.fffffffffffffp+796, chars_format::scientific, 239,
        "8."
        "3350180410998169172269207488928628461469624892824269375236724291801232590148953843142759183319184142718959"
        "5652482614313354843558994920763101227805438239183935003093899001804283558460033334914167112875576076628795"
        "630188102234666636993888256e+239"},
    {0x1.fffffffffffffp+797, chars_format::scientific, 240,
        "1."
        "6670036082199633834453841497785725692293924978564853875047344858360246518029790768628551836663836828543791"
        "9130496522862670968711798984152620245561087647836787000618779800360856711692006666982833422575115215325759"
        "1260376204469333273987776512e+240"},
    {0x1.fffffffffffffp+798, chars_format::scientific, 240,
        "3."
        "3340072164399267668907682995571451384587849957129707750094689716720493036059581537257103673327673657087583"
        "8260993045725341937423597968305240491122175295673574001237559600721713423384013333965666845150230430651518"
        "2520752408938666547975553024e+240"},
    {0x1.fffffffffffffp+799, chars_format::scientific, 240,
        "6."
        "6680144328798535337815365991142902769175699914259415500189379433440986072119163074514207346655347314175167"
        "6521986091450683874847195936610480982244350591347148002475119201443426846768026667931333690300460861303036"
        "5041504817877333095951106048e+240"},
    {0x1.fffffffffffffp+800, chars_format::scientific, 241,
        "1."
        "3336028865759707067563073198228580553835139982851883100037875886688197214423832614902841469331069462835033"
        "5304397218290136774969439187322096196448870118269429600495023840288685369353605333586266738060092172260607"
        "30083009635754666191902212096e+241"},
    {0x1.fffffffffffffp+801, chars_format::scientific, 241,
        "2."
        "6672057731519414135126146396457161107670279965703766200075751773376394428847665229805682938662138925670067"
        "0608794436580273549938878374644192392897740236538859200990047680577370738707210667172533476120184344521214"
        "60166019271509332383804424192e+241"},
    {0x1.fffffffffffffp+802, chars_format::scientific, 241,
        "5."
        "3344115463038828270252292792914322215340559931407532400151503546752788857695330459611365877324277851340134"
        "1217588873160547099877756749288384785795480473077718401980095361154741477414421334345066952240368689042429"
        "20332038543018664767608848384e+241"},
    {0x1.fffffffffffffp+803, chars_format::scientific, 242,
        "1."
        "0668823092607765654050458558582864443068111986281506480030300709350557771539066091922273175464855570268026"
        "8243517774632109419975551349857676957159096094615543680396019072230948295482884266869013390448073737808485"
        "840664077086037329535217696768e+242"},
    {0x1.fffffffffffffp+804, chars_format::scientific, 242,
        "2."
        "1337646185215531308100917117165728886136223972563012960060601418701115543078132183844546350929711140536053"
        "6487035549264218839951102699715353914318192189231087360792038144461896590965768533738026780896147475616971"
        "681328154172074659070435393536e+242"},
    {0x1.fffffffffffffp+805, chars_format::scientific, 242,
        "4."
        "2675292370431062616201834234331457772272447945126025920121202837402231086156264367689092701859422281072107"
        "2974071098528437679902205399430707828636384378462174721584076288923793181931537067476053561792294951233943"
        "362656308344149318140870787072e+242"},
    {0x1.fffffffffffffp+806, chars_format::scientific, 242,
        "8."
        "5350584740862125232403668468662915544544895890252051840242405674804462172312528735378185403718844562144214"
        "5948142197056875359804410798861415657272768756924349443168152577847586363863074134952107123584589902467886"
        "725312616688298636281741574144e+242"},
    {0x1.fffffffffffffp+807, chars_format::scientific, 243,
        "1."
        "7070116948172425046480733693732583108908979178050410368048481134960892434462505747075637080743768912428842"
        "9189628439411375071960882159772283131454553751384869888633630515569517272772614826990421424716917980493577"
        "3450625233376597272563483148288e+243"},
    {0x1.fffffffffffffp+808, chars_format::scientific, 243,
        "3."
        "4140233896344850092961467387465166217817958356100820736096962269921784868925011494151274161487537824857685"
        "8379256878822750143921764319544566262909107502769739777267261031139034545545229653980842849433835960987154"
        "6901250466753194545126966296576e+243"},
    {0x1.fffffffffffffp+809, chars_format::scientific, 243,
        "6."
        "8280467792689700185922934774930332435635916712201641472193924539843569737850022988302548322975075649715371"
        "6758513757645500287843528639089132525818215005539479554534522062278069091090459307961685698867671921974309"
        "3802500933506389090253932593152e+243"},
    {0x1.fffffffffffffp+810, chars_format::scientific, 244,
        "1."
        "3656093558537940037184586954986066487127183342440328294438784907968713947570004597660509664595015129943074"
        "3351702751529100057568705727817826505163643001107895910906904412455613818218091861592337139773534384394861"
        "87605001867012778180507865186304e+244"},
    {0x1.fffffffffffffp+811, chars_format::scientific, 244,
        "2."
        "7312187117075880074369173909972132974254366684880656588877569815937427895140009195321019329190030259886148"
        "6703405503058200115137411455635653010327286002215791821813808824911227636436183723184674279547068768789723"
        "75210003734025556361015730372608e+244"},
    {0x1.fffffffffffffp+812, chars_format::scientific, 244,
        "5."
        "4624374234151760148738347819944265948508733369761313177755139631874855790280018390642038658380060519772297"
        "3406811006116400230274822911271306020654572004431583643627617649822455272872367446369348559094137537579447"
        "50420007468051112722031460745216e+244"},
    {0x1.fffffffffffffp+813, chars_format::scientific, 245,
        "1."
        "0924874846830352029747669563988853189701746673952262635551027926374971158056003678128407731676012103954459"
        "4681362201223280046054964582254261204130914400886316728725523529964491054574473489273869711818827507515889"
        "500840014936102225444062921490432e+245"},
    {0x1.fffffffffffffp+814, chars_format::scientific, 245,
        "2."
        "1849749693660704059495339127977706379403493347904525271102055852749942316112007356256815463352024207908918"
        "9362724402446560092109929164508522408261828801772633457451047059928982109148946978547739423637655015031779"
        "001680029872204450888125842980864e+245"},
    {0x1.fffffffffffffp+815, chars_format::scientific, 245,
        "4."
        "3699499387321408118990678255955412758806986695809050542204111705499884632224014712513630926704048415817837"
        "8725448804893120184219858329017044816523657603545266914902094119857964218297893957095478847275310030063558"
        "003360059744408901776251685961728e+245"},
    {0x1.fffffffffffffp+816, chars_format::scientific, 245,
        "8."
        "7398998774642816237981356511910825517613973391618101084408223410999769264448029425027261853408096831635675"
        "7450897609786240368439716658034089633047315207090533829804188239715928436595787914190957694550620060127116"
        "006720119488817803552503371923456e+245"},
    {0x1.fffffffffffffp+817, chars_format::scientific, 246,
        "1."
        "7479799754928563247596271302382165103522794678323620216881644682199953852889605885005452370681619366327135"
        "1490179521957248073687943331606817926609463041418106765960837647943185687319157582838191538910124012025423"
        "2013440238977635607105006743846912e+246"},
    {0x1.fffffffffffffp+818, chars_format::scientific, 246,
        "3."
        "4959599509857126495192542604764330207045589356647240433763289364399907705779211770010904741363238732654270"
        "2980359043914496147375886663213635853218926082836213531921675295886371374638315165676383077820248024050846"
        "4026880477955271214210013487693824e+246"},
    {0x1.fffffffffffffp+819, chars_format::scientific, 246,
        "6."
        "9919199019714252990385085209528660414091178713294480867526578728799815411558423540021809482726477465308540"
        "5960718087828992294751773326427271706437852165672427063843350591772742749276630331352766155640496048101692"
        "8053760955910542428420026975387648e+246"},
    {0x1.fffffffffffffp+820, chars_format::scientific, 247,
        "1."
        "3983839803942850598077017041905732082818235742658896173505315745759963082311684708004361896545295493061708"
        "1192143617565798458950354665285454341287570433134485412768670118354548549855326066270553231128099209620338"
        "56107521911821084856840053950775296e+247"},
    {0x1.fffffffffffffp+821, chars_format::scientific, 247,
        "2."
        "7967679607885701196154034083811464165636471485317792347010631491519926164623369416008723793090590986123416"
        "2384287235131596917900709330570908682575140866268970825537340236709097099710652132541106462256198419240677"
        "12215043823642169713680107901550592e+247"},
    {0x1.fffffffffffffp+822, chars_format::scientific, 247,
        "5."
        "5935359215771402392308068167622928331272942970635584694021262983039852329246738832017447586181181972246832"
        "4768574470263193835801418661141817365150281732537941651074680473418194199421304265082212924512396838481354"
        "24430087647284339427360215803101184e+247"},
    {0x1.fffffffffffffp+823, chars_format::scientific, 248,
        "1."
        "1187071843154280478461613633524585666254588594127116938804252596607970465849347766403489517236236394449366"
        "4953714894052638767160283732228363473030056346507588330214936094683638839884260853016442584902479367696270"
        "848860175294568678854720431606202368e+248"},
    {0x1.fffffffffffffp+824, chars_format::scientific, 248,
        "2."
        "2374143686308560956923227267049171332509177188254233877608505193215940931698695532806979034472472788898732"
        "9907429788105277534320567464456726946060112693015176660429872189367277679768521706032885169804958735392541"
        "697720350589137357709440863212404736e+248"},
    {0x1.fffffffffffffp+825, chars_format::scientific, 248,
        "4."
        "4748287372617121913846454534098342665018354376508467755217010386431881863397391065613958068944945577797465"
        "9814859576210555068641134928913453892120225386030353320859744378734555359537043412065770339609917470785083"
        "395440701178274715418881726424809472e+248"},
    {0x1.fffffffffffffp+826, chars_format::scientific, 248,
        "8."
        "9496574745234243827692909068196685330036708753016935510434020772863763726794782131227916137889891155594931"
        "9629719152421110137282269857826907784240450772060706641719488757469110719074086824131540679219834941570166"
        "790881402356549430837763452849618944e+248"},
    {0x1.fffffffffffffp+827, chars_format::scientific, 249,
        "1."
        "7899314949046848765538581813639337066007341750603387102086804154572752745358956426245583227577978231118986"
        "3925943830484222027456453971565381556848090154412141328343897751493822143814817364826308135843966988314033"
        "3581762804713098861675526905699237888e+249"},
    {0x1.fffffffffffffp+828, chars_format::scientific, 249,
        "3."
        "5798629898093697531077163627278674132014683501206774204173608309145505490717912852491166455155956462237972"
        "7851887660968444054912907943130763113696180308824282656687795502987644287629634729652616271687933976628066"
        "7163525609426197723351053811398475776e+249"},
    {0x1.fffffffffffffp+829, chars_format::scientific, 249,
        "7."
        "1597259796187395062154327254557348264029367002413548408347216618291010981435825704982332910311912924475945"
        "5703775321936888109825815886261526227392360617648565313375591005975288575259269459305232543375867953256133"
        "4327051218852395446702107622796951552e+249"},
    {0x1.fffffffffffffp+830, chars_format::scientific, 250,
        "1."
        "4319451959237479012430865450911469652805873400482709681669443323658202196287165140996466582062382584895189"
        "1140755064387377621965163177252305245478472123529713062675118201195057715051853891861046508675173590651226"
        "68654102437704790893404215245593903104e+250"},
    {0x1.fffffffffffffp+831, chars_format::scientific, 250,
        "2."
        "8638903918474958024861730901822939305611746800965419363338886647316404392574330281992933164124765169790378"
        "2281510128774755243930326354504610490956944247059426125350236402390115430103707783722093017350347181302453"
        "37308204875409581786808430491187806208e+250"},
    {0x1.fffffffffffffp+832, chars_format::scientific, 250,
        "5."
        "7277807836949916049723461803645878611223493601930838726677773294632808785148660563985866328249530339580756"
        "4563020257549510487860652709009220981913888494118852250700472804780230860207415567444186034700694362604906"
        "74616409750819163573616860982375612416e+250"},
    {0x1.fffffffffffffp+833, chars_format::scientific, 251,
        "1."
        "1455561567389983209944692360729175722244698720386167745335554658926561757029732112797173265649906067916151"
        "2912604051509902097572130541801844196382777698823770450140094560956046172041483113488837206940138872520981"
        "349232819501638327147233721964751224832e+251"},
    {0x1.fffffffffffffp+834, chars_format::scientific, 251,
        "2."
        "2911123134779966419889384721458351444489397440772335490671109317853123514059464225594346531299812135832302"
        "5825208103019804195144261083603688392765555397647540900280189121912092344082966226977674413880277745041962"
        "698465639003276654294467443929502449664e+251"},
    {0x1.fffffffffffffp+835, chars_format::scientific, 251,
        "4."
        "5822246269559932839778769442916702888978794881544670981342218635706247028118928451188693062599624271664605"
        "1650416206039608390288522167207376785531110795295081800560378243824184688165932453955348827760555490083925"
        "396931278006553308588934887859004899328e+251"},
    {0x1.fffffffffffffp+836, chars_format::scientific, 251,
        "9."
        "1644492539119865679557538885833405777957589763089341962684437271412494056237856902377386125199248543329210"
        "3300832412079216780577044334414753571062221590590163601120756487648369376331864907910697655521110980167850"
        "793862556013106617177869775718009798656e+251"},
    {0x1.fffffffffffffp+837, chars_format::scientific, 252,
        "1."
        "8328898507823973135911507777166681155591517952617868392536887454282498811247571380475477225039849708665842"
        "0660166482415843356115408866882950714212444318118032720224151297529673875266372981582139531104222196033570"
        "1587725112026213234355739551436019597312e+252"},
    {0x1.fffffffffffffp+838, chars_format::scientific, 252,
        "3."
        "6657797015647946271823015554333362311183035905235736785073774908564997622495142760950954450079699417331684"
        "1320332964831686712230817733765901428424888636236065440448302595059347750532745963164279062208444392067140"
        "3175450224052426468711479102872039194624e+252"},
    {0x1.fffffffffffffp+839, chars_format::scientific, 252,
        "7."
        "3315594031295892543646031108666724622366071810471473570147549817129995244990285521901908900159398834663368"
        "2640665929663373424461635467531802856849777272472130880896605190118695501065491926328558124416888784134280"
        "6350900448104852937422958205744078389248e+252"},
    {0x1.fffffffffffffp+840, chars_format::scientific, 253,
        "1."
        "4663118806259178508729206221733344924473214362094294714029509963425999048998057104380381780031879766932673"
        "6528133185932674684892327093506360571369955454494426176179321038023739100213098385265711624883377756826856"
        "12701800896209705874845916411488156778496e+253"},
    {0x1.fffffffffffffp+841, chars_format::scientific, 253,
        "2."
        "9326237612518357017458412443466689848946428724188589428059019926851998097996114208760763560063759533865347"
        "3056266371865349369784654187012721142739910908988852352358642076047478200426196770531423249766755513653712"
        "25403601792419411749691832822976313556992e+253"},
    {0x1.fffffffffffffp+842, chars_format::scientific, 253,
        "5."
        "8652475225036714034916824886933379697892857448377178856118039853703996195992228417521527120127519067730694"
        "6112532743730698739569308374025442285479821817977704704717284152094956400852393541062846499533511027307424"
        "50807203584838823499383665645952627113984e+253"},
    {0x1.fffffffffffffp+843, chars_format::scientific, 254,
        "1."
        "1730495045007342806983364977386675939578571489675435771223607970740799239198445683504305424025503813546138"
        "9222506548746139747913861674805088457095964363595540940943456830418991280170478708212569299906702205461484"
        "901614407169677646998767331291905254227968e+254"},
    {0x1.fffffffffffffp+844, chars_format::scientific, 254,
        "2."
        "3460990090014685613966729954773351879157142979350871542447215941481598478396891367008610848051007627092277"
        "8445013097492279495827723349610176914191928727191081881886913660837982560340957416425138599813404410922969"
        "803228814339355293997534662583810508455936e+254"},
    {0x1.fffffffffffffp+845, chars_format::scientific, 254,
        "4."
        "6921980180029371227933459909546703758314285958701743084894431882963196956793782734017221696102015254184555"
        "6890026194984558991655446699220353828383857454382163763773827321675965120681914832850277199626808821845939"
        "606457628678710587995069325167621016911872e+254"},
    {0x1.fffffffffffffp+846, chars_format::scientific, 254,
        "9."
        "3843960360058742455866919819093407516628571917403486169788863765926393913587565468034443392204030508369111"
        "3780052389969117983310893398440707656767714908764327527547654643351930241363829665700554399253617643691879"
        "212915257357421175990138650335242033823744e+254"},
    {0x1.fffffffffffffp+847, chars_format::scientific, 255,
        "1."
        "8768792072011748491173383963818681503325714383480697233957772753185278782717513093606888678440806101673822"
        "2756010477993823596662178679688141531353542981752865505509530928670386048272765933140110879850723528738375"
        "8425830514714842351980277300670484067647488e+255"},
    {0x1.fffffffffffffp+848, chars_format::scientific, 255,
        "3."
        "7537584144023496982346767927637363006651428766961394467915545506370557565435026187213777356881612203347644"
        "5512020955987647193324357359376283062707085963505731011019061857340772096545531866280221759701447057476751"
        "6851661029429684703960554601340968135294976e+255"},
    {0x1.fffffffffffffp+849, chars_format::scientific, 255,
        "7."
        "5075168288046993964693535855274726013302857533922788935831091012741115130870052374427554713763224406695289"
        "1024041911975294386648714718752566125414171927011462022038123714681544193091063732560443519402894114953503"
        "3703322058859369407921109202681936270589952e+255"},
    {0x1.fffffffffffffp+850, chars_format::scientific, 256,
        "1."
        "5015033657609398792938707171054945202660571506784557787166218202548223026174010474885510942752644881339057"
        "8204808382395058877329742943750513225082834385402292404407624742936308838618212746512088703880578822990700"
        "67406644117718738815842218405363872541179904e+256"},
    {0x1.fffffffffffffp+851, chars_format::scientific, 256,
        "3."
        "0030067315218797585877414342109890405321143013569115574332436405096446052348020949771021885505289762678115"
        "6409616764790117754659485887501026450165668770804584808815249485872617677236425493024177407761157645981401"
        "34813288235437477631684436810727745082359808e+256"},
    {0x1.fffffffffffffp+852, chars_format::scientific, 256,
        "6."
        "0060134630437595171754828684219780810642286027138231148664872810192892104696041899542043771010579525356231"
        "2819233529580235509318971775002052900331337541609169617630498971745235354472850986048354815522315291962802"
        "69626576470874955263368873621455490164719616e+256"},
    {0x1.fffffffffffffp+853, chars_format::scientific, 257,
        "1."
        "2012026926087519034350965736843956162128457205427646229732974562038578420939208379908408754202115905071246"
        "2563846705916047101863794355000410580066267508321833923526099794349047070894570197209670963104463058392560"
        "539253152941749910526737747242910980329439232e+257"},
    {0x1.fffffffffffffp+854, chars_format::scientific, 257,
        "2."
        "4024053852175038068701931473687912324256914410855292459465949124077156841878416759816817508404231810142492"
        "5127693411832094203727588710000821160132535016643667847052199588698094141789140394419341926208926116785121"
        "078506305883499821053475494485821960658878464e+257"},
    {0x1.fffffffffffffp+855, chars_format::scientific, 257,
        "4."
        "8048107704350076137403862947375824648513828821710584918931898248154313683756833519633635016808463620284985"
        "0255386823664188407455177420001642320265070033287335694104399177396188283578280788838683852417852233570242"
        "157012611766999642106950988971643921317756928e+257"},
    {0x1.fffffffffffffp+856, chars_format::scientific, 257,
        "9."
        "6096215408700152274807725894751649297027657643421169837863796496308627367513667039267270033616927240569970"
        "0510773647328376814910354840003284640530140066574671388208798354792376567156561577677367704835704467140484"
        "314025223533999284213901977943287842635513856e+257"},
    {0x1.fffffffffffffp+857, chars_format::scientific, 258,
        "1."
        "9219243081740030454961545178950329859405531528684233967572759299261725473502733407853454006723385448113994"
        "0102154729465675362982070968000656928106028013314934277641759670958475313431312315535473540967140893428096"
        "8628050447067998568427803955886575685271027712e+258"},
    {0x1.fffffffffffffp+858, chars_format::scientific, 258,
        "3."
        "8438486163480060909923090357900659718811063057368467935145518598523450947005466815706908013446770896227988"
        "0204309458931350725964141936001313856212056026629868555283519341916950626862624631070947081934281786856193"
        "7256100894135997136855607911773151370542055424e+258"},
    {0x1.fffffffffffffp+859, chars_format::scientific, 258,
        "7."
        "6876972326960121819846180715801319437622126114736935870291037197046901894010933631413816026893541792455976"
        "0408618917862701451928283872002627712424112053259737110567038683833901253725249262141894163868563573712387"
        "4512201788271994273711215823546302741084110848e+258"},
    {0x1.fffffffffffffp+860, chars_format::scientific, 259,
        "1."
        "5375394465392024363969236143160263887524425222947387174058207439409380378802186726282763205378708358491195"
        "2081723783572540290385656774400525542484822410651947422113407736766780250745049852428378832773712714742477"
        "49024403576543988547422431647092605482168221696e+259"},
    {0x1.fffffffffffffp+861, chars_format::scientific, 259,
        "3."
        "0750788930784048727938472286320527775048850445894774348116414878818760757604373452565526410757416716982390"
        "4163447567145080580771313548801051084969644821303894844226815473533560501490099704856757665547425429484954"
        "98048807153087977094844863294185210964336443392e+259"},
    {0x1.fffffffffffffp+862, chars_format::scientific, 259,
        "6."
        "1501577861568097455876944572641055550097700891789548696232829757637521515208746905131052821514833433964780"
        "8326895134290161161542627097602102169939289642607789688453630947067121002980199409713515331094850858969909"
        "96097614306175954189689726588370421928672886784e+259"},
    {0x1.fffffffffffffp+863, chars_format::scientific, 260,
        "1."
        "2300315572313619491175388914528211110019540178357909739246565951527504303041749381026210564302966686792956"
        "1665379026858032232308525419520420433987857928521557937690726189413424200596039881942703066218970171793981"
        "992195228612351908379379453176740843857345773568e+260"},
    {0x1.fffffffffffffp+864, chars_format::scientific, 260,
        "2."
        "4600631144627238982350777829056422220039080356715819478493131903055008606083498762052421128605933373585912"
        "3330758053716064464617050839040840867975715857043115875381452378826848401192079763885406132437940343587963"
        "984390457224703816758758906353481687714691547136e+260"},
    {0x1.fffffffffffffp+865, chars_format::scientific, 260,
        "4."
        "9201262289254477964701555658112844440078160713431638956986263806110017212166997524104842257211866747171824"
        "6661516107432128929234101678081681735951431714086231750762904757653696802384159527770812264875880687175927"
        "968780914449407633517517812706963375429383094272e+260"},
    {0x1.fffffffffffffp+866, chars_format::scientific, 260,
        "9."
        "8402524578508955929403111316225688880156321426863277913972527612220034424333995048209684514423733494343649"
        "3323032214864257858468203356163363471902863428172463501525809515307393604768319055541624529751761374351855"
        "937561828898815267035035625413926750858766188544e+260"},
    {0x1.fffffffffffffp+867, chars_format::scientific, 261,
        "1."
        "9680504915701791185880622263245137776031264285372655582794505522444006884866799009641936902884746698868729"
        "8664606442972851571693640671232672694380572685634492700305161903061478720953663811108324905950352274870371"
        "1875123657797630534070071250827853501717532377088e+261"},
    {0x1.fffffffffffffp+868, chars_format::scientific, 261,
        "3."
        "9361009831403582371761244526490275552062528570745311165589011044888013769733598019283873805769493397737459"
        "7329212885945703143387281342465345388761145371268985400610323806122957441907327622216649811900704549740742"
        "3750247315595261068140142501655707003435064754176e+261"},
    {0x1.fffffffffffffp+869, chars_format::scientific, 261,
        "7."
        "8722019662807164743522489052980551104125057141490622331178022089776027539467196038567747611538986795474919"
        "4658425771891406286774562684930690777522290742537970801220647612245914883814655244433299623801409099481484"
        "7500494631190522136280285003311414006870129508352e+261"},
    {0x1.fffffffffffffp+870, chars_format::scientific, 262,
        "1."
        "5744403932561432948704497810596110220825011428298124466235604417955205507893439207713549522307797359094983"
        "8931685154378281257354912536986138155504458148507594160244129522449182976762931048886659924760281819896296"
        "95000989262381044272560570006622828013740259016704e+262"},
    {0x1.fffffffffffffp+871, chars_format::scientific, 262,
        "3."
        "1488807865122865897408995621192220441650022856596248932471208835910411015786878415427099044615594718189967"
        "7863370308756562514709825073972276311008916297015188320488259044898365953525862097773319849520563639792593"
        "90001978524762088545121140013245656027480518033408e+262"},
    {0x1.fffffffffffffp+872, chars_format::scientific, 262,
        "6."
        "2977615730245731794817991242384440883300045713192497864942417671820822031573756830854198089231189436379935"
        "5726740617513125029419650147944552622017832594030376640976518089796731907051724195546639699041127279585187"
        "80003957049524177090242280026491312054961036066816e+262"},
    {0x1.fffffffffffffp+873, chars_format::scientific, 263,
        "1."
        "2595523146049146358963598248476888176660009142638499572988483534364164406314751366170839617846237887275987"
        "1145348123502625005883930029588910524403566518806075328195303617959346381410344839109327939808225455917037"
        "560007914099048354180484560052982624109922072133632e+263"},
    {0x1.fffffffffffffp+874, chars_format::scientific, 263,
        "2."
        "5191046292098292717927196496953776353320018285276999145976967068728328812629502732341679235692475774551974"
        "2290696247005250011767860059177821048807133037612150656390607235918692762820689678218655879616450911834075"
        "120015828198096708360969120105965248219844144267264e+263"},
    {0x1.fffffffffffffp+875, chars_format::scientific, 263,
        "5."
        "0382092584196585435854392993907552706640036570553998291953934137456657625259005464683358471384951549103948"
        "4581392494010500023535720118355642097614266075224301312781214471837385525641379356437311759232901823668150"
        "240031656396193416721938240211930496439688288534528e+263"},
    {0x1.fffffffffffffp+876, chars_format::scientific, 264,
        "1."
        "0076418516839317087170878598781510541328007314110799658390786827491331525051801092936671694276990309820789"
        "6916278498802100004707144023671128419522853215044860262556242894367477105128275871287462351846580364733630"
        "0480063312792386833443876480423860992879376577069056e+264"},
    {0x1.fffffffffffffp+877, chars_format::scientific, 264,
        "2."
        "0152837033678634174341757197563021082656014628221599316781573654982663050103602185873343388553980619641579"
        "3832556997604200009414288047342256839045706430089720525112485788734954210256551742574924703693160729467260"
        "0960126625584773666887752960847721985758753154138112e+264"},
    {0x1.fffffffffffffp+878, chars_format::scientific, 264,
        "4."
        "0305674067357268348683514395126042165312029256443198633563147309965326100207204371746686777107961239283158"
        "7665113995208400018828576094684513678091412860179441050224971577469908420513103485149849407386321458934520"
        "1920253251169547333775505921695443971517506308276224e+264"},
    {0x1.fffffffffffffp+879, chars_format::scientific, 264,
        "8."
        "0611348134714536697367028790252084330624058512886397267126294619930652200414408743493373554215922478566317"
        "5330227990416800037657152189369027356182825720358882100449943154939816841026206970299698814772642917869040"
        "3840506502339094667551011843390887943035012616552448e+264"},
    {0x1.fffffffffffffp+880, chars_format::scientific, 265,
        "1."
        "6122269626942907339473405758050416866124811702577279453425258923986130440082881748698674710843184495713263"
        "5066045598083360007531430437873805471236565144071776420089988630987963368205241394059939762954528583573808"
        "07681013004678189335102023686781775886070025233104896e+265"},
    {0x1.fffffffffffffp+881, chars_format::scientific, 265,
        "3."
        "2244539253885814678946811516100833732249623405154558906850517847972260880165763497397349421686368991426527"
        "0132091196166720015062860875747610942473130288143552840179977261975926736410482788119879525909057167147616"
        "15362026009356378670204047373563551772140050466209792e+265"},
    {0x1.fffffffffffffp+882, chars_format::scientific, 265,
        "6."
        "4489078507771629357893623032201667464499246810309117813701035695944521760331526994794698843372737982853054"
        "0264182392333440030125721751495221884946260576287105680359954523951853472820965576239759051818114334295232"
        "30724052018712757340408094747127103544280100932419584e+265"},
    {0x1.fffffffffffffp+883, chars_format::scientific, 266,
        "1."
        "2897815701554325871578724606440333492899849362061823562740207139188904352066305398958939768674547596570610"
        "8052836478466688006025144350299044376989252115257421136071990904790370694564193115247951810363622866859046"
        "461448104037425514680816189494254207088560201864839168e+266"},
    {0x1.fffffffffffffp+884, chars_format::scientific, 266,
        "2."
        "5795631403108651743157449212880666985799698724123647125480414278377808704132610797917879537349095193141221"
        "6105672956933376012050288700598088753978504230514842272143981809580741389128386230495903620727245733718092"
        "922896208074851029361632378988508414177120403729678336e+266"},
    {0x1.fffffffffffffp+885, chars_format::scientific, 266,
        "5."
        "1591262806217303486314898425761333971599397448247294250960828556755617408265221595835759074698190386282443"
        "2211345913866752024100577401196177507957008461029684544287963619161482778256772460991807241454491467436185"
        "845792416149702058723264757977016828354240807459356672e+266"},
    {0x1.fffffffffffffp+886, chars_format::scientific, 267,
        "1."
        "0318252561243460697262979685152266794319879489649458850192165711351123481653044319167151814939638077256488"
        "6442269182773350404820115480239235501591401692205936908857592723832296555651354492198361448290898293487237"
        "1691584832299404117446529515954033656708481614918713344e+267"},
    {0x1.fffffffffffffp+887, chars_format::scientific, 267,
        "2."
        "0636505122486921394525959370304533588639758979298917700384331422702246963306088638334303629879276154512977"
        "2884538365546700809640230960478471003182803384411873817715185447664593111302708984396722896581796586974474"
        "3383169664598808234893059031908067313416963229837426688e+267"},
    {0x1.fffffffffffffp+888, chars_format::scientific, 267,
        "4."
        "1273010244973842789051918740609067177279517958597835400768662845404493926612177276668607259758552309025954"
        "5769076731093401619280461920956942006365606768823747635430370895329186222605417968793445793163593173948948"
        "6766339329197616469786118063816134626833926459674853376e+267"},
    {0x1.fffffffffffffp+889, chars_format::scientific, 267,
        "8."
        "2546020489947685578103837481218134354559035917195670801537325690808987853224354553337214519517104618051909"
        "1538153462186803238560923841913884012731213537647495270860741790658372445210835937586891586327186347897897"
        "3532678658395232939572236127632269253667852919349706752e+267"},
    {0x1.fffffffffffffp+890, chars_format::scientific, 268,
        "1."
        "6509204097989537115620767496243626870911807183439134160307465138161797570644870910667442903903420923610381"
        "8307630692437360647712184768382776802546242707529499054172148358131674489042167187517378317265437269579579"
        "47065357316790465879144472255264538507335705838699413504e+268"},
    {0x1.fffffffffffffp+891, chars_format::scientific, 268,
        "3."
        "3018408195979074231241534992487253741823614366878268320614930276323595141289741821334885807806841847220763"
        "6615261384874721295424369536765553605092485415058998108344296716263348978084334375034756634530874539159158"
        "94130714633580931758288944510529077014671411677398827008e+268"},
    {0x1.fffffffffffffp+892, chars_format::scientific, 268,
        "6."
        "6036816391958148462483069984974507483647228733756536641229860552647190282579483642669771615613683694441527"
        "3230522769749442590848739073531107210184970830117996216688593432526697956168668750069513269061749078318317"
        "88261429267161863516577889021058154029342823354797654016e+268"},
    {0x1.fffffffffffffp+893, chars_format::scientific, 269,
        "1."
        "3207363278391629692496613996994901496729445746751307328245972110529438056515896728533954323122736738888305"
        "4646104553949888518169747814706221442036994166023599243337718686505339591233733750013902653812349815663663"
        "576522858534323727033155778042116308058685646709595308032e+269"},
    {0x1.fffffffffffffp+894, chars_format::scientific, 269,
        "2."
        "6414726556783259384993227993989802993458891493502614656491944221058876113031793457067908646245473477776610"
        "9292209107899777036339495629412442884073988332047198486675437373010679182467467500027805307624699631327327"
        "153045717068647454066311556084232616117371293419190616064e+269"},
    {0x1.fffffffffffffp+895, chars_format::scientific, 269,
        "5."
        "2829453113566518769986455987979605986917782987005229312983888442117752226063586914135817292490946955553221"
        "8584418215799554072678991258824885768147976664094396973350874746021358364934935000055610615249399262654654"
        "306091434137294908132623112168465232234742586838381232128e+269"},
    {0x1.fffffffffffffp+896, chars_format::scientific, 270,
        "1."
        "0565890622713303753997291197595921197383556597401045862596777688423550445212717382827163458498189391110644"
        "3716883643159910814535798251764977153629595332818879394670174949204271672986987000011122123049879852530930"
        "8612182868274589816265246224336930464469485173676762464256e+270"},
    {0x1.fffffffffffffp+897, chars_format::scientific, 270,
        "2."
        "1131781245426607507994582395191842394767113194802091725193555376847100890425434765654326916996378782221288"
        "7433767286319821629071596503529954307259190665637758789340349898408543345973974000022244246099759705061861"
        "7224365736549179632530492448673860928938970347353524928512e+270"},
    {0x1.fffffffffffffp+898, chars_format::scientific, 270,
        "4."
        "2263562490853215015989164790383684789534226389604183450387110753694201780850869531308653833992757564442577"
        "4867534572639643258143193007059908614518381331275517578680699796817086691947948000044488492199519410123723"
        "4448731473098359265060984897347721857877940694707049857024e+270"},
    {0x1.fffffffffffffp+899, chars_format::scientific, 270,
        "8."
        "4527124981706430031978329580767369579068452779208366900774221507388403561701739062617307667985515128885154"
        "9735069145279286516286386014119817229036762662551035157361399593634173383895896000088976984399038820247446"
        "8897462946196718530121969794695443715755881389414099714048e+270"},
    {0x1.fffffffffffffp+900, chars_format::scientific, 271,
        "1."
        "6905424996341286006395665916153473915813690555841673380154844301477680712340347812523461533597103025777030"
        "9947013829055857303257277202823963445807352532510207031472279918726834676779179200017795396879807764049489"
        "37794925892393437060243939589390887431511762778828199428096e+271"},
    {0x1.fffffffffffffp+901, chars_format::scientific, 271,
        "3."
        "3810849992682572012791331832306947831627381111683346760309688602955361424680695625046923067194206051554061"
        "9894027658111714606514554405647926891614705065020414062944559837453669353558358400035590793759615528098978"
        "75589851784786874120487879178781774863023525557656398856192e+271"},
    {0x1.fffffffffffffp+902, chars_format::scientific, 271,
        "6."
        "7621699985365144025582663664613895663254762223366693520619377205910722849361391250093846134388412103108123"
        "9788055316223429213029108811295853783229410130040828125889119674907338707116716800071181587519231056197957"
        "51179703569573748240975758357563549726047051115312797712384e+271"},
    {0x1.fffffffffffffp+903, chars_format::scientific, 272,
        "1."
        "3524339997073028805116532732922779132650952444673338704123875441182144569872278250018769226877682420621624"
        "7957611063244685842605821762259170756645882026008165625177823934981467741423343360014236317503846211239591"
        "502359407139147496481951516715127099452094102230625595424768e+272"},
    {0x1.fffffffffffffp+904, chars_format::scientific, 272,
        "2."
        "7048679994146057610233065465845558265301904889346677408247750882364289139744556500037538453755364841243249"
        "5915222126489371685211643524518341513291764052016331250355647869962935482846686720028472635007692422479183"
        "004718814278294992963903033430254198904188204461251190849536e+272"},
    {0x1.fffffffffffffp+905, chars_format::scientific, 272,
        "5."
        "4097359988292115220466130931691116530603809778693354816495501764728578279489113000075076907510729682486499"
        "1830444252978743370423287049036683026583528104032662500711295739925870965693373440056945270015384844958366"
        "009437628556589985927806066860508397808376408922502381699072e+272"},
    {0x1.fffffffffffffp+906, chars_format::scientific, 273,
        "1."
        "0819471997658423044093226186338223306120761955738670963299100352945715655897822600015015381502145936497299"
        "8366088850595748674084657409807336605316705620806532500142259147985174193138674688011389054003076968991673"
        "2018875257113179971855612133721016795616752817845004763398144e+273"},
    {0x1.fffffffffffffp+907, chars_format::scientific, 273,
        "2."
        "1638943995316846088186452372676446612241523911477341926598200705891431311795645200030030763004291872994599"
        "6732177701191497348169314819614673210633411241613065000284518295970348386277349376022778108006153937983346"
        "4037750514226359943711224267442033591233505635690009526796288e+273"},
    {0x1.fffffffffffffp+908, chars_format::scientific, 273,
        "4."
        "3277887990633692176372904745352893224483047822954683853196401411782862623591290400060061526008583745989199"
        "3464355402382994696338629639229346421266822483226130000569036591940696772554698752045556216012307875966692"
        "8075501028452719887422448534884067182467011271380019053592576e+273"},
    {0x1.fffffffffffffp+909, chars_format::scientific, 273,
        "8."
        "6555775981267384352745809490705786448966095645909367706392802823565725247182580800120123052017167491978398"
        "6928710804765989392677259278458692842533644966452260001138073183881393545109397504091112432024615751933385"
        "6151002056905439774844897069768134364934022542760038107185152e+273"},
    {0x1.fffffffffffffp+910, chars_format::scientific, 274,
        "1."
        "7311155196253476870549161898141157289793219129181873541278560564713145049436516160024024610403433498395679"
        "7385742160953197878535451855691738568506728993290452000227614636776278709021879500818222486404923150386677"
        "12302004113810879549689794139536268729868045085520076214370304e+274"},
    {0x1.fffffffffffffp+911, chars_format::scientific, 274,
        "3."
        "4622310392506953741098323796282314579586438258363747082557121129426290098873032320048049220806866996791359"
        "4771484321906395757070903711383477137013457986580904000455229273552557418043759001636444972809846300773354"
        "24604008227621759099379588279072537459736090171040152428740608e+274"},
    {0x1.fffffffffffffp+912, chars_format::scientific, 274,
        "6."
        "9244620785013907482196647592564629159172876516727494165114242258852580197746064640096098441613733993582718"
        "9542968643812791514141807422766954274026915973161808000910458547105114836087518003272889945619692601546708"
        "49208016455243518198759176558145074919472180342080304857481216e+274"},
    {0x1.fffffffffffffp+913, chars_format::scientific, 275,
        "1."
        "3848924157002781496439329518512925831834575303345498833022848451770516039549212928019219688322746798716543"
        "7908593728762558302828361484553390854805383194632361600182091709421022967217503600654577989123938520309341"
        "698416032910487036397518353116290149838944360684160609714962432e+275"},
    {0x1.fffffffffffffp+914, chars_format::scientific, 275,
        "2."
        "7697848314005562992878659037025851663669150606690997666045696903541032079098425856038439376645493597433087"
        "5817187457525116605656722969106781709610766389264723200364183418842045934435007201309155978247877040618683"
        "396832065820974072795036706232580299677888721368321219429924864e+275"},
    {0x1.fffffffffffffp+915, chars_format::scientific, 275,
        "5."
        "5395696628011125985757318074051703327338301213381995332091393807082064158196851712076878753290987194866175"
        "1634374915050233211313445938213563419221532778529446400728366837684091868870014402618311956495754081237366"
        "793664131641948145590073412465160599355777442736642438859849728e+275"},
    {0x1.fffffffffffffp+916, chars_format::scientific, 276,
        "1."
        "1079139325602225197151463614810340665467660242676399066418278761416412831639370342415375750658197438973235"
        "0326874983010046642262689187642712683844306555705889280145673367536818373774002880523662391299150816247473"
        "3587328263283896291180146824930321198711554885473284877719699456e+276"},
    {0x1.fffffffffffffp+917, chars_format::scientific, 276,
        "2."
        "2158278651204450394302927229620681330935320485352798132836557522832825663278740684830751501316394877946470"
        "0653749966020093284525378375285425367688613111411778560291346735073636747548005761047324782598301632494946"
        "7174656526567792582360293649860642397423109770946569755439398912e+276"},
    {0x1.fffffffffffffp+918, chars_format::scientific, 276,
        "4."
        "4316557302408900788605854459241362661870640970705596265673115045665651326557481369661503002632789755892940"
        "1307499932040186569050756750570850735377226222823557120582693470147273495096011522094649565196603264989893"
        "4349313053135585164720587299721284794846219541893139510878797824e+276"},
    {0x1.fffffffffffffp+919, chars_format::scientific, 276,
        "8."
        "8633114604817801577211708918482725323741281941411192531346230091331302653114962739323006005265579511785880"
        "2614999864080373138101513501141701470754452445647114241165386940294546990192023044189299130393206529979786"
        "8698626106271170329441174599442569589692439083786279021757595648e+276"},
    {0x1.fffffffffffffp+920, chars_format::scientific, 277,
        "1."
        "7726622920963560315442341783696545064748256388282238506269246018266260530622992547864601201053115902357176"
        "0522999972816074627620302700228340294150890489129422848233077388058909398038404608837859826078641305995957"
        "37397252212542340658882349198885139179384878167572558043515191296e+277"},
    {0x1.fffffffffffffp+921, chars_format::scientific, 277,
        "3."
        "5453245841927120630884683567393090129496512776564477012538492036532521061245985095729202402106231804714352"
        "1045999945632149255240605400456680588301780978258845696466154776117818796076809217675719652157282611991914"
        "74794504425084681317764698397770278358769756335145116087030382592e+277"},
    {0x1.fffffffffffffp+922, chars_format::scientific, 277,
        "7."
        "0906491683854241261769367134786180258993025553128954025076984073065042122491970191458404804212463609428704"
        "2091999891264298510481210800913361176603561956517691392932309552235637592153618435351439304314565223983829"
        "49589008850169362635529396795540556717539512670290232174060765184e+277"},
    {0x1.fffffffffffffp+923, chars_format::scientific, 278,
        "1."
        "4181298336770848252353873426957236051798605110625790805015396814613008424498394038291680960842492721885740"
        "8418399978252859702096242160182672235320712391303538278586461910447127518430723687070287860862913044796765"
        "899178017700338725271058793591081113435079025340580464348121530368e+278"},
    {0x1.fffffffffffffp+924, chars_format::scientific, 278,
        "2."
        "8362596673541696504707746853914472103597210221251581610030793629226016848996788076583361921684985443771481"
        "6836799956505719404192484320365344470641424782607076557172923820894255036861447374140575721725826089593531"
        "798356035400677450542117587182162226870158050681160928696243060736e+278"},
    {0x1.fffffffffffffp+925, chars_format::scientific, 278,
        "5."
        "6725193347083393009415493707828944207194420442503163220061587258452033697993576153166723843369970887542963"
        "3673599913011438808384968640730688941282849565214153114345847641788510073722894748281151443451652179187063"
        "596712070801354901084235174364324453740316101362321857392486121472e+278"},
    {0x1.fffffffffffffp+926, chars_format::scientific, 279,
        "1."
        "1345038669416678601883098741565788841438884088500632644012317451690406739598715230633344768673994177508592"
        "6734719982602287761676993728146137788256569913042830622869169528357702014744578949656230288690330435837412"
        "7193424141602709802168470348728648907480632202724643714784972242944e+279"},
    {0x1.fffffffffffffp+927, chars_format::scientific, 279,
        "2."
        "2690077338833357203766197483131577682877768177001265288024634903380813479197430461266689537347988355017185"
        "3469439965204575523353987456292275576513139826085661245738339056715404029489157899312460577380660871674825"
        "4386848283205419604336940697457297814961264405449287429569944485888e+279"},
    {0x1.fffffffffffffp+928, chars_format::scientific, 279,
        "4."
        "5380154677666714407532394966263155365755536354002530576049269806761626958394860922533379074695976710034370"
        "6938879930409151046707974912584551153026279652171322491476678113430808058978315798624921154761321743349650"
        "8773696566410839208673881394914595629922528810898574859139888971776e+279"},
    {0x1.fffffffffffffp+929, chars_format::scientific, 279,
        "9."
        "0760309355333428815064789932526310731511072708005061152098539613523253916789721845066758149391953420068741"
        "3877759860818302093415949825169102306052559304342644982953356226861616117956631597249842309522643486699301"
        "7547393132821678417347762789829191259845057621797149718279777943552e+279"},
    {0x1.fffffffffffffp+930, chars_format::scientific, 280,
        "1."
        "8152061871066685763012957986505262146302214541601012230419707922704650783357944369013351629878390684013748"
        "2775551972163660418683189965033820461210511860868528996590671245372323223591326319449968461904528697339860"
        "35094786265643356834695525579658382519690115243594299436559555887104e+280"},
    {0x1.fffffffffffffp+931, chars_format::scientific, 280,
        "3."
        "6304123742133371526025915973010524292604429083202024460839415845409301566715888738026703259756781368027496"
        "5551103944327320837366379930067640922421023721737057993181342490744646447182652638899936923809057394679720"
        "70189572531286713669391051159316765039380230487188598873119111774208e+280"},
    {0x1.fffffffffffffp+932, chars_format::scientific, 280,
        "7."
        "2608247484266743052051831946021048585208858166404048921678831690818603133431777476053406519513562736054993"
        "1102207888654641674732759860135281844842047443474115986362684981489292894365305277799873847618114789359441"
        "40379145062573427338782102318633530078760460974377197746238223548416e+280"},
    {0x1.fffffffffffffp+933, chars_format::scientific, 281,
        "1."
        "4521649496853348610410366389204209717041771633280809784335766338163720626686355495210681303902712547210998"
        "6220441577730928334946551972027056368968409488694823197272536996297858578873061055559974769523622957871888"
        "280758290125146854677564204637267060157520921948754395492476447096832e+281"},
    {0x1.fffffffffffffp+934, chars_format::scientific, 281,
        "2."
        "9043298993706697220820732778408419434083543266561619568671532676327441253372710990421362607805425094421997"
        "2440883155461856669893103944054112737936818977389646394545073992595717157746122111119949539047245915743776"
        "561516580250293709355128409274534120315041843897508790984952894193664e+281"},
    {0x1.fffffffffffffp+935, chars_format::scientific, 281,
        "5."
        "8086597987413394441641465556816838868167086533123239137343065352654882506745421980842725215610850188843994"
        "4881766310923713339786207888108225475873637954779292789090147985191434315492244222239899078094491831487553"
        "123033160500587418710256818549068240630083687795017581969905788387328e+281"},
    {0x1.fffffffffffffp+936, chars_format::scientific, 282,
        "1."
        "1617319597482678888328293111363367773633417306624647827468613070530976501349084396168545043122170037768798"
        "8976353262184742667957241577621645095174727590955858557818029597038286863098448844447979815618898366297510"
        "6246066321001174837420513637098136481260167375590035163939811576774656e+282"},
    {0x1.fffffffffffffp+937, chars_format::scientific, 282,
        "2."
        "3234639194965357776656586222726735547266834613249295654937226141061953002698168792337090086244340075537597"
        "7952706524369485335914483155243290190349455181911717115636059194076573726196897688895959631237796732595021"
        "2492132642002349674841027274196272962520334751180070327879623153549312e+282"},
    {0x1.fffffffffffffp+938, chars_format::scientific, 282,
        "4."
        "6469278389930715553313172445453471094533669226498591309874452282123906005396337584674180172488680151075195"
        "5905413048738970671828966310486580380698910363823434231272118388153147452393795377791919262475593465190042"
        "4984265284004699349682054548392545925040669502360140655759246307098624e+282"},
    {0x1.fffffffffffffp+939, chars_format::scientific, 282,
        "9."
        "2938556779861431106626344890906942189067338452997182619748904564247812010792675169348360344977360302150391"
        "1810826097477941343657932620973160761397820727646868462544236776306294904787590755583838524951186930380084"
        "9968530568009398699364109096785091850081339004720281311518492614197248e+282"},
    {0x1.fffffffffffffp+940, chars_format::scientific, 283,
        "1."
        "8587711355972286221325268978181388437813467690599436523949780912849562402158535033869672068995472060430078"
        "2362165219495588268731586524194632152279564145529373692508847355261258980957518151116767704990237386076016"
        "99937061136018797398728218193570183700162678009440562623036985228394496e+283"},
    {0x1.fffffffffffffp+941, chars_format::scientific, 283,
        "3."
        "7175422711944572442650537956362776875626935381198873047899561825699124804317070067739344137990944120860156"
        "4724330438991176537463173048389264304559128291058747385017694710522517961915036302233535409980474772152033"
        "99874122272037594797456436387140367400325356018881125246073970456788992e+283"},
    {0x1.fffffffffffffp+942, chars_format::scientific, 283,
        "7."
        "4350845423889144885301075912725553751253870762397746095799123651398249608634140135478688275981888241720312"
        "9448660877982353074926346096778528609118256582117494770035389421045035923830072604467070819960949544304067"
        "99748244544075189594912872774280734800650712037762250492147940913577984e+283"},
    {0x1.fffffffffffffp+943, chars_format::scientific, 284,
        "1."
        "4870169084777828977060215182545110750250774152479549219159824730279649921726828027095737655196377648344062"
        "5889732175596470614985269219355705721823651316423498954007077884209007184766014520893414163992189908860813"
        "599496489088150379189825745548561469601301424075524500984295881827155968e+284"},
    {0x1.fffffffffffffp+944, chars_format::scientific, 284,
        "2."
        "9740338169555657954120430365090221500501548304959098438319649460559299843453656054191475310392755296688125"
        "1779464351192941229970538438711411443647302632846997908014155768418014369532029041786828327984379817721627"
        "198992978176300758379651491097122939202602848151049001968591763654311936e+284"},
    {0x1.fffffffffffffp+945, chars_format::scientific, 284,
        "5."
        "9480676339111315908240860730180443001003096609918196876639298921118599686907312108382950620785510593376250"
        "3558928702385882459941076877422822887294605265693995816028311536836028739064058083573656655968759635443254"
        "397985956352601516759302982194245878405205696302098003937183527308623872e+284"},
    {0x1.fffffffffffffp+946, chars_format::scientific, 285,
        "1."
        "1896135267822263181648172146036088600200619321983639375327859784223719937381462421676590124157102118675250"
        "0711785740477176491988215375484564577458921053138799163205662307367205747812811616714731331193751927088650"
        "8795971912705203033518605964388491756810411392604196007874367054617247744e+285"},
    {0x1.fffffffffffffp+947, chars_format::scientific, 285,
        "2."
        "3792270535644526363296344292072177200401238643967278750655719568447439874762924843353180248314204237350500"
        "1423571480954352983976430750969129154917842106277598326411324614734411495625623233429462662387503854177301"
        "7591943825410406067037211928776983513620822785208392015748734109234495488e+285"},
    {0x1.fffffffffffffp+948, chars_format::scientific, 285,
        "4."
        "7584541071289052726592688584144354400802477287934557501311439136894879749525849686706360496628408474701000"
        "2847142961908705967952861501938258309835684212555196652822649229468822991251246466858925324775007708354603"
        "5183887650820812134074423857553967027241645570416784031497468218468990976e+285"},
    {0x1.fffffffffffffp+949, chars_format::scientific, 285,
        "9."
        "5169082142578105453185377168288708801604954575869115002622878273789759499051699373412720993256816949402000"
        "5694285923817411935905723003876516619671368425110393305645298458937645982502492933717850649550015416709207"
        "0367775301641624268148847715107934054483291140833568062994936436937981952e+285"},
    {0x1.fffffffffffffp+950, chars_format::scientific, 286,
        "1."
        "9033816428515621090637075433657741760320990915173823000524575654757951899810339874682544198651363389880400"
        "1138857184763482387181144600775303323934273685022078661129059691787529196500498586743570129910003083341841"
        "40735550603283248536297695430215868108966582281667136125989872873875963904e+286"},
    {0x1.fffffffffffffp+951, chars_format::scientific, 286,
        "3."
        "8067632857031242181274150867315483520641981830347646001049151309515903799620679749365088397302726779760800"
        "2277714369526964774362289201550606647868547370044157322258119383575058393000997173487140259820006166683682"
        "81471101206566497072595390860431736217933164563334272251979745747751927808e+286"},
    {0x1.fffffffffffffp+952, chars_format::scientific, 286,
        "7."
        "6135265714062484362548301734630967041283963660695292002098302619031807599241359498730176794605453559521600"
        "4555428739053929548724578403101213295737094740088314644516238767150116786001994346974280519640012333367365"
        "62942202413132994145190781720863472435866329126668544503959491495503855616e+286"},
    {0x1.fffffffffffffp+953, chars_format::scientific, 287,
        "1."
        "5227053142812496872509660346926193408256792732139058400419660523806361519848271899746035358921090711904320"
        "0911085747810785909744915680620242659147418948017662928903247753430023357200398869394856103928002466673473"
        "125884404826265988290381563441726944871732658253337089007918982991007711232e+287"},
    {0x1.fffffffffffffp+954, chars_format::scientific, 287,
        "3."
        "0454106285624993745019320693852386816513585464278116800839321047612723039696543799492070717842181423808640"
        "1822171495621571819489831361240485318294837896035325857806495506860046714400797738789712207856004933346946"
        "251768809652531976580763126883453889743465316506674178015837965982015422464e+287"},
    {0x1.fffffffffffffp+955, chars_format::scientific, 287,
        "6."
        "0908212571249987490038641387704773633027170928556233601678642095225446079393087598984141435684362847617280"
        "3644342991243143638979662722480970636589675792070651715612991013720093428801595477579424415712009866693892"
        "503537619305063953161526253766907779486930633013348356031675931964030844928e+287"},
    {0x1.fffffffffffffp+956, chars_format::scientific, 288,
        "1."
        "2181642514249997498007728277540954726605434185711246720335728419045089215878617519796828287136872569523456"
        "0728868598248628727795932544496194127317935158414130343122598202744018685760319095515884883142401973338778"
        "5007075238610127906323052507533815558973861266026696712063351863928061689856e+288"},
    {0x1.fffffffffffffp+957, chars_format::scientific, 288,
        "2."
        "4363285028499994996015456555081909453210868371422493440671456838090178431757235039593656574273745139046912"
        "1457737196497257455591865088992388254635870316828260686245196405488037371520638191031769766284803946677557"
        "0014150477220255812646105015067631117947722532053393424126703727856123379712e+288"},
    {0x1.fffffffffffffp+958, chars_format::scientific, 288,
        "4."
        "8726570056999989992030913110163818906421736742844986881342913676180356863514470079187313148547490278093824"
        "2915474392994514911183730177984776509271740633656521372490392810976074743041276382063539532569607893355114"
        "0028300954440511625292210030135262235895445064106786848253407455712246759424e+288"},
    {0x1.fffffffffffffp+959, chars_format::scientific, 288,
        "9."
        "7453140113999979984061826220327637812843473485689973762685827352360713727028940158374626297094980556187648"
        "5830948785989029822367460355969553018543481267313042744980785621952149486082552764127079065139215786710228"
        "0056601908881023250584420060270524471790890128213573696506814911424493518848e+288"},
    {0x1.fffffffffffffp+960, chars_format::scientific, 289,
        "1."
        "9490628022799995996812365244065527562568694697137994752537165470472142745405788031674925259418996111237529"
        "7166189757197805964473492071193910603708696253462608548996157124390429897216510552825415813027843157342045"
        "60113203817762046501168840120541048943581780256427147393013629822848987037696e+289"},
    {0x1.fffffffffffffp+961, chars_format::scientific, 289,
        "3."
        "8981256045599991993624730488131055125137389394275989505074330940944285490811576063349850518837992222475059"
        "4332379514395611928946984142387821207417392506925217097992314248780859794433021105650831626055686314684091"
        "20226407635524093002337680241082097887163560512854294786027259645697974075392e+289"},
    {0x1.fffffffffffffp+962, chars_format::scientific, 289,
        "7."
        "7962512091199983987249460976262110250274778788551979010148661881888570981623152126699701037675984444950118"
        "8664759028791223857893968284775642414834785013850434195984628497561719588866042211301663252111372629368182"
        "40452815271048186004675360482164195774327121025708589572054519291395948150784e+289"},
    {0x1.fffffffffffffp+963, chars_format::scientific, 290,
        "1."
        "5592502418239996797449892195252422050054955757710395802029732376377714196324630425339940207535196888990023"
        "7732951805758244771578793656955128482966957002770086839196925699512343917773208442260332650422274525873636"
        "480905630542096372009350720964328391548654242051417179144109038582791896301568e+290"},
    {0x1.fffffffffffffp+964, chars_format::scientific, 290,
        "3."
        "1185004836479993594899784390504844100109911515420791604059464752755428392649260850679880415070393777980047"
        "5465903611516489543157587313910256965933914005540173678393851399024687835546416884520665300844549051747272"
        "961811261084192744018701441928656783097308484102834358288218077165583792603136e+290"},
    {0x1.fffffffffffffp+965, chars_format::scientific, 290,
        "6."
        "2370009672959987189799568781009688200219823030841583208118929505510856785298521701359760830140787555960095"
        "0931807223032979086315174627820513931867828011080347356787702798049375671092833769041330601689098103494545"
        "923622522168385488037402883857313566194616968205668716576436154331167585206272e+290"},
    {0x1.fffffffffffffp+966, chars_format::scientific, 291,
        "1."
        "2474001934591997437959913756201937640043964606168316641623785901102171357059704340271952166028157511192019"
        "0186361444606595817263034925564102786373565602216069471357540559609875134218566753808266120337819620698909"
        "1847245044336770976074805767714627132389233936411337433152872308662335170412544e+291"},
    {0x1.fffffffffffffp+967, chars_format::scientific, 291,
        "2."
        "4948003869183994875919827512403875280087929212336633283247571802204342714119408680543904332056315022384038"
        "0372722889213191634526069851128205572747131204432138942715081119219750268437133507616532240675639241397818"
        "3694490088673541952149611535429254264778467872822674866305744617324670340825088e+291"},
    {0x1.fffffffffffffp+968, chars_format::scientific, 291,
        "4."
        "9896007738367989751839655024807750560175858424673266566495143604408685428238817361087808664112630044768076"
        "0745445778426383269052139702256411145494262408864277885430162238439500536874267015233064481351278482795636"
        "7388980177347083904299223070858508529556935745645349732611489234649340681650176e+291"},
    {0x1.fffffffffffffp+969, chars_format::scientific, 291,
        "9."
        "9792015476735979503679310049615501120351716849346533132990287208817370856477634722175617328225260089536152"
        "1490891556852766538104279404512822290988524817728555770860324476879001073748534030466128962702556965591273"
        "4777960354694167808598446141717017059113871491290699465222978469298681363300352e+291"},
    {0x1.fffffffffffffp+970, chars_format::scientific, 292,
        "1."
        "9958403095347195900735862009923100224070343369869306626598057441763474171295526944435123465645052017907230"
        "4298178311370553307620855880902564458197704963545711154172064895375800214749706806093225792540511393118254"
        "69555920709388335617196892283434034118227742982581398930445956938597362726600704e+292"},
    {0x1.fffffffffffffp+971, chars_format::scientific, 292,
        "3."
        "9916806190694391801471724019846200448140686739738613253196114883526948342591053888870246931290104035814460"
        "8596356622741106615241711761805128916395409927091422308344129790751600429499413612186451585081022786236509"
        "39111841418776671234393784566868068236455485965162797860891913877194725453201408e+292"},
    {0x1.fffffffffffffp+972, chars_format::scientific, 292,
        "7."
        "9833612381388783602943448039692400896281373479477226506392229767053896685182107777740493862580208071628921"
        "7192713245482213230483423523610257832790819854182844616688259581503200858998827224372903170162045572473018"
        "78223682837553342468787569133736136472910971930325595721783827754389450906402816e+292"},
    {0x1.fffffffffffffp+973, chars_format::scientific, 293,
        "1."
        "5966722476277756720588689607938480179256274695895445301278445953410779337036421555548098772516041614325784"
        "3438542649096442646096684704722051566558163970836568923337651916300640171799765444874580634032409114494603"
        "756447365675106684937575138267472272945821943860651191443567655508778901812805632e+293"},
    {0x1.fffffffffffffp+974, chars_format::scientific, 293,
        "3."
        "1933444952555513441177379215876960358512549391790890602556891906821558674072843111096197545032083228651568"
        "6877085298192885292193369409444103133116327941673137846675303832601280343599530889749161268064818228989207"
        "512894731350213369875150276534944545891643887721302382887135311017557803625611264e+293"},
    {0x1.fffffffffffffp+975, chars_format::scientific, 293,
        "6."
        "3866889905111026882354758431753920717025098783581781205113783813643117348145686222192395090064166457303137"
        "3754170596385770584386738818888206266232655883346275693350607665202560687199061779498322536129636457978415"
        "025789462700426739750300553069889091783287775442604765774270622035115607251222528e+293"},
    {0x1.fffffffffffffp+976, chars_format::scientific, 294,
        "1."
        "2773377981022205376470951686350784143405019756716356241022756762728623469629137244438479018012833291460627"
        "4750834119277154116877347763777641253246531176669255138670121533040512137439812355899664507225927291595683"
        "0051578925400853479500601106139778183566575550885209531548541244070231214502445056e+294"},
    {0x1.fffffffffffffp+977, chars_format::scientific, 294,
        "2."
        "5546755962044410752941903372701568286810039513432712482045513525457246939258274488876958036025666582921254"
        "9501668238554308233754695527555282506493062353338510277340243066081024274879624711799329014451854583191366"
        "0103157850801706959001202212279556367133151101770419063097082488140462429004890112e+294"},
    {0x1.fffffffffffffp+978, chars_format::scientific, 294,
        "5."
        "1093511924088821505883806745403136573620079026865424964091027050914493878516548977753916072051333165842509"
        "9003336477108616467509391055110565012986124706677020554680486132162048549759249423598658028903709166382732"
        "0206315701603413918002404424559112734266302203540838126194164976280924858009780224e+294"},
    {0x1.fffffffffffffp+979, chars_format::scientific, 295,
        "1."
        "0218702384817764301176761349080627314724015805373084992818205410182898775703309795550783214410266633168501"
        "9800667295421723293501878211022113002597224941335404110936097226432409709951849884719731605780741833276546"
        "40412631403206827836004808849118225468532604407081676252388329952561849716019560448e+295"},
    {0x1.fffffffffffffp+980, chars_format::scientific, 295,
        "2."
        "0437404769635528602353522698161254629448031610746169985636410820365797551406619591101566428820533266337003"
        "9601334590843446587003756422044226005194449882670808221872194452864819419903699769439463211561483666553092"
        "80825262806413655672009617698236450937065208814163352504776659905123699432039120896e+295"},
    {0x1.fffffffffffffp+981, chars_format::scientific, 295,
        "4."
        "0874809539271057204707045396322509258896063221492339971272821640731595102813239182203132857641066532674007"
        "9202669181686893174007512844088452010388899765341616443744388905729638839807399538878926423122967333106185"
        "61650525612827311344019235396472901874130417628326705009553319810247398864078241792e+295"},
    {0x1.fffffffffffffp+982, chars_format::scientific, 295,
        "8."
        "1749619078542114409414090792645018517792126442984679942545643281463190205626478364406265715282133065348015"
        "8405338363373786348015025688176904020777799530683232887488777811459277679614799077757852846245934666212371"
        "23301051225654622688038470792945803748260835256653410019106639620494797728156483584e+295"},
    {0x1.fffffffffffffp+983, chars_format::scientific, 296,
        "1."
        "6349923815708422881882818158529003703558425288596935988509128656292638041125295672881253143056426613069603"
        "1681067672674757269603005137635380804155559906136646577497755562291855535922959815551570569249186933242474"
        "246602102451309245376076941585891607496521670513306820038213279240989595456312967168e+296"},
    {0x1.fffffffffffffp+984, chars_format::scientific, 296,
        "3."
        "2699847631416845763765636317058007407116850577193871977018257312585276082250591345762506286112853226139206"
        "3362135345349514539206010275270761608311119812273293154995511124583711071845919631103141138498373866484948"
        "493204204902618490752153883171783214993043341026613640076426558481979190912625934336e+296"},
    {0x1.fffffffffffffp+985, chars_format::scientific, 296,
        "6."
        "5399695262833691527531272634116014814233701154387743954036514625170552164501182691525012572225706452278412"
        "6724270690699029078412020550541523216622239624546586309991022249167422143691839262206282276996747732969896"
        "986408409805236981504307766343566429986086682053227280152853116963958381825251868672e+296"},
    {0x1.fffffffffffffp+986, chars_format::scientific, 297,
        "1."
        "3079939052566738305506254526823202962846740230877548790807302925034110432900236538305002514445141290455682"
        "5344854138139805815682404110108304643324447924909317261998204449833484428738367852441256455399349546593979"
        "3972816819610473963008615532687132859972173364106454560305706233927916763650503737344e+297"},
    {0x1.fffffffffffffp+987, chars_format::scientific, 297,
        "2."
        "6159878105133476611012509053646405925693480461755097581614605850068220865800473076610005028890282580911365"
        "0689708276279611631364808220216609286648895849818634523996408899666968857476735704882512910798699093187958"
        "7945633639220947926017231065374265719944346728212909120611412467855833527301007474688e+297"},
    {0x1.fffffffffffffp+988, chars_format::scientific, 297,
        "5."
        "2319756210266953222025018107292811851386960923510195163229211700136441731600946153220010057780565161822730"
        "1379416552559223262729616440433218573297791699637269047992817799333937714953471409765025821597398186375917"
        "5891267278441895852034462130748531439888693456425818241222824935711667054602014949376e+297"},
    {0x1.fffffffffffffp+989, chars_format::scientific, 298,
        "1."
        "0463951242053390644405003621458562370277392184702039032645842340027288346320189230644002011556113032364546"
        "0275883310511844652545923288086643714659558339927453809598563559866787542990694281953005164319479637275183"
        "51782534556883791704068924261497062879777386912851636482445649871423334109204029898752e+298"},
    {0x1.fffffffffffffp+990, chars_format::scientific, 298,
        "2."
        "0927902484106781288810007242917124740554784369404078065291684680054576692640378461288004023112226064729092"
        "0551766621023689305091846576173287429319116679854907619197127119733575085981388563906010328638959274550367"
        "03565069113767583408137848522994125759554773825703272964891299742846668218408059797504e+298"},
    {0x1.fffffffffffffp+991, chars_format::scientific, 298,
        "4."
        "1855804968213562577620014485834249481109568738808156130583369360109153385280756922576008046224452129458184"
        "1103533242047378610183693152346574858638233359709815238394254239467150171962777127812020657277918549100734"
        "07130138227535166816275697045988251519109547651406545929782599485693336436816119595008e+298"},
    {0x1.fffffffffffffp+992, chars_format::scientific, 298,
        "8."
        "3711609936427125155240028971668498962219137477616312261166738720218306770561513845152016092448904258916368"
        "2207066484094757220367386304693149717276466719419630476788508478934300343925554255624041314555837098201468"
        "14260276455070333632551394091976503038219095302813091859565198971386672873632239190016e+298"},
    {0x1.fffffffffffffp+993, chars_format::scientific, 299,
        "1."
        "6742321987285425031048005794333699792443827495523262452233347744043661354112302769030403218489780851783273"
        "6441413296818951444073477260938629943455293343883926095357701695786860068785110851124808262911167419640293"
        "628520552910140667265102788183953006076438190605626183719130397942773345747264478380032e+299"},
    {0x1.fffffffffffffp+994, chars_format::scientific, 299,
        "3."
        "3484643974570850062096011588667399584887654991046524904466695488087322708224605538060806436979561703566547"
        "2882826593637902888146954521877259886910586687767852190715403391573720137570221702249616525822334839280587"
        "257041105820281334530205576367906012152876381211252367438260795885546691494528956760064e+299"},
    {0x1.fffffffffffffp+995, chars_format::scientific, 299,
        "6."
        "6969287949141700124192023177334799169775309982093049808933390976174645416449211076121612873959123407133094"
        "5765653187275805776293909043754519773821173375535704381430806783147440275140443404499233051644669678561174"
        "514082211640562669060411152735812024305752762422504734876521591771093382989057913520128e+299"},
    {0x1.fffffffffffffp+996, chars_format::scientific, 300,
        "1."
        "3393857589828340024838404635466959833955061996418609961786678195234929083289842215224322574791824681426618"
        "9153130637455161155258781808750903954764234675107140876286161356629488055028088680899846610328933935712234"
        "9028164423281125338120822305471624048611505524845009469753043183542186765978115827040256e+300"},
    {0x1.fffffffffffffp+997, chars_format::scientific, 300,
        "2."
        "6787715179656680049676809270933919667910123992837219923573356390469858166579684430448645149583649362853237"
        "8306261274910322310517563617501807909528469350214281752572322713258976110056177361799693220657867871424469"
        "8056328846562250676241644610943248097223011049690018939506086367084373531956231654080512e+300"},
    {0x1.fffffffffffffp+998, chars_format::scientific, 300,
        "5."
        "3575430359313360099353618541867839335820247985674439847146712780939716333159368860897290299167298725706475"
        "6612522549820644621035127235003615819056938700428563505144645426517952220112354723599386441315735742848939"
        "6112657693124501352483289221886496194446022099380037879012172734168747063912463308161024e+300"},
    {0x1.fffffffffffffp+999, chars_format::scientific, 301,
        "1."
        "0715086071862672019870723708373567867164049597134887969429342556187943266631873772179458059833459745141295"
        "1322504509964128924207025447000723163811387740085712701028929085303590444022470944719877288263147148569787"
        "92225315386249002704966578443772992388892044198760075758024345468337494127824926616322048e+301"},
    {0x1.fffffffffffffp+1000, chars_format::scientific, 301,
        "2."
        "1430172143725344039741447416747135734328099194269775938858685112375886533263747544358916119666919490282590"
        "2645009019928257848414050894001446327622775480171425402057858170607180888044941889439754576526294297139575"
        "84450630772498005409933156887545984777784088397520151516048690936674988255649853232644096e+301"},
    {0x1.fffffffffffffp+1001, chars_format::scientific, 301,
        "4."
        "2860344287450688079482894833494271468656198388539551877717370224751773066527495088717832239333838980565180"
        "5290018039856515696828101788002892655245550960342850804115716341214361776089883778879509153052588594279151"
        "68901261544996010819866313775091969555568176795040303032097381873349976511299706465288192e+301"},
    {0x1.fffffffffffffp+1002, chars_format::scientific, 301,
        "8."
        "5720688574901376158965789666988542937312396777079103755434740449503546133054990177435664478667677961130361"
        "0580036079713031393656203576005785310491101920685701608231432682428723552179767557759018306105177188558303"
        "37802523089992021639732627550183939111136353590080606064194763746699953022599412930576384e+301"},
    {0x1.fffffffffffffp+1003, chars_format::scientific, 302,
        "1."
        "7144137714980275231793157933397708587462479355415820751086948089900709226610998035487132895733535592226072"
        "2116007215942606278731240715201157062098220384137140321646286536485744710435953511551803661221035437711660"
        "675605046179984043279465255100367878222272707180161212128389527493399906045198825861152768e+302"},
    {0x1.fffffffffffffp+1004, chars_format::scientific, 302,
        "3."
        "4288275429960550463586315866795417174924958710831641502173896179801418453221996070974265791467071184452144"
        "4232014431885212557462481430402314124196440768274280643292573072971489420871907023103607322442070875423321"
        "351210092359968086558930510200735756444545414360322424256779054986799812090397651722305536e+302"},
    {0x1.fffffffffffffp+1005, chars_format::scientific, 302,
        "6."
        "8576550859921100927172631733590834349849917421663283004347792359602836906443992141948531582934142368904288"
        "8464028863770425114924962860804628248392881536548561286585146145942978841743814046207214644884141750846642"
        "702420184719936173117861020401471512889090828720644848513558109973599624180795303444611072e+302"},
    {0x1.fffffffffffffp+1006, chars_format::scientific, 303,
        "1."
        "3715310171984220185434526346718166869969983484332656600869558471920567381288798428389706316586828473780857"
        "7692805772754085022984992572160925649678576307309712257317029229188595768348762809241442928976828350169328"
        "5404840369439872346235722040802943025778181657441289697027116219947199248361590606889222144e+303"},
    {0x1.fffffffffffffp+1007, chars_format::scientific, 303,
        "2."
        "7430620343968440370869052693436333739939966968665313201739116943841134762577596856779412633173656947561715"
        "5385611545508170045969985144321851299357152614619424514634058458377191536697525618482885857953656700338657"
        "0809680738879744692471444081605886051556363314882579394054232439894398496723181213778444288e+303"},
    {0x1.fffffffffffffp+1008, chars_format::scientific, 303,
        "5."
        "4861240687936880741738105386872667479879933937330626403478233887682269525155193713558825266347313895123431"
        "0771223091016340091939970288643702598714305229238849029268116916754383073395051236965771715907313400677314"
        "1619361477759489384942888163211772103112726629765158788108464879788796993446362427556888576e+303"},
    {0x1.fffffffffffffp+1009, chars_format::scientific, 304,
        "1."
        "0972248137587376148347621077374533495975986787466125280695646777536453905031038742711765053269462779024686"
        "2154244618203268018387994057728740519742861045847769805853623383350876614679010247393154343181462680135462"
        "83238722955518978769885776326423544206225453259530317576216929759577593986892724855113777152e+304"},
    {0x1.fffffffffffffp+1010, chars_format::scientific, 304,
        "2."
        "1944496275174752296695242154749066991951973574932250561391293555072907810062077485423530106538925558049372"
        "4308489236406536036775988115457481039485722091695539611707246766701753229358020494786308686362925360270925"
        "66477445911037957539771552652847088412450906519060635152433859519155187973785449710227554304e+304"},
    {0x1.fffffffffffffp+1011, chars_format::scientific, 304,
        "4."
        "3888992550349504593390484309498133983903947149864501122782587110145815620124154970847060213077851116098744"
        "8616978472813072073551976230914962078971444183391079223414493533403506458716040989572617372725850720541851"
        "32954891822075915079543105305694176824901813038121270304867719038310375947570899420455108608e+304"},
    {0x1.fffffffffffffp+1012, chars_format::scientific, 304,
        "8."
        "7777985100699009186780968618996267967807894299729002245565174220291631240248309941694120426155702232197489"
        "7233956945626144147103952461829924157942888366782158446828987066807012917432081979145234745451701441083702"
        "65909783644151830159086210611388353649803626076242540609735438076620751895141798840910217216e+304"},
    {0x1.fffffffffffffp+1013, chars_format::scientific, 305,
        "1."
        "7555597020139801837356193723799253593561578859945800449113034844058326248049661988338824085231140446439497"
        "9446791389125228829420790492365984831588577673356431689365797413361402583486416395829046949090340288216740"
        "531819567288303660318172421222776707299607252152485081219470876153241503790283597681820434432e+305"},
    {0x1.fffffffffffffp+1014, chars_format::scientific, 305,
        "3."
        "5111194040279603674712387447598507187123157719891600898226069688116652496099323976677648170462280892878995"
        "8893582778250457658841580984731969663177155346712863378731594826722805166972832791658093898180680576433481"
        "063639134576607320636344842445553414599214504304970162438941752306483007580567195363640868864e+305"},
    {0x1.fffffffffffffp+1015, chars_format::scientific, 305,
        "7."
        "0222388080559207349424774895197014374246315439783201796452139376233304992198647953355296340924561785757991"
        "7787165556500915317683161969463939326354310693425726757463189653445610333945665583316187796361361152866962"
        "127278269153214641272689684891106829198429008609940324877883504612966015161134390727281737728e+305"},
    {0x1.fffffffffffffp+1016, chars_format::scientific, 306,
        "1."
        "4044477616111841469884954979039402874849263087956640359290427875246660998439729590671059268184912357151598"
        "3557433111300183063536632393892787865270862138685145351492637930689122066789133116663237559272272230573392"
        "4254556538306429282545379369782213658396858017219880649755767009225932030322268781454563475456e+306"},
    {0x1.fffffffffffffp+1017, chars_format::scientific, 306,
        "2."
        "8088955232223682939769909958078805749698526175913280718580855750493321996879459181342118536369824714303196"
        "7114866222600366127073264787785575730541724277370290702985275861378244133578266233326475118544544461146784"
        "8509113076612858565090758739564427316793716034439761299511534018451864060644537562909126950912e+306"},
    {0x1.fffffffffffffp+1018, chars_format::scientific, 306,
        "5."
        "6177910464447365879539819916157611499397052351826561437161711500986643993758918362684237072739649428606393"
        "4229732445200732254146529575571151461083448554740581405970551722756488267156532466652950237089088922293569"
        "7018226153225717130181517479128854633587432068879522599023068036903728121289075125818253901824e+306"},
    {0x1.fffffffffffffp+1019, chars_format::scientific, 307,
        "1."
        "1235582092889473175907963983231522299879410470365312287432342300197328798751783672536847414547929885721278"
        "6845946489040146450829305915114230292216689710948116281194110344551297653431306493330590047417817784458713"
        "94036452306451434260363034958257709267174864137759045198046136073807456242578150251636507803648e+307"},
    {0x1.fffffffffffffp+1020, chars_format::scientific, 307,
        "2."
        "2471164185778946351815927966463044599758820940730624574864684600394657597503567345073694829095859771442557"
        "3691892978080292901658611830228460584433379421896232562388220689102595306862612986661180094835635568917427"
        "88072904612902868520726069916515418534349728275518090396092272147614912485156300503273015607296e+307"},
    {0x1.fffffffffffffp+1021, chars_format::scientific, 307,
        "4."
        "4942328371557892703631855932926089199517641881461249149729369200789315195007134690147389658191719542885114"
        "7383785956160585803317223660456921168866758843792465124776441378205190613725225973322360189671271137834855"
        "76145809225805737041452139833030837068699456551036180792184544295229824970312601006546031214592e+307"},
    {0x1.fffffffffffffp+1022, chars_format::scientific, 307,
        "8."
        "9884656743115785407263711865852178399035283762922498299458738401578630390014269380294779316383439085770229"
        "4767571912321171606634447320913842337733517687584930249552882756410381227450451946644720379342542275669711"
        "52291618451611474082904279666061674137398913102072361584369088590459649940625202013092062429184e+307"},
    {0x1.fffffffffffffp+1023, chars_format::scientific, 308,
        "1."
        "7976931348623157081452742373170435679807056752584499659891747680315726078002853876058955863276687817154045"
        "8953514382464234321326889464182768467546703537516986049910576551282076245490090389328944075868508455133942"
        "304583236903222948165808559332123348274797826204144723168738177180919299881250404026184124858368e+308"},
};

#endif // DOUBLE_SCIENTIFIC_PRECISION_TO_CHARS_TEST_CASES_4_HPP
