/*############################################################################
  # Copyright (C) 2020 Intel Corporation
  #
  # SPDX-License-Identifier: MIT
  ############################################################################*/

#include "./cpu_workstream.h"
#include "vpl/mfxvideo.h"

// stubs
mfxStatus MFXVideoVPP_Query(mfxSession session, mfxVideoParam *in, mfxVideoParam *out) {
    VPL_TRACE_FUNC;
    RET_IF_FALSE(session, MFX_ERR_INVALID_HANDLE);
    RET_IF_FALSE(out, MFX_ERR_NULL_PTR);
    RET_IF_FALSE(out->Protected == 0, MFX_ERR_UNSUPPORTED);
    RET_IF_FALSE(in == nullptr || in->Protected == 0, MFX_ERR_UNSUPPORTED);

    return CpuVPP::VPPQuery(in, out);
}

mfxStatus MFXVideoVPP_QueryIOSurf(mfxSession session,
                                  mfxVideoParam *par,
                                  mfxFrameAllocRequest request[2]) {
    VPL_TRACE_FUNC;
    RET_IF_FALSE(session, MFX_ERR_INVALID_HANDLE);
    RET_IF_FALSE(par, MFX_ERR_NULL_PTR);
    RET_IF_FALSE(request, MFX_ERR_NULL_PTR);

    return CpuVPP::VPPQueryIOSurf(par, request);
}

mfxStatus MFXVideoVPP_Init(mfxSession session, mfxVideoParam *par) {
    VPL_TRACE_FUNC;
    RET_IF_FALSE(session, MFX_ERR_INVALID_HANDLE);
    RET_IF_FALSE(par, MFX_ERR_NULL_PTR);
    CpuWorkstream *ws = reinterpret_cast<CpuWorkstream *>(session);
    if (ws->GetVPP())
        return MFX_ERR_UNDEFINED_BEHAVIOR;

    std::unique_ptr<CpuVPP> vpp(new CpuVPP);
    RET_IF_FALSE(vpp, MFX_ERR_MEMORY_ALLOC);
    vpp->SetSession(ws);
    mfxStatus sts = vpp->InitVPP(par);

    if (sts < MFX_ERR_NONE)
        return sts;
    else
        ws->SetVPP(vpp.release());

    return sts;
}

mfxStatus MFXVideoVPP_Close(mfxSession session) {
    VPL_TRACE_FUNC;
    RET_IF_FALSE(session, MFX_ERR_INVALID_HANDLE);

    CpuWorkstream *ws = reinterpret_cast<CpuWorkstream *>(session);
    ws->SetVPP(nullptr);

    return MFX_ERR_NONE;
}

mfxStatus MFXVideoVPP_GetVideoParam(mfxSession session, mfxVideoParam *par) {
    VPL_TRACE_FUNC;
    RET_IF_FALSE(session, MFX_ERR_INVALID_HANDLE);
    RET_IF_FALSE(par, MFX_ERR_NULL_PTR);

    CpuWorkstream *ws = reinterpret_cast<CpuWorkstream *>(session);
    CpuVPP *vpp       = ws->GetVPP();
    RET_IF_FALSE(vpp, MFX_ERR_NOT_INITIALIZED);

    return vpp->GetVideoParam(par);
}

mfxStatus MFXVideoVPP_RunFrameVPPAsync(mfxSession session,
                                       mfxFrameSurface1 *in,
                                       mfxFrameSurface1 *out,
                                       mfxExtVppAuxData *aux,
                                       mfxSyncPoint *syncp) {
    VPL_TRACE_FUNC;
    RET_IF_FALSE(session, MFX_ERR_INVALID_HANDLE);
    RET_IF_FALSE(out && syncp, MFX_ERR_NULL_PTR);

    CpuWorkstream *ws = reinterpret_cast<CpuWorkstream *>(session);
    CpuVPP *vpp       = ws->GetVPP();
    RET_IF_FALSE(vpp, MFX_ERR_NOT_INITIALIZED);

    *syncp = (mfxSyncPoint)(0x12345678);

    return vpp->ProcessFrame(in, out, aux);
}

mfxStatus MFXVideoVPP_Reset(mfxSession session, mfxVideoParam *par) {
    VPL_TRACE_FUNC;
    RET_IF_FALSE(session, MFX_ERR_INVALID_HANDLE);
    RET_IF_FALSE(par, MFX_ERR_NULL_PTR);

    CpuWorkstream *ws = reinterpret_cast<CpuWorkstream *>(session);
    CpuVPP *vpp       = ws->GetVPP();
    RET_IF_FALSE(vpp, MFX_ERR_NOT_INITIALIZED);

    mfxVideoParam oldParam = { 0 };
    vpp->GetVideoParam(&oldParam);
    RET_ERROR(vpp->IsSameVideoParam(par, &oldParam));

    RET_ERROR(MFXVideoVPP_Close(session));
    return MFXVideoVPP_Init(session, par);
}

mfxStatus MFXVideoVPP_GetVPPStat(mfxSession session, mfxVPPStat *stat) {
    VPL_TRACE_FUNC;
    return MFX_ERR_NOT_IMPLEMENTED;
}

mfxStatus MFXVideoVPP_ProcessFrameAsync(mfxSession session,
                                        mfxFrameSurface1 *in,
                                        mfxFrameSurface1 **out) {
    VPL_TRACE_FUNC;
    RET_IF_FALSE(session, MFX_ERR_INVALID_HANDLE);

    CpuWorkstream *ws = reinterpret_cast<CpuWorkstream *>(session);
    CpuVPP *vpp       = ws->GetVPP();
    RET_IF_FALSE(vpp, MFX_ERR_NOT_INITIALIZED);

    if (*out == 0) {
        // get a ref-counted surface for vpp into
        // behavior is equivalent to the application calling this and then
        //   passing the surface into ProcessFrameAsync()
        RET_ERROR(MFXMemory_GetSurfaceForVPPOut(session, out));
        (*out)->FrameInterface->Map(*out, MFX_MAP_WRITE);
    }

    mfxStatus sts = vpp->ProcessFrame(in, *out, NULL);
    return sts;
}
