/*
 * This file is part of libsh4lt.
 *
 * libsh4lt is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
 * @file   json-serializer.h
 *
 * @brief serialize and deserialize an information tree using json
 *
 *
 */

#ifndef SH4LT_INFORMATION_TREE_JSON_H_
#define SH4LT_INFORMATION_TREE_JSON_H_

#include <string>
#include "../utils/bool-log.hpp"
#include "./information-tree.hpp"

namespace sh4lt::infotree::json {

/**
 * Serialize an InfoTree into a JSON string
 * representation.
 *
 * \param tree Tree to serialize
 *
 * \return Serialized string
 */
auto serialize(InfoTree::ptrc tree) -> std::string;

/**
 * Parse a JSON string and create an Infotree.
 *
 * \param json_string JSON string to parse
 *
 * \return Tree resulting from the deserialization, or empty shared pointer
 * in case of JSON error in json_string. Error message can be obtained with
 * is_valid function.
 *
 */
auto deserialize(const std::string& json_string) -> InfoTree::ptr;

/**
 * Check JSON string validity and provides error message if invalid.
 *
 * \param Serialized JSON string to parse
 *
 * \return The BoolLog object that contains validity and possible error message
 *
 */
auto is_valid(const std::string& json_string) -> BoolLog;

}  // namespace sh4lt
#endif
