/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#ifndef SH4LT_LOGGER_H_
#define SH4LT_LOGGER_H_

#include <memory>
#include <string>

#define SH4LT_MakeLevel(NAME)                                                \
public:                                                                        \
  template <typename... Targs>                                                 \
  void NAME(const char *format, const ::std::string &value, Targs... Fargs) {  \
    on_##NAME(make_string(format, std::forward<const std::string &>(value),    \
                          std::forward<Targs>(Fargs)...));                     \
  }                                                                            \
  void NAME(const char *format) { on_##NAME(make_string(format)); }            \
                                                                               \
private:                                                                       \
  virtual void on_##NAME(std::string &&) = 0;

namespace sh4lt::logger {

class Logger {
 public:
  using ptr = std::shared_ptr<Logger>;
  virtual ~Logger() = default;
  Logger() = default;
  Logger(Logger&&) = default;
  Logger(const Logger&) = default;
  auto operator=(const Logger&) -> Logger& = default;
  auto operator=(Logger&&) -> Logger& = default;
  SH4LT_MakeLevel(error);
  SH4LT_MakeLevel(critical);
  SH4LT_MakeLevel(warning);
  SH4LT_MakeLevel(message);
  SH4LT_MakeLevel(info);
  SH4LT_MakeLevel(debug);

 private:
  auto make_string(const char* format) -> std::string { return format; }
  template <typename... Targs>
  auto make_string(const char* format, const std::string& value, Targs... Fargs) -> std::string {
    std::string res;
    for (; *format != '\0'; format++) {
      if (*format == '%') {
        res.append(value);
        return res.append(make_string(format + 1, Fargs...));
      }
      res.append(format, 1);
    }
    return res;
  }
};

} // namespace sh4lt
#endif
