%feature("docstring") OT::LinearModelAlgorithm
"Class used to create a linear model from numerical samples.

Parameters
----------
XSample : 2-d sequence of float
    The input samples of a model.

YSample : 2-d sequence of float
    The output samples of a model, must be of dimension 1.

basis : :class:`~openturns.Basis`
    Optional.
    The :math:`\\phi` basis .

See Also
--------
LinearModelResult

Notes
-----
This class is used in order to create a linear model from an input sample
and an output sample.
Let :math:`n` be the sample size and let :math:`n_x` be the input
sample dimension.
This class fits a linear regression model between the scalar
variable :math:`Y` and the :math:`n_x`-dimensional vector :math:`\\vect{X} = (X_i)_{1 \\leq i \\leq n_x}`.
The linear model can be estimated with or without a functional basis.

If no basis is specified, the model is:

.. math::

    Y = a_0 + \\sum_{i=1}^{n_x} a_i X_i + \\epsilon

where :math:`a_0, a_1, ..., a_{n_x} \\in \\Rset`
are unknown coefficients and :math:`\\epsilon` is a random variable with zero mean and constant
(unknown) variance :math:`\\sigma^2` independent from
the coefficients :math:`\\{a_i\\}_{i = 0, ..., n_x}`.
The algorithm estimates the coefficients
:math:`a_0, a_1, ..., a_{n_x}` of the linear model.
Moreover, the method estimates the variance :math:`\\sigma^2`.

If a functional basis is specified, let :math:`p \\in \\Nset` be the number of
functions in the basis.
For :math:`j \\in \\{1, ..., p\\}`, let :math:`\\phi_j : \\Rset^{n_x} \\rightarrow \\Rset`
be the :math:`i`-th basis function.
The linear model is:

.. math::

    Y = \\sum_{j=1}^p a_j \\phi_j(\\vect{X}) + \\epsilon

where :math:`\\epsilon` is a random variable with zero mean and constant
(and unknown) variance :math:`\\sigma^2` and :math:`a_1, ..., a_p \\in \\Rset`
are unknown coefficients.
The algorithm estimates the coefficients
:math:`a_1, ..., a_p` of the linear model.
Moreover, the method estimates the variance :math:`\\sigma^2`.

The coefficients :math:`a_i` are evaluated using a linear least squares method,
by default the `QR` method.
User might also choose `SVD` or `Cholesky` by
setting the `LinearModelAlgorithm-DecompositionMethod` key of the :class:`~openturns.ResourceMap`.
Here are a few guidelines to choose the appropriate decomposition method:

- The Cholesky can be safely used if the functional basis is orthogonal
  and the sample is drawn from the corresponding distribution,
  because this ensures that the columns of the design matrix are
  asymptotically orthogonal when the sample size increases.
  In this case, evaluating the Gram matrix does not increase
  the condition number.
- Selecting the decomposition method can also be based on the sample size.

Please read the :meth:`~openturns.LeastSquaresMethod.Build` help page
for details on this topic.

The :class:`~openturns.LinearModelAnalysis` class can be used for a detailed
analysis of the linear model result.

No scaling is involved in this method.
The scaling of the data, if any, is the responsibility of the user of the algorithm.
This may be useful if, for example, we use a linear model (without functional basis)
with very different input magnitudes and use the Cholesky decomposition
applied to the associated Gram matrix.
In this case, the Cholesky method may fail to produce accurate results.

Examples
--------
>>> import openturns as ot
>>> distribution = ot.Normal()
>>> func = ot.SymbolicFunction(['x1','x2', 'x3'], ['x1 + x2 + sin(x2 * 2 * pi_)/5 + 1e-3 * x3^2'])
>>> dimension = 3
>>> distribution = ot.ComposedDistribution([ot.Normal()]*dimension)
>>> input_sample = distribution.getSample(20)
>>> output_sample = func(input_sample)
>>> algo = ot.LinearModelAlgorithm(input_sample, output_sample)
>>> algo.run()
>>> result = algo.getResult()"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAlgorithm::getInputSample
"Accessor to the input sample.

Returns
-------
inputSample : :class:`~openturns.Sample`
    The Xsample which had been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAlgorithm::getBasis
"Accessor to the input basis.

Returns
-------
basis : :class:`~openturns.Basis`
    The basis which had been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAlgorithm::getOutputSample
"Accessor to the output sample.

Returns
-------
outputSample : :class:`~openturns.Sample`
    The Ysample which had been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAlgorithm::getResult
"Accessor to the computed linear model.

Returns
-------
result : :class:`~openturns.LinearModelResult`
    The linear model built from numerical samples, along with other useful information."

