// Copyright (C) 2021 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0

#include "filepath.h"

#include "algorithm.h"
#include "devicefileaccess.h"
#include "environment.h"
#include "filestreamermanager.h"
#include "fileutils.h"
#include "hostosinfo.h"
#include "qtcassert.h"
#include "stringtable.h"
#include "textcodec.h"
#include "utilstr.h"

#include <QByteArray>
#include <QDateTime>
#include <QDebug>
#include <QDirIterator>
#include <QFileInfo>
#include <QLoggingCategory>
#include <QRegularExpression>
#include <QStringView>
#include <QTemporaryDir>
#include <QTemporaryFile>
#include <QUrl>

#ifdef Q_OS_WIN
#ifdef QTCREATOR_PCH_H
#define CALLBACK WINAPI
#endif
#include <qt_windows.h>
#include <shlobj.h>
#endif

namespace Utils {

Q_LOGGING_CATEGORY(fpLog, "qtc.filepath", QtWarningMsg);

static DeviceFileHooks &deviceFileHooks()
{
    static DeviceFileHooks theDeviceHooks;
    return theDeviceHooks;
}

void DeviceFileHooks::setupDeviceFileHooks(const DeviceFileHooks &hooks)
{
    static bool wasAlreadySet = false;
    QTC_ASSERT(!wasAlreadySet, return);
    wasAlreadySet = true;
    deviceFileHooks() = hooks;
}

static void logError(const QString &context, const QString &errorMsg)
{
    qCDebug(fpLog) << context << ":" << errorMsg;
}

static bool isWindowsDriveLetter(QChar ch)
{
    return (ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z');
}

/*!
    \class Utils::FilePath
    \inmodule QtCreator

    \brief The FilePath class is an abstraction for handles to objects
    in a (possibly remote) file system, similar to a URL or, in the local
    case, a path to a file or directory.

    Ideally, all of \QC code should use FilePath for this purpose,
    but for historic reasons there are still large parts using QString.

    FilePaths are internally stored as triple of strings, with one
    part ("scheme") identifying an access method, a second part ("host")
    a file system (e.g. a host) and a third part ("path") identifying
    a (potential) object on the systems.

    FilePath follows the Unix paradigm of "everything is a file":
    There is no conceptional difference between FilePaths referring
    to plain files or directories.

    A FilePath is implicitly associated with an operating system via its
    host part. The path part of a FilePath is internally stored
    with forward slashes, independent of the associated OS.

    The path parts of FilePaths associated with Windows (and macOS,
    unless selected otherwise in the settings) are compared case-insensitively
    to each other.
    Note that comparisons for equivalence generally need out-of-band
    knowledge, as there may be multiple FilePath representations for
    the same file (e.g. different access methods may lead to the same
    file).

    There are several conversions between FilePath and other string-like
    representations:

    \list

    \li FilePath::fromUserInput()

        Convert string-like data from sources originating outside of
        \QC, e.g. from human input in GUI controls, from environment
        variables and from command-line parameters to \QC.

        The input can contain both slashes and backslashes and will
        be parsed and normalized.

    \li FilePath::nativePath()

        Converts the FilePath to the slash convention of the associated
        OS and drops the scheme and host parts.

        This is useful to interact with the facilities of the associated
        OS, e.g. when passing this FilePath as an argument to a command
        executed on the associated OS.

        \note The FilePath passed as executable to a CommandLine is typically
        not touched by user code. The Process will use it to determine
        the remote system and apply the necessary conversions internally.

    \li FilePath::toFSPathString()

        Converts the FilePath to a [drive:]/__qtc_devices__/scheme/host/path
        string.

        The result works in most cases also with remote setups and QDir/QFileInfo
        but is slower than direct FilePath use and should only be used when
        proper porting to FilePath is too difficult, or not possible, e.g.
        when external Qt based libraries are used that do not use FilePath.

    \li FilePath::toUserOutput()

        Converts the FilePath to the slash convention of the associated
        OS and retains the scheme and host parts.

        This is rarely useful for remote paths as there is practically
        no consumer of this style.

    \li FilePath::displayName()

        Converts the FilePath to the slash convention of the associated
        OS and adds the scheme and host as a " on <device>" suffix.

        This is useful for static user-facing output in the GUI.

    \li FilePath::fromVariant(), FilePath::toVariant()

        These are used to interface QVariant-based API, e.g.
        settings or item model (internal) data.

    \li FilePath::toUrlishString()

        Converts the FilePath to a scheme://host/path string.

        Mostly used in legacy code and for debugging reasons.

        In almost all cases, use of one of the more specialized conversion
        above is more appropriate.

    \endlist

    Conversion of string-like data should always happen at the outer boundary
    of \QC code, using \c fromUserInput() for in-bound communication,
    and depending on the medium \c nativePath() or \c displayName() for out-bound
    communication.

    Communication with QVariant based Qt API should use \c fromVariant() and
    \c toVariant().

    Uses of \c toUrlishString() should be phased out by transforming
    code from QString based file path to FilePath. An exception here are
    fragments of paths of a FilePath that are later used with \c pathAppended()
    or similar which should be kept as QString.

    UNC paths will retain their "//" begin, and are recognizable by this.
*/

FilePath::FilePath()
{
}

/*!
   Constructs a FilePath from \a info.
*/
FilePath FilePath::fromFileInfo(const QFileInfo &info)
{
    return FilePath::fromString(info.absoluteFilePath());
}

/*!
   Returns a QFileInfo.
*/
QFileInfo FilePath::toFileInfo() const
{
    return QFileInfo(toFSPathString());
}

/*!
    Constructs a FilePath from \a variant.

    \sa toVariant()
*/
FilePath FilePath::fromVariant(const QVariant &variant)
{
    return fromSettings(variant); // FIXME: Use variant.value<FilePath>()
}

/*!
    Constructs a FilePath from \a url.

    \sa toVariant()
*/
FilePath FilePath::fromUrl(const QUrl &url)
{
    FilePath result;
    if (url.isLocalFile())
        return FilePath::fromString(url.toLocalFile());
    return FilePath::fromParts(url.scheme(), url.host(), url.path());
}

FilePath FilePath::fromParts(const QStringView scheme, const QStringView host, const QStringView path)
{
    FilePath result;
    result.setParts(scheme, host, path);
    return result;
}

FilePath FilePath::fromPathPart(const QStringView path)
{
    FilePath result;
    result.m_data = path.toString();
    result.m_pathLen = path.size();
    return result;
}

FilePath FilePath::fromPathComponents(const QList<QStringView> &components)
{
    auto it = components.cbegin();
    if (it == components.cend())
        return FilePath();

    QString path;
    if (components.front() == '/') {
        path = '/';
        ++it;
    }
    for (; it != components.cend(); ++it) {
        path += *it;
        if (it + 1 != components.cend())
            path += '/';
    }

    return FilePath::fromPathPart(path);
}

FilePath FilePath::currentWorkingPath()
{
    return FilePath::fromString(QDir::currentPath());
}

bool FilePath::isRootPath() const
{
    if (!isLocal()) {
        QStringView path = pathView();
        if (osType() != OsTypeWindows)
            return path == QLatin1String("/");

        // Remote windows paths look like this: "/c:/", so we remove the leading '/'
        if (path.startsWith('/'))
            path = path.mid(1);

        if (path.length() > 3)
            return false;

        if (!startsWithDriveLetter())
            return false;

        if (path.length() == 3 && path[2] != QLatin1Char('/'))
            return false;

        return true;
    }

    return QDir(path()).isRoot();
}

bool FilePath::isResourceFile() const
{
    if (scheme() == u"qrc")
        return true;
    if (!isLocal())
        return false;
    return pathView().startsWith(':');
}

QString FilePath::encodedHost() const
{
    QString result = host().toString();
    result.replace('%', "%25");
    result.replace('/', "%2f");
    return result;
}

QString decodeHost(QString host)
{
    return host.replace("%25", "%").replace("%2f", "/");
}

/*!
    Returns a QString for passing through QString based APIs.

    \note This is obsolete API and should be replaced by extended use
    of proper \c FilePath, or, in case this is not possible by \c toFSPathString().

    This uses a scheme://host/path setup and is, together with
    fromString, used to pass FilePath through \c QString using
    code paths.

    The result is not useful for use with \cQDir and \c QFileInfo
    and gets destroyed by some operations like \c QFileInfo::canonicalFile.

    \sa toFSPathString()
*/
QString FilePath::toUrlishString() const
{
    if (isLocal())
        return path();

    if (pathView().isEmpty())
        return scheme() + "://" + encodedHost();

    if (isRelativePath())
        return scheme() + "://" + encodedHost() + "/./" + pathView();
    if (isWindowsDriveLetter(pathView().at(0)))
        return scheme() + "://" + encodedHost() + "/" + pathView();
    return scheme() + "://" + encodedHost() + pathView();
}

bool FilePath::equals(const FilePath &first, const FilePath &second, Qt::CaseSensitivity cs)
{
    if (first.m_hash != 0 && second.m_hash != 0 && first.m_hash != second.m_hash)
        return false;

    return first.pathView().compare(second.pathView(), cs) == 0
           && first.host() == second.host()
           && first.scheme() == second.scheme();
}

/*!
 * Returns \c true if this file path compares equal to \a other case-sensitively.
 * This is relevant on semi-case sensitive systems like Windows with NTFS.
 * \sa {https://bugreports.qt.io/browse/QTCREATORBUG-30846}{QTCREATORBUG-30846}
 */
bool FilePath::equalsCaseSensitive(const FilePath &other) const
{
    return equals(*this, other, Qt::CaseSensitive);
}

/*!
    \class Utils::FilePathWatcher
    \inmodule QtCreator

    \brief The FilePathWatcher class watches a file at a path for editing,
    renaming, or deletion.
*/

/*!
    Returns a FilePathWatcher for each given path.

    The returned FilePathWatcher emits its signal when the file at this path
    is modified, renamed, or deleted. The signal is emitted in the calling thread.
    If called from a non-main thread, it might take a while until the signal
    starts to be emitted.

    \sa FilePathWatcher
*/
std::vector<Result<std::unique_ptr<FilePathWatcher>>> FilePath::watch(const FilePaths &paths)
{
    if (paths.isEmpty())
        return {};
    if (paths.size() == 1) {
        return paths.at(0).fileAccess()->watch({paths.at(0)});
    }
    using WatchResult = std::vector<Result<std::unique_ptr<FilePathWatcher>>>;
    WatchResult allResults;
    allResults.reserve(paths.size());
    // Sort into device file access instances, so we can call watch() in bulk on them
    std::unordered_map<DeviceFileAccess *, FilePaths> access;
    for (const FilePath &path : paths)
        access[path.fileAccess()].append(path);
    for (const auto &fileAccess : access) {
        WatchResult results = fileAccess.first->watch(fileAccess.second);
        for (auto &result : results)
            allResults.push_back(std::move(result));
    }
    return allResults;
}

/*!
    \deprecated Use FilePaths::watch() or static FilePath::watch() instead.
*/
Utils::Result<std::unique_ptr<FilePathWatcher>> FilePath::watch() const
{
    std::vector<Result<std::unique_ptr<FilePathWatcher>>> watches = FilePath::watch({*this});
    QTC_ASSERT(
        !watches.empty(),
        return ResultError(
            Tr::tr("Internal DeviceFileAccess error: watch() did not return a result.")));
    return std::move(watches.at(0));
}

void FilePath::openTerminal(const Environment &env) const
{
    deviceFileHooks().openTerminal(*this, env);
}

/*!
    Returns a copy with interned data.
*/
FilePath FilePath::intern() const
{
    FilePath result = *this;
    result.m_data = StringTable::insert(m_data);
    return result;
}

/*!
    Returns a QString for passing on to QString based APIs.

    This uses a /__qtc_devices__/host/path setup.

    This works in most cases also with remote setups and QDir/QFileInfo etc.
    but is slower than direct FilePath use and should only be used when
    proper porting to FilePath is too difficult, or not possible, e.g.
    when external Qt based libraries are used that do not use FilePath.

    \sa fromUserInput()
*/
QString FilePath::toFSPathString() const
{
    if (scheme().isEmpty())
        return path();

    if (pathView().isEmpty())
        return specialRootPath() + '/' + scheme() + '/' + encodedHost();

    if (isRelativePath())
        return specialRootPath() + '/' + scheme() + '/' + encodedHost() + "/./" + pathView();
    return specialRootPath() + '/' + scheme() + '/' + encodedHost() + pathView();
}

QUrl FilePath::toUrl() const
{
    if (isLocal())
        return QUrl::fromLocalFile(toFSPathString());
    QUrl url;
    url.setScheme(scheme().toString());
    url.setHost(host().toString());
    url.setPath(path());
    return url;
}

/*!
    Returns a QString to display to the user, including the device prefix.

    Converts the separators to the native format of the system
    this path belongs to.
*/
QString FilePath::toUserOutput() const
{
    QString tmp = toUrlishString();
    if (!isLocal())
        return tmp;

    if (osType() == OsTypeWindows)
        tmp.replace('/', '\\');
    return tmp;
}

/*!
    Returns a QString to pass to target system native commands, without the device prefix.

    Converts the separators to the native format of the system
    this path belongs to.
*/

QString FilePath::nativePath() const
{
    QString data = path();
    if (osType() == OsTypeWindows)
        data.replace('/', '\\');
    return data;
}

QStringView FilePath::fileNameView() const
{
    const QStringView fp = pathView();
    return fp.mid(fp.lastIndexOf('/') + 1);
}

QString FilePath::fileName() const
{
    return fileNameView().toString();
}

QString FilePath::fileNameWithPathComponents(int pathComponents) const
{
    QString fullPath = path();

    if (pathComponents < 0)
        return fullPath;
    const QChar slash = QLatin1Char('/');
    int i = fullPath.lastIndexOf(slash);
    if (pathComponents == 0 || i == -1)
        return fullPath.mid(i + 1);
    int component = i + 1;
    // skip adjacent slashes
    while (i > 0 && fullPath.at(--i) == slash)
        ;
    while (i >= 0 && --pathComponents >= 0) {
        i = fullPath.lastIndexOf(slash, i);
        component = i + 1;
        while (i > 0 && fullPath.at(--i) == slash)
            ;
    }

    if (i > 0 && fullPath.lastIndexOf(slash, i) != -1)
        return fullPath.mid(component);

    // If there are no more slashes before the found one, return the entire string
    return toUrlishString();
}

/*!
    Returns the base name of the file without the path.

    The base name consists of all characters in the file up to
    (but not including) the first '.' character.
*/
QString FilePath::baseName() const
{
    const QString &name = fileName();
    return name.left(name.indexOf('.'));
}

/*!
   Returns the complete base name of the file without the path.

    The complete base name consists of all characters in the file up to
    (but not including) the last '.' character. In case of ".ui.qml"
    it will be treated as one suffix.
*/
QString FilePath::completeBaseName() const
{
    const QString &name = fileName();
    if (name.endsWith(".ui.qml"))
        return name.left(name.length() - QString(".ui.qml").length());
    return name.left(name.lastIndexOf('.'));
}

/*!
   Returns the suffix (extension) of the file.

    The suffix consists of all characters in the file after
    (but not including) the last '.'. In case of ".ui.qml" it will
    be treated as one suffix.
*/
QStringView FilePath::suffixView() const
{
    const QStringView name = fileNameView();
    if (name.endsWith(u".ui.qml"))
        return u"ui.qml";
    const int index = name.lastIndexOf('.');
    if (index >= 0)
        return name.mid(index + 1);
    return {};
}

QString FilePath::suffix() const
{
    return suffixView().toString();
}

/*!
   Returns the complete suffix (extension) of the file.

    The complete suffix consists of all characters in the file after
    (but not including) the first '.'.
*/
QString FilePath::completeSuffix() const
{
    const QString &name = fileName();
    const int index = name.indexOf('.');
    if (index >= 0)
        return name.mid(index + 1);
    return {};
}

QList<QStringView> FilePath::pathComponents() const
{
    QList<QStringView> result;
    QStringView path = pathView();
    int start = 0;

    if (osType() == OsTypeWindows && startsWithDriveLetter()) {
        start = 2;
        result.append(path.mid(0, 2));
        if (path.size() == 2)
            return result;
        else if (path.at(2) == '/') {
            result.append(path.mid(2, 1));
            start = 3;
        }
    }

    while (start < path.size()) {
        int end = path.indexOf('/', start);
        if (end == -1)
            end = path.size();
        // Consider "/" as a path component
        if (end == 0)
            result.append(path.mid(start, 1));
        else
            result.append(path.mid(start, end - start));
        start = end + 1;
    }
    return result;
}

QStringView FilePath::scheme() const
{
    return QStringView{m_data}.mid(m_pathLen, m_schemeLen);
}

QStringView FilePath::host() const
{
    return QStringView{m_data}.mid(m_pathLen + m_schemeLen, m_hostLen);
}

QStringView FilePath::pathView() const
{
    return QStringView(m_data.constData(), m_pathLen);
}

QString FilePath::path() const
{
    return pathView().toString();
}

void FilePath::setParts(const QStringView scheme, const QStringView host, QStringView path)
{
    QTC_CHECK(!scheme.contains('/'));

    if (path.length() >= 3 && path[0] == '/' && path[1] == '.' && path[2] == '/')
        path = path.mid(3);

    m_hash = 0;

    // The equivalent of:
    //   m_data = path.toString() + scheme.toString() + host.toString();
    // but with less copying.
    // Note: The QStringBuilder optimization does not currently work in this case.
    m_data.resize(0);
    m_data.reserve(m_schemeLen + m_hostLen + m_pathLen);
    m_data.append(path);
    m_data.append(scheme);
    m_data.append(host);

    m_schemeLen = scheme.size();
    m_hostLen = host.size();
    m_pathLen = path.size();
}

/*!
    \brief Re-uses or creates a directory in this location.

    Returns true if the directory is writable afterwards.

    \sa createDir()
*/
Result<> FilePath::ensureWritableDir() const
{
    return fileAccess()->ensureWritableDirectory(*this);
}

bool FilePath::ensureExistingFile() const
{
    const Result<> res = fileAccess()->ensureExistingFile(*this);
    if (!res) {
        logError("ensureExistingFile", res.error());
        return false;
    }
    return true;
}

/*!
    Returns a bool indicating on whether a process with this FilePath's
    native path is likely to start.

    This is equivalent to \l isExecutableFile() in general.
    On Windows, it might append various suffixes depending on
    \a matchScope.
*/
std::optional<FilePath> FilePath::refersToExecutableFile(MatchScope matchScope) const
{
    const auto res = fileAccess()->refersToExecutableFile(*this,  matchScope);
    if (!res) {
        logError("refersToExecutableFile", res.error());
        return {};
    }
    return *res;
}

Result<FilePath> FilePath::tmpDir() const
{
    if (!isLocal()) {
        const Result<Environment> env = deviceEnvironmentWithError();
        if (!env)
            return ResultError(env.error());

        if (env->hasKey("TMPDIR"))
            return withNewPath(env->value("TMPDIR")).cleanPath();
        if (env->hasKey("TEMP"))
            return withNewPath(env->value("TEMP")).cleanPath();
        if (env->hasKey("TMP"))
            return withNewPath(env->value("TMP")).cleanPath();

        if (osType() != OsTypeWindows)
            return withNewPath("/tmp");
        return ResultError(
            Tr::tr("Could not find temporary directory on device %1").arg(displayName()));
    }

    return FilePath::fromUserInput(QDir::tempPath());
}

Result<FilePath> FilePath::createTempFile() const
{
    if (isLocal()) {
        QTemporaryFile file(path());
        file.setAutoRemove(false);
        if (file.open())
            return FilePath::fromString(file.fileName());

        return ResultError(Tr::tr("Could not create temporary file: %1").arg(file.errorString()));
    }

    return fileAccess()->createTempFile(*this);
}

Result<FilePath> FilePath::createTempDir() const
{
    if (isLocal()) {
        QTemporaryDir dir(path());
        dir.setAutoRemove(false);
        if (!dir.path().isEmpty())
            return FilePath::fromString(dir.path());

        return ResultError(
            Tr::tr("Could not create temporary directory: %1").arg(dir.errorString()));
    }

    return fileAccess()->createTempDir(*this);
}

bool FilePath::hasHardLinks() const
{
    const Result<bool> res = fileAccess()->hasHardLinks(*this);
    if (!res)
        logError("hasHardlinks", res.error());
    return res.has_value() ? res.value() : false;
}

/*!
    \brief Creates a directory in this location.

    Returns true if the directory could be created, false if not,
    even if it existed before.

    \sa ensureWritableDir()
*/
bool FilePath::createDir() const
{
    const Result<> res = fileAccess()->createDirectory(*this);
    return res.has_value();
}

FilePaths FilePath::dirEntries(const FileFilter &filter, QDir::SortFlags sort) const
{
    FilePaths result;

    iterateDirectory(
        [&result](const FilePath &path) { result.append(path); return IterationPolicy::Continue; },
        filter
     );

    // FIXME: Not all flags supported here.
    const QDir::SortFlags sortBy = (sort & QDir::SortByMask);

    using Predicate = std::function<bool(const FilePath &, const FilePath &)>;

    std::function<void(FilePaths &, Predicate)> sortWithFolders =
        [](FilePaths &result, Predicate predicate) { Utils::sort(result, predicate); };

    if (sort & QDir::DirsFirst) {
        sortWithFolders = [](FilePaths &result, Predicate predicate) {
            Predicate folderFilter = [predicate](const FilePath &path1, const FilePath &path2) {
                if (path1.isDir() && !path2.isDir())
                    return true;
                if (!path1.isDir() && path2.isDir())
                    return false;
                return predicate(path1, path2);
            };
            Utils::sort(result, folderFilter);
        };
    }

    if (sortBy == QDir::Name) {
        sortWithFolders(result, [](const FilePath &path1, const FilePath &path2) {
            return path1.fileName() < path2.fileName();
        });
    } else if (sortBy == QDir::Time) {
        sortWithFolders(result, [](const FilePath &path1, const FilePath &path2) {
            return path1.lastModified() < path2.lastModified();
        });
    }

    if (sort != QDir::NoSort && (sort & QDir::Reversed))
        std::reverse(result.begin(), result.end());

    return result;
}

FilePaths FilePath::dirEntries(QDir::Filters filters) const
{
    return dirEntries(FileFilter({}, filters));
}

/*!
    Runs \a callBack on each directory entry matching the \a filter.
*/

void FilePath::iterateDirectory(const IterateDirCallback &callBack, const FileFilter &filter) const
{
    fileAccess()->iterateDirectory(*this, callBack, filter);
}

void FilePath::iterateDirectories(const FilePaths &dirs,
                                  const IterateDirCallback &callBack,
                                  const FileFilter &filter)
{
    for (const FilePath &dir : dirs)
        dir.iterateDirectory(callBack, filter);
}

Result<QByteArray> FilePath::fileContents(qint64 maxSize, qint64 offset) const
{
    return fileAccess()->fileContents(*this, maxSize, offset);
}

bool FilePath::ensureReachable(const FilePath &other) const
{
    if (!isLocal()) {
        QTC_ASSERT(deviceFileHooks().ensureReachable, return false);
        return deviceFileHooks().ensureReachable(*this, other);
    }
    return other.isLocal();
}

Result<qint64> FilePath::writeFileContents(const QByteArray &data) const
{
    return fileAccess()->writeFileContents(*this, data);
}

FileStreamHandle FilePath::asyncCopy(const Continuation<> &cont, const FilePath &target) const
{
    return FileStreamerManager::copy(cont, *this, target);
}

FileStreamHandle FilePath::asyncRead(const Continuation<QByteArray> &cont) const
{
    return FileStreamerManager::read(cont, *this);
}

FileStreamHandle FilePath::asyncWrite(const Continuation<qint64> &cont, const QByteArray &data) const
{
    return FileStreamerManager::write(cont, *this, data);
}

bool FilePath::isLocal() const
{
    return m_schemeLen == 0 || scheme() == u"file";
}

bool FilePath::isSameDevice(const FilePath &other) const
{
    if (isLocal() != other.isLocal())
        return false;
    if (isLocal() && other.isLocal())
        return true;

    QTC_ASSERT(deviceFileHooks().isSameDevice, return true);
    return deviceFileHooks().isSameDevice(*this, other);
}

bool FilePath::isSameFile(const FilePath &other) const
{
    if (*this == other)
        return true;

    if (!isSameDevice(other))
        return false;

    const Result<QByteArray> fileId = fileAccess()->fileId(*this);
    if (!fileId) {
        logError("isSameFile", fileId.error());
        return false;
    }
    const Result<QByteArray> otherFileId = fileAccess()->fileId(other);
    if (!otherFileId) {
        logError("isSameFile", otherFileId.error());
        return false;
    }

    if (fileId->isEmpty() || otherFileId->isEmpty())
        return false;

    return *fileId == *otherFileId;
}

static FilePaths appendExeExtensions(const FilePath &executable,
                                     FilePath::MatchScope matchScope)
{
    FilePaths result = {executable};
    const QStringView suffix = executable.suffixView();
    if (executable.osType() == OsTypeWindows && suffix.isEmpty()) {
        switch (matchScope) {
            case FilePath::ExactMatchOnly:
                break;
            case FilePath::WithExeSuffix:
                result.append(executable.stringAppended(".exe"));
                break;
            case FilePath::WithBatSuffix:
                result.append(executable.stringAppended(".bat"));
                break;
            case FilePath::WithExeOrBatSuffix:
                result.append(executable.stringAppended(".exe"));
                result.append(executable.stringAppended(".bat"));
                break;
            case FilePath::WithAnySuffix: {
                // Check all the executable extensions on windows:
                // PATHEXT is only used if the executable has no extension
                static const QStringList extensions = Environment::systemEnvironment()
                    .expandedValueForKey("PATHEXT").split(';');
                for (const QString &ext : extensions)
                    result.append(executable.stringAppended(ext.toLower()));
                break;
            }
            default:
                break;
        }
    }
    return result;
}

bool FilePath::isSameExecutable(const FilePath &other) const
{
    if (*this == other)
        return true;

    if (!isSameDevice(other))
        return false;

    const FilePaths exe1List = appendExeExtensions(*this, WithAnySuffix);
    const FilePaths exe2List = appendExeExtensions(other, WithAnySuffix);
    for (const FilePath &f1 : exe1List) {
        for (const FilePath &f2 : exe2List) {
            if (f1.isSameFile(f2))
                return true;
        }
    }
    return false;
}

/*!
    Returns an empty FilePath if this is not a symbolic link.
*/
FilePath FilePath::symLinkTarget() const
{
    const Result<FilePath> res = fileAccess()->symLinkTarget(*this);
    if (!res) {
        logError("symLinkTarget", res.error());
        return {};
    }
    return *res;
}

FilePath FilePath::withExecutableSuffix() const
{
    return withNewPath(OsSpecificAspects::withExecutableSuffix(osType(), path()));
}

FilePath FilePath::withSuffix(const QString &suffix) const
{
    return withNewPath(path() + suffix);
}

static bool startsWithWindowsDriveLetterAndSlash(QStringView path)
{
    return path.size() > 2 && path[1] == ':' && path[2] == '/' && isWindowsDriveLetter(path[0]);
}

static bool startsWithWindowsDriveLetter(QStringView path)
{
    if (path.size() > 2 && startsWithWindowsDriveLetterAndSlash(path))
        return true;
    return path.size() == 2 && path[1] == ':' && isWindowsDriveLetter(path[0]);
}

// This is a compromise across platforms. It will mis-classify "C:/..." as absolute
// paths on non-Windows, and QRC paths ":abc..." (wrongly) as non-absolute.

static bool isAbsolutePathHelper(QStringView path)
{
    if (path.startsWith('/'))
        return true;
    if (startsWithWindowsDriveLetterAndSlash(path))
        return true;
    if (path.startsWith(u":/")) // QRC
        return true;
    return false;
}

int FilePath::rootLength(const QStringView path)
{
    if (path.size() == 0)
        return 0;

    if (path.size() == 1)
        return path[0] == '/' ? 1 : 0;

    if (path[0] == '/' && path[1] == '/') { // UNC, FIXME: Incomplete
        if (path.size() == 2)
            return 2; // case deviceless UNC root - assuming there's such a thing.
        const int pos = path.indexOf('/', 2);
        if (pos == -1)
            return path.size(); // case   //localhost
        return pos + 1;     // case   //localhost/*
    }

    if (startsWithWindowsDriveLetterAndSlash(path))
        return 3; // FIXME-ish: same assumption as elsewhere: we assume "x:/" only ever appears as root
    if (path.size() == 2 && startsWithWindowsDriveLetter(path))
        return 2;

    if (path[0] == '/')
        return 1;

    return 0;
}

int FilePath::schemeAndHostLength(const QStringView path)
{
    static const QLatin1String colonSlashSlash("://");

    const int sep = path.indexOf(colonSlashSlash);
    if (sep == -1)
        return 0;

    const int pos = path.indexOf('/', sep + 3);
    if (pos == -1) // Just   scheme://host
        return path.size();

    return pos + 1;  // scheme://host/ plus something
}

static QString normalizePathSegmentHelper(const QString &name)
{
    const int len = name.length();

    if (len == 0 || name.contains("%{"))
        return name;

    int i = len - 1;
    QVarLengthArray<char16_t> outVector(len);
    int used = len;
    char16_t *out = outVector.data();
    const ushort *p = reinterpret_cast<const ushort *>(name.data());
    const ushort *prefix = p;
    int up = 0;

    const int prefixLength = name.at(0) == u'/' ? 1 : 0;

    p += prefixLength;
    i -= prefixLength;

    // replicate trailing slash (i > 0 checks for emptiness of input string p)
    // except for remote paths because there can be /../ or /./ ending
    if (i > 0 && p[i] == '/') {
        out[--used] = '/';
        --i;
    }

    while (i >= 0) {
        if (p[i] == '/') {
            --i;
            continue;
        }

        // remove current directory
        if (p[i] == '.' && (i == 0 || p[i - 1] == '/')) {
            --i;
            continue;
        }

        // detect up dir
        if (i >= 1 && p[i] == '.' && p[i - 1] == '.' && (i < 2 || p[i - 2] == '/')) {
            ++up;
            i -= i >= 2 ? 3 : 2;
            continue;
        }

        // prepend a slash before copying when not empty
        if (!up && used != len && out[used] != '/')
            out[--used] = '/';

        // skip or copy
        while (i >= 0) {
            if (p[i] == '/') {
                --i;
                break;
            }

            // actual copy
            if (!up)
                out[--used] = p[i];
            --i;
        }

        // decrement up after copying/skipping
        if (up)
            --up;
    }

    // Indicate failure when ".." are left over for an absolute path.
    //    if (ok)
    //        *ok = prefixLength == 0 || up == 0;

    // add remaining '..'
    while (up) {
        if (used != len && out[used] != '/') // is not empty and there isn't already a '/'
            out[--used] = '/';
        out[--used] = '.';
        out[--used] = '.';
        --up;
    }

    bool isEmpty = used == len;

    if (prefixLength) {
        if (!isEmpty && out[used] == '/') {
            // Even though there is a prefix the out string is a slash. This happens, if the input
            // string only consists of a prefix followed by one or more slashes. Just skip the slash.
            ++used;
        }
        for (int i = prefixLength - 1; i >= 0; --i)
            out[--used] = prefix[i];
    } else {
        if (isEmpty) {
            // After resolving the input path, the resulting string is empty (e.g. "foo/.."). Return
            // a dot in that case.
            out[--used] = '.';
        } else if (out[used] == '/') {
            // After parsing the input string, out only contains a slash. That happens whenever all
            // parts are resolved and there is a trailing slash ("./" or "foo/../" for example).
            // Prepend a dot to have the correct return value.
            out[--used] = '.';
        }
    }

    // If path was not modified return the original value
    if (used == 0)
        return name;
    return QString::fromUtf16(out + used, len - used);
}

QString doCleanPath(const QString &input_)
{
    QString input = input_;
    if (input.contains('\\'))
        input.replace('\\', '/');

    if (input.startsWith("//?/")) {
        input = input.mid(4);
        if (input.startsWith("UNC/"))
            input = '/' + input.mid(3); // trick it into reporting two slashs at start
    }

    int prefixLen = 0;
    const int shLen = FilePath::schemeAndHostLength(input);
    if (shLen > 0) {
        prefixLen = shLen + FilePath::rootLength(input.mid(shLen));
    } else {
        prefixLen = FilePath::rootLength(input);
        if (prefixLen > 0 && input.at(prefixLen - 1) == '/')
            --prefixLen;
    }

    // We need to preserve a leading ./ for remote paths.
    bool reAddDotSlash = false;

    QString path = input.mid(prefixLen);
    if (shLen > 0 && path.startsWith("./"))
        reAddDotSlash = true;

    path = normalizePathSegmentHelper(path);

    if (reAddDotSlash) {
        if (!path.startsWith("./"))
            path.prepend("./");
        else
            QTC_CHECK(false);
    }

    // Strip away last slash except for root directories
    if (path.size() > 1 && path.endsWith(u'/'))
        path.chop(1);

    return input.left(prefixLen) + path;
}

/*!
    Finds the parent directory of the file path.

    Returns an empty file path if the file path is already
    a root level directory.

    Returns a file path with the last segment removed.
*/
FilePath FilePath::parentDir() const
{
    const QString basePath = path();
    if (basePath.isEmpty())
        return {};

    const QString path = basePath
                         + (basePath.endsWith('/') ? QLatin1String("..") : QLatin1String("/.."));
    const QString parent = doCleanPath(path);
    if (parent == path)
        return *this;

    return withNewPath(parent);
}

FilePath FilePath::absolutePath() const
{
    if (isLocal() && isEmpty())
        return *this;
    const FilePath parentPath = isAbsolutePath()
                                    ? parentDir()
                                    : FilePath::currentWorkingPath().resolvePath(*this).parentDir();
    return parentPath.isEmpty() ? *this : parentPath;
}

FilePath FilePath::absoluteFilePath() const
{
    if (isAbsolutePath())
        return cleanPath();
    if (isLocal() && isEmpty())
        return cleanPath();

    return FilePath::currentWorkingPath().resolvePath(*this);
}

const QString &FilePath::specialRootName()
{
    static const QString rootName = "__qtc_devices__";
    return rootName;
}

const QString &FilePath::specialRootPath()
{
    static const QString rootPath = HostOsInfo::root().path() + u"__qtc_devices__";
    return rootPath;
}

const QString &FilePath::specialDeviceRootName()
{
    static const QString deviceRootName = "device";
    return deviceRootName;
}

const QString &FilePath::specialDeviceRootPath()
{
    static const QString deviceRootPath = HostOsInfo::root().path() + u"__qtc_devices__/device";
    return deviceRootPath;
}

FilePath FilePath::normalizedPathName() const
{
    FilePath result = *this;
    if (isLocal()) // FIXME: Assumes no remote Windows and Mac for now.
        result.setParts(scheme(), host(), FileUtils::normalizedPathName(path()));
    return result;
}

/*!
    Converts the file path to the slash convention of the associated
    OS and adds the scheme and host as a " on <device>" suffix.

    This is useful for static user-facing output in the GUI.

    If \a args is not empty, it is added to the output after the file path:
    "<path> <args> on <device>".
*/
QString FilePath::displayName(const QString &args) const
{
    QString deviceName;
    if (!isLocal()) {
        QTC_ASSERT(deviceFileHooks().deviceDisplayName, return nativePath());
        deviceName = deviceFileHooks().deviceDisplayName(*this);
    }

    const QString fullPath = nativePath();

    if (args.isEmpty()) {
        if (deviceName.isEmpty())
            return fullPath;

        return Tr::tr("%1 on %2", "File on device").arg(fullPath, deviceName);
    }

    if (deviceName.isEmpty())
        return fullPath + ' ' + args;

    return Tr::tr("%1 %2 on %3", "File and args on device")
            .arg(fullPath, args, deviceName);
}

/*!
    \fn FilePath FilePath::fromString(const QString &filepath)

    Constructs a FilePath from \a filepath

    \a filepath is not checked for validity. It can be given in the following forms:

    \list
    \li  /some/absolute/local/path
    \li  some/relative/path
    \li  scheme://host/absolute/path
    \li  scheme://host/./relative/path    \note the ./ is verbatim part of the path
    \endlist

    Some decoding happens when parsing the \a filepath
    A sequence %25 present in the host part is replaced by % in the host name,
    a sequence %2f present in the host part is replaced by / in the host name.

    The path part might consist of several parts separated by /, independent
    of the platform or file system.

    To create FilePath objects from strings possibly containing backslashes as
    path separator, use \c fromUserInput.

    \sa toFSPathString, toUserOutput, fromUserInput
*/
FilePath FilePath::fromString(const QString &filepath)
{
    FilePath fn;
    fn.setFromString(filepath);
    return fn;
}

void FilePath::setPath(QStringView path)
{
    setParts(scheme(), host(), path);
}

void FilePath::setFromString(QStringView fileNameView)
{
    static const QStringView qtcDevSlash(u"__qtc_devices__/");
    static const QStringView colonSlashSlash(u"://");

#if 1
    // FIXME: Remove below once the calling code is adjusted
    QString dummy;
    if (fileNameView.contains(u'\\')) {
        QTC_CHECK(false);
        dummy = fileNameView.toString();
        dummy.replace('\\', '/');
        fileNameView = dummy;
    }
#endif

    const QChar slash('/');

    bool startsWithQtcSlashDev = false;
    QStringView withoutQtcDeviceRoot = fileNameView;
    if (fileNameView.startsWith(slash) && fileNameView.mid(1).startsWith(qtcDevSlash)) {
        startsWithQtcSlashDev = true;
        withoutQtcDeviceRoot = withoutQtcDeviceRoot.mid(1 + qtcDevSlash.size());
    } else if (fileNameView.size() > 3 && isWindowsDriveLetter(fileNameView.at(0))
               && fileNameView.at(1) == ':' && fileNameView.mid(3).startsWith(qtcDevSlash)) {
        startsWithQtcSlashDev = true;
        withoutQtcDeviceRoot = withoutQtcDeviceRoot.mid(3 + qtcDevSlash.size());
    }

    if (startsWithQtcSlashDev) {
        const int firstSlash = withoutQtcDeviceRoot.indexOf(slash);

        if (firstSlash != -1) {
            const QString scheme = withoutQtcDeviceRoot.left(firstSlash).toString();
            const int secondSlash = withoutQtcDeviceRoot.indexOf(slash, firstSlash + 1);
            const QString host = decodeHost(
                withoutQtcDeviceRoot.mid(firstSlash + 1, secondSlash - firstSlash - 1).toString());
            if (secondSlash != -1) {
                QStringView path = withoutQtcDeviceRoot.mid(secondSlash);
                setParts(scheme, host, path);
                return;
            }

            setParts(scheme, host, u"/");
            return;
        }

        setParts({}, {}, fileNameView);
        return;
    }

    const int firstSlash = fileNameView.indexOf(slash);
    const int schemeEnd = fileNameView.indexOf(colonSlashSlash);
    if (schemeEnd != -1 && schemeEnd < firstSlash) {
        // This is a pseudo Url, we can't use QUrl here sadly.
        const QStringView scheme = fileNameView.left(schemeEnd);
        int hostEnd = fileNameView.indexOf(slash, schemeEnd + 3);
        const QString host = decodeHost(
                    fileNameView.mid(schemeEnd + 3, hostEnd - schemeEnd - 3).toString());

        QStringView path = fileNameView.mid(hostEnd);
        if (!path.isEmpty() && path[0] == '/' && startsWithWindowsDriveLetter(path.mid(1)))
            hostEnd++;

        setParts(scheme, host, hostEnd != -1 ? fileNameView.mid(hostEnd) : QStringView());
        return;
    }

    setParts({}, {}, fileNameView);
}

static Result<DeviceFileAccess *> getFileAccess(const FilePath &filePath)
{
    if (filePath.isLocal())
        return DesktopDeviceFileAccess::instance();

    if (!deviceFileHooks().fileAccess) {
        // Happens during startup and in tst_fsengine
        QTC_CHECK(false);
        return DesktopDeviceFileAccess::instance();
    }

    return deviceFileHooks().fileAccess(filePath);
}

DeviceFileAccess *FilePath::fileAccess() const
{
    static DeviceFileAccess dummy;
    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    QTC_ASSERT_RESULT(access, return &dummy);
    return *access;
}

bool FilePath::hasFileAccess() const
{
    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    return access.has_value();
}

FilePathInfo FilePath::filePathInfo() const
{
    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return {};

    const Result<FilePathInfo> res = (*access)->filePathInfo(*this);
    if (!res) {
        logError("filePathInfo", res.error());
        return {};
    }
    return *res;
}

/*!
   Returns a bool indicating whether a file or directory with this FilePath exists.
*/
bool FilePath::exists() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->exists(*this);
    if (!res) {
        logError("exists", res.error());
        return false;
    }
    return *res;
}

/*!
    Returns a bool indicating whether this is an executable file.
*/
bool FilePath::isExecutableFile() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->isExecutableFile(*this);
    if (!res) {
        logError("isExecutableFile", res.error());
        return false;
    }
    return *res;
}

/*!
    Returns a bool indicating whether this is a writable directory.
*/
bool FilePath::isWritableDir() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->isWritableDirectory(*this);
    if (!res) {
        logError("isWritableDir", res.error());
        return false;
    }
    return *res;
}

/*!
    Returns a bool indicating whether this is a writable file.
*/
bool FilePath::isWritableFile() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->isWritableFile(*this);
    if (!res) {
        logError("isWritableFile", res.error());
        return false;
    }
    return *res;
}

bool FilePath::isReadableFile() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->isReadableFile(*this);
    if (!res) {
        logError("isReadableFile", res.error());
        return false;
    }
    return *res;
}

bool FilePath::isReadableDir() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->isReadableDirectory(*this);
    if (!res) {
        logError("isReadableDir", res.error());
        return false;
    }
    return *res;
}

bool FilePath::isFile() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->isFile(*this);
    if (!res) {
        logError("isFile", res.error());
        return false;
    }
    return *res;
}

bool FilePath::isDir() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->isDirectory(*this);
    if (!res) {
        logError("isDir", res.error());
        return false;
    }
    return *res;
}

bool FilePath::isSymLink() const
{
    if (isEmpty())
        return false;

    const Result<DeviceFileAccess *> access = getFileAccess(*this);
    if (!access)
        return false;

    const Result<bool> res = (*access)->isSymLink(*this);
    if (!res) {
        logError("isSymLink", res.error());
        return false;
    }
    return *res;
}

/*!
    Constructs a FilePath from \a filepath. The \a defaultExtension is appended
    to \a filepath if that does not have an extension already.

    \a filepath is not checked for validity.
*/
FilePath FilePath::fromStringWithExtension(const QString &filepath, const QString &defaultExtension)
{
    if (filepath.isEmpty() || defaultExtension.isEmpty())
        return FilePath::fromString(filepath);

    FilePath rc = FilePath::fromString(filepath);
    // Add extension unless user specified something else
    const QChar dot = QLatin1Char('.');
    if (!rc.fileName().contains(dot)) {
        if (!defaultExtension.startsWith(dot))
            rc = rc.stringAppended(dot);
        rc = rc.stringAppended(defaultExtension);
    }
    return rc;
}

/*!
    Constructs a FilePath from \a filePath

    The path \a filePath is cleaned, and ~ is replaced by the home path.
*/
FilePath FilePath::fromUserInput(const QString &filePath)
{
    const QString expandedPath = filePath == "~" ? QDir::homePath()
                                 : filePath.startsWith("~/")
                                     ? (QDir::homePath() + "/" + filePath.mid(2))
                                     : filePath;
    return FilePath::fromString(doCleanPath(expandedPath));
}

/*!
    Constructs a FilePath from \a filename with \a filenameSize, which is
    encoded as UTF-8.

   \a filename is not checked for validity.
*/
FilePath FilePath::fromUtf8(const char *filename, int filenameSize)
{
    return FilePath::fromString(QString::fromUtf8(filename, filenameSize));
}

FilePath FilePath::fromSettings(const QVariant &variant)
{
    if (variant.typeId() == QMetaType::QUrl) {
        const QUrl url = variant.toUrl();
        return FilePath::fromParts(url.scheme(), url.host(), url.path());
    }

    // The installer sometimes fails and adds "docker:/..." instead of "docker://...
    // So we fix these paths here in those cases.
    QString data = variant.toString();
    if (data.length() > 8 && data.startsWith("docker:/") && data[8] != '/') {
        qWarning() << "Broken path in settings:" << data << ", applying workaround.";
        data.insert(8, '/');
    }

    return FilePath::fromUserInput(data);
}

FilePaths FilePath::fromSettingsList(const QVariant &variant)
{
    return FilePaths::fromSettings(variant);
}

QVariant FilePath::toSettingsList(const FilePaths &filePaths)
{
    return filePaths.toSettings();
}

QVariant FilePath::toSettings() const
{
    return toUrlishString();
}

FilePaths FilePaths::fromSettings(const QVariant &variant)
{
    return FilePaths(transform(variant.toList(), &FilePath::fromSettings));
}

QVariant FilePaths::toSettings() const
{
    return transform(*this, &FilePath::toSettings);
}

FilePaths FilePaths::fromStrings(const QStringList &fileNames)
{
    return transform(fileNames, &FilePath::fromString);
}

FilePaths FilePaths::resolvePaths(const FilePath &anchor, const QStringList &fileNames)
{
    FilePaths result;
    for (const QString &fileName : fileNames)
        result.append(anchor.resolvePath(fileName));
    return result;
}

QStringList FilePaths::toFsPathStrings() const
{
    return transform(*this, &FilePath::toFSPathString);
}

QString FilePaths::toUserOutput(const QString &separator) const
{
    return transform(*this, &FilePath::toUserOutput).join(separator);
}

/*!
    Returns the FilePath as a variant.

    To be used for type-agnostic internal interfaces like storage in
    QAbstractItemModels.
*/
QVariant FilePath::toVariant() const
{
    // FIXME: Use qVariantFromValue
    return toSettings();
}

/*!
    Returns whether FilePath is a child of \a s.
*/
bool FilePath::isChildOf(const FilePath &s) const
{
    if (!s.isSameDevice(*this))
        return false;
    if (s.isEmpty())
        return false;
    const QStringView p = pathView();
    const QStringView sp = s.pathView();
    if (!p.startsWith(sp, caseSensitivity()))
        return false;
    if (p.size() <= sp.size())
        return false;
    // s is root, '/' was already tested in startsWith
    if (sp.endsWith(QLatin1Char('/')))
        return true;
    // s is a directory, next character should be '/' (/tmpdir is NOT a child of /tmp)
    return sp.isEmpty() || p.at(sp.size()) == QLatin1Char('/');
}

/*!
    Returns whether \c path() starts with \a s.
*/
bool FilePath::startsWith(const QString &s) const
{
    return pathView().startsWith(s, caseSensitivity());
}

/*!
   Returns whether \c path() ends with \a s.
*/
bool FilePath::endsWith(const QString &s) const
{
    return pathView().endsWith(s, caseSensitivity());
}

/*!
   Returns whether \c path() contains \a s.
*/
bool FilePath::contains(const QString &s) const
{
    return pathView().contains(s, caseSensitivity());
}

/*!
    \brief Checks whether the FilePath starts with a drive letter.
    Returns whether FilePath starts with a drive letter
*/
bool FilePath::startsWithDriveLetter() const
{
    QStringView p = pathView();
    return p.size() >= 2 && isWindowsDriveLetter(p[0]) && p.at(1) == ':';
}

/*!
    \brief Relative path from \a parent to this.

    Returns a empty \c FilePath if this is not a child of \a parent.
    \a parent is the Parent to calculate the relative path to.
    That is, this never returns a path starting with "../"

    Returns the relative path of this to \a parent if this is a child of \a parent.
*/
FilePath FilePath::relativeChildPath(const FilePath &parent) const
{
    FilePath res;
    if (isChildOf(parent)) {
        QStringView p = pathView().mid(parent.pathView().size());
        if (p.startsWith('/'))
            p = p.mid(1);
        res.setParts(scheme(), host(), p);
    }
    return res;
}

static QString calcRelativePath(QStringView absolutePath, QStringView absoluteAnchorPath)
{
    if (absolutePath.isEmpty() || absoluteAnchorPath.isEmpty())
        return QString();
    // TODO using split() instead of parsing the strings by char index is slow
    // and needs more memory (but the easiest implementation for now)
    const QList<QStringView> splits1 = absolutePath.split('/');
    const QList<QStringView> splits2 = absoluteAnchorPath.split('/');
    int i = 0;
    while (i < splits1.count() && i < splits2.count() && splits1.at(i) == splits2.at(i))
        ++i;
    QString relativePath;
    int j = i;
    bool addslash = false;
    while (j < splits2.count()) {
        if (!splits2.at(j).isEmpty()) {
            if (addslash)
                relativePath += '/';
            relativePath += u"..";
            addslash = true;
        }
        ++j;
    }
    while (i < splits1.count()) {
        if (!splits1.at(i).isEmpty()) {
            if (addslash)
                relativePath += '/';
            relativePath += splits1.at(i);
            addslash = true;
        }
        ++i;
    }
    if (relativePath.isEmpty())
        return QString(".");
    return relativePath;
}

/*!
    Returns the relative path of FilePath from the directory \a anchorDir.
    FilePath and anchor directory may be files or directories.

    Example usage:

    \code
        FilePath filePath("/foo/b/ar/file.txt");
        FilePath relativePath = filePath.relativePathFrom("/foo/c");
        qDebug() << relativePath
    \endcode

    The debug output will be "../b/ar/file.txt".
*/

QString FilePath::relativePathFromDir(const FilePath &anchorDir) const
{
    QTC_ASSERT(isSameDevice(anchorDir), return path());

    const FilePath absPath = absoluteFilePath();
    const FilePath absoluteAnchorPath = anchorDir.absoluteFilePath();

    QString relativeFilePath = calcRelativePath(absPath.pathView(), absoluteAnchorPath.pathView());

    return relativeFilePath;
}

QString FilePath::relativeNativePathFromDir(const FilePath &anchorDir) const
{
    QString rel = relativePathFromDir(anchorDir);
    if (osType() == OsTypeWindows)
        rel.replace('/', '\\');
    return rel;
}

/*!
    Returns a path corresponding to \a newPath object on the
    same device as the current object.

    This may involve device-specific translations like converting
    windows style paths to unix style paths with suitable file
    system case or handling of drive letters: C:/dev/src -> /c/dev/src

    Example usage:
    \code
        localDir = FilePath("/tmp/workingdir");
        executable = FilePath::fromUrl("docker://123/bin/ls")
        realDir = executable.withNewMappedPath(localDir)
        assert(realDir == FilePath::fromUrl("docker://123/tmp/workingdir"))
    \endcode
*/
FilePath FilePath::withNewMappedPath(const FilePath &newPath) const
{
    FilePath res;
    res.setParts(scheme(), host(), fileAccess()->mapToDevicePath(newPath.path()));
    return res;
}

/*!
    Returns a path with the \a n characters of the local path removed.
    Example usage:
    \code
        backup = FilePath("/tmp/example.txt.backup");
        real = backup.chopped(7);
        assert(real == FilePath("/tmp/example.txt"))
    \endcode
*/

FilePath FilePath::chopped(int n) const
{
    FilePath res;
    res.setParts(scheme(), host(), path().chopped(n));
    return res;
}

/*!
    Returns a FilePath with local path \a newPath on the same device
    as the current object.

    Example usage:
    \code
        devicePath = FilePath("docker://123/tmp");
        newPath = devicePath.withNewPath("/bin/ls");
        assert(realDir == FilePath::fromUrl("docker://123/bin/ls"))
    \endcode
*/
FilePath FilePath::withNewPath(const QString &newPath) const
{
    FilePath res;
    res.setParts(scheme(), host(), newPath);
    return res;
}

/*!
    Search for a binary corresponding to this object on each directory entry
    specified by \a dirs matching the \a filter with the \a matchScope of the
    file path.

    Example usage:
    \code
        binary = FilePath::fromUrl("docker://123/./make);
        fullPath = binary.searchInDirectories(binary.deviceEnvironment().path());
        assert(fullPath == FilePath::fromUrl("docker://123/usr/bin/make"))
    \endcode
*/

FilePath FilePath::searchInDirectories(const FilePaths &dirs,
                                       const FilePathPredicate &filter,
                                       MatchScope matchScope) const
{
    if (isEmpty())
        return {};

    const FilePaths execs = appendExeExtensions(*this, matchScope);

    if (isAbsolutePath()) {
        for (const FilePath &filePath : execs) {
            if (filePath.isExecutableFile() && (!filter || filter(filePath)))
                return filePath;
        }
        return {};
    }

    QSet<FilePath> alreadyCheckedDirectories;

    for (const FilePath &dir : dirs) {
        // Compare the initial size of the set with the size after insertion to check
        // if the directory was already checked.
        const int initialCount = alreadyCheckedDirectories.count();
        alreadyCheckedDirectories.insert(dir);
        const bool wasAlreadyChecked = alreadyCheckedDirectories.count() == initialCount;

        if (dir.isEmpty() || wasAlreadyChecked)
            continue;

        for (const FilePath &exe : execs) {
            const FilePath filePath = dir / exe.path();
            if (filePath.isExecutableFile() && (!filter || filter(filePath)))
                return filePath;
        }
    }

    return {};
}

FilePaths FilePath::searchAllInDirectories(const FilePaths &dirs,
                                           const FilePathPredicate &filter,
                                           MatchScope matchScope) const
{
    if (isEmpty())
        return {};

    const FilePaths execs = appendExeExtensions(*this, matchScope);

    FilePaths result;
    if (isAbsolutePath()) {
        for (const FilePath &filePath : execs) {
            if (filePath.isExecutableFile() && (!filter || filter(filePath)))
                result.append(filePath);
        }
        return result;
    }

    QSet<FilePath> alreadyCheckedDirectories;

    for (const FilePath &dir : dirs) {
        // Compare the initial size of the set with the size after insertion to check
        // if the directory was already checked.
        const int initialCount = alreadyCheckedDirectories.count();
        alreadyCheckedDirectories.insert(dir);
        const bool wasAlreadyChecked = alreadyCheckedDirectories.count() == initialCount;

        if (dir.isEmpty() || wasAlreadyChecked)
            continue;

        for (const FilePath &exe : execs) {
            const FilePath filePath = dir / exe.path();
            if (filePath.isExecutableFile() && (!filter || filter(filePath)))
                result.append(filePath);
        }
    }

    return result;
}

static FilePaths dirsFromPath(const FilePath &anchor,
                              const FilePaths &additionalDirs,
                              FilePath::PathAmending amending)
{
    FilePaths directories = anchor.devicePathEnvironmentVariable();

    if (!additionalDirs.isEmpty()) {
        if (amending == FilePath::AppendToPath)
            directories.append(additionalDirs);
        else
            directories = additionalDirs + directories;
    }

    return directories;
}

FilePath FilePath::searchInPath() const
{
    return searchInPath({});
}

FilePath FilePath::searchInPath(const FilePaths &additionalDirs,
                                PathAmending amending,
                                const FilePathPredicate &filter,
                                MatchScope matchScope) const
{
    if (isAbsolutePath())
        return *this;

    const FilePaths directories = dirsFromPath(*this, additionalDirs, amending);
    return searchInDirectories(directories, filter, matchScope);
}

FilePaths FilePath::searchAllInPath() const
{
    return searchAllInPath({});
}

FilePaths FilePath::searchAllInPath(const FilePaths &additionalDirs,
                                    PathAmending amending,
                                    const FilePathPredicate &filter,
                                    MatchScope matchScope) const
{
    const FilePaths directories = dirsFromPath(*this, additionalDirs, amending);
    return searchAllInDirectories(directories, filter, matchScope);
}

FilePath FilePath::searchHereAndInParents(const QString &fileName, QDir::Filter type) const
{
    return searchHereAndInParents(QStringList{fileName}, type);
}

FilePath FilePath::searchHereAndInParents(const QStringList &fileNames, QDir::Filter type) const
{
    const bool wantFile = type == QDir::Files;
    const bool wantDir = type == QDir::Dirs;
    QTC_ASSERT(wantFile || wantDir, return {});

    FilePath file;
    const auto constraint = [wantFile, wantDir, &fileNames, &file](const FilePath &dir) {
        for (const QString &fileName : fileNames) {
            const FilePath candidate = dir.pathAppended(fileName);
            if ((wantFile && candidate.isFile()) || (wantDir && file.isDir())) {
                file = candidate;
                return IterationPolicy::Stop;
            }
        }
        return IterationPolicy::Continue;
    };
    searchHereAndInParents(constraint);
    return file;
}

void FilePath::searchHereAndInParents(const std::function<IterationPolicy(const FilePath &)> &constraint) const
{
    QTC_ASSERT(!isEmpty(), return);

    FilePath dir = *this;
    if (!isDir())
        dir = dir.parentDir();

    for (const FilePath &parent : PathAndParents(dir)) {
        if (constraint(parent) == IterationPolicy::Stop)
            break;
    }
}

Environment FilePath::deviceEnvironment() const
{
    Result<Environment> env = deviceEnvironmentWithError();
    QTC_ASSERT_RESULT(env, return {});
    return *env;
}

Result<Environment> FilePath::deviceEnvironmentWithError() const
{
    if (!isLocal()) {
        QTC_ASSERT(deviceFileHooks().environment, return {});
        return deviceFileHooks().environment(*this);
    }
    return Environment::systemEnvironment();
}

FilePaths FilePath::devicePathEnvironmentVariable() const
{
    FilePaths result = deviceEnvironment().path();
    if (!isLocal()) {
        for (FilePath &dir : result)
            dir.setParts(this->scheme(), this->host(), dir.path());
    }
    return result;
}

QString FilePath::formatFilePaths(const FilePaths &files, const QString &separator)
{
    const QStringList nativeFiles = transform(files, &FilePath::toUserOutput);
    return nativeFiles.join(separator);
}

void FilePath::removeDuplicates(FilePaths &files)
{
    files = Utils::filteredUnique(files);
}

void FilePath::sort(FilePaths &files)
{
    std::sort(files.begin(), files.end(), [](const FilePath &a, const FilePath &b) {
        const int scheme = a.scheme().compare(b.scheme());
        if (scheme != 0)
            return scheme < 0;

        const int host = a.host().compare(b.host());
        if (host != 0)
            return host < 0;

        return a.pathView() < b.pathView();
    });
}

void join(QString &left, const QString &right)
{
    QStringView r(right);
    if (r.startsWith('/'))
        r = r.mid(1);

    if (!left.endsWith('/'))
        left += '/';
    left += r;
}

FilePath FilePath::pathAppended(const QString &path) const
{
    if (path.isEmpty())
        return *this;

    QString other = path;
    other.replace('\\', '/');

    // FIXME: This should possibly be a QTC_UNEXPECTED later
    // but triggers too often currently.
    if (isEmpty())
        return withNewPath(other);

    QString p = this->path();
    join(p, other);

    return withNewPath(p);
}

FilePath FilePath::stringAppended(const QString &str) const
{
    return FilePath::fromString(toUrlishString() + str);
}

std::optional<FilePath> FilePath::tailRemoved(const QString &str) const
{
    if (pathView().endsWith(str))
        return withNewPath(pathView().chopped(str.size()).toString());
    return {};
}

std::optional<FilePath> FilePath::prefixRemoved(const QString &str) const
{
    if (pathView().startsWith(str))
        return withNewPath(pathView().mid(str.size()).toString());
    return {};
}

QDateTime FilePath::lastModified() const
{
    const Result<QDateTime> res = fileAccess()->lastModified(*this);
    if (!res) {
        logError("lastModified", res.error());
        return {};
    }
    return *res;
}

QFile::Permissions FilePath::permissions() const
{
    const Result<QFile::Permissions> res = fileAccess()->permissions(*this);
    if (!res) {
        logError("permissions", res.error());
        return {};
    }
    return *res;
}

Result<> FilePath::setPermissions(QFile::Permissions permissions) const
{
    const Result<> res = fileAccess()->setPermissions(*this, permissions);
    if (!res)
        logError("setPermissions", res.error());
    return res;
}

Result<> FilePath::makeWritable() const
{
    return setPermissions(permissions() | QFile::WriteUser);
}

OsType FilePath::osType() const
{
    if (isLocal())
        return HostOsInfo::hostOs();

    QTC_ASSERT(deviceFileHooks().osType, return HostOsInfo::hostOs());
    return deviceFileHooks().osType(*this);
}

Result<> FilePath::removeFile() const
{
    return fileAccess()->removeFile(*this);
}

/*!
    Removes the directory this filePath refers too and its subdirectories recursively.
*/
Result<> FilePath::removeRecursively() const
{
    return fileAccess()->removeRecursively(*this);
}

Result<> FilePath::copyRecursively(const FilePath &target) const
{
    return fileAccess()->copyRecursively(*this, target);
}

Result<> FilePath::copyFile(const FilePath &target) const
{
    if (!isSameDevice(target)) {
        // FIXME: This does not scale.
        const Result<QByteArray> contents = fileContents();
        if (!contents) {
            return ResultError(
                Tr::tr("Error while trying to copy file: %1").arg(contents.error()));
        }

        const QFile::Permissions perms = permissions();
        const Result<qint64> copyResult = target.writeFileContents(*contents);

        if (!copyResult)
            return ResultError(Tr::tr("Could not copy file: %1").arg(copyResult.error()));

        if (!target.setPermissions(perms)) {
            target.removeFile();
            return ResultError(
                Tr::tr("Could not set permissions on \"%1\"").arg(target.toUrlishString()));
        }

        return ResultOk;
    }
    return fileAccess()->copyFile(*this, target);
}

/*!
    Creates a symbolic link \a symLink pointing to this file path.

    \note On Windows, this function will not work unless you have admin rights
          or developer mode is enabled.
*/
Result<> FilePath::createSymLink(const FilePath &symLink) const
{
    if (!isSameDevice(symLink)) {
        return ResultError(
            Tr::tr(
                "Cannot create symbolic link to \"%1\" at \"%2\": Paths do not refer to the "
                "same device.").arg(toUserOutput(), symLink.toUserOutput()));
    }
    return fileAccess()->createSymLink(*this, symLink);
}

Result<> FilePath::renameFile(const FilePath &target) const
{
    if (isSameDevice(target))
        return fileAccess()->renameFile(*this, target);

    const Result<> copyResult = copyFile(target);
    if (!copyResult)
        return copyResult;

    const Result<> removeResult = removeFile();
    if (removeResult)
        return ResultOk;

    // If we fail to remove the source file, we remove the target file to return to the
    // original state.
    Result<> rmResult = target.removeFile();
    QTC_CHECK_RESULT(rmResult);
    return ResultError(
        Tr::tr("Failed to move %1 to %2. Removing the source file failed: %3")
            .arg(toUserOutput())
            .arg(target.toUserOutput())
            .arg(rmResult.error()));
}

qint64 FilePath::fileSize() const
{
    const Result<qint64> res = fileAccess()->fileSize(*this);
    if (!res) {
        logError("fileSize", res.error());
        return {};
    }
    return *res;
}

QString FilePath::owner() const
{
    const Result<QString> res = fileAccess()->owner(*this);
    if (!res) {
        logError("owner", res.error());
        return {};
    }
    return *res;
}

uint FilePath::ownerId() const
{
    const Result<uint> res = fileAccess()->ownerId(*this);
    if (!res) {
        logError("ownerId", res.error());
        return {};
    }
    return *res;
}

QString FilePath::group() const
{
    const Result<QString> res = fileAccess()->group(*this);
    if (!res) {
        logError("group", res.error());
        return {};
    }
    return *res;
}

uint FilePath::groupId() const
{
    const Result<uint> res = fileAccess()->groupId(*this);
    if (!res) {
        logError("groupId", res.error());
        return {};
    }
    return *res;
}

qint64 FilePath::bytesAvailable() const
{
    const Result<qint64> res = fileAccess()->bytesAvailable(*this);
    if (!res) {
        logError("bytesAvailable", res.error());
        return {};
    }
    return *res;
}

/*!
    \brief Checks if this is newer than \a timeStamp.

    The time stamp \a timeStamp to compare with.
    Returns \c true if this is newer than \a timeStamp.
    If this is a directory, the function will recursively check all files and return
    \c true if one of them is newer than \a timeStamp. If this is a single file, \c true will
    be returned if the file is newer than \a timeStamp.

     Returns whether at least one file in the file path has a newer date than
     \a timeStamp.
*/
bool FilePath::isNewerThan(const QDateTime &timeStamp) const
{
    if (!exists() || lastModified() > timeStamp)
        return true;
    if (isDir()) {
        const FilePaths dirContents = dirEntries(QDir::Files | QDir::Dirs | QDir::NoDotAndDotDot);
        for (const FilePath &entry : dirContents) {
            if (entry.isNewerThan(timeStamp))
                return true;
        }
    }
    return false;
}

/*!
    \brief Returns the caseSensitivity of the path.

    This is currently only based on the Host OS.
    For device paths, \c Qt::CaseSensitive is always returned.
*/
Qt::CaseSensitivity FilePath::caseSensitivity() const
{
    if (m_schemeLen == 0)
        return HostOsInfo::fileNameCaseSensitivity();

    // FIXME: This could or possibly should the target device's file name case sensitivity
    // into account by diverting to IDevice. However, as this is expensive and we are
    // in time-critical path here, we go with "good enough" for now:
    // The first approximation is "most things are case-sensitive".
    return Qt::CaseSensitive;
}

/*!
    \brief Returns the separator of path components for this path.

    Returns the path separator of the path.
*/
QChar FilePath::pathComponentSeparator() const
{
    return osType() == OsTypeWindows ? u'\\' : u'/';
}

/*!
    \brief Returns the path list separator for the device this path belongs to.

    Returns the path list separator of the device for this path.
*/
QChar FilePath::pathListSeparator() const
{
    return osType() == OsTypeWindows ? u';' : u':';
}

TextEncoding FilePath::processStdOutEncoding() const
{
    return fileAccess()->processStdOutEncoding(*this);
}

TextEncoding FilePath::processStdErrEncoding() const
{
    return fileAccess()->processStdErrEncoding(*this);
}

/*!
    \brief Recursively resolves symlinks if this is a symlink.

    To resolve symlinks anywhere in the path, see canonicalPath.
    Unlike QFileInfo::canonicalFilePath(), this function will still return the expected deepest
    target file even if the symlink is dangling.

    \note Maximum recursion depth == 16.

    Returns the symlink target file path.
*/
FilePath FilePath::resolveSymlinks() const
{
    FilePath current = *this;
    int links = 16;
    while (links--) {
        const QList<QStringView> components = current.pathComponents();
        FilePath pathToTest;
        bool resolved = false;
        for (const QStringView &component : components) {
            if (pathToTest.isEmpty())
                pathToTest = current.withNewPath(component.toString());
            else
                pathToTest = pathToTest / component.toString();
            if (!resolved) {
                const FilePath target = pathToTest.symLinkTarget();
                if (!target.isEmpty()) {
                    resolved = true;
                    pathToTest = target;
                }
            }
        }
        if (!resolved)
            return current;
        current = pathToTest;
    }
    return current;
}

/*!
    Recursively resolves possibly present symlinks in this file name.

    On Windows, also resolves SUBST and re-mounted NTFS drives.
    Unlike QFileInfo::canonicalFilePath(), this function will not return an empty
    string if path doesn't exist.

    Returns the canonical path.
*/
FilePath FilePath::canonicalPath() const
{
    if (!isLocal()) {
        // FIXME: Not a full solution, but it stays on the right device.
        return *this;
    }

#ifdef Q_OS_WIN
    DWORD flagsAndAttrs = FILE_ATTRIBUTE_NORMAL;
    if (isDir())
        flagsAndAttrs |= FILE_FLAG_BACKUP_SEMANTICS;
    const HANDLE fileHandle = CreateFile(
                toUserOutput().toStdWString().c_str(),
                GENERIC_READ,
                FILE_SHARE_READ,
                nullptr,
                OPEN_EXISTING,
                flagsAndAttrs,
                nullptr);
    if (fileHandle != INVALID_HANDLE_VALUE) {
        TCHAR normalizedPath[MAX_PATH];
        const auto length = GetFinalPathNameByHandleW(
                    fileHandle,
                    normalizedPath,
                    MAX_PATH,
                    FILE_NAME_NORMALIZED);
        CloseHandle(fileHandle);
        if (length > 0)
            return fromUserInput(QString::fromStdWString(std::wstring(normalizedPath, length)));
    }
#endif

    const QString result = toFileInfo().canonicalFilePath();
    if (!result.isEmpty())
        return fromString(result);

    return *this;
}

FilePath FilePath::operator/(const QString &str) const
{
    return pathAppended(str);
}

FilePath &FilePath::operator/=(const QString &str)
{
    *this = pathAppended(str);
    return *this;
}

/*!
    \brief Clears all parts of the FilePath.
*/
void FilePath::clear()
{
    *this = {};
}

/*!
    \brief Checks if the path() is empty.

    Returns true if the path() is empty.
    The Host and Scheme of the part are ignored.
*/
bool FilePath::isEmpty() const
{
    return m_pathLen == 0;
}

/*!
    \brief Converts the path to a possibly shortened path with native separators.

    Like QDir::toNativeSeparators(), but use prefix '~' instead of $HOME on unix systems when an
    absolute path is given.

    Returns the possibly shortened path with native separators.
*/
QString FilePath::shortNativePath() const
{
    if (HostOsInfo::isAnyUnixHost()) {
        const FilePath home = FileUtils::homePath();
        if (isChildOf(home)) {
            return QLatin1Char('~') + QDir::separator()
                + QDir::toNativeSeparators(relativeChildPath(home).toUrlishString());
        }
    }
    return toUserOutput();
}

/*!
    \brief Checks whether the path is relative.

    Returns true if the path starts neither with a slash, nor with a letter
    and a colon, nor with a colon and a slash.

    \note This is independent of the platform on which \QC currently runs,
    so this does not necessarily match the platform's definition of
    a relative path. Use with care, and try to avoid.

    \sa isAbsolutePath()
*/
bool FilePath::isRelativePath() const
{
    return !isAbsolutePath();
}

/*!
    \brief Checks whether the path is absolute.

    Returns true if the path starts with a slash, or with a letter and a colon,
    or with a colon and a slash.

    \note This is independent of the platform on which \QC currently runs,
    so this does not necessarily match the platform's definition of
    an absolute path. Use with care, and try to avoid.

    \sa isRelativePath()
*/
bool FilePath::isAbsolutePath() const
{
    return isAbsolutePathHelper(pathView());
}

/*!
    \brief Appends the \a tail to this, if the tail is a relative path.

    Returns the tail if the tail is absolute, otherwise this + tail.
*/
FilePath FilePath::resolvePath(const FilePath &tail) const
{
    if (tail.isEmpty())
        return cleanPath();
    if (tail.isRelativePath())
        return pathAppended(tail.path()).cleanPath();
    return tail;
}

/*!
    \brief Appends the \a tail to this, if the tail is a relative path.

    Returns this with new path tail if the tail is absolute, otherwise this + tail.
*/
FilePath FilePath::resolvePath(const QString &tail) const
{
    if (tail.isEmpty())
        return *this;
    const FilePath clean = FilePath::fromUserInput(tail);
    if (clean.isAbsolutePath())
        return withNewPath(clean.path());
    return pathAppended(clean.path()).cleanPath();
}

Result<FilePath> FilePath::localSource() const
{
    if (isLocal())
        return *this;

    QTC_ASSERT(deviceFileHooks().localSource,
               return ResultError(Tr::tr("No \"localSource\" device hook set.")));
    return deviceFileHooks().localSource(*this);
}

/*!
    \brief Cleans path part similar to \c QDir::cleanPath().

    \list
    \li directory separators normalized (that is, platform-native
     separators converted to "/") and redundant ones removed, and "."s and ".."s
      resolved (as far as possible).
    \li Symbolic links are kept. This function does not return the
      canonical path, but rather the simplest version of the input.
      For example, "./local" becomes "local", "local/../bin" becomes
      "bin" and "/local/usr/../bin" becomes "/local/bin".
    \endlist
*/
FilePath FilePath::cleanPath() const
{
    return withNewPath(doCleanPath(path()));
}

/*!
    On Linux/Mac replace user's home path with ~ in the \c toString()
    result for this path after cleaning.

    If path is not sub of home path, or when running on Windows, returns the input
*/
QString FilePath::withTildeHomePath() const
{
    if (osType() == OsTypeWindows)
        return toUrlishString();

    if (!isLocal())
        return toUrlishString();

    static const QString homePath = QDir::homePath();

    QString outPath = cleanPath().absoluteFilePath().path();
    if (outPath.startsWith(homePath))
       return '~' + outPath.mid(homePath.size());

    return toUrlishString();
}

FilePath FilePaths::commonPath() const
{
    if (isEmpty())
        return {};

    if (count() == 1)
        return constFirst();

    const FilePath &first = constFirst();
    const FilePaths others = mid(1);
    FilePath result;

    // Common scheme
    const QStringView commonScheme = first.scheme();
    auto sameScheme = [&commonScheme] (const FilePath &fp) {
        return commonScheme == fp.scheme();
    };
    if (!allOf(others, sameScheme))
        return result;
    result.setParts(commonScheme, {}, {});

    // Common host
    const QStringView commonHost = first.host();
    auto sameHost = [&commonHost] (const FilePath &fp) {
        return commonHost == fp.host();
    };
    if (!allOf(others, sameHost))
        return result;
    result.setParts(commonScheme, commonHost, {});

    // Common path
    QString commonPath;
    auto sameBasePath = [&commonPath] (const FilePath &fp) {
        return QString(fp.path() + '/').startsWith(commonPath);
    };
    const QStringList pathSegments = first.path().split('/');
    for (const QString &segment : pathSegments) {
        commonPath += segment + '/';
        if (!allOf(others, sameBasePath))
            return result;
        result.setParts(commonScheme, commonHost, commonPath.chopped(1));
    }

    return result;
}

void FilePaths::sort()
{
    std::sort(begin(), end(), std::less<FilePath>());
}

std::vector<Utils::Result<std::unique_ptr<FilePathWatcher>>> FilePaths::watch() const
{
    return FilePath::watch(*this);
}

QTextStream &operator<<(QTextStream &s, const FilePath &fn)
{
    return s << fn.toUrlishString();
}

// FileFilter
FileFilter::FileFilter(const QStringList &nameFilters,
                       const QDir::Filters fileFilters,
                       const QDirIterator::IteratorFlags flags)
    : nameFilters(nameFilters),
      fileFilters(fileFilters),
      iteratorFlags(flags)
{
    QTC_CHECK(this->fileFilters != QDir::Filters());
}

QStringList FileFilter::asFindArguments(const QString &path) const
{
    QStringList arguments;

    const QDir::Filters filters = fileFilters;

    if (iteratorFlags.testFlag(QDirIterator::FollowSymlinks))
        arguments << "-L";
    else
        arguments << "-H";

    arguments << path;

    if (!iteratorFlags.testFlag(QDirIterator::Subdirectories))
        arguments.append({"-maxdepth", "1"});

    QStringList filterOptions;

    if (!(filters & QDir::Hidden))
        filterOptions << "!" << "-name" << ".*";

    QStringList filterFilesAndDirs;
    if (filters.testFlag(QDir::Dirs))
        filterFilesAndDirs << "-type" << "d";
    if (filters.testFlag(QDir::Files)) {
        if (!filterFilesAndDirs.isEmpty())
            filterFilesAndDirs << "-o";
        filterFilesAndDirs << "-type" << "f";
    }

    if (!filters.testFlag(QDir::NoSymLinks)) {
        if (!filterFilesAndDirs.isEmpty())
            filterFilesAndDirs << "-o";
        filterFilesAndDirs << "-type" << "l";
    }

    if (!filterFilesAndDirs.isEmpty())
        filterOptions << "(" << filterFilesAndDirs << ")";

    QStringList accessOptions;
    if (filters & QDir::Readable)
        accessOptions << "-readable";
    if (filters & QDir::Writable) {
        if (!accessOptions.isEmpty())
            accessOptions << "-o";
        accessOptions << "-writable";
    }
    if (filters & QDir::Executable) {
        if (!accessOptions.isEmpty())
            accessOptions << "-o";
        accessOptions << "-executable";
    }

    if (!accessOptions.isEmpty())
        filterOptions << "(" << accessOptions << ")";

    QTC_CHECK(filters ^ QDir::AllDirs);
    QTC_CHECK(filters ^ QDir::Drives);
    QTC_CHECK(filters ^ QDir::NoDot);
    QTC_CHECK(filters ^ QDir::NoDotDot);
    QTC_CHECK(filters ^ QDir::System);

    const QString nameOption = (filters & QDir::CaseSensitive) ? QString{"-name"}
                                                               : QString{"-iname"};
    if (!nameFilters.isEmpty()) {
        bool isFirst = true;
        filterOptions << "(";
        for (const QString &current : nameFilters) {
            if (!isFirst)
                filterOptions << "-o";
            filterOptions << nameOption << current;
            isFirst = false;
        }
        filterOptions << ")";
    }
    arguments << filterOptions;
    return arguments;
}

QTCREATOR_UTILS_EXPORT bool operator==(const FilePath &first, const FilePath &second)
{
    return FilePath::equals(first, second, first.caseSensitivity());
}

QTCREATOR_UTILS_EXPORT bool operator!=(const FilePath &first, const FilePath &second)
{
    return !(first == second);
}

QTCREATOR_UTILS_EXPORT bool operator<(const FilePath &first, const FilePath &second)
{
    const bool firstNeedsDevice = !first.isLocal();
    const bool secondNeedsDevice = !second.isLocal();

    // If either needs a device, we have to compare host and scheme first.
    if (firstNeedsDevice || secondNeedsDevice) {
        // Paths needing a device are "larger" than those not needing one.
        if (firstNeedsDevice < secondNeedsDevice)
            return true;
        else if (firstNeedsDevice > secondNeedsDevice)
            return false;

        // First we sort by scheme ...
        const int s = first.scheme().compare(second.scheme());
        if (s != 0)
            return s < 0;

        // than by host ...
        const int h = first.host().compare(second.host());
        if (h != 0)
            return h < 0;
    }

    const int p = first.pathView().compare(second.pathView(), first.caseSensitivity());
    return p < 0;
}

QTCREATOR_UTILS_EXPORT bool operator<=(const FilePath &first, const FilePath &second)
{
    return !(second < first);
}

QTCREATOR_UTILS_EXPORT bool operator>(const FilePath &first, const FilePath &second)
{
    return second < first;
}

QTCREATOR_UTILS_EXPORT bool operator>=(const FilePath &first, const FilePath &second)
{
    return !(first < second);
}

QTCREATOR_UTILS_EXPORT size_t qHash(const FilePath &filePath, uint seed)
{
    Q_UNUSED(seed)

    if (filePath.m_hash == 0) {
        if (filePath.caseSensitivity() == Qt::CaseSensitive)
            filePath.m_hash = qHash(QStringView(filePath.m_data), 0);
        else
            filePath.m_hash = qHash(filePath.m_data.toCaseFolded(), 0);
    }

    return filePath.m_hash;
}

QTCREATOR_UTILS_EXPORT size_t qHash(const FilePath &filePath)
{
    return qHash(filePath, 0);
}

QTCREATOR_UTILS_EXPORT QDebug operator<<(QDebug dbg, const FilePath &c)
{
    return dbg << c.toUrlishString();
}

FilePaths firstPaths(const FilePairs &pairs)
{
    return transform(pairs, &FilePair::first);
}

FilePaths secondPaths(const FilePairs &pairs)
{
    return transform(pairs, &FilePair::second);
}

PathAndParents::PathAndParents(const FilePath &p)
    : m_path(p)
{}

PathAndParents::PathAndParents(const FilePath &p, const FilePath &last)
    : m_path(p)
    , m_lastPath(last)
{
    QTC_CHECK(last.isEmpty() || p == last || p.isChildOf(last));
}

PathAndParents::iterator PathAndParents::begin() const
{
    return iterator(m_path);
}

PathAndParents::iterator PathAndParents::end() const
{
    if (m_lastPath.isEmpty())
        return iterator(FilePath());

    const FilePath &endPath = m_lastPath.parentDir();
    if (endPath == m_lastPath) // Did the user specify "root" as the last path?
        return iterator(FilePath());
    QTC_ASSERT(m_path == m_lastPath || m_path.isChildOf(m_lastPath), return iterator(FilePath()));
    return iterator(endPath);
}

PathAndParents::iterator::iterator(const FilePath &p)
    : current(p)
{}

PathAndParents::iterator &PathAndParents::iterator::operator++()
{
    const FilePath newParent = current.parentDir();
    if (newParent == current)
        current = FilePath(); // Reached the root, stop iterating.
    else
        current = newParent;
    return *this;
}

PathAndParents::iterator PathAndParents::iterator::operator++(int)
{
    iterator temp = *this;
    ++*this;
    return temp;
}

bool PathAndParents::iterator::operator!=(const iterator &other) const
{
    return !(*this == other);
}

bool PathAndParents::iterator::operator==(const iterator &other) const
{
    return current == other.current;
}

const FilePath &PathAndParents::iterator::operator*() const
{
    return current;
}

} // namespace Utils
